#   Copyright 2013 Nebula Inc.
#
#   Licensed under the Apache License, Version 2.0 (the "License"); you may
#   not use this file except in compliance with the License. You may obtain
#   a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#   WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#   License for the specific language governing permissions and limitations
#   under the License.
#

"""Object v1 action implementations"""

import logging

from osc_lib.cli import format_columns
from osc_lib.cli import parseractions
from osc_lib.command import command
from osc_lib import exceptions
from osc_lib import utils

from openstackclient.common import pagination
from openstackclient.i18n import _


LOG = logging.getLogger(__name__)


class CreateObject(command.Lister):
    _description = _("Upload object to container")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Container for new object'),
        )
        parser.add_argument(
            'objects',
            metavar='<filename>',
            nargs="+",
            help=_('Local filename(s) to upload'),
        )
        parser.add_argument(
            '--name',
            metavar='<name>',
            help=_(
                'Upload a file and rename it. '
                'Can only be used when uploading a single object'
            ),
        )
        return parser

    def take_action(self, parsed_args):
        if parsed_args.name:
            if len(parsed_args.objects) > 1:
                msg = _(
                    'Attempting to upload multiple objects and '
                    'using --name is not permitted'
                )
                raise exceptions.CommandError(msg)
        results = []
        for obj in parsed_args.objects:
            if len(obj) > 1024:
                LOG.warning(
                    _(
                        'Object name is %s characters long, default limit'
                        ' is 1024'
                    ),
                    len(obj),
                )
            data = self.app.client_manager.object_store.object_create(
                container=parsed_args.container,
                object=obj,
                name=parsed_args.name,
            )
            results.append(data)

        columns = ("object", "container", "etag")
        return (
            columns,
            (
                utils.get_dict_properties(
                    s,
                    columns,
                    formatters={},
                )
                for s in results
            ),
        )


class DeleteObject(command.Command):
    _description = _("Delete object from container")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Delete object(s) from <container>'),
        )
        parser.add_argument(
            'objects',
            metavar='<object>',
            nargs="+",
            help=_('Object(s) to delete'),
        )
        return parser

    def take_action(self, parsed_args):
        for obj in parsed_args.objects:
            self.app.client_manager.object_store.object_delete(
                container=parsed_args.container,
                object=obj,
            )


class ListObject(command.Lister):
    _description = _("List objects")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            "container",
            metavar="<container>",
            help=_("Container to list"),
        )
        parser.add_argument(
            "--prefix",
            metavar="<prefix>",
            help=_("Filter list using <prefix>"),
        )
        parser.add_argument(
            "--delimiter",
            metavar="<delimiter>",
            help=_("Roll up items with <delimiter>"),
        )
        pagination.add_marker_pagination_option_to_parser(parser)
        parser.add_argument(
            "--end-marker",
            metavar="<end-marker>",
            help=_("End anchor for paging"),
        )
        parser.add_argument(
            '--long',
            action='store_true',
            default=False,
            help=_('List additional fields in output'),
        )
        parser.add_argument(
            '--all',
            action='store_true',
            default=False,
            help=_('List all objects in container (default is 10000)'),
        )
        return parser

    def take_action(self, parsed_args):
        columns: tuple[str, ...] = ('Name',)
        if parsed_args.long:
            columns += ('Bytes', 'Hash', 'Content Type', 'Last Modified')

        kwargs = {}
        if parsed_args.prefix:
            kwargs['prefix'] = parsed_args.prefix
        if parsed_args.delimiter:
            kwargs['delimiter'] = parsed_args.delimiter
        if parsed_args.marker:
            kwargs['marker'] = parsed_args.marker
        if parsed_args.end_marker:
            kwargs['end_marker'] = parsed_args.end_marker
        if parsed_args.limit:
            kwargs['limit'] = parsed_args.limit
        if parsed_args.all:
            kwargs['full_listing'] = True

        data = self.app.client_manager.object_store.object_list(
            container=parsed_args.container, **kwargs
        )

        return (
            columns,
            (
                utils.get_dict_properties(
                    s,
                    columns,
                    formatters={},
                )
                for s in data
            ),
        )


class SaveObject(command.Command):
    _description = _("Save object locally")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            "--file",
            metavar="<filename>",
            help=_(
                "Destination filename (defaults to object name); using '-'"
                " as the filename will print the file to stdout"
            ),
        )
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Download <object> from <container>'),
        )
        parser.add_argument(
            "object",
            metavar="<object>",
            help=_("Object to save"),
        )
        return parser

    def take_action(self, parsed_args):
        self.app.client_manager.object_store.object_save(
            container=parsed_args.container,
            object=parsed_args.object,
            file=parsed_args.file,
        )


class SetObject(command.Command):
    _description = _("Set object properties")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Modify <object> from <container>'),
        )
        parser.add_argument(
            'object',
            metavar='<object>',
            help=_('Object to modify'),
        )
        parser.add_argument(
            "--property",
            metavar="<key=value>",
            required=True,
            action=parseractions.KeyValueAction,
            help=_(
                "Set a property on this object "
                "(repeat option to set multiple properties)"
            ),
        )
        return parser

    def take_action(self, parsed_args):
        self.app.client_manager.object_store.object_set(
            parsed_args.container,
            parsed_args.object,
            properties=parsed_args.property,
        )


class ShowObject(command.ShowOne):
    _description = _("Display object details")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Display <object> from <container>'),
        )
        parser.add_argument(
            'object',
            metavar='<object>',
            help=_('Object to display'),
        )
        return parser

    def take_action(self, parsed_args):
        data = self.app.client_manager.object_store.object_show(
            container=parsed_args.container,
            object=parsed_args.object,
        )
        if 'properties' in data:
            data['properties'] = format_columns.DictColumn(data['properties'])

        return zip(*sorted(data.items()))


class UnsetObject(command.Command):
    _description = _("Unset object properties")

    def get_parser(self, prog_name):
        parser = super().get_parser(prog_name)
        parser.add_argument(
            'container',
            metavar='<container>',
            help=_('Modify <object> from <container>'),
        )
        parser.add_argument(
            'object',
            metavar='<object>',
            help=_('Object to modify'),
        )
        parser.add_argument(
            '--property',
            metavar='<key>',
            required=True,
            action='append',
            default=[],
            help=_(
                'Property to remove from object '
                '(repeat option to remove multiple properties)'
            ),
        )
        return parser

    def take_action(self, parsed_args):
        self.app.client_manager.object_store.object_unset(
            parsed_args.container,
            parsed_args.object,
            properties=parsed_args.property,
        )
