import { type RemoteConnection } from '../connection.ts';
import type { RemoteReceiverOptions } from './shared.ts';
/**
 * Takes care of mapping remote elements to matching HTML elements
 * on the host page. If you implement your UI with [custom elements](https://developer.mozilla.org/en-US/docs/Web/Web_Components/Using_custom_elements),
 * `DOMRemoteReceiver` is a simple option that avoids much of the
 * manual work required when using the basic `RemoteReceiver`.
 */
export declare class DOMRemoteReceiver {
    /**
     * The root element that will contain the host implementations of
     * all nodes attached to the remote tree. To connect the receiver to
     * a new element, call the `connect()` method.
     */
    readonly root: DocumentFragment | Element;
    /**
     * An object that can synchronize a tree of elements between two JavaScript
     * environments. This object acts as a “thin waist”, allowing for efficient
     * communication of changes between a “remote” environment (usually, a JavaScript
     * sandbox, such as an `iframe` or Web Worker) and a “host” environment
     * (usually, a top-level browser page).
     */
    readonly connection: RemoteConnection;
    private readonly attached;
    constructor({ root, retain, release, call, cache, }?: RemoteReceiverOptions & {
        /**
         * The root element for this receiver. This acts as a shortcut for calling
         * `connect()` after creating the receiver.
         */
        root?: Element;
        /**
         * Customizes how [remote methods](https://github.com/Shopify/remote-dom/blob/main/packages/core#remotemethods)
         * are called. By default, the receiver will call a matching method found on
         * the HTML element that represents the remote element. However, you may want to
         * customize this behavior in order to avoid exposing methods on your HTML
         * elements that should not be callable by the remote environment.
         *
         * @param element The HTML element representing the remote element the method is being called on.
         * @param method The name of the method being called.
         * @param args Arguments passed to the method from the remote environment.
         *
         * @example
         * const receiver = new DOMRemoteReceiver({
         *   call(element, method, ...args) {
         *     // Prevent calling any methods that start with an underscore
         *     if (method.startsWith('_')) {
         *       throw new Error(`Cannot call method ${method}`);
         *     }
         *
         *     return element[method](...args);
         *   },
         * });
         */
        call?(element: Element, method: string, ...args: any[]): any;
        /**
         * Controls how DOM elements created in based on remote elements are retained
         * once they are disconnected from the remote environment.
         */
        cache?: {
            /**
             * A timeout in milliseconds after which a detached element will be released.
             */
            maxAge?: number;
        };
    });
    /**
     * Connects the receiver to a new root element. The representation of
     * any child elements of the remote root will be appended to this node
     * as children, and the `root` property will be updated to point to the
     * new element.
     */
    connect(element: Element): void;
    /**
     * Disconnects the receiver from its current root element. Any current
     * children of the root element will be moved to a `DocumentFragment`
     * instead, so they can be re-attached to a new element later.
     */
    disconnect(): DocumentFragment;
}
//# sourceMappingURL=DOMRemoteReceiver.d.ts.map