/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util;

import ghidra.framework.store.local.ItemPropertyFile;
import ghidra.util.Msg;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.xml.XmlUtilities;
import ghidra.xml.NonThreadedXmlPullParserImpl;
import ghidra.xml.XmlElement;
import java.io.File;
import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public class PropertyFile {
    public static final String PROPERTY_EXT = ".prp";
    protected File propertyFile;
    protected String storageName;
    private Map<String, PropertyMapEntry> basicInfoMap = new HashMap<String, PropertyMapEntry>();
    private static final ErrorHandler HANDLER = new ErrorHandler(){

        @Override
        public void warning(SAXParseException exception) throws SAXException {
            throw exception;
        }

        @Override
        public void error(SAXParseException exception) throws SAXException {
            throw exception;
        }

        @Override
        public void fatalError(SAXParseException exception) throws SAXException {
            throw exception;
        }
    };

    public PropertyFile(File dir, String storageName) throws IOException {
        if (!dir.isAbsolute()) {
            throw new IllegalArgumentException("dir must be specified by an absolute path");
        }
        this.storageName = storageName;
        this.propertyFile = new File(dir, storageName + PROPERTY_EXT);
        if (this.propertyFile.exists()) {
            this.readState();
        }
    }

    protected boolean contains(String key) {
        return this.basicInfoMap.containsKey(key);
    }

    public boolean isReadOnly() {
        return !this.propertyFile.canWrite();
    }

    public File getParentStorageDirectory() {
        return this.propertyFile.getParentFile();
    }

    public String getStorageName() {
        return this.storageName;
    }

    public int getInt(String propertyName, int defaultValue) {
        PropertyMapEntry entry = this.basicInfoMap.get(propertyName);
        if (entry == null || entry.entityType != PropertyEntryType.INT_TYPE) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(entry.value);
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public void putInt(String propertyName, int value) {
        this.basicInfoMap.put(propertyName, new PropertyMapEntry(PropertyEntryType.INT_TYPE, Integer.toString(value)));
    }

    public long getLong(String propertyName, long defaultValue) {
        PropertyMapEntry entry = this.basicInfoMap.get(propertyName);
        if (entry == null || entry.entityType != PropertyEntryType.LONG_TYPE) {
            return defaultValue;
        }
        try {
            return Long.parseLong(entry.value);
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public void putLong(String propertyName, long value) {
        this.basicInfoMap.put(propertyName, new PropertyMapEntry(PropertyEntryType.LONG_TYPE, Long.toString(value)));
    }

    public String getString(String propertyName, String defaultValue) {
        PropertyMapEntry entry = this.basicInfoMap.get(propertyName);
        if (entry == null || entry.entityType != PropertyEntryType.STRING_TYPE) {
            return defaultValue;
        }
        return entry.value;
    }

    public void putString(String propertyName, String value) {
        if (value == null) {
            this.basicInfoMap.remove(propertyName);
        } else {
            this.basicInfoMap.put(propertyName, new PropertyMapEntry(PropertyEntryType.STRING_TYPE, value));
        }
    }

    public boolean getBoolean(String propertyName, boolean defaultValue) {
        PropertyMapEntry entry = this.basicInfoMap.get(propertyName);
        if (entry == null || entry.entityType != PropertyEntryType.BOOLEAN_TYPE) {
            return defaultValue;
        }
        return Boolean.parseBoolean(entry.value);
    }

    public void putBoolean(String propertyName, boolean value) {
        this.basicInfoMap.put(propertyName, new PropertyMapEntry(PropertyEntryType.BOOLEAN_TYPE, Boolean.toString(value)));
    }

    public void remove(String propertyName) {
        this.basicInfoMap.remove(propertyName);
    }

    public long lastModified() {
        return this.propertyFile.lastModified();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeState() throws IOException {
        try (PrintWriter writer = new PrintWriter(this.propertyFile);){
            writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
            writer.println("<FILE_INFO>");
            writer.println("    <BASIC_INFO>");
            for (Map.Entry<String, PropertyMapEntry> entry : this.basicInfoMap.entrySet()) {
                String propertyName = entry.getKey();
                String propertyType = entry.getValue().entityType.rep;
                String propertyValue = entry.getValue().value;
                writer.print("        <STATE NAME=\"");
                writer.print(XmlUtilities.escapeElementEntities((String)propertyName));
                writer.print("\" TYPE=\"");
                writer.print(XmlUtilities.escapeElementEntities((String)propertyType));
                writer.print("\" VALUE=\"");
                writer.print(XmlUtilities.escapeElementEntities((String)propertyValue));
                writer.println("\" />");
            }
            writer.println("    </BASIC_INFO>");
            writer.println("</FILE_INFO>");
        }
    }

    public void readState() throws IOException {
        NonThreadedXmlPullParserImpl parser = null;
        try {
            XmlElement state;
            parser = new NonThreadedXmlPullParserImpl(this.propertyFile, HANDLER, false);
            XmlElement file_info = parser.start(new String[]{"FILE_INFO"});
            XmlElement basic_info = parser.start(new String[]{"BASIC_INFO"});
            while ((state = parser.softStart(new String[]{"STATE"})) != null) {
                String propertyName = state.getAttribute("NAME");
                String propertyTypeString = state.getAttribute("TYPE");
                String propertyValue = state.getAttribute("VALUE");
                PropertyEntryType propertyType = PropertyEntryType.lookup(propertyTypeString);
                this.basicInfoMap.put(propertyName, new PropertyMapEntry(propertyType, propertyValue));
                parser.end(state);
            }
            parser.end(basic_info);
            parser.end(file_info);
        }
        catch (Exception e) {
            String msg = "XML parse error in properties file";
            Msg.error((Object)this, (Object)(msg + ": " + String.valueOf(this.propertyFile)));
            throw new InvalidObjectException(msg);
        }
        finally {
            if (parser != null) {
                parser.dispose();
            }
        }
    }

    public void moveTo(File newStorageParent, String newStorageName) throws DuplicateFileException, IOException {
        if (!newStorageParent.equals(this.propertyFile.getParentFile()) || !newStorageName.equals(this.storageName)) {
            File newPropertyFile = new File(newStorageParent, newStorageName + PROPERTY_EXT);
            if (newPropertyFile.exists()) {
                throw new DuplicateFileException(String.valueOf(newPropertyFile) + " already exists");
            }
            if (!this.propertyFile.renameTo(newPropertyFile)) {
                throw new IOException("move failed");
            }
            this.propertyFile = newPropertyFile;
            this.storageName = newStorageName;
        }
    }

    public boolean exists() {
        return this.propertyFile.exists();
    }

    public void delete() {
        this.propertyFile.delete();
    }

    public int hashCode() {
        return this.propertyFile.hashCode();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        ItemPropertyFile other = (ItemPropertyFile)obj;
        return this.propertyFile.equals(other.propertyFile);
    }

    private record PropertyMapEntry(PropertyEntryType entityType, String value) {
    }

    private static enum PropertyEntryType {
        INT_TYPE("int"),
        LONG_TYPE("long"),
        BOOLEAN_TYPE("boolean"),
        STRING_TYPE("string");

        private final String rep;

        private PropertyEntryType(String rep) {
            this.rep = rep;
        }

        public static PropertyEntryType lookup(String rep) {
            for (PropertyEntryType entryType : PropertyEntryType.values()) {
                if (!rep.equals(entryType.rep)) continue;
                return entryType;
            }
            return null;
        }
    }
}

