/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.features.bsim.query.description;

import java.io.IOException;
import java.io.Writer;
import java.util.Objects;

import generic.lsh.vector.LSHVector;
import generic.lsh.vector.LSHVectorFactory;
import ghidra.util.xml.SpecXmlUtils;
import ghidra.xml.XmlElement;
import ghidra.xml.XmlPullParser;

public class VectorResult {
	public long vectorid;			// Id of vector
	public double sim;				// Similarity score
	public double signif;			// Significance score
	public int hitcount;			// Number of duplicate results
	public LSHVector vec;

	public VectorResult() {
		vectorid = 0;
		hitcount = 0;
		vec = null;
	}

	public VectorResult(long vid, int cnt, double sm, double sg, LSHVector v) {
		vectorid = vid;
		sim = sm;
		signif = sg;
		hitcount = cnt;
		vec = v;
	}

	public void saveXml(Writer write) throws IOException {
		StringBuilder buf = new StringBuilder();
		buf.append("<vec");
		SpecXmlUtils.encodeUnsignedIntegerAttribute(buf, "id", vectorid);
		buf.append(">\n");
		buf.append(" <hit>").append(hitcount).append("</hit>\n");
		buf.append(" <sim>").append(Double.toString(sim)).append("</sim>\n");
		buf.append(" <sig>").append(Double.toString(signif)).append("</sig>\n");
		write.append(buf.toString());
		if (vec != null)
			vec.saveXml(write);
		write.append("</vec>\n");
	}

	public void restoreXml(XmlPullParser parser, LSHVectorFactory vectorFactory) {
		XmlElement el = parser.start("vec");
		vectorid = SpecXmlUtils.decodeLong(el.getAttribute("id"));
		parser.start("hit");
		hitcount = SpecXmlUtils.decodeInt(parser.end().getText());
		parser.start("sim");
		sim = Double.parseDouble(parser.end().getText());
		parser.start("sig");
		signif = Double.parseDouble(parser.end().getText());
		if (parser.peek().isStart()) {
			vec = vectorFactory.restoreVectorFromXml(parser);
		}
		parser.end();
	}

	//generated by Eclipse
	@Override
	public int hashCode() {
		return Objects.hash(hitcount, signif, sim, vec, vectorid);
	}

	//generated by Eclipse
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		VectorResult other = (VectorResult) obj;
		return hitcount == other.hitcount &&
			Double.doubleToLongBits(signif) == Double.doubleToLongBits(other.signif) &&
			Double.doubleToLongBits(sim) == Double.doubleToLongBits(other.sim) &&
			Objects.equals(vec, other.vec) && vectorid == other.vectorid;
	}

}
