/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import * as fs from 'node:fs/promises';
import * as path from 'node:path';
import { debugLogger } from '@google/gemini-cli-core';
import chalk from 'chalk';
import { escapeAnsiCtrlCodes } from '../../ui/utils/textUtils.js';
export const INSTALL_WARNING_MESSAGE = chalk.yellow('The extension you are about to install may have been created by a third-party developer and sourced from a public repository. Google does not vet, endorse, or guarantee the functionality or security of extensions. Please carefully inspect any extension and its source code before installing to understand the permissions it requires and the actions it may perform.');
export const SKILLS_WARNING_MESSAGE = chalk.yellow("Agent skills inject specialized instructions and domain-specific knowledge into the agent's system prompt. This can change how the agent interprets your requests and interacts with your environment. Review the skill definitions at the location(s) provided below to ensure they meet your security standards.");
/**
 * Builds a consent string for installing agent skills.
 */
export async function skillsConsentString(skills, source, targetDir, isLink = false) {
    const action = isLink ? 'Linking' : 'Installing';
    const output = [];
    output.push(`${action} agent skill(s) from "${source}".`);
    output.push(`\nThe following agent skill(s) will be ${action.toLowerCase()}:\n`);
    output.push(...(await renderSkillsList(skills)));
    if (targetDir) {
        const destLabel = isLink ? 'Link' : 'Install';
        output.push(`${destLabel} Destination: ${targetDir}`);
    }
    output.push('\n' + SKILLS_WARNING_MESSAGE);
    return output.join('\n');
}
/**
 * Requests consent from the user to perform an action, by reading a Y/n
 * character from stdin.
 *
 * This should not be called from interactive mode as it will break the CLI.
 *
 * @param consentDescription The description of the thing they will be consenting to.
 * @returns boolean, whether they consented or not.
 */
export async function requestConsentNonInteractive(consentDescription) {
    debugLogger.log(consentDescription);
    const result = await promptForConsentNonInteractive('Do you want to continue? [Y/n]: ');
    return result;
}
/**
 * Requests consent from the user to perform an action, in interactive mode.
 *
 * This should not be called from non-interactive mode as it will not work.
 *
 * @param consentDescription The description of the thing they will be consenting to.
 * @param addExtensionUpdateConfirmationRequest A function to actually add a prompt to the UI.
 * @returns boolean, whether they consented or not.
 */
export async function requestConsentInteractive(consentDescription, addExtensionUpdateConfirmationRequest) {
    return promptForConsentInteractive(consentDescription + '\n\nDo you want to continue?', addExtensionUpdateConfirmationRequest);
}
/**
 * Asks users a prompt and awaits for a y/n response on stdin.
 *
 * This should not be called from interactive mode as it will break the CLI.
 *
 * @param prompt A yes/no prompt to ask the user
 * @param defaultValue Whether to resolve as true or false on enter.
 * @returns Whether or not the user answers 'y' (yes).
 */
export async function promptForConsentNonInteractive(prompt, defaultValue = true) {
    const readline = await import('node:readline');
    const rl = readline.createInterface({
        input: process.stdin,
        output: process.stdout,
    });
    return new Promise((resolve) => {
        rl.question(prompt, (answer) => {
            rl.close();
            const trimmedAnswer = answer.trim().toLowerCase();
            if (trimmedAnswer === '') {
                resolve(defaultValue);
            }
            else {
                resolve(['y', 'yes'].includes(trimmedAnswer));
            }
        });
    });
}
/**
 * Asks users an interactive yes/no prompt.
 *
 * This should not be called from non-interactive mode as it will break the CLI.
 *
 * @param prompt A markdown prompt to ask the user
 * @param addExtensionUpdateConfirmationRequest Function to update the UI state with the confirmation request.
 * @returns Whether or not the user answers yes.
 */
async function promptForConsentInteractive(prompt, addExtensionUpdateConfirmationRequest) {
    return new Promise((resolve) => {
        addExtensionUpdateConfirmationRequest({
            prompt,
            onConfirm: (resolvedConfirmed) => {
                resolve(resolvedConfirmed);
            },
        });
    });
}
/**
 * Builds a consent string for installing an extension based on it's
 * extensionConfig.
 */
async function extensionConsentString(extensionConfig, hasHooks, skills = []) {
    const sanitizedConfig = escapeAnsiCtrlCodes(extensionConfig);
    const output = [];
    const mcpServerEntries = Object.entries(sanitizedConfig.mcpServers || {});
    output.push(`Installing extension "${sanitizedConfig.name}".`);
    if (mcpServerEntries.length) {
        output.push('This extension will run the following MCP servers:');
        for (const [key, mcpServer] of mcpServerEntries) {
            const isLocal = !!mcpServer.command;
            const source = mcpServer.httpUrl ??
                `${mcpServer.command || ''}${mcpServer.args ? ' ' + mcpServer.args.join(' ') : ''}`;
            output.push(`  * ${key} (${isLocal ? 'local' : 'remote'}): ${source}`);
        }
    }
    if (sanitizedConfig.contextFileName) {
        output.push(`This extension will append info to your gemini.md context using ${sanitizedConfig.contextFileName}`);
    }
    if (sanitizedConfig.excludeTools) {
        output.push(`This extension will exclude the following core tools: ${sanitizedConfig.excludeTools}`);
    }
    if (hasHooks) {
        output.push('⚠️  This extension contains Hooks which can automatically execute commands.');
    }
    if (skills.length > 0) {
        output.push(`\n${chalk.bold('Agent Skills:')}`);
        output.push('\nThis extension will install the following agent skills:\n');
        output.push(...(await renderSkillsList(skills)));
    }
    output.push('\n' + INSTALL_WARNING_MESSAGE);
    if (skills.length > 0) {
        output.push('\n' + SKILLS_WARNING_MESSAGE);
    }
    return output.join('\n');
}
/**
 * Shared logic for formatting a list of agent skills for a consent prompt.
 */
async function renderSkillsList(skills) {
    const output = [];
    for (const skill of skills) {
        output.push(`  * ${chalk.bold(skill.name)}: ${skill.description}`);
        const skillDir = path.dirname(skill.location);
        let fileCountStr = '';
        try {
            const skillDirItems = await fs.readdir(skillDir);
            fileCountStr = ` (${skillDirItems.length} items in directory)`;
        }
        catch {
            fileCountStr = ` ${chalk.red('⚠️ (Could not count items in directory)')}`;
        }
        output.push(chalk.dim(`    (Source: ${skill.location})${fileCountStr}`));
        output.push('');
    }
    return output;
}
/**
 * Requests consent from the user to install an extension (extensionConfig), if
 * there is any difference between the consent string for `extensionConfig` and
 * `previousExtensionConfig`.
 *
 * Always requests consent if previousExtensionConfig is null.
 *
 * Throws if the user does not consent.
 */
export async function maybeRequestConsentOrFail(extensionConfig, requestConsent, hasHooks, previousExtensionConfig, previousHasHooks, skills = [], previousSkills = []) {
    const extensionConsent = await extensionConsentString(extensionConfig, hasHooks, skills);
    if (previousExtensionConfig) {
        const previousExtensionConsent = await extensionConsentString(previousExtensionConfig, previousHasHooks ?? false, previousSkills);
        if (previousExtensionConsent === extensionConsent) {
            return;
        }
    }
    if (!(await requestConsent(extensionConsent))) {
        throw new Error(`Installation cancelled for "${extensionConfig.name}".`);
    }
}
//# sourceMappingURL=consent.js.map