import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { renderWithProviders } from '../test-utils/render.js';
import { Text, useIsScreenReaderEnabled } from 'ink';
import { App } from './App.js';
import {} from './contexts/UIStateContext.js';
import { StreamingState } from './types.js';
import { makeFakeConfig, CoreToolCallStatus } from '@google/gemini-cli-core';
vi.mock('ink', async (importOriginal) => {
    const original = await importOriginal();
    return {
        ...original,
        useIsScreenReaderEnabled: vi.fn(),
    };
});
vi.mock('./components/DialogManager.js', () => ({
    DialogManager: () => _jsx(Text, { children: "DialogManager" }),
}));
vi.mock('./components/Composer.js', () => ({
    Composer: () => _jsx(Text, { children: "Composer" }),
}));
vi.mock('./components/Notifications.js', async () => {
    const { Text, Box } = await import('ink');
    return {
        Notifications: () => (_jsx(Box, { children: _jsx(Text, { children: "Notifications" }) })),
    };
});
vi.mock('./components/QuittingDisplay.js', () => ({
    QuittingDisplay: () => _jsx(Text, { children: "Quitting..." }),
}));
vi.mock('./components/HistoryItemDisplay.js', () => ({
    HistoryItemDisplay: () => _jsx(Text, { children: "HistoryItemDisplay" }),
}));
vi.mock('./components/Footer.js', async () => {
    const { Text, Box } = await import('ink');
    return {
        Footer: () => (_jsx(Box, { children: _jsx(Text, { children: "Footer" }) })),
    };
});
describe('App', () => {
    beforeEach(() => {
        useIsScreenReaderEnabled.mockReturnValue(false);
    });
    const mockUIState = {
        streamingState: StreamingState.Idle,
        cleanUiDetailsVisible: true,
        quittingMessages: null,
        dialogsVisible: false,
        mainControlsRef: {
            current: null,
        },
        rootUiRef: {
            current: null,
        },
        historyManager: {
            addItem: vi.fn(),
            history: [],
            updateItem: vi.fn(),
            clearItems: vi.fn(),
            loadHistory: vi.fn(),
        },
        history: [],
        pendingHistoryItems: [],
        pendingGeminiHistoryItems: [],
        bannerData: {
            defaultText: 'Mock Banner Text',
            warningText: '',
        },
        backgroundShells: new Map(),
    };
    it('should render main content and composer when not quitting', async () => {
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
            useAlternateBuffer: false,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Composer');
        unmount();
    });
    it('should render quitting display when quittingMessages is set', async () => {
        const quittingUIState = {
            ...mockUIState,
            quittingMessages: [{ id: 1, type: 'user', text: 'test' }],
        };
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: quittingUIState,
            useAlternateBuffer: false,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Quitting...');
        unmount();
    });
    it('should render full history in alternate buffer mode when quittingMessages is set', async () => {
        const quittingUIState = {
            ...mockUIState,
            quittingMessages: [{ id: 1, type: 'user', text: 'test' }],
            history: [{ id: 1, type: 'user', text: 'history item' }],
            pendingHistoryItems: [{ type: 'user', text: 'pending item' }],
        };
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: quittingUIState,
            useAlternateBuffer: true,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('HistoryItemDisplay');
        expect(lastFrame()).toContain('Quitting...');
        unmount();
    });
    it('should render dialog manager when dialogs are visible', async () => {
        const dialogUIState = {
            ...mockUIState,
            dialogsVisible: true,
        };
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: dialogUIState,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('DialogManager');
        unmount();
    });
    it.each([
        { key: 'C', stateKey: 'ctrlCPressedOnce' },
        { key: 'D', stateKey: 'ctrlDPressedOnce' },
    ])('should show Ctrl+$key exit prompt when dialogs are visible and $stateKey is true', async ({ key, stateKey }) => {
        const uiState = {
            ...mockUIState,
            dialogsVisible: true,
            [stateKey]: true,
        };
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain(`Press Ctrl+${key} again to exit.`);
        unmount();
    });
    it('should render ScreenReaderAppLayout when screen reader is enabled', async () => {
        useIsScreenReaderEnabled.mockReturnValue(true);
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Footer');
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Composer');
        unmount();
    });
    it('should render DefaultAppLayout when screen reader is not enabled', async () => {
        useIsScreenReaderEnabled.mockReturnValue(false);
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Composer');
        unmount();
    });
    it('should render ToolConfirmationQueue along with Composer when tool is confirming and experiment is on', async () => {
        useIsScreenReaderEnabled.mockReturnValue(false);
        const toolCalls = [
            {
                callId: 'call-1',
                name: 'ls',
                description: 'list directory',
                status: CoreToolCallStatus.AwaitingApproval,
                resultDisplay: '',
                confirmationDetails: {
                    type: 'exec',
                    title: 'Confirm execution',
                    command: 'ls',
                    rootCommand: 'ls',
                    rootCommands: ['ls'],
                },
            },
        ];
        const stateWithConfirmingTool = {
            ...mockUIState,
            pendingHistoryItems: [
                {
                    type: 'tool_group',
                    tools: toolCalls,
                },
            ],
            pendingGeminiHistoryItems: [
                {
                    type: 'tool_group',
                    tools: toolCalls,
                },
            ],
        };
        const configWithExperiment = makeFakeConfig();
        vi.spyOn(configWithExperiment, 'isTrustedFolder').mockReturnValue(true);
        vi.spyOn(configWithExperiment, 'getIdeMode').mockReturnValue(false);
        const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
            uiState: stateWithConfirmingTool,
            config: configWithExperiment,
        });
        await waitUntilReady();
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Action Required'); // From ToolConfirmationQueue
        expect(lastFrame()).toContain('Composer');
        expect(lastFrame()).toMatchSnapshot();
        unmount();
    });
    describe('Snapshots', () => {
        it('renders default layout correctly', async () => {
            useIsScreenReaderEnabled.mockReturnValue(false);
            const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
                uiState: mockUIState,
            });
            await waitUntilReady();
            expect(lastFrame()).toMatchSnapshot();
            unmount();
        });
        it('renders screen reader layout correctly', async () => {
            useIsScreenReaderEnabled.mockReturnValue(true);
            const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
                uiState: mockUIState,
            });
            await waitUntilReady();
            expect(lastFrame()).toMatchSnapshot();
            unmount();
        });
        it('renders with dialogs visible', async () => {
            const dialogUIState = {
                ...mockUIState,
                dialogsVisible: true,
            };
            const { lastFrame, waitUntilReady, unmount } = renderWithProviders(_jsx(App, {}), {
                uiState: dialogUIState,
            });
            await waitUntilReady();
            expect(lastFrame()).toMatchSnapshot();
            unmount();
        });
    });
});
//# sourceMappingURL=App.test.js.map