/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { addMemory, listMemoryFiles, refreshMemory, showMemory, } from '@google/gemini-cli-core';
import { MessageType } from '../types.js';
import { CommandKind } from './types.js';
export const memoryCommand = {
    name: 'memory',
    description: 'Commands for interacting with memory',
    kind: CommandKind.BUILT_IN,
    autoExecute: false,
    subCommands: [
        {
            name: 'show',
            description: 'Show the current memory contents',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: async (context) => {
                const config = context.services.config;
                if (!config)
                    return;
                const result = showMemory(config);
                context.ui.addItem({
                    type: MessageType.INFO,
                    text: result.content,
                }, Date.now());
            },
        },
        {
            name: 'add',
            description: 'Add content to the memory',
            kind: CommandKind.BUILT_IN,
            autoExecute: false,
            action: (context, args) => {
                const result = addMemory(args);
                if (result.type === 'message') {
                    return result;
                }
                context.ui.addItem({
                    type: MessageType.INFO,
                    text: `Attempting to save to memory: "${args.trim()}"`,
                }, Date.now());
                return result;
            },
        },
        {
            name: 'refresh',
            description: 'Refresh the memory from the source',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: async (context) => {
                context.ui.addItem({
                    type: MessageType.INFO,
                    text: 'Refreshing memory from source files...',
                }, Date.now());
                try {
                    const config = context.services.config;
                    if (config) {
                        const result = await refreshMemory(config);
                        context.ui.addItem({
                            type: MessageType.INFO,
                            text: result.content,
                        }, Date.now());
                    }
                }
                catch (error) {
                    context.ui.addItem({
                        type: MessageType.ERROR,
                        // eslint-disable-next-line @typescript-eslint/no-unsafe-type-assertion
                        text: `Error refreshing memory: ${error.message}`,
                    }, Date.now());
                }
            },
        },
        {
            name: 'list',
            description: 'Lists the paths of the GEMINI.md files in use',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: async (context) => {
                const config = context.services.config;
                if (!config)
                    return;
                const result = listMemoryFiles(config);
                context.ui.addItem({
                    type: MessageType.INFO,
                    text: result.content,
                }, Date.now());
            },
        },
    ],
};
//# sourceMappingURL=memoryCommand.js.map