/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { CommandKind } from './types.js';
import process from 'node:process';
import { MessageType } from '../types.js';
import { IdeClient, UserAccountManager, debugLogger, getVersion, } from '@google/gemini-cli-core';
export const aboutCommand = {
    name: 'about',
    description: 'Show version info',
    kind: CommandKind.BUILT_IN,
    autoExecute: true,
    action: async (context) => {
        const osVersion = process.platform;
        let sandboxEnv = 'no sandbox';
        if (process.env['SANDBOX'] && process.env['SANDBOX'] !== 'sandbox-exec') {
            sandboxEnv = process.env['SANDBOX'];
        }
        else if (process.env['SANDBOX'] === 'sandbox-exec') {
            sandboxEnv = `sandbox-exec (${process.env['SEATBELT_PROFILE'] || 'unknown'})`;
        }
        const modelVersion = context.services.config?.getModel() || 'Unknown';
        const cliVersion = await getVersion();
        const selectedAuthType = context.services.settings.merged.security?.auth?.selectedType || '';
        const gcpProject = process.env['GOOGLE_CLOUD_PROJECT'] || '';
        const ideClient = await getIdeClientName(context);
        const userAccountManager = new UserAccountManager();
        const cachedAccount = userAccountManager.getCachedGoogleAccount();
        debugLogger.log('AboutCommand: Retrieved cached Google account', {
            cachedAccount,
        });
        const userEmail = cachedAccount ?? undefined;
        const aboutItem = {
            type: MessageType.ABOUT,
            cliVersion,
            osVersion,
            sandboxEnv,
            modelVersion,
            selectedAuthType,
            gcpProject,
            ideClient,
            userEmail,
        };
        context.ui.addItem(aboutItem, Date.now());
    },
};
async function getIdeClientName(context) {
    if (!context.services.config?.getIdeMode()) {
        return '';
    }
    const ideClient = await IdeClient.getInstance();
    return ideClient?.getDetectedIdeDisplayName() ?? '';
}
//# sourceMappingURL=aboutCommand.js.map