/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { Storage } from '../config/storage.js';
import { loadSkillsFromDir } from './skillLoader.js';
export {};
export class SkillManager {
    skills = [];
    activeSkillNames = new Set();
    /**
     * Clears all discovered skills.
     */
    clearSkills() {
        this.skills = [];
    }
    /**
     * Discovers skills from standard user and project locations, as well as extensions.
     * Precedence: Extensions (lowest) -> User -> Project (highest).
     */
    async discoverSkills(storage, extensions = []) {
        this.clearSkills();
        // 1. Extension skills (lowest precedence)
        for (const extension of extensions) {
            if (extension.isActive && extension.skills) {
                this.addSkillsWithPrecedence(extension.skills);
            }
        }
        // 2. User skills
        const userSkills = await loadSkillsFromDir(Storage.getUserSkillsDir());
        this.addSkillsWithPrecedence(userSkills);
        // 3. Project skills (highest precedence)
        const projectSkills = await loadSkillsFromDir(storage.getProjectSkillsDir());
        this.addSkillsWithPrecedence(projectSkills);
    }
    addSkillsWithPrecedence(newSkills) {
        const skillMap = new Map();
        for (const skill of [...this.skills, ...newSkills]) {
            skillMap.set(skill.name, skill);
        }
        this.skills = Array.from(skillMap.values());
    }
    /**
     * Returns the list of enabled discovered skills.
     */
    getSkills() {
        return this.skills.filter((s) => !s.disabled);
    }
    /**
     * Returns all discovered skills, including disabled ones.
     */
    getAllSkills() {
        return this.skills;
    }
    /**
     * Filters discovered skills by name.
     */
    filterSkills(predicate) {
        this.skills = this.skills.filter(predicate);
    }
    /**
     * Sets the list of disabled skill names.
     */
    setDisabledSkills(disabledNames) {
        for (const skill of this.skills) {
            skill.disabled = disabledNames.includes(skill.name);
        }
    }
    /**
     * Reads the full content (metadata + body) of a skill by name.
     */
    getSkill(name) {
        return this.skills.find((s) => s.name === name) ?? null;
    }
    /**
     * Activates a skill by name.
     */
    activateSkill(name) {
        this.activeSkillNames.add(name);
    }
    /**
     * Checks if a skill is active.
     */
    isSkillActive(name) {
        return this.activeSkillNames.has(name);
    }
}
//# sourceMappingURL=skillManager.js.map