#
# This file defines dialog boxes used by the 2dF positioner 
# and configuration software.
#
#    Copyright (c)  Anglo-Australian Telescope Board, 1995.
#    Permission granted for use for non-commercial purposes.
#
# Sccs Id:     procs.tcl, Release 1.3, 03/27/00
#
# Based on POS subsystem procs.tcl SCCS version Release 1.12, 01/22/99.
# 
# History:
#     17-Mar-2000 - TJF - A late start to history.  Remove setup_dialog,
#                         cfile_dialog and action_dialog.  These are
#                          not being used by configure and
#                         are calling fs_dialog2 which is to be removed.
#
#-------------------------------------------------------------------------------    
#
#                          G l o b a l s
#
#  This section should list all the global variables used in this file. This 
#  section isn't necessary for the program to work, but it provides a 
#  convenient place to document (briefly) the use to which these variables are
#  put.

global procs_font_1           ;# Font used for the various dialogues.
global tcl_platform           ;# Tcl array giving platform-dependent info. In
                              ;# this case used to pick font names.

#  -----------------------------------------------------------------------------
#
#                    I n i t i a l i s a t i o n s
#
#  Initialisations - performed as the file is loaded.
#
#  Set the font used for the various dialogues.
                              
if { ($tcl_platform(platform) == "windows") ||
           ($tcl_platform(platform) == "macintosh")} {
    set procs_font_1 {Times 14 }
} else {
    set procs_font_1 {Times 14 }
}
 
#
# prompt_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -	Title to display in dialog's decorative frame.
# text -	Message to display in dialog.
# textvariable - Variable to recieve result.
# default -	Index of button that is to display the default ring
#		(-1 means none).
# args -	One or more strings to display in buttons across the
#		bottom of the dialog box.


proc prompt_dialog {w title text textvariable default args} {
    global tk_priv

    # 1. Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w Dialog
    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # 2. Fill the top part with bitmap and message and entry field.

    global procs_font_1
    message $w.msg -width 3i -text $text \
	    -font $procs_font_1
    pack $w.msg -in $w.top -side top -expand 1 -fill x -padx 5m -pady 5m
    entry $w.ent -textvariable $textvariable -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent -in $w.top -side top -expand 1 -fill x -padx 5m -pady 5m


    # 3. Create a row of buttons at the bottom of the dialog.

    set i 0
    foreach but $args {
	button $w.button$i -text $but -command "set tk_priv(button) $i"
	if {$i == $default} {
	    frame $w.default -relief sunken -bd 1
	    raise $w.button$i $w.default
	    pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m
	    pack $w.button$i -in $w.default -padx 2m -pady 2m \
		    -ipadx 2m -ipady 1m
	    bind $w <Return> "$w.button$i flash; set tk_priv(button) $i"
	    bind $w.ent <Return> "$w.button$i flash; set tk_priv(button) $i"
	} else {
	    pack $w.button$i -in $w.bot -side left -expand 1 \
		    -padx 3m -pady 3m -ipadx 2m -ipady 1m
	}
	incr i
    }

    # 4. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

    # 5. Set a grab and claim the focus too.

    set oldFocus [focus]
#    grab $w
    focus $w

    # 6. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.

    tkwait variable tk_priv(button)
    catch { destroy $w }
    focus $oldFocus
    return $tk_priv(button)
}

#
# fs_dialog:
#
# This procedure displays a file selection, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -	Title to display in dialog's decorative frame.
# text -	Message to display in dialog.
# textvariable - Variable to recieve result.
# filter -      Initial filter string

proc fs_dialog {w title text textvariable filter} {
    global fs_priv
    global selection
    global file_list
    global dir_list
    global gfilter
    global gmask

    upvar 1 $textvariable tv
    set selection $tv
    set gmask $filter
    set gfilter [pwd]/$gmask

    # 1. Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w Dialog
    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both -expand 1
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # 2. Fill the top part with bitmap and message and entry field.

    global procs_font_1
    message $w.msg -width 3i -text $text \
	    -font $procs_font_1
    pack $w.msg -in $w.top -side top -fill x -padx 2m -pady 3m
    message $w.mfilter -text Filter -width 150 -anchor w
    pack $w.mfilter -in $w.top -side top -fill x -padx 2m
    entry $w.fil -textvariable gfilter -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.fil -in $w.top -side top -fill x -padx 2m
    frame $w.m
    pack $w.m -in $w.top -side top -expand 1 -fill both
    frame $w.f
    pack $w.f -in $w.m -side right -expand 1 -fill both -padx 2m -pady 2m
    message $w.f.mess -text Files -width 120 -anchor w
    pack $w.f.mess -side top -fill x
    listbox $w.f.list -geometry 20x10 -yscrollcommand "$w.f.vscroll set" \
        -xscrollcommand "$w.f.hscroll set" -relief sunken 
    tk_listboxSingleSelect $w.f.list
    scrollbar $w.f.vscroll -orient vertical -command "$w.f.list yview"
    scrollbar $w.f.hscroll -orient horizontal -command "$w.f.list xview"
    pack $w.f.hscroll -side bottom -fill x
    pack $w.f.list -side left -expand 1 -fill both
    pack $w.f.vscroll -side right -fill y
    bind $w.f.list <ButtonRelease-1> \
         {set selection [lindex $file_list [%W curselection]]}
    frame $w.d
    pack $w.d -in $w.m -side left -expand 1 -fill both -padx 2m -pady 2m
    message $w.d.mess -text Directories -width 120 -anchor w
    pack $w.d.mess -side top -fill x
    listbox $w.d.list -geometry 20x10 -yscrollcommand "$w.d.vscroll set" \
        -xscrollcommand "$w.d.hscroll set" -relief sunken
    tk_listboxSingleSelect $w.d.list
    scrollbar $w.d.vscroll -orient vertical -command "$w.d.list yview"
    scrollbar $w.d.hscroll -orient horizontal -command "$w.d.list xview"
    pack $w.d.hscroll -side bottom -fill x
    pack $w.d.list -side left -expand 1 -fill both
    pack $w.d.vscroll -side right -fill y
    bind $w.d.list <Double-Button-1> \
         "cd \[lindex \$dir_list \[%W curselection\]\]; DoFilter $w; FsSet $w"
    bind $w.d.list <Return> \
         "cd \[lindex \$dir_list \[%W curselection\]\]; DoFilter $w; FsSet $w"
    frame $w.s 
    pack $w.s -in $w.top -side bottom -fill both
    message $w.s.msel -text Selection -width 120 -anchor w
    pack $w.s.msel -side top -fill x -padx 2m
    global procs_font_1
    entry $w.s.ent -textvariable selection -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.s.ent  -side top -fill x -padx 2m -pady 2m

    DoFilter $w

    # 3. Create a row of buttons at the bottom of the dialog.

    button $w.button0 -text OK -command "set fs_priv(button) 0"
    frame $w.default -relief sunken -bd 1
    raise $w.button0 $w.default
    pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m  
    pack $w.button0 -in $w.default -padx 2m -pady 2m \
		    -ipadx 2m -ipady 1m
    bind $w.f.list <Return> "$w.button0 flash; \
                              set fs_priv(button) 0"
    bind $w.s.ent <Return> "$w.button0 flash;  \
                              set fs_priv(button) 0"
    bind $w.fil <Return> "$w.button1 flash; FilButton; DoFilter $w; FsSet $w"
    bind $w.f.list <Double-Button-1> "$w.button0 flash;  \
                              set fs_priv(button) 0"
	    
    button $w.button1 -text Filter -command "FilButton; DoFilter $w; FsSet $w"
    pack $w.button1 -in $w.bot -side left -expand 1 \
		    -padx 3m -pady 3m -ipadx 2m -ipady 1m
    button $w.button2 -text Cancel -command "set fs_priv(button) 2"
    pack $w.button2 -in $w.bot -side left -expand 1 \
		    -padx 3m -pady 3m -ipadx 2m -ipady 1m

    # 4. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
    wm minsize $w 1 1

    FsSet $w

    # 5. Set a grab and claim the focus too.

    set oldFocus [focus]
    grab $w
    focus $w

    # 6. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.

    tkwait variable fs_priv(button)
    if {$fs_priv(button) == 0} { set tv $selection }
    destroy $w
    focus $oldFocus
    puts stdout $tv
    return $fs_priv(button)
}

proc FilButton {} {

    global gfilter
    global gmask

    set gmask [file tail $gfilter]
    catch {cd [file dirname $gfilter]}
}

proc DoFilter {w} {

    global gfilter
    global file_list
    global dir_list
    global gmask

    $w.f.list delete 0 end
    $w.d.list delete 0 end
    set gfilter [pwd]/$gmask
    set file_list [lsort [glob -nocomplain $gmask]]
    set n 0
    $w.f.list delete 0 end
    foreach file $file_list {
       $w.f.list insert $n $file
       incr n
    }
    set dir_list {}
    set temp_list [lsort [glob -nocomplain .* *]]
    foreach file $temp_list {
       if {[file isdirectory $file]} {
          $w.d.list insert $n [pwd]/$file
          incr n
          lappend dir_list [pwd]/$file
       }
    }

}

proc FsSet {w} {

    update
    set lscroll [$w.d.hscroll get]
    set lt [lindex $lscroll 0]
    set lw [lindex $lscroll 1]
    $w.d.list xview [expr $lt-$lw+1] 
}


#
# error_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -	Title to display in dialog's decorative frame.
# text -	Message to display in dialog.
# bitmap -      Bitmap to display in dialog.
# command -     Command to invoke when OK button pressed.

proc error_dialog {w title text bitmap command} {
    # 1. Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w Dialog
    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # 2. Fill the top part with bitmap and message and entry field.

    global procs_font_1
    message $w.msg -width 90m -text $text \
	    -font $procs_font_1
    pack $w.msg -in $w.top -side right -expand 1 -fill x -padx 5m -pady 5m
    if {$bitmap != ""} {
       label $w.top.bitmap -bitmap $bitmap
       pack $w.top.bitmap -side left -padx 5m -pady 5m
    }


    # 3. Create a row of buttons at the bottom of the dialog.

    button $w.button -text OK -command "destroy $w ; $command"
    frame $w.default -relief sunken -bd 1
    raise $w.button $w.default
    pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m
    pack $w.button -in $w.default -padx 2m -pady 2m \
		    -ipadx 2m -ipady 1m
    bind $w <Return> "$w.button flash"; 

    # 4. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
}



proc SetupFileSelection {var file} {

   global $var

   set $var $file
}


#
# startup_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# name -        Variable to receive task name.
# pnname -      variable to receive node name.

proc startup_dialog {w pname pnname} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w "Start Up"
    wm iconname $w Startup
    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.f1
    pack $w.top.f1 -side top -fill x -expand 1

    frame $w.top.f2
    pack $w.top.f2 -side top -fill x -expand 1

    global procs_font_1
    message $w.msg -width 3i -text "Positioner Task:" \
	    -font $procs_font_1
    pack $w.msg -in $w.top.f1 -side left -padx 5m -pady 3m
    entry $w.ent -textvariable $pname -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent -in $w.top.f1 -side right -padx 5m -pady 3m

    message $w.msg2 -width 3i -text "Node Name:" \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.f2 -side left -padx 5m -pady 3m
    entry $w.ent2 -textvariable $pnname -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.f2 -side right  -padx 5m -pady 3m


    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"

  # Withdraw the window, then update all the geometry information
  # so we know how big it wants to be, then center the window in the
  # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

 #  Set a grab and claim the focus too.

    set oldFocus [focus]
    grab $w
    focus $w

 # Wait for the user to respond, then restore the focus 

    tkwait variable tk_priv(button)
    catch { destroy $w }
    focus $oldFocus


    # Return the selected button

    return $tk_priv(button)
}

#
# load_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# name -        Variable to receive task name.
# sim -         Variable to recieve simulation level.
# time_base -   Variable to receive time_base.

proc load_dialog {w pname psim ptime_base} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    upvar #0 $ptime_base tb
    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w "Start Up"
    wm iconname $w Startup
    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    message $w.msg -width 3i -text "Load Positioner Task:" \
	    -font $procs_font_1
    pack $w.msg -in $w.top -side left -expand 1 -fill x -padx 5m -pady 5m
    entry $w.ent -textvariable $pname -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent -in $w.top -side left -expand 1 -fill x -padx 5m -pady 5m


    frame $w.mid.sim
    radiobutton $w.mid.sim.full -value FULL -variable $psim \
            -text "Simulation FULL" -width 16
    radiobutton $w.mid.sim.none -value NONE -variable $psim \
            -text "Simulation NONE" -width 16
    pack $w.mid.sim.full -side top -ipadx 2m -ipady 2m
    pack $w.mid.sim.none -side bottom -ipadx 2m -ipady 2m
    pack $w.mid.sim -in $w.mid -side left -padx 5m -pady 5m

    scale $w.mid.scale -from 1 -to 25 -label "Simulation Time Base" \
          -showvalue 1 -length 90m -orient horizontal -tickinterval 2 \
          -relief raised
    pack $w.mid.scale -side right -padx 5m -pady 5m
    $w.mid.scale set $tb

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"

  # Withdraw the window, then update all the geometry information
  # so we know how big it wants to be, then center the window in the
  # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

 #  Set a grab and claim the focus too.

    set oldFocus [focus]
    grab $w
    focus $w

 # Wait for the user to respond, then restore the focus 

    tkwait variable tk_priv(button)
    set tb [$w.mid.scale get]
    catch { destroy $w }
    focus $oldFocus


    # Return the selected button

    return $tk_priv(button)
}

#
# movepiv_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Pivot -       Variable to recieve pivot number.
# X     -       Variable to receive X position.
# Y     -       Variable to receive Y position.
# Theta -       Variable to receive Theta value.
# Plate -       Variable to receive plate number.
# Flags -       Variable to receive flags.

proc movepiv_dialog {w title pPivot pX pY pTheta pPlate pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2
    frame $w.top.l3
    frame $w.top.l4
    global procs_font_1
    label $w.msg1 -width 8 -text "Pivot:" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 8 -text "X:" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    label $w.msg3 -width 8 -text "Y:" -anchor e \
	    -font $procs_font_1
    pack $w.msg3 -in $w.top.l3 -side left  -padx 2m -pady 2m 
    label $w.msg4 -width 8 -text "Theta:" -anchor e \
	    -font $procs_font_1
    pack $w.msg4 -in $w.top.l4 -side left  -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pPivot -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $pX -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent3 -textvariable $pY -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent3 -in $w.top.l3 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent4 -textvariable $pTheta -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent4 -in $w.top.l4 -side left -ipadx 2m -padx 5m -pady 2m

    pack $w.top.l1 -side top
    pack $w.top.l2 -side top
    pack $w.top.l3 -side top
    pack $w.top.l4 -side top

    # Set up the plate and Flags section

    SetPlateAndFlags $w $pPlate $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent2 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent3 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent4 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# show_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Pivot -       Variable to recieve pivot number.
# X     -       Variable to receive X position.
# Y     -       Variable to receive Y position.
# Theta -       Variable to receive Theta value.

proc show_dialog {w title pPivot pX pY pTheta} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2
    frame $w.top.l3
    frame $w.top.l4
    global procs_font_1
    label $w.msg1 -width 8 -text "Pivot:" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 8 -text "X:" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    label $w.msg3 -width 8 -text "Y:" -anchor e \
	    -font $procs_font_1
    pack $w.msg3 -in $w.top.l3 -side left  -padx 2m -pady 2m 
    label $w.msg4 -width 8 -text "Theta:" -anchor e \
	    -font $procs_font_1
    pack $w.msg4 -in $w.top.l4 -side left  -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pPivot -borderwidth 2 -relief sunken \
	    -font $procs_font_1 
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $pX -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -state disabled
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent3 -textvariable $pY -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -state disabled
    pack $w.ent3 -in $w.top.l3 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent4 -textvariable $pTheta -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -state disabled
    pack $w.ent4 -in $w.top.l4 -side left -ipadx 2m -padx 5m -pady 2m

    pack $w.top.l1 -side top
    pack $w.top.l2 -side top
    pack $w.top.l3 -side top
    pack $w.top.l4 -side top

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    $w.button0 configure -text Apply

    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Return the selected button

    return $tk_priv(button)
}


#
# prange_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# P0F   -       Variable to recieve P0_FIRST.
# P0L   -       Variable to receive P0_LAST.
# P1F   -       Variable to receive P1_FIRST.
# P1L   -       Variable to receive P1_LAST.

proc prange_dialog {w title P0F P0L P1F P1L} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2
    frame $w.top.l3
    frame $w.top.l4
    global procs_font_1
    label $w.msg1 -width 15 -text "Plate 1 First:" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 15 -text "Plate 1 Last:" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    label $w.msg3 -width 15 -text "Plate 2 First:" -anchor e \
	    -font $procs_font_1
    pack $w.msg3 -in $w.top.l3 -side left  -padx 2m -pady 2m 
    label $w.msg4 -width 15 -text "Plate 2 Last:" -anchor e \
	    -font $procs_font_1
    pack $w.msg4 -in $w.top.l4 -side left  -padx 2m -pady 2m
    entry $w.ent1 -textvariable $P0F -borderwidth 2 -relief sunken \
	    -font $procs_font_1 
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $P0L -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent3 -textvariable $P1F -borderwidth 2 -relief sunken \
	    -font $procs_font_1 
    pack $w.ent3 -in $w.top.l3 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent4 -textvariable $P1L -borderwidth 2 -relief sunken \
	    -font $procs_font_1 
    pack $w.ent4 -in $w.top.l4 -side left -ipadx 2m -padx 5m -pady 2m

    pack $w.top.l1 -side top
    pack $w.top.l2 -side top
    pack $w.top.l3 -side top
    pack $w.top.l4 -side top

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent2 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent3 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent4 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"

    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Return the selected button

    return $tk_priv(button)
}

#
# parkpiv_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# First -       Variable to recieve first pivot number.
# Last  -       Variable to receive last pivot number.
# Plate -       Variable to receive plate number.
# Flags -       Variable to receive flags.

proc parkpiv_dialog {w title pFirst pLast pPlate pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    label $w.msg -text "Enter Range of Pivot Numbers" \
	    -font $procs_font_1
    pack $w.msg -in $w.top -side top -pady 5m -padx 5m
    frame $w.top.l1
    frame $w.top.l2
    label $w.msg1 -width 12 -text "First Pivot" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 12 -text "Last Pivot" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pFirst -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $pLast -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m

    pack $w.top.l1 -side top
    pack $w.top.l2 -side top

    # Set up the plate and Flags section

    SetPlateAndFlags $w $pPlate $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent2 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags $pFlags

    # Return the selected button

    return $tk_priv(button)
}


#
# tumble_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Plate -       Variable to receive plate number.
# Angle -       Mark or Angle to rotate to.
# Flags -       Variable to receive flags.

proc tumble_dialog {w title pPlate pAngle pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    global procs_font_1
    message $w.msg -text "Tumble to Configure " \
	    -font $procs_font_1 -width 100m
    pack $w.msg -in $w.top.l1 -side left -pady 5m -padx 5m

    frame $w.top.plates
    radiobutton $w.top.plates.plate1 -value 0 -variable $pPlate -width 12 \
        -text "Plate 0"
    radiobutton $w.top.plates.plate2 -value 1 -variable $pPlate -width 12 \
        -text "Plate 1"
    pack $w.top.plates.plate1 -side top -ipadx 2m
    pack $w.top.plates.plate2 -side bottom -ipadx 2m
    pack $w.top.plates -in $w.top.l1 -side right -padx 10m -pady 5m

    pack $w.top.l1 -side top

    frame $w.top.marks
    radiobutton $w.top.marks.mark0 -value 0 -width 12 \
       -variable $pAngle -text "0 degrees"
    radiobutton $w.top.marks.mark45 -value 45 -width 12 \
       -variable $pAngle -text "45 degrees"
    radiobutton $w.top.marks.mark90 -value 90 -width 12 \
       -variable $pAngle -text "90 degrees"
    radiobutton $w.top.marks.mark135 -value 135 -width 12 \
       -variable $pAngle -text "135 degrees"
    radiobutton $w.top.marks.mark180 -value 180 -width 12 \
       -variable $pAngle -text "180 degrees"
    radiobutton $w.top.marks.mark225 -value 225 -width 12 \
       -variable $pAngle -text "225 degrees"
    radiobutton $w.top.marks.mark270 -value 270 -width 12 \
       -variable $pAngle -text "270 degrees"
    pack $w.top.marks.mark0 -side top -ipadx 2m 
    pack $w.top.marks.mark45 -side top -ipadx 2m 
    pack $w.top.marks.mark90 -side top -ipadx 2m 
    pack $w.top.marks.mark135 -side top -ipadx 2m 
    pack $w.top.marks.mark180 -side top -ipadx 2m 
    pack $w.top.marks.mark225 -side top -ipadx 2m 
    pack $w.top.marks.mark270 -side top -ipadx 2m 
    pack $w.top.marks -in $w.top -side top -padx 10m -pady 5m

    frame $w.top.l2
    global procs_font_1
    label $w.msg1 -width 14 -text "Tumbler Angle" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l2 -side left -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pAngle -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent1 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    pack $w.top.l2 -side top
    

    # Set up the plate and Flags section

    SetFlags5 $w $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags5 $pFlags

    # Return the selected button

    return $tk_priv(button)
}


#
# reset_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Flags -       Variable to receive flags.

proc reset_dialog {w title pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    message $w.msg -text "Reset Positioner Task and Sub-Tasks" \
	    -font $procs_font_1 -width 100m
    pack $w.msg -in $w.top -side top -pady 5m -padx 5m

    # Set up the plate and Flags section

    SetFlags4 $w $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags4 $pFlags

    # Return the selected button

    return $tk_priv(button)
}



#
# rotate_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# PA -          Variable to receive position angle.
# Plate -       Variable to receive plate number.
# Flags -       Variable to receive flags.

proc rotate_dialog {w title pPA pPlate pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    label $w.msg -text "Position Angle (degrees) " -anchor w \
	    -font $procs_font_1 -width 20
    pack $w.msg -in $w.top -side left -pady 5m -padx 5m
    entry $w.ent -textvariable $pPA -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent -in $w.top -side left -ipadx 2m -padx 5m -pady 2m


    # Set up the plate and Flags section

    SetFlags2 $w $pPlate $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# jaw_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# string -      String to Display.
# Dist -          Variable to receive position angle.
# Flags -       Variable to receive flags.

proc jaw_dialog {w title string pState pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.jaw
    radiobutton $w.top.jaw.close -value CLOSE -width 20\
        -variable $pState -text "CLOSE"
    radiobutton $w.top.jaw.open -value OPEN -width 20 \
        -variable $pState -text "OPEN"
    pack $w.top.jaw -padx 10m -pady 5m -side top
    pack $w.top.jaw.open -side top -ipadx 2m -ipady 2m
    pack $w.top.jaw.close -side bottom -ipadx 2m -ipady 2m


    # Set up the plate and Flags section

    SetFlags3 $w $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w

    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags3 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# fidmark_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# string -      String to Display.
# Dist -          Variable to receive position angle.
# Flags -       Variable to receive flags.

proc fidmark_dialog {w title string pDist pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    label $w.msg -text $string -anchor w \
	    -font $procs_font_1 -width 14
    pack $w.msg -in $w.top -side left -pady 5m -padx 5m
    entry $w.ent -textvariable $pDist -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -width 10
    pack $w.ent -in $w.top -side left -ipadx 2m -padx 5m -pady 2m


    # Set up the plate and Flags section

    SetFlags2 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# survey_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# string -      String to Display.
# Area -        Variable to receive area.
# Max -         Variable to receive maximum.
# Min -         Variable to receive minimum.
# Flags -       Variable to receive flags.

proc survey_dialog {w title string pArea pMax pMin pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    label $w.msg -text "Survey Fiducial Marks" -anchor w \
	    -font $procs_font_1 -width 22
    pack $w.msg -in $w.top -side top -pady 5m -padx 5m

    frame $w.top.area
    pack $w.top.area -side top

    radiobutton $w.top.area.all -text ALL -value ALL -variable $pArea \
        -width 12 -anchor w
    pack $w.top.area.all -side top
    radiobutton $w.top.area.coeffs -text COEFFS -value COEFFS \
        -variable $pArea -width 12 -anchor w
    pack $w.top.area.coeffs -side top
    radiobutton $w.top.area.temp -text TEMP -value TEMP -variable $pArea \
         -width 12 -anchor w
    pack $w.top.area.temp -side top
    radiobutton $w.top.area.flex -text FLEX -value FLEX -variable $pArea \
         -width 12 -anchor w
    pack $w.top.area.flex -side top


    frame $w.top.max
    pack $w.top.max -side top
    global procs_font_1
    label $w.top.max.msg -text "Maximum:" -anchor w \
	    -font $procs_font_1 -width 10
    pack $w.top.max.msg -side left -pady 2m -padx 2m
    entry $w.top.max.ent -textvariable $pMax -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -width 10
    pack $w.top.max.ent -side left -ipadx 2m -padx 2m -pady 2m
    frame $w.top.min
    pack $w.top.min -side top
    label $w.top.min.msg -text "Minimum:" -anchor w \
	    -font $procs_font_1 -width 10
    pack $w.top.min.msg -side left -pady 2m -padx 2m
    entry $w.top.min.ent -textvariable $pMin -borderwidth 2 -relief sunken \
	    -font $procs_font_1 -width 10
    pack $w.top.min.ent -side left -ipadx 2m -padx 2m -pady 2m


    # Set up the plate and Flags section

    SetFlags6 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.top.max.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.top.min.ent <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags6 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# moveg_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# X     -       Variable to receive X position.
# Y     -       Variable to receive Y position.
# Z     -       Variable to receive Z position.
# Theta -       Variable to receive Theta value.
# Flags -       Variable to receive flags.

proc moveg_dialog {w title pX pY pZ pTheta pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2
    frame $w.top.l3
    frame $w.top.l4
    global procs_font_1
    label $w.msg1 -width 8 -text "X:" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 8 -text "Y:" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    label $w.msg3 -width 8 -text "Z:" -anchor e \
	    -font $procs_font_1
    pack $w.msg3 -in $w.top.l3 -side left  -padx 2m -pady 2m 
    label $w.msg4 -width 8 -text "Theta:" -anchor e \
	    -font $procs_font_1
    pack $w.msg4 -in $w.top.l4 -side left  -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pX -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $pY -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent3 -textvariable $pZ -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent3 -in $w.top.l3 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent4 -textvariable $pTheta -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent4 -in $w.top.l4 -side left -ipadx 2m -padx 5m -pady 2m

    pack $w.top.l1 -side top
    pack $w.top.l2 -side top
    pack $w.top.l3 -side top
    pack $w.top.l4 -side top

    # Set up the plate and Flags section

    SetFlags2 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent2 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent3 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent4 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# movef_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# X     -       Variable to receive X position.
# Y     -       Variable to receive Y position.

# Flags -       Variable to receive flags.

proc movef_dialog {w title pX pY pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2
    global procs_font_1
    label $w.msg1 -width 8 -text "X:" -anchor e \
	    -font $procs_font_1
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.msg2 -width 8 -text "Y:" -anchor e \
	    -font $procs_font_1
    pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
    entry $w.ent1 -textvariable $pX -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 5m -pady 2m
    entry $w.ent2 -textvariable $pY -borderwidth 2 -relief sunken \
	    -font $procs_font_1
    pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 5m -pady 2m
    pack $w.top.l1 -side top
    pack $w.top.l2 -side top

    # Set up the plate and Flags section

    SetFlags2 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w
    bind $w.ent1 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
    bind $w.ent2 <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# park_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Flags -       Variable to receive flags.

proc park_dialog {w title pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    message $w.msg -text $title \
	    -font $procs_font_1 -width 100m
    pack $w.msg -in $w.top -side top -pady 5m -padx 5m

    # Set up the plate and Flags section

    SetFlags2 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

#
# home_dialog:
#
# This procedure displays a dialog box, waits for a button in the dialog
# to be invoked, then returns the index of the selected button.
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Axis  -       Axis to Home
# Flags -       Variable to receive flags.

proc home_dialog {w title pAxis pFlags} {
    global tk_priv

    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    global procs_font_1
    message $w.msg -text $title \
	    -font $procs_font_1 -width 100m
    pack $w.msg -in $w.top -side top -pady 5m -padx 5m


    # Set up the axis section

    frame $w.mid.axis
    radiobutton $w.mid.axis.x -value X -variable $pAxis -text "X" -anchor w
    radiobutton $w.mid.axis.y -value Y -variable $pAxis -text "Y" -anchor w

    if {$title == "Home Gripper"} {
      radiobutton $w.mid.axis.z -value Z -variable $pAxis -text "Z" -anchor w
      radiobutton $w.mid.axis.theta -value THETA -variable $pAxis \
         -text "THETA" -anchor w
      radiobutton $w.mid.axis.jaw -value JAW -variable $pAxis \
         -text "JAW" -anchor w
    }

    pack $w.mid.axis.x -side top -ipadx 2m -fill x
    pack $w.mid.axis.y -side top -ipadx 2m -fill x
    if {$title == "Home Gripper"} {
       pack $w.mid.axis.z -side top -ipadx 2m -fill x
       pack $w.mid.axis.theta -side top -ipadx 2m -fill x
       pack $w.mid.axis.jaw -side top -ipadx 2m -fill x
    }
    pack $w.mid.axis  -side left -padx 10m -pady 5m


    # Set up the plate and Flags section

    SetFlags2 $w 0 $pFlags

    # Set up the OK and Cancel butons

    SetUpButtons $w


    # Set up the window and wait for response

    SetUpWindowAndWait $w

    # Set the flags variable

    CompleteFlags2 $pFlags

    # Return the selected button

    return $tk_priv(button)
}

proc SetPlateAndFlags {w pPlate pFlags} {    

    frame $w.mid.plates
    radiobutton $w.mid.plates.plate0 -value 0 -variable $pPlate -text "Plate 0"
    radiobutton $w.mid.plates.plate1 -value 1 -variable $pPlate -text "Plate 1"
    pack $w.mid.plates.plate0 -side top -ipadx 2m -ipady 2m
    pack $w.mid.plates.plate1 -side bottom -ipadx 2m -ipady 2m
    pack $w.mid.plates -in $w.mid -side left -padx 10m -pady 5m


    global Mode
    global FieldCheck
    global OrderCheck
    global ExeFile
    global FileCheck
    global IntCheck
    global DebugDisplay
    global DisplayText
    global BlindPickup
    global JawFull

    set FieldCheck 1
    set OrderCheck 1
    set ExeFile 1
    set FileCheck 1
    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set BlindPickup 0
    set JawFull 0

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.nofield -state $npstate -anchor w \
       -text "Check Target Field" -width 28 -variable FieldCheck
    pack $w.mid.flags.nofield -side top
    checkbutton $w.mid.flags.noorder -state $npstate -anchor w \
       -text "Check Order of Moves" -width 28 -variable OrderCheck
    pack $w.mid.flags.noorder -side top
    checkbutton $w.mid.flags.noexe -state normal -anchor w \
       -text "Execute CommandFile" -width 28 -variable ExeFile
    pack $w.mid.flags.noexe -side top
    checkbutton $w.mid.flags.nofile -state $npstate -anchor w \
       -text "Check Command File" -width 28 -variable FileCheck
    pack $w.mid.flags.nofile -side top
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    pack $w.mid.flags.debug -side top
    checkbutton $w.mid.flags.blind -state $npstate -anchor w \
       -text "Blind Button Pickup" -width 28 -variable BlindPickup
    pack $w.mid.flags.blind -side top
    checkbutton $w.mid.flags.jawfull -state normal -anchor w \
       -text "Open Jaw Wide" -width 28 -variable JawFull
    pack $w.mid.flags.jawfull -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags1 {w pFlags} {    


    global Mode
    global FieldCheck
    global OrderCheck
    global ExeFile
    global FileCheck
    global IntCheck
    global DebugDisplay
    global DisplayText
    global BlindPickup
    global JawFull

    set FieldCheck 1
    set OrderCheck 1
    set ExeFile 1
    set FileCheck 1
    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set BlindPickup 0
    set JawFull 0

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.nofield -state $npstate -anchor w \
       -text "Check Target Field" -width 28 -variable FieldCheck
    pack $w.mid.flags.nofield -side top
    checkbutton $w.mid.flags.noorder -state $npstate -anchor w \
       -text "Check Order of Moves" -width 28 -variable OrderCheck
    pack $w.mid.flags.noorder -side top
    checkbutton $w.mid.flags.noexe -state normal -anchor w \
       -text "Execute CommandFile" -width 28 -variable ExeFile
    pack $w.mid.flags.noexe -side top
    checkbutton $w.mid.flags.nofile -state $npstate -anchor w \
       -text "Check Command File" -width 28 -variable FileCheck
    pack $w.mid.flags.nofile -side top
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    pack $w.mid.flags.debug -side top
    checkbutton $w.mid.flags.blind -state $npstate -anchor w \
       -text "Blind Button Pickup" -width 28 -variable BlindPickup
    pack $w.mid.flags.blind -side top
    checkbutton $w.mid.flags.jawfull -state normal -anchor w \
       -text "Open Jaw Wide" -width 28 -variable JawFull
    pack $w.mid.flags.jawfull -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlagsCF {w pFlags} {    

    global Mode
    global FieldCheck
    global OrderCheck
    global ExeFile
    global FileCheck
    global IntCheck
    global DebugDisplay
    global DisplayText
    global BlindPickup
    global JawFull

    set FieldCheck 1
    set OrderCheck 1
    set ExeFile 1
    set FileCheck 1
    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set BlindPickup 0
    set JawFull 0

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.nofile -state $npstate -anchor w \
       -text "Check Command File" -width 28 -variable FileCheck
    pack $w.mid.flags.nofile -side top
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    pack $w.mid.flags.debug -side top
    checkbutton $w.mid.flags.blind -state $npstate -anchor w \
       -text "Blind Button Pickup" -width 28 -variable BlindPickup
    pack $w.mid.flags.blind -side top
    checkbutton $w.mid.flags.jawfull -state normal -anchor w \
       -text "Open Jaw Wide" -width 28 -variable JawFull
    pack $w.mid.flags.jawfull -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags2 {w pPlate pFlags} {    


    global Mode
    global IntCheck
    global DebugDisplay
    global DisplayText

    if {$pPlate != 0} {
    frame $w.mid.plates
    radiobutton $w.mid.plates.plate0 -value 0 -variable $pPlate -text "Plate 0"
    radiobutton $w.mid.plates.plate1 -value 1 -variable $pPlate -text "Plate 1"
    pack $w.mid.plates.plate0 -side top -ipadx 2m -ipady 2m
    pack $w.mid.plates.plate1 -side bottom -ipadx 2m -ipady 2m
    pack $w.mid.plates -in $w.mid -side left -padx 10m -pady 5m
    }

    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    pack $w.mid.flags.debug -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags3 {w pFlags} {    


    global Mode
    global IntCheck
    global DebugDisplay
    global DisplayText
    global JawFull

    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set JawFull 0

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    checkbutton $w.mid.flags.jawfull -state normal -anchor w \
       -text "Jaw Full" -width 28 -variable JawFull
    pack $w.mid.flags.debug -side top
    pack $w.mid.flags.jawfull -side top  
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags4 {w pFlags} {    


    global Mode
    global ResetGrip
    global ResetFpi
    global ResetPlates
    global ResetDelta

    set ResetGrip 0
    set ResetFpi 0
    set ResetPlates 0
    set ResetDelta 0

    frame $w.mid.flags
    
    checkbutton $w.mid.flags.grip -state normal -anchor w \
       -text "Hard Reset Gripper Gantry Task" -width 40 -variable ResetGrip
    pack $w.mid.flags.grip -side top
    checkbutton $w.mid.flags.fpi -state normal -anchor w \
       -text "Hard Reset FPI Gantry Task" -width 40 -variable ResetFpi
    checkbutton $w.mid.flags.plates -state normal -anchor w \
       -text "Hard Reset Tumbler/Rotator Task" -width 40 -variable ResetPlates
    checkbutton $w.mid.flags.delta -state normal -anchor w \
       -text "Hard Reset Delta Config Task" -width 40 -variable ResetDelta
    pack $w.mid.flags.fpi -side top
    pack $w.mid.flags.plates -side top  
    pack $w.mid.flags.delta -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags5 {w pFlags} {    


    global Mode
    global IntCheck
    global DebugDisplay
    global DisplayText
    global Angle
    global Mark
    global TumbleOnly

    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set Angle 0
    set Mark 0
    set TumbleOnly 0

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    checkbutton $w.mid.flags.angle -state normal -anchor w \
       -text "Angle" -width 28 -variable Angle
    checkbutton $w.mid.flags.mark -state normal -anchor w \
       -text "Mark" -width 28 -variable Mark
    checkbutton $w.mid.flags.tumble -state $npstate -anchor w \
       -text "Tumble Only" -width 28 -variable TumbleOnly
    pack $w.mid.flags.debug -side top
    pack $w.mid.flags.angle -side top 
    pack $w.mid.flags.mark -side top 
    pack $w.mid.flags.tumble -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc SetFlags6 {w pPlate pFlags} {    


    global Mode
    global IntCheck
    global DebugDisplay
    global DisplayText
    global SetCoeffs

    if {$pPlate != 0} {
    frame $w.mid.plates
    radiobutton $w.mid.plates.plate0 -value 0 -variable $pPlate -text "Plate 0"
    radiobutton $w.mid.plates.plate1 -value 1 -variable $pPlate -text "Plate 1"
    pack $w.mid.plates.plate0 -side top -ipadx 2m -ipady 2m
    pack $w.mid.plates.plate1 -side bottom -ipadx 2m -ipady 2m
    pack $w.mid.plates -in $w.mid -side left -padx 10m -pady 5m
    }

    set IntCheck 1
    set DebugDisplay 0
    set DisplayText 1
    set SetCoeffs 1

    frame $w.mid.flags
    if {$Mode == "PROTECTED"} {
      set npstate disabled
    } else {
      set npstate normal
    }
    
    checkbutton $w.mid.flags.noint -state $npstate -anchor w \
       -text "Check Interlocks" -width 28 -variable IntCheck
    pack $w.mid.flags.noint -side top
    checkbutton $w.mid.flags.display -state normal -anchor w \
       -text "Display Informational Text" -width 28 -variable DisplayText
    pack $w.mid.flags.display -side top
    checkbutton $w.mid.flags.debug -state normal -anchor w \
       -text "Display Debugging Text" -width 28 -variable DebugDisplay
    pack $w.mid.flags.debug -side top
    checkbutton $w.mid.flags.set -state normal -anchor w \
       -text "Set Coeffs" -width 28 -variable SetCoeffs
    pack $w.mid.flags.set -side top
    pack $w.mid.flags -side left -padx 10m -pady 5m   

} 

proc CompleteFlags {pFlags} {

    global FieldCheck
    global OrderCheck
    global ExeFile
    global FileCheck
    global IntCheck
    global DebugDisplay
    global DisplayText
    global BlindPickup
    global JawFull


    upvar 2 $pFlags F
    set F ""
    if {$FieldCheck == 0} { lappend F NO_FIELD_CHECK }
    if {$OrderCheck == 0} { lappend F NO_ORDER_CHECK }
    if {$ExeFile == 0} { lappend F NO_FILE_EXE }
    if {$FileCheck == 0} { lappend F NO_FILE_CHECK }
    if {$IntCheck == 0} { lappend F NO_INT_CHECK }
    if {$DebugDisplay == 1} { lappend F DEBUG }
    if {$DisplayText == 1} {lappend F DISPLAY }
    if {$BlindPickup == 1} { lappend F BLIND }
    if {$JawFull == 1} { lappend F JAW_FULL }

}

proc CompleteFlags2 {pFlags} {

    global IntCheck
    global DebugDisplay
    global DisplayText


    upvar 2 $pFlags F
    set F ""
    if {$IntCheck == 0} { lappend F NO_INT_CHECK }
    if {$DebugDisplay == 1} { lappend F DEBUG }
    if {$DisplayText == 1} { lappend F DISPLAY }

}

proc CompleteFlags3 {pFlags} {

    global IntCheck
    global DebugDisplay
    global DisplayText
    global JawFull


    upvar 2 $pFlags F
    set F ""
    if {$IntCheck == 0} { lappend F NO_INT_CHECK }
    if {$DebugDisplay == 1} { lappend F DEBUG }
    if {$DisplayText == 1} { lappend F DISPLAY }
    if {$JawFull == 1} { lappend F JAW_FULL }

}

proc CompleteFlags4 {pFlags} {

    global ResetGrip
    global ResetFpi
    global ResetPlates
    global ResetDelta


    upvar 2 $pFlags F
    set F ""
    if {$ResetGrip == 1} { lappend F TDFGRIP }
    if {$ResetFpi == 1} { lappend F TDFFPI }
    if {$ResetPlates == 1} { lappend F TDFPLATES }
    if {$ResetDelta == 1} { lappend F TDFDELTA }

}

proc CompleteFlags5 {pFlags} {

    global IntCheck
    global DebugDisplay
    global DisplayText
    global Angle
    global Mark
    global TumbleOnly


    upvar 2 $pFlags F
    set F ""
    if {$IntCheck == 0} { lappend F NO_INT_CHECK }
    if {$DebugDisplay == 1} { lappend F DEBUG }
    if {$DisplayText == 1} { lappend F DISPLAY }
    if {$Angle == 1} { lappend F ANGLE }
    if {$Mark == 1} { lappend F MARK }
    if {$TumbleOnly == 1} { lappend F TUMBLE_ONLY }

}

proc CompleteFlags6 {pFlags} {

    global IntCheck
    global DebugDisplay
    global DisplayText
    global SetCoeffs


    upvar 2 $pFlags F
    set F ""
    if {$IntCheck == 0} { lappend F NO_INT_CHECK }
    if {$DebugDisplay == 1} { lappend F DEBUG }
    if {$DisplayText == 1} { lappend F DISPLAY }
    if {$SetCoeffs == 1} { lappend F SET_COEFFS }

}



proc SetUpButtons {w} {

    global tk_priv

    button $w.button0 -text OK -command "set tk_priv(button) 0"
    frame $w.default -relief sunken -bd 1
    raise $w.button0 $w.default
    pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m  
    pack $w.button0 -in $w.default -padx 2m -pady 2m \
		    -ipadx 2m -ipady 1m
    bind $w <Return> "$w.button0 flash; \
                              set tk_priv(button) 0"
	    
    button $w.button2 -text Cancel -command "set tk_priv(button) 2"
    pack $w.button2 -in $w.bot -side left -expand 1 \
		    -padx 3m -pady 3m -ipadx 2m -ipady 1m

}


proc SetUpWindowAndWait {w} {

  # Withdraw the window, then update all the geometry information
  # so we know how big it wants to be, then center the window in the
  # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

 #  Set a grab and claim the focus too.

    set oldFocus [focus]
 #   grab $w
    focus $w

 # Wait for the user to respond, then restore the focus 

    tkwait variable tk_priv(button)
    catch { destroy $w }
    focus $oldFocus

} 





