! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
#define ESMF_FILENAME "ESMF_ArrayCreate.F90"
!==============================================================================







!==============================================================================
! ESMF ArrayCreate module
module ESMF_ArrayCreateMod
!
!==============================================================================
!
! This file contains the ArrayCreate() methods.
!
!------------------------------------------------------------------------------
! INCLUDES
#include "ESMF.h"

!------------------------------------------------------------------------------
!BOPI
! !MODULE: ESMF_ArrayCreateMod - Provide TKR overloading for ESMF_ArrayCreate()
!
! !DESCRIPTION:
!
! The code in this file is part of the {\tt ESMF\_Array} class Fortran API.
!
!
!------------------------------------------------------------------------------
! !USES:
  use ESMF_UtilTypesMod ! ESMF utility types
  use ESMF_InitMacrosMod ! ESMF initializer macros
  use ESMF_BaseMod ! ESMF base class
  use ESMF_LogErrMod ! ESMF error handling
  use ESMF_LocalArrayMod
  use ESMF_ArraySpecMod
  use ESMF_VMMod
  use ESMF_DELayoutMod
  use ESMF_DistGridMod
  use ESMF_RHandleMod
  use ESMF_F90InterfaceMod ! ESMF Fortran-C++ interface helper

  implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
  private

!------------------------------------------------------------------------------
! ! ESMF_Array
!
!------------------------------------------------------------------------------

  ! Fortran class type to hold pointer to C++ object
  type ESMF_Array
#ifndef ESMF_NO_SEQUENCE
  sequence
#endif
! private
    type(ESMF_Pointer) :: this
    ESMF_INIT_DECLARE_NAMED_ALIAS
  end type

!------------------------------------------------------------------------------
! !PUBLIC TYPES:
  public ESMF_Array

!------------------------------------------------------------------------------
!
! !PUBLIC MEMBER FUNCTIONS:

! - ESMF-public methods:
  public operator(==)
  public operator(/=)
  public ESMF_ArrayCopy
  public ESMF_ArrayCreate
  public ESMF_ArrayDestroy

! - ESMF-internal methods:
  public ESMF_ArrayGetInit
  public ESMF_ArraySetInitCreated
  public ESMF_ArrayGetThis
  public ESMF_ArraySetThis
  public ESMF_ArraySetThisNull
  public ESMF_ArraySetPLocalDeInternal
  public ESMF_ArrayCopyThis
  public ESMF_ArrayIsProxy
! ESMF_ArrayCreateFromGrid - defined in ESMF_Grid.F90 due to circularity issues

!EOPI
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
  character(*), parameter, private :: version = &
    '$Id$'

!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================


! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_ArrayCreate -- Generic interface

! !INTERFACE:
  interface ESMF_ArrayCreate

! !PRIVATE MEMBER FUNCTIONS:
!
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateFrmPtr1DI1 
 module procedure ESMF_ArrayCreateFrmPtr2DI1 
 module procedure ESMF_ArrayCreateFrmPtr3DI1 
 module procedure ESMF_ArrayCreateFrmPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtr5DI1 
 module procedure ESMF_ArrayCreateFrmPtr6DI1 
 module procedure ESMF_ArrayCreateFrmPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateFrmPtr1DI2 
 module procedure ESMF_ArrayCreateFrmPtr2DI2 
 module procedure ESMF_ArrayCreateFrmPtr3DI2 
 module procedure ESMF_ArrayCreateFrmPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtr5DI2 
 module procedure ESMF_ArrayCreateFrmPtr6DI2 
 module procedure ESMF_ArrayCreateFrmPtr7DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateFrmPtr1DI4 
 module procedure ESMF_ArrayCreateFrmPtr1DI8 
 module procedure ESMF_ArrayCreateFrmPtr1DR4 
 module procedure ESMF_ArrayCreateFrmPtr1DR8 
 module procedure ESMF_ArrayCreateFrmPtr2DI4 
 module procedure ESMF_ArrayCreateFrmPtr2DI8 
 module procedure ESMF_ArrayCreateFrmPtr2DR4 
 module procedure ESMF_ArrayCreateFrmPtr2DR8 
 module procedure ESMF_ArrayCreateFrmPtr3DI4 
 module procedure ESMF_ArrayCreateFrmPtr3DI8 
 module procedure ESMF_ArrayCreateFrmPtr3DR4 
 module procedure ESMF_ArrayCreateFrmPtr3DR8 
 module procedure ESMF_ArrayCreateFrmPtr4DI4 
 module procedure ESMF_ArrayCreateFrmPtr4DI8 
 module procedure ESMF_ArrayCreateFrmPtr4DR4 
 module procedure ESMF_ArrayCreateFrmPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtr5DI4 
 module procedure ESMF_ArrayCreateFrmPtr5DI8 
 module procedure ESMF_ArrayCreateFrmPtr5DR4 
 module procedure ESMF_ArrayCreateFrmPtr5DR8 
 module procedure ESMF_ArrayCreateFrmPtr6DI4 
 module procedure ESMF_ArrayCreateFrmPtr6DI8 
 module procedure ESMF_ArrayCreateFrmPtr6DR4 
 module procedure ESMF_ArrayCreateFrmPtr6DR8 
 module procedure ESMF_ArrayCreateFrmPtr7DI4 
 module procedure ESMF_ArrayCreateFrmPtr7DI8 
 module procedure ESMF_ArrayCreateFrmPtr7DR4 
 module procedure ESMF_ArrayCreateFrmPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateFrmPtrArb1DI1 
 module procedure ESMF_ArrayCreateFrmPtrArb2DI1 
 module procedure ESMF_ArrayCreateFrmPtrArb3DI1 
 module procedure ESMF_ArrayCreateFrmPtrArb4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArb5DI1 
 module procedure ESMF_ArrayCreateFrmPtrArb6DI1 
 module procedure ESMF_ArrayCreateFrmPtrArb7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateFrmPtrArb1DI2 
 module procedure ESMF_ArrayCreateFrmPtrArb2DI2 
 module procedure ESMF_ArrayCreateFrmPtrArb3DI2 
 module procedure ESMF_ArrayCreateFrmPtrArb4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArb5DI2 
 module procedure ESMF_ArrayCreateFrmPtrArb6DI2 
 module procedure ESMF_ArrayCreateFrmPtrArb7DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateFrmPtrArb1DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb1DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb1DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb1DR8 
 module procedure ESMF_ArrayCreateFrmPtrArb2DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb2DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb2DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb2DR8 
 module procedure ESMF_ArrayCreateFrmPtrArb3DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb3DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb3DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb3DR8 
 module procedure ESMF_ArrayCreateFrmPtrArb4DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb4DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb4DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArb5DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb5DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb5DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb5DR8 
 module procedure ESMF_ArrayCreateFrmPtrArb6DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb6DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb6DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb6DR8 
 module procedure ESMF_ArrayCreateFrmPtrArb7DI4 
 module procedure ESMF_ArrayCreateFrmPtrArb7DI8 
 module procedure ESMF_ArrayCreateFrmPtrArb7DR4 
 module procedure ESMF_ArrayCreateFrmPtrArb7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DI1 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DI1 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DI1 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DI1 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DI1 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DI2 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DI2 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DI2 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DI2 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DI2 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI81DR8 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI82DR8 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI83DR8 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI84DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI85DR8 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI86DR8 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DI4 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DI8 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DR4 
 module procedure ESMF_ArrayCreateFrmPtrArbI87DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateAsmdSp1DI1 
 module procedure ESMF_ArrayCreateAsmdSp2DI1 
 module procedure ESMF_ArrayCreateAsmdSp3DI1 
 module procedure ESMF_ArrayCreateAsmdSp4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSp5DI1 
 module procedure ESMF_ArrayCreateAsmdSp6DI1 
 module procedure ESMF_ArrayCreateAsmdSp7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateAsmdSp1DI2 
 module procedure ESMF_ArrayCreateAsmdSp2DI2 
 module procedure ESMF_ArrayCreateAsmdSp3DI2 
 module procedure ESMF_ArrayCreateAsmdSp4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSp5DI2 
 module procedure ESMF_ArrayCreateAsmdSp6DI2 
 module procedure ESMF_ArrayCreateAsmdSp7DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateAsmdSp1DI4 
 module procedure ESMF_ArrayCreateAsmdSp1DI8 
 module procedure ESMF_ArrayCreateAsmdSp1DR4 
 module procedure ESMF_ArrayCreateAsmdSp1DR8 
 module procedure ESMF_ArrayCreateAsmdSp2DI4 
 module procedure ESMF_ArrayCreateAsmdSp2DI8 
 module procedure ESMF_ArrayCreateAsmdSp2DR4 
 module procedure ESMF_ArrayCreateAsmdSp2DR8 
 module procedure ESMF_ArrayCreateAsmdSp3DI4 
 module procedure ESMF_ArrayCreateAsmdSp3DI8 
 module procedure ESMF_ArrayCreateAsmdSp3DR4 
 module procedure ESMF_ArrayCreateAsmdSp3DR8 
 module procedure ESMF_ArrayCreateAsmdSp4DI4 
 module procedure ESMF_ArrayCreateAsmdSp4DI8 
 module procedure ESMF_ArrayCreateAsmdSp4DR4 
 module procedure ESMF_ArrayCreateAsmdSp4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSp5DI4 
 module procedure ESMF_ArrayCreateAsmdSp5DI8 
 module procedure ESMF_ArrayCreateAsmdSp5DR4 
 module procedure ESMF_ArrayCreateAsmdSp5DR8 
 module procedure ESMF_ArrayCreateAsmdSp6DI4 
 module procedure ESMF_ArrayCreateAsmdSp6DI8 
 module procedure ESMF_ArrayCreateAsmdSp6DR4 
 module procedure ESMF_ArrayCreateAsmdSp6DR8 
 module procedure ESMF_ArrayCreateAsmdSp7DI4 
 module procedure ESMF_ArrayCreateAsmdSp7DI8 
 module procedure ESMF_ArrayCreateAsmdSp7DR4 
 module procedure ESMF_ArrayCreateAsmdSp7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateAsmdSpArb1DI1 
 module procedure ESMF_ArrayCreateAsmdSpArb2DI1 
 module procedure ESMF_ArrayCreateAsmdSpArb3DI1 
 module procedure ESMF_ArrayCreateAsmdSpArb4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArb5DI1 
 module procedure ESMF_ArrayCreateAsmdSpArb6DI1 
 module procedure ESMF_ArrayCreateAsmdSpArb7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateAsmdSpArb1DI2 
 module procedure ESMF_ArrayCreateAsmdSpArb2DI2 
 module procedure ESMF_ArrayCreateAsmdSpArb3DI2 
 module procedure ESMF_ArrayCreateAsmdSpArb4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArb5DI2 
 module procedure ESMF_ArrayCreateAsmdSpArb6DI2 
 module procedure ESMF_ArrayCreateAsmdSpArb7DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateAsmdSpArb1DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb1DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb1DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb1DR8 
 module procedure ESMF_ArrayCreateAsmdSpArb2DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb2DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb2DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb2DR8 
 module procedure ESMF_ArrayCreateAsmdSpArb3DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb3DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb3DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb3DR8 
 module procedure ESMF_ArrayCreateAsmdSpArb4DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb4DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb4DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArb5DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb5DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb5DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb5DR8 
 module procedure ESMF_ArrayCreateAsmdSpArb6DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb6DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb6DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb6DR8 
 module procedure ESMF_ArrayCreateAsmdSpArb7DI4 
 module procedure ESMF_ArrayCreateAsmdSpArb7DI8 
 module procedure ESMF_ArrayCreateAsmdSpArb7DR4 
 module procedure ESMF_ArrayCreateAsmdSpArb7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DI1 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DI1 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DI1 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DI1 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DI1 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DI2 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DI2 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DI2 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DI2 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DI2 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DI2 
#endif 
#endif 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI81DR8 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI82DR8 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI83DR8 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI84DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI85DR8 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI86DR8 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DI4 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DI8 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DR4 
 module procedure ESMF_ArrayCreateAsmdSpArbI87DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    module procedure ESMF_ArrayCreateLocalArray
    module procedure ESMF_ArrayCreateLocalArrayArb
    module procedure ESMF_ArrayCreateLocalArrayArbI8
    module procedure ESMF_ArrayCreateAllocate
    module procedure ESMF_ArrayCreateAllocateArb
    module procedure ESMF_ArrayCreateAllocateArbI8
    module procedure ESMF_ArrayCreateAllocateAS
    module procedure ESMF_ArrayCreateAllocateASArb
    module procedure ESMF_ArrayCreateAllocateASArbI8
    module procedure ESMF_ArrayCreateFromArray

! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_ArrayCreate} functions.
!EOPI
  end interface

  interface ESMF_ArraySetPLocalDeInternal
    module procedure ESMF_ArraySetPLocalDeInternalI4
    module procedure ESMF_ArraySetPLocalDeInternalI8
  end interface


!===============================================================================
! ArrayOperator() interfaces
!===============================================================================

! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_ArrayAssignment(=) - Array assignment
!
! !INTERFACE:
! interface assignment(=)
! array1 = array2
!
! !ARGUMENTS:
! type(ESMF_Array) :: array1
! type(ESMF_Array) :: array2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Assign array1 as an alias to the same ESMF Array object in memory
! as array2. If array2 is invalid, then array1 will be equally invalid after
! the assignment.
!
! The arguments are:
! \begin{description}
! \item[array1]
! The {\tt ESMF\_Array} object on the left hand side of the assignment.
! \item[array2]
! The {\tt ESMF\_Array} object on the right hand side of the assignment.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_ArrayOperator(==) - Array equality operator
!
! !INTERFACE:
  interface operator(==)
! if (array1 == array2) then ... endif
! OR
! result = (array1 == array2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_Array), intent(in) :: array1
! type(ESMF_Array), intent(in) :: array2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether array1 and array2 are valid aliases to the same ESMF
! Array object in memory. For a more general comparison of two ESMF Arrays,
! going beyond the simple alias test, the ESMF\_ArrayMatch() function (not yet
! implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[array1]
! The {\tt ESMF\_Array} object on the left hand side of the equality
! operation.
! \item[array2]
! The {\tt ESMF\_Array} object on the right hand side of the equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_ArrayEQ

  end interface
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_ArrayOperator(/=) - Array not equal operator
!
! !INTERFACE:
  interface operator(/=)
! if (array1 /= array2) then ... endif
! OR
! result = (array1 /= array2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_Array), intent(in) :: array1
! type(ESMF_Array), intent(in) :: array2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether array1 and array2 are {\it not} valid aliases to the
! same ESMF Array object in memory. For a more general comparison of two ESMF
! Arrays, going beyond the simple alias test, the ESMF\_ArrayMatch() function
! (not yet implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[array1]
! The {\tt ESMF\_Array} object on the left hand side of the non-equality
! operation.
! \item[array2]
! The {\tt ESMF\_Array} object on the right hand side of the non-equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_ArrayNE

  end interface
!------------------------------------------------------------------------------


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

contains

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_ArrayEQ()"
!BOPI
! !IROUTINE: ESMF_ArrayEQ - Compare two Calendars for equality
!
! !INTERFACE:
  impure elemental function ESMF_ArrayEQ(array1, array2)
!
! !RETURN VALUE:
    logical :: ESMF_ArrayEQ

! !ARGUMENTS:
    type(ESMF_Array), intent(in) :: array1
    type(ESMF_Array), intent(in) :: array2

! !DESCRIPTION:
! Test if both {\tt array1} and {\tt array2} alias the same ESMF Array
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE ainit1, ainit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ! TODO: Consider moving this logic to C++: use Base class? status?
    ! Or replicate logic for C interface also.

    ! check inputs
    ainit1 = ESMF_ArrayGetInit(array1)
    ainit2 = ESMF_ArrayGetInit(array2)

    ! TODO: this line must remain split in two for SunOS f90 8.3 127000-03
    if (ainit1 .eq. ESMF_INIT_CREATED .and. &
      ainit2 .eq. ESMF_INIT_CREATED) then
      ESMF_ArrayEQ = array1%this .eq. array2%this
    else
      ESMF_ArrayEQ = .false.
    endif

  end function ESMF_ArrayEQ
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_ArrayNE()"
!BOPI
! !IROUTINE: ESMF_ArrayNE - Compare two Calendars for non-equality
!
! !INTERFACE:
  impure elemental function ESMF_ArrayNE(array1, array2)
!
! !RETURN VALUE:
    logical :: ESMF_ArrayNE

! !ARGUMENTS:
    type(ESMF_Array), intent(in) :: array1
    type(ESMF_Array), intent(in) :: array2

! !DESCRIPTION:
! Test if both {\tt array1} and {\tt array2} alias the same ESMF Array
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE ainit1, ainit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ESMF_ArrayNE = .not.ESMF_ArrayEQ(array1, array2)

  end function ESMF_ArrayNE
!-------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_ArrayCopy()"
!BOP
! !IROUTINE: ESMF_ArrayCopy - Copy data from one Array object to another

! !INTERFACE:
  subroutine ESMF_ArrayCopy(arrayOut, arrayIn, keywordEnforcer, rc)
!
! !ARGUMENTS:
       type(ESMF_Array), intent(inout) :: arrayOut
       type(ESMF_Array), intent(in) :: arrayIn
       type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
       integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Copy data from one {\tt ESMF\_Array} object to another.
!
! The arguments are:
! \begin{description}
! \item[arrayOut]
! {\tt ESMF\_Array} object into which to copy the data. The incoming
! {\tt arrayOut} must already references a matching memory allocation.
! \item[arrayIn]
! {\tt ESMF\_Array} object that holds the data to be copied.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! Initialize return code; assume failure until success is certain
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_ArrayGetInit, arrayIn, rc)
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_ArrayGetInit, arrayOut, rc)

    ! Call into the C++ interface
    call c_ESMC_ArrayCopy(arrayOut, arrayIn, localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_ArrayCopy
!------------------------------------------------------------------------------


!===============================================================================
! ArrayCreate() interfaces
!===============================================================================
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
! -------------------------- ESMF-public method ----------------------------- 
!BOP 
! !IROUTINE: ESMF_ArrayCreate - Create Array object from Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_ArrayCreate() 
! function ESMF_ArrayCreateFrmPtr<rank><type><kind>(distgrid, farrayPtr, & 
! keywordEnforcer, datacopyflag, distgridToArrayMap, computationalEdgeLWidth, & 
! computationalEdgeUWidth, computationalLWidth, & 
! computationalUWidth, totalLWidth, & 
! totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Array) :: ESMF_ArrayCreateDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_DistGrid), intent(in) :: distgrid 
! <type> (ESMF_KIND_<kind>), pointer :: farrayPtr(<rank>) 
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: distgridToArrayMap(:) 
! integer, intent(in), optional :: computationalEdgeLWidth(:) 
! integer, intent(in), optional :: computationalEdgeUWidth(:) 
! integer, intent(in), optional :: computationalLWidth(:) 
! integer, intent(in), optional :: computationalUWidth(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Array} object from existing local native Fortran 
! arrays with pointer attribute. The decomposition and distribution is 
! specified by the {\tt distgrid} argument. Each PET must issue this call 
! with identical arguments in order to create a consistent Array object. 
! The only exception is the {\tt farrayPtr} argument which will be different 
! on each PET. The bounds of the local arrays are preserved by this call and 
! determine the bounds of the total region of the 
! resulting Array object. Bounds of the DE-local exclusive regions are set 
! to be consistent with the total regions and the specified {\tt distgrid} 
! argument. Bounds for Array dimensions that are not distributed are 
! automatically set to the bounds provided by {\tt farrayPtr}. 
! 
! This interface requires a 1 DE per PET decomposition. The Array object will 
! not be created and an error will be returned if this condition is not met. 
! 
! The not distributed Array dimensions form a tensor of rank = array.rank - 
! distgrid.dimCount. The widths of the computational region are set to 
! the provided value, or zero by default, for all tensor elements. Use 
! {\tt ESMF\_ArraySet()} to change these default settings after the 
! Array object has been created. 
! 
! The return value is the newly created {\tt ESMF\_Array} object. 
! 
! The arguments are: 
! \begin{description} 
! \item[distgrid] 
! {\tt ESMF\_DistGrid} object that describes how the array is decomposed and 
! distributed over DEs. The dimCount of distgrid must be smaller or equal 
! to the rank of {\tt farrayPtr}. 
! \item[farrayPtr] 
! Valid native Fortran array with pointer attribute. Memory must be 
! associated with the actual argument. The type/kind/rank information of 
! {\tt farrayPtr} will be used to set {\tt Array}'s properties 
! accordingly. The shape of {\tt farrayPtr} will be checked against the 
! information contained in the {\tt distgrid}. The bounds of 
! {\tt farrayPtr} will be preserved by this call and the bounds of the 
! resulting Array object are set accordingly. 
! \item[{[datacopyflag]}] 
! Specifies whether the Array object will reference the memory allocation 
! provided by {\tt farrayPtr} directly or will copy the data from 
! {\tt farrayPtr} into a new memory allocation. For valid values see 
! \ref{const:datacopyflag}. The default is {\tt ESMF\_DATACOPY\_REFERENCE}. 
! Note that the {\tt ESMF\_DATACOPY\_REFERENCE} option may not be safe 
! when providing an array slice in {\tt farrayPtr}. 
! \item[{[distgridToArrayMap]}] 
! List that contains as many elements as is indicated by 
! {\tt distgrids}'s dimCount. The list elements map each dimension of 
! the DistGrid object to a dimension in {\tt farrayPtr} by specifying the 
! appropriate Array dimension index. The default is to map all of 
! {\tt distgrid}'s dimensions against the lower dimensions of the 
! {\tt farrayPtr} argument in sequence, i.e. {\tt distgridToArrayMap = 
! (/1, 2, .../)}. 
! Unmapped {\tt farrayPtr} dimensions are not decomposed dimensions and 
! form a tensor of rank = Array.rank - DistGrid.dimCount. 
! All {\tt distgridToArrayMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Array rank. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Array rank is less than the DistGrid dimCount then the default 
! distgridToArrayMap will contain zeros for the dimCount - rank 
! rightmost entries. A zero entry in the {\tt distgridToArrayMap} 
! indicates that the particular DistGrid dimension will be replicating 
! the Array across the DEs along this direction. 
! \item[{[computationalEdgeLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalEdgeUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[computationalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[totalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the total memory 
! region with respect to the lower corner of the exclusive region. 
! The default is to accommodate the union of exclusive and computational 
! region exactly. 
! \item[{[totalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the total memory 
! region with respect to the upper corner of the exclusive region. 
! The default is a vector that contains the remaining number of elements 
! in each direction as to fit the union of exclusive and computational 
! region into the memory region provided by the {\tt farrayPtr} argument. 
! \item[{[name]}] 
! Name of the Array object. 
! \item[{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Di1(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Di2(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr1Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr1Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr1Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr1Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr1Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr1Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr2Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr2Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr2Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr2Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr2Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr2Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr3Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr3Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr3Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr3Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr3Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr3Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr4Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr4Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr4Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr4Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr4Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr5Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr5Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr5Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr5Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr5Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr5Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr6Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr6Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr6Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr6Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr6Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr6Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Di4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Di8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Dr4(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtr" 
 function ESMF_ArrayCreateFrmPtr7Dr8(distgrid, farrayPtr, & 
 keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtr7Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateFrmPtr7Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Must pass the datacopyflag_opt flag into LocalArrayCreate() to ensure 
 ! that the LocalArray object contained in the localarrayList, passed to 
 ! ArrayCreate() below, is suitable to be placed inside of the Array 
 ! object directly. ArrayCreate() does _not_ create LocalArray objects 
 ! internally for ESMF_INDEX_USER. 
 allocate(localarrayList(1)) 
 localarrayList(1) = ESMF_LocalArrayCreate(farrayPtr, datacopyflag=datacopyflag_opt, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=ESMF_INDEX_USER, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have placed the 
 ! LocalArray object created above into the Array object. It will be 
 ! destroyed when the Array object is destroyed. Only the localarrayList 
 ! must be cleaned up here. 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateFrmPtr7Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateFrmPtr7Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
! -------------------------- ESMF-public method ----------------------------- 
!BOP 
! !IROUTINE: ESMF_ArrayCreate - Create Array object from Fortran array pointer w/ arbitrary seqIndices for halo
! 
! !INTERFACE: 
! ! Private name; call using ESMF_ArrayCreate() 
! function ESMF_ArrayCreateFrmPtrArb<indexkind><rank><type><kind>(distgrid, & 
! farrayPtr, haloSeqIndexList, keywordEnforcer, datacopyflag, & 
! distgridToArrayMap, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Array) :: ESMF_ArrayCreateDataPtrArb<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_DistGrid), intent(in) :: distgrid 
! <type> (ESMF_KIND_<kind>), pointer :: farrayPtr(<rank>) 
! integer(ESMF_KIND_<indexkind>), intent(in) :: haloSeqIndexList(:) 
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: distgridToArrayMap(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Array} object from existing local native Fortran 
! arrays with pointer attribute, according to distgrid. Besides 
! {\tt farrayPtr} each PET must issue this call with identical arguments in 
! order to create a consistent Array object. The bounds of the local arrays 
! are preserved by this call and determine the bounds of the total region of 
! the resulting Array object. Bounds of the DE-local exclusive regions are 
! set to be consistent with the total regions and the specified distgrid 
! argument. Bounds for Array dimensions that are not distributed are 
! automatically set to the bounds provided by {\tt farrayPtr}. 
! 
! This interface requires a 1 DE per PET decomposition. The Array object will 
! not be created and an error will be returned if this condition is not met. 
! 
! The not distributed Array dimensions form a tensor of rank = array.rank - 
! distgrid.dimCount. The widths of the computational region are set to 
! the provided value, or zero by default, for all tensor elements. Use 
! {\tt ESMF\_ArraySet()} to change these default settings after the 
! Array object has been created. 
! 
! The return value is the newly created {\tt ESMF\_Array} object. 
! 
! The arguments are: 
! \begin{description} 
! \item[distgrid] 
! {\tt ESMF\_DistGrid} object that describes how the array is decomposed and 
! distributed over DEs. The dimCount of distgrid must be smaller or equal 
! to the rank of {\tt farrayPtr}. 
! \item[farrayPtr] 
! Valid native Fortran array with pointer attribute. Memory must be 
! associated with the actual argument. The type/kind/rank information of 
! {\tt farrayPtr} will be used to set {\tt Array}'s properties 
! accordingly. The shape of {\tt farrayPtr} will be checked against the 
! information contained in the {\tt distgrid}. The bounds of 
! {\tt farrayPtr} will be preserved by this call and the bounds of the 
! resulting Array object are set accordingly. 
! \item[haloSeqIndexList] 
! One dimensional array containing sequence indices of local halo region. 
! The size (and content) of {\tt haloSeqIndexList} can (and typically will) 
! be different on each PET. 
! The {\tt haloSeqIndexList} argument is of integer type, but can be of 
! different kind in order to support both 32-bit ({\tt ESMF\_KIND\_I4}) 
! and 64-bit ({\tt ESMF\_KIND\_I8}) indexing. 
! \item[{[datacopyflag]}] 
! Specifies whether the Array object will reference the memory allocation 
! provided by {\tt farrayPtr} directly or will copy the data from 
! {\tt farrayPtr} into a new memory allocation. For valid values see 
! \ref{const:datacopyflag}. The default is {\tt ESMF\_DATACOPY\_REFERENCE}. 
! Note that the {\tt ESMF\_DATACOPY\_REFERENCE} option may not be safe 
! when providing an array slice in {\tt farrayPtr}. 
! \item[{[distgridToArrayMap]}] 
! List that contains as many elements as is indicated by 
! {\tt distgrids}'s dimCount. The list elements map each dimension of 
! the DistGrid object to a dimension in {\tt farrayPtr} by specifying the 
! appropriate Array dimension index. The default is to map all of 
! {\tt distgrid}'s dimensions against the lower dimensions of the 
! {\tt farrayPtr} argument in sequence, i.e. {\tt distgridToArrayMap = 
! (/1, 2, .../)}. 
! Unmapped {\tt farrayPtr} dimensions are not decomposed dimensions and 
! form a tensor of rank = Array.rank - DistGrid.dimCount. 
! All {\tt distgridToArrayMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Array rank. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Array rank is less than the DistGrid dimCount then the default 
! distgridToArrayMap will contain zeros for the dimCount - rank 
! rightmost entries. A zero entry in the {\tt distgridToArrayMap} 
! indicates that the particular DistGrid dimension will be replicating 
! the Array across the DEs along this direction. 
! \item[{[name]}] 
! Name of the Array object. 
! \item[{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb1Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb1Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb1Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb1Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb1Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb1Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb2Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb2Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb2Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb2Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb2Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb2Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb3Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb3Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb3Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb3Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb3Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb3Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb4Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb4Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb4Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb4Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb4Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb5Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb5Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb5Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb5Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb5Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb5Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb6Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb6Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb6Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb6Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb6Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb6Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArb" 
 function ESMF_ArrayCreateFrmPtrArb7Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArb7Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArb7Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArb7Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArb7Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArb7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Di1(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Di1%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Di1 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Di1, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Di2(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Di2%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Di2 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Di2, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI81Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI81Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI81Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI81Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI81Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI81Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI82Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI82Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI82Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI82Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI82Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI82Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI83Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI83Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI83Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI83Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI83Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI83Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI84Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI84Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI84Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI84Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI84Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI84Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI85Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI85Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI85Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI85Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI85Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI85Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI86Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI86Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI86Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI86Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI86Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI86Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Di4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Di4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Di4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Di4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Di8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Di8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Di8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Di8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Dr4(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Dr4%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Dr4 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Dr4, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateFrmPtrArbI8" 
 function ESMF_ArrayCreateFrmPtrArbI87Dr8(distgrid, farrayPtr, & 
 haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateFrmPtrArbI87Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),pointer :: farrayPtr 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 integer :: localHaloCount 
 integer :: dimCount, localDeCount 
 type(ESMF_DELayout) :: delayout 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! invalidate return value 
 ESMF_ArrayCreateFrmPtrArbI87Dr8%this = ESMF_NULL_POINTER 

 ! Check init status of arguments used directly 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Ensure incoming DistGrid is 1D 
 call ESMF_DistGridGet(distgrid, dimCount=dimCount, delayout=delayout, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (dimCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface requires 1D DistGrid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Ensure there is only 1 DE/PET 
 call ESMF_DELayoutGet(delayout, localDeCount=localDeCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (localDeCount /= 1) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- this ArrayCreate() interface only supports 1DE/PETs", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare for actual ArrayCreate() call 
 localHaloCount = size(haloSeqIndexList) 

 ! Call into conventional interface to create the Array object 
 ESMF_ArrayCreateFrmPtrArbI87Dr8 = & 
 ESMF_ArrayCreate(distgrid, farrayPtr, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap, totalUWidth=(/localHaloCount/), & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Set the arbitrary sequence indices in the Array halo rim 
 call ESMF_ArraySetPLocalDeInternal( & 
 ESMF_ArrayCreateFrmPtrArbI87Dr8, & 
 localDe=0, rimSeqIndex=haloSeqIndexList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateFrmPtrArbI87Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
! -------------------------- ESMF-public method ----------------------------- 
!BOP 
! !IROUTINE: ESMF_ArrayCreate - Create Array object from Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_ArrayCreate() 
! function ESMF_ArrayCreateAsmdSp<rank><type><kind>(distgrid, farray, & 
! indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
! computationalEdgeLWidth, computationalEdgeUWidth, computationalLWidth, & 
! computationalUWidth, totalLWidth, & 
! totalUWidth, undistLBound, undistUBound, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Array) :: ESMF_ArrayCreateDataAssmdShape<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_DistGrid), intent(in) :: distgrid 
! <type> (ESMF_KIND_<kind>), intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: distgridToArrayMap(:) 
! integer, intent(in), optional :: computationalEdgeLWidth(:) 
! integer, intent(in), optional :: computationalEdgeUWidth(:) 
! integer, intent(in), optional :: computationalLWidth(:) 
! integer, intent(in), optional :: computationalUWidth(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(in), optional :: undistLBound(:) 
! integer, intent(in), optional :: undistUBound(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Array} object from an existing local native Fortran 
! array. The decomposition and distribution is 
! specified by the {\tt distgrid} argument. Each PET must issue this call 
! with identical arguments in order to create a consistent Array object. 
! The only exception is the {\tt farray} argument which will be different 
! on each PET. The local arrays provided must be dimensioned according to 
! the DE-local total region. Bounds of the exclusive regions are set as 
! specified in the {\tt distgrid} argument. Bounds for Array dimensions 
! that are not distributed can be chosen freely using the 
! {\tt undistLBound} and {\tt undistUBound} arguments. 
! 
! This interface requires a 1 DE per PET decomposition. The Array object will 
! not be created and an error will be returned if this condition is not met. 
! 
! The not distributed Array dimensions form a tensor of rank = array.rank - 
! distgrid.dimCount. The widths of the computational region are set to 
! the provided value, or zero by default, for all tensor elements. Use 
! {\tt ESMF\_ArraySet()} to change these default settings after the 
! Array object has been created. 
! 
! The return value is the newly created {\tt ESMF\_Array} object. 
! 
! The arguments are: 
! \begin{description} 
! \item[distgrid] 
! {\tt ESMF\_DistGrid} object that describes how the array is decomposed and 
! distributed over DEs. The dimCount of distgrid must be smaller or equal 
! to the rank of farray. 
! \item[farray] 
! Valid native Fortran array, i.e. memory must be associated with the 
! actual argument. The type/kind/rank information of {\tt farray} will be 
! used to set {\tt Array}'s properties accordingly. The shape of 
! {\tt farray} will be checked against the information contained in the 
! {\tt distgrid}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item[{[datacopyflag]}] 
! Specifies whether the Array object will reference the memory allocation 
! provided by {\tt farray} directly or will copy the data from 
! {\tt farray} into a new memory allocation. For valid values see 
! \ref{const:datacopyflag}. The default is {\tt ESMF\_DATACOPY\_REFERENCE}. 
! Note that the {\tt ESMF\_DATACOPY\_REFERENCE} option may not be safe 
! when providing an array slice in {\tt farray}. 
! \item[{[distgridToArrayMap]}] 
! List that contains as many elements as is indicated by 
! {\tt distgrids}'s dimCount. The list elements map each dimension of 
! the DistGrid object to a dimension in {\tt farray} by specifying the 
! appropriate Array dimension index. The default is to map all of 
! {\tt distgrid}'s dimensions against the lower dimensions of the 
! {\tt farray} argument in sequence, i.e. {\tt distgridToArrayMap = 
! (/1, 2, .../)}. 
! Unmapped {\tt farray} dimensions are not decomposed dimensions and 
! form a tensor of rank = Array.rank - DistGrid.dimCount. 
! All {\tt distgridToArrayMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Array rank. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Array rank is less than the DistGrid dimCount then the default 
! distgridToArrayMap will contain zeros for the dimCount - rank 
! rightmost entries. A zero entry in the {\tt distgridToArrayMap} 
! indicates that the particular DistGrid dimension will be replicating 
! the Array across the DEs along this direction. 
! \item[{[computationalEdgeLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalEdgeUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[computationalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[totalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the total memory 
! region with respect to the lower corner of the exclusive region. 
! The default is to accommodate the union of exclusive and computational 
! region exactly. 
! \item[{[totalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the total memory 
! region with respect to the upper corner of the exclusive region. 
! The default is a vector that contains the remaining number of elements 
! in each direction as to fit the union of exclusive and computational 
! region into the memory region provided by the {\tt farray} argument. 
! \item[{[undistLBound]}] 
! Lower bounds for the array dimensions that are not distributed. 
! By default lbound is 1. 
! \item[{[undistUBound]}] 
! Upper bounds for the array dimensions that are not distributed. 
! By default ubound is equal to the extent of the corresponding 
! dimension in {\tt farray}. 
! \item[{[name]}] 
! Name of the Array object. 
! \item[{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Di1(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Di2(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp1Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp1Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp1Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp1Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp1Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp1Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp2Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp2Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp2Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp2Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp2Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp2Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp3Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp3Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp3Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp3Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp3Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp3Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp4Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp4Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp4Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp4Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp4Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp5Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp5Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp5Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp5Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp5Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp5Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp6Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp6Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp6Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp6Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp6Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp6Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Di4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Di8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Dr4(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSp" 
 function ESMF_ArrayCreateAsmdSp7Dr8(distgrid, farray, & 
 indexflag, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSp7Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSp7Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSp7Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSp7Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSp7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
! -------------------------- ESMF-public method ----------------------------- 
!BOP 
! !IROUTINE: ESMF_ArrayCreate - Create Array object from Fortran array w/ arbitrary seqIndices for halo
! 
! !INTERFACE: 
! ! Private name; call using ESMF_ArrayCreate() 
! function ESMF_ArrayCreateAsmdSpArb<indexkind><rank><type><kind>(distgrid, & 
! farray, indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, & 
! distgridToArrayMap, computationalEdgeLWidth, computationalEdgeUWidth, & 
! computationalLWidth, computationalUWidth, totalLWidth, totalUWidth, & 
! undistLBound, undistUBound, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Array) :: ESMF_ArrayCreateDataAssmdShapeArb<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_DistGrid), intent(in) :: distgrid 
! <type> (ESMF_KIND_<kind>), intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
! integer(ESMF_KIND_<indexkind>), intent(in) :: haloSeqIndexList(:) 
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: distgridToArrayMap(:) 
! integer, intent(in), optional :: computationalEdgeLWidth(:) 
! integer, intent(in), optional :: computationalEdgeUWidth(:) 
! integer, intent(in), optional :: computationalLWidth(:) 
! integer, intent(in), optional :: computationalUWidth(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(in), optional :: undistLBound(:) 
! integer, intent(in), optional :: undistUBound(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Array} object from an existing local native Fortran 
! array. The decomposition and distribution is 
! specified by the {\tt distgrid} argument. Each PET must issue this call 
! with identical arguments in order to create a consistent Array object. 
! The only exception is the {\tt farray} argument which will be different 
! on each PET. The local arrays provided must be dimensioned according to 
! the DE-local total region. Bounds of the exclusive regions are set as 
! specified in the {\tt distgrid} argument. Bounds for Array dimensions 
! that are not distributed can be chosen freely using the 
! {\tt undistLBound} and {\tt undistUBound} arguments. 
! 
! This interface requires a 1 DE per PET decomposition. The Array object will 
! not be created and an error will be returned if this condition is not met. 
! 
! The not distributed Array dimensions form a tensor of rank = array.rank - 
! distgrid.dimCount. The widths of the computational region are set to 
! the provided value, or zero by default, for all tensor elements. Use 
! {\tt ESMF\_ArraySet()} to change these default settings after the 
! Array object has been created. 
! 
! The return value is the newly created {\tt ESMF\_Array} object. 
! 
! The arguments are: 
! \begin{description} 
! \item[distgrid] 
! {\tt ESMF\_DistGrid} object that describes how the array is decomposed and 
! distributed over DEs. The dimCount of distgrid must be smaller or equal 
! to the rank of farray. 
! \item[farray] 
! Valid native Fortran array, i.e. memory must be associated with the 
! actual argument. The type/kind/rank information of {\tt farray} will be 
! used to set {\tt Array}'s properties accordingly. The shape of 
! {\tt farray} will be checked against the information contained in the 
! {\tt distgrid}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item[haloSeqIndexList] 
! One dimensional array containing sequence indices of local halo region. 
! The size (and content) of {\tt haloSeqIndexList} can (and typically will) 
! be different on each PET. 
! The {\tt haloSeqIndexList} argument is of integer type, but can be of 
! different kind in order to support both 32-bit ({\tt ESMF\_KIND\_I4}) 
! and 64-bit ({\tt ESMF\_KIND\_I8}) indexing. 
! \item[{[datacopyflag]}] 
! Specifies whether the Array object will reference the memory allocation 
! provided by {\tt farray} directly or will copy the data from 
! {\tt farray} into a new memory allocation. For valid values see 
! \ref{const:datacopyflag}. The default is {\tt ESMF\_DATACOPY\_REFERENCE}. 
! Note that the {\tt ESMF\_DATACOPY\_REFERENCE} option may not be safe 
! when providing an array slice in {\tt farray}. 
! \item[{[distgridToArrayMap]}] 
! List that contains as many elements as is indicated by 
! {\tt distgrids}'s dimCount. The list elements map each dimension of 
! the DistGrid object to a dimension in {\tt farray} by specifying the 
! appropriate Array dimension index. The default is to map all of 
! {\tt distgrid}'s dimensions against the lower dimensions of the 
! {\tt farray} argument in sequence, i.e. {\tt distgridToArrayMap = 
! (/1, 2, .../)}. 
! Unmapped {\tt farray} dimensions are not decomposed dimensions and 
! form a tensor of rank = Array.rank - DistGrid.dimCount. 
! All {\tt distgridToArrayMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Array rank. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Array rank is less than the DistGrid dimCount then the default 
! distgridToArrayMap will contain zeros for the dimCount - rank 
! rightmost entries. A zero entry in the {\tt distgridToArrayMap} 
! indicates that the particular DistGrid dimension will be replicating 
! the Array across the DEs along this direction. 
! \item[{[computationalEdgeLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalEdgeUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region for DEs 
! that are located on the edge of a tile. 
! The default is a zero vector. 
! \item[{[computationalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the computational 
! region with respect to the lower corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[computationalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the computational 
! region with respect to the upper corner of the exclusive region. 
! The default is a zero vector. 
! \item[{[totalLWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the lower corner of the total memory 
! region with respect to the lower corner of the exclusive region. 
! The default is to accommodate the union of exclusive and computational 
! region exactly. 
! \item[{[totalUWidth]}] 
! This vector argument must have dimCount elements, where dimCount is 
! specified in distgrid. It specifies the upper corner of the total memory 
! region with respect to the upper corner of the exclusive region. 
! The default is a vector that contains the remaining number of elements 
! in each direction as to fit the union of exclusive and computational 
! region into the memory region provided by the {\tt farray} argument. 
! \item[{[undistLBound]}] 
! Lower bounds for the array dimensions that are not distributed. 
! By default lbound is 1. 
! \item[{[undistUBound]}] 
! Upper bounds for the array dimensions that are not distributed. 
! By default ubound is equal to the extent of the corresponding 
! dimension in {\tt farray}. 
! \item[{[name]}] 
! Name of the Array object. 
! \item[{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb1Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb1Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb1Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb1Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb1Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb1Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb2Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb2Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb2Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb2Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb2Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb2Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb3Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb3Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb3Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb3Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb3Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb3Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb4Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb4Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb4Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb4Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb4Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb5Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb5Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb5Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb5Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb5Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb5Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb6Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb6Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb6Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb6Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb6Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb6Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArb" 
 function ESMF_ArrayCreateAsmdSpArb7Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArb7Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I4), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArb7Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArb7Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArb7Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArb7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI85Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI85Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI85Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI86Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI86Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI86Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI86Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI86Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI86Di1 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI87Di1(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI87Di1 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i1),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI87Di1 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI87Di1 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI87Di1) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI87Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI85Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI85Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI85Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI86Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI86Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI86Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI86Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI86Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI86Di2 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI87Di2(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI87Di2 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i2),dimension(:,:,:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI87Di2 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI87Di2 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI87Di2) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI87Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI81Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI81Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI81Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI81Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI81Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI81Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI82Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI82Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI82Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI82Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI82Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI82Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI83Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI83Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI83Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI83Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI83Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI83Dr8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI84Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI84Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI84Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI84Dr8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI84Dr8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI84Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Di4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Di4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Di4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI85Di4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI85Di4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI85Di4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Di8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Di8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 integer (ESMF_KIND_i8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Di8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI85Di8 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI85Di8) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI85Di8 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Dr4(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Dr4 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r4),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Dr4 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Garbage collection. The ArrayCreate() call will have created a new 
 ! LocalArray object internally, so the temporary object created above 
 ! must be deleted here. 
 call ESMF_LocalArrayDestroy(localarrayList(1), rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 deallocate(localarrayList) 

 ! Set return value 
 ESMF_ArrayCreateAsmdSpArbI85Dr4 = array 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_ArrayCreateAsmdSpArbI85Dr4) 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_ArrayCreateAsmdSpArbI85Dr4 
!---------------------------------------------------------------------------- 
 
! -------------------------- ESMF-public method ----------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_ArrayCreateAsmdSpArbI8" 
 function ESMF_ArrayCreateAsmdSpArbI85Dr8(distgrid, farray, & 
 indexflag, haloSeqIndexList, keywordEnforcer, datacopyflag, distgridToArrayMap, & 
 computationalEdgeLWidth, computationalEdgeUWidth, & 
 computationalLWidth, computationalUWidth, totalLWidth, & 
 totalUWidth, undistLBound, undistUBound, name, rc) 

 type(ESMF_Array) :: ESMF_ArrayCreateAsmdSpArbI85Dr8 

 type(ESMF_DistGrid), intent(in) :: distgrid 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),intent(in),target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 integer(ESMF_KIND_I8), intent(in) :: haloSeqIndexList(:) 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: distgridToArrayMap(:) 
 integer, intent(in), optional :: computationalEdgeLWidth(:) 
 integer, intent(in), optional :: computationalEdgeUWidth(:) 
 integer, intent(in), optional :: computationalLWidth(:) 
 integer, intent(in), optional :: computationalUWidth(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(in), optional :: undistLBound(:) 
 integer, intent(in), optional :: undistUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! Local variables 
 integer :: localrc ! local return code 
 type(ESMF_Array) :: array ! new Array 
 type(ESMF_LocalArray), allocatable :: localarrayList(:) ! helper variable 
 real (ESMF_KIND_r8),dimension(:,:,:,:,:),pointer :: fptr ! helper variable 
 type(ESMF_DataCopy_Flag) :: datacopyflag_opt ! helper variable 

 ! Initialize return code; assume failure until success is certain 
 localrc = ESMF_RC_NOT_IMPL 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 

 ! Mark this Array object as invalid 
 array%this = ESMF_NULL_POINTER 
 ESMF_ArrayCreateAsmdSpArbI85Dr8 = array 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_DistGridGetInit, distgrid, rc) 

 ! Set copy/ref behavior 
 datacopyflag_opt = ESMF_DATACOPY_REFERENCE ! default 
 if (present(datacopyflag)) datacopyflag_opt = datacopyflag 

 ! Check that indexflag does not specify mode that is not supported here 
 if (indexflag==ESMF_INDEX_USER) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- ESMF_INDEX_USER not supported as input value", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 ! Prepare the LocalArray list to be used in the ArrayCreate() call. 
 ! Do not pass the datacopyflag_opt flag into LocalArrayCreate() to prevent 
 ! unncessary data movements at this point. The ArrayCreate() below will 
 ! create a new LocalArray internally (to be used inside of the Array 
 ! object), and that is where the datacopyflag_opt will need to be considered. 
 allocate(localarrayList(1)) 
 fptr => farray 
 localarrayList(1) = ESMF_LocalArrayCreate(fptr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create the Array object by calling into overloaded ArrayCreate() interf. 
 array = ESMF_ArrayCreate(distgrid, localarrayList, & 
 haloSeqIndexList=haloSeqIndexList, indexflag=indexflag, & 
 datacopyflag=datacopyflag_opt, distgridToArrayMap=distgridToArrayMap, & 
 computationalEdgeLWidth=computationalEdgeLWidth, & 
 computationalEdgeUWidth=computationalEdgeUWidth, & 
 computationalLWidth=computationalLWidth, & 
 computationalUWidth=computationalUWidth, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 undistLBound=undistLBound, undistUBound=undis