/* A new version of pix */
/* Use a combination of statistical input files and overlays generated by xds to
	edit overlays to match needs of user */
/* $log$ $Id: npix.c,v 1.5 2006/09/26 04:59:59 mscohen Exp mscohen $ */

#include "ImgLib.h"

typedef struct {
	char    InFileName[255];
	char    Ovlfname[256];
	char    progname[256];
	IMAGE   inIm;
	Boolean HaveStats;
	long    VolSize;
	FILE    *inFile;
	FILE    *ProcFile;
	float   *StatData;
	char    *Ovl;
	char    *EditedOvl;
	Boolean cluster;
	Boolean autowrite;
	short   timePt;
	float   thresh;
	
} UtilGlobal;
UtilGlobal u;

Boolean verbose = false;

/* PROTOTYPES */
OSErr OpenStatFile();
OSErr GetNewStatFile( void );
OSErr ReadOverlay( char *fname );
OSErr SelectAll( void );
OSErr exp_overlay( Boolean defaultname );
OSErr Export( void );
OSErr import( void );
OSErr UnionOvl( void );
OSErr AddOvl( void );
OSErr threshold( short sign );
OSErr Double( void );
OSErr Half( void );
OSErr npixMain( int argc, char *argv[], char *id );
OSErr MakeMenu( char *result );
OSErr DoResult( char result, Boolean *Done );
void  show_usage( char *progname );
void  Help( void );
void show_usage( char *progname )
{
	fprintf(stderr,"\nUsage: %s Namebase_of_bfloat_file\n\n"
	"Selects pixels based on threshod and write into default.ovl.\n"
	"Outputs fbloat file containing only selected pixels\n\n", progname);
}

/* doubles the size of the overlay file and saves to user-selected name */
OSErr Double( void )
{
	OSErr error = noErr;
	long  i;
	int   x, y, xs, ys;
	FILE  *ovlfile;
	char  fname[256];
	
	xs = u.inIm.dim.isoX;
	ys = u.inIm.dim.isoY;
	
	if( u.autowrite ) {
		sprintf( fname, "default.ovl" );
	} else {
		printf( "Enter file name for new overlay: " );
		scanf( "%s", fname );
	}

	error = ck_fopen( &ovlfile, fname, "w" );
	RETURNONERROR;

	for( y=0; y<ys; y++ ) {
		for( x=0; x<xs; x++ ) {
			if( u.EditedOvl[y*xs + x] ) {
				fprintf( ovlfile, "%d %d\t", x*2, y*2 );
				fprintf( ovlfile, "%d %d\t", x*2+1, y*2 );
				fprintf( ovlfile, "%d %d\t", x*2, y*2+1 );
				fprintf( ovlfile, "%d %d\t", x*2+1, y*2+1 );
				i+=4;
			}
		}
	} 
	
	error = ck_fclose( ovlfile );
	printf( "\tExported %ld locations to %s\n", i, fname );

	return error;
} 
	
/* import an overlay file and reduce its size */
OSErr Half( void )
{
	OSErr error = noErr;
	long  i, pixCount;
	int   x, y, xs, ys;
	FILE  *ovlfile;
	char  fname[256];
	
	xs = u.inIm.dim.isoX;
	ys = u.inIm.dim.isoY;

	error = vclr( u.Ovl, u.VolSize, T_CHAR );
	RETURNONERROR;

	error = vclr( u.EditedOvl, u.VolSize, T_CHAR );
	RETURNONERROR;

	printf( "Enter name of overlay file to reduce: " );
	scanf( "%s", fname );
	
	error = ck_fopen( &ovlfile, u.Ovlfname, "r" );
	RETURNONERROR;
	
	while( fscanf( ovlfile, "%d %d", &x, &y ) > 0 ) {
		y /= 2;
		x /= 2;
		u.Ovl[ u.inIm.dim.isoX * y + x ] = 1;
		u.EditedOvl[ u.inIm.dim.isoX * y + x ] = 1;
	}

	error = ck_fclose( ovlfile );

	if( u.autowrite ) {
		error = exp_overlay( true );
	}

	pixCount = 0;
	for( i=0; i<u.VolSize; i++ ) {
		if( u.Ovl[i] ) {
			pixCount++;
		}
	}
	printf( "\nOverlay has %ld locations", pixCount );

	return error;
}

/* Add a new overlay to the active set */
OSErr AddOvl( void )
{
	OSErr error = noErr;
	long i;
	long  pixCount = 0;
	FILE  *OvlFile;
	char  fname[256];
	char  *NewOvl;
	int   xs = u.inIm.dim.isoX;
	int   ys = u.inIm.dim.isoY;
	int   x, y;

if( verbose ) DBG( "AddOvl" );

	printf( "Enter name of overlay to add to current subselection: " );
	scanf( "%s",fname );
	error = ck_fopen( &OvlFile, fname, "r" );

	NewOvl = (char *)ck_calloc( xs * ys, sizeof( char ), "NewOvl" );

	while( fscanf( OvlFile, "%d %d", &x, &y ) > 0 ) {
		NewOvl[ u.inIm.dim.isoX * y + x ] = 1;
	}
	ck_fclose( OvlFile );
	
	for( i=0; i<u.VolSize; i++ ) {
		if( NewOvl[i] || u.EditedOvl[i] ) {
			u.Ovl[i] = u.EditedOvl[i] = 1;
			pixCount++;
		} else {
			u.Ovl[i] = u.EditedOvl[i] = 0;
		}
	}
	
	free( NewOvl );

	if( u.autowrite ) {
		error = exp_overlay( true );
	}

	printf( "New overlay has %ld locations\n", pixCount );
	return error;
}

/* Add a new overlay to the active set */
OSErr UnionOvl( void )
{
	OSErr error = noErr;
	long i;
	long  pixCount;
	FILE  *OvlFile;
	char  fname[256];
	char  *NewOvl;
	int   xs = u.inIm.dim.isoX;
	int   ys = u.inIm.dim.isoY;
	int   x, y;

if( verbose ) DBG( "UnionOvl" );

	printf( "Enter name of overlay to merge with current subselection: " );
	scanf( "%s",fname );
	error = ck_fopen( &OvlFile, fname, "r" );

	NewOvl = (char *)ck_calloc( xs * ys, sizeof( char ), "NewOvl" );

	while( fscanf( OvlFile, "%d %d", &x, &y ) > 0 ) {
		NewOvl[ u.inIm.dim.isoX * y + x ] = 1;
	}
	ck_fclose( OvlFile );
	
	for( i=0; i<u.VolSize; i++ ) {
		if( NewOvl[i] & u.EditedOvl[i] ) {
			u.Ovl[i] = u.EditedOvl[i] = 1;
			pixCount++;
		} else {
			u.Ovl[i] = u.EditedOvl[i] = 0;
		}
	}
	
	free( NewOvl );

	if( u.autowrite ) {
		error = exp_overlay( true );
	}

	printf( "New overlay has %ld locations\n", pixCount );
	return error;
}


/* enable all pixels in the overlay file for sub-selection */
OSErr SelectAll( void )
{
	OSErr error = noErr;
	long  i;
	
if( verbose ) DBG( "SelectAll" );

	for( i=0; i<u.VolSize; i++ ) {
		u.EditedOvl[i] = u.Ovl[i] = 1;
	}
	printf( "Selected %ld locations\n", i );
	return error;
} 

/* read in a named overlay file */
OSErr import( void )
{
	OSErr error = noErr;

	printf( "Name of overlay file to import: " );
	scanf( "%s", u.Ovlfname );
	
	if( !strlen( u.Ovlfname ) ) {
		sprintf( u.Ovlfname, "default.ovl" );
	}

	error = ReadOverlay( u.Ovlfname );
	return error;
} 

/* save a copy of the statistical image using with only suprathrshold pixels */
OSErr Export( void )
{
	OSErr error = noErr;
	FILE  *outFile;
	float zero = 0.0;
	long  i, count;
	char  fname[256];
	char  basename[256];
	IMAGE outIm;
	
if( verbose ) DBG( "export" );

	memcpy( &outIm, &u.inIm, sizeof( IMAGE ));	// copy the input header to the output header
	outIm.dim.timePts = 1;

	printf( "Enter export file name and extension (.bfloat or .img): " );
	scanf( "%s", fname );

	error = FindOutputType( fname, &outIm.file_type, &outIm.data_type );
	RETURNONERROR;

	strcpy( basename, fname );
	*strrchr(basename,'.') = '\0';	// terminate before extension

	error = CreateHeader( outIm.file_type, &outIm, basename );
	RETURNONERROR;
	
	error = ck_fopen( &outFile, fname, "wb" );
	RETURNONERROR;

	for( i=0, count=0; i<u.VolSize; i++ ) {
		if( u.EditedOvl[i] ) {
			count += fwrite( &u.StatData[i], sizeof( float ), 1, outFile );
		} else {
			count += fwrite( &zero, sizeof( float ), 1, outFile );
		}
	}

	error = ck_fclose( outFile );
	return error;
}
	
/* read in a new overlay file by name */
OSErr ReadOverlay( char *fname )
{
	OSErr error = noErr;
	FILE  *OvlFile;
	long  i = 0;
	int   x, y;

if( verbose ) DBG( "ReadOverlay" );

	error = ck_fopen( &OvlFile, u.Ovlfname, "r" );
	if( error ) {
		error = ck_fopen( &OvlFile, u.Ovlfname, "w" );
		RETURNONERROR;
		error = ck_fclose( OvlFile );
		RETURNONERROR;
		error = ck_fopen( &OvlFile, u.Ovlfname, "r" );
	}	
	
	error = vclr( u.Ovl, u.VolSize, T_CHAR );
	RETURNONERROR;

	while( fscanf( OvlFile, "%d %d", &x, &y ) > 0 ) {
		u.Ovl[ u.inIm.dim.isoX * y + x ] = 1;
		u.EditedOvl[ u.inIm.dim.isoX * y + x ] = 1;
		i++;
	}

	error = ck_fclose( OvlFile );
	printf( "\nRead %ld locations", i );

	return error;
}

/* select only locations above a user-specified threshold */
/* impose constraint that clusters must be at least 2 pixels in plane */
OSErr threshold( short sign )
{
	OSErr error = noErr;
	long i;
	long  pixCount;
	char  *mask;
	int   xs = u.inIm.dim.isoX;
	int   ys = u.inIm.dim.isoY;
	int   maskX;
	int   maskY;
	int   x, y;
	float average;
	
	maskX = xs+2;
	maskY = ys+2;

if( verbose ) DBG( "thresh" );

	printf( "Enter threshold: " );
	scanf( "%f", &u.thresh );
	printf( "\n" );

/* prepare EditedOvl with the thresholded data */
	for( i=0; i<u.VolSize; i++ ) {
		if( sign > 0 ) {
			u.EditedOvl[i] = u.Ovl[i] * (u.StatData[i] >= u.thresh);
		} else {
			u.EditedOvl[i] = u.Ovl[i] * (u.StatData[i] <= u.thresh);
		}
	}

	if( u.cluster ) {
/* mask is allocated for an extra two bytes to allow compilation on intel to respect memory boundaries */
		mask = (char *)ck_calloc( maskX * maskY + 2, sizeof( char ), "Mask" );
	
/* fill the mask locations with ovl file data */
		for( y=0; y<ys; y++ ) {
			memcpy( &mask[(y+1) * maskX + 1], &u.EditedOvl[y*xs], xs );
		}
		
		for( y=1; y<ys-1; y++ ) {
			for( x=1; x<xs-1; x++ ) {
				i = y*xs + x;
				if( mask[i] ) {
					if( mask[i-1] || mask[i+1] || mask[i-maskX] || mask[i+maskX] ) {
						mask[i] = 1;
					} else {
						mask[i] = 0;
					}
				}
			}
		}

		/* copy the mask back to u.EditedOvl */
		printf( "maskX: %d\n", maskX );
		printf( "ys: %d\n", ys );
		
		for( y=0; y<ys; y++ ) {
			memcpy( &u.EditedOvl[y*xs], &mask[(y+1) * maskX + 1], maskX );
			printf( "%d\n", y );
		}
		free( mask );
	}

	if( u.autowrite ) {
		error = exp_overlay( true );
	}

	pixCount = 0;
	average = 0;
	for( i=0; i<u.VolSize; i++ ) {
		if( u.EditedOvl[i] ) {
			average += u.StatData[i];
			pixCount++;
		}
	}

	if( sign > 0 ) {
		printf( "\t%d pixels greater than %0.3f\t", pixCount, u.thresh );
	} else {
		printf( "\t%d pixels less than %0.3f\t", pixCount, u.thresh );
	}
	printf( "Average = %0.3f\n", average/pixCount );
	return error;
}

/* export an overlay file for use by xds */
OSErr exp_overlay( Boolean defaultname )
{
	OSErr error = noErr;
	int  x, y, xs, ys;
	char fname[256];
	FILE *ovlfile;
	long i=0;
	
if( verbose ) DBG( "exp_overlay" );

	xs = u.inIm.dim.isoX;
	ys = u.inIm.dim.isoY;
	
	if( defaultname ) {
		sprintf( fname, "default.ovl" );
	} else {
		printf( "Overlay file name?: " );
		scanf( "%s", fname );
	}
	error = ck_fopen( &ovlfile, fname, "w" );
	RETURNONERROR;
	

	for( y=0; y<ys; y++ ) {
		for( x=0; x<xs; x++ ) {
			if( u.EditedOvl[y*xs + x] ) {
				fprintf( ovlfile, "%d %d\t", x, y );
				i++;
			}
		}
	} 
	
	error = ck_fclose( ovlfile );
	printf( "\tExported %ld locations to %s.\n", i, fname );

	return error;
} 

/* select a new statistical image */
OSErr GetNewStatFile( void )
{
	OSErr error = noErr;
	
if( verbose ) DBG( "GetNewStatFile" );

	printf( "Enter the name of a new stats file: " );
	scanf( "%s", u.InFileName );

	error = OpenStatFile();
	return error;
}

/* read in the contents of a new statistical image
   always reads from u.InFileName */
OSErr OpenStatFile()
{
	OSErr   error = noErr;
	long    inVolSize;
	
	if( u.HaveStats ) {
		free( u.StatData );
		free( u.Ovl );
	}

	error = UC_Readheader( u.InFileName, &u.inIm );
	RETURNONERROR;
	
	if( u.inIm.data_type != T_FLOAT ) {
		printf( "This program only works with floating point images.\n" );
		return 0;
	}

	u.VolSize  = u.inIm.dim.isoX * u.inIm.dim.isoY * u.inIm.dim.n_slices;
	inVolSize  = u.VolSize * get_datasize( u.inIm.data_type );
	u.StatData = (float *)ck_malloc( inVolSize,  "Storage for input data" );
	u.Ovl      = (char *)ck_calloc( u.VolSize, sizeof( char ), "Overlay mask" );

	if( !u.HaveStats ) {
/* u.EditedOvl is made bigger to compensated for the intel memory allocation */
		u.EditedOvl= (char *)ck_calloc( u.VolSize + 2, sizeof( char ), "Edited Overlay mask" );
	}

	error = ck_fopen( &u.inFile, u.InFileName,  "rb" );
	RETURNONERROR;

	u.inIm.data_type = T_FLOAT;
	error = GetSelectedVolume( u.inFile, &u.inIm, (void *)u.StatData, u.timePt );
	RETURNONERROR;

	error = ck_fclose( u.inFile );
	RETURNONERROR;
	
	u.HaveStats = true;
	return error;
}
	
OSErr MakeMenu( char *result )
{
	OSErr error = noErr;
	char  input[3];
	char  status[4];
	
	printf( "\n\nOPTIONS:\n" );
	printf( "\tt:  Select t-thresholded pixels [ %0.3f ]\n", u.thresh );
	printf( "\tT:  Select (-t)-thresholded pixels\n" );
	printf( "\tf:  New Data File [ %s ]\n", u.InFileName );
	printf( "\tA:  Select entire image\n" );
	printf( "\tE:  Export pix-thresholded bfloat file\n" );
	printf( "\ts:  Save default.ovl\n" );
	printf( "\tI:  Import default.ovl\n" );
	printf( "\te:  Export overlay\n" );
	printf( "\ti:  Import overlay by name [ %s ]\n", u.Ovlfname );
	printf( "\t#:  Select a different image number [ %d ]\n", u.timePt ); 
	printf( "\t+:  Add new overlay to subselection\n" );
	printf( "\t&:  Import overlay and select conjunction\n" );
	printf( "\tZ:  Double overlay matrix size\n" );
	printf( "\tz:  Half overlay matrix size\n" );
	printf( "    ======== modes =======\n" );
	if( u.cluster ) {
		sprintf( status, "ON" );
	} else {
		sprintf( status, "OFF" );
	}
	printf( "\tC:  Toggle cluster status [ %s ]\n", status );
	if( u.autowrite ) {
		sprintf( status, "YES" );
	} else {
		sprintf( status, "NO" );
	}
	printf( "\tU:  Update default.ovl after each change [ %s ]\n\n", status );
	printf( "\t?:  A little help, please\n" );
	printf( "\tq:  Quit\n\n" );
	printf( "Selection: " );
	
	scanf( "%s", input );
	*result = input[0];

	return error;
}

OSErr DoResult( char result, Boolean *Done )
{
	OSErr error = noErr;

	switch( result )
	{
		case 't':
			error = threshold( 1 );
		break;
		
		case 'T':
			error = threshold( -1 );
		break;
		
		case 'f':
			error = GetNewStatFile();
		break;

		case 'i':
			error = import();
		break;

		case 'I':
			sprintf( u.Ovlfname, "default.ovl" );
			error = ReadOverlay( u.Ovlfname );
		break;

		case 'E':
			error = Export();
		break;

		case 'A':
			error = SelectAll();
		break;

		case 'e':
			error = exp_overlay( false );	// save with new file name
		break;

		case 's':
			error = exp_overlay( true );	// save as default.ovl
		break;

		case '#':
			printf( "Enter image number: " );
			scanf( "%d", &u.timePt );
		break;

		case 'Z':
			error = Double();
		break;

		case 'z':
			error = Half();
		break;

		case 'C':
			u.cluster = !u.cluster;
		break;

		case '+':
			error = AddOvl();
		break;
		
		case '&':
			error = UnionOvl();
		break;
		
		case 'U':
			u.autowrite = !u.autowrite;
		break;

		case '?':
			Help();
		break;

		case 'q':
		case 'Q':
			*Done = true;
		break;

		default:
			printf( "Unknown option" );
	}
	return error;
}

void Help( void )
{
	printf( "\n%s is used to select subregions of images with intensities above a chosen threshold\n", u.progname );
	printf( "Most operations are intended for use with 'xds' and communicate by way of the file, default.ovl\n"
	"though you may choose to use a different file name.\n\n"
	"Typically, you will load a statistical image, such as those that are craeted by CC_gr\n"
	"You will then either select all locations ('A') or use default.ovl created by xds to choose a region.\n"
	"With %s, you will then use 't' or 'T' to select all pixels from the statistical file\n", u.progname );
	printf( "with intensities greater than, or less than (respectively) a threshold you choose.\n\n"
	"Sometimes, you will want to merge more than one selection choosing, for example, locations that\n"
	"exceed a threshold on multiple statistical images.\n"
	"To do so, you will select a region on a first image, then import a new statistical image ('f') and\n"
	"re-threshold.\n\n"
	"When clustering is turned on, the program will only accept locations with at least one suprathreshold\n"
	"nearest neighbor.\n\n"
	"When 'Update default.ovl' is enabled, the file 'default.ovl' will be rewritten with each change.\n\n"
	"The 'Z' and 'z' options are provided as utility functions to allow you to change the size of the overlay\n"
	"overlay to correspond to different image sizes.\n\n"
	"The 'E' option creates an image with the statistical values of only the suprathreshold locations.\n"
	"This can be used in programs such as 'overlayMac' to create colorized overlays.\n\n"
	"If you have bugs, desired features or lots of money to send to me, I can be reached at mscohen@ucla.edu.\n" );
}

OSErr npixMain( int argc, char *argv[], char *id )
{
	OSErr error = noErr;
	Boolean Done = false;
	char  result;

	u.HaveStats = false;
	u.cluster = true;
	u.autowrite = true;
	u.timePt = 0;
	u.thresh = 0;
	strcpy( u.Ovlfname, "default.ovl" );

	if( argc == 1 ) {
		error = GetNewStatFile();
		RETURNONERROR;
	} else if( argc != 2 ) {
		show_usage( argv[0] );
		return 0;
	} else {
		strcpy( u.InFileName, argv[1] );
	}
	RETURNONERROR;

	error = OpenStatFile();
	RETURNONERROR;
	
	sprintf( u.Ovlfname, "default.ovl" );
	error = ReadOverlay( u.Ovlfname );
	RETURNONERROR;
	
	while( !Done ) {
		error = MakeMenu( &result );
		RETURNONERROR;
		
		error = DoResult( result, &Done );
		RETURNONERROR;
	}

	free( u.StatData );
	free( u.Ovl );
	free( u.EditedOvl );
	return error;
}

int main( int argc, char *argv[] )
{
	OSErr error = noErr;
	char  buffer[128], versionStr[128];
	static  char id[] = "$Revision: 1.5 $$Date: 2006/09/26 04:59:59 $";
	
	sprintf( u.progname, argv[0] );
	sscanf( id, "%s%s%s%s", buffer, versionStr, buffer, buffer );
	strcat( versionStr, " " );
	strcat( versionStr, buffer );

	printf( "\n%s version ....... %s\n", argv[0], versionStr );
	GetVLibVer( buffer );
	printf( "VLib version ....... %s\n", buffer );
	GetImgLibVer( buffer );
	printf( "ImgLib version ..... %s\n", buffer );
	error = npixMain( argc, argv, id );
		ILError( error, "Running main program" );

	return 0;
}
