/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <math.h>
#include <gtk/gtkgl.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "gettext.h"
#include "quaternion.h"
#include "visualization.h"

#define _(String) gettext(String)

/*
 * Handler for mouse clicks.  First argument is visualization widget, second
 * argument is event structure, third argument is unused.
 */
static gboolean on_button_press(GtkWidget * widget, GdkEventButton * event, gpointer);

/*
 * Handler for mouse motion.  First argument is visualization widget, second
 * argument is event structure, third argument is unused.
 */
static gboolean on_motion_notify(GtkWidget * widget, GdkEventMotion * event, gpointer);

/*
 * Handler for expose event.  First argument is visualization widget, second
 * argument is event structure, third argument is unused.
 */
static gboolean on_expose(GtkWidget * widget, GdkEventExpose * event, gpointer);

/*
 * Handler for destroying save dialog.  First argument is pointer to
 * visualization widget, second argument is unused.
 */
static void     on_save_destroy(GtkWidget * widget, gpointer);

GtkWidget      *
visualization_create()
{
	GdkGLConfig    *config;	/* OpenGL frame buffer configuration object. */
	GtkWidget      *visualization;	/* Visualization widget. */

	/* Create OpenGL frame buffer configuration object. */
	config = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGBA | GDK_GL_MODE_DEPTH | GDK_GL_MODE_DOUBLE);
	assert(config != NULL);

	/* Create and setup visualization widget. */
	visualization = gtk_drawing_area_new();
	gtk_widget_set_size_request(visualization, 400, 400);
	gtk_widget_set_events(visualization, gtk_widget_get_events(visualization) | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK | GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);
	gtk_widget_set_gl_capability(visualization, config, NULL, TRUE, GDK_GL_RGBA_TYPE);

	/* Register handlers for visualization widget signals. */
	g_signal_connect(G_OBJECT(visualization), "button_press_event", G_CALLBACK(on_button_press), NULL);
	g_signal_connect(G_OBJECT(visualization), "motion_notify_event", G_CALLBACK(on_motion_notify), NULL);
	g_signal_connect(G_OBJECT(visualization), "expose_event", G_CALLBACK(on_expose), NULL);

	return visualization;
}

void
on_visualization_save(GtkWidget * widget, gpointer data)
{
	GtkWidget      *save;	/* Save dialog widget. */
	char           *name;	/* File name. */
	Scene          *scene;	/* Scene object. */
	GdkGLConfig    *config;	/* OpenGL frame buffer configuration object. */
	GdkPixmap      *pixmap;	/* Pixmap object to render scene to. */
	GdkGLDrawable  *drawable;	/* Pixmap drawable. */
	GdkGLContext   *context;/* OpenGL rendering context. */
	GdkPixbuf      *pixbuf;	/* Pixels buffer. */

	/*
	 * If save dialog widget not already exists, then create and remember
	 * one and register its destroy handler.
	 */
	save = (GtkWidget *)
		g_object_get_data(G_OBJECT(widget), "save");
	if (save == NULL) {
		save = gtk_file_selection_new(_("Save current map in PNG format"));
		g_object_set_data(G_OBJECT(widget), "save", save);
		g_signal_connect_swapped(G_OBJECT(save), "destroy", G_CALLBACK(on_save_destroy), widget);
	}
	/* Show save dialog,... */
	if (gtk_dialog_run(GTK_DIALOG(save)) == GTK_RESPONSE_OK) {
		/* ...get file name from user,... */
		name = (char *)
			gtk_file_selection_get_filename(GTK_FILE_SELECTION(save));

		/* ...get scene object,... */
		scene = (Scene *) g_object_get_data(G_OBJECT(widget), "scene");
		assert(scene != NULL);

		/* ...create OpenGL frame buffer configuration object... */
		config = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGBA | GDK_GL_MODE_DEPTH | GDK_GL_MODE_SINGLE);
		assert(config != NULL);

		/* ...and pixmap object,... */
		pixmap = gdk_pixmap_new(widget->window, widget->allocation.width, widget->allocation.height, gdk_gl_config_get_depth(config));
		gdk_pixmap_set_gl_capability(pixmap, config, NULL);

		/* ...render scene to pixmap,... */
		drawable = gdk_pixmap_get_gl_drawable(pixmap);
		context = gdk_gl_context_new(drawable, NULL, FALSE, GDK_GL_RGBA_TYPE);
		gdk_gl_drawable_gl_begin(drawable, context);
		scene_render(scene, widget->allocation.width, widget->allocation.height);
		gdk_gl_drawable_swap_buffers(drawable);
		gdk_gl_drawable_gl_end(drawable);

		/* ...save pixmap pixels to file... */
		pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, widget->allocation.width, widget->allocation.height);
		gdk_pixbuf_get_from_drawable(pixbuf, pixmap, gdk_colormap_get_system(), 0, 0, 0, 0, widget->allocation.width, widget->allocation.height);
		gdk_pixbuf_save(pixbuf, name, "png", NULL, NULL);

		/* ...and destroy used objects. */
		g_object_unref(G_OBJECT(config));
		g_object_unref(G_OBJECT(pixmap));
		g_object_unref(G_OBJECT(context));
		g_object_unref(G_OBJECT(pixbuf));
	}
	/* Hide save dialog. */
	gtk_widget_hide(save);
}

static          gboolean
on_button_press(GtkWidget * widget, GdkEventButton * event, gpointer data)
{
	/* Remember current mouse pointer position. */
	g_object_set_data(G_OBJECT(widget), "mouse_x", GINT_TO_POINTER((int)event->x));
	g_object_set_data(G_OBJECT(widget), "mouse_y", GINT_TO_POINTER((int)event->y));

	return TRUE;
}

static          gboolean
on_motion_notify(GtkWidget * widget, GdkEventMotion * event, gpointer data)
{
	int             x, y;	/* Current mouse pointer position. */
	int             mouse_x, mouse_y;	/* Previous mouse pointer
						 * position. */
	GdkModifierType state;	/* Mouse buttons state. */
	Scene          *scene;	/* Scene object. */
	int             delta_x, delta_y;	/* Mouse pointer position
						 * increments. */
	Quaternion      q = {0, 0, 0, 1}, r;	/* Quaternions used to
						 * calculate new value of
						 * cumulative scene rotation. */
	Vector          axis;	/* Rotation asix. */

	/* Get current mouse pointer position and mouse buttons state. */
	gdk_window_get_pointer(widget->window, &x, &y, &state);

	/* Handle only mouse motion while right mouse button pressed. */
	if (!(state & GDK_BUTTON3_MASK))
		return FALSE;

	/* Get scene object. */
	scene = (Scene *) g_object_get_data(G_OBJECT(widget), "scene");
	assert(scene != NULL);

	/*
	 * Get previous mouse pointer position and calculate mouse pointer
	 * position increments.  Remember current mouse pointer position.
	 */
	mouse_x = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "mouse_x"));
	mouse_y = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "mouse_y"));
	delta_x = x - mouse_x;
	delta_y = y - mouse_y;
	g_object_set_data(G_OBJECT(widget), "mouse_x", GINT_TO_POINTER(x));
	g_object_set_data(G_OBJECT(widget), "mouse_y", GINT_TO_POINTER(y));

	/*
	 * Calculate rotations around eye x- and y-axis that are proportional
	 * to mouse pointer position increments around window y- and x-axis.
	 * Add these rotations to cumulative scene rotation.
	 */
	axis[X] = 0, axis[Y] = 1, axis[Z] = 0;
	quaternion_from_axis_angle(axis, M_PI * delta_x / widget->allocation.width, r);
	quaternion_mult(q, r);
	axis[X] = 1, axis[Y] = axis[Z] = 0;
	quaternion_from_axis_angle(axis, M_PI * delta_y / widget->allocation.height, r);
	quaternion_mult(q, r);
	quaternion_from_axis_angle(scene->axis, scene->angle, r);
	quaternion_mult(q, r);
	quaternion_to_axis_angle(q, scene->axis, &scene->angle);

	/* Force reapaint of visualization window. */
	gdk_window_invalidate_rect(widget->window, NULL, FALSE);

	return TRUE;
}

static          gboolean
on_expose(GtkWidget * widget, GdkEventExpose * event, gpointer data)
{
	Scene          *scene;	/* Scene object. */
	GdkGLDrawable  *drawable;	/* Visualization window drawable. */
	GdkGLContext   *context;/* OpenGL rendering context. */

	/* Get scene object. */
	scene = (Scene *) g_object_get_data(G_OBJECT(widget), "scene");
	assert(scene != NULL);

	/* Render scene to visualization window. */
	drawable = gtk_widget_get_gl_drawable(widget);
	context = gtk_widget_get_gl_context(widget);
	gdk_gl_drawable_gl_begin(drawable, context);
	scene_render(scene, widget->allocation.width, widget->allocation.height);
	gdk_gl_drawable_swap_buffers(drawable);
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static void
on_save_destroy(GtkWidget * widget, gpointer data)
{
	/* Forget about save dialog widget. */
	g_object_set_data(G_OBJECT(widget), "save", NULL);
}
