/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "signal.h"

Signal         *
signal_create()
{
	Signal         *signal;	/* Signal object. */

	/* Create signal object. */
	signal = (Signal *) malloc(sizeof(Signal));
	assert(signal != NULL);

	/* Initialize new signal object. */
	signal->file = NULL;
	signal->label = NULL;
	signal->transducer = NULL;
	signal->prefiltering = NULL;

	return signal;
}

void
signal_destroy(Signal * signal)
{
	/* If necessary, close input file. */
	if (signal->file != NULL)
		fclose(signal->file);

	/*
	 * Free memory used for signal name, transducer type and prefiltering
	 * indicator, as well as for signal object itself.
	 */
	free(signal->label);
	free(signal->transducer);
	free(signal->prefiltering);
	free(signal);
}

void
signal_load_edf(Signal * signal, char *name, int number)
{
	char            buffer[81];	/* Input buffer. */
	int             count;	/* Number of signals in input file. */
	int             offset;	/* Number of bytes in input file header. */
	int             result;	/* Result returned by input functions. */
	int             i;	/* Loop index. */

	/* Open input file. */
	if (signal->file != NULL)
		fclose(signal->file);
	signal->file = fopen(name, "r");
	assert(signal->file != NULL);

	/* Read number of signals in input file. */
	fseek(signal->file, 252, SEEK_SET);
	result = fread(buffer, 4, 1, signal->file);
	assert(result == 1);
	buffer[4] = 0;
	count = atoi(buffer);

	/* Read signal name. */
	fseek(signal->file, 256 + number * 16, SEEK_SET);
	result = fread(buffer, 16, 1, signal->file);
	assert(result == 1);
	buffer[16] = 0;
	for (i = 15; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	signal->label = (char *)malloc((i + 2) * sizeof(char));
	assert(signal->label != NULL);
	strcpy(signal->label, buffer);

	/* Read signal transducer type. */
	fseek(signal->file, 256 + count * 16 + number * 80, SEEK_SET);
	result = fread(buffer, 80, 1, signal->file);
	assert(result == 1);
	buffer[80] = 0;
	for (i = 79; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	signal->transducer = (char *)malloc((i + 2) * sizeof(char));
	assert(signal->transducer != NULL);
	strcpy(signal->transducer, buffer);

	/* Read and calculate signal unit (must be in volts). */
	fseek(signal->file, 256 + count * 96 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	for (i = 7; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	switch (i) {
	case 0:
		assert(buffer[i] == 'V');
		signal->unit = 1e6;
		break;

	case 1:
		assert(buffer[i] == 'V');
		switch (buffer[i - 1]) {
		case 'Y':
			signal->unit = 1e30;
			break;

		case 'Z':
			signal->unit = 1e27;
			break;

		case 'E':
			signal->unit = 1e24;
			break;

		case 'P':
			signal->unit = 1e21;
			break;

		case 'T':
			signal->unit = 1e18;
			break;

		case 'G':
			signal->unit = 1e15;
			break;

		case 'M':
			signal->unit = 1e12;
			break;

		case 'K':
			signal->unit = 1e9;
			break;

		case 'H':
			signal->unit = 1e8;
			break;

		case 'D':
			signal->unit = 1e7;
			break;

		case 'd':
			signal->unit = 1e5;
			break;

		case 'c':
			signal->unit = 1e4;
			break;

		case 'm':
			signal->unit = 1e3;
			break;

		case 'u':
			signal->unit = 1;
			break;

		case 'n':
			signal->unit = 1e-3;
			break;

		case 'p':
			signal->unit = 1e-6;
			break;

		case 'f':
			signal->unit = 1e-9;
			break;

		case 'a':
			signal->unit = 1e-12;
			break;

		case 'z':
			signal->unit = 1e-15;
			break;

		case 'y':
			signal->unit = 1e-18;
			break;

		default:
			assert(0);
		}
		break;

	default:
		assert(0);
	}

	/* Read physical minimum and maximum. */
	fseek(signal->file, 256 + count * 104 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	signal->physical_lo = atof(buffer);
	fseek(signal->file, 256 + count * 112 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	signal->physical_hi = atof(buffer);

	/* Read digital minimum and maximum. */
	fseek(signal->file, 256 + count * 120 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	signal->digital_lo = atoi(buffer);
	fseek(signal->file, 256 + count * 128 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	signal->digital_hi = atoi(buffer);

	/* Read prefiltering indicator. */
	fseek(signal->file, 256 + count * 136 + number * 80, SEEK_SET);
	result = fread(buffer, 80, 1, signal->file);
	assert(result == 1);
	buffer[80] = 0;
	for (i = 79; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	signal->prefiltering = (char *)malloc((i + 2) * sizeof(char));
	assert(signal->prefiltering != NULL);
	strcpy(signal->prefiltering, buffer);

	/* Read number of samples in each data record. */
	fseek(signal->file, 256 + count * 216 + number * 8, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	signal->count = atoi(buffer);

	/* Read reserved bytes. */
	fseek(signal->file, 256 + count * 224 + number * 32, SEEK_SET);
	result = fread(buffer, 32, 1, signal->file);
	assert(result == 1);

	/*
	 * Calculate coefficients for conversion from digital to physical
	 * signal value.
	 */
	signal->c1 = (signal->physical_hi - signal->physical_lo) * signal->unit / (signal->digital_hi - signal->digital_lo);
	signal->c0 = signal->physical_lo * signal->unit - signal->digital_lo * signal->c1;

	/* Read number of bytes in input file header. */
	fseek(signal->file, 184, SEEK_SET);
	result = fread(buffer, 8, 1, signal->file);
	assert(result == 1);
	buffer[8] = 0;
	offset = atoi(buffer);

	/*
	 * Calculate offset of first signal sample in input file and number
	 * of bytes to skip between two successive signal data records.
	 */
	signal->offset = offset;
	signal->skip = 0;
	for (i = 0; i < number; i++) {
		fseek(signal->file, 256 + count * 216 + i * 8, SEEK_SET);
		result = fread(buffer, 8, 1, signal->file);
		assert(result == 1);
		buffer[8] = 0;
		signal->offset += 2 * atoi(buffer);
		signal->skip += 2 * atoi(buffer);
	}
	signal->skip += 2 * signal->count;
	for (i = number + 1; i < count; i++) {
		fseek(signal->file, 256 + count * 216 + i * 8, SEEK_SET);
		result = fread(buffer, 8, 1, signal->file);
		assert(result == 1);
		buffer[8] = 0;
		signal->skip += 2 * atoi(buffer);
	}
}

void
signal_values(Signal * signal, int from, int count, float *values)
{
	div_t           offset;	/* Number of data record and offset in this
				 * record of current signal value. */
	char            buffer[2];	/* Signal value, as read from input
					 * file. */
	int             i;	/* Loop index. */

	/*
	 * Calculate number of data record and offset in this record of first
	 * requested signal value.
	 */
	offset = div(from, signal->count);
	fseek(signal->file, signal->offset + offset.quot * signal->skip + 2 * offset.rem, SEEK_SET);

	/* Read signal values. */
	for (i = 0; i < count; i++) {
		/*
		 * Read raw value from input file (values are written in file
		 * in little-endian format) and calculate corresponding
		 * signal physical value.
		 */
		fread(buffer, 2, 1, signal->file);
		values[i] = signal->c0 + signal->c1 * ((buffer[1] << 8) | buffer[0]);

		/*
		 * Calculate number of data record and offset in this record
		 * of next signal value.
		 */
		offset.rem++;
		if (offset.rem == signal->count) {
			fseek(signal->file, signal->skip - 2 * signal->count, SEEK_CUR);
			offset.rem = 0;
		}
	}
}
