/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "scene.h"

Scene          *
scene_create(char *mapped, char *not_mapped)
{
	Scene          *scene;	/* Scene object. */

	/* Create scene object. */
	scene = (Scene *) malloc(sizeof(Scene));
	assert(scene != NULL);

	/*
	 * Load mapped and not mapped objects from corresponding files and
	 * create copy of mapped object.
	 */
	scene->mapped = object_create(mapped);
	scene->copy = object_copy(scene->mapped);
	scene->not_mapped = object_create(not_mapped);

	/* Initialize cumulative rotation axis and angle. */
	scene->axis[X] = scene->axis[Y] = 0, scene->axis[Z] = 1;
	scene->angle = M_PI;

	return scene;
}

void
scene_destroy(Scene * scene)
{
	/* Destroy mapped object and its copy, as well as not mapped object. */
	object_destroy(scene->mapped);
	object_destroy(scene->copy);
	object_destroy(scene->not_mapped);

	/* Free memory used by scene object. */
	free(scene);
}

void
scene_render(Scene * scene, int width, int height)
{
	Vector          lo, hi;	/* Lower and upper corners of scene bounding
				 * box. */
	float           angle;	/* Field of view angle. */
	float           aspect;	/* View frustum aspect ratio. */
	Vector          from, to, up;	/* Viewer position and orientation
					 * vectors. */
	GLfloat         light_ambient[] = {0.8, 0.8, 0.8, 1};	/* Ligh ambient color. */
	GLfloat         light_diffuse[] = {0.8, 0.8, 0.8, 1};	/* Ligh diffuse color. */
	GLfloat         light_specular[4] = {0.8, 0.8, 0.8, 1};	/* Ligh specular color. */
	GLfloat         light_position[] = {0, 0, 0, 1};	/* Light position. */

	/* Calculate scene bounding box. */
	lo[X] = (scene->mapped->lo[X] < scene->not_mapped->lo[X]) ? scene->mapped->lo[X] : scene->not_mapped->lo[X];
	lo[Y] = (scene->mapped->lo[Y] < scene->not_mapped->lo[Y]) ? scene->mapped->lo[Y] : scene->not_mapped->lo[Y];
	lo[Z] = (scene->mapped->lo[Z] < scene->not_mapped->lo[Z]) ? scene->mapped->lo[Z] : scene->not_mapped->lo[Z];
	hi[X] = (scene->mapped->hi[X] > scene->not_mapped->hi[X]) ? scene->mapped->hi[X] : scene->not_mapped->hi[X];
	hi[Y] = (scene->mapped->hi[Y] > scene->not_mapped->hi[Y]) ? scene->mapped->hi[Y] : scene->not_mapped->hi[Y];
	hi[Z] = (scene->mapped->hi[Z] > scene->not_mapped->hi[Z]) ? scene->mapped->hi[Z] : scene->not_mapped->hi[Z];

	/*
	 * Calculate viewer position and orientation such that all parts of
	 * scene are visible.
	 */
	angle = M_PI / 12;
	aspect = (float)width / height;
	from[X] = from[Y] = 0, from[Z] = (hi[X] - lo[X]) > (hi[Y] - lo[Y]) ? (hi[X] - lo[X]) / 2 / tan(angle / 2 * ((aspect < 1) ? aspect : 1)) + (hi[Z] - lo[Z]) : (hi[Y] - lo[Y]) / 2 / tan(angle / 2 * ((aspect < 1) ? aspect : 1)) + (hi[Z] - lo[Z]);
	to[X] = to[Y] = to[Z] = 0;
	up[X] = 0, up[Y] = 1, up[Z] = 0;

	/* Initialize color and depth buffers. */
	glEnable(GL_DEPTH_TEST);
	glClearColor(0, 0, 0, 0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* Setup viewport and projection. */
	glViewport(0, 0, width, height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(angle * 180 / M_PI, aspect, from[Z] / 5, 5 * from[Z]);

	/* Initialize modelview matrix. */
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	/* Setup light as headlight with given parameters. */
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
	glLightfv(GL_LIGHT0, GL_POSITION, light_position);

	/* Apply viewing parameters. */
	gluLookAt(from[X], from[Y], from[Z], to[X], to[Y], to[Z], up[X], up[Y], up[Z]);

	/* Apply cumulative rotation. */
	glRotatef(scene->angle * 180 / M_PI, scene->axis[X], scene->axis[Y], scene->axis[Z]);

	/*
	 * Setup material diffuse reflection coefficients to track current
	 * color.
	 */
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);

	/* Render scene objects. */
	object_render(scene->mapped);
	object_render(scene->not_mapped);
}
