/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <stdlib.h>
#include "gettext.h"
#include "gradient.h"
#include "interpolation.h"
#include "legend_controls.h"

#define _(String) gettext(String)

/*
 * Handler for current low boundary spin button change.  First argument is
 * main application widget, second argument is unused.
 */
static void     on_lo_value_changed(GtkWidget * widget, gpointer data);

/*
 * Handler for current high boundary spin button change.  First argument is
 * main application widget, second argument is unused.
 */
static void     on_hi_value_changed(GtkWidget * widget, gpointer data);

GtkWidget      *
legend_controls_create(GtkWidget * window)
{
	Interpolation  *interpolation;	/* Interpolation object. */
	GtkTooltips    *tooltips;	/* Tooltips object. */
	GtkWidget      *frame;	/* Frame containing time controls. */
	GtkWidget      *eventbox;	/* Event box widget. */
	GtkWidget      *hbox;	/* Horizontal box for widgets packing. */
	GtkWidget      *spin;	/* Spin widget. */
	GtkWidget      *label;	/* Label widget. */
	GtkWidget      *gradient;	/* Gradient widget. */
	float          *lo, *hi, *min, *max;	/* Arrays with current and
						 * widest boundaries for
						 * interpolation of score
						 * values. */

	/* Get interpolation object. */
	interpolation = (Interpolation *) g_object_get_data(G_OBJECT(window), "interpolation");
	assert(interpolation != NULL);

	/* Get tooltips object. */
	tooltips = GTK_TOOLTIPS(g_object_get_data(G_OBJECT(window), "tooltips"));

	/* Create frame widget for legend controls layout. */
	frame = gtk_frame_new(_("Legend"));

	/* Create and setup horizontal box for widgets packing. */
	hbox = gtk_hbox_new(FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
	gtk_container_add(GTK_CONTAINER(frame), hbox);

	/*
	 * Get arrays with current and widest boundaries for interpolation of
	 * score values.
	 */
	lo = (float *)g_object_get_data(G_OBJECT(window), "lo");
	assert(lo != NULL);
	hi = (float *)g_object_get_data(G_OBJECT(window), "hi");
	assert(hi != NULL);
	min = (float *)g_object_get_data(G_OBJECT(window), "min");
	assert(min != NULL);
	max = (float *)g_object_get_data(G_OBJECT(window), "max");
	assert(max != NULL);

	/* Create and setup current low boundary label widget. */
	label = gtk_label_new(_("Low"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/* Create and setup current low boundary spin widget. */
	spin = gtk_spin_button_new_with_range(min[0], hi[0], (max[0] - min[0]) / 100);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), spin, _("Score value mapped to legend leftmost color"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), lo[0]);
	gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 2);
	g_object_set_data(G_OBJECT(window), "lo_spin", spin);
	g_signal_connect_swapped(G_OBJECT(spin), "value_changed", G_CALLBACK(on_lo_value_changed), G_OBJECT(window));

	/* Create and setup gradient widget. */
	eventbox = gtk_event_box_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), eventbox, _("Color map"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), eventbox, TRUE, TRUE, 0);
	gradient = gradient_create(interpolation);
	gtk_widget_set_size_request(gradient, 80, 20);
	gtk_container_add(GTK_CONTAINER(eventbox), gradient);

	/* Create and setup current high boundary label widget. */
	label = gtk_label_new(_("High"));
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/* Create and setup current high boundary spin widget. */
	spin = gtk_spin_button_new_with_range(lo[0], max[0], (max[0] - min[0]) / 100);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), spin, _("Score value mapped to legend rightmost color"), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), hi[0]);
	gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 2);
	g_object_set_data(G_OBJECT(window), "hi_spin", spin);
	g_signal_connect_swapped(G_OBJECT(spin), "value_changed", G_CALLBACK(on_hi_value_changed), G_OBJECT(window));

	/*
	 * Initialize and remember flag that will control if handlers for
	 * current low and high boundaries enabled.  When current score
	 * changed, ranges and values for current low and high boundary spin
	 * buttons has to be adjusted.  There exist no atomic operation to
	 * accomplish this and range change could provoke undesirable value
	 * change.  In order to avoid this, value change handlers should be
	 * disabled during range change.
	 */
	g_object_set_data(G_OBJECT(window), "change_flag", GINT_TO_POINTER(1));

	return frame;
}

static void
on_lo_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *spin;	/* Spin widget. */
	float          *lo_curr;/* Current low boundary. */
	double          lo, hi;	/* High boundary spin range values. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* If handler disabled, return immediately. */
	if (!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "change_flag")))
		return;

	/* Get spin widget. */
	spin = g_object_get_data(G_OBJECT(widget), "lo_spin");
	assert(spin != NULL);

	/* Adjust current low boundary according to spin changed value. */
	lo_curr = (float *)g_object_get_data(G_OBJECT(widget), "lo_curr");
	assert(lo_curr != NULL);
	*lo_curr = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));

	/*
	 * Adjust high boundary spin range according to new low boundary
	 * value.
	 */
	spin = g_object_get_data(G_OBJECT(widget), "hi_spin");
	assert(spin != NULL);
	gtk_spin_button_get_range(GTK_SPIN_BUTTON(spin), &lo, &hi);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), *lo_curr, hi);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_hi_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *spin;	/* Spin widget. */
	float          *hi_curr;/* Current high boundary. */
	double          lo, hi;	/* Low boundary spin range values. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* If handler disabled, return immediately. */
	if (!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "change_flag")))
		return;

	/* Get spin widget. */
	spin = g_object_get_data(G_OBJECT(widget), "hi_spin");
	assert(spin != NULL);

	/* Adjust current high boundary according to spin changed value. */
	hi_curr = (float *)g_object_get_data(G_OBJECT(widget), "hi_curr");
	assert(hi_curr != NULL);
	*hi_curr = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));

	/*
	 * Adjust low boundary spin range according to new low boundary
	 * value.
	 */
	spin = g_object_get_data(G_OBJECT(widget), "lo_spin");
	assert(spin != NULL);
	gtk_spin_button_get_range(GTK_SPIN_BUTTON(spin), &lo, &hi);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), lo, *hi_curr);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}
