/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <float.h>
#include <stdlib.h>
#include <string.h>
#include "interpolation.h"

Interpolation  *
interpolation_create(Object * object, Input * input, Sensors * sensors)
{
	Interpolation  *interpolation;	/* Interpolation object. */
	Vector          v;	/* Vector between current vertex and current
				 * sensor. */
	float           distance;	/* Length of above vector. */
	float           den, mul;	/* Sum of inverse values of distances
					 * between current vertex and its
					 * nearest sensors and its inverse
					 * value. */
	int             i, j, k;/* Loop indices. */

	/* Create and initialize interpolation object. */
	interpolation = (Interpolation *) malloc(sizeof(Interpolation));
	assert(interpolation != NULL);
	interpolation->count = object->vertex_count;
	interpolation->indices = (int *)malloc(MIN_COUNT * interpolation->count * sizeof(int));
	assert(interpolation->indices != NULL);
	interpolation->coefficients = (float *)malloc(MIN_COUNT * interpolation->count * sizeof(float));
	assert(interpolation->coefficients != NULL);

	/* For each model vertex... */
	for (i = 0; i < object->vertex_count; i++) {
		/*
		 * ...initialize indices of nearest sensors and interpolation
		 * coefficients,...
		 */
		for (j = 0; j < MIN_COUNT; j++) {
			interpolation->indices[i * MIN_COUNT + j] = -1;
			interpolation->coefficients[i * MIN_COUNT + j] = FLT_MAX;
		}

		/* ...then for each signal from input file... */
		for (j = 0; j < input->count; j++) {
			/*
			 * ...calculate distance between vertex and
			 * corresponding sensor...
			 */
			vector_copy(v, object->vertices[i].point);
			vector_sub(v, sensors->points[input->sensors[j]]);
			distance = vector_norm(v);

			/*
			 * ...and if distance lower than any of already
			 * calculated distances (distances are remembered as
			 * vertex interpolation coefficients at this stage),
			 * remember signal index and distance...
			 */
			for (k = 0; k < MIN_COUNT; k++)
				if (distance < interpolation->coefficients[MIN_COUNT * i + k]) {
					memmove(interpolation->indices + MIN_COUNT * i + k + 1, interpolation->indices + MIN_COUNT * i + k, (MIN_COUNT - (k + 1)) * sizeof(int));
					memmove(interpolation->coefficients + MIN_COUNT * i + k + 1, interpolation->coefficients + MIN_COUNT * i + k, (MIN_COUNT - (k + 1)) * sizeof(float));
					interpolation->indices[MIN_COUNT * i + k] = j;
					interpolation->coefficients[MIN_COUNT * i + k] = distance;
					break;
				}
		}

		/*
		 * Invert distances from vertex to its nearest sensors and
		 * calculate sum of these inversed values.  If any vertex is
		 * at same position as any sensor, then set final
		 * interpolation coefficients values immediately.
		 */
		den = 0;
		for (j = 0; j < MIN_COUNT; j++) {
			if (interpolation->coefficients[MIN_COUNT * i + j] == 0) {
				for (k = 0; k < MIN_COUNT; k++)
					interpolation->coefficients[MIN_COUNT * i + k] = 0;
				interpolation->coefficients[MIN_COUNT * i + j] = 1;
				break;
			}
			interpolation->coefficients[MIN_COUNT * i + j] = 1 / interpolation->coefficients[MIN_COUNT * i + j];
			den += interpolation->coefficients[MIN_COUNT * i + j];
		}

		/*
		 * Calculate final values of interpolation coefficients by
		 * dividing inversed distances from each sensor with sum of
		 * these distances.
		 */
		if (j == MIN_COUNT) {
			mul = 1 / den;
			for (j = 0; j < MIN_COUNT; j++)
				interpolation->coefficients[MIN_COUNT * i + j] *= mul;
		}
	}

	return interpolation;
}

void
interpolation_destroy(Interpolation * interpolation)
{
	/*
	 * Free memory used for indices to nearest sensors, interpolation
	 * coefficients and interpolation object.
	 */
	free(interpolation->indices);
	free(interpolation->coefficients);
	free(interpolation);
}

void
interpolation_interpolate(Interpolation * interpolation, float value, float lo, float hi, Color color)
{
	/* Clamp value to given range. */
	if (value <= lo)
		value = lo;
	else if (value >= hi)
		value = hi;
	else
		value = (value - lo) / (hi - lo);

	/*
	 * Map value to color.  Values from first half of range are mapped to
	 * colors between blue and green, values from second half or range
	 * are mapped to colors between green and red.
	 */
	if (value < 0.5) {
		color[R] = 0;
		color[G] = 1.6 * value;
		color[B] = 0.8 - 1.6 * value;
	} else {
		color[R] = 1.6 * value - 0.8;
		color[G] = 1.6 - 1.6 * value;
		color[B] = 0;
	}
}

void
interpolation_process(Interpolation * interpolation, Object * object, int count, float *values, float lo, float hi)
{
	float           value;	/* Interpolated value. */
	int             i, j;	/* Loop indices. */

	/* For each vertex... */
	for (i = 0; i < interpolation->count; i++) {
		/*
		 * ...interpolate vertex score value from score values of
		 * nearest sensors using precalculated interpolation
		 * coefficients...
		 */
		value = 0;
		for (j = 0; j < MIN_COUNT; j++)
			value += interpolation->coefficients[MIN_COUNT * i + j] * values[interpolation->indices[MIN_COUNT * i + j]];

		/*
		 * ...and map interpoolated vertex score value to vertex
		 * color.
		 */
		interpolation_interpolate(interpolation, value, lo, hi, object->vertices[i].color);
	}
}
