/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <stdlib.h>
#include "gradient.h"

/*
 * Handler for destroying gradient widget.  First argument is pointer to
 * gradient widget, second argument is unused.
 */
static void     on_destroy(GtkWidget * widget, gpointer);

/*
 * Handler for configure event.  First argument is gradient widget, second
 * argument is event structure, third argument is unused.
 */
static gboolean on_configure(GtkWidget * widget, GdkEventConfigure * event, gpointer);

/*
 * Handler for expose event.  First argument is gradient widget, second
 * argument is event structure, third argument is unused.
 */
static gboolean on_expose(GtkWidget * widget, GdkEventExpose * event, gpointer);

GtkWidget      *
gradient_create(Interpolation * interpolation)
{
	GtkWidget      *gradient;	/* Gradient widget. */

	/* Create gradient widget. */
	gradient = gtk_drawing_area_new();

	/* Register handlers for gradient widget signals. */
	g_signal_connect(G_OBJECT(gradient), "destroy", G_CALLBACK(on_destroy), NULL);
	g_signal_connect(G_OBJECT(gradient), "configure_event", G_CALLBACK(on_configure), NULL);
	g_signal_connect(G_OBJECT(gradient), "expose_event", G_CALLBACK(on_expose), NULL);

	/*
	 * Remember interpolation object with gradient widget.  Also
	 * initialize remembered legend pixels buffer.
	 */
	g_object_set_data(G_OBJECT(gradient), "interpolation", interpolation);
	g_object_set_data(G_OBJECT(gradient), "pixels", NULL);

	return gradient;
}

static void
on_destroy(GtkWidget * widget, gpointer data)
{
	char           *pixels;	/* Legend pixels buffer. */

	/* Destroy legend pixels buffer. */
	pixels = (char *)g_object_get_data(G_OBJECT(widget), "pixels");
	if (pixels != NULL)
		free(pixels);
}

static          gboolean
on_configure(GtkWidget * widget, GdkEventConfigure * event, gpointer data)
{
	Interpolation  *interpolation;	/* Interpolation object. */
	Color           color;	/* Interpolated color. */
	char           *pixels, *curr;	/* Legend pixels buffer and current
					 * pixel from buffer. */
	int             i, j;	/* Loop indices. */

	/* Get interpolation object. */
	interpolation = (Interpolation *) g_object_get_data(G_OBJECT(widget), "interpolation");
	assert(interpolation != NULL);

	/*
	 * Resize legend pixels buffer according to new gradient widget
	 * dimensions.
	 */
	pixels = (char *)g_object_get_data(G_OBJECT(widget), "pixels");
	if (pixels != NULL)
		free(pixels);
	pixels = (char *)malloc(widget->allocation.width * widget->allocation.height * 3 * sizeof(char));
	assert(pixels != NULL);

	/*
	 * Fill legend pixels buffer. For each column of pixels, interpolate
	 * column index to appropriate color using interpolation object and
	 * then draw column pixels.
	 */
	for (i = 0; i < widget->allocation.width; i++) {
		interpolation_interpolate(interpolation, i, 0, widget->allocation.width - 1, color);
		for (curr = pixels + 3 * i, j = 0; j < widget->allocation.height; j++, curr += 3 * widget->allocation.width) {
			*curr = (char)(255 * color[R]);
			*(curr + 1) = (char)(255 * color[G]);
			*(curr + 2) = (char)(255 * color[B]);
		}
	}

	/* Remember new legend pixels buffer. */
	g_object_set_data(G_OBJECT(widget), "pixels", pixels);

	/* Force repaint of gradient window. */
	gdk_window_invalidate_rect(widget->window, NULL, FALSE);

	return TRUE;
}

static          gboolean
on_expose(GtkWidget * widget, GdkEventExpose * event, gpointer data)
{
	char           *pixels;	/* Legend pixels buffer. */

	/* Get legend pixels buffer. */
	pixels = (char *)
		g_object_get_data(G_OBJECT(widget), "pixels");
	assert(pixels != NULL);

	/* Copy legend pixels to gradient window. */
	gdk_draw_rgb_image(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], 0, 0, widget->allocation.width, widget->allocation.height, GDK_RGB_DITHER_NORMAL, pixels, widget->allocation.width * 3);

	return FALSE;
}
