/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <math.h>
#include <stdlib.h>

/* Maximum input vector length is 2^POW_MAX. */
#define POW_MAX 15

void
fft(int p, float *in_re, float *in_im, float *out_re, float *out_im)
{
	static int      n = 0;	/* Vector length. */
	static int      shuffled[1 << POW_MAX];	/* Cashed values of shuffled
						 * indices. */
	static float    w_re[1 << (POW_MAX - 1)], w_im[1 << (POW_MAX - 1)];	/* Cashed values of
										 * powers of 2*pi/n=W
										 * value. */
	int             index;
	int             w_pow;
	int             step;
	int             lo, hi;
	int             w_curr;
	float           re, im;
	int             i, j;

	/* Check input argument validity. */
	assert(p >= 1 && p <= POW_MAX);

	/*
	 * If vector length different from previous function call, regenerate
	 * caches for shuffled indices and W value powers.
	 */
	if (n != 1 << p) {
		/* Remember vector length. */
		n = 1 << p;

		/* Calculate shuffled indices. */
		for (i = 0; i < n; i++)
			for (index = i, shuffled[i] = index & 1, j = 1; j < p; j++) {
				index >>= 1;
				shuffled[i] <<= 1;
				shuffled[i] |= index & 1;
			}

		/* Calculate powers of W value. */
		w_re[0] = 1, w_im[0] = 0;
		w_re[1] = cos(2 * M_PI / n), w_im[1] = sin(2 * M_PI / n);
		for (i = 2; i < n / 2; i++) {
			w_re[i] = w_re[i - 1] * w_re[1] - w_im[i - 1] * w_im[1];
			w_im[i] = w_im[i - 1] * w_re[1] + w_re[i - 1] * w_im[1];
		}
	}
	/* Initialize output vector. */
	for (i = 0; i < n; i++) {
		out_re[i] = in_re[shuffled[i]];
		out_im[i] = in_im[shuffled[i]];
	}

	/* In each step,... */
	for (step = 1, w_pow = n / 2; step < n; step *= 2, w_pow /= 2)
		/* ...for each pair of input vector elements... */
		for (lo = 0, hi = step; hi < n; lo += step, hi += step)
			for (i = 0, w_curr = 0; i < step; i++, lo++, hi++, w_curr += w_pow) {
				/* ...calculate new values of pair elements. */
				re = out_re[hi] * w_re[w_curr] + out_im[hi] * w_im[w_curr];
				im = out_im[hi] * w_re[w_curr] - out_re[hi] * w_im[w_curr];
				out_re[hi] = out_re[lo] - re;
				out_im[hi] = out_im[lo] - im;
				out_re[lo] += re;
				out_im[lo] += im;
			}
}
