/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.atconfiguration;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.*;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.hamcrest.CoreMatchers;

import org.junit.Assert;
import org.junit.Test;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanClassLoaderAware;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.support.ConfigurationSupport;
import org.springframework.config.java.support.FactoryBeanObjectCreationException;

import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ResourceLoaderAware;

import org.springframework.core.io.ResourceLoader;

import test.common.beans.TestBean;


/**
 * System tests for {@link ConfigurationSupport} base class.
 *
 * @author  Rod Johnson
 * @author  Chris Beams
 */
public class ConfigurationSupportTests {

    @Test
    public void testFactoryBeanCallbacks() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(FactoryBeanConfig.class);
        assertEquals("whatever", ctx.getBean("factoryTestBean"));
    }

    @Configuration
    public static class FactoryBeanConfig extends ConfigurationSupport {
        @Bean
        public Object factoryTestBean() {
            return getObject(new TestFactoryBeanForBeanFactory("whatever"));
        }
    }

    private static class TestFactoryBeanForBeanFactory implements FactoryBean, InitializingBean, BeanFactoryAware,
                                                                  BeanClassLoaderAware {
        private final Object expectedReturnedObject;

        private BeanFactory bf;

        private ClassLoader beanClassLoader;

        private boolean afterPropertiesSetCalled;

        public TestFactoryBeanForBeanFactory(Object expObject) { this.expectedReturnedObject = expObject; }

        public void afterPropertiesSet() throws Exception { afterPropertiesSetCalled = true; }

        public void setBeanClassLoader(ClassLoader cl) { this.beanClassLoader = cl; }

        public void setBeanFactory(BeanFactory bf) throws BeansException { this.bf = bf; }

        public Object getObject() throws Exception {
            assertTrue(afterPropertiesSetCalled);
            assertNotNull("Must have beanFactory set", bf);
            assertNotNull("Must have beanClassLoader set", beanClassLoader);

            return expectedReturnedObject;
        }

        public Class<?> getObjectType() { return null; }

        public boolean isSingleton() { return true; }
    }


    // -------------------------------------------------------------------------

    @Test
    public void testApplicationContextCallbacks() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ApplicationContextConfig.class);
        assertEquals("whatever", ctx.getBean("factoryTestBean"));
    }

    @Configuration
    public static class ApplicationContextConfig extends ConfigurationSupport {
        @Bean
        public Object factoryTestBean() {
            return getObject(new TestFactoryBeanForAppContext("whatever"));
        }
    }

    private static class TestFactoryBeanForAppContext extends TestFactoryBeanForBeanFactory
        implements ResourceLoaderAware, ApplicationContextAware {
        private ResourceLoader rl;

        private ApplicationContext ac;

        public TestFactoryBeanForAppContext(Object o) { super(o); }

        public void setResourceLoader(ResourceLoader rl) { this.rl = rl; }

        public void setApplicationContext(ApplicationContext ac) throws BeansException { this.ac = ac; }

        @Override
        public Object getObject() throws Exception {
            assertNotNull("Must have ResourceLoader set", rl);
            assertNotNull("Must have ApplicationContext set", ac);

            return super.getObject();
        }

    }


    // -------------------------------------------------------------------------

    @Test
    public void testTypeSafeGetObject() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(TypeSafeFactoryBeanConfig.class);
        assertEquals("localhost", ctx.getBean("hostname"));
    }

    @Configuration
    public static class TypeSafeFactoryBeanConfig extends ConfigurationSupport {
        public @Bean String hostname() {
            return this.getObject(new HostnameFactoryBean(), String.class);
        }

    }

    private static class HostnameFactoryBean implements FactoryBean {
        public Object getObject() throws Exception { return "localhost"; }
        public Class<?> getObjectType() { return String.class; }
        public boolean isSingleton() { return true; }
    }

    @Test(expected=BeanCreationException.class)
    public void testTypeSafeGetObjectWithWrongType() {
        new JavaConfigApplicationContext(InvalidTypeSafeFactoryBeanConfig.class);
    }

    @Configuration
    public static class InvalidTypeSafeFactoryBeanConfig extends ConfigurationSupport {
        public @Bean TestBean hostname() {
            return this.getObject(new HostnameFactoryBean(), TestBean.class);
        }

    }


    // -------------------------------------------------------------------------

    @Test
    public void testStringBasedGetBean() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(Config1.class, Config2.class);
        assertThat("xyz", CoreMatchers.equalTo(((TestBean) ctx.getBean("foo")).getName()));
    }

    @Test
    public void testTypeSafeGetBean() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(TypeSafeConfig.class, Config2.class);
        assertThat("xyz", CoreMatchers.equalTo(((TestBean) ctx.getBean("foo")).getName()));
    }

    @Test
    public void testTypeSafeGetBeanWithStringQualifier() {
        JavaConfigApplicationContext ctx =
            new JavaConfigApplicationContext(TypeSafeConfigWithStringQualifier.class,
                                             Config2.class);
        assertThat("xyz", CoreMatchers.equalTo(((TestBean) ctx.getBean("foo")).getName()));
    }

    @Test
    public void testGetObjectWrapsAndRethrowsExceptions() {
        try {
            new JavaConfigApplicationContext(ConfigWithFactoryBeanThatThrows.class);
            fail("should have thrown " + FactoryBeanObjectCreationException.class.getSimpleName());
        } catch (Throwable t) {
            Throwable[] throwables = ExceptionUtils.getThrowables(t);

            // the rootmost cause should be the originally thrown exception
            Assert.assertEquals(ConfigWithFactoryBeanThatThrows.FailingFactoryBean.exception, throwables[throwables.length-1]);

            // the exception wrapping the root cause should be FactoryBeanObjectCreationException
            Assert.assertThat(throwables[throwables.length-2], instanceOf(FactoryBeanObjectCreationException.class));
        }
    }

    @Configuration
    static class ConfigWithFactoryBeanThatThrows extends ConfigurationSupport {
        public @Bean Object failure() {
            return getObject(new FailingFactoryBean());
        }

        private static class FailingFactoryBean implements FactoryBean {
            static Exception exception = new Exception("failed");
            @SuppressWarnings("unchecked")
            public Class getObjectType() { return Object.class; }
            public boolean isSingleton() { return true; }
            public Object getObject() throws Exception { throw exception; }
        }
    }


    @Configuration
    public static class Config1 extends ConfigurationSupport {
        @Bean
        public TestBean foo() { return new TestBean((String) getBean("name")); }
    }

    @Configuration
    public static class Config2 {
        @Bean
        public String name() { return "xyz"; }
    }

    @Configuration
    public static class Config3 extends ConfigurationSupport {
        @Bean
        public TestBean foo() {
            // return new TestBean(getBean(String.class));
            return new TestBean((String) getBean("name"));
        }
    }

    @Configuration
    public static class TypeSafeConfig extends ConfigurationSupport {
        @Bean
        public TestBean foo() { return new TestBean(getBean(String.class)); }
    }

    @Configuration
    public static class TypeSafeConfigWithStringQualifier extends ConfigurationSupport {
        @Bean
        public TestBean foo() { return new TestBean(getBean(String.class, "name")); }
    }
}
