#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Import/Export
- Name     : PostgreSQL
- ID       : db_pgsql

Description
----------
Accessing PostgreSQL databases.
'''

from PySAGA.helper import Tool_Wrapper

def List_PostgreSQL_Connections(CONNECTIONS=None, Verbose=2):
    '''
    List PostgreSQL Connections
    ----------
    [db_pgsql.0]\n
    Lists all PostgreSQL sources.\n
    Arguments
    ----------
    - CONNECTIONS [`output table`] : Connections

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '0', 'List PostgreSQL Connections')
    if Tool.is_Okay():
        Tool.Set_Output('CONNECTIONS', CONNECTIONS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_0(CONNECTIONS=None, Verbose=2):
    '''
    List PostgreSQL Connections
    ----------
    [db_pgsql.0]\n
    Lists all PostgreSQL sources.\n
    Arguments
    ----------
    - CONNECTIONS [`output table`] : Connections

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '0', 'List PostgreSQL Connections')
    if Tool.is_Okay():
        Tool.Set_Output('CONNECTIONS', CONNECTIONS)
        return Tool.Execute(Verbose)
    return False

def Connect_to_PostgreSQL(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Connect to PostgreSQL
    ----------
    [db_pgsql.1]\n
    Connect to PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '1', 'Connect to PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_1(PG_HOST=None, PG_PORT=None, PG_DB=None, PG_USER=None, PG_PWD=None, Verbose=2):
    '''
    Connect to PostgreSQL
    ----------
    [db_pgsql.1]\n
    Connect to PostgreSQL data source.\n
    Arguments
    ----------
    - PG_HOST [`text`] : Host. Default: localhost
    - PG_PORT [`integer number`] : Port. Minimum: 0 Default: 5432
    - PG_DB [`text`] : Database
    - PG_USER [`text`] : User. Default: postgres
    - PG_PWD [`text`] : Password. Default: postgres

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '1', 'Connect to PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('PG_HOST', PG_HOST)
        Tool.Set_Option('PG_PORT', PG_PORT)
        Tool.Set_Option('PG_DB', PG_DB)
        Tool.Set_Option('PG_USER', PG_USER)
        Tool.Set_Option('PG_PWD', PG_PWD)
        return Tool.Execute(Verbose)
    return False

def Disconnect_from_PostgreSQL(CONNECTION=None, TRANSACT=None, Verbose=2):
    '''
    Disconnect from PostgreSQL
    ----------
    [db_pgsql.2]\n
    Disconnect PostgreSQL data source.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '2', 'Disconnect from PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_2(CONNECTION=None, TRANSACT=None, Verbose=2):
    '''
    Disconnect from PostgreSQL
    ----------
    [db_pgsql.2]\n
    Disconnect PostgreSQL data source.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '2', 'Disconnect from PostgreSQL')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Disconnect_All(TRANSACT=None, Verbose=2):
    '''
    Disconnect All
    ----------
    [db_pgsql.3]\n
    Disconnects all PostgreSQL connections.\n
    Arguments
    ----------
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '3', 'Disconnect All')
    if Tool.is_Okay():
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_3(TRANSACT=None, Verbose=2):
    '''
    Disconnect All
    ----------
    [db_pgsql.3]\n
    Disconnects all PostgreSQL connections.\n
    Arguments
    ----------
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '3', 'Disconnect All')
    if Tool.is_Okay():
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Begin_Transaction(CONNECTION=None, SAVEPOINT=None, Verbose=2):
    '''
    Begin Transaction
    ----------
    [db_pgsql.4]\n
    Begins a transaction, which will be finished later with a commit or rollback. Tries to add a save point, if already in transaction mode.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SAVEPOINT [`text`] : Save Point. Default: SAVEPOINT_01

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '4', 'Begin Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_4(CONNECTION=None, SAVEPOINT=None, Verbose=2):
    '''
    Begin Transaction
    ----------
    [db_pgsql.4]\n
    Begins a transaction, which will be finished later with a commit or rollback. Tries to add a save point, if already in transaction mode.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SAVEPOINT [`text`] : Save Point. Default: SAVEPOINT_01

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '4', 'Begin Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def CommitRollback_Transaction(CONNECTION=None, TRANSACT=None, SAVEPOINT=None, Verbose=2):
    '''
    Commit/Rollback Transaction
    ----------
    [db_pgsql.5]\n
    Execute a commit or rollback on open transactions with PostgreSQL source.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1
    - SAVEPOINT [`text`] : Save Point

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '5', 'Commit/Rollback Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TRANSACT', TRANSACT)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_5(CONNECTION=None, TRANSACT=None, SAVEPOINT=None, Verbose=2):
    '''
    Commit/Rollback Transaction
    ----------
    [db_pgsql.5]\n
    Execute a commit or rollback on open transactions with PostgreSQL source.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1
    - SAVEPOINT [`text`] : Save Point

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '5', 'Commit/Rollback Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TRANSACT', TRANSACT)
        Tool.Set_Option('SAVEPOINT', SAVEPOINT)
        return Tool.Execute(Verbose)
    return False

def Execute_SQL(TABLES=None, CONNECTION=None, SQL=None, OUTPUT=None, STOP=None, Verbose=2):
    '''
    Execute SQL
    ----------
    [db_pgsql.6]\n
    Execute SQL commands on a connected PostgreSQL source. Separate different commands with a semicolon (';').\n
    Arguments
    ----------
    - TABLES [`output table list`] : Tables
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SQL [`long text`] : SQL Statement. Default: CREATE TABLE myTable1 (Col1 VARCHAR(255) PRIMARY KEY, Col2 INTEGER);
INSERT INTO myTable1 (Col1, Col2) VALUES('First Value', 1);
DROP TABLE myTable1;

    - OUTPUT [`choice`] : Output. Available Choices: [0] none [1] message window [2] tables Default: 2
    - STOP [`boolean`] : Stop on Error. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '6', 'Execute SQL')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SQL', SQL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('STOP', STOP)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_6(TABLES=None, CONNECTION=None, SQL=None, OUTPUT=None, STOP=None, Verbose=2):
    '''
    Execute SQL
    ----------
    [db_pgsql.6]\n
    Execute SQL commands on a connected PostgreSQL source. Separate different commands with a semicolon (';').\n
    Arguments
    ----------
    - TABLES [`output table list`] : Tables
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SQL [`long text`] : SQL Statement. Default: CREATE TABLE myTable1 (Col1 VARCHAR(255) PRIMARY KEY, Col2 INTEGER);
INSERT INTO myTable1 (Col1, Col2) VALUES('First Value', 1);
DROP TABLE myTable1;

    - OUTPUT [`choice`] : Output. Available Choices: [0] none [1] message window [2] tables Default: 2
    - STOP [`boolean`] : Stop on Error. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '6', 'Execute SQL')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SQL', SQL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('STOP', STOP)
        return Tool.Execute(Verbose)
    return False

def List_Tables(TABLES=None, CONNECTION=None, Verbose=2):
    '''
    List Tables
    ----------
    [db_pgsql.10]\n
    Lists all tables of an PostgreSQL data source.\n
    Arguments
    ----------
    - TABLES [`output table`] : Tables
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '10', 'List Tables')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_10(TABLES=None, CONNECTION=None, Verbose=2):
    '''
    List Tables
    ----------
    [db_pgsql.10]\n
    Lists all tables of an PostgreSQL data source.\n
    Arguments
    ----------
    - TABLES [`output table`] : Tables
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '10', 'List Tables')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        return Tool.Execute(Verbose)
    return False

def List_Table_Fields(TABLE=None, CONNECTION=None, DB_TABLE=None, VERBOSE=None, Verbose=2):
    '''
    List Table Fields
    ----------
    [db_pgsql.11]\n
    Loads table information from PostgreSQL data source.\n
    Arguments
    ----------
    - TABLE [`output table`] : Field Description
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table
    - VERBOSE [`boolean`] : Verbose. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '11', 'List Table Fields')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('VERBOSE', VERBOSE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_11(TABLE=None, CONNECTION=None, DB_TABLE=None, VERBOSE=None, Verbose=2):
    '''
    List Table Fields
    ----------
    [db_pgsql.11]\n
    Loads table information from PostgreSQL data source.\n
    Arguments
    ----------
    - TABLE [`output table`] : Field Description
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table
    - VERBOSE [`boolean`] : Verbose. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '11', 'List Table Fields')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('VERBOSE', VERBOSE)
        return Tool.Execute(Verbose)
    return False

def Import_Table(TABLE=None, CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Import Table
    ----------
    [db_pgsql.12]\n
    Imports a table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '12', 'Import Table')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_12(TABLE=None, CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Import Table
    ----------
    [db_pgsql.12]\n
    Imports a table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '12', 'Import Table')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Export_Table(TABLE=None, CONNECTION=None, TABLE_PK=None, TABLE_NN=None, TABLE_UQ=None, NAME=None, EXISTS=None, Verbose=2):
    '''
    Export Table
    ----------
    [db_pgsql.13]\n
    Exports a table to a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`input table`] : Table
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TABLE_PK [`table fields`] : Primary Key
    - TABLE_NN [`table fields`] : Not Null
    - TABLE_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '13', 'Export Table')
    if Tool.is_Okay():
        Tool.Set_Input ('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TABLE_PK', TABLE_PK)
        Tool.Set_Option('TABLE_NN', TABLE_NN)
        Tool.Set_Option('TABLE_UQ', TABLE_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_13(TABLE=None, CONNECTION=None, TABLE_PK=None, TABLE_NN=None, TABLE_UQ=None, NAME=None, EXISTS=None, Verbose=2):
    '''
    Export Table
    ----------
    [db_pgsql.13]\n
    Exports a table to a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`input table`] : Table
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TABLE_PK [`table fields`] : Primary Key
    - TABLE_NN [`table fields`] : Not Null
    - TABLE_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '13', 'Export Table')
    if Tool.is_Okay():
        Tool.Set_Input ('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TABLE_PK', TABLE_PK)
        Tool.Set_Option('TABLE_NN', TABLE_NN)
        Tool.Set_Option('TABLE_UQ', TABLE_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        return Tool.Execute(Verbose)
    return False

def Drop_Table(CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Drop Table
    ----------
    [db_pgsql.14]\n
    Deletes a table from a PostgreSQL database.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '14', 'Drop Table')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_14(CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Drop Table
    ----------
    [db_pgsql.14]\n
    Deletes a table from a PostgreSQL database.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '14', 'Drop Table')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Import_Table_from_Query(TABLE=None, CONNECTION=None, TABLES=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query
    ----------
    [db_pgsql.15]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TABLES [`text`] : Tables
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '15', 'Import Table from Query')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_15(TABLE=None, CONNECTION=None, TABLES=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query
    ----------
    [db_pgsql.15]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - TABLES [`text`] : Tables
    - FIELDS [`text`] : Fields
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '15', 'Import Table from Query')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Import_Table_from_Query_GUI(TABLE=None, CONNECTION=None, WHERE=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query (GUI)
    ----------
    [db_pgsql.16]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - WHERE [`text`] : Where
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '16', 'Import Table from Query (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_16(TABLE=None, CONNECTION=None, WHERE=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from Query (GUI)
    ----------
    [db_pgsql.16]\n
    Import a SQL table from a PostgreSQL database.\n
    Arguments
    ----------
    - TABLE [`output table`] : Query Result
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - WHERE [`text`] : Where
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '16', 'Import Table from Query (GUI)')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Import_Shapes(SHAPES=None, COLLECTION=None, CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Import Shapes
    ----------
    [db_pgsql.20]\n
    Imports shapes from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - COLLECTION [`output shapes list`] : Shapes
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '20', 'Import Shapes')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_20(SHAPES=None, COLLECTION=None, CONNECTION=None, DB_TABLE=None, Verbose=2):
    '''
    Import Shapes
    ----------
    [db_pgsql.20]\n
    Imports shapes from a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`output shapes`] : Shapes
    - COLLECTION [`output shapes list`] : Shapes
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '20', 'Import Shapes')
    if Tool.is_Okay():
        Tool.Set_Output('SHAPES', SHAPES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        return Tool.Execute(Verbose)
    return False

def Export_Shapes(SHAPES=None, CONNECTION=None, SHAPES_PK=None, SHAPES_NN=None, SHAPES_UQ=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, CRS_GEOGCS=None, CRS_PROJCS=None, Verbose=2):
    '''
    Export Shapes
    ----------
    [db_pgsql.21]\n
    Exports shapes to a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SHAPES_PK [`table fields`] : Primary Key
    - SHAPES_NN [`table fields`] : Not Null
    - SHAPES_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name. if empty shapes layers's name is used as table name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG
    - CRS_GEOGCS [`choice`] : Geographic Coordinate Systems. Available Choices: [0] <select> [1] 1_Ceres_2015 [2] 4_Vesta_2015 [3] AGD66 [4] AGD84 [5] ATF (Paris) [6] ATS77 [7] Abidjan 1987 [8] Accra [9] Aden 1925 [10] Adindan [11] Afgooye [12] Agadez [13] Ain el Abd [14] Albanian 1987 [15] American Samoa 1962 [16] Amersfoort [17] Ammassalik 1958 [18] Anguilla 1957 [19] Antigua 1943 [20] Aratu [21] Arc 1950 [22] Arc 1960 [23] Ascension Island 1958 [24] Astro DOS 71 [25] Australian Antarctic [26] Australian Antarctic [27] Australian Antarctic [28] Ayabelle Lighthouse [29] Azores Central 1948 [30] Azores Central 1995 [31] Azores Occidental 1939 [32] Azores Oriental 1940 [33] Azores Oriental 1995 [34] BDA2000 [35] BDA2000 [36] BGS2005 [37] BGS2005 [38] Barbados 1938 [39] Batavia [40] Batavia (Jakarta) [41] Beduaram [42] Beijing 1954 [43] Bekaa Valley 1920 [44] Belge 1950 [45] Belge 1950 (Brussels) [46] Belge 1972 [47] Bellevue [48] Bermuda 1957 [49] Bern 1898 (Bern) [50] Bern 1938 [51] Bioko [52] Bissau [53] Bogota 1975 [54] Bogota 1975 (Bogota) [55] Bukit Rimpah [56] CGRS93 [57] CGRS93 [58] CH1903 [59] CH1903+ [60] CHTRF95 [61] CHTRF95 [62] CHTRF95 [63] CIGD11 [64] CIGD11 [65] CR-SIRGAS [66] CR-SIRGAS [67] CR05 [68] CR05 [69] CSG67 [70] Cadastre 1997 [71] Cadastre 1997 [72] California_SRS_Epoch_2017.50_(NAD83) [73] Camacupa [74] Camp Area Astro [75] Campo Inchauspe [76] Cape [77] Cape Canaveral [78] Carthage [79] Carthage (Paris) [80] Chatham Islands 1971 [81] Chatham Islands 1979 [82] China Geodetic Coordinate System 2000 [83] China Geodetic Coordinate System 2000 [84] Chos Malal 1914 [85] Chua [86] Cocos Islands 1965 [87] Combani 1950 [88] Conakry 1905 [89] Corrego Alegre 1961 [90] Corrego Alegre 1970-72 [91] Cote d'Ivoire [92] DB_REF [93] DB_REF [94] DGN95 [95] DGN95 [96] DHDN [97] DRUKREF 03 [98] DRUKREF 03 [99] Dabola 1981 [100] Datum 73 [101] Dealul Piscului 1930 [102] Dealul Piscului 1970 [103] Deception Island [104] Deir ez Zor [105] Diego Garcia 1969 [106] Dominica 1945 [107] Douala [108] Douala 1948 [109] ED50 [110] ED50(ED77) [111] ED79 [112] ED87 [113] ELD79 [114] EST92 [115] EST97 [116] EST97 [117] EST97 [118] ETRF2000 [119] ETRF2000 [120] ETRF2005 [121] ETRF2014 [122] ETRF89 [123] ETRF90 [124] ETRF90 [125] ETRF91 [126] ETRF91 [127] ETRF92 [128] ETRF92 [129] ETRF93 [130] ETRF93 [131] ETRF94 [132] ETRF94 [133] ETRF96 [134] ETRF96 [135] ETRF97 [136] ETRF97 [137] ETRS89 [138] ETRS89 [139] ETRS89 [140] Easter Island 1967 [141] Egypt 1907 [142] Egypt 1930 [143] Egypt Gulf of Suez S-650 TL [144] FD54 [145] FD58 [146] FEH2010 [147] FEH2010 [148] Fahud [149] Fatu Iva 72 [150] Fiji 1956 [151] Fiji 1986 [152] Fort Marigot [153] GCGD59 [154] GCS_Adrastea_2000 [155] GCS_Alaskan_Islands [156] GCS_Amalthea_2000 [157] GCS_American_Samoa_1962 (deprecated) [158] GCS_Ananke_2000 [159] GCS_Anna_1_1965 (deprecated) [160] GCS_Ariel_2000 [161] GCS_Ascension_Island_1958 (deprecated) [162] GCS_Assumed_Geographic_1 (deprecated) [163] GCS_Astro_1952 (deprecated) [164] GCS_Atlas_2000 [165] GCS_Ayabelle (deprecated) [166] GCS_Bab_South [167] GCS_Beacon_E_1945 (deprecated) [168] GCS_Belinda_2000 [169] GCS_Bellevue_IGN (deprecated) [170] GCS_Bermuda_2000 (deprecated) [171] GCS_Bianca_2000 [172] GCS_CGRS_1993 (deprecated) [173] GCS_CR05 (deprecated) [174] GCS_Callisto_2000 [175] GCS_Calypso_2000 [176] GCS_Camp_Area (deprecated) [177] GCS_Canton_1966 (deprecated) [178] GCS_Cape_Canaveral (deprecated) [179] GCS_Carme_2000 [180] GCS_Carthage (deprecated) [181] GCS_Carthage_Grad [182] GCS_Charon_2000 [183] GCS_Chatham_Island_1971 (deprecated) [184] GCS_Chatham_Islands_1979 (deprecated) [185] GCS_Cordelia_2000 [186] GCS_Cressida_2000 [187] GCS_D48 [188] GCS_DOS_1968 [189] GCS_DOS_71_4 (deprecated) [190] GCS_Datum_Lisboa_Bessel [191] GCS_Datum_Lisboa_Hayford [192] GCS_Deception_Island (deprecated) [193] GCS_Deimos_2000 [194] GCS_Desdemona_2000 [195] GCS_Despina_2000 [196] GCS_Dione_2000 [197] GCS_ETRF_1989 [198] GCS_EUREF_FIN [199] GCS_Easter_Island_1967 (deprecated) [200] GCS_Elara_2000 [201] GCS_Enceladus_2000 [202] GCS_Epimetheus_2000 [203] GCS_Estonia_1937 [204] GCS_Europa_2000 [205] GCS_European_1979 (deprecated) [206] GCS_Everest_Bangladesh [207] GCS_Everest_India_Nepal [208] GCS_Everest_Modified_1969 [209] GCS_Fischer_1960 [210] GCS_Fischer_1968 [211] GCS_Fischer_Modified [212] GCS_Fort_Thomas_1955 [213] GCS_GDBD2009 (deprecated) [214] GCS_GUX_1 [215] GCS_Galatea_2000 [216] GCS_Gan_1970 (deprecated) [217] GCS_Ganymede_2000 [218] GCS_Graciosa_Base_SW_1948 [219] GCS_Guam_1963 (deprecated) [220] GCS_Gunung_Segara (deprecated) [221] GCS_HD1909 (deprecated) [222] GCS_Helene_2000 [223] GCS_Hermannskogel [224] GCS_Himalia_2000 [225] GCS_Hjorsey_1955 (deprecated) [226] GCS_Hong_Kong_1963_67 (deprecated) [227] GCS_Hong_Kong_1980 (deprecated) [228] GCS_Hough_1960 [229] GCS_Hyperion_2000 [230] GCS_IGRS (deprecated) [231] GCS_IGS08 [232] GCS_ISN_2004 (deprecated) [233] GCS_ISTS_061_1968 (deprecated) [234] GCS_ISTS_073_1969 (deprecated) [235] GCS_ITRF_1988 [236] GCS_ITRF_1989 [237] GCS_ITRF_1990 [238] GCS_ITRF_1991 [239] GCS_ITRF_1992 [240] GCS_ITRF_1993 [241] GCS_ITRF_1994 [242] GCS_ITRF_1996 [243] GCS_ITRF_1997 [244] GCS_ITRF_2000 [245] GCS_ITRF_2005 [246] GCS_ITRF_2008 [247] GCS_Iapetus_2000 [248] GCS_International_1967 (deprecated) [249] GCS_Io_2000 [250] GCS_JAD_2001 (deprecated) [251] GCS_JGD_2000 (deprecated) [252] GCS_JGD_2011 (deprecated) [253] GCS_Janus_2000 [254] GCS_Johnston_Island_1961 (deprecated) [255] GCS_Jordan [256] GCS_Juliet_2000 [257] GCS_Jupiter_2000 [258] GCS_Kerguelen_Island_1949 (deprecated) [259] GCS_Kusaie_1951 (deprecated) [260] GCS_Kyrg-06 [261] GCS_LC5_1961 [262] GCS_Larissa_2000 [263] GCS_Leda_2000 [264] GCS_Lysithea_2000 [265] GCS_MACAO_2008 [266] GCS_MGI_1901 (deprecated) [267] GCS_MONREF_1997 [268] GCS_MSK_1942 [269] GCS_Majuro [270] GCS_Mars_1979 [271] GCS_Mars_2000 [272] GCS_Merchich_Degree [273] GCS_Mercury_2000 [274] GCS_Metis_2000 [275] GCS_Midway_1961 (deprecated) [276] GCS_Mimas_2000 [277] GCS_Miranda_2000 [278] GCS_Moon_2000 [279] GCS_NAD_1983_2011 (deprecated) [280] GCS_NAD_1983_CORS96 (deprecated) [281] GCS_NAD_1983_HARN_Adj_MN_Anoka [282] GCS_NAD_1983_HARN_Adj_MN_Becker [283] GCS_NAD_1983_HARN_Adj_MN_Beltrami_North [284] GCS_NAD_1983_HARN_Adj_MN_Beltrami_South [285] GCS_NAD_1983_HARN_Adj_MN_Benton [286] GCS_NAD_1983_HARN_Adj_MN_Big_Stone [287] GCS_NAD_1983_HARN_Adj_MN_Blue_Earth [288] GCS_NAD_1983_HARN_Adj_MN_Brown [289] GCS_NAD_1983_HARN_Adj_MN_Carlton [290] GCS_NAD_1983_HARN_Adj_MN_Carver [291] GCS_NAD_1983_HARN_Adj_MN_Cass_North [292] GCS_NAD_1983_HARN_Adj_MN_Cass_South [293] GCS_NAD_1983_HARN_Adj_MN_Chippewa [294] GCS_NAD_1983_HARN_Adj_MN_Chisago [295] GCS_NAD_1983_HARN_Adj_MN_Cook_North [296] GCS_NAD_1983_HARN_Adj_MN_Cook_South [297] GCS_NAD_1983_HARN_Adj_MN_Cottonwood [298] GCS_NAD_1983_HARN_Adj_MN_Crow_Wing [299] GCS_NAD_1983_HARN_Adj_MN_Dakota [300] GCS_NAD_1983_HARN_Adj_MN_Dodge [301] GCS_NAD_1983_HARN_Adj_MN_Douglas [302] GCS_NAD_1983_HARN_Adj_MN_Faribault [303] GCS_NAD_1983_HARN_Adj_MN_Fillmore [304] GCS_NAD_1983_HARN_Adj_MN_Freeborn [305] GCS_NAD_1983_HARN_Adj_MN_Goodhue [306] GCS_NAD_1983_HARN_Adj_MN_Grant [307] GCS_NAD_1983_HARN_Adj_MN_Hennepin [308] GCS_NAD_1983_HARN_Adj_MN_Houston [309] GCS_NAD_1983_HARN_Adj_MN_Isanti [310] GCS_NAD_1983_HARN_Adj_MN_Itasca_North [311] GCS_NAD_1983_HARN_Adj_MN_Itasca_South [312] GCS_NAD_1983_HARN_Adj_MN_Jackson [313] GCS_NAD_1983_HARN_Adj_MN_Kanabec [314] GCS_NAD_1983_HARN_Adj_MN_Kandiyohi [315] GCS_NAD_1983_HARN_Adj_MN_Kittson [316] GCS_NAD_1983_HARN_Adj_MN_Koochiching [317] GCS_NAD_1983_HARN_Adj_MN_Lac_Qui_Parle [318] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North [319] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South [320] GCS_NAD_1983_HARN_Adj_MN_Le_Sueur [321] GCS_NAD_1983_HARN_Adj_MN_Lincoln [322] GCS_NAD_1983_HARN_Adj_MN_Lyon [323] GCS_NAD_1983_HARN_Adj_MN_Mahnomen [324] GCS_NAD_1983_HARN_Adj_MN_Marshall [325] GCS_NAD_1983_HARN_Adj_MN_Martin [326] GCS_NAD_1983_HARN_Adj_MN_McLeod [327] GCS_NAD_1983_HARN_Adj_MN_Meeker [328] GCS_NAD_1983_HARN_Adj_MN_Morrison [329] GCS_NAD_1983_HARN_Adj_MN_Mower [330] GCS_NAD_1983_HARN_Adj_MN_Murray [331] GCS_NAD_1983_HARN_Adj_MN_Nicollet [332] GCS_NAD_1983_HARN_Adj_MN_Nobles [333] GCS_NAD_1983_HARN_Adj_MN_Norman [334] GCS_NAD_1983_HARN_Adj_MN_Olmsted [335] GCS_NAD_1983_HARN_Adj_MN_Ottertail [336] GCS_NAD_1983_HARN_Adj_MN_Pennington [337] GCS_NAD_1983_HARN_Adj_MN_Pine [338] GCS_NAD_1983_HARN_Adj_MN_Pipestone [339] GCS_NAD_1983_HARN_Adj_MN_Polk [340] GCS_NAD_1983_HARN_Adj_MN_Pope [341] GCS_NAD_1983_HARN_Adj_MN_Ramsey [342] GCS_NAD_1983_HARN_Adj_MN_Red_Lake [343] GCS_NAD_1983_HARN_Adj_MN_Redwood [344] GCS_NAD_1983_HARN_Adj_MN_Renville [345] GCS_NAD_1983_HARN_Adj_MN_Rice [346] GCS_NAD_1983_HARN_Adj_MN_Rock [347] GCS_NAD_1983_HARN_Adj_MN_Roseau [348] GCS_NAD_1983_HARN_Adj_MN_Scott [349] GCS_NAD_1983_HARN_Adj_MN_Sherburne [350] GCS_NAD_1983_HARN_Adj_MN_Sibley [351] GCS_NAD_1983_HARN_Adj_MN_St_Louis [352] GCS_NAD_1983_HARN_Adj_MN_St_Louis_Central [353] GCS_NAD_1983_HARN_Adj_MN_St_Louis_North [354] GCS_NAD_1983_HARN_Adj_MN_St_Louis_South [355] GCS_NAD_1983_HARN_Adj_MN_Stearns [356] GCS_NAD_1983_HARN_Adj_MN_Steele [357] GCS_NAD_1983_HARN_Adj_MN_Stevens [358] GCS_NAD_1983_HARN_Adj_MN_Swift [359] GCS_NAD_1983_HARN_Adj_MN_Todd [360] GCS_NAD_1983_HARN_Adj_MN_Traverse [361] GCS_NAD_1983_HARN_Adj_MN_Wabasha [362] GCS_NAD_1983_HARN_Adj_MN_Wadena [363] GCS_NAD_1983_HARN_Adj_MN_Waseca [364] GCS_NAD_1983_HARN_Adj_MN_Watonwan [365] GCS_NAD_1983_HARN_Adj_MN_Winona [366] GCS_NAD_1983_HARN_Adj_MN_Wright [367] GCS_NAD_1983_HARN_Adj_MN_Yellow_Medicine [368] GCS_NAD_1983_HARN_Adj_WI_Adams (deprecated) [369] GCS_NAD_1983_HARN_Adj_WI_Ashland (deprecated) [370] GCS_NAD_1983_HARN_Adj_WI_Barron (deprecated) [371] GCS_NAD_1983_HARN_Adj_WI_Bayfield (deprecated) [372] GCS_NAD_1983_HARN_Adj_WI_Brown (deprecated) [373] GCS_NAD_1983_HARN_Adj_WI_Buffalo (deprecated) [374] GCS_NAD_1983_HARN_Adj_WI_Burnett (deprecated) [375] GCS_NAD_1983_HARN_Adj_WI_Calumet (deprecated) [376] GCS_NAD_1983_HARN_Adj_WI_Chippewa [377] GCS_NAD_1983_HARN_Adj_WI_Clark (deprecated) [378] GCS_NAD_1983_HARN_Adj_WI_Columbia (deprecated) [379] GCS_NAD_1983_HARN_Adj_WI_Crawford (deprecated) [380] GCS_NAD_1983_HARN_Adj_WI_Dane (deprecated) [381] GCS_NAD_1983_HARN_Adj_WI_Dodge (deprecated) [382] GCS_NAD_1983_HARN_Adj_WI_Door (deprecated) [383] GCS_NAD_1983_HARN_Adj_WI_Douglas (deprecated) [384] GCS_NAD_1983_HARN_Adj_WI_Dunn (deprecated) [385] GCS_NAD_1983_HARN_Adj_WI_EauClaire (deprecated) [386] GCS_NAD_1983_HARN_Adj_WI_Florence (deprecated) [387] GCS_NAD_1983_HARN_Adj_WI_FondduLac (deprecated) [388] GCS_NAD_1983_HARN_Adj_WI_Forest (deprecated) [389] GCS_NAD_1983_HARN_Adj_WI_Grant (deprecated) [390] GCS_NAD_1983_HARN_Adj_WI_Green (deprecated) [391] GCS_NAD_1983_HARN_Adj_WI_GreenLake (deprecated) [392] GCS_NAD_1983_HARN_Adj_WI_Iowa (deprecated) [393] GCS_NAD_1983_HARN_Adj_WI_Iron (deprecated) [394] GCS_NAD_1983_HARN_Adj_WI_Jackson (deprecated) [395] GCS_NAD_1983_HARN_Adj_WI_Jefferson (deprecated) [396] GCS_NAD_1983_HARN_Adj_WI_Juneau (deprecated) [397] GCS_NAD_1983_HARN_Adj_WI_Kenosha (deprecated) [398] GCS_NAD_1983_HARN_Adj_WI_Kewaunee (deprecated) [399] GCS_NAD_1983_HARN_Adj_WI_LaCrosse (deprecated) [400] GCS_NAD_1983_HARN_Adj_WI_Lafayette (deprecated) [401] GCS_NAD_1983_HARN_Adj_WI_Langlade (deprecated) [402] GCS_NAD_1983_HARN_Adj_WI_Lincoln (deprecated) [403] GCS_NAD_1983_HARN_Adj_WI_Manitowoc (deprecated) [404] GCS_NAD_1983_HARN_Adj_WI_Marathon (deprecated) [405] GCS_NAD_1983_HARN_Adj_WI_Marinette (deprecated) [406] GCS_NAD_1983_HARN_Adj_WI_Marquette (deprecated) [407] GCS_NAD_1983_HARN_Adj_WI_Menominee (deprecated) [408] GCS_NAD_1983_HARN_Adj_WI_Milwaukee (deprecated) [409] GCS_NAD_1983_HARN_Adj_WI_Monroe (deprecated) [410] GCS_NAD_1983_HARN_Adj_WI_Oconto (deprecated) [411] GCS_NAD_1983_HARN_Adj_WI_Oneida (deprecated) [412] GCS_NAD_1983_HARN_Adj_WI_Outagamie (deprecated) [413] GCS_NAD_1983_HARN_Adj_WI_Ozaukee (deprecated) [414] GCS_NAD_1983_HARN_Adj_WI_Pepin (deprecated) [415] GCS_NAD_1983_HARN_Adj_WI_Pierce (deprecated) [416] GCS_NAD_1983_HARN_Adj_WI_Polk (deprecated) [417] GCS_NAD_1983_HARN_Adj_WI_Portage (deprecated) [418] GCS_NAD_1983_HARN_Adj_WI_Price (deprecated) [419] GCS_NAD_1983_HARN_Adj_WI_Racine (deprecated) [420] GCS_NAD_1983_HARN_Adj_WI_Richland (deprecated) [421] GCS_NAD_1983_HARN_Adj_WI_Rock (deprecated) [422] GCS_NAD_1983_HARN_Adj_WI_Rusk (deprecated) [423] GCS_NAD_1983_HARN_Adj_WI_Sauk (deprecated) [424] GCS_NAD_1983_HARN_Adj_WI_Sawyer (deprecated) [425] GCS_NAD_1983_HARN_Adj_WI_Shawano (deprecated) [426] GCS_NAD_1983_HARN_Adj_WI_Sheboygan (deprecated) [427] GCS_NAD_1983_HARN_Adj_WI_StCroix (deprecated) [428] GCS_NAD_1983_HARN_Adj_WI_Taylor (deprecated) [429] GCS_NAD_1983_HARN_Adj_WI_Trempealeau (deprecated) [430] GCS_NAD_1983_HARN_Adj_WI_Vernon (deprecated) [431] GCS_NAD_1983_HARN_Adj_WI_Vilas (deprecated) [432] GCS_NAD_1983_HARN_Adj_WI_Walworth (deprecated) [433] GCS_NAD_1983_HARN_Adj_WI_Washburn (deprecated) [434] GCS_NAD_1983_HARN_Adj_WI_Washington (deprecated) [435] GCS_NAD_1983_HARN_Adj_WI_Waukesha (deprecated) [436] GCS_NAD_1983_HARN_Adj_WI_Waupaca (deprecated) [437] GCS_NAD_1983_HARN_Adj_WI_Waushara (deprecated) [438] GCS_NAD_1983_HARN_Adj_WI_Winnebago (deprecated) [439] GCS_NAD_1983_HARN_Adj_WI_Wood (deprecated) [440] GCS_NAD_1983_MA11 (deprecated) [441] GCS_NAD_1983_MARP00 [442] GCS_NAD_1983_PA11 (deprecated) [443] GCS_NAD_1983_PACP00 [444] GCS_NZGD_2000 (deprecated) [445] GCS_Naiad_2000 [446] GCS_Nepal_Nagarkot (deprecated) [447] GCS_Neptune_2000 [448] GCS_Nereid_2000 [449] GCS_Oberon_2000 [450] GCS_Observatorio_Meteorologico_1939 [451] GCS_Observatorio_Meteorologico_1965 [452] GCS_Ocotepeque_1935 (deprecated) [453] GCS_Old_Hawaiian_Intl_1924 [454] GCS_Oman [455] GCS_Ophelia_2000 [456] GCS_PTRA08 (deprecated) [457] GCS_Pan_2000 [458] GCS_Pandora_2000 [459] GCS_Pasiphae_2000 [460] GCS_Phobos_2000 [461] GCS_Phoebe_2000 [462] GCS_Pico_de_Las_Nieves (deprecated) [463] GCS_Pitcairn_1967 (deprecated) [464] GCS_Pluto_2000 [465] GCS_Pohnpei [466] GCS_Point_58 (deprecated) [467] GCS_Portia_2000 [468] GCS_Porto_Santo_1936 (deprecated) [469] GCS_Prometheus_2000 [470] GCS_Proteus_2000 [471] GCS_Puck_2000 [472] GCS_REGVEN (deprecated) [473] GCS_RGF_1993 (deprecated) [474] GCS_Reunion_1947 (deprecated) [475] GCS_Rhea_2000 [476] GCS_Roma_1940 [477] GCS_Rosalind_2000 [478] GCS_S42_Hungary [479] GCS_Santo_DOS_1965 (deprecated) [480] GCS_Sao_Braz [481] GCS_Saturn_2000 [482] GCS_Selvagem_Grande_1938 (deprecated) [483] GCS_Sierra_Leone_1960 [484] GCS_Sinope_2000 [485] GCS_South_Asia_Singapore [486] GCS_Sphere_ARC_INFO [487] GCS_Sphere_EMEP [488] GCS_Sphere_GRS_1980_Mean_Radius [489] GCS_TWD_1967 (deprecated) [490] GCS_TWD_1997 (deprecated) [491] GCS_Telesto_2000 [492] GCS_Tern_Island_1961 (deprecated) [493] GCS_Tethys_2000 [494] GCS_Thalassa_2000 [495] GCS_Thebe_2000 [496] GCS_Titan_2000 [497] GCS_Titania_2000 [498] GCS_Tristan_1968 (deprecated) [499] GCS_Triton_2000 [500] GCS_Umbriel_2000 [501] GCS_Uranus_2000 [502] GCS_Venus_1985 [503] GCS_Venus_2000 [504] GCS_Viti_Levu_1916 (deprecated) [505] GCS_Voirol_1875 (deprecated) [506] GCS_Voirol_1875_Grad [507] GCS_Voirol_1879_Grad [508] GCS_Voirol_Unifie_1960 (deprecated) [509] GCS_Voirol_Unifie_1960 (deprecated) [510] GCS_Voirol_Unifie_1960_Degree (deprecated) [511] GCS_Voirol_Unifie_1960_Paris (deprecated) [512] GCS_WGS_1966 (deprecated) [513] GCS_WGS_1984_Major_Auxiliary_Sphere (deprecated) [514] GCS_Wake_Eniwetok_1960 (deprecated) [515] GCS_Wake_Island_1952 (deprecated) [516] GCS_Walbeck [517] GCS_Xrail84 [518] GDA2020 [519] GDA2020 [520] GDA94 [521] GDA94 [522] GDA94 [523] GDBD2009 [524] GDBD2009 [525] GDM2000 [526] GDM2000 [527] GGD [528] GGRS87 [529] GR96 [530] GR96 [531] GSK-2011 [532] GSK-2011 [533] Gambia [534] Gan 1970 [535] Gandajika [536] Gandajika 1970 [537] Garoua [538] Garoua [539] Grand Comoros [540] Greek [541] Greek (Athens) [542] Grenada 1953 [543] Guadeloupe 1948 [544] Guam 1963 [545] Gulshan 303 [546] Gusterberg (Ferro) [547] Guyane Francaise [548] HD1909 [549] HD72 [550] HTRS96 [551] HTRS96 [552] Hanoi 1972 [553] Hartebeesthoek94 [554] Hartebeesthoek94 [555] Hartebeesthoek94 [556] Helle 1954 [557] Herat North [558] Hito XVIII 1963 [559] Hjorsey 1955 [560] Hong Kong 1963 [561] Hong Kong 1963(67) [562] Hong Kong 1980 [563] Hong Kong Geodetic CS [564] Hong Kong Geodetic CS [565] Hu Tzu Shan 1950 [566] ID74 [567] IG05 Intermediate CRS [568] IG05 Intermediate CRS [569] IG05/12 Intermediate CRS [570] IG05/12 Intermediate CRS [571] IGC 1962 6th Parallel South [572] IGCB 1955 [573] IGD05 [574] IGD05 [575] IGD05 [576] IGD05 [577] IGD05/12 [578] IGD05/12 [579] IGD05/12 [580] IGD05/12 [581] IGM95 [582] IGM95 [583] IGN 1962 Kerguelen [584] IGN Astro 1960 [585] IGN53 Mare [586] IGN56 Lifou [587] IGN63 Hiva Oa [588] IGN72 Grand Terre [589] IGN72 Grande Terre [590] IGN72 Nuku Hiva [591] IGRS [592] IGRS [593] IGS00 [594] IGS00 [595] IGS05 [596] IGS05 [597] IGS08 [598] IGS08 [599] IGS14 [600] IGS14 [601] IGS14 [602] IGS97 [603] IGS97 [604] IGb00 [605] IGb00 [606] IGb08 [607] IGb08 [608] IKBD-92 [609] IRENET95 [610] IRENET95 [611] IRENET95 [612] ISN2004 [613] ISN2004 [614] ISN2016 [615] ISN2016 [616] ISN93 [617] ISN93 [618] ISN93 [619] ITRF2000 [620] ITRF2000 [621] ITRF2005 [622] ITRF2005 [623] ITRF2008 [624] ITRF2008 [625] ITRF2014 [626] ITRF2014 [627] ITRF2014 [628] ITRF88 [629] ITRF88 [630] ITRF89 [631] ITRF89 [632] ITRF90 [633] ITRF90 [634] ITRF91 [635] ITRF91 [636] ITRF92 [637] ITRF92 [638] ITRF93 [639] ITRF93 [640] ITRF94 [641] ITRF94 [642] ITRF96 [643] ITRF96 [644] ITRF97 [645] ITRF97 [646] Indian 1954 [647] Indian 1960 [648] Indian 1975 [649] Israel 1993 [650] Iwo Jima 1945 [651] JAD2001 [652] JAD2001 [653] JAD69 [654] JGD2000 [655] JGD2000 [656] JGD2000 [657] JGD2011 [658] JGD2011 [659] Jamaica 1875 [660] Johnston Island 1961 [661] Jouik 1961 [662] K0 1949 [663] KKJ [664] KOC [665] KUDAMS [666] Kalianpur 1880 [667] Kalianpur 1937 [668] Kalianpur 1962 [669] Kalianpur 1975 [670] Kandawala [671] Karbala 1979 [672] Kasai 1953 [673] Katanga 1955 [674] Kertau (RSO) [675] Kertau 1968 [676] Korea 2000 [677] Korea 2000 [678] Korean 1985 [679] Korean 1995 [680] Kousseri [681] Kusaie 1951 [682] Kyrg-06 [683] Kyrg-06 [684] LGD2006 [685] LGD2006 [686] LKS92 [687] LKS92 [688] LKS92 [689] LKS94 [690] LKS94 [691] LKS94 [692] LKS94 (ETRS89) [693] La Canoa [694] Lake [695] Lao 1993 [696] Lao 1993 [697] Lao 1997 [698] Lao 1997 [699] Le Pouce 1934 [700] Leigon [701] Liberia 1964 [702] Lisbon [703] Lisbon (Lisbon) [704] Lisbon 1890 [705] Lisbon 1890 (Lisbon) [706] Locodjo 1965 [707] Loma Quintana [708] Lome [709] Luxembourg 1930 [710] Luzon 1911 [711] M'poraloko [712] MACARIO SOLIS [713] MACARIO SOLIS [714] MAGNA-SIRGAS [715] MAGNA-SIRGAS [716] MARGEN [717] MARGEN [718] MGI [719] MGI (Ferro) [720] MGI 1901 [721] MOLDREF99 [722] MOLDREF99 [723] MOP78 [724] MTRF-2000 [725] MTRF-2000 [726] Macao 1920 [727] Macao 2008 [728] Macao 2008 [729] Madeira 1936 [730] Madrid 1870 (Madrid) [731] Madzansua [732] Mahe 1971 [733] Makassar [734] Makassar (Jakarta) [735] Malongo 1987 [736] Manoca [737] Manoca 1962 [738] Marcus Island 1952 [739] Mars_2000_(Sphere) [740] Marshall Islands 1960 [741] Martinique 1938 [742] Massawa [743] Maupiti 83 [744] Mauritania 1999 [745] Mauritania 1999 [746] Mauritania 1999 [747] Merchich [748] Mercury_2015 [749] Mexico ITRF2008 [750] Mexico ITRF2008 [751] Mexico ITRF92 [752] Mexico ITRF92 [753] Mhast [754] Mhast (offshore) [755] Mhast (onshore) [756] Mhast 1951 [757] Midway 1961 [758] Minna [759] Monte Mario [760] Monte Mario (Rome) [761] Montserrat 1958 [762] Moorea 87 [763] Mount Dillon [764] Moznet [765] Moznet [766] Moznet [767] NAD27 [768] NAD27 Michigan [769] NAD27(76) [770] NAD27(CGQ77) [771] NAD83 [772] NAD83(2011) [773] NAD83(2011) [774] NAD83(CORS96) [775] NAD83(CORS96) [776] NAD83(CSRS) [777] NAD83(CSRS) [778] NAD83(CSRS) [779] NAD83(CSRS)v2 [780] NAD83(CSRS)v2 [781] NAD83(CSRS)v3 [782] NAD83(CSRS)v3 [783] NAD83(CSRS)v4 [784] NAD83(CSRS)v4 [785] NAD83(CSRS)v5 [786] NAD83(CSRS)v5 [787] NAD83(CSRS)v6 [788] NAD83(CSRS)v6 [789] NAD83(CSRS)v7 [790] NAD83(CSRS)v7 [791] NAD83(CSRS96) [792] NAD83(CSRS96) [793] NAD83(CSRS98) [794] NAD83(FBN) [795] NAD83(FBN) [796] NAD83(FBN) [797] NAD83(HARN Corrected) [798] NAD83(HARN Corrected) [799] NAD83(HARN) [800] NAD83(HARN) [801] NAD83(HARN) [802] NAD83(MA11) [803] NAD83(MA11) [804] NAD83(NSRS2007) [805] NAD83(NSRS2007) [806] NAD83(PA11) [807] NAD83(PA11) [808] NDG (Paris) [809] NEA74 Noumea [810] NGN [811] NGO 1948 [812] NGO 1948 (Oslo) [813] NSWC 9Z-2 [814] NTF [815] NTF (Paris) [816] NZGD2000 [817] NZGD2000 [818] NZGD2000 [819] NZGD49 [820] Nahrwan 1934 [821] Nahrwan 1967 [822] Nakhl-e Ghanem [823] Naparima 1955 [824] Naparima 1972 [825] Nepal 1981 [826] New Beijing [827] Nord Sahara 1959 [828] Nord Sahara 1959 (Paris) [829] Nouakchott 1965 [830] ONGD14 [831] ONGD14 [832] OS(SN)80 [833] OSGB 1936 [834] OSGB70 [835] OSNI 1952 [836] Observatario [837] Ocotepeque 1935 [838] Old Hawaiian [839] PD/83 [840] PNG94 [841] PNG94 [842] POSGAR [843] POSGAR 2007 [844] POSGAR 2007 [845] POSGAR 94 [846] POSGAR 94 [847] POSGAR 98 [848] POSGAR 98 [849] POSGAR 98 [850] PRS92 [851] PRS92 [852] PSAD56 [853] PSD93 [854] PTRA08 [855] PTRA08 [856] PZ-90 [857] PZ-90 [858] PZ-90.02 [859] PZ-90.02 [860] PZ-90.11 [861] PZ-90.11 [862] Padang [863] Padang (Jakarta) [864] Palestine 1923 [865] Pampa del Castillo [866] Panama-Colon 1911 [867] Perroud 1950 [868] Peru96 [869] Peru96 [870] Petrels 1972 [871] Phoenix Islands 1966 [872] Pico de las Nieves 1984 [873] Pitcairn 1967 [874] Pitcairn 2006 [875] Point 58 [876] Pointe Noire [877] Popular Visualisation CRS [878] Porto Santo [879] Porto Santo 1995 [880] Principe [881] Puerto Rico [882] Pulkovo 1942 [883] Pulkovo 1942(58) [884] Pulkovo 1942(83) [885] Pulkovo 1995 [886] QND95 [887] Qatar 1948 [888] Qatar 1974 [889] Qornoq [890] Qornoq 1927 [891] RD/83 [892] RDN2008 [893] RDN2008 [894] REGCAN95 [895] REGCAN95 [896] REGVEN [897] REGVEN [898] REGVEN [899] RGAF09 [900] RGAF09 [901] RGAF09 (lon-lat) [902] RGAF09 (lon-lat) [903] RGF93 [904] RGF93 [905] RGF93 [906] RGF93 (lon-lat) [907] RGF93 (lon-lat) [908] RGFG95 [909] RGFG95 [910] RGFG95 [911] RGFG95 (lon-lat) [912] RGFG95 (lon-lat) [913] RGM04 [914] RGM04 [915] RGM04 (lon-lat) [916] RGM04 (lon-lat) [917] RGNC 1991 [918] RGNC 1991 [919] RGNC91-93 [920] RGNC91-93 [921] RGNC91-93 [922] RGPF [923] RGPF [924] RGR92 [925] RGR92 [926] RGR92 [927] RGR92 (lon-lat) [928] RGR92 (lon-lat) [929] RGRDC 2005 [930] RGRDC 2005 [931] RGSPM06 [932] RGSPM06 [933] RGSPM06 (lon-lat) [934] RGSPM06 (lon-lat) [935] RGTAAF07 [936] RGTAAF07 [937] RGTAAF07 (lon-lat) [938] RGTAAF07 (lon-lat) [939] RGTAAF07 (lon-lat) [940] RGWF96 [941] RGWF96 [942] RGWF96 (lon-lat) [943] RGWF96 (lon-lat) [944] RRAF 1991 [945] RRAF 1991 [946] RRAF 1991 [947] RRAF 1991 [948] RRAF 1991 [949] RSRGD2000 [950] RSRGD2000 [951] RT38 [952] RT38 (Stockholm) [953] RT90 [954] Rassadiran [955] Reunion 1947 [956] Reykjavik 1900 [957] S-JTSK [958] S-JTSK (Ferro) [959] S-JTSK [JTSK03] [960] S-JTSK/05 [961] S-JTSK/05 (Ferro) [962] SAD69 [963] SAD69 [964] SAD69(96) [965] SHGD2015 [966] SHGD2015 [967] SIGD61 [968] SIRGAS 1995 [969] SIRGAS 1995 [970] SIRGAS 1995 [971] SIRGAS 2000 [972] SIRGAS 2000 [973] SIRGAS-Chile [974] SIRGAS-Chile [975] SIRGAS-ROU98 [976] SIRGAS-ROU98 [977] SIRGAS_ES2007.8 [978] SIRGAS_ES2007.8 [979] SLD99 [980] SRB_ETRS89 [981] SRB_ETRS89 [982] SREF98 [983] SREF98 [984] ST71 Belep [985] ST84 Ile des Pins [986] ST87 Ouvea [987] ST87 Ouvea [988] SVY21 [989] SWEREF99 [990] SWEREF99 [991] SWEREF99 [992] Saint Pierre et Miquelon 1950 [993] Samboja [994] Santo 1965 [995] Sao Tome [996] Sapper Hill 1943 [997] Schwarzeck [998] Scoresbysund 1952 [999] Segara [1000] Segara (Jakarta) [1001] Segora [1002] Selvagem Grande [1003] Serindung [1004] Sibun Gorge 1922 [1005] Sierra Leone 1924 [1006] Sierra Leone 1968 [1007] Slovenia 1996 [1008] Slovenia 1996 [1009] Solomon 1968 [1010] South East Island 1943 [1011] South Georgia 1968 [1012] South Yemen [1013] St. George Island [1014] St. Helena Tritan [1015] St. Helena Tritan [1016] St. Kitts 1955 [1017] St. Lawrence Island [1018] St. Lucia 1955 [1019] St. Paul Island [1020] St. Stephen (Ferro) [1021] St. Vincent 1945 [1022] Sudan [1023] Sun_2015 [1024] TC(1948) [1025] TGD2005 [1026] TGD2005 [1027] TM65 [1028] TM75 [1029] TUREF [1030] TUREF [1031] TWD67 [1032] TWD97 [1033] TWD97 [1034] Tahaa 54 [1035] Tahiti 52 [1036] Tahiti 79 [1037] Tananarive [1038] Tananarive (Paris) [1039] Tern Island 1961 [1040] Tete [1041] Timbalai 1948 [1042] Tokyo [1043] Tokyo 1892 [1044] Trinidad 1903 [1045] Tristan 1968 [1046] UCS-2000 [1047] UCS-2000 [1048] Unknown datum based upon the Airy 1830 ellipsoid [1049] Unknown datum based upon the Airy Modified 1849 ellipsoid [1050] Unknown datum based upon the Australian National Spheroid [1051] Unknown datum based upon the Authalic Sphere [1052] Unknown datum based upon the Average Terrestrial System 1977 ellipsoid [1053] Unknown datum based upon the Bessel 1841 ellipsoid [1054] Unknown datum based upon the Bessel Modified ellipsoid [1055] Unknown datum based upon the Bessel Namibia ellipsoid [1056] Unknown datum based upon the Clarke 1858 ellipsoid [1057] Unknown datum based upon the Clarke 1866 Michigan ellipsoid [1058] Unknown datum based upon the Clarke 1866 ellipsoid [1059] Unknown datum based upon the Clarke 1880 (Arc) ellipsoid [1060] Unknown datum based upon the Clarke 1880 (Benoit) ellipsoid [1061] Unknown datum based upon the Clarke 1880 (IGN) ellipsoid [1062] Unknown datum based upon the Clarke 1880 (RGS) ellipsoid [1063] Unknown datum based upon the Clarke 1880 (SGA 1922) ellipsoid [1064] Unknown datum based upon the Clarke 1880 ellipsoid [1065] Unknown datum based upon the Everest (1830 Definition) ellipsoid [1066] Unknown datum based upon the Everest 1830 (1937 Adjustment) ellipsoid [1067] Unknown datum based upon the Everest 1830 (1962 Definition) ellipsoid [1068] Unknown datum based upon the Everest 1830 (1967 Definition) ellipsoid [1069] Unknown datum based upon the Everest 1830 (1975 Definition) ellipsoid [1070] Unknown datum based upon the Everest 1830 Modified ellipsoid [1071] Unknown datum based upon the GEM 10C ellipsoid [1072] Unknown datum based upon the GRS 1967 ellipsoid [1073] Unknown datum based upon the GRS 1980 ellipsoid [1074] Unknown datum based upon the Helmert 1906 ellipsoid [1075] Unknown datum based upon the Indonesian National Spheroid [1076] Unknown datum based upon the International 1924 ellipsoid [1077] Unknown datum based upon the Krassowsky 1940 ellipsoid [1078] Unknown datum based upon the NWL 9D ellipsoid [1079] Unknown datum based upon the OSU86F ellipsoid [1080] Unknown datum based upon the OSU91A ellipsoid [1081] Unknown datum based upon the Plessis 1817 ellipsoid [1082] Unknown datum based upon the Struve 1860 ellipsoid [1083] Unknown datum based upon the WGS 72 ellipsoid [1084] Unknown datum based upon the WGS 84 ellipsoid [1085] Unknown datum based upon the War Office ellipsoid [1086] Unspecified datum based upon the Clarke 1866 Authalic Sphere [1087] Unspecified datum based upon the GRS 1980 Authalic Sphere [1088] Unspecified datum based upon the Hughes 1980 ellipsoid [1089] Unspecified datum based upon the International 1924 Authalic Sphere [1090] VN-2000 [1091] Vanua Levu 1915 [1092] Vientiane 1982 [1093] Viti Levu 1912 [1094] Viti Levu 1916 [1095] Voirol 1875 [1096] Voirol 1875 (Paris) [1097] Voirol 1879 [1098] Voirol 1879 (Paris) [1099] WGS 66 [1100] WGS 66 [1101] WGS 72 [1102] WGS 72 [1103] WGS 72BE [1104] WGS 72BE [1105] WGS 84 [1106] WGS 84 [1107] WGS 84 [1108] WGS 84 (3D) [1109] WGS 84 (G1150) [1110] WGS 84 (G1674) [1111] WGS 84 (G1762) [1112] WGS 84 (G730) [1113] WGS 84 (G873) [1114] WGS 84 (Transit) [1115] WGS_1984_(G1150) [1116] WGS_1984_(G1674) [1117] WGS_1984_(G1762) [1118] WGS_1984_(G730) [1119] WGS_1984_(G873) [1120] WGS_1984_(Transit) [1121] Wake Island 1952 [1122] Xian 1980 [1123] Yacare [1124] Yemen NGN96 [1125] Yemen NGN96 [1126] Yemen NGN96 [1127] Yoff [1128] Zanderij [1129] fk89 Default: 0
    - CRS_PROJCS [`choice`] : Projected Coordinate Systems. Available Choices: [0] <select> [1] AGD66 / ACT Standard Grid [2] AGD66 / AMG zone 48 (deprecated) [3] AGD66 / AMG zone 49 [4] AGD66 / AMG zone 50 [5] AGD66 / AMG zone 51 [6] AGD66 / AMG zone 52 [7] AGD66 / AMG zone 53 [8] AGD66 / AMG zone 54 [9] AGD66 / AMG zone 55 [10] AGD66 / AMG zone 56 [11] AGD66 / AMG zone 57 [12] AGD66 / AMG zone 58 [13] AGD66 / Vicgrid66 [14] AGD84 / AMG zone 48 (deprecated) [15] AGD84 / AMG zone 49 [16] AGD84 / AMG zone 50 [17] AGD84 / AMG zone 51 [18] AGD84 / AMG zone 52 [19] AGD84 / AMG zone 53 [20] AGD84 / AMG zone 54 [21] AGD84 / AMG zone 55 [22] AGD84 / AMG zone 56 [23] AGD84 / AMG zone 57 (deprecated) [24] AGD84 / AMG zone 58 (deprecated) [25] AGD_1966_ACT_Grid_AGC_Zone [26] AGD_1966_ISG_54_2 [27] AGD_1966_ISG_54_3 [28] AGD_1966_ISG_55_1 [29] AGD_1966_ISG_55_2 [30] AGD_1966_ISG_55_3 [31] AGD_1966_ISG_56_1 [32] AGD_1966_ISG_56_2 [33] AGD_1966_ISG_56_3 [34] AGD_1966_VICGRID (deprecated) [35] ATF (Paris) / Nord de Guerre [36] ATS77 / MTM Nova Scotia zone 4 [37] ATS77 / MTM Nova Scotia zone 5 [38] ATS77 / New Brunswick Stereographic (ATS77) [39] ATS77 / Prince Edward Isl. Stereographic (ATS77) [40] ATS77 / UTM zone 19N [41] ATS77 / UTM zone 20N [42] Abidjan 1987 / TM 5 NW [43] Abidjan 1987 / UTM zone 29N [44] Abidjan 1987 / UTM zone 30N [45] Accra / Ghana National Grid [46] Accra / TM 1 NW [47] Adindan / UTM zone 35N [48] Adindan / UTM zone 36N [49] Adindan / UTM zone 37N [50] Adindan / UTM zone 38N [51] Afgooye / UTM zone 38N [52] Afgooye / UTM zone 39N [53] Africa_Albers_Equal_Area_Conic [54] Africa_Equidistant_Conic [55] Africa_Lambert_Conformal_Conic [56] Africa_Sinusoidal [57] Ain el Abd / Aramco Lambert [58] Ain el Abd / Bahrain Grid [59] Ain el Abd / UTM zone 36N [60] Ain el Abd / UTM zone 37N [61] Ain el Abd / UTM zone 38N [62] Ain el Abd / UTM zone 39N [63] Ain el Abd / UTM zone 40N [64] Ain_el_Abd_1970_Aramco_Lambert_2 [65] Albanian 1987 / Gauss Kruger zone 4 (deprecated) [66] Albanian 1987 / Gauss-Kruger zone 4 [67] American Samoa 1962 / American Samoa Lambert [68] American Samoa 1962 / American Samoa Lambert (deprecated) [69] American Samoa 1962 / American Samoa Lambert (deprecated) [70] American_Samoa_1962_StatePlane_American_Samoa_FIPS_5300 [71] American_Samoa_1962_UTM_Zone_2S [72] Amersfoort / RD New [73] Amersfoort / RD Old [74] Anguilla 1957 / British West Indies Grid [75] Antigua 1943 / British West Indies Grid [76] Aratu / UTM zone 22S [77] Aratu / UTM zone 23S [78] Aratu / UTM zone 24S [79] Aratu / UTM zone 25S [80] Arc 1950 / UTM zone 34S [81] Arc 1950 / UTM zone 35S [82] Arc 1950 / UTM zone 36S [83] Arc 1960 / UTM zone 35N [84] Arc 1960 / UTM zone 35S [85] Arc 1960 / UTM zone 36N [86] Arc 1960 / UTM zone 36S [87] Arc 1960 / UTM zone 37N [88] Arc 1960 / UTM zone 37S [89] Asia_Lambert_Conformal_Conic [90] Asia_North_Albers_Equal_Area_Conic [91] Asia_North_Equidistant_Conic [92] Asia_North_Lambert_Conformal_Conic [93] Asia_South_Albers_Equal_Area_Conic [94] Asia_South_Equidistant_Conic [95] Asia_South_Lambert_Conformal_Conic [96] Astro DOS 71 / SHLG71 [97] Astro DOS 71 / UTM zone 30S [98] Azores Central 1948 / UTM zone 26N [99] Azores Central 1995 / UTM zone 26N [100] Azores Occidental 1939 / UTM zone 25N [101] Azores Oriental 1940 / UTM zone 26N [102] Azores Oriental 1995 / UTM zone 26N [103] BDA2000 / Bermuda 2000 National Grid [104] BGS2005 / CCS2005 [105] BGS2005 / UTM zone 34N [106] BGS2005 / UTM zone 34N (N-E) [107] BGS2005 / UTM zone 35N [108] BGS2005 / UTM zone 35N (N-E) [109] BGS2005 / UTM zone 36N [110] BUTM 2010 [111] Bab_South_Palau_Azimuthal_Equidistant [112] Barbados 1938 / Barbados National Grid [113] Barbados 1938 / British West Indies Grid [114] Batavia (Jakarta) / NEIEZ [115] Batavia (Jakarta) / NEIEZ (deprecated) [116] Batavia / NEIEZ [117] Batavia / TM 109 SE [118] Batavia / UTM zone 48S [119] Batavia / UTM zone 49S [120] Batavia / UTM zone 50S [121] Beduaram / TM 13 NE [122] Beijing 1954 / 3-degree Gauss-Kruger CM 102E [123] Beijing 1954 / 3-degree Gauss-Kruger CM 105E [124] Beijing 1954 / 3-degree Gauss-Kruger CM 108E [125] Beijing 1954 / 3-degree Gauss-Kruger CM 111E [126] Beijing 1954 / 3-degree Gauss-Kruger CM 114E [127] Beijing 1954 / 3-degree Gauss-Kruger CM 117E [128] Beijing 1954 / 3-degree Gauss-Kruger CM 120E [129] Beijing 1954 / 3-degree Gauss-Kruger CM 123E [130] Beijing 1954 / 3-degree Gauss-Kruger CM 126E [131] Beijing 1954 / 3-degree Gauss-Kruger CM 129E [132] Beijing 1954 / 3-degree Gauss-Kruger CM 132E [133] Beijing 1954 / 3-degree Gauss-Kruger CM 135E [134] Beijing 1954 / 3-degree Gauss-Kruger CM 75E [135] Beijing 1954 / 3-degree Gauss-Kruger CM 78E [136] Beijing 1954 / 3-degree Gauss-Kruger CM 81E [137] Beijing 1954 / 3-degree Gauss-Kruger CM 84E [138] Beijing 1954 / 3-degree Gauss-Kruger CM 87E [139] Beijing 1954 / 3-degree Gauss-Kruger CM 90E [140] Beijing 1954 / 3-degree Gauss-Kruger CM 93E [141] Beijing 1954 / 3-degree Gauss-Kruger CM 96E [142] Beijing 1954 / 3-degree Gauss-Kruger CM 99E [143] Beijing 1954 / 3-degree Gauss-Kruger zone 25 [144] Beijing 1954 / 3-degree Gauss-Kruger zone 26 [145] Beijing 1954 / 3-degree Gauss-Kruger zone 27 [146] Beijing 1954 / 3-degree Gauss-Kruger zone 28 [147] Beijing 1954 / 3-degree Gauss-Kruger zone 29 [148] Beijing 1954 / 3-degree Gauss-Kruger zone 30 [149] Beijing 1954 / 3-degree Gauss-Kruger zone 31 [150] Beijing 1954 / 3-degree Gauss-Kruger zone 32 [151] Beijing 1954 / 3-degree Gauss-Kruger zone 33 [152] Beijing 1954 / 3-degree Gauss-Kruger zone 34 [153] Beijing 1954 / 3-degree Gauss-Kruger zone 35 [154] Beijing 1954 / 3-degree Gauss-Kruger zone 36 [155] Beijing 1954 / 3-degree Gauss-Kruger zone 37 [156] Beijing 1954 / 3-degree Gauss-Kruger zone 38 [157] Beijing 1954 / 3-degree Gauss-Kruger zone 39 [158] Beijing 1954 / 3-degree Gauss-Kruger zone 40 [159] Beijing 1954 / 3-degree Gauss-Kruger zone 41 [160] Beijing 1954 / 3-degree Gauss-Kruger zone 42 [161] Beijing 1954 / 3-degree Gauss-Kruger zone 43 [162] Beijing 1954 / 3-degree Gauss-Kruger zone 44 [163] Beijing 1954 / 3-degree Gauss-Kruger zone 45 [164] Beijing 1954 / Gauss-Kruger 13N (deprecated) [165] Beijing 1954 / Gauss-Kruger 14N (deprecated) [166] Beijing 1954 / Gauss-Kruger 15N (deprecated) [167] Beijing 1954 / Gauss-Kruger 16N (deprecated) [168] Beijing 1954 / Gauss-Kruger 17N (deprecated) [169] Beijing 1954 / Gauss-Kruger 18N (deprecated) [170] Beijing 1954 / Gauss-Kruger 19N (deprecated) [171] Beijing 1954 / Gauss-Kruger 20N (deprecated) [172] Beijing 1954 / Gauss-Kruger 21N (deprecated) [173] Beijing 1954 / Gauss-Kruger 22N (deprecated) [174] Beijing 1954 / Gauss-Kruger 23N (deprecated) [175] Beijing 1954 / Gauss-Kruger CM 105E [176] Beijing 1954 / Gauss-Kruger CM 111E [177] Beijing 1954 / Gauss-Kruger CM 117E [178] Beijing 1954 / Gauss-Kruger CM 123E [179] Beijing 1954 / Gauss-Kruger CM 129E [180] Beijing 1954 / Gauss-Kruger CM 135E [181] Beijing 1954 / Gauss-Kruger CM 75E [182] Beijing 1954 / Gauss-Kruger CM 81E [183] Beijing 1954 / Gauss-Kruger CM 87E [184] Beijing 1954 / Gauss-Kruger CM 93E [185] Beijing 1954 / Gauss-Kruger CM 99E [186] Beijing 1954 / Gauss-Kruger zone 13 [187] Beijing 1954 / Gauss-Kruger zone 14 [188] Beijing 1954 / Gauss-Kruger zone 15 [189] Beijing 1954 / Gauss-Kruger zone 16 [190] Beijing 1954 / Gauss-Kruger zone 17 [191] Beijing 1954 / Gauss-Kruger zone 18 [192] Beijing 1954 / Gauss-Kruger zone 19 [193] Beijing 1954 / Gauss-Kruger zone 20 [194] Beijing 1954 / Gauss-Kruger zone 21 [195] Beijing 1954 / Gauss-Kruger zone 22 [196] Beijing 1954 / Gauss-Kruger zone 23 [197] Belge 1950 (Brussels) / Belge Lambert 50 [198] Belge 1972 / Belge Lambert 72 [199] Belge 1972 / Belgian Lambert 72 [200] Belge_Lambert_1972_bad_FE_FN [201] Belge_Lambert_2008 (deprecated) [202] Bermuda 1957 / UTM zone 20N [203] Bermuda_2000_National_Grid (deprecated) [204] Bern 1898 (Bern) / LV03C [205] Bissau / UTM zone 28N [206] Bogota 1975 / Colombia Bogota zone [207] Bogota 1975 / Colombia Bogota zone (deprecated) [208] Bogota 1975 / Colombia East [209] Bogota 1975 / Colombia East (deprecated) [210] Bogota 1975 / Colombia East Central zone [211] Bogota 1975 / Colombia East Central zone (deprecated) [212] Bogota 1975 / Colombia West zone [213] Bogota 1975 / Colombia West zone (deprecated) [214] Bogota 1975 / UTM zone 17N (deprecated) [215] Bogota 1975 / UTM zone 18N [216] Bogota_Ciudad_Bogota [217] CGCS2000 / 3-degree Gauss-Kruger CM 102E [218] CGCS2000 / 3-degree Gauss-Kruger CM 105E [219] CGCS2000 / 3-degree Gauss-Kruger CM 108E [220] CGCS2000 / 3-degree Gauss-Kruger CM 111E [221] CGCS2000 / 3-degree Gauss-Kruger CM 114E [222] CGCS2000 / 3-degree Gauss-Kruger CM 117E [223] CGCS2000 / 3-degree Gauss-Kruger CM 120E [224] CGCS2000 / 3-degree Gauss-Kruger CM 123E [225] CGCS2000 / 3-degree Gauss-Kruger CM 126E [226] CGCS2000 / 3-degree Gauss-Kruger CM 129E [227] CGCS2000 / 3-degree Gauss-Kruger CM 132E [228] CGCS2000 / 3-degree Gauss-Kruger CM 135E [229] CGCS2000 / 3-degree Gauss-Kruger CM 75E [230] CGCS2000 / 3-degree Gauss-Kruger CM 78E [231] CGCS2000 / 3-degree Gauss-Kruger CM 81E [232] CGCS2000 / 3-degree Gauss-Kruger CM 84E [233] CGCS2000 / 3-degree Gauss-Kruger CM 87E [234] CGCS2000 / 3-degree Gauss-Kruger CM 90E [235] CGCS2000 / 3-degree Gauss-Kruger CM 93E [236] CGCS2000 / 3-degree Gauss-Kruger CM 96E [237] CGCS2000 / 3-degree Gauss-Kruger CM 99E [238] CGCS2000 / 3-degree Gauss-Kruger zone 25 [239] CGCS2000 / 3-degree Gauss-Kruger zone 26 [240] CGCS2000 / 3-degree Gauss-Kruger zone 27 [241] CGCS2000 / 3-degree Gauss-Kruger zone 28 [242] CGCS2000 / 3-degree Gauss-Kruger zone 29 [243] CGCS2000 / 3-degree Gauss-Kruger zone 30 [244] CGCS2000 / 3-degree Gauss-Kruger zone 31 [245] CGCS2000 / 3-degree Gauss-Kruger zone 32 [246] CGCS2000 / 3-degree Gauss-Kruger zone 33 [247] CGCS2000 / 3-degree Gauss-Kruger zone 34 [248] CGCS2000 / 3-degree Gauss-Kruger zone 35 [249] CGCS2000 / 3-degree Gauss-Kruger zone 36 [250] CGCS2000 / 3-degree Gauss-Kruger zone 37 [251] CGCS2000 / 3-degree Gauss-Kruger zone 38 [252] CGCS2000 / 3-degree Gauss-Kruger zone 39 [253] CGCS2000 / 3-degree Gauss-Kruger zone 40 [254] CGCS2000 / 3-degree Gauss-Kruger zone 41 [255] CGCS2000 / 3-degree Gauss-Kruger zone 42 [256] CGCS2000 / 3-degree Gauss-Kruger zone 43 [257] CGCS2000 / 3-degree Gauss-Kruger zone 44 [258] CGCS2000 / 3-degree Gauss-Kruger zone 45 [259] CGCS2000 / Gauss-Kruger CM 105E [260] CGCS2000 / Gauss-Kruger CM 111E [261] CGCS2000 / Gauss-Kruger CM 117E [262] CGCS2000 / Gauss-Kruger CM 123E [263] CGCS2000 / Gauss-Kruger CM 129E [264] CGCS2000 / Gauss-Kruger CM 135E [265] CGCS2000 / Gauss-Kruger CM 75E [266] CGCS2000 / Gauss-Kruger CM 81E [267] CGCS2000 / Gauss-Kruger CM 87E [268] CGCS2000 / Gauss-Kruger CM 93E [269] CGCS2000 / Gauss-Kruger CM 99E [270] CGCS2000 / Gauss-Kruger zone 13 [271] CGCS2000 / Gauss-Kruger zone 14 [272] CGCS2000 / Gauss-Kruger zone 15 [273] CGCS2000 / Gauss-Kruger zone 16 [274] CGCS2000 / Gauss-Kruger zone 17 [275] CGCS2000 / Gauss-Kruger zone 18 [276] CGCS2000 / Gauss-Kruger zone 19 [277] CGCS2000 / Gauss-Kruger zone 20 [278] CGCS2000 / Gauss-Kruger zone 21 [279] CGCS2000 / Gauss-Kruger zone 22 [280] CGCS2000 / Gauss-Kruger zone 23 [281] CGRS93 / Cyprus Local Transverse Mercator [282] CGRS_1993_LTM [283] CH1903 / LV03 [284] CH1903 / LV03C-G [285] CH1903+ / LV95 [286] CI1971 / Chatham Islands Map Grid [287] CI1979 / Chatham Islands Map Grid [288] COB_NAD83_2007 [289] CR-SIRGAS / CRTM05 [290] CR-SIRGAS / UTM zone 16N [291] CR-SIRGAS / UTM zone 17N [292] CR05 / CRTM05 [293] CRTM05 (deprecated) [294] CSG67 / UTM zone 21N [295] CSG67 / UTM zone 22N [296] Cadastre 1997 / UTM zone 38S [297] Cadastre 1997 / UTM zone 38S (deprecated) [298] Camacupa / TM 11.30 SE [299] Camacupa / TM 12 SE [300] Camacupa / UTM zone 32S [301] Camacupa / UTM zone 33S [302] Campo Inchauspe / Argentina 1 [303] Campo Inchauspe / Argentina 2 [304] Campo Inchauspe / Argentina 3 [305] Campo Inchauspe / Argentina 4 [306] Campo Inchauspe / Argentina 5 [307] Campo Inchauspe / Argentina 6 [308] Campo Inchauspe / Argentina 7 [309] Campo Inchauspe / UTM zone 19S [310] Campo Inchauspe / UTM zone 20S [311] Canada_Albers_Equal_Area_Conic [312] Canada_Lambert_Conformal_Conic [313] Cape / Lo15 [314] Cape / Lo17 [315] Cape / Lo19 [316] Cape / Lo21 [317] Cape / Lo23 [318] Cape / Lo25 [319] Cape / Lo27 [320] Cape / Lo29 [321] Cape / Lo31 [322] Cape / Lo33 [323] Cape / UTM zone 34S [324] Cape / UTM zone 35S [325] Cape / UTM zone 36S (deprecated) [326] Cape_Lo15 [327] Cape_Lo17 [328] Cape_Lo19 [329] Cape_Lo21 [330] Cape_Lo23 [331] Cape_Lo25 [332] Cape_Lo27 [333] Cape_Lo29 [334] Cape_Lo31 [335] Cape_Lo33 [336] Carthage (Paris) / Tunisia Mining Grid [337] Carthage / Nord Tunisie [338] Carthage / Sud Tunisie [339] Carthage / TM 11 NE [340] Carthage / UTM zone 32N [341] Cayman Islands National Grid 2011 [342] Cayman Islands National Grid 2011 (deprecated) [343] Chatham_Islands_1979_Map_Grid (deprecated) [344] Chos Malal 1914 / Argentina 2 [345] Chua / UTM zone 23S [346] Chua_UTM_Zone_23S (deprecated) [347] Colombia_West_West_Zone [348] Combani 1950 / UTM zone 38S [349] Conakry 1905 / UTM zone 28N [350] Conakry 1905 / UTM zone 29N [351] Corrego Alegre 1961 / UTM zone 21S [352] Corrego Alegre 1961 / UTM zone 22S [353] Corrego Alegre 1961 / UTM zone 23S [354] Corrego Alegre 1961 / UTM zone 24S [355] Corrego Alegre 1970-72 / UTM zone 21S [356] Corrego Alegre 1970-72 / UTM zone 22S [357] Corrego Alegre 1970-72 / UTM zone 23S [358] Corrego Alegre 1970-72 / UTM zone 24S [359] Corrego Alegre 1970-72 / UTM zone 25S [360] County ST74 [361] D48_Slovenia_TM [362] DB_REF / 3-degree Gauss-Kruger zone 2 (E-N) [363] DB_REF / 3-degree Gauss-Kruger zone 3 (E-N) [364] DB_REF / 3-degree Gauss-Kruger zone 4 (E-N) [365] DB_REF / 3-degree Gauss-Kruger zone 5 (E-N) [366] DGN95 / Indonesia TM-3 zone 46.2 [367] DGN95 / Indonesia TM-3 zone 47.1 [368] DGN95 / Indonesia TM-3 zone 47.2 [369] DGN95 / Indonesia TM-3 zone 48.1 [370] DGN95 / Indonesia TM-3 zone 48.2 [371] DGN95 / Indonesia TM-3 zone 49.1 [372] DGN95 / Indonesia TM-3 zone 49.2 [373] DGN95 / Indonesia TM-3 zone 50.1 [374] DGN95 / Indonesia TM-3 zone 50.2 [375] DGN95 / Indonesia TM-3 zone 51.1 [376] DGN95 / Indonesia TM-3 zone 51.2 [377] DGN95 / Indonesia TM-3 zone 52.1 [378] DGN95 / Indonesia TM-3 zone 52.2 [379] DGN95 / Indonesia TM-3 zone 53.1 [380] DGN95 / Indonesia TM-3 zone 53.2 [381] DGN95 / Indonesia TM-3 zone 54.1 [382] DGN95 / UTM zone 46N [383] DGN95 / UTM zone 47N [384] DGN95 / UTM zone 47S [385] DGN95 / UTM zone 48N [386] DGN95 / UTM zone 48S [387] DGN95 / UTM zone 49N [388] DGN95 / UTM zone 49S [389] DGN95 / UTM zone 50N [390] DGN95 / UTM zone 50S [391] DGN95 / UTM zone 51N [392] DGN95 / UTM zone 51S [393] DGN95 / UTM zone 52N [394] DGN95 / UTM zone 52S [395] DGN95 / UTM zone 53S [396] DGN95 / UTM zone 54S [397] DHDN / 3-degree Gauss zone 1 (deprecated) [398] DHDN / 3-degree Gauss zone 2 (deprecated) [399] DHDN / 3-degree Gauss zone 3 (deprecated) [400] DHDN / 3-degree Gauss zone 4 (deprecated) [401] DHDN / 3-degree Gauss zone 5 (deprecated) [402] DHDN / 3-degree Gauss-Kruger zone 1 [403] DHDN / 3-degree Gauss-Kruger zone 1 (E-N) [404] DHDN / 3-degree Gauss-Kruger zone 2 [405] DHDN / 3-degree Gauss-Kruger zone 2 (E-N) [406] DHDN / 3-degree Gauss-Kruger zone 3 [407] DHDN / 3-degree Gauss-Kruger zone 3 (E-N) [408] DHDN / 3-degree Gauss-Kruger zone 4 [409] DHDN / 3-degree Gauss-Kruger zone 4 (E-N) [410] DHDN / 3-degree Gauss-Kruger zone 5 [411] DHDN / 3-degree Gauss-Kruger zone 5 (E-N) [412] DHDN / Soldner Berlin [413] DRUKREF 03 / Bhutan National Grid [414] DRUKREF 03 / Bumthang TM [415] DRUKREF 03 / Chhukha TM [416] DRUKREF 03 / Dagana TM [417] DRUKREF 03 / Gasa TM [418] DRUKREF 03 / Ha TM [419] DRUKREF 03 / Lhuentse TM [420] DRUKREF 03 / Mongar TM [421] DRUKREF 03 / Paro TM [422] DRUKREF 03 / Pemagatshel TM [423] DRUKREF 03 / Punakha TM [424] DRUKREF 03 / Samdrup Jongkhar TM [425] DRUKREF 03 / Samtse TM [426] DRUKREF 03 / Sarpang TM [427] DRUKREF 03 / Thimphu TM [428] DRUKREF 03 / Trashigang TM [429] DRUKREF 03 / Trongsa TM [430] DRUKREF 03 / Tsirang TM [431] DRUKREF 03 / Wangdue Phodrang TM [432] DRUKREF 03 / Yangtse TM [433] DRUKREF 03 / Zhemgang TM [434] Dabola 1981 / UTM zone 28N [435] Dabola 1981 / UTM zone 28N (deprecated) [436] Dabola 1981 / UTM zone 29N [437] Dabola 1981 / UTM zone 29N (deprecated) [438] Datum 73 / Modified Portuguese Grid [439] Datum 73 / Modified Portuguese Grid (deprecated) [440] Datum 73 / UTM zone 29N [441] Datum_73_Hayford_Gauss_IGeoE [442] Datum_73_Hayford_Gauss_IPCC [443] Dealul Piscului 1930 / Stereo 33 [444] Dealul Piscului 1970/ Stereo 70 (deprecated) [445] Deir ez Zor / Levant Stereographic [446] Deir ez Zor / Levant Zone [447] Deir ez Zor / Syria Lambert [448] Dominica 1945 / British West Indies Grid [449] Douala / UTM zone 32N (deprecated) [450] Douala 1948 / AEF west [451] Douala 1948 / AOF west (deprecated) [452] ED50 / 3-degree Gauss-Kruger zone 10 [453] ED50 / 3-degree Gauss-Kruger zone 11 [454] ED50 / 3-degree Gauss-Kruger zone 12 [455] ED50 / 3-degree Gauss-Kruger zone 13 [456] ED50 / 3-degree Gauss-Kruger zone 14 [457] ED50 / 3-degree Gauss-Kruger zone 15 [458] ED50 / 3-degree Gauss-Kruger zone 9 [459] ED50 / France EuroLambert (deprecated) [460] ED50 / Iraq National Grid [461] ED50 / Jordan TM [462] ED50 / SPBA LCC [463] ED50 / TM 0 N [464] ED50 / TM 5 NE [465] ED50 / TM 6 NE [466] ED50 / TM27 [467] ED50 / TM30 [468] ED50 / TM33 [469] ED50 / TM36 [470] ED50 / TM39 [471] ED50 / TM42 [472] ED50 / TM45 [473] ED50 / UTM zone 28N [474] ED50 / UTM zone 29N [475] ED50 / UTM zone 30N [476] ED50 / UTM zone 31N [477] ED50 / UTM zone 32N [478] ED50 / UTM zone 33N [479] ED50 / UTM zone 34N [480] ED50 / UTM zone 35N [481] ED50 / UTM zone 36N [482] ED50 / UTM zone 37N [483] ED50 / UTM zone 38N [484] ED50(ED77) / UTM zone 38N [485] ED50(ED77) / UTM zone 39N [486] ED50(ED77) / UTM zone 40N [487] ED50(ED77) / UTM zone 41N [488] ED_1950_Turkey_10 [489] ED_1950_Turkey_10 (deprecated) [490] ED_1950_Turkey_11 [491] ED_1950_Turkey_11 (deprecated) [492] ED_1950_Turkey_12 [493] ED_1950_Turkey_12 (deprecated) [494] ED_1950_Turkey_13 [495] ED_1950_Turkey_13 (deprecated) [496] ED_1950_Turkey_14 [497] ED_1950_Turkey_14 (deprecated) [498] ED_1950_Turkey_15 [499] ED_1950_Turkey_15 (deprecated) [500] ED_1950_Turkey_9 [501] ED_1950_Turkey_9 (deprecated) [502] ELD79 / Libya zone 10 [503] ELD79 / Libya zone 11 [504] ELD79 / Libya zone 12 [505] ELD79 / Libya zone 13 [506] ELD79 / Libya zone 5 [507] ELD79 / Libya zone 6 [508] ELD79 / Libya zone 7 [509] ELD79 / Libya zone 8 [510] ELD79 / Libya zone 9 [511] ELD79 / TM 12 NE [512] ELD79 / UTM zone 32N [513] ELD79 / UTM zone 33N [514] ELD79 / UTM zone 34N [515] ELD79 / UTM zone 35N [516] EMEP_150_Kilometer_Grid [517] EMEP_50_Kilometer_Grid [518] ETRF_1989_TM_Baltic_1993 (deprecated) [519] ETRF_1989_UTM_Zone_28N (deprecated) [520] ETRF_1989_UTM_Zone_29N (deprecated) [521] ETRF_1989_UTM_Zone_30N (deprecated) [522] ETRF_1989_UTM_Zone_31N (deprecated) [523] ETRF_1989_UTM_Zone_32N (deprecated) [524] ETRF_1989_UTM_Zone_33N (deprecated) [525] ETRF_1989_UTM_Zone_34N (deprecated) [526] ETRF_1989_UTM_Zone_35N (deprecated) [527] ETRF_1989_UTM_Zone_36N (deprecated) [528] ETRF_1989_UTM_Zone_37N (deprecated) [529] ETRF_1989_UTM_Zone_38N (deprecated) [530] ETRS89 / Albania LCC 2010 [531] ETRS89 / Albania TM 2010 [532] ETRS89 / Austria Lambert [533] ETRS89 / Belgian Lambert 2005 [534] ETRS89 / Belgian Lambert 2008 [535] ETRS89 / DKTM1 [536] ETRS89 / DKTM2 [537] ETRS89 / DKTM3 [538] ETRS89 / DKTM4 [539] ETRS89 / EPSG Arctic zone 2-22 [540] ETRS89 / EPSG Arctic zone 3-11 [541] ETRS89 / EPSG Arctic zone 4-26 [542] ETRS89 / EPSG Arctic zone 4-28 [543] ETRS89 / EPSG Arctic zone 5-11 [544] ETRS89 / EPSG Arctic zone 5-13 [545] ETRS89 / EPSG Arctic zone 5-47 [546] ETRS89 / ETRS-GK19FIN [547] ETRS89 / ETRS-GK20FIN [548] ETRS89 / ETRS-GK21FIN [549] ETRS89 / ETRS-GK22FIN [550] ETRS89 / ETRS-GK23FIN [551] ETRS89 / ETRS-GK24FIN [552] ETRS89 / ETRS-GK25FIN [553] ETRS89 / ETRS-GK26FIN [554] ETRS89 / ETRS-GK27FIN [555] ETRS89 / ETRS-GK28FIN [556] ETRS89 / ETRS-GK29FIN [557] ETRS89 / ETRS-GK30FIN [558] ETRS89 / ETRS-GK31FIN [559] ETRS89 / Faroe TM [560] ETRS89 / GK19FIN [561] ETRS89 / GK20FIN [562] ETRS89 / GK21FIN [563] ETRS89 / GK22FIN [564] ETRS89 / GK23FIN [565] ETRS89 / GK24FIN [566] ETRS89 / GK25FIN [567] ETRS89 / GK26FIN [568] ETRS89 / GK27FIN [569] ETRS89 / GK28FIN [570] ETRS89 / GK29FIN [571] ETRS89 / GK30FIN [572] ETRS89 / GK31FIN [573] ETRS89 / Gauss-Kruger CM 9E [574] ETRS89 / Guernsey Grid [575] ETRS89 / Jersey Transverse Mercator [576] ETRS89 / Kp2000 Bornholm [577] ETRS89 / Kp2000 Jutland [578] ETRS89 / Kp2000 Zealand [579] ETRS89 / LAEA Europe [580] ETRS89 / LCC Europe [581] ETRS89 / LCC Germany (E-N) [582] ETRS89 / LCC Germany (N-E) [583] ETRS89 / NTM zone 10 [584] ETRS89 / NTM zone 10 (deprecated) [585] ETRS89 / NTM zone 11 [586] ETRS89 / NTM zone 11 (deprecated) [587] ETRS89 / NTM zone 12 [588] ETRS89 / NTM zone 12 (deprecated) [589] ETRS89 / NTM zone 13 [590] ETRS89 / NTM zone 13 (deprecated) [591] ETRS89 / NTM zone 14 [592] ETRS89 / NTM zone 14 (deprecated) [593] ETRS89 / NTM zone 15 [594] ETRS89 / NTM zone 15 (deprecated) [595] ETRS89 / NTM zone 16 [596] ETRS89 / NTM zone 16 (deprecated) [597] ETRS89 / NTM zone 17 [598] ETRS89 / NTM zone 17 (deprecated) [599] ETRS89 / NTM zone 18 [600] ETRS89 / NTM zone 18 (deprecated) [601] ETRS89 / NTM zone 19 [602] ETRS89 / NTM zone 19 (deprecated) [603] ETRS89 / NTM zone 20 [604] ETRS89 / NTM zone 20 (deprecated) [605] ETRS89 / NTM zone 21 [606] ETRS89 / NTM zone 21 (deprecated) [607] ETRS89 / NTM zone 22 [608] ETRS89 / NTM zone 22 (deprecated) [609] ETRS89 / NTM zone 23 [610] ETRS89 / NTM zone 23 (deprecated) [611] ETRS89 / NTM zone 24 [612] ETRS89 / NTM zone 24 (deprecated) [613] ETRS89 / NTM zone 25 [614] ETRS89 / NTM zone 25 (deprecated) [615] ETRS89 / NTM zone 26 [616] ETRS89 / NTM zone 26 (deprecated) [617] ETRS89 / NTM zone 27 [618] ETRS89 / NTM zone 27 (deprecated) [619] ETRS89 / NTM zone 28 [620] ETRS89 / NTM zone 28 (deprecated) [621] ETRS89 / NTM zone 29 [622] ETRS89 / NTM zone 29 (deprecated) [623] ETRS89 / NTM zone 30 [624] ETRS89 / NTM zone 30 (deprecated) [625] ETRS89 / NTM zone 5 [626] ETRS89 / NTM zone 5 (deprecated) [627] ETRS89 / NTM zone 6 [628] ETRS89 / NTM zone 6 (deprecated) [629] ETRS89 / NTM zone 7 [630] ETRS89 / NTM zone 7 (deprecated) [631] ETRS89 / NTM zone 8 [632] ETRS89 / NTM zone 8 (deprecated) [633] ETRS89 / NTM zone 9 [634] ETRS89 / NTM zone 9 (deprecated) [635] ETRS89 / Poland CS2000 zone 5 [636] ETRS89 / Poland CS2000 zone 6 [637] ETRS89 / Poland CS2000 zone 7 [638] ETRS89 / Poland CS2000 zone 8 [639] ETRS89 / Poland CS92 [640] ETRS89 / Portugal TM06 [641] ETRS89 / TM 30 NE [642] ETRS89 / TM Baltic93 [643] ETRS89 / TM26 (deprecated) [644] ETRS89 / TM27 (deprecated) [645] ETRS89 / TM35FIN(E,N) [646] ETRS89 / TM35FIN(N,E) [647] ETRS89 / TM38 (deprecated) [648] ETRS89 / TM39 (deprecated) [649] ETRS89 / UTM zone 28N [650] ETRS89 / UTM zone 28N (N-E) [651] ETRS89 / UTM zone 29N [652] ETRS89 / UTM zone 29N (N-E) [653] ETRS89 / UTM zone 30N [654] ETRS89 / UTM zone 30N (N-E) [655] ETRS89 / UTM zone 31N [656] ETRS89 / UTM zone 31N (N-E) [657] ETRS89 / UTM zone 31N (N-zE) [658] ETRS89 / UTM zone 31N (zE-N) [659] ETRS89 / UTM zone 32N [660] ETRS89 / UTM zone 32N (N-E) [661] ETRS89 / UTM zone 32N (N-zE) [662] ETRS89 / UTM zone 32N (zE-N) [663] ETRS89 / UTM zone 33N [664] ETRS89 / UTM zone 33N (N-E) [665] ETRS89 / UTM zone 33N (N-zE) [666] ETRS89 / UTM zone 33N (zE-N) [667] ETRS89 / UTM zone 34N [668] ETRS89 / UTM zone 34N (N-E) [669] ETRS89 / UTM zone 35N [670] ETRS89 / UTM zone 35N (N-E) [671] ETRS89 / UTM zone 36N [672] ETRS89 / UTM zone 36N (N-E) [673] ETRS89 / UTM zone 37N [674] ETRS89 / UTM zone 37N (N-E) [675] ETRS89 / UTM zone 38N (deprecated) [676] ETRS_1989_DKTM1 (deprecated) [677] ETRS_1989_DKTM2 (deprecated) [678] ETRS_1989_DKTM3 (deprecated) [679] ETRS_1989_DKTM4 (deprecated) [680] ETRS_1989_Kosovo_Grid [681] ETRS_1989_Slovenia_TM [682] ETRS_1989_UTM_Zone_26N [683] ETRS_1989_UTM_Zone_27N [684] ETRS_1989_UTM_Zone_32N_7stellen [685] ETRS_1989_UTM_Zone_32N_8stellen [686] ETRS_1989_UTM_Zone_33N_7stellen [687] ETRS_1989_UTM_Zone_33N_8stellen [688] ETRS_1989_UTM_Zone_39N [689] ETRS_1989_UTM_Zone_N32 (deprecated) [690] ETRS_1989_UWPP_1992 [691] ETRS_1989_UWPP_2000_PAS_5 [692] ETRS_1989_UWPP_2000_PAS_6 [693] ETRS_1989_UWPP_2000_PAS_7 [694] ETRS_1989_UWPP_2000_PAS_8 [695] EUREF_FIN_TM35FIN (deprecated) [696] Egypt 1907 / Blue Belt [697] Egypt 1907 / Extended Purple Belt [698] Egypt 1907 / Purple Belt [699] Egypt 1907 / Red Belt [700] Egypt Gulf of Suez S-650 TL / Red Belt [701] Estonian Coordinate System of 1992 [702] Estonian Coordinate System of 1997 [703] Europe_Albers_Equal_Area_Conic [704] Europe_Equidistant_Conic [705] Europe_Lambert_Conformal_Conic [706] Everest_Modified_1969_RSO_Malaya_Meters [707] FD54 / UTM zone 29N [708] FD58 / Iraq zone [709] FEH2010 / Fehmarnbelt TM [710] Fahud / UTM zone 39N [711] Fahud / UTM zone 40N [712] Fatu Iva 72 / UTM zone 7S [713] Fiji 1956 / UTM zone 1S [714] Fiji 1956 / UTM zone 60S [715] Fiji 1986 / Fiji Map Grid [716] Fiji 1986 / Fiji Map Grid (deprecated) [717] Fort Marigot / UTM zone 20N [718] GDA2020 / ALB2020 [719] GDA2020 / BCG2020 [720] GDA2020 / BIO2020 [721] GDA2020 / BRO2020 [722] GDA2020 / CARN2020 [723] GDA2020 / CIG2020 [724] GDA2020 / CKIG2020 [725] GDA2020 / COL2020 [726] GDA2020 / ESP2020 [727] GDA2020 / EXM2020 [728] GDA2020 / GA LCC [729] GDA2020 / GCG2020 [730] GDA2020 / GOLD2020 [731] GDA2020 / JCG2020 [732] GDA2020 / KALB2020 [733] GDA2020 / KAR2020 [734] GDA2020 / KUN2020 [735] GDA2020 / LCG2020 [736] GDA2020 / MGA zone 46 [737] GDA2020 / MGA zone 47 [738] GDA2020 / MGA zone 48 [739] GDA2020 / MGA zone 49 [740] GDA2020 / MGA zone 50 [741] GDA2020 / MGA zone 51 [742] GDA2020 / MGA zone 52 [743] GDA2020 / MGA zone 53 [744] GDA2020 / MGA zone 54 [745] GDA2020 / MGA zone 55 [746] GDA2020 / MGA zone 56 [747] GDA2020 / MGA zone 57 [748] GDA2020 / MGA zone 58 [749] GDA2020 / MGA zone 59 [750] GDA2020 / MRCG2020 [751] GDA2020 / NSW Lambert [752] GDA2020 / PCG2020 [753] GDA2020 / PHG2020 [754] GDA2020 / SA Lambert [755] GDA2020 / Vicgrid [756] GDA94 / Australian Albers [757] GDA94 / BCSG02 [758] GDA94 / CIG94 [759] GDA94 / CKIG94 [760] GDA94 / Geoscience Australia Lambert [761] GDA94 / MGA zone 41 (deprecated) [762] GDA94 / MGA zone 42 (deprecated) [763] GDA94 / MGA zone 43 (deprecated) [764] GDA94 / MGA zone 44 (deprecated) [765] GDA94 / MGA zone 46 [766] GDA94 / MGA zone 47 [767] GDA94 / MGA zone 48 [768] GDA94 / MGA zone 49 [769] GDA94 / MGA zone 50 [770] GDA94 / MGA zone 51 [771] GDA94 / MGA zone 52 [772] GDA94 / MGA zone 53 [773] GDA94 / MGA zone 54 [774] GDA94 / MGA zone 55 [775] GDA94 / MGA zone 56 [776] GDA94 / MGA zone 57 [777] GDA94 / MGA zone 58 [778] GDA94 / MGA zone 59 [779] GDA94 / NSW Lambert [780] GDA94 / SA Lambert [781] GDA94 / Vicgrid [782] GDA94 / WEIPA94 [783] GDA_1994_Perth_Coastal_Grid_1994 [784] GDA_1994_South_Australia_Lambert (deprecated) [785] GDA_1994_VICGRID94 (deprecated) [786] GDBD2009 / Brunei BRSO [787] GDBD2009_GEORSO (deprecated) [788] GDM2000 / East Malaysia BRSO [789] GDM2000 / Johor Grid [790] GDM2000 / Kedah and Perlis Grid [791] GDM2000 / Kelantan Grid [792] GDM2000 / Pahang Grid [793] GDM2000 / Peninsula RSO [794] GDM2000 / Perak Grid [795] GDM2000 / Pinang Grid [796] GDM2000 / Selangor Grid [797] GDM2000 / Sembilan and Melaka Grid [798] GDM2000 / Terengganu Grid [799] GGRS87 / Greek Grid [800] GOES-16_East_ABI_Fixed_Grid_ITRF2000_incorrect_GCS (deprecated) [801] GOES-16_East_ABI_Fixed_Grid_ITRF2008 [802] GR96 / EPSG Arctic zone 1-25 [803] GR96 / EPSG Arctic zone 2-18 [804] GR96 / EPSG Arctic zone 2-20 [805] GR96 / EPSG Arctic zone 3-29 [806] GR96 / EPSG Arctic zone 3-31 [807] GR96 / EPSG Arctic zone 3-33 [808] GR96 / EPSG Arctic zone 4-20 [809] GR96 / EPSG Arctic zone 4-22 [810] GR96 / EPSG Arctic zone 4-24 [811] GR96 / EPSG Arctic zone 5-41 [812] GR96 / EPSG Arctic zone 5-43 [813] GR96 / EPSG Arctic zone 5-45 [814] GR96 / EPSG Arctic zone 6-26 [815] GR96 / EPSG Arctic zone 6-28 [816] GR96 / EPSG Arctic zone 6-30 [817] GR96 / EPSG Arctic zone 7-11 [818] GR96 / EPSG Arctic zone 7-13 [819] GR96 / EPSG Arctic zone 8-20 [820] GR96 / EPSG Arctic zone 8-22 [821] GR96 / UTM zone 18N [822] GR96 / UTM zone 19N [823] GR96 / UTM zone 20N [824] GR96 / UTM zone 21N [825] GR96 / UTM zone 22N [826] GR96 / UTM zone 23N [827] GR96 / UTM zone 24N [828] GR96 / UTM zone 25N [829] GR96 / UTM zone 26N [830] GR96 / UTM zone 27N [831] GR96 / UTM zone 28N [832] GR96 / UTM zone 29N [833] Garoua / UTM zone 33N [834] Garoua / UTM zone 33N (deprecated) [835] Germany_Zone_1 [836] Germany_Zone_2 [837] Germany_Zone_3 [838] Germany_Zone_4 [839] Germany_Zone_5 [840] Graciosa_Base_SW_1948_UTM_Zone_26N [841] Grand Cayman 1959 / UTM zone 17N (deprecated) [842] Grand Cayman National Grid 1959 [843] Grand Comoros / UTM zone 38S [844] Grenada 1953 / British West Indies Grid [845] Guadeloupe 1948 / UTM zone 20N [846] Guam 1963 / Yap Islands [847] Guam_Geodetic_Network_1993 [848] Guam_Geodetic_Triangulation_Network_1963 [849] Guernsey_Grid [850] Gulshan 303 / Bangladesh Transverse Mercator [851] Gusterberg Grid (Ferro) [852] HD72 / EOV [853] HTRS96 / Croatia LCC [854] HTRS96 / Croatia TM [855] HTRS96 / UTM zone 33N [856] HTRS96 / UTM zone 34N [857] Hanoi 1972 / GK 106 NE [858] Hanoi 1972 / Gauss-Kruger zone 18 [859] Hanoi 1972 / Gauss-Kruger zone 19 [860] Hartebeesthoek94 / Lo15 [861] Hartebeesthoek94 / Lo17 [862] Hartebeesthoek94 / Lo19 [863] Hartebeesthoek94 / Lo21 [864] Hartebeesthoek94 / Lo23 [865] Hartebeesthoek94 / Lo25 [866] Hartebeesthoek94 / Lo27 [867] Hartebeesthoek94 / Lo29 [868] Hartebeesthoek94 / Lo31 [869] Hartebeesthoek94 / Lo33 [870] Hartebeesthoek94_Lo15 [871] Hartebeesthoek94_Lo17 [872] Hartebeesthoek94_Lo19 [873] Hartebeesthoek94_Lo19_(E-N) [874] Hartebeesthoek94_Lo21 [875] Hartebeesthoek94_Lo21_(E-N) [876] Hartebeesthoek94_Lo23 [877] Hartebeesthoek94_Lo23_(E-N) [878] Hartebeesthoek94_Lo25 [879] Hartebeesthoek94_Lo25_(E-N) [880] Hartebeesthoek94_Lo27 [881] Hartebeesthoek94_Lo27_(E-N) [882] Hartebeesthoek94_Lo29 [883] Hartebeesthoek94_Lo29_(E-N) [884] Hartebeesthoek94_Lo31 [885] Hartebeesthoek94_Lo31_(E-N) [886] Hartebeesthoek94_Lo33 [887] Hawaii_Albers_Equal_Area_Conic [888] Helle 1954 / Jan Mayen Grid [889] Hito XVIII 1963 / Argentina 2 [890] Hito XVIII 1963 / UTM zone 19S [891] Hjorsey 1955 / UTM zone 26N [892] Hjorsey 1955 / UTM zone 27N [893] Hjorsey 1955 / UTM zone 28N [894] Hong Kong 1963 Grid System [895] Hong Kong 1963 Grid System (deprecated) [896] Hong Kong 1980 Grid System [897] Hong_Kong_1980_Grid (deprecated) [898] Hong_Kong_1980_UTM_Zone_49N [899] Hong_Kong_1980_UTM_Zone_50N [900] Hu Tzu Shan 1950 / UTM zone 51N [901] ID74 / UTM zone 46N [902] ID74 / UTM zone 46S (deprecated) [903] ID74 / UTM zone 47N [904] ID74 / UTM zone 47S [905] ID74 / UTM zone 48N [906] ID74 / UTM zone 48S [907] ID74 / UTM zone 49N [908] ID74 / UTM zone 49S [909] ID74 / UTM zone 50N [910] ID74 / UTM zone 50S [911] ID74 / UTM zone 51N [912] ID74 / UTM zone 51S [913] ID74 / UTM zone 52N [914] ID74 / UTM zone 52S [915] ID74 / UTM zone 53N (deprecated) [916] ID74 / UTM zone 53S [917] ID74 / UTM zone 54S [918] IGC 1962 / Congo TM zone 12 [919] IGC 1962 / Congo TM zone 14 [920] IGC 1962 / Congo TM zone 16 [921] IGC 1962 / Congo TM zone 18 [922] IGC 1962 / Congo TM zone 20 [923] IGC 1962 / Congo TM zone 22 [924] IGC 1962 / Congo TM zone 24 [925] IGC 1962 / Congo TM zone 26 [926] IGC 1962 / Congo TM zone 28 [927] IGC 1962 / Congo TM zone 30 [928] IGCB 1955 / Congo TM zone 12 [929] IGCB 1955 / Congo TM zone 14 [930] IGCB 1955 / Congo TM zone 16 [931] IGCB 1955 / UTM zone 33S [932] IGM95 / UTM zone 32N [933] IGM95 / UTM zone 33N [934] IGN 1962 Kerguelen / UTM zone 42S [935] IGN Astro 1960 / UTM zone 28N [936] IGN Astro 1960 / UTM zone 29N [937] IGN Astro 1960 / UTM zone 30N [938] IGN53 Mare / UTM zone 58S [939] IGN53 Mare / UTM zone 59S [940] IGN56 Lifou / UTM zone 58S [941] IGN63 Hiva Oa / UTM zone 7S [942] IGN72 Grand Terre / UTM zone 58S (deprecated) [943] IGN72 Grande Terre / UTM zone 58S [944] IGN72 Nuku Hiva / UTM zone 7S [945] IGRS / UTM zone 37N [946] IGRS / UTM zone 38N [947] IGRS / UTM zone 39N [948] IRENET95 / Irish Transverse Mercator [949] IRENET95 / UTM zone 29N [950] ISN2004 / LAEA Europe [951] ISN2004 / LCC Europe [952] ISN2004 / Lambert 2004 [953] ISN2016 / Lambert 2016 [954] ISN93 / Lambert 1993 [955] ISN_2004_Lambert_2004 (deprecated) [956] Indian 1954 / UTM zone 46N [957] Indian 1954 / UTM zone 47N [958] Indian 1954 / UTM zone 48N [959] Indian 1960 / TM 106 NE [960] Indian 1960 / UTM zone 48N [961] Indian 1960 / UTM zone 49N [962] Indian 1975 / UTM zone 47N [963] Indian 1975 / UTM zone 48N [964] Israel 1993 / Israeli TM Grid [965] Israeli Grid 05 [966] Israeli Grid 05/12 [967] JAD2001 / Jamaica Metric Grid [968] JAD2001 / UTM zone 17N [969] JAD2001 / UTM zone 18N [970] JAD69 / Jamaica National Grid [971] JAD_2001_Jamaica_Grid (deprecated) [972] JAXA Snow Depth Polar Stereographic North [973] JGD2000 / Japan Plane Rectangular CS I [974] JGD2000 / Japan Plane Rectangular CS II [975] JGD2000 / Japan Plane Rectangular CS III [976] JGD2000 / Japan Plane Rectangular CS IV [977] JGD2000 / Japan Plane Rectangular CS IX [978] JGD2000 / Japan Plane Rectangular CS V [979] JGD2000 / Japan Plane Rectangular CS VI [980] JGD2000 / Japan Plane Rectangular CS VII [981] JGD2000 / Japan Plane Rectangular CS VIII [982] JGD2000 / Japan Plane Rectangular CS X [983] JGD2000 / Japan Plane Rectangular CS XI [984] JGD2000 / Japan Plane Rectangular CS XII [985] JGD2000 / Japan Plane Rectangular CS XIII [986] JGD2000 / Japan Plane Rectangular CS XIV [987] JGD2000 / Japan Plane Rectangular CS XIX [988] JGD2000 / Japan Plane Rectangular CS XV [989] JGD2000 / Japan Plane Rectangular CS XVI [990] JGD2000 / Japan Plane Rectangular CS XVII [991] JGD2000 / Japan Plane Rectangular CS XVIII [992] JGD2000 / UTM zone 51N [993] JGD2000 / UTM zone 52N [994] JGD2000 / UTM zone 53N [995] JGD2000 / UTM zone 54N [996] JGD2000 / UTM zone 55N [997] JGD2011 / Japan Plane Rectangular CS I [998] JGD2011 / Japan Plane Rectangular CS II [999] JGD2011 / Japan Plane Rectangular CS III [1000] JGD2011 / Japan Plane Rectangular CS IV [1001] JGD2011 / Japan Plane Rectangular CS IX [1002] JGD2011 / Japan Plane Rectangular CS V [1003] JGD2011 / Japan Plane Rectangular CS VI [1004] JGD2011 / Japan Plane Rectangular CS VII [1005] JGD2011 / Japan Plane Rectangular CS VIII [1006] JGD2011 / Japan Plane Rectangular CS X [1007] JGD2011 / Japan Plane Rectangular CS XI [1008] JGD2011 / Japan Plane Rectangular CS XII [1009] JGD2011 / Japan Plane Rectangular CS XIII [1010] JGD2011 / Japan Plane Rectangular CS XIV [1011] JGD2011 / Japan Plane Rectangular CS XIX [1012] JGD2011 / Japan Plane Rectangular CS XV [1013] JGD2011 / Japan Plane Rectangular CS XVI [1014] JGD2011 / Japan Plane Rectangular CS XVII [1015] JGD2011 / Japan Plane Rectangular CS XVIII [1016] JGD2011 / UTM zone 51N [1017] JGD2011 / UTM zone 52N [1018] JGD2011 / UTM zone 53N [1019] JGD2011 / UTM zone 54N [1020] JGD2011 / UTM zone 55N [1021] JGD_2000_UTM_Zone_51N (deprecated) [1022] JGD_2000_UTM_Zone_52N (deprecated) [1023] JGD_2000_UTM_Zone_53N (deprecated) [1024] JGD_2000_UTM_Zone_54N (deprecated) [1025] JGD_2000_UTM_Zone_55N (deprecated) [1026] JGD_2000_UTM_Zone_56N [1027] JGD_2011_Japan_Zone_1 (deprecated) [1028] JGD_2011_Japan_Zone_10 (deprecated) [1029] JGD_2011_Japan_Zone_11 (deprecated) [1030] JGD_2011_Japan_Zone_12 (deprecated) [1031] JGD_2011_Japan_Zone_13 (deprecated) [1032] JGD_2011_Japan_Zone_14 (deprecated) [1033] JGD_2011_Japan_Zone_15 (deprecated) [1034] JGD_2011_Japan_Zone_16 (deprecated) [1035] JGD_2011_Japan_Zone_17 (deprecated) [1036] JGD_2011_Japan_Zone_18 (deprecated) [1037] JGD_2011_Japan_Zone_19 (deprecated) [1038] JGD_2011_Japan_Zone_2 (deprecated) [1039] JGD_2011_Japan_Zone_3 (deprecated) [1040] JGD_2011_Japan_Zone_4 (deprecated) [1041] JGD_2011_Japan_Zone_5 (deprecated) [1042] JGD_2011_Japan_Zone_6 (deprecated) [1043] JGD_2011_Japan_Zone_7 (deprecated) [1044] JGD_2011_Japan_Zone_8 (deprecated) [1045] JGD_2011_Japan_Zone_9 (deprecated) [1046] JGD_2011_UTM_Zone_51N (deprecated) [1047] JGD_2011_UTM_Zone_52N (deprecated) [1048] JGD_2011_UTM_Zone_53N (deprecated) [1049] JGD_2011_UTM_Zone_54N (deprecated) [1050] JGD_2011_UTM_Zone_55N (deprecated) [1051] JGD_2011_UTM_Zone_56N [1052] Jamaica 1875 / Jamaica (Old Grid) [1053] Jordan_JTM [1054] K0 1949 / UTM zone 42S (deprecated) [1055] KKJ / Finland Uniform Coordinate System [1056] KKJ / Finland zone 0 [1057] KKJ / Finland zone 1 [1058] KKJ / Finland zone 2 [1059] KKJ / Finland zone 4 [1060] KKJ / Finland zone 5 [1061] KOC Lambert [1062] KUDAMS / KTM [1063] KUDAMS / KTM (deprecated) [1064] Kalianpur 1880 / India zone 0 [1065] Kalianpur 1880 / India zone I [1066] Kalianpur 1880 / India zone IIIa [1067] Kalianpur 1880 / India zone IIa [1068] Kalianpur 1880 / India zone IIb [1069] Kalianpur 1880 / India zone IVa [1070] Kalianpur 1937 / India zone IIb [1071] Kalianpur 1937 / UTM zone 45N [1072] Kalianpur 1937 / UTM zone 46N [1073] Kalianpur 1962 / India zone I [1074] Kalianpur 1962 / India zone IIa [1075] Kalianpur 1962 / UTM zone 41N [1076] Kalianpur 1962 / UTM zone 42N [1077] Kalianpur 1962 / UTM zone 43N [1078] Kalianpur 1975 / India zone I [1079] Kalianpur 1975 / India zone IIIa [1080] Kalianpur 1975 / India zone IIa [1081] Kalianpur 1975 / India zone IIb [1082] Kalianpur 1975 / India zone IVa [1083] Kalianpur 1975 / UTM zone 42N [1084] Kalianpur 1975 / UTM zone 43N [1085] Kalianpur 1975 / UTM zone 44N [1086] Kalianpur 1975 / UTM zone 45N [1087] Kalianpur 1975 / UTM zone 46N [1088] Kalianpur 1975 / UTM zone 47N [1089] Kandawala / Sri Lanka Grid [1090] Kandawala_Ceylon_Belt_Indian_Yards_1937 [1091] Kandawala_Ceylon_Belt_Meters [1092] Karbala 1979 / Iraq National Grid [1093] Karbala 1979 / UTM zone 37N [1094] Karbala 1979 / UTM zone 38N [1095] Karbala 1979 / UTM zone 39N [1096] Kasai 1953 / Congo TM zone 22 [1097] Kasai 1953 / Congo TM zone 24 [1098] Katanga 1955 / Katanga Gauss zone A [1099] Katanga 1955 / Katanga Gauss zone B [1100] Katanga 1955 / Katanga Gauss zone C [1101] Katanga 1955 / Katanga Gauss zone D [1102] Katanga 1955 / Katanga Lambert [1103] Katanga 1955 / Katanga Lambert (deprecated) [1104] Katanga 1955 / Katanga Lambert (deprecated) [1105] Katanga 1955 / Katanga TM (deprecated) [1106] Katanga_1955_Katanga_Lambert (deprecated) [1107] Kertau (RSO) / RSO Malaya (ch) [1108] Kertau (RSO) / RSO Malaya (m) [1109] Kertau / R.S.O. Malaya (ch) (deprecated) [1110] Kertau 1968 / Johor Grid [1111] Kertau 1968 / Kedah and Perlis Grid [1112] Kertau 1968 / Kelantan Grid [1113] Kertau 1968 / Pahang Grid [1114] Kertau 1968 / Perak Revised Grid [1115] Kertau 1968 / Pinang Grid [1116] Kertau 1968 / Selangor Grid [1117] Kertau 1968 / Sembilan and Melaka Grid [1118] Kertau 1968 / Singapore Grid [1119] Kertau 1968 / Terengganu Grid [1120] Kertau 1968 / UTM zone 47N [1121] Kertau 1968 / UTM zone 48N [1122] Kertau_RSO_Malaya_Meters [1123] Korea 2000 / Central Belt [1124] Korea 2000 / Central Belt 2010 [1125] Korea 2000 / Central Belt Jeju [1126] Korea 2000 / East Belt [1127] Korea 2000 / East Belt 2010 [1128] Korea 2000 / East Sea Belt [1129] Korea 2000 / East Sea Belt 2010 [1130] Korea 2000 / Unified CS [1131] Korea 2000 / West Belt [1132] Korea 2000 / West Belt 2010 [1133] Korea_2000_Korea_Central_Belt_2010 (deprecated) [1134] Korea_2000_Korea_East_Belt_2010 (deprecated) [1135] Korea_2000_Korea_East_Sea_Belt_2010 (deprecated) [1136] Korea_2000_Korea_Unified_Coordinate_System (deprecated) [1137] Korea_2000_Korea_West_Belt_2010 (deprecated) [1138] Korean 1985 / Central Belt [1139] Korean 1985 / Central Belt Jeju [1140] Korean 1985 / East Belt [1141] Korean 1985 / East Sea Belt [1142] Korean 1985 / Modified Central Belt [1143] Korean 1985 / Modified Central Belt Jeju [1144] Korean 1985 / Modified East Belt [1145] Korean 1985 / Modified East Sea Belt [1146] Korean 1985 / Modified West Belt [1147] Korean 1985 / Unified CS [1148] Korean 1985 / West Belt [1149] Korean_1985_Korea_Unified_Coordinate_System (deprecated) [1150] Korean_1985_Modified_Korea_Central_Belt (deprecated) [1151] Korean_1985_Modified_Korea_Central_Belt_Jeju (deprecated) [1152] Korean_1985_Modified_Korea_East_Belt (deprecated) [1153] Korean_1985_Modified_Korea_East_Sea_Belt (deprecated) [1154] Korean_1985_Modified_Korea_West_Belt (deprecated) [1155] Kousseri / UTM zone 33N [1156] Kyrg-06 / zone 1 [1157] Kyrg-06 / zone 2 [1158] Kyrg-06 / zone 3 [1159] Kyrg-06 / zone 4 [1160] Kyrg-06 / zone 5 [1161] Kyrg-06_TM_Zone_1 [1162] Kyrg-06_TM_Zone_2 [1163] Kyrg-06_TM_Zone_3 [1164] Kyrg-06_TM_Zone_4 [1165] Kyrg-06_TM_Zone_5 [1166] LGD2006 / Libya TM [1167] LGD2006 / Libya TM zone 10 [1168] LGD2006 / Libya TM zone 11 [1169] LGD2006 / Libya TM zone 12 [1170] LGD2006 / Libya TM zone 13 [1171] LGD2006 / Libya TM zone 5 [1172] LGD2006 / Libya TM zone 6 [1173] LGD2006 / Libya TM zone 7 [1174] LGD2006 / Libya TM zone 8 [1175] LGD2006 / Libya TM zone 9 [1176] LGD2006 / UTM zone 32N [1177] LGD2006 / UTM zone 33N [1178] LGD2006 / UTM zone 34N [1179] LGD2006 / UTM zone 35N [1180] LKS92 / Latvia TM [1181] LKS94 / Lithuania TM [1182] LKS_1992_Latvia_TM_0 [1183] La Canoa / UTM zone 18N [1184] La Canoa / UTM zone 19N [1185] La Canoa / UTM zone 20N [1186] La_Canoa_UTM_Zone_21N [1187] Lake / Maracaibo Grid [1188] Lake / Maracaibo Grid M1 [1189] Lake / Maracaibo Grid M3 [1190] Lake / Maracaibo La Rosa Grid [1191] Le Pouce 1934 / Mauritius Grid [1192] Leigon / Ghana Metre Grid [1193] Lietuvos Koordinoei Sistema 1994 (deprecated) [1194] Lisboa_Bessel_Bonne [1195] Lisboa_Hayford_Gauss_IGeoE [1196] Lisboa_Hayford_Gauss_IPCC [1197] Lisbon (Lisbon) / Portuguese Grid [1198] Lisbon (Lisbon) / Portuguese National Grid [1199] Lisbon / Portuguese Grid New [1200] Little Cayman 1961 / UTM zone 17N (deprecated) [1201] Locodjo 1965 / TM 5 NW [1202] Locodjo 1965 / UTM zone 29N [1203] Locodjo 1965 / UTM zone 30N [1204] Lome / UTM zone 31N [1205] London_Survey_Grid [1206] Luxembourg 1930 / Gauss [1207] Luzon 1911 / Philippines zone I [1208] Luzon 1911 / Philippines zone II [1209] Luzon 1911 / Philippines zone III [1210] Luzon 1911 / Philippines zone IV [1211] Luzon 1911 / Philippines zone V [1212] Luzon_1911_UTM_Zone_50N [1213] Luzon_1911_UTM_Zone_51N [1214] Luzon_1911_UTM_Zone_52N [1215] M'poraloko / UTM zone 32N [1216] M'poraloko / UTM zone 32S [1217] MAGNA-SIRGAS / Colombia Bogota zone [1218] MAGNA-SIRGAS / Colombia East Central zone [1219] MAGNA-SIRGAS / Colombia East zone [1220] MAGNA-SIRGAS / Colombia Far West zone [1221] MAGNA-SIRGAS / Colombia West zone [1222] MAGNA_Ciudad_Bogota [1223] MARGEN / UTM zone 19S [1224] MARGEN / UTM zone 20S [1225] MARGEN / UTM zone 21S [1226] MGI (Ferro) / Austria Central Zone [1227] MGI (Ferro) / Austria Central Zone (deprecated) [1228] MGI (Ferro) / Austria East Zone [1229] MGI (Ferro) / Austria East Zone (deprecated) [1230] MGI (Ferro) / Austria GK Central Zone [1231] MGI (Ferro) / Austria GK East Zone [1232] MGI (Ferro) / Austria GK West Zone [1233] MGI (Ferro) / Austria West Zone [1234] MGI (Ferro) / Austria West Zone (deprecated) [1235] MGI (Ferro) / M28 [1236] MGI (Ferro) / M31 [1237] MGI (Ferro) / M34 [1238] MGI / 3-degree Gauss zone 5 (deprecated) [1239] MGI / 3-degree Gauss zone 6 (deprecated) [1240] MGI / 3-degree Gauss zone 7 (deprecated) [1241] MGI / 3-degree Gauss zone 8 (deprecated) [1242] MGI / Austria GK Central [1243] MGI / Austria GK East [1244] MGI / Austria GK M28 [1245] MGI / Austria GK M31 [1246] MGI / Austria GK M34 [1247] MGI / Austria GK West [1248] MGI / Austria Lambert [1249] MGI / Austria Lambert (deprecated) [1250] MGI / Austria M28 [1251] MGI / Austria M31 [1252] MGI / Austria M34 [1253] MGI / Balkans zone 5 (deprecated) [1254] MGI / Balkans zone 6 (deprecated) [1255] MGI / Balkans zone 7 (deprecated) [1256] MGI / Balkans zone 8 (deprecated) [1257] MGI / Balkans zone 8 (deprecated) [1258] MGI / M28 (deprecated) [1259] MGI / M31 (deprecated) [1260] MGI / M34 (deprecated) [1261] MGI / Slovene National Grid (deprecated) [1262] MGI / Slovenia Grid (deprecated) [1263] MGI 1901 / Balkans zone 5 [1264] MGI 1901 / Balkans zone 5 [1265] MGI 1901 / Balkans zone 6 [1266] MGI 1901 / Balkans zone 6 [1267] MGI 1901 / Balkans zone 7 [1268] MGI 1901 / Balkans zone 8 [1269] MGI 1901 / Balkans zone 8 [1270] MGI 1901 / Slovene National Grid [1271] MGI 1901 / Slovenia Grid [1272] MGI 1901 / Slovenia Grid [1273] MOLDREF99 / Moldova TM [1274] MONREF_1997_UTM_Zone_46N [1275] MONREF_1997_UTM_Zone_47N [1276] MONREF_1997_UTM_Zone_48N [1277] MONREF_1997_UTM_Zone_49N [1278] MONREF_1997_UTM_Zone_50N [1279] MOP78 / UTM zone 1S [1280] MTRF-2000 / UTM zone 36N [1281] MTRF-2000 / UTM zone 37N [1282] MTRF-2000 / UTM zone 38N [1283] MTRF-2000 / UTM zone 39N [1284] MTRF-2000 / UTM zone 40N [1285] Macao 1920 / Macao Grid [1286] Macao_2008_Macao_Grid [1287] Macedonia State Coordinate System [1288] Macedonia State Coordinate System zone 7 [1289] Madeira 1936 / UTM zone 28N (deprecated) [1290] Madrid 1870 (Madrid) / Spain [1291] Makassar (Jakarta) / NEIEZ [1292] Makassar (Jakarta) / NEIEZ (deprecated) [1293] Makassar / NEIEZ [1294] Malongo 1987 / UTM zone 32S [1295] Malongo 1987 / UTM zone 33S [1296] Manoca 1962 / UTM zone 32N [1297] Martinique 1938 / UTM zone 20N [1298] Massawa / UTM zone 37N [1299] Maupiti 83 / UTM zone 5S [1300] Mauritania 1999 / UTM zone 28N [1301] Mauritania 1999 / UTM zone 28N (deprecated) [1302] Mauritania 1999 / UTM zone 29N [1303] Mauritania 1999 / UTM zone 29N (deprecated) [1304] Mauritania 1999 / UTM zone 30N [1305] Mauritania 1999 / UTM zone 30N (deprecated) [1306] Merchich / Nord Maroc [1307] Merchich / Sahara (deprecated) [1308] Merchich / Sahara Nord [1309] Merchich / Sahara Sud [1310] Merchich / Sud Maroc [1311] Merchich_Degree_UTM_Zone_28N [1312] Mexican_Datum_1993_UTM_Zone_11N (deprecated) [1313] Mexican_Datum_1993_UTM_Zone_12N (deprecated) [1314] Mexican_Datum_1993_UTM_Zone_13N (deprecated) [1315] Mexican_Datum_1993_UTM_Zone_14N (deprecated) [1316] Mexican_Datum_1993_UTM_Zone_15N (deprecated) [1317] Mexican_Datum_1993_UTM_Zone_16N (deprecated) [1318] Mexico ITRF2008 / LCC [1319] Mexico ITRF2008 / UTM zone 11N [1320] Mexico ITRF2008 / UTM zone 12N [1321] Mexico ITRF2008 / UTM zone 13N [1322] Mexico ITRF2008 / UTM zone 14N [1323] Mexico ITRF2008 / UTM zone 15N [1324] Mexico ITRF2008 / UTM zone 16N [1325] Mexico ITRF92 / LCC [1326] Mexico ITRF92 / UTM zone 11N [1327] Mexico ITRF92 / UTM zone 12N [1328] Mexico ITRF92 / UTM zone 13N [1329] Mexico ITRF92 / UTM zone 14N [1330] Mexico ITRF92 / UTM zone 15N [1331] Mexico ITRF92 / UTM zone 16N [1332] Mhast (offshore) / UTM zone 32S [1333] Mhast (onshore) / UTM zone 32S [1334] Mhast / UTM zone 32S (deprecated) [1335] Minna / Nigeria East Belt [1336] Minna / Nigeria Mid Belt [1337] Minna / Nigeria West Belt [1338] Minna / UTM zone 31N [1339] Minna / UTM zone 32N [1340] Monte Mario (Rome) / Italy zone 1 (deprecated) [1341] Monte Mario (Rome) / Italy zone 2 (deprecated) [1342] Monte Mario / Italy zone 1 [1343] Monte Mario / Italy zone 2 [1344] Monte Mario / TM Emilia-Romagna [1345] Monte_Mario_Italy_1 (deprecated) [1346] Monte_Mario_Italy_2 (deprecated) [1347] Montserrat 1958 / British West Indies Grid [1348] Moorea 87 / UTM zone 6S [1349] Mount Dillon / Tobago Grid [1350] Moznet / UTM zone 36S [1351] Moznet / UTM zone 37S [1352] Moznet / UTM zone 38S [1353] NAD Michigan / Michigan Central (deprecated) [1354] NAD Michigan / Michigan East (deprecated) [1355] NAD Michigan / Michigan North (deprecated) [1356] NAD Michigan / Michigan Old Central (deprecated) [1357] NAD Michigan / Michigan South (deprecated) [1358] NAD Michigan / Michigan West (deprecated) [1359] NAD27 / Alabama East [1360] NAD27 / Alabama West [1361] NAD27 / Alaska Albers [1362] NAD27 / Alaska zone 1 [1363] NAD27 / Alaska zone 10 [1364] NAD27 / Alaska zone 2 [1365] NAD27 / Alaska zone 3 [1366] NAD27 / Alaska zone 4 [1367] NAD27 / Alaska zone 5 [1368] NAD27 / Alaska zone 6 [1369] NAD27 / Alaska zone 7 [1370] NAD27 / Alaska zone 8 [1371] NAD27 / Alaska zone 9 [1372] NAD27 / Alberta 3TM ref merid 111 W [1373] NAD27 / Alberta 3TM ref merid 114 W [1374] NAD27 / Alberta 3TM ref merid 117 W [1375] NAD27 / Alberta 3TM ref merid 120 W [1376] NAD27 / Alberta 3TM ref merid 120 W (deprecated) [1377] NAD27 / Arizona Central [1378] NAD27 / Arizona East [1379] NAD27 / Arizona West [1380] NAD27 / Arkansas North [1381] NAD27 / Arkansas South [1382] NAD27 / BLM 10N (ftUS) [1383] NAD27 / BLM 11N (ftUS) [1384] NAD27 / BLM 12N (ftUS) [1385] NAD27 / BLM 13N (ftUS) [1386] NAD27 / BLM 14N (feet) (deprecated) [1387] NAD27 / BLM 14N (ftUS) [1388] NAD27 / BLM 15N (feet) (deprecated) [1389] NAD27 / BLM 15N (ftUS) [1390] NAD27 / BLM 16N (feet) (deprecated) [1391] NAD27 / BLM 16N (ftUS) [1392] NAD27 / BLM 17N (feet) (deprecated) [1393] NAD27 / BLM 17N (ftUS) [1394] NAD27 / BLM 18N (ftUS) [1395] NAD27 / BLM 19N (ftUS) [1396] NAD27 / BLM 1N (ftUS) [1397] NAD27 / BLM 2N (ftUS) [1398] NAD27 / BLM 3N (ftUS) [1399] NAD27 / BLM 4N (ftUS) [1400] NAD27 / BLM 59N (ftUS) [1401] NAD27 / BLM 5N (ftUS) [1402] NAD27 / BLM 60N (ftUS) [1403] NAD27 / BLM 6N (ftUS) [1404] NAD27 / BLM 7N (ftUS) [1405] NAD27 / BLM 8N (ftUS) [1406] NAD27 / BLM 9N (ftUS) [1407] NAD27 / California Albers [1408] NAD27 / California zone I [1409] NAD27 / California zone II [1410] NAD27 / California zone III [1411] NAD27 / California zone IV [1412] NAD27 / California zone V [1413] NAD27 / California zone VI [1414] NAD27 / California zone VII [1415] NAD27 / California zone VII (deprecated) [1416] NAD27 / Colorado Central [1417] NAD27 / Colorado North [1418] NAD27 / Colorado South [1419] NAD27 / Connecticut [1420] NAD27 / Conus Albers [1421] NAD27 / Cuba Norte [1422] NAD27 / Cuba Norte (deprecated) [1423] NAD27 / Cuba Sur [1424] NAD27 / Cuba Sur (deprecated) [1425] NAD27 / Delaware [1426] NAD27 / Florida East [1427] NAD27 / Florida North [1428] NAD27 / Florida West [1429] NAD27 / Georgia East [1430] NAD27 / Georgia West [1431] NAD27 / Guatemala Norte (deprecated) [1432] NAD27 / Guatemala Sur (deprecated) [1433] NAD27 / Idaho Central [1434] NAD27 / Idaho East [1435] NAD27 / Idaho West [1436] NAD27 / Illinois East [1437] NAD27 / Illinois West [1438] NAD27 / Indiana East [1439] NAD27 / Indiana West [1440] NAD27 / Iowa North [1441] NAD27 / Iowa South [1442] NAD27 / Kansas North [1443] NAD27 / Kansas South [1444] NAD27 / Kentucky North [1445] NAD27 / Kentucky South [1446] NAD27 / Louisiana North [1447] NAD27 / Louisiana Offshore [1448] NAD27 / Louisiana South [1449] NAD27 / MTM zone 1 [1450] NAD27 / MTM zone 10 [1451] NAD27 / MTM zone 2 [1452] NAD27 / MTM zone 3 [1453] NAD27 / MTM zone 4 [1454] NAD27 / MTM zone 5 [1455] NAD27 / MTM zone 6 [1456] NAD27 / MTQ Lambert [1457] NAD27 / Maine East [1458] NAD27 / Maine West [1459] NAD27 / Maryland [1460] NAD27 / Massachusetts Island [1461] NAD27 / Massachusetts Mainland [1462] NAD27 / Michigan East [1463] NAD27 / Michigan Old Central [1464] NAD27 / Michigan West [1465] NAD27 / Minnesota Central [1466] NAD27 / Minnesota North [1467] NAD27 / Minnesota South [1468] NAD27 / Mississippi East [1469] NAD27 / Mississippi West [1470] NAD27 / Missouri Central [1471] NAD27 / Missouri East [1472] NAD27 / Missouri West [1473] NAD27 / Montana Central [1474] NAD27 / Montana North [1475] NAD27 / Montana South [1476] NAD27 / Nebraska North [1477] NAD27 / Nebraska South [1478] NAD27 / Nevada Central [1479] NAD27 / Nevada East [1480] NAD27 / Nevada West [1481] NAD27 / New Brunswick Stereographic (NAD27) [1482] NAD27 / New Hampshire [1483] NAD27 / New Jersey [1484] NAD27 / New Mexico Central [1485] NAD27 / New Mexico East [1486] NAD27 / New Mexico West [1487] NAD27 / New York Central [1488] NAD27 / New York East [1489] NAD27 / New York Long Island [1490] NAD27 / New York Long Island (deprecated) [1491] NAD27 / New York West [1492] NAD27 / North Carolina [1493] NAD27 / North Dakota North [1494] NAD27 / North Dakota South [1495] NAD27 / Ohio North [1496] NAD27 / Ohio South [1497] NAD27 / Oklahoma North [1498] NAD27 / Oklahoma South [1499] NAD27 / Oregon North [1500] NAD27 / Oregon South [1501] NAD27 / Pennsylvania North [1502] NAD27 / Pennsylvania South [1503] NAD27 / Pennsylvania South (deprecated) [1504] NAD27 / Quebec Lambert [1505] NAD27 / Rhode Island [1506] NAD27 / Shackleford [1507] NAD27 / South Carolina North [1508] NAD27 / South Carolina South [1509] NAD27 / South Dakota North [1510] NAD27 / South Dakota South [1511] NAD27 / Tennessee [1512] NAD27 / Tennessee (deprecated) [1513] NAD27 / Texas Central [1514] NAD27 / Texas North [1515] NAD27 / Texas North Central [1516] NAD27 / Texas South [1517] NAD27 / Texas South Central [1518] NAD27 / UTM zone 10N [1519] NAD27 / UTM zone 11N [1520] NAD27 / UTM zone 12N [1521] NAD27 / UTM zone 13N [1522] NAD27 / UTM zone 14N [1523] NAD27 / UTM zone 15N [1524] NAD27 / UTM zone 16N [1525] NAD27 / UTM zone 17N [1526] NAD27 / UTM zone 18N [1527] NAD27 / UTM zone 19N [1528] NAD27 / UTM zone 1N [1529] NAD27 / UTM zone 20N [1530] NAD27 / UTM zone 21N [1531] NAD27 / UTM zone 22N [1532] NAD27 / UTM zone 2N [1533] NAD27 / UTM zone 3N [1534] NAD27 / UTM zone 4N [1535] NAD27 / UTM zone 59N [1536] NAD27 / UTM zone 5N [1537] NAD27 / UTM zone 60N [1538] NAD27 / UTM zone 6N [1539] NAD27 / UTM zone 7N [1540] NAD27 / UTM zone 8N [1541] NAD27 / UTM zone 9N [1542] NAD27 / Utah Central [1543] NAD27 / Utah North [1544] NAD27 / Utah South [1545] NAD27 / Vermont [1546] NAD27 / Virginia North [1547] NAD27 / Virginia South [1548] NAD27 / Washington North [1549] NAD27 / Washington South [1550] NAD27 / West Virginia North [1551] NAD27 / West Virginia South [1552] NAD27 / Wisconsin Central [1553] NAD27 / Wisconsin North [1554] NAD27 / Wisconsin South [1555] NAD27 / Wisconsin Transverse Mercator [1556] NAD27 / Wyoming East [1557] NAD27 / Wyoming East Central [1558] NAD27 / Wyoming West [1559] NAD27 / Wyoming West Central [1560] NAD27(76) / MTM zone 10 [1561] NAD27(76) / MTM zone 11 [1562] NAD27(76) / MTM zone 12 [1563] NAD27(76) / MTM zone 13 [1564] NAD27(76) / MTM zone 14 [1565] NAD27(76) / MTM zone 15 [1566] NAD27(76) / MTM zone 16 [1567] NAD27(76) / MTM zone 17 [1568] NAD27(76) / MTM zone 8 [1569] NAD27(76) / MTM zone 9 [1570] NAD27(76) / UTM zone 15N [1571] NAD27(76) / UTM zone 16N [1572] NAD27(76) / UTM zone 17N [1573] NAD27(76) / UTM zone 18N [1574] NAD27(CGQ77) / Quebec Lambert [1575] NAD27(CGQ77) / SCoPQ zone 10 [1576] NAD27(CGQ77) / SCoPQ zone 2 (deprecated) [1577] NAD27(CGQ77) / SCoPQ zone 3 [1578] NAD27(CGQ77) / SCoPQ zone 4 [1579] NAD27(CGQ77) / SCoPQ zone 5 [1580] NAD27(CGQ77) / SCoPQ zone 6 [1581] NAD27(CGQ77) / SCoPQ zone 7 [1582] NAD27(CGQ77) / SCoPQ zone 8 [1583] NAD27(CGQ77) / SCoPQ zone 9 [1584] NAD27(CGQ77) / UTM zone 17N [1585] NAD27(CGQ77) / UTM zone 18N [1586] NAD27(CGQ77) / UTM zone 19N [1587] NAD27(CGQ77) / UTM zone 20N [1588] NAD27(CGQ77) / UTM zone 21N [1589] NAD83 / Alabama East [1590] NAD83 / Alabama West [1591] NAD83 / Alaska Albers [1592] NAD83 / Alaska zone 1 [1593] NAD83 / Alaska zone 10 [1594] NAD83 / Alaska zone 2 [1595] NAD83 / Alaska zone 3 [1596] NAD83 / Alaska zone 4 [1597] NAD83 / Alaska zone 5 [1598] NAD83 / Alaska zone 6 [1599] NAD83 / Alaska zone 7 [1600] NAD83 / Alaska zone 8 [1601] NAD83 / Alaska zone 9 [1602] NAD83 / Alberta 10-TM (Forest) [1603] NAD83 / Alberta 10-TM (Resource) [1604] NAD83 / Alberta 3TM ref merid 111 W [1605] NAD83 / Alberta 3TM ref merid 114 W [1606] NAD83 / Alberta 3TM ref merid 117 W [1607] NAD83 / Alberta 3TM ref merid 120 W [1608] NAD83 / Alberta 3TM ref merid 120 W (deprecated) [1609] NAD83 / Arizona Central [1610] NAD83 / Arizona Central (ft) [1611] NAD83 / Arizona East [1612] NAD83 / Arizona East (ft) [1613] NAD83 / Arizona West [1614] NAD83 / Arizona West (ft) [1615] NAD83 / Arkansas North [1616] NAD83 / Arkansas North (ftUS) [1617] NAD83 / Arkansas South [1618] NAD83 / Arkansas South (ftUS) [1619] NAD83 / BC Albers [1620] NAD83 / BLM 10N (ftUS) [1621] NAD83 / BLM 11N (ftUS) [1622] NAD83 / BLM 12N (ftUS) [1623] NAD83 / BLM 13N (ftUS) [1624] NAD83 / BLM 14N (ftUS) [1625] NAD83 / BLM 15N (ftUS) [1626] NAD83 / BLM 16N (ftUS) [1627] NAD83 / BLM 17N (ftUS) [1628] NAD83 / BLM 18N (ftUS) [1629] NAD83 / BLM 19N (ftUS) [1630] NAD83 / BLM 1N (ftUS) [1631] NAD83 / BLM 2N (ftUS) [1632] NAD83 / BLM 3N (ftUS) [1633] NAD83 / BLM 4N (ftUS) [1634] NAD83 / BLM 59N (ftUS) [1635] NAD83 / BLM 5N (ftUS) [1636] NAD83 / BLM 60N (ftUS) [1637] NAD83 / BLM 6N (ftUS) [1638] NAD83 / BLM 7N (ftUS) [1639] NAD83 / BLM 8N (ftUS) [1640] NAD83 / BLM 9N (ftUS) [1641] NAD83 / California Albers [1642] NAD83 / California zone 1 [1643] NAD83 / California zone 1 (ftUS) [1644] NAD83 / California zone 2 [1645] NAD83 / California zone 2 (ftUS) [1646] NAD83 / California zone 3 [1647] NAD83 / California zone 3 (ftUS) [1648] NAD83 / California zone 4 [1649] NAD83 / California zone 4 (ftUS) [1650] NAD83 / California zone 5 [1651] NAD83 / California zone 5 (ftUS) [1652] NAD83 / California zone 6 [1653] NAD83 / California zone 6 (ftUS) [1654] NAD83 / Canada Atlas Lambert [1655] NAD83 / Colorado Central [1656] NAD83 / Colorado Central (ftUS) [1657] NAD83 / Colorado North [1658] NAD83 / Colorado North (ftUS) [1659] NAD83 / Colorado South [1660] NAD83 / Colorado South (ftUS) [1661] NAD83 / Connecticut [1662] NAD83 / Connecticut (ftUS) [1663] NAD83 / Conus Albers [1664] NAD83 / Delaware [1665] NAD83 / Delaware (ftUS) [1666] NAD83 / Florida East [1667] NAD83 / Florida East (ftUS) [1668] NAD83 / Florida GDL Albers [1669] NAD83 / Florida North [1670] NAD83 / Florida North (ftUS) [1671] NAD83 / Florida West [1672] NAD83 / Florida West (ftUS) [1673] NAD83 / Georgia East [1674] NAD83 / Georgia East (ftUS) [1675] NAD83 / Georgia West [1676] NAD83 / Georgia West (ftUS) [1677] NAD83 / Great Lakes Albers [1678] NAD83 / Great Lakes and St Lawrence Albers [1679] NAD83 / Hawaii zone 1 [1680] NAD83 / Hawaii zone 2 [1681] NAD83 / Hawaii zone 3 [1682] NAD83 / Hawaii zone 3 (ftUS) [1683] NAD83 / Hawaii zone 4 [1684] NAD83 / Hawaii zone 5 [1685] NAD83 / Idaho Central [1686] NAD83 / Idaho Central (ftUS) [1687] NAD83 / Idaho East [1688] NAD83 / Idaho East (ftUS) [1689] NAD83 / Idaho Transverse Mercator [1690] NAD83 / Idaho West [1691] NAD83 / Idaho West (ftUS) [1692] NAD83 / Illinois East [1693] NAD83 / Illinois East (ftUS) [1694] NAD83 / Illinois West [1695] NAD83 / Illinois West (ftUS) [1696] NAD83 / Indiana East [1697] NAD83 / Indiana East (ftUS) [1698] NAD83 / Indiana East (ftUS) (deprecated) [1699] NAD83 / Indiana West [1700] NAD83 / Indiana West (ftUS) [1701] NAD83 / Indiana West (ftUS) (deprecated) [1702] NAD83 / Iowa North [1703] NAD83 / Iowa North (ftUS) [1704] NAD83 / Iowa South [1705] NAD83 / Iowa South (ftUS) [1706] NAD83 / Kansas LCC [1707] NAD83 / Kansas LCC (ftUS) [1708] NAD83 / Kansas North [1709] NAD83 / Kansas North (ftUS) [1710] NAD83 / Kansas South [1711] NAD83 / Kansas South (ftUS) [1712] NAD83 / Kentucky North [1713] NAD83 / Kentucky North (deprecated) [1714] NAD83 / Kentucky North (ftUS) [1715] NAD83 / Kentucky Single Zone [1716] NAD83 / Kentucky Single Zone (ftUS) [1717] NAD83 / Kentucky South [1718] NAD83 / Kentucky South (ftUS) [1719] NAD83 / Louisiana North [1720] NAD83 / Louisiana North (ftUS) [1721] NAD83 / Louisiana Offshore [1722] NAD83 / Louisiana Offshore (ftUS) [1723] NAD83 / Louisiana South [1724] NAD83 / Louisiana South (ftUS) [1725] NAD83 / MTM zone 1 [1726] NAD83 / MTM zone 10 [1727] NAD83 / MTM zone 11 [1728] NAD83 / MTM zone 12 [1729] NAD83 / MTM zone 13 [1730] NAD83 / MTM zone 14 [1731] NAD83 / MTM zone 15 [1732] NAD83 / MTM zone 16 [1733] NAD83 / MTM zone 17 [1734] NAD83 / MTM zone 2 [1735] NAD83 / MTM zone 3 [1736] NAD83 / MTM zone 4 [1737] NAD83 / MTM zone 5 [1738] NAD83 / MTM zone 6 [1739] NAD83 / MTM zone 7 [1740] NAD83 / MTM zone 8 [1741] NAD83 / MTM zone 9 [1742] NAD83 / MTQ Lambert [1743] NAD83 / Maine CS2000 Central [1744] NAD83 / Maine CS2000 Central (deprecated) [1745] NAD83 / Maine CS2000 East [1746] NAD83 / Maine CS2000 West [1747] NAD83 / Maine East [1748] NAD83 / Maine East (ftUS) [1749] NAD83 / Maine East (ftUS) (deprecated) [1750] NAD83 / Maine West [1751] NAD83 / Maine West (ftUS) [1752] NAD83 / Maine West (ftUS) (deprecated) [1753] NAD83 / Maryland [1754] NAD83 / Maryland (ftUS) [1755] NAD83 / Massachusetts Island [1756] NAD83 / Massachusetts Island (ftUS) [1757] NAD83 / Massachusetts Mainland [1758] NAD83 / Massachusetts Mainland (ftUS) [1759] NAD83 / Michigan Central [1760] NAD83 / Michigan Central (ft) [1761] NAD83 / Michigan North [1762] NAD83 / Michigan North (ft) [1763] NAD83 / Michigan Oblique Mercator [1764] NAD83 / Michigan South [1765] NAD83 / Michigan South (ft) [1766] NAD83 / Minnesota Central [1767] NAD83 / Minnesota Central (ftUS) [1768] NAD83 / Minnesota Central (ftUS) (deprecated) [1769] NAD83 / Minnesota North [1770] NAD83 / Minnesota North (ftUS) [1771] NAD83 / Minnesota North (ftUS) (deprecated) [1772] NAD83 / Minnesota South [1773] NAD83 / Minnesota South (ftUS) [1774] NAD83 / Minnesota South (ftUS) (deprecated) [1775] NAD83 / Mississippi East [1776] NAD83 / Mississippi East (ftUS) [1777] NAD83 / Mississippi TM [1778] NAD83 / Mississippi West [1779] NAD83 / Mississippi West (ftUS) [1780] NAD83 / Missouri Central [1781] NAD83 / Missouri East [1782] NAD83 / Missouri West [1783] NAD83 / Montana [1784] NAD83 / Montana (ft) [1785] NAD83 / NCRS Las Vegas (ftUS) [1786] NAD83 / NCRS Las Vegas (m) [1787] NAD83 / NCRS Las Vegas high (ftUS) [1788] NAD83 / NCRS Las Vegas high (m) [1789] NAD83 / NWT Lambert [1790] NAD83 / Nebraska [1791] NAD83 / Nebraska (ftUS) [1792] NAD83 / Nebraska (ftUS) (deprecated) [1793] NAD83 / Nevada Central [1794] NAD83 / Nevada Central (ftUS) [1795] NAD83 / Nevada East [1796] NAD83 / Nevada East (ftUS) [1797] NAD83 / Nevada West [1798] NAD83 / Nevada West (ftUS) [1799] NAD83 / New Hampshire [1800] NAD83 / New Hampshire (ftUS) [1801] NAD83 / New Jersey [1802] NAD83 / New Jersey (ftUS) [1803] NAD83 / New Mexico Central [1804] NAD83 / New Mexico Central (ftUS) [1805] NAD83 / New Mexico East [1806] NAD83 / New Mexico East (ftUS) [1807] NAD83 / New Mexico West [1808] NAD83 / New Mexico West (ftUS) [1809] NAD83 / New York Central [1810] NAD83 / New York Central (ftUS) [1811] NAD83 / New York East [1812] NAD83 / New York East (ftUS) [1813] NAD83 / New York Long Island [1814] NAD83 / New York Long Island (ftUS) [1815] NAD83 / New York West [1816] NAD83 / New York West (ftUS) [1817] NAD83 / North Carolina [1818] NAD83 / North Carolina (ftUS) [1819] NAD83 / North Dakota North [1820] NAD83 / North Dakota North (ft) [1821] NAD83 / North Dakota South [1822] NAD83 / North Dakota South (ft) [1823] NAD83 / Ohio North [1824] NAD83 / Ohio North (ftUS) [1825] NAD83 / Ohio South [1826] NAD83 / Ohio South (ftUS) [1827] NAD83 / Oklahoma North [1828] NAD83 / Oklahoma North (ftUS) [1829] NAD83 / Oklahoma South [1830] NAD83 / Oklahoma South (ftUS) [1831] NAD83 / Ontario MNR Lambert [1832] NAD83 / Oregon GIC Lambert (ft) [1833] NAD83 / Oregon LCC (m) [1834] NAD83 / Oregon North [1835] NAD83 / Oregon North (ft) [1836] NAD83 / Oregon South [1837] NAD83 / Oregon South (ft) [1838] NAD83 / Pennsylvania North [1839] NAD83 / Pennsylvania North (ftUS) [1840] NAD83 / Pennsylvania South [1841] NAD83 / Pennsylvania South (ftUS) [1842] NAD83 / Puerto Rico & Virgin Is. [1843] NAD83 / Quebec Albers [1844] NAD83 / Quebec Lambert [1845] NAD83 / Rhode Island [1846] NAD83 / Rhode Island (ftUS) [1847] NAD83 / SCoPQ zone 2 (deprecated) [1848] NAD83 / South Carolina [1849] NAD83 / South Carolina (ft) [1850] NAD83 / South Dakota North [1851] NAD83 / South Dakota North (ftUS) [1852] NAD83 / South Dakota North (ftUS) (deprecated) [1853] NAD83 / South Dakota South [1854] NAD83 / South Dakota South (ftUS) [1855] NAD83 / Statistics Canada Lambert [1856] NAD83 / Tennessee [1857] NAD83 / Tennessee (ftUS) [1858] NAD83 / Teranet Ontario Lambert [1859] NAD83 / Texas Central [1860] NAD83 / Texas Central (ftUS) [1861] NAD83 / Texas Centric Albers Equal Area [1862] NAD83 / Texas Centric Lambert Conformal [1863] NAD83 / Texas North [1864] NAD83 / Texas North (ftUS) [1865] NAD83 / Texas North Central [1866] NAD83 / Texas North Central (ftUS) [1867] NAD83 / Texas South [1868] NAD83 / Texas South (ftUS) [1869] NAD83 / Texas South Central [1870] NAD83 / Texas South Central (ftUS) [1871] NAD83 / Texas State Mapping System [1872] NAD83 / UTM zone 10N [1873] NAD83 / UTM zone 11N [1874] NAD83 / UTM zone 12N [1875] NAD83 / UTM zone 13N [1876] NAD83 / UTM zone 14N [1877] NAD83 / UTM zone 15N [1878] NAD83 / UTM zone 16N [1879] NAD83 / UTM zone 17N [1880] NAD83 / UTM zone 18N [1881] NAD83 / UTM zone 19N [1882] NAD83 / UTM zone 1N [1883] NAD83 / UTM zone 20N [1884] NAD83 / UTM zone 21N [1885] NAD83 / UTM zone 22N [1886] NAD83 / UTM zone 23N [1887] NAD83 / UTM zone 2N [1888] NAD83 / UTM zone 3N [1889] NAD83 / UTM zone 4N [1890] NAD83 / UTM zone 59N [1891] NAD83 / UTM zone 5N [1892] NAD83 / UTM zone 60N [1893] NAD83 / UTM zone 6N [1894] NAD83 / UTM zone 7N [1895] NAD83 / UTM zone 8N [1896] NAD83 / UTM zone 9N [1897] NAD83 / Utah Central [1898] NAD83 / Utah Central (ft) [1899] NAD83 / Utah Central (ftUS) [1900] NAD83 / Utah North [1901] NAD83 / Utah North (ft) [1902] NAD83 / Utah North (ftUS) [1903] NAD83 / Utah South [1904] NAD83 / Utah South (ft) [1905] NAD83 / Utah South (ftUS) [1906] NAD83 / Vermont [1907] NAD83 / Vermont (ftUS) [1908] NAD83 / Virginia Lambert [1909] NAD83 / Virginia North [1910] NAD83 / Virginia North (ftUS) [1911] NAD83 / Virginia South [1912] NAD83 / Virginia South (ftUS) [1913] NAD83 / Washington North [1914] NAD83 / Washington North (ftUS) [1915] NAD83 / Washington South [1916] NAD83 / Washington South (ftUS) [1917] NAD83 / West Virginia North [1918] NAD83 / West Virginia North (ftUS) [1919] NAD83 / West Virginia North (ftUS) (deprecated) [1920] NAD83 / West Virginia South [1921] NAD83 / West Virginia South (ftUS) [1922] NAD83 / West Virginia South (ftUS) (deprecated) [1923] NAD83 / Wisconsin Central [1924] NAD83 / Wisconsin Central (ftUS) [1925] NAD83 / Wisconsin North [1926] NAD83 / Wisconsin North (ftUS) [1927] NAD83 / Wisconsin South [1928] NAD83 / Wisconsin South (ftUS) [1929] NAD83 / Wisconsin Transverse Mercator [1930] NAD83 / Wyoming East [1931] NAD83 / Wyoming East (ftUS) [1932] NAD83 / Wyoming East Central [1933] NAD83 / Wyoming East Central (ftUS) [1934] NAD83 / Wyoming West [1935] NAD83 / Wyoming West (ftUS) [1936] NAD83 / Wyoming West Central [1937] NAD83 / Wyoming West Central (ftUS) [1938] NAD83 / Yukon Albers [1939] NAD83(2011) / Alabama East [1940] NAD83(2011) / Alabama West [1941] NAD83(2011) / Alaska Albers [1942] NAD83(2011) / Alaska zone 1 [1943] NAD83(2011) / Alaska zone 10 [1944] NAD83(2011) / Alaska zone 2 [1945] NAD83(2011) / Alaska zone 3 [1946] NAD83(2011) / Alaska zone 4 [1947] NAD83(2011) / Alaska zone 5 [1948] NAD83(2011) / Alaska zone 6 [1949] NAD83(2011) / Alaska zone 7 [1950] NAD83(2011) / Alaska zone 8 [1951] NAD83(2011) / Alaska zone 9 [1952] NAD83(2011) / Arizona Central [1953] NAD83(2011) / Arizona Central (ft) [1954] NAD83(2011) / Arizona East [1955] NAD83(2011) / Arizona East (ft) [1956] NAD83(2011) / Arizona West [1957] NAD83(2011) / Arizona West (ft) [1958] NAD83(2011) / Arkansas North [1959] NAD83(2011) / Arkansas North (ftUS) [1960] NAD83(2011) / Arkansas South [1961] NAD83(2011) / Arkansas South (ftUS) [1962] NAD83(2011) / California Albers [1963] NAD83(2011) / California zone 1 [1964] NAD83(2011) / California zone 1 (ftUS) [1965] NAD83(2011) / California zone 2 [1966] NAD83(2011) / California zone 2 (ftUS) [1967] NAD83(2011) / California zone 3 [1968] NAD83(2011) / California zone 3 (ftUS) [1969] NAD83(2011) / California zone 4 [1970] NAD83(2011) / California zone 4 (ftUS) [1971] NAD83(2011) / California zone 5 [1972] NAD83(2011) / California zone 5 (ftUS) [1973] NAD83(2011) / California zone 6 [1974] NAD83(2011) / California zone 6 (ftUS) [1975] NAD83(2011) / Colorado Central [1976] NAD83(2011) / Colorado Central (ftUS) [1977] NAD83(2011) / Colorado North [1978] NAD83(2011) / Colorado North (ftUS) [1979] NAD83(2011) / Colorado South [1980] NAD83(2011) / Colorado South (ftUS) [1981] NAD83(2011) / Connecticut [1982] NAD83(2011) / Connecticut (ftUS) [1983] NAD83(2011) / Conus Albers [1984] NAD83(2011) / Delaware [1985] NAD83(2011) / Delaware (ftUS) [1986] NAD83(2011) / EPSG Arctic zone 5-29 [1987] NAD83(2011) / EPSG Arctic zone 5-31 [1988] NAD83(2011) / EPSG Arctic zone 6-14 [1989] NAD83(2011) / EPSG Arctic zone 6-16 [1990] NAD83(2011) / Florida East [1991] NAD83(2011) / Florida East (ftUS) [1992] NAD83(2011) / Florida GDL Albers [1993] NAD83(2011) / Florida North [1994] NAD83(2011) / Florida North (ftUS) [1995] NAD83(2011) / Florida West [1996] NAD83(2011) / Florida West (ftUS) [1997] NAD83(2011) / Georgia East [1998] NAD83(2011) / Georgia East (ftUS) [1999] NAD83(2011) / Georgia West [2000] NAD83(2011) / Georgia West (ftUS) [2001] NAD83(2011) / IaRCS zone 1 [2002] NAD83(2011) / IaRCS zone 10 [2003] NAD83(2011) / IaRCS zone 11 [2004] NAD83(2011) / IaRCS zone 12 [2005] NAD83(2011) / IaRCS zone 13 [2006] NAD83(2011) / IaRCS zone 14 [2007] NAD83(2011) / IaRCS zone 2 [2008] NAD83(2011) / IaRCS zone 3 [2009] NAD83(2011) / IaRCS zone 4 [2010] NAD83(2011) / IaRCS zone 5 [2011] NAD83(2011) / IaRCS zone 6 [2012] NAD83(2011) / IaRCS zone 7 [2013] NAD83(2011) / IaRCS zone 8 [2014] NAD83(2011) / IaRCS zone 9 [2015] NAD83(2011) / Idaho Central [2016] NAD83(2011) / Idaho Central (ftUS) [2017] NAD83(2011) / Idaho East [2018] NAD83(2011) / Idaho East (ftUS) [2019] NAD83(2011) / Idaho West [2020] NAD83(2011) / Idaho West (ftUS) [2021] NAD83(2011) / Illinois East [2022] NAD83(2011) / Illinois East (ftUS) [2023] NAD83(2011) / Illinois West [2024] NAD83(2011) / Illinois West (ftUS) [2025] NAD83(2011) / InGCS Adams (ftUS) [2026] NAD83(2011) / InGCS Adams (m) [2027] NAD83(2011) / InGCS Allen (ftUS) [2028] NAD83(2011) / InGCS Allen (m) [2029] NAD83(2011) / InGCS Bartholomew (ftUS) [2030] NAD83(2011) / InGCS Bartholomew (m) [2031] NAD83(2011) / InGCS Benton (ftUS) [2032] NAD83(2011) / InGCS Benton (m) [2033] NAD83(2011) / InGCS Blackford-Delaware (ftUS) [2034] NAD83(2011) / InGCS Blackford-Delaware (m) [2035] NAD83(2011) / InGCS Boone-Hendricks (ftUS) [2036] NAD83(2011) / InGCS Boone-Hendricks (m) [2037] NAD83(2011) / InGCS Brown (ftUS) [2038] NAD83(2011) / InGCS Brown (m) [2039] NAD83(2011) / InGCS Carroll (ftUS) [2040] NAD83(2011) / InGCS Carroll (m) [2041] NAD83(2011) / InGCS Cass (ftUS) [2042] NAD83(2011) / InGCS Cass (m) [2043] NAD83(2011) / InGCS Clark-Floyd-Scott (ftUS) [2044] NAD83(2011) / InGCS Clark-Floyd-Scott (m) [2045] NAD83(2011) / InGCS Clay (ftUS) [2046] NAD83(2011) / InGCS Clay (m) [2047] NAD83(2011) / InGCS Clinton (ftUS) [2048] NAD83(2011) / InGCS Clinton (m) [2049] NAD83(2011) / InGCS Crawford-Lawrence-Orange (ftUS) [2050] NAD83(2011) / InGCS Crawford-Lawrence-Orange (m) [2051] NAD83(2011) / InGCS Daviess-Greene (ftUS) [2052] NAD83(2011) / InGCS Daviess-Greene (m) [2053] NAD83(2011) / InGCS DeKalb (ftUS) [2054] NAD83(2011) / InGCS DeKalb (m) [2055] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (ftUS) [2056] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (m) [2057] NAD83(2011) / InGCS Decatur-Rush (ftUS) [2058] NAD83(2011) / InGCS Decatur-Rush (m) [2059] NAD83(2011) / InGCS Dubois-Martin (ftUS) [2060] NAD83(2011) / InGCS Dubois-Martin (m) [2061] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (ftUS) [2062] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (m) [2063] NAD83(2011) / InGCS Fayette-Franklin-Union (ftUS) [2064] NAD83(2011) / InGCS Fayette-Franklin-Union (m) [2065] NAD83(2011) / InGCS Fountain-Warren (ftUS) [2066] NAD83(2011) / InGCS Fountain-Warren (m) [2067] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (ftUS) [2068] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (m) [2069] NAD83(2011) / InGCS Gibson (ftUS) [2070] NAD83(2011) / InGCS Gibson (m) [2071] NAD83(2011) / InGCS Grant (ftUS) [2072] NAD83(2011) / InGCS Grant (m) [2073] NAD83(2011) / InGCS Hamilton-Tipton (ftUS) [2074] NAD83(2011) / InGCS Hamilton-Tipton (m) [2075] NAD83(2011) / InGCS Hancock-Madison (ftUS) [2076] NAD83(2011) / InGCS Hancock-Madison (m) [2077] NAD83(2011) / InGCS Harrison-Washington (ftUS) [2078] NAD83(2011) / InGCS Harrison-Washington (m) [2079] NAD83(2011) / InGCS Henry (ftUS) [2080] NAD83(2011) / InGCS Henry (m) [2081] NAD83(2011) / InGCS Howard-Miami (ftUS) [2082] NAD83(2011) / InGCS Howard-Miami (m) [2083] NAD83(2011) / InGCS Huntington-Whitley (ftUS) [2084] NAD83(2011) / InGCS Huntington-Whitley (m) [2085] NAD83(2011) / InGCS Jackson (ftUS) [2086] NAD83(2011) / InGCS Jackson (m) [2087] NAD83(2011) / InGCS Jasper-Porter (ftUS) [2088] NAD83(2011) / InGCS Jasper-Porter (m) [2089] NAD83(2011) / InGCS Jay (ftUS) [2090] NAD83(2011) / InGCS Jay (m) [2091] NAD83(2011) / InGCS Jefferson (ftUS) [2092] NAD83(2011) / InGCS Jefferson (m) [2093] NAD83(2011) / InGCS Jennings (ftUS) [2094] NAD83(2011) / InGCS Jennings (m) [2095] NAD83(2011) / InGCS Johnson-Marion (ftUS) [2096] NAD83(2011) / InGCS Johnson-Marion (m) [2097] NAD83(2011) / InGCS Knox (ftUS) [2098] NAD83(2011) / InGCS Knox (m) [2099] NAD83(2011) / InGCS LaGrange-Noble (ftUS) [2100] NAD83(2011) / InGCS LaGrange-Noble (m) [2101] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (ftUS) [2102] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (m) [2103] NAD83(2011) / InGCS Lake-Newton (ftUS) [2104] NAD83(2011) / InGCS Lake-Newton (m) [2105] NAD83(2011) / InGCS Monroe-Morgan (ftUS) [2106] NAD83(2011) / InGCS Monroe-Morgan (m) [2107] NAD83(2011) / InGCS Montgomery-Putnam (ftUS) [2108] NAD83(2011) / InGCS Montgomery-Putnam (m) [2109] NAD83(2011) / InGCS Owen (ftUS) [2110] NAD83(2011) / InGCS Owen (m) [2111] NAD83(2011) / InGCS Parke-Vermillion (ftUS) [2112] NAD83(2011) / InGCS Parke-Vermillion (m) [2113] NAD83(2011) / InGCS Perry (ftUS) [2114] NAD83(2011) / InGCS Perry (m) [2115] NAD83(2011) / InGCS Pike-Warrick (ftUS) [2116] NAD83(2011) / InGCS Pike-Warrick (m) [2117] NAD83(2011) / InGCS Posey (ftUS) [2118] NAD83(2011) / InGCS Posey (m) [2119] NAD83(2011) / InGCS Randolph-Wayne (ftUS) [2120] NAD83(2011) / InGCS Randolph-Wayne (m) [2121] NAD83(2011) / InGCS Ripley (ftUS) [2122] NAD83(2011) / InGCS Ripley (m) [2123] NAD83(2011) / InGCS Shelby (ftUS) [2124] NAD83(2011) / InGCS Shelby (m) [2125] NAD83(2011) / InGCS Spencer (ftUS) [2126] NAD83(2011) / InGCS Spencer (m) [2127] NAD83(2011) / InGCS Steuben (ftUS) [2128] NAD83(2011) / InGCS Steuben (m) [2129] NAD83(2011) / InGCS Sullivan (ftUS) [2130] NAD83(2011) / InGCS Sullivan (m) [2131] NAD83(2011) / InGCS Tippecanoe-White (ftUS) [2132] NAD83(2011) / InGCS Tippecanoe-White (m) [2133] NAD83(2011) / InGCS Vanderburgh (ftUS) [2134] NAD83(2011) / InGCS Vanderburgh (m) [2135] NAD83(2011) / InGCS Vigo (ftUS) [2136] NAD83(2011) / InGCS Vigo (m) [2137] NAD83(2011) / InGCS Wells (ftUS) [2138] NAD83(2011) / InGCS Wells (m) [2139] NAD83(2011) / Indiana East [2140] NAD83(2011) / Indiana East (ftUS) [2141] NAD83(2011) / Indiana West [2142] NAD83(2011) / Indiana West (ftUS) [2143] NAD83(2011) / Iowa North [2144] NAD83(2011) / Iowa North (ftUS) [2145] NAD83(2011) / Iowa South [2146] NAD83(2011) / Iowa South (ftUS) [2147] NAD83(2011) / KS RCS zone 1 [2148] NAD83(2011) / KS RCS zone 10 [2149] NAD83(2011) / KS RCS zone 11 [2150] NAD83(2011) / KS RCS zone 12 [2151] NAD83(2011) / KS RCS zone 13 [2152] NAD83(2011) / KS RCS zone 14 [2153] NAD83(2011) / KS RCS zone 15 [2154] NAD83(2011) / KS RCS zone 16 [2155] NAD83(2011) / KS RCS zone 17 [2156] NAD83(2011) / KS RCS zone 18 [2157] NAD83(2011) / KS RCS zone 19 [2158] NAD83(2011) / KS RCS zone 2 [2159] NAD83(2011) / KS RCS zone 20 [2160] NAD83(2011) / KS RCS zone 3 [2161] NAD83(2011) / KS RCS zone 4 [2162] NAD83(2011) / KS RCS zone 5 [2163] NAD83(2011) / KS RCS zone 6 [2164] NAD83(2011) / KS RCS zone 7 [2165] NAD83(2011) / KS RCS zone 8 [2166] NAD83(2011) / KS RCS zone 9 [2167] NAD83(2011) / Kansas LCC [2168] NAD83(2011) / Kansas LCC (ftUS) [2169] NAD83(2011) / Kansas North [2170] NAD83(2011) / Kansas North (ftUS) [2171] NAD83(2011) / Kansas South [2172] NAD83(2011) / Kansas South (ftUS) [2173] NAD83(2011) / Kentucky North [2174] NAD83(2011) / Kentucky North (ftUS) [2175] NAD83(2011) / Kentucky Single Zone [2176] NAD83(2011) / Kentucky Single Zone (ftUS) [2177] NAD83(2011) / Kentucky South [2178] NAD83(2011) / Kentucky South (ftUS) [2179] NAD83(2011) / Louisiana North [2180] NAD83(2011) / Louisiana North (ftUS) [2181] NAD83(2011) / Louisiana South [2182] NAD83(2011) / Louisiana South (ftUS) [2183] NAD83(2011) / Maine CS2000 Central [2184] NAD83(2011) / Maine CS2000 East [2185] NAD83(2011) / Maine CS2000 West [2186] NAD83(2011) / Maine East [2187] NAD83(2011) / Maine East (ftUS) [2188] NAD83(2011) / Maine West [2189] NAD83(2011) / Maine West (ftUS) [2190] NAD83(2011) / Maryland [2191] NAD83(2011) / Maryland (ftUS) [2192] NAD83(2011) / Massachusetts Island [2193] NAD83(2011) / Massachusetts Island (ftUS) [2194] NAD83(2011) / Massachusetts Mainland [2195] NAD83(2011) / Massachusetts Mainland (ftUS) [2196] NAD83(2011) / Michigan Central [2197] NAD83(2011) / Michigan Central (ft) [2198] NAD83(2011) / Michigan North [2199] NAD83(2011) / Michigan North (ft) [2200] NAD83(2011) / Michigan Oblique Mercator [2201] NAD83(2011) / Michigan South [2202] NAD83(2011) / Michigan South (ft) [2203] NAD83(2011) / Minnesota Central [2204] NAD83(2011) / Minnesota Central (ftUS) [2205] NAD83(2011) / Minnesota North [2206] NAD83(2011) / Minnesota North (ftUS) [2207] NAD83(2011) / Minnesota South [2208] NAD83(2011) / Minnesota South (ftUS) [2209] NAD83(2011) / Mississippi East [2210] NAD83(2011) / Mississippi East (ftUS) [2211] NAD83(2011) / Mississippi TM [2212] NAD83(2011) / Mississippi West [2213] NAD83(2011) / Mississippi West (ftUS) [2214] NAD83(2011) / Missouri Central [2215] NAD83(2011) / Missouri East [2216] NAD83(2011) / Missouri West [2217] NAD83(2011) / Montana [2218] NAD83(2011) / Montana (ft) [2219] NAD83(2011) / NCRS Las Vegas (ftUS) [2220] NAD83(2011) / NCRS Las Vegas (m) [2221] NAD83(2011) / NCRS Las Vegas high (ftUS) [2222] NAD83(2011) / NCRS Las Vegas high (m) [2223] NAD83(2011) / Nebraska [2224] NAD83(2011) / Nebraska (ftUS) [2225] NAD83(2011) / Nebraska (ftUS) (deprecated) [2226] NAD83(2011) / Nevada Central [2227] NAD83(2011) / Nevada Central (ftUS) [2228] NAD83(2011) / Nevada East [2229] NAD83(2011) / Nevada East (ftUS) [2230] NAD83(2011) / Nevada West [2231] NAD83(2011) / Nevada West (ftUS) [2232] NAD83(2011) / New Hampshire [2233] NAD83(2011) / New Hampshire (ftUS) [2234] NAD83(2011) / New Jersey [2235] NAD83(2011) / New Jersey (ftUS) [2236] NAD83(2011) / New Mexico Central [2237] NAD83(2011) / New Mexico Central (ftUS) [2238] NAD83(2011) / New Mexico East [2239] NAD83(2011) / New Mexico East (ftUS) [2240] NAD83(2011) / New Mexico West [2241] NAD83(2011) / New Mexico West (ftUS) [2242] NAD83(2011) / New York Central [2243] NAD83(2011) / New York Central (ftUS) [2244] NAD83(2011) / New York East [2245] NAD83(2011) / New York East (ftUS) [2246] NAD83(2011) / New York Long Island [2247] NAD83(2011) / New York Long Island (ftUS) [2248] NAD83(2011) / New York West [2249] NAD83(2011) / New York West (ftUS) [2250] NAD83(2011) / North Carolina [2251] NAD83(2011) / North Carolina (ftUS) [2252] NAD83(2011) / North Dakota North [2253] NAD83(2011) / North Dakota North (ft) [2254] NAD83(2011) / North Dakota South [2255] NAD83(2011) / North Dakota South (ft) [2256] NAD83(2011) / Ohio North [2257] NAD83(2011) / Ohio North (ftUS) [2258] NAD83(2011) / Ohio South [2259] NAD83(2011) / Ohio South (ftUS) [2260] NAD83(2011) / Oklahoma North [2261] NAD83(2011) / Oklahoma North (ftUS) [2262] NAD83(2011) / Oklahoma South [2263] NAD83(2011) / Oklahoma South (ftUS) [2264] NAD83(2011) / Oregon Baker zone (ft) [2265] NAD83(2011) / Oregon Baker zone (m) [2266] NAD83(2011) / Oregon Bend-Burns zone (ft) [2267] NAD83(2011) / Oregon Bend-Burns zone (m) [2268] NAD83(2011) / Oregon Bend-Klamath Falls zone (ft) [2269] NAD83(2011) / Oregon Bend-Klamath Falls zone (m) [2270] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (ft) [2271] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (m) [2272] NAD83(2011) / Oregon Burns-Harper zone (ft) [2273] NAD83(2011) / Oregon Burns-Harper zone (m) [2274] NAD83(2011) / Oregon Canyon City-Burns zone (ft) [2275] NAD83(2011) / Oregon Canyon City-Burns zone (m) [2276] NAD83(2011) / Oregon Canyonville-Grants Pass zone (ft) [2277] NAD83(2011) / Oregon Canyonville-Grants Pass zone (m) [2278] NAD83(2011) / Oregon Coast Range North zone (ft) [2279] NAD83(2011) / Oregon Coast Range North zone (m) [2280] NAD83(2011) / Oregon Coast zone (ft) [2281] NAD83(2011) / Oregon Coast zone (m) [2282] NAD83(2011) / Oregon Columbia River East zone (ft) [2283] NAD83(2011) / Oregon Columbia River East zone (m) [2284] NAD83(2011) / Oregon Columbia River West zone (ft) [2285] NAD83(2011) / Oregon Columbia River West zone (m) [2286] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (ft) [2287] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (m) [2288] NAD83(2011) / Oregon Dayville-Prairie City zone (ft) [2289] NAD83(2011) / Oregon Dayville-Prairie City zone (m) [2290] NAD83(2011) / Oregon Denio-Burns zone (ft) [2291] NAD83(2011) / Oregon Denio-Burns zone (m) [2292] NAD83(2011) / Oregon Dufur-Madras zone (ft) [2293] NAD83(2011) / Oregon Dufur-Madras zone (m) [2294] NAD83(2011) / Oregon Eugene zone (ft) [2295] NAD83(2011) / Oregon Eugene zone (m) [2296] NAD83(2011) / Oregon GIC Lambert (ft) [2297] NAD83(2011) / Oregon Grants Pass-Ashland zone (ft) [2298] NAD83(2011) / Oregon Grants Pass-Ashland zone (m) [2299] NAD83(2011) / Oregon Gresham-Warm Springs zone (ft) [2300] NAD83(2011) / Oregon Gresham-Warm Springs zone (m) [2301] NAD83(2011) / Oregon Halfway zone (ft) [2302] NAD83(2011) / Oregon Halfway zone (m) [2303] NAD83(2011) / Oregon LCC (m) [2304] NAD83(2011) / Oregon La Grande zone (ft) [2305] NAD83(2011) / Oregon La Grande zone (m) [2306] NAD83(2011) / Oregon Medford-Diamond Lake zone (ft) [2307] NAD83(2011) / Oregon Medford-Diamond Lake zone (m) [2308] NAD83(2011) / Oregon Mitchell zone (ft) [2309] NAD83(2011) / Oregon Mitchell zone (m) [2310] NAD83(2011) / Oregon North [2311] NAD83(2011) / Oregon North (ft) [2312] NAD83(2011) / Oregon North Central zone (ft) [2313] NAD83(2011) / Oregon North Central zone (m) [2314] NAD83(2011) / Oregon Ochoco Summit zone (ft) [2315] NAD83(2011) / Oregon Ochoco Summit zone (m) [2316] NAD83(2011) / Oregon Ontario zone (ft) [2317] NAD83(2011) / Oregon Ontario zone (m) [2318] NAD83(2011) / Oregon Owyhee zone (ft) [2319] NAD83(2011) / Oregon Owyhee zone (m) [2320] NAD83(2011) / Oregon Pendleton zone (ft) [2321] NAD83(2011) / Oregon Pendleton zone (m) [2322] NAD83(2011) / Oregon Pendleton-La Grande zone (ft) [2323] NAD83(2011) / Oregon Pendleton-La Grande zone (m) [2324] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (ft) [2325] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (m) [2326] NAD83(2011) / Oregon Portland zone (ft) [2327] NAD83(2011) / Oregon Portland zone (m) [2328] NAD83(2011) / Oregon Prairie City-Brogan zone (ft) [2329] NAD83(2011) / Oregon Prairie City-Brogan zone (m) [2330] NAD83(2011) / Oregon Riley-Lakeview zone (ft) [2331] NAD83(2011) / Oregon Riley-Lakeview zone (m) [2332] NAD83(2011) / Oregon Salem zone (ft) [2333] NAD83(2011) / Oregon Salem zone (m) [2334] NAD83(2011) / Oregon Santiam Pass zone (ft) [2335] NAD83(2011) / Oregon Santiam Pass zone (m) [2336] NAD83(2011) / Oregon Siskiyou Pass zone (ft) [2337] NAD83(2011) / Oregon Siskiyou Pass zone (m) [2338] NAD83(2011) / Oregon South [2339] NAD83(2011) / Oregon South (ft) [2340] NAD83(2011) / Oregon Ukiah-Fox zone (ft) [2341] NAD83(2011) / Oregon Ukiah-Fox zone (m) [2342] NAD83(2011) / Oregon Wallowa zone (ft) [2343] NAD83(2011) / Oregon Wallowa zone (m) [2344] NAD83(2011) / Oregon Warner Highway zone (ft) [2345] NAD83(2011) / Oregon Warner Highway zone (m) [2346] NAD83(2011) / Oregon Willamette Pass zone (ft) [2347] NAD83(2011) / Oregon Willamette Pass zone (m) [2348] NAD83(2011) / PCCS zone 1 (ft) [2349] NAD83(2011) / PCCS zone 2 (ft) [2350] NAD83(2011) / PCCS zone 3 (ft) [2351] NAD83(2011) / PCCS zone 4 (ft) [2352] NAD83(2011) / Pennsylvania North [2353] NAD83(2011) / Pennsylvania North (ftUS) [2354] NAD83(2011) / Pennsylvania South [2355] NAD83(2011) / Pennsylvania South (ftUS) [2356] NAD83(2011) / Puerto Rico and Virgin Is. [2357] NAD83(2011) / RMTCRS Billings (ft) [2358] NAD83(2011) / RMTCRS Billings (m) [2359] NAD83(2011) / RMTCRS Blackfeet (ft) [2360] NAD83(2011) / RMTCRS Blackfeet (m) [2361] NAD83(2011) / RMTCRS Bobcat (ft) [2362] NAD83(2011) / RMTCRS Bobcat (m) [2363] NAD83(2011) / RMTCRS Crow (ft) [2364] NAD83(2011) / RMTCRS Crow (m) [2365] NAD83(2011) / RMTCRS Fort Belknap (ft) [2366] NAD83(2011) / RMTCRS Fort Belknap (m) [2367] NAD83(2011) / RMTCRS Fort Peck Assiniboine (ft) [2368] NAD83(2011) / RMTCRS Fort Peck Assiniboine (m) [2369] NAD83(2011) / RMTCRS Fort Peck Sioux (ft) [2370] NAD83(2011) / RMTCRS Fort Peck Sioux (m) [2371] NAD83(2011) / RMTCRS Milk River (ft) [2372] NAD83(2011) / RMTCRS Milk River (m) [2373] NAD83(2011) / RMTCRS St Mary (ft) [2374] NAD83(2011) / RMTCRS St Mary (m) [2375] NAD83(2011) / RMTCRS Wind River (ftUS) [2376] NAD83(2011) / RMTCRS Wind River (m) [2377] NAD83(2011) / Rhode Island [2378] NAD83(2011) / Rhode Island (ftUS) [2379] NAD83(2011) / San Francisco CS13 [2380] NAD83(2011) / San Francisco CS13 (deprecated) [2381] NAD83(2011) / San Francisco CS13 (ftUS) [2382] NAD83(2011) / San Francisco CS13 (ftUS) (deprecated) [2383] NAD83(2011) / South Carolina [2384] NAD83(2011) / South Carolina (ft) [2385] NAD83(2011) / South Dakota North [2386] NAD83(2011) / South Dakota North (ftUS) [2387] NAD83(2011) / South Dakota South [2388] NAD83(2011) / South Dakota South (ftUS) [2389] NAD83(2011) / Tennessee [2390] NAD83(2011) / Tennessee (ftUS) [2391] NAD83(2011) / Texas Central [2392] NAD83(2011) / Texas Central (ftUS) [2393] NAD83(2011) / Texas Centric Albers Equal Area [2394] NAD83(2011) / Texas Centric Lambert Conformal [2395] NAD83(2011) / Texas North [2396] NAD83(2011) / Texas North (ftUS) [2397] NAD83(2011) / Texas North Central [2398] NAD83(2011) / Texas North Central (ftUS) [2399] NAD83(2011) / Texas South [2400] NAD83(2011) / Texas South (ftUS) [2401] NAD83(2011) / Texas South Central [2402] NAD83(2011) / Texas South Central (ftUS) [2403] NAD83(2011) / UTM zone 10N [2404] NAD83(2011) / UTM zone 11N [2405] NAD83(2011) / UTM zone 12N [2406] NAD83(2011) / UTM zone 13N [2407] NAD83(2011) / UTM zone 14N [2408] NAD83(2011) / UTM zone 15N [2409] NAD83(2011) / UTM zone 16N [2410] NAD83(2011) / UTM zone 17N [2411] NAD83(2011) / UTM zone 18N [2412] NAD83(2011) / UTM zone 19N [2413] NAD83(2011) / UTM zone 1N [2414] NAD83(2011) / UTM zone 2N [2415] NAD83(2011) / UTM zone 3N [2416] NAD83(2011) / UTM zone 4N [2417] NAD83(2011) / UTM zone 59N [2418] NAD83(2011) / UTM zone 5N [2419] NAD83(2011) / UTM zone 60N [2420] NAD83(2011) / UTM zone 6N [2421] NAD83(2011) / UTM zone 7N [2422] NAD83(2011) / UTM zone 8N [2423] NAD83(2011) / UTM zone 9N [2424] NAD83(2011) / Utah Central [2425] NAD83(2011) / Utah Central (ftUS) [2426] NAD83(2011) / Utah North [2427] NAD83(2011) / Utah North (ftUS) [2428] NAD83(2011) / Utah South [2429] NAD83(2011) / Utah South (ftUS) [2430] NAD83(2011) / Vermont [2431] NAD83(2011) / Vermont (ftUS) [2432] NAD83(2011) / Virginia Lambert [2433] NAD83(2011) / Virginia North [2434] NAD83(2011) / Virginia North (ftUS) [2435] NAD83(2011) / Virginia South [2436] NAD83(2011) / Virginia South (ftUS) [2437] NAD83(2011) / WISCRS Adams and Juneau (ftUS) [2438] NAD83(2011) / WISCRS Adams and Juneau (m) [2439] NAD83(2011) / WISCRS Ashland (ftUS) [2440] NAD83(2011) / WISCRS Ashland (m) [2441] NAD83(2011) / WISCRS Barron (ftUS) [2442] NAD83(2011) / WISCRS Barron (m) [2443] NAD83(2011) / WISCRS Bayfield (ftUS) [2444] NAD83(2011) / WISCRS Bayfield (m) [2445] NAD83(2011) / WISCRS Brown (ftUS) [2446] NAD83(2011) / WISCRS Brown (m) [2447] NAD83(2011) / WISCRS Buffalo (ftUS) [2448] NAD83(2011) / WISCRS Buffalo (m) [2449] NAD83(2011) / WISCRS Burnett (ftUS) [2450] NAD83(2011) / WISCRS Burnett (m) [2451] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2452] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2453] NAD83(2011) / WISCRS Chippewa (ftUS) [2454] NAD83(2011) / WISCRS Chippewa (m) [2455] NAD83(2011) / WISCRS Clark (ftUS) [2456] NAD83(2011) / WISCRS Clark (m) [2457] NAD83(2011) / WISCRS Columbia (ftUS) [2458] NAD83(2011) / WISCRS Columbia (m) [2459] NAD83(2011) / WISCRS Crawford (ftUS) [2460] NAD83(2011) / WISCRS Crawford (m) [2461] NAD83(2011) / WISCRS Dane (ftUS) [2462] NAD83(2011) / WISCRS Dane (m) [2463] NAD83(2011) / WISCRS Dodge and Jefferson (ftUS) [2464] NAD83(2011) / WISCRS Dodge and Jefferson (m) [2465] NAD83(2011) / WISCRS Door (ftUS) [2466] NAD83(2011) / WISCRS Door (m) [2467] NAD83(2011) / WISCRS Douglas (ftUS) [2468] NAD83(2011) / WISCRS Douglas (m) [2469] NAD83(2011) / WISCRS Dunn (ftUS) [2470] NAD83(2011) / WISCRS Dunn (m) [2471] NAD83(2011) / WISCRS Eau Claire (ftUS) [2472] NAD83(2011) / WISCRS Eau Claire (m) [2473] NAD83(2011) / WISCRS Florence (ftUS) [2474] NAD83(2011) / WISCRS Florence (m) [2475] NAD83(2011) / WISCRS Forest (ftUS) [2476] NAD83(2011) / WISCRS Forest (m) [2477] NAD83(2011) / WISCRS Grant (ftUS) [2478] NAD83(2011) / WISCRS Grant (m) [2479] NAD83(2011) / WISCRS Green Lake and Marquette (ftUS) [2480] NAD83(2011) / WISCRS Green Lake and Marquette (m) [2481] NAD83(2011) / WISCRS Green and Lafayette (ftUS) [2482] NAD83(2011) / WISCRS Green and Lafayette (m) [2483] NAD83(2011) / WISCRS Iowa (ftUS) [2484] NAD83(2011) / WISCRS Iowa (m) [2485] NAD83(2011) / WISCRS Iron (ftUS) [2486] NAD83(2011) / WISCRS Iron (m) [2487] NAD83(2011) / WISCRS Jackson (ftUS) [2488] NAD83(2011) / WISCRS Jackson (m) [2489] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [2490] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [2491] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [2492] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [2493] NAD83(2011) / WISCRS La Crosse (ftUS) [2494] NAD83(2011) / WISCRS La Crosse (m) [2495] NAD83(2011) / WISCRS Langlade (ftUS) [2496] NAD83(2011) / WISCRS Langlade (m) [2497] NAD83(2011) / WISCRS Lincoln (ftUS) [2498] NAD83(2011) / WISCRS Lincoln (m) [2499] NAD83(2011) / WISCRS Marathon (ftUS) [2500] NAD83(2011) / WISCRS Marathon (m) [2501] NAD83(2011) / WISCRS Marinette (ftUS) [2502] NAD83(2011) / WISCRS Marinette (m) [2503] NAD83(2011) / WISCRS Menominee (ftUS) [2504] NAD83(2011) / WISCRS Menominee (m) [2505] NAD83(2011) / WISCRS Monroe (ftUS) [2506] NAD83(2011) / WISCRS Monroe (m) [2507] NAD83(2011) / WISCRS Oconto (ftUS) [2508] NAD83(2011) / WISCRS Oconto (m) [2509] NAD83(2011) / WISCRS Oneida (ftUS) [2510] NAD83(2011) / WISCRS Oneida (m) [2511] NAD83(2011) / WISCRS Pepin and Pierce (ftUS) [2512] NAD83(2011) / WISCRS Pepin and Pierce (m) [2513] NAD83(2011) / WISCRS Polk (ftUS) [2514] NAD83(2011) / WISCRS Polk (m) [2515] NAD83(2011) / WISCRS Portage (ftUS) [2516] NAD83(2011) / WISCRS Portage (m) [2517] NAD83(2011) / WISCRS Price (ftUS) [2518] NAD83(2011) / WISCRS Price (m) [2519] NAD83(2011) / WISCRS Richland (ftUS) [2520] NAD83(2011) / WISCRS Richland (m) [2521] NAD83(2011) / WISCRS Rock (ftUS) [2522] NAD83(2011) / WISCRS Rock (m) [2523] NAD83(2011) / WISCRS Rusk (ftUS) [2524] NAD83(2011) / WISCRS Rusk (m) [2525] NAD83(2011) / WISCRS Sauk (ftUS) [2526] NAD83(2011) / WISCRS Sauk (m) [2527] NAD83(2011) / WISCRS Sawyer (ftUS) [2528] NAD83(2011) / WISCRS Sawyer (m) [2529] NAD83(2011) / WISCRS Shawano (ftUS) [2530] NAD83(2011) / WISCRS Shawano (m) [2531] NAD83(2011) / WISCRS St. Croix (ftUS) [2532] NAD83(2011) / WISCRS St. Croix (m) [2533] NAD83(2011) / WISCRS Taylor (ftUS) [2534] NAD83(2011) / WISCRS Taylor (m) [2535] NAD83(2011) / WISCRS Trempealeau (ftUS) [2536] NAD83(2011) / WISCRS Trempealeau (m) [2537] NAD83(2011) / WISCRS Vernon (ftUS) [2538] NAD83(2011) / WISCRS Vernon (m) [2539] NAD83(2011) / WISCRS Vilas (ftUS) [2540] NAD83(2011) / WISCRS Vilas (m) [2541] NAD83(2011) / WISCRS Walworth (ftUS) [2542] NAD83(2011) / WISCRS Walworth (m) [2543] NAD83(2011) / WISCRS Washburn (ftUS) [2544] NAD83(2011) / WISCRS Washburn (m) [2545] NAD83(2011) / WISCRS Washington (ftUS) [2546] NAD83(2011) / WISCRS Washington (m) [2547] NAD83(2011) / WISCRS Waukesha (ftUS) [2548] NAD83(2011) / WISCRS Waukesha (m) [2549] NAD83(2011) / WISCRS Waupaca (ftUS) [2550] NAD83(2011) / WISCRS Waupaca (m) [2551] NAD83(2011) / WISCRS Waushara (ftUS) [2552] NAD83(2011) / WISCRS Waushara (m) [2553] NAD83(2011) / WISCRS Wood (ftUS) [2554] NAD83(2011) / WISCRS Wood (m) [2555] NAD83(2011) / Washington North [2556] NAD83(2011) / Washington North (ftUS) [2557] NAD83(2011) / Washington South [2558] NAD83(2011) / Washington South (ftUS) [2559] NAD83(2011) / West Virginia North [2560] NAD83(2011) / West Virginia North (ftUS) [2561] NAD83(2011) / West Virginia South [2562] NAD83(2011) / West Virginia South (ftUS) [2563] NAD83(2011) / Wisconsin Central [2564] NAD83(2011) / Wisconsin Central (deprecated) [2565] NAD83(2011) / Wisconsin Central (ftUS) [2566] NAD83(2011) / Wisconsin North [2567] NAD83(2011) / Wisconsin North (ftUS) [2568] NAD83(2011) / Wisconsin South [2569] NAD83(2011) / Wisconsin South (ftUS) [2570] NAD83(2011) / Wisconsin Transverse Mercator [2571] NAD83(2011) / Wyoming East [2572] NAD83(2011) / Wyoming East (ftUS) [2573] NAD83(2011) / Wyoming East Central [2574] NAD83(2011) / Wyoming East Central (ftUS) [2575] NAD83(2011) / Wyoming West [2576] NAD83(2011) / Wyoming West (ftUS) [2577] NAD83(2011) / Wyoming West Central [2578] NAD83(2011) / Wyoming West Central (ftUS) [2579] NAD83(CORS96) / Oregon Baker zone (ft) [2580] NAD83(CORS96) / Oregon Baker zone (m) [2581] NAD83(CORS96) / Oregon Bend-Burns zone (ft) [2582] NAD83(CORS96) / Oregon Bend-Burns zone (m) [2583] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (ft) [2584] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (m) [2585] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (ft) [2586] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (m) [2587] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (ft) [2588] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (m) [2589] NAD83(CORS96) / Oregon Coast zone (ft) [2590] NAD83(CORS96) / Oregon Coast zone (m) [2591] NAD83(CORS96) / Oregon Columbia River East zone (ft) [2592] NAD83(CORS96) / Oregon Columbia River East zone (m) [2593] NAD83(CORS96) / Oregon Columbia River West zone (ft) [2594] NAD83(CORS96) / Oregon Columbia River West zone (m) [2595] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (ft) [2596] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (m) [2597] NAD83(CORS96) / Oregon Dufur-Madras zone (ft) [2598] NAD83(CORS96) / Oregon Dufur-Madras zone (m) [2599] NAD83(CORS96) / Oregon Eugene zone (ft) [2600] NAD83(CORS96) / Oregon Eugene zone (m) [2601] NAD83(CORS96) / Oregon GIC Lambert (ft) [2602] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (ft) [2603] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (m) [2604] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (ft) [2605] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (m) [2606] NAD83(CORS96) / Oregon LCC (m) [2607] NAD83(CORS96) / Oregon La Grande zone (ft) [2608] NAD83(CORS96) / Oregon La Grande zone (m) [2609] NAD83(CORS96) / Oregon North [2610] NAD83(CORS96) / Oregon North (ft) [2611] NAD83(CORS96) / Oregon Ontario zone (ft) [2612] NAD83(CORS96) / Oregon Ontario zone (m) [2613] NAD83(CORS96) / Oregon Pendleton zone (ft) [2614] NAD83(CORS96) / Oregon Pendleton zone (m) [2615] NAD83(CORS96) / Oregon Pendleton-La Grande zone (ft) [2616] NAD83(CORS96) / Oregon Pendleton-La Grande zone (m) [2617] NAD83(CORS96) / Oregon Portland zone (ft) [2618] NAD83(CORS96) / Oregon Portland zone (m) [2619] NAD83(CORS96) / Oregon Salem zone (ft) [2620] NAD83(CORS96) / Oregon Salem zone (m) [2621] NAD83(CORS96) / Oregon Santiam Pass zone (ft) [2622] NAD83(CORS96) / Oregon Santiam Pass zone (m) [2623] NAD83(CORS96) / Oregon South [2624] NAD83(CORS96) / Oregon South (ft) [2625] NAD83(CORS96) / Puerto Rico and Virgin Is. [2626] NAD83(CSRS) / Alberta 10-TM (Forest) [2627] NAD83(CSRS) / Alberta 10-TM (Resource) [2628] NAD83(CSRS) / Alberta 3TM ref merid 111 W [2629] NAD83(CSRS) / Alberta 3TM ref merid 114 W [2630] NAD83(CSRS) / Alberta 3TM ref merid 117 W [2631] NAD83(CSRS) / Alberta 3TM ref merid 120 W [2632] NAD83(CSRS) / Alberta 3TM ref merid 120 W (deprecated) [2633] NAD83(CSRS) / BC Albers [2634] NAD83(CSRS) / Canada Atlas Lambert [2635] NAD83(CSRS) / EPSG Arctic zone 1-23 [2636] NAD83(CSRS) / EPSG Arctic zone 2-14 [2637] NAD83(CSRS) / EPSG Arctic zone 2-16 [2638] NAD83(CSRS) / EPSG Arctic zone 3-25 [2639] NAD83(CSRS) / EPSG Arctic zone 3-27 [2640] NAD83(CSRS) / EPSG Arctic zone 3-29 [2641] NAD83(CSRS) / EPSG Arctic zone 4-14 [2642] NAD83(CSRS) / EPSG Arctic zone 4-16 [2643] NAD83(CSRS) / EPSG Arctic zone 4-18 [2644] NAD83(CSRS) / EPSG Arctic zone 5-33 [2645] NAD83(CSRS) / EPSG Arctic zone 5-35 [2646] NAD83(CSRS) / EPSG Arctic zone 5-37 [2647] NAD83(CSRS) / EPSG Arctic zone 5-39 [2648] NAD83(CSRS) / EPSG Arctic zone 6-18 [2649] NAD83(CSRS) / EPSG Arctic zone 6-20 [2650] NAD83(CSRS) / EPSG Arctic zone 6-22 [2651] NAD83(CSRS) / EPSG Arctic zone 6-24 [2652] NAD83(CSRS) / MTM zone 1 [2653] NAD83(CSRS) / MTM zone 10 [2654] NAD83(CSRS) / MTM zone 11 [2655] NAD83(CSRS) / MTM zone 12 [2656] NAD83(CSRS) / MTM zone 13 [2657] NAD83(CSRS) / MTM zone 14 [2658] NAD83(CSRS) / MTM zone 15 [2659] NAD83(CSRS) / MTM zone 16 [2660] NAD83(CSRS) / MTM zone 17 [2661] NAD83(CSRS) / MTM zone 2 [2662] NAD83(CSRS) / MTM zone 3 [2663] NAD83(CSRS) / MTM zone 4 [2664] NAD83(CSRS) / MTM zone 5 [2665] NAD83(CSRS) / MTM zone 6 [2666] NAD83(CSRS) / MTM zone 7 [2667] NAD83(CSRS) / MTM zone 8 [2668] NAD83(CSRS) / MTM zone 9 [2669] NAD83(CSRS) / MTQ Lambert [2670] NAD83(CSRS) / NWT Lambert [2671] NAD83(CSRS) / New Brunswick Stereographic [2672] NAD83(CSRS) / Ontario MNR Lambert [2673] NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83) [2674] NAD83(CSRS) / Quebec Albers [2675] NAD83(CSRS) / Quebec Lambert [2676] NAD83(CSRS) / SCoPQ zone 2 (deprecated) [2677] NAD83(CSRS) / Statistics Canada Lambert [2678] NAD83(CSRS) / Teranet Ontario Lambert [2679] NAD83(CSRS) / UTM zone 10N [2680] NAD83(CSRS) / UTM zone 11N [2681] NAD83(CSRS) / UTM zone 12N [2682] NAD83(CSRS) / UTM zone 13N [2683] NAD83(CSRS) / UTM zone 14N [2684] NAD83(CSRS) / UTM zone 15N [2685] NAD83(CSRS) / UTM zone 16N [2686] NAD83(CSRS) / UTM zone 17N [2687] NAD83(CSRS) / UTM zone 18N [2688] NAD83(CSRS) / UTM zone 19N [2689] NAD83(CSRS) / UTM zone 20N [2690] NAD83(CSRS) / UTM zone 21N [2691] NAD83(CSRS) / UTM zone 22N [2692] NAD83(CSRS) / UTM zone 7N [2693] NAD83(CSRS) / UTM zone 8N [2694] NAD83(CSRS) / UTM zone 9N [2695] NAD83(CSRS) / Yukon Albers [2696] NAD83(CSRS)v6 / MTM Nova Scotia zone 4 [2697] NAD83(CSRS)v6 / MTM Nova Scotia zone 5 [2698] NAD83(CSRS98) / MTM zone 10 (deprecated) [2699] NAD83(CSRS98) / MTM zone 3 (deprecated) [2700] NAD83(CSRS98) / MTM zone 4 (deprecated) [2701] NAD83(CSRS98) / MTM zone 5 (deprecated) [2702] NAD83(CSRS98) / MTM zone 6 (deprecated) [2703] NAD83(CSRS98) / MTM zone 7 (deprecated) [2704] NAD83(CSRS98) / MTM zone 8 (deprecated) [2705] NAD83(CSRS98) / MTM zone 9 (deprecated) [2706] NAD83(CSRS98) / New Brunswick Stereo (deprecated) [2707] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2708] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2709] NAD83(CSRS98) / SCoPQ zone 2 (deprecated) [2710] NAD83(CSRS98) / UTM zone 11N (deprecated) [2711] NAD83(CSRS98) / UTM zone 12N (deprecated) [2712] NAD83(CSRS98) / UTM zone 13N (deprecated) [2713] NAD83(CSRS98) / UTM zone 17N (deprecated) [2714] NAD83(CSRS98) / UTM zone 18N (deprecated) [2715] NAD83(CSRS98) / UTM zone 19N (deprecated) [2716] NAD83(CSRS98) / UTM zone 20N (deprecated) [2717] NAD83(CSRS98) / UTM zone 21N (deprecated) [2718] NAD83(HARN) / Alabama East [2719] NAD83(HARN) / Alabama West [2720] NAD83(HARN) / Arizona Central [2721] NAD83(HARN) / Arizona Central (ft) [2722] NAD83(HARN) / Arizona East [2723] NAD83(HARN) / Arizona East (ft) [2724] NAD83(HARN) / Arizona West [2725] NAD83(HARN) / Arizona West (ft) [2726] NAD83(HARN) / Arkansas North [2727] NAD83(HARN) / Arkansas North (ftUS) [2728] NAD83(HARN) / Arkansas South [2729] NAD83(HARN) / Arkansas South (ftUS) [2730] NAD83(HARN) / California Albers [2731] NAD83(HARN) / California zone 1 [2732] NAD83(HARN) / California zone 1 (ftUS) [2733] NAD83(HARN) / California zone 2 [2734] NAD83(HARN) / California zone 2 (ftUS) [2735] NAD83(HARN) / California zone 3 [2736] NAD83(HARN) / California zone 3 (ftUS) [2737] NAD83(HARN) / California zone 4 [2738] NAD83(HARN) / California zone 4 (ftUS) [2739] NAD83(HARN) / California zone 5 [2740] NAD83(HARN) / California zone 5 (ftUS) [2741] NAD83(HARN) / California zone 6 [2742] NAD83(HARN) / California zone 6 (ftUS) [2743] NAD83(HARN) / Colorado Central [2744] NAD83(HARN) / Colorado Central (ftUS) [2745] NAD83(HARN) / Colorado North [2746] NAD83(HARN) / Colorado North (ftUS) [2747] NAD83(HARN) / Colorado South [2748] NAD83(HARN) / Colorado South (ftUS) [2749] NAD83(HARN) / Connecticut [2750] NAD83(HARN) / Connecticut (ftUS) [2751] NAD83(HARN) / Conus Albers [2752] NAD83(HARN) / Delaware [2753] NAD83(HARN) / Delaware (ftUS) [2754] NAD83(HARN) / Florida East [2755] NAD83(HARN) / Florida East (ftUS) [2756] NAD83(HARN) / Florida GDL Albers [2757] NAD83(HARN) / Florida North [2758] NAD83(HARN) / Florida North (ftUS) [2759] NAD83(HARN) / Florida West [2760] NAD83(HARN) / Florida West (ftUS) [2761] NAD83(HARN) / Georgia East [2762] NAD83(HARN) / Georgia East (ftUS) [2763] NAD83(HARN) / Georgia West [2764] NAD83(HARN) / Georgia West (ftUS) [2765] NAD83(HARN) / Guam Map Grid [2766] NAD83(HARN) / Hawaii zone 1 [2767] NAD83(HARN) / Hawaii zone 2 [2768] NAD83(HARN) / Hawaii zone 3 [2769] NAD83(HARN) / Hawaii zone 3 (ftUS) [2770] NAD83(HARN) / Hawaii zone 4 [2771] NAD83(HARN) / Hawaii zone 5 [2772] NAD83(HARN) / Idaho Central [2773] NAD83(HARN) / Idaho Central (ftUS) [2774] NAD83(HARN) / Idaho East [2775] NAD83(HARN) / Idaho East (ftUS) [2776] NAD83(HARN) / Idaho West [2777] NAD83(HARN) / Idaho West (ftUS) [2778] NAD83(HARN) / Illinois East [2779] NAD83(HARN) / Illinois East (ftUS) [2780] NAD83(HARN) / Illinois West [2781] NAD83(HARN) / Illinois West (ftUS) [2782] NAD83(HARN) / Indiana East [2783] NAD83(HARN) / Indiana East (ftUS) [2784] NAD83(HARN) / Indiana East (ftUS) (deprecated) [2785] NAD83(HARN) / Indiana West [2786] NAD83(HARN) / Indiana West (ftUS) [2787] NAD83(HARN) / Indiana West (ftUS) (deprecated) [2788] NAD83(HARN) / Iowa North [2789] NAD83(HARN) / Iowa North (ftUS) [2790] NAD83(HARN) / Iowa South [2791] NAD83(HARN) / Iowa South (ftUS) [2792] NAD83(HARN) / Kansas North [2793] NAD83(HARN) / Kansas North (ftUS) [2794] NAD83(HARN) / Kansas South [2795] NAD83(HARN) / Kansas South (ftUS) [2796] NAD83(HARN) / Kentucky North [2797] NAD83(HARN) / Kentucky North (ftUS) [2798] NAD83(HARN) / Kentucky Single Zone [2799] NAD83(HARN) / Kentucky Single Zone (ftUS) [2800] NAD83(HARN) / Kentucky South [2801] NAD83(HARN) / Kentucky South (ftUS) [2802] NAD83(HARN) / Louisiana North [2803] NAD83(HARN) / Louisiana North (ftUS) [2804] NAD83(HARN) / Louisiana South [2805] NAD83(HARN) / Louisiana South (ftUS) [2806] NAD83(HARN) / Maine CS2000 Central [2807] NAD83(HARN) / Maine CS2000 Central (deprecated) [2808] NAD83(HARN) / Maine CS2000 East [2809] NAD83(HARN) / Maine CS2000 West [2810] NAD83(HARN) / Maine East [2811] NAD83(HARN) / Maine East (ftUS) [2812] NAD83(HARN) / Maine East (ftUS) (deprecated) [2813] NAD83(HARN) / Maine West [2814] NAD83(HARN) / Maine West (ftUS) [2815] NAD83(HARN) / Maine West (ftUS) (deprecated) [2816] NAD83(HARN) / Maryland [2817] NAD83(HARN) / Maryland (ftUS) [2818] NAD83(HARN) / Massachusetts Island [2819] NAD83(HARN) / Massachusetts Island (ftUS) [2820] NAD83(HARN) / Massachusetts Mainland [2821] NAD83(HARN) / Massachusetts Mainland (ftUS) [2822] NAD83(HARN) / Michigan Central [2823] NAD83(HARN) / Michigan Central (ft) [2824] NAD83(HARN) / Michigan North [2825] NAD83(HARN) / Michigan North (ft) [2826] NAD83(HARN) / Michigan Oblique Mercator [2827] NAD83(HARN) / Michigan South [2828] NAD83(HARN) / Michigan South (ft) [2829] NAD83(HARN) / Minnesota Central [2830] NAD83(HARN) / Minnesota Central (ftUS) [2831] NAD83(HARN) / Minnesota Central (ftUS) (deprecated) [2832] NAD83(HARN) / Minnesota North [2833] NAD83(HARN) / Minnesota North (ftUS) [2834] NAD83(HARN) / Minnesota North (ftUS) (deprecated) [2835] NAD83(HARN) / Minnesota South [2836] NAD83(HARN) / Minnesota South (ftUS) [2837] NAD83(HARN) / Minnesota South (ftUS) (deprecated) [2838] NAD83(HARN) / Mississippi East [2839] NAD83(HARN) / Mississippi East (ftUS) [2840] NAD83(HARN) / Mississippi TM [2841] NAD83(HARN) / Mississippi West [2842] NAD83(HARN) / Mississippi West (ftUS) [2843] NAD83(HARN) / Missouri Central [2844] NAD83(HARN) / Missouri East [2845] NAD83(HARN) / Missouri West [2846] NAD83(HARN) / Montana [2847] NAD83(HARN) / Montana (ft) [2848] NAD83(HARN) / Nebraska [2849] NAD83(HARN) / Nebraska (ftUS) [2850] NAD83(HARN) / Nebraska (ftUS) (deprecated) [2851] NAD83(HARN) / Nevada Central [2852] NAD83(HARN) / Nevada Central (ftUS) [2853] NAD83(HARN) / Nevada East [2854] NAD83(HARN) / Nevada East (ftUS) [2855] NAD83(HARN) / Nevada West [2856] NAD83(HARN) / Nevada West (ftUS) [2857] NAD83(HARN) / New Hampshire [2858] NAD83(HARN) / New Hampshire (ftUS) [2859] NAD83(HARN) / New Jersey [2860] NAD83(HARN) / New Jersey (ftUS) [2861] NAD83(HARN) / New Mexico Central [2862] NAD83(HARN) / New Mexico Central (ftUS) [2863] NAD83(HARN) / New Mexico East [2864] NAD83(HARN) / New Mexico East (ftUS) [2865] NAD83(HARN) / New Mexico West [2866] NAD83(HARN) / New Mexico West (ftUS) [2867] NAD83(HARN) / New York Central [2868] NAD83(HARN) / New York Central (ftUS) [2869] NAD83(HARN) / New York East [2870] NAD83(HARN) / New York East (ftUS) [2871] NAD83(HARN) / New York Long Island [2872] NAD83(HARN) / New York Long Island (ftUS) [2873] NAD83(HARN) / New York West [2874] NAD83(HARN) / New York West (ftUS) [2875] NAD83(HARN) / North Carolina [2876] NAD83(HARN) / North Carolina (ftUS) [2877] NAD83(HARN) / North Carolina (ftUS) (deprecated) [2878] NAD83(HARN) / North Dakota North [2879] NAD83(HARN) / North Dakota North (ft) [2880] NAD83(HARN) / North Dakota South [2881] NAD83(HARN) / North Dakota South (ft) [2882] NAD83(HARN) / Ohio North [2883] NAD83(HARN) / Ohio North (ftUS) [2884] NAD83(HARN) / Ohio South [2885] NAD83(HARN) / Ohio South (ftUS) [2886] NAD83(HARN) / Oklahoma North [2887] NAD83(HARN) / Oklahoma North (ftUS) [2888] NAD83(HARN) / Oklahoma South [2889] NAD83(HARN) / Oklahoma South (ftUS) [2890] NAD83(HARN) / Oregon GIC Lambert (ft) [2891] NAD83(HARN) / Oregon LCC (m) [2892] NAD83(HARN) / Oregon North [2893] NAD83(HARN) / Oregon North (ft) [2894] NAD83(HARN) / Oregon South [2895] NAD83(HARN) / Oregon South (ft) [2896] NAD83(HARN) / Pennsylvania North [2897] NAD83(HARN) / Pennsylvania North (ftUS) [2898] NAD83(HARN) / Pennsylvania South [2899] NAD83(HARN) / Pennsylvania South (ftUS) [2900] NAD83(HARN) / Puerto Rico and Virgin Is. [2901] NAD83(HARN) / Rhode Island [2902] NAD83(HARN) / Rhode Island (ftUS) [2903] NAD83(HARN) / South Carolina [2904] NAD83(HARN) / South Carolina (ft) [2905] NAD83(HARN) / South Dakota North [2906] NAD83(HARN) / South Dakota North (ftUS) [2907] NAD83(HARN) / South Dakota South [2908] NAD83(HARN) / South Dakota South (ftUS) [2909] NAD83(HARN) / Tennessee [2910] NAD83(HARN) / Tennessee (ftUS) [2911] NAD83(HARN) / Texas Central [2912] NAD83(HARN) / Texas Central (ftUS) [2913] NAD83(HARN) / Texas Centric Albers Equal Area [2914] NAD83(HARN) / Texas Centric Lambert Conformal [2915] NAD83(HARN) / Texas North [2916] NAD83(HARN) / Texas North (ftUS) [2917] NAD83(HARN) / Texas North Central [2918] NAD83(HARN) / Texas North Central (ftUS) [2919] NAD83(HARN) / Texas South [2920] NAD83(HARN) / Texas South (ftUS) [2921] NAD83(HARN) / Texas South Central [2922] NAD83(HARN) / Texas South Central (ftUS) [2923] NAD83(HARN) / UTM zone 10N [2924] NAD83(HARN) / UTM zone 11N [2925] NAD83(HARN) / UTM zone 12N [2926] NAD83(HARN) / UTM zone 13N [2927] NAD83(HARN) / UTM zone 14N [2928] NAD83(HARN) / UTM zone 15N [2929] NAD83(HARN) / UTM zone 16N [2930] NAD83(HARN) / UTM zone 17N [2931] NAD83(HARN) / UTM zone 18N [2932] NAD83(HARN) / UTM zone 19N [2933] NAD83(HARN) / UTM zone 2S [2934] NAD83(HARN) / UTM zone 4N [2935] NAD83(HARN) / UTM zone 59S (deprecated) [2936] NAD83(HARN) / UTM zone 5N [2937] NAD83(HARN) / Utah Central [2938] NAD83(HARN) / Utah Central (ft) [2939] NAD83(HARN) / Utah Central (ftUS) [2940] NAD83(HARN) / Utah North [2941] NAD83(HARN) / Utah North (ft) [2942] NAD83(HARN) / Utah North (ftUS) [2943] NAD83(HARN) / Utah South [2944] NAD83(HARN) / Utah South (ft) [2945] NAD83(HARN) / Utah South (ftUS) [2946] NAD83(HARN) / Vermont [2947] NAD83(HARN) / Vermont (ftUS) [2948] NAD83(HARN) / Virginia Lambert [2949] NAD83(HARN) / Virginia North [2950] NAD83(HARN) / Virginia North (ftUS) [2951] NAD83(HARN) / Virginia South [2952] NAD83(HARN) / Virginia South (ftUS) [2953] NAD83(HARN) / WISCRS Adams and Juneau (ftUS) [2954] NAD83(HARN) / WISCRS Adams and Juneau (m) [2955] NAD83(HARN) / WISCRS Ashland (ftUS) [2956] NAD83(HARN) / WISCRS Ashland (m) [2957] NAD83(HARN) / WISCRS Barron (ftUS) [2958] NAD83(HARN) / WISCRS Barron (m) [2959] NAD83(HARN) / WISCRS Bayfield (ftUS) [2960] NAD83(HARN) / WISCRS Bayfield (m) [2961] NAD83(HARN) / WISCRS Brown (ftUS) [2962] NAD83(HARN) / WISCRS Brown (m) [2963] NAD83(HARN) / WISCRS Buffalo (ftUS) [2964] NAD83(HARN) / WISCRS Buffalo (m) [2965] NAD83(HARN) / WISCRS Burnett (ftUS) [2966] NAD83(HARN) / WISCRS Burnett (m) [2967] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2968] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2969] NAD83(HARN) / WISCRS Chippewa (ftUS) [2970] NAD83(HARN) / WISCRS Chippewa (m) [2971] NAD83(HARN) / WISCRS Clark (ftUS) [2972] NAD83(HARN) / WISCRS Clark (m) [2973] NAD83(HARN) / WISCRS Columbia (ftUS) [2974] NAD83(HARN) / WISCRS Columbia (m) [2975] NAD83(HARN) / WISCRS Crawford (ftUS) [2976] NAD83(HARN) / WISCRS Crawford (m) [2977] NAD83(HARN) / WISCRS Dane (ftUS) [2978] NAD83(HARN) / WISCRS Dane (m) [2979] NAD83(HARN) / WISCRS Dodge and Jefferson (ftUS) [2980] NAD83(HARN) / WISCRS Dodge and Jefferson (m) [2981] NAD83(HARN) / WISCRS Door (ftUS) [2982] NAD83(HARN) / WISCRS Door (m) [2983] NAD83(HARN) / WISCRS Douglas (ftUS) [2984] NAD83(HARN) / WISCRS Douglas (m) [2985] NAD83(HARN) / WISCRS Dunn (ftUS) [2986] NAD83(HARN) / WISCRS Dunn (m) [2987] NAD83(HARN) / WISCRS Eau Claire (ftUS) [2988] NAD83(HARN) / WISCRS Eau Claire (m) [2989] NAD83(HARN) / WISCRS Florence (ftUS) [2990] NAD83(HARN) / WISCRS Florence (m) [2991] NAD83(HARN) / WISCRS Forest (ftUS) [2992] NAD83(HARN) / WISCRS Forest (m) [2993] NAD83(HARN) / WISCRS Grant (ftUS) [2994] NAD83(HARN) / WISCRS Grant (m) [2995] NAD83(HARN) / WISCRS Green Lake and Marquette (ftUS) [2996] NAD83(HARN) / WISCRS Green Lake and Marquette (m) [2997] NAD83(HARN) / WISCRS Green and Lafayette (ftUS) [2998] NAD83(HARN) / WISCRS Green and Lafayette (m) [2999] NAD83(HARN) / WISCRS Iowa (ftUS) [3000] NAD83(HARN) / WISCRS Iowa (m) [3001] NAD83(HARN) / WISCRS Iron (ftUS) [3002] NAD83(HARN) / WISCRS Iron (m) [3003] NAD83(HARN) / WISCRS Jackson (ftUS) [3004] NAD83(HARN) / WISCRS Jackson (m) [3005] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [3006] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [3007] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [3008] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [3009] NAD83(HARN) / WISCRS La Crosse (ftUS) [3010] NAD83(HARN) / WISCRS La Crosse (m) [3011] NAD83(HARN) / WISCRS Langlade (ftUS) [3012] NAD83(HARN) / WISCRS Langlade (m) [3013] NAD83(HARN) / WISCRS Lincoln (ftUS) [3014] NAD83(HARN) / WISCRS Lincoln (m) [3015] NAD83(HARN) / WISCRS Marathon (ftUS) [3016] NAD83(HARN) / WISCRS Marathon (m) [3017] NAD83(HARN) / WISCRS Marinette (ftUS) [3018] NAD83(HARN) / WISCRS Marinette (m) [3019] NAD83(HARN) / WISCRS Menominee (ftUS) [3020] NAD83(HARN) / WISCRS Menominee (m) [3021] NAD83(HARN) / WISCRS Monroe (ftUS) [3022] NAD83(HARN) / WISCRS Monroe (m) [3023] NAD83(HARN) / WISCRS Oconto (ftUS) [3024] NAD83(HARN) / WISCRS Oconto (m) [3025] NAD83(HARN) / WISCRS Oneida (ftUS) [3026] NAD83(HARN) / WISCRS Oneida (m) [3027] NAD83(HARN) / WISCRS Pepin and Pierce (ftUS) [3028] NAD83(HARN) / WISCRS Pepin and Pierce (m) [3029] NAD83(HARN) / WISCRS Polk (ftUS) [3030] NAD83(HARN) / WISCRS Polk (m) [3031] NAD83(HARN) / WISCRS Portage (ftUS) [3032] NAD83(HARN) / WISCRS Portage (m) [3033] NAD83(HARN) / WISCRS Price (ftUS) [3034] NAD83(HARN) / WISCRS Price (m) [3035] NAD83(HARN) / WISCRS Richland (ftUS) [3036] NAD83(HARN) / WISCRS Richland (m) [3037] NAD83(HARN) / WISCRS Rock (ftUS) [3038] NAD83(HARN) / WISCRS Rock (m) [3039] NAD83(HARN) / WISCRS Rusk (ftUS) [3040] NAD83(HARN) / WISCRS Rusk (m) [3041] NAD83(HARN) / WISCRS Sauk (ftUS) [3042] NAD83(HARN) / WISCRS Sauk (m) [3043] NAD83(HARN) / WISCRS Sawyer (ftUS) [3044] NAD83(HARN) / WISCRS Sawyer (m) [3045] NAD83(HARN) / WISCRS Shawano (ftUS) [3046] NAD83(HARN) / WISCRS Shawano (m) [3047] NAD83(HARN) / WISCRS St. Croix (ftUS) [3048] NAD83(HARN) / WISCRS St. Croix (m) [3049] NAD83(HARN) / WISCRS Taylor (ftUS) [3050] NAD83(HARN) / WISCRS Taylor (m) [3051] NAD83(HARN) / WISCRS Trempealeau (ftUS) [3052] NAD83(HARN) / WISCRS Trempealeau (m) [3053] NAD83(HARN) / WISCRS Vernon (ftUS) [3054] NAD83(HARN) / WISCRS Vernon (m) [3055] NAD83(HARN) / WISCRS Vilas (ftUS) [3056] NAD83(HARN) / WISCRS Vilas (m) [3057] NAD83(HARN) / WISCRS Walworth (ftUS) [3058] NAD83(HARN) / WISCRS Walworth (m) [3059] NAD83(HARN) / WISCRS Washburn (ftUS) [3060] NAD83(HARN) / WISCRS Washburn (m) [3061] NAD83(HARN) / WISCRS Washington (ftUS) [3062] NAD83(HARN) / WISCRS Washington (m) [3063] NAD83(HARN) / WISCRS Waukesha (ftUS) [3064] NAD83(HARN) / WISCRS Waukesha (m) [3065] NAD83(HARN) / WISCRS Waupaca (ftUS) [3066] NAD83(HARN) / WISCRS Waupaca (m) [3067] NAD83(HARN) / WISCRS Waushara (ftUS) [3068] NAD83(HARN) / WISCRS Waushara (m) [3069] NAD83(HARN) / WISCRS Wood (ftUS) [3070] NAD83(HARN) / WISCRS Wood (m) [3071] NAD83(HARN) / Washington North [3072] NAD83(HARN) / Washington North (ftUS) [3073] NAD83(HARN) / Washington South [3074] NAD83(HARN) / Washington South (ftUS) [3075] NAD83(HARN) / West Virginia North [3076] NAD83(HARN) / West Virginia North (ftUS) [3077] NAD83(HARN) / West Virginia North (ftUS) (deprecated) [3078] NAD83(HARN) / West Virginia South [3079] NAD83(HARN) / West Virginia South (ftUS) [3080] NAD83(HARN) / West Virginia South (ftUS) (deprecated) [3081] NAD83(HARN) / Wisconsin Central [3082] NAD83(HARN) / Wisconsin Central (ftUS) [3083] NAD83(HARN) / Wisconsin North [3084] NAD83(HARN) / Wisconsin North (ftUS) [3085] NAD83(HARN) / Wisconsin South [3086] NAD83(HARN) / Wisconsin South (ftUS) [3087] NAD83(HARN) / Wisconsin Transverse Mercator [3088] NAD83(HARN) / Wyoming East [3089] NAD83(HARN) / Wyoming East (ftUS) [3090] NAD83(HARN) / Wyoming East Central [3091] NAD83(HARN) / Wyoming East Central (ftUS) [3092] NAD83(HARN) / Wyoming West [3093] NAD83(HARN) / Wyoming West (ftUS) [3094] NAD83(HARN) / Wyoming West Central [3095] NAD83(HARN) / Wyoming West Central (ftUS) [3096] NAD83(MA11) / Guam Map Grid [3097] NAD83(MA11) / UTM zone 54N [3098] NAD83(MA11) / UTM zone 55N [3099] NAD83(NSRS2007) / Alabama East [3100] NAD83(NSRS2007) / Alabama West [3101] NAD83(NSRS2007) / Alaska Albers [3102] NAD83(NSRS2007) / Alaska zone 1 [3103] NAD83(NSRS2007) / Alaska zone 10 [3104] NAD83(NSRS2007) / Alaska zone 2 [3105] NAD83(NSRS2007) / Alaska zone 3 [3106] NAD83(NSRS2007) / Alaska zone 4 [3107] NAD83(NSRS2007) / Alaska zone 5 [3108] NAD83(NSRS2007) / Alaska zone 6 [3109] NAD83(NSRS2007) / Alaska zone 7 [3110] NAD83(NSRS2007) / Alaska zone 8 [3111] NAD83(NSRS2007) / Alaska zone 9 [3112] NAD83(NSRS2007) / Arizona Central [3113] NAD83(NSRS2007) / Arizona Central (ft) [3114] NAD83(NSRS2007) / Arizona East [3115] NAD83(NSRS2007) / Arizona East (ft) [3116] NAD83(NSRS2007) / Arizona West [3117] NAD83(NSRS2007) / Arizona West (ft) [3118] NAD83(NSRS2007) / Arkansas North [3119] NAD83(NSRS2007) / Arkansas North (ftUS) [3120] NAD83(NSRS2007) / Arkansas South [3121] NAD83(NSRS2007) / Arkansas South (ftUS) [3122] NAD83(NSRS2007) / California Albers [3123] NAD83(NSRS2007) / California zone 1 [3124] NAD83(NSRS2007) / California zone 1 (ftUS) [3125] NAD83(NSRS2007) / California zone 2 [3126] NAD83(NSRS2007) / California zone 2 (ftUS) [3127] NAD83(NSRS2007) / California zone 3 [3128] NAD83(NSRS2007) / California zone 3 (ftUS) [3129] NAD83(NSRS2007) / California zone 4 [3130] NAD83(NSRS2007) / California zone 4 (ftUS) [3131] NAD83(NSRS2007) / California zone 5 [3132] NAD83(NSRS2007) / California zone 5 (ftUS) [3133] NAD83(NSRS2007) / California zone 6 [3134] NAD83(NSRS2007) / California zone 6 (ftUS) [3135] NAD83(NSRS2007) / Colorado Central [3136] NAD83(NSRS2007) / Colorado Central (ftUS) [3137] NAD83(NSRS2007) / Colorado North [3138] NAD83(NSRS2007) / Colorado North (ftUS) [3139] NAD83(NSRS2007) / Colorado South [3140] NAD83(NSRS2007) / Colorado South (ftUS) [3141] NAD83(NSRS2007) / Connecticut [3142] NAD83(NSRS2007) / Connecticut (ftUS) [3143] NAD83(NSRS2007) / Conus Albers [3144] NAD83(NSRS2007) / Delaware [3145] NAD83(NSRS2007) / Delaware (ftUS) [3146] NAD83(NSRS2007) / EPSG Arctic zone 5-29 [3147] NAD83(NSRS2007) / EPSG Arctic zone 5-31 [3148] NAD83(NSRS2007) / EPSG Arctic zone 6-14 [3149] NAD83(NSRS2007) / EPSG Arctic zone 6-16 [3150] NAD83(NSRS2007) / Florida East [3151] NAD83(NSRS2007) / Florida East (ftUS) [3152] NAD83(NSRS2007) / Florida GDL Albers [3153] NAD83(NSRS2007) / Florida North [3154] NAD83(NSRS2007) / Florida North (ftUS) [3155] NAD83(NSRS2007) / Florida West [3156] NAD83(NSRS2007) / Florida West (ftUS) [3157] NAD83(NSRS2007) / Georgia East [3158] NAD83(NSRS2007) / Georgia East (ftUS) [3159] NAD83(NSRS2007) / Georgia West [3160] NAD83(NSRS2007) / Georgia West (ftUS) [3161] NAD83(NSRS2007) / Idaho Central [3162] NAD83(NSRS2007) / Idaho Central (ftUS) [3163] NAD83(NSRS2007) / Idaho East [3164] NAD83(NSRS2007) / Idaho East (ftUS) [3165] NAD83(NSRS2007) / Idaho West [3166] NAD83(NSRS2007) / Idaho West (ftUS) [3167] NAD83(NSRS2007) / Illinois East [3168] NAD83(NSRS2007) / Illinois East (ftUS) [3169] NAD83(NSRS2007) / Illinois West [3170] NAD83(NSRS2007) / Illinois West (ftUS) [3171] NAD83(NSRS2007) / Indiana East [3172] NAD83(NSRS2007) / Indiana East (ftUS) [3173] NAD83(NSRS2007) / Indiana West [3174] NAD83(NSRS2007) / Indiana West (ftUS) [3175] NAD83(NSRS2007) / Iowa North [3176] NAD83(NSRS2007) / Iowa North (ftUS) [3177] NAD83(NSRS2007) / Iowa South [3178] NAD83(NSRS2007) / Iowa South (ftUS) [3179] NAD83(NSRS2007) / Kansas North [3180] NAD83(NSRS2007) / Kansas North (ftUS) [3181] NAD83(NSRS2007) / Kansas South [3182] NAD83(NSRS2007) / Kansas South (ftUS) [3183] NAD83(NSRS2007) / Kentucky North [3184] NAD83(NSRS2007) / Kentucky North (ftUS) [3185] NAD83(NSRS2007) / Kentucky Single Zone [3186] NAD83(NSRS2007) / Kentucky Single Zone (ftUS) [3187] NAD83(NSRS2007) / Kentucky South [3188] NAD83(NSRS2007) / Kentucky South (ftUS) [3189] NAD83(NSRS2007) / Louisiana North [3190] NAD83(NSRS2007) / Louisiana North (ftUS) [3191] NAD83(NSRS2007) / Louisiana South [3192] NAD83(NSRS2007) / Louisiana South (ftUS) [3193] NAD83(NSRS2007) / Maine CS2000 Central [3194] NAD83(NSRS2007) / Maine CS2000 East [3195] NAD83(NSRS2007) / Maine CS2000 West [3196] NAD83(NSRS2007) / Maine East [3197] NAD83(NSRS2007) / Maine East (ftUS) [3198] NAD83(NSRS2007) / Maine East (ftUS) (deprecated) [3199] NAD83(NSRS2007) / Maine West [3200] NAD83(NSRS2007) / Maine West (ftUS) [3201] NAD83(NSRS2007) / Maine West (ftUS) (deprecated) [3202] NAD83(NSRS2007) / Maryland [3203] NAD83(NSRS2007) / Maryland (ftUS) [3204] NAD83(NSRS2007) / Massachusetts Island [3205] NAD83(NSRS2007) / Massachusetts Island (ftUS) [3206] NAD83(NSRS2007) / Massachusetts Mainland [3207] NAD83(NSRS2007) / Massachusetts Mainland (ftUS) [3208] NAD83(NSRS2007) / Michigan Central [3209] NAD83(NSRS2007) / Michigan Central (ft) [3210] NAD83(NSRS2007) / Michigan North [3211] NAD83(NSRS2007) / Michigan North (ft) [3212] NAD83(NSRS2007) / Michigan Oblique Mercator [3213] NAD83(NSRS2007) / Michigan South [3214] NAD83(NSRS2007) / Michigan South (ft) [3215] NAD83(NSRS2007) / Minnesota Central [3216] NAD83(NSRS2007) / Minnesota Central (ftUS) [3217] NAD83(NSRS2007) / Minnesota Central (ftUS) (deprecated) [3218] NAD83(NSRS2007) / Minnesota North [3219] NAD83(NSRS2007) / Minnesota North (ftUS) [3220] NAD83(NSRS2007) / Minnesota North (ftUS) (deprecated) [3221] NAD83(NSRS2007) / Minnesota South [3222] NAD83(NSRS2007) / Minnesota South (ftUS) [3223] NAD83(NSRS2007) / Minnesota South (ftUS) (deprecated) [3224] NAD83(NSRS2007) / Mississippi East [3225] NAD83(NSRS2007) / Mississippi East (ftUS) [3226] NAD83(NSRS2007) / Mississippi TM [3227] NAD83(NSRS2007) / Mississippi West [3228] NAD83(NSRS2007) / Mississippi West (ftUS) [3229] NAD83(NSRS2007) / Missouri Central [3230] NAD83(NSRS2007) / Missouri East [3231] NAD83(NSRS2007) / Missouri West [3232] NAD83(NSRS2007) / Montana [3233] NAD83(NSRS2007) / Montana (ft) [3234] NAD83(NSRS2007) / Nebraska [3235] NAD83(NSRS2007) / Nebraska (ftUS) [3236] NAD83(NSRS2007) / Nebraska (ftUS) (deprecated) [3237] NAD83(NSRS2007) / Nevada Central [3238] NAD83(NSRS2007) / Nevada Central (ftUS) [3239] NAD83(NSRS2007) / Nevada East [3240] NAD83(NSRS2007) / Nevada East (ftUS) [3241] NAD83(NSRS2007) / Nevada West [3242] NAD83(NSRS2007) / Nevada West (ftUS) [3243] NAD83(NSRS2007) / New Hampshire [3244] NAD83(NSRS2007) / New Hampshire (ftUS) [3245] NAD83(NSRS2007) / New Jersey [3246] NAD83(NSRS2007) / New Jersey (ftUS) [3247] NAD83(NSRS2007) / New Mexico Central [3248] NAD83(NSRS2007) / New Mexico Central (ftUS) [3249] NAD83(NSRS2007) / New Mexico East [3250] NAD83(NSRS2007) / New Mexico East (ftUS) [3251] NAD83(NSRS2007) / New Mexico West [3252] NAD83(NSRS2007) / New Mexico West (ftUS) [3253] NAD83(NSRS2007) / New York Central [3254] NAD83(NSRS2007) / New York Central (ftUS) [3255] NAD83(NSRS2007) / New York East [3256] NAD83(NSRS2007) / New York East (ftUS) [3257] NAD83(NSRS2007) / New York Long Island [3258] NAD83(NSRS2007) / New York Long Island (ftUS) [3259] NAD83(NSRS2007) / New York West [3260] NAD83(NSRS2007) / New York West (ftUS) [3261] NAD83(NSRS2007) / North Carolina [3262] NAD83(NSRS2007) / North Carolina (ftUS) [3263] NAD83(NSRS2007) / North Dakota North [3264] NAD83(NSRS2007) / North Dakota North (ft) [3265] NAD83(NSRS2007) / North Dakota South [3266] NAD83(NSRS2007) / North Dakota South (ft) [3267] NAD83(NSRS2007) / Ohio North [3268] NAD83(NSRS2007) / Ohio North (ftUS) [3269] NAD83(NSRS2007) / Ohio South [3270] NAD83(NSRS2007) / Ohio South (ftUS) [3271] NAD83(NSRS2007) / Oklahoma North [3272] NAD83(NSRS2007) / Oklahoma North (ftUS) [3273] NAD83(NSRS2007) / Oklahoma South [3274] NAD83(NSRS2007) / Oklahoma South (ftUS) [3275] NAD83(NSRS2007) / Oregon GIC Lambert (ft) [3276] NAD83(NSRS2007) / Oregon LCC (m) [3277] NAD83(NSRS2007) / Oregon North [3278] NAD83(NSRS2007) / Oregon North (ft) [3279] NAD83(NSRS2007) / Oregon South [3280] NAD83(NSRS2007) / Oregon South (ft) [3281] NAD83(NSRS2007) / Pennsylvania North [3282] NAD83(NSRS2007) / Pennsylvania North (ftUS) [3283] NAD83(NSRS2007) / Pennsylvania South [3284] NAD83(NSRS2007) / Pennsylvania South (ftUS) [3285] NAD83(NSRS2007) / Puerto Rico and Virgin Is. [3286] NAD83(NSRS2007) / Rhode Island [3287] NAD83(NSRS2007) / Rhode Island (ftUS) [3288] NAD83(NSRS2007) / South Carolina [3289] NAD83(NSRS2007) / South Carolina (ft) [3290] NAD83(NSRS2007) / South Dakota North [3291] NAD83(NSRS2007) / South Dakota North (ftUS) [3292] NAD83(NSRS2007) / South Dakota South [3293] NAD83(NSRS2007) / South Dakota South (ftUS) [3294] NAD83(NSRS2007) / Tennessee [3295] NAD83(NSRS2007) / Tennessee (ftUS) [3296] NAD83(NSRS2007) / Texas Central [3297] NAD83(NSRS2007) / Texas Central (ftUS) [3298] NAD83(NSRS2007) / Texas Centric Albers Equal Area [3299] NAD83(NSRS2007) / Texas Centric Lambert Conformal [3300] NAD83(NSRS2007) / Texas North [3301] NAD83(NSRS2007) / Texas North (ftUS) [3302] NAD83(NSRS2007) / Texas North Central [3303] NAD83(NSRS2007) / Texas North Central (ftUS) [3304] NAD83(NSRS2007) / Texas South [3305] NAD83(NSRS2007) / Texas South (ftUS) [3306] NAD83(NSRS2007) / Texas South Central [3307] NAD83(NSRS2007) / Texas South Central (ftUS) [3308] NAD83(NSRS2007) / UTM zone 10N [3309] NAD83(NSRS2007) / UTM zone 11N [3310] NAD83(NSRS2007) / UTM zone 12N [3311] NAD83(NSRS2007) / UTM zone 13N [3312] NAD83(NSRS2007) / UTM zone 14N [3313] NAD83(NSRS2007) / UTM zone 15N [3314] NAD83(NSRS2007) / UTM zone 16N [3315] NAD83(NSRS2007) / UTM zone 17N [3316] NAD83(NSRS2007) / UTM zone 18N [3317] NAD83(NSRS2007) / UTM zone 19N [3318] NAD83(NSRS2007) / UTM zone 1N [3319] NAD83(NSRS2007) / UTM zone 2N [3320] NAD83(NSRS2007) / UTM zone 3N [3321] NAD83(NSRS2007) / UTM zone 4N [3322] NAD83(NSRS2007) / UTM zone 59N [3323] NAD83(NSRS2007) / UTM zone 5N [3324] NAD83(NSRS2007) / UTM zone 60N [3325] NAD83(NSRS2007) / UTM zone 6N [3326] NAD83(NSRS2007) / UTM zone 7N [3327] NAD83(NSRS2007) / UTM zone 8N [3328] NAD83(NSRS2007) / UTM zone 9N [3329] NAD83(NSRS2007) / Utah Central [3330] NAD83(NSRS2007) / Utah Central (ft) [3331] NAD83(NSRS2007) / Utah Central (ftUS) [3332] NAD83(NSRS2007) / Utah North [3333] NAD83(NSRS2007) / Utah North (ft) [3334] NAD83(NSRS2007) / Utah North (ftUS) [3335] NAD83(NSRS2007) / Utah South [3336] NAD83(NSRS2007) / Utah South (ft) [3337] NAD83(NSRS2007) / Utah South (ftUS) [3338] NAD83(NSRS2007) / Vermont [3339] NAD83(NSRS2007) / Vermont (ftUS) [3340] NAD83(NSRS2007) / Virginia Lambert [3341] NAD83(NSRS2007) / Virginia North [3342] NAD83(NSRS2007) / Virginia North (ftUS) [3343] NAD83(NSRS2007) / Virginia South [3344] NAD83(NSRS2007) / Virginia South (ftUS) [3345] NAD83(NSRS2007) / Washington North [3346] NAD83(NSRS2007) / Washington North (ftUS) [3347] NAD83(NSRS2007) / Washington South [3348] NAD83(NSRS2007) / Washington South (ftUS) [3349] NAD83(NSRS2007) / West Virginia North [3350] NAD83(NSRS2007) / West Virginia North (ftUS) [3351] NAD83(NSRS2007) / West Virginia North (ftUS) (deprecated) [3352] NAD83(NSRS2007) / West Virginia South [3353] NAD83(NSRS2007) / West Virginia South (ftUS) [3354] NAD83(NSRS2007) / West Virginia South (ftUS) (deprecated) [3355] NAD83(NSRS2007) / Wisconsin Central [3356] NAD83(NSRS2007) / Wisconsin Central (ftUS) [3357] NAD83(NSRS2007) / Wisconsin North [3358] NAD83(NSRS2007) / Wisconsin North (ftUS) [3359] NAD83(NSRS2007) / Wisconsin South [3360] NAD83(NSRS2007) / Wisconsin South (ftUS) [3361] NAD83(NSRS2007) / Wisconsin Transverse Mercator [3362] NAD83(NSRS2007) / Wyoming East [3363] NAD83(NSRS2007) / Wyoming East (ftUS) [3364] NAD83(NSRS2007) / Wyoming East Central [3365] NAD83(NSRS2007) / Wyoming East Central (ftUS) [3366] NAD83(NSRS2007) / Wyoming West [3367] NAD83(NSRS2007) / Wyoming West (ftUS) [3368] NAD83(NSRS2007) / Wyoming West Central [3369] NAD83(NSRS2007) / Wyoming West Central (ftUS) [3370] NAD83(PA11) / Hawaii zone 1 [3371] NAD83(PA11) / Hawaii zone 2 [3372] NAD83(PA11) / Hawaii zone 3 [3373] NAD83(PA11) / Hawaii zone 3 (ftUS) [3374] NAD83(PA11) / Hawaii zone 4 [3375] NAD83(PA11) / Hawaii zone 5 [3376] NAD83(PA11) / UTM zone 2S [3377] NAD83(PA11) / UTM zone 4N [3378] NAD83(PA11) / UTM zone 5N [3379] NAD_1927_10TM_AEP_Forest [3380] NAD_1927_10TM_AEP_Resource [3381] NAD_1927_3TM_111 (deprecated) [3382] NAD_1927_3TM_114 (deprecated) [3383] NAD_1927_3TM_117 (deprecated) [3384] NAD_1927_3TM_120 (deprecated) [3385] NAD_1927_Alaska_Albers_Meters [3386] NAD_1927_Georgia_Statewide_Albers [3387] NAD_1927_Michigan_GeoRef_Feet_US [3388] NAD_1927_Michigan_GeoRef_Meters [3389] NAD_1927_StatePlane_California_V_Ventura [3390] NAD_1927_StatePlane_Guam_FIPS_5400 [3391] NAD_1927_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [3392] NAD_1927_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [3393] NAD_1927_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [3394] NAD_1927_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [3395] NAD_1927_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [3396] NAD_1927_StatePlane_Michigan_Central_FIPS_2112 [3397] NAD_1927_StatePlane_Michigan_North_FIPS_2111 [3398] NAD_1927_StatePlane_Michigan_South_FIPS_2113 [3399] NAD_1927_StatePlane_Puerto_Rico_FIPS_5201 [3400] NAD_1927_StatePlane_Virgin_Islands_St_Croix_FIPS_5202 [3401] NAD_1927_Texas_Statewide_Mapping_System (deprecated) [3402] NAD_1927_UTM_Zone_1N (deprecated) [3403] NAD_1927_UTM_Zone_2N (deprecated) [3404] NAD_1927_UTM_Zone_59N (deprecated) [3405] NAD_1927_UTM_Zone_60N (deprecated) [3406] NAD_1983_10TM_AEP_Forest (deprecated) [3407] NAD_1983_10TM_AEP_Resource (deprecated) [3408] NAD_1983_2011_Alaska_Albers (deprecated) [3409] NAD_1983_2011_California_Teale_Albers (deprecated) [3410] NAD_1983_2011_Contiguous_USA_Albers (deprecated) [3411] NAD_1983_2011_Fargo_Ground_Coordinate_System [3412] NAD_1983_2011_Florida_GDL_Albers (deprecated) [3413] NAD_1983_2011_Maine_2000_Central_Zone (deprecated) [3414] NAD_1983_2011_Maine_2000_East_Zone (deprecated) [3415] NAD_1983_2011_Maine_2000_West_Zone (deprecated) [3416] NAD_1983_2011_Michigan_GeoRef_Meters (deprecated) [3417] NAD_1983_2011_Mississippi_TM (deprecated) [3418] NAD_1983_2011_Oregon_Statewide_Lambert (deprecated) [3419] NAD_1983_2011_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3420] NAD_1983_2011_StatePlane_Alabama_East_FIPS_0101 (deprecated) [3421] NAD_1983_2011_StatePlane_Alabama_West_FIPS_0102 (deprecated) [3422] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010 (deprecated) [3423] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010_Feet [3424] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001 (deprecated) [3425] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001_Feet [3426] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002 (deprecated) [3427] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002_Feet [3428] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003 (deprecated) [3429] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003_Feet [3430] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004 (deprecated) [3431] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004_Feet [3432] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005 (deprecated) [3433] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005_Feet [3434] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006 (deprecated) [3435] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006_Feet [3436] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007 (deprecated) [3437] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007_Feet [3438] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008 (deprecated) [3439] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008_Feet [3440] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009 (deprecated) [3441] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009_Feet [3442] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [3443] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl (deprecated) [3444] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201 (deprecated) [3445] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201_Ft_Intl (deprecated) [3446] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203 (deprecated) [3447] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203_Ft_Intl (deprecated) [3448] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [3449] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301_Ft_US (deprecated) [3450] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [3451] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302_Ft_US (deprecated) [3452] NAD_1983_2011_StatePlane_California_III_FIPS_0403 (deprecated) [3453] NAD_1983_2011_StatePlane_California_III_FIPS_0403_Ft_US (deprecated) [3454] NAD_1983_2011_StatePlane_California_II_FIPS_0402 (deprecated) [3455] NAD_1983_2011_StatePlane_California_II_FIPS_0402_Ft_US (deprecated) [3456] NAD_1983_2011_StatePlane_California_IV_FIPS_0404 (deprecated) [3457] NAD_1983_2011_StatePlane_California_IV_FIPS_0404_Ft_US (deprecated) [3458] NAD_1983_2011_StatePlane_California_I_FIPS_0401 (deprecated) [3459] NAD_1983_2011_StatePlane_California_I_FIPS_0401_Ft_US (deprecated) [3460] NAD_1983_2011_StatePlane_California_VI_FIPS_0406 (deprecated) [3461] NAD_1983_2011_StatePlane_California_VI_FIPS_0406_Ft_US (deprecated) [3462] NAD_1983_2011_StatePlane_California_V_FIPS_0405 (deprecated) [3463] NAD_1983_2011_StatePlane_California_V_FIPS_0405_Ft_US (deprecated) [3464] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [3465] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502_Ft_US (deprecated) [3466] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501 (deprecated) [3467] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501_Ft_US (deprecated) [3468] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503 (deprecated) [3469] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503_Ft_US (deprecated) [3470] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600 (deprecated) [3471] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600_Ft_US (deprecated) [3472] NAD_1983_2011_StatePlane_Delaware_FIPS_0700 (deprecated) [3473] NAD_1983_2011_StatePlane_Delaware_FIPS_0700_Ft_US (deprecated) [3474] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901 (deprecated) [3475] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901_Ft_US (deprecated) [3476] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903 (deprecated) [3477] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903_Ft_US (deprecated) [3478] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902 (deprecated) [3479] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902_Ft_US (deprecated) [3480] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001 (deprecated) [3481] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001_Ft_US (deprecated) [3482] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002 (deprecated) [3483] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002_Ft_US (deprecated) [3484] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [3485] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102_Ft_US (deprecated) [3486] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101 (deprecated) [3487] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101_Ft_US (deprecated) [3488] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103 (deprecated) [3489] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103_Ft_US (deprecated) [3490] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201 (deprecated) [3491] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201_Ft_US (deprecated) [3492] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202 (deprecated) [3493] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202_Ft_US (deprecated) [3494] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301 (deprecated) [3495] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301_Ft_US (deprecated) [3496] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302 (deprecated) [3497] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302_Ft_US (deprecated) [3498] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401 (deprecated) [3499] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401_Ft_US (deprecated) [3500] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402 (deprecated) [3501] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402_Ft_US (deprecated) [3502] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501 (deprecated) [3503] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501_Ft_US (deprecated) [3504] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502 (deprecated) [3505] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502_Ft_US (deprecated) [3506] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600 (deprecated) [3507] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600_Ft_US (deprecated) [3508] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [3509] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601_Ft_US (deprecated) [3510] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [3511] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602_Ft_US (deprecated) [3512] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [3513] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701_Ft_US (deprecated) [3514] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [3515] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702_Ft_US (deprecated) [3516] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801 (deprecated) [3517] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801_Ft_US (deprecated) [3518] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802 (deprecated) [3519] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802_Ft_US (deprecated) [3520] NAD_1983_2011_StatePlane_Maryland_FIPS_1900 (deprecated) [3521] NAD_1983_2011_StatePlane_Maryland_FIPS_1900_Ft_US (deprecated) [3522] NAD_1983_2011_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS (deprecated) [3523] NAD_1983_2011_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [3524] NAD_1983_2011_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [3525] NAD_1983_2011_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS (deprecated) [3526] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [3527] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl (deprecated) [3528] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111 (deprecated) [3529] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111_Ft_Intl (deprecated) [3530] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113 (deprecated) [3531] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113_Ft_Intl (deprecated) [3532] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [3533] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202_Ft_US (deprecated) [3534] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [3535] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201_Ft_US (deprecated) [3536] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [3537] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203_Ft_US (deprecated) [3538] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [3539] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301_Ft_US (deprecated) [3540] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [3541] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302_Ft_US (deprecated) [3542] NAD_1983_2011_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [3543] NAD_1983_2011_StatePlane_Missouri_East_FIPS_2401 (deprecated) [3544] NAD_1983_2011_StatePlane_Missouri_West_FIPS_2403 (deprecated) [3545] NAD_1983_2011_StatePlane_Montana_FIPS_2500 (deprecated) [3546] NAD_1983_2011_StatePlane_Montana_FIPS_2500_Ft_Intl (deprecated) [3547] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600 (deprecated) [3548] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600_Ft_US (deprecated) [3549] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [3550] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702_Ft_US (deprecated) [3551] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701 (deprecated) [3552] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701_Ft_US (deprecated) [3553] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703 (deprecated) [3554] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703_Ft_US (deprecated) [3555] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [3556] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800_Ft_US (deprecated) [3557] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900 (deprecated) [3558] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900_Ft_US (deprecated) [3559] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [3560] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US (deprecated) [3561] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [3562] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001_Ft_US (deprecated) [3563] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [3564] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003_Ft_US (deprecated) [3565] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102 (deprecated) [3566] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102_Ft_US (deprecated) [3567] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101 (deprecated) [3568] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101_Ft_US (deprecated) [3569] NAD_1983_2011_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US (deprecated) [3570] NAD_1983_2011_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [3571] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103 (deprecated) [3572] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103_Ft_US (deprecated) [3573] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200 (deprecated) [3574] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200_Ft_US (deprecated) [3575] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [3576] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301_FtI (deprecated) [3577] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [3578] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302_FtI (deprecated) [3579] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401 (deprecated) [3580] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401_Ft_US (deprecated) [3581] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402 (deprecated) [3582] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402_Ft_US (deprecated) [3583] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [3584] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501_Ft_US (deprecated) [3585] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [3586] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502_Ft_US (deprecated) [3587] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3588] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3589] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3590] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3591] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701 (deprecated) [3592] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US (deprecated) [3593] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702 (deprecated) [3594] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US (deprecated) [3595] NAD_1983_2011_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [3596] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [3597] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800_Ft_US (deprecated) [3598] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900 (deprecated) [3599] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900_Ft_Intl (deprecated) [3600] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [3601] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001_Ft_US (deprecated) [3602] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [3603] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002_Ft_US (deprecated) [3604] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100 (deprecated) [3605] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100_Ft_US (deprecated) [3606] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203 (deprecated) [3607] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203_Ft_US (deprecated) [3608] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [3609] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202_FtUS (deprecated) [3610] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201 (deprecated) [3611] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201_Ft_US (deprecated) [3612] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [3613] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204_FtUS (deprecated) [3614] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205 (deprecated) [3615] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205_Ft_US (deprecated) [3616] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302 (deprecated) [3617] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3618] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_US (deprecated) [3619] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301 (deprecated) [3620] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3621] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_US (deprecated) [3622] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303 (deprecated) [3623] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3624] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_US (deprecated) [3625] NAD_1983_2011_StatePlane_Vermont_FIPS_4400 (deprecated) [3626] NAD_1983_2011_StatePlane_Vermont_FIPS_4400_Ft_US (deprecated) [3627] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501 (deprecated) [3628] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501_Ft_US (deprecated) [3629] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502 (deprecated) [3630] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502_Ft_US (deprecated) [3631] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601 (deprecated) [3632] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601_Ft_US (deprecated) [3633] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602 (deprecated) [3634] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602_Ft_US (deprecated) [3635] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [3636] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701_FtUS (deprecated) [3637] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [3638] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702_FtUS (deprecated) [3639] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [3640] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US (deprecated) [3641] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [3642] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801_Ft_US (deprecated) [3643] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [3644] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803_Ft_US (deprecated) [3645] NAD_1983_2011_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US (deprecated) [3646] NAD_1983_2011_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [3647] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [3648] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901_Ft_US (deprecated) [3649] NAD_1983_2011_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US (deprecated) [3650] NAD_1983_2011_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [3651] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [3652] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904_Ft_US (deprecated) [3653] NAD_1983_2011_Texas_Centric_Mapping_System_Albers (deprecated) [3654] NAD_1983_2011_Texas_Centric_Mapping_System_Lambert (deprecated) [3655] NAD_1983_2011_UTM_Zone_10N (deprecated) [3656] NAD_1983_2011_UTM_Zone_11N (deprecated) [3657] NAD_1983_2011_UTM_Zone_12N (deprecated) [3658] NAD_1983_2011_UTM_Zone_13N (deprecated) [3659] NAD_1983_2011_UTM_Zone_14N (deprecated) [3660] NAD_1983_2011_UTM_Zone_15N (deprecated) [3661] NAD_1983_2011_UTM_Zone_16N (deprecated) [3662] NAD_1983_2011_UTM_Zone_17N (deprecated) [3663] NAD_1983_2011_UTM_Zone_18N (deprecated) [3664] NAD_1983_2011_UTM_Zone_19N (deprecated) [3665] NAD_1983_2011_UTM_Zone_1N (deprecated) [3666] NAD_1983_2011_UTM_Zone_20N [3667] NAD_1983_2011_UTM_Zone_2N (deprecated) [3668] NAD_1983_2011_UTM_Zone_3N (deprecated) [3669] NAD_1983_2011_UTM_Zone_4N (deprecated) [3670] NAD_1983_2011_UTM_Zone_59N (deprecated) [3671] NAD_1983_2011_UTM_Zone_5N (deprecated) [3672] NAD_1983_2011_UTM_Zone_60N (deprecated) [3673] NAD_1983_2011_UTM_Zone_6N (deprecated) [3674] NAD_1983_2011_UTM_Zone_7N (deprecated) [3675] NAD_1983_2011_UTM_Zone_8N (deprecated) [3676] NAD_1983_2011_UTM_Zone_9N (deprecated) [3677] NAD_1983_2011_Wisconsin_TM (deprecated) [3678] NAD_1983_2011_Wisconsin_TM_US_Ft [3679] NAD_1983_3TM_111 (deprecated) [3680] NAD_1983_3TM_114 (deprecated) [3681] NAD_1983_3TM_117 (deprecated) [3682] NAD_1983_3TM_120 (deprecated) [3683] NAD_1983_Alaska_Albers (deprecated) [3684] NAD_1983_Albers_BLM_MT_ND_SD [3685] NAD_1983_BC_Environment_Albers (deprecated) [3686] NAD_1983_CORS96_Alaska_Albers [3687] NAD_1983_CORS96_Maine_2000_Central_Zone [3688] NAD_1983_CORS96_Maine_2000_East_Zone [3689] NAD_1983_CORS96_Maine_2000_West_Zone [3690] NAD_1983_CORS96_Oregon_Statewide_Lambert (deprecated) [3691] NAD_1983_CORS96_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3692] NAD_1983_CORS96_StatePlane_Alabama_East_FIPS_0101 [3693] NAD_1983_CORS96_StatePlane_Alabama_West_FIPS_0102 [3694] NAD_1983_CORS96_StatePlane_Alaska_10_FIPS_5010 [3695] NAD_1983_CORS96_StatePlane_Alaska_1_FIPS_5001 [3696] NAD_1983_CORS96_StatePlane_Alaska_2_FIPS_5002 [3697] NAD_1983_CORS96_StatePlane_Alaska_3_FIPS_5003 [3698] NAD_1983_CORS96_StatePlane_Alaska_4_FIPS_5004 [3699] NAD_1983_CORS96_StatePlane_Alaska_5_FIPS_5005 [3700] NAD_1983_CORS96_StatePlane_Alaska_6_FIPS_5006 [3701] NAD_1983_CORS96_StatePlane_Alaska_7_FIPS_5007 [3702] NAD_1983_CORS96_StatePlane_Alaska_8_FIPS_5008 [3703] NAD_1983_CORS96_StatePlane_Alaska_9_FIPS_5009 [3704] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202 [3705] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl [3706] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201 [3707] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201_Ft_Intl [3708] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203 [3709] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203_Ft_Intl [3710] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301 [3711] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301_Ft_US [3712] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302 [3713] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302_Ft_US [3714] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403 [3715] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403_Ft_US [3716] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402 [3717] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402_Ft_US [3718] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404 [3719] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404_Ft_US [3720] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401 [3721] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401_Ft_US [3722] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406 [3723] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406_Ft_US [3724] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405 [3725] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405_Ft_US [3726] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502 [3727] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502_Ft_US [3728] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501 [3729] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501_Ft_US [3730] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503 [3731] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503_Ft_US [3732] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600 [3733] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600_Ft_US [3734] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700 [3735] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700_Ft_US [3736] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901 [3737] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901_Ft_US [3738] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903 [3739] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903_Ft_US [3740] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902 [3741] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902_Ft_US [3742] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001 [3743] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001_Ft_US [3744] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002 [3745] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002_Ft_US [3746] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102 [3747] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102_Ft_US [3748] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101 [3749] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101_Ft_US [3750] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103 [3751] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103_Ft_US [3752] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201 [3753] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201_Ft_US [3754] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202 [3755] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202_Ft_US [3756] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301 [3757] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301_Ft_US [3758] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302 [3759] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302_Ft_US [3760] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401 [3761] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401_Ft_US [3762] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402 [3763] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402_Ft_US [3764] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501 [3765] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501_Ft_US [3766] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502 [3767] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502_Ft_US [3768] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600 [3769] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600_Ft_US [3770] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601 [3771] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601_Ft_US [3772] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602 [3773] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602_Ft_US [3774] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701 [3775] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701_Ft_US [3776] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702 [3777] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702_Ft_US [3778] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801 [3779] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801_Ft_US [3780] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802 [3781] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802_Ft_US [3782] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900 [3783] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900_Ft_US [3784] NAD_1983_CORS96_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS [3785] NAD_1983_CORS96_StatePlane_Massachusetts_Island_FIPS_2002 [3786] NAD_1983_CORS96_StatePlane_Massachusetts_Mainland_FIPS_2001 [3787] NAD_1983_CORS96_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS [3788] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112 [3789] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl [3790] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111 [3791] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111_Ft_Intl [3792] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113 [3793] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113_Ft_Intl [3794] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202 [3795] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202_Ft_US [3796] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201 [3797] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201_Ft_US [3798] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203 [3799] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203_Ft_US [3800] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301 [3801] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301_Ft_US [3802] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302 [3803] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302_Ft_US [3804] NAD_1983_CORS96_StatePlane_Missouri_Central_FIPS_2402 [3805] NAD_1983_CORS96_StatePlane_Missouri_East_FIPS_2401 [3806] NAD_1983_CORS96_StatePlane_Missouri_West_FIPS_2403 [3807] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500 [3808] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500_Ft_Intl [3809] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600 [3810] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600_Ft_US [3811] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702 [3812] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702_Ft_US [3813] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701 [3814] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701_Ft_US [3815] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703 [3816] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703_Ft_US [3817] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800 [3818] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800_Ft_US [3819] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900 [3820] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900_Ft_US [3821] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002 [3822] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US [3823] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001 [3824] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001_Ft_US [3825] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003 [3826] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003_Ft_US [3827] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102 [3828] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102_Ft_US [3829] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101 [3830] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101_Ft_US [3831] NAD_1983_CORS96_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US [3832] NAD_1983_CORS96_StatePlane_New_York_Long_Island_FIPS_3104 [3833] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103 [3834] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103_Ft_US [3835] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200 [3836] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200_Ft_US [3837] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301 [3838] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301_FtI [3839] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302 [3840] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302_FtI [3841] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401 [3842] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401_Ft_US [3843] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402 [3844] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402_Ft_US [3845] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501 [3846] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501_Ft_US [3847] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502 [3848] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502_Ft_US [3849] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3850] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3851] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3852] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3853] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701 [3854] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US [3855] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702 [3856] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US [3857] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800 [3858] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800_Ft_US [3859] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900 [3860] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900_Ft_Intl [3861] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001 [3862] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001_Ft_US [3863] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002 [3864] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002_Ft_US [3865] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100 [3866] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100_Ft_US [3867] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203 [3868] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203_Ft_US [3869] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202 [3870] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202_FtUS [3871] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201 [3872] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201_Ft_US [3873] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204 [3874] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204_FtUS [3875] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205 [3876] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205_Ft_US [3877] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302 [3878] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3879] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_US [3880] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301 [3881] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3882] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_US [3883] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303 [3884] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3885] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_US [3886] NAD_1983_CORS96_StatePlane_Vermont_FIPS_4400 [3887] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501 [3888] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501_Ft_US [3889] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502 [3890] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502_Ft_US [3891] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601 [3892] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601_Ft_US [3893] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602 [3894] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602_Ft_US [3895] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701 [3896] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701_FtUS [3897] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702 [3898] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702_FtUS [3899] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802 [3900] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US [3901] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801 [3902] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801_Ft_US [3903] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803 [3904] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803_Ft_US [3905] NAD_1983_CORS96_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US [3906] NAD_1983_CORS96_StatePlane_Wyoming_East_Central_FIPS_4902 [3907] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901 [3908] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901_Ft_US [3909] NAD_1983_CORS96_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US [3910] NAD_1983_CORS96_StatePlane_Wyoming_West_Central_FIPS_4903 [3911] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904 [3912] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904_Ft_US [3913] NAD_1983_CORS96_UTM_Zone_10N [3914] NAD_1983_CORS96_UTM_Zone_11N [3915] NAD_1983_CORS96_UTM_Zone_12N [3916] NAD_1983_CORS96_UTM_Zone_13N [3917] NAD_1983_CORS96_UTM_Zone_14N [3918] NAD_1983_CORS96_UTM_Zone_15N [3919] NAD_1983_CORS96_UTM_Zone_16N [3920] NAD_1983_CORS96_UTM_Zone_17N [3921] NAD_1983_CORS96_UTM_Zone_18N [3922] NAD_1983_CORS96_UTM_Zone_19N [3923] NAD_1983_CORS96_UTM_Zone_1N [3924] NAD_1983_CORS96_UTM_Zone_20N [3925] NAD_1983_CORS96_UTM_Zone_2N [3926] NAD_1983_CORS96_UTM_Zone_3N [3927] NAD_1983_CORS96_UTM_Zone_4N [3928] NAD_1983_CORS96_UTM_Zone_59N [3929] NAD_1983_CORS96_UTM_Zone_5N [3930] NAD_1983_CORS96_UTM_Zone_60N [3931] NAD_1983_CORS96_UTM_Zone_6N [3932] NAD_1983_CORS96_UTM_Zone_7N [3933] NAD_1983_CORS96_UTM_Zone_8N [3934] NAD_1983_CORS96_UTM_Zone_9N [3935] NAD_1983_CSRS_UTM_Zone_14N (deprecated) [3936] NAD_1983_CSRS_UTM_Zone_15N (deprecated) [3937] NAD_1983_CSRS_UTM_Zone_16N (deprecated) [3938] NAD_1983_California_Teale_Albers_FtUS [3939] NAD_1983_Fargo_Ground_Coordinate_System [3940] NAD_1983_Georgia_Statewide_Lambert [3941] NAD_1983_HARN_Adj_MN_Aitkin_Feet [3942] NAD_1983_HARN_Adj_MN_Aitkin_Meters [3943] NAD_1983_HARN_Adj_MN_Anoka_Feet [3944] NAD_1983_HARN_Adj_MN_Anoka_Meters [3945] NAD_1983_HARN_Adj_MN_Becker_Feet [3946] NAD_1983_HARN_Adj_MN_Becker_Meters [3947] NAD_1983_HARN_Adj_MN_Beltrami_North_Feet [3948] NAD_1983_HARN_Adj_MN_Beltrami_North_Meters [3949] NAD_1983_HARN_Adj_MN_Beltrami_South_Feet [3950] NAD_1983_HARN_Adj_MN_Beltrami_South_Meters [3951] NAD_1983_HARN_Adj_MN_Benton_Feet [3952] NAD_1983_HARN_Adj_MN_Benton_Meters [3953] NAD_1983_HARN_Adj_MN_Big_Stone_Feet [3954] NAD_1983_HARN_Adj_MN_Big_Stone_Meters [3955] NAD_1983_HARN_Adj_MN_Blue_Earth_Feet [3956] NAD_1983_HARN_Adj_MN_Blue_Earth_Meters [3957] NAD_1983_HARN_Adj_MN_Brown_Feet [3958] NAD_1983_HARN_Adj_MN_Brown_Meters [3959] NAD_1983_HARN_Adj_MN_Carlton_Feet [3960] NAD_1983_HARN_Adj_MN_Carlton_Meters [3961] NAD_1983_HARN_Adj_MN_Carver_Feet [3962] NAD_1983_HARN_Adj_MN_Carver_Meters [3963] NAD_1983_HARN_Adj_MN_Cass_North_Feet [3964] NAD_1983_HARN_Adj_MN_Cass_North_Meters [3965] NAD_1983_HARN_Adj_MN_Cass_South_Feet [3966] NAD_1983_HARN_Adj_MN_Cass_South_Meters [3967] NAD_1983_HARN_Adj_MN_Chippewa_Feet [3968] NAD_1983_HARN_Adj_MN_Chippewa_Meters [3969] NAD_1983_HARN_Adj_MN_Chisago_Feet [3970] NAD_1983_HARN_Adj_MN_Chisago_Meters [3971] NAD_1983_HARN_Adj_MN_Clay_Feet [3972] NAD_1983_HARN_Adj_MN_Clay_Meters [3973] NAD_1983_HARN_Adj_MN_Clearwater_Feet [3974] NAD_1983_HARN_Adj_MN_Clearwater_Meters [3975] NAD_1983_HARN_Adj_MN_Cook_North_Feet [3976] NAD_1983_HARN_Adj_MN_Cook_North_Meters [3977] NAD_1983_HARN_Adj_MN_Cook_South_Feet [3978] NAD_1983_HARN_Adj_MN_Cook_South_Meters [3979] NAD_1983_HARN_Adj_MN_Cottonwood_Feet [3980] NAD_1983_HARN_Adj_MN_Cottonwood_Meters [3981] NAD_1983_HARN_Adj_MN_Crow_Wing_Feet [3982] NAD_1983_HARN_Adj_MN_Crow_Wing_Meters [3983] NAD_1983_HARN_Adj_MN_Dakota_Feet [3984] NAD_1983_HARN_Adj_MN_Dakota_Meters [3985] NAD_1983_HARN_Adj_MN_Dodge_Feet [3986] NAD_1983_HARN_Adj_MN_Dodge_Meters [3987] NAD_1983_HARN_Adj_MN_Douglas_Feet [3988] NAD_1983_HARN_Adj_MN_Douglas_Meters [3989] NAD_1983_HARN_Adj_MN_Faribault_Feet [3990] NAD_1983_HARN_Adj_MN_Faribault_Meters [3991] NAD_1983_HARN_Adj_MN_Fillmore_Feet [3992] NAD_1983_HARN_Adj_MN_Fillmore_Meters [3993] NAD_1983_HARN_Adj_MN_Freeborn_Feet [3994] NAD_1983_HARN_Adj_MN_Freeborn_Meters [3995] NAD_1983_HARN_Adj_MN_Goodhue_Feet [3996] NAD_1983_HARN_Adj_MN_Goodhue_Meters [3997] NAD_1983_HARN_Adj_MN_Grant_Feet [3998] NAD_1983_HARN_Adj_MN_Grant_Meters [3999] NAD_1983_HARN_Adj_MN_Hennepin_Feet [4000] NAD_1983_HARN_Adj_MN_Hennepin_Meters [4001] NAD_1983_HARN_Adj_MN_Houston_Feet [4002] NAD_1983_HARN_Adj_MN_Houston_Meters [4003] NAD_1983_HARN_Adj_MN_Hubbard_Feet [4004] NAD_1983_HARN_Adj_MN_Hubbard_Meters [4005] NAD_1983_HARN_Adj_MN_Isanti_Feet [4006] NAD_1983_HARN_Adj_MN_Isanti_Meters [4007] NAD_1983_HARN_Adj_MN_Itasca_North_Feet [4008] NAD_1983_HARN_Adj_MN_Itasca_North_Meters [4009] NAD_1983_HARN_Adj_MN_Itasca_South_Feet [4010] NAD_1983_HARN_Adj_MN_Itasca_South_Meters [4011] NAD_1983_HARN_Adj_MN_Jackson_Feet [4012] NAD_1983_HARN_Adj_MN_Jackson_Meters [4013] NAD_1983_HARN_Adj_MN_Kanabec_Feet [4014] NAD_1983_HARN_Adj_MN_Kanabec_Meters [4015] NAD_1983_HARN_Adj_MN_Kandiyohi_Feet [4016] NAD_1983_HARN_Adj_MN_Kandiyohi_Meters [4017] NAD_1983_HARN_Adj_MN_Kittson_Feet [4018] NAD_1983_HARN_Adj_MN_Kittson_Meters [4019] NAD_1983_HARN_Adj_MN_Koochiching_Feet [4020] NAD_1983_HARN_Adj_MN_Koochiching_Meters [4021] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Feet [4022] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Meters [4023] NAD_1983_HARN_Adj_MN_Lake_Feet [4024] NAD_1983_HARN_Adj_MN_Lake_Meters [4025] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Feet [4026] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Meters [4027] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Feet [4028] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Meters [4029] NAD_1983_HARN_Adj_MN_Le_Sueur_Feet [4030] NAD_1983_HARN_Adj_MN_Le_Sueur_Meters [4031] NAD_1983_HARN_Adj_MN_Lincoln_Feet [4032] NAD_1983_HARN_Adj_MN_Lincoln_Meters [4033] NAD_1983_HARN_Adj_MN_Lyon_Feet [4034] NAD_1983_HARN_Adj_MN_Lyon_Meters [4035] NAD_1983_HARN_Adj_MN_Mahnomen_Feet [4036] NAD_1983_HARN_Adj_MN_Mahnomen_Meters [4037] NAD_1983_HARN_Adj_MN_Marshall_Feet [4038] NAD_1983_HARN_Adj_MN_Marshall_Meters [4039] NAD_1983_HARN_Adj_MN_Martin_Feet [4040] NAD_1983_HARN_Adj_MN_Martin_Meters [4041] NAD_1983_HARN_Adj_MN_McLeod_Feet [4042] NAD_1983_HARN_Adj_MN_McLeod_Meters [4043] NAD_1983_HARN_Adj_MN_Meeker_Feet [4044] NAD_1983_HARN_Adj_MN_Meeker_Meters [4045] NAD_1983_HARN_Adj_MN_Mille_Lacs_Feet [4046] NAD_1983_HARN_Adj_MN_Mille_Lacs_Meters [4047] NAD_1983_HARN_Adj_MN_Morrison_Feet [4048] NAD_1983_HARN_Adj_MN_Morrison_Meters [4049] NAD_1983_HARN_Adj_MN_Mower_Feet [4050] NAD_1983_HARN_Adj_MN_Mower_Meters [4051] NAD_1983_HARN_Adj_MN_Murray_Feet [4052] NAD_1983_HARN_Adj_MN_Murray_Meters [4053] NAD_1983_HARN_Adj_MN_Nicollet_Feet [4054] NAD_1983_HARN_Adj_MN_Nicollet_Meters [4055] NAD_1983_HARN_Adj_MN_Nobles_Feet [4056] NAD_1983_HARN_Adj_MN_Nobles_Meters [4057] NAD_1983_HARN_Adj_MN_Norman_Feet [4058] NAD_1983_HARN_Adj_MN_Norman_Meters [4059] NAD_1983_HARN_Adj_MN_Olmsted_Feet [4060] NAD_1983_HARN_Adj_MN_Olmsted_Meters [4061] NAD_1983_HARN_Adj_MN_Ottertail_Feet [4062] NAD_1983_HARN_Adj_MN_Ottertail_Meters [4063] NAD_1983_HARN_Adj_MN_Pennington_Feet [4064] NAD_1983_HARN_Adj_MN_Pennington_Meters [4065] NAD_1983_HARN_Adj_MN_Pine_Feet [4066] NAD_1983_HARN_Adj_MN_Pine_Meters [4067] NAD_1983_HARN_Adj_MN_Pipestone_Feet [4068] NAD_1983_HARN_Adj_MN_Pipestone_Meters [4069] NAD_1983_HARN_Adj_MN_Polk_Feet [4070] NAD_1983_HARN_Adj_MN_Polk_Meters [4071] NAD_1983_HARN_Adj_MN_Pope_Feet [4072] NAD_1983_HARN_Adj_MN_Pope_Meters [4073] NAD_1983_HARN_Adj_MN_Ramsey_Feet [4074] NAD_1983_HARN_Adj_MN_Ramsey_Meters [4075] NAD_1983_HARN_Adj_MN_Red_Lake_Feet [4076] NAD_1983_HARN_Adj_MN_Red_Lake_Meters [4077] NAD_1983_HARN_Adj_MN_Redwood_Feet [4078] NAD_1983_HARN_Adj_MN_Redwood_Meters [4079] NAD_1983_HARN_Adj_MN_Renville_Feet [4080] NAD_1983_HARN_Adj_MN_Renville_Meters [4081] NAD_1983_HARN_Adj_MN_Rice_Feet [4082] NAD_1983_HARN_Adj_MN_Rice_Meters [4083] NAD_1983_HARN_Adj_MN_Rock_Feet [4084] NAD_1983_HARN_Adj_MN_Rock_Meters [4085] NAD_1983_HARN_Adj_MN_Roseau_Feet [4086] NAD_1983_HARN_Adj_MN_Roseau_Meters [4087] NAD_1983_HARN_Adj_MN_Scott_Feet [4088] NAD_1983_HARN_Adj_MN_Scott_Meters [4089] NAD_1983_HARN_Adj_MN_Sherburne_Feet [4090] NAD_1983_HARN_Adj_MN_Sherburne_Meters [4091] NAD_1983_HARN_Adj_MN_Sibley_Feet [4092] NAD_1983_HARN_Adj_MN_Sibley_Meters [4093] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Feet [4094] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Meters [4095] NAD_1983_HARN_Adj_MN_St_Louis_Central_Feet [4096] NAD_1983_HARN_Adj_MN_St_Louis_Central_Meters [4097] NAD_1983_HARN_Adj_MN_St_Louis_North_Feet [4098] NAD_1983_HARN_Adj_MN_St_Louis_North_Meters [4099] NAD_1983_HARN_Adj_MN_St_Louis_South_Feet [4100] NAD_1983_HARN_Adj_MN_St_Louis_South_Meters [4101] NAD_1983_HARN_Adj_MN_Stearns_Feet [4102] NAD_1983_HARN_Adj_MN_Stearns_Meters [4103] NAD_1983_HARN_Adj_MN_Steele_Feet [4104] NAD_1983_HARN_Adj_MN_Steele_Meters [4105] NAD_1983_HARN_Adj_MN_Stevens_Feet [4106] NAD_1983_HARN_Adj_MN_Stevens_Meters [4107] NAD_1983_HARN_Adj_MN_Swift_Feet [4108] NAD_1983_HARN_Adj_MN_Swift_Meters [4109] NAD_1983_HARN_Adj_MN_Todd_Feet [4110] NAD_1983_HARN_Adj_MN_Todd_Meters [4111] NAD_1983_HARN_Adj_MN_Traverse_Feet [4112] NAD_1983_HARN_Adj_MN_Traverse_Meters [4113] NAD_1983_HARN_Adj_MN_Wabasha_Feet [4114] NAD_1983_HARN_Adj_MN_Wabasha_Meters [4115] NAD_1983_HARN_Adj_MN_Wadena_Feet [4116] NAD_1983_HARN_Adj_MN_Wadena_Meters [4117] NAD_1983_HARN_Adj_MN_Waseca_Feet [4118] NAD_1983_HARN_Adj_MN_Waseca_Meters [4119] NAD_1983_HARN_Adj_MN_Washington_Feet [4120] NAD_1983_HARN_Adj_MN_Washington_Meters [4121] NAD_1983_HARN_Adj_MN_Watonwan_Feet [4122] NAD_1983_HARN_Adj_MN_Watonwan_Meters [4123] NAD_1983_HARN_Adj_MN_Wilkin_Feet [4124] NAD_1983_HARN_Adj_MN_Wilkin_Meters [4125] NAD_1983_HARN_Adj_MN_Winona_Feet [4126] NAD_1983_HARN_Adj_MN_Winona_Meters [4127] NAD_1983_HARN_Adj_MN_Wright_Feet [4128] NAD_1983_HARN_Adj_MN_Wright_Meters [4129] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Feet [4130] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Meters [4131] NAD_1983_HARN_Adj_WI_Adams_Feet (deprecated) [4132] NAD_1983_HARN_Adj_WI_Adams_Meters (deprecated) [4133] NAD_1983_HARN_Adj_WI_Ashland_Feet (deprecated) [4134] NAD_1983_HARN_Adj_WI_Ashland_Meters (deprecated) [4135] NAD_1983_HARN_Adj_WI_Barron_Feet (deprecated) [4136] NAD_1983_HARN_Adj_WI_Barron_Meters (deprecated) [4137] NAD_1983_HARN_Adj_WI_Bayfield_Feet (deprecated) [4138] NAD_1983_HARN_Adj_WI_Bayfield_Meters (deprecated) [4139] NAD_1983_HARN_Adj_WI_Brown_Feet (deprecated) [4140] NAD_1983_HARN_Adj_WI_Brown_Meters (deprecated) [4141] NAD_1983_HARN_Adj_WI_Buffalo_Feet (deprecated) [4142] NAD_1983_HARN_Adj_WI_Buffalo_Meters (deprecated) [4143] NAD_1983_HARN_Adj_WI_Burnett_Feet (deprecated) [4144] NAD_1983_HARN_Adj_WI_Burnett_Meters (deprecated) [4145] NAD_1983_HARN_Adj_WI_Calumet_Feet (deprecated) [4146] NAD_1983_HARN_Adj_WI_Calumet_Meters (deprecated) [4147] NAD_1983_HARN_Adj_WI_Chippewa_Feet [4148] NAD_1983_HARN_Adj_WI_Chippewa_Meters [4149] NAD_1983_HARN_Adj_WI_Clark_Feet (deprecated) [4150] NAD_1983_HARN_Adj_WI_Clark_Meters (deprecated) [4151] NAD_1983_HARN_Adj_WI_Columbia_Feet (deprecated) [4152] NAD_1983_HARN_Adj_WI_Columbia_Meters (deprecated) [4153] NAD_1983_HARN_Adj_WI_Crawford_Feet (deprecated) [4154] NAD_1983_HARN_Adj_WI_Crawford_Meters (deprecated) [4155] NAD_1983_HARN_Adj_WI_Dane_Feet (deprecated) [4156] NAD_1983_HARN_Adj_WI_Dane_Meters (deprecated) [4157] NAD_1983_HARN_Adj_WI_Dodge_Feet (deprecated) [4158] NAD_1983_HARN_Adj_WI_Dodge_Meters (deprecated) [4159] NAD_1983_HARN_Adj_WI_Door_Feet (deprecated) [4160] NAD_1983_HARN_Adj_WI_Door_Meters (deprecated) [4161] NAD_1983_HARN_Adj_WI_Douglas_Feet (deprecated) [4162] NAD_1983_HARN_Adj_WI_Douglas_Meters (deprecated) [4163] NAD_1983_HARN_Adj_WI_Dunn_Feet (deprecated) [4164] NAD_1983_HARN_Adj_WI_Dunn_Meters (deprecated) [4165] NAD_1983_HARN_Adj_WI_EauClaire_Feet (deprecated) [4166] NAD_1983_HARN_Adj_WI_EauClaire_Meters (deprecated) [4167] NAD_1983_HARN_Adj_WI_Florence_Feet (deprecated) [4168] NAD_1983_HARN_Adj_WI_Florence_Meters (deprecated) [4169] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Feet (deprecated) [4170] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Meters (deprecated) [4171] NAD_1983_HARN_Adj_WI_Forest_Feet (deprecated) [4172] NAD_1983_HARN_Adj_WI_Forest_Meters (deprecated) [4173] NAD_1983_HARN_Adj_WI_Grant_Feet (deprecated) [4174] NAD_1983_HARN_Adj_WI_Grant_Meters (deprecated) [4175] NAD_1983_HARN_Adj_WI_GreenLake_Feet (deprecated) [4176] NAD_1983_HARN_Adj_WI_GreenLake_Meters (deprecated) [4177] NAD_1983_HARN_Adj_WI_Green_Feet (deprecated) [4178] NAD_1983_HARN_Adj_WI_Green_Meters (deprecated) [4179] NAD_1983_HARN_Adj_WI_Iowa_Feet (deprecated) [4180] NAD_1983_HARN_Adj_WI_Iowa_Meters (deprecated) [4181] NAD_1983_HARN_Adj_WI_Iron_Feet (deprecated) [4182] NAD_1983_HARN_Adj_WI_Iron_Meters (deprecated) [4183] NAD_1983_HARN_Adj_WI_Jackson_Feet (deprecated) [4184] NAD_1983_HARN_Adj_WI_Jackson_Meters (deprecated) [4185] NAD_1983_HARN_Adj_WI_Jefferson_Feet (deprecated) [4186] NAD_1983_HARN_Adj_WI_Jefferson_Meters (deprecated) [4187] NAD_1983_HARN_Adj_WI_Juneau_Feet (deprecated) [4188] NAD_1983_HARN_Adj_WI_Juneau_Meters (deprecated) [4189] NAD_1983_HARN_Adj_WI_Kenosha_Feet (deprecated) [4190] NAD_1983_HARN_Adj_WI_Kenosha_Meters (deprecated) [4191] NAD_1983_HARN_Adj_WI_Kewaunee_Feet (deprecated) [4192] NAD_1983_HARN_Adj_WI_Kewaunee_Meters (deprecated) [4193] NAD_1983_HARN_Adj_WI_LaCrosse_Feet (deprecated) [4194] NAD_1983_HARN_Adj_WI_LaCrosse_Meters (deprecated) [4195] NAD_1983_HARN_Adj_WI_Lafayette_Feet (deprecated) [4196] NAD_1983_HARN_Adj_WI_Lafayette_Meters (deprecated) [4197] NAD_1983_HARN_Adj_WI_Langlade_Feet (deprecated) [4198] NAD_1983_HARN_Adj_WI_Langlade_Meters (deprecated) [4199] NAD_1983_HARN_Adj_WI_Lincoln_Feet (deprecated) [4200] NAD_1983_HARN_Adj_WI_Lincoln_Meters (deprecated) [4201] NAD_1983_HARN_Adj_WI_Manitowoc_Feet (deprecated) [4202] NAD_1983_HARN_Adj_WI_Manitowoc_Meters (deprecated) [4203] NAD_1983_HARN_Adj_WI_Marathon_Feet (deprecated) [4204] NAD_1983_HARN_Adj_WI_Marathon_Meters (deprecated) [4205] NAD_1983_HARN_Adj_WI_Marinette_Feet (deprecated) [4206] NAD_1983_HARN_Adj_WI_Marinette_Meters (deprecated) [4207] NAD_1983_HARN_Adj_WI_Marquette_Feet (deprecated) [4208] NAD_1983_HARN_Adj_WI_Marquette_Meters (deprecated) [4209] NAD_1983_HARN_Adj_WI_Menominee_Feet (deprecated) [4210] NAD_1983_HARN_Adj_WI_Menominee_Meters (deprecated) [4211] NAD_1983_HARN_Adj_WI_Milwaukee_Feet (deprecated) [4212] NAD_1983_HARN_Adj_WI_Milwaukee_Meters (deprecated) [4213] NAD_1983_HARN_Adj_WI_Monroe_Feet (deprecated) [4214] NAD_1983_HARN_Adj_WI_Monroe_Meters (deprecated) [4215] NAD_1983_HARN_Adj_WI_Oconto_Feet (deprecated) [4216] NAD_1983_HARN_Adj_WI_Oconto_Meters (deprecated) [4217] NAD_1983_HARN_Adj_WI_Oneida_Feet (deprecated) [4218] NAD_1983_HARN_Adj_WI_Oneida_Meters (deprecated) [4219] NAD_1983_HARN_Adj_WI_Outagamie_Feet (deprecated) [4220] NAD_1983_HARN_Adj_WI_Outagamie_Meters (deprecated) [4221] NAD_1983_HARN_Adj_WI_Ozaukee_Feet (deprecated) [4222] NAD_1983_HARN_Adj_WI_Ozaukee_Meters (deprecated) [4223] NAD_1983_HARN_Adj_WI_Pepin_Feet (deprecated) [4224] NAD_1983_HARN_Adj_WI_Pepin_Meters (deprecated) [4225] NAD_1983_HARN_Adj_WI_Pierce_Feet (deprecated) [4226] NAD_1983_HARN_Adj_WI_Pierce_Meters (deprecated) [4227] NAD_1983_HARN_Adj_WI_Polk_Feet (deprecated) [4228] NAD_1983_HARN_Adj_WI_Polk_Meters (deprecated) [4229] NAD_1983_HARN_Adj_WI_Portage_Feet (deprecated) [4230] NAD_1983_HARN_Adj_WI_Portage_Meters (deprecated) [4231] NAD_1983_HARN_Adj_WI_Price_Feet (deprecated) [4232] NAD_1983_HARN_Adj_WI_Price_Meters (deprecated) [4233] NAD_1983_HARN_Adj_WI_Racine_Feet (deprecated) [4234] NAD_1983_HARN_Adj_WI_Racine_Meters (deprecated) [4235] NAD_1983_HARN_Adj_WI_Richland_Feet (deprecated) [4236] NAD_1983_HARN_Adj_WI_Richland_Meters (deprecated) [4237] NAD_1983_HARN_Adj_WI_Rock_Feet (deprecated) [4238] NAD_1983_HARN_Adj_WI_Rock_Meters (deprecated) [4239] NAD_1983_HARN_Adj_WI_Rusk_Feet (deprecated) [4240] NAD_1983_HARN_Adj_WI_Rusk_Meters (deprecated) [4241] NAD_1983_HARN_Adj_WI_Sauk_Feet (deprecated) [4242] NAD_1983_HARN_Adj_WI_Sauk_Meters (deprecated) [4243] NAD_1983_HARN_Adj_WI_Sawyer_Feet (deprecated) [4244] NAD_1983_HARN_Adj_WI_Sawyer_Meters (deprecated) [4245] NAD_1983_HARN_Adj_WI_Shawano_Feet (deprecated) [4246] NAD_1983_HARN_Adj_WI_Shawano_Meters (deprecated) [4247] NAD_1983_HARN_Adj_WI_Sheboygan_Feet (deprecated) [4248] NAD_1983_HARN_Adj_WI_Sheboygan_Meters (deprecated) [4249] NAD_1983_HARN_Adj_WI_St_Croix_Feet (deprecated) [4250] NAD_1983_HARN_Adj_WI_St_Croix_Meters (deprecated) [4251] NAD_1983_HARN_Adj_WI_Taylor_Feet (deprecated) [4252] NAD_1983_HARN_Adj_WI_Taylor_Meters (deprecated) [4253] NAD_1983_HARN_Adj_WI_Trempealeau_Feet (deprecated) [4254] NAD_1983_HARN_Adj_WI_Trempealeau_Meters (deprecated) [4255] NAD_1983_HARN_Adj_WI_Vernon_Feet (deprecated) [4256] NAD_1983_HARN_Adj_WI_Vernon_Meters (deprecated) [4257] NAD_1983_HARN_Adj_WI_Vilas_Feet (deprecated) [4258] NAD_1983_HARN_Adj_WI_Vilas_Meters (deprecated) [4259] NAD_1983_HARN_Adj_WI_Walworth_Feet (deprecated) [4260] NAD_1983_HARN_Adj_WI_Walworth_Meters (deprecated) [4261] NAD_1983_HARN_Adj_WI_Washburn_Feet (deprecated) [4262] NAD_1983_HARN_Adj_WI_Washburn_Meters (deprecated) [4263] NAD_1983_HARN_Adj_WI_Washington_Feet (deprecated) [4264] NAD_1983_HARN_Adj_WI_Washington_Meters (deprecated) [4265] NAD_1983_HARN_Adj_WI_Waukesha_Feet (deprecated) [4266] NAD_1983_HARN_Adj_WI_Waukesha_Meters (deprecated) [4267] NAD_1983_HARN_Adj_WI_Waupaca_Feet (deprecated) [4268] NAD_1983_HARN_Adj_WI_Waupaca_Meters (deprecated) [4269] NAD_1983_HARN_Adj_WI_Waushara_Feet (deprecated) [4270] NAD_1983_HARN_Adj_WI_Waushara_Meters (deprecated) [4271] NAD_1983_HARN_Adj_WI_Winnebago_Feet (deprecated) [4272] NAD_1983_HARN_Adj_WI_Winnebago_Meters (deprecated) [4273] NAD_1983_HARN_Adj_WI_Wood_Feet (deprecated) [4274] NAD_1983_HARN_Adj_WI_Wood_Meters (deprecated) [4275] NAD_1983_HARN_Fargo_Ground_Coordinate_System [4276] NAD_1983_HARN_Guam_Map_Grid (deprecated) [4277] NAD_1983_HARN_Maine_2000_Central_Zone (deprecated) [4278] NAD_1983_HARN_Maine_2000_East_Zone (deprecated) [4279] NAD_1983_HARN_Maine_2000_West_Zone (deprecated) [4280] NAD_1983_HARN_Mississippi_TM (deprecated) [4281] NAD_1983_HARN_StatePlane_Alabama_East_FIPS_0101 (deprecated) [4282] NAD_1983_HARN_StatePlane_Alabama_West_FIPS_0102 (deprecated) [4283] NAD_1983_HARN_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [4284] NAD_1983_HARN_StatePlane_Arizona_East_FIPS_0201 (deprecated) [4285] NAD_1983_HARN_StatePlane_Arizona_West_FIPS_0203 (deprecated) [4286] NAD_1983_HARN_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [4287] NAD_1983_HARN_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [4288] NAD_1983_HARN_StatePlane_California_III_FIPS_0403 (deprecated) [4289] NAD_1983_HARN_StatePlane_California_II_FIPS_0402 (deprecated) [4290] NAD_1983_HARN_StatePlane_California_IV_FIPS_0404 (deprecated) [4291] NAD_1983_HARN_StatePlane_California_I_FIPS_0401 (deprecated) [4292] NAD_1983_HARN_StatePlane_California_VI_FIPS_0406 (deprecated) [4293] NAD_1983_HARN_StatePlane_California_V_FIPS_0405 (deprecated) [4294] NAD_1983_HARN_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [4295] NAD_1983_HARN_StatePlane_Colorado_North_FIPS_0501 (deprecated) [4296] NAD_1983_HARN_StatePlane_Colorado_South_FIPS_0503 (deprecated) [4297] NAD_1983_HARN_StatePlane_Connecticut_FIPS_0600 (deprecated) [4298] NAD_1983_HARN_StatePlane_Delaware_FIPS_0700 (deprecated) [4299] NAD_1983_HARN_StatePlane_Florida_East_FIPS_0901 (deprecated) [4300] NAD_1983_HARN_StatePlane_Florida_North_FIPS_0903 (deprecated) [4301] NAD_1983_HARN_StatePlane_Florida_West_FIPS_0902 (deprecated) [4302] NAD_1983_HARN_StatePlane_Georgia_East_FIPS_1001 (deprecated) [4303] NAD_1983_HARN_StatePlane_Georgia_West_FIPS_1002 (deprecated) [4304] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4305] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101_Feet [4306] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4307] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102_Feet [4308] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4309] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4310] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4311] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104_Feet [4312] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4313] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105_Feet [4314] NAD_1983_HARN_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [4315] NAD_1983_HARN_StatePlane_Idaho_East_FIPS_1101 (deprecated) [4316] NAD_1983_HARN_StatePlane_Idaho_West_FIPS_1103 (deprecated) [4317] NAD_1983_HARN_StatePlane_Illinois_East_FIPS_1201 (deprecated) [4318] NAD_1983_HARN_StatePlane_Illinois_West_FIPS_1202 (deprecated) [4319] NAD_1983_HARN_StatePlane_Indiana_East_FIPS_1301 (deprecated) [4320] NAD_1983_HARN_StatePlane_Indiana_West_FIPS_1302 (deprecated) [4321] NAD_1983_HARN_StatePlane_Iowa_North_FIPS_1401 (deprecated) [4322] NAD_1983_HARN_StatePlane_Iowa_South_FIPS_1402 (deprecated) [4323] NAD_1983_HARN_StatePlane_Kansas_North_FIPS_1501 (deprecated) [4324] NAD_1983_HARN_StatePlane_Kansas_South_FIPS_1502 (deprecated) [4325] NAD_1983_HARN_StatePlane_Kentucky_FIPS_1600 (deprecated) [4326] NAD_1983_HARN_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [4327] NAD_1983_HARN_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [4328] NAD_1983_HARN_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [4329] NAD_1983_HARN_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [4330] NAD_1983_HARN_StatePlane_Maine_East_FIPS_1801 (deprecated) [4331] NAD_1983_HARN_StatePlane_Maine_West_FIPS_1802 (deprecated) [4332] NAD_1983_HARN_StatePlane_Maryland_FIPS_1900 (deprecated) [4333] NAD_1983_HARN_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [4334] NAD_1983_HARN_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [4335] NAD_1983_HARN_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [4336] NAD_1983_HARN_StatePlane_Michigan_North_FIPS_2111 (deprecated) [4337] NAD_1983_HARN_StatePlane_Michigan_South_FIPS_2113 (deprecated) [4338] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [4339] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4340] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [4341] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4342] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [4343] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4344] NAD_1983_HARN_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [4345] NAD_1983_HARN_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [4346] NAD_1983_HARN_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [4347] NAD_1983_HARN_StatePlane_Missouri_East_FIPS_2401 (deprecated) [4348] NAD_1983_HARN_StatePlane_Missouri_West_FIPS_2403 (deprecated) [4349] NAD_1983_HARN_StatePlane_Montana_FIPS_2500 (deprecated) [4350] NAD_1983_HARN_StatePlane_Nebraska_FIPS_2600 (deprecated) [4351] NAD_1983_HARN_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [4352] NAD_1983_HARN_StatePlane_Nevada_East_FIPS_2701 (deprecated) [4353] NAD_1983_HARN_StatePlane_Nevada_West_FIPS_2703 (deprecated) [4354] NAD_1983_HARN_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [4355] NAD_1983_HARN_StatePlane_New_Jersey_FIPS_2900 (deprecated) [4356] NAD_1983_HARN_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [4357] NAD_1983_HARN_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [4358] NAD_1983_HARN_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [4359] NAD_1983_HARN_StatePlane_New_York_Central_FIPS_3102 (deprecated) [4360] NAD_1983_HARN_StatePlane_New_York_East_FIPS_3101 (deprecated) [4361] NAD_1983_HARN_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [4362] NAD_1983_HARN_StatePlane_New_York_West_FIPS_3103 (deprecated) [4363] NAD_1983_HARN_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [4364] NAD_1983_HARN_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [4365] NAD_1983_HARN_StatePlane_Ohio_North_FIPS_3401 (deprecated) [4366] NAD_1983_HARN_StatePlane_Ohio_South_FIPS_3402 (deprecated) [4367] NAD_1983_HARN_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [4368] NAD_1983_HARN_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [4369] NAD_1983_HARN_StatePlane_Oregon_North_FIPS_3601 (deprecated) [4370] NAD_1983_HARN_StatePlane_Oregon_South_FIPS_3602 (deprecated) [4371] NAD_1983_HARN_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200 (deprecated) [4372] NAD_1983_HARN_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [4373] NAD_1983_HARN_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [4374] NAD_1983_HARN_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [4375] NAD_1983_HARN_StatePlane_Tennessee_FIPS_4100 (deprecated) [4376] NAD_1983_HARN_StatePlane_Texas_Central_FIPS_4203 (deprecated) [4377] NAD_1983_HARN_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [4378] NAD_1983_HARN_StatePlane_Texas_North_FIPS_4201 (deprecated) [4379] NAD_1983_HARN_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [4380] NAD_1983_HARN_StatePlane_Texas_South_FIPS_4205 (deprecated) [4381] NAD_1983_HARN_StatePlane_Utah_Central_FIPS_4302 (deprecated) [4382] NAD_1983_HARN_StatePlane_Utah_North_FIPS_4301 (deprecated) [4383] NAD_1983_HARN_StatePlane_Utah_South_FIPS_4303 (deprecated) [4384] NAD_1983_HARN_StatePlane_Vermont_FIPS_4400 (deprecated) [4385] NAD_1983_HARN_StatePlane_Virginia_North_FIPS_4501 (deprecated) [4386] NAD_1983_HARN_StatePlane_Virginia_South_FIPS_4502 (deprecated) [4387] NAD_1983_HARN_StatePlane_Washington_North_FIPS_4601 (deprecated) [4388] NAD_1983_HARN_StatePlane_Washington_South_FIPS_4602 (deprecated) [4389] NAD_1983_HARN_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [4390] NAD_1983_HARN_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [4391] NAD_1983_HARN_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [4392] NAD_1983_HARN_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [4393] NAD_1983_HARN_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [4394] NAD_1983_HARN_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [4395] NAD_1983_HARN_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [4396] NAD_1983_HARN_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [4397] NAD_1983_HARN_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [4398] NAD_1983_HARN_UTM_Zone_11N (deprecated) [4399] NAD_1983_HARN_UTM_Zone_12N (deprecated) [4400] NAD_1983_HARN_UTM_Zone_13N (deprecated) [4401] NAD_1983_HARN_UTM_Zone_18N (deprecated) [4402] NAD_1983_HARN_UTM_Zone_2S (deprecated) [4403] NAD_1983_HARN_UTM_Zone_4N (deprecated) [4404] NAD_1983_HARN_UTM_Zone_5N (deprecated) [4405] NAD_1983_HARN_WISCRS_Adams_County_Feet [4406] NAD_1983_HARN_WISCRS_Adams_County_Meters [4407] NAD_1983_HARN_WISCRS_Ashland_County_Feet (deprecated) [4408] NAD_1983_HARN_WISCRS_Ashland_County_Meters (deprecated) [4409] NAD_1983_HARN_WISCRS_Barron_County_Feet (deprecated) [4410] NAD_1983_HARN_WISCRS_Barron_County_Meters (deprecated) [4411] NAD_1983_HARN_WISCRS_Bayfield_County_Feet (deprecated) [4412] NAD_1983_HARN_WISCRS_Bayfield_County_Meters (deprecated) [4413] NAD_1983_HARN_WISCRS_Brown_County_Feet (deprecated) [4414] NAD_1983_HARN_WISCRS_Brown_County_Meters (deprecated) [4415] NAD_1983_HARN_WISCRS_Buffalo_County_Feet (deprecated) [4416] NAD_1983_HARN_WISCRS_Buffalo_County_Meters (deprecated) [4417] NAD_1983_HARN_WISCRS_Burnett_County_Feet (deprecated) [4418] NAD_1983_HARN_WISCRS_Burnett_County_Meters (deprecated) [4419] NAD_1983_HARN_WISCRS_Calumet_County_Feet [4420] NAD_1983_HARN_WISCRS_Calumet_County_Meters [4421] NAD_1983_HARN_WISCRS_Chippewa_County_Feet (deprecated) [4422] NAD_1983_HARN_WISCRS_Chippewa_County_Meters (deprecated) [4423] NAD_1983_HARN_WISCRS_Clark_County_Feet (deprecated) [4424] NAD_1983_HARN_WISCRS_Clark_County_Meters (deprecated) [4425] NAD_1983_HARN_WISCRS_Columbia_County_Feet (deprecated) [4426] NAD_1983_HARN_WISCRS_Columbia_County_Meters (deprecated) [4427] NAD_1983_HARN_WISCRS_Crawford_County_Feet (deprecated) [4428] NAD_1983_HARN_WISCRS_Crawford_County_Meters (deprecated) [4429] NAD_1983_HARN_WISCRS_Dane_County_Feet (deprecated) [4430] NAD_1983_HARN_WISCRS_Dane_County_Meters (deprecated) [4431] NAD_1983_HARN_WISCRS_Dodge_County_Feet [4432] NAD_1983_HARN_WISCRS_Dodge_County_Meters [4433] NAD_1983_HARN_WISCRS_Door_County_Feet (deprecated) [4434] NAD_1983_HARN_WISCRS_Door_County_Meters (deprecated) [4435] NAD_1983_HARN_WISCRS_Douglas_County_Feet (deprecated) [4436] NAD_1983_HARN_WISCRS_Douglas_County_Meters (deprecated) [4437] NAD_1983_HARN_WISCRS_Dunn_County_Feet (deprecated) [4438] NAD_1983_HARN_WISCRS_Dunn_County_Meters (deprecated) [4439] NAD_1983_HARN_WISCRS_EauClaire_County_Feet (deprecated) [4440] NAD_1983_HARN_WISCRS_EauClaire_County_Meters (deprecated) [4441] NAD_1983_HARN_WISCRS_Florence_County_Feet (deprecated) [4442] NAD_1983_HARN_WISCRS_Florence_County_Meters (deprecated) [4443] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Feet [4444] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Meters [4445] NAD_1983_HARN_WISCRS_Forest_County_Feet (deprecated) [4446] NAD_1983_HARN_WISCRS_Forest_County_Meters (deprecated) [4447] NAD_1983_HARN_WISCRS_Grant_County_Feet (deprecated) [4448] NAD_1983_HARN_WISCRS_Grant_County_Meters (deprecated) [4449] NAD_1983_HARN_WISCRS_GreenLake_County_Feet [4450] NAD_1983_HARN_WISCRS_GreenLake_County_Meters [4451] NAD_1983_HARN_WISCRS_Green_County_Feet [4452] NAD_1983_HARN_WISCRS_Green_County_Meters [4453] NAD_1983_HARN_WISCRS_Iowa_County_Feet (deprecated) [4454] NAD_1983_HARN_WISCRS_Iowa_County_Meters (deprecated) [4455] NAD_1983_HARN_WISCRS_Iron_County_Feet (deprecated) [4456] NAD_1983_HARN_WISCRS_Iron_County_Meters (deprecated) [4457] NAD_1983_HARN_WISCRS_Jackson_County_Feet (deprecated) [4458] NAD_1983_HARN_WISCRS_Jackson_County_Meters (deprecated) [4459] NAD_1983_HARN_WISCRS_Jefferson_County_Feet [4460] NAD_1983_HARN_WISCRS_Jefferson_County_Meters [4461] NAD_1983_HARN_WISCRS_Juneau_County_Feet [4462] NAD_1983_HARN_WISCRS_Juneau_County_Meters [4463] NAD_1983_HARN_WISCRS_Kenosha_County_Feet [4464] NAD_1983_HARN_WISCRS_Kenosha_County_Meters [4465] NAD_1983_HARN_WISCRS_Kewaunee_County_Feet [4466] NAD_1983_HARN_WISCRS_Kewaunee_County_Meters [4467] NAD_1983_HARN_WISCRS_LaCrosse_County_Feet (deprecated) [4468] NAD_1983_HARN_WISCRS_LaCrosse_County_Meters (deprecated) [4469] NAD_1983_HARN_WISCRS_Lafayette_County_Feet [4470] NAD_1983_HARN_WISCRS_Lafayette_County_Meters [4471] NAD_1983_HARN_WISCRS_Langlade_County_Feet (deprecated) [4472] NAD_1983_HARN_WISCRS_Langlade_County_Meters (deprecated) [4473] NAD_1983_HARN_WISCRS_Lincoln_County_Feet (deprecated) [4474] NAD_1983_HARN_WISCRS_Lincoln_County_Meters (deprecated) [4475] NAD_1983_HARN_WISCRS_Manitowoc_County_Feet [4476] NAD_1983_HARN_WISCRS_Manitowoc_County_Meters [4477] NAD_1983_HARN_WISCRS_Marathon_County_Feet (deprecated) [4478] NAD_1983_HARN_WISCRS_Marathon_County_Meters (deprecated) [4479] NAD_1983_HARN_WISCRS_Marinette_County_Feet (deprecated) [4480] NAD_1983_HARN_WISCRS_Marinette_County_Meters (deprecated) [4481] NAD_1983_HARN_WISCRS_Marquette_County_Feet [4482] NAD_1983_HARN_WISCRS_Marquette_County_Meters [4483] NAD_1983_HARN_WISCRS_Menominee_County_Feet (deprecated) [4484] NAD_1983_HARN_WISCRS_Menominee_County_Meters (deprecated) [4485] NAD_1983_HARN_WISCRS_Milwaukee_County_Feet [4486] NAD_1983_HARN_WISCRS_Milwaukee_County_Meters [4487] NAD_1983_HARN_WISCRS_Monroe_County_Feet (deprecated) [4488] NAD_1983_HARN_WISCRS_Monroe_County_Meters (deprecated) [4489] NAD_1983_HARN_WISCRS_Oconto_County_Feet (deprecated) [4490] NAD_1983_HARN_WISCRS_Oconto_County_Meters (deprecated) [4491] NAD_1983_HARN_WISCRS_Oneida_County_Feet (deprecated) [4492] NAD_1983_HARN_WISCRS_Oneida_County_Meters (deprecated) [4493] NAD_1983_HARN_WISCRS_Outagamie_County_Feet [4494] NAD_1983_HARN_WISCRS_Outagamie_County_Meters [4495] NAD_1983_HARN_WISCRS_Ozaukee_County_Feet [4496] NAD_1983_HARN_WISCRS_Ozaukee_County_Meters [4497] NAD_1983_HARN_WISCRS_Pepin_County_Feet [4498] NAD_1983_HARN_WISCRS_Pepin_County_Meters [4499] NAD_1983_HARN_WISCRS_Pierce_County_Feet [4500] NAD_1983_HARN_WISCRS_Pierce_County_Meters [4501] NAD_1983_HARN_WISCRS_Polk_County_Feet (deprecated) [4502] NAD_1983_HARN_WISCRS_Polk_County_Meters (deprecated) [4503] NAD_1983_HARN_WISCRS_Portage_County_Feet (deprecated) [4504] NAD_1983_HARN_WISCRS_Portage_County_Meters (deprecated) [4505] NAD_1983_HARN_WISCRS_Price_County_Feet (deprecated) [4506] NAD_1983_HARN_WISCRS_Price_County_Meters (deprecated) [4507] NAD_1983_HARN_WISCRS_Racine_County_Feet [4508] NAD_1983_HARN_WISCRS_Racine_County_Meters [4509] NAD_1983_HARN_WISCRS_Richland_County_Feet (deprecated) [4510] NAD_1983_HARN_WISCRS_Richland_County_Meters (deprecated) [4511] NAD_1983_HARN_WISCRS_Rock_County_Feet (deprecated) [4512] NAD_1983_HARN_WISCRS_Rock_County_Meters (deprecated) [4513] NAD_1983_HARN_WISCRS_Rusk_County_Feet (deprecated) [4514] NAD_1983_HARN_WISCRS_Rusk_County_Meters (deprecated) [4515] NAD_1983_HARN_WISCRS_Sauk_County_Feet (deprecated) [4516] NAD_1983_HARN_WISCRS_Sauk_County_Meters (deprecated) [4517] NAD_1983_HARN_WISCRS_Sawyer_County_Feet (deprecated) [4518] NAD_1983_HARN_WISCRS_Sawyer_County_Meters (deprecated) [4519] NAD_1983_HARN_WISCRS_Shawano_County_Feet (deprecated) [4520] NAD_1983_HARN_WISCRS_Shawano_County_Meters (deprecated) [4521] NAD_1983_HARN_WISCRS_Sheboygan_County_Feet [4522] NAD_1983_HARN_WISCRS_Sheboygan_County_Meters [4523] NAD_1983_HARN_WISCRS_St_Croix_County_Feet (deprecated) [4524] NAD_1983_HARN_WISCRS_St_Croix_County_Meters (deprecated) [4525] NAD_1983_HARN_WISCRS_Taylor_County_Feet (deprecated) [4526] NAD_1983_HARN_WISCRS_Taylor_County_Meters (deprecated) [4527] NAD_1983_HARN_WISCRS_Trempealeau_County_Feet (deprecated) [4528] NAD_1983_HARN_WISCRS_Trempealeau_County_Meters (deprecated) [4529] NAD_1983_HARN_WISCRS_Vernon_County_Feet (deprecated) [4530] NAD_1983_HARN_WISCRS_Vernon_County_Meters (deprecated) [4531] NAD_1983_HARN_WISCRS_Vilas_County_Feet (deprecated) [4532] NAD_1983_HARN_WISCRS_Vilas_County_Meters (deprecated) [4533] NAD_1983_HARN_WISCRS_Walworth_County_Feet (deprecated) [4534] NAD_1983_HARN_WISCRS_Walworth_County_Meters (deprecated) [4535] NAD_1983_HARN_WISCRS_Washburn_County_Feet (deprecated) [4536] NAD_1983_HARN_WISCRS_Washburn_County_Meters (deprecated) [4537] NAD_1983_HARN_WISCRS_Washington_County_Feet (deprecated) [4538] NAD_1983_HARN_WISCRS_Washington_County_Meters (deprecated) [4539] NAD_1983_HARN_WISCRS_Waukesha_County_Feet (deprecated) [4540] NAD_1983_HARN_WISCRS_Waukesha_County_Meters (deprecated) [4541] NAD_1983_HARN_WISCRS_Waupaca_County_Feet (deprecated) [4542] NAD_1983_HARN_WISCRS_Waupaca_County_Meters (deprecated) [4543] NAD_1983_HARN_WISCRS_Waushara_County_Feet (deprecated) [4544] NAD_1983_HARN_WISCRS_Waushara_County_Meters (deprecated) [4545] NAD_1983_HARN_WISCRS_Winnebago_County_Feet [4546] NAD_1983_HARN_WISCRS_Winnebago_County_Meters [4547] NAD_1983_HARN_WISCRS_Wood_County_Feet (deprecated) [4548] NAD_1983_HARN_WISCRS_Wood_County_Meters (deprecated) [4549] NAD_1983_HARN_Wisconsin_TM_US_Ft [4550] NAD_1983_Idaho-Ada_County [4551] NAD_1983_Idaho_TM [4552] NAD_1983_MA11_Guam_Map_Grid (deprecated) [4553] NAD_1983_MA11_UTM_Zone_55N [4554] NAD_1983_Maine_2000_Central_Zone (deprecated) [4555] NAD_1983_Maine_2000_East_Zone (deprecated) [4556] NAD_1983_Maine_2000_West_Zone (deprecated) [4557] NAD_1983_Michigan_GeoRef_Feet_US [4558] NAD_1983_Michigan_GeoRef_Meters (deprecated) [4559] NAD_1983_Mississippi_TM (deprecated) [4560] NAD_1983_NSRS2007_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [4561] NAD_1983_NSRS2007_UTM_Zone_20N [4562] NAD_1983_NSRS2007_Wisconsin_TM_US_Ft [4563] NAD_1983_Nebraska_Lancaster_County_FtUS [4564] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4565] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101_Feet [4566] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4567] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102_Feet [4568] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4569] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4570] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4571] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104_Feet [4572] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4573] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105_Feet [4574] NAD_1983_PA11_UTM_Zone_2S (deprecated) [4575] NAD_1983_PA11_UTM_Zone_4N (deprecated) [4576] NAD_1983_PA11_UTM_Zone_5N (deprecated) [4577] NAD_1983_PACP00_UTM_Zone_2S [4578] NAD_1983_PACP00_UTM_Zone_4N [4579] NAD_1983_PACP00_UTM_Zone_5N [4580] NAD_1983_StatePlane_Alabama_East_FIPS_0101_Feet [4581] NAD_1983_StatePlane_Alabama_West_FIPS_0102_Feet [4582] NAD_1983_StatePlane_Alaska_10_FIPS_5010_Feet [4583] NAD_1983_StatePlane_Alaska_1_FIPS_5001_Feet [4584] NAD_1983_StatePlane_Alaska_2_FIPS_5002_Feet [4585] NAD_1983_StatePlane_Alaska_3_FIPS_5003_Feet [4586] NAD_1983_StatePlane_Alaska_4_FIPS_5004_Feet [4587] NAD_1983_StatePlane_Alaska_5_FIPS_5005_Feet [4588] NAD_1983_StatePlane_Alaska_6_FIPS_5006_Feet [4589] NAD_1983_StatePlane_Alaska_7_FIPS_5007_Feet [4590] NAD_1983_StatePlane_Alaska_8_FIPS_5008_Feet [4591] NAD_1983_StatePlane_Alaska_9_FIPS_5009_Feet [4592] NAD_1983_StatePlane_Arizona_Central_FIPS_0202_Feet [4593] NAD_1983_StatePlane_Arizona_East_FIPS_0201_Feet [4594] NAD_1983_StatePlane_Arizona_West_FIPS_0203_Feet [4595] NAD_1983_StatePlane_Arkansas_North_FIPS_0301_Feet (deprecated) [4596] NAD_1983_StatePlane_Arkansas_South_FIPS_0302_Feet (deprecated) [4597] NAD_1983_StatePlane_California_III_FIPS_0403_Feet (deprecated) [4598] NAD_1983_StatePlane_California_II_FIPS_0402_Feet (deprecated) [4599] NAD_1983_StatePlane_California_IV_FIPS_0404_Feet (deprecated) [4600] NAD_1983_StatePlane_California_I_FIPS_0401_Feet (deprecated) [4601] NAD_1983_StatePlane_California_VI_FIPS_0406_Feet (deprecated) [4602] NAD_1983_StatePlane_California_V_FIPS_0405_Feet (deprecated) [4603] NAD_1983_StatePlane_Colorado_Central_FIPS_0502_Feet (deprecated) [4604] NAD_1983_StatePlane_Colorado_North_FIPS_0501_Feet (deprecated) [4605] NAD_1983_StatePlane_Colorado_South_FIPS_0503_Feet (deprecated) [4606] NAD_1983_StatePlane_Connecticut_FIPS_0600_Feet (deprecated) [4607] NAD_1983_StatePlane_Delaware_FIPS_0700_Feet (deprecated) [4608] NAD_1983_StatePlane_Florida_East_FIPS_0901_Feet (deprecated) [4609] NAD_1983_StatePlane_Florida_North_FIPS_0903_Feet (deprecated) [4610] NAD_1983_StatePlane_Florida_West_FIPS_0902_Feet (deprecated) [4611] NAD_1983_StatePlane_Georgia_East_FIPS_1001_Feet (deprecated) [4612] NAD_1983_StatePlane_Georgia_West_FIPS_1002_Feet (deprecated) [4613] NAD_1983_StatePlane_Guam_FIPS_5400 [4614] NAD_1983_StatePlane_Guam_FIPS_5400_Feet [4615] NAD_1983_StatePlane_Hawaii_1_FIPS_5101_Feet [4616] NAD_1983_StatePlane_Hawaii_2_FIPS_5102_Feet [4617] NAD_1983_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4618] NAD_1983_StatePlane_Hawaii_4_FIPS_5104_Feet [4619] NAD_1983_StatePlane_Hawaii_5_FIPS_5105_Feet [4620] NAD_1983_StatePlane_Idaho_Central_FIPS_1102_Feet (deprecated) [4621] NAD_1983_StatePlane_Idaho_East_FIPS_1101_Feet (deprecated) [4622] NAD_1983_StatePlane_Idaho_West_FIPS_1103_Feet (deprecated) [4623] NAD_1983_StatePlane_Illinois_East_FIPS_1201_Feet (deprecated) [4624] NAD_1983_StatePlane_Illinois_West_FIPS_1202_Feet (deprecated) [4625] NAD_1983_StatePlane_Indiana_East_FIPS_1301_Feet (deprecated) [4626] NAD_1983_StatePlane_Indiana_West_FIPS_1302_Feet (deprecated) [4627] NAD_1983_StatePlane_Iowa_North_FIPS_1401_Feet (deprecated) [4628] NAD_1983_StatePlane_Iowa_South_FIPS_1402_Feet (deprecated) [4629] NAD_1983_StatePlane_Kansas_North_FIPS_1501_Feet (deprecated) [4630] NAD_1983_StatePlane_Kansas_South_FIPS_1502_Feet (deprecated) [4631] NAD_1983_StatePlane_Kentucky_FIPS_1600 (deprecated) [4632] NAD_1983_StatePlane_Kentucky_FIPS_1600_Feet (deprecated) [4633] NAD_1983_StatePlane_Kentucky_North_FIPS_1601_Feet (deprecated) [4634] NAD_1983_StatePlane_Kentucky_South_FIPS_1602_Feet (deprecated) [4635] NAD_1983_StatePlane_Louisiana_North_FIPS_1701_Feet (deprecated) [4636] NAD_1983_StatePlane_Louisiana_South_FIPS_1702_Feet (deprecated) [4637] NAD_1983_StatePlane_Maine_East_FIPS_1801_Feet (deprecated) [4638] NAD_1983_StatePlane_Maine_West_FIPS_1802_Feet (deprecated) [4639] NAD_1983_StatePlane_Maryland_FIPS_1900_Feet (deprecated) [4640] NAD_1983_StatePlane_Massachusetts_Island_FIPS_2002_Feet (deprecated) [4641] NAD_1983_StatePlane_Massachusetts_Mainland_FIPS_2001_Feet (deprecated) [4642] NAD_1983_StatePlane_Michigan_Central_FIPS_2112_Feet [4643] NAD_1983_StatePlane_Michigan_North_FIPS_2111_Feet [4644] NAD_1983_StatePlane_Michigan_South_FIPS_2113_Feet [4645] NAD_1983_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4646] NAD_1983_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4647] NAD_1983_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4648] NAD_1983_StatePlane_Mississippi_East_FIPS_2301_Feet (deprecated) [4649] NAD_1983_StatePlane_Mississippi_West_FIPS_2302_Feet (deprecated) [4650] NAD_1983_StatePlane_Missouri_Central_FIPS_2402_Feet [4651] NAD_1983_StatePlane_Missouri_East_FIPS_2401_Feet [4652] NAD_1983_StatePlane_Missouri_West_FIPS_2403_Feet [4653] NAD_1983_StatePlane_Montana_FIPS_2500_Feet [4654] NAD_1983_StatePlane_Nebraska_FIPS_2600_Feet (deprecated) [4655] NAD_1983_StatePlane_Nevada_Central_FIPS_2702_Feet (deprecated) [4656] NAD_1983_StatePlane_Nevada_East_FIPS_2701_Feet (deprecated) [4657] NAD_1983_StatePlane_Nevada_West_FIPS_2703_Feet (deprecated) [4658] NAD_1983_StatePlane_New_Hampshire_FIPS_2800_Feet (deprecated) [4659] NAD_1983_StatePlane_New_Jersey_FIPS_2900_Feet (deprecated) [4660] NAD_1983_StatePlane_New_Mexico_Central_FIPS_3002_Feet (deprecated) [4661] NAD_1983_StatePlane_New_Mexico_East_FIPS_3001_Feet (deprecated) [4662] NAD_1983_StatePlane_New_Mexico_West_FIPS_3003_Feet (deprecated) [4663] NAD_1983_StatePlane_New_York_Central_FIPS_3102_Feet (deprecated) [4664] NAD_1983_StatePlane_New_York_East_FIPS_3101_Feet (deprecated) [4665] NAD_1983_StatePlane_New_York_Long_Island_FIPS_3104_Feet (deprecated) [4666] NAD_1983_StatePlane_New_York_West_FIPS_3103_Feet (deprecated) [4667] NAD_1983_StatePlane_North_Carolina_FIPS_3200_Feet (deprecated) [4668] NAD_1983_StatePlane_North_Dakota_North_FIPS_3301_Feet [4669] NAD_1983_StatePlane_North_Dakota_South_FIPS_3302_Feet [4670] NAD_1983_StatePlane_Ohio_North_FIPS_3401_Feet (deprecated) [4671] NAD_1983_StatePlane_Ohio_South_FIPS_3402_Feet (deprecated) [4672] NAD_1983_StatePlane_Oklahoma_North_FIPS_3501_Feet (deprecated) [4673] NAD_1983_StatePlane_Oklahoma_South_FIPS_3502_Feet (deprecated) [4674] NAD_1983_StatePlane_Oregon_North_FIPS_3601_Feet [4675] NAD_1983_StatePlane_Oregon_South_FIPS_3602_Feet [4676] NAD_1983_StatePlane_Pennsylvania_North_FIPS_3701_Feet (deprecated) [4677] NAD_1983_StatePlane_Pennsylvania_South_FIPS_3702_Feet (deprecated) [4678] NAD_1983_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200_Feet [4679] NAD_1983_StatePlane_Rhode_Island_FIPS_3800_Feet (deprecated) [4680] NAD_1983_StatePlane_South_Carolina_FIPS_3900_Feet [4681] NAD_1983_StatePlane_South_Dakota_North_FIPS_4001_Feet (deprecated) [4682] NAD_1983_StatePlane_South_Dakota_South_FIPS_4002_Feet (deprecated) [4683] NAD_1983_StatePlane_Tennessee_FIPS_4100_Feet (deprecated) [4684] NAD_1983_StatePlane_Texas_Central_FIPS_4203_Feet (deprecated) [4685] NAD_1983_StatePlane_Texas_North_Central_FIPS_4202_Feet (deprecated) [4686] NAD_1983_StatePlane_Texas_North_FIPS_4201_Feet (deprecated) [4687] NAD_1983_StatePlane_Texas_South_Central_FIPS_4204_Feet (deprecated) [4688] NAD_1983_StatePlane_Texas_South_FIPS_4205_Feet (deprecated) [4689] NAD_1983_StatePlane_Utah_Central_FIPS_4302_Feet (deprecated) [4690] NAD_1983_StatePlane_Utah_North_FIPS_4301_Feet (deprecated) [4691] NAD_1983_StatePlane_Utah_South_FIPS_4303_Feet (deprecated) [4692] NAD_1983_StatePlane_Vermont_FIPS_4400_Feet (deprecated) [4693] NAD_1983_StatePlane_Virginia_North_FIPS_4501_Feet (deprecated) [4694] NAD_1983_StatePlane_Virginia_South_FIPS_4502_Feet (deprecated) [4695] NAD_1983_StatePlane_Washington_North_FIPS_4601_Feet (deprecated) [4696] NAD_1983_StatePlane_Washington_South_FIPS_4602_Feet (deprecated) [4697] NAD_1983_StatePlane_West_Virginia_North_FIPS_4701_Feet (deprecated) [4698] NAD_1983_StatePlane_West_Virginia_South_FIPS_4702_Feet (deprecated) [4699] NAD_1983_StatePlane_Wisconsin_Central_FIPS_4802_Feet (deprecated) [4700] NAD_1983_StatePlane_Wisconsin_North_FIPS_4801_Feet (deprecated) [4701] NAD_1983_StatePlane_Wisconsin_South_FIPS_4803_Feet (deprecated) [4702] NAD_1983_StatePlane_Wyoming_East_Central_FIPS_4902_Feet (deprecated) [4703] NAD_1983_StatePlane_Wyoming_East_FIPS_4901_Feet (deprecated) [4704] NAD_1983_StatePlane_Wyoming_West_Central_FIPS_4903_Feet (deprecated) [4705] NAD_1983_StatePlane_Wyoming_West_FIPS_4904_Feet (deprecated) [4706] NAD_1983_Texas_Centric_Mapping_System_Albers (deprecated) [4707] NAD_1983_Texas_Centric_Mapping_System_Lambert (deprecated) [4708] NAD_1983_Texas_Statewide_Mapping_System (deprecated) [4709] NAD_1983_USFS_R6_Albers [4710] NAD_1983_USFS_R9_Albers [4711] NAD_1983_UTM_Zone_1N (deprecated) [4712] NAD_1983_UTM_Zone_2N (deprecated) [4713] NAD_1983_UTM_Zone_58N [4714] NAD_1983_UTM_Zone_59N (deprecated) [4715] NAD_1983_UTM_Zone_60N (deprecated) [4716] NAD_1983_Wisconsin_TM_US_Ft [4717] NAD_1983_WyLAM [4718] NEA74 Noumea / Noumea Lambert [4719] NEA74 Noumea / Noumea Lambert 2 [4720] NEA74 Noumea / UTM zone 58S [4721] NGN / UTM zone 38N [4722] NGN / UTM zone 39N [4723] NGO 1948 (Oslo) / NGO zone I [4724] NGO 1948 (Oslo) / NGO zone II [4725] NGO 1948 (Oslo) / NGO zone III [4726] NGO 1948 (Oslo) / NGO zone IV [4727] NGO 1948 (Oslo) / NGO zone V [4728] NGO 1948 (Oslo) / NGO zone VI [4729] NGO 1948 (Oslo) / NGO zone VII [4730] NGO 1948 (Oslo) / NGO zone VIII [4731] NGO_1948_Baerum_Kommune [4732] NGO_1948_Bergenhalvoen [4733] NGO_1948_Norway_Zone_1 [4734] NGO_1948_Norway_Zone_2 [4735] NGO_1948_Norway_Zone_3 [4736] NGO_1948_Norway_Zone_4 [4737] NGO_1948_Norway_Zone_5 [4738] NGO_1948_Norway_Zone_6 [4739] NGO_1948_Norway_Zone_7 [4740] NGO_1948_Norway_Zone_8 [4741] NGO_1948_Oslo_Baerum_Kommune [4742] NGO_1948_Oslo_Bergenhalvoen [4743] NGO_1948_Oslo_Kommune [4744] NGO_1948_Oslo_Oslo_Kommune [4745] NGO_1948_UTM_Zone_32N [4746] NGO_1948_UTM_Zone_33N [4747] NGO_1948_UTM_Zone_34N [4748] NGO_1948_UTM_Zone_35N [4749] NSIDC EASE-Grid Global [4750] NSIDC EASE-Grid North [4751] NSIDC EASE-Grid South [4752] NSIDC Sea Ice Polar Stereographic North [4753] NSIDC Sea Ice Polar Stereographic South [4754] NTF (Paris) / Centre France (deprecated) [4755] NTF (Paris) / Corse (deprecated) [4756] NTF (Paris) / France I (deprecated) [4757] NTF (Paris) / France II (deprecated) [4758] NTF (Paris) / France III (deprecated) [4759] NTF (Paris) / France IV (deprecated) [4760] NTF (Paris) / Lambert Centre France [4761] NTF (Paris) / Lambert Corse [4762] NTF (Paris) / Lambert Nord France [4763] NTF (Paris) / Lambert Sud France [4764] NTF (Paris) / Lambert zone I [4765] NTF (Paris) / Lambert zone II [4766] NTF (Paris) / Lambert zone III [4767] NTF (Paris) / Lambert zone IV [4768] NTF (Paris) / Nord France (deprecated) [4769] NTF (Paris) / Sud France (deprecated) [4770] NTF_France_III_degrees [4771] NTF_France_II_degrees [4772] NTF_France_IV_degrees [4773] NTF_France_I_degrees [4774] NTF_Lambert_Zone_I [4775] NTF_Lambert_Zone_II [4776] NTF_Lambert_Zone_III [4777] NTF_Lambert_Zone_IV [4778] NZGD2000 / Amuri 2000 [4779] NZGD2000 / Antipodes Islands TM 2000 [4780] NZGD2000 / Auckland Islands TM 2000 [4781] NZGD2000 / Bay of Plenty 2000 [4782] NZGD2000 / Bluff 2000 [4783] NZGD2000 / Buller 2000 [4784] NZGD2000 / Campbell Island TM 2000 [4785] NZGD2000 / Chatham Island Circuit 2000 [4786] NZGD2000 / Chatham Islands TM 2000 [4787] NZGD2000 / Collingwood 2000 [4788] NZGD2000 / Gawler 2000 [4789] NZGD2000 / Grey 2000 [4790] NZGD2000 / Hawkes Bay 2000 [4791] NZGD2000 / Hokitika 2000 [4792] NZGD2000 / Jacksons Bay 2000 [4793] NZGD2000 / Karamea 2000 [4794] NZGD2000 / Lindis Peak 2000 [4795] NZGD2000 / Marlborough 2000 [4796] NZGD2000 / Mount Eden 2000 [4797] NZGD2000 / Mount Nicholas 2000 [4798] NZGD2000 / Mount Pleasant 2000 [4799] NZGD2000 / Mount York 2000 [4800] NZGD2000 / NZCS2000 [4801] NZGD2000 / Nelson 2000 [4802] NZGD2000 / New Zealand Transverse Mercator 2000 [4803] NZGD2000 / North Taieri 2000 [4804] NZGD2000 / Observation Point 2000 [4805] NZGD2000 / Okarito 2000 [4806] NZGD2000 / Poverty Bay 2000 [4807] NZGD2000 / Raoul Island TM 2000 [4808] NZGD2000 / Taranaki 2000 [4809] NZGD2000 / Timaru 2000 [4810] NZGD2000 / Tuhirangi 2000 [4811] NZGD2000 / UTM zone 1S [4812] NZGD2000 / UTM zone 58S [4813] NZGD2000 / UTM zone 59S [4814] NZGD2000 / UTM zone 60S [4815] NZGD2000 / Wairarapa 2000 [4816] NZGD2000 / Wanganui 2000 [4817] NZGD2000 / Wellington 2000 [4818] NZGD49 / Amuri Circuit [4819] NZGD49 / Bay of Plenty Circuit [4820] NZGD49 / Bluff Circuit [4821] NZGD49 / Buller Circuit [4822] NZGD49 / Collingwood Circuit [4823] NZGD49 / Gawler Circuit [4824] NZGD49 / Grey Circuit [4825] NZGD49 / Hawkes Bay Circuit [4826] NZGD49 / Hokitika Circuit [4827] NZGD49 / Jacksons Bay Circuit [4828] NZGD49 / Karamea Circuit [4829] NZGD49 / Lindis Peak Circuit [4830] NZGD49 / Marlborough Circuit [4831] NZGD49 / Mount Eden Circuit [4832] NZGD49 / Mount Nicholas Circuit [4833] NZGD49 / Mount Pleasant Circuit [4834] NZGD49 / Mount York Circuit [4835] NZGD49 / Nelson Circuit [4836] NZGD49 / New Zealand Map Grid [4837] NZGD49 / North Island Grid [4838] NZGD49 / North Taieri Circuit [4839] NZGD49 / Observation Point Circuit [4840] NZGD49 / Okarito Circuit [4841] NZGD49 / Poverty Bay Circuit [4842] NZGD49 / South Island Grid [4843] NZGD49 / Taranaki Circuit [4844] NZGD49 / Timaru Circuit [4845] NZGD49 / Tuhirangi Circuit [4846] NZGD49 / UTM zone 58S [4847] NZGD49 / UTM zone 59S [4848] NZGD49 / UTM zone 60S [4849] NZGD49 / Wairarapa Circuit [4850] NZGD49 / Wanganui Circuit [4851] NZGD49 / Wellington Circuit [4852] NZGD_2000_Chatham_Island_Circuit (deprecated) [4853] Nahrwan 1934 / Iraq zone [4854] Nahrwan 1934 / UTM zone 37N [4855] Nahrwan 1934 / UTM zone 38N [4856] Nahrwan 1934 / UTM zone 39N [4857] Nahrwan 1967 / UTM zone 37N (deprecated) [4858] Nahrwan 1967 / UTM zone 38N (deprecated) [4859] Nahrwan 1967 / UTM zone 39N [4860] Nahrwan 1967 / UTM zone 40N [4861] Nakhl-e Ghanem / UTM zone 39N [4862] Naparima 1955 / UTM zone 20N [4863] Naparima 1972 / UTM zone 20N [4864] Nepal_Nagarkot_TM [4865] New Beijing / 3-degree Gauss-Kruger CM 102E [4866] New Beijing / 3-degree Gauss-Kruger CM 105E [4867] New Beijing / 3-degree Gauss-Kruger CM 108E [4868] New Beijing / 3-degree Gauss-Kruger CM 111E [4869] New Beijing / 3-degree Gauss-Kruger CM 114E [4870] New Beijing / 3-degree Gauss-Kruger CM 117E [4871] New Beijing / 3-degree Gauss-Kruger CM 120E [4872] New Beijing / 3-degree Gauss-Kruger CM 123E [4873] New Beijing / 3-degree Gauss-Kruger CM 126E [4874] New Beijing / 3-degree Gauss-Kruger CM 129E [4875] New Beijing / 3-degree Gauss-Kruger CM 132E [4876] New Beijing / 3-degree Gauss-Kruger CM 135E [4877] New Beijing / 3-degree Gauss-Kruger CM 75E [4878] New Beijing / 3-degree Gauss-Kruger CM 78E [4879] New Beijing / 3-degree Gauss-Kruger CM 81E [4880] New Beijing / 3-degree Gauss-Kruger CM 84E [4881] New Beijing / 3-degree Gauss-Kruger CM 87E [4882] New Beijing / 3-degree Gauss-Kruger CM 90E [4883] New Beijing / 3-degree Gauss-Kruger CM 93E [4884] New Beijing / 3-degree Gauss-Kruger CM 96E [4885] New Beijing / 3-degree Gauss-Kruger CM 99E [4886] New Beijing / 3-degree Gauss-Kruger zone 25 [4887] New Beijing / 3-degree Gauss-Kruger zone 26 [4888] New Beijing / 3-degree Gauss-Kruger zone 27 [4889] New Beijing / 3-degree Gauss-Kruger zone 28 [4890] New Beijing / 3-degree Gauss-Kruger zone 29 [4891] New Beijing / 3-degree Gauss-Kruger zone 30 [4892] New Beijing / 3-degree Gauss-Kruger zone 31 [4893] New Beijing / 3-degree Gauss-Kruger zone 32 [4894] New Beijing / 3-degree Gauss-Kruger zone 33 [4895] New Beijing / 3-degree Gauss-Kruger zone 34 [4896] New Beijing / 3-degree Gauss-Kruger zone 35 [4897] New Beijing / 3-degree Gauss-Kruger zone 36 [4898] New Beijing / 3-degree Gauss-Kruger zone 37 [4899] New Beijing / 3-degree Gauss-Kruger zone 38 [4900] New Beijing / 3-degree Gauss-Kruger zone 39 [4901] New Beijing / 3-degree Gauss-Kruger zone 40 [4902] New Beijing / 3-degree Gauss-Kruger zone 41 [4903] New Beijing / 3-degree Gauss-Kruger zone 42 [4904] New Beijing / 3-degree Gauss-Kruger zone 43 [4905] New Beijing / 3-degree Gauss-Kruger zone 44 [4906] New Beijing / 3-degree Gauss-Kruger zone 45 [4907] New Beijing / Gauss-Kruger CM 105E [4908] New Beijing / Gauss-Kruger CM 111E [4909] New Beijing / Gauss-Kruger CM 117E [4910] New Beijing / Gauss-Kruger CM 123E [4911] New Beijing / Gauss-Kruger CM 129E [4912] New Beijing / Gauss-Kruger CM 135E [4913] New Beijing / Gauss-Kruger CM 75E [4914] New Beijing / Gauss-Kruger CM 81E [4915] New Beijing / Gauss-Kruger CM 87E [4916] New Beijing / Gauss-Kruger CM 93E [4917] New Beijing / Gauss-Kruger CM 99E [4918] New Beijing / Gauss-Kruger zone 13 [4919] New Beijing / Gauss-Kruger zone 14 [4920] New Beijing / Gauss-Kruger zone 15 [4921] New Beijing / Gauss-Kruger zone 16 [4922] New Beijing / Gauss-Kruger zone 17 [4923] New Beijing / Gauss-Kruger zone 18 [4924] New Beijing / Gauss-Kruger zone 19 [4925] New Beijing / Gauss-Kruger zone 20 [4926] New Beijing / Gauss-Kruger zone 21 [4927] New Beijing / Gauss-Kruger zone 22 [4928] New Beijing / Gauss-Kruger zone 23 [4929] New_Beijing_3_Degree_Gauss_Kruger_CM_132E [4930] Nord Sahara 1959 / Nord Algerie [4931] Nord Sahara 1959 / Sud Algerie [4932] Nord Sahara 1959 / UTM zone 29N [4933] Nord Sahara 1959 / UTM zone 30N [4934] Nord Sahara 1959 / UTM zone 31N [4935] Nord Sahara 1959 / UTM zone 32N [4936] Nord_Algerie (deprecated) [4937] Nord_Algerie_Ancienne_Degree [4938] Nord_Algerie_Degree (deprecated) [4939] Nord_Maroc_Degree [4940] North_America_Albers_Equal_Area_Conic [4941] North_America_Equidistant_Conic [4942] North_America_Lambert_Conformal_Conic [4943] North_Pole_Azimuthal_Equidistant [4944] North_Pole_Gnomonic [4945] North_Pole_Lambert_Azimuthal_Equal_Area [4946] North_Pole_Orthographic [4947] North_Pole_Stereographic [4948] OCRS_Baker_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4949] OCRS_Baker_NAD_1983_CORS96_TM_Meters (deprecated) [4950] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4951] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Meters (deprecated) [4952] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4953] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Meters (deprecated) [4954] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4955] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Meters (deprecated) [4956] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4957] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4958] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4959] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Meters (deprecated) [4960] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4961] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Meters (deprecated) [4962] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4963] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Meters (deprecated) [4964] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4965] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Meters (deprecated) [4966] OCRS_Eugene_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4967] OCRS_Eugene_NAD_1983_CORS96_TM_Meters (deprecated) [4968] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4969] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Meters (deprecated) [4970] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4971] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Meters (deprecated) [4972] OCRS_La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4973] OCRS_La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4974] OCRS_Ontario_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4975] OCRS_Ontario_NAD_1983_CORS96_TM_Meters (deprecated) [4976] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4977] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Meters (deprecated) [4978] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4979] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4980] OCRS_Pendleton_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4981] OCRS_Pendleton_NAD_1983_CORS96_TM_Meters (deprecated) [4982] OCRS_Portland_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4983] OCRS_Portland_NAD_1983_CORS96_LCC_Meters (deprecated) [4984] OCRS_Salem_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4985] OCRS_Salem_NAD_1983_CORS96_TM_Meters (deprecated) [4986] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4987] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4988] ONGD14 / UTM zone 39N [4989] ONGD14 / UTM zone 40N [4990] ONGD14 / UTM zone 41N [4991] OSGB 1936 / British National Grid [4992] OSGB36_Highways_England_A10H1 [4993] OSGB36_Highways_England_A10H2 [4994] OSGB36_Highways_England_A11H1 [4995] OSGB36_Highways_England_A11H2 [4996] OSGB36_Highways_England_A11H3 [4997] OSGB36_Highways_England_A12H1 [4998] OSGB36_Highways_England_A12H2 [4999] OSGB36_Highways_England_A12H3 [5000] OSGB36_Highways_England_A13H1 [5001] OSGB36_Highways_England_A13H2 [5002] OSGB36_Highways_England_A14H1 [5003] OSGB36_Highways_England_A14H2 [5004] OSGB36_Highways_England_A15H1 [5005] OSGB36_Highways_England_A15H2 [5006] OSGB36_Highways_England_A16H1 [5007] OSGB36_Highways_England_A16H2 [5008] OSGB36_Highways_England_A17H1 [5009] OSGB36_Highways_England_A17H2 [5010] OSGB36_Highways_England_A18H1 [5011] OSGB36_Highways_England_A18H2 [5012] OSGB36_Highways_England_A19H1 [5013] OSGB36_Highways_England_A19H2 [5014] OSGB36_Highways_England_A1H1 [5015] OSGB36_Highways_England_A20H1 [5016] OSGB36_Highways_England_A20H2 [5017] OSGB36_Highways_England_A21H1 [5018] OSGB36_Highways_England_A21H2 [5019] OSGB36_Highways_England_A22H1 [5020] OSGB36_Highways_England_A22H2 [5021] OSGB36_Highways_England_A23H1 [5022] OSGB36_Highways_England_A23H2 [5023] OSGB36_Highways_England_A24H1 [5024] OSGB36_Highways_England_A24H2 [5025] OSGB36_Highways_England_A25H1 [5026] OSGB36_Highways_England_A25H2 [5027] OSGB36_Highways_England_A26H1 [5028] OSGB36_Highways_England_A26H2 [5029] OSGB36_Highways_England_A27H1 [5030] OSGB36_Highways_England_A27H2 [5031] OSGB36_Highways_England_A28H1 [5032] OSGB36_Highways_England_A28H2 [5033] OSGB36_Highways_England_A29H1 [5034] OSGB36_Highways_England_A2H1 [5035] OSGB36_Highways_England_A30H1 [5036] OSGB36_Highways_England_A3H1 [5037] OSGB36_Highways_England_A3H2 [5038] OSGB36_Highways_England_A4H1 [5039] OSGB36_Highways_England_A5H1 [5040] OSGB36_Highways_England_A5H2 [5041] OSGB36_Highways_England_A6H1 [5042] OSGB36_Highways_England_A6H2 [5043] OSGB36_Highways_England_A7H1 [5044] OSGB36_Highways_England_A7H2 [5045] OSGB36_Highways_England_A8H1 [5046] OSGB36_Highways_England_A8H2 [5047] OSGB36_Highways_England_A9H1 [5048] OSGB36_Highways_England_A9H2 [5049] OSGB36_Highways_England_B15H1 [5050] OSGB36_Highways_England_B15H2 [5051] OSGB36_Highways_England_B15H3 [5052] OSGB36_Highways_England_B16H1 [5053] OSGB36_Highways_England_B16H2 [5054] OSGB36_Highways_England_B16H3 [5055] OSGB36_Highways_England_B16H4 [5056] OSGB36_Highways_England_B17H1 [5057] OSGB36_Highways_England_B17H2 [5058] OSGB36_Highways_England_B18H1 [5059] OSGB36_Highways_England_B18H2 [5060] OSGB36_Highways_England_B19H1 [5061] OSGB36_Highways_England_B20H1 [5062] OSGB36_Highways_England_B21H1 [5063] OSGB36_Highways_England_B22H1 [5064] OSGB36_Highways_England_B23H1 [5065] OSGB36_Highways_England_B24H1 [5066] OSGB36_Highways_England_B25H1 [5067] OSGB36_Highways_England_B26H1 [5068] OSGB36_Highways_England_B27H1 [5069] OSGB36_Highways_England_B28H1 [5070] OSGB36_Highways_England_B29H1 [5071] OSGB36_Highways_England_B30H1 [5072] OSGB36_Highways_England_B31H1 [5073] OSGB36_Highways_England_B32H1 [5074] OSGB36_Highways_England_C13H1 [5075] OSGB36_Highways_England_C14H1 [5076] OSGB36_Highways_England_C14H2 [5077] OSGB36_Highways_England_C14H3 [5078] OSGB36_Highways_England_C14H4 [5079] OSGB36_Highways_England_C15H1 [5080] OSGB36_Highways_England_C15H2 [5081] OSGB36_Highways_England_C15H3 [5082] OSGB36_Highways_England_C15H4 [5083] OSGB36_Highways_England_C15H5 [5084] OSGB36_Highways_England_C16H1 [5085] OSGB36_Highways_England_C16H2 [5086] OSGB36_Highways_England_C16H3 [5087] OSGB36_Highways_England_C16H4 [5088] OSGB36_Highways_England_C17H1 [5089] OSGB36_Highways_England_C17H2 [5090] OSGB36_Highways_England_C18H1 [5091] OSGB36_Highways_England_C18H2 [5092] OSGB36_Highways_England_C19H1 [5093] OSGB36_Highways_England_C19H2 [5094] OSNI 1952 / Irish National Grid [5095] Observatorio_Meteorologico_1939_UTM_Zone_25N [5096] Observatorio_Meteorologico_1965_Macau_Grid [5097] Ocotepeque 1935 / Costa Rica Norte [5098] Ocotepeque 1935 / Costa Rica Sur [5099] Ocotepeque 1935 / El Salvador Lambert [5100] Ocotepeque 1935 / Guatemala Norte [5101] Ocotepeque 1935 / Guatemala Norte (deprecated) [5102] Ocotepeque 1935 / Guatemala Sur [5103] Ocotepeque 1935 / Nicaragua Norte [5104] Ocotepeque 1935 / Nicaragua Sur [5105] Ocotepeque_1935_Costa_Rica_Lambert_Norte [5106] Ocotepeque_1935_Costa_Rica_Lambert_Sur [5107] Old Hawaiian / Hawaii zone 1 [5108] Old Hawaiian / Hawaii zone 2 [5109] Old Hawaiian / Hawaii zone 3 [5110] Old Hawaiian / Hawaii zone 4 [5111] Old Hawaiian / Hawaii zone 5 [5112] Old_Hawaiian_UTM_Zone_4N [5113] Old_Hawaiian_UTM_Zone_5N [5114] PD/83 / 3-degree Gauss-Kruger zone 3 [5115] PD/83 / 3-degree Gauss-Kruger zone 3 (E-N) [5116] PD/83 / 3-degree Gauss-Kruger zone 4 [5117] PD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5118] PNG94 / PNGMG94 zone 54 [5119] PNG94 / PNGMG94 zone 55 [5120] PNG94 / PNGMG94 zone 56 [5121] POSGAR 2007 / Argentina 1 [5122] POSGAR 2007 / Argentina 2 [5123] POSGAR 2007 / Argentina 3 [5124] POSGAR 2007 / Argentina 4 [5125] POSGAR 2007 / Argentina 5 [5126] POSGAR 2007 / Argentina 6 [5127] POSGAR 2007 / Argentina 7 [5128] POSGAR 94 / Argentina 1 [5129] POSGAR 94 / Argentina 2 [5130] POSGAR 94 / Argentina 3 [5131] POSGAR 94 / Argentina 4 [5132] POSGAR 94 / Argentina 5 [5133] POSGAR 94 / Argentina 6 [5134] POSGAR 94 / Argentina 7 [5135] POSGAR 98 / Argentina 1 [5136] POSGAR 98 / Argentina 2 [5137] POSGAR 98 / Argentina 3 [5138] POSGAR 98 / Argentina 4 [5139] POSGAR 98 / Argentina 5 [5140] POSGAR 98 / Argentina 6 [5141] POSGAR 98 / Argentina 7 [5142] PRS92 / Philippines zone 1 [5143] PRS92 / Philippines zone 2 [5144] PRS92 / Philippines zone 3 [5145] PRS92 / Philippines zone 4 [5146] PRS92 / Philippines zone 5 [5147] PRS_1992_UTM_Zone_50N [5148] PRS_1992_UTM_Zone_51N [5149] PRS_1992_UTM_Zone_52N [5150] PSAD56 / ICN Regional [5151] PSAD56 / Peru central zone [5152] PSAD56 / Peru east zone [5153] PSAD56 / Peru west zone [5154] PSAD56 / UTM zone 17N [5155] PSAD56 / UTM zone 17S [5156] PSAD56 / UTM zone 18N [5157] PSAD56 / UTM zone 18S [5158] PSAD56 / UTM zone 19N [5159] PSAD56 / UTM zone 19S [5160] PSAD56 / UTM zone 20N [5161] PSAD56 / UTM zone 20S [5162] PSAD56 / UTM zone 21N [5163] PSAD56 / UTM zone 21S [5164] PSAD56 / UTM zone 22S [5165] PSD93 / UTM zone 39N [5166] PSD93 / UTM zone 40N [5167] PTRA08 / LAEA Europe [5168] PTRA08 / LCC Europe [5169] PTRA08 / UTM zone 25N [5170] PTRA08 / UTM zone 26N [5171] PTRA08 / UTM zone 28N [5172] PTRA08_UTM_Zone_25N (deprecated) [5173] PTRA08_UTM_Zone_26N (deprecated) [5174] PTRA08_UTM_Zone_28N (deprecated) [5175] Palestine 1923 / Israeli CS Grid [5176] Palestine 1923 / Palestine Belt [5177] Palestine 1923 / Palestine Grid [5178] Palestine 1923 / Palestine Grid modified [5179] Palestine_1923_Palestine_Grid_TM (deprecated) [5180] Pampa del Castillo / Argentina 2 [5181] Panama-Colon 1911 / Panama Lambert [5182] Panama-Colon 1911 / Panama Polyconic [5183] Panhandle_Energy_Albers [5184] Peru96 / UTM zone 17S [5185] Peru96 / UTM zone 17S (deprecated) [5186] Peru96 / UTM zone 18S [5187] Peru96 / UTM zone 19S [5188] Pitcairn 1967 / UTM zone 9S [5189] Pitcairn 2006 / Pitcairn TM 2006 [5190] Pohnpei_Az_Eq_1971 [5191] Pointe Noire / UTM zone 32S [5192] Popular Visualisation CRS / Mercator (deprecated) [5193] Popular Visualisation CRS / Mercator (deprecated) [5194] Porto Santo / UTM zone 28N [5195] Porto Santo 1995 / UTM zone 28N [5196] Porto_Santo_1936_UTM_Zone_28N (deprecated) [5197] Puerto Rico / St. Croix [5198] Puerto Rico / UTM zone 20N [5199] Puerto Rico State Plane CS of 1927 [5200] Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E [5201] Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E [5202] Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E [5203] Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E [5204] Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E [5205] Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E [5206] Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E [5207] Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E [5208] Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E [5209] Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E [5210] Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E [5211] Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E [5212] Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E [5213] Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E [5214] Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E [5215] Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E [5216] Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E [5217] Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E [5218] Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E [5219] Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E [5220] Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E [5221] Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E [5222] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E [5223] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W [5224] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E [5225] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W [5226] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E [5227] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W [5228] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E [5229] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W [5230] Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E [5231] Pulkovo 1942 / 3-degree Gauss-Kruger CM 18E (deprecated) [5232] Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E [5233] Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E [5234] Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E [5235] Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E [5236] Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E [5237] Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E [5238] Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E [5239] Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E [5240] Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E [5241] Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E [5242] Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E [5243] Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E [5244] Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E [5245] Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E [5246] Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E [5247] Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E [5248] Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E [5249] Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E [5250] Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E [5251] Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E [5252] Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E [5253] Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E [5254] Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E [5255] Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E [5256] Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E [5257] Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E [5258] Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E [5259] Pulkovo 1942 / 3-degree Gauss-Kruger zone 10 [5260] Pulkovo 1942 / 3-degree Gauss-Kruger zone 11 [5261] Pulkovo 1942 / 3-degree Gauss-Kruger zone 12 [5262] Pulkovo 1942 / 3-degree Gauss-Kruger zone 13 [5263] Pulkovo 1942 / 3-degree Gauss-Kruger zone 14 [5264] Pulkovo 1942 / 3-degree Gauss-Kruger zone 15 [5265] Pulkovo 1942 / 3-degree Gauss-Kruger zone 16 [5266] Pulkovo 1942 / 3-degree Gauss-Kruger zone 17 [5267] Pulkovo 1942 / 3-degree Gauss-Kruger zone 18 [5268] Pulkovo 1942 / 3-degree Gauss-Kruger zone 19 [5269] Pulkovo 1942 / 3-degree Gauss-Kruger zone 20 [5270] Pulkovo 1942 / 3-degree Gauss-Kruger zone 21 [5271] Pulkovo 1942 / 3-degree Gauss-Kruger zone 22 [5272] Pulkovo 1942 / 3-degree Gauss-Kruger zone 23 [5273] Pulkovo 1942 / 3-degree Gauss-Kruger zone 24 [5274] Pulkovo 1942 / 3-degree Gauss-Kruger zone 25 [5275] Pulkovo 1942 / 3-degree Gauss-Kruger zone 26 [5276] Pulkovo 1942 / 3-degree Gauss-Kruger zone 27 [5277] Pulkovo 1942 / 3-degree Gauss-Kruger zone 28 [5278] Pulkovo 1942 / 3-degree Gauss-Kruger zone 29 [5279] Pulkovo 1942 / 3-degree Gauss-Kruger zone 30 [5280] Pulkovo 1942 / 3-degree Gauss-Kruger zone 31 [5281] Pulkovo 1942 / 3-degree Gauss-Kruger zone 32 [5282] Pulkovo 1942 / 3-degree Gauss-Kruger zone 33 [5283] Pulkovo 1942 / 3-degree Gauss-Kruger zone 34 [5284] Pulkovo 1942 / 3-degree Gauss-Kruger zone 35 [5285] Pulkovo 1942 / 3-degree Gauss-Kruger zone 36 [5286] Pulkovo 1942 / 3-degree Gauss-Kruger zone 37 [5287] Pulkovo 1942 / 3-degree Gauss-Kruger zone 38 [5288] Pulkovo 1942 / 3-degree Gauss-Kruger zone 39 [5289] Pulkovo 1942 / 3-degree Gauss-Kruger zone 40 [5290] Pulkovo 1942 / 3-degree Gauss-Kruger zone 41 [5291] Pulkovo 1942 / 3-degree Gauss-Kruger zone 42 [5292] Pulkovo 1942 / 3-degree Gauss-Kruger zone 43 [5293] Pulkovo 1942 / 3-degree Gauss-Kruger zone 44 [5294] Pulkovo 1942 / 3-degree Gauss-Kruger zone 45 [5295] Pulkovo 1942 / 3-degree Gauss-Kruger zone 46 [5296] Pulkovo 1942 / 3-degree Gauss-Kruger zone 47 [5297] Pulkovo 1942 / 3-degree Gauss-Kruger zone 48 [5298] Pulkovo 1942 / 3-degree Gauss-Kruger zone 49 [5299] Pulkovo 1942 / 3-degree Gauss-Kruger zone 50 [5300] Pulkovo 1942 / 3-degree Gauss-Kruger zone 51 [5301] Pulkovo 1942 / 3-degree Gauss-Kruger zone 52 [5302] Pulkovo 1942 / 3-degree Gauss-Kruger zone 53 [5303] Pulkovo 1942 / 3-degree Gauss-Kruger zone 54 [5304] Pulkovo 1942 / 3-degree Gauss-Kruger zone 55 [5305] Pulkovo 1942 / 3-degree Gauss-Kruger zone 56 [5306] Pulkovo 1942 / 3-degree Gauss-Kruger zone 57 [5307] Pulkovo 1942 / 3-degree Gauss-Kruger zone 58 [5308] Pulkovo 1942 / 3-degree Gauss-Kruger zone 59 [5309] Pulkovo 1942 / 3-degree Gauss-Kruger zone 6 (deprecated) [5310] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 [5311] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 (deprecated) [5312] Pulkovo 1942 / 3-degree Gauss-Kruger zone 61 [5313] Pulkovo 1942 / 3-degree Gauss-Kruger zone 62 [5314] Pulkovo 1942 / 3-degree Gauss-Kruger zone 63 [5315] Pulkovo 1942 / 3-degree Gauss-Kruger zone 64 [5316] Pulkovo 1942 / 3-degree Gauss-Kruger zone 7 [5317] Pulkovo 1942 / 3-degree Gauss-Kruger zone 8 [5318] Pulkovo 1942 / 3-degree Gauss-Kruger zone 9 [5319] Pulkovo 1942 / CS63 zone A1 [5320] Pulkovo 1942 / CS63 zone A2 [5321] Pulkovo 1942 / CS63 zone A3 [5322] Pulkovo 1942 / CS63 zone A4 [5323] Pulkovo 1942 / CS63 zone C0 [5324] Pulkovo 1942 / CS63 zone C1 [5325] Pulkovo 1942 / CS63 zone C2 [5326] Pulkovo 1942 / CS63 zone K2 [5327] Pulkovo 1942 / CS63 zone K3 [5328] Pulkovo 1942 / CS63 zone K4 [5329] Pulkovo 1942 / CS63 zone X1 [5330] Pulkovo 1942 / CS63 zone X2 [5331] Pulkovo 1942 / CS63 zone X3 [5332] Pulkovo 1942 / CS63 zone X4 [5333] Pulkovo 1942 / CS63 zone X5 [5334] Pulkovo 1942 / CS63 zone X6 [5335] Pulkovo 1942 / CS63 zone X7 [5336] Pulkovo 1942 / Caspian Sea Mercator [5337] Pulkovo 1942 / Gauss-Kruger 10N (deprecated) [5338] Pulkovo 1942 / Gauss-Kruger 11N (deprecated) [5339] Pulkovo 1942 / Gauss-Kruger 12N (deprecated) [5340] Pulkovo 1942 / Gauss-Kruger 13N (deprecated) [5341] Pulkovo 1942 / Gauss-Kruger 14N (deprecated) [5342] Pulkovo 1942 / Gauss-Kruger 15N (deprecated) [5343] Pulkovo 1942 / Gauss-Kruger 16N (deprecated) [5344] Pulkovo 1942 / Gauss-Kruger 17N (deprecated) [5345] Pulkovo 1942 / Gauss-Kruger 18N (deprecated) [5346] Pulkovo 1942 / Gauss-Kruger 19N (deprecated) [5347] Pulkovo 1942 / Gauss-Kruger 20N (deprecated) [5348] Pulkovo 1942 / Gauss-Kruger 21N (deprecated) [5349] Pulkovo 1942 / Gauss-Kruger 22N (deprecated) [5350] Pulkovo 1942 / Gauss-Kruger 23N (deprecated) [5351] Pulkovo 1942 / Gauss-Kruger 24N (deprecated) [5352] Pulkovo 1942 / Gauss-Kruger 25N (deprecated) [5353] Pulkovo 1942 / Gauss-Kruger 26N (deprecated) [5354] Pulkovo 1942 / Gauss-Kruger 27N (deprecated) [5355] Pulkovo 1942 / Gauss-Kruger 28N (deprecated) [5356] Pulkovo 1942 / Gauss-Kruger 29N (deprecated) [5357] Pulkovo 1942 / Gauss-Kruger 2N (deprecated) [5358] Pulkovo 1942 / Gauss-Kruger 30N (deprecated) [5359] Pulkovo 1942 / Gauss-Kruger 31N (deprecated) [5360] Pulkovo 1942 / Gauss-Kruger 32N (deprecated) [5361] Pulkovo 1942 / Gauss-Kruger 3N (deprecated) [5362] Pulkovo 1942 / Gauss-Kruger 4N (deprecated) [5363] Pulkovo 1942 / Gauss-Kruger 5N (deprecated) [5364] Pulkovo 1942 / Gauss-Kruger 6N (deprecated) [5365] Pulkovo 1942 / Gauss-Kruger 7N (deprecated) [5366] Pulkovo 1942 / Gauss-Kruger 8N (deprecated) [5367] Pulkovo 1942 / Gauss-Kruger 9N (deprecated) [5368] Pulkovo 1942 / Gauss-Kruger CM 105E [5369] Pulkovo 1942 / Gauss-Kruger CM 111E [5370] Pulkovo 1942 / Gauss-Kruger CM 117E [5371] Pulkovo 1942 / Gauss-Kruger CM 123E [5372] Pulkovo 1942 / Gauss-Kruger CM 129E [5373] Pulkovo 1942 / Gauss-Kruger CM 135E [5374] Pulkovo 1942 / Gauss-Kruger CM 141E [5375] Pulkovo 1942 / Gauss-Kruger CM 147E [5376] Pulkovo 1942 / Gauss-Kruger CM 153E [5377] Pulkovo 1942 / Gauss-Kruger CM 159E [5378] Pulkovo 1942 / Gauss-Kruger CM 15E (deprecated) [5379] Pulkovo 1942 / Gauss-Kruger CM 165E [5380] Pulkovo 1942 / Gauss-Kruger CM 171E [5381] Pulkovo 1942 / Gauss-Kruger CM 171W [5382] Pulkovo 1942 / Gauss-Kruger CM 177E [5383] Pulkovo 1942 / Gauss-Kruger CM 177W [5384] Pulkovo 1942 / Gauss-Kruger CM 21E [5385] Pulkovo 1942 / Gauss-Kruger CM 27E [5386] Pulkovo 1942 / Gauss-Kruger CM 33E [5387] Pulkovo 1942 / Gauss-Kruger CM 39E [5388] Pulkovo 1942 / Gauss-Kruger CM 45E [5389] Pulkovo 1942 / Gauss-Kruger CM 51E [5390] Pulkovo 1942 / Gauss-Kruger CM 57E [5391] Pulkovo 1942 / Gauss-Kruger CM 63E [5392] Pulkovo 1942 / Gauss-Kruger CM 69E [5393] Pulkovo 1942 / Gauss-Kruger CM 75E [5394] Pulkovo 1942 / Gauss-Kruger CM 81E [5395] Pulkovo 1942 / Gauss-Kruger CM 87E [5396] Pulkovo 1942 / Gauss-Kruger CM 93E [5397] Pulkovo 1942 / Gauss-Kruger CM 99E [5398] Pulkovo 1942 / Gauss-Kruger CM 9E (deprecated) [5399] Pulkovo 1942 / Gauss-Kruger zone 10 [5400] Pulkovo 1942 / Gauss-Kruger zone 11 [5401] Pulkovo 1942 / Gauss-Kruger zone 12 [5402] Pulkovo 1942 / Gauss-Kruger zone 13 [5403] Pulkovo 1942 / Gauss-Kruger zone 14 [5404] Pulkovo 1942 / Gauss-Kruger zone 15 [5405] Pulkovo 1942 / Gauss-Kruger zone 16 [5406] Pulkovo 1942 / Gauss-Kruger zone 17 [5407] Pulkovo 1942 / Gauss-Kruger zone 18 [5408] Pulkovo 1942 / Gauss-Kruger zone 19 [5409] Pulkovo 1942 / Gauss-Kruger zone 2 (deprecated) [5410] Pulkovo 1942 / Gauss-Kruger zone 20 [5411] Pulkovo 1942 / Gauss-Kruger zone 21 [5412] Pulkovo 1942 / Gauss-Kruger zone 22 [5413] Pulkovo 1942 / Gauss-Kruger zone 23 [5414] Pulkovo 1942 / Gauss-Kruger zone 24 [5415] Pulkovo 1942 / Gauss-Kruger zone 25 [5416] Pulkovo 1942 / Gauss-Kruger zone 26 [5417] Pulkovo 1942 / Gauss-Kruger zone 27 [5418] Pulkovo 1942 / Gauss-Kruger zone 28 [5419] Pulkovo 1942 / Gauss-Kruger zone 29 [5420] Pulkovo 1942 / Gauss-Kruger zone 3 (deprecated) [5421] Pulkovo 1942 / Gauss-Kruger zone 30 [5422] Pulkovo 1942 / Gauss-Kruger zone 31 [5423] Pulkovo 1942 / Gauss-Kruger zone 32 [5424] Pulkovo 1942 / Gauss-Kruger zone 4 [5425] Pulkovo 1942 / Gauss-Kruger zone 5 [5426] Pulkovo 1942 / Gauss-Kruger zone 6 [5427] Pulkovo 1942 / Gauss-Kruger zone 7 [5428] Pulkovo 1942 / Gauss-Kruger zone 8 [5429] Pulkovo 1942 / Gauss-Kruger zone 9 [5430] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 10 [5431] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 [5432] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 (E-N) [5433] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 [5434] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 (E-N) [5435] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 [5436] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 (E-N) [5437] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 6 [5438] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 7 [5439] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 8 [5440] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 9 [5441] Pulkovo 1942(58) / GUGiK-80 [5442] Pulkovo 1942(58) / Gauss-Kruger zone 2 [5443] Pulkovo 1942(58) / Gauss-Kruger zone 2 (E-N) [5444] Pulkovo 1942(58) / Gauss-Kruger zone 3 [5445] Pulkovo 1942(58) / Gauss-Kruger zone 3 (E-N) [5446] Pulkovo 1942(58) / Gauss-Kruger zone 4 [5447] Pulkovo 1942(58) / Gauss-Kruger zone 5 [5448] Pulkovo 1942(58) / Poland zone I [5449] Pulkovo 1942(58) / Poland zone I (deprecated) [5450] Pulkovo 1942(58) / Poland zone II [5451] Pulkovo 1942(58) / Poland zone III [5452] Pulkovo 1942(58) / Poland zone IV [5453] Pulkovo 1942(58) / Poland zone V [5454] Pulkovo 1942(58) / Stereo70 [5455] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 [5456] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 (E-N) [5457] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 [5458] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 (E-N) [5459] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 [5460] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 (E-N) [5461] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 6 [5462] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 [5463] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 (deprecated) [5464] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 [5465] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 (deprecated) [5466] Pulkovo 1942(83) / Gauss Kruger zone 3 (deprecated) [5467] Pulkovo 1942(83) / Gauss Kruger zone 4 (deprecated) [5468] Pulkovo 1942(83) / Gauss Kruger zone 5 (deprecated) [5469] Pulkovo 1942(83) / Gauss-Kruger zone 2 [5470] Pulkovo 1942(83) / Gauss-Kruger zone 2 (E-N) [5471] Pulkovo 1942(83) / Gauss-Kruger zone 3 [5472] Pulkovo 1942(83) / Gauss-Kruger zone 3 (E-N) [5473] Pulkovo 1942(83) / Gauss-Kruger zone 4 [5474] Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E [5475] Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E [5476] Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E [5477] Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E [5478] Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E [5479] Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E [5480] Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E [5481] Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E [5482] Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E [5483] Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E [5484] Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E [5485] Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E [5486] Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E [5487] Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E [5488] Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E [5489] Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E [5490] Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E [5491] Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E [5492] Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E [5493] Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E [5494] Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E [5495] Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E [5496] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E [5497] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W [5498] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E [5499] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W [5500] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E [5501] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W [5502] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E [5503] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W [5504] Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E [5505] Pulkovo 1995 / 3-degree Gauss-Kruger CM 18E (deprecated) [5506] Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E [5507] Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E [5508] Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E [5509] Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E [5510] Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E [5511] Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E [5512] Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E [5513] Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E [5514] Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E [5515] Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E [5516] Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E [5517] Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E [5518] Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E [5519] Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E [5520] Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E [5521] Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E [5522] Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E [5523] Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E [5524] Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E [5525] Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E [5526] Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E [5527] Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E [5528] Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E [5529] Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E [5530] Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E [5531] Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E [5532] Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E [5533] Pulkovo 1995 / 3-degree Gauss-Kruger zone 10 [5534] Pulkovo 1995 / 3-degree Gauss-Kruger zone 11 [5535] Pulkovo 1995 / 3-degree Gauss-Kruger zone 12 [5536] Pulkovo 1995 / 3-degree Gauss-Kruger zone 13 [5537] Pulkovo 1995 / 3-degree Gauss-Kruger zone 14 [5538] Pulkovo 1995 / 3-degree Gauss-Kruger zone 15 [5539] Pulkovo 1995 / 3-degree Gauss-Kruger zone 16 [5540] Pulkovo 1995 / 3-degree Gauss-Kruger zone 17 [5541] Pulkovo 1995 / 3-degree Gauss-Kruger zone 18 [5542] Pulkovo 1995 / 3-degree Gauss-Kruger zone 19 [5543] Pulkovo 1995 / 3-degree Gauss-Kruger zone 20 [5544] Pulkovo 1995 / 3-degree Gauss-Kruger zone 21 [5545] Pulkovo 1995 / 3-degree Gauss-Kruger zone 22 [5546] Pulkovo 1995 / 3-degree Gauss-Kruger zone 23 [5547] Pulkovo 1995 / 3-degree Gauss-Kruger zone 24 [5548] Pulkovo 1995 / 3-degree Gauss-Kruger zone 25 [5549] Pulkovo 1995 / 3-degree Gauss-Kruger zone 26 [5550] Pulkovo 1995 / 3-degree Gauss-Kruger zone 27 [5551] Pulkovo 1995 / 3-degree Gauss-Kruger zone 28 [5552] Pulkovo 1995 / 3-degree Gauss-Kruger zone 29 [5553] Pulkovo 1995 / 3-degree Gauss-Kruger zone 30 [5554] Pulkovo 1995 / 3-degree Gauss-Kruger zone 31 [5555] Pulkovo 1995 / 3-degree Gauss-Kruger zone 32 [5556] Pulkovo 1995 / 3-degree Gauss-Kruger zone 33 [5557] Pulkovo 1995 / 3-degree Gauss-Kruger zone 34 [5558] Pulkovo 1995 / 3-degree Gauss-Kruger zone 35 [5559] Pulkovo 1995 / 3-degree Gauss-Kruger zone 36 [5560] Pulkovo 1995 / 3-degree Gauss-Kruger zone 37 [5561] Pulkovo 1995 / 3-degree Gauss-Kruger zone 38 [5562] Pulkovo 1995 / 3-degree Gauss-Kruger zone 39 [5563] Pulkovo 1995 / 3-degree Gauss-Kruger zone 40 [5564] Pulkovo 1995 / 3-degree Gauss-Kruger zone 41 [5565] Pulkovo 1995 / 3-degree Gauss-Kruger zone 42 [5566] Pulkovo 1995 / 3-degree Gauss-Kruger zone 43 [5567] Pulkovo 1995 / 3-degree Gauss-Kruger zone 44 [5568] Pulkovo 1995 / 3-degree Gauss-Kruger zone 45 [5569] Pulkovo 1995 / 3-degree Gauss-Kruger zone 46 [5570] Pulkovo 1995 / 3-degree Gauss-Kruger zone 47 [5571] Pulkovo 1995 / 3-degree Gauss-Kruger zone 48 [5572] Pulkovo 1995 / 3-degree Gauss-Kruger zone 49 [5573] Pulkovo 1995 / 3-degree Gauss-Kruger zone 50 [5574] Pulkovo 1995 / 3-degree Gauss-Kruger zone 51 [5575] Pulkovo 1995 / 3-degree Gauss-Kruger zone 52 [5576] Pulkovo 1995 / 3-degree Gauss-Kruger zone 53 [5577] Pulkovo 1995 / 3-degree Gauss-Kruger zone 54 [5578] Pulkovo 1995 / 3-degree Gauss-Kruger zone 55 [5579] Pulkovo 1995 / 3-degree Gauss-Kruger zone 56 [5580] Pulkovo 1995 / 3-degree Gauss-Kruger zone 57 [5581] Pulkovo 1995 / 3-degree Gauss-Kruger zone 58 [5582] Pulkovo 1995 / 3-degree Gauss-Kruger zone 59 [5583] Pulkovo 1995 / 3-degree Gauss-Kruger zone 6 (deprecated) [5584] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 [5585] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 (deprecated) [5586] Pulkovo 1995 / 3-degree Gauss-Kruger zone 61 [5587] Pulkovo 1995 / 3-degree Gauss-Kruger zone 62 [5588] Pulkovo 1995 / 3-degree Gauss-Kruger zone 63 [5589] Pulkovo 1995 / 3-degree Gauss-Kruger zone 64 [5590] Pulkovo 1995 / 3-degree Gauss-Kruger zone 7 [5591] Pulkovo 1995 / 3-degree Gauss-Kruger zone 8 [5592] Pulkovo 1995 / 3-degree Gauss-Kruger zone 9 [5593] Pulkovo 1995 / Gauss-Kruger 10N (deprecated) [5594] Pulkovo 1995 / Gauss-Kruger 11N (deprecated) [5595] Pulkovo 1995 / Gauss-Kruger 12N (deprecated) [5596] Pulkovo 1995 / Gauss-Kruger 13N (deprecated) [5597] Pulkovo 1995 / Gauss-Kruger 14N (deprecated) [5598] Pulkovo 1995 / Gauss-Kruger 15N (deprecated) [5599] Pulkovo 1995 / Gauss-Kruger 16N (deprecated) [5600] Pulkovo 1995 / Gauss-Kruger 17N (deprecated) [5601] Pulkovo 1995 / Gauss-Kruger 18N (deprecated) [5602] Pulkovo 1995 / Gauss-Kruger 19N (deprecated) [5603] Pulkovo 1995 / Gauss-Kruger 20N (deprecated) [5604] Pulkovo 1995 / Gauss-Kruger 21N (deprecated) [5605] Pulkovo 1995 / Gauss-Kruger 22N (deprecated) [5606] Pulkovo 1995 / Gauss-Kruger 23N (deprecated) [5607] Pulkovo 1995 / Gauss-Kruger 24N (deprecated) [5608] Pulkovo 1995 / Gauss-Kruger 25N (deprecated) [5609] Pulkovo 1995 / Gauss-Kruger 26N (deprecated) [5610] Pulkovo 1995 / Gauss-Kruger 27N (deprecated) [5611] Pulkovo 1995 / Gauss-Kruger 28N (deprecated) [5612] Pulkovo 1995 / Gauss-Kruger 29N (deprecated) [5613] Pulkovo 1995 / Gauss-Kruger 30N (deprecated) [5614] Pulkovo 1995 / Gauss-Kruger 31N (deprecated) [5615] Pulkovo 1995 / Gauss-Kruger 32N (deprecated) [5616] Pulkovo 1995 / Gauss-Kruger 4N (deprecated) [5617] Pulkovo 1995 / Gauss-Kruger 5N (deprecated) [5618] Pulkovo 1995 / Gauss-Kruger 6N (deprecated) [5619] Pulkovo 1995 / Gauss-Kruger 7N (deprecated) [5620] Pulkovo 1995 / Gauss-Kruger 8N (deprecated) [5621] Pulkovo 1995 / Gauss-Kruger 9N (deprecated) [5622] Pulkovo 1995 / Gauss-Kruger CM 105E [5623] Pulkovo 1995 / Gauss-Kruger CM 111E [5624] Pulkovo 1995 / Gauss-Kruger CM 117E [5625] Pulkovo 1995 / Gauss-Kruger CM 123E [5626] Pulkovo 1995 / Gauss-Kruger CM 129E [5627] Pulkovo 1995 / Gauss-Kruger CM 135E [5628] Pulkovo 1995 / Gauss-Kruger CM 141E [5629] Pulkovo 1995 / Gauss-Kruger CM 147E [5630] Pulkovo 1995 / Gauss-Kruger CM 153E [5631] Pulkovo 1995 / Gauss-Kruger CM 159E [5632] Pulkovo 1995 / Gauss-Kruger CM 165E [5633] Pulkovo 1995 / Gauss-Kruger CM 171E [5634] Pulkovo 1995 / Gauss-Kruger CM 171W [5635] Pulkovo 1995 / Gauss-Kruger CM 177E [5636] Pulkovo 1995 / Gauss-Kruger CM 177W [5637] Pulkovo 1995 / Gauss-Kruger CM 21E [5638] Pulkovo 1995 / Gauss-Kruger CM 27E [5639] Pulkovo 1995 / Gauss-Kruger CM 33E [5640] Pulkovo 1995 / Gauss-Kruger CM 39E [5641] Pulkovo 1995 / Gauss-Kruger CM 45E [5642] Pulkovo 1995 / Gauss-Kruger CM 51E [5643] Pulkovo 1995 / Gauss-Kruger CM 57E [5644] Pulkovo 1995 / Gauss-Kruger CM 63E [5645] Pulkovo 1995 / Gauss-Kruger CM 69E [5646] Pulkovo 1995 / Gauss-Kruger CM 75E [5647] Pulkovo 1995 / Gauss-Kruger CM 81E [5648] Pulkovo 1995 / Gauss-Kruger CM 87E [5649] Pulkovo 1995 / Gauss-Kruger CM 93E [5650] Pulkovo 1995 / Gauss-Kruger CM 99E [5651] Pulkovo 1995 / Gauss-Kruger zone 10 [5652] Pulkovo 1995 / Gauss-Kruger zone 11 [5653] Pulkovo 1995 / Gauss-Kruger zone 12 [5654] Pulkovo 1995 / Gauss-Kruger zone 13 [5655] Pulkovo 1995 / Gauss-Kruger zone 14 [5656] Pulkovo 1995 / Gauss-Kruger zone 15 [5657] Pulkovo 1995 / Gauss-Kruger zone 16 [5658] Pulkovo 1995 / Gauss-Kruger zone 17 [5659] Pulkovo 1995 / Gauss-Kruger zone 18 [5660] Pulkovo 1995 / Gauss-Kruger zone 19 [5661] Pulkovo 1995 / Gauss-Kruger zone 20 [5662] Pulkovo 1995 / Gauss-Kruger zone 21 [5663] Pulkovo 1995 / Gauss-Kruger zone 22 [5664] Pulkovo 1995 / Gauss-Kruger zone 23 [5665] Pulkovo 1995 / Gauss-Kruger zone 24 [5666] Pulkovo 1995 / Gauss-Kruger zone 25 [5667] Pulkovo 1995 / Gauss-Kruger zone 26 [5668] Pulkovo 1995 / Gauss-Kruger zone 27 [5669] Pulkovo 1995 / Gauss-Kruger zone 28 [5670] Pulkovo 1995 / Gauss-Kruger zone 29 [5671] Pulkovo 1995 / Gauss-Kruger zone 30 [5672] Pulkovo 1995 / Gauss-Kruger zone 31 [5673] Pulkovo 1995 / Gauss-Kruger zone 32 [5674] Pulkovo 1995 / Gauss-Kruger zone 4 [5675] Pulkovo 1995 / Gauss-Kruger zone 5 [5676] Pulkovo 1995 / Gauss-Kruger zone 6 [5677] Pulkovo 1995 / Gauss-Kruger zone 7 [5678] Pulkovo 1995 / Gauss-Kruger zone 8 [5679] Pulkovo 1995 / Gauss-Kruger zone 9 [5680] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_7 (deprecated) [5681] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_8 (deprecated) [5682] Pulkovo_1995_GK_Zone_2 [5683] Pulkovo_1995_GK_Zone_2N [5684] Pulkovo_1995_GK_Zone_3 [5685] Pulkovo_1995_GK_Zone_3N [5686] QND95 / Qatar National Grid [5687] QND_1995_UTM_39N [5688] Qatar 1948 / Qatar Grid [5689] Qatar 1974 / Qatar National Grid [5690] Qornoq 1927 / UTM zone 22N [5691] Qornoq 1927 / UTM zone 23N [5692] RD/83 / 3-degree Gauss-Kruger zone 4 [5693] RD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5694] RD/83 / 3-degree Gauss-Kruger zone 5 [5695] RD/83 / 3-degree Gauss-Kruger zone 5 (E-N) [5696] RDN2008 / Italy zone (E-N) [5697] RDN2008 / Italy zone (N-E) [5698] RDN2008 / UTM zone 32N [5699] RDN2008 / UTM zone 32N (N-E) [5700] RDN2008 / UTM zone 33N [5701] RDN2008 / UTM zone 33N (N-E) [5702] RDN2008 / UTM zone 34N [5703] RDN2008 / UTM zone 34N (N-E) [5704] RDN2008 / Zone 12 (E-N) [5705] RDN2008 / Zone 12 (N-E) [5706] REGCAN95 / LAEA Europe [5707] REGCAN95 / LCC Europe [5708] REGCAN95 / UTM zone 27N [5709] REGCAN95 / UTM zone 28N [5710] REGCAN95_UTM_Zone_27N (deprecated) [5711] REGCAN95_UTM_Zone_28N (deprecated) [5712] REGVEN / UTM zone 18N [5713] REGVEN / UTM zone 19N [5714] REGVEN / UTM zone 20N [5715] RGAF09 / UTM zone 20N [5716] RGF93 / CC42 [5717] RGF93 / CC43 [5718] RGF93 / CC44 [5719] RGF93 / CC45 [5720] RGF93 / CC46 [5721] RGF93 / CC47 [5722] RGF93 / CC48 [5723] RGF93 / CC49 [5724] RGF93 / CC50 [5725] RGF93 / Lambert-93 [5726] RGFG95 / UTM zone 21N [5727] RGFG95 / UTM zone 22N [5728] RGF_1993_Lambert_93 (deprecated) [5729] RGM04 / UTM zone 38S [5730] RGNC 1991 / Lambert New Caledonia (deprecated) [5731] RGNC91-93 / Lambert New Caledonia [5732] RGNC91-93 / UTM zone 57S [5733] RGNC91-93 / UTM zone 58S [5734] RGNC91-93 / UTM zone 59S [5735] RGPF / UTM zone 5S [5736] RGPF / UTM zone 6S [5737] RGPF / UTM zone 7S [5738] RGPF / UTM zone 8S [5739] RGR92 / UTM zone 39S [5740] RGR92 / UTM zone 40S [5741] RGRDC 2005 / Congo TM zone 12 [5742] RGRDC 2005 / Congo TM zone 14 [5743] RGRDC 2005 / Congo TM zone 16 [5744] RGRDC 2005 / Congo TM zone 18 [5745] RGRDC 2005 / Congo TM zone 20 [5746] RGRDC 2005 / Congo TM zone 22 [5747] RGRDC 2005 / Congo TM zone 24 [5748] RGRDC 2005 / Congo TM zone 26 [5749] RGRDC 2005 / Congo TM zone 28 [5750] RGRDC 2005 / Congo TM zone 30 [5751] RGRDC 2005 / UTM zone 33S [5752] RGRDC 2005 / UTM zone 34S [5753] RGRDC 2005 / UTM zone 35S [5754] RGRDC_2005_Congo_TM_Zone_12 (deprecated) [5755] RGRDC_2005_Congo_TM_Zone_14 (deprecated) [5756] RGRDC_2005_Congo_TM_Zone_16 (deprecated) [5757] RGRDC_2005_Congo_TM_Zone_18 (deprecated) [5758] RGRDC_2005_Congo_TM_Zone_20 (deprecated) [5759] RGRDC_2005_Congo_TM_Zone_22 (deprecated) [5760] RGRDC_2005_Congo_TM_Zone_24 (deprecated) [5761] RGRDC_2005_Congo_TM_Zone_26 (deprecated) [5762] RGRDC_2005_Congo_TM_Zone_28 (deprecated) [5763] RGRDC_2005_UTM_Zone_33S (deprecated) [5764] RGRDC_2005_UTM_Zone_34S (deprecated) [5765] RGRDC_2005_UTM_Zone_35S (deprecated) [5766] RGSPM06 / UTM zone 21N [5767] RGTAAF07 / UTM zone 37S [5768] RGTAAF07 / UTM zone 38S [5769] RGTAAF07 / UTM zone 39S [5770] RGTAAF07 / UTM zone 40S [5771] RGTAAF07 / UTM zone 41S [5772] RGTAAF07 / UTM zone 42S [5773] RGTAAF07 / UTM zone 43S [5774] RGTAAF07 / UTM zone 44S [5775] RGTAAF07 / UTM zone 53S [5776] RGTAAF07 / UTM zone 54S [5777] RGWF96 / UTM zone 1S [5778] RRAF 1991 / UTM zone 20N [5779] RRAF 1991 / UTM zone 20N (deprecated) [5780] RSRGD2000 / BCLC2000 [5781] RSRGD2000 / DGLC2000 [5782] RSRGD2000 / MSLC2000 [5783] RSRGD2000 / PCLC2000 [5784] RSRGD2000 / RSPS2000 [5785] RT38 0 gon [5786] RT38 2.5 gon O [5787] RT38 2.5 gon V [5788] RT38 2.5 gon W (deprecated) [5789] RT38 5 gon O [5790] RT38 5 gon V [5791] RT38 7.5 gon V [5792] RT90 0 gon [5793] RT90 2.5 gon O [5794] RT90 2.5 gon V [5795] RT90 2.5 gon W (deprecated) [5796] RT90 5 gon O [5797] RT90 5 gon V [5798] RT90 7.5 gon V [5799] Rassadiran / Nakhl e Taqi [5800] Reunion 1947 / TM Reunion [5801] Reunion 1947 / TM Reunion (deprecated) [5802] Roma_1940_Gauss_Boaga_Est [5803] Roma_1940_Gauss_Boaga_Ovest [5804] S-JTSK (Ferro) / Krovak [5805] S-JTSK (Ferro) / Krovak East North [5806] S-JTSK / Krovak [5807] S-JTSK / Krovak East North [5808] S-JTSK [JTSK03] / Krovak [5809] S-JTSK [JTSK03] / Krovak East North [5810] S-JTSK_Ferro_Krovak_East_North (deprecated) [5811] S-JTSK_Krovak (deprecated) [5812] S-JTSK_Krovak_East_North (deprecated) [5813] SAD69 / Brazil Polyconic [5814] SAD69 / Brazil Polyconic (deprecated) [5815] SAD69 / UTM zone 17N [5816] SAD69 / UTM zone 17S [5817] SAD69 / UTM zone 17S (deprecated) [5818] SAD69 / UTM zone 18N [5819] SAD69 / UTM zone 18N (deprecated) [5820] SAD69 / UTM zone 18S [5821] SAD69 / UTM zone 18S (deprecated) [5822] SAD69 / UTM zone 19N [5823] SAD69 / UTM zone 19N (deprecated) [5824] SAD69 / UTM zone 19S [5825] SAD69 / UTM zone 19S (deprecated) [5826] SAD69 / UTM zone 20N [5827] SAD69 / UTM zone 20N (deprecated) [5828] SAD69 / UTM zone 20S [5829] SAD69 / UTM zone 20S (deprecated) [5830] SAD69 / UTM zone 21N [5831] SAD69 / UTM zone 21N (deprecated) [5832] SAD69 / UTM zone 21S [5833] SAD69 / UTM zone 21S (deprecated) [5834] SAD69 / UTM zone 22N [5835] SAD69 / UTM zone 22N (deprecated) [5836] SAD69 / UTM zone 22S [5837] SAD69 / UTM zone 22S (deprecated) [5838] SAD69 / UTM zone 23S [5839] SAD69 / UTM zone 23S (deprecated) [5840] SAD69 / UTM zone 24S [5841] SAD69 / UTM zone 24S (deprecated) [5842] SAD69 / UTM zone 25S [5843] SAD69 / UTM zone 25S (deprecated) [5844] SAD69(96) / Brazil Polyconic [5845] SAD69(96) / UTM zone 18S [5846] SAD69(96) / UTM zone 19S [5847] SAD69(96) / UTM zone 20S [5848] SAD69(96) / UTM zone 21S [5849] SAD69(96) / UTM zone 22S [5850] SAD69(96) / UTM zone 22S (deprecated) [5851] SAD69(96) / UTM zone 23S [5852] SAD69(96) / UTM zone 24S [5853] SAD69(96) / UTM zone 25S [5854] SHMG2015 [5855] SIRGAS 1995 / UTM zone 17N [5856] SIRGAS 1995 / UTM zone 17S [5857] SIRGAS 1995 / UTM zone 18N [5858] SIRGAS 1995 / UTM zone 18S [5859] SIRGAS 1995 / UTM zone 19N [5860] SIRGAS 1995 / UTM zone 19S [5861] SIRGAS 1995 / UTM zone 20N [5862] SIRGAS 1995 / UTM zone 20S [5863] SIRGAS 1995 / UTM zone 21N [5864] SIRGAS 1995 / UTM zone 21S [5865] SIRGAS 1995 / UTM zone 22N [5866] SIRGAS 1995 / UTM zone 22S [5867] SIRGAS 1995 / UTM zone 23S [5868] SIRGAS 1995 / UTM zone 24S [5869] SIRGAS 1995 / UTM zone 25S [5870] SIRGAS 2000 / Brazil Mercator [5871] SIRGAS 2000 / Brazil Polyconic [5872] SIRGAS 2000 / UTM zone 11N [5873] SIRGAS 2000 / UTM zone 12N [5874] SIRGAS 2000 / UTM zone 13N [5875] SIRGAS 2000 / UTM zone 14N [5876] SIRGAS 2000 / UTM zone 15N [5877] SIRGAS 2000 / UTM zone 16N [5878] SIRGAS 2000 / UTM zone 17N [5879] SIRGAS 2000 / UTM zone 17S [5880] SIRGAS 2000 / UTM zone 18N [5881] SIRGAS 2000 / UTM zone 18S [5882] SIRGAS 2000 / UTM zone 19N [5883] SIRGAS 2000 / UTM zone 19S [5884] SIRGAS 2000 / UTM zone 20N [5885] SIRGAS 2000 / UTM zone 20S [5886] SIRGAS 2000 / UTM zone 21N [5887] SIRGAS 2000 / UTM zone 21S [5888] SIRGAS 2000 / UTM zone 22N [5889] SIRGAS 2000 / UTM zone 22S [5890] SIRGAS 2000 / UTM zone 23N [5891] SIRGAS 2000 / UTM zone 23S [5892] SIRGAS 2000 / UTM zone 24N [5893] SIRGAS 2000 / UTM zone 24S [5894] SIRGAS 2000 / UTM zone 25S [5895] SIRGAS 2000 / UTM zone 26S [5896] SIRGAS-Chile / UTM zone 18S [5897] SIRGAS-Chile / UTM zone 19S [5898] SIRGAS-ROU98 / UTM zone 21S [5899] SIRGAS-ROU98 / UTM zone 22S [5900] SIRGAS_UTM_Zone_17N (deprecated) [5901] SIRGAS_UTM_Zone_18N (deprecated) [5902] SIRGAS_UTM_Zone_19N (deprecated) [5903] SIRGAS_UTM_Zone_20N (deprecated) [5904] SIRGAS_UTM_Zone_21N (deprecated) [5905] SIRGAS_UTM_Zone_22N (deprecated) [5906] SLD99 / Sri Lanka Grid 1999 [5907] SRB_ETRS89 / UTM zone 34N [5908] ST71 Belep / UTM zone 58S [5909] ST74 [5910] ST84 Ile des Pins / UTM zone 58S [5911] ST87 Ouvea / UTM zone 58S [5912] ST87 Ouvea / UTM zone 58S (deprecated) [5913] SVY21 / Singapore TM [5914] SWEREF99 / RT90 0 gon emulation [5915] SWEREF99 / RT90 2.5 gon O emulation [5916] SWEREF99 / RT90 2.5 gon V emulation [5917] SWEREF99 / RT90 5 gon O emulation [5918] SWEREF99 / RT90 5 gon V emulation [5919] SWEREF99 / RT90 7.5 gon V emulation [5920] SWEREF99 12 00 [5921] SWEREF99 13 30 [5922] SWEREF99 14 15 [5923] SWEREF99 15 00 [5924] SWEREF99 15 45 [5925] SWEREF99 16 30 [5926] SWEREF99 17 15 [5927] SWEREF99 18 00 [5928] SWEREF99 18 45 [5929] SWEREF99 20 15 [5930] SWEREF99 21 45 [5931] SWEREF99 23 15 [5932] SWEREF99 TM [5933] Sahara_Degree [5934] Saint Pierre et Miquelon 1950 / UTM zone 21N [5935] Saipan_Az_Eq_1969 [5936] Samboja / UTM zone 50S (deprecated) [5937] Sao_Braz_UTM_Zone_26N [5938] Sapper Hill 1943 / UTM zone 20S [5939] Sapper Hill 1943 / UTM zone 21S [5940] Schwarzeck / Lo22/11 [5941] Schwarzeck / Lo22/13 [5942] Schwarzeck / Lo22/15 [5943] Schwarzeck / Lo22/17 [5944] Schwarzeck / Lo22/19 [5945] Schwarzeck / Lo22/21 [5946] Schwarzeck / Lo22/23 [5947] Schwarzeck / Lo22/25 [5948] Schwarzeck / UTM zone 33S [5949] Segara (Jakarta) / NEIEZ [5950] Segara (Jakarta) / NEIEZ (deprecated) [5951] Segara / NEIEZ [5952] Segara / UTM zone 50S [5953] Selvagem Grande / UTM zone 28N [5954] Selvagem_Grande_1938_UTM_Zone_28N (deprecated) [5955] Sibun Gorge 1922 / Colony Grid [5956] Sibun Gorge 1922 / Colony Grid (deprecated) [5957] Sierra Leone 1924 / New Colony Grid [5958] Sierra Leone 1924 / New War Office Grid [5959] Sierra Leone 1968 / UTM zone 28N [5960] Sierra Leone 1968 / UTM zone 29N [5961] Sister Islands National Grid 1961 [5962] Slovenia 1996 / Slovene National Grid [5963] Slovenia 1996 / UTM zone 33N [5964] South East Island 1943 / UTM zone 40N [5965] South Yemen / Gauss Kruger zone 8 (deprecated) [5966] South Yemen / Gauss Kruger zone 9 (deprecated) [5967] South Yemen / Gauss-Kruger zone 8 [5968] South Yemen / Gauss-Kruger zone 9 [5969] South_America_Albers_Equal_Area_Conic [5970] South_America_Equidistant_Conic [5971] South_America_Lambert_Conformal_Conic [5972] South_Pole_Azimuthal_Equidistant [5973] South_Pole_Gnomonic [5974] South_Pole_Lambert_Azimuthal_Equal_Area [5975] South_Pole_Orthographic [5976] South_Pole_Stereographic [5977] Sphere_Aitoff [5978] Sphere_Azimuthal_Equidistant [5979] Sphere_Bonne [5980] Sphere_Cassini [5981] Sphere_Craster_Parabolic [5982] Sphere_Cylindrical_Equal_Area [5983] Sphere_Eckert_I [5984] Sphere_Eckert_II [5985] Sphere_Eckert_III [5986] Sphere_Eckert_IV [5987] Sphere_Eckert_V [5988] Sphere_Eckert_VI [5989] Sphere_Equidistant_Conic [5990] Sphere_Equidistant_Cylindrical [5991] Sphere_Gall_Stereographic [5992] Sphere_Hotine [5993] Sphere_Loximuthal [5994] Sphere_Mercator [5995] Sphere_Miller_Cylindrical [5996] Sphere_Mollweide [5997] Sphere_Plate_Carree [5998] Sphere_Polyconic [5999] Sphere_Quartic_Authalic [6000] Sphere_Robinson [6001] Sphere_Sinusoidal [6002] Sphere_Stereographic [6003] Sphere_Two_Point_Equidistant [6004] Sphere_Van_der_Grinten_I [6005] Sphere_Wagner_IV [6006] Sphere_Wagner_V [6007] Sphere_Wagner_VII [6008] Sphere_Winkel_I [6009] Sphere_Winkel_II [6010] Sphere_Winkel_Tripel_NGS [6011] St. Helena Tritan / SHLG(Tritan) [6012] St. Helena Tritan / UTM zone 30S [6013] St. Kitts 1955 / British West Indies Grid [6014] St. Lucia 1955 / British West Indies Grid [6015] St. Stephen Grid (Ferro) [6016] St. Vincent 45 / British West Indies Grid [6017] Sud_Algerie (deprecated) [6018] Sud_Algerie_Ancienne_Degree [6019] Sud_Algerie_Degree (deprecated) [6020] Sud_Maroc_Degree [6021] Sudan / UTM zone 35N (deprecated) [6022] Sudan / UTM zone 36N (deprecated) [6023] TC(1948) / UTM zone 39N [6024] TC(1948) / UTM zone 40N [6025] TGD2005 / Tonga Map Grid [6026] TM65 / Irish Grid [6027] TM65 / Irish National Grid (deprecated) [6028] TM75 / Irish Grid [6029] TUREF / 3-degree Gauss-Kruger zone 10 [6030] TUREF / 3-degree Gauss-Kruger zone 11 [6031] TUREF / 3-degree Gauss-Kruger zone 12 [6032] TUREF / 3-degree Gauss-Kruger zone 13 [6033] TUREF / 3-degree Gauss-Kruger zone 14 [6034] TUREF / 3-degree Gauss-Kruger zone 15 [6035] TUREF / 3-degree Gauss-Kruger zone 9 [6036] TUREF / LAEA Europe [6037] TUREF / LCC Europe [6038] TUREF / TM27 [6039] TUREF / TM30 [6040] TUREF / TM33 [6041] TUREF / TM36 [6042] TUREF / TM39 [6043] TUREF / TM42 [6044] TUREF / TM45 [6045] TWD67 / TM2 zone 119 [6046] TWD67 / TM2 zone 121 [6047] TWD97 / TM2 zone 119 [6048] TWD97 / TM2 zone 121 [6049] TWD_1967_TM_Penghu (deprecated) [6050] TWD_1967_TM_Taiwan (deprecated) [6051] TWD_1997_TM_Penghu (deprecated) [6052] TWD_1997_TM_Taiwan (deprecated) [6053] Tahaa 54 / UTM zone 5S [6054] Tahiti 52 / UTM zone 6S [6055] Tahiti 79 / UTM zone 6S [6056] Tananarive (Paris) / Laborde Grid [6057] Tananarive (Paris) / Laborde Grid (deprecated) [6058] Tananarive (Paris) / Laborde Grid approximation [6059] Tananarive / Laborde Grid [6060] Tananarive / UTM zone 38S [6061] Tananarive / UTM zone 39S [6062] Tananarive_1925_Laborde_Grid [6063] Tete / UTM zone 36S [6064] Tete / UTM zone 37S [6065] The_World_From_Space [6066] Timbalai 1948 / RSO Borneo (ch) [6067] Timbalai 1948 / RSO Borneo (ftSe) [6068] Timbalai 1948 / RSO Borneo (m) [6069] Timbalai 1948 / UTM zone 49N [6070] Timbalai 1948 / UTM zone 50N [6071] Tokyo / Japan Plane Rectangular CS I [6072] Tokyo / Japan Plane Rectangular CS II [6073] Tokyo / Japan Plane Rectangular CS III [6074] Tokyo / Japan Plane Rectangular CS IV [6075] Tokyo / Japan Plane Rectangular CS IX [6076] Tokyo / Japan Plane Rectangular CS V [6077] Tokyo / Japan Plane Rectangular CS VI [6078] Tokyo / Japan Plane Rectangular CS VII [6079] Tokyo / Japan Plane Rectangular CS VIII [6080] Tokyo / Japan Plane Rectangular CS X [6081] Tokyo / Japan Plane Rectangular CS XI [6082] Tokyo / Japan Plane Rectangular CS XII [6083] Tokyo / Japan Plane Rectangular CS XIII [6084] Tokyo / Japan Plane Rectangular CS XIV [6085] Tokyo / Japan Plane Rectangular CS XIX [6086] Tokyo / Japan Plane Rectangular CS XV [6087] Tokyo / Japan Plane Rectangular CS XVI [6088] Tokyo / Japan Plane Rectangular CS XVII [6089] Tokyo / Japan Plane Rectangular CS XVIII [6090] Tokyo / UTM zone 51N [6091] Tokyo / UTM zone 52N [6092] Tokyo / UTM zone 53N [6093] Tokyo / UTM zone 54N [6094] Tokyo / UTM zone 55N [6095] Tokyo 1892 / Korea Central Belt [6096] Tokyo 1892 / Korea East Belt [6097] Tokyo 1892 / Korea East Sea Belt [6098] Tokyo 1892 / Korea West Belt [6099] Tokyo_UTM_Zone_51N (deprecated) [6100] Tokyo_UTM_Zone_52N (deprecated) [6101] Tokyo_UTM_Zone_53N (deprecated) [6102] Tokyo_UTM_Zone_54N (deprecated) [6103] Tokyo_UTM_Zone_55N (deprecated) [6104] Tokyo_UTM_Zone_56N [6105] Trinidad 1903 / Trinidad Grid [6106] Trinidad 1903 / Trinidad Grid (ftCla) [6107] UCS-2000 / 3-degree Gauss-Kruger CM 21E (deprecated) [6108] UCS-2000 / 3-degree Gauss-Kruger CM 24E (deprecated) [6109] UCS-2000 / 3-degree Gauss-Kruger CM 27E (deprecated) [6110] UCS-2000 / 3-degree Gauss-Kruger CM 30E (deprecated) [6111] UCS-2000 / 3-degree Gauss-Kruger CM 33E (deprecated) [6112] UCS-2000 / 3-degree Gauss-Kruger CM 36E (deprecated) [6113] UCS-2000 / 3-degree Gauss-Kruger CM 39E (deprecated) [6114] UCS-2000 / 3-degree Gauss-Kruger zone 10 (deprecated) [6115] UCS-2000 / 3-degree Gauss-Kruger zone 11 (deprecated) [6116] UCS-2000 / 3-degree Gauss-Kruger zone 12 (deprecated) [6117] UCS-2000 / 3-degree Gauss-Kruger zone 13 (deprecated) [6118] UCS-2000 / 3-degree Gauss-Kruger zone 7 (deprecated) [6119] UCS-2000 / 3-degree Gauss-Kruger zone 8 (deprecated) [6120] UCS-2000 / 3-degree Gauss-Kruger zone 9 (deprecated) [6121] UCS-2000 / Gauss-Kruger CM 21E [6122] UCS-2000 / Gauss-Kruger CM 27E [6123] UCS-2000 / Gauss-Kruger CM 33E [6124] UCS-2000 / Gauss-Kruger CM 39E [6125] UCS-2000 / Gauss-Kruger zone 4 [6126] UCS-2000 / Gauss-Kruger zone 5 [6127] UCS-2000 / Gauss-Kruger zone 6 [6128] UCS-2000 / Gauss-Kruger zone 7 [6129] UCS-2000 / Ukraine TM zone 10 [6130] UCS-2000 / Ukraine TM zone 11 [6131] UCS-2000 / Ukraine TM zone 12 [6132] UCS-2000 / Ukraine TM zone 13 [6133] UCS-2000 / Ukraine TM zone 7 [6134] UCS-2000 / Ukraine TM zone 8 [6135] UCS-2000 / Ukraine TM zone 9 [6136] US National Atlas Equal Area [6137] USA_Contiguous_Albers_Equal_Area_Conic [6138] USA_Contiguous_Albers_Equal_Area_Conic_USGS_version [6139] USA_Contiguous_Equidistant_Conic [6140] USA_Contiguous_Lambert_Conformal_Conic [6141] UWPP_1992 [6142] UWPP_2000_PAS_5 [6143] UWPP_2000_PAS_6 [6144] UWPP_2000_PAS_7 [6145] UWPP_2000_PAS_8 [6146] VN-2000 / TM-3 Da Nang zone [6147] VN-2000 / TM-3 Da Nang zone [6148] VN-2000 / TM-3 zone 481 [6149] VN-2000 / TM-3 zone 481 [6150] VN-2000 / TM-3 zone 482 [6151] VN-2000 / TM-3 zone 482 [6152] VN-2000 / TM-3 zone 491 [6153] VN-2000 / TM-3 zone 491 [6154] VN-2000 / UTM zone 48N [6155] VN-2000 / UTM zone 49N [6156] Viti Levu 1912 / Viti Levu Grid [6157] Voirol 1875 / Nord Algerie (ancienne) [6158] Voirol 1875 / Sud Algerie (ancienne) [6159] Voirol 1879 / Nord Algerie (ancienne) [6160] Voirol 1879 / Sud Algerie (ancienne) [6161] WGS 72 / UTM zone 10N [6162] WGS 72 / UTM zone 10S [6163] WGS 72 / UTM zone 11N [6164] WGS 72 / UTM zone 11S [6165] WGS 72 / UTM zone 12N [6166] WGS 72 / UTM zone 12S [6167] WGS 72 / UTM zone 13N [6168] WGS 72 / UTM zone 13S [6169] WGS 72 / UTM zone 14N [6170] WGS 72 / UTM zone 14S [6171] WGS 72 / UTM zone 15N [6172] WGS 72 / UTM zone 15S [6173] WGS 72 / UTM zone 16N [6174] WGS 72 / UTM zone 16S [6175] WGS 72 / UTM zone 17N [6176] WGS 72 / UTM zone 17S [6177] WGS 72 / UTM zone 18N [6178] WGS 72 / UTM zone 18S [6179] WGS 72 / UTM zone 19N [6180] WGS 72 / UTM zone 19S [6181] WGS 72 / UTM zone 1N [6182] WGS 72 / UTM zone 1S [6183] WGS 72 / UTM zone 20N [6184] WGS 72 / UTM zone 20S [6185] WGS 72 / UTM zone 21N [6186] WGS 72 / UTM zone 21S [6187] WGS 72 / UTM zone 22N [6188] WGS 72 / UTM zone 22S [6189] WGS 72 / UTM zone 23N [6190] WGS 72 / UTM zone 23S [6191] WGS 72 / UTM zone 24N [6192] WGS 72 / UTM zone 24S [6193] WGS 72 / UTM zone 25N [6194] WGS 72 / UTM zone 25S [6195] WGS 72 / UTM zone 26N [6196] WGS 72 / UTM zone 26S [6197] WGS 72 / UTM zone 27N [6198] WGS 72 / UTM zone 27S [6199] WGS 72 / UTM zone 28N [6200] WGS 72 / UTM zone 28S [6201] WGS 72 / UTM zone 29N [6202] WGS 72 / UTM zone 29S [6203] WGS 72 / UTM zone 2N [6204] WGS 72 / UTM zone 2S [6205] WGS 72 / UTM zone 30N [6206] WGS 72 / UTM zone 30S [6207] WGS 72 / UTM zone 31N [6208] WGS 72 / UTM zone 31S [6209] WGS 72 / UTM zone 32N [6210] WGS 72 / UTM zone 32S [6211] WGS 72 / UTM zone 33N [6212] WGS 72 / UTM zone 33S [6213] WGS 72 / UTM zone 34N [6214] WGS 72 / UTM zone 34S [6215] WGS 72 / UTM zone 35N [6216] WGS 72 / UTM zone 35S [6217] WGS 72 / UTM zone 36N [6218] WGS 72 / UTM zone 36S [6219] WGS 72 / UTM zone 37N [6220] WGS 72 / UTM zone 37S [6221] WGS 72 / UTM zone 38N [6222] WGS 72 / UTM zone 38S [6223] WGS 72 / UTM zone 39N [6224] WGS 72 / UTM zone 39S [6225] WGS 72 / UTM zone 3N [6226] WGS 72 / UTM zone 3S [6227] WGS 72 / UTM zone 40N [6228] WGS 72 / UTM zone 40S [6229] WGS 72 / UTM zone 41N [6230] WGS 72 / UTM zone 41S [6231] WGS 72 / UTM zone 42N [6232] WGS 72 / UTM zone 42S [6233] WGS 72 / UTM zone 43N [6234] WGS 72 / UTM zone 43S [6235] WGS 72 / UTM zone 44N [6236] WGS 72 / UTM zone 44S [6237] WGS 72 / UTM zone 45N [6238] WGS 72 / UTM zone 45S [6239] WGS 72 / UTM zone 46N [6240] WGS 72 / UTM zone 46S [6241] WGS 72 / UTM zone 47N [6242] WGS 72 / UTM zone 47S [6243] WGS 72 / UTM zone 48N [6244] WGS 72 / UTM zone 48S [6245] WGS 72 / UTM zone 49N [6246] WGS 72 / UTM zone 49S [6247] WGS 72 / UTM zone 4N [6248] WGS 72 / UTM zone 4S [6249] WGS 72 / UTM zone 50N [6250] WGS 72 / UTM zone 50S [6251] WGS 72 / UTM zone 51N [6252] WGS 72 / UTM zone 51S [6253] WGS 72 / UTM zone 52N [6254] WGS 72 / UTM zone 52S [6255] WGS 72 / UTM zone 53N [6256] WGS 72 / UTM zone 53S [6257] WGS 72 / UTM zone 54N [6258] WGS 72 / UTM zone 54S [6259] WGS 72 / UTM zone 55N [6260] WGS 72 / UTM zone 55S [6261] WGS 72 / UTM zone 56N [6262] WGS 72 / UTM zone 56S [6263] WGS 72 / UTM zone 57N [6264] WGS 72 / UTM zone 57S [6265] WGS 72 / UTM zone 58N [6266] WGS 72 / UTM zone 58S [6267] WGS 72 / UTM zone 59N [6268] WGS 72 / UTM zone 59S [6269] WGS 72 / UTM zone 5N [6270] WGS 72 / UTM zone 5S [6271] WGS 72 / UTM zone 60N [6272] WGS 72 / UTM zone 60S [6273] WGS 72 / UTM zone 6N [6274] WGS 72 / UTM zone 6S [6275] WGS 72 / UTM zone 7N [6276] WGS 72 / UTM zone 7S [6277] WGS 72 / UTM zone 8N [6278] WGS 72 / UTM zone 8S [6279] WGS 72 / UTM zone 9N [6280] WGS 72 / UTM zone 9S [6281] WGS 72BE / South China Sea Lambert [6282] WGS 72BE / TM 106 NE [6283] WGS 72BE / UTM zone 10N [6284] WGS 72BE / UTM zone 10S [6285] WGS 72BE / UTM zone 11N [6286] WGS 72BE / UTM zone 11S [6287] WGS 72BE / UTM zone 12N [6288] WGS 72BE / UTM zone 12S [6289] WGS 72BE / UTM zone 13N [6290] WGS 72BE / UTM zone 13S [6291] WGS 72BE / UTM zone 14N [6292] WGS 72BE / UTM zone 14S [6293] WGS 72BE / UTM zone 15N [6294] WGS 72BE / UTM zone 15S [6295] WGS 72BE / UTM zone 16N [6296] WGS 72BE / UTM zone 16S [6297] WGS 72BE / UTM zone 17N [6298] WGS 72BE / UTM zone 17S [6299] WGS 72BE / UTM zone 18N [6300] WGS 72BE / UTM zone 18S [6301] WGS 72BE / UTM zone 19N [6302] WGS 72BE / UTM zone 19S [6303] WGS 72BE / UTM zone 1N [6304] WGS 72BE / UTM zone 1S [6305] WGS 72BE / UTM zone 20N [6306] WGS 72BE / UTM zone 20S [6307] WGS 72BE / UTM zone 21N [6308] WGS 72BE / UTM zone 21S [6309] WGS 72BE / UTM zone 22N [6310] WGS 72BE / UTM zone 22S [6311] WGS 72BE / UTM zone 23N [6312] WGS 72BE / UTM zone 23S [6313] WGS 72BE / UTM zone 24N [6314] WGS 72BE / UTM zone 24S [6315] WGS 72BE / UTM zone 25N [6316] WGS 72BE / UTM zone 25S [6317] WGS 72BE / UTM zone 26N [6318] WGS 72BE / UTM zone 26S [6319] WGS 72BE / UTM zone 27N [6320] WGS 72BE / UTM zone 27S [6321] WGS 72BE / UTM zone 28N [6322] WGS 72BE / UTM zone 28S [6323] WGS 72BE / UTM zone 29N [6324] WGS 72BE / UTM zone 29S [6325] WGS 72BE / UTM zone 2N [6326] WGS 72BE / UTM zone 2S [6327] WGS 72BE / UTM zone 30N [6328] WGS 72BE / UTM zone 30S [6329] WGS 72BE / UTM zone 31N [6330] WGS 72BE / UTM zone 31S [6331] WGS 72BE / UTM zone 32N [6332] WGS 72BE / UTM zone 32S [6333] WGS 72BE / UTM zone 33N [6334] WGS 72BE / UTM zone 33S [6335] WGS 72BE / UTM zone 34N [6336] WGS 72BE / UTM zone 34S [6337] WGS 72BE / UTM zone 35N [6338] WGS 72BE / UTM zone 35S [6339] WGS 72BE / UTM zone 36N [6340] WGS 72BE / UTM zone 36S [6341] WGS 72BE / UTM zone 37N [6342] WGS 72BE / UTM zone 37S [6343] WGS 72BE / UTM zone 38N [6344] WGS 72BE / UTM zone 38S [6345] WGS 72BE / UTM zone 39N [6346] WGS 72BE / UTM zone 39S [6347] WGS 72BE / UTM zone 3N [6348] WGS 72BE / UTM zone 3S [6349] WGS 72BE / UTM zone 40N [6350] WGS 72BE / UTM zone 40S [6351] WGS 72BE / UTM zone 41N [6352] WGS 72BE / UTM zone 41S [6353] WGS 72BE / UTM zone 42N [6354] WGS 72BE / UTM zone 42S [6355] WGS 72BE / UTM zone 43N [6356] WGS 72BE / UTM zone 43S [6357] WGS 72BE / UTM zone 44N [6358] WGS 72BE / UTM zone 44S [6359] WGS 72BE / UTM zone 45N [6360] WGS 72BE / UTM zone 45S [6361] WGS 72BE / UTM zone 46N [6362] WGS 72BE / UTM zone 46S [6363] WGS 72BE / UTM zone 47N [6364] WGS 72BE / UTM zone 47S [6365] WGS 72BE / UTM zone 48N [6366] WGS 72BE / UTM zone 48S [6367] WGS 72BE / UTM zone 49N [6368] WGS 72BE / UTM zone 49S [6369] WGS 72BE / UTM zone 4N [6370] WGS 72BE / UTM zone 4S [6371] WGS 72BE / UTM zone 50N [6372] WGS 72BE / UTM zone 50S [6373] WGS 72BE / UTM zone 51N [6374] WGS 72BE / UTM zone 51S [6375] WGS 72BE / UTM zone 52N [6376] WGS 72BE / UTM zone 52S [6377] WGS 72BE / UTM zone 53N [6378] WGS 72BE / UTM zone 53S [6379] WGS 72BE / UTM zone 54N [6380] WGS 72BE / UTM zone 54S [6381] WGS 72BE / UTM zone 55N [6382] WGS 72BE / UTM zone 55S [6383] WGS 72BE / UTM zone 56N [6384] WGS 72BE / UTM zone 56S [6385] WGS 72BE / UTM zone 57N [6386] WGS 72BE / UTM zone 57S [6387] WGS 72BE / UTM zone 58N [6388] WGS 72BE / UTM zone 58S [6389] WGS 72BE / UTM zone 59N [6390] WGS 72BE / UTM zone 59S [6391] WGS 72BE / UTM zone 5N [6392] WGS 72BE / UTM zone 5S [6393] WGS 72BE / UTM zone 60N [6394] WGS 72BE / UTM zone 60S [6395] WGS 72BE / UTM zone 6N [6396] WGS 72BE / UTM zone 6S [6397] WGS 72BE / UTM zone 7N [6398] WGS 72BE / UTM zone 7S [6399] WGS 72BE / UTM zone 8N [6400] WGS 72BE / UTM zone 8S [6401] WGS 72BE / UTM zone 9N [6402] WGS 72BE / UTM zone 9S [6403] WGS 84 / Andaman and Nicobar [6404] WGS 84 / Andhra Pradesh [6405] WGS 84 / Antarctic Polar Stereographic [6406] WGS 84 / Arctic Polar Stereographic [6407] WGS 84 / Arunachal Pradesh [6408] WGS 84 / Assam [6409] WGS 84 / Australian Antarctic Lambert [6410] WGS 84 / Australian Antarctic Polar Stereographic [6411] WGS 84 / Australian Centre for Remote Sensing Lambert [6412] WGS 84 / BLM 14N (ftUS) [6413] WGS 84 / BLM 15N (ftUS) [6414] WGS 84 / BLM 16N (ftUS) [6415] WGS 84 / BLM 17N (ftUS) [6416] WGS 84 / Bihar [6417] WGS 84 / CIG92 [6418] WGS 84 / CKIG92 [6419] WGS 84 / Cape Verde National [6420] WGS 84 / Chhattisgarh [6421] WGS 84 / Delhi [6422] WGS 84 / Dubai Local TM [6423] WGS 84 / EPSG Alaska Polar Stereographic [6424] WGS 84 / EPSG Arctic Regional zone A1 [6425] WGS 84 / EPSG Arctic Regional zone A2 [6426] WGS 84 / EPSG Arctic Regional zone A3 [6427] WGS 84 / EPSG Arctic Regional zone A4 [6428] WGS 84 / EPSG Arctic Regional zone A5 [6429] WGS 84 / EPSG Arctic Regional zone B1 [6430] WGS 84 / EPSG Arctic Regional zone B2 [6431] WGS 84 / EPSG Arctic Regional zone B3 [6432] WGS 84 / EPSG Arctic Regional zone B4 [6433] WGS 84 / EPSG Arctic Regional zone B5 [6434] WGS 84 / EPSG Arctic Regional zone C1 [6435] WGS 84 / EPSG Arctic Regional zone C2 [6436] WGS 84 / EPSG Arctic Regional zone C3 [6437] WGS 84 / EPSG Arctic Regional zone C4 [6438] WGS 84 / EPSG Arctic Regional zone C5 [6439] WGS 84 / EPSG Arctic zone 1-21 [6440] WGS 84 / EPSG Arctic zone 1-27 [6441] WGS 84 / EPSG Arctic zone 1-29 [6442] WGS 84 / EPSG Arctic zone 1-31 [6443] WGS 84 / EPSG Arctic zone 2-10 [6444] WGS 84 / EPSG Arctic zone 2-12 [6445] WGS 84 / EPSG Arctic zone 2-24 [6446] WGS 84 / EPSG Arctic zone 2-26 [6447] WGS 84 / EPSG Arctic zone 2-28 [6448] WGS 84 / EPSG Arctic zone 3-13 [6449] WGS 84 / EPSG Arctic zone 3-15 [6450] WGS 84 / EPSG Arctic zone 3-17 [6451] WGS 84 / EPSG Arctic zone 3-19 [6452] WGS 84 / EPSG Arctic zone 3-21 [6453] WGS 84 / EPSG Arctic zone 3-23 [6454] WGS 84 / EPSG Arctic zone 4-12 [6455] WGS 84 / EPSG Arctic zone 4-30 [6456] WGS 84 / EPSG Arctic zone 4-32 [6457] WGS 84 / EPSG Arctic zone 4-34 [6458] WGS 84 / EPSG Arctic zone 4-36 [6459] WGS 84 / EPSG Arctic zone 4-38 [6460] WGS 84 / EPSG Arctic zone 4-40 [6461] WGS 84 / EPSG Arctic zone 5-15 [6462] WGS 84 / EPSG Arctic zone 5-17 [6463] WGS 84 / EPSG Arctic zone 5-19 [6464] WGS 84 / EPSG Arctic zone 5-21 [6465] WGS 84 / EPSG Arctic zone 5-23 [6466] WGS 84 / EPSG Arctic zone 5-25 [6467] WGS 84 / EPSG Arctic zone 5-27 [6468] WGS 84 / EPSG Canada Polar Stereographic [6469] WGS 84 / EPSG Greenland Polar Stereographic [6470] WGS 84 / EPSG Norway Polar Stereographic [6471] WGS 84 / EPSG Russia Polar Stereographic [6472] WGS 84 / Gabon TM [6473] WGS 84 / Gabon TM 2011 [6474] WGS 84 / Goa [6475] WGS 84 / Gujarat [6476] WGS 84 / Haryana [6477] WGS 84 / Himachal Pradesh [6478] WGS 84 / IBCAO Polar Stereographic [6479] WGS 84 / India NSF LCC [6480] WGS 84 / India Northeast [6481] WGS 84 / Jammu and Kashmir [6482] WGS 84 / Jharkhand [6483] WGS 84 / Karnataka [6484] WGS 84 / Kerala [6485] WGS 84 / Lakshadweep [6486] WGS 84 / Madhya Pradesh [6487] WGS 84 / Maharashtra [6488] WGS 84 / Manipur [6489] WGS 84 / Meghalaya [6490] WGS 84 / Mercator 41 [6491] WGS 84 / Mercator 41 (deprecated) [6492] WGS 84 / Mizoram [6493] WGS 84 / NSIDC EASE-Grid 2.0 Global [6494] WGS 84 / NSIDC EASE-Grid 2.0 North [6495] WGS 84 / NSIDC EASE-Grid 2.0 South [6496] WGS 84 / NSIDC EASE-Grid Global (deprecated) [6497] WGS 84 / NSIDC EASE-Grid North (deprecated) [6498] WGS 84 / NSIDC EASE-Grid South (deprecated) [6499] WGS 84 / NSIDC Sea Ice Polar Stereographic North [6500] WGS 84 / NSIDC Sea Ice Polar Stereographic South [6501] WGS 84 / Nagaland [6502] WGS 84 / North Pole LAEA Alaska [6503] WGS 84 / North Pole LAEA Atlantic [6504] WGS 84 / North Pole LAEA Bering Sea [6505] WGS 84 / North Pole LAEA Canada [6506] WGS 84 / North Pole LAEA Europe [6507] WGS 84 / North Pole LAEA Russia [6508] WGS 84 / Orissa [6509] WGS 84 / PDC Mercator [6510] WGS 84 / PDC Mercator (deprecated) [6511] WGS 84 / Plate Carree (deprecated) [6512] WGS 84 / Pseudo-Mercator [6513] WGS 84 / Punjab [6514] WGS 84 / Rajasthan [6515] WGS 84 / SCAR IMW SP19-20 [6516] WGS 84 / SCAR IMW SP21-22 [6517] WGS 84 / SCAR IMW SP23-24 [6518] WGS 84 / SCAR IMW SQ01-02 [6519] WGS 84 / SCAR IMW SQ19-20 [6520] WGS 84 / SCAR IMW SQ21-22 [6521] WGS 84 / SCAR IMW SQ37-38 [6522] WGS 84 / SCAR IMW SQ39-40 [6523] WGS 84 / SCAR IMW SQ41-42 [6524] WGS 84 / SCAR IMW SQ43-44 [6525] WGS 84 / SCAR IMW SQ45-46 [6526] WGS 84 / SCAR IMW SQ47-48 [6527] WGS 84 / SCAR IMW SQ49-50 [6528] WGS 84 / SCAR IMW SQ51-52 [6529] WGS 84 / SCAR IMW SQ53-54 [6530] WGS 84 / SCAR IMW SQ55-56 [6531] WGS 84 / SCAR IMW SQ57-58 [6532] WGS 84 / SCAR IMW SR13-14 [6533] WGS 84 / SCAR IMW SR15-16 [6534] WGS 84 / SCAR IMW SR17-18 [6535] WGS 84 / SCAR IMW SR19-20 [6536] WGS 84 / SCAR IMW SR27-28 [6537] WGS 84 / SCAR IMW SR29-30 [6538] WGS 84 / SCAR IMW SR31-32 [6539] WGS 84 / SCAR IMW SR33-34 [6540] WGS 84 / SCAR IMW SR35-36 [6541] WGS 84 / SCAR IMW SR37-38 [6542] WGS 84 / SCAR IMW SR39-40 [6543] WGS 84 / SCAR IMW SR41-42 [6544] WGS 84 / SCAR IMW SR43-44 [6545] WGS 84 / SCAR IMW SR45-46 [6546] WGS 84 / SCAR IMW SR47-48 [6547] WGS 84 / SCAR IMW SR49-50 [6548] WGS 84 / SCAR IMW SR51-52 [6549] WGS 84 / SCAR IMW SR53-54 [6550] WGS 84 / SCAR IMW SR55-56 [6551] WGS 84 / SCAR IMW SR57-58 [6552] WGS 84 / SCAR IMW SR59-60 [6553] WGS 84 / SCAR IMW SS04-06 [6554] WGS 84 / SCAR IMW SS07-09 [6555] WGS 84 / SCAR IMW SS10-12 [6556] WGS 84 / SCAR IMW SS13-15 [6557] WGS 84 / SCAR IMW SS16-18 [6558] WGS 84 / SCAR IMW SS19-21 [6559] WGS 84 / SCAR IMW SS25-27 [6560] WGS 84 / SCAR IMW SS28-30 [6561] WGS 84 / SCAR IMW SS31-33 [6562] WGS 84 / SCAR IMW SS34-36 [6563] WGS 84 / SCAR IMW SS37-39 [6564] WGS 84 / SCAR IMW SS40-42 [6565] WGS 84 / SCAR IMW SS43-45 [6566] WGS 84 / SCAR IMW SS46-48 [6567] WGS 84 / SCAR IMW SS49-51 [6568] WGS 84 / SCAR IMW SS52-54 [6569] WGS 84 / SCAR IMW SS55-57 [6570] WGS 84 / SCAR IMW SS58-60 [6571] WGS 84 / SCAR IMW ST01-04 [6572] WGS 84 / SCAR IMW ST05-08 [6573] WGS 84 / SCAR IMW ST09-12 [6574] WGS 84 / SCAR IMW ST13-16 [6575] WGS 84 / SCAR IMW ST17-20 [6576] WGS 84 / SCAR IMW ST21-24 [6577] WGS 84 / SCAR IMW ST25-28 [6578] WGS 84 / SCAR IMW ST29-32 [6579] WGS 84 / SCAR IMW ST33-36 [6580] WGS 84 / SCAR IMW ST37-40 [6581] WGS 84 / SCAR IMW ST41-44 [6582] WGS 84 / SCAR IMW ST45-48 [6583] WGS 84 / SCAR IMW ST49-52 [6584] WGS 84 / SCAR IMW ST53-56 [6585] WGS 84 / SCAR IMW ST57-60 [6586] WGS 84 / SCAR IMW SU01-05 [6587] WGS 84 / SCAR IMW SU06-10 [6588] WGS 84 / SCAR IMW SU11-15 [6589] WGS 84 / SCAR IMW SU16-20 [6590] WGS 84 / SCAR IMW SU21-25 [6591] WGS 84 / SCAR IMW SU26-30 [6592] WGS 84 / SCAR IMW SU31-35 [6593] WGS 84 / SCAR IMW SU36-40 [6594] WGS 84 / SCAR IMW SU41-45 [6595] WGS 84 / SCAR IMW SU46-50 [6596] WGS 84 / SCAR IMW SU51-55 [6597] WGS 84 / SCAR IMW SU56-60 [6598] WGS 84 / SCAR IMW SV01-10 [6599] WGS 84 / SCAR IMW SV11-20 [6600] WGS 84 / SCAR IMW SV21-30 [6601] WGS 84 / SCAR IMW SV31-40 [6602] WGS 84 / SCAR IMW SV41-50 [6603] WGS 84 / SCAR IMW SV51-60 [6604] WGS 84 / SCAR IMW SW01-60 [6605] WGS 84 / Sikkim [6606] WGS 84 / South Georgia Lambert [6607] WGS 84 / TM 116 SE [6608] WGS 84 / TM 12 SE [6609] WGS 84 / TM 132 SE [6610] WGS 84 / TM 36 SE [6611] WGS 84 / TM 6 NE [6612] WGS 84 / TM 60 SW [6613] WGS 84 / TM Zone 20N (ftUS) [6614] WGS 84 / TM Zone 21N (ftUS) [6615] WGS 84 / TMzn35N [6616] WGS 84 / TMzn36N [6617] WGS 84 / Tamil Nadu [6618] WGS 84 / Tripura [6619] WGS 84 / UPS North (E,N) [6620] WGS 84 / UPS North (N,E) [6621] WGS 84 / UPS South (E,N) [6622] WGS 84 / UPS South (N,E) [6623] WGS 84 / USGS Transantarctic Mountains [6624] WGS 84 / UTM zone 10N [6625] WGS 84 / UTM zone 10S [6626] WGS 84 / UTM zone 11N [6627] WGS 84 / UTM zone 11S [6628] WGS 84 / UTM zone 12N [6629] WGS 84 / UTM zone 12S [6630] WGS 84 / UTM zone 13N [6631] WGS 84 / UTM zone 13S [6632] WGS 84 / UTM zone 14N [6633] WGS 84 / UTM zone 14S [6634] WGS 84 / UTM zone 15N [6635] WGS 84 / UTM zone 15S [6636] WGS 84 / UTM zone 16N [6637] WGS 84 / UTM zone 16S [6638] WGS 84 / UTM zone 17N [6639] WGS 84 / UTM zone 17S [6640] WGS 84 / UTM zone 18N [6641] WGS 84 / UTM zone 18S [6642] WGS 84 / UTM zone 19N [6643] WGS 84 / UTM zone 19S [6644] WGS 84 / UTM zone 1N [6645] WGS 84 / UTM zone 1S [6646] WGS 84 / UTM zone 20N [6647] WGS 84 / UTM zone 20S [6648] WGS 84 / UTM zone 21N [6649] WGS 84 / UTM zone 21S [6650] WGS 84 / UTM zone 22N [6651] WGS 84 / UTM zone 22S [6652] WGS 84 / UTM zone 23N [6653] WGS 84 / UTM zone 23S [6654] WGS 84 / UTM zone 24N [6655] WGS 84 / UTM zone 24S [6656] WGS 84 / UTM zone 25N [6657] WGS 84 / UTM zone 25S [6658] WGS 84 / UTM zone 26N [6659] WGS 84 / UTM zone 26S [6660] WGS 84 / UTM zone 27N [6661] WGS 84 / UTM zone 27S [6662] WGS 84 / UTM zone 28N [6663] WGS 84 / UTM zone 28S [6664] WGS 84 / UTM zone 29N [6665] WGS 84 / UTM zone 29S [6666] WGS 84 / UTM zone 2N [6667] WGS 84 / UTM zone 2S [6668] WGS 84 / UTM zone 30N [6669] WGS 84 / UTM zone 30S [6670] WGS 84 / UTM zone 31N [6671] WGS 84 / UTM zone 31S [6672] WGS 84 / UTM zone 32N [6673] WGS 84 / UTM zone 32S [6674] WGS 84 / UTM zone 33N [6675] WGS 84 / UTM zone 33S [6676] WGS 84 / UTM zone 34N [6677] WGS 84 / UTM zone 34S [6678] WGS 84 / UTM zone 35N [6679] WGS 84 / UTM zone 35S [6680] WGS 84 / UTM zone 36N [6681] WGS 84 / UTM zone 36S [6682] WGS 84 / UTM zone 37N [6683] WGS 84 / UTM zone 37S [6684] WGS 84 / UTM zone 38N [6685] WGS 84 / UTM zone 38S [6686] WGS 84 / UTM zone 39N [6687] WGS 84 / UTM zone 39S [6688] WGS 84 / UTM zone 3N [6689] WGS 84 / UTM zone 3S [6690] WGS 84 / UTM zone 40N [6691] WGS 84 / UTM zone 40S [6692] WGS 84 / UTM zone 41N [6693] WGS 84 / UTM zone 41S [6694] WGS 84 / UTM zone 42N [6695] WGS 84 / UTM zone 42S [6696] WGS 84 / UTM zone 43N [6697] WGS 84 / UTM zone 43S [6698] WGS 84 / UTM zone 44N [6699] WGS 84 / UTM zone 44S [6700] WGS 84 / UTM zone 45N [6701] WGS 84 / UTM zone 45S [6702] WGS 84 / UTM zone 46N [6703] WGS 84 / UTM zone 46S [6704] WGS 84 / UTM zone 47N [6705] WGS 84 / UTM zone 47S [6706] WGS 84 / UTM zone 48N [6707] WGS 84 / UTM zone 48S [6708] WGS 84 / UTM zone 49N [6709] WGS 84 / UTM zone 49S [6710] WGS 84 / UTM zone 4N [6711] WGS 84 / UTM zone 4S [6712] WGS 84 / UTM zone 50N [6713] WGS 84 / UTM zone 50S [6714] WGS 84 / UTM zone 51N [6715] WGS 84 / UTM zone 51S [6716] WGS 84 / UTM zone 52N [6717] WGS 84 / UTM zone 52S [6718] WGS 84 / UTM zone 53N [6719] WGS 84 / UTM zone 53S [6720] WGS 84 / UTM zone 54N [6721] WGS 84 / UTM zone 54S [6722] WGS 84 / UTM zone 55N [6723] WGS 84 / UTM zone 55S [6724] WGS 84 / UTM zone 56N [6725] WGS 84 / UTM zone 56S [6726] WGS 84 / UTM zone 57N [6727] WGS 84 / UTM zone 57S [6728] WGS 84 / UTM zone 58N [6729] WGS 84 / UTM zone 58S [6730] WGS 84 / UTM zone 59N [6731] WGS 84 / UTM zone 59S [6732] WGS 84 / UTM zone 5N [6733] WGS 84 / UTM zone 5S [6734] WGS 84 / UTM zone 60N [6735] WGS 84 / UTM zone 60S [6736] WGS 84 / UTM zone 6N [6737] WGS 84 / UTM zone 6S [6738] WGS 84 / UTM zone 7N [6739] WGS 84 / UTM zone 7S [6740] WGS 84 / UTM zone 8N [6741] WGS 84 / UTM zone 8S [6742] WGS 84 / UTM zone 9N [6743] WGS 84 / UTM zone 9S [6744] WGS 84 / Uttar Pradesh [6745] WGS 84 / Uttaranchal [6746] WGS 84 / West Bengal [6747] WGS 84 / World Equidistant Cylindrical [6748] WGS 84 / World Equidistant Cylindrical (deprecated) [6749] WGS 84 / World Mercator [6750] WGS_1984_ARC_System_Zone_01 [6751] WGS_1984_ARC_System_Zone_02 [6752] WGS_1984_ARC_System_Zone_03 [6753] WGS_1984_ARC_System_Zone_04 [6754] WGS_1984_ARC_System_Zone_05 [6755] WGS_1984_ARC_System_Zone_06 [6756] WGS_1984_ARC_System_Zone_07 [6757] WGS_1984_ARC_System_Zone_08 [6758] WGS_1984_ARC_System_Zone_09 [6759] WGS_1984_ARC_System_Zone_10 [6760] WGS_1984_ARC_System_Zone_11 [6761] WGS_1984_ARC_System_Zone_12 [6762] WGS_1984_ARC_System_Zone_13 [6763] WGS_1984_ARC_System_Zone_14 [6764] WGS_1984_ARC_System_Zone_15 [6765] WGS_1984_ARC_System_Zone_16 [6766] WGS_1984_ARC_System_Zone_17 [6767] WGS_1984_ARC_System_Zone_18 [6768] WGS_1984_Australian_Centre_for_Remote_Sensing_Lambert (deprecated) [6769] WGS_1984_California_Teale_Albers_FtUS [6770] WGS_1984_Canada_Atlas_LCC [6771] WGS_1984_Cape_Verde_Grid (deprecated) [6772] WGS_1984_Costa_Rica_TM_90 [6773] WGS_1984_Web_Mercator (deprecated) [6774] WGS_1984_Web_Mercator_Auxiliary_Sphere (deprecated) [6775] World Equidistant Cylindrical (Sphere) [6776] World Equidistant Cylindrical (Sphere) (deprecated) [6777] World_Aitoff [6778] World_Azimuthal_Equidistant [6779] World_Bonne [6780] World_Cassini [6781] World_Craster_Parabolic [6782] World_Cylindrical_Equal_Area [6783] World_Eckert_I [6784] World_Eckert_II [6785] World_Eckert_III [6786] World_Eckert_IV [6787] World_Eckert_V [6788] World_Eckert_VI [6789] World_Equidistant_Conic [6790] World_Equidistant_Cylindrical [6791] World_Gall_Stereographic [6792] World_Goode_Homolosine_Land [6793] World_Goode_Homolosine_Ocean [6794] World_Hotine [6795] World_Loximuthal [6796] World_Mercator [6797] World_Miller_Cylindrical [6798] World_Mollweide [6799] World_Plate_Carree [6800] World_Polyconic [6801] World_Quartic_Authalic [6802] World_Robinson [6803] World_Sinusoidal [6804] World_Stereographic [6805] World_Two_Point_Equidistant [6806] World_Van_der_Grinten_I [6807] World_Wagner_IV [6808] World_Wagner_V [6809] World_Wagner_VII [6810] World_Winkel_I [6811] World_Winkel_II [6812] World_Winkel_Tripel_NGS [6813] Xian 1980 / 3-degree Gauss-Kruger CM 102E [6814] Xian 1980 / 3-degree Gauss-Kruger CM 105E [6815] Xian 1980 / 3-degree Gauss-Kruger CM 108E [6816] Xian 1980 / 3-degree Gauss-Kruger CM 111E [6817] Xian 1980 / 3-degree Gauss-Kruger CM 114E [6818] Xian 1980 / 3-degree Gauss-Kruger CM 117E [6819] Xian 1980 / 3-degree Gauss-Kruger CM 120E [6820] Xian 1980 / 3-degree Gauss-Kruger CM 123E [6821] Xian 1980 / 3-degree Gauss-Kruger CM 126E [6822] Xian 1980 / 3-degree Gauss-Kruger CM 129E [6823] Xian 1980 / 3-degree Gauss-Kruger CM 132E [6824] Xian 1980 / 3-degree Gauss-Kruger CM 135E [6825] Xian 1980 / 3-degree Gauss-Kruger CM 75E [6826] Xian 1980 / 3-degree Gauss-Kruger CM 78E [6827] Xian 1980 / 3-degree Gauss-Kruger CM 81E [6828] Xian 1980 / 3-degree Gauss-Kruger CM 84E [6829] Xian 1980 / 3-degree Gauss-Kruger CM 87E [6830] Xian 1980 / 3-degree Gauss-Kruger CM 90E [6831] Xian 1980 / 3-degree Gauss-Kruger CM 93E [6832] Xian 1980 / 3-degree Gauss-Kruger CM 96E [6833] Xian 1980 / 3-degree Gauss-Kruger CM 99E [6834] Xian 1980 / 3-degree Gauss-Kruger zone 25 [6835] Xian 1980 / 3-degree Gauss-Kruger zone 26 [6836] Xian 1980 / 3-degree Gauss-Kruger zone 27 [6837] Xian 1980 / 3-degree Gauss-Kruger zone 28 [6838] Xian 1980 / 3-degree Gauss-Kruger zone 29 [6839] Xian 1980 / 3-degree Gauss-Kruger zone 30 [6840] Xian 1980 / 3-degree Gauss-Kruger zone 31 [6841] Xian 1980 / 3-degree Gauss-Kruger zone 32 [6842] Xian 1980 / 3-degree Gauss-Kruger zone 33 [6843] Xian 1980 / 3-degree Gauss-Kruger zone 34 [6844] Xian 1980 / 3-degree Gauss-Kruger zone 35 [6845] Xian 1980 / 3-degree Gauss-Kruger zone 36 [6846] Xian 1980 / 3-degree Gauss-Kruger zone 37 [6847] Xian 1980 / 3-degree Gauss-Kruger zone 38 [6848] Xian 1980 / 3-degree Gauss-Kruger zone 39 [6849] Xian 1980 / 3-degree Gauss-Kruger zone 40 [6850] Xian 1980 / 3-degree Gauss-Kruger zone 41 [6851] Xian 1980 / 3-degree Gauss-Kruger zone 42 [6852] Xian 1980 / 3-degree Gauss-Kruger zone 43 [6853] Xian 1980 / 3-degree Gauss-Kruger zone 44 [6854] Xian 1980 / 3-degree Gauss-Kruger zone 45 [6855] Xian 1980 / Gauss-Kruger CM 105E [6856] Xian 1980 / Gauss-Kruger CM 111E [6857] Xian 1980 / Gauss-Kruger CM 117E [6858] Xian 1980 / Gauss-Kruger CM 123E [6859] Xian 1980 / Gauss-Kruger CM 129E [6860] Xian 1980 / Gauss-Kruger CM 135E [6861] Xian 1980 / Gauss-Kruger CM 75E [6862] Xian 1980 / Gauss-Kruger CM 81E [6863] Xian 1980 / Gauss-Kruger CM 87E [6864] Xian 1980 / Gauss-Kruger CM 93E [6865] Xian 1980 / Gauss-Kruger CM 99E [6866] Xian 1980 / Gauss-Kruger zone 13 [6867] Xian 1980 / Gauss-Kruger zone 14 [6868] Xian 1980 / Gauss-Kruger zone 15 [6869] Xian 1980 / Gauss-Kruger zone 16 [6870] Xian 1980 / Gauss-Kruger zone 17 [6871] Xian 1980 / Gauss-Kruger zone 18 [6872] Xian 1980 / Gauss-Kruger zone 19 [6873] Xian 1980 / Gauss-Kruger zone 20 [6874] Xian 1980 / Gauss-Kruger zone 21 [6875] Xian 1980 / Gauss-Kruger zone 22 [6876] Xian 1980 / Gauss-Kruger zone 23 [6877] Yemen NGN96 / UTM zone 37N [6878] Yemen NGN96 / UTM zone 38N [6879] Yemen NGN96 / UTM zone 39N [6880] Yemen NGN96 / UTM zone 40N [6881] Yoff / UTM zone 28N [6882] Zanderij / Suriname Old TM [6883] Zanderij / Suriname TM [6884] Zanderij / TM 54 NW [6885] Zanderij / UTM zone 21N Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '21', 'Export Shapes')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SHAPES_PK', SHAPES_PK)
        Tool.Set_Option('SHAPES_NN', SHAPES_NN)
        Tool.Set_Option('SHAPES_UQ', SHAPES_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        Tool.Set_Option('CRS_GEOGCS', CRS_GEOGCS)
        Tool.Set_Option('CRS_PROJCS', CRS_PROJCS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_21(SHAPES=None, CONNECTION=None, SHAPES_PK=None, SHAPES_NN=None, SHAPES_UQ=None, NAME=None, EXISTS=None, CRS_CODE=None, CRS_AUTHORITY=None, CRS_GEOGCS=None, CRS_PROJCS=None, Verbose=2):
    '''
    Export Shapes
    ----------
    [db_pgsql.21]\n
    Exports shapes to a PostGIS database.\n
    Arguments
    ----------
    - SHAPES [`input shapes`] : Shapes
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - SHAPES_PK [`table fields`] : Primary Key
    - SHAPES_NN [`table fields`] : Not Null
    - SHAPES_UQ [`table fields`] : Unique
    - NAME [`text`] : Table Name. if empty shapes layers's name is used as table name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG
    - CRS_GEOGCS [`choice`] : Geographic Coordinate Systems. Available Choices: [0] <select> [1] 1_Ceres_2015 [2] 4_Vesta_2015 [3] AGD66 [4] AGD84 [5] ATF (Paris) [6] ATS77 [7] Abidjan 1987 [8] Accra [9] Aden 1925 [10] Adindan [11] Afgooye [12] Agadez [13] Ain el Abd [14] Albanian 1987 [15] American Samoa 1962 [16] Amersfoort [17] Ammassalik 1958 [18] Anguilla 1957 [19] Antigua 1943 [20] Aratu [21] Arc 1950 [22] Arc 1960 [23] Ascension Island 1958 [24] Astro DOS 71 [25] Australian Antarctic [26] Australian Antarctic [27] Australian Antarctic [28] Ayabelle Lighthouse [29] Azores Central 1948 [30] Azores Central 1995 [31] Azores Occidental 1939 [32] Azores Oriental 1940 [33] Azores Oriental 1995 [34] BDA2000 [35] BDA2000 [36] BGS2005 [37] BGS2005 [38] Barbados 1938 [39] Batavia [40] Batavia (Jakarta) [41] Beduaram [42] Beijing 1954 [43] Bekaa Valley 1920 [44] Belge 1950 [45] Belge 1950 (Brussels) [46] Belge 1972 [47] Bellevue [48] Bermuda 1957 [49] Bern 1898 (Bern) [50] Bern 1938 [51] Bioko [52] Bissau [53] Bogota 1975 [54] Bogota 1975 (Bogota) [55] Bukit Rimpah [56] CGRS93 [57] CGRS93 [58] CH1903 [59] CH1903+ [60] CHTRF95 [61] CHTRF95 [62] CHTRF95 [63] CIGD11 [64] CIGD11 [65] CR-SIRGAS [66] CR-SIRGAS [67] CR05 [68] CR05 [69] CSG67 [70] Cadastre 1997 [71] Cadastre 1997 [72] California_SRS_Epoch_2017.50_(NAD83) [73] Camacupa [74] Camp Area Astro [75] Campo Inchauspe [76] Cape [77] Cape Canaveral [78] Carthage [79] Carthage (Paris) [80] Chatham Islands 1971 [81] Chatham Islands 1979 [82] China Geodetic Coordinate System 2000 [83] China Geodetic Coordinate System 2000 [84] Chos Malal 1914 [85] Chua [86] Cocos Islands 1965 [87] Combani 1950 [88] Conakry 1905 [89] Corrego Alegre 1961 [90] Corrego Alegre 1970-72 [91] Cote d'Ivoire [92] DB_REF [93] DB_REF [94] DGN95 [95] DGN95 [96] DHDN [97] DRUKREF 03 [98] DRUKREF 03 [99] Dabola 1981 [100] Datum 73 [101] Dealul Piscului 1930 [102] Dealul Piscului 1970 [103] Deception Island [104] Deir ez Zor [105] Diego Garcia 1969 [106] Dominica 1945 [107] Douala [108] Douala 1948 [109] ED50 [110] ED50(ED77) [111] ED79 [112] ED87 [113] ELD79 [114] EST92 [115] EST97 [116] EST97 [117] EST97 [118] ETRF2000 [119] ETRF2000 [120] ETRF2005 [121] ETRF2014 [122] ETRF89 [123] ETRF90 [124] ETRF90 [125] ETRF91 [126] ETRF91 [127] ETRF92 [128] ETRF92 [129] ETRF93 [130] ETRF93 [131] ETRF94 [132] ETRF94 [133] ETRF96 [134] ETRF96 [135] ETRF97 [136] ETRF97 [137] ETRS89 [138] ETRS89 [139] ETRS89 [140] Easter Island 1967 [141] Egypt 1907 [142] Egypt 1930 [143] Egypt Gulf of Suez S-650 TL [144] FD54 [145] FD58 [146] FEH2010 [147] FEH2010 [148] Fahud [149] Fatu Iva 72 [150] Fiji 1956 [151] Fiji 1986 [152] Fort Marigot [153] GCGD59 [154] GCS_Adrastea_2000 [155] GCS_Alaskan_Islands [156] GCS_Amalthea_2000 [157] GCS_American_Samoa_1962 (deprecated) [158] GCS_Ananke_2000 [159] GCS_Anna_1_1965 (deprecated) [160] GCS_Ariel_2000 [161] GCS_Ascension_Island_1958 (deprecated) [162] GCS_Assumed_Geographic_1 (deprecated) [163] GCS_Astro_1952 (deprecated) [164] GCS_Atlas_2000 [165] GCS_Ayabelle (deprecated) [166] GCS_Bab_South [167] GCS_Beacon_E_1945 (deprecated) [168] GCS_Belinda_2000 [169] GCS_Bellevue_IGN (deprecated) [170] GCS_Bermuda_2000 (deprecated) [171] GCS_Bianca_2000 [172] GCS_CGRS_1993 (deprecated) [173] GCS_CR05 (deprecated) [174] GCS_Callisto_2000 [175] GCS_Calypso_2000 [176] GCS_Camp_Area (deprecated) [177] GCS_Canton_1966 (deprecated) [178] GCS_Cape_Canaveral (deprecated) [179] GCS_Carme_2000 [180] GCS_Carthage (deprecated) [181] GCS_Carthage_Grad [182] GCS_Charon_2000 [183] GCS_Chatham_Island_1971 (deprecated) [184] GCS_Chatham_Islands_1979 (deprecated) [185] GCS_Cordelia_2000 [186] GCS_Cressida_2000 [187] GCS_D48 [188] GCS_DOS_1968 [189] GCS_DOS_71_4 (deprecated) [190] GCS_Datum_Lisboa_Bessel [191] GCS_Datum_Lisboa_Hayford [192] GCS_Deception_Island (deprecated) [193] GCS_Deimos_2000 [194] GCS_Desdemona_2000 [195] GCS_Despina_2000 [196] GCS_Dione_2000 [197] GCS_ETRF_1989 [198] GCS_EUREF_FIN [199] GCS_Easter_Island_1967 (deprecated) [200] GCS_Elara_2000 [201] GCS_Enceladus_2000 [202] GCS_Epimetheus_2000 [203] GCS_Estonia_1937 [204] GCS_Europa_2000 [205] GCS_European_1979 (deprecated) [206] GCS_Everest_Bangladesh [207] GCS_Everest_India_Nepal [208] GCS_Everest_Modified_1969 [209] GCS_Fischer_1960 [210] GCS_Fischer_1968 [211] GCS_Fischer_Modified [212] GCS_Fort_Thomas_1955 [213] GCS_GDBD2009 (deprecated) [214] GCS_GUX_1 [215] GCS_Galatea_2000 [216] GCS_Gan_1970 (deprecated) [217] GCS_Ganymede_2000 [218] GCS_Graciosa_Base_SW_1948 [219] GCS_Guam_1963 (deprecated) [220] GCS_Gunung_Segara (deprecated) [221] GCS_HD1909 (deprecated) [222] GCS_Helene_2000 [223] GCS_Hermannskogel [224] GCS_Himalia_2000 [225] GCS_Hjorsey_1955 (deprecated) [226] GCS_Hong_Kong_1963_67 (deprecated) [227] GCS_Hong_Kong_1980 (deprecated) [228] GCS_Hough_1960 [229] GCS_Hyperion_2000 [230] GCS_IGRS (deprecated) [231] GCS_IGS08 [232] GCS_ISN_2004 (deprecated) [233] GCS_ISTS_061_1968 (deprecated) [234] GCS_ISTS_073_1969 (deprecated) [235] GCS_ITRF_1988 [236] GCS_ITRF_1989 [237] GCS_ITRF_1990 [238] GCS_ITRF_1991 [239] GCS_ITRF_1992 [240] GCS_ITRF_1993 [241] GCS_ITRF_1994 [242] GCS_ITRF_1996 [243] GCS_ITRF_1997 [244] GCS_ITRF_2000 [245] GCS_ITRF_2005 [246] GCS_ITRF_2008 [247] GCS_Iapetus_2000 [248] GCS_International_1967 (deprecated) [249] GCS_Io_2000 [250] GCS_JAD_2001 (deprecated) [251] GCS_JGD_2000 (deprecated) [252] GCS_JGD_2011 (deprecated) [253] GCS_Janus_2000 [254] GCS_Johnston_Island_1961 (deprecated) [255] GCS_Jordan [256] GCS_Juliet_2000 [257] GCS_Jupiter_2000 [258] GCS_Kerguelen_Island_1949 (deprecated) [259] GCS_Kusaie_1951 (deprecated) [260] GCS_Kyrg-06 [261] GCS_LC5_1961 [262] GCS_Larissa_2000 [263] GCS_Leda_2000 [264] GCS_Lysithea_2000 [265] GCS_MACAO_2008 [266] GCS_MGI_1901 (deprecated) [267] GCS_MONREF_1997 [268] GCS_MSK_1942 [269] GCS_Majuro [270] GCS_Mars_1979 [271] GCS_Mars_2000 [272] GCS_Merchich_Degree [273] GCS_Mercury_2000 [274] GCS_Metis_2000 [275] GCS_Midway_1961 (deprecated) [276] GCS_Mimas_2000 [277] GCS_Miranda_2000 [278] GCS_Moon_2000 [279] GCS_NAD_1983_2011 (deprecated) [280] GCS_NAD_1983_CORS96 (deprecated) [281] GCS_NAD_1983_HARN_Adj_MN_Anoka [282] GCS_NAD_1983_HARN_Adj_MN_Becker [283] GCS_NAD_1983_HARN_Adj_MN_Beltrami_North [284] GCS_NAD_1983_HARN_Adj_MN_Beltrami_South [285] GCS_NAD_1983_HARN_Adj_MN_Benton [286] GCS_NAD_1983_HARN_Adj_MN_Big_Stone [287] GCS_NAD_1983_HARN_Adj_MN_Blue_Earth [288] GCS_NAD_1983_HARN_Adj_MN_Brown [289] GCS_NAD_1983_HARN_Adj_MN_Carlton [290] GCS_NAD_1983_HARN_Adj_MN_Carver [291] GCS_NAD_1983_HARN_Adj_MN_Cass_North [292] GCS_NAD_1983_HARN_Adj_MN_Cass_South [293] GCS_NAD_1983_HARN_Adj_MN_Chippewa [294] GCS_NAD_1983_HARN_Adj_MN_Chisago [295] GCS_NAD_1983_HARN_Adj_MN_Cook_North [296] GCS_NAD_1983_HARN_Adj_MN_Cook_South [297] GCS_NAD_1983_HARN_Adj_MN_Cottonwood [298] GCS_NAD_1983_HARN_Adj_MN_Crow_Wing [299] GCS_NAD_1983_HARN_Adj_MN_Dakota [300] GCS_NAD_1983_HARN_Adj_MN_Dodge [301] GCS_NAD_1983_HARN_Adj_MN_Douglas [302] GCS_NAD_1983_HARN_Adj_MN_Faribault [303] GCS_NAD_1983_HARN_Adj_MN_Fillmore [304] GCS_NAD_1983_HARN_Adj_MN_Freeborn [305] GCS_NAD_1983_HARN_Adj_MN_Goodhue [306] GCS_NAD_1983_HARN_Adj_MN_Grant [307] GCS_NAD_1983_HARN_Adj_MN_Hennepin [308] GCS_NAD_1983_HARN_Adj_MN_Houston [309] GCS_NAD_1983_HARN_Adj_MN_Isanti [310] GCS_NAD_1983_HARN_Adj_MN_Itasca_North [311] GCS_NAD_1983_HARN_Adj_MN_Itasca_South [312] GCS_NAD_1983_HARN_Adj_MN_Jackson [313] GCS_NAD_1983_HARN_Adj_MN_Kanabec [314] GCS_NAD_1983_HARN_Adj_MN_Kandiyohi [315] GCS_NAD_1983_HARN_Adj_MN_Kittson [316] GCS_NAD_1983_HARN_Adj_MN_Koochiching [317] GCS_NAD_1983_HARN_Adj_MN_Lac_Qui_Parle [318] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North [319] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South [320] GCS_NAD_1983_HARN_Adj_MN_Le_Sueur [321] GCS_NAD_1983_HARN_Adj_MN_Lincoln [322] GCS_NAD_1983_HARN_Adj_MN_Lyon [323] GCS_NAD_1983_HARN_Adj_MN_Mahnomen [324] GCS_NAD_1983_HARN_Adj_MN_Marshall [325] GCS_NAD_1983_HARN_Adj_MN_Martin [326] GCS_NAD_1983_HARN_Adj_MN_McLeod [327] GCS_NAD_1983_HARN_Adj_MN_Meeker [328] GCS_NAD_1983_HARN_Adj_MN_Morrison [329] GCS_NAD_1983_HARN_Adj_MN_Mower [330] GCS_NAD_1983_HARN_Adj_MN_Murray [331] GCS_NAD_1983_HARN_Adj_MN_Nicollet [332] GCS_NAD_1983_HARN_Adj_MN_Nobles [333] GCS_NAD_1983_HARN_Adj_MN_Norman [334] GCS_NAD_1983_HARN_Adj_MN_Olmsted [335] GCS_NAD_1983_HARN_Adj_MN_Ottertail [336] GCS_NAD_1983_HARN_Adj_MN_Pennington [337] GCS_NAD_1983_HARN_Adj_MN_Pine [338] GCS_NAD_1983_HARN_Adj_MN_Pipestone [339] GCS_NAD_1983_HARN_Adj_MN_Polk [340] GCS_NAD_1983_HARN_Adj_MN_Pope [341] GCS_NAD_1983_HARN_Adj_MN_Ramsey [342] GCS_NAD_1983_HARN_Adj_MN_Red_Lake [343] GCS_NAD_1983_HARN_Adj_MN_Redwood [344] GCS_NAD_1983_HARN_Adj_MN_Renville [345] GCS_NAD_1983_HARN_Adj_MN_Rice [346] GCS_NAD_1983_HARN_Adj_MN_Rock [347] GCS_NAD_1983_HARN_Adj_MN_Roseau [348] GCS_NAD_1983_HARN_Adj_MN_Scott [349] GCS_NAD_1983_HARN_Adj_MN_Sherburne [350] GCS_NAD_1983_HARN_Adj_MN_Sibley [351] GCS_NAD_1983_HARN_Adj_MN_St_Louis [352] GCS_NAD_1983_HARN_Adj_MN_St_Louis_Central [353] GCS_NAD_1983_HARN_Adj_MN_St_Louis_North [354] GCS_NAD_1983_HARN_Adj_MN_St_Louis_South [355] GCS_NAD_1983_HARN_Adj_MN_Stearns [356] GCS_NAD_1983_HARN_Adj_MN_Steele [357] GCS_NAD_1983_HARN_Adj_MN_Stevens [358] GCS_NAD_1983_HARN_Adj_MN_Swift [359] GCS_NAD_1983_HARN_Adj_MN_Todd [360] GCS_NAD_1983_HARN_Adj_MN_Traverse [361] GCS_NAD_1983_HARN_Adj_MN_Wabasha [362] GCS_NAD_1983_HARN_Adj_MN_Wadena [363] GCS_NAD_1983_HARN_Adj_MN_Waseca [364] GCS_NAD_1983_HARN_Adj_MN_Watonwan [365] GCS_NAD_1983_HARN_Adj_MN_Winona [366] GCS_NAD_1983_HARN_Adj_MN_Wright [367] GCS_NAD_1983_HARN_Adj_MN_Yellow_Medicine [368] GCS_NAD_1983_HARN_Adj_WI_Adams (deprecated) [369] GCS_NAD_1983_HARN_Adj_WI_Ashland (deprecated) [370] GCS_NAD_1983_HARN_Adj_WI_Barron (deprecated) [371] GCS_NAD_1983_HARN_Adj_WI_Bayfield (deprecated) [372] GCS_NAD_1983_HARN_Adj_WI_Brown (deprecated) [373] GCS_NAD_1983_HARN_Adj_WI_Buffalo (deprecated) [374] GCS_NAD_1983_HARN_Adj_WI_Burnett (deprecated) [375] GCS_NAD_1983_HARN_Adj_WI_Calumet (deprecated) [376] GCS_NAD_1983_HARN_Adj_WI_Chippewa [377] GCS_NAD_1983_HARN_Adj_WI_Clark (deprecated) [378] GCS_NAD_1983_HARN_Adj_WI_Columbia (deprecated) [379] GCS_NAD_1983_HARN_Adj_WI_Crawford (deprecated) [380] GCS_NAD_1983_HARN_Adj_WI_Dane (deprecated) [381] GCS_NAD_1983_HARN_Adj_WI_Dodge (deprecated) [382] GCS_NAD_1983_HARN_Adj_WI_Door (deprecated) [383] GCS_NAD_1983_HARN_Adj_WI_Douglas (deprecated) [384] GCS_NAD_1983_HARN_Adj_WI_Dunn (deprecated) [385] GCS_NAD_1983_HARN_Adj_WI_EauClaire (deprecated) [386] GCS_NAD_1983_HARN_Adj_WI_Florence (deprecated) [387] GCS_NAD_1983_HARN_Adj_WI_FondduLac (deprecated) [388] GCS_NAD_1983_HARN_Adj_WI_Forest (deprecated) [389] GCS_NAD_1983_HARN_Adj_WI_Grant (deprecated) [390] GCS_NAD_1983_HARN_Adj_WI_Green (deprecated) [391] GCS_NAD_1983_HARN_Adj_WI_GreenLake (deprecated) [392] GCS_NAD_1983_HARN_Adj_WI_Iowa (deprecated) [393] GCS_NAD_1983_HARN_Adj_WI_Iron (deprecated) [394] GCS_NAD_1983_HARN_Adj_WI_Jackson (deprecated) [395] GCS_NAD_1983_HARN_Adj_WI_Jefferson (deprecated) [396] GCS_NAD_1983_HARN_Adj_WI_Juneau (deprecated) [397] GCS_NAD_1983_HARN_Adj_WI_Kenosha (deprecated) [398] GCS_NAD_1983_HARN_Adj_WI_Kewaunee (deprecated) [399] GCS_NAD_1983_HARN_Adj_WI_LaCrosse (deprecated) [400] GCS_NAD_1983_HARN_Adj_WI_Lafayette (deprecated) [401] GCS_NAD_1983_HARN_Adj_WI_Langlade (deprecated) [402] GCS_NAD_1983_HARN_Adj_WI_Lincoln (deprecated) [403] GCS_NAD_1983_HARN_Adj_WI_Manitowoc (deprecated) [404] GCS_NAD_1983_HARN_Adj_WI_Marathon (deprecated) [405] GCS_NAD_1983_HARN_Adj_WI_Marinette (deprecated) [406] GCS_NAD_1983_HARN_Adj_WI_Marquette (deprecated) [407] GCS_NAD_1983_HARN_Adj_WI_Menominee (deprecated) [408] GCS_NAD_1983_HARN_Adj_WI_Milwaukee (deprecated) [409] GCS_NAD_1983_HARN_Adj_WI_Monroe (deprecated) [410] GCS_NAD_1983_HARN_Adj_WI_Oconto (deprecated) [411] GCS_NAD_1983_HARN_Adj_WI_Oneida (deprecated) [412] GCS_NAD_1983_HARN_Adj_WI_Outagamie (deprecated) [413] GCS_NAD_1983_HARN_Adj_WI_Ozaukee (deprecated) [414] GCS_NAD_1983_HARN_Adj_WI_Pepin (deprecated) [415] GCS_NAD_1983_HARN_Adj_WI_Pierce (deprecated) [416] GCS_NAD_1983_HARN_Adj_WI_Polk (deprecated) [417] GCS_NAD_1983_HARN_Adj_WI_Portage (deprecated) [418] GCS_NAD_1983_HARN_Adj_WI_Price (deprecated) [419] GCS_NAD_1983_HARN_Adj_WI_Racine (deprecated) [420] GCS_NAD_1983_HARN_Adj_WI_Richland (deprecated) [421] GCS_NAD_1983_HARN_Adj_WI_Rock (deprecated) [422] GCS_NAD_1983_HARN_Adj_WI_Rusk (deprecated) [423] GCS_NAD_1983_HARN_Adj_WI_Sauk (deprecated) [424] GCS_NAD_1983_HARN_Adj_WI_Sawyer (deprecated) [425] GCS_NAD_1983_HARN_Adj_WI_Shawano (deprecated) [426] GCS_NAD_1983_HARN_Adj_WI_Sheboygan (deprecated) [427] GCS_NAD_1983_HARN_Adj_WI_StCroix (deprecated) [428] GCS_NAD_1983_HARN_Adj_WI_Taylor (deprecated) [429] GCS_NAD_1983_HARN_Adj_WI_Trempealeau (deprecated) [430] GCS_NAD_1983_HARN_Adj_WI_Vernon (deprecated) [431] GCS_NAD_1983_HARN_Adj_WI_Vilas (deprecated) [432] GCS_NAD_1983_HARN_Adj_WI_Walworth (deprecated) [433] GCS_NAD_1983_HARN_Adj_WI_Washburn (deprecated) [434] GCS_NAD_1983_HARN_Adj_WI_Washington (deprecated) [435] GCS_NAD_1983_HARN_Adj_WI_Waukesha (deprecated) [436] GCS_NAD_1983_HARN_Adj_WI_Waupaca (deprecated) [437] GCS_NAD_1983_HARN_Adj_WI_Waushara (deprecated) [438] GCS_NAD_1983_HARN_Adj_WI_Winnebago (deprecated) [439] GCS_NAD_1983_HARN_Adj_WI_Wood (deprecated) [440] GCS_NAD_1983_MA11 (deprecated) [441] GCS_NAD_1983_MARP00 [442] GCS_NAD_1983_PA11 (deprecated) [443] GCS_NAD_1983_PACP00 [444] GCS_NZGD_2000 (deprecated) [445] GCS_Naiad_2000 [446] GCS_Nepal_Nagarkot (deprecated) [447] GCS_Neptune_2000 [448] GCS_Nereid_2000 [449] GCS_Oberon_2000 [450] GCS_Observatorio_Meteorologico_1939 [451] GCS_Observatorio_Meteorologico_1965 [452] GCS_Ocotepeque_1935 (deprecated) [453] GCS_Old_Hawaiian_Intl_1924 [454] GCS_Oman [455] GCS_Ophelia_2000 [456] GCS_PTRA08 (deprecated) [457] GCS_Pan_2000 [458] GCS_Pandora_2000 [459] GCS_Pasiphae_2000 [460] GCS_Phobos_2000 [461] GCS_Phoebe_2000 [462] GCS_Pico_de_Las_Nieves (deprecated) [463] GCS_Pitcairn_1967 (deprecated) [464] GCS_Pluto_2000 [465] GCS_Pohnpei [466] GCS_Point_58 (deprecated) [467] GCS_Portia_2000 [468] GCS_Porto_Santo_1936 (deprecated) [469] GCS_Prometheus_2000 [470] GCS_Proteus_2000 [471] GCS_Puck_2000 [472] GCS_REGVEN (deprecated) [473] GCS_RGF_1993 (deprecated) [474] GCS_Reunion_1947 (deprecated) [475] GCS_Rhea_2000 [476] GCS_Roma_1940 [477] GCS_Rosalind_2000 [478] GCS_S42_Hungary [479] GCS_Santo_DOS_1965 (deprecated) [480] GCS_Sao_Braz [481] GCS_Saturn_2000 [482] GCS_Selvagem_Grande_1938 (deprecated) [483] GCS_Sierra_Leone_1960 [484] GCS_Sinope_2000 [485] GCS_South_Asia_Singapore [486] GCS_Sphere_ARC_INFO [487] GCS_Sphere_EMEP [488] GCS_Sphere_GRS_1980_Mean_Radius [489] GCS_TWD_1967 (deprecated) [490] GCS_TWD_1997 (deprecated) [491] GCS_Telesto_2000 [492] GCS_Tern_Island_1961 (deprecated) [493] GCS_Tethys_2000 [494] GCS_Thalassa_2000 [495] GCS_Thebe_2000 [496] GCS_Titan_2000 [497] GCS_Titania_2000 [498] GCS_Tristan_1968 (deprecated) [499] GCS_Triton_2000 [500] GCS_Umbriel_2000 [501] GCS_Uranus_2000 [502] GCS_Venus_1985 [503] GCS_Venus_2000 [504] GCS_Viti_Levu_1916 (deprecated) [505] GCS_Voirol_1875 (deprecated) [506] GCS_Voirol_1875_Grad [507] GCS_Voirol_1879_Grad [508] GCS_Voirol_Unifie_1960 (deprecated) [509] GCS_Voirol_Unifie_1960 (deprecated) [510] GCS_Voirol_Unifie_1960_Degree (deprecated) [511] GCS_Voirol_Unifie_1960_Paris (deprecated) [512] GCS_WGS_1966 (deprecated) [513] GCS_WGS_1984_Major_Auxiliary_Sphere (deprecated) [514] GCS_Wake_Eniwetok_1960 (deprecated) [515] GCS_Wake_Island_1952 (deprecated) [516] GCS_Walbeck [517] GCS_Xrail84 [518] GDA2020 [519] GDA2020 [520] GDA94 [521] GDA94 [522] GDA94 [523] GDBD2009 [524] GDBD2009 [525] GDM2000 [526] GDM2000 [527] GGD [528] GGRS87 [529] GR96 [530] GR96 [531] GSK-2011 [532] GSK-2011 [533] Gambia [534] Gan 1970 [535] Gandajika [536] Gandajika 1970 [537] Garoua [538] Garoua [539] Grand Comoros [540] Greek [541] Greek (Athens) [542] Grenada 1953 [543] Guadeloupe 1948 [544] Guam 1963 [545] Gulshan 303 [546] Gusterberg (Ferro) [547] Guyane Francaise [548] HD1909 [549] HD72 [550] HTRS96 [551] HTRS96 [552] Hanoi 1972 [553] Hartebeesthoek94 [554] Hartebeesthoek94 [555] Hartebeesthoek94 [556] Helle 1954 [557] Herat North [558] Hito XVIII 1963 [559] Hjorsey 1955 [560] Hong Kong 1963 [561] Hong Kong 1963(67) [562] Hong Kong 1980 [563] Hong Kong Geodetic CS [564] Hong Kong Geodetic CS [565] Hu Tzu Shan 1950 [566] ID74 [567] IG05 Intermediate CRS [568] IG05 Intermediate CRS [569] IG05/12 Intermediate CRS [570] IG05/12 Intermediate CRS [571] IGC 1962 6th Parallel South [572] IGCB 1955 [573] IGD05 [574] IGD05 [575] IGD05 [576] IGD05 [577] IGD05/12 [578] IGD05/12 [579] IGD05/12 [580] IGD05/12 [581] IGM95 [582] IGM95 [583] IGN 1962 Kerguelen [584] IGN Astro 1960 [585] IGN53 Mare [586] IGN56 Lifou [587] IGN63 Hiva Oa [588] IGN72 Grand Terre [589] IGN72 Grande Terre [590] IGN72 Nuku Hiva [591] IGRS [592] IGRS [593] IGS00 [594] IGS00 [595] IGS05 [596] IGS05 [597] IGS08 [598] IGS08 [599] IGS14 [600] IGS14 [601] IGS14 [602] IGS97 [603] IGS97 [604] IGb00 [605] IGb00 [606] IGb08 [607] IGb08 [608] IKBD-92 [609] IRENET95 [610] IRENET95 [611] IRENET95 [612] ISN2004 [613] ISN2004 [614] ISN2016 [615] ISN2016 [616] ISN93 [617] ISN93 [618] ISN93 [619] ITRF2000 [620] ITRF2000 [621] ITRF2005 [622] ITRF2005 [623] ITRF2008 [624] ITRF2008 [625] ITRF2014 [626] ITRF2014 [627] ITRF2014 [628] ITRF88 [629] ITRF88 [630] ITRF89 [631] ITRF89 [632] ITRF90 [633] ITRF90 [634] ITRF91 [635] ITRF91 [636] ITRF92 [637] ITRF92 [638] ITRF93 [639] ITRF93 [640] ITRF94 [641] ITRF94 [642] ITRF96 [643] ITRF96 [644] ITRF97 [645] ITRF97 [646] Indian 1954 [647] Indian 1960 [648] Indian 1975 [649] Israel 1993 [650] Iwo Jima 1945 [651] JAD2001 [652] JAD2001 [653] JAD69 [654] JGD2000 [655] JGD2000 [656] JGD2000 [657] JGD2011 [658] JGD2011 [659] Jamaica 1875 [660] Johnston Island 1961 [661] Jouik 1961 [662] K0 1949 [663] KKJ [664] KOC [665] KUDAMS [666] Kalianpur 1880 [667] Kalianpur 1937 [668] Kalianpur 1962 [669] Kalianpur 1975 [670] Kandawala [671] Karbala 1979 [672] Kasai 1953 [673] Katanga 1955 [674] Kertau (RSO) [675] Kertau 1968 [676] Korea 2000 [677] Korea 2000 [678] Korean 1985 [679] Korean 1995 [680] Kousseri [681] Kusaie 1951 [682] Kyrg-06 [683] Kyrg-06 [684] LGD2006 [685] LGD2006 [686] LKS92 [687] LKS92 [688] LKS92 [689] LKS94 [690] LKS94 [691] LKS94 [692] LKS94 (ETRS89) [693] La Canoa [694] Lake [695] Lao 1993 [696] Lao 1993 [697] Lao 1997 [698] Lao 1997 [699] Le Pouce 1934 [700] Leigon [701] Liberia 1964 [702] Lisbon [703] Lisbon (Lisbon) [704] Lisbon 1890 [705] Lisbon 1890 (Lisbon) [706] Locodjo 1965 [707] Loma Quintana [708] Lome [709] Luxembourg 1930 [710] Luzon 1911 [711] M'poraloko [712] MACARIO SOLIS [713] MACARIO SOLIS [714] MAGNA-SIRGAS [715] MAGNA-SIRGAS [716] MARGEN [717] MARGEN [718] MGI [719] MGI (Ferro) [720] MGI 1901 [721] MOLDREF99 [722] MOLDREF99 [723] MOP78 [724] MTRF-2000 [725] MTRF-2000 [726] Macao 1920 [727] Macao 2008 [728] Macao 2008 [729] Madeira 1936 [730] Madrid 1870 (Madrid) [731] Madzansua [732] Mahe 1971 [733] Makassar [734] Makassar (Jakarta) [735] Malongo 1987 [736] Manoca [737] Manoca 1962 [738] Marcus Island 1952 [739] Mars_2000_(Sphere) [740] Marshall Islands 1960 [741] Martinique 1938 [742] Massawa [743] Maupiti 83 [744] Mauritania 1999 [745] Mauritania 1999 [746] Mauritania 1999 [747] Merchich [748] Mercury_2015 [749] Mexico ITRF2008 [750] Mexico ITRF2008 [751] Mexico ITRF92 [752] Mexico ITRF92 [753] Mhast [754] Mhast (offshore) [755] Mhast (onshore) [756] Mhast 1951 [757] Midway 1961 [758] Minna [759] Monte Mario [760] Monte Mario (Rome) [761] Montserrat 1958 [762] Moorea 87 [763] Mount Dillon [764] Moznet [765] Moznet [766] Moznet [767] NAD27 [768] NAD27 Michigan [769] NAD27(76) [770] NAD27(CGQ77) [771] NAD83 [772] NAD83(2011) [773] NAD83(2011) [774] NAD83(CORS96) [775] NAD83(CORS96) [776] NAD83(CSRS) [777] NAD83(CSRS) [778] NAD83(CSRS) [779] NAD83(CSRS)v2 [780] NAD83(CSRS)v2 [781] NAD83(CSRS)v3 [782] NAD83(CSRS)v3 [783] NAD83(CSRS)v4 [784] NAD83(CSRS)v4 [785] NAD83(CSRS)v5 [786] NAD83(CSRS)v5 [787] NAD83(CSRS)v6 [788] NAD83(CSRS)v6 [789] NAD83(CSRS)v7 [790] NAD83(CSRS)v7 [791] NAD83(CSRS96) [792] NAD83(CSRS96) [793] NAD83(CSRS98) [794] NAD83(FBN) [795] NAD83(FBN) [796] NAD83(FBN) [797] NAD83(HARN Corrected) [798] NAD83(HARN Corrected) [799] NAD83(HARN) [800] NAD83(HARN) [801] NAD83(HARN) [802] NAD83(MA11) [803] NAD83(MA11) [804] NAD83(NSRS2007) [805] NAD83(NSRS2007) [806] NAD83(PA11) [807] NAD83(PA11) [808] NDG (Paris) [809] NEA74 Noumea [810] NGN [811] NGO 1948 [812] NGO 1948 (Oslo) [813] NSWC 9Z-2 [814] NTF [815] NTF (Paris) [816] NZGD2000 [817] NZGD2000 [818] NZGD2000 [819] NZGD49 [820] Nahrwan 1934 [821] Nahrwan 1967 [822] Nakhl-e Ghanem [823] Naparima 1955 [824] Naparima 1972 [825] Nepal 1981 [826] New Beijing [827] Nord Sahara 1959 [828] Nord Sahara 1959 (Paris) [829] Nouakchott 1965 [830] ONGD14 [831] ONGD14 [832] OS(SN)80 [833] OSGB 1936 [834] OSGB70 [835] OSNI 1952 [836] Observatario [837] Ocotepeque 1935 [838] Old Hawaiian [839] PD/83 [840] PNG94 [841] PNG94 [842] POSGAR [843] POSGAR 2007 [844] POSGAR 2007 [845] POSGAR 94 [846] POSGAR 94 [847] POSGAR 98 [848] POSGAR 98 [849] POSGAR 98 [850] PRS92 [851] PRS92 [852] PSAD56 [853] PSD93 [854] PTRA08 [855] PTRA08 [856] PZ-90 [857] PZ-90 [858] PZ-90.02 [859] PZ-90.02 [860] PZ-90.11 [861] PZ-90.11 [862] Padang [863] Padang (Jakarta) [864] Palestine 1923 [865] Pampa del Castillo [866] Panama-Colon 1911 [867] Perroud 1950 [868] Peru96 [869] Peru96 [870] Petrels 1972 [871] Phoenix Islands 1966 [872] Pico de las Nieves 1984 [873] Pitcairn 1967 [874] Pitcairn 2006 [875] Point 58 [876] Pointe Noire [877] Popular Visualisation CRS [878] Porto Santo [879] Porto Santo 1995 [880] Principe [881] Puerto Rico [882] Pulkovo 1942 [883] Pulkovo 1942(58) [884] Pulkovo 1942(83) [885] Pulkovo 1995 [886] QND95 [887] Qatar 1948 [888] Qatar 1974 [889] Qornoq [890] Qornoq 1927 [891] RD/83 [892] RDN2008 [893] RDN2008 [894] REGCAN95 [895] REGCAN95 [896] REGVEN [897] REGVEN [898] REGVEN [899] RGAF09 [900] RGAF09 [901] RGAF09 (lon-lat) [902] RGAF09 (lon-lat) [903] RGF93 [904] RGF93 [905] RGF93 [906] RGF93 (lon-lat) [907] RGF93 (lon-lat) [908] RGFG95 [909] RGFG95 [910] RGFG95 [911] RGFG95 (lon-lat) [912] RGFG95 (lon-lat) [913] RGM04 [914] RGM04 [915] RGM04 (lon-lat) [916] RGM04 (lon-lat) [917] RGNC 1991 [918] RGNC 1991 [919] RGNC91-93 [920] RGNC91-93 [921] RGNC91-93 [922] RGPF [923] RGPF [924] RGR92 [925] RGR92 [926] RGR92 [927] RGR92 (lon-lat) [928] RGR92 (lon-lat) [929] RGRDC 2005 [930] RGRDC 2005 [931] RGSPM06 [932] RGSPM06 [933] RGSPM06 (lon-lat) [934] RGSPM06 (lon-lat) [935] RGTAAF07 [936] RGTAAF07 [937] RGTAAF07 (lon-lat) [938] RGTAAF07 (lon-lat) [939] RGTAAF07 (lon-lat) [940] RGWF96 [941] RGWF96 [942] RGWF96 (lon-lat) [943] RGWF96 (lon-lat) [944] RRAF 1991 [945] RRAF 1991 [946] RRAF 1991 [947] RRAF 1991 [948] RRAF 1991 [949] RSRGD2000 [950] RSRGD2000 [951] RT38 [952] RT38 (Stockholm) [953] RT90 [954] Rassadiran [955] Reunion 1947 [956] Reykjavik 1900 [957] S-JTSK [958] S-JTSK (Ferro) [959] S-JTSK [JTSK03] [960] S-JTSK/05 [961] S-JTSK/05 (Ferro) [962] SAD69 [963] SAD69 [964] SAD69(96) [965] SHGD2015 [966] SHGD2015 [967] SIGD61 [968] SIRGAS 1995 [969] SIRGAS 1995 [970] SIRGAS 1995 [971] SIRGAS 2000 [972] SIRGAS 2000 [973] SIRGAS-Chile [974] SIRGAS-Chile [975] SIRGAS-ROU98 [976] SIRGAS-ROU98 [977] SIRGAS_ES2007.8 [978] SIRGAS_ES2007.8 [979] SLD99 [980] SRB_ETRS89 [981] SRB_ETRS89 [982] SREF98 [983] SREF98 [984] ST71 Belep [985] ST84 Ile des Pins [986] ST87 Ouvea [987] ST87 Ouvea [988] SVY21 [989] SWEREF99 [990] SWEREF99 [991] SWEREF99 [992] Saint Pierre et Miquelon 1950 [993] Samboja [994] Santo 1965 [995] Sao Tome [996] Sapper Hill 1943 [997] Schwarzeck [998] Scoresbysund 1952 [999] Segara [1000] Segara (Jakarta) [1001] Segora [1002] Selvagem Grande [1003] Serindung [1004] Sibun Gorge 1922 [1005] Sierra Leone 1924 [1006] Sierra Leone 1968 [1007] Slovenia 1996 [1008] Slovenia 1996 [1009] Solomon 1968 [1010] South East Island 1943 [1011] South Georgia 1968 [1012] South Yemen [1013] St. George Island [1014] St. Helena Tritan [1015] St. Helena Tritan [1016] St. Kitts 1955 [1017] St. Lawrence Island [1018] St. Lucia 1955 [1019] St. Paul Island [1020] St. Stephen (Ferro) [1021] St. Vincent 1945 [1022] Sudan [1023] Sun_2015 [1024] TC(1948) [1025] TGD2005 [1026] TGD2005 [1027] TM65 [1028] TM75 [1029] TUREF [1030] TUREF [1031] TWD67 [1032] TWD97 [1033] TWD97 [1034] Tahaa 54 [1035] Tahiti 52 [1036] Tahiti 79 [1037] Tananarive [1038] Tananarive (Paris) [1039] Tern Island 1961 [1040] Tete [1041] Timbalai 1948 [1042] Tokyo [1043] Tokyo 1892 [1044] Trinidad 1903 [1045] Tristan 1968 [1046] UCS-2000 [1047] UCS-2000 [1048] Unknown datum based upon the Airy 1830 ellipsoid [1049] Unknown datum based upon the Airy Modified 1849 ellipsoid [1050] Unknown datum based upon the Australian National Spheroid [1051] Unknown datum based upon the Authalic Sphere [1052] Unknown datum based upon the Average Terrestrial System 1977 ellipsoid [1053] Unknown datum based upon the Bessel 1841 ellipsoid [1054] Unknown datum based upon the Bessel Modified ellipsoid [1055] Unknown datum based upon the Bessel Namibia ellipsoid [1056] Unknown datum based upon the Clarke 1858 ellipsoid [1057] Unknown datum based upon the Clarke 1866 Michigan ellipsoid [1058] Unknown datum based upon the Clarke 1866 ellipsoid [1059] Unknown datum based upon the Clarke 1880 (Arc) ellipsoid [1060] Unknown datum based upon the Clarke 1880 (Benoit) ellipsoid [1061] Unknown datum based upon the Clarke 1880 (IGN) ellipsoid [1062] Unknown datum based upon the Clarke 1880 (RGS) ellipsoid [1063] Unknown datum based upon the Clarke 1880 (SGA 1922) ellipsoid [1064] Unknown datum based upon the Clarke 1880 ellipsoid [1065] Unknown datum based upon the Everest (1830 Definition) ellipsoid [1066] Unknown datum based upon the Everest 1830 (1937 Adjustment) ellipsoid [1067] Unknown datum based upon the Everest 1830 (1962 Definition) ellipsoid [1068] Unknown datum based upon the Everest 1830 (1967 Definition) ellipsoid [1069] Unknown datum based upon the Everest 1830 (1975 Definition) ellipsoid [1070] Unknown datum based upon the Everest 1830 Modified ellipsoid [1071] Unknown datum based upon the GEM 10C ellipsoid [1072] Unknown datum based upon the GRS 1967 ellipsoid [1073] Unknown datum based upon the GRS 1980 ellipsoid [1074] Unknown datum based upon the Helmert 1906 ellipsoid [1075] Unknown datum based upon the Indonesian National Spheroid [1076] Unknown datum based upon the International 1924 ellipsoid [1077] Unknown datum based upon the Krassowsky 1940 ellipsoid [1078] Unknown datum based upon the NWL 9D ellipsoid [1079] Unknown datum based upon the OSU86F ellipsoid [1080] Unknown datum based upon the OSU91A ellipsoid [1081] Unknown datum based upon the Plessis 1817 ellipsoid [1082] Unknown datum based upon the Struve 1860 ellipsoid [1083] Unknown datum based upon the WGS 72 ellipsoid [1084] Unknown datum based upon the WGS 84 ellipsoid [1085] Unknown datum based upon the War Office ellipsoid [1086] Unspecified datum based upon the Clarke 1866 Authalic Sphere [1087] Unspecified datum based upon the GRS 1980 Authalic Sphere [1088] Unspecified datum based upon the Hughes 1980 ellipsoid [1089] Unspecified datum based upon the International 1924 Authalic Sphere [1090] VN-2000 [1091] Vanua Levu 1915 [1092] Vientiane 1982 [1093] Viti Levu 1912 [1094] Viti Levu 1916 [1095] Voirol 1875 [1096] Voirol 1875 (Paris) [1097] Voirol 1879 [1098] Voirol 1879 (Paris) [1099] WGS 66 [1100] WGS 66 [1101] WGS 72 [1102] WGS 72 [1103] WGS 72BE [1104] WGS 72BE [1105] WGS 84 [1106] WGS 84 [1107] WGS 84 [1108] WGS 84 (3D) [1109] WGS 84 (G1150) [1110] WGS 84 (G1674) [1111] WGS 84 (G1762) [1112] WGS 84 (G730) [1113] WGS 84 (G873) [1114] WGS 84 (Transit) [1115] WGS_1984_(G1150) [1116] WGS_1984_(G1674) [1117] WGS_1984_(G1762) [1118] WGS_1984_(G730) [1119] WGS_1984_(G873) [1120] WGS_1984_(Transit) [1121] Wake Island 1952 [1122] Xian 1980 [1123] Yacare [1124] Yemen NGN96 [1125] Yemen NGN96 [1126] Yemen NGN96 [1127] Yoff [1128] Zanderij [1129] fk89 Default: 0
    - CRS_PROJCS [`choice`] : Projected Coordinate Systems. Available Choices: [0] <select> [1] AGD66 / ACT Standard Grid [2] AGD66 / AMG zone 48 (deprecated) [3] AGD66 / AMG zone 49 [4] AGD66 / AMG zone 50 [5] AGD66 / AMG zone 51 [6] AGD66 / AMG zone 52 [7] AGD66 / AMG zone 53 [8] AGD66 / AMG zone 54 [9] AGD66 / AMG zone 55 [10] AGD66 / AMG zone 56 [11] AGD66 / AMG zone 57 [12] AGD66 / AMG zone 58 [13] AGD66 / Vicgrid66 [14] AGD84 / AMG zone 48 (deprecated) [15] AGD84 / AMG zone 49 [16] AGD84 / AMG zone 50 [17] AGD84 / AMG zone 51 [18] AGD84 / AMG zone 52 [19] AGD84 / AMG zone 53 [20] AGD84 / AMG zone 54 [21] AGD84 / AMG zone 55 [22] AGD84 / AMG zone 56 [23] AGD84 / AMG zone 57 (deprecated) [24] AGD84 / AMG zone 58 (deprecated) [25] AGD_1966_ACT_Grid_AGC_Zone [26] AGD_1966_ISG_54_2 [27] AGD_1966_ISG_54_3 [28] AGD_1966_ISG_55_1 [29] AGD_1966_ISG_55_2 [30] AGD_1966_ISG_55_3 [31] AGD_1966_ISG_56_1 [32] AGD_1966_ISG_56_2 [33] AGD_1966_ISG_56_3 [34] AGD_1966_VICGRID (deprecated) [35] ATF (Paris) / Nord de Guerre [36] ATS77 / MTM Nova Scotia zone 4 [37] ATS77 / MTM Nova Scotia zone 5 [38] ATS77 / New Brunswick Stereographic (ATS77) [39] ATS77 / Prince Edward Isl. Stereographic (ATS77) [40] ATS77 / UTM zone 19N [41] ATS77 / UTM zone 20N [42] Abidjan 1987 / TM 5 NW [43] Abidjan 1987 / UTM zone 29N [44] Abidjan 1987 / UTM zone 30N [45] Accra / Ghana National Grid [46] Accra / TM 1 NW [47] Adindan / UTM zone 35N [48] Adindan / UTM zone 36N [49] Adindan / UTM zone 37N [50] Adindan / UTM zone 38N [51] Afgooye / UTM zone 38N [52] Afgooye / UTM zone 39N [53] Africa_Albers_Equal_Area_Conic [54] Africa_Equidistant_Conic [55] Africa_Lambert_Conformal_Conic [56] Africa_Sinusoidal [57] Ain el Abd / Aramco Lambert [58] Ain el Abd / Bahrain Grid [59] Ain el Abd / UTM zone 36N [60] Ain el Abd / UTM zone 37N [61] Ain el Abd / UTM zone 38N [62] Ain el Abd / UTM zone 39N [63] Ain el Abd / UTM zone 40N [64] Ain_el_Abd_1970_Aramco_Lambert_2 [65] Albanian 1987 / Gauss Kruger zone 4 (deprecated) [66] Albanian 1987 / Gauss-Kruger zone 4 [67] American Samoa 1962 / American Samoa Lambert [68] American Samoa 1962 / American Samoa Lambert (deprecated) [69] American Samoa 1962 / American Samoa Lambert (deprecated) [70] American_Samoa_1962_StatePlane_American_Samoa_FIPS_5300 [71] American_Samoa_1962_UTM_Zone_2S [72] Amersfoort / RD New [73] Amersfoort / RD Old [74] Anguilla 1957 / British West Indies Grid [75] Antigua 1943 / British West Indies Grid [76] Aratu / UTM zone 22S [77] Aratu / UTM zone 23S [78] Aratu / UTM zone 24S [79] Aratu / UTM zone 25S [80] Arc 1950 / UTM zone 34S [81] Arc 1950 / UTM zone 35S [82] Arc 1950 / UTM zone 36S [83] Arc 1960 / UTM zone 35N [84] Arc 1960 / UTM zone 35S [85] Arc 1960 / UTM zone 36N [86] Arc 1960 / UTM zone 36S [87] Arc 1960 / UTM zone 37N [88] Arc 1960 / UTM zone 37S [89] Asia_Lambert_Conformal_Conic [90] Asia_North_Albers_Equal_Area_Conic [91] Asia_North_Equidistant_Conic [92] Asia_North_Lambert_Conformal_Conic [93] Asia_South_Albers_Equal_Area_Conic [94] Asia_South_Equidistant_Conic [95] Asia_South_Lambert_Conformal_Conic [96] Astro DOS 71 / SHLG71 [97] Astro DOS 71 / UTM zone 30S [98] Azores Central 1948 / UTM zone 26N [99] Azores Central 1995 / UTM zone 26N [100] Azores Occidental 1939 / UTM zone 25N [101] Azores Oriental 1940 / UTM zone 26N [102] Azores Oriental 1995 / UTM zone 26N [103] BDA2000 / Bermuda 2000 National Grid [104] BGS2005 / CCS2005 [105] BGS2005 / UTM zone 34N [106] BGS2005 / UTM zone 34N (N-E) [107] BGS2005 / UTM zone 35N [108] BGS2005 / UTM zone 35N (N-E) [109] BGS2005 / UTM zone 36N [110] BUTM 2010 [111] Bab_South_Palau_Azimuthal_Equidistant [112] Barbados 1938 / Barbados National Grid [113] Barbados 1938 / British West Indies Grid [114] Batavia (Jakarta) / NEIEZ [115] Batavia (Jakarta) / NEIEZ (deprecated) [116] Batavia / NEIEZ [117] Batavia / TM 109 SE [118] Batavia / UTM zone 48S [119] Batavia / UTM zone 49S [120] Batavia / UTM zone 50S [121] Beduaram / TM 13 NE [122] Beijing 1954 / 3-degree Gauss-Kruger CM 102E [123] Beijing 1954 / 3-degree Gauss-Kruger CM 105E [124] Beijing 1954 / 3-degree Gauss-Kruger CM 108E [125] Beijing 1954 / 3-degree Gauss-Kruger CM 111E [126] Beijing 1954 / 3-degree Gauss-Kruger CM 114E [127] Beijing 1954 / 3-degree Gauss-Kruger CM 117E [128] Beijing 1954 / 3-degree Gauss-Kruger CM 120E [129] Beijing 1954 / 3-degree Gauss-Kruger CM 123E [130] Beijing 1954 / 3-degree Gauss-Kruger CM 126E [131] Beijing 1954 / 3-degree Gauss-Kruger CM 129E [132] Beijing 1954 / 3-degree Gauss-Kruger CM 132E [133] Beijing 1954 / 3-degree Gauss-Kruger CM 135E [134] Beijing 1954 / 3-degree Gauss-Kruger CM 75E [135] Beijing 1954 / 3-degree Gauss-Kruger CM 78E [136] Beijing 1954 / 3-degree Gauss-Kruger CM 81E [137] Beijing 1954 / 3-degree Gauss-Kruger CM 84E [138] Beijing 1954 / 3-degree Gauss-Kruger CM 87E [139] Beijing 1954 / 3-degree Gauss-Kruger CM 90E [140] Beijing 1954 / 3-degree Gauss-Kruger CM 93E [141] Beijing 1954 / 3-degree Gauss-Kruger CM 96E [142] Beijing 1954 / 3-degree Gauss-Kruger CM 99E [143] Beijing 1954 / 3-degree Gauss-Kruger zone 25 [144] Beijing 1954 / 3-degree Gauss-Kruger zone 26 [145] Beijing 1954 / 3-degree Gauss-Kruger zone 27 [146] Beijing 1954 / 3-degree Gauss-Kruger zone 28 [147] Beijing 1954 / 3-degree Gauss-Kruger zone 29 [148] Beijing 1954 / 3-degree Gauss-Kruger zone 30 [149] Beijing 1954 / 3-degree Gauss-Kruger zone 31 [150] Beijing 1954 / 3-degree Gauss-Kruger zone 32 [151] Beijing 1954 / 3-degree Gauss-Kruger zone 33 [152] Beijing 1954 / 3-degree Gauss-Kruger zone 34 [153] Beijing 1954 / 3-degree Gauss-Kruger zone 35 [154] Beijing 1954 / 3-degree Gauss-Kruger zone 36 [155] Beijing 1954 / 3-degree Gauss-Kruger zone 37 [156] Beijing 1954 / 3-degree Gauss-Kruger zone 38 [157] Beijing 1954 / 3-degree Gauss-Kruger zone 39 [158] Beijing 1954 / 3-degree Gauss-Kruger zone 40 [159] Beijing 1954 / 3-degree Gauss-Kruger zone 41 [160] Beijing 1954 / 3-degree Gauss-Kruger zone 42 [161] Beijing 1954 / 3-degree Gauss-Kruger zone 43 [162] Beijing 1954 / 3-degree Gauss-Kruger zone 44 [163] Beijing 1954 / 3-degree Gauss-Kruger zone 45 [164] Beijing 1954 / Gauss-Kruger 13N (deprecated) [165] Beijing 1954 / Gauss-Kruger 14N (deprecated) [166] Beijing 1954 / Gauss-Kruger 15N (deprecated) [167] Beijing 1954 / Gauss-Kruger 16N (deprecated) [168] Beijing 1954 / Gauss-Kruger 17N (deprecated) [169] Beijing 1954 / Gauss-Kruger 18N (deprecated) [170] Beijing 1954 / Gauss-Kruger 19N (deprecated) [171] Beijing 1954 / Gauss-Kruger 20N (deprecated) [172] Beijing 1954 / Gauss-Kruger 21N (deprecated) [173] Beijing 1954 / Gauss-Kruger 22N (deprecated) [174] Beijing 1954 / Gauss-Kruger 23N (deprecated) [175] Beijing 1954 / Gauss-Kruger CM 105E [176] Beijing 1954 / Gauss-Kruger CM 111E [177] Beijing 1954 / Gauss-Kruger CM 117E [178] Beijing 1954 / Gauss-Kruger CM 123E [179] Beijing 1954 / Gauss-Kruger CM 129E [180] Beijing 1954 / Gauss-Kruger CM 135E [181] Beijing 1954 / Gauss-Kruger CM 75E [182] Beijing 1954 / Gauss-Kruger CM 81E [183] Beijing 1954 / Gauss-Kruger CM 87E [184] Beijing 1954 / Gauss-Kruger CM 93E [185] Beijing 1954 / Gauss-Kruger CM 99E [186] Beijing 1954 / Gauss-Kruger zone 13 [187] Beijing 1954 / Gauss-Kruger zone 14 [188] Beijing 1954 / Gauss-Kruger zone 15 [189] Beijing 1954 / Gauss-Kruger zone 16 [190] Beijing 1954 / Gauss-Kruger zone 17 [191] Beijing 1954 / Gauss-Kruger zone 18 [192] Beijing 1954 / Gauss-Kruger zone 19 [193] Beijing 1954 / Gauss-Kruger zone 20 [194] Beijing 1954 / Gauss-Kruger zone 21 [195] Beijing 1954 / Gauss-Kruger zone 22 [196] Beijing 1954 / Gauss-Kruger zone 23 [197] Belge 1950 (Brussels) / Belge Lambert 50 [198] Belge 1972 / Belge Lambert 72 [199] Belge 1972 / Belgian Lambert 72 [200] Belge_Lambert_1972_bad_FE_FN [201] Belge_Lambert_2008 (deprecated) [202] Bermuda 1957 / UTM zone 20N [203] Bermuda_2000_National_Grid (deprecated) [204] Bern 1898 (Bern) / LV03C [205] Bissau / UTM zone 28N [206] Bogota 1975 / Colombia Bogota zone [207] Bogota 1975 / Colombia Bogota zone (deprecated) [208] Bogota 1975 / Colombia East [209] Bogota 1975 / Colombia East (deprecated) [210] Bogota 1975 / Colombia East Central zone [211] Bogota 1975 / Colombia East Central zone (deprecated) [212] Bogota 1975 / Colombia West zone [213] Bogota 1975 / Colombia West zone (deprecated) [214] Bogota 1975 / UTM zone 17N (deprecated) [215] Bogota 1975 / UTM zone 18N [216] Bogota_Ciudad_Bogota [217] CGCS2000 / 3-degree Gauss-Kruger CM 102E [218] CGCS2000 / 3-degree Gauss-Kruger CM 105E [219] CGCS2000 / 3-degree Gauss-Kruger CM 108E [220] CGCS2000 / 3-degree Gauss-Kruger CM 111E [221] CGCS2000 / 3-degree Gauss-Kruger CM 114E [222] CGCS2000 / 3-degree Gauss-Kruger CM 117E [223] CGCS2000 / 3-degree Gauss-Kruger CM 120E [224] CGCS2000 / 3-degree Gauss-Kruger CM 123E [225] CGCS2000 / 3-degree Gauss-Kruger CM 126E [226] CGCS2000 / 3-degree Gauss-Kruger CM 129E [227] CGCS2000 / 3-degree Gauss-Kruger CM 132E [228] CGCS2000 / 3-degree Gauss-Kruger CM 135E [229] CGCS2000 / 3-degree Gauss-Kruger CM 75E [230] CGCS2000 / 3-degree Gauss-Kruger CM 78E [231] CGCS2000 / 3-degree Gauss-Kruger CM 81E [232] CGCS2000 / 3-degree Gauss-Kruger CM 84E [233] CGCS2000 / 3-degree Gauss-Kruger CM 87E [234] CGCS2000 / 3-degree Gauss-Kruger CM 90E [235] CGCS2000 / 3-degree Gauss-Kruger CM 93E [236] CGCS2000 / 3-degree Gauss-Kruger CM 96E [237] CGCS2000 / 3-degree Gauss-Kruger CM 99E [238] CGCS2000 / 3-degree Gauss-Kruger zone 25 [239] CGCS2000 / 3-degree Gauss-Kruger zone 26 [240] CGCS2000 / 3-degree Gauss-Kruger zone 27 [241] CGCS2000 / 3-degree Gauss-Kruger zone 28 [242] CGCS2000 / 3-degree Gauss-Kruger zone 29 [243] CGCS2000 / 3-degree Gauss-Kruger zone 30 [244] CGCS2000 / 3-degree Gauss-Kruger zone 31 [245] CGCS2000 / 3-degree Gauss-Kruger zone 32 [246] CGCS2000 / 3-degree Gauss-Kruger zone 33 [247] CGCS2000 / 3-degree Gauss-Kruger zone 34 [248] CGCS2000 / 3-degree Gauss-Kruger zone 35 [249] CGCS2000 / 3-degree Gauss-Kruger zone 36 [250] CGCS2000 / 3-degree Gauss-Kruger zone 37 [251] CGCS2000 / 3-degree Gauss-Kruger zone 38 [252] CGCS2000 / 3-degree Gauss-Kruger zone 39 [253] CGCS2000 / 3-degree Gauss-Kruger zone 40 [254] CGCS2000 / 3-degree Gauss-Kruger zone 41 [255] CGCS2000 / 3-degree Gauss-Kruger zone 42 [256] CGCS2000 / 3-degree Gauss-Kruger zone 43 [257] CGCS2000 / 3-degree Gauss-Kruger zone 44 [258] CGCS2000 / 3-degree Gauss-Kruger zone 45 [259] CGCS2000 / Gauss-Kruger CM 105E [260] CGCS2000 / Gauss-Kruger CM 111E [261] CGCS2000 / Gauss-Kruger CM 117E [262] CGCS2000 / Gauss-Kruger CM 123E [263] CGCS2000 / Gauss-Kruger CM 129E [264] CGCS2000 / Gauss-Kruger CM 135E [265] CGCS2000 / Gauss-Kruger CM 75E [266] CGCS2000 / Gauss-Kruger CM 81E [267] CGCS2000 / Gauss-Kruger CM 87E [268] CGCS2000 / Gauss-Kruger CM 93E [269] CGCS2000 / Gauss-Kruger CM 99E [270] CGCS2000 / Gauss-Kruger zone 13 [271] CGCS2000 / Gauss-Kruger zone 14 [272] CGCS2000 / Gauss-Kruger zone 15 [273] CGCS2000 / Gauss-Kruger zone 16 [274] CGCS2000 / Gauss-Kruger zone 17 [275] CGCS2000 / Gauss-Kruger zone 18 [276] CGCS2000 / Gauss-Kruger zone 19 [277] CGCS2000 / Gauss-Kruger zone 20 [278] CGCS2000 / Gauss-Kruger zone 21 [279] CGCS2000 / Gauss-Kruger zone 22 [280] CGCS2000 / Gauss-Kruger zone 23 [281] CGRS93 / Cyprus Local Transverse Mercator [282] CGRS_1993_LTM [283] CH1903 / LV03 [284] CH1903 / LV03C-G [285] CH1903+ / LV95 [286] CI1971 / Chatham Islands Map Grid [287] CI1979 / Chatham Islands Map Grid [288] COB_NAD83_2007 [289] CR-SIRGAS / CRTM05 [290] CR-SIRGAS / UTM zone 16N [291] CR-SIRGAS / UTM zone 17N [292] CR05 / CRTM05 [293] CRTM05 (deprecated) [294] CSG67 / UTM zone 21N [295] CSG67 / UTM zone 22N [296] Cadastre 1997 / UTM zone 38S [297] Cadastre 1997 / UTM zone 38S (deprecated) [298] Camacupa / TM 11.30 SE [299] Camacupa / TM 12 SE [300] Camacupa / UTM zone 32S [301] Camacupa / UTM zone 33S [302] Campo Inchauspe / Argentina 1 [303] Campo Inchauspe / Argentina 2 [304] Campo Inchauspe / Argentina 3 [305] Campo Inchauspe / Argentina 4 [306] Campo Inchauspe / Argentina 5 [307] Campo Inchauspe / Argentina 6 [308] Campo Inchauspe / Argentina 7 [309] Campo Inchauspe / UTM zone 19S [310] Campo Inchauspe / UTM zone 20S [311] Canada_Albers_Equal_Area_Conic [312] Canada_Lambert_Conformal_Conic [313] Cape / Lo15 [314] Cape / Lo17 [315] Cape / Lo19 [316] Cape / Lo21 [317] Cape / Lo23 [318] Cape / Lo25 [319] Cape / Lo27 [320] Cape / Lo29 [321] Cape / Lo31 [322] Cape / Lo33 [323] Cape / UTM zone 34S [324] Cape / UTM zone 35S [325] Cape / UTM zone 36S (deprecated) [326] Cape_Lo15 [327] Cape_Lo17 [328] Cape_Lo19 [329] Cape_Lo21 [330] Cape_Lo23 [331] Cape_Lo25 [332] Cape_Lo27 [333] Cape_Lo29 [334] Cape_Lo31 [335] Cape_Lo33 [336] Carthage (Paris) / Tunisia Mining Grid [337] Carthage / Nord Tunisie [338] Carthage / Sud Tunisie [339] Carthage / TM 11 NE [340] Carthage / UTM zone 32N [341] Cayman Islands National Grid 2011 [342] Cayman Islands National Grid 2011 (deprecated) [343] Chatham_Islands_1979_Map_Grid (deprecated) [344] Chos Malal 1914 / Argentina 2 [345] Chua / UTM zone 23S [346] Chua_UTM_Zone_23S (deprecated) [347] Colombia_West_West_Zone [348] Combani 1950 / UTM zone 38S [349] Conakry 1905 / UTM zone 28N [350] Conakry 1905 / UTM zone 29N [351] Corrego Alegre 1961 / UTM zone 21S [352] Corrego Alegre 1961 / UTM zone 22S [353] Corrego Alegre 1961 / UTM zone 23S [354] Corrego Alegre 1961 / UTM zone 24S [355] Corrego Alegre 1970-72 / UTM zone 21S [356] Corrego Alegre 1970-72 / UTM zone 22S [357] Corrego Alegre 1970-72 / UTM zone 23S [358] Corrego Alegre 1970-72 / UTM zone 24S [359] Corrego Alegre 1970-72 / UTM zone 25S [360] County ST74 [361] D48_Slovenia_TM [362] DB_REF / 3-degree Gauss-Kruger zone 2 (E-N) [363] DB_REF / 3-degree Gauss-Kruger zone 3 (E-N) [364] DB_REF / 3-degree Gauss-Kruger zone 4 (E-N) [365] DB_REF / 3-degree Gauss-Kruger zone 5 (E-N) [366] DGN95 / Indonesia TM-3 zone 46.2 [367] DGN95 / Indonesia TM-3 zone 47.1 [368] DGN95 / Indonesia TM-3 zone 47.2 [369] DGN95 / Indonesia TM-3 zone 48.1 [370] DGN95 / Indonesia TM-3 zone 48.2 [371] DGN95 / Indonesia TM-3 zone 49.1 [372] DGN95 / Indonesia TM-3 zone 49.2 [373] DGN95 / Indonesia TM-3 zone 50.1 [374] DGN95 / Indonesia TM-3 zone 50.2 [375] DGN95 / Indonesia TM-3 zone 51.1 [376] DGN95 / Indonesia TM-3 zone 51.2 [377] DGN95 / Indonesia TM-3 zone 52.1 [378] DGN95 / Indonesia TM-3 zone 52.2 [379] DGN95 / Indonesia TM-3 zone 53.1 [380] DGN95 / Indonesia TM-3 zone 53.2 [381] DGN95 / Indonesia TM-3 zone 54.1 [382] DGN95 / UTM zone 46N [383] DGN95 / UTM zone 47N [384] DGN95 / UTM zone 47S [385] DGN95 / UTM zone 48N [386] DGN95 / UTM zone 48S [387] DGN95 / UTM zone 49N [388] DGN95 / UTM zone 49S [389] DGN95 / UTM zone 50N [390] DGN95 / UTM zone 50S [391] DGN95 / UTM zone 51N [392] DGN95 / UTM zone 51S [393] DGN95 / UTM zone 52N [394] DGN95 / UTM zone 52S [395] DGN95 / UTM zone 53S [396] DGN95 / UTM zone 54S [397] DHDN / 3-degree Gauss zone 1 (deprecated) [398] DHDN / 3-degree Gauss zone 2 (deprecated) [399] DHDN / 3-degree Gauss zone 3 (deprecated) [400] DHDN / 3-degree Gauss zone 4 (deprecated) [401] DHDN / 3-degree Gauss zone 5 (deprecated) [402] DHDN / 3-degree Gauss-Kruger zone 1 [403] DHDN / 3-degree Gauss-Kruger zone 1 (E-N) [404] DHDN / 3-degree Gauss-Kruger zone 2 [405] DHDN / 3-degree Gauss-Kruger zone 2 (E-N) [406] DHDN / 3-degree Gauss-Kruger zone 3 [407] DHDN / 3-degree Gauss-Kruger zone 3 (E-N) [408] DHDN / 3-degree Gauss-Kruger zone 4 [409] DHDN / 3-degree Gauss-Kruger zone 4 (E-N) [410] DHDN / 3-degree Gauss-Kruger zone 5 [411] DHDN / 3-degree Gauss-Kruger zone 5 (E-N) [412] DHDN / Soldner Berlin [413] DRUKREF 03 / Bhutan National Grid [414] DRUKREF 03 / Bumthang TM [415] DRUKREF 03 / Chhukha TM [416] DRUKREF 03 / Dagana TM [417] DRUKREF 03 / Gasa TM [418] DRUKREF 03 / Ha TM [419] DRUKREF 03 / Lhuentse TM [420] DRUKREF 03 / Mongar TM [421] DRUKREF 03 / Paro TM [422] DRUKREF 03 / Pemagatshel TM [423] DRUKREF 03 / Punakha TM [424] DRUKREF 03 / Samdrup Jongkhar TM [425] DRUKREF 03 / Samtse TM [426] DRUKREF 03 / Sarpang TM [427] DRUKREF 03 / Thimphu TM [428] DRUKREF 03 / Trashigang TM [429] DRUKREF 03 / Trongsa TM [430] DRUKREF 03 / Tsirang TM [431] DRUKREF 03 / Wangdue Phodrang TM [432] DRUKREF 03 / Yangtse TM [433] DRUKREF 03 / Zhemgang TM [434] Dabola 1981 / UTM zone 28N [435] Dabola 1981 / UTM zone 28N (deprecated) [436] Dabola 1981 / UTM zone 29N [437] Dabola 1981 / UTM zone 29N (deprecated) [438] Datum 73 / Modified Portuguese Grid [439] Datum 73 / Modified Portuguese Grid (deprecated) [440] Datum 73 / UTM zone 29N [441] Datum_73_Hayford_Gauss_IGeoE [442] Datum_73_Hayford_Gauss_IPCC [443] Dealul Piscului 1930 / Stereo 33 [444] Dealul Piscului 1970/ Stereo 70 (deprecated) [445] Deir ez Zor / Levant Stereographic [446] Deir ez Zor / Levant Zone [447] Deir ez Zor / Syria Lambert [448] Dominica 1945 / British West Indies Grid [449] Douala / UTM zone 32N (deprecated) [450] Douala 1948 / AEF west [451] Douala 1948 / AOF west (deprecated) [452] ED50 / 3-degree Gauss-Kruger zone 10 [453] ED50 / 3-degree Gauss-Kruger zone 11 [454] ED50 / 3-degree Gauss-Kruger zone 12 [455] ED50 / 3-degree Gauss-Kruger zone 13 [456] ED50 / 3-degree Gauss-Kruger zone 14 [457] ED50 / 3-degree Gauss-Kruger zone 15 [458] ED50 / 3-degree Gauss-Kruger zone 9 [459] ED50 / France EuroLambert (deprecated) [460] ED50 / Iraq National Grid [461] ED50 / Jordan TM [462] ED50 / SPBA LCC [463] ED50 / TM 0 N [464] ED50 / TM 5 NE [465] ED50 / TM 6 NE [466] ED50 / TM27 [467] ED50 / TM30 [468] ED50 / TM33 [469] ED50 / TM36 [470] ED50 / TM39 [471] ED50 / TM42 [472] ED50 / TM45 [473] ED50 / UTM zone 28N [474] ED50 / UTM zone 29N [475] ED50 / UTM zone 30N [476] ED50 / UTM zone 31N [477] ED50 / UTM zone 32N [478] ED50 / UTM zone 33N [479] ED50 / UTM zone 34N [480] ED50 / UTM zone 35N [481] ED50 / UTM zone 36N [482] ED50 / UTM zone 37N [483] ED50 / UTM zone 38N [484] ED50(ED77) / UTM zone 38N [485] ED50(ED77) / UTM zone 39N [486] ED50(ED77) / UTM zone 40N [487] ED50(ED77) / UTM zone 41N [488] ED_1950_Turkey_10 [489] ED_1950_Turkey_10 (deprecated) [490] ED_1950_Turkey_11 [491] ED_1950_Turkey_11 (deprecated) [492] ED_1950_Turkey_12 [493] ED_1950_Turkey_12 (deprecated) [494] ED_1950_Turkey_13 [495] ED_1950_Turkey_13 (deprecated) [496] ED_1950_Turkey_14 [497] ED_1950_Turkey_14 (deprecated) [498] ED_1950_Turkey_15 [499] ED_1950_Turkey_15 (deprecated) [500] ED_1950_Turkey_9 [501] ED_1950_Turkey_9 (deprecated) [502] ELD79 / Libya zone 10 [503] ELD79 / Libya zone 11 [504] ELD79 / Libya zone 12 [505] ELD79 / Libya zone 13 [506] ELD79 / Libya zone 5 [507] ELD79 / Libya zone 6 [508] ELD79 / Libya zone 7 [509] ELD79 / Libya zone 8 [510] ELD79 / Libya zone 9 [511] ELD79 / TM 12 NE [512] ELD79 / UTM zone 32N [513] ELD79 / UTM zone 33N [514] ELD79 / UTM zone 34N [515] ELD79 / UTM zone 35N [516] EMEP_150_Kilometer_Grid [517] EMEP_50_Kilometer_Grid [518] ETRF_1989_TM_Baltic_1993 (deprecated) [519] ETRF_1989_UTM_Zone_28N (deprecated) [520] ETRF_1989_UTM_Zone_29N (deprecated) [521] ETRF_1989_UTM_Zone_30N (deprecated) [522] ETRF_1989_UTM_Zone_31N (deprecated) [523] ETRF_1989_UTM_Zone_32N (deprecated) [524] ETRF_1989_UTM_Zone_33N (deprecated) [525] ETRF_1989_UTM_Zone_34N (deprecated) [526] ETRF_1989_UTM_Zone_35N (deprecated) [527] ETRF_1989_UTM_Zone_36N (deprecated) [528] ETRF_1989_UTM_Zone_37N (deprecated) [529] ETRF_1989_UTM_Zone_38N (deprecated) [530] ETRS89 / Albania LCC 2010 [531] ETRS89 / Albania TM 2010 [532] ETRS89 / Austria Lambert [533] ETRS89 / Belgian Lambert 2005 [534] ETRS89 / Belgian Lambert 2008 [535] ETRS89 / DKTM1 [536] ETRS89 / DKTM2 [537] ETRS89 / DKTM3 [538] ETRS89 / DKTM4 [539] ETRS89 / EPSG Arctic zone 2-22 [540] ETRS89 / EPSG Arctic zone 3-11 [541] ETRS89 / EPSG Arctic zone 4-26 [542] ETRS89 / EPSG Arctic zone 4-28 [543] ETRS89 / EPSG Arctic zone 5-11 [544] ETRS89 / EPSG Arctic zone 5-13 [545] ETRS89 / EPSG Arctic zone 5-47 [546] ETRS89 / ETRS-GK19FIN [547] ETRS89 / ETRS-GK20FIN [548] ETRS89 / ETRS-GK21FIN [549] ETRS89 / ETRS-GK22FIN [550] ETRS89 / ETRS-GK23FIN [551] ETRS89 / ETRS-GK24FIN [552] ETRS89 / ETRS-GK25FIN [553] ETRS89 / ETRS-GK26FIN [554] ETRS89 / ETRS-GK27FIN [555] ETRS89 / ETRS-GK28FIN [556] ETRS89 / ETRS-GK29FIN [557] ETRS89 / ETRS-GK30FIN [558] ETRS89 / ETRS-GK31FIN [559] ETRS89 / Faroe TM [560] ETRS89 / GK19FIN [561] ETRS89 / GK20FIN [562] ETRS89 / GK21FIN [563] ETRS89 / GK22FIN [564] ETRS89 / GK23FIN [565] ETRS89 / GK24FIN [566] ETRS89 / GK25FIN [567] ETRS89 / GK26FIN [568] ETRS89 / GK27FIN [569] ETRS89 / GK28FIN [570] ETRS89 / GK29FIN [571] ETRS89 / GK30FIN [572] ETRS89 / GK31FIN [573] ETRS89 / Gauss-Kruger CM 9E [574] ETRS89 / Guernsey Grid [575] ETRS89 / Jersey Transverse Mercator [576] ETRS89 / Kp2000 Bornholm [577] ETRS89 / Kp2000 Jutland [578] ETRS89 / Kp2000 Zealand [579] ETRS89 / LAEA Europe [580] ETRS89 / LCC Europe [581] ETRS89 / LCC Germany (E-N) [582] ETRS89 / LCC Germany (N-E) [583] ETRS89 / NTM zone 10 [584] ETRS89 / NTM zone 10 (deprecated) [585] ETRS89 / NTM zone 11 [586] ETRS89 / NTM zone 11 (deprecated) [587] ETRS89 / NTM zone 12 [588] ETRS89 / NTM zone 12 (deprecated) [589] ETRS89 / NTM zone 13 [590] ETRS89 / NTM zone 13 (deprecated) [591] ETRS89 / NTM zone 14 [592] ETRS89 / NTM zone 14 (deprecated) [593] ETRS89 / NTM zone 15 [594] ETRS89 / NTM zone 15 (deprecated) [595] ETRS89 / NTM zone 16 [596] ETRS89 / NTM zone 16 (deprecated) [597] ETRS89 / NTM zone 17 [598] ETRS89 / NTM zone 17 (deprecated) [599] ETRS89 / NTM zone 18 [600] ETRS89 / NTM zone 18 (deprecated) [601] ETRS89 / NTM zone 19 [602] ETRS89 / NTM zone 19 (deprecated) [603] ETRS89 / NTM zone 20 [604] ETRS89 / NTM zone 20 (deprecated) [605] ETRS89 / NTM zone 21 [606] ETRS89 / NTM zone 21 (deprecated) [607] ETRS89 / NTM zone 22 [608] ETRS89 / NTM zone 22 (deprecated) [609] ETRS89 / NTM zone 23 [610] ETRS89 / NTM zone 23 (deprecated) [611] ETRS89 / NTM zone 24 [612] ETRS89 / NTM zone 24 (deprecated) [613] ETRS89 / NTM zone 25 [614] ETRS89 / NTM zone 25 (deprecated) [615] ETRS89 / NTM zone 26 [616] ETRS89 / NTM zone 26 (deprecated) [617] ETRS89 / NTM zone 27 [618] ETRS89 / NTM zone 27 (deprecated) [619] ETRS89 / NTM zone 28 [620] ETRS89 / NTM zone 28 (deprecated) [621] ETRS89 / NTM zone 29 [622] ETRS89 / NTM zone 29 (deprecated) [623] ETRS89 / NTM zone 30 [624] ETRS89 / NTM zone 30 (deprecated) [625] ETRS89 / NTM zone 5 [626] ETRS89 / NTM zone 5 (deprecated) [627] ETRS89 / NTM zone 6 [628] ETRS89 / NTM zone 6 (deprecated) [629] ETRS89 / NTM zone 7 [630] ETRS89 / NTM zone 7 (deprecated) [631] ETRS89 / NTM zone 8 [632] ETRS89 / NTM zone 8 (deprecated) [633] ETRS89 / NTM zone 9 [634] ETRS89 / NTM zone 9 (deprecated) [635] ETRS89 / Poland CS2000 zone 5 [636] ETRS89 / Poland CS2000 zone 6 [637] ETRS89 / Poland CS2000 zone 7 [638] ETRS89 / Poland CS2000 zone 8 [639] ETRS89 / Poland CS92 [640] ETRS89 / Portugal TM06 [641] ETRS89 / TM 30 NE [642] ETRS89 / TM Baltic93 [643] ETRS89 / TM26 (deprecated) [644] ETRS89 / TM27 (deprecated) [645] ETRS89 / TM35FIN(E,N) [646] ETRS89 / TM35FIN(N,E) [647] ETRS89 / TM38 (deprecated) [648] ETRS89 / TM39 (deprecated) [649] ETRS89 / UTM zone 28N [650] ETRS89 / UTM zone 28N (N-E) [651] ETRS89 / UTM zone 29N [652] ETRS89 / UTM zone 29N (N-E) [653] ETRS89 / UTM zone 30N [654] ETRS89 / UTM zone 30N (N-E) [655] ETRS89 / UTM zone 31N [656] ETRS89 / UTM zone 31N (N-E) [657] ETRS89 / UTM zone 31N (N-zE) [658] ETRS89 / UTM zone 31N (zE-N) [659] ETRS89 / UTM zone 32N [660] ETRS89 / UTM zone 32N (N-E) [661] ETRS89 / UTM zone 32N (N-zE) [662] ETRS89 / UTM zone 32N (zE-N) [663] ETRS89 / UTM zone 33N [664] ETRS89 / UTM zone 33N (N-E) [665] ETRS89 / UTM zone 33N (N-zE) [666] ETRS89 / UTM zone 33N (zE-N) [667] ETRS89 / UTM zone 34N [668] ETRS89 / UTM zone 34N (N-E) [669] ETRS89 / UTM zone 35N [670] ETRS89 / UTM zone 35N (N-E) [671] ETRS89 / UTM zone 36N [672] ETRS89 / UTM zone 36N (N-E) [673] ETRS89 / UTM zone 37N [674] ETRS89 / UTM zone 37N (N-E) [675] ETRS89 / UTM zone 38N (deprecated) [676] ETRS_1989_DKTM1 (deprecated) [677] ETRS_1989_DKTM2 (deprecated) [678] ETRS_1989_DKTM3 (deprecated) [679] ETRS_1989_DKTM4 (deprecated) [680] ETRS_1989_Kosovo_Grid [681] ETRS_1989_Slovenia_TM [682] ETRS_1989_UTM_Zone_26N [683] ETRS_1989_UTM_Zone_27N [684] ETRS_1989_UTM_Zone_32N_7stellen [685] ETRS_1989_UTM_Zone_32N_8stellen [686] ETRS_1989_UTM_Zone_33N_7stellen [687] ETRS_1989_UTM_Zone_33N_8stellen [688] ETRS_1989_UTM_Zone_39N [689] ETRS_1989_UTM_Zone_N32 (deprecated) [690] ETRS_1989_UWPP_1992 [691] ETRS_1989_UWPP_2000_PAS_5 [692] ETRS_1989_UWPP_2000_PAS_6 [693] ETRS_1989_UWPP_2000_PAS_7 [694] ETRS_1989_UWPP_2000_PAS_8 [695] EUREF_FIN_TM35FIN (deprecated) [696] Egypt 1907 / Blue Belt [697] Egypt 1907 / Extended Purple Belt [698] Egypt 1907 / Purple Belt [699] Egypt 1907 / Red Belt [700] Egypt Gulf of Suez S-650 TL / Red Belt [701] Estonian Coordinate System of 1992 [702] Estonian Coordinate System of 1997 [703] Europe_Albers_Equal_Area_Conic [704] Europe_Equidistant_Conic [705] Europe_Lambert_Conformal_Conic [706] Everest_Modified_1969_RSO_Malaya_Meters [707] FD54 / UTM zone 29N [708] FD58 / Iraq zone [709] FEH2010 / Fehmarnbelt TM [710] Fahud / UTM zone 39N [711] Fahud / UTM zone 40N [712] Fatu Iva 72 / UTM zone 7S [713] Fiji 1956 / UTM zone 1S [714] Fiji 1956 / UTM zone 60S [715] Fiji 1986 / Fiji Map Grid [716] Fiji 1986 / Fiji Map Grid (deprecated) [717] Fort Marigot / UTM zone 20N [718] GDA2020 / ALB2020 [719] GDA2020 / BCG2020 [720] GDA2020 / BIO2020 [721] GDA2020 / BRO2020 [722] GDA2020 / CARN2020 [723] GDA2020 / CIG2020 [724] GDA2020 / CKIG2020 [725] GDA2020 / COL2020 [726] GDA2020 / ESP2020 [727] GDA2020 / EXM2020 [728] GDA2020 / GA LCC [729] GDA2020 / GCG2020 [730] GDA2020 / GOLD2020 [731] GDA2020 / JCG2020 [732] GDA2020 / KALB2020 [733] GDA2020 / KAR2020 [734] GDA2020 / KUN2020 [735] GDA2020 / LCG2020 [736] GDA2020 / MGA zone 46 [737] GDA2020 / MGA zone 47 [738] GDA2020 / MGA zone 48 [739] GDA2020 / MGA zone 49 [740] GDA2020 / MGA zone 50 [741] GDA2020 / MGA zone 51 [742] GDA2020 / MGA zone 52 [743] GDA2020 / MGA zone 53 [744] GDA2020 / MGA zone 54 [745] GDA2020 / MGA zone 55 [746] GDA2020 / MGA zone 56 [747] GDA2020 / MGA zone 57 [748] GDA2020 / MGA zone 58 [749] GDA2020 / MGA zone 59 [750] GDA2020 / MRCG2020 [751] GDA2020 / NSW Lambert [752] GDA2020 / PCG2020 [753] GDA2020 / PHG2020 [754] GDA2020 / SA Lambert [755] GDA2020 / Vicgrid [756] GDA94 / Australian Albers [757] GDA94 / BCSG02 [758] GDA94 / CIG94 [759] GDA94 / CKIG94 [760] GDA94 / Geoscience Australia Lambert [761] GDA94 / MGA zone 41 (deprecated) [762] GDA94 / MGA zone 42 (deprecated) [763] GDA94 / MGA zone 43 (deprecated) [764] GDA94 / MGA zone 44 (deprecated) [765] GDA94 / MGA zone 46 [766] GDA94 / MGA zone 47 [767] GDA94 / MGA zone 48 [768] GDA94 / MGA zone 49 [769] GDA94 / MGA zone 50 [770] GDA94 / MGA zone 51 [771] GDA94 / MGA zone 52 [772] GDA94 / MGA zone 53 [773] GDA94 / MGA zone 54 [774] GDA94 / MGA zone 55 [775] GDA94 / MGA zone 56 [776] GDA94 / MGA zone 57 [777] GDA94 / MGA zone 58 [778] GDA94 / MGA zone 59 [779] GDA94 / NSW Lambert [780] GDA94 / SA Lambert [781] GDA94 / Vicgrid [782] GDA94 / WEIPA94 [783] GDA_1994_Perth_Coastal_Grid_1994 [784] GDA_1994_South_Australia_Lambert (deprecated) [785] GDA_1994_VICGRID94 (deprecated) [786] GDBD2009 / Brunei BRSO [787] GDBD2009_GEORSO (deprecated) [788] GDM2000 / East Malaysia BRSO [789] GDM2000 / Johor Grid [790] GDM2000 / Kedah and Perlis Grid [791] GDM2000 / Kelantan Grid [792] GDM2000 / Pahang Grid [793] GDM2000 / Peninsula RSO [794] GDM2000 / Perak Grid [795] GDM2000 / Pinang Grid [796] GDM2000 / Selangor Grid [797] GDM2000 / Sembilan and Melaka Grid [798] GDM2000 / Terengganu Grid [799] GGRS87 / Greek Grid [800] GOES-16_East_ABI_Fixed_Grid_ITRF2000_incorrect_GCS (deprecated) [801] GOES-16_East_ABI_Fixed_Grid_ITRF2008 [802] GR96 / EPSG Arctic zone 1-25 [803] GR96 / EPSG Arctic zone 2-18 [804] GR96 / EPSG Arctic zone 2-20 [805] GR96 / EPSG Arctic zone 3-29 [806] GR96 / EPSG Arctic zone 3-31 [807] GR96 / EPSG Arctic zone 3-33 [808] GR96 / EPSG Arctic zone 4-20 [809] GR96 / EPSG Arctic zone 4-22 [810] GR96 / EPSG Arctic zone 4-24 [811] GR96 / EPSG Arctic zone 5-41 [812] GR96 / EPSG Arctic zone 5-43 [813] GR96 / EPSG Arctic zone 5-45 [814] GR96 / EPSG Arctic zone 6-26 [815] GR96 / EPSG Arctic zone 6-28 [816] GR96 / EPSG Arctic zone 6-30 [817] GR96 / EPSG Arctic zone 7-11 [818] GR96 / EPSG Arctic zone 7-13 [819] GR96 / EPSG Arctic zone 8-20 [820] GR96 / EPSG Arctic zone 8-22 [821] GR96 / UTM zone 18N [822] GR96 / UTM zone 19N [823] GR96 / UTM zone 20N [824] GR96 / UTM zone 21N [825] GR96 / UTM zone 22N [826] GR96 / UTM zone 23N [827] GR96 / UTM zone 24N [828] GR96 / UTM zone 25N [829] GR96 / UTM zone 26N [830] GR96 / UTM zone 27N [831] GR96 / UTM zone 28N [832] GR96 / UTM zone 29N [833] Garoua / UTM zone 33N [834] Garoua / UTM zone 33N (deprecated) [835] Germany_Zone_1 [836] Germany_Zone_2 [837] Germany_Zone_3 [838] Germany_Zone_4 [839] Germany_Zone_5 [840] Graciosa_Base_SW_1948_UTM_Zone_26N [841] Grand Cayman 1959 / UTM zone 17N (deprecated) [842] Grand Cayman National Grid 1959 [843] Grand Comoros / UTM zone 38S [844] Grenada 1953 / British West Indies Grid [845] Guadeloupe 1948 / UTM zone 20N [846] Guam 1963 / Yap Islands [847] Guam_Geodetic_Network_1993 [848] Guam_Geodetic_Triangulation_Network_1963 [849] Guernsey_Grid [850] Gulshan 303 / Bangladesh Transverse Mercator [851] Gusterberg Grid (Ferro) [852] HD72 / EOV [853] HTRS96 / Croatia LCC [854] HTRS96 / Croatia TM [855] HTRS96 / UTM zone 33N [856] HTRS96 / UTM zone 34N [857] Hanoi 1972 / GK 106 NE [858] Hanoi 1972 / Gauss-Kruger zone 18 [859] Hanoi 1972 / Gauss-Kruger zone 19 [860] Hartebeesthoek94 / Lo15 [861] Hartebeesthoek94 / Lo17 [862] Hartebeesthoek94 / Lo19 [863] Hartebeesthoek94 / Lo21 [864] Hartebeesthoek94 / Lo23 [865] Hartebeesthoek94 / Lo25 [866] Hartebeesthoek94 / Lo27 [867] Hartebeesthoek94 / Lo29 [868] Hartebeesthoek94 / Lo31 [869] Hartebeesthoek94 / Lo33 [870] Hartebeesthoek94_Lo15 [871] Hartebeesthoek94_Lo17 [872] Hartebeesthoek94_Lo19 [873] Hartebeesthoek94_Lo19_(E-N) [874] Hartebeesthoek94_Lo21 [875] Hartebeesthoek94_Lo21_(E-N) [876] Hartebeesthoek94_Lo23 [877] Hartebeesthoek94_Lo23_(E-N) [878] Hartebeesthoek94_Lo25 [879] Hartebeesthoek94_Lo25_(E-N) [880] Hartebeesthoek94_Lo27 [881] Hartebeesthoek94_Lo27_(E-N) [882] Hartebeesthoek94_Lo29 [883] Hartebeesthoek94_Lo29_(E-N) [884] Hartebeesthoek94_Lo31 [885] Hartebeesthoek94_Lo31_(E-N) [886] Hartebeesthoek94_Lo33 [887] Hawaii_Albers_Equal_Area_Conic [888] Helle 1954 / Jan Mayen Grid [889] Hito XVIII 1963 / Argentina 2 [890] Hito XVIII 1963 / UTM zone 19S [891] Hjorsey 1955 / UTM zone 26N [892] Hjorsey 1955 / UTM zone 27N [893] Hjorsey 1955 / UTM zone 28N [894] Hong Kong 1963 Grid System [895] Hong Kong 1963 Grid System (deprecated) [896] Hong Kong 1980 Grid System [897] Hong_Kong_1980_Grid (deprecated) [898] Hong_Kong_1980_UTM_Zone_49N [899] Hong_Kong_1980_UTM_Zone_50N [900] Hu Tzu Shan 1950 / UTM zone 51N [901] ID74 / UTM zone 46N [902] ID74 / UTM zone 46S (deprecated) [903] ID74 / UTM zone 47N [904] ID74 / UTM zone 47S [905] ID74 / UTM zone 48N [906] ID74 / UTM zone 48S [907] ID74 / UTM zone 49N [908] ID74 / UTM zone 49S [909] ID74 / UTM zone 50N [910] ID74 / UTM zone 50S [911] ID74 / UTM zone 51N [912] ID74 / UTM zone 51S [913] ID74 / UTM zone 52N [914] ID74 / UTM zone 52S [915] ID74 / UTM zone 53N (deprecated) [916] ID74 / UTM zone 53S [917] ID74 / UTM zone 54S [918] IGC 1962 / Congo TM zone 12 [919] IGC 1962 / Congo TM zone 14 [920] IGC 1962 / Congo TM zone 16 [921] IGC 1962 / Congo TM zone 18 [922] IGC 1962 / Congo TM zone 20 [923] IGC 1962 / Congo TM zone 22 [924] IGC 1962 / Congo TM zone 24 [925] IGC 1962 / Congo TM zone 26 [926] IGC 1962 / Congo TM zone 28 [927] IGC 1962 / Congo TM zone 30 [928] IGCB 1955 / Congo TM zone 12 [929] IGCB 1955 / Congo TM zone 14 [930] IGCB 1955 / Congo TM zone 16 [931] IGCB 1955 / UTM zone 33S [932] IGM95 / UTM zone 32N [933] IGM95 / UTM zone 33N [934] IGN 1962 Kerguelen / UTM zone 42S [935] IGN Astro 1960 / UTM zone 28N [936] IGN Astro 1960 / UTM zone 29N [937] IGN Astro 1960 / UTM zone 30N [938] IGN53 Mare / UTM zone 58S [939] IGN53 Mare / UTM zone 59S [940] IGN56 Lifou / UTM zone 58S [941] IGN63 Hiva Oa / UTM zone 7S [942] IGN72 Grand Terre / UTM zone 58S (deprecated) [943] IGN72 Grande Terre / UTM zone 58S [944] IGN72 Nuku Hiva / UTM zone 7S [945] IGRS / UTM zone 37N [946] IGRS / UTM zone 38N [947] IGRS / UTM zone 39N [948] IRENET95 / Irish Transverse Mercator [949] IRENET95 / UTM zone 29N [950] ISN2004 / LAEA Europe [951] ISN2004 / LCC Europe [952] ISN2004 / Lambert 2004 [953] ISN2016 / Lambert 2016 [954] ISN93 / Lambert 1993 [955] ISN_2004_Lambert_2004 (deprecated) [956] Indian 1954 / UTM zone 46N [957] Indian 1954 / UTM zone 47N [958] Indian 1954 / UTM zone 48N [959] Indian 1960 / TM 106 NE [960] Indian 1960 / UTM zone 48N [961] Indian 1960 / UTM zone 49N [962] Indian 1975 / UTM zone 47N [963] Indian 1975 / UTM zone 48N [964] Israel 1993 / Israeli TM Grid [965] Israeli Grid 05 [966] Israeli Grid 05/12 [967] JAD2001 / Jamaica Metric Grid [968] JAD2001 / UTM zone 17N [969] JAD2001 / UTM zone 18N [970] JAD69 / Jamaica National Grid [971] JAD_2001_Jamaica_Grid (deprecated) [972] JAXA Snow Depth Polar Stereographic North [973] JGD2000 / Japan Plane Rectangular CS I [974] JGD2000 / Japan Plane Rectangular CS II [975] JGD2000 / Japan Plane Rectangular CS III [976] JGD2000 / Japan Plane Rectangular CS IV [977] JGD2000 / Japan Plane Rectangular CS IX [978] JGD2000 / Japan Plane Rectangular CS V [979] JGD2000 / Japan Plane Rectangular CS VI [980] JGD2000 / Japan Plane Rectangular CS VII [981] JGD2000 / Japan Plane Rectangular CS VIII [982] JGD2000 / Japan Plane Rectangular CS X [983] JGD2000 / Japan Plane Rectangular CS XI [984] JGD2000 / Japan Plane Rectangular CS XII [985] JGD2000 / Japan Plane Rectangular CS XIII [986] JGD2000 / Japan Plane Rectangular CS XIV [987] JGD2000 / Japan Plane Rectangular CS XIX [988] JGD2000 / Japan Plane Rectangular CS XV [989] JGD2000 / Japan Plane Rectangular CS XVI [990] JGD2000 / Japan Plane Rectangular CS XVII [991] JGD2000 / Japan Plane Rectangular CS XVIII [992] JGD2000 / UTM zone 51N [993] JGD2000 / UTM zone 52N [994] JGD2000 / UTM zone 53N [995] JGD2000 / UTM zone 54N [996] JGD2000 / UTM zone 55N [997] JGD2011 / Japan Plane Rectangular CS I [998] JGD2011 / Japan Plane Rectangular CS II [999] JGD2011 / Japan Plane Rectangular CS III [1000] JGD2011 / Japan Plane Rectangular CS IV [1001] JGD2011 / Japan Plane Rectangular CS IX [1002] JGD2011 / Japan Plane Rectangular CS V [1003] JGD2011 / Japan Plane Rectangular CS VI [1004] JGD2011 / Japan Plane Rectangular CS VII [1005] JGD2011 / Japan Plane Rectangular CS VIII [1006] JGD2011 / Japan Plane Rectangular CS X [1007] JGD2011 / Japan Plane Rectangular CS XI [1008] JGD2011 / Japan Plane Rectangular CS XII [1009] JGD2011 / Japan Plane Rectangular CS XIII [1010] JGD2011 / Japan Plane Rectangular CS XIV [1011] JGD2011 / Japan Plane Rectangular CS XIX [1012] JGD2011 / Japan Plane Rectangular CS XV [1013] JGD2011 / Japan Plane Rectangular CS XVI [1014] JGD2011 / Japan Plane Rectangular CS XVII [1015] JGD2011 / Japan Plane Rectangular CS XVIII [1016] JGD2011 / UTM zone 51N [1017] JGD2011 / UTM zone 52N [1018] JGD2011 / UTM zone 53N [1019] JGD2011 / UTM zone 54N [1020] JGD2011 / UTM zone 55N [1021] JGD_2000_UTM_Zone_51N (deprecated) [1022] JGD_2000_UTM_Zone_52N (deprecated) [1023] JGD_2000_UTM_Zone_53N (deprecated) [1024] JGD_2000_UTM_Zone_54N (deprecated) [1025] JGD_2000_UTM_Zone_55N (deprecated) [1026] JGD_2000_UTM_Zone_56N [1027] JGD_2011_Japan_Zone_1 (deprecated) [1028] JGD_2011_Japan_Zone_10 (deprecated) [1029] JGD_2011_Japan_Zone_11 (deprecated) [1030] JGD_2011_Japan_Zone_12 (deprecated) [1031] JGD_2011_Japan_Zone_13 (deprecated) [1032] JGD_2011_Japan_Zone_14 (deprecated) [1033] JGD_2011_Japan_Zone_15 (deprecated) [1034] JGD_2011_Japan_Zone_16 (deprecated) [1035] JGD_2011_Japan_Zone_17 (deprecated) [1036] JGD_2011_Japan_Zone_18 (deprecated) [1037] JGD_2011_Japan_Zone_19 (deprecated) [1038] JGD_2011_Japan_Zone_2 (deprecated) [1039] JGD_2011_Japan_Zone_3 (deprecated) [1040] JGD_2011_Japan_Zone_4 (deprecated) [1041] JGD_2011_Japan_Zone_5 (deprecated) [1042] JGD_2011_Japan_Zone_6 (deprecated) [1043] JGD_2011_Japan_Zone_7 (deprecated) [1044] JGD_2011_Japan_Zone_8 (deprecated) [1045] JGD_2011_Japan_Zone_9 (deprecated) [1046] JGD_2011_UTM_Zone_51N (deprecated) [1047] JGD_2011_UTM_Zone_52N (deprecated) [1048] JGD_2011_UTM_Zone_53N (deprecated) [1049] JGD_2011_UTM_Zone_54N (deprecated) [1050] JGD_2011_UTM_Zone_55N (deprecated) [1051] JGD_2011_UTM_Zone_56N [1052] Jamaica 1875 / Jamaica (Old Grid) [1053] Jordan_JTM [1054] K0 1949 / UTM zone 42S (deprecated) [1055] KKJ / Finland Uniform Coordinate System [1056] KKJ / Finland zone 0 [1057] KKJ / Finland zone 1 [1058] KKJ / Finland zone 2 [1059] KKJ / Finland zone 4 [1060] KKJ / Finland zone 5 [1061] KOC Lambert [1062] KUDAMS / KTM [1063] KUDAMS / KTM (deprecated) [1064] Kalianpur 1880 / India zone 0 [1065] Kalianpur 1880 / India zone I [1066] Kalianpur 1880 / India zone IIIa [1067] Kalianpur 1880 / India zone IIa [1068] Kalianpur 1880 / India zone IIb [1069] Kalianpur 1880 / India zone IVa [1070] Kalianpur 1937 / India zone IIb [1071] Kalianpur 1937 / UTM zone 45N [1072] Kalianpur 1937 / UTM zone 46N [1073] Kalianpur 1962 / India zone I [1074] Kalianpur 1962 / India zone IIa [1075] Kalianpur 1962 / UTM zone 41N [1076] Kalianpur 1962 / UTM zone 42N [1077] Kalianpur 1962 / UTM zone 43N [1078] Kalianpur 1975 / India zone I [1079] Kalianpur 1975 / India zone IIIa [1080] Kalianpur 1975 / India zone IIa [1081] Kalianpur 1975 / India zone IIb [1082] Kalianpur 1975 / India zone IVa [1083] Kalianpur 1975 / UTM zone 42N [1084] Kalianpur 1975 / UTM zone 43N [1085] Kalianpur 1975 / UTM zone 44N [1086] Kalianpur 1975 / UTM zone 45N [1087] Kalianpur 1975 / UTM zone 46N [1088] Kalianpur 1975 / UTM zone 47N [1089] Kandawala / Sri Lanka Grid [1090] Kandawala_Ceylon_Belt_Indian_Yards_1937 [1091] Kandawala_Ceylon_Belt_Meters [1092] Karbala 1979 / Iraq National Grid [1093] Karbala 1979 / UTM zone 37N [1094] Karbala 1979 / UTM zone 38N [1095] Karbala 1979 / UTM zone 39N [1096] Kasai 1953 / Congo TM zone 22 [1097] Kasai 1953 / Congo TM zone 24 [1098] Katanga 1955 / Katanga Gauss zone A [1099] Katanga 1955 / Katanga Gauss zone B [1100] Katanga 1955 / Katanga Gauss zone C [1101] Katanga 1955 / Katanga Gauss zone D [1102] Katanga 1955 / Katanga Lambert [1103] Katanga 1955 / Katanga Lambert (deprecated) [1104] Katanga 1955 / Katanga Lambert (deprecated) [1105] Katanga 1955 / Katanga TM (deprecated) [1106] Katanga_1955_Katanga_Lambert (deprecated) [1107] Kertau (RSO) / RSO Malaya (ch) [1108] Kertau (RSO) / RSO Malaya (m) [1109] Kertau / R.S.O. Malaya (ch) (deprecated) [1110] Kertau 1968 / Johor Grid [1111] Kertau 1968 / Kedah and Perlis Grid [1112] Kertau 1968 / Kelantan Grid [1113] Kertau 1968 / Pahang Grid [1114] Kertau 1968 / Perak Revised Grid [1115] Kertau 1968 / Pinang Grid [1116] Kertau 1968 / Selangor Grid [1117] Kertau 1968 / Sembilan and Melaka Grid [1118] Kertau 1968 / Singapore Grid [1119] Kertau 1968 / Terengganu Grid [1120] Kertau 1968 / UTM zone 47N [1121] Kertau 1968 / UTM zone 48N [1122] Kertau_RSO_Malaya_Meters [1123] Korea 2000 / Central Belt [1124] Korea 2000 / Central Belt 2010 [1125] Korea 2000 / Central Belt Jeju [1126] Korea 2000 / East Belt [1127] Korea 2000 / East Belt 2010 [1128] Korea 2000 / East Sea Belt [1129] Korea 2000 / East Sea Belt 2010 [1130] Korea 2000 / Unified CS [1131] Korea 2000 / West Belt [1132] Korea 2000 / West Belt 2010 [1133] Korea_2000_Korea_Central_Belt_2010 (deprecated) [1134] Korea_2000_Korea_East_Belt_2010 (deprecated) [1135] Korea_2000_Korea_East_Sea_Belt_2010 (deprecated) [1136] Korea_2000_Korea_Unified_Coordinate_System (deprecated) [1137] Korea_2000_Korea_West_Belt_2010 (deprecated) [1138] Korean 1985 / Central Belt [1139] Korean 1985 / Central Belt Jeju [1140] Korean 1985 / East Belt [1141] Korean 1985 / East Sea Belt [1142] Korean 1985 / Modified Central Belt [1143] Korean 1985 / Modified Central Belt Jeju [1144] Korean 1985 / Modified East Belt [1145] Korean 1985 / Modified East Sea Belt [1146] Korean 1985 / Modified West Belt [1147] Korean 1985 / Unified CS [1148] Korean 1985 / West Belt [1149] Korean_1985_Korea_Unified_Coordinate_System (deprecated) [1150] Korean_1985_Modified_Korea_Central_Belt (deprecated) [1151] Korean_1985_Modified_Korea_Central_Belt_Jeju (deprecated) [1152] Korean_1985_Modified_Korea_East_Belt (deprecated) [1153] Korean_1985_Modified_Korea_East_Sea_Belt (deprecated) [1154] Korean_1985_Modified_Korea_West_Belt (deprecated) [1155] Kousseri / UTM zone 33N [1156] Kyrg-06 / zone 1 [1157] Kyrg-06 / zone 2 [1158] Kyrg-06 / zone 3 [1159] Kyrg-06 / zone 4 [1160] Kyrg-06 / zone 5 [1161] Kyrg-06_TM_Zone_1 [1162] Kyrg-06_TM_Zone_2 [1163] Kyrg-06_TM_Zone_3 [1164] Kyrg-06_TM_Zone_4 [1165] Kyrg-06_TM_Zone_5 [1166] LGD2006 / Libya TM [1167] LGD2006 / Libya TM zone 10 [1168] LGD2006 / Libya TM zone 11 [1169] LGD2006 / Libya TM zone 12 [1170] LGD2006 / Libya TM zone 13 [1171] LGD2006 / Libya TM zone 5 [1172] LGD2006 / Libya TM zone 6 [1173] LGD2006 / Libya TM zone 7 [1174] LGD2006 / Libya TM zone 8 [1175] LGD2006 / Libya TM zone 9 [1176] LGD2006 / UTM zone 32N [1177] LGD2006 / UTM zone 33N [1178] LGD2006 / UTM zone 34N [1179] LGD2006 / UTM zone 35N [1180] LKS92 / Latvia TM [1181] LKS94 / Lithuania TM [1182] LKS_1992_Latvia_TM_0 [1183] La Canoa / UTM zone 18N [1184] La Canoa / UTM zone 19N [1185] La Canoa / UTM zone 20N [1186] La_Canoa_UTM_Zone_21N [1187] Lake / Maracaibo Grid [1188] Lake / Maracaibo Grid M1 [1189] Lake / Maracaibo Grid M3 [1190] Lake / Maracaibo La Rosa Grid [1191] Le Pouce 1934 / Mauritius Grid [1192] Leigon / Ghana Metre Grid [1193] Lietuvos Koordinoei Sistema 1994 (deprecated) [1194] Lisboa_Bessel_Bonne [1195] Lisboa_Hayford_Gauss_IGeoE [1196] Lisboa_Hayford_Gauss_IPCC [1197] Lisbon (Lisbon) / Portuguese Grid [1198] Lisbon (Lisbon) / Portuguese National Grid [1199] Lisbon / Portuguese Grid New [1200] Little Cayman 1961 / UTM zone 17N (deprecated) [1201] Locodjo 1965 / TM 5 NW [1202] Locodjo 1965 / UTM zone 29N [1203] Locodjo 1965 / UTM zone 30N [1204] Lome / UTM zone 31N [1205] London_Survey_Grid [1206] Luxembourg 1930 / Gauss [1207] Luzon 1911 / Philippines zone I [1208] Luzon 1911 / Philippines zone II [1209] Luzon 1911 / Philippines zone III [1210] Luzon 1911 / Philippines zone IV [1211] Luzon 1911 / Philippines zone V [1212] Luzon_1911_UTM_Zone_50N [1213] Luzon_1911_UTM_Zone_51N [1214] Luzon_1911_UTM_Zone_52N [1215] M'poraloko / UTM zone 32N [1216] M'poraloko / UTM zone 32S [1217] MAGNA-SIRGAS / Colombia Bogota zone [1218] MAGNA-SIRGAS / Colombia East Central zone [1219] MAGNA-SIRGAS / Colombia East zone [1220] MAGNA-SIRGAS / Colombia Far West zone [1221] MAGNA-SIRGAS / Colombia West zone [1222] MAGNA_Ciudad_Bogota [1223] MARGEN / UTM zone 19S [1224] MARGEN / UTM zone 20S [1225] MARGEN / UTM zone 21S [1226] MGI (Ferro) / Austria Central Zone [1227] MGI (Ferro) / Austria Central Zone (deprecated) [1228] MGI (Ferro) / Austria East Zone [1229] MGI (Ferro) / Austria East Zone (deprecated) [1230] MGI (Ferro) / Austria GK Central Zone [1231] MGI (Ferro) / Austria GK East Zone [1232] MGI (Ferro) / Austria GK West Zone [1233] MGI (Ferro) / Austria West Zone [1234] MGI (Ferro) / Austria West Zone (deprecated) [1235] MGI (Ferro) / M28 [1236] MGI (Ferro) / M31 [1237] MGI (Ferro) / M34 [1238] MGI / 3-degree Gauss zone 5 (deprecated) [1239] MGI / 3-degree Gauss zone 6 (deprecated) [1240] MGI / 3-degree Gauss zone 7 (deprecated) [1241] MGI / 3-degree Gauss zone 8 (deprecated) [1242] MGI / Austria GK Central [1243] MGI / Austria GK East [1244] MGI / Austria GK M28 [1245] MGI / Austria GK M31 [1246] MGI / Austria GK M34 [1247] MGI / Austria GK West [1248] MGI / Austria Lambert [1249] MGI / Austria Lambert (deprecated) [1250] MGI / Austria M28 [1251] MGI / Austria M31 [1252] MGI / Austria M34 [1253] MGI / Balkans zone 5 (deprecated) [1254] MGI / Balkans zone 6 (deprecated) [1255] MGI / Balkans zone 7 (deprecated) [1256] MGI / Balkans zone 8 (deprecated) [1257] MGI / Balkans zone 8 (deprecated) [1258] MGI / M28 (deprecated) [1259] MGI / M31 (deprecated) [1260] MGI / M34 (deprecated) [1261] MGI / Slovene National Grid (deprecated) [1262] MGI / Slovenia Grid (deprecated) [1263] MGI 1901 / Balkans zone 5 [1264] MGI 1901 / Balkans zone 5 [1265] MGI 1901 / Balkans zone 6 [1266] MGI 1901 / Balkans zone 6 [1267] MGI 1901 / Balkans zone 7 [1268] MGI 1901 / Balkans zone 8 [1269] MGI 1901 / Balkans zone 8 [1270] MGI 1901 / Slovene National Grid [1271] MGI 1901 / Slovenia Grid [1272] MGI 1901 / Slovenia Grid [1273] MOLDREF99 / Moldova TM [1274] MONREF_1997_UTM_Zone_46N [1275] MONREF_1997_UTM_Zone_47N [1276] MONREF_1997_UTM_Zone_48N [1277] MONREF_1997_UTM_Zone_49N [1278] MONREF_1997_UTM_Zone_50N [1279] MOP78 / UTM zone 1S [1280] MTRF-2000 / UTM zone 36N [1281] MTRF-2000 / UTM zone 37N [1282] MTRF-2000 / UTM zone 38N [1283] MTRF-2000 / UTM zone 39N [1284] MTRF-2000 / UTM zone 40N [1285] Macao 1920 / Macao Grid [1286] Macao_2008_Macao_Grid [1287] Macedonia State Coordinate System [1288] Macedonia State Coordinate System zone 7 [1289] Madeira 1936 / UTM zone 28N (deprecated) [1290] Madrid 1870 (Madrid) / Spain [1291] Makassar (Jakarta) / NEIEZ [1292] Makassar (Jakarta) / NEIEZ (deprecated) [1293] Makassar / NEIEZ [1294] Malongo 1987 / UTM zone 32S [1295] Malongo 1987 / UTM zone 33S [1296] Manoca 1962 / UTM zone 32N [1297] Martinique 1938 / UTM zone 20N [1298] Massawa / UTM zone 37N [1299] Maupiti 83 / UTM zone 5S [1300] Mauritania 1999 / UTM zone 28N [1301] Mauritania 1999 / UTM zone 28N (deprecated) [1302] Mauritania 1999 / UTM zone 29N [1303] Mauritania 1999 / UTM zone 29N (deprecated) [1304] Mauritania 1999 / UTM zone 30N [1305] Mauritania 1999 / UTM zone 30N (deprecated) [1306] Merchich / Nord Maroc [1307] Merchich / Sahara (deprecated) [1308] Merchich / Sahara Nord [1309] Merchich / Sahara Sud [1310] Merchich / Sud Maroc [1311] Merchich_Degree_UTM_Zone_28N [1312] Mexican_Datum_1993_UTM_Zone_11N (deprecated) [1313] Mexican_Datum_1993_UTM_Zone_12N (deprecated) [1314] Mexican_Datum_1993_UTM_Zone_13N (deprecated) [1315] Mexican_Datum_1993_UTM_Zone_14N (deprecated) [1316] Mexican_Datum_1993_UTM_Zone_15N (deprecated) [1317] Mexican_Datum_1993_UTM_Zone_16N (deprecated) [1318] Mexico ITRF2008 / LCC [1319] Mexico ITRF2008 / UTM zone 11N [1320] Mexico ITRF2008 / UTM zone 12N [1321] Mexico ITRF2008 / UTM zone 13N [1322] Mexico ITRF2008 / UTM zone 14N [1323] Mexico ITRF2008 / UTM zone 15N [1324] Mexico ITRF2008 / UTM zone 16N [1325] Mexico ITRF92 / LCC [1326] Mexico ITRF92 / UTM zone 11N [1327] Mexico ITRF92 / UTM zone 12N [1328] Mexico ITRF92 / UTM zone 13N [1329] Mexico ITRF92 / UTM zone 14N [1330] Mexico ITRF92 / UTM zone 15N [1331] Mexico ITRF92 / UTM zone 16N [1332] Mhast (offshore) / UTM zone 32S [1333] Mhast (onshore) / UTM zone 32S [1334] Mhast / UTM zone 32S (deprecated) [1335] Minna / Nigeria East Belt [1336] Minna / Nigeria Mid Belt [1337] Minna / Nigeria West Belt [1338] Minna / UTM zone 31N [1339] Minna / UTM zone 32N [1340] Monte Mario (Rome) / Italy zone 1 (deprecated) [1341] Monte Mario (Rome) / Italy zone 2 (deprecated) [1342] Monte Mario / Italy zone 1 [1343] Monte Mario / Italy zone 2 [1344] Monte Mario / TM Emilia-Romagna [1345] Monte_Mario_Italy_1 (deprecated) [1346] Monte_Mario_Italy_2 (deprecated) [1347] Montserrat 1958 / British West Indies Grid [1348] Moorea 87 / UTM zone 6S [1349] Mount Dillon / Tobago Grid [1350] Moznet / UTM zone 36S [1351] Moznet / UTM zone 37S [1352] Moznet / UTM zone 38S [1353] NAD Michigan / Michigan Central (deprecated) [1354] NAD Michigan / Michigan East (deprecated) [1355] NAD Michigan / Michigan North (deprecated) [1356] NAD Michigan / Michigan Old Central (deprecated) [1357] NAD Michigan / Michigan South (deprecated) [1358] NAD Michigan / Michigan West (deprecated) [1359] NAD27 / Alabama East [1360] NAD27 / Alabama West [1361] NAD27 / Alaska Albers [1362] NAD27 / Alaska zone 1 [1363] NAD27 / Alaska zone 10 [1364] NAD27 / Alaska zone 2 [1365] NAD27 / Alaska zone 3 [1366] NAD27 / Alaska zone 4 [1367] NAD27 / Alaska zone 5 [1368] NAD27 / Alaska zone 6 [1369] NAD27 / Alaska zone 7 [1370] NAD27 / Alaska zone 8 [1371] NAD27 / Alaska zone 9 [1372] NAD27 / Alberta 3TM ref merid 111 W [1373] NAD27 / Alberta 3TM ref merid 114 W [1374] NAD27 / Alberta 3TM ref merid 117 W [1375] NAD27 / Alberta 3TM ref merid 120 W [1376] NAD27 / Alberta 3TM ref merid 120 W (deprecated) [1377] NAD27 / Arizona Central [1378] NAD27 / Arizona East [1379] NAD27 / Arizona West [1380] NAD27 / Arkansas North [1381] NAD27 / Arkansas South [1382] NAD27 / BLM 10N (ftUS) [1383] NAD27 / BLM 11N (ftUS) [1384] NAD27 / BLM 12N (ftUS) [1385] NAD27 / BLM 13N (ftUS) [1386] NAD27 / BLM 14N (feet) (deprecated) [1387] NAD27 / BLM 14N (ftUS) [1388] NAD27 / BLM 15N (feet) (deprecated) [1389] NAD27 / BLM 15N (ftUS) [1390] NAD27 / BLM 16N (feet) (deprecated) [1391] NAD27 / BLM 16N (ftUS) [1392] NAD27 / BLM 17N (feet) (deprecated) [1393] NAD27 / BLM 17N (ftUS) [1394] NAD27 / BLM 18N (ftUS) [1395] NAD27 / BLM 19N (ftUS) [1396] NAD27 / BLM 1N (ftUS) [1397] NAD27 / BLM 2N (ftUS) [1398] NAD27 / BLM 3N (ftUS) [1399] NAD27 / BLM 4N (ftUS) [1400] NAD27 / BLM 59N (ftUS) [1401] NAD27 / BLM 5N (ftUS) [1402] NAD27 / BLM 60N (ftUS) [1403] NAD27 / BLM 6N (ftUS) [1404] NAD27 / BLM 7N (ftUS) [1405] NAD27 / BLM 8N (ftUS) [1406] NAD27 / BLM 9N (ftUS) [1407] NAD27 / California Albers [1408] NAD27 / California zone I [1409] NAD27 / California zone II [1410] NAD27 / California zone III [1411] NAD27 / California zone IV [1412] NAD27 / California zone V [1413] NAD27 / California zone VI [1414] NAD27 / California zone VII [1415] NAD27 / California zone VII (deprecated) [1416] NAD27 / Colorado Central [1417] NAD27 / Colorado North [1418] NAD27 / Colorado South [1419] NAD27 / Connecticut [1420] NAD27 / Conus Albers [1421] NAD27 / Cuba Norte [1422] NAD27 / Cuba Norte (deprecated) [1423] NAD27 / Cuba Sur [1424] NAD27 / Cuba Sur (deprecated) [1425] NAD27 / Delaware [1426] NAD27 / Florida East [1427] NAD27 / Florida North [1428] NAD27 / Florida West [1429] NAD27 / Georgia East [1430] NAD27 / Georgia West [1431] NAD27 / Guatemala Norte (deprecated) [1432] NAD27 / Guatemala Sur (deprecated) [1433] NAD27 / Idaho Central [1434] NAD27 / Idaho East [1435] NAD27 / Idaho West [1436] NAD27 / Illinois East [1437] NAD27 / Illinois West [1438] NAD27 / Indiana East [1439] NAD27 / Indiana West [1440] NAD27 / Iowa North [1441] NAD27 / Iowa South [1442] NAD27 / Kansas North [1443] NAD27 / Kansas South [1444] NAD27 / Kentucky North [1445] NAD27 / Kentucky South [1446] NAD27 / Louisiana North [1447] NAD27 / Louisiana Offshore [1448] NAD27 / Louisiana South [1449] NAD27 / MTM zone 1 [1450] NAD27 / MTM zone 10 [1451] NAD27 / MTM zone 2 [1452] NAD27 / MTM zone 3 [1453] NAD27 / MTM zone 4 [1454] NAD27 / MTM zone 5 [1455] NAD27 / MTM zone 6 [1456] NAD27 / MTQ Lambert [1457] NAD27 / Maine East [1458] NAD27 / Maine West [1459] NAD27 / Maryland [1460] NAD27 / Massachusetts Island [1461] NAD27 / Massachusetts Mainland [1462] NAD27 / Michigan East [1463] NAD27 / Michigan Old Central [1464] NAD27 / Michigan West [1465] NAD27 / Minnesota Central [1466] NAD27 / Minnesota North [1467] NAD27 / Minnesota South [1468] NAD27 / Mississippi East [1469] NAD27 / Mississippi West [1470] NAD27 / Missouri Central [1471] NAD27 / Missouri East [1472] NAD27 / Missouri West [1473] NAD27 / Montana Central [1474] NAD27 / Montana North [1475] NAD27 / Montana South [1476] NAD27 / Nebraska North [1477] NAD27 / Nebraska South [1478] NAD27 / Nevada Central [1479] NAD27 / Nevada East [1480] NAD27 / Nevada West [1481] NAD27 / New Brunswick Stereographic (NAD27) [1482] NAD27 / New Hampshire [1483] NAD27 / New Jersey [1484] NAD27 / New Mexico Central [1485] NAD27 / New Mexico East [1486] NAD27 / New Mexico West [1487] NAD27 / New York Central [1488] NAD27 / New York East [1489] NAD27 / New York Long Island [1490] NAD27 / New York Long Island (deprecated) [1491] NAD27 / New York West [1492] NAD27 / North Carolina [1493] NAD27 / North Dakota North [1494] NAD27 / North Dakota South [1495] NAD27 / Ohio North [1496] NAD27 / Ohio South [1497] NAD27 / Oklahoma North [1498] NAD27 / Oklahoma South [1499] NAD27 / Oregon North [1500] NAD27 / Oregon South [1501] NAD27 / Pennsylvania North [1502] NAD27 / Pennsylvania South [1503] NAD27 / Pennsylvania South (deprecated) [1504] NAD27 / Quebec Lambert [1505] NAD27 / Rhode Island [1506] NAD27 / Shackleford [1507] NAD27 / South Carolina North [1508] NAD27 / South Carolina South [1509] NAD27 / South Dakota North [1510] NAD27 / South Dakota South [1511] NAD27 / Tennessee [1512] NAD27 / Tennessee (deprecated) [1513] NAD27 / Texas Central [1514] NAD27 / Texas North [1515] NAD27 / Texas North Central [1516] NAD27 / Texas South [1517] NAD27 / Texas South Central [1518] NAD27 / UTM zone 10N [1519] NAD27 / UTM zone 11N [1520] NAD27 / UTM zone 12N [1521] NAD27 / UTM zone 13N [1522] NAD27 / UTM zone 14N [1523] NAD27 / UTM zone 15N [1524] NAD27 / UTM zone 16N [1525] NAD27 / UTM zone 17N [1526] NAD27 / UTM zone 18N [1527] NAD27 / UTM zone 19N [1528] NAD27 / UTM zone 1N [1529] NAD27 / UTM zone 20N [1530] NAD27 / UTM zone 21N [1531] NAD27 / UTM zone 22N [1532] NAD27 / UTM zone 2N [1533] NAD27 / UTM zone 3N [1534] NAD27 / UTM zone 4N [1535] NAD27 / UTM zone 59N [1536] NAD27 / UTM zone 5N [1537] NAD27 / UTM zone 60N [1538] NAD27 / UTM zone 6N [1539] NAD27 / UTM zone 7N [1540] NAD27 / UTM zone 8N [1541] NAD27 / UTM zone 9N [1542] NAD27 / Utah Central [1543] NAD27 / Utah North [1544] NAD27 / Utah South [1545] NAD27 / Vermont [1546] NAD27 / Virginia North [1547] NAD27 / Virginia South [1548] NAD27 / Washington North [1549] NAD27 / Washington South [1550] NAD27 / West Virginia North [1551] NAD27 / West Virginia South [1552] NAD27 / Wisconsin Central [1553] NAD27 / Wisconsin North [1554] NAD27 / Wisconsin South [1555] NAD27 / Wisconsin Transverse Mercator [1556] NAD27 / Wyoming East [1557] NAD27 / Wyoming East Central [1558] NAD27 / Wyoming West [1559] NAD27 / Wyoming West Central [1560] NAD27(76) / MTM zone 10 [1561] NAD27(76) / MTM zone 11 [1562] NAD27(76) / MTM zone 12 [1563] NAD27(76) / MTM zone 13 [1564] NAD27(76) / MTM zone 14 [1565] NAD27(76) / MTM zone 15 [1566] NAD27(76) / MTM zone 16 [1567] NAD27(76) / MTM zone 17 [1568] NAD27(76) / MTM zone 8 [1569] NAD27(76) / MTM zone 9 [1570] NAD27(76) / UTM zone 15N [1571] NAD27(76) / UTM zone 16N [1572] NAD27(76) / UTM zone 17N [1573] NAD27(76) / UTM zone 18N [1574] NAD27(CGQ77) / Quebec Lambert [1575] NAD27(CGQ77) / SCoPQ zone 10 [1576] NAD27(CGQ77) / SCoPQ zone 2 (deprecated) [1577] NAD27(CGQ77) / SCoPQ zone 3 [1578] NAD27(CGQ77) / SCoPQ zone 4 [1579] NAD27(CGQ77) / SCoPQ zone 5 [1580] NAD27(CGQ77) / SCoPQ zone 6 [1581] NAD27(CGQ77) / SCoPQ zone 7 [1582] NAD27(CGQ77) / SCoPQ zone 8 [1583] NAD27(CGQ77) / SCoPQ zone 9 [1584] NAD27(CGQ77) / UTM zone 17N [1585] NAD27(CGQ77) / UTM zone 18N [1586] NAD27(CGQ77) / UTM zone 19N [1587] NAD27(CGQ77) / UTM zone 20N [1588] NAD27(CGQ77) / UTM zone 21N [1589] NAD83 / Alabama East [1590] NAD83 / Alabama West [1591] NAD83 / Alaska Albers [1592] NAD83 / Alaska zone 1 [1593] NAD83 / Alaska zone 10 [1594] NAD83 / Alaska zone 2 [1595] NAD83 / Alaska zone 3 [1596] NAD83 / Alaska zone 4 [1597] NAD83 / Alaska zone 5 [1598] NAD83 / Alaska zone 6 [1599] NAD83 / Alaska zone 7 [1600] NAD83 / Alaska zone 8 [1601] NAD83 / Alaska zone 9 [1602] NAD83 / Alberta 10-TM (Forest) [1603] NAD83 / Alberta 10-TM (Resource) [1604] NAD83 / Alberta 3TM ref merid 111 W [1605] NAD83 / Alberta 3TM ref merid 114 W [1606] NAD83 / Alberta 3TM ref merid 117 W [1607] NAD83 / Alberta 3TM ref merid 120 W [1608] NAD83 / Alberta 3TM ref merid 120 W (deprecated) [1609] NAD83 / Arizona Central [1610] NAD83 / Arizona Central (ft) [1611] NAD83 / Arizona East [1612] NAD83 / Arizona East (ft) [1613] NAD83 / Arizona West [1614] NAD83 / Arizona West (ft) [1615] NAD83 / Arkansas North [1616] NAD83 / Arkansas North (ftUS) [1617] NAD83 / Arkansas South [1618] NAD83 / Arkansas South (ftUS) [1619] NAD83 / BC Albers [1620] NAD83 / BLM 10N (ftUS) [1621] NAD83 / BLM 11N (ftUS) [1622] NAD83 / BLM 12N (ftUS) [1623] NAD83 / BLM 13N (ftUS) [1624] NAD83 / BLM 14N (ftUS) [1625] NAD83 / BLM 15N (ftUS) [1626] NAD83 / BLM 16N (ftUS) [1627] NAD83 / BLM 17N (ftUS) [1628] NAD83 / BLM 18N (ftUS) [1629] NAD83 / BLM 19N (ftUS) [1630] NAD83 / BLM 1N (ftUS) [1631] NAD83 / BLM 2N (ftUS) [1632] NAD83 / BLM 3N (ftUS) [1633] NAD83 / BLM 4N (ftUS) [1634] NAD83 / BLM 59N (ftUS) [1635] NAD83 / BLM 5N (ftUS) [1636] NAD83 / BLM 60N (ftUS) [1637] NAD83 / BLM 6N (ftUS) [1638] NAD83 / BLM 7N (ftUS) [1639] NAD83 / BLM 8N (ftUS) [1640] NAD83 / BLM 9N (ftUS) [1641] NAD83 / California Albers [1642] NAD83 / California zone 1 [1643] NAD83 / California zone 1 (ftUS) [1644] NAD83 / California zone 2 [1645] NAD83 / California zone 2 (ftUS) [1646] NAD83 / California zone 3 [1647] NAD83 / California zone 3 (ftUS) [1648] NAD83 / California zone 4 [1649] NAD83 / California zone 4 (ftUS) [1650] NAD83 / California zone 5 [1651] NAD83 / California zone 5 (ftUS) [1652] NAD83 / California zone 6 [1653] NAD83 / California zone 6 (ftUS) [1654] NAD83 / Canada Atlas Lambert [1655] NAD83 / Colorado Central [1656] NAD83 / Colorado Central (ftUS) [1657] NAD83 / Colorado North [1658] NAD83 / Colorado North (ftUS) [1659] NAD83 / Colorado South [1660] NAD83 / Colorado South (ftUS) [1661] NAD83 / Connecticut [1662] NAD83 / Connecticut (ftUS) [1663] NAD83 / Conus Albers [1664] NAD83 / Delaware [1665] NAD83 / Delaware (ftUS) [1666] NAD83 / Florida East [1667] NAD83 / Florida East (ftUS) [1668] NAD83 / Florida GDL Albers [1669] NAD83 / Florida North [1670] NAD83 / Florida North (ftUS) [1671] NAD83 / Florida West [1672] NAD83 / Florida West (ftUS) [1673] NAD83 / Georgia East [1674] NAD83 / Georgia East (ftUS) [1675] NAD83 / Georgia West [1676] NAD83 / Georgia West (ftUS) [1677] NAD83 / Great Lakes Albers [1678] NAD83 / Great Lakes and St Lawrence Albers [1679] NAD83 / Hawaii zone 1 [1680] NAD83 / Hawaii zone 2 [1681] NAD83 / Hawaii zone 3 [1682] NAD83 / Hawaii zone 3 (ftUS) [1683] NAD83 / Hawaii zone 4 [1684] NAD83 / Hawaii zone 5 [1685] NAD83 / Idaho Central [1686] NAD83 / Idaho Central (ftUS) [1687] NAD83 / Idaho East [1688] NAD83 / Idaho East (ftUS) [1689] NAD83 / Idaho Transverse Mercator [1690] NAD83 / Idaho West [1691] NAD83 / Idaho West (ftUS) [1692] NAD83 / Illinois East [1693] NAD83 / Illinois East (ftUS) [1694] NAD83 / Illinois West [1695] NAD83 / Illinois West (ftUS) [1696] NAD83 / Indiana East [1697] NAD83 / Indiana East (ftUS) [1698] NAD83 / Indiana East (ftUS) (deprecated) [1699] NAD83 / Indiana West [1700] NAD83 / Indiana West (ftUS) [1701] NAD83 / Indiana West (ftUS) (deprecated) [1702] NAD83 / Iowa North [1703] NAD83 / Iowa North (ftUS) [1704] NAD83 / Iowa South [1705] NAD83 / Iowa South (ftUS) [1706] NAD83 / Kansas LCC [1707] NAD83 / Kansas LCC (ftUS) [1708] NAD83 / Kansas North [1709] NAD83 / Kansas North (ftUS) [1710] NAD83 / Kansas South [1711] NAD83 / Kansas South (ftUS) [1712] NAD83 / Kentucky North [1713] NAD83 / Kentucky North (deprecated) [1714] NAD83 / Kentucky North (ftUS) [1715] NAD83 / Kentucky Single Zone [1716] NAD83 / Kentucky Single Zone (ftUS) [1717] NAD83 / Kentucky South [1718] NAD83 / Kentucky South (ftUS) [1719] NAD83 / Louisiana North [1720] NAD83 / Louisiana North (ftUS) [1721] NAD83 / Louisiana Offshore [1722] NAD83 / Louisiana Offshore (ftUS) [1723] NAD83 / Louisiana South [1724] NAD83 / Louisiana South (ftUS) [1725] NAD83 / MTM zone 1 [1726] NAD83 / MTM zone 10 [1727] NAD83 / MTM zone 11 [1728] NAD83 / MTM zone 12 [1729] NAD83 / MTM zone 13 [1730] NAD83 / MTM zone 14 [1731] NAD83 / MTM zone 15 [1732] NAD83 / MTM zone 16 [1733] NAD83 / MTM zone 17 [1734] NAD83 / MTM zone 2 [1735] NAD83 / MTM zone 3 [1736] NAD83 / MTM zone 4 [1737] NAD83 / MTM zone 5 [1738] NAD83 / MTM zone 6 [1739] NAD83 / MTM zone 7 [1740] NAD83 / MTM zone 8 [1741] NAD83 / MTM zone 9 [1742] NAD83 / MTQ Lambert [1743] NAD83 / Maine CS2000 Central [1744] NAD83 / Maine CS2000 Central (deprecated) [1745] NAD83 / Maine CS2000 East [1746] NAD83 / Maine CS2000 West [1747] NAD83 / Maine East [1748] NAD83 / Maine East (ftUS) [1749] NAD83 / Maine East (ftUS) (deprecated) [1750] NAD83 / Maine West [1751] NAD83 / Maine West (ftUS) [1752] NAD83 / Maine West (ftUS) (deprecated) [1753] NAD83 / Maryland [1754] NAD83 / Maryland (ftUS) [1755] NAD83 / Massachusetts Island [1756] NAD83 / Massachusetts Island (ftUS) [1757] NAD83 / Massachusetts Mainland [1758] NAD83 / Massachusetts Mainland (ftUS) [1759] NAD83 / Michigan Central [1760] NAD83 / Michigan Central (ft) [1761] NAD83 / Michigan North [1762] NAD83 / Michigan North (ft) [1763] NAD83 / Michigan Oblique Mercator [1764] NAD83 / Michigan South [1765] NAD83 / Michigan South (ft) [1766] NAD83 / Minnesota Central [1767] NAD83 / Minnesota Central (ftUS) [1768] NAD83 / Minnesota Central (ftUS) (deprecated) [1769] NAD83 / Minnesota North [1770] NAD83 / Minnesota North (ftUS) [1771] NAD83 / Minnesota North (ftUS) (deprecated) [1772] NAD83 / Minnesota South [1773] NAD83 / Minnesota South (ftUS) [1774] NAD83 / Minnesota South (ftUS) (deprecated) [1775] NAD83 / Mississippi East [1776] NAD83 / Mississippi East (ftUS) [1777] NAD83 / Mississippi TM [1778] NAD83 / Mississippi West [1779] NAD83 / Mississippi West (ftUS) [1780] NAD83 / Missouri Central [1781] NAD83 / Missouri East [1782] NAD83 / Missouri West [1783] NAD83 / Montana [1784] NAD83 / Montana (ft) [1785] NAD83 / NCRS Las Vegas (ftUS) [1786] NAD83 / NCRS Las Vegas (m) [1787] NAD83 / NCRS Las Vegas high (ftUS) [1788] NAD83 / NCRS Las Vegas high (m) [1789] NAD83 / NWT Lambert [1790] NAD83 / Nebraska [1791] NAD83 / Nebraska (ftUS) [1792] NAD83 / Nebraska (ftUS) (deprecated) [1793] NAD83 / Nevada Central [1794] NAD83 / Nevada Central (ftUS) [1795] NAD83 / Nevada East [1796] NAD83 / Nevada East (ftUS) [1797] NAD83 / Nevada West [1798] NAD83 / Nevada West (ftUS) [1799] NAD83 / New Hampshire [1800] NAD83 / New Hampshire (ftUS) [1801] NAD83 / New Jersey [1802] NAD83 / New Jersey (ftUS) [1803] NAD83 / New Mexico Central [1804] NAD83 / New Mexico Central (ftUS) [1805] NAD83 / New Mexico East [1806] NAD83 / New Mexico East (ftUS) [1807] NAD83 / New Mexico West [1808] NAD83 / New Mexico West (ftUS) [1809] NAD83 / New York Central [1810] NAD83 / New York Central (ftUS) [1811] NAD83 / New York East [1812] NAD83 / New York East (ftUS) [1813] NAD83 / New York Long Island [1814] NAD83 / New York Long Island (ftUS) [1815] NAD83 / New York West [1816] NAD83 / New York West (ftUS) [1817] NAD83 / North Carolina [1818] NAD83 / North Carolina (ftUS) [1819] NAD83 / North Dakota North [1820] NAD83 / North Dakota North (ft) [1821] NAD83 / North Dakota South [1822] NAD83 / North Dakota South (ft) [1823] NAD83 / Ohio North [1824] NAD83 / Ohio North (ftUS) [1825] NAD83 / Ohio South [1826] NAD83 / Ohio South (ftUS) [1827] NAD83 / Oklahoma North [1828] NAD83 / Oklahoma North (ftUS) [1829] NAD83 / Oklahoma South [1830] NAD83 / Oklahoma South (ftUS) [1831] NAD83 / Ontario MNR Lambert [1832] NAD83 / Oregon GIC Lambert (ft) [1833] NAD83 / Oregon LCC (m) [1834] NAD83 / Oregon North [1835] NAD83 / Oregon North (ft) [1836] NAD83 / Oregon South [1837] NAD83 / Oregon South (ft) [1838] NAD83 / Pennsylvania North [1839] NAD83 / Pennsylvania North (ftUS) [1840] NAD83 / Pennsylvania South [1841] NAD83 / Pennsylvania South (ftUS) [1842] NAD83 / Puerto Rico & Virgin Is. [1843] NAD83 / Quebec Albers [1844] NAD83 / Quebec Lambert [1845] NAD83 / Rhode Island [1846] NAD83 / Rhode Island (ftUS) [1847] NAD83 / SCoPQ zone 2 (deprecated) [1848] NAD83 / South Carolina [1849] NAD83 / South Carolina (ft) [1850] NAD83 / South Dakota North [1851] NAD83 / South Dakota North (ftUS) [1852] NAD83 / South Dakota North (ftUS) (deprecated) [1853] NAD83 / South Dakota South [1854] NAD83 / South Dakota South (ftUS) [1855] NAD83 / Statistics Canada Lambert [1856] NAD83 / Tennessee [1857] NAD83 / Tennessee (ftUS) [1858] NAD83 / Teranet Ontario Lambert [1859] NAD83 / Texas Central [1860] NAD83 / Texas Central (ftUS) [1861] NAD83 / Texas Centric Albers Equal Area [1862] NAD83 / Texas Centric Lambert Conformal [1863] NAD83 / Texas North [1864] NAD83 / Texas North (ftUS) [1865] NAD83 / Texas North Central [1866] NAD83 / Texas North Central (ftUS) [1867] NAD83 / Texas South [1868] NAD83 / Texas South (ftUS) [1869] NAD83 / Texas South Central [1870] NAD83 / Texas South Central (ftUS) [1871] NAD83 / Texas State Mapping System [1872] NAD83 / UTM zone 10N [1873] NAD83 / UTM zone 11N [1874] NAD83 / UTM zone 12N [1875] NAD83 / UTM zone 13N [1876] NAD83 / UTM zone 14N [1877] NAD83 / UTM zone 15N [1878] NAD83 / UTM zone 16N [1879] NAD83 / UTM zone 17N [1880] NAD83 / UTM zone 18N [1881] NAD83 / UTM zone 19N [1882] NAD83 / UTM zone 1N [1883] NAD83 / UTM zone 20N [1884] NAD83 / UTM zone 21N [1885] NAD83 / UTM zone 22N [1886] NAD83 / UTM zone 23N [1887] NAD83 / UTM zone 2N [1888] NAD83 / UTM zone 3N [1889] NAD83 / UTM zone 4N [1890] NAD83 / UTM zone 59N [1891] NAD83 / UTM zone 5N [1892] NAD83 / UTM zone 60N [1893] NAD83 / UTM zone 6N [1894] NAD83 / UTM zone 7N [1895] NAD83 / UTM zone 8N [1896] NAD83 / UTM zone 9N [1897] NAD83 / Utah Central [1898] NAD83 / Utah Central (ft) [1899] NAD83 / Utah Central (ftUS) [1900] NAD83 / Utah North [1901] NAD83 / Utah North (ft) [1902] NAD83 / Utah North (ftUS) [1903] NAD83 / Utah South [1904] NAD83 / Utah South (ft) [1905] NAD83 / Utah South (ftUS) [1906] NAD83 / Vermont [1907] NAD83 / Vermont (ftUS) [1908] NAD83 / Virginia Lambert [1909] NAD83 / Virginia North [1910] NAD83 / Virginia North (ftUS) [1911] NAD83 / Virginia South [1912] NAD83 / Virginia South (ftUS) [1913] NAD83 / Washington North [1914] NAD83 / Washington North (ftUS) [1915] NAD83 / Washington South [1916] NAD83 / Washington South (ftUS) [1917] NAD83 / West Virginia North [1918] NAD83 / West Virginia North (ftUS) [1919] NAD83 / West Virginia North (ftUS) (deprecated) [1920] NAD83 / West Virginia South [1921] NAD83 / West Virginia South (ftUS) [1922] NAD83 / West Virginia South (ftUS) (deprecated) [1923] NAD83 / Wisconsin Central [1924] NAD83 / Wisconsin Central (ftUS) [1925] NAD83 / Wisconsin North [1926] NAD83 / Wisconsin North (ftUS) [1927] NAD83 / Wisconsin South [1928] NAD83 / Wisconsin South (ftUS) [1929] NAD83 / Wisconsin Transverse Mercator [1930] NAD83 / Wyoming East [1931] NAD83 / Wyoming East (ftUS) [1932] NAD83 / Wyoming East Central [1933] NAD83 / Wyoming East Central (ftUS) [1934] NAD83 / Wyoming West [1935] NAD83 / Wyoming West (ftUS) [1936] NAD83 / Wyoming West Central [1937] NAD83 / Wyoming West Central (ftUS) [1938] NAD83 / Yukon Albers [1939] NAD83(2011) / Alabama East [1940] NAD83(2011) / Alabama West [1941] NAD83(2011) / Alaska Albers [1942] NAD83(2011) / Alaska zone 1 [1943] NAD83(2011) / Alaska zone 10 [1944] NAD83(2011) / Alaska zone 2 [1945] NAD83(2011) / Alaska zone 3 [1946] NAD83(2011) / Alaska zone 4 [1947] NAD83(2011) / Alaska zone 5 [1948] NAD83(2011) / Alaska zone 6 [1949] NAD83(2011) / Alaska zone 7 [1950] NAD83(2011) / Alaska zone 8 [1951] NAD83(2011) / Alaska zone 9 [1952] NAD83(2011) / Arizona Central [1953] NAD83(2011) / Arizona Central (ft) [1954] NAD83(2011) / Arizona East [1955] NAD83(2011) / Arizona East (ft) [1956] NAD83(2011) / Arizona West [1957] NAD83(2011) / Arizona West (ft) [1958] NAD83(2011) / Arkansas North [1959] NAD83(2011) / Arkansas North (ftUS) [1960] NAD83(2011) / Arkansas South [1961] NAD83(2011) / Arkansas South (ftUS) [1962] NAD83(2011) / California Albers [1963] NAD83(2011) / California zone 1 [1964] NAD83(2011) / California zone 1 (ftUS) [1965] NAD83(2011) / California zone 2 [1966] NAD83(2011) / California zone 2 (ftUS) [1967] NAD83(2011) / California zone 3 [1968] NAD83(2011) / California zone 3 (ftUS) [1969] NAD83(2011) / California zone 4 [1970] NAD83(2011) / California zone 4 (ftUS) [1971] NAD83(2011) / California zone 5 [1972] NAD83(2011) / California zone 5 (ftUS) [1973] NAD83(2011) / California zone 6 [1974] NAD83(2011) / California zone 6 (ftUS) [1975] NAD83(2011) / Colorado Central [1976] NAD83(2011) / Colorado Central (ftUS) [1977] NAD83(2011) / Colorado North [1978] NAD83(2011) / Colorado North (ftUS) [1979] NAD83(2011) / Colorado South [1980] NAD83(2011) / Colorado South (ftUS) [1981] NAD83(2011) / Connecticut [1982] NAD83(2011) / Connecticut (ftUS) [1983] NAD83(2011) / Conus Albers [1984] NAD83(2011) / Delaware [1985] NAD83(2011) / Delaware (ftUS) [1986] NAD83(2011) / EPSG Arctic zone 5-29 [1987] NAD83(2011) / EPSG Arctic zone 5-31 [1988] NAD83(2011) / EPSG Arctic zone 6-14 [1989] NAD83(2011) / EPSG Arctic zone 6-16 [1990] NAD83(2011) / Florida East [1991] NAD83(2011) / Florida East (ftUS) [1992] NAD83(2011) / Florida GDL Albers [1993] NAD83(2011) / Florida North [1994] NAD83(2011) / Florida North (ftUS) [1995] NAD83(2011) / Florida West [1996] NAD83(2011) / Florida West (ftUS) [1997] NAD83(2011) / Georgia East [1998] NAD83(2011) / Georgia East (ftUS) [1999] NAD83(2011) / Georgia West [2000] NAD83(2011) / Georgia West (ftUS) [2001] NAD83(2011) / IaRCS zone 1 [2002] NAD83(2011) / IaRCS zone 10 [2003] NAD83(2011) / IaRCS zone 11 [2004] NAD83(2011) / IaRCS zone 12 [2005] NAD83(2011) / IaRCS zone 13 [2006] NAD83(2011) / IaRCS zone 14 [2007] NAD83(2011) / IaRCS zone 2 [2008] NAD83(2011) / IaRCS zone 3 [2009] NAD83(2011) / IaRCS zone 4 [2010] NAD83(2011) / IaRCS zone 5 [2011] NAD83(2011) / IaRCS zone 6 [2012] NAD83(2011) / IaRCS zone 7 [2013] NAD83(2011) / IaRCS zone 8 [2014] NAD83(2011) / IaRCS zone 9 [2015] NAD83(2011) / Idaho Central [2016] NAD83(2011) / Idaho Central (ftUS) [2017] NAD83(2011) / Idaho East [2018] NAD83(2011) / Idaho East (ftUS) [2019] NAD83(2011) / Idaho West [2020] NAD83(2011) / Idaho West (ftUS) [2021] NAD83(2011) / Illinois East [2022] NAD83(2011) / Illinois East (ftUS) [2023] NAD83(2011) / Illinois West [2024] NAD83(2011) / Illinois West (ftUS) [2025] NAD83(2011) / InGCS Adams (ftUS) [2026] NAD83(2011) / InGCS Adams (m) [2027] NAD83(2011) / InGCS Allen (ftUS) [2028] NAD83(2011) / InGCS Allen (m) [2029] NAD83(2011) / InGCS Bartholomew (ftUS) [2030] NAD83(2011) / InGCS Bartholomew (m) [2031] NAD83(2011) / InGCS Benton (ftUS) [2032] NAD83(2011) / InGCS Benton (m) [2033] NAD83(2011) / InGCS Blackford-Delaware (ftUS) [2034] NAD83(2011) / InGCS Blackford-Delaware (m) [2035] NAD83(2011) / InGCS Boone-Hendricks (ftUS) [2036] NAD83(2011) / InGCS Boone-Hendricks (m) [2037] NAD83(2011) / InGCS Brown (ftUS) [2038] NAD83(2011) / InGCS Brown (m) [2039] NAD83(2011) / InGCS Carroll (ftUS) [2040] NAD83(2011) / InGCS Carroll (m) [2041] NAD83(2011) / InGCS Cass (ftUS) [2042] NAD83(2011) / InGCS Cass (m) [2043] NAD83(2011) / InGCS Clark-Floyd-Scott (ftUS) [2044] NAD83(2011) / InGCS Clark-Floyd-Scott (m) [2045] NAD83(2011) / InGCS Clay (ftUS) [2046] NAD83(2011) / InGCS Clay (m) [2047] NAD83(2011) / InGCS Clinton (ftUS) [2048] NAD83(2011) / InGCS Clinton (m) [2049] NAD83(2011) / InGCS Crawford-Lawrence-Orange (ftUS) [2050] NAD83(2011) / InGCS Crawford-Lawrence-Orange (m) [2051] NAD83(2011) / InGCS Daviess-Greene (ftUS) [2052] NAD83(2011) / InGCS Daviess-Greene (m) [2053] NAD83(2011) / InGCS DeKalb (ftUS) [2054] NAD83(2011) / InGCS DeKalb (m) [2055] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (ftUS) [2056] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (m) [2057] NAD83(2011) / InGCS Decatur-Rush (ftUS) [2058] NAD83(2011) / InGCS Decatur-Rush (m) [2059] NAD83(2011) / InGCS Dubois-Martin (ftUS) [2060] NAD83(2011) / InGCS Dubois-Martin (m) [2061] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (ftUS) [2062] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (m) [2063] NAD83(2011) / InGCS Fayette-Franklin-Union (ftUS) [2064] NAD83(2011) / InGCS Fayette-Franklin-Union (m) [2065] NAD83(2011) / InGCS Fountain-Warren (ftUS) [2066] NAD83(2011) / InGCS Fountain-Warren (m) [2067] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (ftUS) [2068] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (m) [2069] NAD83(2011) / InGCS Gibson (ftUS) [2070] NAD83(2011) / InGCS Gibson (m) [2071] NAD83(2011) / InGCS Grant (ftUS) [2072] NAD83(2011) / InGCS Grant (m) [2073] NAD83(2011) / InGCS Hamilton-Tipton (ftUS) [2074] NAD83(2011) / InGCS Hamilton-Tipton (m) [2075] NAD83(2011) / InGCS Hancock-Madison (ftUS) [2076] NAD83(2011) / InGCS Hancock-Madison (m) [2077] NAD83(2011) / InGCS Harrison-Washington (ftUS) [2078] NAD83(2011) / InGCS Harrison-Washington (m) [2079] NAD83(2011) / InGCS Henry (ftUS) [2080] NAD83(2011) / InGCS Henry (m) [2081] NAD83(2011) / InGCS Howard-Miami (ftUS) [2082] NAD83(2011) / InGCS Howard-Miami (m) [2083] NAD83(2011) / InGCS Huntington-Whitley (ftUS) [2084] NAD83(2011) / InGCS Huntington-Whitley (m) [2085] NAD83(2011) / InGCS Jackson (ftUS) [2086] NAD83(2011) / InGCS Jackson (m) [2087] NAD83(2011) / InGCS Jasper-Porter (ftUS) [2088] NAD83(2011) / InGCS Jasper-Porter (m) [2089] NAD83(2011) / InGCS Jay (ftUS) [2090] NAD83(2011) / InGCS Jay (m) [2091] NAD83(2011) / InGCS Jefferson (ftUS) [2092] NAD83(2011) / InGCS Jefferson (m) [2093] NAD83(2011) / InGCS Jennings (ftUS) [2094] NAD83(2011) / InGCS Jennings (m) [2095] NAD83(2011) / InGCS Johnson-Marion (ftUS) [2096] NAD83(2011) / InGCS Johnson-Marion (m) [2097] NAD83(2011) / InGCS Knox (ftUS) [2098] NAD83(2011) / InGCS Knox (m) [2099] NAD83(2011) / InGCS LaGrange-Noble (ftUS) [2100] NAD83(2011) / InGCS LaGrange-Noble (m) [2101] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (ftUS) [2102] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (m) [2103] NAD83(2011) / InGCS Lake-Newton (ftUS) [2104] NAD83(2011) / InGCS Lake-Newton (m) [2105] NAD83(2011) / InGCS Monroe-Morgan (ftUS) [2106] NAD83(2011) / InGCS Monroe-Morgan (m) [2107] NAD83(2011) / InGCS Montgomery-Putnam (ftUS) [2108] NAD83(2011) / InGCS Montgomery-Putnam (m) [2109] NAD83(2011) / InGCS Owen (ftUS) [2110] NAD83(2011) / InGCS Owen (m) [2111] NAD83(2011) / InGCS Parke-Vermillion (ftUS) [2112] NAD83(2011) / InGCS Parke-Vermillion (m) [2113] NAD83(2011) / InGCS Perry (ftUS) [2114] NAD83(2011) / InGCS Perry (m) [2115] NAD83(2011) / InGCS Pike-Warrick (ftUS) [2116] NAD83(2011) / InGCS Pike-Warrick (m) [2117] NAD83(2011) / InGCS Posey (ftUS) [2118] NAD83(2011) / InGCS Posey (m) [2119] NAD83(2011) / InGCS Randolph-Wayne (ftUS) [2120] NAD83(2011) / InGCS Randolph-Wayne (m) [2121] NAD83(2011) / InGCS Ripley (ftUS) [2122] NAD83(2011) / InGCS Ripley (m) [2123] NAD83(2011) / InGCS Shelby (ftUS) [2124] NAD83(2011) / InGCS Shelby (m) [2125] NAD83(2011) / InGCS Spencer (ftUS) [2126] NAD83(2011) / InGCS Spencer (m) [2127] NAD83(2011) / InGCS Steuben (ftUS) [2128] NAD83(2011) / InGCS Steuben (m) [2129] NAD83(2011) / InGCS Sullivan (ftUS) [2130] NAD83(2011) / InGCS Sullivan (m) [2131] NAD83(2011) / InGCS Tippecanoe-White (ftUS) [2132] NAD83(2011) / InGCS Tippecanoe-White (m) [2133] NAD83(2011) / InGCS Vanderburgh (ftUS) [2134] NAD83(2011) / InGCS Vanderburgh (m) [2135] NAD83(2011) / InGCS Vigo (ftUS) [2136] NAD83(2011) / InGCS Vigo (m) [2137] NAD83(2011) / InGCS Wells (ftUS) [2138] NAD83(2011) / InGCS Wells (m) [2139] NAD83(2011) / Indiana East [2140] NAD83(2011) / Indiana East (ftUS) [2141] NAD83(2011) / Indiana West [2142] NAD83(2011) / Indiana West (ftUS) [2143] NAD83(2011) / Iowa North [2144] NAD83(2011) / Iowa North (ftUS) [2145] NAD83(2011) / Iowa South [2146] NAD83(2011) / Iowa South (ftUS) [2147] NAD83(2011) / KS RCS zone 1 [2148] NAD83(2011) / KS RCS zone 10 [2149] NAD83(2011) / KS RCS zone 11 [2150] NAD83(2011) / KS RCS zone 12 [2151] NAD83(2011) / KS RCS zone 13 [2152] NAD83(2011) / KS RCS zone 14 [2153] NAD83(2011) / KS RCS zone 15 [2154] NAD83(2011) / KS RCS zone 16 [2155] NAD83(2011) / KS RCS zone 17 [2156] NAD83(2011) / KS RCS zone 18 [2157] NAD83(2011) / KS RCS zone 19 [2158] NAD83(2011) / KS RCS zone 2 [2159] NAD83(2011) / KS RCS zone 20 [2160] NAD83(2011) / KS RCS zone 3 [2161] NAD83(2011) / KS RCS zone 4 [2162] NAD83(2011) / KS RCS zone 5 [2163] NAD83(2011) / KS RCS zone 6 [2164] NAD83(2011) / KS RCS zone 7 [2165] NAD83(2011) / KS RCS zone 8 [2166] NAD83(2011) / KS RCS zone 9 [2167] NAD83(2011) / Kansas LCC [2168] NAD83(2011) / Kansas LCC (ftUS) [2169] NAD83(2011) / Kansas North [2170] NAD83(2011) / Kansas North (ftUS) [2171] NAD83(2011) / Kansas South [2172] NAD83(2011) / Kansas South (ftUS) [2173] NAD83(2011) / Kentucky North [2174] NAD83(2011) / Kentucky North (ftUS) [2175] NAD83(2011) / Kentucky Single Zone [2176] NAD83(2011) / Kentucky Single Zone (ftUS) [2177] NAD83(2011) / Kentucky South [2178] NAD83(2011) / Kentucky South (ftUS) [2179] NAD83(2011) / Louisiana North [2180] NAD83(2011) / Louisiana North (ftUS) [2181] NAD83(2011) / Louisiana South [2182] NAD83(2011) / Louisiana South (ftUS) [2183] NAD83(2011) / Maine CS2000 Central [2184] NAD83(2011) / Maine CS2000 East [2185] NAD83(2011) / Maine CS2000 West [2186] NAD83(2011) / Maine East [2187] NAD83(2011) / Maine East (ftUS) [2188] NAD83(2011) / Maine West [2189] NAD83(2011) / Maine West (ftUS) [2190] NAD83(2011) / Maryland [2191] NAD83(2011) / Maryland (ftUS) [2192] NAD83(2011) / Massachusetts Island [2193] NAD83(2011) / Massachusetts Island (ftUS) [2194] NAD83(2011) / Massachusetts Mainland [2195] NAD83(2011) / Massachusetts Mainland (ftUS) [2196] NAD83(2011) / Michigan Central [2197] NAD83(2011) / Michigan Central (ft) [2198] NAD83(2011) / Michigan North [2199] NAD83(2011) / Michigan North (ft) [2200] NAD83(2011) / Michigan Oblique Mercator [2201] NAD83(2011) / Michigan South [2202] NAD83(2011) / Michigan South (ft) [2203] NAD83(2011) / Minnesota Central [2204] NAD83(2011) / Minnesota Central (ftUS) [2205] NAD83(2011) / Minnesota North [2206] NAD83(2011) / Minnesota North (ftUS) [2207] NAD83(2011) / Minnesota South [2208] NAD83(2011) / Minnesota South (ftUS) [2209] NAD83(2011) / Mississippi East [2210] NAD83(2011) / Mississippi East (ftUS) [2211] NAD83(2011) / Mississippi TM [2212] NAD83(2011) / Mississippi West [2213] NAD83(2011) / Mississippi West (ftUS) [2214] NAD83(2011) / Missouri Central [2215] NAD83(2011) / Missouri East [2216] NAD83(2011) / Missouri West [2217] NAD83(2011) / Montana [2218] NAD83(2011) / Montana (ft) [2219] NAD83(2011) / NCRS Las Vegas (ftUS) [2220] NAD83(2011) / NCRS Las Vegas (m) [2221] NAD83(2011) / NCRS Las Vegas high (ftUS) [2222] NAD83(2011) / NCRS Las Vegas high (m) [2223] NAD83(2011) / Nebraska [2224] NAD83(2011) / Nebraska (ftUS) [2225] NAD83(2011) / Nebraska (ftUS) (deprecated) [2226] NAD83(2011) / Nevada Central [2227] NAD83(2011) / Nevada Central (ftUS) [2228] NAD83(2011) / Nevada East [2229] NAD83(2011) / Nevada East (ftUS) [2230] NAD83(2011) / Nevada West [2231] NAD83(2011) / Nevada West (ftUS) [2232] NAD83(2011) / New Hampshire [2233] NAD83(2011) / New Hampshire (ftUS) [2234] NAD83(2011) / New Jersey [2235] NAD83(2011) / New Jersey (ftUS) [2236] NAD83(2011) / New Mexico Central [2237] NAD83(2011) / New Mexico Central (ftUS) [2238] NAD83(2011) / New Mexico East [2239] NAD83(2011) / New Mexico East (ftUS) [2240] NAD83(2011) / New Mexico West [2241] NAD83(2011) / New Mexico West (ftUS) [2242] NAD83(2011) / New York Central [2243] NAD83(2011) / New York Central (ftUS) [2244] NAD83(2011) / New York East [2245] NAD83(2011) / New York East (ftUS) [2246] NAD83(2011) / New York Long Island [2247] NAD83(2011) / New York Long Island (ftUS) [2248] NAD83(2011) / New York West [2249] NAD83(2011) / New York West (ftUS) [2250] NAD83(2011) / North Carolina [2251] NAD83(2011) / North Carolina (ftUS) [2252] NAD83(2011) / North Dakota North [2253] NAD83(2011) / North Dakota North (ft) [2254] NAD83(2011) / North Dakota South [2255] NAD83(2011) / North Dakota South (ft) [2256] NAD83(2011) / Ohio North [2257] NAD83(2011) / Ohio North (ftUS) [2258] NAD83(2011) / Ohio South [2259] NAD83(2011) / Ohio South (ftUS) [2260] NAD83(2011) / Oklahoma North [2261] NAD83(2011) / Oklahoma North (ftUS) [2262] NAD83(2011) / Oklahoma South [2263] NAD83(2011) / Oklahoma South (ftUS) [2264] NAD83(2011) / Oregon Baker zone (ft) [2265] NAD83(2011) / Oregon Baker zone (m) [2266] NAD83(2011) / Oregon Bend-Burns zone (ft) [2267] NAD83(2011) / Oregon Bend-Burns zone (m) [2268] NAD83(2011) / Oregon Bend-Klamath Falls zone (ft) [2269] NAD83(2011) / Oregon Bend-Klamath Falls zone (m) [2270] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (ft) [2271] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (m) [2272] NAD83(2011) / Oregon Burns-Harper zone (ft) [2273] NAD83(2011) / Oregon Burns-Harper zone (m) [2274] NAD83(2011) / Oregon Canyon City-Burns zone (ft) [2275] NAD83(2011) / Oregon Canyon City-Burns zone (m) [2276] NAD83(2011) / Oregon Canyonville-Grants Pass zone (ft) [2277] NAD83(2011) / Oregon Canyonville-Grants Pass zone (m) [2278] NAD83(2011) / Oregon Coast Range North zone (ft) [2279] NAD83(2011) / Oregon Coast Range North zone (m) [2280] NAD83(2011) / Oregon Coast zone (ft) [2281] NAD83(2011) / Oregon Coast zone (m) [2282] NAD83(2011) / Oregon Columbia River East zone (ft) [2283] NAD83(2011) / Oregon Columbia River East zone (m) [2284] NAD83(2011) / Oregon Columbia River West zone (ft) [2285] NAD83(2011) / Oregon Columbia River West zone (m) [2286] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (ft) [2287] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (m) [2288] NAD83(2011) / Oregon Dayville-Prairie City zone (ft) [2289] NAD83(2011) / Oregon Dayville-Prairie City zone (m) [2290] NAD83(2011) / Oregon Denio-Burns zone (ft) [2291] NAD83(2011) / Oregon Denio-Burns zone (m) [2292] NAD83(2011) / Oregon Dufur-Madras zone (ft) [2293] NAD83(2011) / Oregon Dufur-Madras zone (m) [2294] NAD83(2011) / Oregon Eugene zone (ft) [2295] NAD83(2011) / Oregon Eugene zone (m) [2296] NAD83(2011) / Oregon GIC Lambert (ft) [2297] NAD83(2011) / Oregon Grants Pass-Ashland zone (ft) [2298] NAD83(2011) / Oregon Grants Pass-Ashland zone (m) [2299] NAD83(2011) / Oregon Gresham-Warm Springs zone (ft) [2300] NAD83(2011) / Oregon Gresham-Warm Springs zone (m) [2301] NAD83(2011) / Oregon Halfway zone (ft) [2302] NAD83(2011) / Oregon Halfway zone (m) [2303] NAD83(2011) / Oregon LCC (m) [2304] NAD83(2011) / Oregon La Grande zone (ft) [2305] NAD83(2011) / Oregon La Grande zone (m) [2306] NAD83(2011) / Oregon Medford-Diamond Lake zone (ft) [2307] NAD83(2011) / Oregon Medford-Diamond Lake zone (m) [2308] NAD83(2011) / Oregon Mitchell zone (ft) [2309] NAD83(2011) / Oregon Mitchell zone (m) [2310] NAD83(2011) / Oregon North [2311] NAD83(2011) / Oregon North (ft) [2312] NAD83(2011) / Oregon North Central zone (ft) [2313] NAD83(2011) / Oregon North Central zone (m) [2314] NAD83(2011) / Oregon Ochoco Summit zone (ft) [2315] NAD83(2011) / Oregon Ochoco Summit zone (m) [2316] NAD83(2011) / Oregon Ontario zone (ft) [2317] NAD83(2011) / Oregon Ontario zone (m) [2318] NAD83(2011) / Oregon Owyhee zone (ft) [2319] NAD83(2011) / Oregon Owyhee zone (m) [2320] NAD83(2011) / Oregon Pendleton zone (ft) [2321] NAD83(2011) / Oregon Pendleton zone (m) [2322] NAD83(2011) / Oregon Pendleton-La Grande zone (ft) [2323] NAD83(2011) / Oregon Pendleton-La Grande zone (m) [2324] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (ft) [2325] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (m) [2326] NAD83(2011) / Oregon Portland zone (ft) [2327] NAD83(2011) / Oregon Portland zone (m) [2328] NAD83(2011) / Oregon Prairie City-Brogan zone (ft) [2329] NAD83(2011) / Oregon Prairie City-Brogan zone (m) [2330] NAD83(2011) / Oregon Riley-Lakeview zone (ft) [2331] NAD83(2011) / Oregon Riley-Lakeview zone (m) [2332] NAD83(2011) / Oregon Salem zone (ft) [2333] NAD83(2011) / Oregon Salem zone (m) [2334] NAD83(2011) / Oregon Santiam Pass zone (ft) [2335] NAD83(2011) / Oregon Santiam Pass zone (m) [2336] NAD83(2011) / Oregon Siskiyou Pass zone (ft) [2337] NAD83(2011) / Oregon Siskiyou Pass zone (m) [2338] NAD83(2011) / Oregon South [2339] NAD83(2011) / Oregon South (ft) [2340] NAD83(2011) / Oregon Ukiah-Fox zone (ft) [2341] NAD83(2011) / Oregon Ukiah-Fox zone (m) [2342] NAD83(2011) / Oregon Wallowa zone (ft) [2343] NAD83(2011) / Oregon Wallowa zone (m) [2344] NAD83(2011) / Oregon Warner Highway zone (ft) [2345] NAD83(2011) / Oregon Warner Highway zone (m) [2346] NAD83(2011) / Oregon Willamette Pass zone (ft) [2347] NAD83(2011) / Oregon Willamette Pass zone (m) [2348] NAD83(2011) / PCCS zone 1 (ft) [2349] NAD83(2011) / PCCS zone 2 (ft) [2350] NAD83(2011) / PCCS zone 3 (ft) [2351] NAD83(2011) / PCCS zone 4 (ft) [2352] NAD83(2011) / Pennsylvania North [2353] NAD83(2011) / Pennsylvania North (ftUS) [2354] NAD83(2011) / Pennsylvania South [2355] NAD83(2011) / Pennsylvania South (ftUS) [2356] NAD83(2011) / Puerto Rico and Virgin Is. [2357] NAD83(2011) / RMTCRS Billings (ft) [2358] NAD83(2011) / RMTCRS Billings (m) [2359] NAD83(2011) / RMTCRS Blackfeet (ft) [2360] NAD83(2011) / RMTCRS Blackfeet (m) [2361] NAD83(2011) / RMTCRS Bobcat (ft) [2362] NAD83(2011) / RMTCRS Bobcat (m) [2363] NAD83(2011) / RMTCRS Crow (ft) [2364] NAD83(2011) / RMTCRS Crow (m) [2365] NAD83(2011) / RMTCRS Fort Belknap (ft) [2366] NAD83(2011) / RMTCRS Fort Belknap (m) [2367] NAD83(2011) / RMTCRS Fort Peck Assiniboine (ft) [2368] NAD83(2011) / RMTCRS Fort Peck Assiniboine (m) [2369] NAD83(2011) / RMTCRS Fort Peck Sioux (ft) [2370] NAD83(2011) / RMTCRS Fort Peck Sioux (m) [2371] NAD83(2011) / RMTCRS Milk River (ft) [2372] NAD83(2011) / RMTCRS Milk River (m) [2373] NAD83(2011) / RMTCRS St Mary (ft) [2374] NAD83(2011) / RMTCRS St Mary (m) [2375] NAD83(2011) / RMTCRS Wind River (ftUS) [2376] NAD83(2011) / RMTCRS Wind River (m) [2377] NAD83(2011) / Rhode Island [2378] NAD83(2011) / Rhode Island (ftUS) [2379] NAD83(2011) / San Francisco CS13 [2380] NAD83(2011) / San Francisco CS13 (deprecated) [2381] NAD83(2011) / San Francisco CS13 (ftUS) [2382] NAD83(2011) / San Francisco CS13 (ftUS) (deprecated) [2383] NAD83(2011) / South Carolina [2384] NAD83(2011) / South Carolina (ft) [2385] NAD83(2011) / South Dakota North [2386] NAD83(2011) / South Dakota North (ftUS) [2387] NAD83(2011) / South Dakota South [2388] NAD83(2011) / South Dakota South (ftUS) [2389] NAD83(2011) / Tennessee [2390] NAD83(2011) / Tennessee (ftUS) [2391] NAD83(2011) / Texas Central [2392] NAD83(2011) / Texas Central (ftUS) [2393] NAD83(2011) / Texas Centric Albers Equal Area [2394] NAD83(2011) / Texas Centric Lambert Conformal [2395] NAD83(2011) / Texas North [2396] NAD83(2011) / Texas North (ftUS) [2397] NAD83(2011) / Texas North Central [2398] NAD83(2011) / Texas North Central (ftUS) [2399] NAD83(2011) / Texas South [2400] NAD83(2011) / Texas South (ftUS) [2401] NAD83(2011) / Texas South Central [2402] NAD83(2011) / Texas South Central (ftUS) [2403] NAD83(2011) / UTM zone 10N [2404] NAD83(2011) / UTM zone 11N [2405] NAD83(2011) / UTM zone 12N [2406] NAD83(2011) / UTM zone 13N [2407] NAD83(2011) / UTM zone 14N [2408] NAD83(2011) / UTM zone 15N [2409] NAD83(2011) / UTM zone 16N [2410] NAD83(2011) / UTM zone 17N [2411] NAD83(2011) / UTM zone 18N [2412] NAD83(2011) / UTM zone 19N [2413] NAD83(2011) / UTM zone 1N [2414] NAD83(2011) / UTM zone 2N [2415] NAD83(2011) / UTM zone 3N [2416] NAD83(2011) / UTM zone 4N [2417] NAD83(2011) / UTM zone 59N [2418] NAD83(2011) / UTM zone 5N [2419] NAD83(2011) / UTM zone 60N [2420] NAD83(2011) / UTM zone 6N [2421] NAD83(2011) / UTM zone 7N [2422] NAD83(2011) / UTM zone 8N [2423] NAD83(2011) / UTM zone 9N [2424] NAD83(2011) / Utah Central [2425] NAD83(2011) / Utah Central (ftUS) [2426] NAD83(2011) / Utah North [2427] NAD83(2011) / Utah North (ftUS) [2428] NAD83(2011) / Utah South [2429] NAD83(2011) / Utah South (ftUS) [2430] NAD83(2011) / Vermont [2431] NAD83(2011) / Vermont (ftUS) [2432] NAD83(2011) / Virginia Lambert [2433] NAD83(2011) / Virginia North [2434] NAD83(2011) / Virginia North (ftUS) [2435] NAD83(2011) / Virginia South [2436] NAD83(2011) / Virginia South (ftUS) [2437] NAD83(2011) / WISCRS Adams and Juneau (ftUS) [2438] NAD83(2011) / WISCRS Adams and Juneau (m) [2439] NAD83(2011) / WISCRS Ashland (ftUS) [2440] NAD83(2011) / WISCRS Ashland (m) [2441] NAD83(2011) / WISCRS Barron (ftUS) [2442] NAD83(2011) / WISCRS Barron (m) [2443] NAD83(2011) / WISCRS Bayfield (ftUS) [2444] NAD83(2011) / WISCRS Bayfield (m) [2445] NAD83(2011) / WISCRS Brown (ftUS) [2446] NAD83(2011) / WISCRS Brown (m) [2447] NAD83(2011) / WISCRS Buffalo (ftUS) [2448] NAD83(2011) / WISCRS Buffalo (m) [2449] NAD83(2011) / WISCRS Burnett (ftUS) [2450] NAD83(2011) / WISCRS Burnett (m) [2451] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2452] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2453] NAD83(2011) / WISCRS Chippewa (ftUS) [2454] NAD83(2011) / WISCRS Chippewa (m) [2455] NAD83(2011) / WISCRS Clark (ftUS) [2456] NAD83(2011) / WISCRS Clark (m) [2457] NAD83(2011) / WISCRS Columbia (ftUS) [2458] NAD83(2011) / WISCRS Columbia (m) [2459] NAD83(2011) / WISCRS Crawford (ftUS) [2460] NAD83(2011) / WISCRS Crawford (m) [2461] NAD83(2011) / WISCRS Dane (ftUS) [2462] NAD83(2011) / WISCRS Dane (m) [2463] NAD83(2011) / WISCRS Dodge and Jefferson (ftUS) [2464] NAD83(2011) / WISCRS Dodge and Jefferson (m) [2465] NAD83(2011) / WISCRS Door (ftUS) [2466] NAD83(2011) / WISCRS Door (m) [2467] NAD83(2011) / WISCRS Douglas (ftUS) [2468] NAD83(2011) / WISCRS Douglas (m) [2469] NAD83(2011) / WISCRS Dunn (ftUS) [2470] NAD83(2011) / WISCRS Dunn (m) [2471] NAD83(2011) / WISCRS Eau Claire (ftUS) [2472] NAD83(2011) / WISCRS Eau Claire (m) [2473] NAD83(2011) / WISCRS Florence (ftUS) [2474] NAD83(2011) / WISCRS Florence (m) [2475] NAD83(2011) / WISCRS Forest (ftUS) [2476] NAD83(2011) / WISCRS Forest (m) [2477] NAD83(2011) / WISCRS Grant (ftUS) [2478] NAD83(2011) / WISCRS Grant (m) [2479] NAD83(2011) / WISCRS Green Lake and Marquette (ftUS) [2480] NAD83(2011) / WISCRS Green Lake and Marquette (m) [2481] NAD83(2011) / WISCRS Green and Lafayette (ftUS) [2482] NAD83(2011) / WISCRS Green and Lafayette (m) [2483] NAD83(2011) / WISCRS Iowa (ftUS) [2484] NAD83(2011) / WISCRS Iowa (m) [2485] NAD83(2011) / WISCRS Iron (ftUS) [2486] NAD83(2011) / WISCRS Iron (m) [2487] NAD83(2011) / WISCRS Jackson (ftUS) [2488] NAD83(2011) / WISCRS Jackson (m) [2489] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [2490] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [2491] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [2492] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [2493] NAD83(2011) / WISCRS La Crosse (ftUS) [2494] NAD83(2011) / WISCRS La Crosse (m) [2495] NAD83(2011) / WISCRS Langlade (ftUS) [2496] NAD83(2011) / WISCRS Langlade (m) [2497] NAD83(2011) / WISCRS Lincoln (ftUS) [2498] NAD83(2011) / WISCRS Lincoln (m) [2499] NAD83(2011) / WISCRS Marathon (ftUS) [2500] NAD83(2011) / WISCRS Marathon (m) [2501] NAD83(2011) / WISCRS Marinette (ftUS) [2502] NAD83(2011) / WISCRS Marinette (m) [2503] NAD83(2011) / WISCRS Menominee (ftUS) [2504] NAD83(2011) / WISCRS Menominee (m) [2505] NAD83(2011) / WISCRS Monroe (ftUS) [2506] NAD83(2011) / WISCRS Monroe (m) [2507] NAD83(2011) / WISCRS Oconto (ftUS) [2508] NAD83(2011) / WISCRS Oconto (m) [2509] NAD83(2011) / WISCRS Oneida (ftUS) [2510] NAD83(2011) / WISCRS Oneida (m) [2511] NAD83(2011) / WISCRS Pepin and Pierce (ftUS) [2512] NAD83(2011) / WISCRS Pepin and Pierce (m) [2513] NAD83(2011) / WISCRS Polk (ftUS) [2514] NAD83(2011) / WISCRS Polk (m) [2515] NAD83(2011) / WISCRS Portage (ftUS) [2516] NAD83(2011) / WISCRS Portage (m) [2517] NAD83(2011) / WISCRS Price (ftUS) [2518] NAD83(2011) / WISCRS Price (m) [2519] NAD83(2011) / WISCRS Richland (ftUS) [2520] NAD83(2011) / WISCRS Richland (m) [2521] NAD83(2011) / WISCRS Rock (ftUS) [2522] NAD83(2011) / WISCRS Rock (m) [2523] NAD83(2011) / WISCRS Rusk (ftUS) [2524] NAD83(2011) / WISCRS Rusk (m) [2525] NAD83(2011) / WISCRS Sauk (ftUS) [2526] NAD83(2011) / WISCRS Sauk (m) [2527] NAD83(2011) / WISCRS Sawyer (ftUS) [2528] NAD83(2011) / WISCRS Sawyer (m) [2529] NAD83(2011) / WISCRS Shawano (ftUS) [2530] NAD83(2011) / WISCRS Shawano (m) [2531] NAD83(2011) / WISCRS St. Croix (ftUS) [2532] NAD83(2011) / WISCRS St. Croix (m) [2533] NAD83(2011) / WISCRS Taylor (ftUS) [2534] NAD83(2011) / WISCRS Taylor (m) [2535] NAD83(2011) / WISCRS Trempealeau (ftUS) [2536] NAD83(2011) / WISCRS Trempealeau (m) [2537] NAD83(2011) / WISCRS Vernon (ftUS) [2538] NAD83(2011) / WISCRS Vernon (m) [2539] NAD83(2011) / WISCRS Vilas (ftUS) [2540] NAD83(2011) / WISCRS Vilas (m) [2541] NAD83(2011) / WISCRS Walworth (ftUS) [2542] NAD83(2011) / WISCRS Walworth (m) [2543] NAD83(2011) / WISCRS Washburn (ftUS) [2544] NAD83(2011) / WISCRS Washburn (m) [2545] NAD83(2011) / WISCRS Washington (ftUS) [2546] NAD83(2011) / WISCRS Washington (m) [2547] NAD83(2011) / WISCRS Waukesha (ftUS) [2548] NAD83(2011) / WISCRS Waukesha (m) [2549] NAD83(2011) / WISCRS Waupaca (ftUS) [2550] NAD83(2011) / WISCRS Waupaca (m) [2551] NAD83(2011) / WISCRS Waushara (ftUS) [2552] NAD83(2011) / WISCRS Waushara (m) [2553] NAD83(2011) / WISCRS Wood (ftUS) [2554] NAD83(2011) / WISCRS Wood (m) [2555] NAD83(2011) / Washington North [2556] NAD83(2011) / Washington North (ftUS) [2557] NAD83(2011) / Washington South [2558] NAD83(2011) / Washington South (ftUS) [2559] NAD83(2011) / West Virginia North [2560] NAD83(2011) / West Virginia North (ftUS) [2561] NAD83(2011) / West Virginia South [2562] NAD83(2011) / West Virginia South (ftUS) [2563] NAD83(2011) / Wisconsin Central [2564] NAD83(2011) / Wisconsin Central (deprecated) [2565] NAD83(2011) / Wisconsin Central (ftUS) [2566] NAD83(2011) / Wisconsin North [2567] NAD83(2011) / Wisconsin North (ftUS) [2568] NAD83(2011) / Wisconsin South [2569] NAD83(2011) / Wisconsin South (ftUS) [2570] NAD83(2011) / Wisconsin Transverse Mercator [2571] NAD83(2011) / Wyoming East [2572] NAD83(2011) / Wyoming East (ftUS) [2573] NAD83(2011) / Wyoming East Central [2574] NAD83(2011) / Wyoming East Central (ftUS) [2575] NAD83(2011) / Wyoming West [2576] NAD83(2011) / Wyoming West (ftUS) [2577] NAD83(2011) / Wyoming West Central [2578] NAD83(2011) / Wyoming West Central (ftUS) [2579] NAD83(CORS96) / Oregon Baker zone (ft) [2580] NAD83(CORS96) / Oregon Baker zone (m) [2581] NAD83(CORS96) / Oregon Bend-Burns zone (ft) [2582] NAD83(CORS96) / Oregon Bend-Burns zone (m) [2583] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (ft) [2584] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (m) [2585] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (ft) [2586] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (m) [2587] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (ft) [2588] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (m) [2589] NAD83(CORS96) / Oregon Coast zone (ft) [2590] NAD83(CORS96) / Oregon Coast zone (m) [2591] NAD83(CORS96) / Oregon Columbia River East zone (ft) [2592] NAD83(CORS96) / Oregon Columbia River East zone (m) [2593] NAD83(CORS96) / Oregon Columbia River West zone (ft) [2594] NAD83(CORS96) / Oregon Columbia River West zone (m) [2595] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (ft) [2596] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (m) [2597] NAD83(CORS96) / Oregon Dufur-Madras zone (ft) [2598] NAD83(CORS96) / Oregon Dufur-Madras zone (m) [2599] NAD83(CORS96) / Oregon Eugene zone (ft) [2600] NAD83(CORS96) / Oregon Eugene zone (m) [2601] NAD83(CORS96) / Oregon GIC Lambert (ft) [2602] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (ft) [2603] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (m) [2604] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (ft) [2605] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (m) [2606] NAD83(CORS96) / Oregon LCC (m) [2607] NAD83(CORS96) / Oregon La Grande zone (ft) [2608] NAD83(CORS96) / Oregon La Grande zone (m) [2609] NAD83(CORS96) / Oregon North [2610] NAD83(CORS96) / Oregon North (ft) [2611] NAD83(CORS96) / Oregon Ontario zone (ft) [2612] NAD83(CORS96) / Oregon Ontario zone (m) [2613] NAD83(CORS96) / Oregon Pendleton zone (ft) [2614] NAD83(CORS96) / Oregon Pendleton zone (m) [2615] NAD83(CORS96) / Oregon Pendleton-La Grande zone (ft) [2616] NAD83(CORS96) / Oregon Pendleton-La Grande zone (m) [2617] NAD83(CORS96) / Oregon Portland zone (ft) [2618] NAD83(CORS96) / Oregon Portland zone (m) [2619] NAD83(CORS96) / Oregon Salem zone (ft) [2620] NAD83(CORS96) / Oregon Salem zone (m) [2621] NAD83(CORS96) / Oregon Santiam Pass zone (ft) [2622] NAD83(CORS96) / Oregon Santiam Pass zone (m) [2623] NAD83(CORS96) / Oregon South [2624] NAD83(CORS96) / Oregon South (ft) [2625] NAD83(CORS96) / Puerto Rico and Virgin Is. [2626] NAD83(CSRS) / Alberta 10-TM (Forest) [2627] NAD83(CSRS) / Alberta 10-TM (Resource) [2628] NAD83(CSRS) / Alberta 3TM ref merid 111 W [2629] NAD83(CSRS) / Alberta 3TM ref merid 114 W [2630] NAD83(CSRS) / Alberta 3TM ref merid 117 W [2631] NAD83(CSRS) / Alberta 3TM ref merid 120 W [2632] NAD83(CSRS) / Alberta 3TM ref merid 120 W (deprecated) [2633] NAD83(CSRS) / BC Albers [2634] NAD83(CSRS) / Canada Atlas Lambert [2635] NAD83(CSRS) / EPSG Arctic zone 1-23 [2636] NAD83(CSRS) / EPSG Arctic zone 2-14 [2637] NAD83(CSRS) / EPSG Arctic zone 2-16 [2638] NAD83(CSRS) / EPSG Arctic zone 3-25 [2639] NAD83(CSRS) / EPSG Arctic zone 3-27 [2640] NAD83(CSRS) / EPSG Arctic zone 3-29 [2641] NAD83(CSRS) / EPSG Arctic zone 4-14 [2642] NAD83(CSRS) / EPSG Arctic zone 4-16 [2643] NAD83(CSRS) / EPSG Arctic zone 4-18 [2644] NAD83(CSRS) / EPSG Arctic zone 5-33 [2645] NAD83(CSRS) / EPSG Arctic zone 5-35 [2646] NAD83(CSRS) / EPSG Arctic zone 5-37 [2647] NAD83(CSRS) / EPSG Arctic zone 5-39 [2648] NAD83(CSRS) / EPSG Arctic zone 6-18 [2649] NAD83(CSRS) / EPSG Arctic zone 6-20 [2650] NAD83(CSRS) / EPSG Arctic zone 6-22 [2651] NAD83(CSRS) / EPSG Arctic zone 6-24 [2652] NAD83(CSRS) / MTM zone 1 [2653] NAD83(CSRS) / MTM zone 10 [2654] NAD83(CSRS) / MTM zone 11 [2655] NAD83(CSRS) / MTM zone 12 [2656] NAD83(CSRS) / MTM zone 13 [2657] NAD83(CSRS) / MTM zone 14 [2658] NAD83(CSRS) / MTM zone 15 [2659] NAD83(CSRS) / MTM zone 16 [2660] NAD83(CSRS) / MTM zone 17 [2661] NAD83(CSRS) / MTM zone 2 [2662] NAD83(CSRS) / MTM zone 3 [2663] NAD83(CSRS) / MTM zone 4 [2664] NAD83(CSRS) / MTM zone 5 [2665] NAD83(CSRS) / MTM zone 6 [2666] NAD83(CSRS) / MTM zone 7 [2667] NAD83(CSRS) / MTM zone 8 [2668] NAD83(CSRS) / MTM zone 9 [2669] NAD83(CSRS) / MTQ Lambert [2670] NAD83(CSRS) / NWT Lambert [2671] NAD83(CSRS) / New Brunswick Stereographic [2672] NAD83(CSRS) / Ontario MNR Lambert [2673] NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83) [2674] NAD83(CSRS) / Quebec Albers [2675] NAD83(CSRS) / Quebec Lambert [2676] NAD83(CSRS) / SCoPQ zone 2 (deprecated) [2677] NAD83(CSRS) / Statistics Canada Lambert [2678] NAD83(CSRS) / Teranet Ontario Lambert [2679] NAD83(CSRS) / UTM zone 10N [2680] NAD83(CSRS) / UTM zone 11N [2681] NAD83(CSRS) / UTM zone 12N [2682] NAD83(CSRS) / UTM zone 13N [2683] NAD83(CSRS) / UTM zone 14N [2684] NAD83(CSRS) / UTM zone 15N [2685] NAD83(CSRS) / UTM zone 16N [2686] NAD83(CSRS) / UTM zone 17N [2687] NAD83(CSRS) / UTM zone 18N [2688] NAD83(CSRS) / UTM zone 19N [2689] NAD83(CSRS) / UTM zone 20N [2690] NAD83(CSRS) / UTM zone 21N [2691] NAD83(CSRS) / UTM zone 22N [2692] NAD83(CSRS) / UTM zone 7N [2693] NAD83(CSRS) / UTM zone 8N [2694] NAD83(CSRS) / UTM zone 9N [2695] NAD83(CSRS) / Yukon Albers [2696] NAD83(CSRS)v6 / MTM Nova Scotia zone 4 [2697] NAD83(CSRS)v6 / MTM Nova Scotia zone 5 [2698] NAD83(CSRS98) / MTM zone 10 (deprecated) [2699] NAD83(CSRS98) / MTM zone 3 (deprecated) [2700] NAD83(CSRS98) / MTM zone 4 (deprecated) [2701] NAD83(CSRS98) / MTM zone 5 (deprecated) [2702] NAD83(CSRS98) / MTM zone 6 (deprecated) [2703] NAD83(CSRS98) / MTM zone 7 (deprecated) [2704] NAD83(CSRS98) / MTM zone 8 (deprecated) [2705] NAD83(CSRS98) / MTM zone 9 (deprecated) [2706] NAD83(CSRS98) / New Brunswick Stereo (deprecated) [2707] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2708] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2709] NAD83(CSRS98) / SCoPQ zone 2 (deprecated) [2710] NAD83(CSRS98) / UTM zone 11N (deprecated) [2711] NAD83(CSRS98) / UTM zone 12N (deprecated) [2712] NAD83(CSRS98) / UTM zone 13N (deprecated) [2713] NAD83(CSRS98) / UTM zone 17N (deprecated) [2714] NAD83(CSRS98) / UTM zone 18N (deprecated) [2715] NAD83(CSRS98) / UTM zone 19N (deprecated) [2716] NAD83(CSRS98) / UTM zone 20N (deprecated) [2717] NAD83(CSRS98) / UTM zone 21N (deprecated) [2718] NAD83(HARN) / Alabama East [2719] NAD83(HARN) / Alabama West [2720] NAD83(HARN) / Arizona Central [2721] NAD83(HARN) / Arizona Central (ft) [2722] NAD83(HARN) / Arizona East [2723] NAD83(HARN) / Arizona East (ft) [2724] NAD83(HARN) / Arizona West [2725] NAD83(HARN) / Arizona West (ft) [2726] NAD83(HARN) / Arkansas North [2727] NAD83(HARN) / Arkansas North (ftUS) [2728] NAD83(HARN) / Arkansas South [2729] NAD83(HARN) / Arkansas South (ftUS) [2730] NAD83(HARN) / California Albers [2731] NAD83(HARN) / California zone 1 [2732] NAD83(HARN) / California zone 1 (ftUS) [2733] NAD83(HARN) / California zone 2 [2734] NAD83(HARN) / California zone 2 (ftUS) [2735] NAD83(HARN) / California zone 3 [2736] NAD83(HARN) / California zone 3 (ftUS) [2737] NAD83(HARN) / California zone 4 [2738] NAD83(HARN) / California zone 4 (ftUS) [2739] NAD83(HARN) / California zone 5 [2740] NAD83(HARN) / California zone 5 (ftUS) [2741] NAD83(HARN) / California zone 6 [2742] NAD83(HARN) / California zone 6 (ftUS) [2743] NAD83(HARN) / Colorado Central [2744] NAD83(HARN) / Colorado Central (ftUS) [2745] NAD83(HARN) / Colorado North [2746] NAD83(HARN) / Colorado North (ftUS) [2747] NAD83(HARN) / Colorado South [2748] NAD83(HARN) / Colorado South (ftUS) [2749] NAD83(HARN) / Connecticut [2750] NAD83(HARN) / Connecticut (ftUS) [2751] NAD83(HARN) / Conus Albers [2752] NAD83(HARN) / Delaware [2753] NAD83(HARN) / Delaware (ftUS) [2754] NAD83(HARN) / Florida East [2755] NAD83(HARN) / Florida East (ftUS) [2756] NAD83(HARN) / Florida GDL Albers [2757] NAD83(HARN) / Florida North [2758] NAD83(HARN) / Florida North (ftUS) [2759] NAD83(HARN) / Florida West [2760] NAD83(HARN) / Florida West (ftUS) [2761] NAD83(HARN) / Georgia East [2762] NAD83(HARN) / Georgia East (ftUS) [2763] NAD83(HARN) / Georgia West [2764] NAD83(HARN) / Georgia West (ftUS) [2765] NAD83(HARN) / Guam Map Grid [2766] NAD83(HARN) / Hawaii zone 1 [2767] NAD83(HARN) / Hawaii zone 2 [2768] NAD83(HARN) / Hawaii zone 3 [2769] NAD83(HARN) / Hawaii zone 3 (ftUS) [2770] NAD83(HARN) / Hawaii zone 4 [2771] NAD83(HARN) / Hawaii zone 5 [2772] NAD83(HARN) / Idaho Central [2773] NAD83(HARN) / Idaho Central (ftUS) [2774] NAD83(HARN) / Idaho East [2775] NAD83(HARN) / Idaho East (ftUS) [2776] NAD83(HARN) / Idaho West [2777] NAD83(HARN) / Idaho West (ftUS) [2778] NAD83(HARN) / Illinois East [2779] NAD83(HARN) / Illinois East (ftUS) [2780] NAD83(HARN) / Illinois West [2781] NAD83(HARN) / Illinois West (ftUS) [2782] NAD83(HARN) / Indiana East [2783] NAD83(HARN) / Indiana East (ftUS) [2784] NAD83(HARN) / Indiana East (ftUS) (deprecated) [2785] NAD83(HARN) / Indiana West [2786] NAD83(HARN) / Indiana West (ftUS) [2787] NAD83(HARN) / Indiana West (ftUS) (deprecated) [2788] NAD83(HARN) / Iowa North [2789] NAD83(HARN) / Iowa North (ftUS) [2790] NAD83(HARN) / Iowa South [2791] NAD83(HARN) / Iowa South (ftUS) [2792] NAD83(HARN) / Kansas North [2793] NAD83(HARN) / Kansas North (ftUS) [2794] NAD83(HARN) / Kansas South [2795] NAD83(HARN) / Kansas South (ftUS) [2796] NAD83(HARN) / Kentucky North [2797] NAD83(HARN) / Kentucky North (ftUS) [2798] NAD83(HARN) / Kentucky Single Zone [2799] NAD83(HARN) / Kentucky Single Zone (ftUS) [2800] NAD83(HARN) / Kentucky South [2801] NAD83(HARN) / Kentucky South (ftUS) [2802] NAD83(HARN) / Louisiana North [2803] NAD83(HARN) / Louisiana North (ftUS) [2804] NAD83(HARN) / Louisiana South [2805] NAD83(HARN) / Louisiana South (ftUS) [2806] NAD83(HARN) / Maine CS2000 Central [2807] NAD83(HARN) / Maine CS2000 Central (deprecated) [2808] NAD83(HARN) / Maine CS2000 East [2809] NAD83(HARN) / Maine CS2000 West [2810] NAD83(HARN) / Maine East [2811] NAD83(HARN) / Maine East (ftUS) [2812] NAD83(HARN) / Maine East (ftUS) (deprecated) [2813] NAD83(HARN) / Maine West [2814] NAD83(HARN) / Maine West (ftUS) [2815] NAD83(HARN) / Maine West (ftUS) (deprecated) [2816] NAD83(HARN) / Maryland [2817] NAD83(HARN) / Maryland (ftUS) [2818] NAD83(HARN) / Massachusetts Island [2819] NAD83(HARN) / Massachusetts Island (ftUS) [2820] NAD83(HARN) / Massachusetts Mainland [2821] NAD83(HARN) / Massachusetts Mainland (ftUS) [2822] NAD83(HARN) / Michigan Central [2823] NAD83(HARN) / Michigan Central (ft) [2824] NAD83(HARN) / Michigan North [2825] NAD83(HARN) / Michigan North (ft) [2826] NAD83(HARN) / Michigan Oblique Mercator [2827] NAD83(HARN) / Michigan South [2828] NAD83(HARN) / Michigan South (ft) [2829] NAD83(HARN) / Minnesota Central [2830] NAD83(HARN) / Minnesota Central (ftUS) [2831] NAD83(HARN) / Minnesota Central (ftUS) (deprecated) [2832] NAD83(HARN) / Minnesota North [2833] NAD83(HARN) / Minnesota North (ftUS) [2834] NAD83(HARN) / Minnesota North (ftUS) (deprecated) [2835] NAD83(HARN) / Minnesota South [2836] NAD83(HARN) / Minnesota South (ftUS) [2837] NAD83(HARN) / Minnesota South (ftUS) (deprecated) [2838] NAD83(HARN) / Mississippi East [2839] NAD83(HARN) / Mississippi East (ftUS) [2840] NAD83(HARN) / Mississippi TM [2841] NAD83(HARN) / Mississippi West [2842] NAD83(HARN) / Mississippi West (ftUS) [2843] NAD83(HARN) / Missouri Central [2844] NAD83(HARN) / Missouri East [2845] NAD83(HARN) / Missouri West [2846] NAD83(HARN) / Montana [2847] NAD83(HARN) / Montana (ft) [2848] NAD83(HARN) / Nebraska [2849] NAD83(HARN) / Nebraska (ftUS) [2850] NAD83(HARN) / Nebraska (ftUS) (deprecated) [2851] NAD83(HARN) / Nevada Central [2852] NAD83(HARN) / Nevada Central (ftUS) [2853] NAD83(HARN) / Nevada East [2854] NAD83(HARN) / Nevada East (ftUS) [2855] NAD83(HARN) / Nevada West [2856] NAD83(HARN) / Nevada West (ftUS) [2857] NAD83(HARN) / New Hampshire [2858] NAD83(HARN) / New Hampshire (ftUS) [2859] NAD83(HARN) / New Jersey [2860] NAD83(HARN) / New Jersey (ftUS) [2861] NAD83(HARN) / New Mexico Central [2862] NAD83(HARN) / New Mexico Central (ftUS) [2863] NAD83(HARN) / New Mexico East [2864] NAD83(HARN) / New Mexico East (ftUS) [2865] NAD83(HARN) / New Mexico West [2866] NAD83(HARN) / New Mexico West (ftUS) [2867] NAD83(HARN) / New York Central [2868] NAD83(HARN) / New York Central (ftUS) [2869] NAD83(HARN) / New York East [2870] NAD83(HARN) / New York East (ftUS) [2871] NAD83(HARN) / New York Long Island [2872] NAD83(HARN) / New York Long Island (ftUS) [2873] NAD83(HARN) / New York West [2874] NAD83(HARN) / New York West (ftUS) [2875] NAD83(HARN) / North Carolina [2876] NAD83(HARN) / North Carolina (ftUS) [2877] NAD83(HARN) / North Carolina (ftUS) (deprecated) [2878] NAD83(HARN) / North Dakota North [2879] NAD83(HARN) / North Dakota North (ft) [2880] NAD83(HARN) / North Dakota South [2881] NAD83(HARN) / North Dakota South (ft) [2882] NAD83(HARN) / Ohio North [2883] NAD83(HARN) / Ohio North (ftUS) [2884] NAD83(HARN) / Ohio South [2885] NAD83(HARN) / Ohio South (ftUS) [2886] NAD83(HARN) / Oklahoma North [2887] NAD83(HARN) / Oklahoma North (ftUS) [2888] NAD83(HARN) / Oklahoma South [2889] NAD83(HARN) / Oklahoma South (ftUS) [2890] NAD83(HARN) / Oregon GIC Lambert (ft) [2891] NAD83(HARN) / Oregon LCC (m) [2892] NAD83(HARN) / Oregon North [2893] NAD83(HARN) / Oregon North (ft) [2894] NAD83(HARN) / Oregon South [2895] NAD83(HARN) / Oregon South (ft) [2896] NAD83(HARN) / Pennsylvania North [2897] NAD83(HARN) / Pennsylvania North (ftUS) [2898] NAD83(HARN) / Pennsylvania South [2899] NAD83(HARN) / Pennsylvania South (ftUS) [2900] NAD83(HARN) / Puerto Rico and Virgin Is. [2901] NAD83(HARN) / Rhode Island [2902] NAD83(HARN) / Rhode Island (ftUS) [2903] NAD83(HARN) / South Carolina [2904] NAD83(HARN) / South Carolina (ft) [2905] NAD83(HARN) / South Dakota North [2906] NAD83(HARN) / South Dakota North (ftUS) [2907] NAD83(HARN) / South Dakota South [2908] NAD83(HARN) / South Dakota South (ftUS) [2909] NAD83(HARN) / Tennessee [2910] NAD83(HARN) / Tennessee (ftUS) [2911] NAD83(HARN) / Texas Central [2912] NAD83(HARN) / Texas Central (ftUS) [2913] NAD83(HARN) / Texas Centric Albers Equal Area [2914] NAD83(HARN) / Texas Centric Lambert Conformal [2915] NAD83(HARN) / Texas North [2916] NAD83(HARN) / Texas North (ftUS) [2917] NAD83(HARN) / Texas North Central [2918] NAD83(HARN) / Texas North Central (ftUS) [2919] NAD83(HARN) / Texas South [2920] NAD83(HARN) / Texas South (ftUS) [2921] NAD83(HARN) / Texas South Central [2922] NAD83(HARN) / Texas South Central (ftUS) [2923] NAD83(HARN) / UTM zone 10N [2924] NAD83(HARN) / UTM zone 11N [2925] NAD83(HARN) / UTM zone 12N [2926] NAD83(HARN) / UTM zone 13N [2927] NAD83(HARN) / UTM zone 14N [2928] NAD83(HARN) / UTM zone 15N [2929] NAD83(HARN) / UTM zone 16N [2930] NAD83(HARN) / UTM zone 17N [2931] NAD83(HARN) / UTM zone 18N [2932] NAD83(HARN) / UTM zone 19N [2933] NAD83(HARN) / UTM zone 2S [2934] NAD83(HARN) / UTM zone 4N [2935] NAD83(HARN) / UTM zone 59S (deprecated) [2936] NAD83(HARN) / UTM zone 5N [2937] NAD83(HARN) / Utah Central [2938] NAD83(HARN) / Utah Central (ft) [2939] NAD83(HARN) / Utah Central (ftUS) [2940] NAD83(HARN) / Utah North [2941] NAD83(HARN) / Utah North (ft) [2942] NAD83(HARN) / Utah North (ftUS) [2943] NAD83(HARN) / Utah South [2944] NAD83(HARN) / Utah South (ft) [2945] NAD83(HARN) / Utah South (ftUS) [2946] NAD83(HARN) / Vermont [2947] NAD83(HARN) / Vermont (ftUS) [2948] NAD83(HARN) / Virginia Lambert [2949] NAD83(HARN) / Virginia North [2950] NAD83(HARN) / Virginia North (ftUS) [2951] NAD83(HARN) / Virginia South [2952] NAD83(HARN) / Virginia South (ftUS) [2953] NAD83(HARN) / WISCRS Adams and Juneau (ftUS) [2954] NAD83(HARN) / WISCRS Adams and Juneau (m) [2955] NAD83(HARN) / WISCRS Ashland (ftUS) [2956] NAD83(HARN) / WISCRS Ashland (m) [2957] NAD83(HARN) / WISCRS Barron (ftUS) [2958] NAD83(HARN) / WISCRS Barron (m) [2959] NAD83(HARN) / WISCRS Bayfield (ftUS) [2960] NAD83(HARN) / WISCRS Bayfield (m) [2961] NAD83(HARN) / WISCRS Brown (ftUS) [2962] NAD83(HARN) / WISCRS Brown (m) [2963] NAD83(HARN) / WISCRS Buffalo (ftUS) [2964] NAD83(HARN) / WISCRS Buffalo (m) [2965] NAD83(HARN) / WISCRS Burnett (ftUS) [2966] NAD83(HARN) / WISCRS Burnett (m) [2967] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2968] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2969] NAD83(HARN) / WISCRS Chippewa (ftUS) [2970] NAD83(HARN) / WISCRS Chippewa (m) [2971] NAD83(HARN) / WISCRS Clark (ftUS) [2972] NAD83(HARN) / WISCRS Clark (m) [2973] NAD83(HARN) / WISCRS Columbia (ftUS) [2974] NAD83(HARN) / WISCRS Columbia (m) [2975] NAD83(HARN) / WISCRS Crawford (ftUS) [2976] NAD83(HARN) / WISCRS Crawford (m) [2977] NAD83(HARN) / WISCRS Dane (ftUS) [2978] NAD83(HARN) / WISCRS Dane (m) [2979] NAD83(HARN) / WISCRS Dodge and Jefferson (ftUS) [2980] NAD83(HARN) / WISCRS Dodge and Jefferson (m) [2981] NAD83(HARN) / WISCRS Door (ftUS) [2982] NAD83(HARN) / WISCRS Door (m) [2983] NAD83(HARN) / WISCRS Douglas (ftUS) [2984] NAD83(HARN) / WISCRS Douglas (m) [2985] NAD83(HARN) / WISCRS Dunn (ftUS) [2986] NAD83(HARN) / WISCRS Dunn (m) [2987] NAD83(HARN) / WISCRS Eau Claire (ftUS) [2988] NAD83(HARN) / WISCRS Eau Claire (m) [2989] NAD83(HARN) / WISCRS Florence (ftUS) [2990] NAD83(HARN) / WISCRS Florence (m) [2991] NAD83(HARN) / WISCRS Forest (ftUS) [2992] NAD83(HARN) / WISCRS Forest (m) [2993] NAD83(HARN) / WISCRS Grant (ftUS) [2994] NAD83(HARN) / WISCRS Grant (m) [2995] NAD83(HARN) / WISCRS Green Lake and Marquette (ftUS) [2996] NAD83(HARN) / WISCRS Green Lake and Marquette (m) [2997] NAD83(HARN) / WISCRS Green and Lafayette (ftUS) [2998] NAD83(HARN) / WISCRS Green and Lafayette (m) [2999] NAD83(HARN) / WISCRS Iowa (ftUS) [3000] NAD83(HARN) / WISCRS Iowa (m) [3001] NAD83(HARN) / WISCRS Iron (ftUS) [3002] NAD83(HARN) / WISCRS Iron (m) [3003] NAD83(HARN) / WISCRS Jackson (ftUS) [3004] NAD83(HARN) / WISCRS Jackson (m) [3005] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [3006] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [3007] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [3008] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [3009] NAD83(HARN) / WISCRS La Crosse (ftUS) [3010] NAD83(HARN) / WISCRS La Crosse (m) [3011] NAD83(HARN) / WISCRS Langlade (ftUS) [3012] NAD83(HARN) / WISCRS Langlade (m) [3013] NAD83(HARN) / WISCRS Lincoln (ftUS) [3014] NAD83(HARN) / WISCRS Lincoln (m) [3015] NAD83(HARN) / WISCRS Marathon (ftUS) [3016] NAD83(HARN) / WISCRS Marathon (m) [3017] NAD83(HARN) / WISCRS Marinette (ftUS) [3018] NAD83(HARN) / WISCRS Marinette (m) [3019] NAD83(HARN) / WISCRS Menominee (ftUS) [3020] NAD83(HARN) / WISCRS Menominee (m) [3021] NAD83(HARN) / WISCRS Monroe (ftUS) [3022] NAD83(HARN) / WISCRS Monroe (m) [3023] NAD83(HARN) / WISCRS Oconto (ftUS) [3024] NAD83(HARN) / WISCRS Oconto (m) [3025] NAD83(HARN) / WISCRS Oneida (ftUS) [3026] NAD83(HARN) / WISCRS Oneida (m) [3027] NAD83(HARN) / WISCRS Pepin and Pierce (ftUS) [3028] NAD83(HARN) / WISCRS Pepin and Pierce (m) [3029] NAD83(HARN) / WISCRS Polk (ftUS) [3030] NAD83(HARN) / WISCRS Polk (m) [3031] NAD83(HARN) / WISCRS Portage (ftUS) [3032] NAD83(HARN) / WISCRS Portage (m) [3033] NAD83(HARN) / WISCRS Price (ftUS) [3034] NAD83(HARN) / WISCRS Price (m) [3035] NAD83(HARN) / WISCRS Richland (ftUS) [3036] NAD83(HARN) / WISCRS Richland (m) [3037] NAD83(HARN) / WISCRS Rock (ftUS) [3038] NAD83(HARN) / WISCRS Rock (m) [3039] NAD83(HARN) / WISCRS Rusk (ftUS) [3040] NAD83(HARN) / WISCRS Rusk (m) [3041] NAD83(HARN) / WISCRS Sauk (ftUS) [3042] NAD83(HARN) / WISCRS Sauk (m) [3043] NAD83(HARN) / WISCRS Sawyer (ftUS) [3044] NAD83(HARN) / WISCRS Sawyer (m) [3045] NAD83(HARN) / WISCRS Shawano (ftUS) [3046] NAD83(HARN) / WISCRS Shawano (m) [3047] NAD83(HARN) / WISCRS St. Croix (ftUS) [3048] NAD83(HARN) / WISCRS St. Croix (m) [3049] NAD83(HARN) / WISCRS Taylor (ftUS) [3050] NAD83(HARN) / WISCRS Taylor (m) [3051] NAD83(HARN) / WISCRS Trempealeau (ftUS) [3052] NAD83(HARN) / WISCRS Trempealeau (m) [3053] NAD83(HARN) / WISCRS Vernon (ftUS) [3054] NAD83(HARN) / WISCRS Vernon (m) [3055] NAD83(HARN) / WISCRS Vilas (ftUS) [3056] NAD83(HARN) / WISCRS Vilas (m) [3057] NAD83(HARN) / WISCRS Walworth (ftUS) [3058] NAD83(HARN) / WISCRS Walworth (m) [3059] NAD83(HARN) / WISCRS Washburn (ftUS) [3060] NAD83(HARN) / WISCRS Washburn (m) [3061] NAD83(HARN) / WISCRS Washington (ftUS) [3062] NAD83(HARN) / WISCRS Washington (m) [3063] NAD83(HARN) / WISCRS Waukesha (ftUS) [3064] NAD83(HARN) / WISCRS Waukesha (m) [3065] NAD83(HARN) / WISCRS Waupaca (ftUS) [3066] NAD83(HARN) / WISCRS Waupaca (m) [3067] NAD83(HARN) / WISCRS Waushara (ftUS) [3068] NAD83(HARN) / WISCRS Waushara (m) [3069] NAD83(HARN) / WISCRS Wood (ftUS) [3070] NAD83(HARN) / WISCRS Wood (m) [3071] NAD83(HARN) / Washington North [3072] NAD83(HARN) / Washington North (ftUS) [3073] NAD83(HARN) / Washington South [3074] NAD83(HARN) / Washington South (ftUS) [3075] NAD83(HARN) / West Virginia North [3076] NAD83(HARN) / West Virginia North (ftUS) [3077] NAD83(HARN) / West Virginia North (ftUS) (deprecated) [3078] NAD83(HARN) / West Virginia South [3079] NAD83(HARN) / West Virginia South (ftUS) [3080] NAD83(HARN) / West Virginia South (ftUS) (deprecated) [3081] NAD83(HARN) / Wisconsin Central [3082] NAD83(HARN) / Wisconsin Central (ftUS) [3083] NAD83(HARN) / Wisconsin North [3084] NAD83(HARN) / Wisconsin North (ftUS) [3085] NAD83(HARN) / Wisconsin South [3086] NAD83(HARN) / Wisconsin South (ftUS) [3087] NAD83(HARN) / Wisconsin Transverse Mercator [3088] NAD83(HARN) / Wyoming East [3089] NAD83(HARN) / Wyoming East (ftUS) [3090] NAD83(HARN) / Wyoming East Central [3091] NAD83(HARN) / Wyoming East Central (ftUS) [3092] NAD83(HARN) / Wyoming West [3093] NAD83(HARN) / Wyoming West (ftUS) [3094] NAD83(HARN) / Wyoming West Central [3095] NAD83(HARN) / Wyoming West Central (ftUS) [3096] NAD83(MA11) / Guam Map Grid [3097] NAD83(MA11) / UTM zone 54N [3098] NAD83(MA11) / UTM zone 55N [3099] NAD83(NSRS2007) / Alabama East [3100] NAD83(NSRS2007) / Alabama West [3101] NAD83(NSRS2007) / Alaska Albers [3102] NAD83(NSRS2007) / Alaska zone 1 [3103] NAD83(NSRS2007) / Alaska zone 10 [3104] NAD83(NSRS2007) / Alaska zone 2 [3105] NAD83(NSRS2007) / Alaska zone 3 [3106] NAD83(NSRS2007) / Alaska zone 4 [3107] NAD83(NSRS2007) / Alaska zone 5 [3108] NAD83(NSRS2007) / Alaska zone 6 [3109] NAD83(NSRS2007) / Alaska zone 7 [3110] NAD83(NSRS2007) / Alaska zone 8 [3111] NAD83(NSRS2007) / Alaska zone 9 [3112] NAD83(NSRS2007) / Arizona Central [3113] NAD83(NSRS2007) / Arizona Central (ft) [3114] NAD83(NSRS2007) / Arizona East [3115] NAD83(NSRS2007) / Arizona East (ft) [3116] NAD83(NSRS2007) / Arizona West [3117] NAD83(NSRS2007) / Arizona West (ft) [3118] NAD83(NSRS2007) / Arkansas North [3119] NAD83(NSRS2007) / Arkansas North (ftUS) [3120] NAD83(NSRS2007) / Arkansas South [3121] NAD83(NSRS2007) / Arkansas South (ftUS) [3122] NAD83(NSRS2007) / California Albers [3123] NAD83(NSRS2007) / California zone 1 [3124] NAD83(NSRS2007) / California zone 1 (ftUS) [3125] NAD83(NSRS2007) / California zone 2 [3126] NAD83(NSRS2007) / California zone 2 (ftUS) [3127] NAD83(NSRS2007) / California zone 3 [3128] NAD83(NSRS2007) / California zone 3 (ftUS) [3129] NAD83(NSRS2007) / California zone 4 [3130] NAD83(NSRS2007) / California zone 4 (ftUS) [3131] NAD83(NSRS2007) / California zone 5 [3132] NAD83(NSRS2007) / California zone 5 (ftUS) [3133] NAD83(NSRS2007) / California zone 6 [3134] NAD83(NSRS2007) / California zone 6 (ftUS) [3135] NAD83(NSRS2007) / Colorado Central [3136] NAD83(NSRS2007) / Colorado Central (ftUS) [3137] NAD83(NSRS2007) / Colorado North [3138] NAD83(NSRS2007) / Colorado North (ftUS) [3139] NAD83(NSRS2007) / Colorado South [3140] NAD83(NSRS2007) / Colorado South (ftUS) [3141] NAD83(NSRS2007) / Connecticut [3142] NAD83(NSRS2007) / Connecticut (ftUS) [3143] NAD83(NSRS2007) / Conus Albers [3144] NAD83(NSRS2007) / Delaware [3145] NAD83(NSRS2007) / Delaware (ftUS) [3146] NAD83(NSRS2007) / EPSG Arctic zone 5-29 [3147] NAD83(NSRS2007) / EPSG Arctic zone 5-31 [3148] NAD83(NSRS2007) / EPSG Arctic zone 6-14 [3149] NAD83(NSRS2007) / EPSG Arctic zone 6-16 [3150] NAD83(NSRS2007) / Florida East [3151] NAD83(NSRS2007) / Florida East (ftUS) [3152] NAD83(NSRS2007) / Florida GDL Albers [3153] NAD83(NSRS2007) / Florida North [3154] NAD83(NSRS2007) / Florida North (ftUS) [3155] NAD83(NSRS2007) / Florida West [3156] NAD83(NSRS2007) / Florida West (ftUS) [3157] NAD83(NSRS2007) / Georgia East [3158] NAD83(NSRS2007) / Georgia East (ftUS) [3159] NAD83(NSRS2007) / Georgia West [3160] NAD83(NSRS2007) / Georgia West (ftUS) [3161] NAD83(NSRS2007) / Idaho Central [3162] NAD83(NSRS2007) / Idaho Central (ftUS) [3163] NAD83(NSRS2007) / Idaho East [3164] NAD83(NSRS2007) / Idaho East (ftUS) [3165] NAD83(NSRS2007) / Idaho West [3166] NAD83(NSRS2007) / Idaho West (ftUS) [3167] NAD83(NSRS2007) / Illinois East [3168] NAD83(NSRS2007) / Illinois East (ftUS) [3169] NAD83(NSRS2007) / Illinois West [3170] NAD83(NSRS2007) / Illinois West (ftUS) [3171] NAD83(NSRS2007) / Indiana East [3172] NAD83(NSRS2007) / Indiana East (ftUS) [3173] NAD83(NSRS2007) / Indiana West [3174] NAD83(NSRS2007) / Indiana West (ftUS) [3175] NAD83(NSRS2007) / Iowa North [3176] NAD83(NSRS2007) / Iowa North (ftUS) [3177] NAD83(NSRS2007) / Iowa South [3178] NAD83(NSRS2007) / Iowa South (ftUS) [3179] NAD83(NSRS2007) / Kansas North [3180] NAD83(NSRS2007) / Kansas North (ftUS) [3181] NAD83(NSRS2007) / Kansas South [3182] NAD83(NSRS2007) / Kansas South (ftUS) [3183] NAD83(NSRS2007) / Kentucky North [3184] NAD83(NSRS2007) / Kentucky North (ftUS) [3185] NAD83(NSRS2007) / Kentucky Single Zone [3186] NAD83(NSRS2007) / Kentucky Single Zone (ftUS) [3187] NAD83(NSRS2007) / Kentucky South [3188] NAD83(NSRS2007) / Kentucky South (ftUS) [3189] NAD83(NSRS2007) / Louisiana North [3190] NAD83(NSRS2007) / Louisiana North (ftUS) [3191] NAD83(NSRS2007) / Louisiana South [3192] NAD83(NSRS2007) / Louisiana South (ftUS) [3193] NAD83(NSRS2007) / Maine CS2000 Central [3194] NAD83(NSRS2007) / Maine CS2000 East [3195] NAD83(NSRS2007) / Maine CS2000 West [3196] NAD83(NSRS2007) / Maine East [3197] NAD83(NSRS2007) / Maine East (ftUS) [3198] NAD83(NSRS2007) / Maine East (ftUS) (deprecated) [3199] NAD83(NSRS2007) / Maine West [3200] NAD83(NSRS2007) / Maine West (ftUS) [3201] NAD83(NSRS2007) / Maine West (ftUS) (deprecated) [3202] NAD83(NSRS2007) / Maryland [3203] NAD83(NSRS2007) / Maryland (ftUS) [3204] NAD83(NSRS2007) / Massachusetts Island [3205] NAD83(NSRS2007) / Massachusetts Island (ftUS) [3206] NAD83(NSRS2007) / Massachusetts Mainland [3207] NAD83(NSRS2007) / Massachusetts Mainland (ftUS) [3208] NAD83(NSRS2007) / Michigan Central [3209] NAD83(NSRS2007) / Michigan Central (ft) [3210] NAD83(NSRS2007) / Michigan North [3211] NAD83(NSRS2007) / Michigan North (ft) [3212] NAD83(NSRS2007) / Michigan Oblique Mercator [3213] NAD83(NSRS2007) / Michigan South [3214] NAD83(NSRS2007) / Michigan South (ft) [3215] NAD83(NSRS2007) / Minnesota Central [3216] NAD83(NSRS2007) / Minnesota Central (ftUS) [3217] NAD83(NSRS2007) / Minnesota Central (ftUS) (deprecated) [3218] NAD83(NSRS2007) / Minnesota North [3219] NAD83(NSRS2007) / Minnesota North (ftUS) [3220] NAD83(NSRS2007) / Minnesota North (ftUS) (deprecated) [3221] NAD83(NSRS2007) / Minnesota South [3222] NAD83(NSRS2007) / Minnesota South (ftUS) [3223] NAD83(NSRS2007) / Minnesota South (ftUS) (deprecated) [3224] NAD83(NSRS2007) / Mississippi East [3225] NAD83(NSRS2007) / Mississippi East (ftUS) [3226] NAD83(NSRS2007) / Mississippi TM [3227] NAD83(NSRS2007) / Mississippi West [3228] NAD83(NSRS2007) / Mississippi West (ftUS) [3229] NAD83(NSRS2007) / Missouri Central [3230] NAD83(NSRS2007) / Missouri East [3231] NAD83(NSRS2007) / Missouri West [3232] NAD83(NSRS2007) / Montana [3233] NAD83(NSRS2007) / Montana (ft) [3234] NAD83(NSRS2007) / Nebraska [3235] NAD83(NSRS2007) / Nebraska (ftUS) [3236] NAD83(NSRS2007) / Nebraska (ftUS) (deprecated) [3237] NAD83(NSRS2007) / Nevada Central [3238] NAD83(NSRS2007) / Nevada Central (ftUS) [3239] NAD83(NSRS2007) / Nevada East [3240] NAD83(NSRS2007) / Nevada East (ftUS) [3241] NAD83(NSRS2007) / Nevada West [3242] NAD83(NSRS2007) / Nevada West (ftUS) [3243] NAD83(NSRS2007) / New Hampshire [3244] NAD83(NSRS2007) / New Hampshire (ftUS) [3245] NAD83(NSRS2007) / New Jersey [3246] NAD83(NSRS2007) / New Jersey (ftUS) [3247] NAD83(NSRS2007) / New Mexico Central [3248] NAD83(NSRS2007) / New Mexico Central (ftUS) [3249] NAD83(NSRS2007) / New Mexico East [3250] NAD83(NSRS2007) / New Mexico East (ftUS) [3251] NAD83(NSRS2007) / New Mexico West [3252] NAD83(NSRS2007) / New Mexico West (ftUS) [3253] NAD83(NSRS2007) / New York Central [3254] NAD83(NSRS2007) / New York Central (ftUS) [3255] NAD83(NSRS2007) / New York East [3256] NAD83(NSRS2007) / New York East (ftUS) [3257] NAD83(NSRS2007) / New York Long Island [3258] NAD83(NSRS2007) / New York Long Island (ftUS) [3259] NAD83(NSRS2007) / New York West [3260] NAD83(NSRS2007) / New York West (ftUS) [3261] NAD83(NSRS2007) / North Carolina [3262] NAD83(NSRS2007) / North Carolina (ftUS) [3263] NAD83(NSRS2007) / North Dakota North [3264] NAD83(NSRS2007) / North Dakota North (ft) [3265] NAD83(NSRS2007) / North Dakota South [3266] NAD83(NSRS2007) / North Dakota South (ft) [3267] NAD83(NSRS2007) / Ohio North [3268] NAD83(NSRS2007) / Ohio North (ftUS) [3269] NAD83(NSRS2007) / Ohio South [3270] NAD83(NSRS2007) / Ohio South (ftUS) [3271] NAD83(NSRS2007) / Oklahoma North [3272] NAD83(NSRS2007) / Oklahoma North (ftUS) [3273] NAD83(NSRS2007) / Oklahoma South [3274] NAD83(NSRS2007) / Oklahoma South (ftUS) [3275] NAD83(NSRS2007) / Oregon GIC Lambert (ft) [3276] NAD83(NSRS2007) / Oregon LCC (m) [3277] NAD83(NSRS2007) / Oregon North [3278] NAD83(NSRS2007) / Oregon North (ft) [3279] NAD83(NSRS2007) / Oregon South [3280] NAD83(NSRS2007) / Oregon South (ft) [3281] NAD83(NSRS2007) / Pennsylvania North [3282] NAD83(NSRS2007) / Pennsylvania North (ftUS) [3283] NAD83(NSRS2007) / Pennsylvania South [3284] NAD83(NSRS2007) / Pennsylvania South (ftUS) [3285] NAD83(NSRS2007) / Puerto Rico and Virgin Is. [3286] NAD83(NSRS2007) / Rhode Island [3287] NAD83(NSRS2007) / Rhode Island (ftUS) [3288] NAD83(NSRS2007) / South Carolina [3289] NAD83(NSRS2007) / South Carolina (ft) [3290] NAD83(NSRS2007) / South Dakota North [3291] NAD83(NSRS2007) / South Dakota North (ftUS) [3292] NAD83(NSRS2007) / South Dakota South [3293] NAD83(NSRS2007) / South Dakota South (ftUS) [3294] NAD83(NSRS2007) / Tennessee [3295] NAD83(NSRS2007) / Tennessee (ftUS) [3296] NAD83(NSRS2007) / Texas Central [3297] NAD83(NSRS2007) / Texas Central (ftUS) [3298] NAD83(NSRS2007) / Texas Centric Albers Equal Area [3299] NAD83(NSRS2007) / Texas Centric Lambert Conformal [3300] NAD83(NSRS2007) / Texas North [3301] NAD83(NSRS2007) / Texas North (ftUS) [3302] NAD83(NSRS2007) / Texas North Central [3303] NAD83(NSRS2007) / Texas North Central (ftUS) [3304] NAD83(NSRS2007) / Texas South [3305] NAD83(NSRS2007) / Texas South (ftUS) [3306] NAD83(NSRS2007) / Texas South Central [3307] NAD83(NSRS2007) / Texas South Central (ftUS) [3308] NAD83(NSRS2007) / UTM zone 10N [3309] NAD83(NSRS2007) / UTM zone 11N [3310] NAD83(NSRS2007) / UTM zone 12N [3311] NAD83(NSRS2007) / UTM zone 13N [3312] NAD83(NSRS2007) / UTM zone 14N [3313] NAD83(NSRS2007) / UTM zone 15N [3314] NAD83(NSRS2007) / UTM zone 16N [3315] NAD83(NSRS2007) / UTM zone 17N [3316] NAD83(NSRS2007) / UTM zone 18N [3317] NAD83(NSRS2007) / UTM zone 19N [3318] NAD83(NSRS2007) / UTM zone 1N [3319] NAD83(NSRS2007) / UTM zone 2N [3320] NAD83(NSRS2007) / UTM zone 3N [3321] NAD83(NSRS2007) / UTM zone 4N [3322] NAD83(NSRS2007) / UTM zone 59N [3323] NAD83(NSRS2007) / UTM zone 5N [3324] NAD83(NSRS2007) / UTM zone 60N [3325] NAD83(NSRS2007) / UTM zone 6N [3326] NAD83(NSRS2007) / UTM zone 7N [3327] NAD83(NSRS2007) / UTM zone 8N [3328] NAD83(NSRS2007) / UTM zone 9N [3329] NAD83(NSRS2007) / Utah Central [3330] NAD83(NSRS2007) / Utah Central (ft) [3331] NAD83(NSRS2007) / Utah Central (ftUS) [3332] NAD83(NSRS2007) / Utah North [3333] NAD83(NSRS2007) / Utah North (ft) [3334] NAD83(NSRS2007) / Utah North (ftUS) [3335] NAD83(NSRS2007) / Utah South [3336] NAD83(NSRS2007) / Utah South (ft) [3337] NAD83(NSRS2007) / Utah South (ftUS) [3338] NAD83(NSRS2007) / Vermont [3339] NAD83(NSRS2007) / Vermont (ftUS) [3340] NAD83(NSRS2007) / Virginia Lambert [3341] NAD83(NSRS2007) / Virginia North [3342] NAD83(NSRS2007) / Virginia North (ftUS) [3343] NAD83(NSRS2007) / Virginia South [3344] NAD83(NSRS2007) / Virginia South (ftUS) [3345] NAD83(NSRS2007) / Washington North [3346] NAD83(NSRS2007) / Washington North (ftUS) [3347] NAD83(NSRS2007) / Washington South [3348] NAD83(NSRS2007) / Washington South (ftUS) [3349] NAD83(NSRS2007) / West Virginia North [3350] NAD83(NSRS2007) / West Virginia North (ftUS) [3351] NAD83(NSRS2007) / West Virginia North (ftUS) (deprecated) [3352] NAD83(NSRS2007) / West Virginia South [3353] NAD83(NSRS2007) / West Virginia South (ftUS) [3354] NAD83(NSRS2007) / West Virginia South (ftUS) (deprecated) [3355] NAD83(NSRS2007) / Wisconsin Central [3356] NAD83(NSRS2007) / Wisconsin Central (ftUS) [3357] NAD83(NSRS2007) / Wisconsin North [3358] NAD83(NSRS2007) / Wisconsin North (ftUS) [3359] NAD83(NSRS2007) / Wisconsin South [3360] NAD83(NSRS2007) / Wisconsin South (ftUS) [3361] NAD83(NSRS2007) / Wisconsin Transverse Mercator [3362] NAD83(NSRS2007) / Wyoming East [3363] NAD83(NSRS2007) / Wyoming East (ftUS) [3364] NAD83(NSRS2007) / Wyoming East Central [3365] NAD83(NSRS2007) / Wyoming East Central (ftUS) [3366] NAD83(NSRS2007) / Wyoming West [3367] NAD83(NSRS2007) / Wyoming West (ftUS) [3368] NAD83(NSRS2007) / Wyoming West Central [3369] NAD83(NSRS2007) / Wyoming West Central (ftUS) [3370] NAD83(PA11) / Hawaii zone 1 [3371] NAD83(PA11) / Hawaii zone 2 [3372] NAD83(PA11) / Hawaii zone 3 [3373] NAD83(PA11) / Hawaii zone 3 (ftUS) [3374] NAD83(PA11) / Hawaii zone 4 [3375] NAD83(PA11) / Hawaii zone 5 [3376] NAD83(PA11) / UTM zone 2S [3377] NAD83(PA11) / UTM zone 4N [3378] NAD83(PA11) / UTM zone 5N [3379] NAD_1927_10TM_AEP_Forest [3380] NAD_1927_10TM_AEP_Resource [3381] NAD_1927_3TM_111 (deprecated) [3382] NAD_1927_3TM_114 (deprecated) [3383] NAD_1927_3TM_117 (deprecated) [3384] NAD_1927_3TM_120 (deprecated) [3385] NAD_1927_Alaska_Albers_Meters [3386] NAD_1927_Georgia_Statewide_Albers [3387] NAD_1927_Michigan_GeoRef_Feet_US [3388] NAD_1927_Michigan_GeoRef_Meters [3389] NAD_1927_StatePlane_California_V_Ventura [3390] NAD_1927_StatePlane_Guam_FIPS_5400 [3391] NAD_1927_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [3392] NAD_1927_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [3393] NAD_1927_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [3394] NAD_1927_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [3395] NAD_1927_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [3396] NAD_1927_StatePlane_Michigan_Central_FIPS_2112 [3397] NAD_1927_StatePlane_Michigan_North_FIPS_2111 [3398] NAD_1927_StatePlane_Michigan_South_FIPS_2113 [3399] NAD_1927_StatePlane_Puerto_Rico_FIPS_5201 [3400] NAD_1927_StatePlane_Virgin_Islands_St_Croix_FIPS_5202 [3401] NAD_1927_Texas_Statewide_Mapping_System (deprecated) [3402] NAD_1927_UTM_Zone_1N (deprecated) [3403] NAD_1927_UTM_Zone_2N (deprecated) [3404] NAD_1927_UTM_Zone_59N (deprecated) [3405] NAD_1927_UTM_Zone_60N (deprecated) [3406] NAD_1983_10TM_AEP_Forest (deprecated) [3407] NAD_1983_10TM_AEP_Resource (deprecated) [3408] NAD_1983_2011_Alaska_Albers (deprecated) [3409] NAD_1983_2011_California_Teale_Albers (deprecated) [3410] NAD_1983_2011_Contiguous_USA_Albers (deprecated) [3411] NAD_1983_2011_Fargo_Ground_Coordinate_System [3412] NAD_1983_2011_Florida_GDL_Albers (deprecated) [3413] NAD_1983_2011_Maine_2000_Central_Zone (deprecated) [3414] NAD_1983_2011_Maine_2000_East_Zone (deprecated) [3415] NAD_1983_2011_Maine_2000_West_Zone (deprecated) [3416] NAD_1983_2011_Michigan_GeoRef_Meters (deprecated) [3417] NAD_1983_2011_Mississippi_TM (deprecated) [3418] NAD_1983_2011_Oregon_Statewide_Lambert (deprecated) [3419] NAD_1983_2011_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3420] NAD_1983_2011_StatePlane_Alabama_East_FIPS_0101 (deprecated) [3421] NAD_1983_2011_StatePlane_Alabama_West_FIPS_0102 (deprecated) [3422] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010 (deprecated) [3423] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010_Feet [3424] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001 (deprecated) [3425] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001_Feet [3426] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002 (deprecated) [3427] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002_Feet [3428] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003 (deprecated) [3429] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003_Feet [3430] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004 (deprecated) [3431] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004_Feet [3432] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005 (deprecated) [3433] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005_Feet [3434] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006 (deprecated) [3435] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006_Feet [3436] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007 (deprecated) [3437] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007_Feet [3438] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008 (deprecated) [3439] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008_Feet [3440] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009 (deprecated) [3441] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009_Feet [3442] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [3443] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl (deprecated) [3444] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201 (deprecated) [3445] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201_Ft_Intl (deprecated) [3446] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203 (deprecated) [3447] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203_Ft_Intl (deprecated) [3448] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [3449] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301_Ft_US (deprecated) [3450] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [3451] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302_Ft_US (deprecated) [3452] NAD_1983_2011_StatePlane_California_III_FIPS_0403 (deprecated) [3453] NAD_1983_2011_StatePlane_California_III_FIPS_0403_Ft_US (deprecated) [3454] NAD_1983_2011_StatePlane_California_II_FIPS_0402 (deprecated) [3455] NAD_1983_2011_StatePlane_California_II_FIPS_0402_Ft_US (deprecated) [3456] NAD_1983_2011_StatePlane_California_IV_FIPS_0404 (deprecated) [3457] NAD_1983_2011_StatePlane_California_IV_FIPS_0404_Ft_US (deprecated) [3458] NAD_1983_2011_StatePlane_California_I_FIPS_0401 (deprecated) [3459] NAD_1983_2011_StatePlane_California_I_FIPS_0401_Ft_US (deprecated) [3460] NAD_1983_2011_StatePlane_California_VI_FIPS_0406 (deprecated) [3461] NAD_1983_2011_StatePlane_California_VI_FIPS_0406_Ft_US (deprecated) [3462] NAD_1983_2011_StatePlane_California_V_FIPS_0405 (deprecated) [3463] NAD_1983_2011_StatePlane_California_V_FIPS_0405_Ft_US (deprecated) [3464] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [3465] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502_Ft_US (deprecated) [3466] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501 (deprecated) [3467] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501_Ft_US (deprecated) [3468] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503 (deprecated) [3469] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503_Ft_US (deprecated) [3470] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600 (deprecated) [3471] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600_Ft_US (deprecated) [3472] NAD_1983_2011_StatePlane_Delaware_FIPS_0700 (deprecated) [3473] NAD_1983_2011_StatePlane_Delaware_FIPS_0700_Ft_US (deprecated) [3474] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901 (deprecated) [3475] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901_Ft_US (deprecated) [3476] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903 (deprecated) [3477] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903_Ft_US (deprecated) [3478] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902 (deprecated) [3479] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902_Ft_US (deprecated) [3480] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001 (deprecated) [3481] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001_Ft_US (deprecated) [3482] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002 (deprecated) [3483] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002_Ft_US (deprecated) [3484] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [3485] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102_Ft_US (deprecated) [3486] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101 (deprecated) [3487] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101_Ft_US (deprecated) [3488] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103 (deprecated) [3489] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103_Ft_US (deprecated) [3490] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201 (deprecated) [3491] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201_Ft_US (deprecated) [3492] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202 (deprecated) [3493] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202_Ft_US (deprecated) [3494] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301 (deprecated) [3495] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301_Ft_US (deprecated) [3496] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302 (deprecated) [3497] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302_Ft_US (deprecated) [3498] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401 (deprecated) [3499] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401_Ft_US (deprecated) [3500] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402 (deprecated) [3501] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402_Ft_US (deprecated) [3502] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501 (deprecated) [3503] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501_Ft_US (deprecated) [3504] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502 (deprecated) [3505] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502_Ft_US (deprecated) [3506] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600 (deprecated) [3507] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600_Ft_US (deprecated) [3508] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [3509] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601_Ft_US (deprecated) [3510] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [3511] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602_Ft_US (deprecated) [3512] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [3513] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701_Ft_US (deprecated) [3514] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [3515] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702_Ft_US (deprecated) [3516] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801 (deprecated) [3517] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801_Ft_US (deprecated) [3518] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802 (deprecated) [3519] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802_Ft_US (deprecated) [3520] NAD_1983_2011_StatePlane_Maryland_FIPS_1900 (deprecated) [3521] NAD_1983_2011_StatePlane_Maryland_FIPS_1900_Ft_US (deprecated) [3522] NAD_1983_2011_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS (deprecated) [3523] NAD_1983_2011_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [3524] NAD_1983_2011_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [3525] NAD_1983_2011_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS (deprecated) [3526] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [3527] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl (deprecated) [3528] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111 (deprecated) [3529] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111_Ft_Intl (deprecated) [3530] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113 (deprecated) [3531] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113_Ft_Intl (deprecated) [3532] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [3533] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202_Ft_US (deprecated) [3534] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [3535] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201_Ft_US (deprecated) [3536] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [3537] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203_Ft_US (deprecated) [3538] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [3539] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301_Ft_US (deprecated) [3540] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [3541] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302_Ft_US (deprecated) [3542] NAD_1983_2011_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [3543] NAD_1983_2011_StatePlane_Missouri_East_FIPS_2401 (deprecated) [3544] NAD_1983_2011_StatePlane_Missouri_West_FIPS_2403 (deprecated) [3545] NAD_1983_2011_StatePlane_Montana_FIPS_2500 (deprecated) [3546] NAD_1983_2011_StatePlane_Montana_FIPS_2500_Ft_Intl (deprecated) [3547] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600 (deprecated) [3548] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600_Ft_US (deprecated) [3549] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [3550] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702_Ft_US (deprecated) [3551] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701 (deprecated) [3552] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701_Ft_US (deprecated) [3553] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703 (deprecated) [3554] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703_Ft_US (deprecated) [3555] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [3556] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800_Ft_US (deprecated) [3557] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900 (deprecated) [3558] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900_Ft_US (deprecated) [3559] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [3560] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US (deprecated) [3561] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [3562] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001_Ft_US (deprecated) [3563] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [3564] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003_Ft_US (deprecated) [3565] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102 (deprecated) [3566] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102_Ft_US (deprecated) [3567] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101 (deprecated) [3568] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101_Ft_US (deprecated) [3569] NAD_1983_2011_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US (deprecated) [3570] NAD_1983_2011_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [3571] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103 (deprecated) [3572] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103_Ft_US (deprecated) [3573] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200 (deprecated) [3574] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200_Ft_US (deprecated) [3575] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [3576] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301_FtI (deprecated) [3577] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [3578] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302_FtI (deprecated) [3579] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401 (deprecated) [3580] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401_Ft_US (deprecated) [3581] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402 (deprecated) [3582] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402_Ft_US (deprecated) [3583] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [3584] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501_Ft_US (deprecated) [3585] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [3586] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502_Ft_US (deprecated) [3587] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3588] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3589] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3590] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3591] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701 (deprecated) [3592] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US (deprecated) [3593] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702 (deprecated) [3594] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US (deprecated) [3595] NAD_1983_2011_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [3596] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [3597] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800_Ft_US (deprecated) [3598] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900 (deprecated) [3599] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900_Ft_Intl (deprecated) [3600] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [3601] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001_Ft_US (deprecated) [3602] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [3603] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002_Ft_US (deprecated) [3604] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100 (deprecated) [3605] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100_Ft_US (deprecated) [3606] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203 (deprecated) [3607] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203_Ft_US (deprecated) [3608] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [3609] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202_FtUS (deprecated) [3610] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201 (deprecated) [3611] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201_Ft_US (deprecated) [3612] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [3613] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204_FtUS (deprecated) [3614] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205 (deprecated) [3615] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205_Ft_US (deprecated) [3616] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302 (deprecated) [3617] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3618] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_US (deprecated) [3619] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301 (deprecated) [3620] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3621] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_US (deprecated) [3622] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303 (deprecated) [3623] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3624] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_US (deprecated) [3625] NAD_1983_2011_StatePlane_Vermont_FIPS_4400 (deprecated) [3626] NAD_1983_2011_StatePlane_Vermont_FIPS_4400_Ft_US (deprecated) [3627] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501 (deprecated) [3628] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501_Ft_US (deprecated) [3629] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502 (deprecated) [3630] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502_Ft_US (deprecated) [3631] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601 (deprecated) [3632] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601_Ft_US (deprecated) [3633] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602 (deprecated) [3634] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602_Ft_US (deprecated) [3635] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [3636] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701_FtUS (deprecated) [3637] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [3638] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702_FtUS (deprecated) [3639] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [3640] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US (deprecated) [3641] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [3642] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801_Ft_US (deprecated) [3643] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [3644] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803_Ft_US (deprecated) [3645] NAD_1983_2011_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US (deprecated) [3646] NAD_1983_2011_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [3647] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [3648] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901_Ft_US (deprecated) [3649] NAD_1983_2011_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US (deprecated) [3650] NAD_1983_2011_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [3651] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [3652] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904_Ft_US (deprecated) [3653] NAD_1983_2011_Texas_Centric_Mapping_System_Albers (deprecated) [3654] NAD_1983_2011_Texas_Centric_Mapping_System_Lambert (deprecated) [3655] NAD_1983_2011_UTM_Zone_10N (deprecated) [3656] NAD_1983_2011_UTM_Zone_11N (deprecated) [3657] NAD_1983_2011_UTM_Zone_12N (deprecated) [3658] NAD_1983_2011_UTM_Zone_13N (deprecated) [3659] NAD_1983_2011_UTM_Zone_14N (deprecated) [3660] NAD_1983_2011_UTM_Zone_15N (deprecated) [3661] NAD_1983_2011_UTM_Zone_16N (deprecated) [3662] NAD_1983_2011_UTM_Zone_17N (deprecated) [3663] NAD_1983_2011_UTM_Zone_18N (deprecated) [3664] NAD_1983_2011_UTM_Zone_19N (deprecated) [3665] NAD_1983_2011_UTM_Zone_1N (deprecated) [3666] NAD_1983_2011_UTM_Zone_20N [3667] NAD_1983_2011_UTM_Zone_2N (deprecated) [3668] NAD_1983_2011_UTM_Zone_3N (deprecated) [3669] NAD_1983_2011_UTM_Zone_4N (deprecated) [3670] NAD_1983_2011_UTM_Zone_59N (deprecated) [3671] NAD_1983_2011_UTM_Zone_5N (deprecated) [3672] NAD_1983_2011_UTM_Zone_60N (deprecated) [3673] NAD_1983_2011_UTM_Zone_6N (deprecated) [3674] NAD_1983_2011_UTM_Zone_7N (deprecated) [3675] NAD_1983_2011_UTM_Zone_8N (deprecated) [3676] NAD_1983_2011_UTM_Zone_9N (deprecated) [3677] NAD_1983_2011_Wisconsin_TM (deprecated) [3678] NAD_1983_2011_Wisconsin_TM_US_Ft [3679] NAD_1983_3TM_111 (deprecated) [3680] NAD_1983_3TM_114 (deprecated) [3681] NAD_1983_3TM_117 (deprecated) [3682] NAD_1983_3TM_120 (deprecated) [3683] NAD_1983_Alaska_Albers (deprecated) [3684] NAD_1983_Albers_BLM_MT_ND_SD [3685] NAD_1983_BC_Environment_Albers (deprecated) [3686] NAD_1983_CORS96_Alaska_Albers [3687] NAD_1983_CORS96_Maine_2000_Central_Zone [3688] NAD_1983_CORS96_Maine_2000_East_Zone [3689] NAD_1983_CORS96_Maine_2000_West_Zone [3690] NAD_1983_CORS96_Oregon_Statewide_Lambert (deprecated) [3691] NAD_1983_CORS96_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3692] NAD_1983_CORS96_StatePlane_Alabama_East_FIPS_0101 [3693] NAD_1983_CORS96_StatePlane_Alabama_West_FIPS_0102 [3694] NAD_1983_CORS96_StatePlane_Alaska_10_FIPS_5010 [3695] NAD_1983_CORS96_StatePlane_Alaska_1_FIPS_5001 [3696] NAD_1983_CORS96_StatePlane_Alaska_2_FIPS_5002 [3697] NAD_1983_CORS96_StatePlane_Alaska_3_FIPS_5003 [3698] NAD_1983_CORS96_StatePlane_Alaska_4_FIPS_5004 [3699] NAD_1983_CORS96_StatePlane_Alaska_5_FIPS_5005 [3700] NAD_1983_CORS96_StatePlane_Alaska_6_FIPS_5006 [3701] NAD_1983_CORS96_StatePlane_Alaska_7_FIPS_5007 [3702] NAD_1983_CORS96_StatePlane_Alaska_8_FIPS_5008 [3703] NAD_1983_CORS96_StatePlane_Alaska_9_FIPS_5009 [3704] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202 [3705] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl [3706] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201 [3707] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201_Ft_Intl [3708] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203 [3709] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203_Ft_Intl [3710] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301 [3711] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301_Ft_US [3712] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302 [3713] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302_Ft_US [3714] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403 [3715] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403_Ft_US [3716] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402 [3717] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402_Ft_US [3718] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404 [3719] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404_Ft_US [3720] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401 [3721] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401_Ft_US [3722] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406 [3723] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406_Ft_US [3724] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405 [3725] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405_Ft_US [3726] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502 [3727] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502_Ft_US [3728] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501 [3729] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501_Ft_US [3730] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503 [3731] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503_Ft_US [3732] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600 [3733] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600_Ft_US [3734] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700 [3735] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700_Ft_US [3736] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901 [3737] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901_Ft_US [3738] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903 [3739] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903_Ft_US [3740] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902 [3741] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902_Ft_US [3742] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001 [3743] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001_Ft_US [3744] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002 [3745] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002_Ft_US [3746] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102 [3747] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102_Ft_US [3748] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101 [3749] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101_Ft_US [3750] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103 [3751] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103_Ft_US [3752] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201 [3753] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201_Ft_US [3754] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202 [3755] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202_Ft_US [3756] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301 [3757] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301_Ft_US [3758] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302 [3759] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302_Ft_US [3760] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401 [3761] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401_Ft_US [3762] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402 [3763] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402_Ft_US [3764] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501 [3765] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501_Ft_US [3766] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502 [3767] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502_Ft_US [3768] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600 [3769] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600_Ft_US [3770] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601 [3771] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601_Ft_US [3772] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602 [3773] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602_Ft_US [3774] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701 [3775] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701_Ft_US [3776] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702 [3777] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702_Ft_US [3778] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801 [3779] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801_Ft_US [3780] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802 [3781] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802_Ft_US [3782] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900 [3783] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900_Ft_US [3784] NAD_1983_CORS96_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS [3785] NAD_1983_CORS96_StatePlane_Massachusetts_Island_FIPS_2002 [3786] NAD_1983_CORS96_StatePlane_Massachusetts_Mainland_FIPS_2001 [3787] NAD_1983_CORS96_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS [3788] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112 [3789] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl [3790] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111 [3791] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111_Ft_Intl [3792] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113 [3793] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113_Ft_Intl [3794] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202 [3795] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202_Ft_US [3796] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201 [3797] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201_Ft_US [3798] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203 [3799] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203_Ft_US [3800] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301 [3801] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301_Ft_US [3802] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302 [3803] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302_Ft_US [3804] NAD_1983_CORS96_StatePlane_Missouri_Central_FIPS_2402 [3805] NAD_1983_CORS96_StatePlane_Missouri_East_FIPS_2401 [3806] NAD_1983_CORS96_StatePlane_Missouri_West_FIPS_2403 [3807] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500 [3808] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500_Ft_Intl [3809] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600 [3810] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600_Ft_US [3811] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702 [3812] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702_Ft_US [3813] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701 [3814] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701_Ft_US [3815] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703 [3816] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703_Ft_US [3817] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800 [3818] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800_Ft_US [3819] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900 [3820] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900_Ft_US [3821] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002 [3822] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US [3823] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001 [3824] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001_Ft_US [3825] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003 [3826] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003_Ft_US [3827] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102 [3828] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102_Ft_US [3829] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101 [3830] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101_Ft_US [3831] NAD_1983_CORS96_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US [3832] NAD_1983_CORS96_StatePlane_New_York_Long_Island_FIPS_3104 [3833] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103 [3834] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103_Ft_US [3835] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200 [3836] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200_Ft_US [3837] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301 [3838] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301_FtI [3839] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302 [3840] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302_FtI [3841] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401 [3842] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401_Ft_US [3843] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402 [3844] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402_Ft_US [3845] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501 [3846] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501_Ft_US [3847] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502 [3848] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502_Ft_US [3849] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3850] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3851] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3852] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3853] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701 [3854] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US [3855] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702 [3856] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US [3857] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800 [3858] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800_Ft_US [3859] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900 [3860] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900_Ft_Intl [3861] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001 [3862] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001_Ft_US [3863] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002 [3864] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002_Ft_US [3865] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100 [3866] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100_Ft_US [3867] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203 [3868] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203_Ft_US [3869] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202 [3870] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202_FtUS [3871] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201 [3872] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201_Ft_US [3873] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204 [3874] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204_FtUS [3875] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205 [3876] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205_Ft_US [3877] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302 [3878] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3879] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_US [3880] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301 [3881] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3882] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_US [3883] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303 [3884] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3885] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_US [3886] NAD_1983_CORS96_StatePlane_Vermont_FIPS_4400 [3887] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501 [3888] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501_Ft_US [3889] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502 [3890] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502_Ft_US [3891] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601 [3892] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601_Ft_US [3893] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602 [3894] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602_Ft_US [3895] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701 [3896] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701_FtUS [3897] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702 [3898] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702_FtUS [3899] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802 [3900] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US [3901] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801 [3902] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801_Ft_US [3903] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803 [3904] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803_Ft_US [3905] NAD_1983_CORS96_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US [3906] NAD_1983_CORS96_StatePlane_Wyoming_East_Central_FIPS_4902 [3907] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901 [3908] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901_Ft_US [3909] NAD_1983_CORS96_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US [3910] NAD_1983_CORS96_StatePlane_Wyoming_West_Central_FIPS_4903 [3911] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904 [3912] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904_Ft_US [3913] NAD_1983_CORS96_UTM_Zone_10N [3914] NAD_1983_CORS96_UTM_Zone_11N [3915] NAD_1983_CORS96_UTM_Zone_12N [3916] NAD_1983_CORS96_UTM_Zone_13N [3917] NAD_1983_CORS96_UTM_Zone_14N [3918] NAD_1983_CORS96_UTM_Zone_15N [3919] NAD_1983_CORS96_UTM_Zone_16N [3920] NAD_1983_CORS96_UTM_Zone_17N [3921] NAD_1983_CORS96_UTM_Zone_18N [3922] NAD_1983_CORS96_UTM_Zone_19N [3923] NAD_1983_CORS96_UTM_Zone_1N [3924] NAD_1983_CORS96_UTM_Zone_20N [3925] NAD_1983_CORS96_UTM_Zone_2N [3926] NAD_1983_CORS96_UTM_Zone_3N [3927] NAD_1983_CORS96_UTM_Zone_4N [3928] NAD_1983_CORS96_UTM_Zone_59N [3929] NAD_1983_CORS96_UTM_Zone_5N [3930] NAD_1983_CORS96_UTM_Zone_60N [3931] NAD_1983_CORS96_UTM_Zone_6N [3932] NAD_1983_CORS96_UTM_Zone_7N [3933] NAD_1983_CORS96_UTM_Zone_8N [3934] NAD_1983_CORS96_UTM_Zone_9N [3935] NAD_1983_CSRS_UTM_Zone_14N (deprecated) [3936] NAD_1983_CSRS_UTM_Zone_15N (deprecated) [3937] NAD_1983_CSRS_UTM_Zone_16N (deprecated) [3938] NAD_1983_California_Teale_Albers_FtUS [3939] NAD_1983_Fargo_Ground_Coordinate_System [3940] NAD_1983_Georgia_Statewide_Lambert [3941] NAD_1983_HARN_Adj_MN_Aitkin_Feet [3942] NAD_1983_HARN_Adj_MN_Aitkin_Meters [3943] NAD_1983_HARN_Adj_MN_Anoka_Feet [3944] NAD_1983_HARN_Adj_MN_Anoka_Meters [3945] NAD_1983_HARN_Adj_MN_Becker_Feet [3946] NAD_1983_HARN_Adj_MN_Becker_Meters [3947] NAD_1983_HARN_Adj_MN_Beltrami_North_Feet [3948] NAD_1983_HARN_Adj_MN_Beltrami_North_Meters [3949] NAD_1983_HARN_Adj_MN_Beltrami_South_Feet [3950] NAD_1983_HARN_Adj_MN_Beltrami_South_Meters [3951] NAD_1983_HARN_Adj_MN_Benton_Feet [3952] NAD_1983_HARN_Adj_MN_Benton_Meters [3953] NAD_1983_HARN_Adj_MN_Big_Stone_Feet [3954] NAD_1983_HARN_Adj_MN_Big_Stone_Meters [3955] NAD_1983_HARN_Adj_MN_Blue_Earth_Feet [3956] NAD_1983_HARN_Adj_MN_Blue_Earth_Meters [3957] NAD_1983_HARN_Adj_MN_Brown_Feet [3958] NAD_1983_HARN_Adj_MN_Brown_Meters [3959] NAD_1983_HARN_Adj_MN_Carlton_Feet [3960] NAD_1983_HARN_Adj_MN_Carlton_Meters [3961] NAD_1983_HARN_Adj_MN_Carver_Feet [3962] NAD_1983_HARN_Adj_MN_Carver_Meters [3963] NAD_1983_HARN_Adj_MN_Cass_North_Feet [3964] NAD_1983_HARN_Adj_MN_Cass_North_Meters [3965] NAD_1983_HARN_Adj_MN_Cass_South_Feet [3966] NAD_1983_HARN_Adj_MN_Cass_South_Meters [3967] NAD_1983_HARN_Adj_MN_Chippewa_Feet [3968] NAD_1983_HARN_Adj_MN_Chippewa_Meters [3969] NAD_1983_HARN_Adj_MN_Chisago_Feet [3970] NAD_1983_HARN_Adj_MN_Chisago_Meters [3971] NAD_1983_HARN_Adj_MN_Clay_Feet [3972] NAD_1983_HARN_Adj_MN_Clay_Meters [3973] NAD_1983_HARN_Adj_MN_Clearwater_Feet [3974] NAD_1983_HARN_Adj_MN_Clearwater_Meters [3975] NAD_1983_HARN_Adj_MN_Cook_North_Feet [3976] NAD_1983_HARN_Adj_MN_Cook_North_Meters [3977] NAD_1983_HARN_Adj_MN_Cook_South_Feet [3978] NAD_1983_HARN_Adj_MN_Cook_South_Meters [3979] NAD_1983_HARN_Adj_MN_Cottonwood_Feet [3980] NAD_1983_HARN_Adj_MN_Cottonwood_Meters [3981] NAD_1983_HARN_Adj_MN_Crow_Wing_Feet [3982] NAD_1983_HARN_Adj_MN_Crow_Wing_Meters [3983] NAD_1983_HARN_Adj_MN_Dakota_Feet [3984] NAD_1983_HARN_Adj_MN_Dakota_Meters [3985] NAD_1983_HARN_Adj_MN_Dodge_Feet [3986] NAD_1983_HARN_Adj_MN_Dodge_Meters [3987] NAD_1983_HARN_Adj_MN_Douglas_Feet [3988] NAD_1983_HARN_Adj_MN_Douglas_Meters [3989] NAD_1983_HARN_Adj_MN_Faribault_Feet [3990] NAD_1983_HARN_Adj_MN_Faribault_Meters [3991] NAD_1983_HARN_Adj_MN_Fillmore_Feet [3992] NAD_1983_HARN_Adj_MN_Fillmore_Meters [3993] NAD_1983_HARN_Adj_MN_Freeborn_Feet [3994] NAD_1983_HARN_Adj_MN_Freeborn_Meters [3995] NAD_1983_HARN_Adj_MN_Goodhue_Feet [3996] NAD_1983_HARN_Adj_MN_Goodhue_Meters [3997] NAD_1983_HARN_Adj_MN_Grant_Feet [3998] NAD_1983_HARN_Adj_MN_Grant_Meters [3999] NAD_1983_HARN_Adj_MN_Hennepin_Feet [4000] NAD_1983_HARN_Adj_MN_Hennepin_Meters [4001] NAD_1983_HARN_Adj_MN_Houston_Feet [4002] NAD_1983_HARN_Adj_MN_Houston_Meters [4003] NAD_1983_HARN_Adj_MN_Hubbard_Feet [4004] NAD_1983_HARN_Adj_MN_Hubbard_Meters [4005] NAD_1983_HARN_Adj_MN_Isanti_Feet [4006] NAD_1983_HARN_Adj_MN_Isanti_Meters [4007] NAD_1983_HARN_Adj_MN_Itasca_North_Feet [4008] NAD_1983_HARN_Adj_MN_Itasca_North_Meters [4009] NAD_1983_HARN_Adj_MN_Itasca_South_Feet [4010] NAD_1983_HARN_Adj_MN_Itasca_South_Meters [4011] NAD_1983_HARN_Adj_MN_Jackson_Feet [4012] NAD_1983_HARN_Adj_MN_Jackson_Meters [4013] NAD_1983_HARN_Adj_MN_Kanabec_Feet [4014] NAD_1983_HARN_Adj_MN_Kanabec_Meters [4015] NAD_1983_HARN_Adj_MN_Kandiyohi_Feet [4016] NAD_1983_HARN_Adj_MN_Kandiyohi_Meters [4017] NAD_1983_HARN_Adj_MN_Kittson_Feet [4018] NAD_1983_HARN_Adj_MN_Kittson_Meters [4019] NAD_1983_HARN_Adj_MN_Koochiching_Feet [4020] NAD_1983_HARN_Adj_MN_Koochiching_Meters [4021] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Feet [4022] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Meters [4023] NAD_1983_HARN_Adj_MN_Lake_Feet [4024] NAD_1983_HARN_Adj_MN_Lake_Meters [4025] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Feet [4026] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Meters [4027] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Feet [4028] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Meters [4029] NAD_1983_HARN_Adj_MN_Le_Sueur_Feet [4030] NAD_1983_HARN_Adj_MN_Le_Sueur_Meters [4031] NAD_1983_HARN_Adj_MN_Lincoln_Feet [4032] NAD_1983_HARN_Adj_MN_Lincoln_Meters [4033] NAD_1983_HARN_Adj_MN_Lyon_Feet [4034] NAD_1983_HARN_Adj_MN_Lyon_Meters [4035] NAD_1983_HARN_Adj_MN_Mahnomen_Feet [4036] NAD_1983_HARN_Adj_MN_Mahnomen_Meters [4037] NAD_1983_HARN_Adj_MN_Marshall_Feet [4038] NAD_1983_HARN_Adj_MN_Marshall_Meters [4039] NAD_1983_HARN_Adj_MN_Martin_Feet [4040] NAD_1983_HARN_Adj_MN_Martin_Meters [4041] NAD_1983_HARN_Adj_MN_McLeod_Feet [4042] NAD_1983_HARN_Adj_MN_McLeod_Meters [4043] NAD_1983_HARN_Adj_MN_Meeker_Feet [4044] NAD_1983_HARN_Adj_MN_Meeker_Meters [4045] NAD_1983_HARN_Adj_MN_Mille_Lacs_Feet [4046] NAD_1983_HARN_Adj_MN_Mille_Lacs_Meters [4047] NAD_1983_HARN_Adj_MN_Morrison_Feet [4048] NAD_1983_HARN_Adj_MN_Morrison_Meters [4049] NAD_1983_HARN_Adj_MN_Mower_Feet [4050] NAD_1983_HARN_Adj_MN_Mower_Meters [4051] NAD_1983_HARN_Adj_MN_Murray_Feet [4052] NAD_1983_HARN_Adj_MN_Murray_Meters [4053] NAD_1983_HARN_Adj_MN_Nicollet_Feet [4054] NAD_1983_HARN_Adj_MN_Nicollet_Meters [4055] NAD_1983_HARN_Adj_MN_Nobles_Feet [4056] NAD_1983_HARN_Adj_MN_Nobles_Meters [4057] NAD_1983_HARN_Adj_MN_Norman_Feet [4058] NAD_1983_HARN_Adj_MN_Norman_Meters [4059] NAD_1983_HARN_Adj_MN_Olmsted_Feet [4060] NAD_1983_HARN_Adj_MN_Olmsted_Meters [4061] NAD_1983_HARN_Adj_MN_Ottertail_Feet [4062] NAD_1983_HARN_Adj_MN_Ottertail_Meters [4063] NAD_1983_HARN_Adj_MN_Pennington_Feet [4064] NAD_1983_HARN_Adj_MN_Pennington_Meters [4065] NAD_1983_HARN_Adj_MN_Pine_Feet [4066] NAD_1983_HARN_Adj_MN_Pine_Meters [4067] NAD_1983_HARN_Adj_MN_Pipestone_Feet [4068] NAD_1983_HARN_Adj_MN_Pipestone_Meters [4069] NAD_1983_HARN_Adj_MN_Polk_Feet [4070] NAD_1983_HARN_Adj_MN_Polk_Meters [4071] NAD_1983_HARN_Adj_MN_Pope_Feet [4072] NAD_1983_HARN_Adj_MN_Pope_Meters [4073] NAD_1983_HARN_Adj_MN_Ramsey_Feet [4074] NAD_1983_HARN_Adj_MN_Ramsey_Meters [4075] NAD_1983_HARN_Adj_MN_Red_Lake_Feet [4076] NAD_1983_HARN_Adj_MN_Red_Lake_Meters [4077] NAD_1983_HARN_Adj_MN_Redwood_Feet [4078] NAD_1983_HARN_Adj_MN_Redwood_Meters [4079] NAD_1983_HARN_Adj_MN_Renville_Feet [4080] NAD_1983_HARN_Adj_MN_Renville_Meters [4081] NAD_1983_HARN_Adj_MN_Rice_Feet [4082] NAD_1983_HARN_Adj_MN_Rice_Meters [4083] NAD_1983_HARN_Adj_MN_Rock_Feet [4084] NAD_1983_HARN_Adj_MN_Rock_Meters [4085] NAD_1983_HARN_Adj_MN_Roseau_Feet [4086] NAD_1983_HARN_Adj_MN_Roseau_Meters [4087] NAD_1983_HARN_Adj_MN_Scott_Feet [4088] NAD_1983_HARN_Adj_MN_Scott_Meters [4089] NAD_1983_HARN_Adj_MN_Sherburne_Feet [4090] NAD_1983_HARN_Adj_MN_Sherburne_Meters [4091] NAD_1983_HARN_Adj_MN_Sibley_Feet [4092] NAD_1983_HARN_Adj_MN_Sibley_Meters [4093] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Feet [4094] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Meters [4095] NAD_1983_HARN_Adj_MN_St_Louis_Central_Feet [4096] NAD_1983_HARN_Adj_MN_St_Louis_Central_Meters [4097] NAD_1983_HARN_Adj_MN_St_Louis_North_Feet [4098] NAD_1983_HARN_Adj_MN_St_Louis_North_Meters [4099] NAD_1983_HARN_Adj_MN_St_Louis_South_Feet [4100] NAD_1983_HARN_Adj_MN_St_Louis_South_Meters [4101] NAD_1983_HARN_Adj_MN_Stearns_Feet [4102] NAD_1983_HARN_Adj_MN_Stearns_Meters [4103] NAD_1983_HARN_Adj_MN_Steele_Feet [4104] NAD_1983_HARN_Adj_MN_Steele_Meters [4105] NAD_1983_HARN_Adj_MN_Stevens_Feet [4106] NAD_1983_HARN_Adj_MN_Stevens_Meters [4107] NAD_1983_HARN_Adj_MN_Swift_Feet [4108] NAD_1983_HARN_Adj_MN_Swift_Meters [4109] NAD_1983_HARN_Adj_MN_Todd_Feet [4110] NAD_1983_HARN_Adj_MN_Todd_Meters [4111] NAD_1983_HARN_Adj_MN_Traverse_Feet [4112] NAD_1983_HARN_Adj_MN_Traverse_Meters [4113] NAD_1983_HARN_Adj_MN_Wabasha_Feet [4114] NAD_1983_HARN_Adj_MN_Wabasha_Meters [4115] NAD_1983_HARN_Adj_MN_Wadena_Feet [4116] NAD_1983_HARN_Adj_MN_Wadena_Meters [4117] NAD_1983_HARN_Adj_MN_Waseca_Feet [4118] NAD_1983_HARN_Adj_MN_Waseca_Meters [4119] NAD_1983_HARN_Adj_MN_Washington_Feet [4120] NAD_1983_HARN_Adj_MN_Washington_Meters [4121] NAD_1983_HARN_Adj_MN_Watonwan_Feet [4122] NAD_1983_HARN_Adj_MN_Watonwan_Meters [4123] NAD_1983_HARN_Adj_MN_Wilkin_Feet [4124] NAD_1983_HARN_Adj_MN_Wilkin_Meters [4125] NAD_1983_HARN_Adj_MN_Winona_Feet [4126] NAD_1983_HARN_Adj_MN_Winona_Meters [4127] NAD_1983_HARN_Adj_MN_Wright_Feet [4128] NAD_1983_HARN_Adj_MN_Wright_Meters [4129] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Feet [4130] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Meters [4131] NAD_1983_HARN_Adj_WI_Adams_Feet (deprecated) [4132] NAD_1983_HARN_Adj_WI_Adams_Meters (deprecated) [4133] NAD_1983_HARN_Adj_WI_Ashland_Feet (deprecated) [4134] NAD_1983_HARN_Adj_WI_Ashland_Meters (deprecated) [4135] NAD_1983_HARN_Adj_WI_Barron_Feet (deprecated) [4136] NAD_1983_HARN_Adj_WI_Barron_Meters (deprecated) [4137] NAD_1983_HARN_Adj_WI_Bayfield_Feet (deprecated) [4138] NAD_1983_HARN_Adj_WI_Bayfield_Meters (deprecated) [4139] NAD_1983_HARN_Adj_WI_Brown_Feet (deprecated) [4140] NAD_1983_HARN_Adj_WI_Brown_Meters (deprecated) [4141] NAD_1983_HARN_Adj_WI_Buffalo_Feet (deprecated) [4142] NAD_1983_HARN_Adj_WI_Buffalo_Meters (deprecated) [4143] NAD_1983_HARN_Adj_WI_Burnett_Feet (deprecated) [4144] NAD_1983_HARN_Adj_WI_Burnett_Meters (deprecated) [4145] NAD_1983_HARN_Adj_WI_Calumet_Feet (deprecated) [4146] NAD_1983_HARN_Adj_WI_Calumet_Meters (deprecated) [4147] NAD_1983_HARN_Adj_WI_Chippewa_Feet [4148] NAD_1983_HARN_Adj_WI_Chippewa_Meters [4149] NAD_1983_HARN_Adj_WI_Clark_Feet (deprecated) [4150] NAD_1983_HARN_Adj_WI_Clark_Meters (deprecated) [4151] NAD_1983_HARN_Adj_WI_Columbia_Feet (deprecated) [4152] NAD_1983_HARN_Adj_WI_Columbia_Meters (deprecated) [4153] NAD_1983_HARN_Adj_WI_Crawford_Feet (deprecated) [4154] NAD_1983_HARN_Adj_WI_Crawford_Meters (deprecated) [4155] NAD_1983_HARN_Adj_WI_Dane_Feet (deprecated) [4156] NAD_1983_HARN_Adj_WI_Dane_Meters (deprecated) [4157] NAD_1983_HARN_Adj_WI_Dodge_Feet (deprecated) [4158] NAD_1983_HARN_Adj_WI_Dodge_Meters (deprecated) [4159] NAD_1983_HARN_Adj_WI_Door_Feet (deprecated) [4160] NAD_1983_HARN_Adj_WI_Door_Meters (deprecated) [4161] NAD_1983_HARN_Adj_WI_Douglas_Feet (deprecated) [4162] NAD_1983_HARN_Adj_WI_Douglas_Meters (deprecated) [4163] NAD_1983_HARN_Adj_WI_Dunn_Feet (deprecated) [4164] NAD_1983_HARN_Adj_WI_Dunn_Meters (deprecated) [4165] NAD_1983_HARN_Adj_WI_EauClaire_Feet (deprecated) [4166] NAD_1983_HARN_Adj_WI_EauClaire_Meters (deprecated) [4167] NAD_1983_HARN_Adj_WI_Florence_Feet (deprecated) [4168] NAD_1983_HARN_Adj_WI_Florence_Meters (deprecated) [4169] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Feet (deprecated) [4170] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Meters (deprecated) [4171] NAD_1983_HARN_Adj_WI_Forest_Feet (deprecated) [4172] NAD_1983_HARN_Adj_WI_Forest_Meters (deprecated) [4173] NAD_1983_HARN_Adj_WI_Grant_Feet (deprecated) [4174] NAD_1983_HARN_Adj_WI_Grant_Meters (deprecated) [4175] NAD_1983_HARN_Adj_WI_GreenLake_Feet (deprecated) [4176] NAD_1983_HARN_Adj_WI_GreenLake_Meters (deprecated) [4177] NAD_1983_HARN_Adj_WI_Green_Feet (deprecated) [4178] NAD_1983_HARN_Adj_WI_Green_Meters (deprecated) [4179] NAD_1983_HARN_Adj_WI_Iowa_Feet (deprecated) [4180] NAD_1983_HARN_Adj_WI_Iowa_Meters (deprecated) [4181] NAD_1983_HARN_Adj_WI_Iron_Feet (deprecated) [4182] NAD_1983_HARN_Adj_WI_Iron_Meters (deprecated) [4183] NAD_1983_HARN_Adj_WI_Jackson_Feet (deprecated) [4184] NAD_1983_HARN_Adj_WI_Jackson_Meters (deprecated) [4185] NAD_1983_HARN_Adj_WI_Jefferson_Feet (deprecated) [4186] NAD_1983_HARN_Adj_WI_Jefferson_Meters (deprecated) [4187] NAD_1983_HARN_Adj_WI_Juneau_Feet (deprecated) [4188] NAD_1983_HARN_Adj_WI_Juneau_Meters (deprecated) [4189] NAD_1983_HARN_Adj_WI_Kenosha_Feet (deprecated) [4190] NAD_1983_HARN_Adj_WI_Kenosha_Meters (deprecated) [4191] NAD_1983_HARN_Adj_WI_Kewaunee_Feet (deprecated) [4192] NAD_1983_HARN_Adj_WI_Kewaunee_Meters (deprecated) [4193] NAD_1983_HARN_Adj_WI_LaCrosse_Feet (deprecated) [4194] NAD_1983_HARN_Adj_WI_LaCrosse_Meters (deprecated) [4195] NAD_1983_HARN_Adj_WI_Lafayette_Feet (deprecated) [4196] NAD_1983_HARN_Adj_WI_Lafayette_Meters (deprecated) [4197] NAD_1983_HARN_Adj_WI_Langlade_Feet (deprecated) [4198] NAD_1983_HARN_Adj_WI_Langlade_Meters (deprecated) [4199] NAD_1983_HARN_Adj_WI_Lincoln_Feet (deprecated) [4200] NAD_1983_HARN_Adj_WI_Lincoln_Meters (deprecated) [4201] NAD_1983_HARN_Adj_WI_Manitowoc_Feet (deprecated) [4202] NAD_1983_HARN_Adj_WI_Manitowoc_Meters (deprecated) [4203] NAD_1983_HARN_Adj_WI_Marathon_Feet (deprecated) [4204] NAD_1983_HARN_Adj_WI_Marathon_Meters (deprecated) [4205] NAD_1983_HARN_Adj_WI_Marinette_Feet (deprecated) [4206] NAD_1983_HARN_Adj_WI_Marinette_Meters (deprecated) [4207] NAD_1983_HARN_Adj_WI_Marquette_Feet (deprecated) [4208] NAD_1983_HARN_Adj_WI_Marquette_Meters (deprecated) [4209] NAD_1983_HARN_Adj_WI_Menominee_Feet (deprecated) [4210] NAD_1983_HARN_Adj_WI_Menominee_Meters (deprecated) [4211] NAD_1983_HARN_Adj_WI_Milwaukee_Feet (deprecated) [4212] NAD_1983_HARN_Adj_WI_Milwaukee_Meters (deprecated) [4213] NAD_1983_HARN_Adj_WI_Monroe_Feet (deprecated) [4214] NAD_1983_HARN_Adj_WI_Monroe_Meters (deprecated) [4215] NAD_1983_HARN_Adj_WI_Oconto_Feet (deprecated) [4216] NAD_1983_HARN_Adj_WI_Oconto_Meters (deprecated) [4217] NAD_1983_HARN_Adj_WI_Oneida_Feet (deprecated) [4218] NAD_1983_HARN_Adj_WI_Oneida_Meters (deprecated) [4219] NAD_1983_HARN_Adj_WI_Outagamie_Feet (deprecated) [4220] NAD_1983_HARN_Adj_WI_Outagamie_Meters (deprecated) [4221] NAD_1983_HARN_Adj_WI_Ozaukee_Feet (deprecated) [4222] NAD_1983_HARN_Adj_WI_Ozaukee_Meters (deprecated) [4223] NAD_1983_HARN_Adj_WI_Pepin_Feet (deprecated) [4224] NAD_1983_HARN_Adj_WI_Pepin_Meters (deprecated) [4225] NAD_1983_HARN_Adj_WI_Pierce_Feet (deprecated) [4226] NAD_1983_HARN_Adj_WI_Pierce_Meters (deprecated) [4227] NAD_1983_HARN_Adj_WI_Polk_Feet (deprecated) [4228] NAD_1983_HARN_Adj_WI_Polk_Meters (deprecated) [4229] NAD_1983_HARN_Adj_WI_Portage_Feet (deprecated) [4230] NAD_1983_HARN_Adj_WI_Portage_Meters (deprecated) [4231] NAD_1983_HARN_Adj_WI_Price_Feet (deprecated) [4232] NAD_1983_HARN_Adj_WI_Price_Meters (deprecated) [4233] NAD_1983_HARN_Adj_WI_Racine_Feet (deprecated) [4234] NAD_1983_HARN_Adj_WI_Racine_Meters (deprecated) [4235] NAD_1983_HARN_Adj_WI_Richland_Feet (deprecated) [4236] NAD_1983_HARN_Adj_WI_Richland_Meters (deprecated) [4237] NAD_1983_HARN_Adj_WI_Rock_Feet (deprecated) [4238] NAD_1983_HARN_Adj_WI_Rock_Meters (deprecated) [4239] NAD_1983_HARN_Adj_WI_Rusk_Feet (deprecated) [4240] NAD_1983_HARN_Adj_WI_Rusk_Meters (deprecated) [4241] NAD_1983_HARN_Adj_WI_Sauk_Feet (deprecated) [4242] NAD_1983_HARN_Adj_WI_Sauk_Meters (deprecated) [4243] NAD_1983_HARN_Adj_WI_Sawyer_Feet (deprecated) [4244] NAD_1983_HARN_Adj_WI_Sawyer_Meters (deprecated) [4245] NAD_1983_HARN_Adj_WI_Shawano_Feet (deprecated) [4246] NAD_1983_HARN_Adj_WI_Shawano_Meters (deprecated) [4247] NAD_1983_HARN_Adj_WI_Sheboygan_Feet (deprecated) [4248] NAD_1983_HARN_Adj_WI_Sheboygan_Meters (deprecated) [4249] NAD_1983_HARN_Adj_WI_St_Croix_Feet (deprecated) [4250] NAD_1983_HARN_Adj_WI_St_Croix_Meters (deprecated) [4251] NAD_1983_HARN_Adj_WI_Taylor_Feet (deprecated) [4252] NAD_1983_HARN_Adj_WI_Taylor_Meters (deprecated) [4253] NAD_1983_HARN_Adj_WI_Trempealeau_Feet (deprecated) [4254] NAD_1983_HARN_Adj_WI_Trempealeau_Meters (deprecated) [4255] NAD_1983_HARN_Adj_WI_Vernon_Feet (deprecated) [4256] NAD_1983_HARN_Adj_WI_Vernon_Meters (deprecated) [4257] NAD_1983_HARN_Adj_WI_Vilas_Feet (deprecated) [4258] NAD_1983_HARN_Adj_WI_Vilas_Meters (deprecated) [4259] NAD_1983_HARN_Adj_WI_Walworth_Feet (deprecated) [4260] NAD_1983_HARN_Adj_WI_Walworth_Meters (deprecated) [4261] NAD_1983_HARN_Adj_WI_Washburn_Feet (deprecated) [4262] NAD_1983_HARN_Adj_WI_Washburn_Meters (deprecated) [4263] NAD_1983_HARN_Adj_WI_Washington_Feet (deprecated) [4264] NAD_1983_HARN_Adj_WI_Washington_Meters (deprecated) [4265] NAD_1983_HARN_Adj_WI_Waukesha_Feet (deprecated) [4266] NAD_1983_HARN_Adj_WI_Waukesha_Meters (deprecated) [4267] NAD_1983_HARN_Adj_WI_Waupaca_Feet (deprecated) [4268] NAD_1983_HARN_Adj_WI_Waupaca_Meters (deprecated) [4269] NAD_1983_HARN_Adj_WI_Waushara_Feet (deprecated) [4270] NAD_1983_HARN_Adj_WI_Waushara_Meters (deprecated) [4271] NAD_1983_HARN_Adj_WI_Winnebago_Feet (deprecated) [4272] NAD_1983_HARN_Adj_WI_Winnebago_Meters (deprecated) [4273] NAD_1983_HARN_Adj_WI_Wood_Feet (deprecated) [4274] NAD_1983_HARN_Adj_WI_Wood_Meters (deprecated) [4275] NAD_1983_HARN_Fargo_Ground_Coordinate_System [4276] NAD_1983_HARN_Guam_Map_Grid (deprecated) [4277] NAD_1983_HARN_Maine_2000_Central_Zone (deprecated) [4278] NAD_1983_HARN_Maine_2000_East_Zone (deprecated) [4279] NAD_1983_HARN_Maine_2000_West_Zone (deprecated) [4280] NAD_1983_HARN_Mississippi_TM (deprecated) [4281] NAD_1983_HARN_StatePlane_Alabama_East_FIPS_0101 (deprecated) [4282] NAD_1983_HARN_StatePlane_Alabama_West_FIPS_0102 (deprecated) [4283] NAD_1983_HARN_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [4284] NAD_1983_HARN_StatePlane_Arizona_East_FIPS_0201 (deprecated) [4285] NAD_1983_HARN_StatePlane_Arizona_West_FIPS_0203 (deprecated) [4286] NAD_1983_HARN_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [4287] NAD_1983_HARN_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [4288] NAD_1983_HARN_StatePlane_California_III_FIPS_0403 (deprecated) [4289] NAD_1983_HARN_StatePlane_California_II_FIPS_0402 (deprecated) [4290] NAD_1983_HARN_StatePlane_California_IV_FIPS_0404 (deprecated) [4291] NAD_1983_HARN_StatePlane_California_I_FIPS_0401 (deprecated) [4292] NAD_1983_HARN_StatePlane_California_VI_FIPS_0406 (deprecated) [4293] NAD_1983_HARN_StatePlane_California_V_FIPS_0405 (deprecated) [4294] NAD_1983_HARN_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [4295] NAD_1983_HARN_StatePlane_Colorado_North_FIPS_0501 (deprecated) [4296] NAD_1983_HARN_StatePlane_Colorado_South_FIPS_0503 (deprecated) [4297] NAD_1983_HARN_StatePlane_Connecticut_FIPS_0600 (deprecated) [4298] NAD_1983_HARN_StatePlane_Delaware_FIPS_0700 (deprecated) [4299] NAD_1983_HARN_StatePlane_Florida_East_FIPS_0901 (deprecated) [4300] NAD_1983_HARN_StatePlane_Florida_North_FIPS_0903 (deprecated) [4301] NAD_1983_HARN_StatePlane_Florida_West_FIPS_0902 (deprecated) [4302] NAD_1983_HARN_StatePlane_Georgia_East_FIPS_1001 (deprecated) [4303] NAD_1983_HARN_StatePlane_Georgia_West_FIPS_1002 (deprecated) [4304] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4305] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101_Feet [4306] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4307] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102_Feet [4308] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4309] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4310] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4311] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104_Feet [4312] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4313] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105_Feet [4314] NAD_1983_HARN_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [4315] NAD_1983_HARN_StatePlane_Idaho_East_FIPS_1101 (deprecated) [4316] NAD_1983_HARN_StatePlane_Idaho_West_FIPS_1103 (deprecated) [4317] NAD_1983_HARN_StatePlane_Illinois_East_FIPS_1201 (deprecated) [4318] NAD_1983_HARN_StatePlane_Illinois_West_FIPS_1202 (deprecated) [4319] NAD_1983_HARN_StatePlane_Indiana_East_FIPS_1301 (deprecated) [4320] NAD_1983_HARN_StatePlane_Indiana_West_FIPS_1302 (deprecated) [4321] NAD_1983_HARN_StatePlane_Iowa_North_FIPS_1401 (deprecated) [4322] NAD_1983_HARN_StatePlane_Iowa_South_FIPS_1402 (deprecated) [4323] NAD_1983_HARN_StatePlane_Kansas_North_FIPS_1501 (deprecated) [4324] NAD_1983_HARN_StatePlane_Kansas_South_FIPS_1502 (deprecated) [4325] NAD_1983_HARN_StatePlane_Kentucky_FIPS_1600 (deprecated) [4326] NAD_1983_HARN_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [4327] NAD_1983_HARN_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [4328] NAD_1983_HARN_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [4329] NAD_1983_HARN_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [4330] NAD_1983_HARN_StatePlane_Maine_East_FIPS_1801 (deprecated) [4331] NAD_1983_HARN_StatePlane_Maine_West_FIPS_1802 (deprecated) [4332] NAD_1983_HARN_StatePlane_Maryland_FIPS_1900 (deprecated) [4333] NAD_1983_HARN_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [4334] NAD_1983_HARN_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [4335] NAD_1983_HARN_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [4336] NAD_1983_HARN_StatePlane_Michigan_North_FIPS_2111 (deprecated) [4337] NAD_1983_HARN_StatePlane_Michigan_South_FIPS_2113 (deprecated) [4338] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [4339] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4340] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [4341] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4342] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [4343] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4344] NAD_1983_HARN_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [4345] NAD_1983_HARN_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [4346] NAD_1983_HARN_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [4347] NAD_1983_HARN_StatePlane_Missouri_East_FIPS_2401 (deprecated) [4348] NAD_1983_HARN_StatePlane_Missouri_West_FIPS_2403 (deprecated) [4349] NAD_1983_HARN_StatePlane_Montana_FIPS_2500 (deprecated) [4350] NAD_1983_HARN_StatePlane_Nebraska_FIPS_2600 (deprecated) [4351] NAD_1983_HARN_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [4352] NAD_1983_HARN_StatePlane_Nevada_East_FIPS_2701 (deprecated) [4353] NAD_1983_HARN_StatePlane_Nevada_West_FIPS_2703 (deprecated) [4354] NAD_1983_HARN_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [4355] NAD_1983_HARN_StatePlane_New_Jersey_FIPS_2900 (deprecated) [4356] NAD_1983_HARN_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [4357] NAD_1983_HARN_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [4358] NAD_1983_HARN_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [4359] NAD_1983_HARN_StatePlane_New_York_Central_FIPS_3102 (deprecated) [4360] NAD_1983_HARN_StatePlane_New_York_East_FIPS_3101 (deprecated) [4361] NAD_1983_HARN_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [4362] NAD_1983_HARN_StatePlane_New_York_West_FIPS_3103 (deprecated) [4363] NAD_1983_HARN_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [4364] NAD_1983_HARN_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [4365] NAD_1983_HARN_StatePlane_Ohio_North_FIPS_3401 (deprecated) [4366] NAD_1983_HARN_StatePlane_Ohio_South_FIPS_3402 (deprecated) [4367] NAD_1983_HARN_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [4368] NAD_1983_HARN_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [4369] NAD_1983_HARN_StatePlane_Oregon_North_FIPS_3601 (deprecated) [4370] NAD_1983_HARN_StatePlane_Oregon_South_FIPS_3602 (deprecated) [4371] NAD_1983_HARN_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200 (deprecated) [4372] NAD_1983_HARN_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [4373] NAD_1983_HARN_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [4374] NAD_1983_HARN_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [4375] NAD_1983_HARN_StatePlane_Tennessee_FIPS_4100 (deprecated) [4376] NAD_1983_HARN_StatePlane_Texas_Central_FIPS_4203 (deprecated) [4377] NAD_1983_HARN_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [4378] NAD_1983_HARN_StatePlane_Texas_North_FIPS_4201 (deprecated) [4379] NAD_1983_HARN_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [4380] NAD_1983_HARN_StatePlane_Texas_South_FIPS_4205 (deprecated) [4381] NAD_1983_HARN_StatePlane_Utah_Central_FIPS_4302 (deprecated) [4382] NAD_1983_HARN_StatePlane_Utah_North_FIPS_4301 (deprecated) [4383] NAD_1983_HARN_StatePlane_Utah_South_FIPS_4303 (deprecated) [4384] NAD_1983_HARN_StatePlane_Vermont_FIPS_4400 (deprecated) [4385] NAD_1983_HARN_StatePlane_Virginia_North_FIPS_4501 (deprecated) [4386] NAD_1983_HARN_StatePlane_Virginia_South_FIPS_4502 (deprecated) [4387] NAD_1983_HARN_StatePlane_Washington_North_FIPS_4601 (deprecated) [4388] NAD_1983_HARN_StatePlane_Washington_South_FIPS_4602 (deprecated) [4389] NAD_1983_HARN_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [4390] NAD_1983_HARN_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [4391] NAD_1983_HARN_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [4392] NAD_1983_HARN_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [4393] NAD_1983_HARN_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [4394] NAD_1983_HARN_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [4395] NAD_1983_HARN_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [4396] NAD_1983_HARN_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [4397] NAD_1983_HARN_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [4398] NAD_1983_HARN_UTM_Zone_11N (deprecated) [4399] NAD_1983_HARN_UTM_Zone_12N (deprecated) [4400] NAD_1983_HARN_UTM_Zone_13N (deprecated) [4401] NAD_1983_HARN_UTM_Zone_18N (deprecated) [4402] NAD_1983_HARN_UTM_Zone_2S (deprecated) [4403] NAD_1983_HARN_UTM_Zone_4N (deprecated) [4404] NAD_1983_HARN_UTM_Zone_5N (deprecated) [4405] NAD_1983_HARN_WISCRS_Adams_County_Feet [4406] NAD_1983_HARN_WISCRS_Adams_County_Meters [4407] NAD_1983_HARN_WISCRS_Ashland_County_Feet (deprecated) [4408] NAD_1983_HARN_WISCRS_Ashland_County_Meters (deprecated) [4409] NAD_1983_HARN_WISCRS_Barron_County_Feet (deprecated) [4410] NAD_1983_HARN_WISCRS_Barron_County_Meters (deprecated) [4411] NAD_1983_HARN_WISCRS_Bayfield_County_Feet (deprecated) [4412] NAD_1983_HARN_WISCRS_Bayfield_County_Meters (deprecated) [4413] NAD_1983_HARN_WISCRS_Brown_County_Feet (deprecated) [4414] NAD_1983_HARN_WISCRS_Brown_County_Meters (deprecated) [4415] NAD_1983_HARN_WISCRS_Buffalo_County_Feet (deprecated) [4416] NAD_1983_HARN_WISCRS_Buffalo_County_Meters (deprecated) [4417] NAD_1983_HARN_WISCRS_Burnett_County_Feet (deprecated) [4418] NAD_1983_HARN_WISCRS_Burnett_County_Meters (deprecated) [4419] NAD_1983_HARN_WISCRS_Calumet_County_Feet [4420] NAD_1983_HARN_WISCRS_Calumet_County_Meters [4421] NAD_1983_HARN_WISCRS_Chippewa_County_Feet (deprecated) [4422] NAD_1983_HARN_WISCRS_Chippewa_County_Meters (deprecated) [4423] NAD_1983_HARN_WISCRS_Clark_County_Feet (deprecated) [4424] NAD_1983_HARN_WISCRS_Clark_County_Meters (deprecated) [4425] NAD_1983_HARN_WISCRS_Columbia_County_Feet (deprecated) [4426] NAD_1983_HARN_WISCRS_Columbia_County_Meters (deprecated) [4427] NAD_1983_HARN_WISCRS_Crawford_County_Feet (deprecated) [4428] NAD_1983_HARN_WISCRS_Crawford_County_Meters (deprecated) [4429] NAD_1983_HARN_WISCRS_Dane_County_Feet (deprecated) [4430] NAD_1983_HARN_WISCRS_Dane_County_Meters (deprecated) [4431] NAD_1983_HARN_WISCRS_Dodge_County_Feet [4432] NAD_1983_HARN_WISCRS_Dodge_County_Meters [4433] NAD_1983_HARN_WISCRS_Door_County_Feet (deprecated) [4434] NAD_1983_HARN_WISCRS_Door_County_Meters (deprecated) [4435] NAD_1983_HARN_WISCRS_Douglas_County_Feet (deprecated) [4436] NAD_1983_HARN_WISCRS_Douglas_County_Meters (deprecated) [4437] NAD_1983_HARN_WISCRS_Dunn_County_Feet (deprecated) [4438] NAD_1983_HARN_WISCRS_Dunn_County_Meters (deprecated) [4439] NAD_1983_HARN_WISCRS_EauClaire_County_Feet (deprecated) [4440] NAD_1983_HARN_WISCRS_EauClaire_County_Meters (deprecated) [4441] NAD_1983_HARN_WISCRS_Florence_County_Feet (deprecated) [4442] NAD_1983_HARN_WISCRS_Florence_County_Meters (deprecated) [4443] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Feet [4444] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Meters [4445] NAD_1983_HARN_WISCRS_Forest_County_Feet (deprecated) [4446] NAD_1983_HARN_WISCRS_Forest_County_Meters (deprecated) [4447] NAD_1983_HARN_WISCRS_Grant_County_Feet (deprecated) [4448] NAD_1983_HARN_WISCRS_Grant_County_Meters (deprecated) [4449] NAD_1983_HARN_WISCRS_GreenLake_County_Feet [4450] NAD_1983_HARN_WISCRS_GreenLake_County_Meters [4451] NAD_1983_HARN_WISCRS_Green_County_Feet [4452] NAD_1983_HARN_WISCRS_Green_County_Meters [4453] NAD_1983_HARN_WISCRS_Iowa_County_Feet (deprecated) [4454] NAD_1983_HARN_WISCRS_Iowa_County_Meters (deprecated) [4455] NAD_1983_HARN_WISCRS_Iron_County_Feet (deprecated) [4456] NAD_1983_HARN_WISCRS_Iron_County_Meters (deprecated) [4457] NAD_1983_HARN_WISCRS_Jackson_County_Feet (deprecated) [4458] NAD_1983_HARN_WISCRS_Jackson_County_Meters (deprecated) [4459] NAD_1983_HARN_WISCRS_Jefferson_County_Feet [4460] NAD_1983_HARN_WISCRS_Jefferson_County_Meters [4461] NAD_1983_HARN_WISCRS_Juneau_County_Feet [4462] NAD_1983_HARN_WISCRS_Juneau_County_Meters [4463] NAD_1983_HARN_WISCRS_Kenosha_County_Feet [4464] NAD_1983_HARN_WISCRS_Kenosha_County_Meters [4465] NAD_1983_HARN_WISCRS_Kewaunee_County_Feet [4466] NAD_1983_HARN_WISCRS_Kewaunee_County_Meters [4467] NAD_1983_HARN_WISCRS_LaCrosse_County_Feet (deprecated) [4468] NAD_1983_HARN_WISCRS_LaCrosse_County_Meters (deprecated) [4469] NAD_1983_HARN_WISCRS_Lafayette_County_Feet [4470] NAD_1983_HARN_WISCRS_Lafayette_County_Meters [4471] NAD_1983_HARN_WISCRS_Langlade_County_Feet (deprecated) [4472] NAD_1983_HARN_WISCRS_Langlade_County_Meters (deprecated) [4473] NAD_1983_HARN_WISCRS_Lincoln_County_Feet (deprecated) [4474] NAD_1983_HARN_WISCRS_Lincoln_County_Meters (deprecated) [4475] NAD_1983_HARN_WISCRS_Manitowoc_County_Feet [4476] NAD_1983_HARN_WISCRS_Manitowoc_County_Meters [4477] NAD_1983_HARN_WISCRS_Marathon_County_Feet (deprecated) [4478] NAD_1983_HARN_WISCRS_Marathon_County_Meters (deprecated) [4479] NAD_1983_HARN_WISCRS_Marinette_County_Feet (deprecated) [4480] NAD_1983_HARN_WISCRS_Marinette_County_Meters (deprecated) [4481] NAD_1983_HARN_WISCRS_Marquette_County_Feet [4482] NAD_1983_HARN_WISCRS_Marquette_County_Meters [4483] NAD_1983_HARN_WISCRS_Menominee_County_Feet (deprecated) [4484] NAD_1983_HARN_WISCRS_Menominee_County_Meters (deprecated) [4485] NAD_1983_HARN_WISCRS_Milwaukee_County_Feet [4486] NAD_1983_HARN_WISCRS_Milwaukee_County_Meters [4487] NAD_1983_HARN_WISCRS_Monroe_County_Feet (deprecated) [4488] NAD_1983_HARN_WISCRS_Monroe_County_Meters (deprecated) [4489] NAD_1983_HARN_WISCRS_Oconto_County_Feet (deprecated) [4490] NAD_1983_HARN_WISCRS_Oconto_County_Meters (deprecated) [4491] NAD_1983_HARN_WISCRS_Oneida_County_Feet (deprecated) [4492] NAD_1983_HARN_WISCRS_Oneida_County_Meters (deprecated) [4493] NAD_1983_HARN_WISCRS_Outagamie_County_Feet [4494] NAD_1983_HARN_WISCRS_Outagamie_County_Meters [4495] NAD_1983_HARN_WISCRS_Ozaukee_County_Feet [4496] NAD_1983_HARN_WISCRS_Ozaukee_County_Meters [4497] NAD_1983_HARN_WISCRS_Pepin_County_Feet [4498] NAD_1983_HARN_WISCRS_Pepin_County_Meters [4499] NAD_1983_HARN_WISCRS_Pierce_County_Feet [4500] NAD_1983_HARN_WISCRS_Pierce_County_Meters [4501] NAD_1983_HARN_WISCRS_Polk_County_Feet (deprecated) [4502] NAD_1983_HARN_WISCRS_Polk_County_Meters (deprecated) [4503] NAD_1983_HARN_WISCRS_Portage_County_Feet (deprecated) [4504] NAD_1983_HARN_WISCRS_Portage_County_Meters (deprecated) [4505] NAD_1983_HARN_WISCRS_Price_County_Feet (deprecated) [4506] NAD_1983_HARN_WISCRS_Price_County_Meters (deprecated) [4507] NAD_1983_HARN_WISCRS_Racine_County_Feet [4508] NAD_1983_HARN_WISCRS_Racine_County_Meters [4509] NAD_1983_HARN_WISCRS_Richland_County_Feet (deprecated) [4510] NAD_1983_HARN_WISCRS_Richland_County_Meters (deprecated) [4511] NAD_1983_HARN_WISCRS_Rock_County_Feet (deprecated) [4512] NAD_1983_HARN_WISCRS_Rock_County_Meters (deprecated) [4513] NAD_1983_HARN_WISCRS_Rusk_County_Feet (deprecated) [4514] NAD_1983_HARN_WISCRS_Rusk_County_Meters (deprecated) [4515] NAD_1983_HARN_WISCRS_Sauk_County_Feet (deprecated) [4516] NAD_1983_HARN_WISCRS_Sauk_County_Meters (deprecated) [4517] NAD_1983_HARN_WISCRS_Sawyer_County_Feet (deprecated) [4518] NAD_1983_HARN_WISCRS_Sawyer_County_Meters (deprecated) [4519] NAD_1983_HARN_WISCRS_Shawano_County_Feet (deprecated) [4520] NAD_1983_HARN_WISCRS_Shawano_County_Meters (deprecated) [4521] NAD_1983_HARN_WISCRS_Sheboygan_County_Feet [4522] NAD_1983_HARN_WISCRS_Sheboygan_County_Meters [4523] NAD_1983_HARN_WISCRS_St_Croix_County_Feet (deprecated) [4524] NAD_1983_HARN_WISCRS_St_Croix_County_Meters (deprecated) [4525] NAD_1983_HARN_WISCRS_Taylor_County_Feet (deprecated) [4526] NAD_1983_HARN_WISCRS_Taylor_County_Meters (deprecated) [4527] NAD_1983_HARN_WISCRS_Trempealeau_County_Feet (deprecated) [4528] NAD_1983_HARN_WISCRS_Trempealeau_County_Meters (deprecated) [4529] NAD_1983_HARN_WISCRS_Vernon_County_Feet (deprecated) [4530] NAD_1983_HARN_WISCRS_Vernon_County_Meters (deprecated) [4531] NAD_1983_HARN_WISCRS_Vilas_County_Feet (deprecated) [4532] NAD_1983_HARN_WISCRS_Vilas_County_Meters (deprecated) [4533] NAD_1983_HARN_WISCRS_Walworth_County_Feet (deprecated) [4534] NAD_1983_HARN_WISCRS_Walworth_County_Meters (deprecated) [4535] NAD_1983_HARN_WISCRS_Washburn_County_Feet (deprecated) [4536] NAD_1983_HARN_WISCRS_Washburn_County_Meters (deprecated) [4537] NAD_1983_HARN_WISCRS_Washington_County_Feet (deprecated) [4538] NAD_1983_HARN_WISCRS_Washington_County_Meters (deprecated) [4539] NAD_1983_HARN_WISCRS_Waukesha_County_Feet (deprecated) [4540] NAD_1983_HARN_WISCRS_Waukesha_County_Meters (deprecated) [4541] NAD_1983_HARN_WISCRS_Waupaca_County_Feet (deprecated) [4542] NAD_1983_HARN_WISCRS_Waupaca_County_Meters (deprecated) [4543] NAD_1983_HARN_WISCRS_Waushara_County_Feet (deprecated) [4544] NAD_1983_HARN_WISCRS_Waushara_County_Meters (deprecated) [4545] NAD_1983_HARN_WISCRS_Winnebago_County_Feet [4546] NAD_1983_HARN_WISCRS_Winnebago_County_Meters [4547] NAD_1983_HARN_WISCRS_Wood_County_Feet (deprecated) [4548] NAD_1983_HARN_WISCRS_Wood_County_Meters (deprecated) [4549] NAD_1983_HARN_Wisconsin_TM_US_Ft [4550] NAD_1983_Idaho-Ada_County [4551] NAD_1983_Idaho_TM [4552] NAD_1983_MA11_Guam_Map_Grid (deprecated) [4553] NAD_1983_MA11_UTM_Zone_55N [4554] NAD_1983_Maine_2000_Central_Zone (deprecated) [4555] NAD_1983_Maine_2000_East_Zone (deprecated) [4556] NAD_1983_Maine_2000_West_Zone (deprecated) [4557] NAD_1983_Michigan_GeoRef_Feet_US [4558] NAD_1983_Michigan_GeoRef_Meters (deprecated) [4559] NAD_1983_Mississippi_TM (deprecated) [4560] NAD_1983_NSRS2007_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [4561] NAD_1983_NSRS2007_UTM_Zone_20N [4562] NAD_1983_NSRS2007_Wisconsin_TM_US_Ft [4563] NAD_1983_Nebraska_Lancaster_County_FtUS [4564] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4565] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101_Feet [4566] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4567] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102_Feet [4568] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4569] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4570] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4571] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104_Feet [4572] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4573] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105_Feet [4574] NAD_1983_PA11_UTM_Zone_2S (deprecated) [4575] NAD_1983_PA11_UTM_Zone_4N (deprecated) [4576] NAD_1983_PA11_UTM_Zone_5N (deprecated) [4577] NAD_1983_PACP00_UTM_Zone_2S [4578] NAD_1983_PACP00_UTM_Zone_4N [4579] NAD_1983_PACP00_UTM_Zone_5N [4580] NAD_1983_StatePlane_Alabama_East_FIPS_0101_Feet [4581] NAD_1983_StatePlane_Alabama_West_FIPS_0102_Feet [4582] NAD_1983_StatePlane_Alaska_10_FIPS_5010_Feet [4583] NAD_1983_StatePlane_Alaska_1_FIPS_5001_Feet [4584] NAD_1983_StatePlane_Alaska_2_FIPS_5002_Feet [4585] NAD_1983_StatePlane_Alaska_3_FIPS_5003_Feet [4586] NAD_1983_StatePlane_Alaska_4_FIPS_5004_Feet [4587] NAD_1983_StatePlane_Alaska_5_FIPS_5005_Feet [4588] NAD_1983_StatePlane_Alaska_6_FIPS_5006_Feet [4589] NAD_1983_StatePlane_Alaska_7_FIPS_5007_Feet [4590] NAD_1983_StatePlane_Alaska_8_FIPS_5008_Feet [4591] NAD_1983_StatePlane_Alaska_9_FIPS_5009_Feet [4592] NAD_1983_StatePlane_Arizona_Central_FIPS_0202_Feet [4593] NAD_1983_StatePlane_Arizona_East_FIPS_0201_Feet [4594] NAD_1983_StatePlane_Arizona_West_FIPS_0203_Feet [4595] NAD_1983_StatePlane_Arkansas_North_FIPS_0301_Feet (deprecated) [4596] NAD_1983_StatePlane_Arkansas_South_FIPS_0302_Feet (deprecated) [4597] NAD_1983_StatePlane_California_III_FIPS_0403_Feet (deprecated) [4598] NAD_1983_StatePlane_California_II_FIPS_0402_Feet (deprecated) [4599] NAD_1983_StatePlane_California_IV_FIPS_0404_Feet (deprecated) [4600] NAD_1983_StatePlane_California_I_FIPS_0401_Feet (deprecated) [4601] NAD_1983_StatePlane_California_VI_FIPS_0406_Feet (deprecated) [4602] NAD_1983_StatePlane_California_V_FIPS_0405_Feet (deprecated) [4603] NAD_1983_StatePlane_Colorado_Central_FIPS_0502_Feet (deprecated) [4604] NAD_1983_StatePlane_Colorado_North_FIPS_0501_Feet (deprecated) [4605] NAD_1983_StatePlane_Colorado_South_FIPS_0503_Feet (deprecated) [4606] NAD_1983_StatePlane_Connecticut_FIPS_0600_Feet (deprecated) [4607] NAD_1983_StatePlane_Delaware_FIPS_0700_Feet (deprecated) [4608] NAD_1983_StatePlane_Florida_East_FIPS_0901_Feet (deprecated) [4609] NAD_1983_StatePlane_Florida_North_FIPS_0903_Feet (deprecated) [4610] NAD_1983_StatePlane_Florida_West_FIPS_0902_Feet (deprecated) [4611] NAD_1983_StatePlane_Georgia_East_FIPS_1001_Feet (deprecated) [4612] NAD_1983_StatePlane_Georgia_West_FIPS_1002_Feet (deprecated) [4613] NAD_1983_StatePlane_Guam_FIPS_5400 [4614] NAD_1983_StatePlane_Guam_FIPS_5400_Feet [4615] NAD_1983_StatePlane_Hawaii_1_FIPS_5101_Feet [4616] NAD_1983_StatePlane_Hawaii_2_FIPS_5102_Feet [4617] NAD_1983_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4618] NAD_1983_StatePlane_Hawaii_4_FIPS_5104_Feet [4619] NAD_1983_StatePlane_Hawaii_5_FIPS_5105_Feet [4620] NAD_1983_StatePlane_Idaho_Central_FIPS_1102_Feet (deprecated) [4621] NAD_1983_StatePlane_Idaho_East_FIPS_1101_Feet (deprecated) [4622] NAD_1983_StatePlane_Idaho_West_FIPS_1103_Feet (deprecated) [4623] NAD_1983_StatePlane_Illinois_East_FIPS_1201_Feet (deprecated) [4624] NAD_1983_StatePlane_Illinois_West_FIPS_1202_Feet (deprecated) [4625] NAD_1983_StatePlane_Indiana_East_FIPS_1301_Feet (deprecated) [4626] NAD_1983_StatePlane_Indiana_West_FIPS_1302_Feet (deprecated) [4627] NAD_1983_StatePlane_Iowa_North_FIPS_1401_Feet (deprecated) [4628] NAD_1983_StatePlane_Iowa_South_FIPS_1402_Feet (deprecated) [4629] NAD_1983_StatePlane_Kansas_North_FIPS_1501_Feet (deprecated) [4630] NAD_1983_StatePlane_Kansas_South_FIPS_1502_Feet (deprecated) [4631] NAD_1983_StatePlane_Kentucky_FIPS_1600 (deprecated) [4632] NAD_1983_StatePlane_Kentucky_FIPS_1600_Feet (deprecated) [4633] NAD_1983_StatePlane_Kentucky_North_FIPS_1601_Feet (deprecated) [4634] NAD_1983_StatePlane_Kentucky_South_FIPS_1602_Feet (deprecated) [4635] NAD_1983_StatePlane_Louisiana_North_FIPS_1701_Feet (deprecated) [4636] NAD_1983_StatePlane_Louisiana_South_FIPS_1702_Feet (deprecated) [4637] NAD_1983_StatePlane_Maine_East_FIPS_1801_Feet (deprecated) [4638] NAD_1983_StatePlane_Maine_West_FIPS_1802_Feet (deprecated) [4639] NAD_1983_StatePlane_Maryland_FIPS_1900_Feet (deprecated) [4640] NAD_1983_StatePlane_Massachusetts_Island_FIPS_2002_Feet (deprecated) [4641] NAD_1983_StatePlane_Massachusetts_Mainland_FIPS_2001_Feet (deprecated) [4642] NAD_1983_StatePlane_Michigan_Central_FIPS_2112_Feet [4643] NAD_1983_StatePlane_Michigan_North_FIPS_2111_Feet [4644] NAD_1983_StatePlane_Michigan_South_FIPS_2113_Feet [4645] NAD_1983_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4646] NAD_1983_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4647] NAD_1983_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4648] NAD_1983_StatePlane_Mississippi_East_FIPS_2301_Feet (deprecated) [4649] NAD_1983_StatePlane_Mississippi_West_FIPS_2302_Feet (deprecated) [4650] NAD_1983_StatePlane_Missouri_Central_FIPS_2402_Feet [4651] NAD_1983_StatePlane_Missouri_East_FIPS_2401_Feet [4652] NAD_1983_StatePlane_Missouri_West_FIPS_2403_Feet [4653] NAD_1983_StatePlane_Montana_FIPS_2500_Feet [4654] NAD_1983_StatePlane_Nebraska_FIPS_2600_Feet (deprecated) [4655] NAD_1983_StatePlane_Nevada_Central_FIPS_2702_Feet (deprecated) [4656] NAD_1983_StatePlane_Nevada_East_FIPS_2701_Feet (deprecated) [4657] NAD_1983_StatePlane_Nevada_West_FIPS_2703_Feet (deprecated) [4658] NAD_1983_StatePlane_New_Hampshire_FIPS_2800_Feet (deprecated) [4659] NAD_1983_StatePlane_New_Jersey_FIPS_2900_Feet (deprecated) [4660] NAD_1983_StatePlane_New_Mexico_Central_FIPS_3002_Feet (deprecated) [4661] NAD_1983_StatePlane_New_Mexico_East_FIPS_3001_Feet (deprecated) [4662] NAD_1983_StatePlane_New_Mexico_West_FIPS_3003_Feet (deprecated) [4663] NAD_1983_StatePlane_New_York_Central_FIPS_3102_Feet (deprecated) [4664] NAD_1983_StatePlane_New_York_East_FIPS_3101_Feet (deprecated) [4665] NAD_1983_StatePlane_New_York_Long_Island_FIPS_3104_Feet (deprecated) [4666] NAD_1983_StatePlane_New_York_West_FIPS_3103_Feet (deprecated) [4667] NAD_1983_StatePlane_North_Carolina_FIPS_3200_Feet (deprecated) [4668] NAD_1983_StatePlane_North_Dakota_North_FIPS_3301_Feet [4669] NAD_1983_StatePlane_North_Dakota_South_FIPS_3302_Feet [4670] NAD_1983_StatePlane_Ohio_North_FIPS_3401_Feet (deprecated) [4671] NAD_1983_StatePlane_Ohio_South_FIPS_3402_Feet (deprecated) [4672] NAD_1983_StatePlane_Oklahoma_North_FIPS_3501_Feet (deprecated) [4673] NAD_1983_StatePlane_Oklahoma_South_FIPS_3502_Feet (deprecated) [4674] NAD_1983_StatePlane_Oregon_North_FIPS_3601_Feet [4675] NAD_1983_StatePlane_Oregon_South_FIPS_3602_Feet [4676] NAD_1983_StatePlane_Pennsylvania_North_FIPS_3701_Feet (deprecated) [4677] NAD_1983_StatePlane_Pennsylvania_South_FIPS_3702_Feet (deprecated) [4678] NAD_1983_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200_Feet [4679] NAD_1983_StatePlane_Rhode_Island_FIPS_3800_Feet (deprecated) [4680] NAD_1983_StatePlane_South_Carolina_FIPS_3900_Feet [4681] NAD_1983_StatePlane_South_Dakota_North_FIPS_4001_Feet (deprecated) [4682] NAD_1983_StatePlane_South_Dakota_South_FIPS_4002_Feet (deprecated) [4683] NAD_1983_StatePlane_Tennessee_FIPS_4100_Feet (deprecated) [4684] NAD_1983_StatePlane_Texas_Central_FIPS_4203_Feet (deprecated) [4685] NAD_1983_StatePlane_Texas_North_Central_FIPS_4202_Feet (deprecated) [4686] NAD_1983_StatePlane_Texas_North_FIPS_4201_Feet (deprecated) [4687] NAD_1983_StatePlane_Texas_South_Central_FIPS_4204_Feet (deprecated) [4688] NAD_1983_StatePlane_Texas_South_FIPS_4205_Feet (deprecated) [4689] NAD_1983_StatePlane_Utah_Central_FIPS_4302_Feet (deprecated) [4690] NAD_1983_StatePlane_Utah_North_FIPS_4301_Feet (deprecated) [4691] NAD_1983_StatePlane_Utah_South_FIPS_4303_Feet (deprecated) [4692] NAD_1983_StatePlane_Vermont_FIPS_4400_Feet (deprecated) [4693] NAD_1983_StatePlane_Virginia_North_FIPS_4501_Feet (deprecated) [4694] NAD_1983_StatePlane_Virginia_South_FIPS_4502_Feet (deprecated) [4695] NAD_1983_StatePlane_Washington_North_FIPS_4601_Feet (deprecated) [4696] NAD_1983_StatePlane_Washington_South_FIPS_4602_Feet (deprecated) [4697] NAD_1983_StatePlane_West_Virginia_North_FIPS_4701_Feet (deprecated) [4698] NAD_1983_StatePlane_West_Virginia_South_FIPS_4702_Feet (deprecated) [4699] NAD_1983_StatePlane_Wisconsin_Central_FIPS_4802_Feet (deprecated) [4700] NAD_1983_StatePlane_Wisconsin_North_FIPS_4801_Feet (deprecated) [4701] NAD_1983_StatePlane_Wisconsin_South_FIPS_4803_Feet (deprecated) [4702] NAD_1983_StatePlane_Wyoming_East_Central_FIPS_4902_Feet (deprecated) [4703] NAD_1983_StatePlane_Wyoming_East_FIPS_4901_Feet (deprecated) [4704] NAD_1983_StatePlane_Wyoming_West_Central_FIPS_4903_Feet (deprecated) [4705] NAD_1983_StatePlane_Wyoming_West_FIPS_4904_Feet (deprecated) [4706] NAD_1983_Texas_Centric_Mapping_System_Albers (deprecated) [4707] NAD_1983_Texas_Centric_Mapping_System_Lambert (deprecated) [4708] NAD_1983_Texas_Statewide_Mapping_System (deprecated) [4709] NAD_1983_USFS_R6_Albers [4710] NAD_1983_USFS_R9_Albers [4711] NAD_1983_UTM_Zone_1N (deprecated) [4712] NAD_1983_UTM_Zone_2N (deprecated) [4713] NAD_1983_UTM_Zone_58N [4714] NAD_1983_UTM_Zone_59N (deprecated) [4715] NAD_1983_UTM_Zone_60N (deprecated) [4716] NAD_1983_Wisconsin_TM_US_Ft [4717] NAD_1983_WyLAM [4718] NEA74 Noumea / Noumea Lambert [4719] NEA74 Noumea / Noumea Lambert 2 [4720] NEA74 Noumea / UTM zone 58S [4721] NGN / UTM zone 38N [4722] NGN / UTM zone 39N [4723] NGO 1948 (Oslo) / NGO zone I [4724] NGO 1948 (Oslo) / NGO zone II [4725] NGO 1948 (Oslo) / NGO zone III [4726] NGO 1948 (Oslo) / NGO zone IV [4727] NGO 1948 (Oslo) / NGO zone V [4728] NGO 1948 (Oslo) / NGO zone VI [4729] NGO 1948 (Oslo) / NGO zone VII [4730] NGO 1948 (Oslo) / NGO zone VIII [4731] NGO_1948_Baerum_Kommune [4732] NGO_1948_Bergenhalvoen [4733] NGO_1948_Norway_Zone_1 [4734] NGO_1948_Norway_Zone_2 [4735] NGO_1948_Norway_Zone_3 [4736] NGO_1948_Norway_Zone_4 [4737] NGO_1948_Norway_Zone_5 [4738] NGO_1948_Norway_Zone_6 [4739] NGO_1948_Norway_Zone_7 [4740] NGO_1948_Norway_Zone_8 [4741] NGO_1948_Oslo_Baerum_Kommune [4742] NGO_1948_Oslo_Bergenhalvoen [4743] NGO_1948_Oslo_Kommune [4744] NGO_1948_Oslo_Oslo_Kommune [4745] NGO_1948_UTM_Zone_32N [4746] NGO_1948_UTM_Zone_33N [4747] NGO_1948_UTM_Zone_34N [4748] NGO_1948_UTM_Zone_35N [4749] NSIDC EASE-Grid Global [4750] NSIDC EASE-Grid North [4751] NSIDC EASE-Grid South [4752] NSIDC Sea Ice Polar Stereographic North [4753] NSIDC Sea Ice Polar Stereographic South [4754] NTF (Paris) / Centre France (deprecated) [4755] NTF (Paris) / Corse (deprecated) [4756] NTF (Paris) / France I (deprecated) [4757] NTF (Paris) / France II (deprecated) [4758] NTF (Paris) / France III (deprecated) [4759] NTF (Paris) / France IV (deprecated) [4760] NTF (Paris) / Lambert Centre France [4761] NTF (Paris) / Lambert Corse [4762] NTF (Paris) / Lambert Nord France [4763] NTF (Paris) / Lambert Sud France [4764] NTF (Paris) / Lambert zone I [4765] NTF (Paris) / Lambert zone II [4766] NTF (Paris) / Lambert zone III [4767] NTF (Paris) / Lambert zone IV [4768] NTF (Paris) / Nord France (deprecated) [4769] NTF (Paris) / Sud France (deprecated) [4770] NTF_France_III_degrees [4771] NTF_France_II_degrees [4772] NTF_France_IV_degrees [4773] NTF_France_I_degrees [4774] NTF_Lambert_Zone_I [4775] NTF_Lambert_Zone_II [4776] NTF_Lambert_Zone_III [4777] NTF_Lambert_Zone_IV [4778] NZGD2000 / Amuri 2000 [4779] NZGD2000 / Antipodes Islands TM 2000 [4780] NZGD2000 / Auckland Islands TM 2000 [4781] NZGD2000 / Bay of Plenty 2000 [4782] NZGD2000 / Bluff 2000 [4783] NZGD2000 / Buller 2000 [4784] NZGD2000 / Campbell Island TM 2000 [4785] NZGD2000 / Chatham Island Circuit 2000 [4786] NZGD2000 / Chatham Islands TM 2000 [4787] NZGD2000 / Collingwood 2000 [4788] NZGD2000 / Gawler 2000 [4789] NZGD2000 / Grey 2000 [4790] NZGD2000 / Hawkes Bay 2000 [4791] NZGD2000 / Hokitika 2000 [4792] NZGD2000 / Jacksons Bay 2000 [4793] NZGD2000 / Karamea 2000 [4794] NZGD2000 / Lindis Peak 2000 [4795] NZGD2000 / Marlborough 2000 [4796] NZGD2000 / Mount Eden 2000 [4797] NZGD2000 / Mount Nicholas 2000 [4798] NZGD2000 / Mount Pleasant 2000 [4799] NZGD2000 / Mount York 2000 [4800] NZGD2000 / NZCS2000 [4801] NZGD2000 / Nelson 2000 [4802] NZGD2000 / New Zealand Transverse Mercator 2000 [4803] NZGD2000 / North Taieri 2000 [4804] NZGD2000 / Observation Point 2000 [4805] NZGD2000 / Okarito 2000 [4806] NZGD2000 / Poverty Bay 2000 [4807] NZGD2000 / Raoul Island TM 2000 [4808] NZGD2000 / Taranaki 2000 [4809] NZGD2000 / Timaru 2000 [4810] NZGD2000 / Tuhirangi 2000 [4811] NZGD2000 / UTM zone 1S [4812] NZGD2000 / UTM zone 58S [4813] NZGD2000 / UTM zone 59S [4814] NZGD2000 / UTM zone 60S [4815] NZGD2000 / Wairarapa 2000 [4816] NZGD2000 / Wanganui 2000 [4817] NZGD2000 / Wellington 2000 [4818] NZGD49 / Amuri Circuit [4819] NZGD49 / Bay of Plenty Circuit [4820] NZGD49 / Bluff Circuit [4821] NZGD49 / Buller Circuit [4822] NZGD49 / Collingwood Circuit [4823] NZGD49 / Gawler Circuit [4824] NZGD49 / Grey Circuit [4825] NZGD49 / Hawkes Bay Circuit [4826] NZGD49 / Hokitika Circuit [4827] NZGD49 / Jacksons Bay Circuit [4828] NZGD49 / Karamea Circuit [4829] NZGD49 / Lindis Peak Circuit [4830] NZGD49 / Marlborough Circuit [4831] NZGD49 / Mount Eden Circuit [4832] NZGD49 / Mount Nicholas Circuit [4833] NZGD49 / Mount Pleasant Circuit [4834] NZGD49 / Mount York Circuit [4835] NZGD49 / Nelson Circuit [4836] NZGD49 / New Zealand Map Grid [4837] NZGD49 / North Island Grid [4838] NZGD49 / North Taieri Circuit [4839] NZGD49 / Observation Point Circuit [4840] NZGD49 / Okarito Circuit [4841] NZGD49 / Poverty Bay Circuit [4842] NZGD49 / South Island Grid [4843] NZGD49 / Taranaki Circuit [4844] NZGD49 / Timaru Circuit [4845] NZGD49 / Tuhirangi Circuit [4846] NZGD49 / UTM zone 58S [4847] NZGD49 / UTM zone 59S [4848] NZGD49 / UTM zone 60S [4849] NZGD49 / Wairarapa Circuit [4850] NZGD49 / Wanganui Circuit [4851] NZGD49 / Wellington Circuit [4852] NZGD_2000_Chatham_Island_Circuit (deprecated) [4853] Nahrwan 1934 / Iraq zone [4854] Nahrwan 1934 / UTM zone 37N [4855] Nahrwan 1934 / UTM zone 38N [4856] Nahrwan 1934 / UTM zone 39N [4857] Nahrwan 1967 / UTM zone 37N (deprecated) [4858] Nahrwan 1967 / UTM zone 38N (deprecated) [4859] Nahrwan 1967 / UTM zone 39N [4860] Nahrwan 1967 / UTM zone 40N [4861] Nakhl-e Ghanem / UTM zone 39N [4862] Naparima 1955 / UTM zone 20N [4863] Naparima 1972 / UTM zone 20N [4864] Nepal_Nagarkot_TM [4865] New Beijing / 3-degree Gauss-Kruger CM 102E [4866] New Beijing / 3-degree Gauss-Kruger CM 105E [4867] New Beijing / 3-degree Gauss-Kruger CM 108E [4868] New Beijing / 3-degree Gauss-Kruger CM 111E [4869] New Beijing / 3-degree Gauss-Kruger CM 114E [4870] New Beijing / 3-degree Gauss-Kruger CM 117E [4871] New Beijing / 3-degree Gauss-Kruger CM 120E [4872] New Beijing / 3-degree Gauss-Kruger CM 123E [4873] New Beijing / 3-degree Gauss-Kruger CM 126E [4874] New Beijing / 3-degree Gauss-Kruger CM 129E [4875] New Beijing / 3-degree Gauss-Kruger CM 132E [4876] New Beijing / 3-degree Gauss-Kruger CM 135E [4877] New Beijing / 3-degree Gauss-Kruger CM 75E [4878] New Beijing / 3-degree Gauss-Kruger CM 78E [4879] New Beijing / 3-degree Gauss-Kruger CM 81E [4880] New Beijing / 3-degree Gauss-Kruger CM 84E [4881] New Beijing / 3-degree Gauss-Kruger CM 87E [4882] New Beijing / 3-degree Gauss-Kruger CM 90E [4883] New Beijing / 3-degree Gauss-Kruger CM 93E [4884] New Beijing / 3-degree Gauss-Kruger CM 96E [4885] New Beijing / 3-degree Gauss-Kruger CM 99E [4886] New Beijing / 3-degree Gauss-Kruger zone 25 [4887] New Beijing / 3-degree Gauss-Kruger zone 26 [4888] New Beijing / 3-degree Gauss-Kruger zone 27 [4889] New Beijing / 3-degree Gauss-Kruger zone 28 [4890] New Beijing / 3-degree Gauss-Kruger zone 29 [4891] New Beijing / 3-degree Gauss-Kruger zone 30 [4892] New Beijing / 3-degree Gauss-Kruger zone 31 [4893] New Beijing / 3-degree Gauss-Kruger zone 32 [4894] New Beijing / 3-degree Gauss-Kruger zone 33 [4895] New Beijing / 3-degree Gauss-Kruger zone 34 [4896] New Beijing / 3-degree Gauss-Kruger zone 35 [4897] New Beijing / 3-degree Gauss-Kruger zone 36 [4898] New Beijing / 3-degree Gauss-Kruger zone 37 [4899] New Beijing / 3-degree Gauss-Kruger zone 38 [4900] New Beijing / 3-degree Gauss-Kruger zone 39 [4901] New Beijing / 3-degree Gauss-Kruger zone 40 [4902] New Beijing / 3-degree Gauss-Kruger zone 41 [4903] New Beijing / 3-degree Gauss-Kruger zone 42 [4904] New Beijing / 3-degree Gauss-Kruger zone 43 [4905] New Beijing / 3-degree Gauss-Kruger zone 44 [4906] New Beijing / 3-degree Gauss-Kruger zone 45 [4907] New Beijing / Gauss-Kruger CM 105E [4908] New Beijing / Gauss-Kruger CM 111E [4909] New Beijing / Gauss-Kruger CM 117E [4910] New Beijing / Gauss-Kruger CM 123E [4911] New Beijing / Gauss-Kruger CM 129E [4912] New Beijing / Gauss-Kruger CM 135E [4913] New Beijing / Gauss-Kruger CM 75E [4914] New Beijing / Gauss-Kruger CM 81E [4915] New Beijing / Gauss-Kruger CM 87E [4916] New Beijing / Gauss-Kruger CM 93E [4917] New Beijing / Gauss-Kruger CM 99E [4918] New Beijing / Gauss-Kruger zone 13 [4919] New Beijing / Gauss-Kruger zone 14 [4920] New Beijing / Gauss-Kruger zone 15 [4921] New Beijing / Gauss-Kruger zone 16 [4922] New Beijing / Gauss-Kruger zone 17 [4923] New Beijing / Gauss-Kruger zone 18 [4924] New Beijing / Gauss-Kruger zone 19 [4925] New Beijing / Gauss-Kruger zone 20 [4926] New Beijing / Gauss-Kruger zone 21 [4927] New Beijing / Gauss-Kruger zone 22 [4928] New Beijing / Gauss-Kruger zone 23 [4929] New_Beijing_3_Degree_Gauss_Kruger_CM_132E [4930] Nord Sahara 1959 / Nord Algerie [4931] Nord Sahara 1959 / Sud Algerie [4932] Nord Sahara 1959 / UTM zone 29N [4933] Nord Sahara 1959 / UTM zone 30N [4934] Nord Sahara 1959 / UTM zone 31N [4935] Nord Sahara 1959 / UTM zone 32N [4936] Nord_Algerie (deprecated) [4937] Nord_Algerie_Ancienne_Degree [4938] Nord_Algerie_Degree (deprecated) [4939] Nord_Maroc_Degree [4940] North_America_Albers_Equal_Area_Conic [4941] North_America_Equidistant_Conic [4942] North_America_Lambert_Conformal_Conic [4943] North_Pole_Azimuthal_Equidistant [4944] North_Pole_Gnomonic [4945] North_Pole_Lambert_Azimuthal_Equal_Area [4946] North_Pole_Orthographic [4947] North_Pole_Stereographic [4948] OCRS_Baker_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4949] OCRS_Baker_NAD_1983_CORS96_TM_Meters (deprecated) [4950] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4951] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Meters (deprecated) [4952] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4953] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Meters (deprecated) [4954] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4955] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Meters (deprecated) [4956] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4957] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4958] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4959] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Meters (deprecated) [4960] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4961] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Meters (deprecated) [4962] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4963] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Meters (deprecated) [4964] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4965] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Meters (deprecated) [4966] OCRS_Eugene_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4967] OCRS_Eugene_NAD_1983_CORS96_TM_Meters (deprecated) [4968] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4969] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Meters (deprecated) [4970] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4971] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Meters (deprecated) [4972] OCRS_La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4973] OCRS_La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4974] OCRS_Ontario_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4975] OCRS_Ontario_NAD_1983_CORS96_TM_Meters (deprecated) [4976] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4977] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Meters (deprecated) [4978] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4979] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4980] OCRS_Pendleton_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4981] OCRS_Pendleton_NAD_1983_CORS96_TM_Meters (deprecated) [4982] OCRS_Portland_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4983] OCRS_Portland_NAD_1983_CORS96_LCC_Meters (deprecated) [4984] OCRS_Salem_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4985] OCRS_Salem_NAD_1983_CORS96_TM_Meters (deprecated) [4986] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4987] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4988] ONGD14 / UTM zone 39N [4989] ONGD14 / UTM zone 40N [4990] ONGD14 / UTM zone 41N [4991] OSGB 1936 / British National Grid [4992] OSGB36_Highways_England_A10H1 [4993] OSGB36_Highways_England_A10H2 [4994] OSGB36_Highways_England_A11H1 [4995] OSGB36_Highways_England_A11H2 [4996] OSGB36_Highways_England_A11H3 [4997] OSGB36_Highways_England_A12H1 [4998] OSGB36_Highways_England_A12H2 [4999] OSGB36_Highways_England_A12H3 [5000] OSGB36_Highways_England_A13H1 [5001] OSGB36_Highways_England_A13H2 [5002] OSGB36_Highways_England_A14H1 [5003] OSGB36_Highways_England_A14H2 [5004] OSGB36_Highways_England_A15H1 [5005] OSGB36_Highways_England_A15H2 [5006] OSGB36_Highways_England_A16H1 [5007] OSGB36_Highways_England_A16H2 [5008] OSGB36_Highways_England_A17H1 [5009] OSGB36_Highways_England_A17H2 [5010] OSGB36_Highways_England_A18H1 [5011] OSGB36_Highways_England_A18H2 [5012] OSGB36_Highways_England_A19H1 [5013] OSGB36_Highways_England_A19H2 [5014] OSGB36_Highways_England_A1H1 [5015] OSGB36_Highways_England_A20H1 [5016] OSGB36_Highways_England_A20H2 [5017] OSGB36_Highways_England_A21H1 [5018] OSGB36_Highways_England_A21H2 [5019] OSGB36_Highways_England_A22H1 [5020] OSGB36_Highways_England_A22H2 [5021] OSGB36_Highways_England_A23H1 [5022] OSGB36_Highways_England_A23H2 [5023] OSGB36_Highways_England_A24H1 [5024] OSGB36_Highways_England_A24H2 [5025] OSGB36_Highways_England_A25H1 [5026] OSGB36_Highways_England_A25H2 [5027] OSGB36_Highways_England_A26H1 [5028] OSGB36_Highways_England_A26H2 [5029] OSGB36_Highways_England_A27H1 [5030] OSGB36_Highways_England_A27H2 [5031] OSGB36_Highways_England_A28H1 [5032] OSGB36_Highways_England_A28H2 [5033] OSGB36_Highways_England_A29H1 [5034] OSGB36_Highways_England_A2H1 [5035] OSGB36_Highways_England_A30H1 [5036] OSGB36_Highways_England_A3H1 [5037] OSGB36_Highways_England_A3H2 [5038] OSGB36_Highways_England_A4H1 [5039] OSGB36_Highways_England_A5H1 [5040] OSGB36_Highways_England_A5H2 [5041] OSGB36_Highways_England_A6H1 [5042] OSGB36_Highways_England_A6H2 [5043] OSGB36_Highways_England_A7H1 [5044] OSGB36_Highways_England_A7H2 [5045] OSGB36_Highways_England_A8H1 [5046] OSGB36_Highways_England_A8H2 [5047] OSGB36_Highways_England_A9H1 [5048] OSGB36_Highways_England_A9H2 [5049] OSGB36_Highways_England_B15H1 [5050] OSGB36_Highways_England_B15H2 [5051] OSGB36_Highways_England_B15H3 [5052] OSGB36_Highways_England_B16H1 [5053] OSGB36_Highways_England_B16H2 [5054] OSGB36_Highways_England_B16H3 [5055] OSGB36_Highways_England_B16H4 [5056] OSGB36_Highways_England_B17H1 [5057] OSGB36_Highways_England_B17H2 [5058] OSGB36_Highways_England_B18H1 [5059] OSGB36_Highways_England_B18H2 [5060] OSGB36_Highways_England_B19H1 [5061] OSGB36_Highways_England_B20H1 [5062] OSGB36_Highways_England_B21H1 [5063] OSGB36_Highways_England_B22H1 [5064] OSGB36_Highways_England_B23H1 [5065] OSGB36_Highways_England_B24H1 [5066] OSGB36_Highways_England_B25H1 [5067] OSGB36_Highways_England_B26H1 [5068] OSGB36_Highways_England_B27H1 [5069] OSGB36_Highways_England_B28H1 [5070] OSGB36_Highways_England_B29H1 [5071] OSGB36_Highways_England_B30H1 [5072] OSGB36_Highways_England_B31H1 [5073] OSGB36_Highways_England_B32H1 [5074] OSGB36_Highways_England_C13H1 [5075] OSGB36_Highways_England_C14H1 [5076] OSGB36_Highways_England_C14H2 [5077] OSGB36_Highways_England_C14H3 [5078] OSGB36_Highways_England_C14H4 [5079] OSGB36_Highways_England_C15H1 [5080] OSGB36_Highways_England_C15H2 [5081] OSGB36_Highways_England_C15H3 [5082] OSGB36_Highways_England_C15H4 [5083] OSGB36_Highways_England_C15H5 [5084] OSGB36_Highways_England_C16H1 [5085] OSGB36_Highways_England_C16H2 [5086] OSGB36_Highways_England_C16H3 [5087] OSGB36_Highways_England_C16H4 [5088] OSGB36_Highways_England_C17H1 [5089] OSGB36_Highways_England_C17H2 [5090] OSGB36_Highways_England_C18H1 [5091] OSGB36_Highways_England_C18H2 [5092] OSGB36_Highways_England_C19H1 [5093] OSGB36_Highways_England_C19H2 [5094] OSNI 1952 / Irish National Grid [5095] Observatorio_Meteorologico_1939_UTM_Zone_25N [5096] Observatorio_Meteorologico_1965_Macau_Grid [5097] Ocotepeque 1935 / Costa Rica Norte [5098] Ocotepeque 1935 / Costa Rica Sur [5099] Ocotepeque 1935 / El Salvador Lambert [5100] Ocotepeque 1935 / Guatemala Norte [5101] Ocotepeque 1935 / Guatemala Norte (deprecated) [5102] Ocotepeque 1935 / Guatemala Sur [5103] Ocotepeque 1935 / Nicaragua Norte [5104] Ocotepeque 1935 / Nicaragua Sur [5105] Ocotepeque_1935_Costa_Rica_Lambert_Norte [5106] Ocotepeque_1935_Costa_Rica_Lambert_Sur [5107] Old Hawaiian / Hawaii zone 1 [5108] Old Hawaiian / Hawaii zone 2 [5109] Old Hawaiian / Hawaii zone 3 [5110] Old Hawaiian / Hawaii zone 4 [5111] Old Hawaiian / Hawaii zone 5 [5112] Old_Hawaiian_UTM_Zone_4N [5113] Old_Hawaiian_UTM_Zone_5N [5114] PD/83 / 3-degree Gauss-Kruger zone 3 [5115] PD/83 / 3-degree Gauss-Kruger zone 3 (E-N) [5116] PD/83 / 3-degree Gauss-Kruger zone 4 [5117] PD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5118] PNG94 / PNGMG94 zone 54 [5119] PNG94 / PNGMG94 zone 55 [5120] PNG94 / PNGMG94 zone 56 [5121] POSGAR 2007 / Argentina 1 [5122] POSGAR 2007 / Argentina 2 [5123] POSGAR 2007 / Argentina 3 [5124] POSGAR 2007 / Argentina 4 [5125] POSGAR 2007 / Argentina 5 [5126] POSGAR 2007 / Argentina 6 [5127] POSGAR 2007 / Argentina 7 [5128] POSGAR 94 / Argentina 1 [5129] POSGAR 94 / Argentina 2 [5130] POSGAR 94 / Argentina 3 [5131] POSGAR 94 / Argentina 4 [5132] POSGAR 94 / Argentina 5 [5133] POSGAR 94 / Argentina 6 [5134] POSGAR 94 / Argentina 7 [5135] POSGAR 98 / Argentina 1 [5136] POSGAR 98 / Argentina 2 [5137] POSGAR 98 / Argentina 3 [5138] POSGAR 98 / Argentina 4 [5139] POSGAR 98 / Argentina 5 [5140] POSGAR 98 / Argentina 6 [5141] POSGAR 98 / Argentina 7 [5142] PRS92 / Philippines zone 1 [5143] PRS92 / Philippines zone 2 [5144] PRS92 / Philippines zone 3 [5145] PRS92 / Philippines zone 4 [5146] PRS92 / Philippines zone 5 [5147] PRS_1992_UTM_Zone_50N [5148] PRS_1992_UTM_Zone_51N [5149] PRS_1992_UTM_Zone_52N [5150] PSAD56 / ICN Regional [5151] PSAD56 / Peru central zone [5152] PSAD56 / Peru east zone [5153] PSAD56 / Peru west zone [5154] PSAD56 / UTM zone 17N [5155] PSAD56 / UTM zone 17S [5156] PSAD56 / UTM zone 18N [5157] PSAD56 / UTM zone 18S [5158] PSAD56 / UTM zone 19N [5159] PSAD56 / UTM zone 19S [5160] PSAD56 / UTM zone 20N [5161] PSAD56 / UTM zone 20S [5162] PSAD56 / UTM zone 21N [5163] PSAD56 / UTM zone 21S [5164] PSAD56 / UTM zone 22S [5165] PSD93 / UTM zone 39N [5166] PSD93 / UTM zone 40N [5167] PTRA08 / LAEA Europe [5168] PTRA08 / LCC Europe [5169] PTRA08 / UTM zone 25N [5170] PTRA08 / UTM zone 26N [5171] PTRA08 / UTM zone 28N [5172] PTRA08_UTM_Zone_25N (deprecated) [5173] PTRA08_UTM_Zone_26N (deprecated) [5174] PTRA08_UTM_Zone_28N (deprecated) [5175] Palestine 1923 / Israeli CS Grid [5176] Palestine 1923 / Palestine Belt [5177] Palestine 1923 / Palestine Grid [5178] Palestine 1923 / Palestine Grid modified [5179] Palestine_1923_Palestine_Grid_TM (deprecated) [5180] Pampa del Castillo / Argentina 2 [5181] Panama-Colon 1911 / Panama Lambert [5182] Panama-Colon 1911 / Panama Polyconic [5183] Panhandle_Energy_Albers [5184] Peru96 / UTM zone 17S [5185] Peru96 / UTM zone 17S (deprecated) [5186] Peru96 / UTM zone 18S [5187] Peru96 / UTM zone 19S [5188] Pitcairn 1967 / UTM zone 9S [5189] Pitcairn 2006 / Pitcairn TM 2006 [5190] Pohnpei_Az_Eq_1971 [5191] Pointe Noire / UTM zone 32S [5192] Popular Visualisation CRS / Mercator (deprecated) [5193] Popular Visualisation CRS / Mercator (deprecated) [5194] Porto Santo / UTM zone 28N [5195] Porto Santo 1995 / UTM zone 28N [5196] Porto_Santo_1936_UTM_Zone_28N (deprecated) [5197] Puerto Rico / St. Croix [5198] Puerto Rico / UTM zone 20N [5199] Puerto Rico State Plane CS of 1927 [5200] Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E [5201] Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E [5202] Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E [5203] Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E [5204] Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E [5205] Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E [5206] Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E [5207] Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E [5208] Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E [5209] Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E [5210] Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E [5211] Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E [5212] Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E [5213] Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E [5214] Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E [5215] Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E [5216] Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E [5217] Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E [5218] Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E [5219] Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E [5220] Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E [5221] Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E [5222] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E [5223] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W [5224] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E [5225] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W [5226] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E [5227] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W [5228] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E [5229] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W [5230] Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E [5231] Pulkovo 1942 / 3-degree Gauss-Kruger CM 18E (deprecated) [5232] Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E [5233] Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E [5234] Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E [5235] Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E [5236] Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E [5237] Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E [5238] Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E [5239] Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E [5240] Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E [5241] Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E [5242] Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E [5243] Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E [5244] Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E [5245] Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E [5246] Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E [5247] Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E [5248] Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E [5249] Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E [5250] Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E [5251] Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E [5252] Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E [5253] Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E [5254] Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E [5255] Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E [5256] Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E [5257] Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E [5258] Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E [5259] Pulkovo 1942 / 3-degree Gauss-Kruger zone 10 [5260] Pulkovo 1942 / 3-degree Gauss-Kruger zone 11 [5261] Pulkovo 1942 / 3-degree Gauss-Kruger zone 12 [5262] Pulkovo 1942 / 3-degree Gauss-Kruger zone 13 [5263] Pulkovo 1942 / 3-degree Gauss-Kruger zone 14 [5264] Pulkovo 1942 / 3-degree Gauss-Kruger zone 15 [5265] Pulkovo 1942 / 3-degree Gauss-Kruger zone 16 [5266] Pulkovo 1942 / 3-degree Gauss-Kruger zone 17 [5267] Pulkovo 1942 / 3-degree Gauss-Kruger zone 18 [5268] Pulkovo 1942 / 3-degree Gauss-Kruger zone 19 [5269] Pulkovo 1942 / 3-degree Gauss-Kruger zone 20 [5270] Pulkovo 1942 / 3-degree Gauss-Kruger zone 21 [5271] Pulkovo 1942 / 3-degree Gauss-Kruger zone 22 [5272] Pulkovo 1942 / 3-degree Gauss-Kruger zone 23 [5273] Pulkovo 1942 / 3-degree Gauss-Kruger zone 24 [5274] Pulkovo 1942 / 3-degree Gauss-Kruger zone 25 [5275] Pulkovo 1942 / 3-degree Gauss-Kruger zone 26 [5276] Pulkovo 1942 / 3-degree Gauss-Kruger zone 27 [5277] Pulkovo 1942 / 3-degree Gauss-Kruger zone 28 [5278] Pulkovo 1942 / 3-degree Gauss-Kruger zone 29 [5279] Pulkovo 1942 / 3-degree Gauss-Kruger zone 30 [5280] Pulkovo 1942 / 3-degree Gauss-Kruger zone 31 [5281] Pulkovo 1942 / 3-degree Gauss-Kruger zone 32 [5282] Pulkovo 1942 / 3-degree Gauss-Kruger zone 33 [5283] Pulkovo 1942 / 3-degree Gauss-Kruger zone 34 [5284] Pulkovo 1942 / 3-degree Gauss-Kruger zone 35 [5285] Pulkovo 1942 / 3-degree Gauss-Kruger zone 36 [5286] Pulkovo 1942 / 3-degree Gauss-Kruger zone 37 [5287] Pulkovo 1942 / 3-degree Gauss-Kruger zone 38 [5288] Pulkovo 1942 / 3-degree Gauss-Kruger zone 39 [5289] Pulkovo 1942 / 3-degree Gauss-Kruger zone 40 [5290] Pulkovo 1942 / 3-degree Gauss-Kruger zone 41 [5291] Pulkovo 1942 / 3-degree Gauss-Kruger zone 42 [5292] Pulkovo 1942 / 3-degree Gauss-Kruger zone 43 [5293] Pulkovo 1942 / 3-degree Gauss-Kruger zone 44 [5294] Pulkovo 1942 / 3-degree Gauss-Kruger zone 45 [5295] Pulkovo 1942 / 3-degree Gauss-Kruger zone 46 [5296] Pulkovo 1942 / 3-degree Gauss-Kruger zone 47 [5297] Pulkovo 1942 / 3-degree Gauss-Kruger zone 48 [5298] Pulkovo 1942 / 3-degree Gauss-Kruger zone 49 [5299] Pulkovo 1942 / 3-degree Gauss-Kruger zone 50 [5300] Pulkovo 1942 / 3-degree Gauss-Kruger zone 51 [5301] Pulkovo 1942 / 3-degree Gauss-Kruger zone 52 [5302] Pulkovo 1942 / 3-degree Gauss-Kruger zone 53 [5303] Pulkovo 1942 / 3-degree Gauss-Kruger zone 54 [5304] Pulkovo 1942 / 3-degree Gauss-Kruger zone 55 [5305] Pulkovo 1942 / 3-degree Gauss-Kruger zone 56 [5306] Pulkovo 1942 / 3-degree Gauss-Kruger zone 57 [5307] Pulkovo 1942 / 3-degree Gauss-Kruger zone 58 [5308] Pulkovo 1942 / 3-degree Gauss-Kruger zone 59 [5309] Pulkovo 1942 / 3-degree Gauss-Kruger zone 6 (deprecated) [5310] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 [5311] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 (deprecated) [5312] Pulkovo 1942 / 3-degree Gauss-Kruger zone 61 [5313] Pulkovo 1942 / 3-degree Gauss-Kruger zone 62 [5314] Pulkovo 1942 / 3-degree Gauss-Kruger zone 63 [5315] Pulkovo 1942 / 3-degree Gauss-Kruger zone 64 [5316] Pulkovo 1942 / 3-degree Gauss-Kruger zone 7 [5317] Pulkovo 1942 / 3-degree Gauss-Kruger zone 8 [5318] Pulkovo 1942 / 3-degree Gauss-Kruger zone 9 [5319] Pulkovo 1942 / CS63 zone A1 [5320] Pulkovo 1942 / CS63 zone A2 [5321] Pulkovo 1942 / CS63 zone A3 [5322] Pulkovo 1942 / CS63 zone A4 [5323] Pulkovo 1942 / CS63 zone C0 [5324] Pulkovo 1942 / CS63 zone C1 [5325] Pulkovo 1942 / CS63 zone C2 [5326] Pulkovo 1942 / CS63 zone K2 [5327] Pulkovo 1942 / CS63 zone K3 [5328] Pulkovo 1942 / CS63 zone K4 [5329] Pulkovo 1942 / CS63 zone X1 [5330] Pulkovo 1942 / CS63 zone X2 [5331] Pulkovo 1942 / CS63 zone X3 [5332] Pulkovo 1942 / CS63 zone X4 [5333] Pulkovo 1942 / CS63 zone X5 [5334] Pulkovo 1942 / CS63 zone X6 [5335] Pulkovo 1942 / CS63 zone X7 [5336] Pulkovo 1942 / Caspian Sea Mercator [5337] Pulkovo 1942 / Gauss-Kruger 10N (deprecated) [5338] Pulkovo 1942 / Gauss-Kruger 11N (deprecated) [5339] Pulkovo 1942 / Gauss-Kruger 12N (deprecated) [5340] Pulkovo 1942 / Gauss-Kruger 13N (deprecated) [5341] Pulkovo 1942 / Gauss-Kruger 14N (deprecated) [5342] Pulkovo 1942 / Gauss-Kruger 15N (deprecated) [5343] Pulkovo 1942 / Gauss-Kruger 16N (deprecated) [5344] Pulkovo 1942 / Gauss-Kruger 17N (deprecated) [5345] Pulkovo 1942 / Gauss-Kruger 18N (deprecated) [5346] Pulkovo 1942 / Gauss-Kruger 19N (deprecated) [5347] Pulkovo 1942 / Gauss-Kruger 20N (deprecated) [5348] Pulkovo 1942 / Gauss-Kruger 21N (deprecated) [5349] Pulkovo 1942 / Gauss-Kruger 22N (deprecated) [5350] Pulkovo 1942 / Gauss-Kruger 23N (deprecated) [5351] Pulkovo 1942 / Gauss-Kruger 24N (deprecated) [5352] Pulkovo 1942 / Gauss-Kruger 25N (deprecated) [5353] Pulkovo 1942 / Gauss-Kruger 26N (deprecated) [5354] Pulkovo 1942 / Gauss-Kruger 27N (deprecated) [5355] Pulkovo 1942 / Gauss-Kruger 28N (deprecated) [5356] Pulkovo 1942 / Gauss-Kruger 29N (deprecated) [5357] Pulkovo 1942 / Gauss-Kruger 2N (deprecated) [5358] Pulkovo 1942 / Gauss-Kruger 30N (deprecated) [5359] Pulkovo 1942 / Gauss-Kruger 31N (deprecated) [5360] Pulkovo 1942 / Gauss-Kruger 32N (deprecated) [5361] Pulkovo 1942 / Gauss-Kruger 3N (deprecated) [5362] Pulkovo 1942 / Gauss-Kruger 4N (deprecated) [5363] Pulkovo 1942 / Gauss-Kruger 5N (deprecated) [5364] Pulkovo 1942 / Gauss-Kruger 6N (deprecated) [5365] Pulkovo 1942 / Gauss-Kruger 7N (deprecated) [5366] Pulkovo 1942 / Gauss-Kruger 8N (deprecated) [5367] Pulkovo 1942 / Gauss-Kruger 9N (deprecated) [5368] Pulkovo 1942 / Gauss-Kruger CM 105E [5369] Pulkovo 1942 / Gauss-Kruger CM 111E [5370] Pulkovo 1942 / Gauss-Kruger CM 117E [5371] Pulkovo 1942 / Gauss-Kruger CM 123E [5372] Pulkovo 1942 / Gauss-Kruger CM 129E [5373] Pulkovo 1942 / Gauss-Kruger CM 135E [5374] Pulkovo 1942 / Gauss-Kruger CM 141E [5375] Pulkovo 1942 / Gauss-Kruger CM 147E [5376] Pulkovo 1942 / Gauss-Kruger CM 153E [5377] Pulkovo 1942 / Gauss-Kruger CM 159E [5378] Pulkovo 1942 / Gauss-Kruger CM 15E (deprecated) [5379] Pulkovo 1942 / Gauss-Kruger CM 165E [5380] Pulkovo 1942 / Gauss-Kruger CM 171E [5381] Pulkovo 1942 / Gauss-Kruger CM 171W [5382] Pulkovo 1942 / Gauss-Kruger CM 177E [5383] Pulkovo 1942 / Gauss-Kruger CM 177W [5384] Pulkovo 1942 / Gauss-Kruger CM 21E [5385] Pulkovo 1942 / Gauss-Kruger CM 27E [5386] Pulkovo 1942 / Gauss-Kruger CM 33E [5387] Pulkovo 1942 / Gauss-Kruger CM 39E [5388] Pulkovo 1942 / Gauss-Kruger CM 45E [5389] Pulkovo 1942 / Gauss-Kruger CM 51E [5390] Pulkovo 1942 / Gauss-Kruger CM 57E [5391] Pulkovo 1942 / Gauss-Kruger CM 63E [5392] Pulkovo 1942 / Gauss-Kruger CM 69E [5393] Pulkovo 1942 / Gauss-Kruger CM 75E [5394] Pulkovo 1942 / Gauss-Kruger CM 81E [5395] Pulkovo 1942 / Gauss-Kruger CM 87E [5396] Pulkovo 1942 / Gauss-Kruger CM 93E [5397] Pulkovo 1942 / Gauss-Kruger CM 99E [5398] Pulkovo 1942 / Gauss-Kruger CM 9E (deprecated) [5399] Pulkovo 1942 / Gauss-Kruger zone 10 [5400] Pulkovo 1942 / Gauss-Kruger zone 11 [5401] Pulkovo 1942 / Gauss-Kruger zone 12 [5402] Pulkovo 1942 / Gauss-Kruger zone 13 [5403] Pulkovo 1942 / Gauss-Kruger zone 14 [5404] Pulkovo 1942 / Gauss-Kruger zone 15 [5405] Pulkovo 1942 / Gauss-Kruger zone 16 [5406] Pulkovo 1942 / Gauss-Kruger zone 17 [5407] Pulkovo 1942 / Gauss-Kruger zone 18 [5408] Pulkovo 1942 / Gauss-Kruger zone 19 [5409] Pulkovo 1942 / Gauss-Kruger zone 2 (deprecated) [5410] Pulkovo 1942 / Gauss-Kruger zone 20 [5411] Pulkovo 1942 / Gauss-Kruger zone 21 [5412] Pulkovo 1942 / Gauss-Kruger zone 22 [5413] Pulkovo 1942 / Gauss-Kruger zone 23 [5414] Pulkovo 1942 / Gauss-Kruger zone 24 [5415] Pulkovo 1942 / Gauss-Kruger zone 25 [5416] Pulkovo 1942 / Gauss-Kruger zone 26 [5417] Pulkovo 1942 / Gauss-Kruger zone 27 [5418] Pulkovo 1942 / Gauss-Kruger zone 28 [5419] Pulkovo 1942 / Gauss-Kruger zone 29 [5420] Pulkovo 1942 / Gauss-Kruger zone 3 (deprecated) [5421] Pulkovo 1942 / Gauss-Kruger zone 30 [5422] Pulkovo 1942 / Gauss-Kruger zone 31 [5423] Pulkovo 1942 / Gauss-Kruger zone 32 [5424] Pulkovo 1942 / Gauss-Kruger zone 4 [5425] Pulkovo 1942 / Gauss-Kruger zone 5 [5426] Pulkovo 1942 / Gauss-Kruger zone 6 [5427] Pulkovo 1942 / Gauss-Kruger zone 7 [5428] Pulkovo 1942 / Gauss-Kruger zone 8 [5429] Pulkovo 1942 / Gauss-Kruger zone 9 [5430] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 10 [5431] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 [5432] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 (E-N) [5433] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 [5434] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 (E-N) [5435] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 [5436] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 (E-N) [5437] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 6 [5438] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 7 [5439] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 8 [5440] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 9 [5441] Pulkovo 1942(58) / GUGiK-80 [5442] Pulkovo 1942(58) / Gauss-Kruger zone 2 [5443] Pulkovo 1942(58) / Gauss-Kruger zone 2 (E-N) [5444] Pulkovo 1942(58) / Gauss-Kruger zone 3 [5445] Pulkovo 1942(58) / Gauss-Kruger zone 3 (E-N) [5446] Pulkovo 1942(58) / Gauss-Kruger zone 4 [5447] Pulkovo 1942(58) / Gauss-Kruger zone 5 [5448] Pulkovo 1942(58) / Poland zone I [5449] Pulkovo 1942(58) / Poland zone I (deprecated) [5450] Pulkovo 1942(58) / Poland zone II [5451] Pulkovo 1942(58) / Poland zone III [5452] Pulkovo 1942(58) / Poland zone IV [5453] Pulkovo 1942(58) / Poland zone V [5454] Pulkovo 1942(58) / Stereo70 [5455] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 [5456] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 (E-N) [5457] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 [5458] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 (E-N) [5459] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 [5460] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 (E-N) [5461] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 6 [5462] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 [5463] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 (deprecated) [5464] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 [5465] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 (deprecated) [5466] Pulkovo 1942(83) / Gauss Kruger zone 3 (deprecated) [5467] Pulkovo 1942(83) / Gauss Kruger zone 4 (deprecated) [5468] Pulkovo 1942(83) / Gauss Kruger zone 5 (deprecated) [5469] Pulkovo 1942(83) / Gauss-Kruger zone 2 [5470] Pulkovo 1942(83) / Gauss-Kruger zone 2 (E-N) [5471] Pulkovo 1942(83) / Gauss-Kruger zone 3 [5472] Pulkovo 1942(83) / Gauss-Kruger zone 3 (E-N) [5473] Pulkovo 1942(83) / Gauss-Kruger zone 4 [5474] Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E [5475] Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E [5476] Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E [5477] Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E [5478] Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E [5479] Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E [5480] Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E [5481] Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E [5482] Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E [5483] Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E [5484] Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E [5485] Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E [5486] Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E [5487] Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E [5488] Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E [5489] Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E [5490] Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E [5491] Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E [5492] Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E [5493] Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E [5494] Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E [5495] Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E [5496] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E [5497] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W [5498] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E [5499] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W [5500] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E [5501] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W [5502] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E [5503] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W [5504] Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E [5505] Pulkovo 1995 / 3-degree Gauss-Kruger CM 18E (deprecated) [5506] Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E [5507] Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E [5508] Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E [5509] Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E [5510] Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E [5511] Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E [5512] Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E [5513] Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E [5514] Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E [5515] Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E [5516] Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E [5517] Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E [5518] Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E [5519] Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E [5520] Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E [5521] Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E [5522] Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E [5523] Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E [5524] Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E [5525] Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E [5526] Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E [5527] Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E [5528] Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E [5529] Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E [5530] Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E [5531] Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E [5532] Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E [5533] Pulkovo 1995 / 3-degree Gauss-Kruger zone 10 [5534] Pulkovo 1995 / 3-degree Gauss-Kruger zone 11 [5535] Pulkovo 1995 / 3-degree Gauss-Kruger zone 12 [5536] Pulkovo 1995 / 3-degree Gauss-Kruger zone 13 [5537] Pulkovo 1995 / 3-degree Gauss-Kruger zone 14 [5538] Pulkovo 1995 / 3-degree Gauss-Kruger zone 15 [5539] Pulkovo 1995 / 3-degree Gauss-Kruger zone 16 [5540] Pulkovo 1995 / 3-degree Gauss-Kruger zone 17 [5541] Pulkovo 1995 / 3-degree Gauss-Kruger zone 18 [5542] Pulkovo 1995 / 3-degree Gauss-Kruger zone 19 [5543] Pulkovo 1995 / 3-degree Gauss-Kruger zone 20 [5544] Pulkovo 1995 / 3-degree Gauss-Kruger zone 21 [5545] Pulkovo 1995 / 3-degree Gauss-Kruger zone 22 [5546] Pulkovo 1995 / 3-degree Gauss-Kruger zone 23 [5547] Pulkovo 1995 / 3-degree Gauss-Kruger zone 24 [5548] Pulkovo 1995 / 3-degree Gauss-Kruger zone 25 [5549] Pulkovo 1995 / 3-degree Gauss-Kruger zone 26 [5550] Pulkovo 1995 / 3-degree Gauss-Kruger zone 27 [5551] Pulkovo 1995 / 3-degree Gauss-Kruger zone 28 [5552] Pulkovo 1995 / 3-degree Gauss-Kruger zone 29 [5553] Pulkovo 1995 / 3-degree Gauss-Kruger zone 30 [5554] Pulkovo 1995 / 3-degree Gauss-Kruger zone 31 [5555] Pulkovo 1995 / 3-degree Gauss-Kruger zone 32 [5556] Pulkovo 1995 / 3-degree Gauss-Kruger zone 33 [5557] Pulkovo 1995 / 3-degree Gauss-Kruger zone 34 [5558] Pulkovo 1995 / 3-degree Gauss-Kruger zone 35 [5559] Pulkovo 1995 / 3-degree Gauss-Kruger zone 36 [5560] Pulkovo 1995 / 3-degree Gauss-Kruger zone 37 [5561] Pulkovo 1995 / 3-degree Gauss-Kruger zone 38 [5562] Pulkovo 1995 / 3-degree Gauss-Kruger zone 39 [5563] Pulkovo 1995 / 3-degree Gauss-Kruger zone 40 [5564] Pulkovo 1995 / 3-degree Gauss-Kruger zone 41 [5565] Pulkovo 1995 / 3-degree Gauss-Kruger zone 42 [5566] Pulkovo 1995 / 3-degree Gauss-Kruger zone 43 [5567] Pulkovo 1995 / 3-degree Gauss-Kruger zone 44 [5568] Pulkovo 1995 / 3-degree Gauss-Kruger zone 45 [5569] Pulkovo 1995 / 3-degree Gauss-Kruger zone 46 [5570] Pulkovo 1995 / 3-degree Gauss-Kruger zone 47 [5571] Pulkovo 1995 / 3-degree Gauss-Kruger zone 48 [5572] Pulkovo 1995 / 3-degree Gauss-Kruger zone 49 [5573] Pulkovo 1995 / 3-degree Gauss-Kruger zone 50 [5574] Pulkovo 1995 / 3-degree Gauss-Kruger zone 51 [5575] Pulkovo 1995 / 3-degree Gauss-Kruger zone 52 [5576] Pulkovo 1995 / 3-degree Gauss-Kruger zone 53 [5577] Pulkovo 1995 / 3-degree Gauss-Kruger zone 54 [5578] Pulkovo 1995 / 3-degree Gauss-Kruger zone 55 [5579] Pulkovo 1995 / 3-degree Gauss-Kruger zone 56 [5580] Pulkovo 1995 / 3-degree Gauss-Kruger zone 57 [5581] Pulkovo 1995 / 3-degree Gauss-Kruger zone 58 [5582] Pulkovo 1995 / 3-degree Gauss-Kruger zone 59 [5583] Pulkovo 1995 / 3-degree Gauss-Kruger zone 6 (deprecated) [5584] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 [5585] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 (deprecated) [5586] Pulkovo 1995 / 3-degree Gauss-Kruger zone 61 [5587] Pulkovo 1995 / 3-degree Gauss-Kruger zone 62 [5588] Pulkovo 1995 / 3-degree Gauss-Kruger zone 63 [5589] Pulkovo 1995 / 3-degree Gauss-Kruger zone 64 [5590] Pulkovo 1995 / 3-degree Gauss-Kruger zone 7 [5591] Pulkovo 1995 / 3-degree Gauss-Kruger zone 8 [5592] Pulkovo 1995 / 3-degree Gauss-Kruger zone 9 [5593] Pulkovo 1995 / Gauss-Kruger 10N (deprecated) [5594] Pulkovo 1995 / Gauss-Kruger 11N (deprecated) [5595] Pulkovo 1995 / Gauss-Kruger 12N (deprecated) [5596] Pulkovo 1995 / Gauss-Kruger 13N (deprecated) [5597] Pulkovo 1995 / Gauss-Kruger 14N (deprecated) [5598] Pulkovo 1995 / Gauss-Kruger 15N (deprecated) [5599] Pulkovo 1995 / Gauss-Kruger 16N (deprecated) [5600] Pulkovo 1995 / Gauss-Kruger 17N (deprecated) [5601] Pulkovo 1995 / Gauss-Kruger 18N (deprecated) [5602] Pulkovo 1995 / Gauss-Kruger 19N (deprecated) [5603] Pulkovo 1995 / Gauss-Kruger 20N (deprecated) [5604] Pulkovo 1995 / Gauss-Kruger 21N (deprecated) [5605] Pulkovo 1995 / Gauss-Kruger 22N (deprecated) [5606] Pulkovo 1995 / Gauss-Kruger 23N (deprecated) [5607] Pulkovo 1995 / Gauss-Kruger 24N (deprecated) [5608] Pulkovo 1995 / Gauss-Kruger 25N (deprecated) [5609] Pulkovo 1995 / Gauss-Kruger 26N (deprecated) [5610] Pulkovo 1995 / Gauss-Kruger 27N (deprecated) [5611] Pulkovo 1995 / Gauss-Kruger 28N (deprecated) [5612] Pulkovo 1995 / Gauss-Kruger 29N (deprecated) [5613] Pulkovo 1995 / Gauss-Kruger 30N (deprecated) [5614] Pulkovo 1995 / Gauss-Kruger 31N (deprecated) [5615] Pulkovo 1995 / Gauss-Kruger 32N (deprecated) [5616] Pulkovo 1995 / Gauss-Kruger 4N (deprecated) [5617] Pulkovo 1995 / Gauss-Kruger 5N (deprecated) [5618] Pulkovo 1995 / Gauss-Kruger 6N (deprecated) [5619] Pulkovo 1995 / Gauss-Kruger 7N (deprecated) [5620] Pulkovo 1995 / Gauss-Kruger 8N (deprecated) [5621] Pulkovo 1995 / Gauss-Kruger 9N (deprecated) [5622] Pulkovo 1995 / Gauss-Kruger CM 105E [5623] Pulkovo 1995 / Gauss-Kruger CM 111E [5624] Pulkovo 1995 / Gauss-Kruger CM 117E [5625] Pulkovo 1995 / Gauss-Kruger CM 123E [5626] Pulkovo 1995 / Gauss-Kruger CM 129E [5627] Pulkovo 1995 / Gauss-Kruger CM 135E [5628] Pulkovo 1995 / Gauss-Kruger CM 141E [5629] Pulkovo 1995 / Gauss-Kruger CM 147E [5630] Pulkovo 1995 / Gauss-Kruger CM 153E [5631] Pulkovo 1995 / Gauss-Kruger CM 159E [5632] Pulkovo 1995 / Gauss-Kruger CM 165E [5633] Pulkovo 1995 / Gauss-Kruger CM 171E [5634] Pulkovo 1995 / Gauss-Kruger CM 171W [5635] Pulkovo 1995 / Gauss-Kruger CM 177E [5636] Pulkovo 1995 / Gauss-Kruger CM 177W [5637] Pulkovo 1995 / Gauss-Kruger CM 21E [5638] Pulkovo 1995 / Gauss-Kruger CM 27E [5639] Pulkovo 1995 / Gauss-Kruger CM 33E [5640] Pulkovo 1995 / Gauss-Kruger CM 39E [5641] Pulkovo 1995 / Gauss-Kruger CM 45E [5642] Pulkovo 1995 / Gauss-Kruger CM 51E [5643] Pulkovo 1995 / Gauss-Kruger CM 57E [5644] Pulkovo 1995 / Gauss-Kruger CM 63E [5645] Pulkovo 1995 / Gauss-Kruger CM 69E [5646] Pulkovo 1995 / Gauss-Kruger CM 75E [5647] Pulkovo 1995 / Gauss-Kruger CM 81E [5648] Pulkovo 1995 / Gauss-Kruger CM 87E [5649] Pulkovo 1995 / Gauss-Kruger CM 93E [5650] Pulkovo 1995 / Gauss-Kruger CM 99E [5651] Pulkovo 1995 / Gauss-Kruger zone 10 [5652] Pulkovo 1995 / Gauss-Kruger zone 11 [5653] Pulkovo 1995 / Gauss-Kruger zone 12 [5654] Pulkovo 1995 / Gauss-Kruger zone 13 [5655] Pulkovo 1995 / Gauss-Kruger zone 14 [5656] Pulkovo 1995 / Gauss-Kruger zone 15 [5657] Pulkovo 1995 / Gauss-Kruger zone 16 [5658] Pulkovo 1995 / Gauss-Kruger zone 17 [5659] Pulkovo 1995 / Gauss-Kruger zone 18 [5660] Pulkovo 1995 / Gauss-Kruger zone 19 [5661] Pulkovo 1995 / Gauss-Kruger zone 20 [5662] Pulkovo 1995 / Gauss-Kruger zone 21 [5663] Pulkovo 1995 / Gauss-Kruger zone 22 [5664] Pulkovo 1995 / Gauss-Kruger zone 23 [5665] Pulkovo 1995 / Gauss-Kruger zone 24 [5666] Pulkovo 1995 / Gauss-Kruger zone 25 [5667] Pulkovo 1995 / Gauss-Kruger zone 26 [5668] Pulkovo 1995 / Gauss-Kruger zone 27 [5669] Pulkovo 1995 / Gauss-Kruger zone 28 [5670] Pulkovo 1995 / Gauss-Kruger zone 29 [5671] Pulkovo 1995 / Gauss-Kruger zone 30 [5672] Pulkovo 1995 / Gauss-Kruger zone 31 [5673] Pulkovo 1995 / Gauss-Kruger zone 32 [5674] Pulkovo 1995 / Gauss-Kruger zone 4 [5675] Pulkovo 1995 / Gauss-Kruger zone 5 [5676] Pulkovo 1995 / Gauss-Kruger zone 6 [5677] Pulkovo 1995 / Gauss-Kruger zone 7 [5678] Pulkovo 1995 / Gauss-Kruger zone 8 [5679] Pulkovo 1995 / Gauss-Kruger zone 9 [5680] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_7 (deprecated) [5681] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_8 (deprecated) [5682] Pulkovo_1995_GK_Zone_2 [5683] Pulkovo_1995_GK_Zone_2N [5684] Pulkovo_1995_GK_Zone_3 [5685] Pulkovo_1995_GK_Zone_3N [5686] QND95 / Qatar National Grid [5687] QND_1995_UTM_39N [5688] Qatar 1948 / Qatar Grid [5689] Qatar 1974 / Qatar National Grid [5690] Qornoq 1927 / UTM zone 22N [5691] Qornoq 1927 / UTM zone 23N [5692] RD/83 / 3-degree Gauss-Kruger zone 4 [5693] RD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5694] RD/83 / 3-degree Gauss-Kruger zone 5 [5695] RD/83 / 3-degree Gauss-Kruger zone 5 (E-N) [5696] RDN2008 / Italy zone (E-N) [5697] RDN2008 / Italy zone (N-E) [5698] RDN2008 / UTM zone 32N [5699] RDN2008 / UTM zone 32N (N-E) [5700] RDN2008 / UTM zone 33N [5701] RDN2008 / UTM zone 33N (N-E) [5702] RDN2008 / UTM zone 34N [5703] RDN2008 / UTM zone 34N (N-E) [5704] RDN2008 / Zone 12 (E-N) [5705] RDN2008 / Zone 12 (N-E) [5706] REGCAN95 / LAEA Europe [5707] REGCAN95 / LCC Europe [5708] REGCAN95 / UTM zone 27N [5709] REGCAN95 / UTM zone 28N [5710] REGCAN95_UTM_Zone_27N (deprecated) [5711] REGCAN95_UTM_Zone_28N (deprecated) [5712] REGVEN / UTM zone 18N [5713] REGVEN / UTM zone 19N [5714] REGVEN / UTM zone 20N [5715] RGAF09 / UTM zone 20N [5716] RGF93 / CC42 [5717] RGF93 / CC43 [5718] RGF93 / CC44 [5719] RGF93 / CC45 [5720] RGF93 / CC46 [5721] RGF93 / CC47 [5722] RGF93 / CC48 [5723] RGF93 / CC49 [5724] RGF93 / CC50 [5725] RGF93 / Lambert-93 [5726] RGFG95 / UTM zone 21N [5727] RGFG95 / UTM zone 22N [5728] RGF_1993_Lambert_93 (deprecated) [5729] RGM04 / UTM zone 38S [5730] RGNC 1991 / Lambert New Caledonia (deprecated) [5731] RGNC91-93 / Lambert New Caledonia [5732] RGNC91-93 / UTM zone 57S [5733] RGNC91-93 / UTM zone 58S [5734] RGNC91-93 / UTM zone 59S [5735] RGPF / UTM zone 5S [5736] RGPF / UTM zone 6S [5737] RGPF / UTM zone 7S [5738] RGPF / UTM zone 8S [5739] RGR92 / UTM zone 39S [5740] RGR92 / UTM zone 40S [5741] RGRDC 2005 / Congo TM zone 12 [5742] RGRDC 2005 / Congo TM zone 14 [5743] RGRDC 2005 / Congo TM zone 16 [5744] RGRDC 2005 / Congo TM zone 18 [5745] RGRDC 2005 / Congo TM zone 20 [5746] RGRDC 2005 / Congo TM zone 22 [5747] RGRDC 2005 / Congo TM zone 24 [5748] RGRDC 2005 / Congo TM zone 26 [5749] RGRDC 2005 / Congo TM zone 28 [5750] RGRDC 2005 / Congo TM zone 30 [5751] RGRDC 2005 / UTM zone 33S [5752] RGRDC 2005 / UTM zone 34S [5753] RGRDC 2005 / UTM zone 35S [5754] RGRDC_2005_Congo_TM_Zone_12 (deprecated) [5755] RGRDC_2005_Congo_TM_Zone_14 (deprecated) [5756] RGRDC_2005_Congo_TM_Zone_16 (deprecated) [5757] RGRDC_2005_Congo_TM_Zone_18 (deprecated) [5758] RGRDC_2005_Congo_TM_Zone_20 (deprecated) [5759] RGRDC_2005_Congo_TM_Zone_22 (deprecated) [5760] RGRDC_2005_Congo_TM_Zone_24 (deprecated) [5761] RGRDC_2005_Congo_TM_Zone_26 (deprecated) [5762] RGRDC_2005_Congo_TM_Zone_28 (deprecated) [5763] RGRDC_2005_UTM_Zone_33S (deprecated) [5764] RGRDC_2005_UTM_Zone_34S (deprecated) [5765] RGRDC_2005_UTM_Zone_35S (deprecated) [5766] RGSPM06 / UTM zone 21N [5767] RGTAAF07 / UTM zone 37S [5768] RGTAAF07 / UTM zone 38S [5769] RGTAAF07 / UTM zone 39S [5770] RGTAAF07 / UTM zone 40S [5771] RGTAAF07 / UTM zone 41S [5772] RGTAAF07 / UTM zone 42S [5773] RGTAAF07 / UTM zone 43S [5774] RGTAAF07 / UTM zone 44S [5775] RGTAAF07 / UTM zone 53S [5776] RGTAAF07 / UTM zone 54S [5777] RGWF96 / UTM zone 1S [5778] RRAF 1991 / UTM zone 20N [5779] RRAF 1991 / UTM zone 20N (deprecated) [5780] RSRGD2000 / BCLC2000 [5781] RSRGD2000 / DGLC2000 [5782] RSRGD2000 / MSLC2000 [5783] RSRGD2000 / PCLC2000 [5784] RSRGD2000 / RSPS2000 [5785] RT38 0 gon [5786] RT38 2.5 gon O [5787] RT38 2.5 gon V [5788] RT38 2.5 gon W (deprecated) [5789] RT38 5 gon O [5790] RT38 5 gon V [5791] RT38 7.5 gon V [5792] RT90 0 gon [5793] RT90 2.5 gon O [5794] RT90 2.5 gon V [5795] RT90 2.5 gon W (deprecated) [5796] RT90 5 gon O [5797] RT90 5 gon V [5798] RT90 7.5 gon V [5799] Rassadiran / Nakhl e Taqi [5800] Reunion 1947 / TM Reunion [5801] Reunion 1947 / TM Reunion (deprecated) [5802] Roma_1940_Gauss_Boaga_Est [5803] Roma_1940_Gauss_Boaga_Ovest [5804] S-JTSK (Ferro) / Krovak [5805] S-JTSK (Ferro) / Krovak East North [5806] S-JTSK / Krovak [5807] S-JTSK / Krovak East North [5808] S-JTSK [JTSK03] / Krovak [5809] S-JTSK [JTSK03] / Krovak East North [5810] S-JTSK_Ferro_Krovak_East_North (deprecated) [5811] S-JTSK_Krovak (deprecated) [5812] S-JTSK_Krovak_East_North (deprecated) [5813] SAD69 / Brazil Polyconic [5814] SAD69 / Brazil Polyconic (deprecated) [5815] SAD69 / UTM zone 17N [5816] SAD69 / UTM zone 17S [5817] SAD69 / UTM zone 17S (deprecated) [5818] SAD69 / UTM zone 18N [5819] SAD69 / UTM zone 18N (deprecated) [5820] SAD69 / UTM zone 18S [5821] SAD69 / UTM zone 18S (deprecated) [5822] SAD69 / UTM zone 19N [5823] SAD69 / UTM zone 19N (deprecated) [5824] SAD69 / UTM zone 19S [5825] SAD69 / UTM zone 19S (deprecated) [5826] SAD69 / UTM zone 20N [5827] SAD69 / UTM zone 20N (deprecated) [5828] SAD69 / UTM zone 20S [5829] SAD69 / UTM zone 20S (deprecated) [5830] SAD69 / UTM zone 21N [5831] SAD69 / UTM zone 21N (deprecated) [5832] SAD69 / UTM zone 21S [5833] SAD69 / UTM zone 21S (deprecated) [5834] SAD69 / UTM zone 22N [5835] SAD69 / UTM zone 22N (deprecated) [5836] SAD69 / UTM zone 22S [5837] SAD69 / UTM zone 22S (deprecated) [5838] SAD69 / UTM zone 23S [5839] SAD69 / UTM zone 23S (deprecated) [5840] SAD69 / UTM zone 24S [5841] SAD69 / UTM zone 24S (deprecated) [5842] SAD69 / UTM zone 25S [5843] SAD69 / UTM zone 25S (deprecated) [5844] SAD69(96) / Brazil Polyconic [5845] SAD69(96) / UTM zone 18S [5846] SAD69(96) / UTM zone 19S [5847] SAD69(96) / UTM zone 20S [5848] SAD69(96) / UTM zone 21S [5849] SAD69(96) / UTM zone 22S [5850] SAD69(96) / UTM zone 22S (deprecated) [5851] SAD69(96) / UTM zone 23S [5852] SAD69(96) / UTM zone 24S [5853] SAD69(96) / UTM zone 25S [5854] SHMG2015 [5855] SIRGAS 1995 / UTM zone 17N [5856] SIRGAS 1995 / UTM zone 17S [5857] SIRGAS 1995 / UTM zone 18N [5858] SIRGAS 1995 / UTM zone 18S [5859] SIRGAS 1995 / UTM zone 19N [5860] SIRGAS 1995 / UTM zone 19S [5861] SIRGAS 1995 / UTM zone 20N [5862] SIRGAS 1995 / UTM zone 20S [5863] SIRGAS 1995 / UTM zone 21N [5864] SIRGAS 1995 / UTM zone 21S [5865] SIRGAS 1995 / UTM zone 22N [5866] SIRGAS 1995 / UTM zone 22S [5867] SIRGAS 1995 / UTM zone 23S [5868] SIRGAS 1995 / UTM zone 24S [5869] SIRGAS 1995 / UTM zone 25S [5870] SIRGAS 2000 / Brazil Mercator [5871] SIRGAS 2000 / Brazil Polyconic [5872] SIRGAS 2000 / UTM zone 11N [5873] SIRGAS 2000 / UTM zone 12N [5874] SIRGAS 2000 / UTM zone 13N [5875] SIRGAS 2000 / UTM zone 14N [5876] SIRGAS 2000 / UTM zone 15N [5877] SIRGAS 2000 / UTM zone 16N [5878] SIRGAS 2000 / UTM zone 17N [5879] SIRGAS 2000 / UTM zone 17S [5880] SIRGAS 2000 / UTM zone 18N [5881] SIRGAS 2000 / UTM zone 18S [5882] SIRGAS 2000 / UTM zone 19N [5883] SIRGAS 2000 / UTM zone 19S [5884] SIRGAS 2000 / UTM zone 20N [5885] SIRGAS 2000 / UTM zone 20S [5886] SIRGAS 2000 / UTM zone 21N [5887] SIRGAS 2000 / UTM zone 21S [5888] SIRGAS 2000 / UTM zone 22N [5889] SIRGAS 2000 / UTM zone 22S [5890] SIRGAS 2000 / UTM zone 23N [5891] SIRGAS 2000 / UTM zone 23S [5892] SIRGAS 2000 / UTM zone 24N [5893] SIRGAS 2000 / UTM zone 24S [5894] SIRGAS 2000 / UTM zone 25S [5895] SIRGAS 2000 / UTM zone 26S [5896] SIRGAS-Chile / UTM zone 18S [5897] SIRGAS-Chile / UTM zone 19S [5898] SIRGAS-ROU98 / UTM zone 21S [5899] SIRGAS-ROU98 / UTM zone 22S [5900] SIRGAS_UTM_Zone_17N (deprecated) [5901] SIRGAS_UTM_Zone_18N (deprecated) [5902] SIRGAS_UTM_Zone_19N (deprecated) [5903] SIRGAS_UTM_Zone_20N (deprecated) [5904] SIRGAS_UTM_Zone_21N (deprecated) [5905] SIRGAS_UTM_Zone_22N (deprecated) [5906] SLD99 / Sri Lanka Grid 1999 [5907] SRB_ETRS89 / UTM zone 34N [5908] ST71 Belep / UTM zone 58S [5909] ST74 [5910] ST84 Ile des Pins / UTM zone 58S [5911] ST87 Ouvea / UTM zone 58S [5912] ST87 Ouvea / UTM zone 58S (deprecated) [5913] SVY21 / Singapore TM [5914] SWEREF99 / RT90 0 gon emulation [5915] SWEREF99 / RT90 2.5 gon O emulation [5916] SWEREF99 / RT90 2.5 gon V emulation [5917] SWEREF99 / RT90 5 gon O emulation [5918] SWEREF99 / RT90 5 gon V emulation [5919] SWEREF99 / RT90 7.5 gon V emulation [5920] SWEREF99 12 00 [5921] SWEREF99 13 30 [5922] SWEREF99 14 15 [5923] SWEREF99 15 00 [5924] SWEREF99 15 45 [5925] SWEREF99 16 30 [5926] SWEREF99 17 15 [5927] SWEREF99 18 00 [5928] SWEREF99 18 45 [5929] SWEREF99 20 15 [5930] SWEREF99 21 45 [5931] SWEREF99 23 15 [5932] SWEREF99 TM [5933] Sahara_Degree [5934] Saint Pierre et Miquelon 1950 / UTM zone 21N [5935] Saipan_Az_Eq_1969 [5936] Samboja / UTM zone 50S (deprecated) [5937] Sao_Braz_UTM_Zone_26N [5938] Sapper Hill 1943 / UTM zone 20S [5939] Sapper Hill 1943 / UTM zone 21S [5940] Schwarzeck / Lo22/11 [5941] Schwarzeck / Lo22/13 [5942] Schwarzeck / Lo22/15 [5943] Schwarzeck / Lo22/17 [5944] Schwarzeck / Lo22/19 [5945] Schwarzeck / Lo22/21 [5946] Schwarzeck / Lo22/23 [5947] Schwarzeck / Lo22/25 [5948] Schwarzeck / UTM zone 33S [5949] Segara (Jakarta) / NEIEZ [5950] Segara (Jakarta) / NEIEZ (deprecated) [5951] Segara / NEIEZ [5952] Segara / UTM zone 50S [5953] Selvagem Grande / UTM zone 28N [5954] Selvagem_Grande_1938_UTM_Zone_28N (deprecated) [5955] Sibun Gorge 1922 / Colony Grid [5956] Sibun Gorge 1922 / Colony Grid (deprecated) [5957] Sierra Leone 1924 / New Colony Grid [5958] Sierra Leone 1924 / New War Office Grid [5959] Sierra Leone 1968 / UTM zone 28N [5960] Sierra Leone 1968 / UTM zone 29N [5961] Sister Islands National Grid 1961 [5962] Slovenia 1996 / Slovene National Grid [5963] Slovenia 1996 / UTM zone 33N [5964] South East Island 1943 / UTM zone 40N [5965] South Yemen / Gauss Kruger zone 8 (deprecated) [5966] South Yemen / Gauss Kruger zone 9 (deprecated) [5967] South Yemen / Gauss-Kruger zone 8 [5968] South Yemen / Gauss-Kruger zone 9 [5969] South_America_Albers_Equal_Area_Conic [5970] South_America_Equidistant_Conic [5971] South_America_Lambert_Conformal_Conic [5972] South_Pole_Azimuthal_Equidistant [5973] South_Pole_Gnomonic [5974] South_Pole_Lambert_Azimuthal_Equal_Area [5975] South_Pole_Orthographic [5976] South_Pole_Stereographic [5977] Sphere_Aitoff [5978] Sphere_Azimuthal_Equidistant [5979] Sphere_Bonne [5980] Sphere_Cassini [5981] Sphere_Craster_Parabolic [5982] Sphere_Cylindrical_Equal_Area [5983] Sphere_Eckert_I [5984] Sphere_Eckert_II [5985] Sphere_Eckert_III [5986] Sphere_Eckert_IV [5987] Sphere_Eckert_V [5988] Sphere_Eckert_VI [5989] Sphere_Equidistant_Conic [5990] Sphere_Equidistant_Cylindrical [5991] Sphere_Gall_Stereographic [5992] Sphere_Hotine [5993] Sphere_Loximuthal [5994] Sphere_Mercator [5995] Sphere_Miller_Cylindrical [5996] Sphere_Mollweide [5997] Sphere_Plate_Carree [5998] Sphere_Polyconic [5999] Sphere_Quartic_Authalic [6000] Sphere_Robinson [6001] Sphere_Sinusoidal [6002] Sphere_Stereographic [6003] Sphere_Two_Point_Equidistant [6004] Sphere_Van_der_Grinten_I [6005] Sphere_Wagner_IV [6006] Sphere_Wagner_V [6007] Sphere_Wagner_VII [6008] Sphere_Winkel_I [6009] Sphere_Winkel_II [6010] Sphere_Winkel_Tripel_NGS [6011] St. Helena Tritan / SHLG(Tritan) [6012] St. Helena Tritan / UTM zone 30S [6013] St. Kitts 1955 / British West Indies Grid [6014] St. Lucia 1955 / British West Indies Grid [6015] St. Stephen Grid (Ferro) [6016] St. Vincent 45 / British West Indies Grid [6017] Sud_Algerie (deprecated) [6018] Sud_Algerie_Ancienne_Degree [6019] Sud_Algerie_Degree (deprecated) [6020] Sud_Maroc_Degree [6021] Sudan / UTM zone 35N (deprecated) [6022] Sudan / UTM zone 36N (deprecated) [6023] TC(1948) / UTM zone 39N [6024] TC(1948) / UTM zone 40N [6025] TGD2005 / Tonga Map Grid [6026] TM65 / Irish Grid [6027] TM65 / Irish National Grid (deprecated) [6028] TM75 / Irish Grid [6029] TUREF / 3-degree Gauss-Kruger zone 10 [6030] TUREF / 3-degree Gauss-Kruger zone 11 [6031] TUREF / 3-degree Gauss-Kruger zone 12 [6032] TUREF / 3-degree Gauss-Kruger zone 13 [6033] TUREF / 3-degree Gauss-Kruger zone 14 [6034] TUREF / 3-degree Gauss-Kruger zone 15 [6035] TUREF / 3-degree Gauss-Kruger zone 9 [6036] TUREF / LAEA Europe [6037] TUREF / LCC Europe [6038] TUREF / TM27 [6039] TUREF / TM30 [6040] TUREF / TM33 [6041] TUREF / TM36 [6042] TUREF / TM39 [6043] TUREF / TM42 [6044] TUREF / TM45 [6045] TWD67 / TM2 zone 119 [6046] TWD67 / TM2 zone 121 [6047] TWD97 / TM2 zone 119 [6048] TWD97 / TM2 zone 121 [6049] TWD_1967_TM_Penghu (deprecated) [6050] TWD_1967_TM_Taiwan (deprecated) [6051] TWD_1997_TM_Penghu (deprecated) [6052] TWD_1997_TM_Taiwan (deprecated) [6053] Tahaa 54 / UTM zone 5S [6054] Tahiti 52 / UTM zone 6S [6055] Tahiti 79 / UTM zone 6S [6056] Tananarive (Paris) / Laborde Grid [6057] Tananarive (Paris) / Laborde Grid (deprecated) [6058] Tananarive (Paris) / Laborde Grid approximation [6059] Tananarive / Laborde Grid [6060] Tananarive / UTM zone 38S [6061] Tananarive / UTM zone 39S [6062] Tananarive_1925_Laborde_Grid [6063] Tete / UTM zone 36S [6064] Tete / UTM zone 37S [6065] The_World_From_Space [6066] Timbalai 1948 / RSO Borneo (ch) [6067] Timbalai 1948 / RSO Borneo (ftSe) [6068] Timbalai 1948 / RSO Borneo (m) [6069] Timbalai 1948 / UTM zone 49N [6070] Timbalai 1948 / UTM zone 50N [6071] Tokyo / Japan Plane Rectangular CS I [6072] Tokyo / Japan Plane Rectangular CS II [6073] Tokyo / Japan Plane Rectangular CS III [6074] Tokyo / Japan Plane Rectangular CS IV [6075] Tokyo / Japan Plane Rectangular CS IX [6076] Tokyo / Japan Plane Rectangular CS V [6077] Tokyo / Japan Plane Rectangular CS VI [6078] Tokyo / Japan Plane Rectangular CS VII [6079] Tokyo / Japan Plane Rectangular CS VIII [6080] Tokyo / Japan Plane Rectangular CS X [6081] Tokyo / Japan Plane Rectangular CS XI [6082] Tokyo / Japan Plane Rectangular CS XII [6083] Tokyo / Japan Plane Rectangular CS XIII [6084] Tokyo / Japan Plane Rectangular CS XIV [6085] Tokyo / Japan Plane Rectangular CS XIX [6086] Tokyo / Japan Plane Rectangular CS XV [6087] Tokyo / Japan Plane Rectangular CS XVI [6088] Tokyo / Japan Plane Rectangular CS XVII [6089] Tokyo / Japan Plane Rectangular CS XVIII [6090] Tokyo / UTM zone 51N [6091] Tokyo / UTM zone 52N [6092] Tokyo / UTM zone 53N [6093] Tokyo / UTM zone 54N [6094] Tokyo / UTM zone 55N [6095] Tokyo 1892 / Korea Central Belt [6096] Tokyo 1892 / Korea East Belt [6097] Tokyo 1892 / Korea East Sea Belt [6098] Tokyo 1892 / Korea West Belt [6099] Tokyo_UTM_Zone_51N (deprecated) [6100] Tokyo_UTM_Zone_52N (deprecated) [6101] Tokyo_UTM_Zone_53N (deprecated) [6102] Tokyo_UTM_Zone_54N (deprecated) [6103] Tokyo_UTM_Zone_55N (deprecated) [6104] Tokyo_UTM_Zone_56N [6105] Trinidad 1903 / Trinidad Grid [6106] Trinidad 1903 / Trinidad Grid (ftCla) [6107] UCS-2000 / 3-degree Gauss-Kruger CM 21E (deprecated) [6108] UCS-2000 / 3-degree Gauss-Kruger CM 24E (deprecated) [6109] UCS-2000 / 3-degree Gauss-Kruger CM 27E (deprecated) [6110] UCS-2000 / 3-degree Gauss-Kruger CM 30E (deprecated) [6111] UCS-2000 / 3-degree Gauss-Kruger CM 33E (deprecated) [6112] UCS-2000 / 3-degree Gauss-Kruger CM 36E (deprecated) [6113] UCS-2000 / 3-degree Gauss-Kruger CM 39E (deprecated) [6114] UCS-2000 / 3-degree Gauss-Kruger zone 10 (deprecated) [6115] UCS-2000 / 3-degree Gauss-Kruger zone 11 (deprecated) [6116] UCS-2000 / 3-degree Gauss-Kruger zone 12 (deprecated) [6117] UCS-2000 / 3-degree Gauss-Kruger zone 13 (deprecated) [6118] UCS-2000 / 3-degree Gauss-Kruger zone 7 (deprecated) [6119] UCS-2000 / 3-degree Gauss-Kruger zone 8 (deprecated) [6120] UCS-2000 / 3-degree Gauss-Kruger zone 9 (deprecated) [6121] UCS-2000 / Gauss-Kruger CM 21E [6122] UCS-2000 / Gauss-Kruger CM 27E [6123] UCS-2000 / Gauss-Kruger CM 33E [6124] UCS-2000 / Gauss-Kruger CM 39E [6125] UCS-2000 / Gauss-Kruger zone 4 [6126] UCS-2000 / Gauss-Kruger zone 5 [6127] UCS-2000 / Gauss-Kruger zone 6 [6128] UCS-2000 / Gauss-Kruger zone 7 [6129] UCS-2000 / Ukraine TM zone 10 [6130] UCS-2000 / Ukraine TM zone 11 [6131] UCS-2000 / Ukraine TM zone 12 [6132] UCS-2000 / Ukraine TM zone 13 [6133] UCS-2000 / Ukraine TM zone 7 [6134] UCS-2000 / Ukraine TM zone 8 [6135] UCS-2000 / Ukraine TM zone 9 [6136] US National Atlas Equal Area [6137] USA_Contiguous_Albers_Equal_Area_Conic [6138] USA_Contiguous_Albers_Equal_Area_Conic_USGS_version [6139] USA_Contiguous_Equidistant_Conic [6140] USA_Contiguous_Lambert_Conformal_Conic [6141] UWPP_1992 [6142] UWPP_2000_PAS_5 [6143] UWPP_2000_PAS_6 [6144] UWPP_2000_PAS_7 [6145] UWPP_2000_PAS_8 [6146] VN-2000 / TM-3 Da Nang zone [6147] VN-2000 / TM-3 Da Nang zone [6148] VN-2000 / TM-3 zone 481 [6149] VN-2000 / TM-3 zone 481 [6150] VN-2000 / TM-3 zone 482 [6151] VN-2000 / TM-3 zone 482 [6152] VN-2000 / TM-3 zone 491 [6153] VN-2000 / TM-3 zone 491 [6154] VN-2000 / UTM zone 48N [6155] VN-2000 / UTM zone 49N [6156] Viti Levu 1912 / Viti Levu Grid [6157] Voirol 1875 / Nord Algerie (ancienne) [6158] Voirol 1875 / Sud Algerie (ancienne) [6159] Voirol 1879 / Nord Algerie (ancienne) [6160] Voirol 1879 / Sud Algerie (ancienne) [6161] WGS 72 / UTM zone 10N [6162] WGS 72 / UTM zone 10S [6163] WGS 72 / UTM zone 11N [6164] WGS 72 / UTM zone 11S [6165] WGS 72 / UTM zone 12N [6166] WGS 72 / UTM zone 12S [6167] WGS 72 / UTM zone 13N [6168] WGS 72 / UTM zone 13S [6169] WGS 72 / UTM zone 14N [6170] WGS 72 / UTM zone 14S [6171] WGS 72 / UTM zone 15N [6172] WGS 72 / UTM zone 15S [6173] WGS 72 / UTM zone 16N [6174] WGS 72 / UTM zone 16S [6175] WGS 72 / UTM zone 17N [6176] WGS 72 / UTM zone 17S [6177] WGS 72 / UTM zone 18N [6178] WGS 72 / UTM zone 18S [6179] WGS 72 / UTM zone 19N [6180] WGS 72 / UTM zone 19S [6181] WGS 72 / UTM zone 1N [6182] WGS 72 / UTM zone 1S [6183] WGS 72 / UTM zone 20N [6184] WGS 72 / UTM zone 20S [6185] WGS 72 / UTM zone 21N [6186] WGS 72 / UTM zone 21S [6187] WGS 72 / UTM zone 22N [6188] WGS 72 / UTM zone 22S [6189] WGS 72 / UTM zone 23N [6190] WGS 72 / UTM zone 23S [6191] WGS 72 / UTM zone 24N [6192] WGS 72 / UTM zone 24S [6193] WGS 72 / UTM zone 25N [6194] WGS 72 / UTM zone 25S [6195] WGS 72 / UTM zone 26N [6196] WGS 72 / UTM zone 26S [6197] WGS 72 / UTM zone 27N [6198] WGS 72 / UTM zone 27S [6199] WGS 72 / UTM zone 28N [6200] WGS 72 / UTM zone 28S [6201] WGS 72 / UTM zone 29N [6202] WGS 72 / UTM zone 29S [6203] WGS 72 / UTM zone 2N [6204] WGS 72 / UTM zone 2S [6205] WGS 72 / UTM zone 30N [6206] WGS 72 / UTM zone 30S [6207] WGS 72 / UTM zone 31N [6208] WGS 72 / UTM zone 31S [6209] WGS 72 / UTM zone 32N [6210] WGS 72 / UTM zone 32S [6211] WGS 72 / UTM zone 33N [6212] WGS 72 / UTM zone 33S [6213] WGS 72 / UTM zone 34N [6214] WGS 72 / UTM zone 34S [6215] WGS 72 / UTM zone 35N [6216] WGS 72 / UTM zone 35S [6217] WGS 72 / UTM zone 36N [6218] WGS 72 / UTM zone 36S [6219] WGS 72 / UTM zone 37N [6220] WGS 72 / UTM zone 37S [6221] WGS 72 / UTM zone 38N [6222] WGS 72 / UTM zone 38S [6223] WGS 72 / UTM zone 39N [6224] WGS 72 / UTM zone 39S [6225] WGS 72 / UTM zone 3N [6226] WGS 72 / UTM zone 3S [6227] WGS 72 / UTM zone 40N [6228] WGS 72 / UTM zone 40S [6229] WGS 72 / UTM zone 41N [6230] WGS 72 / UTM zone 41S [6231] WGS 72 / UTM zone 42N [6232] WGS 72 / UTM zone 42S [6233] WGS 72 / UTM zone 43N [6234] WGS 72 / UTM zone 43S [6235] WGS 72 / UTM zone 44N [6236] WGS 72 / UTM zone 44S [6237] WGS 72 / UTM zone 45N [6238] WGS 72 / UTM zone 45S [6239] WGS 72 / UTM zone 46N [6240] WGS 72 / UTM zone 46S [6241] WGS 72 / UTM zone 47N [6242] WGS 72 / UTM zone 47S [6243] WGS 72 / UTM zone 48N [6244] WGS 72 / UTM zone 48S [6245] WGS 72 / UTM zone 49N [6246] WGS 72 / UTM zone 49S [6247] WGS 72 / UTM zone 4N [6248] WGS 72 / UTM zone 4S [6249] WGS 72 / UTM zone 50N [6250] WGS 72 / UTM zone 50S [6251] WGS 72 / UTM zone 51N [6252] WGS 72 / UTM zone 51S [6253] WGS 72 / UTM zone 52N [6254] WGS 72 / UTM zone 52S [6255] WGS 72 / UTM zone 53N [6256] WGS 72 / UTM zone 53S [6257] WGS 72 / UTM zone 54N [6258] WGS 72 / UTM zone 54S [6259] WGS 72 / UTM zone 55N [6260] WGS 72 / UTM zone 55S [6261] WGS 72 / UTM zone 56N [6262] WGS 72 / UTM zone 56S [6263] WGS 72 / UTM zone 57N [6264] WGS 72 / UTM zone 57S [6265] WGS 72 / UTM zone 58N [6266] WGS 72 / UTM zone 58S [6267] WGS 72 / UTM zone 59N [6268] WGS 72 / UTM zone 59S [6269] WGS 72 / UTM zone 5N [6270] WGS 72 / UTM zone 5S [6271] WGS 72 / UTM zone 60N [6272] WGS 72 / UTM zone 60S [6273] WGS 72 / UTM zone 6N [6274] WGS 72 / UTM zone 6S [6275] WGS 72 / UTM zone 7N [6276] WGS 72 / UTM zone 7S [6277] WGS 72 / UTM zone 8N [6278] WGS 72 / UTM zone 8S [6279] WGS 72 / UTM zone 9N [6280] WGS 72 / UTM zone 9S [6281] WGS 72BE / South China Sea Lambert [6282] WGS 72BE / TM 106 NE [6283] WGS 72BE / UTM zone 10N [6284] WGS 72BE / UTM zone 10S [6285] WGS 72BE / UTM zone 11N [6286] WGS 72BE / UTM zone 11S [6287] WGS 72BE / UTM zone 12N [6288] WGS 72BE / UTM zone 12S [6289] WGS 72BE / UTM zone 13N [6290] WGS 72BE / UTM zone 13S [6291] WGS 72BE / UTM zone 14N [6292] WGS 72BE / UTM zone 14S [6293] WGS 72BE / UTM zone 15N [6294] WGS 72BE / UTM zone 15S [6295] WGS 72BE / UTM zone 16N [6296] WGS 72BE / UTM zone 16S [6297] WGS 72BE / UTM zone 17N [6298] WGS 72BE / UTM zone 17S [6299] WGS 72BE / UTM zone 18N [6300] WGS 72BE / UTM zone 18S [6301] WGS 72BE / UTM zone 19N [6302] WGS 72BE / UTM zone 19S [6303] WGS 72BE / UTM zone 1N [6304] WGS 72BE / UTM zone 1S [6305] WGS 72BE / UTM zone 20N [6306] WGS 72BE / UTM zone 20S [6307] WGS 72BE / UTM zone 21N [6308] WGS 72BE / UTM zone 21S [6309] WGS 72BE / UTM zone 22N [6310] WGS 72BE / UTM zone 22S [6311] WGS 72BE / UTM zone 23N [6312] WGS 72BE / UTM zone 23S [6313] WGS 72BE / UTM zone 24N [6314] WGS 72BE / UTM zone 24S [6315] WGS 72BE / UTM zone 25N [6316] WGS 72BE / UTM zone 25S [6317] WGS 72BE / UTM zone 26N [6318] WGS 72BE / UTM zone 26S [6319] WGS 72BE / UTM zone 27N [6320] WGS 72BE / UTM zone 27S [6321] WGS 72BE / UTM zone 28N [6322] WGS 72BE / UTM zone 28S [6323] WGS 72BE / UTM zone 29N [6324] WGS 72BE / UTM zone 29S [6325] WGS 72BE / UTM zone 2N [6326] WGS 72BE / UTM zone 2S [6327] WGS 72BE / UTM zone 30N [6328] WGS 72BE / UTM zone 30S [6329] WGS 72BE / UTM zone 31N [6330] WGS 72BE / UTM zone 31S [6331] WGS 72BE / UTM zone 32N [6332] WGS 72BE / UTM zone 32S [6333] WGS 72BE / UTM zone 33N [6334] WGS 72BE / UTM zone 33S [6335] WGS 72BE / UTM zone 34N [6336] WGS 72BE / UTM zone 34S [6337] WGS 72BE / UTM zone 35N [6338] WGS 72BE / UTM zone 35S [6339] WGS 72BE / UTM zone 36N [6340] WGS 72BE / UTM zone 36S [6341] WGS 72BE / UTM zone 37N [6342] WGS 72BE / UTM zone 37S [6343] WGS 72BE / UTM zone 38N [6344] WGS 72BE / UTM zone 38S [6345] WGS 72BE / UTM zone 39N [6346] WGS 72BE / UTM zone 39S [6347] WGS 72BE / UTM zone 3N [6348] WGS 72BE / UTM zone 3S [6349] WGS 72BE / UTM zone 40N [6350] WGS 72BE / UTM zone 40S [6351] WGS 72BE / UTM zone 41N [6352] WGS 72BE / UTM zone 41S [6353] WGS 72BE / UTM zone 42N [6354] WGS 72BE / UTM zone 42S [6355] WGS 72BE / UTM zone 43N [6356] WGS 72BE / UTM zone 43S [6357] WGS 72BE / UTM zone 44N [6358] WGS 72BE / UTM zone 44S [6359] WGS 72BE / UTM zone 45N [6360] WGS 72BE / UTM zone 45S [6361] WGS 72BE / UTM zone 46N [6362] WGS 72BE / UTM zone 46S [6363] WGS 72BE / UTM zone 47N [6364] WGS 72BE / UTM zone 47S [6365] WGS 72BE / UTM zone 48N [6366] WGS 72BE / UTM zone 48S [6367] WGS 72BE / UTM zone 49N [6368] WGS 72BE / UTM zone 49S [6369] WGS 72BE / UTM zone 4N [6370] WGS 72BE / UTM zone 4S [6371] WGS 72BE / UTM zone 50N [6372] WGS 72BE / UTM zone 50S [6373] WGS 72BE / UTM zone 51N [6374] WGS 72BE / UTM zone 51S [6375] WGS 72BE / UTM zone 52N [6376] WGS 72BE / UTM zone 52S [6377] WGS 72BE / UTM zone 53N [6378] WGS 72BE / UTM zone 53S [6379] WGS 72BE / UTM zone 54N [6380] WGS 72BE / UTM zone 54S [6381] WGS 72BE / UTM zone 55N [6382] WGS 72BE / UTM zone 55S [6383] WGS 72BE / UTM zone 56N [6384] WGS 72BE / UTM zone 56S [6385] WGS 72BE / UTM zone 57N [6386] WGS 72BE / UTM zone 57S [6387] WGS 72BE / UTM zone 58N [6388] WGS 72BE / UTM zone 58S [6389] WGS 72BE / UTM zone 59N [6390] WGS 72BE / UTM zone 59S [6391] WGS 72BE / UTM zone 5N [6392] WGS 72BE / UTM zone 5S [6393] WGS 72BE / UTM zone 60N [6394] WGS 72BE / UTM zone 60S [6395] WGS 72BE / UTM zone 6N [6396] WGS 72BE / UTM zone 6S [6397] WGS 72BE / UTM zone 7N [6398] WGS 72BE / UTM zone 7S [6399] WGS 72BE / UTM zone 8N [6400] WGS 72BE / UTM zone 8S [6401] WGS 72BE / UTM zone 9N [6402] WGS 72BE / UTM zone 9S [6403] WGS 84 / Andaman and Nicobar [6404] WGS 84 / Andhra Pradesh [6405] WGS 84 / Antarctic Polar Stereographic [6406] WGS 84 / Arctic Polar Stereographic [6407] WGS 84 / Arunachal Pradesh [6408] WGS 84 / Assam [6409] WGS 84 / Australian Antarctic Lambert [6410] WGS 84 / Australian Antarctic Polar Stereographic [6411] WGS 84 / Australian Centre for Remote Sensing Lambert [6412] WGS 84 / BLM 14N (ftUS) [6413] WGS 84 / BLM 15N (ftUS) [6414] WGS 84 / BLM 16N (ftUS) [6415] WGS 84 / BLM 17N (ftUS) [6416] WGS 84 / Bihar [6417] WGS 84 / CIG92 [6418] WGS 84 / CKIG92 [6419] WGS 84 / Cape Verde National [6420] WGS 84 / Chhattisgarh [6421] WGS 84 / Delhi [6422] WGS 84 / Dubai Local TM [6423] WGS 84 / EPSG Alaska Polar Stereographic [6424] WGS 84 / EPSG Arctic Regional zone A1 [6425] WGS 84 / EPSG Arctic Regional zone A2 [6426] WGS 84 / EPSG Arctic Regional zone A3 [6427] WGS 84 / EPSG Arctic Regional zone A4 [6428] WGS 84 / EPSG Arctic Regional zone A5 [6429] WGS 84 / EPSG Arctic Regional zone B1 [6430] WGS 84 / EPSG Arctic Regional zone B2 [6431] WGS 84 / EPSG Arctic Regional zone B3 [6432] WGS 84 / EPSG Arctic Regional zone B4 [6433] WGS 84 / EPSG Arctic Regional zone B5 [6434] WGS 84 / EPSG Arctic Regional zone C1 [6435] WGS 84 / EPSG Arctic Regional zone C2 [6436] WGS 84 / EPSG Arctic Regional zone C3 [6437] WGS 84 / EPSG Arctic Regional zone C4 [6438] WGS 84 / EPSG Arctic Regional zone C5 [6439] WGS 84 / EPSG Arctic zone 1-21 [6440] WGS 84 / EPSG Arctic zone 1-27 [6441] WGS 84 / EPSG Arctic zone 1-29 [6442] WGS 84 / EPSG Arctic zone 1-31 [6443] WGS 84 / EPSG Arctic zone 2-10 [6444] WGS 84 / EPSG Arctic zone 2-12 [6445] WGS 84 / EPSG Arctic zone 2-24 [6446] WGS 84 / EPSG Arctic zone 2-26 [6447] WGS 84 / EPSG Arctic zone 2-28 [6448] WGS 84 / EPSG Arctic zone 3-13 [6449] WGS 84 / EPSG Arctic zone 3-15 [6450] WGS 84 / EPSG Arctic zone 3-17 [6451] WGS 84 / EPSG Arctic zone 3-19 [6452] WGS 84 / EPSG Arctic zone 3-21 [6453] WGS 84 / EPSG Arctic zone 3-23 [6454] WGS 84 / EPSG Arctic zone 4-12 [6455] WGS 84 / EPSG Arctic zone 4-30 [6456] WGS 84 / EPSG Arctic zone 4-32 [6457] WGS 84 / EPSG Arctic zone 4-34 [6458] WGS 84 / EPSG Arctic zone 4-36 [6459] WGS 84 / EPSG Arctic zone 4-38 [6460] WGS 84 / EPSG Arctic zone 4-40 [6461] WGS 84 / EPSG Arctic zone 5-15 [6462] WGS 84 / EPSG Arctic zone 5-17 [6463] WGS 84 / EPSG Arctic zone 5-19 [6464] WGS 84 / EPSG Arctic zone 5-21 [6465] WGS 84 / EPSG Arctic zone 5-23 [6466] WGS 84 / EPSG Arctic zone 5-25 [6467] WGS 84 / EPSG Arctic zone 5-27 [6468] WGS 84 / EPSG Canada Polar Stereographic [6469] WGS 84 / EPSG Greenland Polar Stereographic [6470] WGS 84 / EPSG Norway Polar Stereographic [6471] WGS 84 / EPSG Russia Polar Stereographic [6472] WGS 84 / Gabon TM [6473] WGS 84 / Gabon TM 2011 [6474] WGS 84 / Goa [6475] WGS 84 / Gujarat [6476] WGS 84 / Haryana [6477] WGS 84 / Himachal Pradesh [6478] WGS 84 / IBCAO Polar Stereographic [6479] WGS 84 / India NSF LCC [6480] WGS 84 / India Northeast [6481] WGS 84 / Jammu and Kashmir [6482] WGS 84 / Jharkhand [6483] WGS 84 / Karnataka [6484] WGS 84 / Kerala [6485] WGS 84 / Lakshadweep [6486] WGS 84 / Madhya Pradesh [6487] WGS 84 / Maharashtra [6488] WGS 84 / Manipur [6489] WGS 84 / Meghalaya [6490] WGS 84 / Mercator 41 [6491] WGS 84 / Mercator 41 (deprecated) [6492] WGS 84 / Mizoram [6493] WGS 84 / NSIDC EASE-Grid 2.0 Global [6494] WGS 84 / NSIDC EASE-Grid 2.0 North [6495] WGS 84 / NSIDC EASE-Grid 2.0 South [6496] WGS 84 / NSIDC EASE-Grid Global (deprecated) [6497] WGS 84 / NSIDC EASE-Grid North (deprecated) [6498] WGS 84 / NSIDC EASE-Grid South (deprecated) [6499] WGS 84 / NSIDC Sea Ice Polar Stereographic North [6500] WGS 84 / NSIDC Sea Ice Polar Stereographic South [6501] WGS 84 / Nagaland [6502] WGS 84 / North Pole LAEA Alaska [6503] WGS 84 / North Pole LAEA Atlantic [6504] WGS 84 / North Pole LAEA Bering Sea [6505] WGS 84 / North Pole LAEA Canada [6506] WGS 84 / North Pole LAEA Europe [6507] WGS 84 / North Pole LAEA Russia [6508] WGS 84 / Orissa [6509] WGS 84 / PDC Mercator [6510] WGS 84 / PDC Mercator (deprecated) [6511] WGS 84 / Plate Carree (deprecated) [6512] WGS 84 / Pseudo-Mercator [6513] WGS 84 / Punjab [6514] WGS 84 / Rajasthan [6515] WGS 84 / SCAR IMW SP19-20 [6516] WGS 84 / SCAR IMW SP21-22 [6517] WGS 84 / SCAR IMW SP23-24 [6518] WGS 84 / SCAR IMW SQ01-02 [6519] WGS 84 / SCAR IMW SQ19-20 [6520] WGS 84 / SCAR IMW SQ21-22 [6521] WGS 84 / SCAR IMW SQ37-38 [6522] WGS 84 / SCAR IMW SQ39-40 [6523] WGS 84 / SCAR IMW SQ41-42 [6524] WGS 84 / SCAR IMW SQ43-44 [6525] WGS 84 / SCAR IMW SQ45-46 [6526] WGS 84 / SCAR IMW SQ47-48 [6527] WGS 84 / SCAR IMW SQ49-50 [6528] WGS 84 / SCAR IMW SQ51-52 [6529] WGS 84 / SCAR IMW SQ53-54 [6530] WGS 84 / SCAR IMW SQ55-56 [6531] WGS 84 / SCAR IMW SQ57-58 [6532] WGS 84 / SCAR IMW SR13-14 [6533] WGS 84 / SCAR IMW SR15-16 [6534] WGS 84 / SCAR IMW SR17-18 [6535] WGS 84 / SCAR IMW SR19-20 [6536] WGS 84 / SCAR IMW SR27-28 [6537] WGS 84 / SCAR IMW SR29-30 [6538] WGS 84 / SCAR IMW SR31-32 [6539] WGS 84 / SCAR IMW SR33-34 [6540] WGS 84 / SCAR IMW SR35-36 [6541] WGS 84 / SCAR IMW SR37-38 [6542] WGS 84 / SCAR IMW SR39-40 [6543] WGS 84 / SCAR IMW SR41-42 [6544] WGS 84 / SCAR IMW SR43-44 [6545] WGS 84 / SCAR IMW SR45-46 [6546] WGS 84 / SCAR IMW SR47-48 [6547] WGS 84 / SCAR IMW SR49-50 [6548] WGS 84 / SCAR IMW SR51-52 [6549] WGS 84 / SCAR IMW SR53-54 [6550] WGS 84 / SCAR IMW SR55-56 [6551] WGS 84 / SCAR IMW SR57-58 [6552] WGS 84 / SCAR IMW SR59-60 [6553] WGS 84 / SCAR IMW SS04-06 [6554] WGS 84 / SCAR IMW SS07-09 [6555] WGS 84 / SCAR IMW SS10-12 [6556] WGS 84 / SCAR IMW SS13-15 [6557] WGS 84 / SCAR IMW SS16-18 [6558] WGS 84 / SCAR IMW SS19-21 [6559] WGS 84 / SCAR IMW SS25-27 [6560] WGS 84 / SCAR IMW SS28-30 [6561] WGS 84 / SCAR IMW SS31-33 [6562] WGS 84 / SCAR IMW SS34-36 [6563] WGS 84 / SCAR IMW SS37-39 [6564] WGS 84 / SCAR IMW SS40-42 [6565] WGS 84 / SCAR IMW SS43-45 [6566] WGS 84 / SCAR IMW SS46-48 [6567] WGS 84 / SCAR IMW SS49-51 [6568] WGS 84 / SCAR IMW SS52-54 [6569] WGS 84 / SCAR IMW SS55-57 [6570] WGS 84 / SCAR IMW SS58-60 [6571] WGS 84 / SCAR IMW ST01-04 [6572] WGS 84 / SCAR IMW ST05-08 [6573] WGS 84 / SCAR IMW ST09-12 [6574] WGS 84 / SCAR IMW ST13-16 [6575] WGS 84 / SCAR IMW ST17-20 [6576] WGS 84 / SCAR IMW ST21-24 [6577] WGS 84 / SCAR IMW ST25-28 [6578] WGS 84 / SCAR IMW ST29-32 [6579] WGS 84 / SCAR IMW ST33-36 [6580] WGS 84 / SCAR IMW ST37-40 [6581] WGS 84 / SCAR IMW ST41-44 [6582] WGS 84 / SCAR IMW ST45-48 [6583] WGS 84 / SCAR IMW ST49-52 [6584] WGS 84 / SCAR IMW ST53-56 [6585] WGS 84 / SCAR IMW ST57-60 [6586] WGS 84 / SCAR IMW SU01-05 [6587] WGS 84 / SCAR IMW SU06-10 [6588] WGS 84 / SCAR IMW SU11-15 [6589] WGS 84 / SCAR IMW SU16-20 [6590] WGS 84 / SCAR IMW SU21-25 [6591] WGS 84 / SCAR IMW SU26-30 [6592] WGS 84 / SCAR IMW SU31-35 [6593] WGS 84 / SCAR IMW SU36-40 [6594] WGS 84 / SCAR IMW SU41-45 [6595] WGS 84 / SCAR IMW SU46-50 [6596] WGS 84 / SCAR IMW SU51-55 [6597] WGS 84 / SCAR IMW SU56-60 [6598] WGS 84 / SCAR IMW SV01-10 [6599] WGS 84 / SCAR IMW SV11-20 [6600] WGS 84 / SCAR IMW SV21-30 [6601] WGS 84 / SCAR IMW SV31-40 [6602] WGS 84 / SCAR IMW SV41-50 [6603] WGS 84 / SCAR IMW SV51-60 [6604] WGS 84 / SCAR IMW SW01-60 [6605] WGS 84 / Sikkim [6606] WGS 84 / South Georgia Lambert [6607] WGS 84 / TM 116 SE [6608] WGS 84 / TM 12 SE [6609] WGS 84 / TM 132 SE [6610] WGS 84 / TM 36 SE [6611] WGS 84 / TM 6 NE [6612] WGS 84 / TM 60 SW [6613] WGS 84 / TM Zone 20N (ftUS) [6614] WGS 84 / TM Zone 21N (ftUS) [6615] WGS 84 / TMzn35N [6616] WGS 84 / TMzn36N [6617] WGS 84 / Tamil Nadu [6618] WGS 84 / Tripura [6619] WGS 84 / UPS North (E,N) [6620] WGS 84 / UPS North (N,E) [6621] WGS 84 / UPS South (E,N) [6622] WGS 84 / UPS South (N,E) [6623] WGS 84 / USGS Transantarctic Mountains [6624] WGS 84 / UTM zone 10N [6625] WGS 84 / UTM zone 10S [6626] WGS 84 / UTM zone 11N [6627] WGS 84 / UTM zone 11S [6628] WGS 84 / UTM zone 12N [6629] WGS 84 / UTM zone 12S [6630] WGS 84 / UTM zone 13N [6631] WGS 84 / UTM zone 13S [6632] WGS 84 / UTM zone 14N [6633] WGS 84 / UTM zone 14S [6634] WGS 84 / UTM zone 15N [6635] WGS 84 / UTM zone 15S [6636] WGS 84 / UTM zone 16N [6637] WGS 84 / UTM zone 16S [6638] WGS 84 / UTM zone 17N [6639] WGS 84 / UTM zone 17S [6640] WGS 84 / UTM zone 18N [6641] WGS 84 / UTM zone 18S [6642] WGS 84 / UTM zone 19N [6643] WGS 84 / UTM zone 19S [6644] WGS 84 / UTM zone 1N [6645] WGS 84 / UTM zone 1S [6646] WGS 84 / UTM zone 20N [6647] WGS 84 / UTM zone 20S [6648] WGS 84 / UTM zone 21N [6649] WGS 84 / UTM zone 21S [6650] WGS 84 / UTM zone 22N [6651] WGS 84 / UTM zone 22S [6652] WGS 84 / UTM zone 23N [6653] WGS 84 / UTM zone 23S [6654] WGS 84 / UTM zone 24N [6655] WGS 84 / UTM zone 24S [6656] WGS 84 / UTM zone 25N [6657] WGS 84 / UTM zone 25S [6658] WGS 84 / UTM zone 26N [6659] WGS 84 / UTM zone 26S [6660] WGS 84 / UTM zone 27N [6661] WGS 84 / UTM zone 27S [6662] WGS 84 / UTM zone 28N [6663] WGS 84 / UTM zone 28S [6664] WGS 84 / UTM zone 29N [6665] WGS 84 / UTM zone 29S [6666] WGS 84 / UTM zone 2N [6667] WGS 84 / UTM zone 2S [6668] WGS 84 / UTM zone 30N [6669] WGS 84 / UTM zone 30S [6670] WGS 84 / UTM zone 31N [6671] WGS 84 / UTM zone 31S [6672] WGS 84 / UTM zone 32N [6673] WGS 84 / UTM zone 32S [6674] WGS 84 / UTM zone 33N [6675] WGS 84 / UTM zone 33S [6676] WGS 84 / UTM zone 34N [6677] WGS 84 / UTM zone 34S [6678] WGS 84 / UTM zone 35N [6679] WGS 84 / UTM zone 35S [6680] WGS 84 / UTM zone 36N [6681] WGS 84 / UTM zone 36S [6682] WGS 84 / UTM zone 37N [6683] WGS 84 / UTM zone 37S [6684] WGS 84 / UTM zone 38N [6685] WGS 84 / UTM zone 38S [6686] WGS 84 / UTM zone 39N [6687] WGS 84 / UTM zone 39S [6688] WGS 84 / UTM zone 3N [6689] WGS 84 / UTM zone 3S [6690] WGS 84 / UTM zone 40N [6691] WGS 84 / UTM zone 40S [6692] WGS 84 / UTM zone 41N [6693] WGS 84 / UTM zone 41S [6694] WGS 84 / UTM zone 42N [6695] WGS 84 / UTM zone 42S [6696] WGS 84 / UTM zone 43N [6697] WGS 84 / UTM zone 43S [6698] WGS 84 / UTM zone 44N [6699] WGS 84 / UTM zone 44S [6700] WGS 84 / UTM zone 45N [6701] WGS 84 / UTM zone 45S [6702] WGS 84 / UTM zone 46N [6703] WGS 84 / UTM zone 46S [6704] WGS 84 / UTM zone 47N [6705] WGS 84 / UTM zone 47S [6706] WGS 84 / UTM zone 48N [6707] WGS 84 / UTM zone 48S [6708] WGS 84 / UTM zone 49N [6709] WGS 84 / UTM zone 49S [6710] WGS 84 / UTM zone 4N [6711] WGS 84 / UTM zone 4S [6712] WGS 84 / UTM zone 50N [6713] WGS 84 / UTM zone 50S [6714] WGS 84 / UTM zone 51N [6715] WGS 84 / UTM zone 51S [6716] WGS 84 / UTM zone 52N [6717] WGS 84 / UTM zone 52S [6718] WGS 84 / UTM zone 53N [6719] WGS 84 / UTM zone 53S [6720] WGS 84 / UTM zone 54N [6721] WGS 84 / UTM zone 54S [6722] WGS 84 / UTM zone 55N [6723] WGS 84 / UTM zone 55S [6724] WGS 84 / UTM zone 56N [6725] WGS 84 / UTM zone 56S [6726] WGS 84 / UTM zone 57N [6727] WGS 84 / UTM zone 57S [6728] WGS 84 / UTM zone 58N [6729] WGS 84 / UTM zone 58S [6730] WGS 84 / UTM zone 59N [6731] WGS 84 / UTM zone 59S [6732] WGS 84 / UTM zone 5N [6733] WGS 84 / UTM zone 5S [6734] WGS 84 / UTM zone 60N [6735] WGS 84 / UTM zone 60S [6736] WGS 84 / UTM zone 6N [6737] WGS 84 / UTM zone 6S [6738] WGS 84 / UTM zone 7N [6739] WGS 84 / UTM zone 7S [6740] WGS 84 / UTM zone 8N [6741] WGS 84 / UTM zone 8S [6742] WGS 84 / UTM zone 9N [6743] WGS 84 / UTM zone 9S [6744] WGS 84 / Uttar Pradesh [6745] WGS 84 / Uttaranchal [6746] WGS 84 / West Bengal [6747] WGS 84 / World Equidistant Cylindrical [6748] WGS 84 / World Equidistant Cylindrical (deprecated) [6749] WGS 84 / World Mercator [6750] WGS_1984_ARC_System_Zone_01 [6751] WGS_1984_ARC_System_Zone_02 [6752] WGS_1984_ARC_System_Zone_03 [6753] WGS_1984_ARC_System_Zone_04 [6754] WGS_1984_ARC_System_Zone_05 [6755] WGS_1984_ARC_System_Zone_06 [6756] WGS_1984_ARC_System_Zone_07 [6757] WGS_1984_ARC_System_Zone_08 [6758] WGS_1984_ARC_System_Zone_09 [6759] WGS_1984_ARC_System_Zone_10 [6760] WGS_1984_ARC_System_Zone_11 [6761] WGS_1984_ARC_System_Zone_12 [6762] WGS_1984_ARC_System_Zone_13 [6763] WGS_1984_ARC_System_Zone_14 [6764] WGS_1984_ARC_System_Zone_15 [6765] WGS_1984_ARC_System_Zone_16 [6766] WGS_1984_ARC_System_Zone_17 [6767] WGS_1984_ARC_System_Zone_18 [6768] WGS_1984_Australian_Centre_for_Remote_Sensing_Lambert (deprecated) [6769] WGS_1984_California_Teale_Albers_FtUS [6770] WGS_1984_Canada_Atlas_LCC [6771] WGS_1984_Cape_Verde_Grid (deprecated) [6772] WGS_1984_Costa_Rica_TM_90 [6773] WGS_1984_Web_Mercator (deprecated) [6774] WGS_1984_Web_Mercator_Auxiliary_Sphere (deprecated) [6775] World Equidistant Cylindrical (Sphere) [6776] World Equidistant Cylindrical (Sphere) (deprecated) [6777] World_Aitoff [6778] World_Azimuthal_Equidistant [6779] World_Bonne [6780] World_Cassini [6781] World_Craster_Parabolic [6782] World_Cylindrical_Equal_Area [6783] World_Eckert_I [6784] World_Eckert_II [6785] World_Eckert_III [6786] World_Eckert_IV [6787] World_Eckert_V [6788] World_Eckert_VI [6789] World_Equidistant_Conic [6790] World_Equidistant_Cylindrical [6791] World_Gall_Stereographic [6792] World_Goode_Homolosine_Land [6793] World_Goode_Homolosine_Ocean [6794] World_Hotine [6795] World_Loximuthal [6796] World_Mercator [6797] World_Miller_Cylindrical [6798] World_Mollweide [6799] World_Plate_Carree [6800] World_Polyconic [6801] World_Quartic_Authalic [6802] World_Robinson [6803] World_Sinusoidal [6804] World_Stereographic [6805] World_Two_Point_Equidistant [6806] World_Van_der_Grinten_I [6807] World_Wagner_IV [6808] World_Wagner_V [6809] World_Wagner_VII [6810] World_Winkel_I [6811] World_Winkel_II [6812] World_Winkel_Tripel_NGS [6813] Xian 1980 / 3-degree Gauss-Kruger CM 102E [6814] Xian 1980 / 3-degree Gauss-Kruger CM 105E [6815] Xian 1980 / 3-degree Gauss-Kruger CM 108E [6816] Xian 1980 / 3-degree Gauss-Kruger CM 111E [6817] Xian 1980 / 3-degree Gauss-Kruger CM 114E [6818] Xian 1980 / 3-degree Gauss-Kruger CM 117E [6819] Xian 1980 / 3-degree Gauss-Kruger CM 120E [6820] Xian 1980 / 3-degree Gauss-Kruger CM 123E [6821] Xian 1980 / 3-degree Gauss-Kruger CM 126E [6822] Xian 1980 / 3-degree Gauss-Kruger CM 129E [6823] Xian 1980 / 3-degree Gauss-Kruger CM 132E [6824] Xian 1980 / 3-degree Gauss-Kruger CM 135E [6825] Xian 1980 / 3-degree Gauss-Kruger CM 75E [6826] Xian 1980 / 3-degree Gauss-Kruger CM 78E [6827] Xian 1980 / 3-degree Gauss-Kruger CM 81E [6828] Xian 1980 / 3-degree Gauss-Kruger CM 84E [6829] Xian 1980 / 3-degree Gauss-Kruger CM 87E [6830] Xian 1980 / 3-degree Gauss-Kruger CM 90E [6831] Xian 1980 / 3-degree Gauss-Kruger CM 93E [6832] Xian 1980 / 3-degree Gauss-Kruger CM 96E [6833] Xian 1980 / 3-degree Gauss-Kruger CM 99E [6834] Xian 1980 / 3-degree Gauss-Kruger zone 25 [6835] Xian 1980 / 3-degree Gauss-Kruger zone 26 [6836] Xian 1980 / 3-degree Gauss-Kruger zone 27 [6837] Xian 1980 / 3-degree Gauss-Kruger zone 28 [6838] Xian 1980 / 3-degree Gauss-Kruger zone 29 [6839] Xian 1980 / 3-degree Gauss-Kruger zone 30 [6840] Xian 1980 / 3-degree Gauss-Kruger zone 31 [6841] Xian 1980 / 3-degree Gauss-Kruger zone 32 [6842] Xian 1980 / 3-degree Gauss-Kruger zone 33 [6843] Xian 1980 / 3-degree Gauss-Kruger zone 34 [6844] Xian 1980 / 3-degree Gauss-Kruger zone 35 [6845] Xian 1980 / 3-degree Gauss-Kruger zone 36 [6846] Xian 1980 / 3-degree Gauss-Kruger zone 37 [6847] Xian 1980 / 3-degree Gauss-Kruger zone 38 [6848] Xian 1980 / 3-degree Gauss-Kruger zone 39 [6849] Xian 1980 / 3-degree Gauss-Kruger zone 40 [6850] Xian 1980 / 3-degree Gauss-Kruger zone 41 [6851] Xian 1980 / 3-degree Gauss-Kruger zone 42 [6852] Xian 1980 / 3-degree Gauss-Kruger zone 43 [6853] Xian 1980 / 3-degree Gauss-Kruger zone 44 [6854] Xian 1980 / 3-degree Gauss-Kruger zone 45 [6855] Xian 1980 / Gauss-Kruger CM 105E [6856] Xian 1980 / Gauss-Kruger CM 111E [6857] Xian 1980 / Gauss-Kruger CM 117E [6858] Xian 1980 / Gauss-Kruger CM 123E [6859] Xian 1980 / Gauss-Kruger CM 129E [6860] Xian 1980 / Gauss-Kruger CM 135E [6861] Xian 1980 / Gauss-Kruger CM 75E [6862] Xian 1980 / Gauss-Kruger CM 81E [6863] Xian 1980 / Gauss-Kruger CM 87E [6864] Xian 1980 / Gauss-Kruger CM 93E [6865] Xian 1980 / Gauss-Kruger CM 99E [6866] Xian 1980 / Gauss-Kruger zone 13 [6867] Xian 1980 / Gauss-Kruger zone 14 [6868] Xian 1980 / Gauss-Kruger zone 15 [6869] Xian 1980 / Gauss-Kruger zone 16 [6870] Xian 1980 / Gauss-Kruger zone 17 [6871] Xian 1980 / Gauss-Kruger zone 18 [6872] Xian 1980 / Gauss-Kruger zone 19 [6873] Xian 1980 / Gauss-Kruger zone 20 [6874] Xian 1980 / Gauss-Kruger zone 21 [6875] Xian 1980 / Gauss-Kruger zone 22 [6876] Xian 1980 / Gauss-Kruger zone 23 [6877] Yemen NGN96 / UTM zone 37N [6878] Yemen NGN96 / UTM zone 38N [6879] Yemen NGN96 / UTM zone 39N [6880] Yemen NGN96 / UTM zone 40N [6881] Yoff / UTM zone 28N [6882] Zanderij / Suriname Old TM [6883] Zanderij / Suriname TM [6884] Zanderij / TM 54 NW [6885] Zanderij / UTM zone 21N Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '21', 'Export Shapes')
    if Tool.is_Okay():
        Tool.Set_Input ('SHAPES', SHAPES)
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('SHAPES_PK', SHAPES_PK)
        Tool.Set_Option('SHAPES_NN', SHAPES_NN)
        Tool.Set_Option('SHAPES_UQ', SHAPES_UQ)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        Tool.Set_Option('CRS_GEOGCS', CRS_GEOGCS)
        Tool.Set_Option('CRS_PROJCS', CRS_PROJCS)
        return Tool.Execute(Verbose)
    return False

def Update_Shapes_SRID(CONNECTION=None, DB_TABLE=None, CRS_CODE=None, CRS_AUTHORITY=None, CRS_GEOGCS=None, CRS_PROJCS=None, Verbose=2):
    '''
    Update Shapes SRID
    ----------
    [db_pgsql.22]\n
    Change the SRID of all geometries in the user-specified column and table.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG
    - CRS_GEOGCS [`choice`] : Geographic Coordinate Systems. Available Choices: [0] <select> [1] 1_Ceres_2015 [2] 4_Vesta_2015 [3] AGD66 [4] AGD84 [5] ATF (Paris) [6] ATS77 [7] Abidjan 1987 [8] Accra [9] Aden 1925 [10] Adindan [11] Afgooye [12] Agadez [13] Ain el Abd [14] Albanian 1987 [15] American Samoa 1962 [16] Amersfoort [17] Ammassalik 1958 [18] Anguilla 1957 [19] Antigua 1943 [20] Aratu [21] Arc 1950 [22] Arc 1960 [23] Ascension Island 1958 [24] Astro DOS 71 [25] Australian Antarctic [26] Australian Antarctic [27] Australian Antarctic [28] Ayabelle Lighthouse [29] Azores Central 1948 [30] Azores Central 1995 [31] Azores Occidental 1939 [32] Azores Oriental 1940 [33] Azores Oriental 1995 [34] BDA2000 [35] BDA2000 [36] BGS2005 [37] BGS2005 [38] Barbados 1938 [39] Batavia [40] Batavia (Jakarta) [41] Beduaram [42] Beijing 1954 [43] Bekaa Valley 1920 [44] Belge 1950 [45] Belge 1950 (Brussels) [46] Belge 1972 [47] Bellevue [48] Bermuda 1957 [49] Bern 1898 (Bern) [50] Bern 1938 [51] Bioko [52] Bissau [53] Bogota 1975 [54] Bogota 1975 (Bogota) [55] Bukit Rimpah [56] CGRS93 [57] CGRS93 [58] CH1903 [59] CH1903+ [60] CHTRF95 [61] CHTRF95 [62] CHTRF95 [63] CIGD11 [64] CIGD11 [65] CR-SIRGAS [66] CR-SIRGAS [67] CR05 [68] CR05 [69] CSG67 [70] Cadastre 1997 [71] Cadastre 1997 [72] California_SRS_Epoch_2017.50_(NAD83) [73] Camacupa [74] Camp Area Astro [75] Campo Inchauspe [76] Cape [77] Cape Canaveral [78] Carthage [79] Carthage (Paris) [80] Chatham Islands 1971 [81] Chatham Islands 1979 [82] China Geodetic Coordinate System 2000 [83] China Geodetic Coordinate System 2000 [84] Chos Malal 1914 [85] Chua [86] Cocos Islands 1965 [87] Combani 1950 [88] Conakry 1905 [89] Corrego Alegre 1961 [90] Corrego Alegre 1970-72 [91] Cote d'Ivoire [92] DB_REF [93] DB_REF [94] DGN95 [95] DGN95 [96] DHDN [97] DRUKREF 03 [98] DRUKREF 03 [99] Dabola 1981 [100] Datum 73 [101] Dealul Piscului 1930 [102] Dealul Piscului 1970 [103] Deception Island [104] Deir ez Zor [105] Diego Garcia 1969 [106] Dominica 1945 [107] Douala [108] Douala 1948 [109] ED50 [110] ED50(ED77) [111] ED79 [112] ED87 [113] ELD79 [114] EST92 [115] EST97 [116] EST97 [117] EST97 [118] ETRF2000 [119] ETRF2000 [120] ETRF2005 [121] ETRF2014 [122] ETRF89 [123] ETRF90 [124] ETRF90 [125] ETRF91 [126] ETRF91 [127] ETRF92 [128] ETRF92 [129] ETRF93 [130] ETRF93 [131] ETRF94 [132] ETRF94 [133] ETRF96 [134] ETRF96 [135] ETRF97 [136] ETRF97 [137] ETRS89 [138] ETRS89 [139] ETRS89 [140] Easter Island 1967 [141] Egypt 1907 [142] Egypt 1930 [143] Egypt Gulf of Suez S-650 TL [144] FD54 [145] FD58 [146] FEH2010 [147] FEH2010 [148] Fahud [149] Fatu Iva 72 [150] Fiji 1956 [151] Fiji 1986 [152] Fort Marigot [153] GCGD59 [154] GCS_Adrastea_2000 [155] GCS_Alaskan_Islands [156] GCS_Amalthea_2000 [157] GCS_American_Samoa_1962 (deprecated) [158] GCS_Ananke_2000 [159] GCS_Anna_1_1965 (deprecated) [160] GCS_Ariel_2000 [161] GCS_Ascension_Island_1958 (deprecated) [162] GCS_Assumed_Geographic_1 (deprecated) [163] GCS_Astro_1952 (deprecated) [164] GCS_Atlas_2000 [165] GCS_Ayabelle (deprecated) [166] GCS_Bab_South [167] GCS_Beacon_E_1945 (deprecated) [168] GCS_Belinda_2000 [169] GCS_Bellevue_IGN (deprecated) [170] GCS_Bermuda_2000 (deprecated) [171] GCS_Bianca_2000 [172] GCS_CGRS_1993 (deprecated) [173] GCS_CR05 (deprecated) [174] GCS_Callisto_2000 [175] GCS_Calypso_2000 [176] GCS_Camp_Area (deprecated) [177] GCS_Canton_1966 (deprecated) [178] GCS_Cape_Canaveral (deprecated) [179] GCS_Carme_2000 [180] GCS_Carthage (deprecated) [181] GCS_Carthage_Grad [182] GCS_Charon_2000 [183] GCS_Chatham_Island_1971 (deprecated) [184] GCS_Chatham_Islands_1979 (deprecated) [185] GCS_Cordelia_2000 [186] GCS_Cressida_2000 [187] GCS_D48 [188] GCS_DOS_1968 [189] GCS_DOS_71_4 (deprecated) [190] GCS_Datum_Lisboa_Bessel [191] GCS_Datum_Lisboa_Hayford [192] GCS_Deception_Island (deprecated) [193] GCS_Deimos_2000 [194] GCS_Desdemona_2000 [195] GCS_Despina_2000 [196] GCS_Dione_2000 [197] GCS_ETRF_1989 [198] GCS_EUREF_FIN [199] GCS_Easter_Island_1967 (deprecated) [200] GCS_Elara_2000 [201] GCS_Enceladus_2000 [202] GCS_Epimetheus_2000 [203] GCS_Estonia_1937 [204] GCS_Europa_2000 [205] GCS_European_1979 (deprecated) [206] GCS_Everest_Bangladesh [207] GCS_Everest_India_Nepal [208] GCS_Everest_Modified_1969 [209] GCS_Fischer_1960 [210] GCS_Fischer_1968 [211] GCS_Fischer_Modified [212] GCS_Fort_Thomas_1955 [213] GCS_GDBD2009 (deprecated) [214] GCS_GUX_1 [215] GCS_Galatea_2000 [216] GCS_Gan_1970 (deprecated) [217] GCS_Ganymede_2000 [218] GCS_Graciosa_Base_SW_1948 [219] GCS_Guam_1963 (deprecated) [220] GCS_Gunung_Segara (deprecated) [221] GCS_HD1909 (deprecated) [222] GCS_Helene_2000 [223] GCS_Hermannskogel [224] GCS_Himalia_2000 [225] GCS_Hjorsey_1955 (deprecated) [226] GCS_Hong_Kong_1963_67 (deprecated) [227] GCS_Hong_Kong_1980 (deprecated) [228] GCS_Hough_1960 [229] GCS_Hyperion_2000 [230] GCS_IGRS (deprecated) [231] GCS_IGS08 [232] GCS_ISN_2004 (deprecated) [233] GCS_ISTS_061_1968 (deprecated) [234] GCS_ISTS_073_1969 (deprecated) [235] GCS_ITRF_1988 [236] GCS_ITRF_1989 [237] GCS_ITRF_1990 [238] GCS_ITRF_1991 [239] GCS_ITRF_1992 [240] GCS_ITRF_1993 [241] GCS_ITRF_1994 [242] GCS_ITRF_1996 [243] GCS_ITRF_1997 [244] GCS_ITRF_2000 [245] GCS_ITRF_2005 [246] GCS_ITRF_2008 [247] GCS_Iapetus_2000 [248] GCS_International_1967 (deprecated) [249] GCS_Io_2000 [250] GCS_JAD_2001 (deprecated) [251] GCS_JGD_2000 (deprecated) [252] GCS_JGD_2011 (deprecated) [253] GCS_Janus_2000 [254] GCS_Johnston_Island_1961 (deprecated) [255] GCS_Jordan [256] GCS_Juliet_2000 [257] GCS_Jupiter_2000 [258] GCS_Kerguelen_Island_1949 (deprecated) [259] GCS_Kusaie_1951 (deprecated) [260] GCS_Kyrg-06 [261] GCS_LC5_1961 [262] GCS_Larissa_2000 [263] GCS_Leda_2000 [264] GCS_Lysithea_2000 [265] GCS_MACAO_2008 [266] GCS_MGI_1901 (deprecated) [267] GCS_MONREF_1997 [268] GCS_MSK_1942 [269] GCS_Majuro [270] GCS_Mars_1979 [271] GCS_Mars_2000 [272] GCS_Merchich_Degree [273] GCS_Mercury_2000 [274] GCS_Metis_2000 [275] GCS_Midway_1961 (deprecated) [276] GCS_Mimas_2000 [277] GCS_Miranda_2000 [278] GCS_Moon_2000 [279] GCS_NAD_1983_2011 (deprecated) [280] GCS_NAD_1983_CORS96 (deprecated) [281] GCS_NAD_1983_HARN_Adj_MN_Anoka [282] GCS_NAD_1983_HARN_Adj_MN_Becker [283] GCS_NAD_1983_HARN_Adj_MN_Beltrami_North [284] GCS_NAD_1983_HARN_Adj_MN_Beltrami_South [285] GCS_NAD_1983_HARN_Adj_MN_Benton [286] GCS_NAD_1983_HARN_Adj_MN_Big_Stone [287] GCS_NAD_1983_HARN_Adj_MN_Blue_Earth [288] GCS_NAD_1983_HARN_Adj_MN_Brown [289] GCS_NAD_1983_HARN_Adj_MN_Carlton [290] GCS_NAD_1983_HARN_Adj_MN_Carver [291] GCS_NAD_1983_HARN_Adj_MN_Cass_North [292] GCS_NAD_1983_HARN_Adj_MN_Cass_South [293] GCS_NAD_1983_HARN_Adj_MN_Chippewa [294] GCS_NAD_1983_HARN_Adj_MN_Chisago [295] GCS_NAD_1983_HARN_Adj_MN_Cook_North [296] GCS_NAD_1983_HARN_Adj_MN_Cook_South [297] GCS_NAD_1983_HARN_Adj_MN_Cottonwood [298] GCS_NAD_1983_HARN_Adj_MN_Crow_Wing [299] GCS_NAD_1983_HARN_Adj_MN_Dakota [300] GCS_NAD_1983_HARN_Adj_MN_Dodge [301] GCS_NAD_1983_HARN_Adj_MN_Douglas [302] GCS_NAD_1983_HARN_Adj_MN_Faribault [303] GCS_NAD_1983_HARN_Adj_MN_Fillmore [304] GCS_NAD_1983_HARN_Adj_MN_Freeborn [305] GCS_NAD_1983_HARN_Adj_MN_Goodhue [306] GCS_NAD_1983_HARN_Adj_MN_Grant [307] GCS_NAD_1983_HARN_Adj_MN_Hennepin [308] GCS_NAD_1983_HARN_Adj_MN_Houston [309] GCS_NAD_1983_HARN_Adj_MN_Isanti [310] GCS_NAD_1983_HARN_Adj_MN_Itasca_North [311] GCS_NAD_1983_HARN_Adj_MN_Itasca_South [312] GCS_NAD_1983_HARN_Adj_MN_Jackson [313] GCS_NAD_1983_HARN_Adj_MN_Kanabec [314] GCS_NAD_1983_HARN_Adj_MN_Kandiyohi [315] GCS_NAD_1983_HARN_Adj_MN_Kittson [316] GCS_NAD_1983_HARN_Adj_MN_Koochiching [317] GCS_NAD_1983_HARN_Adj_MN_Lac_Qui_Parle [318] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North [319] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South [320] GCS_NAD_1983_HARN_Adj_MN_Le_Sueur [321] GCS_NAD_1983_HARN_Adj_MN_Lincoln [322] GCS_NAD_1983_HARN_Adj_MN_Lyon [323] GCS_NAD_1983_HARN_Adj_MN_Mahnomen [324] GCS_NAD_1983_HARN_Adj_MN_Marshall [325] GCS_NAD_1983_HARN_Adj_MN_Martin [326] GCS_NAD_1983_HARN_Adj_MN_McLeod [327] GCS_NAD_1983_HARN_Adj_MN_Meeker [328] GCS_NAD_1983_HARN_Adj_MN_Morrison [329] GCS_NAD_1983_HARN_Adj_MN_Mower [330] GCS_NAD_1983_HARN_Adj_MN_Murray [331] GCS_NAD_1983_HARN_Adj_MN_Nicollet [332] GCS_NAD_1983_HARN_Adj_MN_Nobles [333] GCS_NAD_1983_HARN_Adj_MN_Norman [334] GCS_NAD_1983_HARN_Adj_MN_Olmsted [335] GCS_NAD_1983_HARN_Adj_MN_Ottertail [336] GCS_NAD_1983_HARN_Adj_MN_Pennington [337] GCS_NAD_1983_HARN_Adj_MN_Pine [338] GCS_NAD_1983_HARN_Adj_MN_Pipestone [339] GCS_NAD_1983_HARN_Adj_MN_Polk [340] GCS_NAD_1983_HARN_Adj_MN_Pope [341] GCS_NAD_1983_HARN_Adj_MN_Ramsey [342] GCS_NAD_1983_HARN_Adj_MN_Red_Lake [343] GCS_NAD_1983_HARN_Adj_MN_Redwood [344] GCS_NAD_1983_HARN_Adj_MN_Renville [345] GCS_NAD_1983_HARN_Adj_MN_Rice [346] GCS_NAD_1983_HARN_Adj_MN_Rock [347] GCS_NAD_1983_HARN_Adj_MN_Roseau [348] GCS_NAD_1983_HARN_Adj_MN_Scott [349] GCS_NAD_1983_HARN_Adj_MN_Sherburne [350] GCS_NAD_1983_HARN_Adj_MN_Sibley [351] GCS_NAD_1983_HARN_Adj_MN_St_Louis [352] GCS_NAD_1983_HARN_Adj_MN_St_Louis_Central [353] GCS_NAD_1983_HARN_Adj_MN_St_Louis_North [354] GCS_NAD_1983_HARN_Adj_MN_St_Louis_South [355] GCS_NAD_1983_HARN_Adj_MN_Stearns [356] GCS_NAD_1983_HARN_Adj_MN_Steele [357] GCS_NAD_1983_HARN_Adj_MN_Stevens [358] GCS_NAD_1983_HARN_Adj_MN_Swift [359] GCS_NAD_1983_HARN_Adj_MN_Todd [360] GCS_NAD_1983_HARN_Adj_MN_Traverse [361] GCS_NAD_1983_HARN_Adj_MN_Wabasha [362] GCS_NAD_1983_HARN_Adj_MN_Wadena [363] GCS_NAD_1983_HARN_Adj_MN_Waseca [364] GCS_NAD_1983_HARN_Adj_MN_Watonwan [365] GCS_NAD_1983_HARN_Adj_MN_Winona [366] GCS_NAD_1983_HARN_Adj_MN_Wright [367] GCS_NAD_1983_HARN_Adj_MN_Yellow_Medicine [368] GCS_NAD_1983_HARN_Adj_WI_Adams (deprecated) [369] GCS_NAD_1983_HARN_Adj_WI_Ashland (deprecated) [370] GCS_NAD_1983_HARN_Adj_WI_Barron (deprecated) [371] GCS_NAD_1983_HARN_Adj_WI_Bayfield (deprecated) [372] GCS_NAD_1983_HARN_Adj_WI_Brown (deprecated) [373] GCS_NAD_1983_HARN_Adj_WI_Buffalo (deprecated) [374] GCS_NAD_1983_HARN_Adj_WI_Burnett (deprecated) [375] GCS_NAD_1983_HARN_Adj_WI_Calumet (deprecated) [376] GCS_NAD_1983_HARN_Adj_WI_Chippewa [377] GCS_NAD_1983_HARN_Adj_WI_Clark (deprecated) [378] GCS_NAD_1983_HARN_Adj_WI_Columbia (deprecated) [379] GCS_NAD_1983_HARN_Adj_WI_Crawford (deprecated) [380] GCS_NAD_1983_HARN_Adj_WI_Dane (deprecated) [381] GCS_NAD_1983_HARN_Adj_WI_Dodge (deprecated) [382] GCS_NAD_1983_HARN_Adj_WI_Door (deprecated) [383] GCS_NAD_1983_HARN_Adj_WI_Douglas (deprecated) [384] GCS_NAD_1983_HARN_Adj_WI_Dunn (deprecated) [385] GCS_NAD_1983_HARN_Adj_WI_EauClaire (deprecated) [386] GCS_NAD_1983_HARN_Adj_WI_Florence (deprecated) [387] GCS_NAD_1983_HARN_Adj_WI_FondduLac (deprecated) [388] GCS_NAD_1983_HARN_Adj_WI_Forest (deprecated) [389] GCS_NAD_1983_HARN_Adj_WI_Grant (deprecated) [390] GCS_NAD_1983_HARN_Adj_WI_Green (deprecated) [391] GCS_NAD_1983_HARN_Adj_WI_GreenLake (deprecated) [392] GCS_NAD_1983_HARN_Adj_WI_Iowa (deprecated) [393] GCS_NAD_1983_HARN_Adj_WI_Iron (deprecated) [394] GCS_NAD_1983_HARN_Adj_WI_Jackson (deprecated) [395] GCS_NAD_1983_HARN_Adj_WI_Jefferson (deprecated) [396] GCS_NAD_1983_HARN_Adj_WI_Juneau (deprecated) [397] GCS_NAD_1983_HARN_Adj_WI_Kenosha (deprecated) [398] GCS_NAD_1983_HARN_Adj_WI_Kewaunee (deprecated) [399] GCS_NAD_1983_HARN_Adj_WI_LaCrosse (deprecated) [400] GCS_NAD_1983_HARN_Adj_WI_Lafayette (deprecated) [401] GCS_NAD_1983_HARN_Adj_WI_Langlade (deprecated) [402] GCS_NAD_1983_HARN_Adj_WI_Lincoln (deprecated) [403] GCS_NAD_1983_HARN_Adj_WI_Manitowoc (deprecated) [404] GCS_NAD_1983_HARN_Adj_WI_Marathon (deprecated) [405] GCS_NAD_1983_HARN_Adj_WI_Marinette (deprecated) [406] GCS_NAD_1983_HARN_Adj_WI_Marquette (deprecated) [407] GCS_NAD_1983_HARN_Adj_WI_Menominee (deprecated) [408] GCS_NAD_1983_HARN_Adj_WI_Milwaukee (deprecated) [409] GCS_NAD_1983_HARN_Adj_WI_Monroe (deprecated) [410] GCS_NAD_1983_HARN_Adj_WI_Oconto (deprecated) [411] GCS_NAD_1983_HARN_Adj_WI_Oneida (deprecated) [412] GCS_NAD_1983_HARN_Adj_WI_Outagamie (deprecated) [413] GCS_NAD_1983_HARN_Adj_WI_Ozaukee (deprecated) [414] GCS_NAD_1983_HARN_Adj_WI_Pepin (deprecated) [415] GCS_NAD_1983_HARN_Adj_WI_Pierce (deprecated) [416] GCS_NAD_1983_HARN_Adj_WI_Polk (deprecated) [417] GCS_NAD_1983_HARN_Adj_WI_Portage (deprecated) [418] GCS_NAD_1983_HARN_Adj_WI_Price (deprecated) [419] GCS_NAD_1983_HARN_Adj_WI_Racine (deprecated) [420] GCS_NAD_1983_HARN_Adj_WI_Richland (deprecated) [421] GCS_NAD_1983_HARN_Adj_WI_Rock (deprecated) [422] GCS_NAD_1983_HARN_Adj_WI_Rusk (deprecated) [423] GCS_NAD_1983_HARN_Adj_WI_Sauk (deprecated) [424] GCS_NAD_1983_HARN_Adj_WI_Sawyer (deprecated) [425] GCS_NAD_1983_HARN_Adj_WI_Shawano (deprecated) [426] GCS_NAD_1983_HARN_Adj_WI_Sheboygan (deprecated) [427] GCS_NAD_1983_HARN_Adj_WI_StCroix (deprecated) [428] GCS_NAD_1983_HARN_Adj_WI_Taylor (deprecated) [429] GCS_NAD_1983_HARN_Adj_WI_Trempealeau (deprecated) [430] GCS_NAD_1983_HARN_Adj_WI_Vernon (deprecated) [431] GCS_NAD_1983_HARN_Adj_WI_Vilas (deprecated) [432] GCS_NAD_1983_HARN_Adj_WI_Walworth (deprecated) [433] GCS_NAD_1983_HARN_Adj_WI_Washburn (deprecated) [434] GCS_NAD_1983_HARN_Adj_WI_Washington (deprecated) [435] GCS_NAD_1983_HARN_Adj_WI_Waukesha (deprecated) [436] GCS_NAD_1983_HARN_Adj_WI_Waupaca (deprecated) [437] GCS_NAD_1983_HARN_Adj_WI_Waushara (deprecated) [438] GCS_NAD_1983_HARN_Adj_WI_Winnebago (deprecated) [439] GCS_NAD_1983_HARN_Adj_WI_Wood (deprecated) [440] GCS_NAD_1983_MA11 (deprecated) [441] GCS_NAD_1983_MARP00 [442] GCS_NAD_1983_PA11 (deprecated) [443] GCS_NAD_1983_PACP00 [444] GCS_NZGD_2000 (deprecated) [445] GCS_Naiad_2000 [446] GCS_Nepal_Nagarkot (deprecated) [447] GCS_Neptune_2000 [448] GCS_Nereid_2000 [449] GCS_Oberon_2000 [450] GCS_Observatorio_Meteorologico_1939 [451] GCS_Observatorio_Meteorologico_1965 [452] GCS_Ocotepeque_1935 (deprecated) [453] GCS_Old_Hawaiian_Intl_1924 [454] GCS_Oman [455] GCS_Ophelia_2000 [456] GCS_PTRA08 (deprecated) [457] GCS_Pan_2000 [458] GCS_Pandora_2000 [459] GCS_Pasiphae_2000 [460] GCS_Phobos_2000 [461] GCS_Phoebe_2000 [462] GCS_Pico_de_Las_Nieves (deprecated) [463] GCS_Pitcairn_1967 (deprecated) [464] GCS_Pluto_2000 [465] GCS_Pohnpei [466] GCS_Point_58 (deprecated) [467] GCS_Portia_2000 [468] GCS_Porto_Santo_1936 (deprecated) [469] GCS_Prometheus_2000 [470] GCS_Proteus_2000 [471] GCS_Puck_2000 [472] GCS_REGVEN (deprecated) [473] GCS_RGF_1993 (deprecated) [474] GCS_Reunion_1947 (deprecated) [475] GCS_Rhea_2000 [476] GCS_Roma_1940 [477] GCS_Rosalind_2000 [478] GCS_S42_Hungary [479] GCS_Santo_DOS_1965 (deprecated) [480] GCS_Sao_Braz [481] GCS_Saturn_2000 [482] GCS_Selvagem_Grande_1938 (deprecated) [483] GCS_Sierra_Leone_1960 [484] GCS_Sinope_2000 [485] GCS_South_Asia_Singapore [486] GCS_Sphere_ARC_INFO [487] GCS_Sphere_EMEP [488] GCS_Sphere_GRS_1980_Mean_Radius [489] GCS_TWD_1967 (deprecated) [490] GCS_TWD_1997 (deprecated) [491] GCS_Telesto_2000 [492] GCS_Tern_Island_1961 (deprecated) [493] GCS_Tethys_2000 [494] GCS_Thalassa_2000 [495] GCS_Thebe_2000 [496] GCS_Titan_2000 [497] GCS_Titania_2000 [498] GCS_Tristan_1968 (deprecated) [499] GCS_Triton_2000 [500] GCS_Umbriel_2000 [501] GCS_Uranus_2000 [502] GCS_Venus_1985 [503] GCS_Venus_2000 [504] GCS_Viti_Levu_1916 (deprecated) [505] GCS_Voirol_1875 (deprecated) [506] GCS_Voirol_1875_Grad [507] GCS_Voirol_1879_Grad [508] GCS_Voirol_Unifie_1960 (deprecated) [509] GCS_Voirol_Unifie_1960 (deprecated) [510] GCS_Voirol_Unifie_1960_Degree (deprecated) [511] GCS_Voirol_Unifie_1960_Paris (deprecated) [512] GCS_WGS_1966 (deprecated) [513] GCS_WGS_1984_Major_Auxiliary_Sphere (deprecated) [514] GCS_Wake_Eniwetok_1960 (deprecated) [515] GCS_Wake_Island_1952 (deprecated) [516] GCS_Walbeck [517] GCS_Xrail84 [518] GDA2020 [519] GDA2020 [520] GDA94 [521] GDA94 [522] GDA94 [523] GDBD2009 [524] GDBD2009 [525] GDM2000 [526] GDM2000 [527] GGD [528] GGRS87 [529] GR96 [530] GR96 [531] GSK-2011 [532] GSK-2011 [533] Gambia [534] Gan 1970 [535] Gandajika [536] Gandajika 1970 [537] Garoua [538] Garoua [539] Grand Comoros [540] Greek [541] Greek (Athens) [542] Grenada 1953 [543] Guadeloupe 1948 [544] Guam 1963 [545] Gulshan 303 [546] Gusterberg (Ferro) [547] Guyane Francaise [548] HD1909 [549] HD72 [550] HTRS96 [551] HTRS96 [552] Hanoi 1972 [553] Hartebeesthoek94 [554] Hartebeesthoek94 [555] Hartebeesthoek94 [556] Helle 1954 [557] Herat North [558] Hito XVIII 1963 [559] Hjorsey 1955 [560] Hong Kong 1963 [561] Hong Kong 1963(67) [562] Hong Kong 1980 [563] Hong Kong Geodetic CS [564] Hong Kong Geodetic CS [565] Hu Tzu Shan 1950 [566] ID74 [567] IG05 Intermediate CRS [568] IG05 Intermediate CRS [569] IG05/12 Intermediate CRS [570] IG05/12 Intermediate CRS [571] IGC 1962 6th Parallel South [572] IGCB 1955 [573] IGD05 [574] IGD05 [575] IGD05 [576] IGD05 [577] IGD05/12 [578] IGD05/12 [579] IGD05/12 [580] IGD05/12 [581] IGM95 [582] IGM95 [583] IGN 1962 Kerguelen [584] IGN Astro 1960 [585] IGN53 Mare [586] IGN56 Lifou [587] IGN63 Hiva Oa [588] IGN72 Grand Terre [589] IGN72 Grande Terre [590] IGN72 Nuku Hiva [591] IGRS [592] IGRS [593] IGS00 [594] IGS00 [595] IGS05 [596] IGS05 [597] IGS08 [598] IGS08 [599] IGS14 [600] IGS14 [601] IGS14 [602] IGS97 [603] IGS97 [604] IGb00 [605] IGb00 [606] IGb08 [607] IGb08 [608] IKBD-92 [609] IRENET95 [610] IRENET95 [611] IRENET95 [612] ISN2004 [613] ISN2004 [614] ISN2016 [615] ISN2016 [616] ISN93 [617] ISN93 [618] ISN93 [619] ITRF2000 [620] ITRF2000 [621] ITRF2005 [622] ITRF2005 [623] ITRF2008 [624] ITRF2008 [625] ITRF2014 [626] ITRF2014 [627] ITRF2014 [628] ITRF88 [629] ITRF88 [630] ITRF89 [631] ITRF89 [632] ITRF90 [633] ITRF90 [634] ITRF91 [635] ITRF91 [636] ITRF92 [637] ITRF92 [638] ITRF93 [639] ITRF93 [640] ITRF94 [641] ITRF94 [642] ITRF96 [643] ITRF96 [644] ITRF97 [645] ITRF97 [646] Indian 1954 [647] Indian 1960 [648] Indian 1975 [649] Israel 1993 [650] Iwo Jima 1945 [651] JAD2001 [652] JAD2001 [653] JAD69 [654] JGD2000 [655] JGD2000 [656] JGD2000 [657] JGD2011 [658] JGD2011 [659] Jamaica 1875 [660] Johnston Island 1961 [661] Jouik 1961 [662] K0 1949 [663] KKJ [664] KOC [665] KUDAMS [666] Kalianpur 1880 [667] Kalianpur 1937 [668] Kalianpur 1962 [669] Kalianpur 1975 [670] Kandawala [671] Karbala 1979 [672] Kasai 1953 [673] Katanga 1955 [674] Kertau (RSO) [675] Kertau 1968 [676] Korea 2000 [677] Korea 2000 [678] Korean 1985 [679] Korean 1995 [680] Kousseri [681] Kusaie 1951 [682] Kyrg-06 [683] Kyrg-06 [684] LGD2006 [685] LGD2006 [686] LKS92 [687] LKS92 [688] LKS92 [689] LKS94 [690] LKS94 [691] LKS94 [692] LKS94 (ETRS89) [693] La Canoa [694] Lake [695] Lao 1993 [696] Lao 1993 [697] Lao 1997 [698] Lao 1997 [699] Le Pouce 1934 [700] Leigon [701] Liberia 1964 [702] Lisbon [703] Lisbon (Lisbon) [704] Lisbon 1890 [705] Lisbon 1890 (Lisbon) [706] Locodjo 1965 [707] Loma Quintana [708] Lome [709] Luxembourg 1930 [710] Luzon 1911 [711] M'poraloko [712] MACARIO SOLIS [713] MACARIO SOLIS [714] MAGNA-SIRGAS [715] MAGNA-SIRGAS [716] MARGEN [717] MARGEN [718] MGI [719] MGI (Ferro) [720] MGI 1901 [721] MOLDREF99 [722] MOLDREF99 [723] MOP78 [724] MTRF-2000 [725] MTRF-2000 [726] Macao 1920 [727] Macao 2008 [728] Macao 2008 [729] Madeira 1936 [730] Madrid 1870 (Madrid) [731] Madzansua [732] Mahe 1971 [733] Makassar [734] Makassar (Jakarta) [735] Malongo 1987 [736] Manoca [737] Manoca 1962 [738] Marcus Island 1952 [739] Mars_2000_(Sphere) [740] Marshall Islands 1960 [741] Martinique 1938 [742] Massawa [743] Maupiti 83 [744] Mauritania 1999 [745] Mauritania 1999 [746] Mauritania 1999 [747] Merchich [748] Mercury_2015 [749] Mexico ITRF2008 [750] Mexico ITRF2008 [751] Mexico ITRF92 [752] Mexico ITRF92 [753] Mhast [754] Mhast (offshore) [755] Mhast (onshore) [756] Mhast 1951 [757] Midway 1961 [758] Minna [759] Monte Mario [760] Monte Mario (Rome) [761] Montserrat 1958 [762] Moorea 87 [763] Mount Dillon [764] Moznet [765] Moznet [766] Moznet [767] NAD27 [768] NAD27 Michigan [769] NAD27(76) [770] NAD27(CGQ77) [771] NAD83 [772] NAD83(2011) [773] NAD83(2011) [774] NAD83(CORS96) [775] NAD83(CORS96) [776] NAD83(CSRS) [777] NAD83(CSRS) [778] NAD83(CSRS) [779] NAD83(CSRS)v2 [780] NAD83(CSRS)v2 [781] NAD83(CSRS)v3 [782] NAD83(CSRS)v3 [783] NAD83(CSRS)v4 [784] NAD83(CSRS)v4 [785] NAD83(CSRS)v5 [786] NAD83(CSRS)v5 [787] NAD83(CSRS)v6 [788] NAD83(CSRS)v6 [789] NAD83(CSRS)v7 [790] NAD83(CSRS)v7 [791] NAD83(CSRS96) [792] NAD83(CSRS96) [793] NAD83(CSRS98) [794] NAD83(FBN) [795] NAD83(FBN) [796] NAD83(FBN) [797] NAD83(HARN Corrected) [798] NAD83(HARN Corrected) [799] NAD83(HARN) [800] NAD83(HARN) [801] NAD83(HARN) [802] NAD83(MA11) [803] NAD83(MA11) [804] NAD83(NSRS2007) [805] NAD83(NSRS2007) [806] NAD83(PA11) [807] NAD83(PA11) [808] NDG (Paris) [809] NEA74 Noumea [810] NGN [811] NGO 1948 [812] NGO 1948 (Oslo) [813] NSWC 9Z-2 [814] NTF [815] NTF (Paris) [816] NZGD2000 [817] NZGD2000 [818] NZGD2000 [819] NZGD49 [820] Nahrwan 1934 [821] Nahrwan 1967 [822] Nakhl-e Ghanem [823] Naparima 1955 [824] Naparima 1972 [825] Nepal 1981 [826] New Beijing [827] Nord Sahara 1959 [828] Nord Sahara 1959 (Paris) [829] Nouakchott 1965 [830] ONGD14 [831] ONGD14 [832] OS(SN)80 [833] OSGB 1936 [834] OSGB70 [835] OSNI 1952 [836] Observatario [837] Ocotepeque 1935 [838] Old Hawaiian [839] PD/83 [840] PNG94 [841] PNG94 [842] POSGAR [843] POSGAR 2007 [844] POSGAR 2007 [845] POSGAR 94 [846] POSGAR 94 [847] POSGAR 98 [848] POSGAR 98 [849] POSGAR 98 [850] PRS92 [851] PRS92 [852] PSAD56 [853] PSD93 [854] PTRA08 [855] PTRA08 [856] PZ-90 [857] PZ-90 [858] PZ-90.02 [859] PZ-90.02 [860] PZ-90.11 [861] PZ-90.11 [862] Padang [863] Padang (Jakarta) [864] Palestine 1923 [865] Pampa del Castillo [866] Panama-Colon 1911 [867] Perroud 1950 [868] Peru96 [869] Peru96 [870] Petrels 1972 [871] Phoenix Islands 1966 [872] Pico de las Nieves 1984 [873] Pitcairn 1967 [874] Pitcairn 2006 [875] Point 58 [876] Pointe Noire [877] Popular Visualisation CRS [878] Porto Santo [879] Porto Santo 1995 [880] Principe [881] Puerto Rico [882] Pulkovo 1942 [883] Pulkovo 1942(58) [884] Pulkovo 1942(83) [885] Pulkovo 1995 [886] QND95 [887] Qatar 1948 [888] Qatar 1974 [889] Qornoq [890] Qornoq 1927 [891] RD/83 [892] RDN2008 [893] RDN2008 [894] REGCAN95 [895] REGCAN95 [896] REGVEN [897] REGVEN [898] REGVEN [899] RGAF09 [900] RGAF09 [901] RGAF09 (lon-lat) [902] RGAF09 (lon-lat) [903] RGF93 [904] RGF93 [905] RGF93 [906] RGF93 (lon-lat) [907] RGF93 (lon-lat) [908] RGFG95 [909] RGFG95 [910] RGFG95 [911] RGFG95 (lon-lat) [912] RGFG95 (lon-lat) [913] RGM04 [914] RGM04 [915] RGM04 (lon-lat) [916] RGM04 (lon-lat) [917] RGNC 1991 [918] RGNC 1991 [919] RGNC91-93 [920] RGNC91-93 [921] RGNC91-93 [922] RGPF [923] RGPF [924] RGR92 [925] RGR92 [926] RGR92 [927] RGR92 (lon-lat) [928] RGR92 (lon-lat) [929] RGRDC 2005 [930] RGRDC 2005 [931] RGSPM06 [932] RGSPM06 [933] RGSPM06 (lon-lat) [934] RGSPM06 (lon-lat) [935] RGTAAF07 [936] RGTAAF07 [937] RGTAAF07 (lon-lat) [938] RGTAAF07 (lon-lat) [939] RGTAAF07 (lon-lat) [940] RGWF96 [941] RGWF96 [942] RGWF96 (lon-lat) [943] RGWF96 (lon-lat) [944] RRAF 1991 [945] RRAF 1991 [946] RRAF 1991 [947] RRAF 1991 [948] RRAF 1991 [949] RSRGD2000 [950] RSRGD2000 [951] RT38 [952] RT38 (Stockholm) [953] RT90 [954] Rassadiran [955] Reunion 1947 [956] Reykjavik 1900 [957] S-JTSK [958] S-JTSK (Ferro) [959] S-JTSK [JTSK03] [960] S-JTSK/05 [961] S-JTSK/05 (Ferro) [962] SAD69 [963] SAD69 [964] SAD69(96) [965] SHGD2015 [966] SHGD2015 [967] SIGD61 [968] SIRGAS 1995 [969] SIRGAS 1995 [970] SIRGAS 1995 [971] SIRGAS 2000 [972] SIRGAS 2000 [973] SIRGAS-Chile [974] SIRGAS-Chile [975] SIRGAS-ROU98 [976] SIRGAS-ROU98 [977] SIRGAS_ES2007.8 [978] SIRGAS_ES2007.8 [979] SLD99 [980] SRB_ETRS89 [981] SRB_ETRS89 [982] SREF98 [983] SREF98 [984] ST71 Belep [985] ST84 Ile des Pins [986] ST87 Ouvea [987] ST87 Ouvea [988] SVY21 [989] SWEREF99 [990] SWEREF99 [991] SWEREF99 [992] Saint Pierre et Miquelon 1950 [993] Samboja [994] Santo 1965 [995] Sao Tome [996] Sapper Hill 1943 [997] Schwarzeck [998] Scoresbysund 1952 [999] Segara [1000] Segara (Jakarta) [1001] Segora [1002] Selvagem Grande [1003] Serindung [1004] Sibun Gorge 1922 [1005] Sierra Leone 1924 [1006] Sierra Leone 1968 [1007] Slovenia 1996 [1008] Slovenia 1996 [1009] Solomon 1968 [1010] South East Island 1943 [1011] South Georgia 1968 [1012] South Yemen [1013] St. George Island [1014] St. Helena Tritan [1015] St. Helena Tritan [1016] St. Kitts 1955 [1017] St. Lawrence Island [1018] St. Lucia 1955 [1019] St. Paul Island [1020] St. Stephen (Ferro) [1021] St. Vincent 1945 [1022] Sudan [1023] Sun_2015 [1024] TC(1948) [1025] TGD2005 [1026] TGD2005 [1027] TM65 [1028] TM75 [1029] TUREF [1030] TUREF [1031] TWD67 [1032] TWD97 [1033] TWD97 [1034] Tahaa 54 [1035] Tahiti 52 [1036] Tahiti 79 [1037] Tananarive [1038] Tananarive (Paris) [1039] Tern Island 1961 [1040] Tete [1041] Timbalai 1948 [1042] Tokyo [1043] Tokyo 1892 [1044] Trinidad 1903 [1045] Tristan 1968 [1046] UCS-2000 [1047] UCS-2000 [1048] Unknown datum based upon the Airy 1830 ellipsoid [1049] Unknown datum based upon the Airy Modified 1849 ellipsoid [1050] Unknown datum based upon the Australian National Spheroid [1051] Unknown datum based upon the Authalic Sphere [1052] Unknown datum based upon the Average Terrestrial System 1977 ellipsoid [1053] Unknown datum based upon the Bessel 1841 ellipsoid [1054] Unknown datum based upon the Bessel Modified ellipsoid [1055] Unknown datum based upon the Bessel Namibia ellipsoid [1056] Unknown datum based upon the Clarke 1858 ellipsoid [1057] Unknown datum based upon the Clarke 1866 Michigan ellipsoid [1058] Unknown datum based upon the Clarke 1866 ellipsoid [1059] Unknown datum based upon the Clarke 1880 (Arc) ellipsoid [1060] Unknown datum based upon the Clarke 1880 (Benoit) ellipsoid [1061] Unknown datum based upon the Clarke 1880 (IGN) ellipsoid [1062] Unknown datum based upon the Clarke 1880 (RGS) ellipsoid [1063] Unknown datum based upon the Clarke 1880 (SGA 1922) ellipsoid [1064] Unknown datum based upon the Clarke 1880 ellipsoid [1065] Unknown datum based upon the Everest (1830 Definition) ellipsoid [1066] Unknown datum based upon the Everest 1830 (1937 Adjustment) ellipsoid [1067] Unknown datum based upon the Everest 1830 (1962 Definition) ellipsoid [1068] Unknown datum based upon the Everest 1830 (1967 Definition) ellipsoid [1069] Unknown datum based upon the Everest 1830 (1975 Definition) ellipsoid [1070] Unknown datum based upon the Everest 1830 Modified ellipsoid [1071] Unknown datum based upon the GEM 10C ellipsoid [1072] Unknown datum based upon the GRS 1967 ellipsoid [1073] Unknown datum based upon the GRS 1980 ellipsoid [1074] Unknown datum based upon the Helmert 1906 ellipsoid [1075] Unknown datum based upon the Indonesian National Spheroid [1076] Unknown datum based upon the International 1924 ellipsoid [1077] Unknown datum based upon the Krassowsky 1940 ellipsoid [1078] Unknown datum based upon the NWL 9D ellipsoid [1079] Unknown datum based upon the OSU86F ellipsoid [1080] Unknown datum based upon the OSU91A ellipsoid [1081] Unknown datum based upon the Plessis 1817 ellipsoid [1082] Unknown datum based upon the Struve 1860 ellipsoid [1083] Unknown datum based upon the WGS 72 ellipsoid [1084] Unknown datum based upon the WGS 84 ellipsoid [1085] Unknown datum based upon the War Office ellipsoid [1086] Unspecified datum based upon the Clarke 1866 Authalic Sphere [1087] Unspecified datum based upon the GRS 1980 Authalic Sphere [1088] Unspecified datum based upon the Hughes 1980 ellipsoid [1089] Unspecified datum based upon the International 1924 Authalic Sphere [1090] VN-2000 [1091] Vanua Levu 1915 [1092] Vientiane 1982 [1093] Viti Levu 1912 [1094] Viti Levu 1916 [1095] Voirol 1875 [1096] Voirol 1875 (Paris) [1097] Voirol 1879 [1098] Voirol 1879 (Paris) [1099] WGS 66 [1100] WGS 66 [1101] WGS 72 [1102] WGS 72 [1103] WGS 72BE [1104] WGS 72BE [1105] WGS 84 [1106] WGS 84 [1107] WGS 84 [1108] WGS 84 (3D) [1109] WGS 84 (G1150) [1110] WGS 84 (G1674) [1111] WGS 84 (G1762) [1112] WGS 84 (G730) [1113] WGS 84 (G873) [1114] WGS 84 (Transit) [1115] WGS_1984_(G1150) [1116] WGS_1984_(G1674) [1117] WGS_1984_(G1762) [1118] WGS_1984_(G730) [1119] WGS_1984_(G873) [1120] WGS_1984_(Transit) [1121] Wake Island 1952 [1122] Xian 1980 [1123] Yacare [1124] Yemen NGN96 [1125] Yemen NGN96 [1126] Yemen NGN96 [1127] Yoff [1128] Zanderij [1129] fk89 Default: 0
    - CRS_PROJCS [`choice`] : Projected Coordinate Systems. Available Choices: [0] <select> [1] AGD66 / ACT Standard Grid [2] AGD66 / AMG zone 48 (deprecated) [3] AGD66 / AMG zone 49 [4] AGD66 / AMG zone 50 [5] AGD66 / AMG zone 51 [6] AGD66 / AMG zone 52 [7] AGD66 / AMG zone 53 [8] AGD66 / AMG zone 54 [9] AGD66 / AMG zone 55 [10] AGD66 / AMG zone 56 [11] AGD66 / AMG zone 57 [12] AGD66 / AMG zone 58 [13] AGD66 / Vicgrid66 [14] AGD84 / AMG zone 48 (deprecated) [15] AGD84 / AMG zone 49 [16] AGD84 / AMG zone 50 [17] AGD84 / AMG zone 51 [18] AGD84 / AMG zone 52 [19] AGD84 / AMG zone 53 [20] AGD84 / AMG zone 54 [21] AGD84 / AMG zone 55 [22] AGD84 / AMG zone 56 [23] AGD84 / AMG zone 57 (deprecated) [24] AGD84 / AMG zone 58 (deprecated) [25] AGD_1966_ACT_Grid_AGC_Zone [26] AGD_1966_ISG_54_2 [27] AGD_1966_ISG_54_3 [28] AGD_1966_ISG_55_1 [29] AGD_1966_ISG_55_2 [30] AGD_1966_ISG_55_3 [31] AGD_1966_ISG_56_1 [32] AGD_1966_ISG_56_2 [33] AGD_1966_ISG_56_3 [34] AGD_1966_VICGRID (deprecated) [35] ATF (Paris) / Nord de Guerre [36] ATS77 / MTM Nova Scotia zone 4 [37] ATS77 / MTM Nova Scotia zone 5 [38] ATS77 / New Brunswick Stereographic (ATS77) [39] ATS77 / Prince Edward Isl. Stereographic (ATS77) [40] ATS77 / UTM zone 19N [41] ATS77 / UTM zone 20N [42] Abidjan 1987 / TM 5 NW [43] Abidjan 1987 / UTM zone 29N [44] Abidjan 1987 / UTM zone 30N [45] Accra / Ghana National Grid [46] Accra / TM 1 NW [47] Adindan / UTM zone 35N [48] Adindan / UTM zone 36N [49] Adindan / UTM zone 37N [50] Adindan / UTM zone 38N [51] Afgooye / UTM zone 38N [52] Afgooye / UTM zone 39N [53] Africa_Albers_Equal_Area_Conic [54] Africa_Equidistant_Conic [55] Africa_Lambert_Conformal_Conic [56] Africa_Sinusoidal [57] Ain el Abd / Aramco Lambert [58] Ain el Abd / Bahrain Grid [59] Ain el Abd / UTM zone 36N [60] Ain el Abd / UTM zone 37N [61] Ain el Abd / UTM zone 38N [62] Ain el Abd / UTM zone 39N [63] Ain el Abd / UTM zone 40N [64] Ain_el_Abd_1970_Aramco_Lambert_2 [65] Albanian 1987 / Gauss Kruger zone 4 (deprecated) [66] Albanian 1987 / Gauss-Kruger zone 4 [67] American Samoa 1962 / American Samoa Lambert [68] American Samoa 1962 / American Samoa Lambert (deprecated) [69] American Samoa 1962 / American Samoa Lambert (deprecated) [70] American_Samoa_1962_StatePlane_American_Samoa_FIPS_5300 [71] American_Samoa_1962_UTM_Zone_2S [72] Amersfoort / RD New [73] Amersfoort / RD Old [74] Anguilla 1957 / British West Indies Grid [75] Antigua 1943 / British West Indies Grid [76] Aratu / UTM zone 22S [77] Aratu / UTM zone 23S [78] Aratu / UTM zone 24S [79] Aratu / UTM zone 25S [80] Arc 1950 / UTM zone 34S [81] Arc 1950 / UTM zone 35S [82] Arc 1950 / UTM zone 36S [83] Arc 1960 / UTM zone 35N [84] Arc 1960 / UTM zone 35S [85] Arc 1960 / UTM zone 36N [86] Arc 1960 / UTM zone 36S [87] Arc 1960 / UTM zone 37N [88] Arc 1960 / UTM zone 37S [89] Asia_Lambert_Conformal_Conic [90] Asia_North_Albers_Equal_Area_Conic [91] Asia_North_Equidistant_Conic [92] Asia_North_Lambert_Conformal_Conic [93] Asia_South_Albers_Equal_Area_Conic [94] Asia_South_Equidistant_Conic [95] Asia_South_Lambert_Conformal_Conic [96] Astro DOS 71 / SHLG71 [97] Astro DOS 71 / UTM zone 30S [98] Azores Central 1948 / UTM zone 26N [99] Azores Central 1995 / UTM zone 26N [100] Azores Occidental 1939 / UTM zone 25N [101] Azores Oriental 1940 / UTM zone 26N [102] Azores Oriental 1995 / UTM zone 26N [103] BDA2000 / Bermuda 2000 National Grid [104] BGS2005 / CCS2005 [105] BGS2005 / UTM zone 34N [106] BGS2005 / UTM zone 34N (N-E) [107] BGS2005 / UTM zone 35N [108] BGS2005 / UTM zone 35N (N-E) [109] BGS2005 / UTM zone 36N [110] BUTM 2010 [111] Bab_South_Palau_Azimuthal_Equidistant [112] Barbados 1938 / Barbados National Grid [113] Barbados 1938 / British West Indies Grid [114] Batavia (Jakarta) / NEIEZ [115] Batavia (Jakarta) / NEIEZ (deprecated) [116] Batavia / NEIEZ [117] Batavia / TM 109 SE [118] Batavia / UTM zone 48S [119] Batavia / UTM zone 49S [120] Batavia / UTM zone 50S [121] Beduaram / TM 13 NE [122] Beijing 1954 / 3-degree Gauss-Kruger CM 102E [123] Beijing 1954 / 3-degree Gauss-Kruger CM 105E [124] Beijing 1954 / 3-degree Gauss-Kruger CM 108E [125] Beijing 1954 / 3-degree Gauss-Kruger CM 111E [126] Beijing 1954 / 3-degree Gauss-Kruger CM 114E [127] Beijing 1954 / 3-degree Gauss-Kruger CM 117E [128] Beijing 1954 / 3-degree Gauss-Kruger CM 120E [129] Beijing 1954 / 3-degree Gauss-Kruger CM 123E [130] Beijing 1954 / 3-degree Gauss-Kruger CM 126E [131] Beijing 1954 / 3-degree Gauss-Kruger CM 129E [132] Beijing 1954 / 3-degree Gauss-Kruger CM 132E [133] Beijing 1954 / 3-degree Gauss-Kruger CM 135E [134] Beijing 1954 / 3-degree Gauss-Kruger CM 75E [135] Beijing 1954 / 3-degree Gauss-Kruger CM 78E [136] Beijing 1954 / 3-degree Gauss-Kruger CM 81E [137] Beijing 1954 / 3-degree Gauss-Kruger CM 84E [138] Beijing 1954 / 3-degree Gauss-Kruger CM 87E [139] Beijing 1954 / 3-degree Gauss-Kruger CM 90E [140] Beijing 1954 / 3-degree Gauss-Kruger CM 93E [141] Beijing 1954 / 3-degree Gauss-Kruger CM 96E [142] Beijing 1954 / 3-degree Gauss-Kruger CM 99E [143] Beijing 1954 / 3-degree Gauss-Kruger zone 25 [144] Beijing 1954 / 3-degree Gauss-Kruger zone 26 [145] Beijing 1954 / 3-degree Gauss-Kruger zone 27 [146] Beijing 1954 / 3-degree Gauss-Kruger zone 28 [147] Beijing 1954 / 3-degree Gauss-Kruger zone 29 [148] Beijing 1954 / 3-degree Gauss-Kruger zone 30 [149] Beijing 1954 / 3-degree Gauss-Kruger zone 31 [150] Beijing 1954 / 3-degree Gauss-Kruger zone 32 [151] Beijing 1954 / 3-degree Gauss-Kruger zone 33 [152] Beijing 1954 / 3-degree Gauss-Kruger zone 34 [153] Beijing 1954 / 3-degree Gauss-Kruger zone 35 [154] Beijing 1954 / 3-degree Gauss-Kruger zone 36 [155] Beijing 1954 / 3-degree Gauss-Kruger zone 37 [156] Beijing 1954 / 3-degree Gauss-Kruger zone 38 [157] Beijing 1954 / 3-degree Gauss-Kruger zone 39 [158] Beijing 1954 / 3-degree Gauss-Kruger zone 40 [159] Beijing 1954 / 3-degree Gauss-Kruger zone 41 [160] Beijing 1954 / 3-degree Gauss-Kruger zone 42 [161] Beijing 1954 / 3-degree Gauss-Kruger zone 43 [162] Beijing 1954 / 3-degree Gauss-Kruger zone 44 [163] Beijing 1954 / 3-degree Gauss-Kruger zone 45 [164] Beijing 1954 / Gauss-Kruger 13N (deprecated) [165] Beijing 1954 / Gauss-Kruger 14N (deprecated) [166] Beijing 1954 / Gauss-Kruger 15N (deprecated) [167] Beijing 1954 / Gauss-Kruger 16N (deprecated) [168] Beijing 1954 / Gauss-Kruger 17N (deprecated) [169] Beijing 1954 / Gauss-Kruger 18N (deprecated) [170] Beijing 1954 / Gauss-Kruger 19N (deprecated) [171] Beijing 1954 / Gauss-Kruger 20N (deprecated) [172] Beijing 1954 / Gauss-Kruger 21N (deprecated) [173] Beijing 1954 / Gauss-Kruger 22N (deprecated) [174] Beijing 1954 / Gauss-Kruger 23N (deprecated) [175] Beijing 1954 / Gauss-Kruger CM 105E [176] Beijing 1954 / Gauss-Kruger CM 111E [177] Beijing 1954 / Gauss-Kruger CM 117E [178] Beijing 1954 / Gauss-Kruger CM 123E [179] Beijing 1954 / Gauss-Kruger CM 129E [180] Beijing 1954 / Gauss-Kruger CM 135E [181] Beijing 1954 / Gauss-Kruger CM 75E [182] Beijing 1954 / Gauss-Kruger CM 81E [183] Beijing 1954 / Gauss-Kruger CM 87E [184] Beijing 1954 / Gauss-Kruger CM 93E [185] Beijing 1954 / Gauss-Kruger CM 99E [186] Beijing 1954 / Gauss-Kruger zone 13 [187] Beijing 1954 / Gauss-Kruger zone 14 [188] Beijing 1954 / Gauss-Kruger zone 15 [189] Beijing 1954 / Gauss-Kruger zone 16 [190] Beijing 1954 / Gauss-Kruger zone 17 [191] Beijing 1954 / Gauss-Kruger zone 18 [192] Beijing 1954 / Gauss-Kruger zone 19 [193] Beijing 1954 / Gauss-Kruger zone 20 [194] Beijing 1954 / Gauss-Kruger zone 21 [195] Beijing 1954 / Gauss-Kruger zone 22 [196] Beijing 1954 / Gauss-Kruger zone 23 [197] Belge 1950 (Brussels) / Belge Lambert 50 [198] Belge 1972 / Belge Lambert 72 [199] Belge 1972 / Belgian Lambert 72 [200] Belge_Lambert_1972_bad_FE_FN [201] Belge_Lambert_2008 (deprecated) [202] Bermuda 1957 / UTM zone 20N [203] Bermuda_2000_National_Grid (deprecated) [204] Bern 1898 (Bern) / LV03C [205] Bissau / UTM zone 28N [206] Bogota 1975 / Colombia Bogota zone [207] Bogota 1975 / Colombia Bogota zone (deprecated) [208] Bogota 1975 / Colombia East [209] Bogota 1975 / Colombia East (deprecated) [210] Bogota 1975 / Colombia East Central zone [211] Bogota 1975 / Colombia East Central zone (deprecated) [212] Bogota 1975 / Colombia West zone [213] Bogota 1975 / Colombia West zone (deprecated) [214] Bogota 1975 / UTM zone 17N (deprecated) [215] Bogota 1975 / UTM zone 18N [216] Bogota_Ciudad_Bogota [217] CGCS2000 / 3-degree Gauss-Kruger CM 102E [218] CGCS2000 / 3-degree Gauss-Kruger CM 105E [219] CGCS2000 / 3-degree Gauss-Kruger CM 108E [220] CGCS2000 / 3-degree Gauss-Kruger CM 111E [221] CGCS2000 / 3-degree Gauss-Kruger CM 114E [222] CGCS2000 / 3-degree Gauss-Kruger CM 117E [223] CGCS2000 / 3-degree Gauss-Kruger CM 120E [224] CGCS2000 / 3-degree Gauss-Kruger CM 123E [225] CGCS2000 / 3-degree Gauss-Kruger CM 126E [226] CGCS2000 / 3-degree Gauss-Kruger CM 129E [227] CGCS2000 / 3-degree Gauss-Kruger CM 132E [228] CGCS2000 / 3-degree Gauss-Kruger CM 135E [229] CGCS2000 / 3-degree Gauss-Kruger CM 75E [230] CGCS2000 / 3-degree Gauss-Kruger CM 78E [231] CGCS2000 / 3-degree Gauss-Kruger CM 81E [232] CGCS2000 / 3-degree Gauss-Kruger CM 84E [233] CGCS2000 / 3-degree Gauss-Kruger CM 87E [234] CGCS2000 / 3-degree Gauss-Kruger CM 90E [235] CGCS2000 / 3-degree Gauss-Kruger CM 93E [236] CGCS2000 / 3-degree Gauss-Kruger CM 96E [237] CGCS2000 / 3-degree Gauss-Kruger CM 99E [238] CGCS2000 / 3-degree Gauss-Kruger zone 25 [239] CGCS2000 / 3-degree Gauss-Kruger zone 26 [240] CGCS2000 / 3-degree Gauss-Kruger zone 27 [241] CGCS2000 / 3-degree Gauss-Kruger zone 28 [242] CGCS2000 / 3-degree Gauss-Kruger zone 29 [243] CGCS2000 / 3-degree Gauss-Kruger zone 30 [244] CGCS2000 / 3-degree Gauss-Kruger zone 31 [245] CGCS2000 / 3-degree Gauss-Kruger zone 32 [246] CGCS2000 / 3-degree Gauss-Kruger zone 33 [247] CGCS2000 / 3-degree Gauss-Kruger zone 34 [248] CGCS2000 / 3-degree Gauss-Kruger zone 35 [249] CGCS2000 / 3-degree Gauss-Kruger zone 36 [250] CGCS2000 / 3-degree Gauss-Kruger zone 37 [251] CGCS2000 / 3-degree Gauss-Kruger zone 38 [252] CGCS2000 / 3-degree Gauss-Kruger zone 39 [253] CGCS2000 / 3-degree Gauss-Kruger zone 40 [254] CGCS2000 / 3-degree Gauss-Kruger zone 41 [255] CGCS2000 / 3-degree Gauss-Kruger zone 42 [256] CGCS2000 / 3-degree Gauss-Kruger zone 43 [257] CGCS2000 / 3-degree Gauss-Kruger zone 44 [258] CGCS2000 / 3-degree Gauss-Kruger zone 45 [259] CGCS2000 / Gauss-Kruger CM 105E [260] CGCS2000 / Gauss-Kruger CM 111E [261] CGCS2000 / Gauss-Kruger CM 117E [262] CGCS2000 / Gauss-Kruger CM 123E [263] CGCS2000 / Gauss-Kruger CM 129E [264] CGCS2000 / Gauss-Kruger CM 135E [265] CGCS2000 / Gauss-Kruger CM 75E [266] CGCS2000 / Gauss-Kruger CM 81E [267] CGCS2000 / Gauss-Kruger CM 87E [268] CGCS2000 / Gauss-Kruger CM 93E [269] CGCS2000 / Gauss-Kruger CM 99E [270] CGCS2000 / Gauss-Kruger zone 13 [271] CGCS2000 / Gauss-Kruger zone 14 [272] CGCS2000 / Gauss-Kruger zone 15 [273] CGCS2000 / Gauss-Kruger zone 16 [274] CGCS2000 / Gauss-Kruger zone 17 [275] CGCS2000 / Gauss-Kruger zone 18 [276] CGCS2000 / Gauss-Kruger zone 19 [277] CGCS2000 / Gauss-Kruger zone 20 [278] CGCS2000 / Gauss-Kruger zone 21 [279] CGCS2000 / Gauss-Kruger zone 22 [280] CGCS2000 / Gauss-Kruger zone 23 [281] CGRS93 / Cyprus Local Transverse Mercator [282] CGRS_1993_LTM [283] CH1903 / LV03 [284] CH1903 / LV03C-G [285] CH1903+ / LV95 [286] CI1971 / Chatham Islands Map Grid [287] CI1979 / Chatham Islands Map Grid [288] COB_NAD83_2007 [289] CR-SIRGAS / CRTM05 [290] CR-SIRGAS / UTM zone 16N [291] CR-SIRGAS / UTM zone 17N [292] CR05 / CRTM05 [293] CRTM05 (deprecated) [294] CSG67 / UTM zone 21N [295] CSG67 / UTM zone 22N [296] Cadastre 1997 / UTM zone 38S [297] Cadastre 1997 / UTM zone 38S (deprecated) [298] Camacupa / TM 11.30 SE [299] Camacupa / TM 12 SE [300] Camacupa / UTM zone 32S [301] Camacupa / UTM zone 33S [302] Campo Inchauspe / Argentina 1 [303] Campo Inchauspe / Argentina 2 [304] Campo Inchauspe / Argentina 3 [305] Campo Inchauspe / Argentina 4 [306] Campo Inchauspe / Argentina 5 [307] Campo Inchauspe / Argentina 6 [308] Campo Inchauspe / Argentina 7 [309] Campo Inchauspe / UTM zone 19S [310] Campo Inchauspe / UTM zone 20S [311] Canada_Albers_Equal_Area_Conic [312] Canada_Lambert_Conformal_Conic [313] Cape / Lo15 [314] Cape / Lo17 [315] Cape / Lo19 [316] Cape / Lo21 [317] Cape / Lo23 [318] Cape / Lo25 [319] Cape / Lo27 [320] Cape / Lo29 [321] Cape / Lo31 [322] Cape / Lo33 [323] Cape / UTM zone 34S [324] Cape / UTM zone 35S [325] Cape / UTM zone 36S (deprecated) [326] Cape_Lo15 [327] Cape_Lo17 [328] Cape_Lo19 [329] Cape_Lo21 [330] Cape_Lo23 [331] Cape_Lo25 [332] Cape_Lo27 [333] Cape_Lo29 [334] Cape_Lo31 [335] Cape_Lo33 [336] Carthage (Paris) / Tunisia Mining Grid [337] Carthage / Nord Tunisie [338] Carthage / Sud Tunisie [339] Carthage / TM 11 NE [340] Carthage / UTM zone 32N [341] Cayman Islands National Grid 2011 [342] Cayman Islands National Grid 2011 (deprecated) [343] Chatham_Islands_1979_Map_Grid (deprecated) [344] Chos Malal 1914 / Argentina 2 [345] Chua / UTM zone 23S [346] Chua_UTM_Zone_23S (deprecated) [347] Colombia_West_West_Zone [348] Combani 1950 / UTM zone 38S [349] Conakry 1905 / UTM zone 28N [350] Conakry 1905 / UTM zone 29N [351] Corrego Alegre 1961 / UTM zone 21S [352] Corrego Alegre 1961 / UTM zone 22S [353] Corrego Alegre 1961 / UTM zone 23S [354] Corrego Alegre 1961 / UTM zone 24S [355] Corrego Alegre 1970-72 / UTM zone 21S [356] Corrego Alegre 1970-72 / UTM zone 22S [357] Corrego Alegre 1970-72 / UTM zone 23S [358] Corrego Alegre 1970-72 / UTM zone 24S [359] Corrego Alegre 1970-72 / UTM zone 25S [360] County ST74 [361] D48_Slovenia_TM [362] DB_REF / 3-degree Gauss-Kruger zone 2 (E-N) [363] DB_REF / 3-degree Gauss-Kruger zone 3 (E-N) [364] DB_REF / 3-degree Gauss-Kruger zone 4 (E-N) [365] DB_REF / 3-degree Gauss-Kruger zone 5 (E-N) [366] DGN95 / Indonesia TM-3 zone 46.2 [367] DGN95 / Indonesia TM-3 zone 47.1 [368] DGN95 / Indonesia TM-3 zone 47.2 [369] DGN95 / Indonesia TM-3 zone 48.1 [370] DGN95 / Indonesia TM-3 zone 48.2 [371] DGN95 / Indonesia TM-3 zone 49.1 [372] DGN95 / Indonesia TM-3 zone 49.2 [373] DGN95 / Indonesia TM-3 zone 50.1 [374] DGN95 / Indonesia TM-3 zone 50.2 [375] DGN95 / Indonesia TM-3 zone 51.1 [376] DGN95 / Indonesia TM-3 zone 51.2 [377] DGN95 / Indonesia TM-3 zone 52.1 [378] DGN95 / Indonesia TM-3 zone 52.2 [379] DGN95 / Indonesia TM-3 zone 53.1 [380] DGN95 / Indonesia TM-3 zone 53.2 [381] DGN95 / Indonesia TM-3 zone 54.1 [382] DGN95 / UTM zone 46N [383] DGN95 / UTM zone 47N [384] DGN95 / UTM zone 47S [385] DGN95 / UTM zone 48N [386] DGN95 / UTM zone 48S [387] DGN95 / UTM zone 49N [388] DGN95 / UTM zone 49S [389] DGN95 / UTM zone 50N [390] DGN95 / UTM zone 50S [391] DGN95 / UTM zone 51N [392] DGN95 / UTM zone 51S [393] DGN95 / UTM zone 52N [394] DGN95 / UTM zone 52S [395] DGN95 / UTM zone 53S [396] DGN95 / UTM zone 54S [397] DHDN / 3-degree Gauss zone 1 (deprecated) [398] DHDN / 3-degree Gauss zone 2 (deprecated) [399] DHDN / 3-degree Gauss zone 3 (deprecated) [400] DHDN / 3-degree Gauss zone 4 (deprecated) [401] DHDN / 3-degree Gauss zone 5 (deprecated) [402] DHDN / 3-degree Gauss-Kruger zone 1 [403] DHDN / 3-degree Gauss-Kruger zone 1 (E-N) [404] DHDN / 3-degree Gauss-Kruger zone 2 [405] DHDN / 3-degree Gauss-Kruger zone 2 (E-N) [406] DHDN / 3-degree Gauss-Kruger zone 3 [407] DHDN / 3-degree Gauss-Kruger zone 3 (E-N) [408] DHDN / 3-degree Gauss-Kruger zone 4 [409] DHDN / 3-degree Gauss-Kruger zone 4 (E-N) [410] DHDN / 3-degree Gauss-Kruger zone 5 [411] DHDN / 3-degree Gauss-Kruger zone 5 (E-N) [412] DHDN / Soldner Berlin [413] DRUKREF 03 / Bhutan National Grid [414] DRUKREF 03 / Bumthang TM [415] DRUKREF 03 / Chhukha TM [416] DRUKREF 03 / Dagana TM [417] DRUKREF 03 / Gasa TM [418] DRUKREF 03 / Ha TM [419] DRUKREF 03 / Lhuentse TM [420] DRUKREF 03 / Mongar TM [421] DRUKREF 03 / Paro TM [422] DRUKREF 03 / Pemagatshel TM [423] DRUKREF 03 / Punakha TM [424] DRUKREF 03 / Samdrup Jongkhar TM [425] DRUKREF 03 / Samtse TM [426] DRUKREF 03 / Sarpang TM [427] DRUKREF 03 / Thimphu TM [428] DRUKREF 03 / Trashigang TM [429] DRUKREF 03 / Trongsa TM [430] DRUKREF 03 / Tsirang TM [431] DRUKREF 03 / Wangdue Phodrang TM [432] DRUKREF 03 / Yangtse TM [433] DRUKREF 03 / Zhemgang TM [434] Dabola 1981 / UTM zone 28N [435] Dabola 1981 / UTM zone 28N (deprecated) [436] Dabola 1981 / UTM zone 29N [437] Dabola 1981 / UTM zone 29N (deprecated) [438] Datum 73 / Modified Portuguese Grid [439] Datum 73 / Modified Portuguese Grid (deprecated) [440] Datum 73 / UTM zone 29N [441] Datum_73_Hayford_Gauss_IGeoE [442] Datum_73_Hayford_Gauss_IPCC [443] Dealul Piscului 1930 / Stereo 33 [444] Dealul Piscului 1970/ Stereo 70 (deprecated) [445] Deir ez Zor / Levant Stereographic [446] Deir ez Zor / Levant Zone [447] Deir ez Zor / Syria Lambert [448] Dominica 1945 / British West Indies Grid [449] Douala / UTM zone 32N (deprecated) [450] Douala 1948 / AEF west [451] Douala 1948 / AOF west (deprecated) [452] ED50 / 3-degree Gauss-Kruger zone 10 [453] ED50 / 3-degree Gauss-Kruger zone 11 [454] ED50 / 3-degree Gauss-Kruger zone 12 [455] ED50 / 3-degree Gauss-Kruger zone 13 [456] ED50 / 3-degree Gauss-Kruger zone 14 [457] ED50 / 3-degree Gauss-Kruger zone 15 [458] ED50 / 3-degree Gauss-Kruger zone 9 [459] ED50 / France EuroLambert (deprecated) [460] ED50 / Iraq National Grid [461] ED50 / Jordan TM [462] ED50 / SPBA LCC [463] ED50 / TM 0 N [464] ED50 / TM 5 NE [465] ED50 / TM 6 NE [466] ED50 / TM27 [467] ED50 / TM30 [468] ED50 / TM33 [469] ED50 / TM36 [470] ED50 / TM39 [471] ED50 / TM42 [472] ED50 / TM45 [473] ED50 / UTM zone 28N [474] ED50 / UTM zone 29N [475] ED50 / UTM zone 30N [476] ED50 / UTM zone 31N [477] ED50 / UTM zone 32N [478] ED50 / UTM zone 33N [479] ED50 / UTM zone 34N [480] ED50 / UTM zone 35N [481] ED50 / UTM zone 36N [482] ED50 / UTM zone 37N [483] ED50 / UTM zone 38N [484] ED50(ED77) / UTM zone 38N [485] ED50(ED77) / UTM zone 39N [486] ED50(ED77) / UTM zone 40N [487] ED50(ED77) / UTM zone 41N [488] ED_1950_Turkey_10 [489] ED_1950_Turkey_10 (deprecated) [490] ED_1950_Turkey_11 [491] ED_1950_Turkey_11 (deprecated) [492] ED_1950_Turkey_12 [493] ED_1950_Turkey_12 (deprecated) [494] ED_1950_Turkey_13 [495] ED_1950_Turkey_13 (deprecated) [496] ED_1950_Turkey_14 [497] ED_1950_Turkey_14 (deprecated) [498] ED_1950_Turkey_15 [499] ED_1950_Turkey_15 (deprecated) [500] ED_1950_Turkey_9 [501] ED_1950_Turkey_9 (deprecated) [502] ELD79 / Libya zone 10 [503] ELD79 / Libya zone 11 [504] ELD79 / Libya zone 12 [505] ELD79 / Libya zone 13 [506] ELD79 / Libya zone 5 [507] ELD79 / Libya zone 6 [508] ELD79 / Libya zone 7 [509] ELD79 / Libya zone 8 [510] ELD79 / Libya zone 9 [511] ELD79 / TM 12 NE [512] ELD79 / UTM zone 32N [513] ELD79 / UTM zone 33N [514] ELD79 / UTM zone 34N [515] ELD79 / UTM zone 35N [516] EMEP_150_Kilometer_Grid [517] EMEP_50_Kilometer_Grid [518] ETRF_1989_TM_Baltic_1993 (deprecated) [519] ETRF_1989_UTM_Zone_28N (deprecated) [520] ETRF_1989_UTM_Zone_29N (deprecated) [521] ETRF_1989_UTM_Zone_30N (deprecated) [522] ETRF_1989_UTM_Zone_31N (deprecated) [523] ETRF_1989_UTM_Zone_32N (deprecated) [524] ETRF_1989_UTM_Zone_33N (deprecated) [525] ETRF_1989_UTM_Zone_34N (deprecated) [526] ETRF_1989_UTM_Zone_35N (deprecated) [527] ETRF_1989_UTM_Zone_36N (deprecated) [528] ETRF_1989_UTM_Zone_37N (deprecated) [529] ETRF_1989_UTM_Zone_38N (deprecated) [530] ETRS89 / Albania LCC 2010 [531] ETRS89 / Albania TM 2010 [532] ETRS89 / Austria Lambert [533] ETRS89 / Belgian Lambert 2005 [534] ETRS89 / Belgian Lambert 2008 [535] ETRS89 / DKTM1 [536] ETRS89 / DKTM2 [537] ETRS89 / DKTM3 [538] ETRS89 / DKTM4 [539] ETRS89 / EPSG Arctic zone 2-22 [540] ETRS89 / EPSG Arctic zone 3-11 [541] ETRS89 / EPSG Arctic zone 4-26 [542] ETRS89 / EPSG Arctic zone 4-28 [543] ETRS89 / EPSG Arctic zone 5-11 [544] ETRS89 / EPSG Arctic zone 5-13 [545] ETRS89 / EPSG Arctic zone 5-47 [546] ETRS89 / ETRS-GK19FIN [547] ETRS89 / ETRS-GK20FIN [548] ETRS89 / ETRS-GK21FIN [549] ETRS89 / ETRS-GK22FIN [550] ETRS89 / ETRS-GK23FIN [551] ETRS89 / ETRS-GK24FIN [552] ETRS89 / ETRS-GK25FIN [553] ETRS89 / ETRS-GK26FIN [554] ETRS89 / ETRS-GK27FIN [555] ETRS89 / ETRS-GK28FIN [556] ETRS89 / ETRS-GK29FIN [557] ETRS89 / ETRS-GK30FIN [558] ETRS89 / ETRS-GK31FIN [559] ETRS89 / Faroe TM [560] ETRS89 / GK19FIN [561] ETRS89 / GK20FIN [562] ETRS89 / GK21FIN [563] ETRS89 / GK22FIN [564] ETRS89 / GK23FIN [565] ETRS89 / GK24FIN [566] ETRS89 / GK25FIN [567] ETRS89 / GK26FIN [568] ETRS89 / GK27FIN [569] ETRS89 / GK28FIN [570] ETRS89 / GK29FIN [571] ETRS89 / GK30FIN [572] ETRS89 / GK31FIN [573] ETRS89 / Gauss-Kruger CM 9E [574] ETRS89 / Guernsey Grid [575] ETRS89 / Jersey Transverse Mercator [576] ETRS89 / Kp2000 Bornholm [577] ETRS89 / Kp2000 Jutland [578] ETRS89 / Kp2000 Zealand [579] ETRS89 / LAEA Europe [580] ETRS89 / LCC Europe [581] ETRS89 / LCC Germany (E-N) [582] ETRS89 / LCC Germany (N-E) [583] ETRS89 / NTM zone 10 [584] ETRS89 / NTM zone 10 (deprecated) [585] ETRS89 / NTM zone 11 [586] ETRS89 / NTM zone 11 (deprecated) [587] ETRS89 / NTM zone 12 [588] ETRS89 / NTM zone 12 (deprecated) [589] ETRS89 / NTM zone 13 [590] ETRS89 / NTM zone 13 (deprecated) [591] ETRS89 / NTM zone 14 [592] ETRS89 / NTM zone 14 (deprecated) [593] ETRS89 / NTM zone 15 [594] ETRS89 / NTM zone 15 (deprecated) [595] ETRS89 / NTM zone 16 [596] ETRS89 / NTM zone 16 (deprecated) [597] ETRS89 / NTM zone 17 [598] ETRS89 / NTM zone 17 (deprecated) [599] ETRS89 / NTM zone 18 [600] ETRS89 / NTM zone 18 (deprecated) [601] ETRS89 / NTM zone 19 [602] ETRS89 / NTM zone 19 (deprecated) [603] ETRS89 / NTM zone 20 [604] ETRS89 / NTM zone 20 (deprecated) [605] ETRS89 / NTM zone 21 [606] ETRS89 / NTM zone 21 (deprecated) [607] ETRS89 / NTM zone 22 [608] ETRS89 / NTM zone 22 (deprecated) [609] ETRS89 / NTM zone 23 [610] ETRS89 / NTM zone 23 (deprecated) [611] ETRS89 / NTM zone 24 [612] ETRS89 / NTM zone 24 (deprecated) [613] ETRS89 / NTM zone 25 [614] ETRS89 / NTM zone 25 (deprecated) [615] ETRS89 / NTM zone 26 [616] ETRS89 / NTM zone 26 (deprecated) [617] ETRS89 / NTM zone 27 [618] ETRS89 / NTM zone 27 (deprecated) [619] ETRS89 / NTM zone 28 [620] ETRS89 / NTM zone 28 (deprecated) [621] ETRS89 / NTM zone 29 [622] ETRS89 / NTM zone 29 (deprecated) [623] ETRS89 / NTM zone 30 [624] ETRS89 / NTM zone 30 (deprecated) [625] ETRS89 / NTM zone 5 [626] ETRS89 / NTM zone 5 (deprecated) [627] ETRS89 / NTM zone 6 [628] ETRS89 / NTM zone 6 (deprecated) [629] ETRS89 / NTM zone 7 [630] ETRS89 / NTM zone 7 (deprecated) [631] ETRS89 / NTM zone 8 [632] ETRS89 / NTM zone 8 (deprecated) [633] ETRS89 / NTM zone 9 [634] ETRS89 / NTM zone 9 (deprecated) [635] ETRS89 / Poland CS2000 zone 5 [636] ETRS89 / Poland CS2000 zone 6 [637] ETRS89 / Poland CS2000 zone 7 [638] ETRS89 / Poland CS2000 zone 8 [639] ETRS89 / Poland CS92 [640] ETRS89 / Portugal TM06 [641] ETRS89 / TM 30 NE [642] ETRS89 / TM Baltic93 [643] ETRS89 / TM26 (deprecated) [644] ETRS89 / TM27 (deprecated) [645] ETRS89 / TM35FIN(E,N) [646] ETRS89 / TM35FIN(N,E) [647] ETRS89 / TM38 (deprecated) [648] ETRS89 / TM39 (deprecated) [649] ETRS89 / UTM zone 28N [650] ETRS89 / UTM zone 28N (N-E) [651] ETRS89 / UTM zone 29N [652] ETRS89 / UTM zone 29N (N-E) [653] ETRS89 / UTM zone 30N [654] ETRS89 / UTM zone 30N (N-E) [655] ETRS89 / UTM zone 31N [656] ETRS89 / UTM zone 31N (N-E) [657] ETRS89 / UTM zone 31N (N-zE) [658] ETRS89 / UTM zone 31N (zE-N) [659] ETRS89 / UTM zone 32N [660] ETRS89 / UTM zone 32N (N-E) [661] ETRS89 / UTM zone 32N (N-zE) [662] ETRS89 / UTM zone 32N (zE-N) [663] ETRS89 / UTM zone 33N [664] ETRS89 / UTM zone 33N (N-E) [665] ETRS89 / UTM zone 33N (N-zE) [666] ETRS89 / UTM zone 33N (zE-N) [667] ETRS89 / UTM zone 34N [668] ETRS89 / UTM zone 34N (N-E) [669] ETRS89 / UTM zone 35N [670] ETRS89 / UTM zone 35N (N-E) [671] ETRS89 / UTM zone 36N [672] ETRS89 / UTM zone 36N (N-E) [673] ETRS89 / UTM zone 37N [674] ETRS89 / UTM zone 37N (N-E) [675] ETRS89 / UTM zone 38N (deprecated) [676] ETRS_1989_DKTM1 (deprecated) [677] ETRS_1989_DKTM2 (deprecated) [678] ETRS_1989_DKTM3 (deprecated) [679] ETRS_1989_DKTM4 (deprecated) [680] ETRS_1989_Kosovo_Grid [681] ETRS_1989_Slovenia_TM [682] ETRS_1989_UTM_Zone_26N [683] ETRS_1989_UTM_Zone_27N [684] ETRS_1989_UTM_Zone_32N_7stellen [685] ETRS_1989_UTM_Zone_32N_8stellen [686] ETRS_1989_UTM_Zone_33N_7stellen [687] ETRS_1989_UTM_Zone_33N_8stellen [688] ETRS_1989_UTM_Zone_39N [689] ETRS_1989_UTM_Zone_N32 (deprecated) [690] ETRS_1989_UWPP_1992 [691] ETRS_1989_UWPP_2000_PAS_5 [692] ETRS_1989_UWPP_2000_PAS_6 [693] ETRS_1989_UWPP_2000_PAS_7 [694] ETRS_1989_UWPP_2000_PAS_8 [695] EUREF_FIN_TM35FIN (deprecated) [696] Egypt 1907 / Blue Belt [697] Egypt 1907 / Extended Purple Belt [698] Egypt 1907 / Purple Belt [699] Egypt 1907 / Red Belt [700] Egypt Gulf of Suez S-650 TL / Red Belt [701] Estonian Coordinate System of 1992 [702] Estonian Coordinate System of 1997 [703] Europe_Albers_Equal_Area_Conic [704] Europe_Equidistant_Conic [705] Europe_Lambert_Conformal_Conic [706] Everest_Modified_1969_RSO_Malaya_Meters [707] FD54 / UTM zone 29N [708] FD58 / Iraq zone [709] FEH2010 / Fehmarnbelt TM [710] Fahud / UTM zone 39N [711] Fahud / UTM zone 40N [712] Fatu Iva 72 / UTM zone 7S [713] Fiji 1956 / UTM zone 1S [714] Fiji 1956 / UTM zone 60S [715] Fiji 1986 / Fiji Map Grid [716] Fiji 1986 / Fiji Map Grid (deprecated) [717] Fort Marigot / UTM zone 20N [718] GDA2020 / ALB2020 [719] GDA2020 / BCG2020 [720] GDA2020 / BIO2020 [721] GDA2020 / BRO2020 [722] GDA2020 / CARN2020 [723] GDA2020 / CIG2020 [724] GDA2020 / CKIG2020 [725] GDA2020 / COL2020 [726] GDA2020 / ESP2020 [727] GDA2020 / EXM2020 [728] GDA2020 / GA LCC [729] GDA2020 / GCG2020 [730] GDA2020 / GOLD2020 [731] GDA2020 / JCG2020 [732] GDA2020 / KALB2020 [733] GDA2020 / KAR2020 [734] GDA2020 / KUN2020 [735] GDA2020 / LCG2020 [736] GDA2020 / MGA zone 46 [737] GDA2020 / MGA zone 47 [738] GDA2020 / MGA zone 48 [739] GDA2020 / MGA zone 49 [740] GDA2020 / MGA zone 50 [741] GDA2020 / MGA zone 51 [742] GDA2020 / MGA zone 52 [743] GDA2020 / MGA zone 53 [744] GDA2020 / MGA zone 54 [745] GDA2020 / MGA zone 55 [746] GDA2020 / MGA zone 56 [747] GDA2020 / MGA zone 57 [748] GDA2020 / MGA zone 58 [749] GDA2020 / MGA zone 59 [750] GDA2020 / MRCG2020 [751] GDA2020 / NSW Lambert [752] GDA2020 / PCG2020 [753] GDA2020 / PHG2020 [754] GDA2020 / SA Lambert [755] GDA2020 / Vicgrid [756] GDA94 / Australian Albers [757] GDA94 / BCSG02 [758] GDA94 / CIG94 [759] GDA94 / CKIG94 [760] GDA94 / Geoscience Australia Lambert [761] GDA94 / MGA zone 41 (deprecated) [762] GDA94 / MGA zone 42 (deprecated) [763] GDA94 / MGA zone 43 (deprecated) [764] GDA94 / MGA zone 44 (deprecated) [765] GDA94 / MGA zone 46 [766] GDA94 / MGA zone 47 [767] GDA94 / MGA zone 48 [768] GDA94 / MGA zone 49 [769] GDA94 / MGA zone 50 [770] GDA94 / MGA zone 51 [771] GDA94 / MGA zone 52 [772] GDA94 / MGA zone 53 [773] GDA94 / MGA zone 54 [774] GDA94 / MGA zone 55 [775] GDA94 / MGA zone 56 [776] GDA94 / MGA zone 57 [777] GDA94 / MGA zone 58 [778] GDA94 / MGA zone 59 [779] GDA94 / NSW Lambert [780] GDA94 / SA Lambert [781] GDA94 / Vicgrid [782] GDA94 / WEIPA94 [783] GDA_1994_Perth_Coastal_Grid_1994 [784] GDA_1994_South_Australia_Lambert (deprecated) [785] GDA_1994_VICGRID94 (deprecated) [786] GDBD2009 / Brunei BRSO [787] GDBD2009_GEORSO (deprecated) [788] GDM2000 / East Malaysia BRSO [789] GDM2000 / Johor Grid [790] GDM2000 / Kedah and Perlis Grid [791] GDM2000 / Kelantan Grid [792] GDM2000 / Pahang Grid [793] GDM2000 / Peninsula RSO [794] GDM2000 / Perak Grid [795] GDM2000 / Pinang Grid [796] GDM2000 / Selangor Grid [797] GDM2000 / Sembilan and Melaka Grid [798] GDM2000 / Terengganu Grid [799] GGRS87 / Greek Grid [800] GOES-16_East_ABI_Fixed_Grid_ITRF2000_incorrect_GCS (deprecated) [801] GOES-16_East_ABI_Fixed_Grid_ITRF2008 [802] GR96 / EPSG Arctic zone 1-25 [803] GR96 / EPSG Arctic zone 2-18 [804] GR96 / EPSG Arctic zone 2-20 [805] GR96 / EPSG Arctic zone 3-29 [806] GR96 / EPSG Arctic zone 3-31 [807] GR96 / EPSG Arctic zone 3-33 [808] GR96 / EPSG Arctic zone 4-20 [809] GR96 / EPSG Arctic zone 4-22 [810] GR96 / EPSG Arctic zone 4-24 [811] GR96 / EPSG Arctic zone 5-41 [812] GR96 / EPSG Arctic zone 5-43 [813] GR96 / EPSG Arctic zone 5-45 [814] GR96 / EPSG Arctic zone 6-26 [815] GR96 / EPSG Arctic zone 6-28 [816] GR96 / EPSG Arctic zone 6-30 [817] GR96 / EPSG Arctic zone 7-11 [818] GR96 / EPSG Arctic zone 7-13 [819] GR96 / EPSG Arctic zone 8-20 [820] GR96 / EPSG Arctic zone 8-22 [821] GR96 / UTM zone 18N [822] GR96 / UTM zone 19N [823] GR96 / UTM zone 20N [824] GR96 / UTM zone 21N [825] GR96 / UTM zone 22N [826] GR96 / UTM zone 23N [827] GR96 / UTM zone 24N [828] GR96 / UTM zone 25N [829] GR96 / UTM zone 26N [830] GR96 / UTM zone 27N [831] GR96 / UTM zone 28N [832] GR96 / UTM zone 29N [833] Garoua / UTM zone 33N [834] Garoua / UTM zone 33N (deprecated) [835] Germany_Zone_1 [836] Germany_Zone_2 [837] Germany_Zone_3 [838] Germany_Zone_4 [839] Germany_Zone_5 [840] Graciosa_Base_SW_1948_UTM_Zone_26N [841] Grand Cayman 1959 / UTM zone 17N (deprecated) [842] Grand Cayman National Grid 1959 [843] Grand Comoros / UTM zone 38S [844] Grenada 1953 / British West Indies Grid [845] Guadeloupe 1948 / UTM zone 20N [846] Guam 1963 / Yap Islands [847] Guam_Geodetic_Network_1993 [848] Guam_Geodetic_Triangulation_Network_1963 [849] Guernsey_Grid [850] Gulshan 303 / Bangladesh Transverse Mercator [851] Gusterberg Grid (Ferro) [852] HD72 / EOV [853] HTRS96 / Croatia LCC [854] HTRS96 / Croatia TM [855] HTRS96 / UTM zone 33N [856] HTRS96 / UTM zone 34N [857] Hanoi 1972 / GK 106 NE [858] Hanoi 1972 / Gauss-Kruger zone 18 [859] Hanoi 1972 / Gauss-Kruger zone 19 [860] Hartebeesthoek94 / Lo15 [861] Hartebeesthoek94 / Lo17 [862] Hartebeesthoek94 / Lo19 [863] Hartebeesthoek94 / Lo21 [864] Hartebeesthoek94 / Lo23 [865] Hartebeesthoek94 / Lo25 [866] Hartebeesthoek94 / Lo27 [867] Hartebeesthoek94 / Lo29 [868] Hartebeesthoek94 / Lo31 [869] Hartebeesthoek94 / Lo33 [870] Hartebeesthoek94_Lo15 [871] Hartebeesthoek94_Lo17 [872] Hartebeesthoek94_Lo19 [873] Hartebeesthoek94_Lo19_(E-N) [874] Hartebeesthoek94_Lo21 [875] Hartebeesthoek94_Lo21_(E-N) [876] Hartebeesthoek94_Lo23 [877] Hartebeesthoek94_Lo23_(E-N) [878] Hartebeesthoek94_Lo25 [879] Hartebeesthoek94_Lo25_(E-N) [880] Hartebeesthoek94_Lo27 [881] Hartebeesthoek94_Lo27_(E-N) [882] Hartebeesthoek94_Lo29 [883] Hartebeesthoek94_Lo29_(E-N) [884] Hartebeesthoek94_Lo31 [885] Hartebeesthoek94_Lo31_(E-N) [886] Hartebeesthoek94_Lo33 [887] Hawaii_Albers_Equal_Area_Conic [888] Helle 1954 / Jan Mayen Grid [889] Hito XVIII 1963 / Argentina 2 [890] Hito XVIII 1963 / UTM zone 19S [891] Hjorsey 1955 / UTM zone 26N [892] Hjorsey 1955 / UTM zone 27N [893] Hjorsey 1955 / UTM zone 28N [894] Hong Kong 1963 Grid System [895] Hong Kong 1963 Grid System (deprecated) [896] Hong Kong 1980 Grid System [897] Hong_Kong_1980_Grid (deprecated) [898] Hong_Kong_1980_UTM_Zone_49N [899] Hong_Kong_1980_UTM_Zone_50N [900] Hu Tzu Shan 1950 / UTM zone 51N [901] ID74 / UTM zone 46N [902] ID74 / UTM zone 46S (deprecated) [903] ID74 / UTM zone 47N [904] ID74 / UTM zone 47S [905] ID74 / UTM zone 48N [906] ID74 / UTM zone 48S [907] ID74 / UTM zone 49N [908] ID74 / UTM zone 49S [909] ID74 / UTM zone 50N [910] ID74 / UTM zone 50S [911] ID74 / UTM zone 51N [912] ID74 / UTM zone 51S [913] ID74 / UTM zone 52N [914] ID74 / UTM zone 52S [915] ID74 / UTM zone 53N (deprecated) [916] ID74 / UTM zone 53S [917] ID74 / UTM zone 54S [918] IGC 1962 / Congo TM zone 12 [919] IGC 1962 / Congo TM zone 14 [920] IGC 1962 / Congo TM zone 16 [921] IGC 1962 / Congo TM zone 18 [922] IGC 1962 / Congo TM zone 20 [923] IGC 1962 / Congo TM zone 22 [924] IGC 1962 / Congo TM zone 24 [925] IGC 1962 / Congo TM zone 26 [926] IGC 1962 / Congo TM zone 28 [927] IGC 1962 / Congo TM zone 30 [928] IGCB 1955 / Congo TM zone 12 [929] IGCB 1955 / Congo TM zone 14 [930] IGCB 1955 / Congo TM zone 16 [931] IGCB 1955 / UTM zone 33S [932] IGM95 / UTM zone 32N [933] IGM95 / UTM zone 33N [934] IGN 1962 Kerguelen / UTM zone 42S [935] IGN Astro 1960 / UTM zone 28N [936] IGN Astro 1960 / UTM zone 29N [937] IGN Astro 1960 / UTM zone 30N [938] IGN53 Mare / UTM zone 58S [939] IGN53 Mare / UTM zone 59S [940] IGN56 Lifou / UTM zone 58S [941] IGN63 Hiva Oa / UTM zone 7S [942] IGN72 Grand Terre / UTM zone 58S (deprecated) [943] IGN72 Grande Terre / UTM zone 58S [944] IGN72 Nuku Hiva / UTM zone 7S [945] IGRS / UTM zone 37N [946] IGRS / UTM zone 38N [947] IGRS / UTM zone 39N [948] IRENET95 / Irish Transverse Mercator [949] IRENET95 / UTM zone 29N [950] ISN2004 / LAEA Europe [951] ISN2004 / LCC Europe [952] ISN2004 / Lambert 2004 [953] ISN2016 / Lambert 2016 [954] ISN93 / Lambert 1993 [955] ISN_2004_Lambert_2004 (deprecated) [956] Indian 1954 / UTM zone 46N [957] Indian 1954 / UTM zone 47N [958] Indian 1954 / UTM zone 48N [959] Indian 1960 / TM 106 NE [960] Indian 1960 / UTM zone 48N [961] Indian 1960 / UTM zone 49N [962] Indian 1975 / UTM zone 47N [963] Indian 1975 / UTM zone 48N [964] Israel 1993 / Israeli TM Grid [965] Israeli Grid 05 [966] Israeli Grid 05/12 [967] JAD2001 / Jamaica Metric Grid [968] JAD2001 / UTM zone 17N [969] JAD2001 / UTM zone 18N [970] JAD69 / Jamaica National Grid [971] JAD_2001_Jamaica_Grid (deprecated) [972] JAXA Snow Depth Polar Stereographic North [973] JGD2000 / Japan Plane Rectangular CS I [974] JGD2000 / Japan Plane Rectangular CS II [975] JGD2000 / Japan Plane Rectangular CS III [976] JGD2000 / Japan Plane Rectangular CS IV [977] JGD2000 / Japan Plane Rectangular CS IX [978] JGD2000 / Japan Plane Rectangular CS V [979] JGD2000 / Japan Plane Rectangular CS VI [980] JGD2000 / Japan Plane Rectangular CS VII [981] JGD2000 / Japan Plane Rectangular CS VIII [982] JGD2000 / Japan Plane Rectangular CS X [983] JGD2000 / Japan Plane Rectangular CS XI [984] JGD2000 / Japan Plane Rectangular CS XII [985] JGD2000 / Japan Plane Rectangular CS XIII [986] JGD2000 / Japan Plane Rectangular CS XIV [987] JGD2000 / Japan Plane Rectangular CS XIX [988] JGD2000 / Japan Plane Rectangular CS XV [989] JGD2000 / Japan Plane Rectangular CS XVI [990] JGD2000 / Japan Plane Rectangular CS XVII [991] JGD2000 / Japan Plane Rectangular CS XVIII [992] JGD2000 / UTM zone 51N [993] JGD2000 / UTM zone 52N [994] JGD2000 / UTM zone 53N [995] JGD2000 / UTM zone 54N [996] JGD2000 / UTM zone 55N [997] JGD2011 / Japan Plane Rectangular CS I [998] JGD2011 / Japan Plane Rectangular CS II [999] JGD2011 / Japan Plane Rectangular CS III [1000] JGD2011 / Japan Plane Rectangular CS IV [1001] JGD2011 / Japan Plane Rectangular CS IX [1002] JGD2011 / Japan Plane Rectangular CS V [1003] JGD2011 / Japan Plane Rectangular CS VI [1004] JGD2011 / Japan Plane Rectangular CS VII [1005] JGD2011 / Japan Plane Rectangular CS VIII [1006] JGD2011 / Japan Plane Rectangular CS X [1007] JGD2011 / Japan Plane Rectangular CS XI [1008] JGD2011 / Japan Plane Rectangular CS XII [1009] JGD2011 / Japan Plane Rectangular CS XIII [1010] JGD2011 / Japan Plane Rectangular CS XIV [1011] JGD2011 / Japan Plane Rectangular CS XIX [1012] JGD2011 / Japan Plane Rectangular CS XV [1013] JGD2011 / Japan Plane Rectangular CS XVI [1014] JGD2011 / Japan Plane Rectangular CS XVII [1015] JGD2011 / Japan Plane Rectangular CS XVIII [1016] JGD2011 / UTM zone 51N [1017] JGD2011 / UTM zone 52N [1018] JGD2011 / UTM zone 53N [1019] JGD2011 / UTM zone 54N [1020] JGD2011 / UTM zone 55N [1021] JGD_2000_UTM_Zone_51N (deprecated) [1022] JGD_2000_UTM_Zone_52N (deprecated) [1023] JGD_2000_UTM_Zone_53N (deprecated) [1024] JGD_2000_UTM_Zone_54N (deprecated) [1025] JGD_2000_UTM_Zone_55N (deprecated) [1026] JGD_2000_UTM_Zone_56N [1027] JGD_2011_Japan_Zone_1 (deprecated) [1028] JGD_2011_Japan_Zone_10 (deprecated) [1029] JGD_2011_Japan_Zone_11 (deprecated) [1030] JGD_2011_Japan_Zone_12 (deprecated) [1031] JGD_2011_Japan_Zone_13 (deprecated) [1032] JGD_2011_Japan_Zone_14 (deprecated) [1033] JGD_2011_Japan_Zone_15 (deprecated) [1034] JGD_2011_Japan_Zone_16 (deprecated) [1035] JGD_2011_Japan_Zone_17 (deprecated) [1036] JGD_2011_Japan_Zone_18 (deprecated) [1037] JGD_2011_Japan_Zone_19 (deprecated) [1038] JGD_2011_Japan_Zone_2 (deprecated) [1039] JGD_2011_Japan_Zone_3 (deprecated) [1040] JGD_2011_Japan_Zone_4 (deprecated) [1041] JGD_2011_Japan_Zone_5 (deprecated) [1042] JGD_2011_Japan_Zone_6 (deprecated) [1043] JGD_2011_Japan_Zone_7 (deprecated) [1044] JGD_2011_Japan_Zone_8 (deprecated) [1045] JGD_2011_Japan_Zone_9 (deprecated) [1046] JGD_2011_UTM_Zone_51N (deprecated) [1047] JGD_2011_UTM_Zone_52N (deprecated) [1048] JGD_2011_UTM_Zone_53N (deprecated) [1049] JGD_2011_UTM_Zone_54N (deprecated) [1050] JGD_2011_UTM_Zone_55N (deprecated) [1051] JGD_2011_UTM_Zone_56N [1052] Jamaica 1875 / Jamaica (Old Grid) [1053] Jordan_JTM [1054] K0 1949 / UTM zone 42S (deprecated) [1055] KKJ / Finland Uniform Coordinate System [1056] KKJ / Finland zone 0 [1057] KKJ / Finland zone 1 [1058] KKJ / Finland zone 2 [1059] KKJ / Finland zone 4 [1060] KKJ / Finland zone 5 [1061] KOC Lambert [1062] KUDAMS / KTM [1063] KUDAMS / KTM (deprecated) [1064] Kalianpur 1880 / India zone 0 [1065] Kalianpur 1880 / India zone I [1066] Kalianpur 1880 / India zone IIIa [1067] Kalianpur 1880 / India zone IIa [1068] Kalianpur 1880 / India zone IIb [1069] Kalianpur 1880 / India zone IVa [1070] Kalianpur 1937 / India zone IIb [1071] Kalianpur 1937 / UTM zone 45N [1072] Kalianpur 1937 / UTM zone 46N [1073] Kalianpur 1962 / India zone I [1074] Kalianpur 1962 / India zone IIa [1075] Kalianpur 1962 / UTM zone 41N [1076] Kalianpur 1962 / UTM zone 42N [1077] Kalianpur 1962 / UTM zone 43N [1078] Kalianpur 1975 / India zone I [1079] Kalianpur 1975 / India zone IIIa [1080] Kalianpur 1975 / India zone IIa [1081] Kalianpur 1975 / India zone IIb [1082] Kalianpur 1975 / India zone IVa [1083] Kalianpur 1975 / UTM zone 42N [1084] Kalianpur 1975 / UTM zone 43N [1085] Kalianpur 1975 / UTM zone 44N [1086] Kalianpur 1975 / UTM zone 45N [1087] Kalianpur 1975 / UTM zone 46N [1088] Kalianpur 1975 / UTM zone 47N [1089] Kandawala / Sri Lanka Grid [1090] Kandawala_Ceylon_Belt_Indian_Yards_1937 [1091] Kandawala_Ceylon_Belt_Meters [1092] Karbala 1979 / Iraq National Grid [1093] Karbala 1979 / UTM zone 37N [1094] Karbala 1979 / UTM zone 38N [1095] Karbala 1979 / UTM zone 39N [1096] Kasai 1953 / Congo TM zone 22 [1097] Kasai 1953 / Congo TM zone 24 [1098] Katanga 1955 / Katanga Gauss zone A [1099] Katanga 1955 / Katanga Gauss zone B [1100] Katanga 1955 / Katanga Gauss zone C [1101] Katanga 1955 / Katanga Gauss zone D [1102] Katanga 1955 / Katanga Lambert [1103] Katanga 1955 / Katanga Lambert (deprecated) [1104] Katanga 1955 / Katanga Lambert (deprecated) [1105] Katanga 1955 / Katanga TM (deprecated) [1106] Katanga_1955_Katanga_Lambert (deprecated) [1107] Kertau (RSO) / RSO Malaya (ch) [1108] Kertau (RSO) / RSO Malaya (m) [1109] Kertau / R.S.O. Malaya (ch) (deprecated) [1110] Kertau 1968 / Johor Grid [1111] Kertau 1968 / Kedah and Perlis Grid [1112] Kertau 1968 / Kelantan Grid [1113] Kertau 1968 / Pahang Grid [1114] Kertau 1968 / Perak Revised Grid [1115] Kertau 1968 / Pinang Grid [1116] Kertau 1968 / Selangor Grid [1117] Kertau 1968 / Sembilan and Melaka Grid [1118] Kertau 1968 / Singapore Grid [1119] Kertau 1968 / Terengganu Grid [1120] Kertau 1968 / UTM zone 47N [1121] Kertau 1968 / UTM zone 48N [1122] Kertau_RSO_Malaya_Meters [1123] Korea 2000 / Central Belt [1124] Korea 2000 / Central Belt 2010 [1125] Korea 2000 / Central Belt Jeju [1126] Korea 2000 / East Belt [1127] Korea 2000 / East Belt 2010 [1128] Korea 2000 / East Sea Belt [1129] Korea 2000 / East Sea Belt 2010 [1130] Korea 2000 / Unified CS [1131] Korea 2000 / West Belt [1132] Korea 2000 / West Belt 2010 [1133] Korea_2000_Korea_Central_Belt_2010 (deprecated) [1134] Korea_2000_Korea_East_Belt_2010 (deprecated) [1135] Korea_2000_Korea_East_Sea_Belt_2010 (deprecated) [1136] Korea_2000_Korea_Unified_Coordinate_System (deprecated) [1137] Korea_2000_Korea_West_Belt_2010 (deprecated) [1138] Korean 1985 / Central Belt [1139] Korean 1985 / Central Belt Jeju [1140] Korean 1985 / East Belt [1141] Korean 1985 / East Sea Belt [1142] Korean 1985 / Modified Central Belt [1143] Korean 1985 / Modified Central Belt Jeju [1144] Korean 1985 / Modified East Belt [1145] Korean 1985 / Modified East Sea Belt [1146] Korean 1985 / Modified West Belt [1147] Korean 1985 / Unified CS [1148] Korean 1985 / West Belt [1149] Korean_1985_Korea_Unified_Coordinate_System (deprecated) [1150] Korean_1985_Modified_Korea_Central_Belt (deprecated) [1151] Korean_1985_Modified_Korea_Central_Belt_Jeju (deprecated) [1152] Korean_1985_Modified_Korea_East_Belt (deprecated) [1153] Korean_1985_Modified_Korea_East_Sea_Belt (deprecated) [1154] Korean_1985_Modified_Korea_West_Belt (deprecated) [1155] Kousseri / UTM zone 33N [1156] Kyrg-06 / zone 1 [1157] Kyrg-06 / zone 2 [1158] Kyrg-06 / zone 3 [1159] Kyrg-06 / zone 4 [1160] Kyrg-06 / zone 5 [1161] Kyrg-06_TM_Zone_1 [1162] Kyrg-06_TM_Zone_2 [1163] Kyrg-06_TM_Zone_3 [1164] Kyrg-06_TM_Zone_4 [1165] Kyrg-06_TM_Zone_5 [1166] LGD2006 / Libya TM [1167] LGD2006 / Libya TM zone 10 [1168] LGD2006 / Libya TM zone 11 [1169] LGD2006 / Libya TM zone 12 [1170] LGD2006 / Libya TM zone 13 [1171] LGD2006 / Libya TM zone 5 [1172] LGD2006 / Libya TM zone 6 [1173] LGD2006 / Libya TM zone 7 [1174] LGD2006 / Libya TM zone 8 [1175] LGD2006 / Libya TM zone 9 [1176] LGD2006 / UTM zone 32N [1177] LGD2006 / UTM zone 33N [1178] LGD2006 / UTM zone 34N [1179] LGD2006 / UTM zone 35N [1180] LKS92 / Latvia TM [1181] LKS94 / Lithuania TM [1182] LKS_1992_Latvia_TM_0 [1183] La Canoa / UTM zone 18N [1184] La Canoa / UTM zone 19N [1185] La Canoa / UTM zone 20N [1186] La_Canoa_UTM_Zone_21N [1187] Lake / Maracaibo Grid [1188] Lake / Maracaibo Grid M1 [1189] Lake / Maracaibo Grid M3 [1190] Lake / Maracaibo La Rosa Grid [1191] Le Pouce 1934 / Mauritius Grid [1192] Leigon / Ghana Metre Grid [1193] Lietuvos Koordinoei Sistema 1994 (deprecated) [1194] Lisboa_Bessel_Bonne [1195] Lisboa_Hayford_Gauss_IGeoE [1196] Lisboa_Hayford_Gauss_IPCC [1197] Lisbon (Lisbon) / Portuguese Grid [1198] Lisbon (Lisbon) / Portuguese National Grid [1199] Lisbon / Portuguese Grid New [1200] Little Cayman 1961 / UTM zone 17N (deprecated) [1201] Locodjo 1965 / TM 5 NW [1202] Locodjo 1965 / UTM zone 29N [1203] Locodjo 1965 / UTM zone 30N [1204] Lome / UTM zone 31N [1205] London_Survey_Grid [1206] Luxembourg 1930 / Gauss [1207] Luzon 1911 / Philippines zone I [1208] Luzon 1911 / Philippines zone II [1209] Luzon 1911 / Philippines zone III [1210] Luzon 1911 / Philippines zone IV [1211] Luzon 1911 / Philippines zone V [1212] Luzon_1911_UTM_Zone_50N [1213] Luzon_1911_UTM_Zone_51N [1214] Luzon_1911_UTM_Zone_52N [1215] M'poraloko / UTM zone 32N [1216] M'poraloko / UTM zone 32S [1217] MAGNA-SIRGAS / Colombia Bogota zone [1218] MAGNA-SIRGAS / Colombia East Central zone [1219] MAGNA-SIRGAS / Colombia East zone [1220] MAGNA-SIRGAS / Colombia Far West zone [1221] MAGNA-SIRGAS / Colombia West zone [1222] MAGNA_Ciudad_Bogota [1223] MARGEN / UTM zone 19S [1224] MARGEN / UTM zone 20S [1225] MARGEN / UTM zone 21S [1226] MGI (Ferro) / Austria Central Zone [1227] MGI (Ferro) / Austria Central Zone (deprecated) [1228] MGI (Ferro) / Austria East Zone [1229] MGI (Ferro) / Austria East Zone (deprecated) [1230] MGI (Ferro) / Austria GK Central Zone [1231] MGI (Ferro) / Austria GK East Zone [1232] MGI (Ferro) / Austria GK West Zone [1233] MGI (Ferro) / Austria West Zone [1234] MGI (Ferro) / Austria West Zone (deprecated) [1235] MGI (Ferro) / M28 [1236] MGI (Ferro) / M31 [1237] MGI (Ferro) / M34 [1238] MGI / 3-degree Gauss zone 5 (deprecated) [1239] MGI / 3-degree Gauss zone 6 (deprecated) [1240] MGI / 3-degree Gauss zone 7 (deprecated) [1241] MGI / 3-degree Gauss zone 8 (deprecated) [1242] MGI / Austria GK Central [1243] MGI / Austria GK East [1244] MGI / Austria GK M28 [1245] MGI / Austria GK M31 [1246] MGI / Austria GK M34 [1247] MGI / Austria GK West [1248] MGI / Austria Lambert [1249] MGI / Austria Lambert (deprecated) [1250] MGI / Austria M28 [1251] MGI / Austria M31 [1252] MGI / Austria M34 [1253] MGI / Balkans zone 5 (deprecated) [1254] MGI / Balkans zone 6 (deprecated) [1255] MGI / Balkans zone 7 (deprecated) [1256] MGI / Balkans zone 8 (deprecated) [1257] MGI / Balkans zone 8 (deprecated) [1258] MGI / M28 (deprecated) [1259] MGI / M31 (deprecated) [1260] MGI / M34 (deprecated) [1261] MGI / Slovene National Grid (deprecated) [1262] MGI / Slovenia Grid (deprecated) [1263] MGI 1901 / Balkans zone 5 [1264] MGI 1901 / Balkans zone 5 [1265] MGI 1901 / Balkans zone 6 [1266] MGI 1901 / Balkans zone 6 [1267] MGI 1901 / Balkans zone 7 [1268] MGI 1901 / Balkans zone 8 [1269] MGI 1901 / Balkans zone 8 [1270] MGI 1901 / Slovene National Grid [1271] MGI 1901 / Slovenia Grid [1272] MGI 1901 / Slovenia Grid [1273] MOLDREF99 / Moldova TM [1274] MONREF_1997_UTM_Zone_46N [1275] MONREF_1997_UTM_Zone_47N [1276] MONREF_1997_UTM_Zone_48N [1277] MONREF_1997_UTM_Zone_49N [1278] MONREF_1997_UTM_Zone_50N [1279] MOP78 / UTM zone 1S [1280] MTRF-2000 / UTM zone 36N [1281] MTRF-2000 / UTM zone 37N [1282] MTRF-2000 / UTM zone 38N [1283] MTRF-2000 / UTM zone 39N [1284] MTRF-2000 / UTM zone 40N [1285] Macao 1920 / Macao Grid [1286] Macao_2008_Macao_Grid [1287] Macedonia State Coordinate System [1288] Macedonia State Coordinate System zone 7 [1289] Madeira 1936 / UTM zone 28N (deprecated) [1290] Madrid 1870 (Madrid) / Spain [1291] Makassar (Jakarta) / NEIEZ [1292] Makassar (Jakarta) / NEIEZ (deprecated) [1293] Makassar / NEIEZ [1294] Malongo 1987 / UTM zone 32S [1295] Malongo 1987 / UTM zone 33S [1296] Manoca 1962 / UTM zone 32N [1297] Martinique 1938 / UTM zone 20N [1298] Massawa / UTM zone 37N [1299] Maupiti 83 / UTM zone 5S [1300] Mauritania 1999 / UTM zone 28N [1301] Mauritania 1999 / UTM zone 28N (deprecated) [1302] Mauritania 1999 / UTM zone 29N [1303] Mauritania 1999 / UTM zone 29N (deprecated) [1304] Mauritania 1999 / UTM zone 30N [1305] Mauritania 1999 / UTM zone 30N (deprecated) [1306] Merchich / Nord Maroc [1307] Merchich / Sahara (deprecated) [1308] Merchich / Sahara Nord [1309] Merchich / Sahara Sud [1310] Merchich / Sud Maroc [1311] Merchich_Degree_UTM_Zone_28N [1312] Mexican_Datum_1993_UTM_Zone_11N (deprecated) [1313] Mexican_Datum_1993_UTM_Zone_12N (deprecated) [1314] Mexican_Datum_1993_UTM_Zone_13N (deprecated) [1315] Mexican_Datum_1993_UTM_Zone_14N (deprecated) [1316] Mexican_Datum_1993_UTM_Zone_15N (deprecated) [1317] Mexican_Datum_1993_UTM_Zone_16N (deprecated) [1318] Mexico ITRF2008 / LCC [1319] Mexico ITRF2008 / UTM zone 11N [1320] Mexico ITRF2008 / UTM zone 12N [1321] Mexico ITRF2008 / UTM zone 13N [1322] Mexico ITRF2008 / UTM zone 14N [1323] Mexico ITRF2008 / UTM zone 15N [1324] Mexico ITRF2008 / UTM zone 16N [1325] Mexico ITRF92 / LCC [1326] Mexico ITRF92 / UTM zone 11N [1327] Mexico ITRF92 / UTM zone 12N [1328] Mexico ITRF92 / UTM zone 13N [1329] Mexico ITRF92 / UTM zone 14N [1330] Mexico ITRF92 / UTM zone 15N [1331] Mexico ITRF92 / UTM zone 16N [1332] Mhast (offshore) / UTM zone 32S [1333] Mhast (onshore) / UTM zone 32S [1334] Mhast / UTM zone 32S (deprecated) [1335] Minna / Nigeria East Belt [1336] Minna / Nigeria Mid Belt [1337] Minna / Nigeria West Belt [1338] Minna / UTM zone 31N [1339] Minna / UTM zone 32N [1340] Monte Mario (Rome) / Italy zone 1 (deprecated) [1341] Monte Mario (Rome) / Italy zone 2 (deprecated) [1342] Monte Mario / Italy zone 1 [1343] Monte Mario / Italy zone 2 [1344] Monte Mario / TM Emilia-Romagna [1345] Monte_Mario_Italy_1 (deprecated) [1346] Monte_Mario_Italy_2 (deprecated) [1347] Montserrat 1958 / British West Indies Grid [1348] Moorea 87 / UTM zone 6S [1349] Mount Dillon / Tobago Grid [1350] Moznet / UTM zone 36S [1351] Moznet / UTM zone 37S [1352] Moznet / UTM zone 38S [1353] NAD Michigan / Michigan Central (deprecated) [1354] NAD Michigan / Michigan East (deprecated) [1355] NAD Michigan / Michigan North (deprecated) [1356] NAD Michigan / Michigan Old Central (deprecated) [1357] NAD Michigan / Michigan South (deprecated) [1358] NAD Michigan / Michigan West (deprecated) [1359] NAD27 / Alabama East [1360] NAD27 / Alabama West [1361] NAD27 / Alaska Albers [1362] NAD27 / Alaska zone 1 [1363] NAD27 / Alaska zone 10 [1364] NAD27 / Alaska zone 2 [1365] NAD27 / Alaska zone 3 [1366] NAD27 / Alaska zone 4 [1367] NAD27 / Alaska zone 5 [1368] NAD27 / Alaska zone 6 [1369] NAD27 / Alaska zone 7 [1370] NAD27 / Alaska zone 8 [1371] NAD27 / Alaska zone 9 [1372] NAD27 / Alberta 3TM ref merid 111 W [1373] NAD27 / Alberta 3TM ref merid 114 W [1374] NAD27 / Alberta 3TM ref merid 117 W [1375] NAD27 / Alberta 3TM ref merid 120 W [1376] NAD27 / Alberta 3TM ref merid 120 W (deprecated) [1377] NAD27 / Arizona Central [1378] NAD27 / Arizona East [1379] NAD27 / Arizona West [1380] NAD27 / Arkansas North [1381] NAD27 / Arkansas South [1382] NAD27 / BLM 10N (ftUS) [1383] NAD27 / BLM 11N (ftUS) [1384] NAD27 / BLM 12N (ftUS) [1385] NAD27 / BLM 13N (ftUS) [1386] NAD27 / BLM 14N (feet) (deprecated) [1387] NAD27 / BLM 14N (ftUS) [1388] NAD27 / BLM 15N (feet) (deprecated) [1389] NAD27 / BLM 15N (ftUS) [1390] NAD27 / BLM 16N (feet) (deprecated) [1391] NAD27 / BLM 16N (ftUS) [1392] NAD27 / BLM 17N (feet) (deprecated) [1393] NAD27 / BLM 17N (ftUS) [1394] NAD27 / BLM 18N (ftUS) [1395] NAD27 / BLM 19N (ftUS) [1396] NAD27 / BLM 1N (ftUS) [1397] NAD27 / BLM 2N (ftUS) [1398] NAD27 / BLM 3N (ftUS) [1399] NAD27 / BLM 4N (ftUS) [1400] NAD27 / BLM 59N (ftUS) [1401] NAD27 / BLM 5N (ftUS) [1402] NAD27 / BLM 60N (ftUS) [1403] NAD27 / BLM 6N (ftUS) [1404] NAD27 / BLM 7N (ftUS) [1405] NAD27 / BLM 8N (ftUS) [1406] NAD27 / BLM 9N (ftUS) [1407] NAD27 / California Albers [1408] NAD27 / California zone I [1409] NAD27 / California zone II [1410] NAD27 / California zone III [1411] NAD27 / California zone IV [1412] NAD27 / California zone V [1413] NAD27 / California zone VI [1414] NAD27 / California zone VII [1415] NAD27 / California zone VII (deprecated) [1416] NAD27 / Colorado Central [1417] NAD27 / Colorado North [1418] NAD27 / Colorado South [1419] NAD27 / Connecticut [1420] NAD27 / Conus Albers [1421] NAD27 / Cuba Norte [1422] NAD27 / Cuba Norte (deprecated) [1423] NAD27 / Cuba Sur [1424] NAD27 / Cuba Sur (deprecated) [1425] NAD27 / Delaware [1426] NAD27 / Florida East [1427] NAD27 / Florida North [1428] NAD27 / Florida West [1429] NAD27 / Georgia East [1430] NAD27 / Georgia West [1431] NAD27 / Guatemala Norte (deprecated) [1432] NAD27 / Guatemala Sur (deprecated) [1433] NAD27 / Idaho Central [1434] NAD27 / Idaho East [1435] NAD27 / Idaho West [1436] NAD27 / Illinois East [1437] NAD27 / Illinois West [1438] NAD27 / Indiana East [1439] NAD27 / Indiana West [1440] NAD27 / Iowa North [1441] NAD27 / Iowa South [1442] NAD27 / Kansas North [1443] NAD27 / Kansas South [1444] NAD27 / Kentucky North [1445] NAD27 / Kentucky South [1446] NAD27 / Louisiana North [1447] NAD27 / Louisiana Offshore [1448] NAD27 / Louisiana South [1449] NAD27 / MTM zone 1 [1450] NAD27 / MTM zone 10 [1451] NAD27 / MTM zone 2 [1452] NAD27 / MTM zone 3 [1453] NAD27 / MTM zone 4 [1454] NAD27 / MTM zone 5 [1455] NAD27 / MTM zone 6 [1456] NAD27 / MTQ Lambert [1457] NAD27 / Maine East [1458] NAD27 / Maine West [1459] NAD27 / Maryland [1460] NAD27 / Massachusetts Island [1461] NAD27 / Massachusetts Mainland [1462] NAD27 / Michigan East [1463] NAD27 / Michigan Old Central [1464] NAD27 / Michigan West [1465] NAD27 / Minnesota Central [1466] NAD27 / Minnesota North [1467] NAD27 / Minnesota South [1468] NAD27 / Mississippi East [1469] NAD27 / Mississippi West [1470] NAD27 / Missouri Central [1471] NAD27 / Missouri East [1472] NAD27 / Missouri West [1473] NAD27 / Montana Central [1474] NAD27 / Montana North [1475] NAD27 / Montana South [1476] NAD27 / Nebraska North [1477] NAD27 / Nebraska South [1478] NAD27 / Nevada Central [1479] NAD27 / Nevada East [1480] NAD27 / Nevada West [1481] NAD27 / New Brunswick Stereographic (NAD27) [1482] NAD27 / New Hampshire [1483] NAD27 / New Jersey [1484] NAD27 / New Mexico Central [1485] NAD27 / New Mexico East [1486] NAD27 / New Mexico West [1487] NAD27 / New York Central [1488] NAD27 / New York East [1489] NAD27 / New York Long Island [1490] NAD27 / New York Long Island (deprecated) [1491] NAD27 / New York West [1492] NAD27 / North Carolina [1493] NAD27 / North Dakota North [1494] NAD27 / North Dakota South [1495] NAD27 / Ohio North [1496] NAD27 / Ohio South [1497] NAD27 / Oklahoma North [1498] NAD27 / Oklahoma South [1499] NAD27 / Oregon North [1500] NAD27 / Oregon South [1501] NAD27 / Pennsylvania North [1502] NAD27 / Pennsylvania South [1503] NAD27 / Pennsylvania South (deprecated) [1504] NAD27 / Quebec Lambert [1505] NAD27 / Rhode Island [1506] NAD27 / Shackleford [1507] NAD27 / South Carolina North [1508] NAD27 / South Carolina South [1509] NAD27 / South Dakota North [1510] NAD27 / South Dakota South [1511] NAD27 / Tennessee [1512] NAD27 / Tennessee (deprecated) [1513] NAD27 / Texas Central [1514] NAD27 / Texas North [1515] NAD27 / Texas North Central [1516] NAD27 / Texas South [1517] NAD27 / Texas South Central [1518] NAD27 / UTM zone 10N [1519] NAD27 / UTM zone 11N [1520] NAD27 / UTM zone 12N [1521] NAD27 / UTM zone 13N [1522] NAD27 / UTM zone 14N [1523] NAD27 / UTM zone 15N [1524] NAD27 / UTM zone 16N [1525] NAD27 / UTM zone 17N [1526] NAD27 / UTM zone 18N [1527] NAD27 / UTM zone 19N [1528] NAD27 / UTM zone 1N [1529] NAD27 / UTM zone 20N [1530] NAD27 / UTM zone 21N [1531] NAD27 / UTM zone 22N [1532] NAD27 / UTM zone 2N [1533] NAD27 / UTM zone 3N [1534] NAD27 / UTM zone 4N [1535] NAD27 / UTM zone 59N [1536] NAD27 / UTM zone 5N [1537] NAD27 / UTM zone 60N [1538] NAD27 / UTM zone 6N [1539] NAD27 / UTM zone 7N [1540] NAD27 / UTM zone 8N [1541] NAD27 / UTM zone 9N [1542] NAD27 / Utah Central [1543] NAD27 / Utah North [1544] NAD27 / Utah South [1545] NAD27 / Vermont [1546] NAD27 / Virginia North [1547] NAD27 / Virginia South [1548] NAD27 / Washington North [1549] NAD27 / Washington South [1550] NAD27 / West Virginia North [1551] NAD27 / West Virginia South [1552] NAD27 / Wisconsin Central [1553] NAD27 / Wisconsin North [1554] NAD27 / Wisconsin South [1555] NAD27 / Wisconsin Transverse Mercator [1556] NAD27 / Wyoming East [1557] NAD27 / Wyoming East Central [1558] NAD27 / Wyoming West [1559] NAD27 / Wyoming West Central [1560] NAD27(76) / MTM zone 10 [1561] NAD27(76) / MTM zone 11 [1562] NAD27(76) / MTM zone 12 [1563] NAD27(76) / MTM zone 13 [1564] NAD27(76) / MTM zone 14 [1565] NAD27(76) / MTM zone 15 [1566] NAD27(76) / MTM zone 16 [1567] NAD27(76) / MTM zone 17 [1568] NAD27(76) / MTM zone 8 [1569] NAD27(76) / MTM zone 9 [1570] NAD27(76) / UTM zone 15N [1571] NAD27(76) / UTM zone 16N [1572] NAD27(76) / UTM zone 17N [1573] NAD27(76) / UTM zone 18N [1574] NAD27(CGQ77) / Quebec Lambert [1575] NAD27(CGQ77) / SCoPQ zone 10 [1576] NAD27(CGQ77) / SCoPQ zone 2 (deprecated) [1577] NAD27(CGQ77) / SCoPQ zone 3 [1578] NAD27(CGQ77) / SCoPQ zone 4 [1579] NAD27(CGQ77) / SCoPQ zone 5 [1580] NAD27(CGQ77) / SCoPQ zone 6 [1581] NAD27(CGQ77) / SCoPQ zone 7 [1582] NAD27(CGQ77) / SCoPQ zone 8 [1583] NAD27(CGQ77) / SCoPQ zone 9 [1584] NAD27(CGQ77) / UTM zone 17N [1585] NAD27(CGQ77) / UTM zone 18N [1586] NAD27(CGQ77) / UTM zone 19N [1587] NAD27(CGQ77) / UTM zone 20N [1588] NAD27(CGQ77) / UTM zone 21N [1589] NAD83 / Alabama East [1590] NAD83 / Alabama West [1591] NAD83 / Alaska Albers [1592] NAD83 / Alaska zone 1 [1593] NAD83 / Alaska zone 10 [1594] NAD83 / Alaska zone 2 [1595] NAD83 / Alaska zone 3 [1596] NAD83 / Alaska zone 4 [1597] NAD83 / Alaska zone 5 [1598] NAD83 / Alaska zone 6 [1599] NAD83 / Alaska zone 7 [1600] NAD83 / Alaska zone 8 [1601] NAD83 / Alaska zone 9 [1602] NAD83 / Alberta 10-TM (Forest) [1603] NAD83 / Alberta 10-TM (Resource) [1604] NAD83 / Alberta 3TM ref merid 111 W [1605] NAD83 / Alberta 3TM ref merid 114 W [1606] NAD83 / Alberta 3TM ref merid 117 W [1607] NAD83 / Alberta 3TM ref merid 120 W [1608] NAD83 / Alberta 3TM ref merid 120 W (deprecated) [1609] NAD83 / Arizona Central [1610] NAD83 / Arizona Central (ft) [1611] NAD83 / Arizona East [1612] NAD83 / Arizona East (ft) [1613] NAD83 / Arizona West [1614] NAD83 / Arizona West (ft) [1615] NAD83 / Arkansas North [1616] NAD83 / Arkansas North (ftUS) [1617] NAD83 / Arkansas South [1618] NAD83 / Arkansas South (ftUS) [1619] NAD83 / BC Albers [1620] NAD83 / BLM 10N (ftUS) [1621] NAD83 / BLM 11N (ftUS) [1622] NAD83 / BLM 12N (ftUS) [1623] NAD83 / BLM 13N (ftUS) [1624] NAD83 / BLM 14N (ftUS) [1625] NAD83 / BLM 15N (ftUS) [1626] NAD83 / BLM 16N (ftUS) [1627] NAD83 / BLM 17N (ftUS) [1628] NAD83 / BLM 18N (ftUS) [1629] NAD83 / BLM 19N (ftUS) [1630] NAD83 / BLM 1N (ftUS) [1631] NAD83 / BLM 2N (ftUS) [1632] NAD83 / BLM 3N (ftUS) [1633] NAD83 / BLM 4N (ftUS) [1634] NAD83 / BLM 59N (ftUS) [1635] NAD83 / BLM 5N (ftUS) [1636] NAD83 / BLM 60N (ftUS) [1637] NAD83 / BLM 6N (ftUS) [1638] NAD83 / BLM 7N (ftUS) [1639] NAD83 / BLM 8N (ftUS) [1640] NAD83 / BLM 9N (ftUS) [1641] NAD83 / California Albers [1642] NAD83 / California zone 1 [1643] NAD83 / California zone 1 (ftUS) [1644] NAD83 / California zone 2 [1645] NAD83 / California zone 2 (ftUS) [1646] NAD83 / California zone 3 [1647] NAD83 / California zone 3 (ftUS) [1648] NAD83 / California zone 4 [1649] NAD83 / California zone 4 (ftUS) [1650] NAD83 / California zone 5 [1651] NAD83 / California zone 5 (ftUS) [1652] NAD83 / California zone 6 [1653] NAD83 / California zone 6 (ftUS) [1654] NAD83 / Canada Atlas Lambert [1655] NAD83 / Colorado Central [1656] NAD83 / Colorado Central (ftUS) [1657] NAD83 / Colorado North [1658] NAD83 / Colorado North (ftUS) [1659] NAD83 / Colorado South [1660] NAD83 / Colorado South (ftUS) [1661] NAD83 / Connecticut [1662] NAD83 / Connecticut (ftUS) [1663] NAD83 / Conus Albers [1664] NAD83 / Delaware [1665] NAD83 / Delaware (ftUS) [1666] NAD83 / Florida East [1667] NAD83 / Florida East (ftUS) [1668] NAD83 / Florida GDL Albers [1669] NAD83 / Florida North [1670] NAD83 / Florida North (ftUS) [1671] NAD83 / Florida West [1672] NAD83 / Florida West (ftUS) [1673] NAD83 / Georgia East [1674] NAD83 / Georgia East (ftUS) [1675] NAD83 / Georgia West [1676] NAD83 / Georgia West (ftUS) [1677] NAD83 / Great Lakes Albers [1678] NAD83 / Great Lakes and St Lawrence Albers [1679] NAD83 / Hawaii zone 1 [1680] NAD83 / Hawaii zone 2 [1681] NAD83 / Hawaii zone 3 [1682] NAD83 / Hawaii zone 3 (ftUS) [1683] NAD83 / Hawaii zone 4 [1684] NAD83 / Hawaii zone 5 [1685] NAD83 / Idaho Central [1686] NAD83 / Idaho Central (ftUS) [1687] NAD83 / Idaho East [1688] NAD83 / Idaho East (ftUS) [1689] NAD83 / Idaho Transverse Mercator [1690] NAD83 / Idaho West [1691] NAD83 / Idaho West (ftUS) [1692] NAD83 / Illinois East [1693] NAD83 / Illinois East (ftUS) [1694] NAD83 / Illinois West [1695] NAD83 / Illinois West (ftUS) [1696] NAD83 / Indiana East [1697] NAD83 / Indiana East (ftUS) [1698] NAD83 / Indiana East (ftUS) (deprecated) [1699] NAD83 / Indiana West [1700] NAD83 / Indiana West (ftUS) [1701] NAD83 / Indiana West (ftUS) (deprecated) [1702] NAD83 / Iowa North [1703] NAD83 / Iowa North (ftUS) [1704] NAD83 / Iowa South [1705] NAD83 / Iowa South (ftUS) [1706] NAD83 / Kansas LCC [1707] NAD83 / Kansas LCC (ftUS) [1708] NAD83 / Kansas North [1709] NAD83 / Kansas North (ftUS) [1710] NAD83 / Kansas South [1711] NAD83 / Kansas South (ftUS) [1712] NAD83 / Kentucky North [1713] NAD83 / Kentucky North (deprecated) [1714] NAD83 / Kentucky North (ftUS) [1715] NAD83 / Kentucky Single Zone [1716] NAD83 / Kentucky Single Zone (ftUS) [1717] NAD83 / Kentucky South [1718] NAD83 / Kentucky South (ftUS) [1719] NAD83 / Louisiana North [1720] NAD83 / Louisiana North (ftUS) [1721] NAD83 / Louisiana Offshore [1722] NAD83 / Louisiana Offshore (ftUS) [1723] NAD83 / Louisiana South [1724] NAD83 / Louisiana South (ftUS) [1725] NAD83 / MTM zone 1 [1726] NAD83 / MTM zone 10 [1727] NAD83 / MTM zone 11 [1728] NAD83 / MTM zone 12 [1729] NAD83 / MTM zone 13 [1730] NAD83 / MTM zone 14 [1731] NAD83 / MTM zone 15 [1732] NAD83 / MTM zone 16 [1733] NAD83 / MTM zone 17 [1734] NAD83 / MTM zone 2 [1735] NAD83 / MTM zone 3 [1736] NAD83 / MTM zone 4 [1737] NAD83 / MTM zone 5 [1738] NAD83 / MTM zone 6 [1739] NAD83 / MTM zone 7 [1740] NAD83 / MTM zone 8 [1741] NAD83 / MTM zone 9 [1742] NAD83 / MTQ Lambert [1743] NAD83 / Maine CS2000 Central [1744] NAD83 / Maine CS2000 Central (deprecated) [1745] NAD83 / Maine CS2000 East [1746] NAD83 / Maine CS2000 West [1747] NAD83 / Maine East [1748] NAD83 / Maine East (ftUS) [1749] NAD83 / Maine East (ftUS) (deprecated) [1750] NAD83 / Maine West [1751] NAD83 / Maine West (ftUS) [1752] NAD83 / Maine West (ftUS) (deprecated) [1753] NAD83 / Maryland [1754] NAD83 / Maryland (ftUS) [1755] NAD83 / Massachusetts Island [1756] NAD83 / Massachusetts Island (ftUS) [1757] NAD83 / Massachusetts Mainland [1758] NAD83 / Massachusetts Mainland (ftUS) [1759] NAD83 / Michigan Central [1760] NAD83 / Michigan Central (ft) [1761] NAD83 / Michigan North [1762] NAD83 / Michigan North (ft) [1763] NAD83 / Michigan Oblique Mercator [1764] NAD83 / Michigan South [1765] NAD83 / Michigan South (ft) [1766] NAD83 / Minnesota Central [1767] NAD83 / Minnesota Central (ftUS) [1768] NAD83 / Minnesota Central (ftUS) (deprecated) [1769] NAD83 / Minnesota North [1770] NAD83 / Minnesota North (ftUS) [1771] NAD83 / Minnesota North (ftUS) (deprecated) [1772] NAD83 / Minnesota South [1773] NAD83 / Minnesota South (ftUS) [1774] NAD83 / Minnesota South (ftUS) (deprecated) [1775] NAD83 / Mississippi East [1776] NAD83 / Mississippi East (ftUS) [1777] NAD83 / Mississippi TM [1778] NAD83 / Mississippi West [1779] NAD83 / Mississippi West (ftUS) [1780] NAD83 / Missouri Central [1781] NAD83 / Missouri East [1782] NAD83 / Missouri West [1783] NAD83 / Montana [1784] NAD83 / Montana (ft) [1785] NAD83 / NCRS Las Vegas (ftUS) [1786] NAD83 / NCRS Las Vegas (m) [1787] NAD83 / NCRS Las Vegas high (ftUS) [1788] NAD83 / NCRS Las Vegas high (m) [1789] NAD83 / NWT Lambert [1790] NAD83 / Nebraska [1791] NAD83 / Nebraska (ftUS) [1792] NAD83 / Nebraska (ftUS) (deprecated) [1793] NAD83 / Nevada Central [1794] NAD83 / Nevada Central (ftUS) [1795] NAD83 / Nevada East [1796] NAD83 / Nevada East (ftUS) [1797] NAD83 / Nevada West [1798] NAD83 / Nevada West (ftUS) [1799] NAD83 / New Hampshire [1800] NAD83 / New Hampshire (ftUS) [1801] NAD83 / New Jersey [1802] NAD83 / New Jersey (ftUS) [1803] NAD83 / New Mexico Central [1804] NAD83 / New Mexico Central (ftUS) [1805] NAD83 / New Mexico East [1806] NAD83 / New Mexico East (ftUS) [1807] NAD83 / New Mexico West [1808] NAD83 / New Mexico West (ftUS) [1809] NAD83 / New York Central [1810] NAD83 / New York Central (ftUS) [1811] NAD83 / New York East [1812] NAD83 / New York East (ftUS) [1813] NAD83 / New York Long Island [1814] NAD83 / New York Long Island (ftUS) [1815] NAD83 / New York West [1816] NAD83 / New York West (ftUS) [1817] NAD83 / North Carolina [1818] NAD83 / North Carolina (ftUS) [1819] NAD83 / North Dakota North [1820] NAD83 / North Dakota North (ft) [1821] NAD83 / North Dakota South [1822] NAD83 / North Dakota South (ft) [1823] NAD83 / Ohio North [1824] NAD83 / Ohio North (ftUS) [1825] NAD83 / Ohio South [1826] NAD83 / Ohio South (ftUS) [1827] NAD83 / Oklahoma North [1828] NAD83 / Oklahoma North (ftUS) [1829] NAD83 / Oklahoma South [1830] NAD83 / Oklahoma South (ftUS) [1831] NAD83 / Ontario MNR Lambert [1832] NAD83 / Oregon GIC Lambert (ft) [1833] NAD83 / Oregon LCC (m) [1834] NAD83 / Oregon North [1835] NAD83 / Oregon North (ft) [1836] NAD83 / Oregon South [1837] NAD83 / Oregon South (ft) [1838] NAD83 / Pennsylvania North [1839] NAD83 / Pennsylvania North (ftUS) [1840] NAD83 / Pennsylvania South [1841] NAD83 / Pennsylvania South (ftUS) [1842] NAD83 / Puerto Rico & Virgin Is. [1843] NAD83 / Quebec Albers [1844] NAD83 / Quebec Lambert [1845] NAD83 / Rhode Island [1846] NAD83 / Rhode Island (ftUS) [1847] NAD83 / SCoPQ zone 2 (deprecated) [1848] NAD83 / South Carolina [1849] NAD83 / South Carolina (ft) [1850] NAD83 / South Dakota North [1851] NAD83 / South Dakota North (ftUS) [1852] NAD83 / South Dakota North (ftUS) (deprecated) [1853] NAD83 / South Dakota South [1854] NAD83 / South Dakota South (ftUS) [1855] NAD83 / Statistics Canada Lambert [1856] NAD83 / Tennessee [1857] NAD83 / Tennessee (ftUS) [1858] NAD83 / Teranet Ontario Lambert [1859] NAD83 / Texas Central [1860] NAD83 / Texas Central (ftUS) [1861] NAD83 / Texas Centric Albers Equal Area [1862] NAD83 / Texas Centric Lambert Conformal [1863] NAD83 / Texas North [1864] NAD83 / Texas North (ftUS) [1865] NAD83 / Texas North Central [1866] NAD83 / Texas North Central (ftUS) [1867] NAD83 / Texas South [1868] NAD83 / Texas South (ftUS) [1869] NAD83 / Texas South Central [1870] NAD83 / Texas South Central (ftUS) [1871] NAD83 / Texas State Mapping System [1872] NAD83 / UTM zone 10N [1873] NAD83 / UTM zone 11N [1874] NAD83 / UTM zone 12N [1875] NAD83 / UTM zone 13N [1876] NAD83 / UTM zone 14N [1877] NAD83 / UTM zone 15N [1878] NAD83 / UTM zone 16N [1879] NAD83 / UTM zone 17N [1880] NAD83 / UTM zone 18N [1881] NAD83 / UTM zone 19N [1882] NAD83 / UTM zone 1N [1883] NAD83 / UTM zone 20N [1884] NAD83 / UTM zone 21N [1885] NAD83 / UTM zone 22N [1886] NAD83 / UTM zone 23N [1887] NAD83 / UTM zone 2N [1888] NAD83 / UTM zone 3N [1889] NAD83 / UTM zone 4N [1890] NAD83 / UTM zone 59N [1891] NAD83 / UTM zone 5N [1892] NAD83 / UTM zone 60N [1893] NAD83 / UTM zone 6N [1894] NAD83 / UTM zone 7N [1895] NAD83 / UTM zone 8N [1896] NAD83 / UTM zone 9N [1897] NAD83 / Utah Central [1898] NAD83 / Utah Central (ft) [1899] NAD83 / Utah Central (ftUS) [1900] NAD83 / Utah North [1901] NAD83 / Utah North (ft) [1902] NAD83 / Utah North (ftUS) [1903] NAD83 / Utah South [1904] NAD83 / Utah South (ft) [1905] NAD83 / Utah South (ftUS) [1906] NAD83 / Vermont [1907] NAD83 / Vermont (ftUS) [1908] NAD83 / Virginia Lambert [1909] NAD83 / Virginia North [1910] NAD83 / Virginia North (ftUS) [1911] NAD83 / Virginia South [1912] NAD83 / Virginia South (ftUS) [1913] NAD83 / Washington North [1914] NAD83 / Washington North (ftUS) [1915] NAD83 / Washington South [1916] NAD83 / Washington South (ftUS) [1917] NAD83 / West Virginia North [1918] NAD83 / West Virginia North (ftUS) [1919] NAD83 / West Virginia North (ftUS) (deprecated) [1920] NAD83 / West Virginia South [1921] NAD83 / West Virginia South (ftUS) [1922] NAD83 / West Virginia South (ftUS) (deprecated) [1923] NAD83 / Wisconsin Central [1924] NAD83 / Wisconsin Central (ftUS) [1925] NAD83 / Wisconsin North [1926] NAD83 / Wisconsin North (ftUS) [1927] NAD83 / Wisconsin South [1928] NAD83 / Wisconsin South (ftUS) [1929] NAD83 / Wisconsin Transverse Mercator [1930] NAD83 / Wyoming East [1931] NAD83 / Wyoming East (ftUS) [1932] NAD83 / Wyoming East Central [1933] NAD83 / Wyoming East Central (ftUS) [1934] NAD83 / Wyoming West [1935] NAD83 / Wyoming West (ftUS) [1936] NAD83 / Wyoming West Central [1937] NAD83 / Wyoming West Central (ftUS) [1938] NAD83 / Yukon Albers [1939] NAD83(2011) / Alabama East [1940] NAD83(2011) / Alabama West [1941] NAD83(2011) / Alaska Albers [1942] NAD83(2011) / Alaska zone 1 [1943] NAD83(2011) / Alaska zone 10 [1944] NAD83(2011) / Alaska zone 2 [1945] NAD83(2011) / Alaska zone 3 [1946] NAD83(2011) / Alaska zone 4 [1947] NAD83(2011) / Alaska zone 5 [1948] NAD83(2011) / Alaska zone 6 [1949] NAD83(2011) / Alaska zone 7 [1950] NAD83(2011) / Alaska zone 8 [1951] NAD83(2011) / Alaska zone 9 [1952] NAD83(2011) / Arizona Central [1953] NAD83(2011) / Arizona Central (ft) [1954] NAD83(2011) / Arizona East [1955] NAD83(2011) / Arizona East (ft) [1956] NAD83(2011) / Arizona West [1957] NAD83(2011) / Arizona West (ft) [1958] NAD83(2011) / Arkansas North [1959] NAD83(2011) / Arkansas North (ftUS) [1960] NAD83(2011) / Arkansas South [1961] NAD83(2011) / Arkansas South (ftUS) [1962] NAD83(2011) / California Albers [1963] NAD83(2011) / California zone 1 [1964] NAD83(2011) / California zone 1 (ftUS) [1965] NAD83(2011) / California zone 2 [1966] NAD83(2011) / California zone 2 (ftUS) [1967] NAD83(2011) / California zone 3 [1968] NAD83(2011) / California zone 3 (ftUS) [1969] NAD83(2011) / California zone 4 [1970] NAD83(2011) / California zone 4 (ftUS) [1971] NAD83(2011) / California zone 5 [1972] NAD83(2011) / California zone 5 (ftUS) [1973] NAD83(2011) / California zone 6 [1974] NAD83(2011) / California zone 6 (ftUS) [1975] NAD83(2011) / Colorado Central [1976] NAD83(2011) / Colorado Central (ftUS) [1977] NAD83(2011) / Colorado North [1978] NAD83(2011) / Colorado North (ftUS) [1979] NAD83(2011) / Colorado South [1980] NAD83(2011) / Colorado South (ftUS) [1981] NAD83(2011) / Connecticut [1982] NAD83(2011) / Connecticut (ftUS) [1983] NAD83(2011) / Conus Albers [1984] NAD83(2011) / Delaware [1985] NAD83(2011) / Delaware (ftUS) [1986] NAD83(2011) / EPSG Arctic zone 5-29 [1987] NAD83(2011) / EPSG Arctic zone 5-31 [1988] NAD83(2011) / EPSG Arctic zone 6-14 [1989] NAD83(2011) / EPSG Arctic zone 6-16 [1990] NAD83(2011) / Florida East [1991] NAD83(2011) / Florida East (ftUS) [1992] NAD83(2011) / Florida GDL Albers [1993] NAD83(2011) / Florida North [1994] NAD83(2011) / Florida North (ftUS) [1995] NAD83(2011) / Florida West [1996] NAD83(2011) / Florida West (ftUS) [1997] NAD83(2011) / Georgia East [1998] NAD83(2011) / Georgia East (ftUS) [1999] NAD83(2011) / Georgia West [2000] NAD83(2011) / Georgia West (ftUS) [2001] NAD83(2011) / IaRCS zone 1 [2002] NAD83(2011) / IaRCS zone 10 [2003] NAD83(2011) / IaRCS zone 11 [2004] NAD83(2011) / IaRCS zone 12 [2005] NAD83(2011) / IaRCS zone 13 [2006] NAD83(2011) / IaRCS zone 14 [2007] NAD83(2011) / IaRCS zone 2 [2008] NAD83(2011) / IaRCS zone 3 [2009] NAD83(2011) / IaRCS zone 4 [2010] NAD83(2011) / IaRCS zone 5 [2011] NAD83(2011) / IaRCS zone 6 [2012] NAD83(2011) / IaRCS zone 7 [2013] NAD83(2011) / IaRCS zone 8 [2014] NAD83(2011) / IaRCS zone 9 [2015] NAD83(2011) / Idaho Central [2016] NAD83(2011) / Idaho Central (ftUS) [2017] NAD83(2011) / Idaho East [2018] NAD83(2011) / Idaho East (ftUS) [2019] NAD83(2011) / Idaho West [2020] NAD83(2011) / Idaho West (ftUS) [2021] NAD83(2011) / Illinois East [2022] NAD83(2011) / Illinois East (ftUS) [2023] NAD83(2011) / Illinois West [2024] NAD83(2011) / Illinois West (ftUS) [2025] NAD83(2011) / InGCS Adams (ftUS) [2026] NAD83(2011) / InGCS Adams (m) [2027] NAD83(2011) / InGCS Allen (ftUS) [2028] NAD83(2011) / InGCS Allen (m) [2029] NAD83(2011) / InGCS Bartholomew (ftUS) [2030] NAD83(2011) / InGCS Bartholomew (m) [2031] NAD83(2011) / InGCS Benton (ftUS) [2032] NAD83(2011) / InGCS Benton (m) [2033] NAD83(2011) / InGCS Blackford-Delaware (ftUS) [2034] NAD83(2011) / InGCS Blackford-Delaware (m) [2035] NAD83(2011) / InGCS Boone-Hendricks (ftUS) [2036] NAD83(2011) / InGCS Boone-Hendricks (m) [2037] NAD83(2011) / InGCS Brown (ftUS) [2038] NAD83(2011) / InGCS Brown (m) [2039] NAD83(2011) / InGCS Carroll (ftUS) [2040] NAD83(2011) / InGCS Carroll (m) [2041] NAD83(2011) / InGCS Cass (ftUS) [2042] NAD83(2011) / InGCS Cass (m) [2043] NAD83(2011) / InGCS Clark-Floyd-Scott (ftUS) [2044] NAD83(2011) / InGCS Clark-Floyd-Scott (m) [2045] NAD83(2011) / InGCS Clay (ftUS) [2046] NAD83(2011) / InGCS Clay (m) [2047] NAD83(2011) / InGCS Clinton (ftUS) [2048] NAD83(2011) / InGCS Clinton (m) [2049] NAD83(2011) / InGCS Crawford-Lawrence-Orange (ftUS) [2050] NAD83(2011) / InGCS Crawford-Lawrence-Orange (m) [2051] NAD83(2011) / InGCS Daviess-Greene (ftUS) [2052] NAD83(2011) / InGCS Daviess-Greene (m) [2053] NAD83(2011) / InGCS DeKalb (ftUS) [2054] NAD83(2011) / InGCS DeKalb (m) [2055] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (ftUS) [2056] NAD83(2011) / InGCS Dearborn-Ohio-Switzerland (m) [2057] NAD83(2011) / InGCS Decatur-Rush (ftUS) [2058] NAD83(2011) / InGCS Decatur-Rush (m) [2059] NAD83(2011) / InGCS Dubois-Martin (ftUS) [2060] NAD83(2011) / InGCS Dubois-Martin (m) [2061] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (ftUS) [2062] NAD83(2011) / InGCS Elkhart-Kosciusko-Wabash (m) [2063] NAD83(2011) / InGCS Fayette-Franklin-Union (ftUS) [2064] NAD83(2011) / InGCS Fayette-Franklin-Union (m) [2065] NAD83(2011) / InGCS Fountain-Warren (ftUS) [2066] NAD83(2011) / InGCS Fountain-Warren (m) [2067] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (ftUS) [2068] NAD83(2011) / InGCS Fulton-Marshall-St. Joseph (m) [2069] NAD83(2011) / InGCS Gibson (ftUS) [2070] NAD83(2011) / InGCS Gibson (m) [2071] NAD83(2011) / InGCS Grant (ftUS) [2072] NAD83(2011) / InGCS Grant (m) [2073] NAD83(2011) / InGCS Hamilton-Tipton (ftUS) [2074] NAD83(2011) / InGCS Hamilton-Tipton (m) [2075] NAD83(2011) / InGCS Hancock-Madison (ftUS) [2076] NAD83(2011) / InGCS Hancock-Madison (m) [2077] NAD83(2011) / InGCS Harrison-Washington (ftUS) [2078] NAD83(2011) / InGCS Harrison-Washington (m) [2079] NAD83(2011) / InGCS Henry (ftUS) [2080] NAD83(2011) / InGCS Henry (m) [2081] NAD83(2011) / InGCS Howard-Miami (ftUS) [2082] NAD83(2011) / InGCS Howard-Miami (m) [2083] NAD83(2011) / InGCS Huntington-Whitley (ftUS) [2084] NAD83(2011) / InGCS Huntington-Whitley (m) [2085] NAD83(2011) / InGCS Jackson (ftUS) [2086] NAD83(2011) / InGCS Jackson (m) [2087] NAD83(2011) / InGCS Jasper-Porter (ftUS) [2088] NAD83(2011) / InGCS Jasper-Porter (m) [2089] NAD83(2011) / InGCS Jay (ftUS) [2090] NAD83(2011) / InGCS Jay (m) [2091] NAD83(2011) / InGCS Jefferson (ftUS) [2092] NAD83(2011) / InGCS Jefferson (m) [2093] NAD83(2011) / InGCS Jennings (ftUS) [2094] NAD83(2011) / InGCS Jennings (m) [2095] NAD83(2011) / InGCS Johnson-Marion (ftUS) [2096] NAD83(2011) / InGCS Johnson-Marion (m) [2097] NAD83(2011) / InGCS Knox (ftUS) [2098] NAD83(2011) / InGCS Knox (m) [2099] NAD83(2011) / InGCS LaGrange-Noble (ftUS) [2100] NAD83(2011) / InGCS LaGrange-Noble (m) [2101] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (ftUS) [2102] NAD83(2011) / InGCS LaPorte-Pulaski-Starke (m) [2103] NAD83(2011) / InGCS Lake-Newton (ftUS) [2104] NAD83(2011) / InGCS Lake-Newton (m) [2105] NAD83(2011) / InGCS Monroe-Morgan (ftUS) [2106] NAD83(2011) / InGCS Monroe-Morgan (m) [2107] NAD83(2011) / InGCS Montgomery-Putnam (ftUS) [2108] NAD83(2011) / InGCS Montgomery-Putnam (m) [2109] NAD83(2011) / InGCS Owen (ftUS) [2110] NAD83(2011) / InGCS Owen (m) [2111] NAD83(2011) / InGCS Parke-Vermillion (ftUS) [2112] NAD83(2011) / InGCS Parke-Vermillion (m) [2113] NAD83(2011) / InGCS Perry (ftUS) [2114] NAD83(2011) / InGCS Perry (m) [2115] NAD83(2011) / InGCS Pike-Warrick (ftUS) [2116] NAD83(2011) / InGCS Pike-Warrick (m) [2117] NAD83(2011) / InGCS Posey (ftUS) [2118] NAD83(2011) / InGCS Posey (m) [2119] NAD83(2011) / InGCS Randolph-Wayne (ftUS) [2120] NAD83(2011) / InGCS Randolph-Wayne (m) [2121] NAD83(2011) / InGCS Ripley (ftUS) [2122] NAD83(2011) / InGCS Ripley (m) [2123] NAD83(2011) / InGCS Shelby (ftUS) [2124] NAD83(2011) / InGCS Shelby (m) [2125] NAD83(2011) / InGCS Spencer (ftUS) [2126] NAD83(2011) / InGCS Spencer (m) [2127] NAD83(2011) / InGCS Steuben (ftUS) [2128] NAD83(2011) / InGCS Steuben (m) [2129] NAD83(2011) / InGCS Sullivan (ftUS) [2130] NAD83(2011) / InGCS Sullivan (m) [2131] NAD83(2011) / InGCS Tippecanoe-White (ftUS) [2132] NAD83(2011) / InGCS Tippecanoe-White (m) [2133] NAD83(2011) / InGCS Vanderburgh (ftUS) [2134] NAD83(2011) / InGCS Vanderburgh (m) [2135] NAD83(2011) / InGCS Vigo (ftUS) [2136] NAD83(2011) / InGCS Vigo (m) [2137] NAD83(2011) / InGCS Wells (ftUS) [2138] NAD83(2011) / InGCS Wells (m) [2139] NAD83(2011) / Indiana East [2140] NAD83(2011) / Indiana East (ftUS) [2141] NAD83(2011) / Indiana West [2142] NAD83(2011) / Indiana West (ftUS) [2143] NAD83(2011) / Iowa North [2144] NAD83(2011) / Iowa North (ftUS) [2145] NAD83(2011) / Iowa South [2146] NAD83(2011) / Iowa South (ftUS) [2147] NAD83(2011) / KS RCS zone 1 [2148] NAD83(2011) / KS RCS zone 10 [2149] NAD83(2011) / KS RCS zone 11 [2150] NAD83(2011) / KS RCS zone 12 [2151] NAD83(2011) / KS RCS zone 13 [2152] NAD83(2011) / KS RCS zone 14 [2153] NAD83(2011) / KS RCS zone 15 [2154] NAD83(2011) / KS RCS zone 16 [2155] NAD83(2011) / KS RCS zone 17 [2156] NAD83(2011) / KS RCS zone 18 [2157] NAD83(2011) / KS RCS zone 19 [2158] NAD83(2011) / KS RCS zone 2 [2159] NAD83(2011) / KS RCS zone 20 [2160] NAD83(2011) / KS RCS zone 3 [2161] NAD83(2011) / KS RCS zone 4 [2162] NAD83(2011) / KS RCS zone 5 [2163] NAD83(2011) / KS RCS zone 6 [2164] NAD83(2011) / KS RCS zone 7 [2165] NAD83(2011) / KS RCS zone 8 [2166] NAD83(2011) / KS RCS zone 9 [2167] NAD83(2011) / Kansas LCC [2168] NAD83(2011) / Kansas LCC (ftUS) [2169] NAD83(2011) / Kansas North [2170] NAD83(2011) / Kansas North (ftUS) [2171] NAD83(2011) / Kansas South [2172] NAD83(2011) / Kansas South (ftUS) [2173] NAD83(2011) / Kentucky North [2174] NAD83(2011) / Kentucky North (ftUS) [2175] NAD83(2011) / Kentucky Single Zone [2176] NAD83(2011) / Kentucky Single Zone (ftUS) [2177] NAD83(2011) / Kentucky South [2178] NAD83(2011) / Kentucky South (ftUS) [2179] NAD83(2011) / Louisiana North [2180] NAD83(2011) / Louisiana North (ftUS) [2181] NAD83(2011) / Louisiana South [2182] NAD83(2011) / Louisiana South (ftUS) [2183] NAD83(2011) / Maine CS2000 Central [2184] NAD83(2011) / Maine CS2000 East [2185] NAD83(2011) / Maine CS2000 West [2186] NAD83(2011) / Maine East [2187] NAD83(2011) / Maine East (ftUS) [2188] NAD83(2011) / Maine West [2189] NAD83(2011) / Maine West (ftUS) [2190] NAD83(2011) / Maryland [2191] NAD83(2011) / Maryland (ftUS) [2192] NAD83(2011) / Massachusetts Island [2193] NAD83(2011) / Massachusetts Island (ftUS) [2194] NAD83(2011) / Massachusetts Mainland [2195] NAD83(2011) / Massachusetts Mainland (ftUS) [2196] NAD83(2011) / Michigan Central [2197] NAD83(2011) / Michigan Central (ft) [2198] NAD83(2011) / Michigan North [2199] NAD83(2011) / Michigan North (ft) [2200] NAD83(2011) / Michigan Oblique Mercator [2201] NAD83(2011) / Michigan South [2202] NAD83(2011) / Michigan South (ft) [2203] NAD83(2011) / Minnesota Central [2204] NAD83(2011) / Minnesota Central (ftUS) [2205] NAD83(2011) / Minnesota North [2206] NAD83(2011) / Minnesota North (ftUS) [2207] NAD83(2011) / Minnesota South [2208] NAD83(2011) / Minnesota South (ftUS) [2209] NAD83(2011) / Mississippi East [2210] NAD83(2011) / Mississippi East (ftUS) [2211] NAD83(2011) / Mississippi TM [2212] NAD83(2011) / Mississippi West [2213] NAD83(2011) / Mississippi West (ftUS) [2214] NAD83(2011) / Missouri Central [2215] NAD83(2011) / Missouri East [2216] NAD83(2011) / Missouri West [2217] NAD83(2011) / Montana [2218] NAD83(2011) / Montana (ft) [2219] NAD83(2011) / NCRS Las Vegas (ftUS) [2220] NAD83(2011) / NCRS Las Vegas (m) [2221] NAD83(2011) / NCRS Las Vegas high (ftUS) [2222] NAD83(2011) / NCRS Las Vegas high (m) [2223] NAD83(2011) / Nebraska [2224] NAD83(2011) / Nebraska (ftUS) [2225] NAD83(2011) / Nebraska (ftUS) (deprecated) [2226] NAD83(2011) / Nevada Central [2227] NAD83(2011) / Nevada Central (ftUS) [2228] NAD83(2011) / Nevada East [2229] NAD83(2011) / Nevada East (ftUS) [2230] NAD83(2011) / Nevada West [2231] NAD83(2011) / Nevada West (ftUS) [2232] NAD83(2011) / New Hampshire [2233] NAD83(2011) / New Hampshire (ftUS) [2234] NAD83(2011) / New Jersey [2235] NAD83(2011) / New Jersey (ftUS) [2236] NAD83(2011) / New Mexico Central [2237] NAD83(2011) / New Mexico Central (ftUS) [2238] NAD83(2011) / New Mexico East [2239] NAD83(2011) / New Mexico East (ftUS) [2240] NAD83(2011) / New Mexico West [2241] NAD83(2011) / New Mexico West (ftUS) [2242] NAD83(2011) / New York Central [2243] NAD83(2011) / New York Central (ftUS) [2244] NAD83(2011) / New York East [2245] NAD83(2011) / New York East (ftUS) [2246] NAD83(2011) / New York Long Island [2247] NAD83(2011) / New York Long Island (ftUS) [2248] NAD83(2011) / New York West [2249] NAD83(2011) / New York West (ftUS) [2250] NAD83(2011) / North Carolina [2251] NAD83(2011) / North Carolina (ftUS) [2252] NAD83(2011) / North Dakota North [2253] NAD83(2011) / North Dakota North (ft) [2254] NAD83(2011) / North Dakota South [2255] NAD83(2011) / North Dakota South (ft) [2256] NAD83(2011) / Ohio North [2257] NAD83(2011) / Ohio North (ftUS) [2258] NAD83(2011) / Ohio South [2259] NAD83(2011) / Ohio South (ftUS) [2260] NAD83(2011) / Oklahoma North [2261] NAD83(2011) / Oklahoma North (ftUS) [2262] NAD83(2011) / Oklahoma South [2263] NAD83(2011) / Oklahoma South (ftUS) [2264] NAD83(2011) / Oregon Baker zone (ft) [2265] NAD83(2011) / Oregon Baker zone (m) [2266] NAD83(2011) / Oregon Bend-Burns zone (ft) [2267] NAD83(2011) / Oregon Bend-Burns zone (m) [2268] NAD83(2011) / Oregon Bend-Klamath Falls zone (ft) [2269] NAD83(2011) / Oregon Bend-Klamath Falls zone (m) [2270] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (ft) [2271] NAD83(2011) / Oregon Bend-Redmond-Prineville zone (m) [2272] NAD83(2011) / Oregon Burns-Harper zone (ft) [2273] NAD83(2011) / Oregon Burns-Harper zone (m) [2274] NAD83(2011) / Oregon Canyon City-Burns zone (ft) [2275] NAD83(2011) / Oregon Canyon City-Burns zone (m) [2276] NAD83(2011) / Oregon Canyonville-Grants Pass zone (ft) [2277] NAD83(2011) / Oregon Canyonville-Grants Pass zone (m) [2278] NAD83(2011) / Oregon Coast Range North zone (ft) [2279] NAD83(2011) / Oregon Coast Range North zone (m) [2280] NAD83(2011) / Oregon Coast zone (ft) [2281] NAD83(2011) / Oregon Coast zone (m) [2282] NAD83(2011) / Oregon Columbia River East zone (ft) [2283] NAD83(2011) / Oregon Columbia River East zone (m) [2284] NAD83(2011) / Oregon Columbia River West zone (ft) [2285] NAD83(2011) / Oregon Columbia River West zone (m) [2286] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (ft) [2287] NAD83(2011) / Oregon Cottage Grove-Canyonville zone (m) [2288] NAD83(2011) / Oregon Dayville-Prairie City zone (ft) [2289] NAD83(2011) / Oregon Dayville-Prairie City zone (m) [2290] NAD83(2011) / Oregon Denio-Burns zone (ft) [2291] NAD83(2011) / Oregon Denio-Burns zone (m) [2292] NAD83(2011) / Oregon Dufur-Madras zone (ft) [2293] NAD83(2011) / Oregon Dufur-Madras zone (m) [2294] NAD83(2011) / Oregon Eugene zone (ft) [2295] NAD83(2011) / Oregon Eugene zone (m) [2296] NAD83(2011) / Oregon GIC Lambert (ft) [2297] NAD83(2011) / Oregon Grants Pass-Ashland zone (ft) [2298] NAD83(2011) / Oregon Grants Pass-Ashland zone (m) [2299] NAD83(2011) / Oregon Gresham-Warm Springs zone (ft) [2300] NAD83(2011) / Oregon Gresham-Warm Springs zone (m) [2301] NAD83(2011) / Oregon Halfway zone (ft) [2302] NAD83(2011) / Oregon Halfway zone (m) [2303] NAD83(2011) / Oregon LCC (m) [2304] NAD83(2011) / Oregon La Grande zone (ft) [2305] NAD83(2011) / Oregon La Grande zone (m) [2306] NAD83(2011) / Oregon Medford-Diamond Lake zone (ft) [2307] NAD83(2011) / Oregon Medford-Diamond Lake zone (m) [2308] NAD83(2011) / Oregon Mitchell zone (ft) [2309] NAD83(2011) / Oregon Mitchell zone (m) [2310] NAD83(2011) / Oregon North [2311] NAD83(2011) / Oregon North (ft) [2312] NAD83(2011) / Oregon North Central zone (ft) [2313] NAD83(2011) / Oregon North Central zone (m) [2314] NAD83(2011) / Oregon Ochoco Summit zone (ft) [2315] NAD83(2011) / Oregon Ochoco Summit zone (m) [2316] NAD83(2011) / Oregon Ontario zone (ft) [2317] NAD83(2011) / Oregon Ontario zone (m) [2318] NAD83(2011) / Oregon Owyhee zone (ft) [2319] NAD83(2011) / Oregon Owyhee zone (m) [2320] NAD83(2011) / Oregon Pendleton zone (ft) [2321] NAD83(2011) / Oregon Pendleton zone (m) [2322] NAD83(2011) / Oregon Pendleton-La Grande zone (ft) [2323] NAD83(2011) / Oregon Pendleton-La Grande zone (m) [2324] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (ft) [2325] NAD83(2011) / Oregon Pilot Rock-Ukiah zone (m) [2326] NAD83(2011) / Oregon Portland zone (ft) [2327] NAD83(2011) / Oregon Portland zone (m) [2328] NAD83(2011) / Oregon Prairie City-Brogan zone (ft) [2329] NAD83(2011) / Oregon Prairie City-Brogan zone (m) [2330] NAD83(2011) / Oregon Riley-Lakeview zone (ft) [2331] NAD83(2011) / Oregon Riley-Lakeview zone (m) [2332] NAD83(2011) / Oregon Salem zone (ft) [2333] NAD83(2011) / Oregon Salem zone (m) [2334] NAD83(2011) / Oregon Santiam Pass zone (ft) [2335] NAD83(2011) / Oregon Santiam Pass zone (m) [2336] NAD83(2011) / Oregon Siskiyou Pass zone (ft) [2337] NAD83(2011) / Oregon Siskiyou Pass zone (m) [2338] NAD83(2011) / Oregon South [2339] NAD83(2011) / Oregon South (ft) [2340] NAD83(2011) / Oregon Ukiah-Fox zone (ft) [2341] NAD83(2011) / Oregon Ukiah-Fox zone (m) [2342] NAD83(2011) / Oregon Wallowa zone (ft) [2343] NAD83(2011) / Oregon Wallowa zone (m) [2344] NAD83(2011) / Oregon Warner Highway zone (ft) [2345] NAD83(2011) / Oregon Warner Highway zone (m) [2346] NAD83(2011) / Oregon Willamette Pass zone (ft) [2347] NAD83(2011) / Oregon Willamette Pass zone (m) [2348] NAD83(2011) / PCCS zone 1 (ft) [2349] NAD83(2011) / PCCS zone 2 (ft) [2350] NAD83(2011) / PCCS zone 3 (ft) [2351] NAD83(2011) / PCCS zone 4 (ft) [2352] NAD83(2011) / Pennsylvania North [2353] NAD83(2011) / Pennsylvania North (ftUS) [2354] NAD83(2011) / Pennsylvania South [2355] NAD83(2011) / Pennsylvania South (ftUS) [2356] NAD83(2011) / Puerto Rico and Virgin Is. [2357] NAD83(2011) / RMTCRS Billings (ft) [2358] NAD83(2011) / RMTCRS Billings (m) [2359] NAD83(2011) / RMTCRS Blackfeet (ft) [2360] NAD83(2011) / RMTCRS Blackfeet (m) [2361] NAD83(2011) / RMTCRS Bobcat (ft) [2362] NAD83(2011) / RMTCRS Bobcat (m) [2363] NAD83(2011) / RMTCRS Crow (ft) [2364] NAD83(2011) / RMTCRS Crow (m) [2365] NAD83(2011) / RMTCRS Fort Belknap (ft) [2366] NAD83(2011) / RMTCRS Fort Belknap (m) [2367] NAD83(2011) / RMTCRS Fort Peck Assiniboine (ft) [2368] NAD83(2011) / RMTCRS Fort Peck Assiniboine (m) [2369] NAD83(2011) / RMTCRS Fort Peck Sioux (ft) [2370] NAD83(2011) / RMTCRS Fort Peck Sioux (m) [2371] NAD83(2011) / RMTCRS Milk River (ft) [2372] NAD83(2011) / RMTCRS Milk River (m) [2373] NAD83(2011) / RMTCRS St Mary (ft) [2374] NAD83(2011) / RMTCRS St Mary (m) [2375] NAD83(2011) / RMTCRS Wind River (ftUS) [2376] NAD83(2011) / RMTCRS Wind River (m) [2377] NAD83(2011) / Rhode Island [2378] NAD83(2011) / Rhode Island (ftUS) [2379] NAD83(2011) / San Francisco CS13 [2380] NAD83(2011) / San Francisco CS13 (deprecated) [2381] NAD83(2011) / San Francisco CS13 (ftUS) [2382] NAD83(2011) / San Francisco CS13 (ftUS) (deprecated) [2383] NAD83(2011) / South Carolina [2384] NAD83(2011) / South Carolina (ft) [2385] NAD83(2011) / South Dakota North [2386] NAD83(2011) / South Dakota North (ftUS) [2387] NAD83(2011) / South Dakota South [2388] NAD83(2011) / South Dakota South (ftUS) [2389] NAD83(2011) / Tennessee [2390] NAD83(2011) / Tennessee (ftUS) [2391] NAD83(2011) / Texas Central [2392] NAD83(2011) / Texas Central (ftUS) [2393] NAD83(2011) / Texas Centric Albers Equal Area [2394] NAD83(2011) / Texas Centric Lambert Conformal [2395] NAD83(2011) / Texas North [2396] NAD83(2011) / Texas North (ftUS) [2397] NAD83(2011) / Texas North Central [2398] NAD83(2011) / Texas North Central (ftUS) [2399] NAD83(2011) / Texas South [2400] NAD83(2011) / Texas South (ftUS) [2401] NAD83(2011) / Texas South Central [2402] NAD83(2011) / Texas South Central (ftUS) [2403] NAD83(2011) / UTM zone 10N [2404] NAD83(2011) / UTM zone 11N [2405] NAD83(2011) / UTM zone 12N [2406] NAD83(2011) / UTM zone 13N [2407] NAD83(2011) / UTM zone 14N [2408] NAD83(2011) / UTM zone 15N [2409] NAD83(2011) / UTM zone 16N [2410] NAD83(2011) / UTM zone 17N [2411] NAD83(2011) / UTM zone 18N [2412] NAD83(2011) / UTM zone 19N [2413] NAD83(2011) / UTM zone 1N [2414] NAD83(2011) / UTM zone 2N [2415] NAD83(2011) / UTM zone 3N [2416] NAD83(2011) / UTM zone 4N [2417] NAD83(2011) / UTM zone 59N [2418] NAD83(2011) / UTM zone 5N [2419] NAD83(2011) / UTM zone 60N [2420] NAD83(2011) / UTM zone 6N [2421] NAD83(2011) / UTM zone 7N [2422] NAD83(2011) / UTM zone 8N [2423] NAD83(2011) / UTM zone 9N [2424] NAD83(2011) / Utah Central [2425] NAD83(2011) / Utah Central (ftUS) [2426] NAD83(2011) / Utah North [2427] NAD83(2011) / Utah North (ftUS) [2428] NAD83(2011) / Utah South [2429] NAD83(2011) / Utah South (ftUS) [2430] NAD83(2011) / Vermont [2431] NAD83(2011) / Vermont (ftUS) [2432] NAD83(2011) / Virginia Lambert [2433] NAD83(2011) / Virginia North [2434] NAD83(2011) / Virginia North (ftUS) [2435] NAD83(2011) / Virginia South [2436] NAD83(2011) / Virginia South (ftUS) [2437] NAD83(2011) / WISCRS Adams and Juneau (ftUS) [2438] NAD83(2011) / WISCRS Adams and Juneau (m) [2439] NAD83(2011) / WISCRS Ashland (ftUS) [2440] NAD83(2011) / WISCRS Ashland (m) [2441] NAD83(2011) / WISCRS Barron (ftUS) [2442] NAD83(2011) / WISCRS Barron (m) [2443] NAD83(2011) / WISCRS Bayfield (ftUS) [2444] NAD83(2011) / WISCRS Bayfield (m) [2445] NAD83(2011) / WISCRS Brown (ftUS) [2446] NAD83(2011) / WISCRS Brown (m) [2447] NAD83(2011) / WISCRS Buffalo (ftUS) [2448] NAD83(2011) / WISCRS Buffalo (m) [2449] NAD83(2011) / WISCRS Burnett (ftUS) [2450] NAD83(2011) / WISCRS Burnett (m) [2451] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2452] NAD83(2011) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2453] NAD83(2011) / WISCRS Chippewa (ftUS) [2454] NAD83(2011) / WISCRS Chippewa (m) [2455] NAD83(2011) / WISCRS Clark (ftUS) [2456] NAD83(2011) / WISCRS Clark (m) [2457] NAD83(2011) / WISCRS Columbia (ftUS) [2458] NAD83(2011) / WISCRS Columbia (m) [2459] NAD83(2011) / WISCRS Crawford (ftUS) [2460] NAD83(2011) / WISCRS Crawford (m) [2461] NAD83(2011) / WISCRS Dane (ftUS) [2462] NAD83(2011) / WISCRS Dane (m) [2463] NAD83(2011) / WISCRS Dodge and Jefferson (ftUS) [2464] NAD83(2011) / WISCRS Dodge and Jefferson (m) [2465] NAD83(2011) / WISCRS Door (ftUS) [2466] NAD83(2011) / WISCRS Door (m) [2467] NAD83(2011) / WISCRS Douglas (ftUS) [2468] NAD83(2011) / WISCRS Douglas (m) [2469] NAD83(2011) / WISCRS Dunn (ftUS) [2470] NAD83(2011) / WISCRS Dunn (m) [2471] NAD83(2011) / WISCRS Eau Claire (ftUS) [2472] NAD83(2011) / WISCRS Eau Claire (m) [2473] NAD83(2011) / WISCRS Florence (ftUS) [2474] NAD83(2011) / WISCRS Florence (m) [2475] NAD83(2011) / WISCRS Forest (ftUS) [2476] NAD83(2011) / WISCRS Forest (m) [2477] NAD83(2011) / WISCRS Grant (ftUS) [2478] NAD83(2011) / WISCRS Grant (m) [2479] NAD83(2011) / WISCRS Green Lake and Marquette (ftUS) [2480] NAD83(2011) / WISCRS Green Lake and Marquette (m) [2481] NAD83(2011) / WISCRS Green and Lafayette (ftUS) [2482] NAD83(2011) / WISCRS Green and Lafayette (m) [2483] NAD83(2011) / WISCRS Iowa (ftUS) [2484] NAD83(2011) / WISCRS Iowa (m) [2485] NAD83(2011) / WISCRS Iron (ftUS) [2486] NAD83(2011) / WISCRS Iron (m) [2487] NAD83(2011) / WISCRS Jackson (ftUS) [2488] NAD83(2011) / WISCRS Jackson (m) [2489] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [2490] NAD83(2011) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [2491] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [2492] NAD83(2011) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [2493] NAD83(2011) / WISCRS La Crosse (ftUS) [2494] NAD83(2011) / WISCRS La Crosse (m) [2495] NAD83(2011) / WISCRS Langlade (ftUS) [2496] NAD83(2011) / WISCRS Langlade (m) [2497] NAD83(2011) / WISCRS Lincoln (ftUS) [2498] NAD83(2011) / WISCRS Lincoln (m) [2499] NAD83(2011) / WISCRS Marathon (ftUS) [2500] NAD83(2011) / WISCRS Marathon (m) [2501] NAD83(2011) / WISCRS Marinette (ftUS) [2502] NAD83(2011) / WISCRS Marinette (m) [2503] NAD83(2011) / WISCRS Menominee (ftUS) [2504] NAD83(2011) / WISCRS Menominee (m) [2505] NAD83(2011) / WISCRS Monroe (ftUS) [2506] NAD83(2011) / WISCRS Monroe (m) [2507] NAD83(2011) / WISCRS Oconto (ftUS) [2508] NAD83(2011) / WISCRS Oconto (m) [2509] NAD83(2011) / WISCRS Oneida (ftUS) [2510] NAD83(2011) / WISCRS Oneida (m) [2511] NAD83(2011) / WISCRS Pepin and Pierce (ftUS) [2512] NAD83(2011) / WISCRS Pepin and Pierce (m) [2513] NAD83(2011) / WISCRS Polk (ftUS) [2514] NAD83(2011) / WISCRS Polk (m) [2515] NAD83(2011) / WISCRS Portage (ftUS) [2516] NAD83(2011) / WISCRS Portage (m) [2517] NAD83(2011) / WISCRS Price (ftUS) [2518] NAD83(2011) / WISCRS Price (m) [2519] NAD83(2011) / WISCRS Richland (ftUS) [2520] NAD83(2011) / WISCRS Richland (m) [2521] NAD83(2011) / WISCRS Rock (ftUS) [2522] NAD83(2011) / WISCRS Rock (m) [2523] NAD83(2011) / WISCRS Rusk (ftUS) [2524] NAD83(2011) / WISCRS Rusk (m) [2525] NAD83(2011) / WISCRS Sauk (ftUS) [2526] NAD83(2011) / WISCRS Sauk (m) [2527] NAD83(2011) / WISCRS Sawyer (ftUS) [2528] NAD83(2011) / WISCRS Sawyer (m) [2529] NAD83(2011) / WISCRS Shawano (ftUS) [2530] NAD83(2011) / WISCRS Shawano (m) [2531] NAD83(2011) / WISCRS St. Croix (ftUS) [2532] NAD83(2011) / WISCRS St. Croix (m) [2533] NAD83(2011) / WISCRS Taylor (ftUS) [2534] NAD83(2011) / WISCRS Taylor (m) [2535] NAD83(2011) / WISCRS Trempealeau (ftUS) [2536] NAD83(2011) / WISCRS Trempealeau (m) [2537] NAD83(2011) / WISCRS Vernon (ftUS) [2538] NAD83(2011) / WISCRS Vernon (m) [2539] NAD83(2011) / WISCRS Vilas (ftUS) [2540] NAD83(2011) / WISCRS Vilas (m) [2541] NAD83(2011) / WISCRS Walworth (ftUS) [2542] NAD83(2011) / WISCRS Walworth (m) [2543] NAD83(2011) / WISCRS Washburn (ftUS) [2544] NAD83(2011) / WISCRS Washburn (m) [2545] NAD83(2011) / WISCRS Washington (ftUS) [2546] NAD83(2011) / WISCRS Washington (m) [2547] NAD83(2011) / WISCRS Waukesha (ftUS) [2548] NAD83(2011) / WISCRS Waukesha (m) [2549] NAD83(2011) / WISCRS Waupaca (ftUS) [2550] NAD83(2011) / WISCRS Waupaca (m) [2551] NAD83(2011) / WISCRS Waushara (ftUS) [2552] NAD83(2011) / WISCRS Waushara (m) [2553] NAD83(2011) / WISCRS Wood (ftUS) [2554] NAD83(2011) / WISCRS Wood (m) [2555] NAD83(2011) / Washington North [2556] NAD83(2011) / Washington North (ftUS) [2557] NAD83(2011) / Washington South [2558] NAD83(2011) / Washington South (ftUS) [2559] NAD83(2011) / West Virginia North [2560] NAD83(2011) / West Virginia North (ftUS) [2561] NAD83(2011) / West Virginia South [2562] NAD83(2011) / West Virginia South (ftUS) [2563] NAD83(2011) / Wisconsin Central [2564] NAD83(2011) / Wisconsin Central (deprecated) [2565] NAD83(2011) / Wisconsin Central (ftUS) [2566] NAD83(2011) / Wisconsin North [2567] NAD83(2011) / Wisconsin North (ftUS) [2568] NAD83(2011) / Wisconsin South [2569] NAD83(2011) / Wisconsin South (ftUS) [2570] NAD83(2011) / Wisconsin Transverse Mercator [2571] NAD83(2011) / Wyoming East [2572] NAD83(2011) / Wyoming East (ftUS) [2573] NAD83(2011) / Wyoming East Central [2574] NAD83(2011) / Wyoming East Central (ftUS) [2575] NAD83(2011) / Wyoming West [2576] NAD83(2011) / Wyoming West (ftUS) [2577] NAD83(2011) / Wyoming West Central [2578] NAD83(2011) / Wyoming West Central (ftUS) [2579] NAD83(CORS96) / Oregon Baker zone (ft) [2580] NAD83(CORS96) / Oregon Baker zone (m) [2581] NAD83(CORS96) / Oregon Bend-Burns zone (ft) [2582] NAD83(CORS96) / Oregon Bend-Burns zone (m) [2583] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (ft) [2584] NAD83(CORS96) / Oregon Bend-Klamath Falls zone (m) [2585] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (ft) [2586] NAD83(CORS96) / Oregon Bend-Redmond-Prineville zone (m) [2587] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (ft) [2588] NAD83(CORS96) / Oregon Canyonville-Grants Pass zone (m) [2589] NAD83(CORS96) / Oregon Coast zone (ft) [2590] NAD83(CORS96) / Oregon Coast zone (m) [2591] NAD83(CORS96) / Oregon Columbia River East zone (ft) [2592] NAD83(CORS96) / Oregon Columbia River East zone (m) [2593] NAD83(CORS96) / Oregon Columbia River West zone (ft) [2594] NAD83(CORS96) / Oregon Columbia River West zone (m) [2595] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (ft) [2596] NAD83(CORS96) / Oregon Cottage Grove-Canyonville zone (m) [2597] NAD83(CORS96) / Oregon Dufur-Madras zone (ft) [2598] NAD83(CORS96) / Oregon Dufur-Madras zone (m) [2599] NAD83(CORS96) / Oregon Eugene zone (ft) [2600] NAD83(CORS96) / Oregon Eugene zone (m) [2601] NAD83(CORS96) / Oregon GIC Lambert (ft) [2602] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (ft) [2603] NAD83(CORS96) / Oregon Grants Pass-Ashland zone (m) [2604] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (ft) [2605] NAD83(CORS96) / Oregon Gresham-Warm Springs zone (m) [2606] NAD83(CORS96) / Oregon LCC (m) [2607] NAD83(CORS96) / Oregon La Grande zone (ft) [2608] NAD83(CORS96) / Oregon La Grande zone (m) [2609] NAD83(CORS96) / Oregon North [2610] NAD83(CORS96) / Oregon North (ft) [2611] NAD83(CORS96) / Oregon Ontario zone (ft) [2612] NAD83(CORS96) / Oregon Ontario zone (m) [2613] NAD83(CORS96) / Oregon Pendleton zone (ft) [2614] NAD83(CORS96) / Oregon Pendleton zone (m) [2615] NAD83(CORS96) / Oregon Pendleton-La Grande zone (ft) [2616] NAD83(CORS96) / Oregon Pendleton-La Grande zone (m) [2617] NAD83(CORS96) / Oregon Portland zone (ft) [2618] NAD83(CORS96) / Oregon Portland zone (m) [2619] NAD83(CORS96) / Oregon Salem zone (ft) [2620] NAD83(CORS96) / Oregon Salem zone (m) [2621] NAD83(CORS96) / Oregon Santiam Pass zone (ft) [2622] NAD83(CORS96) / Oregon Santiam Pass zone (m) [2623] NAD83(CORS96) / Oregon South [2624] NAD83(CORS96) / Oregon South (ft) [2625] NAD83(CORS96) / Puerto Rico and Virgin Is. [2626] NAD83(CSRS) / Alberta 10-TM (Forest) [2627] NAD83(CSRS) / Alberta 10-TM (Resource) [2628] NAD83(CSRS) / Alberta 3TM ref merid 111 W [2629] NAD83(CSRS) / Alberta 3TM ref merid 114 W [2630] NAD83(CSRS) / Alberta 3TM ref merid 117 W [2631] NAD83(CSRS) / Alberta 3TM ref merid 120 W [2632] NAD83(CSRS) / Alberta 3TM ref merid 120 W (deprecated) [2633] NAD83(CSRS) / BC Albers [2634] NAD83(CSRS) / Canada Atlas Lambert [2635] NAD83(CSRS) / EPSG Arctic zone 1-23 [2636] NAD83(CSRS) / EPSG Arctic zone 2-14 [2637] NAD83(CSRS) / EPSG Arctic zone 2-16 [2638] NAD83(CSRS) / EPSG Arctic zone 3-25 [2639] NAD83(CSRS) / EPSG Arctic zone 3-27 [2640] NAD83(CSRS) / EPSG Arctic zone 3-29 [2641] NAD83(CSRS) / EPSG Arctic zone 4-14 [2642] NAD83(CSRS) / EPSG Arctic zone 4-16 [2643] NAD83(CSRS) / EPSG Arctic zone 4-18 [2644] NAD83(CSRS) / EPSG Arctic zone 5-33 [2645] NAD83(CSRS) / EPSG Arctic zone 5-35 [2646] NAD83(CSRS) / EPSG Arctic zone 5-37 [2647] NAD83(CSRS) / EPSG Arctic zone 5-39 [2648] NAD83(CSRS) / EPSG Arctic zone 6-18 [2649] NAD83(CSRS) / EPSG Arctic zone 6-20 [2650] NAD83(CSRS) / EPSG Arctic zone 6-22 [2651] NAD83(CSRS) / EPSG Arctic zone 6-24 [2652] NAD83(CSRS) / MTM zone 1 [2653] NAD83(CSRS) / MTM zone 10 [2654] NAD83(CSRS) / MTM zone 11 [2655] NAD83(CSRS) / MTM zone 12 [2656] NAD83(CSRS) / MTM zone 13 [2657] NAD83(CSRS) / MTM zone 14 [2658] NAD83(CSRS) / MTM zone 15 [2659] NAD83(CSRS) / MTM zone 16 [2660] NAD83(CSRS) / MTM zone 17 [2661] NAD83(CSRS) / MTM zone 2 [2662] NAD83(CSRS) / MTM zone 3 [2663] NAD83(CSRS) / MTM zone 4 [2664] NAD83(CSRS) / MTM zone 5 [2665] NAD83(CSRS) / MTM zone 6 [2666] NAD83(CSRS) / MTM zone 7 [2667] NAD83(CSRS) / MTM zone 8 [2668] NAD83(CSRS) / MTM zone 9 [2669] NAD83(CSRS) / MTQ Lambert [2670] NAD83(CSRS) / NWT Lambert [2671] NAD83(CSRS) / New Brunswick Stereographic [2672] NAD83(CSRS) / Ontario MNR Lambert [2673] NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83) [2674] NAD83(CSRS) / Quebec Albers [2675] NAD83(CSRS) / Quebec Lambert [2676] NAD83(CSRS) / SCoPQ zone 2 (deprecated) [2677] NAD83(CSRS) / Statistics Canada Lambert [2678] NAD83(CSRS) / Teranet Ontario Lambert [2679] NAD83(CSRS) / UTM zone 10N [2680] NAD83(CSRS) / UTM zone 11N [2681] NAD83(CSRS) / UTM zone 12N [2682] NAD83(CSRS) / UTM zone 13N [2683] NAD83(CSRS) / UTM zone 14N [2684] NAD83(CSRS) / UTM zone 15N [2685] NAD83(CSRS) / UTM zone 16N [2686] NAD83(CSRS) / UTM zone 17N [2687] NAD83(CSRS) / UTM zone 18N [2688] NAD83(CSRS) / UTM zone 19N [2689] NAD83(CSRS) / UTM zone 20N [2690] NAD83(CSRS) / UTM zone 21N [2691] NAD83(CSRS) / UTM zone 22N [2692] NAD83(CSRS) / UTM zone 7N [2693] NAD83(CSRS) / UTM zone 8N [2694] NAD83(CSRS) / UTM zone 9N [2695] NAD83(CSRS) / Yukon Albers [2696] NAD83(CSRS)v6 / MTM Nova Scotia zone 4 [2697] NAD83(CSRS)v6 / MTM Nova Scotia zone 5 [2698] NAD83(CSRS98) / MTM zone 10 (deprecated) [2699] NAD83(CSRS98) / MTM zone 3 (deprecated) [2700] NAD83(CSRS98) / MTM zone 4 (deprecated) [2701] NAD83(CSRS98) / MTM zone 5 (deprecated) [2702] NAD83(CSRS98) / MTM zone 6 (deprecated) [2703] NAD83(CSRS98) / MTM zone 7 (deprecated) [2704] NAD83(CSRS98) / MTM zone 8 (deprecated) [2705] NAD83(CSRS98) / MTM zone 9 (deprecated) [2706] NAD83(CSRS98) / New Brunswick Stereo (deprecated) [2707] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2708] NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83) (deprecated) [2709] NAD83(CSRS98) / SCoPQ zone 2 (deprecated) [2710] NAD83(CSRS98) / UTM zone 11N (deprecated) [2711] NAD83(CSRS98) / UTM zone 12N (deprecated) [2712] NAD83(CSRS98) / UTM zone 13N (deprecated) [2713] NAD83(CSRS98) / UTM zone 17N (deprecated) [2714] NAD83(CSRS98) / UTM zone 18N (deprecated) [2715] NAD83(CSRS98) / UTM zone 19N (deprecated) [2716] NAD83(CSRS98) / UTM zone 20N (deprecated) [2717] NAD83(CSRS98) / UTM zone 21N (deprecated) [2718] NAD83(HARN) / Alabama East [2719] NAD83(HARN) / Alabama West [2720] NAD83(HARN) / Arizona Central [2721] NAD83(HARN) / Arizona Central (ft) [2722] NAD83(HARN) / Arizona East [2723] NAD83(HARN) / Arizona East (ft) [2724] NAD83(HARN) / Arizona West [2725] NAD83(HARN) / Arizona West (ft) [2726] NAD83(HARN) / Arkansas North [2727] NAD83(HARN) / Arkansas North (ftUS) [2728] NAD83(HARN) / Arkansas South [2729] NAD83(HARN) / Arkansas South (ftUS) [2730] NAD83(HARN) / California Albers [2731] NAD83(HARN) / California zone 1 [2732] NAD83(HARN) / California zone 1 (ftUS) [2733] NAD83(HARN) / California zone 2 [2734] NAD83(HARN) / California zone 2 (ftUS) [2735] NAD83(HARN) / California zone 3 [2736] NAD83(HARN) / California zone 3 (ftUS) [2737] NAD83(HARN) / California zone 4 [2738] NAD83(HARN) / California zone 4 (ftUS) [2739] NAD83(HARN) / California zone 5 [2740] NAD83(HARN) / California zone 5 (ftUS) [2741] NAD83(HARN) / California zone 6 [2742] NAD83(HARN) / California zone 6 (ftUS) [2743] NAD83(HARN) / Colorado Central [2744] NAD83(HARN) / Colorado Central (ftUS) [2745] NAD83(HARN) / Colorado North [2746] NAD83(HARN) / Colorado North (ftUS) [2747] NAD83(HARN) / Colorado South [2748] NAD83(HARN) / Colorado South (ftUS) [2749] NAD83(HARN) / Connecticut [2750] NAD83(HARN) / Connecticut (ftUS) [2751] NAD83(HARN) / Conus Albers [2752] NAD83(HARN) / Delaware [2753] NAD83(HARN) / Delaware (ftUS) [2754] NAD83(HARN) / Florida East [2755] NAD83(HARN) / Florida East (ftUS) [2756] NAD83(HARN) / Florida GDL Albers [2757] NAD83(HARN) / Florida North [2758] NAD83(HARN) / Florida North (ftUS) [2759] NAD83(HARN) / Florida West [2760] NAD83(HARN) / Florida West (ftUS) [2761] NAD83(HARN) / Georgia East [2762] NAD83(HARN) / Georgia East (ftUS) [2763] NAD83(HARN) / Georgia West [2764] NAD83(HARN) / Georgia West (ftUS) [2765] NAD83(HARN) / Guam Map Grid [2766] NAD83(HARN) / Hawaii zone 1 [2767] NAD83(HARN) / Hawaii zone 2 [2768] NAD83(HARN) / Hawaii zone 3 [2769] NAD83(HARN) / Hawaii zone 3 (ftUS) [2770] NAD83(HARN) / Hawaii zone 4 [2771] NAD83(HARN) / Hawaii zone 5 [2772] NAD83(HARN) / Idaho Central [2773] NAD83(HARN) / Idaho Central (ftUS) [2774] NAD83(HARN) / Idaho East [2775] NAD83(HARN) / Idaho East (ftUS) [2776] NAD83(HARN) / Idaho West [2777] NAD83(HARN) / Idaho West (ftUS) [2778] NAD83(HARN) / Illinois East [2779] NAD83(HARN) / Illinois East (ftUS) [2780] NAD83(HARN) / Illinois West [2781] NAD83(HARN) / Illinois West (ftUS) [2782] NAD83(HARN) / Indiana East [2783] NAD83(HARN) / Indiana East (ftUS) [2784] NAD83(HARN) / Indiana East (ftUS) (deprecated) [2785] NAD83(HARN) / Indiana West [2786] NAD83(HARN) / Indiana West (ftUS) [2787] NAD83(HARN) / Indiana West (ftUS) (deprecated) [2788] NAD83(HARN) / Iowa North [2789] NAD83(HARN) / Iowa North (ftUS) [2790] NAD83(HARN) / Iowa South [2791] NAD83(HARN) / Iowa South (ftUS) [2792] NAD83(HARN) / Kansas North [2793] NAD83(HARN) / Kansas North (ftUS) [2794] NAD83(HARN) / Kansas South [2795] NAD83(HARN) / Kansas South (ftUS) [2796] NAD83(HARN) / Kentucky North [2797] NAD83(HARN) / Kentucky North (ftUS) [2798] NAD83(HARN) / Kentucky Single Zone [2799] NAD83(HARN) / Kentucky Single Zone (ftUS) [2800] NAD83(HARN) / Kentucky South [2801] NAD83(HARN) / Kentucky South (ftUS) [2802] NAD83(HARN) / Louisiana North [2803] NAD83(HARN) / Louisiana North (ftUS) [2804] NAD83(HARN) / Louisiana South [2805] NAD83(HARN) / Louisiana South (ftUS) [2806] NAD83(HARN) / Maine CS2000 Central [2807] NAD83(HARN) / Maine CS2000 Central (deprecated) [2808] NAD83(HARN) / Maine CS2000 East [2809] NAD83(HARN) / Maine CS2000 West [2810] NAD83(HARN) / Maine East [2811] NAD83(HARN) / Maine East (ftUS) [2812] NAD83(HARN) / Maine East (ftUS) (deprecated) [2813] NAD83(HARN) / Maine West [2814] NAD83(HARN) / Maine West (ftUS) [2815] NAD83(HARN) / Maine West (ftUS) (deprecated) [2816] NAD83(HARN) / Maryland [2817] NAD83(HARN) / Maryland (ftUS) [2818] NAD83(HARN) / Massachusetts Island [2819] NAD83(HARN) / Massachusetts Island (ftUS) [2820] NAD83(HARN) / Massachusetts Mainland [2821] NAD83(HARN) / Massachusetts Mainland (ftUS) [2822] NAD83(HARN) / Michigan Central [2823] NAD83(HARN) / Michigan Central (ft) [2824] NAD83(HARN) / Michigan North [2825] NAD83(HARN) / Michigan North (ft) [2826] NAD83(HARN) / Michigan Oblique Mercator [2827] NAD83(HARN) / Michigan South [2828] NAD83(HARN) / Michigan South (ft) [2829] NAD83(HARN) / Minnesota Central [2830] NAD83(HARN) / Minnesota Central (ftUS) [2831] NAD83(HARN) / Minnesota Central (ftUS) (deprecated) [2832] NAD83(HARN) / Minnesota North [2833] NAD83(HARN) / Minnesota North (ftUS) [2834] NAD83(HARN) / Minnesota North (ftUS) (deprecated) [2835] NAD83(HARN) / Minnesota South [2836] NAD83(HARN) / Minnesota South (ftUS) [2837] NAD83(HARN) / Minnesota South (ftUS) (deprecated) [2838] NAD83(HARN) / Mississippi East [2839] NAD83(HARN) / Mississippi East (ftUS) [2840] NAD83(HARN) / Mississippi TM [2841] NAD83(HARN) / Mississippi West [2842] NAD83(HARN) / Mississippi West (ftUS) [2843] NAD83(HARN) / Missouri Central [2844] NAD83(HARN) / Missouri East [2845] NAD83(HARN) / Missouri West [2846] NAD83(HARN) / Montana [2847] NAD83(HARN) / Montana (ft) [2848] NAD83(HARN) / Nebraska [2849] NAD83(HARN) / Nebraska (ftUS) [2850] NAD83(HARN) / Nebraska (ftUS) (deprecated) [2851] NAD83(HARN) / Nevada Central [2852] NAD83(HARN) / Nevada Central (ftUS) [2853] NAD83(HARN) / Nevada East [2854] NAD83(HARN) / Nevada East (ftUS) [2855] NAD83(HARN) / Nevada West [2856] NAD83(HARN) / Nevada West (ftUS) [2857] NAD83(HARN) / New Hampshire [2858] NAD83(HARN) / New Hampshire (ftUS) [2859] NAD83(HARN) / New Jersey [2860] NAD83(HARN) / New Jersey (ftUS) [2861] NAD83(HARN) / New Mexico Central [2862] NAD83(HARN) / New Mexico Central (ftUS) [2863] NAD83(HARN) / New Mexico East [2864] NAD83(HARN) / New Mexico East (ftUS) [2865] NAD83(HARN) / New Mexico West [2866] NAD83(HARN) / New Mexico West (ftUS) [2867] NAD83(HARN) / New York Central [2868] NAD83(HARN) / New York Central (ftUS) [2869] NAD83(HARN) / New York East [2870] NAD83(HARN) / New York East (ftUS) [2871] NAD83(HARN) / New York Long Island [2872] NAD83(HARN) / New York Long Island (ftUS) [2873] NAD83(HARN) / New York West [2874] NAD83(HARN) / New York West (ftUS) [2875] NAD83(HARN) / North Carolina [2876] NAD83(HARN) / North Carolina (ftUS) [2877] NAD83(HARN) / North Carolina (ftUS) (deprecated) [2878] NAD83(HARN) / North Dakota North [2879] NAD83(HARN) / North Dakota North (ft) [2880] NAD83(HARN) / North Dakota South [2881] NAD83(HARN) / North Dakota South (ft) [2882] NAD83(HARN) / Ohio North [2883] NAD83(HARN) / Ohio North (ftUS) [2884] NAD83(HARN) / Ohio South [2885] NAD83(HARN) / Ohio South (ftUS) [2886] NAD83(HARN) / Oklahoma North [2887] NAD83(HARN) / Oklahoma North (ftUS) [2888] NAD83(HARN) / Oklahoma South [2889] NAD83(HARN) / Oklahoma South (ftUS) [2890] NAD83(HARN) / Oregon GIC Lambert (ft) [2891] NAD83(HARN) / Oregon LCC (m) [2892] NAD83(HARN) / Oregon North [2893] NAD83(HARN) / Oregon North (ft) [2894] NAD83(HARN) / Oregon South [2895] NAD83(HARN) / Oregon South (ft) [2896] NAD83(HARN) / Pennsylvania North [2897] NAD83(HARN) / Pennsylvania North (ftUS) [2898] NAD83(HARN) / Pennsylvania South [2899] NAD83(HARN) / Pennsylvania South (ftUS) [2900] NAD83(HARN) / Puerto Rico and Virgin Is. [2901] NAD83(HARN) / Rhode Island [2902] NAD83(HARN) / Rhode Island (ftUS) [2903] NAD83(HARN) / South Carolina [2904] NAD83(HARN) / South Carolina (ft) [2905] NAD83(HARN) / South Dakota North [2906] NAD83(HARN) / South Dakota North (ftUS) [2907] NAD83(HARN) / South Dakota South [2908] NAD83(HARN) / South Dakota South (ftUS) [2909] NAD83(HARN) / Tennessee [2910] NAD83(HARN) / Tennessee (ftUS) [2911] NAD83(HARN) / Texas Central [2912] NAD83(HARN) / Texas Central (ftUS) [2913] NAD83(HARN) / Texas Centric Albers Equal Area [2914] NAD83(HARN) / Texas Centric Lambert Conformal [2915] NAD83(HARN) / Texas North [2916] NAD83(HARN) / Texas North (ftUS) [2917] NAD83(HARN) / Texas North Central [2918] NAD83(HARN) / Texas North Central (ftUS) [2919] NAD83(HARN) / Texas South [2920] NAD83(HARN) / Texas South (ftUS) [2921] NAD83(HARN) / Texas South Central [2922] NAD83(HARN) / Texas South Central (ftUS) [2923] NAD83(HARN) / UTM zone 10N [2924] NAD83(HARN) / UTM zone 11N [2925] NAD83(HARN) / UTM zone 12N [2926] NAD83(HARN) / UTM zone 13N [2927] NAD83(HARN) / UTM zone 14N [2928] NAD83(HARN) / UTM zone 15N [2929] NAD83(HARN) / UTM zone 16N [2930] NAD83(HARN) / UTM zone 17N [2931] NAD83(HARN) / UTM zone 18N [2932] NAD83(HARN) / UTM zone 19N [2933] NAD83(HARN) / UTM zone 2S [2934] NAD83(HARN) / UTM zone 4N [2935] NAD83(HARN) / UTM zone 59S (deprecated) [2936] NAD83(HARN) / UTM zone 5N [2937] NAD83(HARN) / Utah Central [2938] NAD83(HARN) / Utah Central (ft) [2939] NAD83(HARN) / Utah Central (ftUS) [2940] NAD83(HARN) / Utah North [2941] NAD83(HARN) / Utah North (ft) [2942] NAD83(HARN) / Utah North (ftUS) [2943] NAD83(HARN) / Utah South [2944] NAD83(HARN) / Utah South (ft) [2945] NAD83(HARN) / Utah South (ftUS) [2946] NAD83(HARN) / Vermont [2947] NAD83(HARN) / Vermont (ftUS) [2948] NAD83(HARN) / Virginia Lambert [2949] NAD83(HARN) / Virginia North [2950] NAD83(HARN) / Virginia North (ftUS) [2951] NAD83(HARN) / Virginia South [2952] NAD83(HARN) / Virginia South (ftUS) [2953] NAD83(HARN) / WISCRS Adams and Juneau (ftUS) [2954] NAD83(HARN) / WISCRS Adams and Juneau (m) [2955] NAD83(HARN) / WISCRS Ashland (ftUS) [2956] NAD83(HARN) / WISCRS Ashland (m) [2957] NAD83(HARN) / WISCRS Barron (ftUS) [2958] NAD83(HARN) / WISCRS Barron (m) [2959] NAD83(HARN) / WISCRS Bayfield (ftUS) [2960] NAD83(HARN) / WISCRS Bayfield (m) [2961] NAD83(HARN) / WISCRS Brown (ftUS) [2962] NAD83(HARN) / WISCRS Brown (m) [2963] NAD83(HARN) / WISCRS Buffalo (ftUS) [2964] NAD83(HARN) / WISCRS Buffalo (m) [2965] NAD83(HARN) / WISCRS Burnett (ftUS) [2966] NAD83(HARN) / WISCRS Burnett (m) [2967] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (ftUS) [2968] NAD83(HARN) / WISCRS Calumet, Fond du Lac, Outagamie and Winnebago (m) [2969] NAD83(HARN) / WISCRS Chippewa (ftUS) [2970] NAD83(HARN) / WISCRS Chippewa (m) [2971] NAD83(HARN) / WISCRS Clark (ftUS) [2972] NAD83(HARN) / WISCRS Clark (m) [2973] NAD83(HARN) / WISCRS Columbia (ftUS) [2974] NAD83(HARN) / WISCRS Columbia (m) [2975] NAD83(HARN) / WISCRS Crawford (ftUS) [2976] NAD83(HARN) / WISCRS Crawford (m) [2977] NAD83(HARN) / WISCRS Dane (ftUS) [2978] NAD83(HARN) / WISCRS Dane (m) [2979] NAD83(HARN) / WISCRS Dodge and Jefferson (ftUS) [2980] NAD83(HARN) / WISCRS Dodge and Jefferson (m) [2981] NAD83(HARN) / WISCRS Door (ftUS) [2982] NAD83(HARN) / WISCRS Door (m) [2983] NAD83(HARN) / WISCRS Douglas (ftUS) [2984] NAD83(HARN) / WISCRS Douglas (m) [2985] NAD83(HARN) / WISCRS Dunn (ftUS) [2986] NAD83(HARN) / WISCRS Dunn (m) [2987] NAD83(HARN) / WISCRS Eau Claire (ftUS) [2988] NAD83(HARN) / WISCRS Eau Claire (m) [2989] NAD83(HARN) / WISCRS Florence (ftUS) [2990] NAD83(HARN) / WISCRS Florence (m) [2991] NAD83(HARN) / WISCRS Forest (ftUS) [2992] NAD83(HARN) / WISCRS Forest (m) [2993] NAD83(HARN) / WISCRS Grant (ftUS) [2994] NAD83(HARN) / WISCRS Grant (m) [2995] NAD83(HARN) / WISCRS Green Lake and Marquette (ftUS) [2996] NAD83(HARN) / WISCRS Green Lake and Marquette (m) [2997] NAD83(HARN) / WISCRS Green and Lafayette (ftUS) [2998] NAD83(HARN) / WISCRS Green and Lafayette (m) [2999] NAD83(HARN) / WISCRS Iowa (ftUS) [3000] NAD83(HARN) / WISCRS Iowa (m) [3001] NAD83(HARN) / WISCRS Iron (ftUS) [3002] NAD83(HARN) / WISCRS Iron (m) [3003] NAD83(HARN) / WISCRS Jackson (ftUS) [3004] NAD83(HARN) / WISCRS Jackson (m) [3005] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (ftUS) [3006] NAD83(HARN) / WISCRS Kenosha, Milwaukee, Ozaukee and Racine (m) [3007] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (ftUS) [3008] NAD83(HARN) / WISCRS Kewaunee, Manitowoc and Sheboygan (m) [3009] NAD83(HARN) / WISCRS La Crosse (ftUS) [3010] NAD83(HARN) / WISCRS La Crosse (m) [3011] NAD83(HARN) / WISCRS Langlade (ftUS) [3012] NAD83(HARN) / WISCRS Langlade (m) [3013] NAD83(HARN) / WISCRS Lincoln (ftUS) [3014] NAD83(HARN) / WISCRS Lincoln (m) [3015] NAD83(HARN) / WISCRS Marathon (ftUS) [3016] NAD83(HARN) / WISCRS Marathon (m) [3017] NAD83(HARN) / WISCRS Marinette (ftUS) [3018] NAD83(HARN) / WISCRS Marinette (m) [3019] NAD83(HARN) / WISCRS Menominee (ftUS) [3020] NAD83(HARN) / WISCRS Menominee (m) [3021] NAD83(HARN) / WISCRS Monroe (ftUS) [3022] NAD83(HARN) / WISCRS Monroe (m) [3023] NAD83(HARN) / WISCRS Oconto (ftUS) [3024] NAD83(HARN) / WISCRS Oconto (m) [3025] NAD83(HARN) / WISCRS Oneida (ftUS) [3026] NAD83(HARN) / WISCRS Oneida (m) [3027] NAD83(HARN) / WISCRS Pepin and Pierce (ftUS) [3028] NAD83(HARN) / WISCRS Pepin and Pierce (m) [3029] NAD83(HARN) / WISCRS Polk (ftUS) [3030] NAD83(HARN) / WISCRS Polk (m) [3031] NAD83(HARN) / WISCRS Portage (ftUS) [3032] NAD83(HARN) / WISCRS Portage (m) [3033] NAD83(HARN) / WISCRS Price (ftUS) [3034] NAD83(HARN) / WISCRS Price (m) [3035] NAD83(HARN) / WISCRS Richland (ftUS) [3036] NAD83(HARN) / WISCRS Richland (m) [3037] NAD83(HARN) / WISCRS Rock (ftUS) [3038] NAD83(HARN) / WISCRS Rock (m) [3039] NAD83(HARN) / WISCRS Rusk (ftUS) [3040] NAD83(HARN) / WISCRS Rusk (m) [3041] NAD83(HARN) / WISCRS Sauk (ftUS) [3042] NAD83(HARN) / WISCRS Sauk (m) [3043] NAD83(HARN) / WISCRS Sawyer (ftUS) [3044] NAD83(HARN) / WISCRS Sawyer (m) [3045] NAD83(HARN) / WISCRS Shawano (ftUS) [3046] NAD83(HARN) / WISCRS Shawano (m) [3047] NAD83(HARN) / WISCRS St. Croix (ftUS) [3048] NAD83(HARN) / WISCRS St. Croix (m) [3049] NAD83(HARN) / WISCRS Taylor (ftUS) [3050] NAD83(HARN) / WISCRS Taylor (m) [3051] NAD83(HARN) / WISCRS Trempealeau (ftUS) [3052] NAD83(HARN) / WISCRS Trempealeau (m) [3053] NAD83(HARN) / WISCRS Vernon (ftUS) [3054] NAD83(HARN) / WISCRS Vernon (m) [3055] NAD83(HARN) / WISCRS Vilas (ftUS) [3056] NAD83(HARN) / WISCRS Vilas (m) [3057] NAD83(HARN) / WISCRS Walworth (ftUS) [3058] NAD83(HARN) / WISCRS Walworth (m) [3059] NAD83(HARN) / WISCRS Washburn (ftUS) [3060] NAD83(HARN) / WISCRS Washburn (m) [3061] NAD83(HARN) / WISCRS Washington (ftUS) [3062] NAD83(HARN) / WISCRS Washington (m) [3063] NAD83(HARN) / WISCRS Waukesha (ftUS) [3064] NAD83(HARN) / WISCRS Waukesha (m) [3065] NAD83(HARN) / WISCRS Waupaca (ftUS) [3066] NAD83(HARN) / WISCRS Waupaca (m) [3067] NAD83(HARN) / WISCRS Waushara (ftUS) [3068] NAD83(HARN) / WISCRS Waushara (m) [3069] NAD83(HARN) / WISCRS Wood (ftUS) [3070] NAD83(HARN) / WISCRS Wood (m) [3071] NAD83(HARN) / Washington North [3072] NAD83(HARN) / Washington North (ftUS) [3073] NAD83(HARN) / Washington South [3074] NAD83(HARN) / Washington South (ftUS) [3075] NAD83(HARN) / West Virginia North [3076] NAD83(HARN) / West Virginia North (ftUS) [3077] NAD83(HARN) / West Virginia North (ftUS) (deprecated) [3078] NAD83(HARN) / West Virginia South [3079] NAD83(HARN) / West Virginia South (ftUS) [3080] NAD83(HARN) / West Virginia South (ftUS) (deprecated) [3081] NAD83(HARN) / Wisconsin Central [3082] NAD83(HARN) / Wisconsin Central (ftUS) [3083] NAD83(HARN) / Wisconsin North [3084] NAD83(HARN) / Wisconsin North (ftUS) [3085] NAD83(HARN) / Wisconsin South [3086] NAD83(HARN) / Wisconsin South (ftUS) [3087] NAD83(HARN) / Wisconsin Transverse Mercator [3088] NAD83(HARN) / Wyoming East [3089] NAD83(HARN) / Wyoming East (ftUS) [3090] NAD83(HARN) / Wyoming East Central [3091] NAD83(HARN) / Wyoming East Central (ftUS) [3092] NAD83(HARN) / Wyoming West [3093] NAD83(HARN) / Wyoming West (ftUS) [3094] NAD83(HARN) / Wyoming West Central [3095] NAD83(HARN) / Wyoming West Central (ftUS) [3096] NAD83(MA11) / Guam Map Grid [3097] NAD83(MA11) / UTM zone 54N [3098] NAD83(MA11) / UTM zone 55N [3099] NAD83(NSRS2007) / Alabama East [3100] NAD83(NSRS2007) / Alabama West [3101] NAD83(NSRS2007) / Alaska Albers [3102] NAD83(NSRS2007) / Alaska zone 1 [3103] NAD83(NSRS2007) / Alaska zone 10 [3104] NAD83(NSRS2007) / Alaska zone 2 [3105] NAD83(NSRS2007) / Alaska zone 3 [3106] NAD83(NSRS2007) / Alaska zone 4 [3107] NAD83(NSRS2007) / Alaska zone 5 [3108] NAD83(NSRS2007) / Alaska zone 6 [3109] NAD83(NSRS2007) / Alaska zone 7 [3110] NAD83(NSRS2007) / Alaska zone 8 [3111] NAD83(NSRS2007) / Alaska zone 9 [3112] NAD83(NSRS2007) / Arizona Central [3113] NAD83(NSRS2007) / Arizona Central (ft) [3114] NAD83(NSRS2007) / Arizona East [3115] NAD83(NSRS2007) / Arizona East (ft) [3116] NAD83(NSRS2007) / Arizona West [3117] NAD83(NSRS2007) / Arizona West (ft) [3118] NAD83(NSRS2007) / Arkansas North [3119] NAD83(NSRS2007) / Arkansas North (ftUS) [3120] NAD83(NSRS2007) / Arkansas South [3121] NAD83(NSRS2007) / Arkansas South (ftUS) [3122] NAD83(NSRS2007) / California Albers [3123] NAD83(NSRS2007) / California zone 1 [3124] NAD83(NSRS2007) / California zone 1 (ftUS) [3125] NAD83(NSRS2007) / California zone 2 [3126] NAD83(NSRS2007) / California zone 2 (ftUS) [3127] NAD83(NSRS2007) / California zone 3 [3128] NAD83(NSRS2007) / California zone 3 (ftUS) [3129] NAD83(NSRS2007) / California zone 4 [3130] NAD83(NSRS2007) / California zone 4 (ftUS) [3131] NAD83(NSRS2007) / California zone 5 [3132] NAD83(NSRS2007) / California zone 5 (ftUS) [3133] NAD83(NSRS2007) / California zone 6 [3134] NAD83(NSRS2007) / California zone 6 (ftUS) [3135] NAD83(NSRS2007) / Colorado Central [3136] NAD83(NSRS2007) / Colorado Central (ftUS) [3137] NAD83(NSRS2007) / Colorado North [3138] NAD83(NSRS2007) / Colorado North (ftUS) [3139] NAD83(NSRS2007) / Colorado South [3140] NAD83(NSRS2007) / Colorado South (ftUS) [3141] NAD83(NSRS2007) / Connecticut [3142] NAD83(NSRS2007) / Connecticut (ftUS) [3143] NAD83(NSRS2007) / Conus Albers [3144] NAD83(NSRS2007) / Delaware [3145] NAD83(NSRS2007) / Delaware (ftUS) [3146] NAD83(NSRS2007) / EPSG Arctic zone 5-29 [3147] NAD83(NSRS2007) / EPSG Arctic zone 5-31 [3148] NAD83(NSRS2007) / EPSG Arctic zone 6-14 [3149] NAD83(NSRS2007) / EPSG Arctic zone 6-16 [3150] NAD83(NSRS2007) / Florida East [3151] NAD83(NSRS2007) / Florida East (ftUS) [3152] NAD83(NSRS2007) / Florida GDL Albers [3153] NAD83(NSRS2007) / Florida North [3154] NAD83(NSRS2007) / Florida North (ftUS) [3155] NAD83(NSRS2007) / Florida West [3156] NAD83(NSRS2007) / Florida West (ftUS) [3157] NAD83(NSRS2007) / Georgia East [3158] NAD83(NSRS2007) / Georgia East (ftUS) [3159] NAD83(NSRS2007) / Georgia West [3160] NAD83(NSRS2007) / Georgia West (ftUS) [3161] NAD83(NSRS2007) / Idaho Central [3162] NAD83(NSRS2007) / Idaho Central (ftUS) [3163] NAD83(NSRS2007) / Idaho East [3164] NAD83(NSRS2007) / Idaho East (ftUS) [3165] NAD83(NSRS2007) / Idaho West [3166] NAD83(NSRS2007) / Idaho West (ftUS) [3167] NAD83(NSRS2007) / Illinois East [3168] NAD83(NSRS2007) / Illinois East (ftUS) [3169] NAD83(NSRS2007) / Illinois West [3170] NAD83(NSRS2007) / Illinois West (ftUS) [3171] NAD83(NSRS2007) / Indiana East [3172] NAD83(NSRS2007) / Indiana East (ftUS) [3173] NAD83(NSRS2007) / Indiana West [3174] NAD83(NSRS2007) / Indiana West (ftUS) [3175] NAD83(NSRS2007) / Iowa North [3176] NAD83(NSRS2007) / Iowa North (ftUS) [3177] NAD83(NSRS2007) / Iowa South [3178] NAD83(NSRS2007) / Iowa South (ftUS) [3179] NAD83(NSRS2007) / Kansas North [3180] NAD83(NSRS2007) / Kansas North (ftUS) [3181] NAD83(NSRS2007) / Kansas South [3182] NAD83(NSRS2007) / Kansas South (ftUS) [3183] NAD83(NSRS2007) / Kentucky North [3184] NAD83(NSRS2007) / Kentucky North (ftUS) [3185] NAD83(NSRS2007) / Kentucky Single Zone [3186] NAD83(NSRS2007) / Kentucky Single Zone (ftUS) [3187] NAD83(NSRS2007) / Kentucky South [3188] NAD83(NSRS2007) / Kentucky South (ftUS) [3189] NAD83(NSRS2007) / Louisiana North [3190] NAD83(NSRS2007) / Louisiana North (ftUS) [3191] NAD83(NSRS2007) / Louisiana South [3192] NAD83(NSRS2007) / Louisiana South (ftUS) [3193] NAD83(NSRS2007) / Maine CS2000 Central [3194] NAD83(NSRS2007) / Maine CS2000 East [3195] NAD83(NSRS2007) / Maine CS2000 West [3196] NAD83(NSRS2007) / Maine East [3197] NAD83(NSRS2007) / Maine East (ftUS) [3198] NAD83(NSRS2007) / Maine East (ftUS) (deprecated) [3199] NAD83(NSRS2007) / Maine West [3200] NAD83(NSRS2007) / Maine West (ftUS) [3201] NAD83(NSRS2007) / Maine West (ftUS) (deprecated) [3202] NAD83(NSRS2007) / Maryland [3203] NAD83(NSRS2007) / Maryland (ftUS) [3204] NAD83(NSRS2007) / Massachusetts Island [3205] NAD83(NSRS2007) / Massachusetts Island (ftUS) [3206] NAD83(NSRS2007) / Massachusetts Mainland [3207] NAD83(NSRS2007) / Massachusetts Mainland (ftUS) [3208] NAD83(NSRS2007) / Michigan Central [3209] NAD83(NSRS2007) / Michigan Central (ft) [3210] NAD83(NSRS2007) / Michigan North [3211] NAD83(NSRS2007) / Michigan North (ft) [3212] NAD83(NSRS2007) / Michigan Oblique Mercator [3213] NAD83(NSRS2007) / Michigan South [3214] NAD83(NSRS2007) / Michigan South (ft) [3215] NAD83(NSRS2007) / Minnesota Central [3216] NAD83(NSRS2007) / Minnesota Central (ftUS) [3217] NAD83(NSRS2007) / Minnesota Central (ftUS) (deprecated) [3218] NAD83(NSRS2007) / Minnesota North [3219] NAD83(NSRS2007) / Minnesota North (ftUS) [3220] NAD83(NSRS2007) / Minnesota North (ftUS) (deprecated) [3221] NAD83(NSRS2007) / Minnesota South [3222] NAD83(NSRS2007) / Minnesota South (ftUS) [3223] NAD83(NSRS2007) / Minnesota South (ftUS) (deprecated) [3224] NAD83(NSRS2007) / Mississippi East [3225] NAD83(NSRS2007) / Mississippi East (ftUS) [3226] NAD83(NSRS2007) / Mississippi TM [3227] NAD83(NSRS2007) / Mississippi West [3228] NAD83(NSRS2007) / Mississippi West (ftUS) [3229] NAD83(NSRS2007) / Missouri Central [3230] NAD83(NSRS2007) / Missouri East [3231] NAD83(NSRS2007) / Missouri West [3232] NAD83(NSRS2007) / Montana [3233] NAD83(NSRS2007) / Montana (ft) [3234] NAD83(NSRS2007) / Nebraska [3235] NAD83(NSRS2007) / Nebraska (ftUS) [3236] NAD83(NSRS2007) / Nebraska (ftUS) (deprecated) [3237] NAD83(NSRS2007) / Nevada Central [3238] NAD83(NSRS2007) / Nevada Central (ftUS) [3239] NAD83(NSRS2007) / Nevada East [3240] NAD83(NSRS2007) / Nevada East (ftUS) [3241] NAD83(NSRS2007) / Nevada West [3242] NAD83(NSRS2007) / Nevada West (ftUS) [3243] NAD83(NSRS2007) / New Hampshire [3244] NAD83(NSRS2007) / New Hampshire (ftUS) [3245] NAD83(NSRS2007) / New Jersey [3246] NAD83(NSRS2007) / New Jersey (ftUS) [3247] NAD83(NSRS2007) / New Mexico Central [3248] NAD83(NSRS2007) / New Mexico Central (ftUS) [3249] NAD83(NSRS2007) / New Mexico East [3250] NAD83(NSRS2007) / New Mexico East (ftUS) [3251] NAD83(NSRS2007) / New Mexico West [3252] NAD83(NSRS2007) / New Mexico West (ftUS) [3253] NAD83(NSRS2007) / New York Central [3254] NAD83(NSRS2007) / New York Central (ftUS) [3255] NAD83(NSRS2007) / New York East [3256] NAD83(NSRS2007) / New York East (ftUS) [3257] NAD83(NSRS2007) / New York Long Island [3258] NAD83(NSRS2007) / New York Long Island (ftUS) [3259] NAD83(NSRS2007) / New York West [3260] NAD83(NSRS2007) / New York West (ftUS) [3261] NAD83(NSRS2007) / North Carolina [3262] NAD83(NSRS2007) / North Carolina (ftUS) [3263] NAD83(NSRS2007) / North Dakota North [3264] NAD83(NSRS2007) / North Dakota North (ft) [3265] NAD83(NSRS2007) / North Dakota South [3266] NAD83(NSRS2007) / North Dakota South (ft) [3267] NAD83(NSRS2007) / Ohio North [3268] NAD83(NSRS2007) / Ohio North (ftUS) [3269] NAD83(NSRS2007) / Ohio South [3270] NAD83(NSRS2007) / Ohio South (ftUS) [3271] NAD83(NSRS2007) / Oklahoma North [3272] NAD83(NSRS2007) / Oklahoma North (ftUS) [3273] NAD83(NSRS2007) / Oklahoma South [3274] NAD83(NSRS2007) / Oklahoma South (ftUS) [3275] NAD83(NSRS2007) / Oregon GIC Lambert (ft) [3276] NAD83(NSRS2007) / Oregon LCC (m) [3277] NAD83(NSRS2007) / Oregon North [3278] NAD83(NSRS2007) / Oregon North (ft) [3279] NAD83(NSRS2007) / Oregon South [3280] NAD83(NSRS2007) / Oregon South (ft) [3281] NAD83(NSRS2007) / Pennsylvania North [3282] NAD83(NSRS2007) / Pennsylvania North (ftUS) [3283] NAD83(NSRS2007) / Pennsylvania South [3284] NAD83(NSRS2007) / Pennsylvania South (ftUS) [3285] NAD83(NSRS2007) / Puerto Rico and Virgin Is. [3286] NAD83(NSRS2007) / Rhode Island [3287] NAD83(NSRS2007) / Rhode Island (ftUS) [3288] NAD83(NSRS2007) / South Carolina [3289] NAD83(NSRS2007) / South Carolina (ft) [3290] NAD83(NSRS2007) / South Dakota North [3291] NAD83(NSRS2007) / South Dakota North (ftUS) [3292] NAD83(NSRS2007) / South Dakota South [3293] NAD83(NSRS2007) / South Dakota South (ftUS) [3294] NAD83(NSRS2007) / Tennessee [3295] NAD83(NSRS2007) / Tennessee (ftUS) [3296] NAD83(NSRS2007) / Texas Central [3297] NAD83(NSRS2007) / Texas Central (ftUS) [3298] NAD83(NSRS2007) / Texas Centric Albers Equal Area [3299] NAD83(NSRS2007) / Texas Centric Lambert Conformal [3300] NAD83(NSRS2007) / Texas North [3301] NAD83(NSRS2007) / Texas North (ftUS) [3302] NAD83(NSRS2007) / Texas North Central [3303] NAD83(NSRS2007) / Texas North Central (ftUS) [3304] NAD83(NSRS2007) / Texas South [3305] NAD83(NSRS2007) / Texas South (ftUS) [3306] NAD83(NSRS2007) / Texas South Central [3307] NAD83(NSRS2007) / Texas South Central (ftUS) [3308] NAD83(NSRS2007) / UTM zone 10N [3309] NAD83(NSRS2007) / UTM zone 11N [3310] NAD83(NSRS2007) / UTM zone 12N [3311] NAD83(NSRS2007) / UTM zone 13N [3312] NAD83(NSRS2007) / UTM zone 14N [3313] NAD83(NSRS2007) / UTM zone 15N [3314] NAD83(NSRS2007) / UTM zone 16N [3315] NAD83(NSRS2007) / UTM zone 17N [3316] NAD83(NSRS2007) / UTM zone 18N [3317] NAD83(NSRS2007) / UTM zone 19N [3318] NAD83(NSRS2007) / UTM zone 1N [3319] NAD83(NSRS2007) / UTM zone 2N [3320] NAD83(NSRS2007) / UTM zone 3N [3321] NAD83(NSRS2007) / UTM zone 4N [3322] NAD83(NSRS2007) / UTM zone 59N [3323] NAD83(NSRS2007) / UTM zone 5N [3324] NAD83(NSRS2007) / UTM zone 60N [3325] NAD83(NSRS2007) / UTM zone 6N [3326] NAD83(NSRS2007) / UTM zone 7N [3327] NAD83(NSRS2007) / UTM zone 8N [3328] NAD83(NSRS2007) / UTM zone 9N [3329] NAD83(NSRS2007) / Utah Central [3330] NAD83(NSRS2007) / Utah Central (ft) [3331] NAD83(NSRS2007) / Utah Central (ftUS) [3332] NAD83(NSRS2007) / Utah North [3333] NAD83(NSRS2007) / Utah North (ft) [3334] NAD83(NSRS2007) / Utah North (ftUS) [3335] NAD83(NSRS2007) / Utah South [3336] NAD83(NSRS2007) / Utah South (ft) [3337] NAD83(NSRS2007) / Utah South (ftUS) [3338] NAD83(NSRS2007) / Vermont [3339] NAD83(NSRS2007) / Vermont (ftUS) [3340] NAD83(NSRS2007) / Virginia Lambert [3341] NAD83(NSRS2007) / Virginia North [3342] NAD83(NSRS2007) / Virginia North (ftUS) [3343] NAD83(NSRS2007) / Virginia South [3344] NAD83(NSRS2007) / Virginia South (ftUS) [3345] NAD83(NSRS2007) / Washington North [3346] NAD83(NSRS2007) / Washington North (ftUS) [3347] NAD83(NSRS2007) / Washington South [3348] NAD83(NSRS2007) / Washington South (ftUS) [3349] NAD83(NSRS2007) / West Virginia North [3350] NAD83(NSRS2007) / West Virginia North (ftUS) [3351] NAD83(NSRS2007) / West Virginia North (ftUS) (deprecated) [3352] NAD83(NSRS2007) / West Virginia South [3353] NAD83(NSRS2007) / West Virginia South (ftUS) [3354] NAD83(NSRS2007) / West Virginia South (ftUS) (deprecated) [3355] NAD83(NSRS2007) / Wisconsin Central [3356] NAD83(NSRS2007) / Wisconsin Central (ftUS) [3357] NAD83(NSRS2007) / Wisconsin North [3358] NAD83(NSRS2007) / Wisconsin North (ftUS) [3359] NAD83(NSRS2007) / Wisconsin South [3360] NAD83(NSRS2007) / Wisconsin South (ftUS) [3361] NAD83(NSRS2007) / Wisconsin Transverse Mercator [3362] NAD83(NSRS2007) / Wyoming East [3363] NAD83(NSRS2007) / Wyoming East (ftUS) [3364] NAD83(NSRS2007) / Wyoming East Central [3365] NAD83(NSRS2007) / Wyoming East Central (ftUS) [3366] NAD83(NSRS2007) / Wyoming West [3367] NAD83(NSRS2007) / Wyoming West (ftUS) [3368] NAD83(NSRS2007) / Wyoming West Central [3369] NAD83(NSRS2007) / Wyoming West Central (ftUS) [3370] NAD83(PA11) / Hawaii zone 1 [3371] NAD83(PA11) / Hawaii zone 2 [3372] NAD83(PA11) / Hawaii zone 3 [3373] NAD83(PA11) / Hawaii zone 3 (ftUS) [3374] NAD83(PA11) / Hawaii zone 4 [3375] NAD83(PA11) / Hawaii zone 5 [3376] NAD83(PA11) / UTM zone 2S [3377] NAD83(PA11) / UTM zone 4N [3378] NAD83(PA11) / UTM zone 5N [3379] NAD_1927_10TM_AEP_Forest [3380] NAD_1927_10TM_AEP_Resource [3381] NAD_1927_3TM_111 (deprecated) [3382] NAD_1927_3TM_114 (deprecated) [3383] NAD_1927_3TM_117 (deprecated) [3384] NAD_1927_3TM_120 (deprecated) [3385] NAD_1927_Alaska_Albers_Meters [3386] NAD_1927_Georgia_Statewide_Albers [3387] NAD_1927_Michigan_GeoRef_Feet_US [3388] NAD_1927_Michigan_GeoRef_Meters [3389] NAD_1927_StatePlane_California_V_Ventura [3390] NAD_1927_StatePlane_Guam_FIPS_5400 [3391] NAD_1927_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [3392] NAD_1927_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [3393] NAD_1927_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [3394] NAD_1927_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [3395] NAD_1927_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [3396] NAD_1927_StatePlane_Michigan_Central_FIPS_2112 [3397] NAD_1927_StatePlane_Michigan_North_FIPS_2111 [3398] NAD_1927_StatePlane_Michigan_South_FIPS_2113 [3399] NAD_1927_StatePlane_Puerto_Rico_FIPS_5201 [3400] NAD_1927_StatePlane_Virgin_Islands_St_Croix_FIPS_5202 [3401] NAD_1927_Texas_Statewide_Mapping_System (deprecated) [3402] NAD_1927_UTM_Zone_1N (deprecated) [3403] NAD_1927_UTM_Zone_2N (deprecated) [3404] NAD_1927_UTM_Zone_59N (deprecated) [3405] NAD_1927_UTM_Zone_60N (deprecated) [3406] NAD_1983_10TM_AEP_Forest (deprecated) [3407] NAD_1983_10TM_AEP_Resource (deprecated) [3408] NAD_1983_2011_Alaska_Albers (deprecated) [3409] NAD_1983_2011_California_Teale_Albers (deprecated) [3410] NAD_1983_2011_Contiguous_USA_Albers (deprecated) [3411] NAD_1983_2011_Fargo_Ground_Coordinate_System [3412] NAD_1983_2011_Florida_GDL_Albers (deprecated) [3413] NAD_1983_2011_Maine_2000_Central_Zone (deprecated) [3414] NAD_1983_2011_Maine_2000_East_Zone (deprecated) [3415] NAD_1983_2011_Maine_2000_West_Zone (deprecated) [3416] NAD_1983_2011_Michigan_GeoRef_Meters (deprecated) [3417] NAD_1983_2011_Mississippi_TM (deprecated) [3418] NAD_1983_2011_Oregon_Statewide_Lambert (deprecated) [3419] NAD_1983_2011_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3420] NAD_1983_2011_StatePlane_Alabama_East_FIPS_0101 (deprecated) [3421] NAD_1983_2011_StatePlane_Alabama_West_FIPS_0102 (deprecated) [3422] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010 (deprecated) [3423] NAD_1983_2011_StatePlane_Alaska_10_FIPS_5010_Feet [3424] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001 (deprecated) [3425] NAD_1983_2011_StatePlane_Alaska_1_FIPS_5001_Feet [3426] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002 (deprecated) [3427] NAD_1983_2011_StatePlane_Alaska_2_FIPS_5002_Feet [3428] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003 (deprecated) [3429] NAD_1983_2011_StatePlane_Alaska_3_FIPS_5003_Feet [3430] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004 (deprecated) [3431] NAD_1983_2011_StatePlane_Alaska_4_FIPS_5004_Feet [3432] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005 (deprecated) [3433] NAD_1983_2011_StatePlane_Alaska_5_FIPS_5005_Feet [3434] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006 (deprecated) [3435] NAD_1983_2011_StatePlane_Alaska_6_FIPS_5006_Feet [3436] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007 (deprecated) [3437] NAD_1983_2011_StatePlane_Alaska_7_FIPS_5007_Feet [3438] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008 (deprecated) [3439] NAD_1983_2011_StatePlane_Alaska_8_FIPS_5008_Feet [3440] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009 (deprecated) [3441] NAD_1983_2011_StatePlane_Alaska_9_FIPS_5009_Feet [3442] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [3443] NAD_1983_2011_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl (deprecated) [3444] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201 (deprecated) [3445] NAD_1983_2011_StatePlane_Arizona_East_FIPS_0201_Ft_Intl (deprecated) [3446] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203 (deprecated) [3447] NAD_1983_2011_StatePlane_Arizona_West_FIPS_0203_Ft_Intl (deprecated) [3448] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [3449] NAD_1983_2011_StatePlane_Arkansas_North_FIPS_0301_Ft_US (deprecated) [3450] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [3451] NAD_1983_2011_StatePlane_Arkansas_South_FIPS_0302_Ft_US (deprecated) [3452] NAD_1983_2011_StatePlane_California_III_FIPS_0403 (deprecated) [3453] NAD_1983_2011_StatePlane_California_III_FIPS_0403_Ft_US (deprecated) [3454] NAD_1983_2011_StatePlane_California_II_FIPS_0402 (deprecated) [3455] NAD_1983_2011_StatePlane_California_II_FIPS_0402_Ft_US (deprecated) [3456] NAD_1983_2011_StatePlane_California_IV_FIPS_0404 (deprecated) [3457] NAD_1983_2011_StatePlane_California_IV_FIPS_0404_Ft_US (deprecated) [3458] NAD_1983_2011_StatePlane_California_I_FIPS_0401 (deprecated) [3459] NAD_1983_2011_StatePlane_California_I_FIPS_0401_Ft_US (deprecated) [3460] NAD_1983_2011_StatePlane_California_VI_FIPS_0406 (deprecated) [3461] NAD_1983_2011_StatePlane_California_VI_FIPS_0406_Ft_US (deprecated) [3462] NAD_1983_2011_StatePlane_California_V_FIPS_0405 (deprecated) [3463] NAD_1983_2011_StatePlane_California_V_FIPS_0405_Ft_US (deprecated) [3464] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [3465] NAD_1983_2011_StatePlane_Colorado_Central_FIPS_0502_Ft_US (deprecated) [3466] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501 (deprecated) [3467] NAD_1983_2011_StatePlane_Colorado_North_FIPS_0501_Ft_US (deprecated) [3468] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503 (deprecated) [3469] NAD_1983_2011_StatePlane_Colorado_South_FIPS_0503_Ft_US (deprecated) [3470] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600 (deprecated) [3471] NAD_1983_2011_StatePlane_Connecticut_FIPS_0600_Ft_US (deprecated) [3472] NAD_1983_2011_StatePlane_Delaware_FIPS_0700 (deprecated) [3473] NAD_1983_2011_StatePlane_Delaware_FIPS_0700_Ft_US (deprecated) [3474] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901 (deprecated) [3475] NAD_1983_2011_StatePlane_Florida_East_FIPS_0901_Ft_US (deprecated) [3476] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903 (deprecated) [3477] NAD_1983_2011_StatePlane_Florida_North_FIPS_0903_Ft_US (deprecated) [3478] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902 (deprecated) [3479] NAD_1983_2011_StatePlane_Florida_West_FIPS_0902_Ft_US (deprecated) [3480] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001 (deprecated) [3481] NAD_1983_2011_StatePlane_Georgia_East_FIPS_1001_Ft_US (deprecated) [3482] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002 (deprecated) [3483] NAD_1983_2011_StatePlane_Georgia_West_FIPS_1002_Ft_US (deprecated) [3484] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [3485] NAD_1983_2011_StatePlane_Idaho_Central_FIPS_1102_Ft_US (deprecated) [3486] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101 (deprecated) [3487] NAD_1983_2011_StatePlane_Idaho_East_FIPS_1101_Ft_US (deprecated) [3488] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103 (deprecated) [3489] NAD_1983_2011_StatePlane_Idaho_West_FIPS_1103_Ft_US (deprecated) [3490] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201 (deprecated) [3491] NAD_1983_2011_StatePlane_Illinois_East_FIPS_1201_Ft_US (deprecated) [3492] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202 (deprecated) [3493] NAD_1983_2011_StatePlane_Illinois_West_FIPS_1202_Ft_US (deprecated) [3494] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301 (deprecated) [3495] NAD_1983_2011_StatePlane_Indiana_East_FIPS_1301_Ft_US (deprecated) [3496] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302 (deprecated) [3497] NAD_1983_2011_StatePlane_Indiana_West_FIPS_1302_Ft_US (deprecated) [3498] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401 (deprecated) [3499] NAD_1983_2011_StatePlane_Iowa_North_FIPS_1401_Ft_US (deprecated) [3500] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402 (deprecated) [3501] NAD_1983_2011_StatePlane_Iowa_South_FIPS_1402_Ft_US (deprecated) [3502] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501 (deprecated) [3503] NAD_1983_2011_StatePlane_Kansas_North_FIPS_1501_Ft_US (deprecated) [3504] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502 (deprecated) [3505] NAD_1983_2011_StatePlane_Kansas_South_FIPS_1502_Ft_US (deprecated) [3506] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600 (deprecated) [3507] NAD_1983_2011_StatePlane_Kentucky_FIPS_1600_Ft_US (deprecated) [3508] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [3509] NAD_1983_2011_StatePlane_Kentucky_North_FIPS_1601_Ft_US (deprecated) [3510] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [3511] NAD_1983_2011_StatePlane_Kentucky_South_FIPS_1602_Ft_US (deprecated) [3512] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [3513] NAD_1983_2011_StatePlane_Louisiana_North_FIPS_1701_Ft_US (deprecated) [3514] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [3515] NAD_1983_2011_StatePlane_Louisiana_South_FIPS_1702_Ft_US (deprecated) [3516] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801 (deprecated) [3517] NAD_1983_2011_StatePlane_Maine_East_FIPS_1801_Ft_US (deprecated) [3518] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802 (deprecated) [3519] NAD_1983_2011_StatePlane_Maine_West_FIPS_1802_Ft_US (deprecated) [3520] NAD_1983_2011_StatePlane_Maryland_FIPS_1900 (deprecated) [3521] NAD_1983_2011_StatePlane_Maryland_FIPS_1900_Ft_US (deprecated) [3522] NAD_1983_2011_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS (deprecated) [3523] NAD_1983_2011_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [3524] NAD_1983_2011_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [3525] NAD_1983_2011_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS (deprecated) [3526] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [3527] NAD_1983_2011_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl (deprecated) [3528] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111 (deprecated) [3529] NAD_1983_2011_StatePlane_Michigan_North_FIPS_2111_Ft_Intl (deprecated) [3530] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113 (deprecated) [3531] NAD_1983_2011_StatePlane_Michigan_South_FIPS_2113_Ft_Intl (deprecated) [3532] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [3533] NAD_1983_2011_StatePlane_Minnesota_Central_FIPS_2202_Ft_US (deprecated) [3534] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [3535] NAD_1983_2011_StatePlane_Minnesota_North_FIPS_2201_Ft_US (deprecated) [3536] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [3537] NAD_1983_2011_StatePlane_Minnesota_South_FIPS_2203_Ft_US (deprecated) [3538] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [3539] NAD_1983_2011_StatePlane_Mississippi_East_FIPS_2301_Ft_US (deprecated) [3540] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [3541] NAD_1983_2011_StatePlane_Mississippi_West_FIPS_2302_Ft_US (deprecated) [3542] NAD_1983_2011_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [3543] NAD_1983_2011_StatePlane_Missouri_East_FIPS_2401 (deprecated) [3544] NAD_1983_2011_StatePlane_Missouri_West_FIPS_2403 (deprecated) [3545] NAD_1983_2011_StatePlane_Montana_FIPS_2500 (deprecated) [3546] NAD_1983_2011_StatePlane_Montana_FIPS_2500_Ft_Intl (deprecated) [3547] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600 (deprecated) [3548] NAD_1983_2011_StatePlane_Nebraska_FIPS_2600_Ft_US (deprecated) [3549] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [3550] NAD_1983_2011_StatePlane_Nevada_Central_FIPS_2702_Ft_US (deprecated) [3551] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701 (deprecated) [3552] NAD_1983_2011_StatePlane_Nevada_East_FIPS_2701_Ft_US (deprecated) [3553] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703 (deprecated) [3554] NAD_1983_2011_StatePlane_Nevada_West_FIPS_2703_Ft_US (deprecated) [3555] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [3556] NAD_1983_2011_StatePlane_New_Hampshire_FIPS_2800_Ft_US (deprecated) [3557] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900 (deprecated) [3558] NAD_1983_2011_StatePlane_New_Jersey_FIPS_2900_Ft_US (deprecated) [3559] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [3560] NAD_1983_2011_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US (deprecated) [3561] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [3562] NAD_1983_2011_StatePlane_New_Mexico_East_FIPS_3001_Ft_US (deprecated) [3563] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [3564] NAD_1983_2011_StatePlane_New_Mexico_West_FIPS_3003_Ft_US (deprecated) [3565] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102 (deprecated) [3566] NAD_1983_2011_StatePlane_New_York_Central_FIPS_3102_Ft_US (deprecated) [3567] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101 (deprecated) [3568] NAD_1983_2011_StatePlane_New_York_East_FIPS_3101_Ft_US (deprecated) [3569] NAD_1983_2011_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US (deprecated) [3570] NAD_1983_2011_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [3571] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103 (deprecated) [3572] NAD_1983_2011_StatePlane_New_York_West_FIPS_3103_Ft_US (deprecated) [3573] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200 (deprecated) [3574] NAD_1983_2011_StatePlane_North_Carolina_FIPS_3200_Ft_US (deprecated) [3575] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [3576] NAD_1983_2011_StatePlane_North_Dakota_North_FIPS_3301_FtI (deprecated) [3577] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [3578] NAD_1983_2011_StatePlane_North_Dakota_South_FIPS_3302_FtI (deprecated) [3579] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401 (deprecated) [3580] NAD_1983_2011_StatePlane_Ohio_North_FIPS_3401_Ft_US (deprecated) [3581] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402 (deprecated) [3582] NAD_1983_2011_StatePlane_Ohio_South_FIPS_3402_Ft_US (deprecated) [3583] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [3584] NAD_1983_2011_StatePlane_Oklahoma_North_FIPS_3501_Ft_US (deprecated) [3585] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [3586] NAD_1983_2011_StatePlane_Oklahoma_South_FIPS_3502_Ft_US (deprecated) [3587] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3588] NAD_1983_2011_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3589] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3590] NAD_1983_2011_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3591] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701 (deprecated) [3592] NAD_1983_2011_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US (deprecated) [3593] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702 (deprecated) [3594] NAD_1983_2011_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US (deprecated) [3595] NAD_1983_2011_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [3596] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [3597] NAD_1983_2011_StatePlane_Rhode_Island_FIPS_3800_Ft_US (deprecated) [3598] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900 (deprecated) [3599] NAD_1983_2011_StatePlane_South_Carolina_FIPS_3900_Ft_Intl (deprecated) [3600] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [3601] NAD_1983_2011_StatePlane_South_Dakota_North_FIPS_4001_Ft_US (deprecated) [3602] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [3603] NAD_1983_2011_StatePlane_South_Dakota_South_FIPS_4002_Ft_US (deprecated) [3604] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100 (deprecated) [3605] NAD_1983_2011_StatePlane_Tennessee_FIPS_4100_Ft_US (deprecated) [3606] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203 (deprecated) [3607] NAD_1983_2011_StatePlane_Texas_Central_FIPS_4203_Ft_US (deprecated) [3608] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [3609] NAD_1983_2011_StatePlane_Texas_North_Central_FIPS_4202_FtUS (deprecated) [3610] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201 (deprecated) [3611] NAD_1983_2011_StatePlane_Texas_North_FIPS_4201_Ft_US (deprecated) [3612] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [3613] NAD_1983_2011_StatePlane_Texas_South_Central_FIPS_4204_FtUS (deprecated) [3614] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205 (deprecated) [3615] NAD_1983_2011_StatePlane_Texas_South_FIPS_4205_Ft_US (deprecated) [3616] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302 (deprecated) [3617] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3618] NAD_1983_2011_StatePlane_Utah_Central_FIPS_4302_Ft_US (deprecated) [3619] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301 (deprecated) [3620] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3621] NAD_1983_2011_StatePlane_Utah_North_FIPS_4301_Ft_US (deprecated) [3622] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303 (deprecated) [3623] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3624] NAD_1983_2011_StatePlane_Utah_South_FIPS_4303_Ft_US (deprecated) [3625] NAD_1983_2011_StatePlane_Vermont_FIPS_4400 (deprecated) [3626] NAD_1983_2011_StatePlane_Vermont_FIPS_4400_Ft_US (deprecated) [3627] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501 (deprecated) [3628] NAD_1983_2011_StatePlane_Virginia_North_FIPS_4501_Ft_US (deprecated) [3629] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502 (deprecated) [3630] NAD_1983_2011_StatePlane_Virginia_South_FIPS_4502_Ft_US (deprecated) [3631] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601 (deprecated) [3632] NAD_1983_2011_StatePlane_Washington_North_FIPS_4601_Ft_US (deprecated) [3633] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602 (deprecated) [3634] NAD_1983_2011_StatePlane_Washington_South_FIPS_4602_Ft_US (deprecated) [3635] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [3636] NAD_1983_2011_StatePlane_West_Virginia_North_FIPS_4701_FtUS (deprecated) [3637] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [3638] NAD_1983_2011_StatePlane_West_Virginia_South_FIPS_4702_FtUS (deprecated) [3639] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [3640] NAD_1983_2011_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US (deprecated) [3641] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [3642] NAD_1983_2011_StatePlane_Wisconsin_North_FIPS_4801_Ft_US (deprecated) [3643] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [3644] NAD_1983_2011_StatePlane_Wisconsin_South_FIPS_4803_Ft_US (deprecated) [3645] NAD_1983_2011_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US (deprecated) [3646] NAD_1983_2011_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [3647] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [3648] NAD_1983_2011_StatePlane_Wyoming_East_FIPS_4901_Ft_US (deprecated) [3649] NAD_1983_2011_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US (deprecated) [3650] NAD_1983_2011_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [3651] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [3652] NAD_1983_2011_StatePlane_Wyoming_West_FIPS_4904_Ft_US (deprecated) [3653] NAD_1983_2011_Texas_Centric_Mapping_System_Albers (deprecated) [3654] NAD_1983_2011_Texas_Centric_Mapping_System_Lambert (deprecated) [3655] NAD_1983_2011_UTM_Zone_10N (deprecated) [3656] NAD_1983_2011_UTM_Zone_11N (deprecated) [3657] NAD_1983_2011_UTM_Zone_12N (deprecated) [3658] NAD_1983_2011_UTM_Zone_13N (deprecated) [3659] NAD_1983_2011_UTM_Zone_14N (deprecated) [3660] NAD_1983_2011_UTM_Zone_15N (deprecated) [3661] NAD_1983_2011_UTM_Zone_16N (deprecated) [3662] NAD_1983_2011_UTM_Zone_17N (deprecated) [3663] NAD_1983_2011_UTM_Zone_18N (deprecated) [3664] NAD_1983_2011_UTM_Zone_19N (deprecated) [3665] NAD_1983_2011_UTM_Zone_1N (deprecated) [3666] NAD_1983_2011_UTM_Zone_20N [3667] NAD_1983_2011_UTM_Zone_2N (deprecated) [3668] NAD_1983_2011_UTM_Zone_3N (deprecated) [3669] NAD_1983_2011_UTM_Zone_4N (deprecated) [3670] NAD_1983_2011_UTM_Zone_59N (deprecated) [3671] NAD_1983_2011_UTM_Zone_5N (deprecated) [3672] NAD_1983_2011_UTM_Zone_60N (deprecated) [3673] NAD_1983_2011_UTM_Zone_6N (deprecated) [3674] NAD_1983_2011_UTM_Zone_7N (deprecated) [3675] NAD_1983_2011_UTM_Zone_8N (deprecated) [3676] NAD_1983_2011_UTM_Zone_9N (deprecated) [3677] NAD_1983_2011_Wisconsin_TM (deprecated) [3678] NAD_1983_2011_Wisconsin_TM_US_Ft [3679] NAD_1983_3TM_111 (deprecated) [3680] NAD_1983_3TM_114 (deprecated) [3681] NAD_1983_3TM_117 (deprecated) [3682] NAD_1983_3TM_120 (deprecated) [3683] NAD_1983_Alaska_Albers (deprecated) [3684] NAD_1983_Albers_BLM_MT_ND_SD [3685] NAD_1983_BC_Environment_Albers (deprecated) [3686] NAD_1983_CORS96_Alaska_Albers [3687] NAD_1983_CORS96_Maine_2000_Central_Zone [3688] NAD_1983_CORS96_Maine_2000_East_Zone [3689] NAD_1983_CORS96_Maine_2000_West_Zone [3690] NAD_1983_CORS96_Oregon_Statewide_Lambert (deprecated) [3691] NAD_1983_CORS96_Oregon_Statewide_Lambert_Ft_Intl (deprecated) [3692] NAD_1983_CORS96_StatePlane_Alabama_East_FIPS_0101 [3693] NAD_1983_CORS96_StatePlane_Alabama_West_FIPS_0102 [3694] NAD_1983_CORS96_StatePlane_Alaska_10_FIPS_5010 [3695] NAD_1983_CORS96_StatePlane_Alaska_1_FIPS_5001 [3696] NAD_1983_CORS96_StatePlane_Alaska_2_FIPS_5002 [3697] NAD_1983_CORS96_StatePlane_Alaska_3_FIPS_5003 [3698] NAD_1983_CORS96_StatePlane_Alaska_4_FIPS_5004 [3699] NAD_1983_CORS96_StatePlane_Alaska_5_FIPS_5005 [3700] NAD_1983_CORS96_StatePlane_Alaska_6_FIPS_5006 [3701] NAD_1983_CORS96_StatePlane_Alaska_7_FIPS_5007 [3702] NAD_1983_CORS96_StatePlane_Alaska_8_FIPS_5008 [3703] NAD_1983_CORS96_StatePlane_Alaska_9_FIPS_5009 [3704] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202 [3705] NAD_1983_CORS96_StatePlane_Arizona_Central_FIPS_0202_Ft_Intl [3706] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201 [3707] NAD_1983_CORS96_StatePlane_Arizona_East_FIPS_0201_Ft_Intl [3708] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203 [3709] NAD_1983_CORS96_StatePlane_Arizona_West_FIPS_0203_Ft_Intl [3710] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301 [3711] NAD_1983_CORS96_StatePlane_Arkansas_North_FIPS_0301_Ft_US [3712] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302 [3713] NAD_1983_CORS96_StatePlane_Arkansas_South_FIPS_0302_Ft_US [3714] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403 [3715] NAD_1983_CORS96_StatePlane_California_III_FIPS_0403_Ft_US [3716] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402 [3717] NAD_1983_CORS96_StatePlane_California_II_FIPS_0402_Ft_US [3718] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404 [3719] NAD_1983_CORS96_StatePlane_California_IV_FIPS_0404_Ft_US [3720] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401 [3721] NAD_1983_CORS96_StatePlane_California_I_FIPS_0401_Ft_US [3722] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406 [3723] NAD_1983_CORS96_StatePlane_California_VI_FIPS_0406_Ft_US [3724] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405 [3725] NAD_1983_CORS96_StatePlane_California_V_FIPS_0405_Ft_US [3726] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502 [3727] NAD_1983_CORS96_StatePlane_Colorado_Central_FIPS_0502_Ft_US [3728] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501 [3729] NAD_1983_CORS96_StatePlane_Colorado_North_FIPS_0501_Ft_US [3730] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503 [3731] NAD_1983_CORS96_StatePlane_Colorado_South_FIPS_0503_Ft_US [3732] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600 [3733] NAD_1983_CORS96_StatePlane_Connecticut_FIPS_0600_Ft_US [3734] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700 [3735] NAD_1983_CORS96_StatePlane_Delaware_FIPS_0700_Ft_US [3736] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901 [3737] NAD_1983_CORS96_StatePlane_Florida_East_FIPS_0901_Ft_US [3738] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903 [3739] NAD_1983_CORS96_StatePlane_Florida_North_FIPS_0903_Ft_US [3740] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902 [3741] NAD_1983_CORS96_StatePlane_Florida_West_FIPS_0902_Ft_US [3742] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001 [3743] NAD_1983_CORS96_StatePlane_Georgia_East_FIPS_1001_Ft_US [3744] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002 [3745] NAD_1983_CORS96_StatePlane_Georgia_West_FIPS_1002_Ft_US [3746] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102 [3747] NAD_1983_CORS96_StatePlane_Idaho_Central_FIPS_1102_Ft_US [3748] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101 [3749] NAD_1983_CORS96_StatePlane_Idaho_East_FIPS_1101_Ft_US [3750] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103 [3751] NAD_1983_CORS96_StatePlane_Idaho_West_FIPS_1103_Ft_US [3752] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201 [3753] NAD_1983_CORS96_StatePlane_Illinois_East_FIPS_1201_Ft_US [3754] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202 [3755] NAD_1983_CORS96_StatePlane_Illinois_West_FIPS_1202_Ft_US [3756] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301 [3757] NAD_1983_CORS96_StatePlane_Indiana_East_FIPS_1301_Ft_US [3758] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302 [3759] NAD_1983_CORS96_StatePlane_Indiana_West_FIPS_1302_Ft_US [3760] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401 [3761] NAD_1983_CORS96_StatePlane_Iowa_North_FIPS_1401_Ft_US [3762] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402 [3763] NAD_1983_CORS96_StatePlane_Iowa_South_FIPS_1402_Ft_US [3764] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501 [3765] NAD_1983_CORS96_StatePlane_Kansas_North_FIPS_1501_Ft_US [3766] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502 [3767] NAD_1983_CORS96_StatePlane_Kansas_South_FIPS_1502_Ft_US [3768] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600 [3769] NAD_1983_CORS96_StatePlane_Kentucky_FIPS_1600_Ft_US [3770] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601 [3771] NAD_1983_CORS96_StatePlane_Kentucky_North_FIPS_1601_Ft_US [3772] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602 [3773] NAD_1983_CORS96_StatePlane_Kentucky_South_FIPS_1602_Ft_US [3774] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701 [3775] NAD_1983_CORS96_StatePlane_Louisiana_North_FIPS_1701_Ft_US [3776] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702 [3777] NAD_1983_CORS96_StatePlane_Louisiana_South_FIPS_1702_Ft_US [3778] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801 [3779] NAD_1983_CORS96_StatePlane_Maine_East_FIPS_1801_Ft_US [3780] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802 [3781] NAD_1983_CORS96_StatePlane_Maine_West_FIPS_1802_Ft_US [3782] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900 [3783] NAD_1983_CORS96_StatePlane_Maryland_FIPS_1900_Ft_US [3784] NAD_1983_CORS96_StatePlane_Massachusetts_Isl_FIPS_2002_FtUS [3785] NAD_1983_CORS96_StatePlane_Massachusetts_Island_FIPS_2002 [3786] NAD_1983_CORS96_StatePlane_Massachusetts_Mainland_FIPS_2001 [3787] NAD_1983_CORS96_StatePlane_Massachusetts_Mnld_FIPS_2001_FtUS [3788] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112 [3789] NAD_1983_CORS96_StatePlane_Michigan_Central_FIPS_2112_Ft_Intl [3790] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111 [3791] NAD_1983_CORS96_StatePlane_Michigan_North_FIPS_2111_Ft_Intl [3792] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113 [3793] NAD_1983_CORS96_StatePlane_Michigan_South_FIPS_2113_Ft_Intl [3794] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202 [3795] NAD_1983_CORS96_StatePlane_Minnesota_Central_FIPS_2202_Ft_US [3796] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201 [3797] NAD_1983_CORS96_StatePlane_Minnesota_North_FIPS_2201_Ft_US [3798] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203 [3799] NAD_1983_CORS96_StatePlane_Minnesota_South_FIPS_2203_Ft_US [3800] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301 [3801] NAD_1983_CORS96_StatePlane_Mississippi_East_FIPS_2301_Ft_US [3802] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302 [3803] NAD_1983_CORS96_StatePlane_Mississippi_West_FIPS_2302_Ft_US [3804] NAD_1983_CORS96_StatePlane_Missouri_Central_FIPS_2402 [3805] NAD_1983_CORS96_StatePlane_Missouri_East_FIPS_2401 [3806] NAD_1983_CORS96_StatePlane_Missouri_West_FIPS_2403 [3807] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500 [3808] NAD_1983_CORS96_StatePlane_Montana_FIPS_2500_Ft_Intl [3809] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600 [3810] NAD_1983_CORS96_StatePlane_Nebraska_FIPS_2600_Ft_US [3811] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702 [3812] NAD_1983_CORS96_StatePlane_Nevada_Central_FIPS_2702_Ft_US [3813] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701 [3814] NAD_1983_CORS96_StatePlane_Nevada_East_FIPS_2701_Ft_US [3815] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703 [3816] NAD_1983_CORS96_StatePlane_Nevada_West_FIPS_2703_Ft_US [3817] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800 [3818] NAD_1983_CORS96_StatePlane_New_Hampshire_FIPS_2800_Ft_US [3819] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900 [3820] NAD_1983_CORS96_StatePlane_New_Jersey_FIPS_2900_Ft_US [3821] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002 [3822] NAD_1983_CORS96_StatePlane_New_Mexico_Central_FIPS_3002_Ft_US [3823] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001 [3824] NAD_1983_CORS96_StatePlane_New_Mexico_East_FIPS_3001_Ft_US [3825] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003 [3826] NAD_1983_CORS96_StatePlane_New_Mexico_West_FIPS_3003_Ft_US [3827] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102 [3828] NAD_1983_CORS96_StatePlane_New_York_Central_FIPS_3102_Ft_US [3829] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101 [3830] NAD_1983_CORS96_StatePlane_New_York_East_FIPS_3101_Ft_US [3831] NAD_1983_CORS96_StatePlane_New_York_Long_Isl_FIPS_3104_Ft_US [3832] NAD_1983_CORS96_StatePlane_New_York_Long_Island_FIPS_3104 [3833] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103 [3834] NAD_1983_CORS96_StatePlane_New_York_West_FIPS_3103_Ft_US [3835] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200 [3836] NAD_1983_CORS96_StatePlane_North_Carolina_FIPS_3200_Ft_US [3837] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301 [3838] NAD_1983_CORS96_StatePlane_North_Dakota_North_FIPS_3301_FtI [3839] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302 [3840] NAD_1983_CORS96_StatePlane_North_Dakota_South_FIPS_3302_FtI [3841] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401 [3842] NAD_1983_CORS96_StatePlane_Ohio_North_FIPS_3401_Ft_US [3843] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402 [3844] NAD_1983_CORS96_StatePlane_Ohio_South_FIPS_3402_Ft_US [3845] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501 [3846] NAD_1983_CORS96_StatePlane_Oklahoma_North_FIPS_3501_Ft_US [3847] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502 [3848] NAD_1983_CORS96_StatePlane_Oklahoma_South_FIPS_3502_Ft_US [3849] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601 (deprecated) [3850] NAD_1983_CORS96_StatePlane_Oregon_North_FIPS_3601_Ft_Intl (deprecated) [3851] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602 (deprecated) [3852] NAD_1983_CORS96_StatePlane_Oregon_South_FIPS_3602_Ft_Intl (deprecated) [3853] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701 [3854] NAD_1983_CORS96_StatePlane_Pennsylvania_North_FIPS_3701_Ft_US [3855] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702 [3856] NAD_1983_CORS96_StatePlane_Pennsylvania_South_FIPS_3702_Ft_US [3857] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800 [3858] NAD_1983_CORS96_StatePlane_Rhode_Island_FIPS_3800_Ft_US [3859] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900 [3860] NAD_1983_CORS96_StatePlane_South_Carolina_FIPS_3900_Ft_Intl [3861] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001 [3862] NAD_1983_CORS96_StatePlane_South_Dakota_North_FIPS_4001_Ft_US [3863] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002 [3864] NAD_1983_CORS96_StatePlane_South_Dakota_South_FIPS_4002_Ft_US [3865] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100 [3866] NAD_1983_CORS96_StatePlane_Tennessee_FIPS_4100_Ft_US [3867] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203 [3868] NAD_1983_CORS96_StatePlane_Texas_Central_FIPS_4203_Ft_US [3869] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202 [3870] NAD_1983_CORS96_StatePlane_Texas_North_Central_FIPS_4202_FtUS [3871] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201 [3872] NAD_1983_CORS96_StatePlane_Texas_North_FIPS_4201_Ft_US [3873] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204 [3874] NAD_1983_CORS96_StatePlane_Texas_South_Central_FIPS_4204_FtUS [3875] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205 [3876] NAD_1983_CORS96_StatePlane_Texas_South_FIPS_4205_Ft_US [3877] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302 [3878] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_Intl [3879] NAD_1983_CORS96_StatePlane_Utah_Central_FIPS_4302_Ft_US [3880] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301 [3881] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_Intl [3882] NAD_1983_CORS96_StatePlane_Utah_North_FIPS_4301_Ft_US [3883] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303 [3884] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_Intl [3885] NAD_1983_CORS96_StatePlane_Utah_South_FIPS_4303_Ft_US [3886] NAD_1983_CORS96_StatePlane_Vermont_FIPS_4400 [3887] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501 [3888] NAD_1983_CORS96_StatePlane_Virginia_North_FIPS_4501_Ft_US [3889] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502 [3890] NAD_1983_CORS96_StatePlane_Virginia_South_FIPS_4502_Ft_US [3891] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601 [3892] NAD_1983_CORS96_StatePlane_Washington_North_FIPS_4601_Ft_US [3893] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602 [3894] NAD_1983_CORS96_StatePlane_Washington_South_FIPS_4602_Ft_US [3895] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701 [3896] NAD_1983_CORS96_StatePlane_West_Virginia_North_FIPS_4701_FtUS [3897] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702 [3898] NAD_1983_CORS96_StatePlane_West_Virginia_South_FIPS_4702_FtUS [3899] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802 [3900] NAD_1983_CORS96_StatePlane_Wisconsin_Central_FIPS_4802_Ft_US [3901] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801 [3902] NAD_1983_CORS96_StatePlane_Wisconsin_North_FIPS_4801_Ft_US [3903] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803 [3904] NAD_1983_CORS96_StatePlane_Wisconsin_South_FIPS_4803_Ft_US [3905] NAD_1983_CORS96_StatePlane_Wyoming_E_Central_FIPS_4902_Ft_US [3906] NAD_1983_CORS96_StatePlane_Wyoming_East_Central_FIPS_4902 [3907] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901 [3908] NAD_1983_CORS96_StatePlane_Wyoming_East_FIPS_4901_Ft_US [3909] NAD_1983_CORS96_StatePlane_Wyoming_W_Central_FIPS_4903_Ft_US [3910] NAD_1983_CORS96_StatePlane_Wyoming_West_Central_FIPS_4903 [3911] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904 [3912] NAD_1983_CORS96_StatePlane_Wyoming_West_FIPS_4904_Ft_US [3913] NAD_1983_CORS96_UTM_Zone_10N [3914] NAD_1983_CORS96_UTM_Zone_11N [3915] NAD_1983_CORS96_UTM_Zone_12N [3916] NAD_1983_CORS96_UTM_Zone_13N [3917] NAD_1983_CORS96_UTM_Zone_14N [3918] NAD_1983_CORS96_UTM_Zone_15N [3919] NAD_1983_CORS96_UTM_Zone_16N [3920] NAD_1983_CORS96_UTM_Zone_17N [3921] NAD_1983_CORS96_UTM_Zone_18N [3922] NAD_1983_CORS96_UTM_Zone_19N [3923] NAD_1983_CORS96_UTM_Zone_1N [3924] NAD_1983_CORS96_UTM_Zone_20N [3925] NAD_1983_CORS96_UTM_Zone_2N [3926] NAD_1983_CORS96_UTM_Zone_3N [3927] NAD_1983_CORS96_UTM_Zone_4N [3928] NAD_1983_CORS96_UTM_Zone_59N [3929] NAD_1983_CORS96_UTM_Zone_5N [3930] NAD_1983_CORS96_UTM_Zone_60N [3931] NAD_1983_CORS96_UTM_Zone_6N [3932] NAD_1983_CORS96_UTM_Zone_7N [3933] NAD_1983_CORS96_UTM_Zone_8N [3934] NAD_1983_CORS96_UTM_Zone_9N [3935] NAD_1983_CSRS_UTM_Zone_14N (deprecated) [3936] NAD_1983_CSRS_UTM_Zone_15N (deprecated) [3937] NAD_1983_CSRS_UTM_Zone_16N (deprecated) [3938] NAD_1983_California_Teale_Albers_FtUS [3939] NAD_1983_Fargo_Ground_Coordinate_System [3940] NAD_1983_Georgia_Statewide_Lambert [3941] NAD_1983_HARN_Adj_MN_Aitkin_Feet [3942] NAD_1983_HARN_Adj_MN_Aitkin_Meters [3943] NAD_1983_HARN_Adj_MN_Anoka_Feet [3944] NAD_1983_HARN_Adj_MN_Anoka_Meters [3945] NAD_1983_HARN_Adj_MN_Becker_Feet [3946] NAD_1983_HARN_Adj_MN_Becker_Meters [3947] NAD_1983_HARN_Adj_MN_Beltrami_North_Feet [3948] NAD_1983_HARN_Adj_MN_Beltrami_North_Meters [3949] NAD_1983_HARN_Adj_MN_Beltrami_South_Feet [3950] NAD_1983_HARN_Adj_MN_Beltrami_South_Meters [3951] NAD_1983_HARN_Adj_MN_Benton_Feet [3952] NAD_1983_HARN_Adj_MN_Benton_Meters [3953] NAD_1983_HARN_Adj_MN_Big_Stone_Feet [3954] NAD_1983_HARN_Adj_MN_Big_Stone_Meters [3955] NAD_1983_HARN_Adj_MN_Blue_Earth_Feet [3956] NAD_1983_HARN_Adj_MN_Blue_Earth_Meters [3957] NAD_1983_HARN_Adj_MN_Brown_Feet [3958] NAD_1983_HARN_Adj_MN_Brown_Meters [3959] NAD_1983_HARN_Adj_MN_Carlton_Feet [3960] NAD_1983_HARN_Adj_MN_Carlton_Meters [3961] NAD_1983_HARN_Adj_MN_Carver_Feet [3962] NAD_1983_HARN_Adj_MN_Carver_Meters [3963] NAD_1983_HARN_Adj_MN_Cass_North_Feet [3964] NAD_1983_HARN_Adj_MN_Cass_North_Meters [3965] NAD_1983_HARN_Adj_MN_Cass_South_Feet [3966] NAD_1983_HARN_Adj_MN_Cass_South_Meters [3967] NAD_1983_HARN_Adj_MN_Chippewa_Feet [3968] NAD_1983_HARN_Adj_MN_Chippewa_Meters [3969] NAD_1983_HARN_Adj_MN_Chisago_Feet [3970] NAD_1983_HARN_Adj_MN_Chisago_Meters [3971] NAD_1983_HARN_Adj_MN_Clay_Feet [3972] NAD_1983_HARN_Adj_MN_Clay_Meters [3973] NAD_1983_HARN_Adj_MN_Clearwater_Feet [3974] NAD_1983_HARN_Adj_MN_Clearwater_Meters [3975] NAD_1983_HARN_Adj_MN_Cook_North_Feet [3976] NAD_1983_HARN_Adj_MN_Cook_North_Meters [3977] NAD_1983_HARN_Adj_MN_Cook_South_Feet [3978] NAD_1983_HARN_Adj_MN_Cook_South_Meters [3979] NAD_1983_HARN_Adj_MN_Cottonwood_Feet [3980] NAD_1983_HARN_Adj_MN_Cottonwood_Meters [3981] NAD_1983_HARN_Adj_MN_Crow_Wing_Feet [3982] NAD_1983_HARN_Adj_MN_Crow_Wing_Meters [3983] NAD_1983_HARN_Adj_MN_Dakota_Feet [3984] NAD_1983_HARN_Adj_MN_Dakota_Meters [3985] NAD_1983_HARN_Adj_MN_Dodge_Feet [3986] NAD_1983_HARN_Adj_MN_Dodge_Meters [3987] NAD_1983_HARN_Adj_MN_Douglas_Feet [3988] NAD_1983_HARN_Adj_MN_Douglas_Meters [3989] NAD_1983_HARN_Adj_MN_Faribault_Feet [3990] NAD_1983_HARN_Adj_MN_Faribault_Meters [3991] NAD_1983_HARN_Adj_MN_Fillmore_Feet [3992] NAD_1983_HARN_Adj_MN_Fillmore_Meters [3993] NAD_1983_HARN_Adj_MN_Freeborn_Feet [3994] NAD_1983_HARN_Adj_MN_Freeborn_Meters [3995] NAD_1983_HARN_Adj_MN_Goodhue_Feet [3996] NAD_1983_HARN_Adj_MN_Goodhue_Meters [3997] NAD_1983_HARN_Adj_MN_Grant_Feet [3998] NAD_1983_HARN_Adj_MN_Grant_Meters [3999] NAD_1983_HARN_Adj_MN_Hennepin_Feet [4000] NAD_1983_HARN_Adj_MN_Hennepin_Meters [4001] NAD_1983_HARN_Adj_MN_Houston_Feet [4002] NAD_1983_HARN_Adj_MN_Houston_Meters [4003] NAD_1983_HARN_Adj_MN_Hubbard_Feet [4004] NAD_1983_HARN_Adj_MN_Hubbard_Meters [4005] NAD_1983_HARN_Adj_MN_Isanti_Feet [4006] NAD_1983_HARN_Adj_MN_Isanti_Meters [4007] NAD_1983_HARN_Adj_MN_Itasca_North_Feet [4008] NAD_1983_HARN_Adj_MN_Itasca_North_Meters [4009] NAD_1983_HARN_Adj_MN_Itasca_South_Feet [4010] NAD_1983_HARN_Adj_MN_Itasca_South_Meters [4011] NAD_1983_HARN_Adj_MN_Jackson_Feet [4012] NAD_1983_HARN_Adj_MN_Jackson_Meters [4013] NAD_1983_HARN_Adj_MN_Kanabec_Feet [4014] NAD_1983_HARN_Adj_MN_Kanabec_Meters [4015] NAD_1983_HARN_Adj_MN_Kandiyohi_Feet [4016] NAD_1983_HARN_Adj_MN_Kandiyohi_Meters [4017] NAD_1983_HARN_Adj_MN_Kittson_Feet [4018] NAD_1983_HARN_Adj_MN_Kittson_Meters [4019] NAD_1983_HARN_Adj_MN_Koochiching_Feet [4020] NAD_1983_HARN_Adj_MN_Koochiching_Meters [4021] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Feet [4022] NAD_1983_HARN_Adj_MN_Lac_Qui_Parle_Meters [4023] NAD_1983_HARN_Adj_MN_Lake_Feet [4024] NAD_1983_HARN_Adj_MN_Lake_Meters [4025] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Feet [4026] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North_Meters [4027] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Feet [4028] NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South_Meters [4029] NAD_1983_HARN_Adj_MN_Le_Sueur_Feet [4030] NAD_1983_HARN_Adj_MN_Le_Sueur_Meters [4031] NAD_1983_HARN_Adj_MN_Lincoln_Feet [4032] NAD_1983_HARN_Adj_MN_Lincoln_Meters [4033] NAD_1983_HARN_Adj_MN_Lyon_Feet [4034] NAD_1983_HARN_Adj_MN_Lyon_Meters [4035] NAD_1983_HARN_Adj_MN_Mahnomen_Feet [4036] NAD_1983_HARN_Adj_MN_Mahnomen_Meters [4037] NAD_1983_HARN_Adj_MN_Marshall_Feet [4038] NAD_1983_HARN_Adj_MN_Marshall_Meters [4039] NAD_1983_HARN_Adj_MN_Martin_Feet [4040] NAD_1983_HARN_Adj_MN_Martin_Meters [4041] NAD_1983_HARN_Adj_MN_McLeod_Feet [4042] NAD_1983_HARN_Adj_MN_McLeod_Meters [4043] NAD_1983_HARN_Adj_MN_Meeker_Feet [4044] NAD_1983_HARN_Adj_MN_Meeker_Meters [4045] NAD_1983_HARN_Adj_MN_Mille_Lacs_Feet [4046] NAD_1983_HARN_Adj_MN_Mille_Lacs_Meters [4047] NAD_1983_HARN_Adj_MN_Morrison_Feet [4048] NAD_1983_HARN_Adj_MN_Morrison_Meters [4049] NAD_1983_HARN_Adj_MN_Mower_Feet [4050] NAD_1983_HARN_Adj_MN_Mower_Meters [4051] NAD_1983_HARN_Adj_MN_Murray_Feet [4052] NAD_1983_HARN_Adj_MN_Murray_Meters [4053] NAD_1983_HARN_Adj_MN_Nicollet_Feet [4054] NAD_1983_HARN_Adj_MN_Nicollet_Meters [4055] NAD_1983_HARN_Adj_MN_Nobles_Feet [4056] NAD_1983_HARN_Adj_MN_Nobles_Meters [4057] NAD_1983_HARN_Adj_MN_Norman_Feet [4058] NAD_1983_HARN_Adj_MN_Norman_Meters [4059] NAD_1983_HARN_Adj_MN_Olmsted_Feet [4060] NAD_1983_HARN_Adj_MN_Olmsted_Meters [4061] NAD_1983_HARN_Adj_MN_Ottertail_Feet [4062] NAD_1983_HARN_Adj_MN_Ottertail_Meters [4063] NAD_1983_HARN_Adj_MN_Pennington_Feet [4064] NAD_1983_HARN_Adj_MN_Pennington_Meters [4065] NAD_1983_HARN_Adj_MN_Pine_Feet [4066] NAD_1983_HARN_Adj_MN_Pine_Meters [4067] NAD_1983_HARN_Adj_MN_Pipestone_Feet [4068] NAD_1983_HARN_Adj_MN_Pipestone_Meters [4069] NAD_1983_HARN_Adj_MN_Polk_Feet [4070] NAD_1983_HARN_Adj_MN_Polk_Meters [4071] NAD_1983_HARN_Adj_MN_Pope_Feet [4072] NAD_1983_HARN_Adj_MN_Pope_Meters [4073] NAD_1983_HARN_Adj_MN_Ramsey_Feet [4074] NAD_1983_HARN_Adj_MN_Ramsey_Meters [4075] NAD_1983_HARN_Adj_MN_Red_Lake_Feet [4076] NAD_1983_HARN_Adj_MN_Red_Lake_Meters [4077] NAD_1983_HARN_Adj_MN_Redwood_Feet [4078] NAD_1983_HARN_Adj_MN_Redwood_Meters [4079] NAD_1983_HARN_Adj_MN_Renville_Feet [4080] NAD_1983_HARN_Adj_MN_Renville_Meters [4081] NAD_1983_HARN_Adj_MN_Rice_Feet [4082] NAD_1983_HARN_Adj_MN_Rice_Meters [4083] NAD_1983_HARN_Adj_MN_Rock_Feet [4084] NAD_1983_HARN_Adj_MN_Rock_Meters [4085] NAD_1983_HARN_Adj_MN_Roseau_Feet [4086] NAD_1983_HARN_Adj_MN_Roseau_Meters [4087] NAD_1983_HARN_Adj_MN_Scott_Feet [4088] NAD_1983_HARN_Adj_MN_Scott_Meters [4089] NAD_1983_HARN_Adj_MN_Sherburne_Feet [4090] NAD_1983_HARN_Adj_MN_Sherburne_Meters [4091] NAD_1983_HARN_Adj_MN_Sibley_Feet [4092] NAD_1983_HARN_Adj_MN_Sibley_Meters [4093] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Feet [4094] NAD_1983_HARN_Adj_MN_St_Louis_CS96_Meters [4095] NAD_1983_HARN_Adj_MN_St_Louis_Central_Feet [4096] NAD_1983_HARN_Adj_MN_St_Louis_Central_Meters [4097] NAD_1983_HARN_Adj_MN_St_Louis_North_Feet [4098] NAD_1983_HARN_Adj_MN_St_Louis_North_Meters [4099] NAD_1983_HARN_Adj_MN_St_Louis_South_Feet [4100] NAD_1983_HARN_Adj_MN_St_Louis_South_Meters [4101] NAD_1983_HARN_Adj_MN_Stearns_Feet [4102] NAD_1983_HARN_Adj_MN_Stearns_Meters [4103] NAD_1983_HARN_Adj_MN_Steele_Feet [4104] NAD_1983_HARN_Adj_MN_Steele_Meters [4105] NAD_1983_HARN_Adj_MN_Stevens_Feet [4106] NAD_1983_HARN_Adj_MN_Stevens_Meters [4107] NAD_1983_HARN_Adj_MN_Swift_Feet [4108] NAD_1983_HARN_Adj_MN_Swift_Meters [4109] NAD_1983_HARN_Adj_MN_Todd_Feet [4110] NAD_1983_HARN_Adj_MN_Todd_Meters [4111] NAD_1983_HARN_Adj_MN_Traverse_Feet [4112] NAD_1983_HARN_Adj_MN_Traverse_Meters [4113] NAD_1983_HARN_Adj_MN_Wabasha_Feet [4114] NAD_1983_HARN_Adj_MN_Wabasha_Meters [4115] NAD_1983_HARN_Adj_MN_Wadena_Feet [4116] NAD_1983_HARN_Adj_MN_Wadena_Meters [4117] NAD_1983_HARN_Adj_MN_Waseca_Feet [4118] NAD_1983_HARN_Adj_MN_Waseca_Meters [4119] NAD_1983_HARN_Adj_MN_Washington_Feet [4120] NAD_1983_HARN_Adj_MN_Washington_Meters [4121] NAD_1983_HARN_Adj_MN_Watonwan_Feet [4122] NAD_1983_HARN_Adj_MN_Watonwan_Meters [4123] NAD_1983_HARN_Adj_MN_Wilkin_Feet [4124] NAD_1983_HARN_Adj_MN_Wilkin_Meters [4125] NAD_1983_HARN_Adj_MN_Winona_Feet [4126] NAD_1983_HARN_Adj_MN_Winona_Meters [4127] NAD_1983_HARN_Adj_MN_Wright_Feet [4128] NAD_1983_HARN_Adj_MN_Wright_Meters [4129] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Feet [4130] NAD_1983_HARN_Adj_MN_Yellow_Medicine_Meters [4131] NAD_1983_HARN_Adj_WI_Adams_Feet (deprecated) [4132] NAD_1983_HARN_Adj_WI_Adams_Meters (deprecated) [4133] NAD_1983_HARN_Adj_WI_Ashland_Feet (deprecated) [4134] NAD_1983_HARN_Adj_WI_Ashland_Meters (deprecated) [4135] NAD_1983_HARN_Adj_WI_Barron_Feet (deprecated) [4136] NAD_1983_HARN_Adj_WI_Barron_Meters (deprecated) [4137] NAD_1983_HARN_Adj_WI_Bayfield_Feet (deprecated) [4138] NAD_1983_HARN_Adj_WI_Bayfield_Meters (deprecated) [4139] NAD_1983_HARN_Adj_WI_Brown_Feet (deprecated) [4140] NAD_1983_HARN_Adj_WI_Brown_Meters (deprecated) [4141] NAD_1983_HARN_Adj_WI_Buffalo_Feet (deprecated) [4142] NAD_1983_HARN_Adj_WI_Buffalo_Meters (deprecated) [4143] NAD_1983_HARN_Adj_WI_Burnett_Feet (deprecated) [4144] NAD_1983_HARN_Adj_WI_Burnett_Meters (deprecated) [4145] NAD_1983_HARN_Adj_WI_Calumet_Feet (deprecated) [4146] NAD_1983_HARN_Adj_WI_Calumet_Meters (deprecated) [4147] NAD_1983_HARN_Adj_WI_Chippewa_Feet [4148] NAD_1983_HARN_Adj_WI_Chippewa_Meters [4149] NAD_1983_HARN_Adj_WI_Clark_Feet (deprecated) [4150] NAD_1983_HARN_Adj_WI_Clark_Meters (deprecated) [4151] NAD_1983_HARN_Adj_WI_Columbia_Feet (deprecated) [4152] NAD_1983_HARN_Adj_WI_Columbia_Meters (deprecated) [4153] NAD_1983_HARN_Adj_WI_Crawford_Feet (deprecated) [4154] NAD_1983_HARN_Adj_WI_Crawford_Meters (deprecated) [4155] NAD_1983_HARN_Adj_WI_Dane_Feet (deprecated) [4156] NAD_1983_HARN_Adj_WI_Dane_Meters (deprecated) [4157] NAD_1983_HARN_Adj_WI_Dodge_Feet (deprecated) [4158] NAD_1983_HARN_Adj_WI_Dodge_Meters (deprecated) [4159] NAD_1983_HARN_Adj_WI_Door_Feet (deprecated) [4160] NAD_1983_HARN_Adj_WI_Door_Meters (deprecated) [4161] NAD_1983_HARN_Adj_WI_Douglas_Feet (deprecated) [4162] NAD_1983_HARN_Adj_WI_Douglas_Meters (deprecated) [4163] NAD_1983_HARN_Adj_WI_Dunn_Feet (deprecated) [4164] NAD_1983_HARN_Adj_WI_Dunn_Meters (deprecated) [4165] NAD_1983_HARN_Adj_WI_EauClaire_Feet (deprecated) [4166] NAD_1983_HARN_Adj_WI_EauClaire_Meters (deprecated) [4167] NAD_1983_HARN_Adj_WI_Florence_Feet (deprecated) [4168] NAD_1983_HARN_Adj_WI_Florence_Meters (deprecated) [4169] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Feet (deprecated) [4170] NAD_1983_HARN_Adj_WI_Fond_du_Lac_Meters (deprecated) [4171] NAD_1983_HARN_Adj_WI_Forest_Feet (deprecated) [4172] NAD_1983_HARN_Adj_WI_Forest_Meters (deprecated) [4173] NAD_1983_HARN_Adj_WI_Grant_Feet (deprecated) [4174] NAD_1983_HARN_Adj_WI_Grant_Meters (deprecated) [4175] NAD_1983_HARN_Adj_WI_GreenLake_Feet (deprecated) [4176] NAD_1983_HARN_Adj_WI_GreenLake_Meters (deprecated) [4177] NAD_1983_HARN_Adj_WI_Green_Feet (deprecated) [4178] NAD_1983_HARN_Adj_WI_Green_Meters (deprecated) [4179] NAD_1983_HARN_Adj_WI_Iowa_Feet (deprecated) [4180] NAD_1983_HARN_Adj_WI_Iowa_Meters (deprecated) [4181] NAD_1983_HARN_Adj_WI_Iron_Feet (deprecated) [4182] NAD_1983_HARN_Adj_WI_Iron_Meters (deprecated) [4183] NAD_1983_HARN_Adj_WI_Jackson_Feet (deprecated) [4184] NAD_1983_HARN_Adj_WI_Jackson_Meters (deprecated) [4185] NAD_1983_HARN_Adj_WI_Jefferson_Feet (deprecated) [4186] NAD_1983_HARN_Adj_WI_Jefferson_Meters (deprecated) [4187] NAD_1983_HARN_Adj_WI_Juneau_Feet (deprecated) [4188] NAD_1983_HARN_Adj_WI_Juneau_Meters (deprecated) [4189] NAD_1983_HARN_Adj_WI_Kenosha_Feet (deprecated) [4190] NAD_1983_HARN_Adj_WI_Kenosha_Meters (deprecated) [4191] NAD_1983_HARN_Adj_WI_Kewaunee_Feet (deprecated) [4192] NAD_1983_HARN_Adj_WI_Kewaunee_Meters (deprecated) [4193] NAD_1983_HARN_Adj_WI_LaCrosse_Feet (deprecated) [4194] NAD_1983_HARN_Adj_WI_LaCrosse_Meters (deprecated) [4195] NAD_1983_HARN_Adj_WI_Lafayette_Feet (deprecated) [4196] NAD_1983_HARN_Adj_WI_Lafayette_Meters (deprecated) [4197] NAD_1983_HARN_Adj_WI_Langlade_Feet (deprecated) [4198] NAD_1983_HARN_Adj_WI_Langlade_Meters (deprecated) [4199] NAD_1983_HARN_Adj_WI_Lincoln_Feet (deprecated) [4200] NAD_1983_HARN_Adj_WI_Lincoln_Meters (deprecated) [4201] NAD_1983_HARN_Adj_WI_Manitowoc_Feet (deprecated) [4202] NAD_1983_HARN_Adj_WI_Manitowoc_Meters (deprecated) [4203] NAD_1983_HARN_Adj_WI_Marathon_Feet (deprecated) [4204] NAD_1983_HARN_Adj_WI_Marathon_Meters (deprecated) [4205] NAD_1983_HARN_Adj_WI_Marinette_Feet (deprecated) [4206] NAD_1983_HARN_Adj_WI_Marinette_Meters (deprecated) [4207] NAD_1983_HARN_Adj_WI_Marquette_Feet (deprecated) [4208] NAD_1983_HARN_Adj_WI_Marquette_Meters (deprecated) [4209] NAD_1983_HARN_Adj_WI_Menominee_Feet (deprecated) [4210] NAD_1983_HARN_Adj_WI_Menominee_Meters (deprecated) [4211] NAD_1983_HARN_Adj_WI_Milwaukee_Feet (deprecated) [4212] NAD_1983_HARN_Adj_WI_Milwaukee_Meters (deprecated) [4213] NAD_1983_HARN_Adj_WI_Monroe_Feet (deprecated) [4214] NAD_1983_HARN_Adj_WI_Monroe_Meters (deprecated) [4215] NAD_1983_HARN_Adj_WI_Oconto_Feet (deprecated) [4216] NAD_1983_HARN_Adj_WI_Oconto_Meters (deprecated) [4217] NAD_1983_HARN_Adj_WI_Oneida_Feet (deprecated) [4218] NAD_1983_HARN_Adj_WI_Oneida_Meters (deprecated) [4219] NAD_1983_HARN_Adj_WI_Outagamie_Feet (deprecated) [4220] NAD_1983_HARN_Adj_WI_Outagamie_Meters (deprecated) [4221] NAD_1983_HARN_Adj_WI_Ozaukee_Feet (deprecated) [4222] NAD_1983_HARN_Adj_WI_Ozaukee_Meters (deprecated) [4223] NAD_1983_HARN_Adj_WI_Pepin_Feet (deprecated) [4224] NAD_1983_HARN_Adj_WI_Pepin_Meters (deprecated) [4225] NAD_1983_HARN_Adj_WI_Pierce_Feet (deprecated) [4226] NAD_1983_HARN_Adj_WI_Pierce_Meters (deprecated) [4227] NAD_1983_HARN_Adj_WI_Polk_Feet (deprecated) [4228] NAD_1983_HARN_Adj_WI_Polk_Meters (deprecated) [4229] NAD_1983_HARN_Adj_WI_Portage_Feet (deprecated) [4230] NAD_1983_HARN_Adj_WI_Portage_Meters (deprecated) [4231] NAD_1983_HARN_Adj_WI_Price_Feet (deprecated) [4232] NAD_1983_HARN_Adj_WI_Price_Meters (deprecated) [4233] NAD_1983_HARN_Adj_WI_Racine_Feet (deprecated) [4234] NAD_1983_HARN_Adj_WI_Racine_Meters (deprecated) [4235] NAD_1983_HARN_Adj_WI_Richland_Feet (deprecated) [4236] NAD_1983_HARN_Adj_WI_Richland_Meters (deprecated) [4237] NAD_1983_HARN_Adj_WI_Rock_Feet (deprecated) [4238] NAD_1983_HARN_Adj_WI_Rock_Meters (deprecated) [4239] NAD_1983_HARN_Adj_WI_Rusk_Feet (deprecated) [4240] NAD_1983_HARN_Adj_WI_Rusk_Meters (deprecated) [4241] NAD_1983_HARN_Adj_WI_Sauk_Feet (deprecated) [4242] NAD_1983_HARN_Adj_WI_Sauk_Meters (deprecated) [4243] NAD_1983_HARN_Adj_WI_Sawyer_Feet (deprecated) [4244] NAD_1983_HARN_Adj_WI_Sawyer_Meters (deprecated) [4245] NAD_1983_HARN_Adj_WI_Shawano_Feet (deprecated) [4246] NAD_1983_HARN_Adj_WI_Shawano_Meters (deprecated) [4247] NAD_1983_HARN_Adj_WI_Sheboygan_Feet (deprecated) [4248] NAD_1983_HARN_Adj_WI_Sheboygan_Meters (deprecated) [4249] NAD_1983_HARN_Adj_WI_St_Croix_Feet (deprecated) [4250] NAD_1983_HARN_Adj_WI_St_Croix_Meters (deprecated) [4251] NAD_1983_HARN_Adj_WI_Taylor_Feet (deprecated) [4252] NAD_1983_HARN_Adj_WI_Taylor_Meters (deprecated) [4253] NAD_1983_HARN_Adj_WI_Trempealeau_Feet (deprecated) [4254] NAD_1983_HARN_Adj_WI_Trempealeau_Meters (deprecated) [4255] NAD_1983_HARN_Adj_WI_Vernon_Feet (deprecated) [4256] NAD_1983_HARN_Adj_WI_Vernon_Meters (deprecated) [4257] NAD_1983_HARN_Adj_WI_Vilas_Feet (deprecated) [4258] NAD_1983_HARN_Adj_WI_Vilas_Meters (deprecated) [4259] NAD_1983_HARN_Adj_WI_Walworth_Feet (deprecated) [4260] NAD_1983_HARN_Adj_WI_Walworth_Meters (deprecated) [4261] NAD_1983_HARN_Adj_WI_Washburn_Feet (deprecated) [4262] NAD_1983_HARN_Adj_WI_Washburn_Meters (deprecated) [4263] NAD_1983_HARN_Adj_WI_Washington_Feet (deprecated) [4264] NAD_1983_HARN_Adj_WI_Washington_Meters (deprecated) [4265] NAD_1983_HARN_Adj_WI_Waukesha_Feet (deprecated) [4266] NAD_1983_HARN_Adj_WI_Waukesha_Meters (deprecated) [4267] NAD_1983_HARN_Adj_WI_Waupaca_Feet (deprecated) [4268] NAD_1983_HARN_Adj_WI_Waupaca_Meters (deprecated) [4269] NAD_1983_HARN_Adj_WI_Waushara_Feet (deprecated) [4270] NAD_1983_HARN_Adj_WI_Waushara_Meters (deprecated) [4271] NAD_1983_HARN_Adj_WI_Winnebago_Feet (deprecated) [4272] NAD_1983_HARN_Adj_WI_Winnebago_Meters (deprecated) [4273] NAD_1983_HARN_Adj_WI_Wood_Feet (deprecated) [4274] NAD_1983_HARN_Adj_WI_Wood_Meters (deprecated) [4275] NAD_1983_HARN_Fargo_Ground_Coordinate_System [4276] NAD_1983_HARN_Guam_Map_Grid (deprecated) [4277] NAD_1983_HARN_Maine_2000_Central_Zone (deprecated) [4278] NAD_1983_HARN_Maine_2000_East_Zone (deprecated) [4279] NAD_1983_HARN_Maine_2000_West_Zone (deprecated) [4280] NAD_1983_HARN_Mississippi_TM (deprecated) [4281] NAD_1983_HARN_StatePlane_Alabama_East_FIPS_0101 (deprecated) [4282] NAD_1983_HARN_StatePlane_Alabama_West_FIPS_0102 (deprecated) [4283] NAD_1983_HARN_StatePlane_Arizona_Central_FIPS_0202 (deprecated) [4284] NAD_1983_HARN_StatePlane_Arizona_East_FIPS_0201 (deprecated) [4285] NAD_1983_HARN_StatePlane_Arizona_West_FIPS_0203 (deprecated) [4286] NAD_1983_HARN_StatePlane_Arkansas_North_FIPS_0301 (deprecated) [4287] NAD_1983_HARN_StatePlane_Arkansas_South_FIPS_0302 (deprecated) [4288] NAD_1983_HARN_StatePlane_California_III_FIPS_0403 (deprecated) [4289] NAD_1983_HARN_StatePlane_California_II_FIPS_0402 (deprecated) [4290] NAD_1983_HARN_StatePlane_California_IV_FIPS_0404 (deprecated) [4291] NAD_1983_HARN_StatePlane_California_I_FIPS_0401 (deprecated) [4292] NAD_1983_HARN_StatePlane_California_VI_FIPS_0406 (deprecated) [4293] NAD_1983_HARN_StatePlane_California_V_FIPS_0405 (deprecated) [4294] NAD_1983_HARN_StatePlane_Colorado_Central_FIPS_0502 (deprecated) [4295] NAD_1983_HARN_StatePlane_Colorado_North_FIPS_0501 (deprecated) [4296] NAD_1983_HARN_StatePlane_Colorado_South_FIPS_0503 (deprecated) [4297] NAD_1983_HARN_StatePlane_Connecticut_FIPS_0600 (deprecated) [4298] NAD_1983_HARN_StatePlane_Delaware_FIPS_0700 (deprecated) [4299] NAD_1983_HARN_StatePlane_Florida_East_FIPS_0901 (deprecated) [4300] NAD_1983_HARN_StatePlane_Florida_North_FIPS_0903 (deprecated) [4301] NAD_1983_HARN_StatePlane_Florida_West_FIPS_0902 (deprecated) [4302] NAD_1983_HARN_StatePlane_Georgia_East_FIPS_1001 (deprecated) [4303] NAD_1983_HARN_StatePlane_Georgia_West_FIPS_1002 (deprecated) [4304] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4305] NAD_1983_HARN_StatePlane_Hawaii_1_FIPS_5101_Feet [4306] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4307] NAD_1983_HARN_StatePlane_Hawaii_2_FIPS_5102_Feet [4308] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4309] NAD_1983_HARN_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4310] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4311] NAD_1983_HARN_StatePlane_Hawaii_4_FIPS_5104_Feet [4312] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4313] NAD_1983_HARN_StatePlane_Hawaii_5_FIPS_5105_Feet [4314] NAD_1983_HARN_StatePlane_Idaho_Central_FIPS_1102 (deprecated) [4315] NAD_1983_HARN_StatePlane_Idaho_East_FIPS_1101 (deprecated) [4316] NAD_1983_HARN_StatePlane_Idaho_West_FIPS_1103 (deprecated) [4317] NAD_1983_HARN_StatePlane_Illinois_East_FIPS_1201 (deprecated) [4318] NAD_1983_HARN_StatePlane_Illinois_West_FIPS_1202 (deprecated) [4319] NAD_1983_HARN_StatePlane_Indiana_East_FIPS_1301 (deprecated) [4320] NAD_1983_HARN_StatePlane_Indiana_West_FIPS_1302 (deprecated) [4321] NAD_1983_HARN_StatePlane_Iowa_North_FIPS_1401 (deprecated) [4322] NAD_1983_HARN_StatePlane_Iowa_South_FIPS_1402 (deprecated) [4323] NAD_1983_HARN_StatePlane_Kansas_North_FIPS_1501 (deprecated) [4324] NAD_1983_HARN_StatePlane_Kansas_South_FIPS_1502 (deprecated) [4325] NAD_1983_HARN_StatePlane_Kentucky_FIPS_1600 (deprecated) [4326] NAD_1983_HARN_StatePlane_Kentucky_North_FIPS_1601 (deprecated) [4327] NAD_1983_HARN_StatePlane_Kentucky_South_FIPS_1602 (deprecated) [4328] NAD_1983_HARN_StatePlane_Louisiana_North_FIPS_1701 (deprecated) [4329] NAD_1983_HARN_StatePlane_Louisiana_South_FIPS_1702 (deprecated) [4330] NAD_1983_HARN_StatePlane_Maine_East_FIPS_1801 (deprecated) [4331] NAD_1983_HARN_StatePlane_Maine_West_FIPS_1802 (deprecated) [4332] NAD_1983_HARN_StatePlane_Maryland_FIPS_1900 (deprecated) [4333] NAD_1983_HARN_StatePlane_Massachusetts_Island_FIPS_2002 (deprecated) [4334] NAD_1983_HARN_StatePlane_Massachusetts_Mainland_FIPS_2001 (deprecated) [4335] NAD_1983_HARN_StatePlane_Michigan_Central_FIPS_2112 (deprecated) [4336] NAD_1983_HARN_StatePlane_Michigan_North_FIPS_2111 (deprecated) [4337] NAD_1983_HARN_StatePlane_Michigan_South_FIPS_2113 (deprecated) [4338] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202 (deprecated) [4339] NAD_1983_HARN_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4340] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201 (deprecated) [4341] NAD_1983_HARN_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4342] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203 (deprecated) [4343] NAD_1983_HARN_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4344] NAD_1983_HARN_StatePlane_Mississippi_East_FIPS_2301 (deprecated) [4345] NAD_1983_HARN_StatePlane_Mississippi_West_FIPS_2302 (deprecated) [4346] NAD_1983_HARN_StatePlane_Missouri_Central_FIPS_2402 (deprecated) [4347] NAD_1983_HARN_StatePlane_Missouri_East_FIPS_2401 (deprecated) [4348] NAD_1983_HARN_StatePlane_Missouri_West_FIPS_2403 (deprecated) [4349] NAD_1983_HARN_StatePlane_Montana_FIPS_2500 (deprecated) [4350] NAD_1983_HARN_StatePlane_Nebraska_FIPS_2600 (deprecated) [4351] NAD_1983_HARN_StatePlane_Nevada_Central_FIPS_2702 (deprecated) [4352] NAD_1983_HARN_StatePlane_Nevada_East_FIPS_2701 (deprecated) [4353] NAD_1983_HARN_StatePlane_Nevada_West_FIPS_2703 (deprecated) [4354] NAD_1983_HARN_StatePlane_New_Hampshire_FIPS_2800 (deprecated) [4355] NAD_1983_HARN_StatePlane_New_Jersey_FIPS_2900 (deprecated) [4356] NAD_1983_HARN_StatePlane_New_Mexico_Central_FIPS_3002 (deprecated) [4357] NAD_1983_HARN_StatePlane_New_Mexico_East_FIPS_3001 (deprecated) [4358] NAD_1983_HARN_StatePlane_New_Mexico_West_FIPS_3003 (deprecated) [4359] NAD_1983_HARN_StatePlane_New_York_Central_FIPS_3102 (deprecated) [4360] NAD_1983_HARN_StatePlane_New_York_East_FIPS_3101 (deprecated) [4361] NAD_1983_HARN_StatePlane_New_York_Long_Island_FIPS_3104 (deprecated) [4362] NAD_1983_HARN_StatePlane_New_York_West_FIPS_3103 (deprecated) [4363] NAD_1983_HARN_StatePlane_North_Dakota_North_FIPS_3301 (deprecated) [4364] NAD_1983_HARN_StatePlane_North_Dakota_South_FIPS_3302 (deprecated) [4365] NAD_1983_HARN_StatePlane_Ohio_North_FIPS_3401 (deprecated) [4366] NAD_1983_HARN_StatePlane_Ohio_South_FIPS_3402 (deprecated) [4367] NAD_1983_HARN_StatePlane_Oklahoma_North_FIPS_3501 (deprecated) [4368] NAD_1983_HARN_StatePlane_Oklahoma_South_FIPS_3502 (deprecated) [4369] NAD_1983_HARN_StatePlane_Oregon_North_FIPS_3601 (deprecated) [4370] NAD_1983_HARN_StatePlane_Oregon_South_FIPS_3602 (deprecated) [4371] NAD_1983_HARN_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200 (deprecated) [4372] NAD_1983_HARN_StatePlane_Rhode_Island_FIPS_3800 (deprecated) [4373] NAD_1983_HARN_StatePlane_South_Dakota_North_FIPS_4001 (deprecated) [4374] NAD_1983_HARN_StatePlane_South_Dakota_South_FIPS_4002 (deprecated) [4375] NAD_1983_HARN_StatePlane_Tennessee_FIPS_4100 (deprecated) [4376] NAD_1983_HARN_StatePlane_Texas_Central_FIPS_4203 (deprecated) [4377] NAD_1983_HARN_StatePlane_Texas_North_Central_FIPS_4202 (deprecated) [4378] NAD_1983_HARN_StatePlane_Texas_North_FIPS_4201 (deprecated) [4379] NAD_1983_HARN_StatePlane_Texas_South_Central_FIPS_4204 (deprecated) [4380] NAD_1983_HARN_StatePlane_Texas_South_FIPS_4205 (deprecated) [4381] NAD_1983_HARN_StatePlane_Utah_Central_FIPS_4302 (deprecated) [4382] NAD_1983_HARN_StatePlane_Utah_North_FIPS_4301 (deprecated) [4383] NAD_1983_HARN_StatePlane_Utah_South_FIPS_4303 (deprecated) [4384] NAD_1983_HARN_StatePlane_Vermont_FIPS_4400 (deprecated) [4385] NAD_1983_HARN_StatePlane_Virginia_North_FIPS_4501 (deprecated) [4386] NAD_1983_HARN_StatePlane_Virginia_South_FIPS_4502 (deprecated) [4387] NAD_1983_HARN_StatePlane_Washington_North_FIPS_4601 (deprecated) [4388] NAD_1983_HARN_StatePlane_Washington_South_FIPS_4602 (deprecated) [4389] NAD_1983_HARN_StatePlane_West_Virginia_North_FIPS_4701 (deprecated) [4390] NAD_1983_HARN_StatePlane_West_Virginia_South_FIPS_4702 (deprecated) [4391] NAD_1983_HARN_StatePlane_Wisconsin_Central_FIPS_4802 (deprecated) [4392] NAD_1983_HARN_StatePlane_Wisconsin_North_FIPS_4801 (deprecated) [4393] NAD_1983_HARN_StatePlane_Wisconsin_South_FIPS_4803 (deprecated) [4394] NAD_1983_HARN_StatePlane_Wyoming_East_Central_FIPS_4902 (deprecated) [4395] NAD_1983_HARN_StatePlane_Wyoming_East_FIPS_4901 (deprecated) [4396] NAD_1983_HARN_StatePlane_Wyoming_West_Central_FIPS_4903 (deprecated) [4397] NAD_1983_HARN_StatePlane_Wyoming_West_FIPS_4904 (deprecated) [4398] NAD_1983_HARN_UTM_Zone_11N (deprecated) [4399] NAD_1983_HARN_UTM_Zone_12N (deprecated) [4400] NAD_1983_HARN_UTM_Zone_13N (deprecated) [4401] NAD_1983_HARN_UTM_Zone_18N (deprecated) [4402] NAD_1983_HARN_UTM_Zone_2S (deprecated) [4403] NAD_1983_HARN_UTM_Zone_4N (deprecated) [4404] NAD_1983_HARN_UTM_Zone_5N (deprecated) [4405] NAD_1983_HARN_WISCRS_Adams_County_Feet [4406] NAD_1983_HARN_WISCRS_Adams_County_Meters [4407] NAD_1983_HARN_WISCRS_Ashland_County_Feet (deprecated) [4408] NAD_1983_HARN_WISCRS_Ashland_County_Meters (deprecated) [4409] NAD_1983_HARN_WISCRS_Barron_County_Feet (deprecated) [4410] NAD_1983_HARN_WISCRS_Barron_County_Meters (deprecated) [4411] NAD_1983_HARN_WISCRS_Bayfield_County_Feet (deprecated) [4412] NAD_1983_HARN_WISCRS_Bayfield_County_Meters (deprecated) [4413] NAD_1983_HARN_WISCRS_Brown_County_Feet (deprecated) [4414] NAD_1983_HARN_WISCRS_Brown_County_Meters (deprecated) [4415] NAD_1983_HARN_WISCRS_Buffalo_County_Feet (deprecated) [4416] NAD_1983_HARN_WISCRS_Buffalo_County_Meters (deprecated) [4417] NAD_1983_HARN_WISCRS_Burnett_County_Feet (deprecated) [4418] NAD_1983_HARN_WISCRS_Burnett_County_Meters (deprecated) [4419] NAD_1983_HARN_WISCRS_Calumet_County_Feet [4420] NAD_1983_HARN_WISCRS_Calumet_County_Meters [4421] NAD_1983_HARN_WISCRS_Chippewa_County_Feet (deprecated) [4422] NAD_1983_HARN_WISCRS_Chippewa_County_Meters (deprecated) [4423] NAD_1983_HARN_WISCRS_Clark_County_Feet (deprecated) [4424] NAD_1983_HARN_WISCRS_Clark_County_Meters (deprecated) [4425] NAD_1983_HARN_WISCRS_Columbia_County_Feet (deprecated) [4426] NAD_1983_HARN_WISCRS_Columbia_County_Meters (deprecated) [4427] NAD_1983_HARN_WISCRS_Crawford_County_Feet (deprecated) [4428] NAD_1983_HARN_WISCRS_Crawford_County_Meters (deprecated) [4429] NAD_1983_HARN_WISCRS_Dane_County_Feet (deprecated) [4430] NAD_1983_HARN_WISCRS_Dane_County_Meters (deprecated) [4431] NAD_1983_HARN_WISCRS_Dodge_County_Feet [4432] NAD_1983_HARN_WISCRS_Dodge_County_Meters [4433] NAD_1983_HARN_WISCRS_Door_County_Feet (deprecated) [4434] NAD_1983_HARN_WISCRS_Door_County_Meters (deprecated) [4435] NAD_1983_HARN_WISCRS_Douglas_County_Feet (deprecated) [4436] NAD_1983_HARN_WISCRS_Douglas_County_Meters (deprecated) [4437] NAD_1983_HARN_WISCRS_Dunn_County_Feet (deprecated) [4438] NAD_1983_HARN_WISCRS_Dunn_County_Meters (deprecated) [4439] NAD_1983_HARN_WISCRS_EauClaire_County_Feet (deprecated) [4440] NAD_1983_HARN_WISCRS_EauClaire_County_Meters (deprecated) [4441] NAD_1983_HARN_WISCRS_Florence_County_Feet (deprecated) [4442] NAD_1983_HARN_WISCRS_Florence_County_Meters (deprecated) [4443] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Feet [4444] NAD_1983_HARN_WISCRS_Fond_du_Lac_County_Meters [4445] NAD_1983_HARN_WISCRS_Forest_County_Feet (deprecated) [4446] NAD_1983_HARN_WISCRS_Forest_County_Meters (deprecated) [4447] NAD_1983_HARN_WISCRS_Grant_County_Feet (deprecated) [4448] NAD_1983_HARN_WISCRS_Grant_County_Meters (deprecated) [4449] NAD_1983_HARN_WISCRS_GreenLake_County_Feet [4450] NAD_1983_HARN_WISCRS_GreenLake_County_Meters [4451] NAD_1983_HARN_WISCRS_Green_County_Feet [4452] NAD_1983_HARN_WISCRS_Green_County_Meters [4453] NAD_1983_HARN_WISCRS_Iowa_County_Feet (deprecated) [4454] NAD_1983_HARN_WISCRS_Iowa_County_Meters (deprecated) [4455] NAD_1983_HARN_WISCRS_Iron_County_Feet (deprecated) [4456] NAD_1983_HARN_WISCRS_Iron_County_Meters (deprecated) [4457] NAD_1983_HARN_WISCRS_Jackson_County_Feet (deprecated) [4458] NAD_1983_HARN_WISCRS_Jackson_County_Meters (deprecated) [4459] NAD_1983_HARN_WISCRS_Jefferson_County_Feet [4460] NAD_1983_HARN_WISCRS_Jefferson_County_Meters [4461] NAD_1983_HARN_WISCRS_Juneau_County_Feet [4462] NAD_1983_HARN_WISCRS_Juneau_County_Meters [4463] NAD_1983_HARN_WISCRS_Kenosha_County_Feet [4464] NAD_1983_HARN_WISCRS_Kenosha_County_Meters [4465] NAD_1983_HARN_WISCRS_Kewaunee_County_Feet [4466] NAD_1983_HARN_WISCRS_Kewaunee_County_Meters [4467] NAD_1983_HARN_WISCRS_LaCrosse_County_Feet (deprecated) [4468] NAD_1983_HARN_WISCRS_LaCrosse_County_Meters (deprecated) [4469] NAD_1983_HARN_WISCRS_Lafayette_County_Feet [4470] NAD_1983_HARN_WISCRS_Lafayette_County_Meters [4471] NAD_1983_HARN_WISCRS_Langlade_County_Feet (deprecated) [4472] NAD_1983_HARN_WISCRS_Langlade_County_Meters (deprecated) [4473] NAD_1983_HARN_WISCRS_Lincoln_County_Feet (deprecated) [4474] NAD_1983_HARN_WISCRS_Lincoln_County_Meters (deprecated) [4475] NAD_1983_HARN_WISCRS_Manitowoc_County_Feet [4476] NAD_1983_HARN_WISCRS_Manitowoc_County_Meters [4477] NAD_1983_HARN_WISCRS_Marathon_County_Feet (deprecated) [4478] NAD_1983_HARN_WISCRS_Marathon_County_Meters (deprecated) [4479] NAD_1983_HARN_WISCRS_Marinette_County_Feet (deprecated) [4480] NAD_1983_HARN_WISCRS_Marinette_County_Meters (deprecated) [4481] NAD_1983_HARN_WISCRS_Marquette_County_Feet [4482] NAD_1983_HARN_WISCRS_Marquette_County_Meters [4483] NAD_1983_HARN_WISCRS_Menominee_County_Feet (deprecated) [4484] NAD_1983_HARN_WISCRS_Menominee_County_Meters (deprecated) [4485] NAD_1983_HARN_WISCRS_Milwaukee_County_Feet [4486] NAD_1983_HARN_WISCRS_Milwaukee_County_Meters [4487] NAD_1983_HARN_WISCRS_Monroe_County_Feet (deprecated) [4488] NAD_1983_HARN_WISCRS_Monroe_County_Meters (deprecated) [4489] NAD_1983_HARN_WISCRS_Oconto_County_Feet (deprecated) [4490] NAD_1983_HARN_WISCRS_Oconto_County_Meters (deprecated) [4491] NAD_1983_HARN_WISCRS_Oneida_County_Feet (deprecated) [4492] NAD_1983_HARN_WISCRS_Oneida_County_Meters (deprecated) [4493] NAD_1983_HARN_WISCRS_Outagamie_County_Feet [4494] NAD_1983_HARN_WISCRS_Outagamie_County_Meters [4495] NAD_1983_HARN_WISCRS_Ozaukee_County_Feet [4496] NAD_1983_HARN_WISCRS_Ozaukee_County_Meters [4497] NAD_1983_HARN_WISCRS_Pepin_County_Feet [4498] NAD_1983_HARN_WISCRS_Pepin_County_Meters [4499] NAD_1983_HARN_WISCRS_Pierce_County_Feet [4500] NAD_1983_HARN_WISCRS_Pierce_County_Meters [4501] NAD_1983_HARN_WISCRS_Polk_County_Feet (deprecated) [4502] NAD_1983_HARN_WISCRS_Polk_County_Meters (deprecated) [4503] NAD_1983_HARN_WISCRS_Portage_County_Feet (deprecated) [4504] NAD_1983_HARN_WISCRS_Portage_County_Meters (deprecated) [4505] NAD_1983_HARN_WISCRS_Price_County_Feet (deprecated) [4506] NAD_1983_HARN_WISCRS_Price_County_Meters (deprecated) [4507] NAD_1983_HARN_WISCRS_Racine_County_Feet [4508] NAD_1983_HARN_WISCRS_Racine_County_Meters [4509] NAD_1983_HARN_WISCRS_Richland_County_Feet (deprecated) [4510] NAD_1983_HARN_WISCRS_Richland_County_Meters (deprecated) [4511] NAD_1983_HARN_WISCRS_Rock_County_Feet (deprecated) [4512] NAD_1983_HARN_WISCRS_Rock_County_Meters (deprecated) [4513] NAD_1983_HARN_WISCRS_Rusk_County_Feet (deprecated) [4514] NAD_1983_HARN_WISCRS_Rusk_County_Meters (deprecated) [4515] NAD_1983_HARN_WISCRS_Sauk_County_Feet (deprecated) [4516] NAD_1983_HARN_WISCRS_Sauk_County_Meters (deprecated) [4517] NAD_1983_HARN_WISCRS_Sawyer_County_Feet (deprecated) [4518] NAD_1983_HARN_WISCRS_Sawyer_County_Meters (deprecated) [4519] NAD_1983_HARN_WISCRS_Shawano_County_Feet (deprecated) [4520] NAD_1983_HARN_WISCRS_Shawano_County_Meters (deprecated) [4521] NAD_1983_HARN_WISCRS_Sheboygan_County_Feet [4522] NAD_1983_HARN_WISCRS_Sheboygan_County_Meters [4523] NAD_1983_HARN_WISCRS_St_Croix_County_Feet (deprecated) [4524] NAD_1983_HARN_WISCRS_St_Croix_County_Meters (deprecated) [4525] NAD_1983_HARN_WISCRS_Taylor_County_Feet (deprecated) [4526] NAD_1983_HARN_WISCRS_Taylor_County_Meters (deprecated) [4527] NAD_1983_HARN_WISCRS_Trempealeau_County_Feet (deprecated) [4528] NAD_1983_HARN_WISCRS_Trempealeau_County_Meters (deprecated) [4529] NAD_1983_HARN_WISCRS_Vernon_County_Feet (deprecated) [4530] NAD_1983_HARN_WISCRS_Vernon_County_Meters (deprecated) [4531] NAD_1983_HARN_WISCRS_Vilas_County_Feet (deprecated) [4532] NAD_1983_HARN_WISCRS_Vilas_County_Meters (deprecated) [4533] NAD_1983_HARN_WISCRS_Walworth_County_Feet (deprecated) [4534] NAD_1983_HARN_WISCRS_Walworth_County_Meters (deprecated) [4535] NAD_1983_HARN_WISCRS_Washburn_County_Feet (deprecated) [4536] NAD_1983_HARN_WISCRS_Washburn_County_Meters (deprecated) [4537] NAD_1983_HARN_WISCRS_Washington_County_Feet (deprecated) [4538] NAD_1983_HARN_WISCRS_Washington_County_Meters (deprecated) [4539] NAD_1983_HARN_WISCRS_Waukesha_County_Feet (deprecated) [4540] NAD_1983_HARN_WISCRS_Waukesha_County_Meters (deprecated) [4541] NAD_1983_HARN_WISCRS_Waupaca_County_Feet (deprecated) [4542] NAD_1983_HARN_WISCRS_Waupaca_County_Meters (deprecated) [4543] NAD_1983_HARN_WISCRS_Waushara_County_Feet (deprecated) [4544] NAD_1983_HARN_WISCRS_Waushara_County_Meters (deprecated) [4545] NAD_1983_HARN_WISCRS_Winnebago_County_Feet [4546] NAD_1983_HARN_WISCRS_Winnebago_County_Meters [4547] NAD_1983_HARN_WISCRS_Wood_County_Feet (deprecated) [4548] NAD_1983_HARN_WISCRS_Wood_County_Meters (deprecated) [4549] NAD_1983_HARN_Wisconsin_TM_US_Ft [4550] NAD_1983_Idaho-Ada_County [4551] NAD_1983_Idaho_TM [4552] NAD_1983_MA11_Guam_Map_Grid (deprecated) [4553] NAD_1983_MA11_UTM_Zone_55N [4554] NAD_1983_Maine_2000_Central_Zone (deprecated) [4555] NAD_1983_Maine_2000_East_Zone (deprecated) [4556] NAD_1983_Maine_2000_West_Zone (deprecated) [4557] NAD_1983_Michigan_GeoRef_Feet_US [4558] NAD_1983_Michigan_GeoRef_Meters (deprecated) [4559] NAD_1983_Mississippi_TM (deprecated) [4560] NAD_1983_NSRS2007_StatePlane_Puerto_Rico_Virgin_Isls_FIPS_5200 (deprecated) [4561] NAD_1983_NSRS2007_UTM_Zone_20N [4562] NAD_1983_NSRS2007_Wisconsin_TM_US_Ft [4563] NAD_1983_Nebraska_Lancaster_County_FtUS [4564] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101 (deprecated) [4565] NAD_1983_PA11_StatePlane_Hawaii_1_FIPS_5101_Feet [4566] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102 (deprecated) [4567] NAD_1983_PA11_StatePlane_Hawaii_2_FIPS_5102_Feet [4568] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103 (deprecated) [4569] NAD_1983_PA11_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4570] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104 (deprecated) [4571] NAD_1983_PA11_StatePlane_Hawaii_4_FIPS_5104_Feet [4572] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105 (deprecated) [4573] NAD_1983_PA11_StatePlane_Hawaii_5_FIPS_5105_Feet [4574] NAD_1983_PA11_UTM_Zone_2S (deprecated) [4575] NAD_1983_PA11_UTM_Zone_4N (deprecated) [4576] NAD_1983_PA11_UTM_Zone_5N (deprecated) [4577] NAD_1983_PACP00_UTM_Zone_2S [4578] NAD_1983_PACP00_UTM_Zone_4N [4579] NAD_1983_PACP00_UTM_Zone_5N [4580] NAD_1983_StatePlane_Alabama_East_FIPS_0101_Feet [4581] NAD_1983_StatePlane_Alabama_West_FIPS_0102_Feet [4582] NAD_1983_StatePlane_Alaska_10_FIPS_5010_Feet [4583] NAD_1983_StatePlane_Alaska_1_FIPS_5001_Feet [4584] NAD_1983_StatePlane_Alaska_2_FIPS_5002_Feet [4585] NAD_1983_StatePlane_Alaska_3_FIPS_5003_Feet [4586] NAD_1983_StatePlane_Alaska_4_FIPS_5004_Feet [4587] NAD_1983_StatePlane_Alaska_5_FIPS_5005_Feet [4588] NAD_1983_StatePlane_Alaska_6_FIPS_5006_Feet [4589] NAD_1983_StatePlane_Alaska_7_FIPS_5007_Feet [4590] NAD_1983_StatePlane_Alaska_8_FIPS_5008_Feet [4591] NAD_1983_StatePlane_Alaska_9_FIPS_5009_Feet [4592] NAD_1983_StatePlane_Arizona_Central_FIPS_0202_Feet [4593] NAD_1983_StatePlane_Arizona_East_FIPS_0201_Feet [4594] NAD_1983_StatePlane_Arizona_West_FIPS_0203_Feet [4595] NAD_1983_StatePlane_Arkansas_North_FIPS_0301_Feet (deprecated) [4596] NAD_1983_StatePlane_Arkansas_South_FIPS_0302_Feet (deprecated) [4597] NAD_1983_StatePlane_California_III_FIPS_0403_Feet (deprecated) [4598] NAD_1983_StatePlane_California_II_FIPS_0402_Feet (deprecated) [4599] NAD_1983_StatePlane_California_IV_FIPS_0404_Feet (deprecated) [4600] NAD_1983_StatePlane_California_I_FIPS_0401_Feet (deprecated) [4601] NAD_1983_StatePlane_California_VI_FIPS_0406_Feet (deprecated) [4602] NAD_1983_StatePlane_California_V_FIPS_0405_Feet (deprecated) [4603] NAD_1983_StatePlane_Colorado_Central_FIPS_0502_Feet (deprecated) [4604] NAD_1983_StatePlane_Colorado_North_FIPS_0501_Feet (deprecated) [4605] NAD_1983_StatePlane_Colorado_South_FIPS_0503_Feet (deprecated) [4606] NAD_1983_StatePlane_Connecticut_FIPS_0600_Feet (deprecated) [4607] NAD_1983_StatePlane_Delaware_FIPS_0700_Feet (deprecated) [4608] NAD_1983_StatePlane_Florida_East_FIPS_0901_Feet (deprecated) [4609] NAD_1983_StatePlane_Florida_North_FIPS_0903_Feet (deprecated) [4610] NAD_1983_StatePlane_Florida_West_FIPS_0902_Feet (deprecated) [4611] NAD_1983_StatePlane_Georgia_East_FIPS_1001_Feet (deprecated) [4612] NAD_1983_StatePlane_Georgia_West_FIPS_1002_Feet (deprecated) [4613] NAD_1983_StatePlane_Guam_FIPS_5400 [4614] NAD_1983_StatePlane_Guam_FIPS_5400_Feet [4615] NAD_1983_StatePlane_Hawaii_1_FIPS_5101_Feet [4616] NAD_1983_StatePlane_Hawaii_2_FIPS_5102_Feet [4617] NAD_1983_StatePlane_Hawaii_3_FIPS_5103_Feet (deprecated) [4618] NAD_1983_StatePlane_Hawaii_4_FIPS_5104_Feet [4619] NAD_1983_StatePlane_Hawaii_5_FIPS_5105_Feet [4620] NAD_1983_StatePlane_Idaho_Central_FIPS_1102_Feet (deprecated) [4621] NAD_1983_StatePlane_Idaho_East_FIPS_1101_Feet (deprecated) [4622] NAD_1983_StatePlane_Idaho_West_FIPS_1103_Feet (deprecated) [4623] NAD_1983_StatePlane_Illinois_East_FIPS_1201_Feet (deprecated) [4624] NAD_1983_StatePlane_Illinois_West_FIPS_1202_Feet (deprecated) [4625] NAD_1983_StatePlane_Indiana_East_FIPS_1301_Feet (deprecated) [4626] NAD_1983_StatePlane_Indiana_West_FIPS_1302_Feet (deprecated) [4627] NAD_1983_StatePlane_Iowa_North_FIPS_1401_Feet (deprecated) [4628] NAD_1983_StatePlane_Iowa_South_FIPS_1402_Feet (deprecated) [4629] NAD_1983_StatePlane_Kansas_North_FIPS_1501_Feet (deprecated) [4630] NAD_1983_StatePlane_Kansas_South_FIPS_1502_Feet (deprecated) [4631] NAD_1983_StatePlane_Kentucky_FIPS_1600 (deprecated) [4632] NAD_1983_StatePlane_Kentucky_FIPS_1600_Feet (deprecated) [4633] NAD_1983_StatePlane_Kentucky_North_FIPS_1601_Feet (deprecated) [4634] NAD_1983_StatePlane_Kentucky_South_FIPS_1602_Feet (deprecated) [4635] NAD_1983_StatePlane_Louisiana_North_FIPS_1701_Feet (deprecated) [4636] NAD_1983_StatePlane_Louisiana_South_FIPS_1702_Feet (deprecated) [4637] NAD_1983_StatePlane_Maine_East_FIPS_1801_Feet (deprecated) [4638] NAD_1983_StatePlane_Maine_West_FIPS_1802_Feet (deprecated) [4639] NAD_1983_StatePlane_Maryland_FIPS_1900_Feet (deprecated) [4640] NAD_1983_StatePlane_Massachusetts_Island_FIPS_2002_Feet (deprecated) [4641] NAD_1983_StatePlane_Massachusetts_Mainland_FIPS_2001_Feet (deprecated) [4642] NAD_1983_StatePlane_Michigan_Central_FIPS_2112_Feet [4643] NAD_1983_StatePlane_Michigan_North_FIPS_2111_Feet [4644] NAD_1983_StatePlane_Michigan_South_FIPS_2113_Feet [4645] NAD_1983_StatePlane_Minnesota_Central_FIPS_2202_Feet (deprecated) [4646] NAD_1983_StatePlane_Minnesota_North_FIPS_2201_Feet (deprecated) [4647] NAD_1983_StatePlane_Minnesota_South_FIPS_2203_Feet (deprecated) [4648] NAD_1983_StatePlane_Mississippi_East_FIPS_2301_Feet (deprecated) [4649] NAD_1983_StatePlane_Mississippi_West_FIPS_2302_Feet (deprecated) [4650] NAD_1983_StatePlane_Missouri_Central_FIPS_2402_Feet [4651] NAD_1983_StatePlane_Missouri_East_FIPS_2401_Feet [4652] NAD_1983_StatePlane_Missouri_West_FIPS_2403_Feet [4653] NAD_1983_StatePlane_Montana_FIPS_2500_Feet [4654] NAD_1983_StatePlane_Nebraska_FIPS_2600_Feet (deprecated) [4655] NAD_1983_StatePlane_Nevada_Central_FIPS_2702_Feet (deprecated) [4656] NAD_1983_StatePlane_Nevada_East_FIPS_2701_Feet (deprecated) [4657] NAD_1983_StatePlane_Nevada_West_FIPS_2703_Feet (deprecated) [4658] NAD_1983_StatePlane_New_Hampshire_FIPS_2800_Feet (deprecated) [4659] NAD_1983_StatePlane_New_Jersey_FIPS_2900_Feet (deprecated) [4660] NAD_1983_StatePlane_New_Mexico_Central_FIPS_3002_Feet (deprecated) [4661] NAD_1983_StatePlane_New_Mexico_East_FIPS_3001_Feet (deprecated) [4662] NAD_1983_StatePlane_New_Mexico_West_FIPS_3003_Feet (deprecated) [4663] NAD_1983_StatePlane_New_York_Central_FIPS_3102_Feet (deprecated) [4664] NAD_1983_StatePlane_New_York_East_FIPS_3101_Feet (deprecated) [4665] NAD_1983_StatePlane_New_York_Long_Island_FIPS_3104_Feet (deprecated) [4666] NAD_1983_StatePlane_New_York_West_FIPS_3103_Feet (deprecated) [4667] NAD_1983_StatePlane_North_Carolina_FIPS_3200_Feet (deprecated) [4668] NAD_1983_StatePlane_North_Dakota_North_FIPS_3301_Feet [4669] NAD_1983_StatePlane_North_Dakota_South_FIPS_3302_Feet [4670] NAD_1983_StatePlane_Ohio_North_FIPS_3401_Feet (deprecated) [4671] NAD_1983_StatePlane_Ohio_South_FIPS_3402_Feet (deprecated) [4672] NAD_1983_StatePlane_Oklahoma_North_FIPS_3501_Feet (deprecated) [4673] NAD_1983_StatePlane_Oklahoma_South_FIPS_3502_Feet (deprecated) [4674] NAD_1983_StatePlane_Oregon_North_FIPS_3601_Feet [4675] NAD_1983_StatePlane_Oregon_South_FIPS_3602_Feet [4676] NAD_1983_StatePlane_Pennsylvania_North_FIPS_3701_Feet (deprecated) [4677] NAD_1983_StatePlane_Pennsylvania_South_FIPS_3702_Feet (deprecated) [4678] NAD_1983_StatePlane_Puerto_Rico_Virgin_Islands_FIPS_5200_Feet [4679] NAD_1983_StatePlane_Rhode_Island_FIPS_3800_Feet (deprecated) [4680] NAD_1983_StatePlane_South_Carolina_FIPS_3900_Feet [4681] NAD_1983_StatePlane_South_Dakota_North_FIPS_4001_Feet (deprecated) [4682] NAD_1983_StatePlane_South_Dakota_South_FIPS_4002_Feet (deprecated) [4683] NAD_1983_StatePlane_Tennessee_FIPS_4100_Feet (deprecated) [4684] NAD_1983_StatePlane_Texas_Central_FIPS_4203_Feet (deprecated) [4685] NAD_1983_StatePlane_Texas_North_Central_FIPS_4202_Feet (deprecated) [4686] NAD_1983_StatePlane_Texas_North_FIPS_4201_Feet (deprecated) [4687] NAD_1983_StatePlane_Texas_South_Central_FIPS_4204_Feet (deprecated) [4688] NAD_1983_StatePlane_Texas_South_FIPS_4205_Feet (deprecated) [4689] NAD_1983_StatePlane_Utah_Central_FIPS_4302_Feet (deprecated) [4690] NAD_1983_StatePlane_Utah_North_FIPS_4301_Feet (deprecated) [4691] NAD_1983_StatePlane_Utah_South_FIPS_4303_Feet (deprecated) [4692] NAD_1983_StatePlane_Vermont_FIPS_4400_Feet (deprecated) [4693] NAD_1983_StatePlane_Virginia_North_FIPS_4501_Feet (deprecated) [4694] NAD_1983_StatePlane_Virginia_South_FIPS_4502_Feet (deprecated) [4695] NAD_1983_StatePlane_Washington_North_FIPS_4601_Feet (deprecated) [4696] NAD_1983_StatePlane_Washington_South_FIPS_4602_Feet (deprecated) [4697] NAD_1983_StatePlane_West_Virginia_North_FIPS_4701_Feet (deprecated) [4698] NAD_1983_StatePlane_West_Virginia_South_FIPS_4702_Feet (deprecated) [4699] NAD_1983_StatePlane_Wisconsin_Central_FIPS_4802_Feet (deprecated) [4700] NAD_1983_StatePlane_Wisconsin_North_FIPS_4801_Feet (deprecated) [4701] NAD_1983_StatePlane_Wisconsin_South_FIPS_4803_Feet (deprecated) [4702] NAD_1983_StatePlane_Wyoming_East_Central_FIPS_4902_Feet (deprecated) [4703] NAD_1983_StatePlane_Wyoming_East_FIPS_4901_Feet (deprecated) [4704] NAD_1983_StatePlane_Wyoming_West_Central_FIPS_4903_Feet (deprecated) [4705] NAD_1983_StatePlane_Wyoming_West_FIPS_4904_Feet (deprecated) [4706] NAD_1983_Texas_Centric_Mapping_System_Albers (deprecated) [4707] NAD_1983_Texas_Centric_Mapping_System_Lambert (deprecated) [4708] NAD_1983_Texas_Statewide_Mapping_System (deprecated) [4709] NAD_1983_USFS_R6_Albers [4710] NAD_1983_USFS_R9_Albers [4711] NAD_1983_UTM_Zone_1N (deprecated) [4712] NAD_1983_UTM_Zone_2N (deprecated) [4713] NAD_1983_UTM_Zone_58N [4714] NAD_1983_UTM_Zone_59N (deprecated) [4715] NAD_1983_UTM_Zone_60N (deprecated) [4716] NAD_1983_Wisconsin_TM_US_Ft [4717] NAD_1983_WyLAM [4718] NEA74 Noumea / Noumea Lambert [4719] NEA74 Noumea / Noumea Lambert 2 [4720] NEA74 Noumea / UTM zone 58S [4721] NGN / UTM zone 38N [4722] NGN / UTM zone 39N [4723] NGO 1948 (Oslo) / NGO zone I [4724] NGO 1948 (Oslo) / NGO zone II [4725] NGO 1948 (Oslo) / NGO zone III [4726] NGO 1948 (Oslo) / NGO zone IV [4727] NGO 1948 (Oslo) / NGO zone V [4728] NGO 1948 (Oslo) / NGO zone VI [4729] NGO 1948 (Oslo) / NGO zone VII [4730] NGO 1948 (Oslo) / NGO zone VIII [4731] NGO_1948_Baerum_Kommune [4732] NGO_1948_Bergenhalvoen [4733] NGO_1948_Norway_Zone_1 [4734] NGO_1948_Norway_Zone_2 [4735] NGO_1948_Norway_Zone_3 [4736] NGO_1948_Norway_Zone_4 [4737] NGO_1948_Norway_Zone_5 [4738] NGO_1948_Norway_Zone_6 [4739] NGO_1948_Norway_Zone_7 [4740] NGO_1948_Norway_Zone_8 [4741] NGO_1948_Oslo_Baerum_Kommune [4742] NGO_1948_Oslo_Bergenhalvoen [4743] NGO_1948_Oslo_Kommune [4744] NGO_1948_Oslo_Oslo_Kommune [4745] NGO_1948_UTM_Zone_32N [4746] NGO_1948_UTM_Zone_33N [4747] NGO_1948_UTM_Zone_34N [4748] NGO_1948_UTM_Zone_35N [4749] NSIDC EASE-Grid Global [4750] NSIDC EASE-Grid North [4751] NSIDC EASE-Grid South [4752] NSIDC Sea Ice Polar Stereographic North [4753] NSIDC Sea Ice Polar Stereographic South [4754] NTF (Paris) / Centre France (deprecated) [4755] NTF (Paris) / Corse (deprecated) [4756] NTF (Paris) / France I (deprecated) [4757] NTF (Paris) / France II (deprecated) [4758] NTF (Paris) / France III (deprecated) [4759] NTF (Paris) / France IV (deprecated) [4760] NTF (Paris) / Lambert Centre France [4761] NTF (Paris) / Lambert Corse [4762] NTF (Paris) / Lambert Nord France [4763] NTF (Paris) / Lambert Sud France [4764] NTF (Paris) / Lambert zone I [4765] NTF (Paris) / Lambert zone II [4766] NTF (Paris) / Lambert zone III [4767] NTF (Paris) / Lambert zone IV [4768] NTF (Paris) / Nord France (deprecated) [4769] NTF (Paris) / Sud France (deprecated) [4770] NTF_France_III_degrees [4771] NTF_France_II_degrees [4772] NTF_France_IV_degrees [4773] NTF_France_I_degrees [4774] NTF_Lambert_Zone_I [4775] NTF_Lambert_Zone_II [4776] NTF_Lambert_Zone_III [4777] NTF_Lambert_Zone_IV [4778] NZGD2000 / Amuri 2000 [4779] NZGD2000 / Antipodes Islands TM 2000 [4780] NZGD2000 / Auckland Islands TM 2000 [4781] NZGD2000 / Bay of Plenty 2000 [4782] NZGD2000 / Bluff 2000 [4783] NZGD2000 / Buller 2000 [4784] NZGD2000 / Campbell Island TM 2000 [4785] NZGD2000 / Chatham Island Circuit 2000 [4786] NZGD2000 / Chatham Islands TM 2000 [4787] NZGD2000 / Collingwood 2000 [4788] NZGD2000 / Gawler 2000 [4789] NZGD2000 / Grey 2000 [4790] NZGD2000 / Hawkes Bay 2000 [4791] NZGD2000 / Hokitika 2000 [4792] NZGD2000 / Jacksons Bay 2000 [4793] NZGD2000 / Karamea 2000 [4794] NZGD2000 / Lindis Peak 2000 [4795] NZGD2000 / Marlborough 2000 [4796] NZGD2000 / Mount Eden 2000 [4797] NZGD2000 / Mount Nicholas 2000 [4798] NZGD2000 / Mount Pleasant 2000 [4799] NZGD2000 / Mount York 2000 [4800] NZGD2000 / NZCS2000 [4801] NZGD2000 / Nelson 2000 [4802] NZGD2000 / New Zealand Transverse Mercator 2000 [4803] NZGD2000 / North Taieri 2000 [4804] NZGD2000 / Observation Point 2000 [4805] NZGD2000 / Okarito 2000 [4806] NZGD2000 / Poverty Bay 2000 [4807] NZGD2000 / Raoul Island TM 2000 [4808] NZGD2000 / Taranaki 2000 [4809] NZGD2000 / Timaru 2000 [4810] NZGD2000 / Tuhirangi 2000 [4811] NZGD2000 / UTM zone 1S [4812] NZGD2000 / UTM zone 58S [4813] NZGD2000 / UTM zone 59S [4814] NZGD2000 / UTM zone 60S [4815] NZGD2000 / Wairarapa 2000 [4816] NZGD2000 / Wanganui 2000 [4817] NZGD2000 / Wellington 2000 [4818] NZGD49 / Amuri Circuit [4819] NZGD49 / Bay of Plenty Circuit [4820] NZGD49 / Bluff Circuit [4821] NZGD49 / Buller Circuit [4822] NZGD49 / Collingwood Circuit [4823] NZGD49 / Gawler Circuit [4824] NZGD49 / Grey Circuit [4825] NZGD49 / Hawkes Bay Circuit [4826] NZGD49 / Hokitika Circuit [4827] NZGD49 / Jacksons Bay Circuit [4828] NZGD49 / Karamea Circuit [4829] NZGD49 / Lindis Peak Circuit [4830] NZGD49 / Marlborough Circuit [4831] NZGD49 / Mount Eden Circuit [4832] NZGD49 / Mount Nicholas Circuit [4833] NZGD49 / Mount Pleasant Circuit [4834] NZGD49 / Mount York Circuit [4835] NZGD49 / Nelson Circuit [4836] NZGD49 / New Zealand Map Grid [4837] NZGD49 / North Island Grid [4838] NZGD49 / North Taieri Circuit [4839] NZGD49 / Observation Point Circuit [4840] NZGD49 / Okarito Circuit [4841] NZGD49 / Poverty Bay Circuit [4842] NZGD49 / South Island Grid [4843] NZGD49 / Taranaki Circuit [4844] NZGD49 / Timaru Circuit [4845] NZGD49 / Tuhirangi Circuit [4846] NZGD49 / UTM zone 58S [4847] NZGD49 / UTM zone 59S [4848] NZGD49 / UTM zone 60S [4849] NZGD49 / Wairarapa Circuit [4850] NZGD49 / Wanganui Circuit [4851] NZGD49 / Wellington Circuit [4852] NZGD_2000_Chatham_Island_Circuit (deprecated) [4853] Nahrwan 1934 / Iraq zone [4854] Nahrwan 1934 / UTM zone 37N [4855] Nahrwan 1934 / UTM zone 38N [4856] Nahrwan 1934 / UTM zone 39N [4857] Nahrwan 1967 / UTM zone 37N (deprecated) [4858] Nahrwan 1967 / UTM zone 38N (deprecated) [4859] Nahrwan 1967 / UTM zone 39N [4860] Nahrwan 1967 / UTM zone 40N [4861] Nakhl-e Ghanem / UTM zone 39N [4862] Naparima 1955 / UTM zone 20N [4863] Naparima 1972 / UTM zone 20N [4864] Nepal_Nagarkot_TM [4865] New Beijing / 3-degree Gauss-Kruger CM 102E [4866] New Beijing / 3-degree Gauss-Kruger CM 105E [4867] New Beijing / 3-degree Gauss-Kruger CM 108E [4868] New Beijing / 3-degree Gauss-Kruger CM 111E [4869] New Beijing / 3-degree Gauss-Kruger CM 114E [4870] New Beijing / 3-degree Gauss-Kruger CM 117E [4871] New Beijing / 3-degree Gauss-Kruger CM 120E [4872] New Beijing / 3-degree Gauss-Kruger CM 123E [4873] New Beijing / 3-degree Gauss-Kruger CM 126E [4874] New Beijing / 3-degree Gauss-Kruger CM 129E [4875] New Beijing / 3-degree Gauss-Kruger CM 132E [4876] New Beijing / 3-degree Gauss-Kruger CM 135E [4877] New Beijing / 3-degree Gauss-Kruger CM 75E [4878] New Beijing / 3-degree Gauss-Kruger CM 78E [4879] New Beijing / 3-degree Gauss-Kruger CM 81E [4880] New Beijing / 3-degree Gauss-Kruger CM 84E [4881] New Beijing / 3-degree Gauss-Kruger CM 87E [4882] New Beijing / 3-degree Gauss-Kruger CM 90E [4883] New Beijing / 3-degree Gauss-Kruger CM 93E [4884] New Beijing / 3-degree Gauss-Kruger CM 96E [4885] New Beijing / 3-degree Gauss-Kruger CM 99E [4886] New Beijing / 3-degree Gauss-Kruger zone 25 [4887] New Beijing / 3-degree Gauss-Kruger zone 26 [4888] New Beijing / 3-degree Gauss-Kruger zone 27 [4889] New Beijing / 3-degree Gauss-Kruger zone 28 [4890] New Beijing / 3-degree Gauss-Kruger zone 29 [4891] New Beijing / 3-degree Gauss-Kruger zone 30 [4892] New Beijing / 3-degree Gauss-Kruger zone 31 [4893] New Beijing / 3-degree Gauss-Kruger zone 32 [4894] New Beijing / 3-degree Gauss-Kruger zone 33 [4895] New Beijing / 3-degree Gauss-Kruger zone 34 [4896] New Beijing / 3-degree Gauss-Kruger zone 35 [4897] New Beijing / 3-degree Gauss-Kruger zone 36 [4898] New Beijing / 3-degree Gauss-Kruger zone 37 [4899] New Beijing / 3-degree Gauss-Kruger zone 38 [4900] New Beijing / 3-degree Gauss-Kruger zone 39 [4901] New Beijing / 3-degree Gauss-Kruger zone 40 [4902] New Beijing / 3-degree Gauss-Kruger zone 41 [4903] New Beijing / 3-degree Gauss-Kruger zone 42 [4904] New Beijing / 3-degree Gauss-Kruger zone 43 [4905] New Beijing / 3-degree Gauss-Kruger zone 44 [4906] New Beijing / 3-degree Gauss-Kruger zone 45 [4907] New Beijing / Gauss-Kruger CM 105E [4908] New Beijing / Gauss-Kruger CM 111E [4909] New Beijing / Gauss-Kruger CM 117E [4910] New Beijing / Gauss-Kruger CM 123E [4911] New Beijing / Gauss-Kruger CM 129E [4912] New Beijing / Gauss-Kruger CM 135E [4913] New Beijing / Gauss-Kruger CM 75E [4914] New Beijing / Gauss-Kruger CM 81E [4915] New Beijing / Gauss-Kruger CM 87E [4916] New Beijing / Gauss-Kruger CM 93E [4917] New Beijing / Gauss-Kruger CM 99E [4918] New Beijing / Gauss-Kruger zone 13 [4919] New Beijing / Gauss-Kruger zone 14 [4920] New Beijing / Gauss-Kruger zone 15 [4921] New Beijing / Gauss-Kruger zone 16 [4922] New Beijing / Gauss-Kruger zone 17 [4923] New Beijing / Gauss-Kruger zone 18 [4924] New Beijing / Gauss-Kruger zone 19 [4925] New Beijing / Gauss-Kruger zone 20 [4926] New Beijing / Gauss-Kruger zone 21 [4927] New Beijing / Gauss-Kruger zone 22 [4928] New Beijing / Gauss-Kruger zone 23 [4929] New_Beijing_3_Degree_Gauss_Kruger_CM_132E [4930] Nord Sahara 1959 / Nord Algerie [4931] Nord Sahara 1959 / Sud Algerie [4932] Nord Sahara 1959 / UTM zone 29N [4933] Nord Sahara 1959 / UTM zone 30N [4934] Nord Sahara 1959 / UTM zone 31N [4935] Nord Sahara 1959 / UTM zone 32N [4936] Nord_Algerie (deprecated) [4937] Nord_Algerie_Ancienne_Degree [4938] Nord_Algerie_Degree (deprecated) [4939] Nord_Maroc_Degree [4940] North_America_Albers_Equal_Area_Conic [4941] North_America_Equidistant_Conic [4942] North_America_Lambert_Conformal_Conic [4943] North_Pole_Azimuthal_Equidistant [4944] North_Pole_Gnomonic [4945] North_Pole_Lambert_Azimuthal_Equal_Area [4946] North_Pole_Orthographic [4947] North_Pole_Stereographic [4948] OCRS_Baker_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4949] OCRS_Baker_NAD_1983_CORS96_TM_Meters (deprecated) [4950] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4951] OCRS_Bend-Burns_NAD_1983_CORS96_LCC_Meters (deprecated) [4952] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4953] OCRS_Bend-Klamath_Falls_NAD_1983_CORS96_TM_Meters (deprecated) [4954] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4955] OCRS_Bend-Redmond-Prineville_NAD_1983_CORS96_LCC_Meters (deprecated) [4956] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4957] OCRS_Canyonville-Grants_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4958] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4959] OCRS_Columbia_River_East_NAD_1983_CORS96_LCC_Meters (deprecated) [4960] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4961] OCRS_Columbia_River_West_NAD_1983_CORS96_OM_Meters (deprecated) [4962] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4963] OCRS_Cottage_Grove-Canyonville_NAD_1983_CORS96_TM_Meters (deprecated) [4964] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4965] OCRS_Dufur-Madras_NAD_1983_CORS96_TM_Meters (deprecated) [4966] OCRS_Eugene_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4967] OCRS_Eugene_NAD_1983_CORS96_TM_Meters (deprecated) [4968] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4969] OCRS_Grants_Pass-Ashland_NAD_1983_CORS96_TM_Meters (deprecated) [4970] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4971] OCRS_Gresham-Warm_Springs_NAD_1983_CORS96_TM_Meters (deprecated) [4972] OCRS_La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4973] OCRS_La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4974] OCRS_Ontario_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4975] OCRS_Ontario_NAD_1983_CORS96_TM_Meters (deprecated) [4976] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Feet_Intl (deprecated) [4977] OCRS_Oregon_Coast_NAD_1983_CORS96_OM_Meters (deprecated) [4978] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4979] OCRS_Pendleton-La_Grande_NAD_1983_CORS96_TM_Meters (deprecated) [4980] OCRS_Pendleton_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4981] OCRS_Pendleton_NAD_1983_CORS96_TM_Meters (deprecated) [4982] OCRS_Portland_NAD_1983_CORS96_LCC_Feet_Intl (deprecated) [4983] OCRS_Portland_NAD_1983_CORS96_LCC_Meters (deprecated) [4984] OCRS_Salem_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4985] OCRS_Salem_NAD_1983_CORS96_TM_Meters (deprecated) [4986] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Feet_Intl (deprecated) [4987] OCRS_Santiam_Pass_NAD_1983_CORS96_TM_Meters (deprecated) [4988] ONGD14 / UTM zone 39N [4989] ONGD14 / UTM zone 40N [4990] ONGD14 / UTM zone 41N [4991] OSGB 1936 / British National Grid [4992] OSGB36_Highways_England_A10H1 [4993] OSGB36_Highways_England_A10H2 [4994] OSGB36_Highways_England_A11H1 [4995] OSGB36_Highways_England_A11H2 [4996] OSGB36_Highways_England_A11H3 [4997] OSGB36_Highways_England_A12H1 [4998] OSGB36_Highways_England_A12H2 [4999] OSGB36_Highways_England_A12H3 [5000] OSGB36_Highways_England_A13H1 [5001] OSGB36_Highways_England_A13H2 [5002] OSGB36_Highways_England_A14H1 [5003] OSGB36_Highways_England_A14H2 [5004] OSGB36_Highways_England_A15H1 [5005] OSGB36_Highways_England_A15H2 [5006] OSGB36_Highways_England_A16H1 [5007] OSGB36_Highways_England_A16H2 [5008] OSGB36_Highways_England_A17H1 [5009] OSGB36_Highways_England_A17H2 [5010] OSGB36_Highways_England_A18H1 [5011] OSGB36_Highways_England_A18H2 [5012] OSGB36_Highways_England_A19H1 [5013] OSGB36_Highways_England_A19H2 [5014] OSGB36_Highways_England_A1H1 [5015] OSGB36_Highways_England_A20H1 [5016] OSGB36_Highways_England_A20H2 [5017] OSGB36_Highways_England_A21H1 [5018] OSGB36_Highways_England_A21H2 [5019] OSGB36_Highways_England_A22H1 [5020] OSGB36_Highways_England_A22H2 [5021] OSGB36_Highways_England_A23H1 [5022] OSGB36_Highways_England_A23H2 [5023] OSGB36_Highways_England_A24H1 [5024] OSGB36_Highways_England_A24H2 [5025] OSGB36_Highways_England_A25H1 [5026] OSGB36_Highways_England_A25H2 [5027] OSGB36_Highways_England_A26H1 [5028] OSGB36_Highways_England_A26H2 [5029] OSGB36_Highways_England_A27H1 [5030] OSGB36_Highways_England_A27H2 [5031] OSGB36_Highways_England_A28H1 [5032] OSGB36_Highways_England_A28H2 [5033] OSGB36_Highways_England_A29H1 [5034] OSGB36_Highways_England_A2H1 [5035] OSGB36_Highways_England_A30H1 [5036] OSGB36_Highways_England_A3H1 [5037] OSGB36_Highways_England_A3H2 [5038] OSGB36_Highways_England_A4H1 [5039] OSGB36_Highways_England_A5H1 [5040] OSGB36_Highways_England_A5H2 [5041] OSGB36_Highways_England_A6H1 [5042] OSGB36_Highways_England_A6H2 [5043] OSGB36_Highways_England_A7H1 [5044] OSGB36_Highways_England_A7H2 [5045] OSGB36_Highways_England_A8H1 [5046] OSGB36_Highways_England_A8H2 [5047] OSGB36_Highways_England_A9H1 [5048] OSGB36_Highways_England_A9H2 [5049] OSGB36_Highways_England_B15H1 [5050] OSGB36_Highways_England_B15H2 [5051] OSGB36_Highways_England_B15H3 [5052] OSGB36_Highways_England_B16H1 [5053] OSGB36_Highways_England_B16H2 [5054] OSGB36_Highways_England_B16H3 [5055] OSGB36_Highways_England_B16H4 [5056] OSGB36_Highways_England_B17H1 [5057] OSGB36_Highways_England_B17H2 [5058] OSGB36_Highways_England_B18H1 [5059] OSGB36_Highways_England_B18H2 [5060] OSGB36_Highways_England_B19H1 [5061] OSGB36_Highways_England_B20H1 [5062] OSGB36_Highways_England_B21H1 [5063] OSGB36_Highways_England_B22H1 [5064] OSGB36_Highways_England_B23H1 [5065] OSGB36_Highways_England_B24H1 [5066] OSGB36_Highways_England_B25H1 [5067] OSGB36_Highways_England_B26H1 [5068] OSGB36_Highways_England_B27H1 [5069] OSGB36_Highways_England_B28H1 [5070] OSGB36_Highways_England_B29H1 [5071] OSGB36_Highways_England_B30H1 [5072] OSGB36_Highways_England_B31H1 [5073] OSGB36_Highways_England_B32H1 [5074] OSGB36_Highways_England_C13H1 [5075] OSGB36_Highways_England_C14H1 [5076] OSGB36_Highways_England_C14H2 [5077] OSGB36_Highways_England_C14H3 [5078] OSGB36_Highways_England_C14H4 [5079] OSGB36_Highways_England_C15H1 [5080] OSGB36_Highways_England_C15H2 [5081] OSGB36_Highways_England_C15H3 [5082] OSGB36_Highways_England_C15H4 [5083] OSGB36_Highways_England_C15H5 [5084] OSGB36_Highways_England_C16H1 [5085] OSGB36_Highways_England_C16H2 [5086] OSGB36_Highways_England_C16H3 [5087] OSGB36_Highways_England_C16H4 [5088] OSGB36_Highways_England_C17H1 [5089] OSGB36_Highways_England_C17H2 [5090] OSGB36_Highways_England_C18H1 [5091] OSGB36_Highways_England_C18H2 [5092] OSGB36_Highways_England_C19H1 [5093] OSGB36_Highways_England_C19H2 [5094] OSNI 1952 / Irish National Grid [5095] Observatorio_Meteorologico_1939_UTM_Zone_25N [5096] Observatorio_Meteorologico_1965_Macau_Grid [5097] Ocotepeque 1935 / Costa Rica Norte [5098] Ocotepeque 1935 / Costa Rica Sur [5099] Ocotepeque 1935 / El Salvador Lambert [5100] Ocotepeque 1935 / Guatemala Norte [5101] Ocotepeque 1935 / Guatemala Norte (deprecated) [5102] Ocotepeque 1935 / Guatemala Sur [5103] Ocotepeque 1935 / Nicaragua Norte [5104] Ocotepeque 1935 / Nicaragua Sur [5105] Ocotepeque_1935_Costa_Rica_Lambert_Norte [5106] Ocotepeque_1935_Costa_Rica_Lambert_Sur [5107] Old Hawaiian / Hawaii zone 1 [5108] Old Hawaiian / Hawaii zone 2 [5109] Old Hawaiian / Hawaii zone 3 [5110] Old Hawaiian / Hawaii zone 4 [5111] Old Hawaiian / Hawaii zone 5 [5112] Old_Hawaiian_UTM_Zone_4N [5113] Old_Hawaiian_UTM_Zone_5N [5114] PD/83 / 3-degree Gauss-Kruger zone 3 [5115] PD/83 / 3-degree Gauss-Kruger zone 3 (E-N) [5116] PD/83 / 3-degree Gauss-Kruger zone 4 [5117] PD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5118] PNG94 / PNGMG94 zone 54 [5119] PNG94 / PNGMG94 zone 55 [5120] PNG94 / PNGMG94 zone 56 [5121] POSGAR 2007 / Argentina 1 [5122] POSGAR 2007 / Argentina 2 [5123] POSGAR 2007 / Argentina 3 [5124] POSGAR 2007 / Argentina 4 [5125] POSGAR 2007 / Argentina 5 [5126] POSGAR 2007 / Argentina 6 [5127] POSGAR 2007 / Argentina 7 [5128] POSGAR 94 / Argentina 1 [5129] POSGAR 94 / Argentina 2 [5130] POSGAR 94 / Argentina 3 [5131] POSGAR 94 / Argentina 4 [5132] POSGAR 94 / Argentina 5 [5133] POSGAR 94 / Argentina 6 [5134] POSGAR 94 / Argentina 7 [5135] POSGAR 98 / Argentina 1 [5136] POSGAR 98 / Argentina 2 [5137] POSGAR 98 / Argentina 3 [5138] POSGAR 98 / Argentina 4 [5139] POSGAR 98 / Argentina 5 [5140] POSGAR 98 / Argentina 6 [5141] POSGAR 98 / Argentina 7 [5142] PRS92 / Philippines zone 1 [5143] PRS92 / Philippines zone 2 [5144] PRS92 / Philippines zone 3 [5145] PRS92 / Philippines zone 4 [5146] PRS92 / Philippines zone 5 [5147] PRS_1992_UTM_Zone_50N [5148] PRS_1992_UTM_Zone_51N [5149] PRS_1992_UTM_Zone_52N [5150] PSAD56 / ICN Regional [5151] PSAD56 / Peru central zone [5152] PSAD56 / Peru east zone [5153] PSAD56 / Peru west zone [5154] PSAD56 / UTM zone 17N [5155] PSAD56 / UTM zone 17S [5156] PSAD56 / UTM zone 18N [5157] PSAD56 / UTM zone 18S [5158] PSAD56 / UTM zone 19N [5159] PSAD56 / UTM zone 19S [5160] PSAD56 / UTM zone 20N [5161] PSAD56 / UTM zone 20S [5162] PSAD56 / UTM zone 21N [5163] PSAD56 / UTM zone 21S [5164] PSAD56 / UTM zone 22S [5165] PSD93 / UTM zone 39N [5166] PSD93 / UTM zone 40N [5167] PTRA08 / LAEA Europe [5168] PTRA08 / LCC Europe [5169] PTRA08 / UTM zone 25N [5170] PTRA08 / UTM zone 26N [5171] PTRA08 / UTM zone 28N [5172] PTRA08_UTM_Zone_25N (deprecated) [5173] PTRA08_UTM_Zone_26N (deprecated) [5174] PTRA08_UTM_Zone_28N (deprecated) [5175] Palestine 1923 / Israeli CS Grid [5176] Palestine 1923 / Palestine Belt [5177] Palestine 1923 / Palestine Grid [5178] Palestine 1923 / Palestine Grid modified [5179] Palestine_1923_Palestine_Grid_TM (deprecated) [5180] Pampa del Castillo / Argentina 2 [5181] Panama-Colon 1911 / Panama Lambert [5182] Panama-Colon 1911 / Panama Polyconic [5183] Panhandle_Energy_Albers [5184] Peru96 / UTM zone 17S [5185] Peru96 / UTM zone 17S (deprecated) [5186] Peru96 / UTM zone 18S [5187] Peru96 / UTM zone 19S [5188] Pitcairn 1967 / UTM zone 9S [5189] Pitcairn 2006 / Pitcairn TM 2006 [5190] Pohnpei_Az_Eq_1971 [5191] Pointe Noire / UTM zone 32S [5192] Popular Visualisation CRS / Mercator (deprecated) [5193] Popular Visualisation CRS / Mercator (deprecated) [5194] Porto Santo / UTM zone 28N [5195] Porto Santo 1995 / UTM zone 28N [5196] Porto_Santo_1936_UTM_Zone_28N (deprecated) [5197] Puerto Rico / St. Croix [5198] Puerto Rico / UTM zone 20N [5199] Puerto Rico State Plane CS of 1927 [5200] Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E [5201] Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E [5202] Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E [5203] Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E [5204] Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E [5205] Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E [5206] Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E [5207] Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E [5208] Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E [5209] Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E [5210] Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E [5211] Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E [5212] Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E [5213] Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E [5214] Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E [5215] Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E [5216] Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E [5217] Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E [5218] Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E [5219] Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E [5220] Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E [5221] Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E [5222] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E [5223] Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W [5224] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E [5225] Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W [5226] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E [5227] Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W [5228] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E [5229] Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W [5230] Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E [5231] Pulkovo 1942 / 3-degree Gauss-Kruger CM 18E (deprecated) [5232] Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E [5233] Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E [5234] Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E [5235] Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E [5236] Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E [5237] Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E [5238] Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E [5239] Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E [5240] Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E [5241] Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E [5242] Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E [5243] Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E [5244] Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E [5245] Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E [5246] Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E [5247] Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E [5248] Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E [5249] Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E [5250] Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E [5251] Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E [5252] Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E [5253] Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E [5254] Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E [5255] Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E [5256] Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E [5257] Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E [5258] Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E [5259] Pulkovo 1942 / 3-degree Gauss-Kruger zone 10 [5260] Pulkovo 1942 / 3-degree Gauss-Kruger zone 11 [5261] Pulkovo 1942 / 3-degree Gauss-Kruger zone 12 [5262] Pulkovo 1942 / 3-degree Gauss-Kruger zone 13 [5263] Pulkovo 1942 / 3-degree Gauss-Kruger zone 14 [5264] Pulkovo 1942 / 3-degree Gauss-Kruger zone 15 [5265] Pulkovo 1942 / 3-degree Gauss-Kruger zone 16 [5266] Pulkovo 1942 / 3-degree Gauss-Kruger zone 17 [5267] Pulkovo 1942 / 3-degree Gauss-Kruger zone 18 [5268] Pulkovo 1942 / 3-degree Gauss-Kruger zone 19 [5269] Pulkovo 1942 / 3-degree Gauss-Kruger zone 20 [5270] Pulkovo 1942 / 3-degree Gauss-Kruger zone 21 [5271] Pulkovo 1942 / 3-degree Gauss-Kruger zone 22 [5272] Pulkovo 1942 / 3-degree Gauss-Kruger zone 23 [5273] Pulkovo 1942 / 3-degree Gauss-Kruger zone 24 [5274] Pulkovo 1942 / 3-degree Gauss-Kruger zone 25 [5275] Pulkovo 1942 / 3-degree Gauss-Kruger zone 26 [5276] Pulkovo 1942 / 3-degree Gauss-Kruger zone 27 [5277] Pulkovo 1942 / 3-degree Gauss-Kruger zone 28 [5278] Pulkovo 1942 / 3-degree Gauss-Kruger zone 29 [5279] Pulkovo 1942 / 3-degree Gauss-Kruger zone 30 [5280] Pulkovo 1942 / 3-degree Gauss-Kruger zone 31 [5281] Pulkovo 1942 / 3-degree Gauss-Kruger zone 32 [5282] Pulkovo 1942 / 3-degree Gauss-Kruger zone 33 [5283] Pulkovo 1942 / 3-degree Gauss-Kruger zone 34 [5284] Pulkovo 1942 / 3-degree Gauss-Kruger zone 35 [5285] Pulkovo 1942 / 3-degree Gauss-Kruger zone 36 [5286] Pulkovo 1942 / 3-degree Gauss-Kruger zone 37 [5287] Pulkovo 1942 / 3-degree Gauss-Kruger zone 38 [5288] Pulkovo 1942 / 3-degree Gauss-Kruger zone 39 [5289] Pulkovo 1942 / 3-degree Gauss-Kruger zone 40 [5290] Pulkovo 1942 / 3-degree Gauss-Kruger zone 41 [5291] Pulkovo 1942 / 3-degree Gauss-Kruger zone 42 [5292] Pulkovo 1942 / 3-degree Gauss-Kruger zone 43 [5293] Pulkovo 1942 / 3-degree Gauss-Kruger zone 44 [5294] Pulkovo 1942 / 3-degree Gauss-Kruger zone 45 [5295] Pulkovo 1942 / 3-degree Gauss-Kruger zone 46 [5296] Pulkovo 1942 / 3-degree Gauss-Kruger zone 47 [5297] Pulkovo 1942 / 3-degree Gauss-Kruger zone 48 [5298] Pulkovo 1942 / 3-degree Gauss-Kruger zone 49 [5299] Pulkovo 1942 / 3-degree Gauss-Kruger zone 50 [5300] Pulkovo 1942 / 3-degree Gauss-Kruger zone 51 [5301] Pulkovo 1942 / 3-degree Gauss-Kruger zone 52 [5302] Pulkovo 1942 / 3-degree Gauss-Kruger zone 53 [5303] Pulkovo 1942 / 3-degree Gauss-Kruger zone 54 [5304] Pulkovo 1942 / 3-degree Gauss-Kruger zone 55 [5305] Pulkovo 1942 / 3-degree Gauss-Kruger zone 56 [5306] Pulkovo 1942 / 3-degree Gauss-Kruger zone 57 [5307] Pulkovo 1942 / 3-degree Gauss-Kruger zone 58 [5308] Pulkovo 1942 / 3-degree Gauss-Kruger zone 59 [5309] Pulkovo 1942 / 3-degree Gauss-Kruger zone 6 (deprecated) [5310] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 [5311] Pulkovo 1942 / 3-degree Gauss-Kruger zone 60 (deprecated) [5312] Pulkovo 1942 / 3-degree Gauss-Kruger zone 61 [5313] Pulkovo 1942 / 3-degree Gauss-Kruger zone 62 [5314] Pulkovo 1942 / 3-degree Gauss-Kruger zone 63 [5315] Pulkovo 1942 / 3-degree Gauss-Kruger zone 64 [5316] Pulkovo 1942 / 3-degree Gauss-Kruger zone 7 [5317] Pulkovo 1942 / 3-degree Gauss-Kruger zone 8 [5318] Pulkovo 1942 / 3-degree Gauss-Kruger zone 9 [5319] Pulkovo 1942 / CS63 zone A1 [5320] Pulkovo 1942 / CS63 zone A2 [5321] Pulkovo 1942 / CS63 zone A3 [5322] Pulkovo 1942 / CS63 zone A4 [5323] Pulkovo 1942 / CS63 zone C0 [5324] Pulkovo 1942 / CS63 zone C1 [5325] Pulkovo 1942 / CS63 zone C2 [5326] Pulkovo 1942 / CS63 zone K2 [5327] Pulkovo 1942 / CS63 zone K3 [5328] Pulkovo 1942 / CS63 zone K4 [5329] Pulkovo 1942 / CS63 zone X1 [5330] Pulkovo 1942 / CS63 zone X2 [5331] Pulkovo 1942 / CS63 zone X3 [5332] Pulkovo 1942 / CS63 zone X4 [5333] Pulkovo 1942 / CS63 zone X5 [5334] Pulkovo 1942 / CS63 zone X6 [5335] Pulkovo 1942 / CS63 zone X7 [5336] Pulkovo 1942 / Caspian Sea Mercator [5337] Pulkovo 1942 / Gauss-Kruger 10N (deprecated) [5338] Pulkovo 1942 / Gauss-Kruger 11N (deprecated) [5339] Pulkovo 1942 / Gauss-Kruger 12N (deprecated) [5340] Pulkovo 1942 / Gauss-Kruger 13N (deprecated) [5341] Pulkovo 1942 / Gauss-Kruger 14N (deprecated) [5342] Pulkovo 1942 / Gauss-Kruger 15N (deprecated) [5343] Pulkovo 1942 / Gauss-Kruger 16N (deprecated) [5344] Pulkovo 1942 / Gauss-Kruger 17N (deprecated) [5345] Pulkovo 1942 / Gauss-Kruger 18N (deprecated) [5346] Pulkovo 1942 / Gauss-Kruger 19N (deprecated) [5347] Pulkovo 1942 / Gauss-Kruger 20N (deprecated) [5348] Pulkovo 1942 / Gauss-Kruger 21N (deprecated) [5349] Pulkovo 1942 / Gauss-Kruger 22N (deprecated) [5350] Pulkovo 1942 / Gauss-Kruger 23N (deprecated) [5351] Pulkovo 1942 / Gauss-Kruger 24N (deprecated) [5352] Pulkovo 1942 / Gauss-Kruger 25N (deprecated) [5353] Pulkovo 1942 / Gauss-Kruger 26N (deprecated) [5354] Pulkovo 1942 / Gauss-Kruger 27N (deprecated) [5355] Pulkovo 1942 / Gauss-Kruger 28N (deprecated) [5356] Pulkovo 1942 / Gauss-Kruger 29N (deprecated) [5357] Pulkovo 1942 / Gauss-Kruger 2N (deprecated) [5358] Pulkovo 1942 / Gauss-Kruger 30N (deprecated) [5359] Pulkovo 1942 / Gauss-Kruger 31N (deprecated) [5360] Pulkovo 1942 / Gauss-Kruger 32N (deprecated) [5361] Pulkovo 1942 / Gauss-Kruger 3N (deprecated) [5362] Pulkovo 1942 / Gauss-Kruger 4N (deprecated) [5363] Pulkovo 1942 / Gauss-Kruger 5N (deprecated) [5364] Pulkovo 1942 / Gauss-Kruger 6N (deprecated) [5365] Pulkovo 1942 / Gauss-Kruger 7N (deprecated) [5366] Pulkovo 1942 / Gauss-Kruger 8N (deprecated) [5367] Pulkovo 1942 / Gauss-Kruger 9N (deprecated) [5368] Pulkovo 1942 / Gauss-Kruger CM 105E [5369] Pulkovo 1942 / Gauss-Kruger CM 111E [5370] Pulkovo 1942 / Gauss-Kruger CM 117E [5371] Pulkovo 1942 / Gauss-Kruger CM 123E [5372] Pulkovo 1942 / Gauss-Kruger CM 129E [5373] Pulkovo 1942 / Gauss-Kruger CM 135E [5374] Pulkovo 1942 / Gauss-Kruger CM 141E [5375] Pulkovo 1942 / Gauss-Kruger CM 147E [5376] Pulkovo 1942 / Gauss-Kruger CM 153E [5377] Pulkovo 1942 / Gauss-Kruger CM 159E [5378] Pulkovo 1942 / Gauss-Kruger CM 15E (deprecated) [5379] Pulkovo 1942 / Gauss-Kruger CM 165E [5380] Pulkovo 1942 / Gauss-Kruger CM 171E [5381] Pulkovo 1942 / Gauss-Kruger CM 171W [5382] Pulkovo 1942 / Gauss-Kruger CM 177E [5383] Pulkovo 1942 / Gauss-Kruger CM 177W [5384] Pulkovo 1942 / Gauss-Kruger CM 21E [5385] Pulkovo 1942 / Gauss-Kruger CM 27E [5386] Pulkovo 1942 / Gauss-Kruger CM 33E [5387] Pulkovo 1942 / Gauss-Kruger CM 39E [5388] Pulkovo 1942 / Gauss-Kruger CM 45E [5389] Pulkovo 1942 / Gauss-Kruger CM 51E [5390] Pulkovo 1942 / Gauss-Kruger CM 57E [5391] Pulkovo 1942 / Gauss-Kruger CM 63E [5392] Pulkovo 1942 / Gauss-Kruger CM 69E [5393] Pulkovo 1942 / Gauss-Kruger CM 75E [5394] Pulkovo 1942 / Gauss-Kruger CM 81E [5395] Pulkovo 1942 / Gauss-Kruger CM 87E [5396] Pulkovo 1942 / Gauss-Kruger CM 93E [5397] Pulkovo 1942 / Gauss-Kruger CM 99E [5398] Pulkovo 1942 / Gauss-Kruger CM 9E (deprecated) [5399] Pulkovo 1942 / Gauss-Kruger zone 10 [5400] Pulkovo 1942 / Gauss-Kruger zone 11 [5401] Pulkovo 1942 / Gauss-Kruger zone 12 [5402] Pulkovo 1942 / Gauss-Kruger zone 13 [5403] Pulkovo 1942 / Gauss-Kruger zone 14 [5404] Pulkovo 1942 / Gauss-Kruger zone 15 [5405] Pulkovo 1942 / Gauss-Kruger zone 16 [5406] Pulkovo 1942 / Gauss-Kruger zone 17 [5407] Pulkovo 1942 / Gauss-Kruger zone 18 [5408] Pulkovo 1942 / Gauss-Kruger zone 19 [5409] Pulkovo 1942 / Gauss-Kruger zone 2 (deprecated) [5410] Pulkovo 1942 / Gauss-Kruger zone 20 [5411] Pulkovo 1942 / Gauss-Kruger zone 21 [5412] Pulkovo 1942 / Gauss-Kruger zone 22 [5413] Pulkovo 1942 / Gauss-Kruger zone 23 [5414] Pulkovo 1942 / Gauss-Kruger zone 24 [5415] Pulkovo 1942 / Gauss-Kruger zone 25 [5416] Pulkovo 1942 / Gauss-Kruger zone 26 [5417] Pulkovo 1942 / Gauss-Kruger zone 27 [5418] Pulkovo 1942 / Gauss-Kruger zone 28 [5419] Pulkovo 1942 / Gauss-Kruger zone 29 [5420] Pulkovo 1942 / Gauss-Kruger zone 3 (deprecated) [5421] Pulkovo 1942 / Gauss-Kruger zone 30 [5422] Pulkovo 1942 / Gauss-Kruger zone 31 [5423] Pulkovo 1942 / Gauss-Kruger zone 32 [5424] Pulkovo 1942 / Gauss-Kruger zone 4 [5425] Pulkovo 1942 / Gauss-Kruger zone 5 [5426] Pulkovo 1942 / Gauss-Kruger zone 6 [5427] Pulkovo 1942 / Gauss-Kruger zone 7 [5428] Pulkovo 1942 / Gauss-Kruger zone 8 [5429] Pulkovo 1942 / Gauss-Kruger zone 9 [5430] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 10 [5431] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 [5432] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3 (E-N) [5433] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 [5434] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4 (E-N) [5435] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 [5436] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5 (E-N) [5437] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 6 [5438] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 7 [5439] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 8 [5440] Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 9 [5441] Pulkovo 1942(58) / GUGiK-80 [5442] Pulkovo 1942(58) / Gauss-Kruger zone 2 [5443] Pulkovo 1942(58) / Gauss-Kruger zone 2 (E-N) [5444] Pulkovo 1942(58) / Gauss-Kruger zone 3 [5445] Pulkovo 1942(58) / Gauss-Kruger zone 3 (E-N) [5446] Pulkovo 1942(58) / Gauss-Kruger zone 4 [5447] Pulkovo 1942(58) / Gauss-Kruger zone 5 [5448] Pulkovo 1942(58) / Poland zone I [5449] Pulkovo 1942(58) / Poland zone I (deprecated) [5450] Pulkovo 1942(58) / Poland zone II [5451] Pulkovo 1942(58) / Poland zone III [5452] Pulkovo 1942(58) / Poland zone IV [5453] Pulkovo 1942(58) / Poland zone V [5454] Pulkovo 1942(58) / Stereo70 [5455] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 [5456] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3 (E-N) [5457] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 [5458] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4 (E-N) [5459] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 [5460] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5 (E-N) [5461] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 6 [5462] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 [5463] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7 (deprecated) [5464] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 [5465] Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8 (deprecated) [5466] Pulkovo 1942(83) / Gauss Kruger zone 3 (deprecated) [5467] Pulkovo 1942(83) / Gauss Kruger zone 4 (deprecated) [5468] Pulkovo 1942(83) / Gauss Kruger zone 5 (deprecated) [5469] Pulkovo 1942(83) / Gauss-Kruger zone 2 [5470] Pulkovo 1942(83) / Gauss-Kruger zone 2 (E-N) [5471] Pulkovo 1942(83) / Gauss-Kruger zone 3 [5472] Pulkovo 1942(83) / Gauss-Kruger zone 3 (E-N) [5473] Pulkovo 1942(83) / Gauss-Kruger zone 4 [5474] Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E [5475] Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E [5476] Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E [5477] Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E [5478] Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E [5479] Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E [5480] Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E [5481] Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E [5482] Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E [5483] Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E [5484] Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E [5485] Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E [5486] Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E [5487] Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E [5488] Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E [5489] Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E [5490] Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E [5491] Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E [5492] Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E [5493] Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E [5494] Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E [5495] Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E [5496] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E [5497] Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W [5498] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E [5499] Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W [5500] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E [5501] Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W [5502] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E [5503] Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W [5504] Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E [5505] Pulkovo 1995 / 3-degree Gauss-Kruger CM 18E (deprecated) [5506] Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E [5507] Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E [5508] Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E [5509] Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E [5510] Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E [5511] Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E [5512] Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E [5513] Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E [5514] Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E [5515] Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E [5516] Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E [5517] Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E [5518] Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E [5519] Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E [5520] Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E [5521] Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E [5522] Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E [5523] Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E [5524] Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E [5525] Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E [5526] Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E [5527] Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E [5528] Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E [5529] Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E [5530] Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E [5531] Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E [5532] Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E [5533] Pulkovo 1995 / 3-degree Gauss-Kruger zone 10 [5534] Pulkovo 1995 / 3-degree Gauss-Kruger zone 11 [5535] Pulkovo 1995 / 3-degree Gauss-Kruger zone 12 [5536] Pulkovo 1995 / 3-degree Gauss-Kruger zone 13 [5537] Pulkovo 1995 / 3-degree Gauss-Kruger zone 14 [5538] Pulkovo 1995 / 3-degree Gauss-Kruger zone 15 [5539] Pulkovo 1995 / 3-degree Gauss-Kruger zone 16 [5540] Pulkovo 1995 / 3-degree Gauss-Kruger zone 17 [5541] Pulkovo 1995 / 3-degree Gauss-Kruger zone 18 [5542] Pulkovo 1995 / 3-degree Gauss-Kruger zone 19 [5543] Pulkovo 1995 / 3-degree Gauss-Kruger zone 20 [5544] Pulkovo 1995 / 3-degree Gauss-Kruger zone 21 [5545] Pulkovo 1995 / 3-degree Gauss-Kruger zone 22 [5546] Pulkovo 1995 / 3-degree Gauss-Kruger zone 23 [5547] Pulkovo 1995 / 3-degree Gauss-Kruger zone 24 [5548] Pulkovo 1995 / 3-degree Gauss-Kruger zone 25 [5549] Pulkovo 1995 / 3-degree Gauss-Kruger zone 26 [5550] Pulkovo 1995 / 3-degree Gauss-Kruger zone 27 [5551] Pulkovo 1995 / 3-degree Gauss-Kruger zone 28 [5552] Pulkovo 1995 / 3-degree Gauss-Kruger zone 29 [5553] Pulkovo 1995 / 3-degree Gauss-Kruger zone 30 [5554] Pulkovo 1995 / 3-degree Gauss-Kruger zone 31 [5555] Pulkovo 1995 / 3-degree Gauss-Kruger zone 32 [5556] Pulkovo 1995 / 3-degree Gauss-Kruger zone 33 [5557] Pulkovo 1995 / 3-degree Gauss-Kruger zone 34 [5558] Pulkovo 1995 / 3-degree Gauss-Kruger zone 35 [5559] Pulkovo 1995 / 3-degree Gauss-Kruger zone 36 [5560] Pulkovo 1995 / 3-degree Gauss-Kruger zone 37 [5561] Pulkovo 1995 / 3-degree Gauss-Kruger zone 38 [5562] Pulkovo 1995 / 3-degree Gauss-Kruger zone 39 [5563] Pulkovo 1995 / 3-degree Gauss-Kruger zone 40 [5564] Pulkovo 1995 / 3-degree Gauss-Kruger zone 41 [5565] Pulkovo 1995 / 3-degree Gauss-Kruger zone 42 [5566] Pulkovo 1995 / 3-degree Gauss-Kruger zone 43 [5567] Pulkovo 1995 / 3-degree Gauss-Kruger zone 44 [5568] Pulkovo 1995 / 3-degree Gauss-Kruger zone 45 [5569] Pulkovo 1995 / 3-degree Gauss-Kruger zone 46 [5570] Pulkovo 1995 / 3-degree Gauss-Kruger zone 47 [5571] Pulkovo 1995 / 3-degree Gauss-Kruger zone 48 [5572] Pulkovo 1995 / 3-degree Gauss-Kruger zone 49 [5573] Pulkovo 1995 / 3-degree Gauss-Kruger zone 50 [5574] Pulkovo 1995 / 3-degree Gauss-Kruger zone 51 [5575] Pulkovo 1995 / 3-degree Gauss-Kruger zone 52 [5576] Pulkovo 1995 / 3-degree Gauss-Kruger zone 53 [5577] Pulkovo 1995 / 3-degree Gauss-Kruger zone 54 [5578] Pulkovo 1995 / 3-degree Gauss-Kruger zone 55 [5579] Pulkovo 1995 / 3-degree Gauss-Kruger zone 56 [5580] Pulkovo 1995 / 3-degree Gauss-Kruger zone 57 [5581] Pulkovo 1995 / 3-degree Gauss-Kruger zone 58 [5582] Pulkovo 1995 / 3-degree Gauss-Kruger zone 59 [5583] Pulkovo 1995 / 3-degree Gauss-Kruger zone 6 (deprecated) [5584] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 [5585] Pulkovo 1995 / 3-degree Gauss-Kruger zone 60 (deprecated) [5586] Pulkovo 1995 / 3-degree Gauss-Kruger zone 61 [5587] Pulkovo 1995 / 3-degree Gauss-Kruger zone 62 [5588] Pulkovo 1995 / 3-degree Gauss-Kruger zone 63 [5589] Pulkovo 1995 / 3-degree Gauss-Kruger zone 64 [5590] Pulkovo 1995 / 3-degree Gauss-Kruger zone 7 [5591] Pulkovo 1995 / 3-degree Gauss-Kruger zone 8 [5592] Pulkovo 1995 / 3-degree Gauss-Kruger zone 9 [5593] Pulkovo 1995 / Gauss-Kruger 10N (deprecated) [5594] Pulkovo 1995 / Gauss-Kruger 11N (deprecated) [5595] Pulkovo 1995 / Gauss-Kruger 12N (deprecated) [5596] Pulkovo 1995 / Gauss-Kruger 13N (deprecated) [5597] Pulkovo 1995 / Gauss-Kruger 14N (deprecated) [5598] Pulkovo 1995 / Gauss-Kruger 15N (deprecated) [5599] Pulkovo 1995 / Gauss-Kruger 16N (deprecated) [5600] Pulkovo 1995 / Gauss-Kruger 17N (deprecated) [5601] Pulkovo 1995 / Gauss-Kruger 18N (deprecated) [5602] Pulkovo 1995 / Gauss-Kruger 19N (deprecated) [5603] Pulkovo 1995 / Gauss-Kruger 20N (deprecated) [5604] Pulkovo 1995 / Gauss-Kruger 21N (deprecated) [5605] Pulkovo 1995 / Gauss-Kruger 22N (deprecated) [5606] Pulkovo 1995 / Gauss-Kruger 23N (deprecated) [5607] Pulkovo 1995 / Gauss-Kruger 24N (deprecated) [5608] Pulkovo 1995 / Gauss-Kruger 25N (deprecated) [5609] Pulkovo 1995 / Gauss-Kruger 26N (deprecated) [5610] Pulkovo 1995 / Gauss-Kruger 27N (deprecated) [5611] Pulkovo 1995 / Gauss-Kruger 28N (deprecated) [5612] Pulkovo 1995 / Gauss-Kruger 29N (deprecated) [5613] Pulkovo 1995 / Gauss-Kruger 30N (deprecated) [5614] Pulkovo 1995 / Gauss-Kruger 31N (deprecated) [5615] Pulkovo 1995 / Gauss-Kruger 32N (deprecated) [5616] Pulkovo 1995 / Gauss-Kruger 4N (deprecated) [5617] Pulkovo 1995 / Gauss-Kruger 5N (deprecated) [5618] Pulkovo 1995 / Gauss-Kruger 6N (deprecated) [5619] Pulkovo 1995 / Gauss-Kruger 7N (deprecated) [5620] Pulkovo 1995 / Gauss-Kruger 8N (deprecated) [5621] Pulkovo 1995 / Gauss-Kruger 9N (deprecated) [5622] Pulkovo 1995 / Gauss-Kruger CM 105E [5623] Pulkovo 1995 / Gauss-Kruger CM 111E [5624] Pulkovo 1995 / Gauss-Kruger CM 117E [5625] Pulkovo 1995 / Gauss-Kruger CM 123E [5626] Pulkovo 1995 / Gauss-Kruger CM 129E [5627] Pulkovo 1995 / Gauss-Kruger CM 135E [5628] Pulkovo 1995 / Gauss-Kruger CM 141E [5629] Pulkovo 1995 / Gauss-Kruger CM 147E [5630] Pulkovo 1995 / Gauss-Kruger CM 153E [5631] Pulkovo 1995 / Gauss-Kruger CM 159E [5632] Pulkovo 1995 / Gauss-Kruger CM 165E [5633] Pulkovo 1995 / Gauss-Kruger CM 171E [5634] Pulkovo 1995 / Gauss-Kruger CM 171W [5635] Pulkovo 1995 / Gauss-Kruger CM 177E [5636] Pulkovo 1995 / Gauss-Kruger CM 177W [5637] Pulkovo 1995 / Gauss-Kruger CM 21E [5638] Pulkovo 1995 / Gauss-Kruger CM 27E [5639] Pulkovo 1995 / Gauss-Kruger CM 33E [5640] Pulkovo 1995 / Gauss-Kruger CM 39E [5641] Pulkovo 1995 / Gauss-Kruger CM 45E [5642] Pulkovo 1995 / Gauss-Kruger CM 51E [5643] Pulkovo 1995 / Gauss-Kruger CM 57E [5644] Pulkovo 1995 / Gauss-Kruger CM 63E [5645] Pulkovo 1995 / Gauss-Kruger CM 69E [5646] Pulkovo 1995 / Gauss-Kruger CM 75E [5647] Pulkovo 1995 / Gauss-Kruger CM 81E [5648] Pulkovo 1995 / Gauss-Kruger CM 87E [5649] Pulkovo 1995 / Gauss-Kruger CM 93E [5650] Pulkovo 1995 / Gauss-Kruger CM 99E [5651] Pulkovo 1995 / Gauss-Kruger zone 10 [5652] Pulkovo 1995 / Gauss-Kruger zone 11 [5653] Pulkovo 1995 / Gauss-Kruger zone 12 [5654] Pulkovo 1995 / Gauss-Kruger zone 13 [5655] Pulkovo 1995 / Gauss-Kruger zone 14 [5656] Pulkovo 1995 / Gauss-Kruger zone 15 [5657] Pulkovo 1995 / Gauss-Kruger zone 16 [5658] Pulkovo 1995 / Gauss-Kruger zone 17 [5659] Pulkovo 1995 / Gauss-Kruger zone 18 [5660] Pulkovo 1995 / Gauss-Kruger zone 19 [5661] Pulkovo 1995 / Gauss-Kruger zone 20 [5662] Pulkovo 1995 / Gauss-Kruger zone 21 [5663] Pulkovo 1995 / Gauss-Kruger zone 22 [5664] Pulkovo 1995 / Gauss-Kruger zone 23 [5665] Pulkovo 1995 / Gauss-Kruger zone 24 [5666] Pulkovo 1995 / Gauss-Kruger zone 25 [5667] Pulkovo 1995 / Gauss-Kruger zone 26 [5668] Pulkovo 1995 / Gauss-Kruger zone 27 [5669] Pulkovo 1995 / Gauss-Kruger zone 28 [5670] Pulkovo 1995 / Gauss-Kruger zone 29 [5671] Pulkovo 1995 / Gauss-Kruger zone 30 [5672] Pulkovo 1995 / Gauss-Kruger zone 31 [5673] Pulkovo 1995 / Gauss-Kruger zone 32 [5674] Pulkovo 1995 / Gauss-Kruger zone 4 [5675] Pulkovo 1995 / Gauss-Kruger zone 5 [5676] Pulkovo 1995 / Gauss-Kruger zone 6 [5677] Pulkovo 1995 / Gauss-Kruger zone 7 [5678] Pulkovo 1995 / Gauss-Kruger zone 8 [5679] Pulkovo 1995 / Gauss-Kruger zone 9 [5680] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_7 (deprecated) [5681] Pulkovo_1942_Adj_1983_3_Degree_GK_Zone_8 (deprecated) [5682] Pulkovo_1995_GK_Zone_2 [5683] Pulkovo_1995_GK_Zone_2N [5684] Pulkovo_1995_GK_Zone_3 [5685] Pulkovo_1995_GK_Zone_3N [5686] QND95 / Qatar National Grid [5687] QND_1995_UTM_39N [5688] Qatar 1948 / Qatar Grid [5689] Qatar 1974 / Qatar National Grid [5690] Qornoq 1927 / UTM zone 22N [5691] Qornoq 1927 / UTM zone 23N [5692] RD/83 / 3-degree Gauss-Kruger zone 4 [5693] RD/83 / 3-degree Gauss-Kruger zone 4 (E-N) [5694] RD/83 / 3-degree Gauss-Kruger zone 5 [5695] RD/83 / 3-degree Gauss-Kruger zone 5 (E-N) [5696] RDN2008 / Italy zone (E-N) [5697] RDN2008 / Italy zone (N-E) [5698] RDN2008 / UTM zone 32N [5699] RDN2008 / UTM zone 32N (N-E) [5700] RDN2008 / UTM zone 33N [5701] RDN2008 / UTM zone 33N (N-E) [5702] RDN2008 / UTM zone 34N [5703] RDN2008 / UTM zone 34N (N-E) [5704] RDN2008 / Zone 12 (E-N) [5705] RDN2008 / Zone 12 (N-E) [5706] REGCAN95 / LAEA Europe [5707] REGCAN95 / LCC Europe [5708] REGCAN95 / UTM zone 27N [5709] REGCAN95 / UTM zone 28N [5710] REGCAN95_UTM_Zone_27N (deprecated) [5711] REGCAN95_UTM_Zone_28N (deprecated) [5712] REGVEN / UTM zone 18N [5713] REGVEN / UTM zone 19N [5714] REGVEN / UTM zone 20N [5715] RGAF09 / UTM zone 20N [5716] RGF93 / CC42 [5717] RGF93 / CC43 [5718] RGF93 / CC44 [5719] RGF93 / CC45 [5720] RGF93 / CC46 [5721] RGF93 / CC47 [5722] RGF93 / CC48 [5723] RGF93 / CC49 [5724] RGF93 / CC50 [5725] RGF93 / Lambert-93 [5726] RGFG95 / UTM zone 21N [5727] RGFG95 / UTM zone 22N [5728] RGF_1993_Lambert_93 (deprecated) [5729] RGM04 / UTM zone 38S [5730] RGNC 1991 / Lambert New Caledonia (deprecated) [5731] RGNC91-93 / Lambert New Caledonia [5732] RGNC91-93 / UTM zone 57S [5733] RGNC91-93 / UTM zone 58S [5734] RGNC91-93 / UTM zone 59S [5735] RGPF / UTM zone 5S [5736] RGPF / UTM zone 6S [5737] RGPF / UTM zone 7S [5738] RGPF / UTM zone 8S [5739] RGR92 / UTM zone 39S [5740] RGR92 / UTM zone 40S [5741] RGRDC 2005 / Congo TM zone 12 [5742] RGRDC 2005 / Congo TM zone 14 [5743] RGRDC 2005 / Congo TM zone 16 [5744] RGRDC 2005 / Congo TM zone 18 [5745] RGRDC 2005 / Congo TM zone 20 [5746] RGRDC 2005 / Congo TM zone 22 [5747] RGRDC 2005 / Congo TM zone 24 [5748] RGRDC 2005 / Congo TM zone 26 [5749] RGRDC 2005 / Congo TM zone 28 [5750] RGRDC 2005 / Congo TM zone 30 [5751] RGRDC 2005 / UTM zone 33S [5752] RGRDC 2005 / UTM zone 34S [5753] RGRDC 2005 / UTM zone 35S [5754] RGRDC_2005_Congo_TM_Zone_12 (deprecated) [5755] RGRDC_2005_Congo_TM_Zone_14 (deprecated) [5756] RGRDC_2005_Congo_TM_Zone_16 (deprecated) [5757] RGRDC_2005_Congo_TM_Zone_18 (deprecated) [5758] RGRDC_2005_Congo_TM_Zone_20 (deprecated) [5759] RGRDC_2005_Congo_TM_Zone_22 (deprecated) [5760] RGRDC_2005_Congo_TM_Zone_24 (deprecated) [5761] RGRDC_2005_Congo_TM_Zone_26 (deprecated) [5762] RGRDC_2005_Congo_TM_Zone_28 (deprecated) [5763] RGRDC_2005_UTM_Zone_33S (deprecated) [5764] RGRDC_2005_UTM_Zone_34S (deprecated) [5765] RGRDC_2005_UTM_Zone_35S (deprecated) [5766] RGSPM06 / UTM zone 21N [5767] RGTAAF07 / UTM zone 37S [5768] RGTAAF07 / UTM zone 38S [5769] RGTAAF07 / UTM zone 39S [5770] RGTAAF07 / UTM zone 40S [5771] RGTAAF07 / UTM zone 41S [5772] RGTAAF07 / UTM zone 42S [5773] RGTAAF07 / UTM zone 43S [5774] RGTAAF07 / UTM zone 44S [5775] RGTAAF07 / UTM zone 53S [5776] RGTAAF07 / UTM zone 54S [5777] RGWF96 / UTM zone 1S [5778] RRAF 1991 / UTM zone 20N [5779] RRAF 1991 / UTM zone 20N (deprecated) [5780] RSRGD2000 / BCLC2000 [5781] RSRGD2000 / DGLC2000 [5782] RSRGD2000 / MSLC2000 [5783] RSRGD2000 / PCLC2000 [5784] RSRGD2000 / RSPS2000 [5785] RT38 0 gon [5786] RT38 2.5 gon O [5787] RT38 2.5 gon V [5788] RT38 2.5 gon W (deprecated) [5789] RT38 5 gon O [5790] RT38 5 gon V [5791] RT38 7.5 gon V [5792] RT90 0 gon [5793] RT90 2.5 gon O [5794] RT90 2.5 gon V [5795] RT90 2.5 gon W (deprecated) [5796] RT90 5 gon O [5797] RT90 5 gon V [5798] RT90 7.5 gon V [5799] Rassadiran / Nakhl e Taqi [5800] Reunion 1947 / TM Reunion [5801] Reunion 1947 / TM Reunion (deprecated) [5802] Roma_1940_Gauss_Boaga_Est [5803] Roma_1940_Gauss_Boaga_Ovest [5804] S-JTSK (Ferro) / Krovak [5805] S-JTSK (Ferro) / Krovak East North [5806] S-JTSK / Krovak [5807] S-JTSK / Krovak East North [5808] S-JTSK [JTSK03] / Krovak [5809] S-JTSK [JTSK03] / Krovak East North [5810] S-JTSK_Ferro_Krovak_East_North (deprecated) [5811] S-JTSK_Krovak (deprecated) [5812] S-JTSK_Krovak_East_North (deprecated) [5813] SAD69 / Brazil Polyconic [5814] SAD69 / Brazil Polyconic (deprecated) [5815] SAD69 / UTM zone 17N [5816] SAD69 / UTM zone 17S [5817] SAD69 / UTM zone 17S (deprecated) [5818] SAD69 / UTM zone 18N [5819] SAD69 / UTM zone 18N (deprecated) [5820] SAD69 / UTM zone 18S [5821] SAD69 / UTM zone 18S (deprecated) [5822] SAD69 / UTM zone 19N [5823] SAD69 / UTM zone 19N (deprecated) [5824] SAD69 / UTM zone 19S [5825] SAD69 / UTM zone 19S (deprecated) [5826] SAD69 / UTM zone 20N [5827] SAD69 / UTM zone 20N (deprecated) [5828] SAD69 / UTM zone 20S [5829] SAD69 / UTM zone 20S (deprecated) [5830] SAD69 / UTM zone 21N [5831] SAD69 / UTM zone 21N (deprecated) [5832] SAD69 / UTM zone 21S [5833] SAD69 / UTM zone 21S (deprecated) [5834] SAD69 / UTM zone 22N [5835] SAD69 / UTM zone 22N (deprecated) [5836] SAD69 / UTM zone 22S [5837] SAD69 / UTM zone 22S (deprecated) [5838] SAD69 / UTM zone 23S [5839] SAD69 / UTM zone 23S (deprecated) [5840] SAD69 / UTM zone 24S [5841] SAD69 / UTM zone 24S (deprecated) [5842] SAD69 / UTM zone 25S [5843] SAD69 / UTM zone 25S (deprecated) [5844] SAD69(96) / Brazil Polyconic [5845] SAD69(96) / UTM zone 18S [5846] SAD69(96) / UTM zone 19S [5847] SAD69(96) / UTM zone 20S [5848] SAD69(96) / UTM zone 21S [5849] SAD69(96) / UTM zone 22S [5850] SAD69(96) / UTM zone 22S (deprecated) [5851] SAD69(96) / UTM zone 23S [5852] SAD69(96) / UTM zone 24S [5853] SAD69(96) / UTM zone 25S [5854] SHMG2015 [5855] SIRGAS 1995 / UTM zone 17N [5856] SIRGAS 1995 / UTM zone 17S [5857] SIRGAS 1995 / UTM zone 18N [5858] SIRGAS 1995 / UTM zone 18S [5859] SIRGAS 1995 / UTM zone 19N [5860] SIRGAS 1995 / UTM zone 19S [5861] SIRGAS 1995 / UTM zone 20N [5862] SIRGAS 1995 / UTM zone 20S [5863] SIRGAS 1995 / UTM zone 21N [5864] SIRGAS 1995 / UTM zone 21S [5865] SIRGAS 1995 / UTM zone 22N [5866] SIRGAS 1995 / UTM zone 22S [5867] SIRGAS 1995 / UTM zone 23S [5868] SIRGAS 1995 / UTM zone 24S [5869] SIRGAS 1995 / UTM zone 25S [5870] SIRGAS 2000 / Brazil Mercator [5871] SIRGAS 2000 / Brazil Polyconic [5872] SIRGAS 2000 / UTM zone 11N [5873] SIRGAS 2000 / UTM zone 12N [5874] SIRGAS 2000 / UTM zone 13N [5875] SIRGAS 2000 / UTM zone 14N [5876] SIRGAS 2000 / UTM zone 15N [5877] SIRGAS 2000 / UTM zone 16N [5878] SIRGAS 2000 / UTM zone 17N [5879] SIRGAS 2000 / UTM zone 17S [5880] SIRGAS 2000 / UTM zone 18N [5881] SIRGAS 2000 / UTM zone 18S [5882] SIRGAS 2000 / UTM zone 19N [5883] SIRGAS 2000 / UTM zone 19S [5884] SIRGAS 2000 / UTM zone 20N [5885] SIRGAS 2000 / UTM zone 20S [5886] SIRGAS 2000 / UTM zone 21N [5887] SIRGAS 2000 / UTM zone 21S [5888] SIRGAS 2000 / UTM zone 22N [5889] SIRGAS 2000 / UTM zone 22S [5890] SIRGAS 2000 / UTM zone 23N [5891] SIRGAS 2000 / UTM zone 23S [5892] SIRGAS 2000 / UTM zone 24N [5893] SIRGAS 2000 / UTM zone 24S [5894] SIRGAS 2000 / UTM zone 25S [5895] SIRGAS 2000 / UTM zone 26S [5896] SIRGAS-Chile / UTM zone 18S [5897] SIRGAS-Chile / UTM zone 19S [5898] SIRGAS-ROU98 / UTM zone 21S [5899] SIRGAS-ROU98 / UTM zone 22S [5900] SIRGAS_UTM_Zone_17N (deprecated) [5901] SIRGAS_UTM_Zone_18N (deprecated) [5902] SIRGAS_UTM_Zone_19N (deprecated) [5903] SIRGAS_UTM_Zone_20N (deprecated) [5904] SIRGAS_UTM_Zone_21N (deprecated) [5905] SIRGAS_UTM_Zone_22N (deprecated) [5906] SLD99 / Sri Lanka Grid 1999 [5907] SRB_ETRS89 / UTM zone 34N [5908] ST71 Belep / UTM zone 58S [5909] ST74 [5910] ST84 Ile des Pins / UTM zone 58S [5911] ST87 Ouvea / UTM zone 58S [5912] ST87 Ouvea / UTM zone 58S (deprecated) [5913] SVY21 / Singapore TM [5914] SWEREF99 / RT90 0 gon emulation [5915] SWEREF99 / RT90 2.5 gon O emulation [5916] SWEREF99 / RT90 2.5 gon V emulation [5917] SWEREF99 / RT90 5 gon O emulation [5918] SWEREF99 / RT90 5 gon V emulation [5919] SWEREF99 / RT90 7.5 gon V emulation [5920] SWEREF99 12 00 [5921] SWEREF99 13 30 [5922] SWEREF99 14 15 [5923] SWEREF99 15 00 [5924] SWEREF99 15 45 [5925] SWEREF99 16 30 [5926] SWEREF99 17 15 [5927] SWEREF99 18 00 [5928] SWEREF99 18 45 [5929] SWEREF99 20 15 [5930] SWEREF99 21 45 [5931] SWEREF99 23 15 [5932] SWEREF99 TM [5933] Sahara_Degree [5934] Saint Pierre et Miquelon 1950 / UTM zone 21N [5935] Saipan_Az_Eq_1969 [5936] Samboja / UTM zone 50S (deprecated) [5937] Sao_Braz_UTM_Zone_26N [5938] Sapper Hill 1943 / UTM zone 20S [5939] Sapper Hill 1943 / UTM zone 21S [5940] Schwarzeck / Lo22/11 [5941] Schwarzeck / Lo22/13 [5942] Schwarzeck / Lo22/15 [5943] Schwarzeck / Lo22/17 [5944] Schwarzeck / Lo22/19 [5945] Schwarzeck / Lo22/21 [5946] Schwarzeck / Lo22/23 [5947] Schwarzeck / Lo22/25 [5948] Schwarzeck / UTM zone 33S [5949] Segara (Jakarta) / NEIEZ [5950] Segara (Jakarta) / NEIEZ (deprecated) [5951] Segara / NEIEZ [5952] Segara / UTM zone 50S [5953] Selvagem Grande / UTM zone 28N [5954] Selvagem_Grande_1938_UTM_Zone_28N (deprecated) [5955] Sibun Gorge 1922 / Colony Grid [5956] Sibun Gorge 1922 / Colony Grid (deprecated) [5957] Sierra Leone 1924 / New Colony Grid [5958] Sierra Leone 1924 / New War Office Grid [5959] Sierra Leone 1968 / UTM zone 28N [5960] Sierra Leone 1968 / UTM zone 29N [5961] Sister Islands National Grid 1961 [5962] Slovenia 1996 / Slovene National Grid [5963] Slovenia 1996 / UTM zone 33N [5964] South East Island 1943 / UTM zone 40N [5965] South Yemen / Gauss Kruger zone 8 (deprecated) [5966] South Yemen / Gauss Kruger zone 9 (deprecated) [5967] South Yemen / Gauss-Kruger zone 8 [5968] South Yemen / Gauss-Kruger zone 9 [5969] South_America_Albers_Equal_Area_Conic [5970] South_America_Equidistant_Conic [5971] South_America_Lambert_Conformal_Conic [5972] South_Pole_Azimuthal_Equidistant [5973] South_Pole_Gnomonic [5974] South_Pole_Lambert_Azimuthal_Equal_Area [5975] South_Pole_Orthographic [5976] South_Pole_Stereographic [5977] Sphere_Aitoff [5978] Sphere_Azimuthal_Equidistant [5979] Sphere_Bonne [5980] Sphere_Cassini [5981] Sphere_Craster_Parabolic [5982] Sphere_Cylindrical_Equal_Area [5983] Sphere_Eckert_I [5984] Sphere_Eckert_II [5985] Sphere_Eckert_III [5986] Sphere_Eckert_IV [5987] Sphere_Eckert_V [5988] Sphere_Eckert_VI [5989] Sphere_Equidistant_Conic [5990] Sphere_Equidistant_Cylindrical [5991] Sphere_Gall_Stereographic [5992] Sphere_Hotine [5993] Sphere_Loximuthal [5994] Sphere_Mercator [5995] Sphere_Miller_Cylindrical [5996] Sphere_Mollweide [5997] Sphere_Plate_Carree [5998] Sphere_Polyconic [5999] Sphere_Quartic_Authalic [6000] Sphere_Robinson [6001] Sphere_Sinusoidal [6002] Sphere_Stereographic [6003] Sphere_Two_Point_Equidistant [6004] Sphere_Van_der_Grinten_I [6005] Sphere_Wagner_IV [6006] Sphere_Wagner_V [6007] Sphere_Wagner_VII [6008] Sphere_Winkel_I [6009] Sphere_Winkel_II [6010] Sphere_Winkel_Tripel_NGS [6011] St. Helena Tritan / SHLG(Tritan) [6012] St. Helena Tritan / UTM zone 30S [6013] St. Kitts 1955 / British West Indies Grid [6014] St. Lucia 1955 / British West Indies Grid [6015] St. Stephen Grid (Ferro) [6016] St. Vincent 45 / British West Indies Grid [6017] Sud_Algerie (deprecated) [6018] Sud_Algerie_Ancienne_Degree [6019] Sud_Algerie_Degree (deprecated) [6020] Sud_Maroc_Degree [6021] Sudan / UTM zone 35N (deprecated) [6022] Sudan / UTM zone 36N (deprecated) [6023] TC(1948) / UTM zone 39N [6024] TC(1948) / UTM zone 40N [6025] TGD2005 / Tonga Map Grid [6026] TM65 / Irish Grid [6027] TM65 / Irish National Grid (deprecated) [6028] TM75 / Irish Grid [6029] TUREF / 3-degree Gauss-Kruger zone 10 [6030] TUREF / 3-degree Gauss-Kruger zone 11 [6031] TUREF / 3-degree Gauss-Kruger zone 12 [6032] TUREF / 3-degree Gauss-Kruger zone 13 [6033] TUREF / 3-degree Gauss-Kruger zone 14 [6034] TUREF / 3-degree Gauss-Kruger zone 15 [6035] TUREF / 3-degree Gauss-Kruger zone 9 [6036] TUREF / LAEA Europe [6037] TUREF / LCC Europe [6038] TUREF / TM27 [6039] TUREF / TM30 [6040] TUREF / TM33 [6041] TUREF / TM36 [6042] TUREF / TM39 [6043] TUREF / TM42 [6044] TUREF / TM45 [6045] TWD67 / TM2 zone 119 [6046] TWD67 / TM2 zone 121 [6047] TWD97 / TM2 zone 119 [6048] TWD97 / TM2 zone 121 [6049] TWD_1967_TM_Penghu (deprecated) [6050] TWD_1967_TM_Taiwan (deprecated) [6051] TWD_1997_TM_Penghu (deprecated) [6052] TWD_1997_TM_Taiwan (deprecated) [6053] Tahaa 54 / UTM zone 5S [6054] Tahiti 52 / UTM zone 6S [6055] Tahiti 79 / UTM zone 6S [6056] Tananarive (Paris) / Laborde Grid [6057] Tananarive (Paris) / Laborde Grid (deprecated) [6058] Tananarive (Paris) / Laborde Grid approximation [6059] Tananarive / Laborde Grid [6060] Tananarive / UTM zone 38S [6061] Tananarive / UTM zone 39S [6062] Tananarive_1925_Laborde_Grid [6063] Tete / UTM zone 36S [6064] Tete / UTM zone 37S [6065] The_World_From_Space [6066] Timbalai 1948 / RSO Borneo (ch) [6067] Timbalai 1948 / RSO Borneo (ftSe) [6068] Timbalai 1948 / RSO Borneo (m) [6069] Timbalai 1948 / UTM zone 49N [6070] Timbalai 1948 / UTM zone 50N [6071] Tokyo / Japan Plane Rectangular CS I [6072] Tokyo / Japan Plane Rectangular CS II [6073] Tokyo / Japan Plane Rectangular CS III [6074] Tokyo / Japan Plane Rectangular CS IV [6075] Tokyo / Japan Plane Rectangular CS IX [6076] Tokyo / Japan Plane Rectangular CS V [6077] Tokyo / Japan Plane Rectangular CS VI [6078] Tokyo / Japan Plane Rectangular CS VII [6079] Tokyo / Japan Plane Rectangular CS VIII [6080] Tokyo / Japan Plane Rectangular CS X [6081] Tokyo / Japan Plane Rectangular CS XI [6082] Tokyo / Japan Plane Rectangular CS XII [6083] Tokyo / Japan Plane Rectangular CS XIII [6084] Tokyo / Japan Plane Rectangular CS XIV [6085] Tokyo / Japan Plane Rectangular CS XIX [6086] Tokyo / Japan Plane Rectangular CS XV [6087] Tokyo / Japan Plane Rectangular CS XVI [6088] Tokyo / Japan Plane Rectangular CS XVII [6089] Tokyo / Japan Plane Rectangular CS XVIII [6090] Tokyo / UTM zone 51N [6091] Tokyo / UTM zone 52N [6092] Tokyo / UTM zone 53N [6093] Tokyo / UTM zone 54N [6094] Tokyo / UTM zone 55N [6095] Tokyo 1892 / Korea Central Belt [6096] Tokyo 1892 / Korea East Belt [6097] Tokyo 1892 / Korea East Sea Belt [6098] Tokyo 1892 / Korea West Belt [6099] Tokyo_UTM_Zone_51N (deprecated) [6100] Tokyo_UTM_Zone_52N (deprecated) [6101] Tokyo_UTM_Zone_53N (deprecated) [6102] Tokyo_UTM_Zone_54N (deprecated) [6103] Tokyo_UTM_Zone_55N (deprecated) [6104] Tokyo_UTM_Zone_56N [6105] Trinidad 1903 / Trinidad Grid [6106] Trinidad 1903 / Trinidad Grid (ftCla) [6107] UCS-2000 / 3-degree Gauss-Kruger CM 21E (deprecated) [6108] UCS-2000 / 3-degree Gauss-Kruger CM 24E (deprecated) [6109] UCS-2000 / 3-degree Gauss-Kruger CM 27E (deprecated) [6110] UCS-2000 / 3-degree Gauss-Kruger CM 30E (deprecated) [6111] UCS-2000 / 3-degree Gauss-Kruger CM 33E (deprecated) [6112] UCS-2000 / 3-degree Gauss-Kruger CM 36E (deprecated) [6113] UCS-2000 / 3-degree Gauss-Kruger CM 39E (deprecated) [6114] UCS-2000 / 3-degree Gauss-Kruger zone 10 (deprecated) [6115] UCS-2000 / 3-degree Gauss-Kruger zone 11 (deprecated) [6116] UCS-2000 / 3-degree Gauss-Kruger zone 12 (deprecated) [6117] UCS-2000 / 3-degree Gauss-Kruger zone 13 (deprecated) [6118] UCS-2000 / 3-degree Gauss-Kruger zone 7 (deprecated) [6119] UCS-2000 / 3-degree Gauss-Kruger zone 8 (deprecated) [6120] UCS-2000 / 3-degree Gauss-Kruger zone 9 (deprecated) [6121] UCS-2000 / Gauss-Kruger CM 21E [6122] UCS-2000 / Gauss-Kruger CM 27E [6123] UCS-2000 / Gauss-Kruger CM 33E [6124] UCS-2000 / Gauss-Kruger CM 39E [6125] UCS-2000 / Gauss-Kruger zone 4 [6126] UCS-2000 / Gauss-Kruger zone 5 [6127] UCS-2000 / Gauss-Kruger zone 6 [6128] UCS-2000 / Gauss-Kruger zone 7 [6129] UCS-2000 / Ukraine TM zone 10 [6130] UCS-2000 / Ukraine TM zone 11 [6131] UCS-2000 / Ukraine TM zone 12 [6132] UCS-2000 / Ukraine TM zone 13 [6133] UCS-2000 / Ukraine TM zone 7 [6134] UCS-2000 / Ukraine TM zone 8 [6135] UCS-2000 / Ukraine TM zone 9 [6136] US National Atlas Equal Area [6137] USA_Contiguous_Albers_Equal_Area_Conic [6138] USA_Contiguous_Albers_Equal_Area_Conic_USGS_version [6139] USA_Contiguous_Equidistant_Conic [6140] USA_Contiguous_Lambert_Conformal_Conic [6141] UWPP_1992 [6142] UWPP_2000_PAS_5 [6143] UWPP_2000_PAS_6 [6144] UWPP_2000_PAS_7 [6145] UWPP_2000_PAS_8 [6146] VN-2000 / TM-3 Da Nang zone [6147] VN-2000 / TM-3 Da Nang zone [6148] VN-2000 / TM-3 zone 481 [6149] VN-2000 / TM-3 zone 481 [6150] VN-2000 / TM-3 zone 482 [6151] VN-2000 / TM-3 zone 482 [6152] VN-2000 / TM-3 zone 491 [6153] VN-2000 / TM-3 zone 491 [6154] VN-2000 / UTM zone 48N [6155] VN-2000 / UTM zone 49N [6156] Viti Levu 1912 / Viti Levu Grid [6157] Voirol 1875 / Nord Algerie (ancienne) [6158] Voirol 1875 / Sud Algerie (ancienne) [6159] Voirol 1879 / Nord Algerie (ancienne) [6160] Voirol 1879 / Sud Algerie (ancienne) [6161] WGS 72 / UTM zone 10N [6162] WGS 72 / UTM zone 10S [6163] WGS 72 / UTM zone 11N [6164] WGS 72 / UTM zone 11S [6165] WGS 72 / UTM zone 12N [6166] WGS 72 / UTM zone 12S [6167] WGS 72 / UTM zone 13N [6168] WGS 72 / UTM zone 13S [6169] WGS 72 / UTM zone 14N [6170] WGS 72 / UTM zone 14S [6171] WGS 72 / UTM zone 15N [6172] WGS 72 / UTM zone 15S [6173] WGS 72 / UTM zone 16N [6174] WGS 72 / UTM zone 16S [6175] WGS 72 / UTM zone 17N [6176] WGS 72 / UTM zone 17S [6177] WGS 72 / UTM zone 18N [6178] WGS 72 / UTM zone 18S [6179] WGS 72 / UTM zone 19N [6180] WGS 72 / UTM zone 19S [6181] WGS 72 / UTM zone 1N [6182] WGS 72 / UTM zone 1S [6183] WGS 72 / UTM zone 20N [6184] WGS 72 / UTM zone 20S [6185] WGS 72 / UTM zone 21N [6186] WGS 72 / UTM zone 21S [6187] WGS 72 / UTM zone 22N [6188] WGS 72 / UTM zone 22S [6189] WGS 72 / UTM zone 23N [6190] WGS 72 / UTM zone 23S [6191] WGS 72 / UTM zone 24N [6192] WGS 72 / UTM zone 24S [6193] WGS 72 / UTM zone 25N [6194] WGS 72 / UTM zone 25S [6195] WGS 72 / UTM zone 26N [6196] WGS 72 / UTM zone 26S [6197] WGS 72 / UTM zone 27N [6198] WGS 72 / UTM zone 27S [6199] WGS 72 / UTM zone 28N [6200] WGS 72 / UTM zone 28S [6201] WGS 72 / UTM zone 29N [6202] WGS 72 / UTM zone 29S [6203] WGS 72 / UTM zone 2N [6204] WGS 72 / UTM zone 2S [6205] WGS 72 / UTM zone 30N [6206] WGS 72 / UTM zone 30S [6207] WGS 72 / UTM zone 31N [6208] WGS 72 / UTM zone 31S [6209] WGS 72 / UTM zone 32N [6210] WGS 72 / UTM zone 32S [6211] WGS 72 / UTM zone 33N [6212] WGS 72 / UTM zone 33S [6213] WGS 72 / UTM zone 34N [6214] WGS 72 / UTM zone 34S [6215] WGS 72 / UTM zone 35N [6216] WGS 72 / UTM zone 35S [6217] WGS 72 / UTM zone 36N [6218] WGS 72 / UTM zone 36S [6219] WGS 72 / UTM zone 37N [6220] WGS 72 / UTM zone 37S [6221] WGS 72 / UTM zone 38N [6222] WGS 72 / UTM zone 38S [6223] WGS 72 / UTM zone 39N [6224] WGS 72 / UTM zone 39S [6225] WGS 72 / UTM zone 3N [6226] WGS 72 / UTM zone 3S [6227] WGS 72 / UTM zone 40N [6228] WGS 72 / UTM zone 40S [6229] WGS 72 / UTM zone 41N [6230] WGS 72 / UTM zone 41S [6231] WGS 72 / UTM zone 42N [6232] WGS 72 / UTM zone 42S [6233] WGS 72 / UTM zone 43N [6234] WGS 72 / UTM zone 43S [6235] WGS 72 / UTM zone 44N [6236] WGS 72 / UTM zone 44S [6237] WGS 72 / UTM zone 45N [6238] WGS 72 / UTM zone 45S [6239] WGS 72 / UTM zone 46N [6240] WGS 72 / UTM zone 46S [6241] WGS 72 / UTM zone 47N [6242] WGS 72 / UTM zone 47S [6243] WGS 72 / UTM zone 48N [6244] WGS 72 / UTM zone 48S [6245] WGS 72 / UTM zone 49N [6246] WGS 72 / UTM zone 49S [6247] WGS 72 / UTM zone 4N [6248] WGS 72 / UTM zone 4S [6249] WGS 72 / UTM zone 50N [6250] WGS 72 / UTM zone 50S [6251] WGS 72 / UTM zone 51N [6252] WGS 72 / UTM zone 51S [6253] WGS 72 / UTM zone 52N [6254] WGS 72 / UTM zone 52S [6255] WGS 72 / UTM zone 53N [6256] WGS 72 / UTM zone 53S [6257] WGS 72 / UTM zone 54N [6258] WGS 72 / UTM zone 54S [6259] WGS 72 / UTM zone 55N [6260] WGS 72 / UTM zone 55S [6261] WGS 72 / UTM zone 56N [6262] WGS 72 / UTM zone 56S [6263] WGS 72 / UTM zone 57N [6264] WGS 72 / UTM zone 57S [6265] WGS 72 / UTM zone 58N [6266] WGS 72 / UTM zone 58S [6267] WGS 72 / UTM zone 59N [6268] WGS 72 / UTM zone 59S [6269] WGS 72 / UTM zone 5N [6270] WGS 72 / UTM zone 5S [6271] WGS 72 / UTM zone 60N [6272] WGS 72 / UTM zone 60S [6273] WGS 72 / UTM zone 6N [6274] WGS 72 / UTM zone 6S [6275] WGS 72 / UTM zone 7N [6276] WGS 72 / UTM zone 7S [6277] WGS 72 / UTM zone 8N [6278] WGS 72 / UTM zone 8S [6279] WGS 72 / UTM zone 9N [6280] WGS 72 / UTM zone 9S [6281] WGS 72BE / South China Sea Lambert [6282] WGS 72BE / TM 106 NE [6283] WGS 72BE / UTM zone 10N [6284] WGS 72BE / UTM zone 10S [6285] WGS 72BE / UTM zone 11N [6286] WGS 72BE / UTM zone 11S [6287] WGS 72BE / UTM zone 12N [6288] WGS 72BE / UTM zone 12S [6289] WGS 72BE / UTM zone 13N [6290] WGS 72BE / UTM zone 13S [6291] WGS 72BE / UTM zone 14N [6292] WGS 72BE / UTM zone 14S [6293] WGS 72BE / UTM zone 15N [6294] WGS 72BE / UTM zone 15S [6295] WGS 72BE / UTM zone 16N [6296] WGS 72BE / UTM zone 16S [6297] WGS 72BE / UTM zone 17N [6298] WGS 72BE / UTM zone 17S [6299] WGS 72BE / UTM zone 18N [6300] WGS 72BE / UTM zone 18S [6301] WGS 72BE / UTM zone 19N [6302] WGS 72BE / UTM zone 19S [6303] WGS 72BE / UTM zone 1N [6304] WGS 72BE / UTM zone 1S [6305] WGS 72BE / UTM zone 20N [6306] WGS 72BE / UTM zone 20S [6307] WGS 72BE / UTM zone 21N [6308] WGS 72BE / UTM zone 21S [6309] WGS 72BE / UTM zone 22N [6310] WGS 72BE / UTM zone 22S [6311] WGS 72BE / UTM zone 23N [6312] WGS 72BE / UTM zone 23S [6313] WGS 72BE / UTM zone 24N [6314] WGS 72BE / UTM zone 24S [6315] WGS 72BE / UTM zone 25N [6316] WGS 72BE / UTM zone 25S [6317] WGS 72BE / UTM zone 26N [6318] WGS 72BE / UTM zone 26S [6319] WGS 72BE / UTM zone 27N [6320] WGS 72BE / UTM zone 27S [6321] WGS 72BE / UTM zone 28N [6322] WGS 72BE / UTM zone 28S [6323] WGS 72BE / UTM zone 29N [6324] WGS 72BE / UTM zone 29S [6325] WGS 72BE / UTM zone 2N [6326] WGS 72BE / UTM zone 2S [6327] WGS 72BE / UTM zone 30N [6328] WGS 72BE / UTM zone 30S [6329] WGS 72BE / UTM zone 31N [6330] WGS 72BE / UTM zone 31S [6331] WGS 72BE / UTM zone 32N [6332] WGS 72BE / UTM zone 32S [6333] WGS 72BE / UTM zone 33N [6334] WGS 72BE / UTM zone 33S [6335] WGS 72BE / UTM zone 34N [6336] WGS 72BE / UTM zone 34S [6337] WGS 72BE / UTM zone 35N [6338] WGS 72BE / UTM zone 35S [6339] WGS 72BE / UTM zone 36N [6340] WGS 72BE / UTM zone 36S [6341] WGS 72BE / UTM zone 37N [6342] WGS 72BE / UTM zone 37S [6343] WGS 72BE / UTM zone 38N [6344] WGS 72BE / UTM zone 38S [6345] WGS 72BE / UTM zone 39N [6346] WGS 72BE / UTM zone 39S [6347] WGS 72BE / UTM zone 3N [6348] WGS 72BE / UTM zone 3S [6349] WGS 72BE / UTM zone 40N [6350] WGS 72BE / UTM zone 40S [6351] WGS 72BE / UTM zone 41N [6352] WGS 72BE / UTM zone 41S [6353] WGS 72BE / UTM zone 42N [6354] WGS 72BE / UTM zone 42S [6355] WGS 72BE / UTM zone 43N [6356] WGS 72BE / UTM zone 43S [6357] WGS 72BE / UTM zone 44N [6358] WGS 72BE / UTM zone 44S [6359] WGS 72BE / UTM zone 45N [6360] WGS 72BE / UTM zone 45S [6361] WGS 72BE / UTM zone 46N [6362] WGS 72BE / UTM zone 46S [6363] WGS 72BE / UTM zone 47N [6364] WGS 72BE / UTM zone 47S [6365] WGS 72BE / UTM zone 48N [6366] WGS 72BE / UTM zone 48S [6367] WGS 72BE / UTM zone 49N [6368] WGS 72BE / UTM zone 49S [6369] WGS 72BE / UTM zone 4N [6370] WGS 72BE / UTM zone 4S [6371] WGS 72BE / UTM zone 50N [6372] WGS 72BE / UTM zone 50S [6373] WGS 72BE / UTM zone 51N [6374] WGS 72BE / UTM zone 51S [6375] WGS 72BE / UTM zone 52N [6376] WGS 72BE / UTM zone 52S [6377] WGS 72BE / UTM zone 53N [6378] WGS 72BE / UTM zone 53S [6379] WGS 72BE / UTM zone 54N [6380] WGS 72BE / UTM zone 54S [6381] WGS 72BE / UTM zone 55N [6382] WGS 72BE / UTM zone 55S [6383] WGS 72BE / UTM zone 56N [6384] WGS 72BE / UTM zone 56S [6385] WGS 72BE / UTM zone 57N [6386] WGS 72BE / UTM zone 57S [6387] WGS 72BE / UTM zone 58N [6388] WGS 72BE / UTM zone 58S [6389] WGS 72BE / UTM zone 59N [6390] WGS 72BE / UTM zone 59S [6391] WGS 72BE / UTM zone 5N [6392] WGS 72BE / UTM zone 5S [6393] WGS 72BE / UTM zone 60N [6394] WGS 72BE / UTM zone 60S [6395] WGS 72BE / UTM zone 6N [6396] WGS 72BE / UTM zone 6S [6397] WGS 72BE / UTM zone 7N [6398] WGS 72BE / UTM zone 7S [6399] WGS 72BE / UTM zone 8N [6400] WGS 72BE / UTM zone 8S [6401] WGS 72BE / UTM zone 9N [6402] WGS 72BE / UTM zone 9S [6403] WGS 84 / Andaman and Nicobar [6404] WGS 84 / Andhra Pradesh [6405] WGS 84 / Antarctic Polar Stereographic [6406] WGS 84 / Arctic Polar Stereographic [6407] WGS 84 / Arunachal Pradesh [6408] WGS 84 / Assam [6409] WGS 84 / Australian Antarctic Lambert [6410] WGS 84 / Australian Antarctic Polar Stereographic [6411] WGS 84 / Australian Centre for Remote Sensing Lambert [6412] WGS 84 / BLM 14N (ftUS) [6413] WGS 84 / BLM 15N (ftUS) [6414] WGS 84 / BLM 16N (ftUS) [6415] WGS 84 / BLM 17N (ftUS) [6416] WGS 84 / Bihar [6417] WGS 84 / CIG92 [6418] WGS 84 / CKIG92 [6419] WGS 84 / Cape Verde National [6420] WGS 84 / Chhattisgarh [6421] WGS 84 / Delhi [6422] WGS 84 / Dubai Local TM [6423] WGS 84 / EPSG Alaska Polar Stereographic [6424] WGS 84 / EPSG Arctic Regional zone A1 [6425] WGS 84 / EPSG Arctic Regional zone A2 [6426] WGS 84 / EPSG Arctic Regional zone A3 [6427] WGS 84 / EPSG Arctic Regional zone A4 [6428] WGS 84 / EPSG Arctic Regional zone A5 [6429] WGS 84 / EPSG Arctic Regional zone B1 [6430] WGS 84 / EPSG Arctic Regional zone B2 [6431] WGS 84 / EPSG Arctic Regional zone B3 [6432] WGS 84 / EPSG Arctic Regional zone B4 [6433] WGS 84 / EPSG Arctic Regional zone B5 [6434] WGS 84 / EPSG Arctic Regional zone C1 [6435] WGS 84 / EPSG Arctic Regional zone C2 [6436] WGS 84 / EPSG Arctic Regional zone C3 [6437] WGS 84 / EPSG Arctic Regional zone C4 [6438] WGS 84 / EPSG Arctic Regional zone C5 [6439] WGS 84 / EPSG Arctic zone 1-21 [6440] WGS 84 / EPSG Arctic zone 1-27 [6441] WGS 84 / EPSG Arctic zone 1-29 [6442] WGS 84 / EPSG Arctic zone 1-31 [6443] WGS 84 / EPSG Arctic zone 2-10 [6444] WGS 84 / EPSG Arctic zone 2-12 [6445] WGS 84 / EPSG Arctic zone 2-24 [6446] WGS 84 / EPSG Arctic zone 2-26 [6447] WGS 84 / EPSG Arctic zone 2-28 [6448] WGS 84 / EPSG Arctic zone 3-13 [6449] WGS 84 / EPSG Arctic zone 3-15 [6450] WGS 84 / EPSG Arctic zone 3-17 [6451] WGS 84 / EPSG Arctic zone 3-19 [6452] WGS 84 / EPSG Arctic zone 3-21 [6453] WGS 84 / EPSG Arctic zone 3-23 [6454] WGS 84 / EPSG Arctic zone 4-12 [6455] WGS 84 / EPSG Arctic zone 4-30 [6456] WGS 84 / EPSG Arctic zone 4-32 [6457] WGS 84 / EPSG Arctic zone 4-34 [6458] WGS 84 / EPSG Arctic zone 4-36 [6459] WGS 84 / EPSG Arctic zone 4-38 [6460] WGS 84 / EPSG Arctic zone 4-40 [6461] WGS 84 / EPSG Arctic zone 5-15 [6462] WGS 84 / EPSG Arctic zone 5-17 [6463] WGS 84 / EPSG Arctic zone 5-19 [6464] WGS 84 / EPSG Arctic zone 5-21 [6465] WGS 84 / EPSG Arctic zone 5-23 [6466] WGS 84 / EPSG Arctic zone 5-25 [6467] WGS 84 / EPSG Arctic zone 5-27 [6468] WGS 84 / EPSG Canada Polar Stereographic [6469] WGS 84 / EPSG Greenland Polar Stereographic [6470] WGS 84 / EPSG Norway Polar Stereographic [6471] WGS 84 / EPSG Russia Polar Stereographic [6472] WGS 84 / Gabon TM [6473] WGS 84 / Gabon TM 2011 [6474] WGS 84 / Goa [6475] WGS 84 / Gujarat [6476] WGS 84 / Haryana [6477] WGS 84 / Himachal Pradesh [6478] WGS 84 / IBCAO Polar Stereographic [6479] WGS 84 / India NSF LCC [6480] WGS 84 / India Northeast [6481] WGS 84 / Jammu and Kashmir [6482] WGS 84 / Jharkhand [6483] WGS 84 / Karnataka [6484] WGS 84 / Kerala [6485] WGS 84 / Lakshadweep [6486] WGS 84 / Madhya Pradesh [6487] WGS 84 / Maharashtra [6488] WGS 84 / Manipur [6489] WGS 84 / Meghalaya [6490] WGS 84 / Mercator 41 [6491] WGS 84 / Mercator 41 (deprecated) [6492] WGS 84 / Mizoram [6493] WGS 84 / NSIDC EASE-Grid 2.0 Global [6494] WGS 84 / NSIDC EASE-Grid 2.0 North [6495] WGS 84 / NSIDC EASE-Grid 2.0 South [6496] WGS 84 / NSIDC EASE-Grid Global (deprecated) [6497] WGS 84 / NSIDC EASE-Grid North (deprecated) [6498] WGS 84 / NSIDC EASE-Grid South (deprecated) [6499] WGS 84 / NSIDC Sea Ice Polar Stereographic North [6500] WGS 84 / NSIDC Sea Ice Polar Stereographic South [6501] WGS 84 / Nagaland [6502] WGS 84 / North Pole LAEA Alaska [6503] WGS 84 / North Pole LAEA Atlantic [6504] WGS 84 / North Pole LAEA Bering Sea [6505] WGS 84 / North Pole LAEA Canada [6506] WGS 84 / North Pole LAEA Europe [6507] WGS 84 / North Pole LAEA Russia [6508] WGS 84 / Orissa [6509] WGS 84 / PDC Mercator [6510] WGS 84 / PDC Mercator (deprecated) [6511] WGS 84 / Plate Carree (deprecated) [6512] WGS 84 / Pseudo-Mercator [6513] WGS 84 / Punjab [6514] WGS 84 / Rajasthan [6515] WGS 84 / SCAR IMW SP19-20 [6516] WGS 84 / SCAR IMW SP21-22 [6517] WGS 84 / SCAR IMW SP23-24 [6518] WGS 84 / SCAR IMW SQ01-02 [6519] WGS 84 / SCAR IMW SQ19-20 [6520] WGS 84 / SCAR IMW SQ21-22 [6521] WGS 84 / SCAR IMW SQ37-38 [6522] WGS 84 / SCAR IMW SQ39-40 [6523] WGS 84 / SCAR IMW SQ41-42 [6524] WGS 84 / SCAR IMW SQ43-44 [6525] WGS 84 / SCAR IMW SQ45-46 [6526] WGS 84 / SCAR IMW SQ47-48 [6527] WGS 84 / SCAR IMW SQ49-50 [6528] WGS 84 / SCAR IMW SQ51-52 [6529] WGS 84 / SCAR IMW SQ53-54 [6530] WGS 84 / SCAR IMW SQ55-56 [6531] WGS 84 / SCAR IMW SQ57-58 [6532] WGS 84 / SCAR IMW SR13-14 [6533] WGS 84 / SCAR IMW SR15-16 [6534] WGS 84 / SCAR IMW SR17-18 [6535] WGS 84 / SCAR IMW SR19-20 [6536] WGS 84 / SCAR IMW SR27-28 [6537] WGS 84 / SCAR IMW SR29-30 [6538] WGS 84 / SCAR IMW SR31-32 [6539] WGS 84 / SCAR IMW SR33-34 [6540] WGS 84 / SCAR IMW SR35-36 [6541] WGS 84 / SCAR IMW SR37-38 [6542] WGS 84 / SCAR IMW SR39-40 [6543] WGS 84 / SCAR IMW SR41-42 [6544] WGS 84 / SCAR IMW SR43-44 [6545] WGS 84 / SCAR IMW SR45-46 [6546] WGS 84 / SCAR IMW SR47-48 [6547] WGS 84 / SCAR IMW SR49-50 [6548] WGS 84 / SCAR IMW SR51-52 [6549] WGS 84 / SCAR IMW SR53-54 [6550] WGS 84 / SCAR IMW SR55-56 [6551] WGS 84 / SCAR IMW SR57-58 [6552] WGS 84 / SCAR IMW SR59-60 [6553] WGS 84 / SCAR IMW SS04-06 [6554] WGS 84 / SCAR IMW SS07-09 [6555] WGS 84 / SCAR IMW SS10-12 [6556] WGS 84 / SCAR IMW SS13-15 [6557] WGS 84 / SCAR IMW SS16-18 [6558] WGS 84 / SCAR IMW SS19-21 [6559] WGS 84 / SCAR IMW SS25-27 [6560] WGS 84 / SCAR IMW SS28-30 [6561] WGS 84 / SCAR IMW SS31-33 [6562] WGS 84 / SCAR IMW SS34-36 [6563] WGS 84 / SCAR IMW SS37-39 [6564] WGS 84 / SCAR IMW SS40-42 [6565] WGS 84 / SCAR IMW SS43-45 [6566] WGS 84 / SCAR IMW SS46-48 [6567] WGS 84 / SCAR IMW SS49-51 [6568] WGS 84 / SCAR IMW SS52-54 [6569] WGS 84 / SCAR IMW SS55-57 [6570] WGS 84 / SCAR IMW SS58-60 [6571] WGS 84 / SCAR IMW ST01-04 [6572] WGS 84 / SCAR IMW ST05-08 [6573] WGS 84 / SCAR IMW ST09-12 [6574] WGS 84 / SCAR IMW ST13-16 [6575] WGS 84 / SCAR IMW ST17-20 [6576] WGS 84 / SCAR IMW ST21-24 [6577] WGS 84 / SCAR IMW ST25-28 [6578] WGS 84 / SCAR IMW ST29-32 [6579] WGS 84 / SCAR IMW ST33-36 [6580] WGS 84 / SCAR IMW ST37-40 [6581] WGS 84 / SCAR IMW ST41-44 [6582] WGS 84 / SCAR IMW ST45-48 [6583] WGS 84 / SCAR IMW ST49-52 [6584] WGS 84 / SCAR IMW ST53-56 [6585] WGS 84 / SCAR IMW ST57-60 [6586] WGS 84 / SCAR IMW SU01-05 [6587] WGS 84 / SCAR IMW SU06-10 [6588] WGS 84 / SCAR IMW SU11-15 [6589] WGS 84 / SCAR IMW SU16-20 [6590] WGS 84 / SCAR IMW SU21-25 [6591] WGS 84 / SCAR IMW SU26-30 [6592] WGS 84 / SCAR IMW SU31-35 [6593] WGS 84 / SCAR IMW SU36-40 [6594] WGS 84 / SCAR IMW SU41-45 [6595] WGS 84 / SCAR IMW SU46-50 [6596] WGS 84 / SCAR IMW SU51-55 [6597] WGS 84 / SCAR IMW SU56-60 [6598] WGS 84 / SCAR IMW SV01-10 [6599] WGS 84 / SCAR IMW SV11-20 [6600] WGS 84 / SCAR IMW SV21-30 [6601] WGS 84 / SCAR IMW SV31-40 [6602] WGS 84 / SCAR IMW SV41-50 [6603] WGS 84 / SCAR IMW SV51-60 [6604] WGS 84 / SCAR IMW SW01-60 [6605] WGS 84 / Sikkim [6606] WGS 84 / South Georgia Lambert [6607] WGS 84 / TM 116 SE [6608] WGS 84 / TM 12 SE [6609] WGS 84 / TM 132 SE [6610] WGS 84 / TM 36 SE [6611] WGS 84 / TM 6 NE [6612] WGS 84 / TM 60 SW [6613] WGS 84 / TM Zone 20N (ftUS) [6614] WGS 84 / TM Zone 21N (ftUS) [6615] WGS 84 / TMzn35N [6616] WGS 84 / TMzn36N [6617] WGS 84 / Tamil Nadu [6618] WGS 84 / Tripura [6619] WGS 84 / UPS North (E,N) [6620] WGS 84 / UPS North (N,E) [6621] WGS 84 / UPS South (E,N) [6622] WGS 84 / UPS South (N,E) [6623] WGS 84 / USGS Transantarctic Mountains [6624] WGS 84 / UTM zone 10N [6625] WGS 84 / UTM zone 10S [6626] WGS 84 / UTM zone 11N [6627] WGS 84 / UTM zone 11S [6628] WGS 84 / UTM zone 12N [6629] WGS 84 / UTM zone 12S [6630] WGS 84 / UTM zone 13N [6631] WGS 84 / UTM zone 13S [6632] WGS 84 / UTM zone 14N [6633] WGS 84 / UTM zone 14S [6634] WGS 84 / UTM zone 15N [6635] WGS 84 / UTM zone 15S [6636] WGS 84 / UTM zone 16N [6637] WGS 84 / UTM zone 16S [6638] WGS 84 / UTM zone 17N [6639] WGS 84 / UTM zone 17S [6640] WGS 84 / UTM zone 18N [6641] WGS 84 / UTM zone 18S [6642] WGS 84 / UTM zone 19N [6643] WGS 84 / UTM zone 19S [6644] WGS 84 / UTM zone 1N [6645] WGS 84 / UTM zone 1S [6646] WGS 84 / UTM zone 20N [6647] WGS 84 / UTM zone 20S [6648] WGS 84 / UTM zone 21N [6649] WGS 84 / UTM zone 21S [6650] WGS 84 / UTM zone 22N [6651] WGS 84 / UTM zone 22S [6652] WGS 84 / UTM zone 23N [6653] WGS 84 / UTM zone 23S [6654] WGS 84 / UTM zone 24N [6655] WGS 84 / UTM zone 24S [6656] WGS 84 / UTM zone 25N [6657] WGS 84 / UTM zone 25S [6658] WGS 84 / UTM zone 26N [6659] WGS 84 / UTM zone 26S [6660] WGS 84 / UTM zone 27N [6661] WGS 84 / UTM zone 27S [6662] WGS 84 / UTM zone 28N [6663] WGS 84 / UTM zone 28S [6664] WGS 84 / UTM zone 29N [6665] WGS 84 / UTM zone 29S [6666] WGS 84 / UTM zone 2N [6667] WGS 84 / UTM zone 2S [6668] WGS 84 / UTM zone 30N [6669] WGS 84 / UTM zone 30S [6670] WGS 84 / UTM zone 31N [6671] WGS 84 / UTM zone 31S [6672] WGS 84 / UTM zone 32N [6673] WGS 84 / UTM zone 32S [6674] WGS 84 / UTM zone 33N [6675] WGS 84 / UTM zone 33S [6676] WGS 84 / UTM zone 34N [6677] WGS 84 / UTM zone 34S [6678] WGS 84 / UTM zone 35N [6679] WGS 84 / UTM zone 35S [6680] WGS 84 / UTM zone 36N [6681] WGS 84 / UTM zone 36S [6682] WGS 84 / UTM zone 37N [6683] WGS 84 / UTM zone 37S [6684] WGS 84 / UTM zone 38N [6685] WGS 84 / UTM zone 38S [6686] WGS 84 / UTM zone 39N [6687] WGS 84 / UTM zone 39S [6688] WGS 84 / UTM zone 3N [6689] WGS 84 / UTM zone 3S [6690] WGS 84 / UTM zone 40N [6691] WGS 84 / UTM zone 40S [6692] WGS 84 / UTM zone 41N [6693] WGS 84 / UTM zone 41S [6694] WGS 84 / UTM zone 42N [6695] WGS 84 / UTM zone 42S [6696] WGS 84 / UTM zone 43N [6697] WGS 84 / UTM zone 43S [6698] WGS 84 / UTM zone 44N [6699] WGS 84 / UTM zone 44S [6700] WGS 84 / UTM zone 45N [6701] WGS 84 / UTM zone 45S [6702] WGS 84 / UTM zone 46N [6703] WGS 84 / UTM zone 46S [6704] WGS 84 / UTM zone 47N [6705] WGS 84 / UTM zone 47S [6706] WGS 84 / UTM zone 48N [6707] WGS 84 / UTM zone 48S [6708] WGS 84 / UTM zone 49N [6709] WGS 84 / UTM zone 49S [6710] WGS 84 / UTM zone 4N [6711] WGS 84 / UTM zone 4S [6712] WGS 84 / UTM zone 50N [6713] WGS 84 / UTM zone 50S [6714] WGS 84 / UTM zone 51N [6715] WGS 84 / UTM zone 51S [6716] WGS 84 / UTM zone 52N [6717] WGS 84 / UTM zone 52S [6718] WGS 84 / UTM zone 53N [6719] WGS 84 / UTM zone 53S [6720] WGS 84 / UTM zone 54N [6721] WGS 84 / UTM zone 54S [6722] WGS 84 / UTM zone 55N [6723] WGS 84 / UTM zone 55S [6724] WGS 84 / UTM zone 56N [6725] WGS 84 / UTM zone 56S [6726] WGS 84 / UTM zone 57N [6727] WGS 84 / UTM zone 57S [6728] WGS 84 / UTM zone 58N [6729] WGS 84 / UTM zone 58S [6730] WGS 84 / UTM zone 59N [6731] WGS 84 / UTM zone 59S [6732] WGS 84 / UTM zone 5N [6733] WGS 84 / UTM zone 5S [6734] WGS 84 / UTM zone 60N [6735] WGS 84 / UTM zone 60S [6736] WGS 84 / UTM zone 6N [6737] WGS 84 / UTM zone 6S [6738] WGS 84 / UTM zone 7N [6739] WGS 84 / UTM zone 7S [6740] WGS 84 / UTM zone 8N [6741] WGS 84 / UTM zone 8S [6742] WGS 84 / UTM zone 9N [6743] WGS 84 / UTM zone 9S [6744] WGS 84 / Uttar Pradesh [6745] WGS 84 / Uttaranchal [6746] WGS 84 / West Bengal [6747] WGS 84 / World Equidistant Cylindrical [6748] WGS 84 / World Equidistant Cylindrical (deprecated) [6749] WGS 84 / World Mercator [6750] WGS_1984_ARC_System_Zone_01 [6751] WGS_1984_ARC_System_Zone_02 [6752] WGS_1984_ARC_System_Zone_03 [6753] WGS_1984_ARC_System_Zone_04 [6754] WGS_1984_ARC_System_Zone_05 [6755] WGS_1984_ARC_System_Zone_06 [6756] WGS_1984_ARC_System_Zone_07 [6757] WGS_1984_ARC_System_Zone_08 [6758] WGS_1984_ARC_System_Zone_09 [6759] WGS_1984_ARC_System_Zone_10 [6760] WGS_1984_ARC_System_Zone_11 [6761] WGS_1984_ARC_System_Zone_12 [6762] WGS_1984_ARC_System_Zone_13 [6763] WGS_1984_ARC_System_Zone_14 [6764] WGS_1984_ARC_System_Zone_15 [6765] WGS_1984_ARC_System_Zone_16 [6766] WGS_1984_ARC_System_Zone_17 [6767] WGS_1984_ARC_System_Zone_18 [6768] WGS_1984_Australian_Centre_for_Remote_Sensing_Lambert (deprecated) [6769] WGS_1984_California_Teale_Albers_FtUS [6770] WGS_1984_Canada_Atlas_LCC [6771] WGS_1984_Cape_Verde_Grid (deprecated) [6772] WGS_1984_Costa_Rica_TM_90 [6773] WGS_1984_Web_Mercator (deprecated) [6774] WGS_1984_Web_Mercator_Auxiliary_Sphere (deprecated) [6775] World Equidistant Cylindrical (Sphere) [6776] World Equidistant Cylindrical (Sphere) (deprecated) [6777] World_Aitoff [6778] World_Azimuthal_Equidistant [6779] World_Bonne [6780] World_Cassini [6781] World_Craster_Parabolic [6782] World_Cylindrical_Equal_Area [6783] World_Eckert_I [6784] World_Eckert_II [6785] World_Eckert_III [6786] World_Eckert_IV [6787] World_Eckert_V [6788] World_Eckert_VI [6789] World_Equidistant_Conic [6790] World_Equidistant_Cylindrical [6791] World_Gall_Stereographic [6792] World_Goode_Homolosine_Land [6793] World_Goode_Homolosine_Ocean [6794] World_Hotine [6795] World_Loximuthal [6796] World_Mercator [6797] World_Miller_Cylindrical [6798] World_Mollweide [6799] World_Plate_Carree [6800] World_Polyconic [6801] World_Quartic_Authalic [6802] World_Robinson [6803] World_Sinusoidal [6804] World_Stereographic [6805] World_Two_Point_Equidistant [6806] World_Van_der_Grinten_I [6807] World_Wagner_IV [6808] World_Wagner_V [6809] World_Wagner_VII [6810] World_Winkel_I [6811] World_Winkel_II [6812] World_Winkel_Tripel_NGS [6813] Xian 1980 / 3-degree Gauss-Kruger CM 102E [6814] Xian 1980 / 3-degree Gauss-Kruger CM 105E [6815] Xian 1980 / 3-degree Gauss-Kruger CM 108E [6816] Xian 1980 / 3-degree Gauss-Kruger CM 111E [6817] Xian 1980 / 3-degree Gauss-Kruger CM 114E [6818] Xian 1980 / 3-degree Gauss-Kruger CM 117E [6819] Xian 1980 / 3-degree Gauss-Kruger CM 120E [6820] Xian 1980 / 3-degree Gauss-Kruger CM 123E [6821] Xian 1980 / 3-degree Gauss-Kruger CM 126E [6822] Xian 1980 / 3-degree Gauss-Kruger CM 129E [6823] Xian 1980 / 3-degree Gauss-Kruger CM 132E [6824] Xian 1980 / 3-degree Gauss-Kruger CM 135E [6825] Xian 1980 / 3-degree Gauss-Kruger CM 75E [6826] Xian 1980 / 3-degree Gauss-Kruger CM 78E [6827] Xian 1980 / 3-degree Gauss-Kruger CM 81E [6828] Xian 1980 / 3-degree Gauss-Kruger CM 84E [6829] Xian 1980 / 3-degree Gauss-Kruger CM 87E [6830] Xian 1980 / 3-degree Gauss-Kruger CM 90E [6831] Xian 1980 / 3-degree Gauss-Kruger CM 93E [6832] Xian 1980 / 3-degree Gauss-Kruger CM 96E [6833] Xian 1980 / 3-degree Gauss-Kruger CM 99E [6834] Xian 1980 / 3-degree Gauss-Kruger zone 25 [6835] Xian 1980 / 3-degree Gauss-Kruger zone 26 [6836] Xian 1980 / 3-degree Gauss-Kruger zone 27 [6837] Xian 1980 / 3-degree Gauss-Kruger zone 28 [6838] Xian 1980 / 3-degree Gauss-Kruger zone 29 [6839] Xian 1980 / 3-degree Gauss-Kruger zone 30 [6840] Xian 1980 / 3-degree Gauss-Kruger zone 31 [6841] Xian 1980 / 3-degree Gauss-Kruger zone 32 [6842] Xian 1980 / 3-degree Gauss-Kruger zone 33 [6843] Xian 1980 / 3-degree Gauss-Kruger zone 34 [6844] Xian 1980 / 3-degree Gauss-Kruger zone 35 [6845] Xian 1980 / 3-degree Gauss-Kruger zone 36 [6846] Xian 1980 / 3-degree Gauss-Kruger zone 37 [6847] Xian 1980 / 3-degree Gauss-Kruger zone 38 [6848] Xian 1980 / 3-degree Gauss-Kruger zone 39 [6849] Xian 1980 / 3-degree Gauss-Kruger zone 40 [6850] Xian 1980 / 3-degree Gauss-Kruger zone 41 [6851] Xian 1980 / 3-degree Gauss-Kruger zone 42 [6852] Xian 1980 / 3-degree Gauss-Kruger zone 43 [6853] Xian 1980 / 3-degree Gauss-Kruger zone 44 [6854] Xian 1980 / 3-degree Gauss-Kruger zone 45 [6855] Xian 1980 / Gauss-Kruger CM 105E [6856] Xian 1980 / Gauss-Kruger CM 111E [6857] Xian 1980 / Gauss-Kruger CM 117E [6858] Xian 1980 / Gauss-Kruger CM 123E [6859] Xian 1980 / Gauss-Kruger CM 129E [6860] Xian 1980 / Gauss-Kruger CM 135E [6861] Xian 1980 / Gauss-Kruger CM 75E [6862] Xian 1980 / Gauss-Kruger CM 81E [6863] Xian 1980 / Gauss-Kruger CM 87E [6864] Xian 1980 / Gauss-Kruger CM 93E [6865] Xian 1980 / Gauss-Kruger CM 99E [6866] Xian 1980 / Gauss-Kruger zone 13 [6867] Xian 1980 / Gauss-Kruger zone 14 [6868] Xian 1980 / Gauss-Kruger zone 15 [6869] Xian 1980 / Gauss-Kruger zone 16 [6870] Xian 1980 / Gauss-Kruger zone 17 [6871] Xian 1980 / Gauss-Kruger zone 18 [6872] Xian 1980 / Gauss-Kruger zone 19 [6873] Xian 1980 / Gauss-Kruger zone 20 [6874] Xian 1980 / Gauss-Kruger zone 21 [6875] Xian 1980 / Gauss-Kruger zone 22 [6876] Xian 1980 / Gauss-Kruger zone 23 [6877] Yemen NGN96 / UTM zone 37N [6878] Yemen NGN96 / UTM zone 38N [6879] Yemen NGN96 / UTM zone 39N [6880] Yemen NGN96 / UTM zone 40N [6881] Yoff / UTM zone 28N [6882] Zanderij / Suriname Old TM [6883] Zanderij / Suriname TM [6884] Zanderij / TM 54 NW [6885] Zanderij / UTM zone 21N Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_pgsql', '22', 'Update Shapes SRID')
    if Tool.is_Okay():
        Tool.Set_Option('CONNECTION', CONNECTION)
        Tool.Set_Option('DB_TABLE', DB_TABLE)
        Tool.Set_Option('CRS_CODE', CRS_CODE)
        Tool.Set_Option('CRS_AUTHORITY', CRS_AUTHORITY)
        Tool.Set_Option('CRS_GEOGCS', CRS_GEOGCS)
        Tool.Set_Option('CRS_PROJCS', CRS_PROJCS)
        return Tool.Execute(Verbose)
    return False

def run_tool_db_pgsql_22(CONNECTION=None, DB_TABLE=None, CRS_CODE=None, CRS_AUTHORITY=None, CRS_GEOGCS=None, CRS_PROJCS=None, Verbose=2):
    '''
    Update Shapes SRID
    ----------
    [db_pgsql.22]\n
    Change the SRID of all geometries in the user-specified column and table.\n
    Arguments
    ----------
    - CONNECTION [`choice`] : Connection. Available Choices: Default: 0
    - DB_TABLE [`text`] : Table
    - CRS_CODE [`integer number`] : Authority Code. Minimum: -1 Default: -1
    - CRS_AUTHORITY [`text`] : Authority. Default: EPSG
    - CRS_GEOGCS [`choice`] : Geographic Coordinate Systems. Available Choices: [0] <select> [1] 1_Ceres_2015 [2] 4_Vesta_2015 [3] AGD66 [4] AGD84 [5] ATF (Paris) [6] ATS77 [7] Abidjan 1987 [8] Accra [9] Aden 1925 [10] Adindan [11] Afgooye [12] Agadez [13] Ain el Abd [14] Albanian 1987 [15] American Samoa 1962 [16] Amersfoort [17] Ammassalik 1958 [18] Anguilla 1957 [19] Antigua 1943 [20] Aratu [21] Arc 1950 [22] Arc 1960 [23] Ascension Island 1958 [24] Astro DOS 71 [25] Australian Antarctic [26] Australian Antarctic [27] Australian Antarctic [28] Ayabelle Lighthouse [29] Azores Central 1948 [30] Azores Central 1995 [31] Azores Occidental 1939 [32] Azores Oriental 1940 [33] Azores Oriental 1995 [34] BDA2000 [35] BDA2000 [36] BGS2005 [37] BGS2005 [38] Barbados 1938 [39] Batavia [40] Batavia (Jakarta) [41] Beduaram [42] Beijing 1954 [43] Bekaa Valley 1920 [44] Belge 1950 [45] Belge 1950 (Brussels) [46] Belge 1972 [47] Bellevue [48] Bermuda 1957 [49] Bern 1898 (Bern) [50] Bern 1938 [51] Bioko [52] Bissau [53] Bogota 1975 [54] Bogota 1975 (Bogota) [55] Bukit Rimpah [56] CGRS93 [57] CGRS93 [58] CH1903 [59] CH1903+ [60] CHTRF95 [61] CHTRF95 [62] CHTRF95 [63] CIGD11 [64] CIGD11 [65] CR-SIRGAS [66] CR-SIRGAS [67] CR05 [68] CR05 [69] CSG67 [70] Cadastre 1997 [71] Cadastre 1997 [72] California_SRS_Epoch_2017.50_(NAD83) [73] Camacupa [74] Camp Area Astro [75] Campo Inchauspe [76] Cape [77] Cape Canaveral [78] Carthage [79] Carthage (Paris) [80] Chatham Islands 1971 [81] Chatham Islands 1979 [82] China Geodetic Coordinate System 2000 [83] China Geodetic Coordinate System 2000 [84] Chos Malal 1914 [85] Chua [86] Cocos Islands 1965 [87] Combani 1950 [88] Conakry 1905 [89] Corrego Alegre 1961 [90] Corrego Alegre 1970-72 [91] Cote d'Ivoire [92] DB_REF [93] DB_REF [94] DGN95 [95] DGN95 [96] DHDN [97] DRUKREF 03 [98] DRUKREF 03 [99] Dabola 1981 [100] Datum 73 [101] Dealul Piscului 1930 [102] Dealul Piscului 1970 [103] Deception Island [104] Deir ez Zor [105] Diego Garcia 1969 [106] Dominica 1945 [107] Douala [108] Douala 1948 [109] ED50 [110] ED50(ED77) [111] ED79 [112] ED87 [113] ELD79 [114] EST92 [115] EST97 [116] EST97 [117] EST97 [118] ETRF2000 [119] ETRF2000 [120] ETRF2005 [121] ETRF2014 [122] ETRF89 [123] ETRF90 [124] ETRF90 [125] ETRF91 [126] ETRF91 [127] ETRF92 [128] ETRF92 [129] ETRF93 [130] ETRF93 [131] ETRF94 [132] ETRF94 [133] ETRF96 [134] ETRF96 [135] ETRF97 [136] ETRF97 [137] ETRS89 [138] ETRS89 [139] ETRS89 [140] Easter Island 1967 [141] Egypt 1907 [142] Egypt 1930 [143] Egypt Gulf of Suez S-650 TL [144] FD54 [145] FD58 [146] FEH2010 [147] FEH2010 [148] Fahud [149] Fatu Iva 72 [150] Fiji 1956 [151] Fiji 1986 [152] Fort Marigot [153] GCGD59 [154] GCS_Adrastea_2000 [155] GCS_Alaskan_Islands [156] GCS_Amalthea_2000 [157] GCS_American_Samoa_1962 (deprecated) [158] GCS_Ananke_2000 [159] GCS_Anna_1_1965 (deprecated) [160] GCS_Ariel_2000 [161] GCS_Ascension_Island_1958 (deprecated) [162] GCS_Assumed_Geographic_1 (deprecated) [163] GCS_Astro_1952 (deprecated) [164] GCS_Atlas_2000 [165] GCS_Ayabelle (deprecated) [166] GCS_Bab_South [167] GCS_Beacon_E_1945 (deprecated) [168] GCS_Belinda_2000 [169] GCS_Bellevue_IGN (deprecated) [170] GCS_Bermuda_2000 (deprecated) [171] GCS_Bianca_2000 [172] GCS_CGRS_1993 (deprecated) [173] GCS_CR05 (deprecated) [174] GCS_Callisto_2000 [175] GCS_Calypso_2000 [176] GCS_Camp_Area (deprecated) [177] GCS_Canton_1966 (deprecated) [178] GCS_Cape_Canaveral (deprecated) [179] GCS_Carme_2000 [180] GCS_Carthage (deprecated) [181] GCS_Carthage_Grad [182] GCS_Charon_2000 [183] GCS_Chatham_Island_1971 (deprecated) [184] GCS_Chatham_Islands_1979 (deprecated) [185] GCS_Cordelia_2000 [186] GCS_Cressida_2000 [187] GCS_D48 [188] GCS_DOS_1968 [189] GCS_DOS_71_4 (deprecated) [190] GCS_Datum_Lisboa_Bessel [191] GCS_Datum_Lisboa_Hayford [192] GCS_Deception_Island (deprecated) [193] GCS_Deimos_2000 [194] GCS_Desdemona_2000 [195] GCS_Despina_2000 [196] GCS_Dione_2000 [197] GCS_ETRF_1989 [198] GCS_EUREF_FIN [199] GCS_Easter_Island_1967 (deprecated) [200] GCS_Elara_2000 [201] GCS_Enceladus_2000 [202] GCS_Epimetheus_2000 [203] GCS_Estonia_1937 [204] GCS_Europa_2000 [205] GCS_European_1979 (deprecated) [206] GCS_Everest_Bangladesh [207] GCS_Everest_India_Nepal [208] GCS_Everest_Modified_1969 [209] GCS_Fischer_1960 [210] GCS_Fischer_1968 [211] GCS_Fischer_Modified [212] GCS_Fort_Thomas_1955 [213] GCS_GDBD2009 (deprecated) [214] GCS_GUX_1 [215] GCS_Galatea_2000 [216] GCS_Gan_1970 (deprecated) [217] GCS_Ganymede_2000 [218] GCS_Graciosa_Base_SW_1948 [219] GCS_Guam_1963 (deprecated) [220] GCS_Gunung_Segara (deprecated) [221] GCS_HD1909 (deprecated) [222] GCS_Helene_2000 [223] GCS_Hermannskogel [224] GCS_Himalia_2000 [225] GCS_Hjorsey_1955 (deprecated) [226] GCS_Hong_Kong_1963_67 (deprecated) [227] GCS_Hong_Kong_1980 (deprecated) [228] GCS_Hough_1960 [229] GCS_Hyperion_2000 [230] GCS_IGRS (deprecated) [231] GCS_IGS08 [232] GCS_ISN_2004 (deprecated) [233] GCS_ISTS_061_1968 (deprecated) [234] GCS_ISTS_073_1969 (deprecated) [235] GCS_ITRF_1988 [236] GCS_ITRF_1989 [237] GCS_ITRF_1990 [238] GCS_ITRF_1991 [239] GCS_ITRF_1992 [240] GCS_ITRF_1993 [241] GCS_ITRF_1994 [242] GCS_ITRF_1996 [243] GCS_ITRF_1997 [244] GCS_ITRF_2000 [245] GCS_ITRF_2005 [246] GCS_ITRF_2008 [247] GCS_Iapetus_2000 [248] GCS_International_1967 (deprecated) [249] GCS_Io_2000 [250] GCS_JAD_2001 (deprecated) [251] GCS_JGD_2000 (deprecated) [252] GCS_JGD_2011 (deprecated) [253] GCS_Janus_2000 [254] GCS_Johnston_Island_1961 (deprecated) [255] GCS_Jordan [256] GCS_Juliet_2000 [257] GCS_Jupiter_2000 [258] GCS_Kerguelen_Island_1949 (deprecated) [259] GCS_Kusaie_1951 (deprecated) [260] GCS_Kyrg-06 [261] GCS_LC5_1961 [262] GCS_Larissa_2000 [263] GCS_Leda_2000 [264] GCS_Lysithea_2000 [265] GCS_MACAO_2008 [266] GCS_MGI_1901 (deprecated) [267] GCS_MONREF_1997 [268] GCS_MSK_1942 [269] GCS_Majuro [270] GCS_Mars_1979 [271] GCS_Mars_2000 [272] GCS_Merchich_Degree [273] GCS_Mercury_2000 [274] GCS_Metis_2000 [275] GCS_Midway_1961 (deprecated) [276] GCS_Mimas_2000 [277] GCS_Miranda_2000 [278] GCS_Moon_2000 [279] GCS_NAD_1983_2011 (deprecated) [280] GCS_NAD_1983_CORS96 (deprecated) [281] GCS_NAD_1983_HARN_Adj_MN_Anoka [282] GCS_NAD_1983_HARN_Adj_MN_Becker [283] GCS_NAD_1983_HARN_Adj_MN_Beltrami_North [284] GCS_NAD_1983_HARN_Adj_MN_Beltrami_South [285] GCS_NAD_1983_HARN_Adj_MN_Benton [286] GCS_NAD_1983_HARN_Adj_MN_Big_Stone [287] GCS_NAD_1983_HARN_Adj_MN_Blue_Earth [288] GCS_NAD_1983_HARN_Adj_MN_Brown [289] GCS_NAD_1983_HARN_Adj_MN_Carlton [290] GCS_NAD_1983_HARN_Adj_MN_Carver [291] GCS_NAD_1983_HARN_Adj_MN_Cass_North [292] GCS_NAD_1983_HARN_Adj_MN_Cass_South [293] GCS_NAD_1983_HARN_Adj_MN_Chippewa [294] GCS_NAD_1983_HARN_Adj_MN_Chisago [295] GCS_NAD_1983_HARN_Adj_MN_Cook_North [296] GCS_NAD_1983_HARN_Adj_MN_Cook_South [297] GCS_NAD_1983_HARN_Adj_MN_Cottonwood [298] GCS_NAD_1983_HARN_Adj_MN_Crow_Wing [299] GCS_NAD_1983_HARN_Adj_MN_Dakota [300] GCS_NAD_1983_HARN_Adj_MN_Dodge [301] GCS_NAD_1983_HARN_Adj_MN_Douglas [302] GCS_NAD_1983_HARN_Adj_MN_Faribault [303] GCS_NAD_1983_HARN_Adj_MN_Fillmore [304] GCS_NAD_1983_HARN_Adj_MN_Freeborn [305] GCS_NAD_1983_HARN_Adj_MN_Goodhue [306] GCS_NAD_1983_HARN_Adj_MN_Grant [307] GCS_NAD_1983_HARN_Adj_MN_Hennepin [308] GCS_NAD_1983_HARN_Adj_MN_Houston [309] GCS_NAD_1983_HARN_Adj_MN_Isanti [310] GCS_NAD_1983_HARN_Adj_MN_Itasca_North [311] GCS_NAD_1983_HARN_Adj_MN_Itasca_South [312] GCS_NAD_1983_HARN_Adj_MN_Jackson [313] GCS_NAD_1983_HARN_Adj_MN_Kanabec [314] GCS_NAD_1983_HARN_Adj_MN_Kandiyohi [315] GCS_NAD_1983_HARN_Adj_MN_Kittson [316] GCS_NAD_1983_HARN_Adj_MN_Koochiching [317] GCS_NAD_1983_HARN_Adj_MN_Lac_Qui_Parle [318] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_North [319] GCS_NAD_1983_HARN_Adj_MN_Lake_of_the_Woods_South [320] GCS_NAD_1983_HARN_Adj_MN_Le_Sueur [321] GCS_NAD_1983_HARN_Adj_MN_Lincoln [322] GCS_NAD_1983_HARN_Adj_MN_Lyon [323] GCS_NAD_1983_HARN_Adj_MN_Mahnomen [324] GCS_NAD_1983_HARN_Adj_MN_Marshall [325] GCS_NAD_1983_HARN_Adj_MN_Martin [326] GCS_NAD_1983_HARN_Adj_MN_McLeod [327] GCS_NAD_1983_HARN_Adj_MN_Meeker [328] GCS_NAD_1983_HARN_Adj_MN_Morrison [329] GCS_NAD_1983_HARN_Adj_MN_Mower [330] GCS_NAD_1983_HARN_Adj_MN_Murray [331] GCS_NAD_1983_HARN_Adj_MN_Nicollet [332] GCS_NAD_1983_HARN_Adj_MN_Nobles [333] GCS_NAD_1983_HARN_Adj_MN_Norman [334] GCS_NAD_1983_HARN_Adj_MN_Olmsted [335] GCS_NAD_1983_HARN_Adj_MN_Ottertail [336] GCS_NAD_1983_HARN_Adj_MN_Pennington [337] GCS_NAD_1983_HARN_Adj_MN_Pine [338] GCS_NAD_1983_HARN_Adj_MN_Pipestone [339] GCS_NAD_1983_HARN_Adj_MN_Polk [340] GCS_NAD_1983_HARN_Adj_MN_Pope [341] GCS_NAD_1983_HARN_Adj_MN_Ramsey [342] GCS_NAD_1983_HARN_Adj_MN_Red_Lake [343] GCS_NAD_1983_HARN_Adj_MN_Redwood [344] GCS_NAD_1983_HARN_Adj_MN_Renville [345] GCS_NAD_1983_HARN_Adj_MN_Rice [346] GCS_NAD_1983_HARN_Adj_MN_Rock [347] GCS_NAD_1983_HARN_Adj_MN_Roseau [348] GCS_NAD_1983_HARN_Adj_MN_Scott [349] GCS_NAD_1983_HARN_Adj_MN_Sherburne [350] GCS_NAD_1983_HARN_Adj_MN_Sibley [351] GCS_NAD_1983_HARN_Adj_MN_St_Louis [352] GCS_NAD_1983_HARN_Adj_MN_St_Louis_Central [353] GCS_NAD_1983_HARN_Adj_MN_St_Louis_North [354] GCS_NAD_1983_HARN_Adj_MN_St_Louis_South [355] GCS_NAD_1983_HARN_Adj_MN_Stearns [356] GCS_NAD_1983_HARN_Adj_MN_Steele [357] GCS_NAD_1983_HARN_Adj_MN_Stevens [358] GCS_NAD_1983_HARN_Adj_MN_Swift [359] GCS_NAD_1983_HARN_Adj_MN_Todd [360] GCS_NAD_1983_HARN_Adj_MN_Traverse [361] GCS_NAD_1983_HARN_Adj_MN_Wabasha [362] GCS_NAD_1983_HARN_Adj_MN_Wadena [363] GCS_NAD_1983_HARN_Adj_MN_Waseca [364] GCS_NAD_1983_HARN_Adj_MN_Watonwan [365] GCS_NAD_1983_HARN_Adj_MN_Winona [366] GCS_NAD_1983_HARN_Adj_MN_Wright [367] GCS_NAD_1983_HARN_Adj_MN_Yellow_Medicine [368] GCS_NAD_1983_HARN_Adj_WI_Adams (deprecated) [369] GCS_NAD_1983_HARN_Adj_WI_Ashland (deprecated) [370] GCS_NAD_1983_HARN_Adj_WI_Barron (deprecated) [371] GCS_NAD_1983_HARN_Adj_WI_Bayfield (deprecated) [372] GCS_NAD_1983_HARN_Adj_WI_Brown (deprecated) [373] GCS_NAD_1983_HARN_Adj_WI_Buffalo (deprecated) [374] GCS_NAD_1983_HARN_Adj_WI_Burnett (deprecated) [375] GCS_NAD_1983_HARN_Adj_WI_Calumet (deprecated) [376] GCS_NAD_1983_HARN_Adj_WI_Chippewa [377] GCS_NAD_1983_HARN_Adj_WI_Clark (deprecated) [378] GCS_NAD_1983_HARN_Adj_WI_Columbia (deprecated) [379] GCS_NAD_1983_HARN_Adj_WI_Crawford (deprecated) [380] GCS_NAD_1983_HARN_Adj_WI_Dane (deprecated) [381] GCS_NAD_1983_HARN_Adj_WI_Dodge (deprecated) [382] GCS_NAD_1983_HARN_Adj_WI_Door (deprecated) [383] GCS_NAD_1983_HARN_Adj_WI_Douglas (deprecated) [384] GCS_NAD_1983_HARN_Adj_WI_Dunn (deprecated) [385] GCS_NAD_1983_HARN_Adj_WI_EauClaire (deprecated) [386] GCS_NAD_1983_HARN_Adj_WI_Florence (deprecated) [387] GCS_NAD_1983_HARN_Adj_WI_FondduLac (deprecated) [388] GCS_NAD_1983_HARN_Adj_WI_Forest (deprecated) [389] GCS_NAD_1983_HARN_Adj_WI_Grant (deprecated) [390] GCS_NAD_1983_HARN_Adj_WI_Green (deprecated) [391] GCS_NAD_1983_HARN_Adj_WI_GreenLake (deprecated) [392] GCS_NAD_1983_HARN_Adj_WI_Iowa (deprecated) [393] GCS_NAD_1983_HARN_Adj_WI_Iron (deprecated) [394] GCS_NAD_1983_HARN_Adj_WI_Jackson (deprec