//! Qore namespace
namespace Qore {
//! Json namespace
namespace Json {
/** @defgroup json_functions JSON Functions
//@{
 */

//! Serializes qore data into a JSON string, formatted with line breaks for easier readability
/** By default the string produced will be in UTF-8 encoding, but this can be overridden by the second argument

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param data the data to serialize to a JSON string
    @param encoding an optional output encoding for the resulting JSON string; if this argument is not passed, then UTF-8 encoding is used by default

    @return the JSON string corresponding to the arguments passed, formatted with line breaks for easier readability

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_json(); camel-case function names were deprecated in json 1.5

    @see @ref jsonserialization
*/
string makeFormattedJSONString(any data, __7_ string encoding);

//! Serializes qore data into a JSON string, without any line breaks
/** By default the string produced will be in UTF-8 encoding, but this can be overridden by the second argument

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param data the data to serialize to a JSON string
    @param encoding an optional output encoding for the resulting JSON string; if this argument is not passed, then UTF-8 encoding is used by default

    @return the JSON string corresponding to the arguments passed, without any line breaks

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_json(); camel-case function names were deprecated in json 1.5

    @see @ref jsonserialization
*/
string makeJSONString(any data, __7_ string encoding);

//! Serializes qore data into a JSON string with optional whitespace formatting (line breaks and indentation)
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code string json = make_json(data); @endcode

    By default the string produced will be in UTF-8 encoding, but this can be overridden by the \a encoding argument

    @param data the data to serialize to a JSON string
    @param format optional formatting flags; see @ref json_generation_constants for more information
    @param encoding an optional output encoding for the resulting JSON string; if this argument is not passed, then UTF-8 encoding is used by default

    @return the JSON string corresponding to the arguments passed

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if an error occurs converting input strings to the target encoding

    @note According to <a href="https://tools.ietf.org/html/rfc7159">RFC 7159</a> JSON strings can only be in UTF-8, UTF-16, or UTF-32 encoding, but this function will allow any encoding to be used to generate JSON strings

    @see @ref jsonserialization

    @since json 1.5 as a replacement for deprecated camel-case makeJSONString() and makeFormattedJSONString()
*/
string make_json(any data, __7_ int format, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseJSON();

//! Parses a JSON string and returns the corresponding %Qore data structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param json_str the JSON string to parse

    @return the %Qore data structure corresponding to the input string

    @throw JSON-PARSE-ERROR syntax error parsing JSON string

    @see @ref jsonserialization

    @deprecated use parse_json(); camel-case function names were deprecated in json 1.5
 */
auto parseJSON(string json_str);

//! Parses a JSON string and returns the corresponding %Qore data structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param json_str the JSON string to parse

    @return the %Qore data structure corresponding to the input string

    @throw JSON-PARSE-ERROR syntax error parsing JSON string

    @par Example:
    @code any data = parse_json(json); @endcode

    @see @ref jsonserialization

    @since json 1.5 as a replacement for deprecated camel-case parseJSON()
 */
auto parse_json(string json_str);

//@}
}
}

//! main Qore-language namespace
namespace Qore {
/** @defgroup json_rpc_functions JSON-RPC Functions
//@{
 */

//! Creates a JSON-RPC 1.1 error response string from the parameters passed, formatted with line breaks for easier readability
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param code the error code to return
    @param error a string error message
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC 1.1 error response, formatted with line breaks for easier readability

    @throw MAKE-JSONRPC11-ERROR-STRING-ERROR the error code is not between 100-999 or empty error message string
    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated do not use; camel-case function names were deprecated in json 1.5, and support for the non-published JSON-RPC 1.1 protocol was dropped in json 1.7

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeFormattedJSONRPC11ErrorString(softint code, string error, any id, any error_msg);

//! Creates a generic JSON-RPC error response string from the parameters passed, formatted with line breaks for easier readability
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC error response, formatted with line breaks for easier readability

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_error(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeFormattedJSONRPCErrorString(any version, any id, any error_msg);

//! Creates a JSON-RPC request string from the parameters passed, formatted with line breaks for easier readability
/** To follow JSON-RPC specifications, the generated string will always be in UTF-8 encoding

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param method_name the name of the JSON-RPC method to call
    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"s
    @param id the ID of the call
    @param request_msg the parameters for the message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC request, formatted with line breaks for easier readability

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_request(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeFormattedJSONRPCRequestString(string method_name, any version, any id, any request_msg);

//! Creates a JSON-RPC response string from the parameters passed, formatted with line breaks for easier readability
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param response_msg the parameters for the message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC response, formatted with line breaks for easier readability

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_response(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeFormattedJSONRPCResponseString(any version, any id, any response_msg);

//! Creates a JSON-RPC 1.1 error response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param code the error code to return
    @param error a string error message
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC 1.1 error response, without any line breaks

    @throw MAKE-JSONRPC11-ERROR-STRING-ERROR the error code is not between 100-999 or empty error message string
    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated do not use; camel-case function names were deprecated in json 1.5, and support for the non-published JSON-RPC 1.1 protocol was dropped in json 1.7

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeJSONRPC11ErrorString(softint code, string error, any id, any error_msg);

//! Creates a generic JSON-RPC error response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC error response, without any line breaks

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_error(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeJSONRPCErrorString(any version, any id, any error_msg);

//! Creates a JSON-RPC request string from the parameters passed, without any line breaks
/** To follow JSON-RPC specifications, the generated string will always be in UTF-8 encoding

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param method_name the name of the JSON-RPC method to call
    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param request_msg the parameters for the message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC request

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_request(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeJSONRPCRequestString(string method_name, any version, any id, any request_msg);

//! Creates a JSON-RPC response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param response_msg the parameters for the message/message payload

    @return a JSON string in UTF-8 encoding representing the JSON-RPC response, without any line breaks

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @deprecated use make_jsonrpc_response(); camel-case function names were deprecated in json 1.5

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string makeJSONRPCResponseString(any version, any id, any response_msg);

//! Creates a JSON-RPC 1.1 error response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param code the error code to return
    @param error a string error message
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload
    @param format optional formatting flags; see @ref json_generation_constants for more information

    @return a JSON string in UTF-8 encoding representing the JSON-RPC 1.1 error response, without any line breaks

    @throw MAKE-JSONRPC11-ERROR-STRING-ERROR the error code is not between 100-999 or empty error message string
    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @par Example:
    @code string json = make_jsonrpc11_error(200, msg, id, error); @endcode

    @since json 1.5 as a replacement for deprecated camel-case makeJSONRPC11ErrorString() and makeFormattedJSONRPC11ErrorString()

    @deprecated do not use; support for the non-published JSON-RPC 1.1 protocol was dropped in json 1.7

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string make_jsonrpc11_error(softint code, string error, any id, any error_msg, __7_ int format);

//! Creates a generic JSON-RPC error response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param error_msg the parameters for the error response message/message payload
    @param format optional formatting flags; see @ref json_generation_constants for more information

    @return a JSON string in UTF-8 encoding representing the JSON-RPC error response, without any line breaks

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @par Example:
    @code string json = make_jsonrpc_error("1.0", id, error); @endcode

    @since
    - json 1.5 added this function as a replacement for deprecated camel-case makeJSONRPCErrorString() and makeFormattedJSONRPCErrorString()
    - json 1.7 added support for JSON-RPC 2.0 and corrections for JSON-RPC 1.0 support

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string make_jsonrpc_error(any version, any id, any error_msg, __7_ int format);

//! Creates a JSON-RPC request string from the parameters passed, without any line breaks
/** To follow JSON-RPC specifications, the generated string will always be in UTF-8 encoding

   @par Code Flags:
    @ref RET_VALUE_ONLY

     @param method_name the name of the JSON-RPC method to call
    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param request_msg the parameters for the message/message payload
    @param format optional formatting flags; see @ref json_generation_constants for more information

    @return a JSON string in UTF-8 encoding representing the JSON-RPC request

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @par Example:
    @code string json = make_jsonrpc_request("omq.system.help", "1.0", id, msg); @endcode

    @since
    - json 1.5 added this function as a replacement for deprecated camel-case makeJSONRPCRequestString() and makeFormattedJSONRPCRequestString()
    - json 1.7 added support for JSON-RPC 2.0 and corrections for JSON-RPC 1.0 support

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string make_jsonrpc_request(string method_name, any version, any id, any request_msg, __7_ int format);

//! Creates a JSON-RPC response string from the parameters passed, without any line breaks
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param version the JSON-RPC version to include in the call under the \c "version" key (for JSON-RPC earlier than 2.0) or the \c "jsonrpc" key (for JSON-RPC equal to or greater than 2.0); will be converted to a string; if the resulting string is empty, the \c "version" key will not be sent; note that for strict JSON-RPC 1.0 compatibility this argument will not be serialized if passed as \c "1.0"
    @param id the ID of the call
    @param response_msg the parameters for the message/message payload
    @param format optional formatting flags; see @ref json_generation_constants for more information

    @return a JSON string in UTF-8 encoding representing the JSON-RPC response, without any line breaks

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)

    @par Example:
    @code string json = make_jsonrpc_response("1.0", id, msg); @endcode

    @since
    - json 1.5 added this function as a replacement for deprecated camel-case makeJSONRPCResponseString() and makeFormattedJSONRPCResponseString()
    - json 1.7 added support for JSON-RPC 2.0 and corrections for JSON-RPC 1.0 support

    @note do not use the \c version argument for strict JSON-RPC 1.0 compatibility (send as @ref nothing "NOTHING")

    @see
    - @ref jsonserialization
    - @ref JSONRPC
 */
string make_jsonrpc_response(any version, any id, any response_msg, __7_ int format);

//@}
}
//! Qore namespace
namespace Qore {
//! Json namespace
namespace Json {
/** @defgroup json_generation_constants JSON Generation Constants
//@{
 */

//! use whitespace formatting including line breaks and indentation to make generated JSON more readable by humans
    const JGF_ADD_FORMATTING = JGF_ADD_FORMATTING;
//! no flags; standard JSON generation without whitespace formatting
    const JGF_NONE = JGF_NONE;
/** @} */
}
}
