
/* $Id: qm.c,v 1.92 2007/12/01 12:29:57 agraef Exp $ */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

bool            debug_long = 0, brkdbg = 0;
volatile bool   brkflag = 0, quitflag = 0;
bool		gc_flag = 0, gc_v = 0;
double		gc_tol = 0.5;
bool		debug = 0;
int		debug_lock = 0;
bool		init_mode = 0;
int		nbreak = 0, nprof = 0;

static THREAD  *init_thr = NULL;

int lastblksz, maxnblks;

char           *qmmsg[] = {

	"Ok",				/* not used */

	"Break",
	"Halt",				/* not used */
	"Quit",				/* not used */
	"Out of memory",
	"Stack overflow",
	"Stack overflow",
	"Symbol table overflow",
	"Error in conditional",
	"Error in external function",
	"Value mismatch in definition",

	"Exception",
	"Fail",				/* not used */
	"FAIL",				/* not used */
	"Catch",			/* not used */

	"File %s not found",
	"Error reading code file",
	"Bad code file format",

	"Syntax error",
	"Unterminated string constant",
	"Invalid character escape in string constant",
	"External object",
	"Unknown symbol",
	"Ambiguous symbol",
	"Not a variable",
	"Cannot redefine const variable",
	"Cannot undefine const variable",
	"Bad format specification",
	"No such directory",
	"Too many nested command files",
	"Error writing %s",
	"Error compiling %s",
	"Too many arguments",

	"Unknown variable %s",
	"Invalid command (type ? for help)",

	"This can't happen"

};

static EXPR *x_copy(EXPR *x);
static void xcopy_cleanup(void);

static int evalf(THREAD *thr, int fno);
static int eval_with_frame(THREAD *thr, 
			   EXPR *x, int fno, EXPR *lvals[2], int *rp, int rc,
			   OPREC *ip, long xbp, int modno, int lineno,
			   int info_addr, byte info_offs);

static void lock_debug(THREAD *thr);
static void unlock_debug(THREAD *thr);
static void set_debug(THREAD *thr, bool debug);

static void debug_thread(THREAD *thr, char *msg);
static int rule(THREAD *thr, int fno, EXPR *lvals[2],
		long xbp, int addr, OPREC *ip,
		int modno, int lineno);
static void binding(THREAD *thr, int failed, int m, byte offs);
static void reduction(THREAD *thr, int fno, long xbp);
static void default_reduction(THREAD *thr, int fno, long xbp);
static void tail_reduction(THREAD *thr, int fno, long xbp, int fno1);

void error(char *s)
{
  flush_shift();
  fprintf(stderr, "! %s\n", s);
  fflush(stderr);
}

int             tmpspsz, tmptbsz;
int             atmpspsz = TMPSPSZ, atmptbsz = TMPTBSZ;

int	        maxargs = MAXARGS;

int		xnblks = 0;
XBLK           *xblk;
EXPR           *xheap;
EXPR           *xfreep;

THREAD	        threads[MAXTHREAD], *thr0 = threads;
short		nthreads, nused;

static THREAD  *nthr;

int push_mark(THREAD *thr, EXPR *h)
{
  if (!thr->mark) {
    if ((thr->mark = (Mark*)aalloc(100, sizeof(Mark)))) {
      thr->markp = thr->mark;
      thr->marksz = 100;
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (thr->markp-thr->mark == thr->marksz) {
    Mark *mark1;
    if ((mark1 = (Mark*)arealloc(thr->mark, thr->marksz, 100, sizeof(Mark)))) {
      thr->mark = mark1;
      thr->markp = thr->mark+thr->marksz;
      thr->marksz += 100;
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  }
  thr->markp->xp = thr->xsp-thr->xst;
  thr->markp->h = h;
  thr->markp++;
  return 1;
}

void pop_mark(THREAD *thr)
{
  if (thr->markp > thr->mark) --thr->markp;
}

int get_mark(THREAD *thr, long *xp, EXPR **h)
{
  if (thr->markp > thr->mark) {
    *xp = thr->markp[-1].xp;
    *h = thr->markp[-1].h;
    return 1;
  } else
    return 0;
}

int have_mark(THREAD *thr)
{
  return thr->markp>thr->mark;
}

static void add_sentinel(THREAD *thr, EXPR *x)
{
  EXPRL **y = &thr->sentinels;
  while (*y) y = &((*y)->next);
  *y = malloc(sizeof(EXPRL));
  if (*y) {
    (*y)->x = x; (*y)->next = NULL;
  }
}

static void free_sentinels(THREAD *thr)
{
  while (thr->sentinels) {
    EXPRL *x = thr->sentinels;
    thr->sentinels = x->next;
    qmfree(thr, x->x);
    free(x);
  }
}

void process_sentinels(THREAD *thr)
{
  while (thr->sentinels) {
    EXPRL *x = thr->sentinels; EXPR *y = NULL;
    thr->sentinels = x->next;
    if (eval(thr, x->x)) y = *--thr->xsp;
    qmfree(thr, x->x); qmfree(thr, y); free(x);
  }
}

/* The fx array stores expression nodes preallocated for the function symbols
   in the symbol table. This offers a significant improvement in performance
   when a program is run, since many operations are simply pushes of function
   symbols. Note that there are two versions of the array; the fx0 array is
   used for irreducible terms (created outside of special forms). */

static EXPR    *fx, *fx0;

/* MULTITHREADING SUPPORT */

#ifdef USE_THREADS

static pthread_key_t thr_key;

pthread_mutex_t global_mutex, tty_mutex, parse_mutex, reads_mutex;

static bool input_suspended;
static pthread_mutex_t input_mutex, init_mutex;
static pthread_cond_t input_cond, init_cond;

#ifndef _WIN32

/* do necessary cleanup at fork time */

void atfork_prepare(void)
{
  THREAD *thr;
  int i;
  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].prepare)
      (*dll_atfork[i].prepare)();
  pthread_mutex_lock(&init_mutex);
  pthread_mutex_lock(&input_mutex);
#if 0
  /* NOTE: We don't lock the tty, parse and reads mutexes, since that would
     suspend a fork in a background thread when some other thread (or the main
     loop) sits waiting reading from a tty or file. */
  pthread_mutex_lock(&tty_mutex);
  pthread_mutex_lock(&parse_mutex);
  pthread_mutex_lock(&reads_mutex);
#endif
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_mutex_lock(&thr->exit_mutex);
}

void atfork_parent(void)
{
  THREAD *thr;
  int i;
  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].parent)
      (*dll_atfork[i].parent)();
  pthread_mutex_unlock(&init_mutex);
  pthread_mutex_unlock(&input_mutex);
#if 0
  /* See notes above. */
  pthread_mutex_unlock(&tty_mutex);
  pthread_mutex_unlock(&parse_mutex);
  pthread_mutex_unlock(&reads_mutex);
#endif
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_mutex_unlock(&thr->exit_mutex);
}

void atfork_child(void)
{
  THREAD *thr, *this = get_thr();
  sigset_t sigset;
  int i;
  pthread_mutexattr_t attr;
  pthread_mutexattr_init(&attr);
  pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);

  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].child)
      (*dll_atfork[i].child)();
  pthread_mutex_init(&global_mutex, NULL);
  pthread_mutex_init(&init_mutex, NULL);
  pthread_cond_init(&init_cond, NULL);
  pthread_mutex_init(&input_mutex, NULL);
  pthread_cond_init(&input_cond, NULL);
  pthread_mutex_init(&tty_mutex, NULL);
  pthread_mutex_init(&parse_mutex, &attr);
  pthread_mutex_init(&reads_mutex, NULL);

  pthread_mutex_lock(&global_mutex);
  if (this->debug) {
    input_suspended = 1;
    debug_lock = 1;
    this->debug_lock = 1;
  } else {
    input_suspended = 0;
    debug_lock = 0;
    this->debug_lock = 0;
  }

  /* this thread is the new main thread, hence we unblock all signals */
  sigemptyset(&sigset);
  pthread_sigmask(SIG_SETMASK, &sigset, NULL);

  /* the forking thread becomes the one and only thread in the child process,
     so update the thread table accordingly */
  if (nused > 1)
    for (thr = thr0; thr < thr0+nthreads; thr++)
      if (thr->used) {
	pthread_mutex_init(&thr->exit_mutex, NULL);
	pthread_cond_init(&thr->exit_cond, NULL);
	if (thr != this) {
	  thr->tty_lock = 0;
	  thr->debug_lock = 0;
	  thr->stats_fini = 1;
	  thr->endtime = clock();
	  thr->active = 0;
	  /* since there is no chance that the other threads will be activated
	     again we can as well collect their resources now */
	  thr->sticky = 1;
	  while (thr->xsp > thr->xst)
	    qmfree(thr, *--thr->xsp);
	  while (thr->asp > thr->ast)
	    free(*--thr->asp);
	  if (thr->xst) free(thr->xst); thr->xst = thr->xsp = NULL;
	  if (thr->ast) free(thr->ast); thr->ast = thr->asp = NULL;
	  if (thr->args) free(thr->args); thr->args = NULL;
	  if (thr->mark) free(thr->mark); thr->mark = NULL;
	  if (thr->sentinels) free_sentinels(thr);
	  if (thr->vartb) free(thr->vartb); thr->vartb = NULL;
	} else {
	  /* update the info of the new main thread */
	  thr->id = pthread_self();
	  /* reinitialize the signal queue */
	  thr->nsig = 0;
	  thr->sigpend = thr->sigblk = 0;
	}
      }

  /*  if (this->debug) debug_thread(this, "thread #%d forked"); */
}

#endif

static void init_threads(void)
{
  pthread_mutexattr_t attr;
  pthread_mutexattr_init(&attr);
  pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);

  pthread_key_create(&thr_key, NULL);

  pthread_mutex_init(&global_mutex, NULL);
  pthread_mutex_init(&init_mutex, NULL);
  pthread_cond_init(&init_cond, NULL);
  pthread_mutex_init(&input_mutex, NULL);
  pthread_cond_init(&input_cond, NULL);
  pthread_mutex_init(&tty_mutex, NULL);
  pthread_mutex_init(&parse_mutex, &attr);
  pthread_mutex_init(&reads_mutex, NULL);

#ifdef HAVE_PTHREAD_ATFORK
  pthread_atfork(atfork_prepare, atfork_parent, atfork_child);
#endif
}

#endif

int init_thread(void)
{
  THREAD *thr;
#ifdef USE_THREADS
  pthread_mutex_lock(&global_mutex);
#else
  if (nthreads) return -1;
#endif
  thr = nthr;
  if (thr)
    nthr = thr->next;
  else if (nthreads >= MAXTHREAD)
    goto err;
  else
    thr = thr0+(nthreads++);

  thr->xstsz = XSTSZ;
  thr->astsz = ASTSZ;
  thr->maxxstsz = thr->maxastsz = stackmax;
  if (thr->maxxstsz > 0 && thr->maxxstsz < thr->xstsz)
    thr->xstsz = thr->maxxstsz;
  if (thr->maxastsz > 0 && thr->maxastsz < thr->astsz)
    thr->astsz = thr->maxastsz;
  if (!(thr->xst = (EXPR **)aalloc(thr->xstsz, sizeof(EXPR*))) ||
      !(thr->ast = (AREC **)aalloc(thr->astsz, sizeof(AREC*))) ||
      !(thr->args = (EXPR**)malloc((maxargs+1)*sizeof(EXPR*))))
    goto alloc_err;

  thr->next = NULL;
  thr->used = 1;
  thr->active = 1;
  thr->released = 0;
  thr->sticky = 0;
  thr->debug_lock = 0;
  thr->level = thr->stoplevel = 0;

  thr->xsp = thr->xst;
  thr->asp = thr->ast;
  thr->qmstat = thr->qmstat_save = OK;
  thr->mode = 0;
  thr->debug = debug;
  thr->brkdbg = brkdbg;
  thr->brkflag = 0;
  thr->nsig = 0;
  thr->sigpend = thr->sigblk = 0;

  thr->mark = thr->markp = NULL;
  thr->marksz = 0;
  thr->sentinels = NULL;

  thr->vartb = 0;
  thr->nvarsyms = thr->avarsyms = 0;
  thr->lastaddr = NONE;

  thr->maxexprs = thr->nexprs = thr->nredns = 0;
  thr->stats_init = (thr>thr0);
  thr->stats_fini = 0;
  thr->starttime = clock();

  thr->baseptr = NULL;

#ifdef USE_THREADS
  pthread_setspecific(thr_key, thr);
  thr->id = pthread_self();
  if (thr == thr0)
    pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
  else {
#ifndef _WIN32
    /* signals are to be handled in the main thread, hence we block all
       signals here */
    sigset_t sigset;
    sigfillset(&sigset);
    pthread_sigmask(SIG_SETMASK, &sigset, NULL);
#endif
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
  }
  pthread_mutex_init(&thr->exit_mutex, NULL);
  pthread_cond_init(&thr->exit_cond, NULL);
#endif
  nused++;
  if (thr->debug && thr > thr0) debug_thread(thr, "thread #%d started");
  return thr-thr0;

 alloc_err:
  if (thr->xst) free(thr->xst);
  if (thr->ast) free(thr->ast);
  if (thr->args) free(thr->args);
  if (thr+1 < thr0+nthreads) {
    thr->next = nthr;
    nthr = thr;
  } else
    nthreads--;
 err:
#ifdef USE_THREADS
  pthread_mutex_unlock(&global_mutex);
#endif
  return -1;
}

void exit_thread(int id)
{
  THREAD *thr = thr0+id;
#ifndef USE_THREADS
  return;
#else
  if (!thr->used || !thr->active) return;
  thr->stats_fini = 1;
  thr->endtime = clock();
  pthread_mutex_lock(&thr->exit_mutex);
  thr->active = 0;
  pthread_cond_broadcast(&thr->exit_cond);
  pthread_mutex_unlock(&thr->exit_mutex);
  if (thr->released) pthread_mutex_lock(&global_mutex);
  unlock_debug(thr);
  if (thr->tty_lock) pthread_mutex_unlock(&tty_mutex);
  pthread_mutex_unlock(&global_mutex);
  if (thr->debug) debug_thread(thr, "thread #%d exited");
#endif
}

static inline AREC *cleanact(THREAD *thr, AREC *act)
{
  if (act->lvals[0]) qmfree(thr, act->lvals[0]);
  if (act->lvals[1]) qmfree(thr, act->lvals[1]);
  return act;
}

void fini_thread(int id)
{
  THREAD *thr;
#ifndef USE_THREADS
  return;
#else
  thr = thr0+id;
  thr->sticky = 1;
  while (thr->xsp > thr->xst)
    qmfree(thr, *--thr->xsp);
  while (thr->asp > thr->ast)
    free(cleanact(thr, *--thr->asp));
  if (thr->xst) free(thr->xst);
  if (thr->ast) free(thr->ast);
  if (thr->args) free(thr->args);
  if (thr->mark) free(thr->mark);
  if (thr->sentinels) free_sentinels(thr);
  if (thr->vartb) free(thr->vartb);
  pthread_mutex_destroy(&thr->exit_mutex);
  pthread_cond_destroy(&thr->exit_cond);
  thr->used = 0;
  if (thr+1 < thr0+nthreads) {
    thr->next = nthr;
    nthr = thr;
  } else
    nthreads--;
  nused--;
#endif
}

THREAD *get_thr(void)
{
#ifdef USE_THREADS
  return (THREAD*)pthread_getspecific(thr_key);
#else
  return thr0;
#endif
}

void kill_threads(void)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_cancel(thr->id);
  pthread_mutex_unlock(&global_mutex);
#endif
}

void wait_threads(void)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used) {
	pthread_mutex_lock(&thr->exit_mutex);
	while (thr->active)
	  pthread_cond_wait(&thr->exit_cond, &thr->exit_mutex);
	pthread_mutex_unlock(&thr->exit_mutex);
      }
  pthread_mutex_lock(&global_mutex);
#endif
}

int this_thread(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  return thr-thr0;
#else
  return 0;
#endif
}

int have_lock(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  return !thr->released;
#else
  return 1;
#endif
}

void release_lock(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  if (thr > thr0) pthread_testcancel();
  unlock_debug(thr);
  pthread_mutex_unlock(&global_mutex);
  thr->released = 1;
#endif
}

void acquire_lock(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  if (init_thr) {
    /* suspend secondary threads until initializations are over */
    pthread_mutex_lock(&init_mutex);
    while (init_thr && thr != init_thr)
      pthread_cond_wait(&init_cond, &init_mutex);
    pthread_mutex_unlock(&init_mutex);
  }
  pthread_mutex_lock(&global_mutex);
  thr->released = 0;
  if (thr > thr0) pthread_testcancel();
  lock_debug(thr);
#endif
}

void start_init(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&init_mutex);
  init_thr = get_thr();
#endif
  init_mode = 1;
#ifdef USE_THREADS
  pthread_mutex_unlock(&init_mutex);
#endif
}

void end_init(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&init_mutex);
#endif
  init_mode = 0;
#ifdef USE_THREADS
  init_thr = NULL;
  pthread_cond_broadcast(&init_cond);
  pthread_mutex_unlock(&init_mutex);
#endif
}

void acquire_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  while (input_suspended)
    pthread_cond_wait(&input_cond, &input_mutex);
  acquire_tty();
  pthread_mutex_unlock(&input_mutex);
#endif
}

void release_input(void)
{
#ifdef USE_THREADS
  release_tty();
#endif
}

void suspend_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  input_suspended = 1;
  pthread_mutex_unlock(&input_mutex);
#endif
}

void resume_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  input_suspended = 0;
  pthread_cond_signal(&input_cond);
  pthread_mutex_unlock(&input_mutex);
#endif
}

void acquire_tty(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  pthread_mutex_lock(&tty_mutex);
  thr->tty_lock = 1;
#endif
}

void release_tty(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  pthread_mutex_unlock(&tty_mutex);
  thr->tty_lock = 0;
#endif
}

/* INITIALIZATION: */

/* init(), reinit(): (re)initialize the Q machine. */

/* Check the C stack direction (pilfered from the Chicken sources). A value >0
   indicates that the stack grows upward, towards higher addresses, <0 that
   the stack grows downward, towards lower addresses, and =0 that the
   direction is not known (this shouldn't happen, though).  */

static int c_stack_dir_tester(int counter, char *baseptr)
{
  if (counter < 100) {
    return c_stack_dir_tester(counter + 1, baseptr);
  } else {
    char tester;
    return &tester - baseptr;
  }
}

static int c_stack_dir(void)
{
  char basechar;
  int dir = c_stack_dir_tester(0, &basechar);
  return (dir>0)?1:(dir<0)?-1:0;
}

int stack_dir;

/* Note: These routines are only invoked from the main thread. */

static initfx(int fno)
{
  int	       *rp;
  int             rc;
  fx[fno].data.xp = fx0[fno].data.xp = NULL;
  fx[fno].refc = fx0[fno].refc = 1;
  fx[fno].fno = fx0[fno].fno = fno;
  fx[fno].type = fx0[fno].type = symtb[fno].type;
  fx[fno].argc = fx0[fno].argc = symtb[fno].argc;
  fx[fno].red =
    fno < BUILTIN && funtb[fno] && nargs[fno] == 0 ||
    symtb[fno].f && symtb[fno].argc == 0 ||
    (symtb[fno].flags & VSYM) ||
    match(thr0, fno, NULL, &rp, &rc);
  fx[fno].raw = fx0[fno].raw = fno == UNQUOTEOP || fno == FORCEOP ||
    fno == MEMOP || (symtb[fno].flags & VSYM);
  fx0[fno].red = (symtb[fno].flags & VSYM)?1:0;
  fx[fno].pipe = fx0[fno].pipe = 0;
  fx[fno].mem = fx0[fno].mem = 0;
  fx[fno].virt = fx0[fno].virt = (symtb[fno].flags & VIRT)?1:0;
}

void init(void)
{
  int 		fno;
  stack_dir = c_stack_dir();
  /* initialize the main thread */
#ifdef USE_THREADS
  init_threads();
#endif
  nthreads = nused = 0; nthr = NULL;
  if (init_thread() == -1)
    fatal("memory overflow");
  /* initialize global data */
  if (!(xblk = (XBLK *)malloc(sizeof(XBLK))) ||
      !(fx = (EXPR *)aalloc(symtbsz, sizeof(EXPR))) ||
      !(fx0 = (EXPR *)aalloc(symtbsz, sizeof(EXPR))))
    fatal("memory overflow");
  initfx(DEFOP);
  initfx(UNDEFOP);
  for (fno = BINARY; fno < symtbsz; fno++)
    initfx(fno);
  for (fno = TYPESYMS; fno < symtbsz; fno++)
    if ((symtb[fno].flags & TSYM) &&
	(symtb[fno].type && (symtb[symtb[fno].type].flags & VIRT) ||
	 matchtype(thr0, UNPARSEOP, fno)))
      symtb[fno].flags |= VIRT;
  xblk->next = NULL;
  xfreep = NULL;
  xheap = xblk->x;
  xnblks = 1;
  lastblksz = memmax % XBLKSZ;
  maxnblks = memmax/XBLKSZ+((memmax <= 0||lastblksz==0)?0:1);
  if (lastblksz == 0) lastblksz = XBLKSZ;
  brkflag = 0;
}

void reinit(void)
{
  int 		fno;
  /* reinitialze the main thread */
  if (thr0->args) free(thr0->args);
  if (!(thr0->args = (EXPR**)malloc((maxargs+1)*sizeof(EXPR*))))
    fatal("memory overflow");
  thr0->qmstat = thr0->qmstat_save = OK;
  thr0->mode = 0;
  thr0->debug = debug;
  thr0->brkdbg = brkdbg;
  thr0->brkflag = 0;
  thr0->nsig = 0;
  thr0->sigpend = thr0->sigblk = 0;
  thr0->maxexprs = thr0->nexprs = thr0->nredns = 0;
  thr0->stats_init = thr0->stats_fini = 0;
  /* reinitialize global data */
  if (fx) free(fx);
  if (fx0) free(fx0);
  if (!(fx = (EXPR *)aalloc(symtbsz, sizeof(EXPR))) ||
      !(fx0 = (EXPR *)aalloc(symtbsz, sizeof(EXPR))))
    fatal("memory overflow");
  initfx(DEFOP);
  initfx(UNDEFOP);
  for (fno = BINARY; fno < symtbsz; fno++)
    initfx(fno);
  for (fno = TYPESYMS; fno < symtbsz; fno++)
    if ((symtb[fno].flags & TSYM) && matchtype(thr0, UNPARSEOP, fno))
      symtb[fno].flags |= VIRT;
  brkflag = 0;
}

/* EXPRESSION HEAP MANAGEMENT: */

unsigned long fexprs = 0;

/* x_alloc() allocates an expression */

static EXPR *x_alloc(THREAD *thr)
{
  EXPR		*x;

  if (x = xfreep) {
    if (++thr->nexprs > thr->maxexprs) thr->maxexprs = thr->nexprs;
    xfreep = xfreep->data.xp;
    --fexprs;
#ifdef USE_THREADS
    x->sticky = 0;
    x->thrid = thr-thr0;
#endif
    return x;
  }
  if (xheap >= xblk->x+XBLKSZ) {
    if (maxnblks <= 0 || xnblks < maxnblks) {
      /* try to allocate a new block */
      XBLK	       *xblk1 = (XBLK*) malloc(sizeof(XBLK));
      if (xblk1) {
	xblk1->next = xblk;
	xblk = xblk1;
	xheap = xblk->x;
	xnblks++;
      } else
	return NULL;
    } else
      return NULL;
  }
  if (maxnblks > 0 && xnblks == maxnblks && xheap-xblk->x >= lastblksz)
    return NULL;
  else {
    x = xheap++;
    if (++thr->nexprs > thr->maxexprs) thr->maxexprs = thr->nexprs;
#ifdef USE_THREADS
    x->sticky = 0;
    x->thrid = thr-thr0;
#endif
    return x;
  }
}

/* x_free() frees an expression by inserting it into the free list */

inline
static void x_free(THREAD *thr, EXPR *x)
{
#ifdef USE_THREADS
  if (!x->sticky)
#endif
    if (thr->nexprs > 0) --thr->nexprs;
  ++fexprs;
  x->data.xp = xfreep;
  xfreep = x;
}

#if 0

/* As of Q 7.7, this code is disabled and the --gc option is not available any
   more. */

/* x_collect() implements the usual stop-and-copy garbage collector which is
   used to defrag the expression heap and return unused memory to the system
   pool. It copies the expression objects reachable from def'ed symbols and
   the stack into fresh memory. If there's not enough memory to allocate the
   new memory arena, then the process is aborted and the current memory arena
   is left unchanged. */

/* Note: This routine is only invoked from the main thread, and only if no
   other threads are currently active. */

static XBLK	       *fblk;	/* the free block list */
static XBLK	       *xblk1;	/* the new heap */
static EXPR	       *xheap1;	/* new heap ptr */

static void x_collect(void)
{
  XBLK *fblk1;
  int fno;
  EXPR **xp;
  int lastblksz, xnblks1;
  unsigned long nheap = ((unsigned long)(xnblks-1))*XBLKSZ+(xheap-xblk->x);
  unsigned long mexprs = nheap-fexprs;

  if (gc_v) {
    flush_shift();
    printf("garbage collecting ... ");
    fflush(stdout);
  }

  /* compute the size of the new memory arena */
  lastblksz = mexprs % XBLKSZ;
  xnblks1 = mexprs/XBLKSZ+((lastblksz==0)?0:1);
  if (xnblks1 <= 0) xnblks1 = 1;

  /* create the new memory arena */
  if ((fblk = fblk1 = (XBLK *)malloc(sizeof(XBLK)))) {
    int i;
    for (i = 1; i < xnblks1; i++)
      if ((fblk1->next = (XBLK *)malloc(sizeof(XBLK))))
	fblk1 = fblk1->next;
      else {
	for (; fblk; fblk = fblk1) {
	  fblk1 = fblk->next;
	  free(fblk);
	}
	break;
      }
  }
  if (fblk)
    fblk1->next = NULL;
  else {
    if (gc_v) printf("failed (not enough memory)\n");
    return;
  }

  /* reset reference counts of preallocated function nodes: */
  for (fno = BINARY; fno < symtbsz; fno++)
    fx[fno].refc = fx0[fno].refc = 1;

  /* initialize block pointers */
  xblk1 = fblk; xheap1 = xblk1->x;
  fblk = fblk->next; xblk1->next = NULL;

  /* copy all referenced objects into the free memory arena */
  if (mexprs > 0) {
    for (fno = INPUTOP; fno < symtbsz+tmptbsz; fno++)
      if (symtb[fno].x)
	symtb[fno].x = (void*) x_copy(symtb[fno].x);
    for (xp = thr0->xst; xp < thr0->xsp; xp++)
      *xp = x_copy(*xp);
    xcopy_cleanup();
  }

  /* discard any remaining free blocks (shouldn't happen but Murphy knows) */
  for (; fblk; fblk = fblk1) {
    fblk1 = fblk->next;
    free(fblk);
  }

  /* discard the old memory arena */
  for (fblk = xblk; fblk; fblk = fblk1) {
    fblk1 = fblk->next;
    free(fblk);
  }

  /* reinitialize the heap: */
  xblk = xblk1;
  xfreep = NULL;
  xheap = xheap1;
  xnblks = xnblks1;
  fexprs = 0;

  if (gc_v) printf("done (moved %ld cells)\n", mexprs);
}

#define MARK -99	/* refc value used to mark "broken hearts" */

#if 0

/* This routine is now implemented non-recursively below. */

static EXPR *x_copy(EXPR *x)
{
  if (x->refc == MARK) {
    /* this is a "broken heart" which has been moved already
       to the location indicated by xp */
    x->data.xp->refc++;
    return x->data.xp;
  } else if (x->fno >= BINARY && x->fno < symtbsz) {
    /* preallocated function node, doesn't have to be copied */
    x->refc++;
    return x;
  } else {
    if (xheap1 >= xblk1->x+XBLKSZ) {
      XBLK *xblk2;
      /* get a new block */
      if (!fblk) fatal(qmmsg[THIS_CANT_HAPPEN]);
      xblk2 = xblk1; xblk1 = fblk;
      fblk = fblk->next; xblk1->next = xblk2;
      xheap1 = xblk1->x;
    }
    /* move x to new location */
    *xheap1 = *x;
    xheap1->refc = 1;
    /* mark x as a broken heart */
    x->refc = MARK;
    x->data.xp = xheap1++;
    /* copy args recursively */
    switch (x->fno) {
    case CONSOP: case PAIROP: case APPOP:
      x->data.xp->data.args.x1 = x_copy(x->data.xp->data.args.x1);
      x->data.xp->data.args.x2 = x_copy(x->data.xp->data.args.x2);
      break;
    case VECTOP: {
      int i, n = x->data.xp->data.vect.n;
      for (i = 0; i < n; i++)
	x->data.xp->data.vect.xv[i] = x_copy(x->data.xp->data.vect.xv[i]);
      break;
    }
    }
    /* return the moved object: */
    return x->data.xp;
  }
}

static void xcopy_cleanup(void)
{
}

#else

/* This operation should be "failsafe", so we actually implement it
   non-recursively. */

static EXPR **xstk = NULL;

static int xstkp = 0, xstka = 0;

static EXPR *x_copy(EXPR *x)
{
  EXPR *y;
  int mark = xstkp;
 loop:
  if (x->refc == MARK) {
    /* this is a "broken heart" which has been moved already
       to the location indicated by xp */
    x->data.xp->refc++;
    y = x->data.xp;
    goto pop;
  } else if (x->fno >= BINARY && x->fno < symtbsz) {
    /* preallocated function node, doesn't have to be copied */
    x->refc++;
    y = x;
    goto pop;
  } else {
    if (xheap1 >= xblk1->x+XBLKSZ) {
      XBLK *xblk2;
      /* get a new block */
      if (!fblk) fatal(qmmsg[THIS_CANT_HAPPEN]);
      xblk2 = xblk1; xblk1 = fblk;
      fblk = fblk->next; xblk1->next = xblk2;
      xheap1 = xblk1->x;
    }
    /* move x to new location */
    *xheap1 = *x;
    xheap1->refc = 1;
    /* mark x as a broken heart */
    x->refc = MARK;
    y = x->data.xp = xheap1++;
    /* copy args recursively */
    switch (y->fno) {
    case CONSOP: case PAIROP: case APPOP:
      if (xstkp >= xstka) {
	if (xstka >= INT_MAX ||
	    !(xstk = xstka?
	      realloc(xstk, (xstka+10240)*sizeof(EXPR*)):
	      malloc(10240*sizeof(EXPR*))))
	  fatal("memory overflow");
	else
	  xstka += 10240;
      }
      xstk[xstkp++] = x;
      x = x->data.args.x1;
      goto loop;
    case VECTOP: {
      int i, n = y->data.vect.n;
      for (i = 0; i < n; i++)
	y->data.vect.xv[i] = x_copy(y->data.vect.xv[i]);
      goto pop;
    }
    default:
    pop:
      while (xstkp > mark && x == xstk[xstkp-1]->data.args.x2) {
	xstk[xstkp-1]->data.xp->data.args.x2 = y;
	x = xstk[--xstkp];
	y = xstk[xstkp]->data.xp;
      }
      if (xstkp > mark) {
	xstk[xstkp-1]->data.xp->data.args.x1 = y;
	x = xstk[xstkp-1]->data.args.x2;
	goto loop;
      }
    }
  }
  /* return the moved object: */
  return y;
}

static void xcopy_cleanup(void)
{
  /* collect temp stack */
  free(xstk);
  xstk = NULL;
  xstkp = xstka = 0;
}

#endif

#endif

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

static inline char *
icfromutf8(iconv_t ic[2], char *s)
{
  if (ic[1] == (iconv_t)-2)
    return NULL;
  if (ic[1] == (iconv_t)-1)
    return s?strdup(s):NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return s?strdup(s):NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

#endif

/* qmfree( x ) decrements the reference count for the expression pointed to by
   x and if the count drops to zero frees the heap space occupied by x. In
   order to achieve robustness and better performance this is implemented
   (mostly) non-recursively using an explicit stack built with the xp field in
   the EXPR data structure. */

void qmfree(THREAD *thr, EXPR *x)
{
  EXPR	       *xp = NULL, *x1;
  if (!x) return;
  do {
    if (!--x->refc) {
      switch (x->fno) {
      case CONSOP: case PAIROP: case APPOP:
      push:
	x1 = x->data.args.x1;
	x->data.xp = xp;
	xp = x;
	x = x1;
	break;
      case STRVALOP:
	free(x->data.s);
	goto pop;
      case INTVALOP:
	mpz_clear(x->data.z);
	goto pop;
      case FILEVALOP:
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
	if (x->data.fargs.ic[0] != (iconv_t)-2 &&
	    x->data.fargs.ic[0] != (iconv_t)-1) {
	  iconv_close(x->data.fargs.ic[0]);
	}
	if (x->data.fargs.ic[1] != (iconv_t)-2 &&
	    x->data.fargs.ic[1] != (iconv_t)-1) {
	  /* In a stateful encoding we might have to emit a terminating shift
	     sequence. */
	  char *s = icfromutf8(x->data.fargs.ic, NULL), *t = s;
	  if (t) {
	    while (*s) putc(*s++, x->data.fp);
	    free(t);
	  }
	  iconv_close(x->data.fargs.ic[1]);
	}
#endif
	if (x->pipe)
	  pclose(x->data.fp);
	else
	  fclose(x->data.fp);
	goto pop;
      case USRVALOP:
	/* invoke external destructor */
	if (x->type)
	  if (symtb[x->type].f) {
	    void (*f)() = symtb[x->type].f;
	    (*f) (x->data.vp);
	  } else if (x->data.vp)
	    free(x->data.vp);
	goto pop;
      case VECTOP: {
	int i, n = x->data.vect.n;
	for (i = 0; i < n; i++)
	  qmfree(thr, x->data.vect.xv[i]);
	if (x->data.vect.xv)
	  free(x->data.vect.xv);
      }
      default:
      pop:
	while (xp && x == xp->data.args.x2) {
	  if (!x->refc) x_free(thr, x);
	  x = xp;
	  xp = x->data.xp;
	}
	if (!x->refc) x_free(thr, x);
	if (xp)
	  x = xp->data.args.x2;
      }
    } else {
#ifdef USE_THREADS
      if (thr->sticky && x->thrid == thr-thr0)
	x->sticky = 1;
#endif
      goto pop;
    }
  } while (xp);
}

/* qmnew( x ) counts a new reference to the expression pointed to by x;
   returns: x. */

inline
EXPR *qmnew(EXPR *x)
{
  if (x) {
    x->refc++;
    return x;
  } else
    return NULL;
}

/* qmsentinel(x) registers a "sentinel" which will be evaluated in a safe
   context asap. */

void qmsentinel(THREAD *thr, EXPR *x)
{
  add_sentinel(thr, x);
}

/* clear(): clear main stack and reinitialize pointers and status. */

/* Note: This routine is only invoked from the main thread. */

void clear(int force_gc)
{
  int defxstsz = XSTSZ, defastsz = ASTSZ; 

  /* free the stack first, to reclaim open file handles and dynamic data */
  while (thr0->xsp > thr0->xst)
    qmfree(thr0, *--thr0->xsp);

  while (thr0->asp > thr0->ast)
    free(cleanact(thr0, *--thr0->asp));

  /* perform garbage collection on the expression heap (only do this if no
     other threads are currently active, and if no sentinels are waiting to be
     processed) */
#ifdef USE_THREADS
  if (nused <= 1 && !thr0->sentinels)
#else
  if (!thr0->sentinels)
#endif
    {
      unsigned long d, dmod;

      d = fexprs/XBLKSZ; dmod = fexprs%XBLKSZ;
      if (dmod > 0 && dmod + (XBLKSZ-(xheap-xblk->x)) > XBLKSZ) d++;
#if 0
      if (force_gc || gc_flag && d > gc_tol*xnblks) x_collect();
#endif
    }

  /* Reallocate evaluation and activation stacks to their original sizes such
     that the evaluation loop doesn't get hooked with the Q machine having
     claimed all available memory for the stacks. */

  thr0->maxxstsz = thr0->maxastsz = stackmax;

  if (thr0->maxxstsz > 0 && thr0->maxxstsz < defxstsz)
    defxstsz = thr0->maxxstsz;
  if (thr0->maxastsz > 0 && thr0->maxastsz < defastsz)
    defastsz = thr0->maxastsz;

  if (thr0->xstsz > defxstsz) {
    thr0->xst = (EXPR**)realloc(thr0->xst, defxstsz*sizeof(EXPR*));
    thr0->xstsz = defxstsz;
  }
  if (thr0->astsz > defastsz) {
    thr0->ast = (AREC**)realloc(thr0->ast, defastsz*sizeof(AREC*));
    thr0->astsz = defastsz;
  }
  if (!thr0->xst || !thr0->ast)
    /* This shouldn't happen, but Murphy knows ... */
    fatal(qmmsg[THIS_CANT_HAPPEN]);

  /* reinitialize status variables */
  
  if (thr0->mark) free(thr0->mark);
  thr0->mark = thr0->markp = NULL;
  thr0->marksz = 0;
  
  thr0->xsp = thr0->xst;
  thr0->asp = thr0->ast;
  thr0->qmstat = thr0->qmstat_save = OK;
  thr0->mode = 0;
  thr0->debug = debug;
  thr0->brkdbg = brkdbg;
  thr0->brkflag = 0;
  thr0->nsig = 0;
  thr0->sigpend = thr0->sigblk = 0;

  if (thr0->sentinels) process_sentinels(thr0);
}

/* SIGNAL HANDLING */

static volatile int defer_sig = 0;
static volatile int saved_sig = 0;

RETSIGTYPE sig_handler(int sig)
{
  THREAD *thr;
  SIGHANDLER_RESTORE(sig, sig_handler);
  thr = get_thr();
  if (!thr || sig <= 0 || sig > NSIGNALS)
    SIGHANDLER_RETURN(0);
  /* prevent race conditions */
  if (defer_sig)
    saved_sig = sig;
  else if (thr->nsig < NSIGNALS) {
    int i;
    for (i = 0; i < thr->nsig && thr->sig[i] != sig; i++) ;
    if (i >= thr->nsig)
      /* enqueue the signal */
      thr->sig[thr->nsig++] = sig;
  }
  SIGHANDLER_RETURN(0);
}

inline
static int get_sig(THREAD *thr)
{
  int sig;
  defer_sig++;
  if (thr->sigblk || !thr->nsig) {
    defer_sig--;
    return 0;
  }
  sig = *thr->sig;
  memmove(thr->sig+1, thr->sig, --thr->nsig);
  defer_sig--;
  if (!defer_sig && saved_sig) {
    raise(saved_sig);
    saved_sig = 0;
  }
  return sig;
}

/* EXPRESSION CONSTRUCTORS: */

/* In case one of these operations fails, NULL is returned and qmstat is
   set to MEM_OVF. */

EXPR *intexpr(THREAD *thr, long i)
{
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_si(z, i); /* this shouldn't fail */
    return mpzexpr(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *uintexpr(THREAD *thr, unsigned long i)
{
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_ui(z, i); /* this shouldn't fail */
    return mpzexpr(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *mpzexpr(THREAD *thr, mpz_t z)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = INTVALOP;
    x->type = INTTYPE;
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    memcpy(x->data.z, z, sizeof(mpz_t));
  } else {
    mpz_clear(z);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *floatexpr(THREAD *thr, double f)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FLOATVALOP;
    x->type = FLOATTYPE;
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    x->data.f = f;
  } else
    thr->qmstat = MEM_OVF;
  return x;
}

#ifdef HAVE_UNICODE
static inline long
u8decode(char *s)
{
  size_t n;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0)
    return (unsigned char)s[0];
  for (n = 0; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)*s) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0;
      if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1 && *s == 0)
    return c;
  else
    return -1;
}
#endif

EXPR *strexpr(THREAD *thr, char *s)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = STRVALOP;
#ifdef HAVE_UNICODE
    x->type = (u8decode(s)>=0)?CHARTYPE:STRTYPE;
#else
    x->type = (s[0]&&!s[1])?CHARTYPE:STRTYPE;
#endif
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    x->data.s = s;
  } else {
    free(s);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *fileexpr(THREAD *thr, FILE *fp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FILEVALOP;
    x->type = FILETYPE;
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    x->data.fp = fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    x->data.fargs.ic[0] = x->data.fargs.ic[1] = (iconv_t)-2;
#endif
  } else {
    fclose(fp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *pipeexpr(THREAD *thr, FILE *fp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FILEVALOP;
    x->type = FILETYPE;
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 1;
    x->virt = 0;
    x->data.fp = fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    x->data.fargs.ic[0] = x->data.fargs.ic[1] = (iconv_t)-2;
#endif
  } else {
    pclose(fp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *vectexpr(THREAD *thr, int n, EXPR **xv)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    int i;
    x->refc = 0;
    x->fno = VECTOP;
    x->type = TUPLETYPE;
    x->argc = 0;
    x->red = x->raw = 0;
    for (i = 0; i < n; i++)
      if (xv[i]->red) {
	x->red = 1;
	break;
      }
    for (i = 0; i < n; i++)
      if (xv[i]->raw) {
	x->raw = 1;
	break;
      }
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    x->data.vect.n = n;
    x->data.vect.xv = xv;
  } else {
    if (xv) {
      int i;
      for (i = 0; i < n; i++)
	qmfree(thr, xv[i]);
      free(xv);
    }
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *usrexpr(THREAD *thr, int type, void *vp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = USRVALOP;
    x->type = type;
    x->argc = 0;
    x->red = x->raw = 0;
    x->mem = 0;
    x->pipe = 0;
    x->virt = 0;
    x->data.vp = vp;
  } else {
    if (type)
      if (symtb[type].f) {
	void (*f)() = symtb[type].f;
	(*f) (vp);
      } else if (vp)
	free(vp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *funexpr(THREAD *thr, int fno)
{
  if (fno < symtbsz && !(symtb[fno].flags&VSYM))
    return (thr->mode?fx:fx0)+fno;
  else {
    EXPR	       *x = x_alloc(thr);
    if (x) {
      x->refc = 0;
      x->fno = fno;
      x->type = symtb[fno].type;
      x->argc = symtb[fno].argc;
      x->red = /*symtb[fno].flags&VSYM?1:*/thr->mode;
      x->raw = (symtb[fno].flags & VSYM)?thr->mode:0;
      x->mem = 0;
      x->pipe = 0;
      x->virt = (symtb[fno].flags & VIRT)?1:0;
    } else
      thr->qmstat = MEM_OVF;
    return x;
  }
}

static EXPR *funexpr2(THREAD *thr, int fno)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = fno;
    x->type = symtb[fno].type;
    x->argc = symtb[fno].argc;
    x->red = /*symtb[fno].flags&VSYM?1:*/thr->mode;
    x->raw = fno == UNQUOTEOP || fno == FORCEOP || fno == MEMOP ||
      ((symtb[fno].flags & VSYM)?thr->mode:0);
    x->mem = 0;
    x->pipe = 0;
    x->virt = (symtb[fno].flags & VIRT)?1:0;
  } else
    thr->qmstat = MEM_OVF;
  return x;
}

#if 1
#define get_argv(x) (((x)->fno==APPOP)?(x)->data.args.argv:symtb[(x)->fno].argv)
#else
static inline unsigned long get_argv(EXPR *x)
{
  return (x->fno==APPOP)?x->data.args.argv:symtb[x->fno].argv;
}
#endif

EXPR *consexpr(THREAD *thr, int fno, EXPR *x1, EXPR *x2)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = fno;
    if (fno == APPOP) {
      unsigned long argv;
      switch (x1->fno) {
      case FLIPOP:
	argv = get_argv(x2);
	x->data.args.argv = ((argv & 1) << 1) | ((argv & 2) >> 1);
	break;
      case COMPOP: case RAPPOP:
	argv = get_argv(x2);
	x->data.args.argv = argv & 1;
	break;
      case APPOP:
	if (x1->data.args.x1->fno == COMPOP) {
	  x->data.args.argv = x1->data.args.argv;
	  argv = get_argv(x2);
	  x->data.args.argv |= argv & 1;
	  break;
	}
	/* else part falls through to default case */
      default:
	argv = get_argv(x1);
	x->data.args.argv = argv >> 1;
	break;
      }
      if (x1->argc) {
	x->type = x1->type;
	x->argc = x1->argc-1;
	x->virt = x1->virt;
      } else {
	x->type = 0;
	x->argc = 0;
	x->virt = 0;
      }
      x->red = thr->mode || x1->red || !(get_argv(x1)&1) && x2->red;
      x->raw = x1->raw || x2->raw;
    } else {
      x->data.args.argv = 0;
      x->type = fno==CONSOP?LISTTYPE:fno==PAIROP?TUPLETYPE:0;
      x->argc = 0;
      x->red = x1->red || x2->red;
      x->raw = x1->raw || x2->raw;
      x->virt = 0;
    }
    x->mem = 0;
    x->pipe = 0;
    x->data.args.x1 = qmnew(x1);
    x->data.args.x2 = qmnew(x2);
  } else
    thr->qmstat = MEM_OVF;
  return x;
}

static EXPR *memexpr(THREAD *thr, EXPR *y)
{
  EXPR *x;
  int fno = y->fno;
  if (y->mem || !y->red || fno < RESERVED && fno != VECTOP)
    return y;
  else if (y->refc == 1) {
    y->mem = 1;
    return y;
  } else if (fno == VECTOP) {
    int n = y->data.vect.n;
    EXPR **xv1 = malloc(n*sizeof(EXPR*)), **xv = y->data.vect.xv;
    if (xv1) {
      int i;
      for (i = 0; i < n; i++)
	xv1[i] = qmnew(xv[i]);
      x = vectexpr(thr, n, xv1);
    } else {
      thr->qmstat = MEM_OVF;
      x = NULL;
    }
  } else if (fno < BINARY)
    x = consexpr(thr, fno, y->data.args.x1, y->data.args.x2);
  else
    x = funexpr2(thr, fno);
  if (x) x->mem = 1;
  return x;
}

/* STACK OPERATIONS: */

/* stack_avail() checks whether there is room on the stack; the stack
   is enlarged if required (in case this fails qmstat is set to XST_OVF
   or MEM_OVF) */

static int stack_avail(THREAD *thr)
{
  if (thr->maxxstsz > 0 && thr->xsp - thr->xst >= thr->maxxstsz) {
    thr->qmstat = XST_OVF;
    return (0);
  } else if (thr->xsp - thr->xst == thr->xstsz) {
    EXPR **xst1;
    int n = XSTSZ/4;
    if (thr->maxxstsz > 0 && thr->xstsz+n > thr->maxxstsz)
      n = thr->maxxstsz-thr->xstsz;
    if (n <= 0 ||
	!(xst1 = (EXPR**)arealloc(thr->xst, thr->xstsz, n,
				  sizeof(EXPR*)))) {
      thr->qmstat = MEM_OVF;
      return (0);
    } else {
      thr->xst = xst1;
      thr->xsp = thr->xst+thr->xstsz;
      thr->xstsz += n;
    }
  }
  return (1);
}

/* push( x ): push a copy of expression x onto the stack; returns:
   zero iff error (cf. qmstat). */

int push(THREAD *thr, EXPR *x)
{
  if (stack_avail(thr)) {
    *thr->xsp++ = qmnew(x);
    return (1);
  } else
    return (0);
}

/* pushint(): push an integer value */

int pushint(THREAD *thr, long i)
{
  if (stack_avail(thr)) {
    EXPR	       *x = intexpr(thr, i);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushuint(): push an unsigned integer value */

int pushuint(THREAD *thr, unsigned long i)
{
  if (stack_avail(thr)) {
    EXPR	       *x = uintexpr(thr, i);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushmpz(): push a big integer value */

int pushmpz(THREAD *thr, mpz_t z)
{
  if (stack_avail(thr)) {
    EXPR	       *x = mpzexpr(thr, z);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfloat(): push a float value */

int pushfloat(THREAD *thr, double f)
{
  if (stack_avail(thr)) {
    EXPR	       *x = floatexpr(thr, f);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushstr(): push a string value (STRVALOP) */

int pushstr(THREAD *thr, char *s)
{
  if (stack_avail(thr)) {
    EXPR	       *x = strexpr(thr, s);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfile(): push a file value */

int pushfile(THREAD *thr, FILE *fp)
{
  if (stack_avail(thr)) {
    EXPR	       *x = fileexpr(thr, fp);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushpipe(): push a file value associated with a pipe */

int pushpipe(THREAD *thr, FILE *fp)
{
  if (stack_avail(thr)) {
    EXPR	       *x = pipeexpr(thr, fp);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushvect(): push a vector (fno = -n) */

int pushvect(THREAD *thr, int n, EXPR **xv)
{
  if (stack_avail(thr)) {
    EXPR	       *x = vectexpr(thr, n, xv);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfun(fno): push a term constructed from fno. */

int pushfun(THREAD *thr, int fno)
{
  if (stack_avail(thr)) {
    EXPR	       *x;
    switch (fno) {
    case PAIROP:
      if (thr->xsp[-1]->fno == VECTOP || thr->xsp[-1]->fno == VOIDOP) {
	int i, n = (thr->xsp[-1]->fno == VOIDOP)?0:thr->xsp[-1]->data.vect.n;
	if (n && thr->xsp[-1]->refc == 1) {
	  /* prepend a new element to an existing vector */
	  EXPR **xv = realloc(thr->xsp[-1]->data.vect.xv, (n+1)*sizeof(EXPR*));
	  if (!xv) {
	    thr->qmstat = MEM_OVF;
	    return 0;
	  } else {
	    for (i = n; i > 0; i--) xv[i] = xv[i-1];
	    xv[0] = thr->xsp[-2];
	    thr->xsp[-1]->data.vect.xv = xv;
	    thr->xsp[-1]->data.vect.n++;
	    thr->xsp[-1]->red = thr->xsp[-1]->red || thr->xsp[-2]->red;
	    thr->xsp[-1]->raw = thr->xsp[-1]->raw || thr->xsp[-2]->raw;
	    thr->xsp[-2] = thr->xsp[-1];
	    thr->xsp--;
	    return 1;
	  }
	} else {
	  /* construct a new vector */
	  EXPR **xv = malloc((n+1)*sizeof(EXPR*));
	  if (!xv) {
	    thr->qmstat = MEM_OVF;
	    return 0;
	  } else {
	    xv[0] = qmnew(thr->xsp[-2]);
	    for (i = 0; i < n; i++)
	      xv[i+1] = qmnew(thr->xsp[-1]->data.vect.xv[i]);
	    if (x = vectexpr(thr, n+1, xv)) {
	      qmfree(thr, *--thr->xsp); qmfree(thr, *--thr->xsp);
	      *thr->xsp++ = qmnew(x);
	      return 1;
	    } else
	      return 0;
	  }
	}
      }
      /* else part falls through to the following case */
    case CONSOP: case APPOP:
      if (x = consexpr(thr, fno, thr->xsp[-2], thr->xsp[-1])) {
	thr->xsp[-2]->refc--, thr->xsp[-1]->refc--;
	thr->xsp--;
	thr->xsp[-1] = qmnew(x);
	return 1;
      } else
	return 0;
    default:
      if (x = funexpr(thr, fno)) {
	*thr->xsp++ = qmnew(x);
	return 1;
      } else
	return 0;
    }
  } else
    return 0;
}

/* ARGUMENTS: */

/* pushlval(): determine a left-hand side or local variable value and push it
   on the stack. xbp denotes the (relative) base pointer pointing to the
   arguments and local variables on the stack. */

static int pushlval(THREAD *thr, int fno, EXPR *lvals[2],
		    long xbp, byte offs, byte plen, PATH p)
{
  EXPR           *x;
  int             i = offs?0:1;
  
  if (offs)
    if (fno == APPOP)
      x = thr->xst[xbp+offs+1];
    else
      x = thr->xst[xbp+offs-1];
  else
    x = lvals[getpath(p, 0)];
  for (; i < plen; i++) {
    if (x->fno == VECTOP) {
      int n = x->data.vect.n, k = 0;
      while (i < plen && getpath(p, i) == 1)
	i++, k++;
      if (i < plen)
	/* assert: getpath(p,i)==0, k is the current element */
	x = x->data.vect.xv[k];
      else if (k == n)
	/* end of the vector has been reached */
	return pushfun(thr, VOIDOP);
      else {
	/* copy vector of remaining elements (assert: n>k>0) */
	EXPR **xv = malloc((n-k)*sizeof(EXPR*));
	int j;
	if (!xv) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	for (j = k; j < n; j++)
	  xv[j-k] = qmnew(x->data.vect.xv[j]);
	return pushvect(thr, n-k, xv);
      }
    } else if (getpath(p, i) == 0)
      x = x->data.args.x1;
    else
      x = x->data.args.x2;
  }
  return push(thr, x);
}

/* GLOBAL ENVIRONMENT: */

/* setvar(): associate a (variable) symbol with an expression */

int setvar(int vno, EXPR *x)
{
  if (!(symtb[vno].flags & VSYM)) {
    thr0->qmstat = BAD_DEF;
    return 0;
  } else if (symtb[vno].x && (symtb[vno].flags & CST)) {
    thr0->qmstat = x?BAD_REDEF:BAD_UNDEF;
    return 0;
  } else {
    if (symtb[vno].x || x)
      symtb[vno].flags |= MODIF;
    else
      symtb[vno].flags &= ~MODIF;
    qmfree(thr0, symtb[vno].x);
    symtb[vno].x = (void*) qmnew(x);
    return 1;
  }
}

/* FUNCTION CALLS AND EXPRESSION EVALUATION: */

/* retval(): collect arguments and local variables and set the return value
   for a function application. */

static inline int retval(THREAD *thr, EXPR **xbp)
{
  EXPR **xp;
  for (xp = xbp; xp < thr->xsp-1; xp++)
    qmfree(thr, *xp);
  *xbp = thr->xsp[-1];
  thr->xsp = xbp+1;
  return 1;
}

/* pushact(): push an activation record on the stack. */

static inline int pushact(THREAD *thr,
			  int fno, EXPR *lvals[2], int *rp, int rc,
			  OPREC *ip, long xbp, int modno, int lineno,
			  int info_addr, byte info_offs)
{
  if (thr->asp-thr->ast == thr->astsz) {
    AREC **ast1;
    int n = ASTSZ/4;
    if (thr->maxastsz > 0 && thr->astsz+n > thr->maxastsz)
      n = thr->maxastsz-thr->astsz;
    if (n <= 0 ||
	!(ast1 = (AREC**)arealloc(thr->ast, thr->astsz, n,
				  sizeof(AREC*)))) {
      thr->qmstat = AST_OVF;
      return (0);
    } else {
      thr->ast = ast1;
      thr->asp = thr->ast+thr->astsz;
      thr->astsz += n;
    }
  }
  if (!(*thr->asp = (AREC*)malloc(sizeof(AREC)))) {
    thr->qmstat = MEM_OVF;
    return (0);
  }
  (*thr->asp)->fno = fno;
  (*thr->asp)->lvals[0] = lvals[0];
  (*thr->asp)->lvals[1] = lvals[1];
  (*thr->asp)->rp = rp;
  (*thr->asp)->rc = rc;
  (*thr->asp)->ip = ip;
  (*thr->asp)->xbp = xbp;
  (*thr->asp)->modno = modno;
  (*thr->asp)->lineno = lineno;
  (*thr->asp)->info_addr = info_addr;
  (*thr->asp)->info_offs = info_offs;
  thr->asp++;
  return (1);
}

/* popact(): pop an activation record from the stack. */

static inline void popact(THREAD *thr,
			  int *fno, EXPR *lvals[2], int **rp, int *rc,
			  OPREC **ip, long *xbp, int *modno, int *lineno,
			  int *info_addr, byte *info_offs)
{
  thr->asp--;
  *fno = (*thr->asp)->fno;
  lvals[0] = (*thr->asp)->lvals[0];
  lvals[1] = (*thr->asp)->lvals[1];
  *rp = (*thr->asp)->rp;
  *rc = (*thr->asp)->rc;
  *ip = (*thr->asp)->ip;
  *xbp = (*thr->asp)->xbp;
  *modno = (*thr->asp)->modno;
  *lineno = (*thr->asp)->lineno;
  *info_addr = (*thr->asp)->info_addr;
  *info_offs = (*thr->asp)->info_offs;
  free(*thr->asp);
}

/* interface to external functions */

static int dl_funcall(THREAD *thr, EXPR* (*f)(), int argc)
{
  EXPR **argv = malloc((argc+1)*sizeof(EXPR*));
  if (argv) {
    bool actmode = thr->mode;
    EXPR* x;
    memcpy(argv, thr->args, (argc+1)*sizeof(EXPR*));
    thr->mode = 1;
    x = (*f) (argc, argv);
    thr->mode = actmode;
    free(argv);
    if (x)
      return push(thr, x);
    else
      return 0;
  } else {
    thr->qmstat = MEM_OVF;
    return 0;
  }
}

/* evalb(): evaluate a built-in (or external) function and perform environment
   variable replacement. */

#define have_fun(fno,type,n) fno < BUILTIN && \
  (funtb[fno] != NULL && nargs[fno] == n || type && fno == USRVALOP && \
   funtb[type] != NULL && nargs[type] == n) || \
  symtb[fno].f && symtb[fno].argc == n

#define exec_fun(thr,fno,type,argc) fno < BUILTIN && \
  (funtb[fno] != NULL && (*funtb[fno]) (thr) || type && fno == USRVALOP && \
   funtb[type] != NULL && (*funtb[type]) (thr)) || \
  symtb[fno].f && dl_funcall(thr, symtb[fno].f, argc)

static int evalb(THREAD *thr, int fno)
{
  int             n, *rp, rc;
  EXPR           *x;

  switch (fno) {
  case CONSOP:
  case PAIROP:
    /* constructor: */
    return (pushfun(thr, fno));
  case APPOP:
    /* application: */
    /* walk down the spine, count arguments: */
    for (n = 1, x = thr->xsp[-2]; n < maxargs && x->fno == APPOP; n++)
      x = x->data.args.x1;
    if (have_fun(x->fno, x->type, n)) {
      int argc = n;
      /* set arguments: */
      thr->self = x;
      thr->args[n] = NULL;
      thr->args[n - 1] = thr->xsp[-1];
      for (x = thr->xsp[-2]; n > 1; x = x->data.args.x1) {
	n--;
	thr->args[n - 1] = x->data.args.x2;
      }
      /* execute function: */
      if (exec_fun(thr, x->fno, x->type, argc)) {
	if (symtb[x->fno].flags&PROF) symtb[x->fno].nredns++;
	thr->nredns++;
	if (thr->debug) reduction(thr, APPOP, thr->xsp-thr->xst-3);
	retval(thr, thr->xsp-3);
	return (1);
      } else
	return (0);
    } else
      return (0);
  case FALSEOP:
  case TRUEOP:
  case NILOP:
  case VOIDOP:
    /* built-in constant: */
    return (pushfun(thr, fno));
  default:
    if (symtb[fno].x)
      /* defined symbol: */
      return (push(thr, symtb[fno].x));
    else if (have_fun(fno, 0, 0)) {
      /* parameterless builtin or external function: */
      thr->self = thr->args[0] = NULL;
      if (exec_fun(thr, fno, 0, 0)) {
	if (symtb[fno].flags&PROF) symtb[fno].nredns++;
	thr->nredns++;
	if (thr->debug) reduction(thr, fno, thr->xsp-thr->xst-1);
	return (1);
      } else
	return (0);
    } else
      /* other symbol: */
      return (0);
  }
}

/* evalb_with_frame constructs an extra stack frame for a user rule calling a
   builtin or external function, to facilitate debugging in nested calls to
   evalu */

static evalb_with_frame(THREAD *thr, 
			int fno, int _fno, EXPR *lvals[2], int *rp, int rc,
			OPREC *ip, long xbp, int modno, int lineno,
			int info_addr, byte info_offs)
{
  int           n;
  EXPR         *x;

  switch (fno) {
  case CONSOP:
  case PAIROP:
    /* constructor: */
    return (pushfun(thr, fno));
  case APPOP:
    /* application: */
    /* walk down the spine, count arguments: */
    for (n = 1, x = thr->xsp[-2]; n < maxargs && x->fno == APPOP; n++)
      x = x->data.args.x1;
    if (have_fun(x->fno, x->type, n)) {
      int argc = n;
      /* set arguments: */
      thr->self = x;
      thr->args[n] = NULL;
      thr->args[n - 1] = thr->xsp[-1];
      for (x = thr->xsp[-2]; n > 1; x = x->data.args.x1) {
	n--;
	thr->args[n - 1] = x->data.args.x2;
      }
      /* execute function: */
      if (!pushact(thr, _fno, lvals, rp, rc, ip, xbp, modno, lineno,
		   info_addr, info_offs))
	return 0;
      if (exec_fun(thr, x->fno, x->type, argc)) {
	free(*--thr->asp);
	if (symtb[x->fno].flags&PROF) symtb[x->fno].nredns++;
	thr->nredns++;
	if (thr->debug) reduction(thr, APPOP, thr->xsp-thr->xst-3);
	retval(thr, thr->xsp-3);
	return (1);
      } else {
	free(*--thr->asp);
	return (0);
      }
    } else
      return (0);
  case FALSEOP:
  case TRUEOP:
  case NILOP:
  case VOIDOP:
    /* built-in constant: */
    return (pushfun(thr, fno));
  default:
    if (symtb[fno].x)
      /* defined symbol: */
      return (push(thr, symtb[fno].x));
    else if (have_fun(fno, 0, 0)) {
      /* parameterless builtin or external function: */
      thr->self = thr->args[0] = NULL;
      if (!pushact(thr, _fno, lvals, rp, rc, ip, xbp, modno, lineno,
		   info_addr, info_offs))
	return 0;
      if (exec_fun(thr, fno, 0, 0)) {
	free(*--thr->asp);
	if (symtb[fno].flags&PROF) symtb[fno].nredns++;
	thr->nredns++;
	if (thr->debug) reduction(thr, fno, thr->xsp-thr->xst-1);
	return (1);
      } else {
	free(*--thr->asp);
	return (0);
      }	
    } else
      /* other symbol: */
      return (0);
  }
}

static int eval_internal(THREAD *thr, EXPR *x);

static int evalr(THREAD *thr, int *fno)
{
  EXPR *x = thr->xsp[-1];
  if (!x->red) {
    *fno = 0;
    return 1;
  } else if (x->mem) {
    int res;
    thr->xsp--;
    res = eval_internal(thr, x);
    qmfree(thr, x);
    *fno = 0;
    return res;
  }
  thr->xsp--;
  switch (x->fno) {
  case CONSOP:
  case PAIROP: {
    /* we do this non-recursively, to prevent stack overflows */
    EXPR *h = x->data.args.x1, *t = x->data.args.x2;
    int n = 0, res;
    while ((res = eval_internal(thr, h)) && ++n && t->fno == x->fno)
      h = t->data.args.x1, t = t->data.args.x2;
    if (!res || !eval_internal(thr, t)) {
      qmfree(thr, x);
      return 0;
    }
    while (n > 0 && (res = pushfun(thr, x->fno))) n--;
    if (!res) {
      qmfree(thr, x);
      return 0;
    }
    *fno = 0;
    qmfree(thr, x);
    return 1;
  }
  case APPOP:
    /* function application: */
    if (!eval_internal(thr, x->data.args.x1)) {
      qmfree(thr, x);
      return 0;
    }
    if (!thr->mode) {
      EXPR *x1 = thr->xsp[-1];
      unsigned long argv = (x1->fno==APPOP)?x1->data.args.argv:
	symtb[x1->fno].argv;
      if (argv & 1) {
	int res;
	thr->mode = 1;
	res = eval_internal(thr, x->data.args.x2);
	thr->mode = 0;
	if (!res) {
	  qmfree(thr, x);
	  return 0;
	}
      } else if (!eval_internal(thr, x->data.args.x2)) {
	qmfree(thr, x);
	return 0;
      }
    } else if (!eval_internal(thr, x->data.args.x2)) {
      qmfree(thr, x);
      return 0;
    }
    *fno = APPOP;
    qmfree(thr, x);
    return 1;
  default:
    if (x->fno == VECTOP) {
      int i, n = x->data.vect.n;
      EXPR **xv = n?malloc(n*sizeof(EXPR*)):NULL;
      if (n && !xv) {
	qmfree(thr, x);
	thr->qmstat = MEM_OVF;
	return 0;
      }
      for (i = 0; i < n; i++)
	if (!eval_internal(thr, x->data.vect.xv[i])) {
	  int j;
	  for (j = 0; j < i; j++)
	    qmfree(thr, xv[j]);
	  free(xv);
	  qmfree(thr, x);
	  return 0;
	} else
	  xv[i] = *--thr->xsp;
      if (!pushvect(thr, n, xv)) {
	qmfree(thr, x);
	return 0;
      }
      *fno = 0;
      qmfree(thr, x);
      return 1;
    } else {
      *fno = x->fno;
      qmfree(thr, x);
      return 1;
    }
  }
}

static int evalr_with_frame(THREAD *thr, int *fno,
			    int _fno, EXPR *lvals[2], int *rp, int rc,
			    OPREC *ip,
			    long xbp, int modno, int lineno,
			    int info_addr, byte info_offs)
{
  EXPR *x = thr->xsp[-1];
  if (!x->red) {
    *fno = 0;
    return 1;
  } else if (x->mem) {
    int res;
    thr->xsp--;
    res = eval_internal(thr, x);
    qmfree(thr, x);
    *fno = 0;
    return res;
  }
  if (!pushact(thr, _fno, lvals, rp, rc, ip, xbp, modno, lineno,
	       info_addr, info_offs))
    return 0;
  thr->xsp--;
  switch (x->fno) {
  case CONSOP:
  case PAIROP: {
    /* we do this non-recursively, to prevent stack overflows */
    EXPR *h = x->data.args.x1, *t = x->data.args.x2;
    int n = 0, res;
    while ((res = eval_internal(thr, h)) && ++n && t->fno == x->fno)
      h = t->data.args.x1, t = t->data.args.x2;
    if (!res || !eval_internal(thr, t)) {
      qmfree(thr, x);
      free(*--thr->asp);
      return 0;
    }
    while (n > 0 && (res = pushfun(thr, x->fno))) n--;
    if (!res) {
      qmfree(thr, x);
      free(*--thr->asp);
      return 0;
    }
    *fno = 0;
    qmfree(thr, x);
    free(*--thr->asp);
    return 1;
  }
  case APPOP:
    /* function application: */
    if (!eval_internal(thr, x->data.args.x1)) {
      qmfree(thr, x);
      free(*--thr->asp);
      return 0;
    }
    if (!thr->mode) {
      EXPR *x1 = thr->xsp[-1];
      unsigned long argv = (x1->fno==APPOP)?x1->data.args.argv:
	symtb[x1->fno].argv;
      if (argv & 1) {
	int res;
	thr->mode = 1;
	res = eval_internal(thr, x->data.args.x2);
	thr->mode = 0;
	if (!res) {
	  qmfree(thr, x);
	  free(*--thr->asp);
	  return 0;
	}
      } else if (!eval_internal(thr, x->data.args.x2)) {
	qmfree(thr, x);
	free(*--thr->asp);
	return 0;
      }
    } else if (!eval_internal(thr, x->data.args.x2)) {
      qmfree(thr, x);
      free(*--thr->asp);
      return 0;
    }
    *fno = APPOP;
    qmfree(thr, x);
    free(*--thr->asp);
    return 1;
  default:
    if (x->fno == VECTOP) {
      int i, n = x->data.vect.n;
      EXPR **xv = n?malloc(n*sizeof(EXPR*)):NULL;
      if (n && !xv) {
	qmfree(thr, x);
	free(*--thr->asp);
	thr->qmstat = MEM_OVF;
	return 0;
      }
      for (i = 0; i < n; i++)
	if (!eval_internal(thr, x->data.vect.xv[i])) {
	  int j;
	  for (j = 0; j < i; j++)
	    qmfree(thr, xv[j]);
	  free(xv);
	  qmfree(thr, x);
	  free(*--thr->asp);
	  return 0;
	} else
	  xv[i] = *--thr->xsp;
      if (!pushvect(thr, n, xv)) {
	qmfree(thr, x);
	free(*--thr->asp);
	return 0;
      }
      *fno = 0;
      qmfree(thr, x);
      free(*--thr->asp);
      return 1;
    } else {
      *fno = x->fno;
      qmfree(thr, x);
      free(*--thr->asp);
      return 1;
    }
  }
}

/*

evalu(): evaluate a user-defined function.

Here you are looking at the very core of the interpreter. evalu() tries to
execute the code for a user-defined rule. If no rule matches, a constructor
term is returned as the default value. If during the execution an error
condition arises (qmstat != OK) the execution is aborted immediately, and
evalu() returns 0; otherwise it returns 1. Exceptions are handled internally
if a handler (catch) is available in the current context.

During execution of a function, the configuration of the Q machine
is described by the following items:

- rp, the rule pointer, pointing to the offset of the currently executing
  rule;

- rc, the count of rules remaining to be processed (including the current
  rule);

- ip, the instruction pointer;

- xbp, the base pointer which points to the first argument of the
  executing rule on the stack (this one actually is an index into the stack
  since the stack may be reallocated during evaluation);

- mode and start, which are used to keep track of special forms.

Besides this, source line information from !INFO instructions is maintained in
the modno and lineno variables.

All state variables except mode are local to evalu in order to make evalu
reentrant. (The mode variable must be made global since it is needed by other
operations like consexpr() and qmunquote(). However, the only way to reenter
evalu() while mode is set is through eval() which takes care of restoring the
original mode value when it finishes. In fact, this only happens through
invokations of the unquote (`), force (~) and memo (&) builtins, since these
are the only operations which are executed while mode is set.)

Tail calls (calls immediately preceding a !RET instruction) are eliminated
automatically by setting the configuration for the new function and returning
the return value of the tail call as the return value of the original call. In
such a case, no additional stack space is required.

*/

/* macros for invoking the debugger */

#define dbg_rule \
if (thr->debug) \
  thr->stoplevel = rule(thr, fno, lvals, xbp, *rp, ip, modno, lineno)

#define dbg_binding(failed,m,offs) \
if (thr->debug) \
     binding(thr, failed, m, offs)

#define dbg_reduction \
if (thr->debug || abp+level<thr->stoplevel) \
{ set_debug(thr, 1); thr->stoplevel = 0; reduction(thr, fno, xbp); }

#define dbg_default_reduction \
if (thr->debug || abp+level<thr->stoplevel) \
{ set_debug(thr, 1); thr->stoplevel = 0; default_reduction(thr, fno, xbp); }

#define dbg_tail_reduction \
if (thr->debug) \
{ set_debug(thr, 1); thr->stoplevel = 0; tail_reduction(thr, fno, xbp, fno1); }

static inline int chkbrk(int fno, EXPR **xbp)
{
  if (nbreak > 0) {
    EXPR *x;
    int brk;
    if (fno == APPOP) {
      /* walk down the spine, find the head symbol */
      for (x = *xbp; x->fno == APPOP; x = x->data.args.x1)
	;
      fno = x->fno;
    }
    brk = (symtb[fno].flags & (BRK|TBRK)) != 0;
    if (symtb[fno].flags & TBRK) {
      symtb[fno].flags &= ~TBRK;
      nbreak--;
    }
    return brk;
  } else
    return 0;
}

static inline void chkprof(int fno, EXPR *lvals[2])
{
  if (nprof > 0) {
    EXPR *x;
    if (fno == APPOP) {
      /* walk down the spine, find the head symbol */
      for (x = lvals[0]; x->fno == APPOP; x = x->data.args.x1)
	;
      fno = x->fno;
    }
    if (symtb[fno].flags & PROF) symtb[fno].nredns++;
  }
}

static int evalu(THREAD *thr, int fno)
{
  int	       *rp, *rp1;
  int           rc, rc1;
  int		fno1;
  EXPR         *lvals[2];
  OPREC	       *ip;
  long          xbp, start;
  long		abp = thr->asp-thr->ast, level = 0, siglevel = 0;
  int		modno = NONE, lineno = NONE, info_addr = -1;
  byte		info_offs = 0;
  long		sig;

  thr->qmstat = OK;

  /* check for matching rule, push default value if none: */
  if (fno == APPOP) {
    lvals[0] = thr->xsp[-2]; lvals[1] = thr->xsp[-1];
  } else {
    lvals[0] = lvals[1] = NULL;
  }
  if (!match(thr, fno, lvals, &rp, &rc)) return pushfun(thr, fno);

  /*
   * at this point, we are dealing with a user-defined rule and have rp and
   * rc set to the corresponding rule pointer and counter
   */

 __start:				/* start a new function call */

  ip = codesp + *rp;
  if (fno == APPOP)
    xbp = thr->xsp - thr->xst - 2;
  else
    xbp = thr->xsp - thr->xst;
  thr->mode = 0;

 __debug:				/* catch debug flag */

#ifdef USE_THREADS
  if (nused > 1) {
    /* allow for cancellation */
    if (thr > thr0) pthread_testcancel();
    /* let another thread hold the candle */
    pthread_mutex_unlock(&global_mutex);
    thr->released = 1;
    if (init_thr) {
      pthread_mutex_lock(&init_mutex);
      while (init_thr && thr != init_thr)
	pthread_cond_wait(&init_cond, &init_mutex);
      pthread_mutex_unlock(&init_mutex);
    }
    pthread_mutex_lock(&global_mutex);
    thr->released = 0;
    if (thr > thr0) pthread_testcancel();
  }
#endif

  if (!thr->debug && thr->brkdbg && chkbrk(fno, thr->xst+xbp)) {
    /* breakpoint on function symbol */
    set_debug(thr, 1);
    thr->qmstat = BREAK;
  }

  /* make sure that debugging information is available at the beginning of a
     rule: */
  if (modno == NONE && ip->opcode == INFOP) {
    modno = ip->opargs.info.modno;
    lineno = ip->opargs.info.lineno;
    info_addr = ip->opargs.info.addr;
    info_offs = ip->opargs.info.offs;
    ip++;
  }

  dbg_rule;

 __loop:				/* execution cycle */

  /* process sentinels */
  if (thr->sentinels) process_sentinels(thr);

  /* catch signals, BREAK, QUIT and other errors: */

  if (thr->qmstat != OK) {
  __fail:
    qmfree(thr, lvals[0]); qmfree(thr, lvals[1]);
    return 0;
  }

  if (thr->sigpend) {
    /* catch has caught a signal and now we're processing the handler; make
       sure we don't get interrupted again until it's done */
    siglevel = (level==0)?0:(level+1);
    thr->sigblk = 1;
    thr->sigpend = 0;
  } else if (level < siglevel)
    /* we're done with the signal handler, reenable signal processing */
    thr->sigblk = 0;
  if ((sig = get_sig(thr))) {
    /* we've received a signal, throw an exception */
    if (pushfun(thr, SYSERROP) && pushint(thr, -sig) &&
	pushfun(thr, APPOP))
      thr->qmstat = XCEPT;
    if (thr->brkdbg) {
      set_debug(thr, 1);
      goto __debug;
    } else {
      thr->sigpend = 1;
      goto __fail;
    }
  } else if (thr->brkflag) {
    /* thread-local break (break function) */
    thr->brkflag = 0;
    set_debug(thr, 1);
    thr->qmstat = BREAK;
    goto __debug;
  } else if (thr > thr0)
    ;
  else if (quitflag) {
    /* quit (quit function or termination signal) */
    thr->qmstat = QUIT;
    goto __fail;
  } else if (brkflag) {
    /* global break (Ctl-C) */
    brkflag = 0;
    thr->qmstat = BREAK;
    if (thr->brkdbg) {
      set_debug(thr, 1);
      goto __debug;
    } else
      goto __fail;
  }

  switch (ip->opcode) {

  case INFOP:
    /* process debugging information */
    modno = ip->opargs.info.modno;
    lineno = ip->opargs.info.lineno;
    info_addr = ip->opargs.info.addr;
    info_offs = ip->opargs.info.offs;
    ip++;
    break;

  case RETOP:
    /* return instruction: set the return value and return */
    chkprof(fno, lvals); thr->nredns++; dbg_reduction;
    retval(thr, thr->xst+xbp);
    qmfree(thr, lvals[0]); qmfree(thr, lvals[1]);
    if (level == 0) {
      if (thr->sentinels) process_sentinels(thr);
      return 1;
    } else {
      popact(thr, &fno, lvals, &rp, &rc, &ip, &xbp, &modno, &lineno,
	     &info_addr, &info_offs);
      level--;
      dbg_rule;
      goto __check;
    }
    break;

  case POPOP:
    /* pop instruction: pop value from stack */
    qmfree(thr, *--thr->xsp);
    ip++;
    break;

  case LVALOP:
    /* lval instruction: push an lval, evaluate if necessary: */
    if (!pushlval(thr, fno, lvals, xbp, ip->opargs.lval.offs, ip->opargs.lval.plen,
		  ip->opargs.lval.p))
      goto __fail;
    else if (!thr->mode && thr->xsp[-1]->red) {
      if ((ip+1)->opcode == RETOP) {
	/* return a reducible lval, process like a tail call */
	if (evalr(thr, &fno1))
	  goto __tail_call;
	else
	  goto __tail_error;
      }
      /* make a new stack frame for a recursive eval */
      if (eval_with_frame(thr, thr->xsp[-1],
			  fno, lvals, rp, rc, ip, xbp, modno, lineno,
			  info_addr, info_offs)) {
	qmfree(thr, thr->xsp[-2]);
	thr->xsp[-2] = thr->xsp[-1];
	thr->xsp--;
      } else if (thr->qmstat == XCEPT_FAIL)
	goto __failed_rule;
      else if (thr->qmstat == XCEPT_FAIL2)
	goto __failed_reduction;
      else
	goto __fail;
    }

  __check:                        /* check mode flag and advance to
				     next instruction */

    /* this code is reached from cases RETOP, LVALOP, QUERYOP, MATCHOP and
       the default case */
	    
    if (!thr->mode && ip->mode) {
      EXPR *x = thr->xsp[-1];
      unsigned long argv = (x->fno==APPOP)?x->data.args.argv:
	symtb[x->fno].argv;
      if (argv & 1) {
	thr->mode = 1;
	start = thr->xsp-thr->xst;
      }
    }
    ip++;
    break;

  case QUERYOP:
    /* query instruction: */
    if (thr->xsp[-1]->fno == FALSEOP) {
      /* skip the current rule */
    __skip_rule:
      {
	long bp = (fno == APPOP?xbp+2:xbp)+ip->opargs.qual.offs;
	/* clean up the stack: */
	while (thr->xsp-thr->xst > bp)
	  qmfree(thr, *--thr->xsp);
      }
      if (ip->opargs.qual.addr >= 0) {
	/* jump to the next rule for the same lhs: */
	ip = codesp + ip->opargs.qual.addr;
	modno = lineno = NONE;
	goto __debug;
      }
      /* abort the current rule */
    __next_rule:
      if (--rc > 0) {
	/* try the next rule: */
	ip = codesp + *++rp;
	modno = lineno = NONE;
	goto __debug;
      } else {
	/* push the default value and return: */
	dbg_default_reduction;
	qmfree(thr, lvals[0]); qmfree(thr, lvals[1]);
	if (!pushfun(thr, fno))
	  return 0;
	else if (level == 0) {
	  if (thr->sentinels) process_sentinels(thr);
	  return 1;
	} else {
	  if (thr->sentinels) process_sentinels(thr);
	  popact(thr, &fno, lvals, &rp, &rc, &ip, &xbp, &modno, &lineno,
		 &info_addr, &info_offs);
	  level--;
	  dbg_rule;
	  goto __check;
	}
      }
    } else if (thr->xsp[-1]->fno == TRUEOP) {
      /* proceed */
      qmfree(thr, *--thr->xsp);
      ip++;
      dbg_rule;
    } else if (!have_mark(thr)) {
      /* error in conditional */
      int _debug = thr->debug;
      thr->qmstat = COND_ERR;
      if (thr->brkdbg) {
	set_debug(thr, 1);
	dbg_rule;
	if (thr->qmstat == OK) thr->qmstat = HALT;
      }
      set_debug(thr, _debug);
      goto __fail;
    } else {
      /* error in conditional, will be handled with catch */
      thr->qmstat = COND_ERR;
      goto __fail;
    }
    break;

  case MATCHOP:
    /* pattern binding: */
    {
      EXPR *pat = thr->xsp[-1];
      if (ip->opargs.qual.m < 0) {
	/* irrefutable match */
	dbg_binding(0, ip->opargs.qual.m,
		    ((thr->xsp-thr->xst)-xbp-((fno==APPOP)?2:0)));
      } else if (!matchp(thr, matchtb[ip->opargs.qual.m], &pat)) {
	dbg_binding(1, ip->opargs.qual.m,
		    ((thr->xsp-thr->xst)-xbp-((fno==APPOP)?2:0)));
	goto __skip_rule;
      } else {
	qmfree(thr, thr->xsp[-1]); thr->xsp[-1] = pat;
	dbg_binding(0, ip->opargs.qual.m,
		    ((thr->xsp-thr->xst)-xbp-((fno==APPOP)?2:0)));
      }
    }
    ip++;
    dbg_rule;
    break;

  case INTVALOP:
    /* push an int value: */
    {
      mpz_t z;
      if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	thr->qmstat = MEM_OVF;
	goto __fail;
      } else if (!pushmpz(thr, z))
	goto __fail;
    }
    ip++;
    break;

  case FLOATVALOP:
    /* push a float value: */
    if (!pushfloat(thr, ip->opargs.fv))
      goto __fail;
    ip++;
    break;
	    
  case STRVALOP:
    /* push a string value: */
    {
      char           *s;
      if ((s = strdup(strsp + ip->opargs.sv)) ==
	  NULL) {
	thr->qmstat = MEM_OVF;
	goto __fail;
      } else if (!pushstr(thr, s))
	goto __fail;
    }
    ip++;
    break;

  case APPOP:
    if (thr->mode && thr->xsp-thr->xst-1 == start)
      thr->mode = 0;
    /* falls through to default case */

  default:
    /* function call: */
    if (thr->mode && !(ip->opcode == APPOP &&
		       (thr->xsp[-2]->fno == UNQUOTEOP ||
			thr->xsp[-2]->fno == FORCEOP))) {
      /* processing special form */
      if (ip->opcode == APPOP && thr->xsp[-2]->fno == MEMOP) {
	EXPR *z = qmnew(memexpr(thr, thr->xsp[-1]));
	if (!z) goto __fail;
	qmfree(thr, *--thr->xsp);
	qmfree(thr, thr->xsp[-1]);
	thr->xsp[-1] = z;
      } else if (!pushfun(thr, ip->opcode))
	goto __fail;
      /* no need to check, mode is enabled already */
      ip++;
      break;
    }
    if ((ip+1)->opcode == RETOP) {
      EXPR *lvals1[2];

      /* process a tail call */
      fno1 = ip->opcode;
    __tail_call:
      /* iterate until no more builtin rules are applicable */
      while (fno1 > 0 &&
	     evalb_with_frame(thr, fno1,
			      fno, lvals, rp, rc, ip, xbp, modno, lineno,
			      info_addr, info_offs)) {
	if (thr->mode || !thr->xsp[-1]->red)
	  fno1 = 0;
	else if (!evalr(thr, &fno1))
	  break;
      }
      /* check whether the resulting term is irreducible; then we're done */
      if (fno1 == 0) {
	ip++;
	break;
      }
    __tail_error:
      /* same error handling as in the non-tail case below */
      if (thr->qmstat == OK)
	;
      else if (thr->qmstat == XCEPT_CATCH)
	goto __handle_exception;
      else if (thr->qmstat == XCEPT_FAIL)
	goto __failed_rule;
      else if (thr->qmstat == XCEPT_FAIL2)
	goto __failed_reduction;
      else if ((thr->qmstat == EXT_ERR || thr->qmstat == XCEPT) &&
	       !have_mark(thr))
	goto __exception;
      else
	goto __fail;
      /* check for an applicable user-defined rule */
      if (fno1 == APPOP) {
	lvals1[0] = thr->xsp[-2]; lvals1[1] = thr->xsp[-1];
      } else {
	lvals1[0] = lvals1[1] = NULL;
      }
      if (match(thr, fno1, lvals1, &rp1, &rc1)) {
	EXPR **xp, **top = thr->xsp;
	chkprof(fno, lvals); thr->nredns++; dbg_tail_reduction;
	/* set up for the new call: */
	rp = rp1;
	rc = rc1;
	fno = fno1;
	ip = codesp + *rp;
	if (fno == APPOP) top -= 2;
	for (xp = thr->xst+xbp; xp < top; xp++)
	  qmfree(thr, *xp);
	if (fno == APPOP) {
	  thr->xst[xbp] = thr->xsp[-2];
	  thr->xst[xbp+1] = thr->xsp[-1];
	  thr->xsp = thr->xst + xbp + 2;
	} else
	  thr->xsp = thr->xst + xbp;
	qmfree(thr, lvals[0]); qmfree(thr, lvals[1]);
	lvals[0] = lvals1[0]; lvals[1] = lvals1[1];
	modno = lineno = NONE;
	goto __debug;
      } else {
	/* push the default value: */
	if (!pushfun(thr, fno1)) goto __fail;
	/* no need to check, since RETOP is the next instruction */
	ip++;
	break;
      }
    }
    /* if we come here we have an embedded subterm (not a tail call) which
       is evaluated recursively */
    fno1 = ip->opcode;
    /* iterate until no more builtin rules are applicable */
    while (fno1 > 0 &&
	   evalb_with_frame(thr, fno1,
			    fno, lvals, rp, rc, ip, xbp, modno, lineno,
			    info_addr, info_offs)) {
      if (thr->mode || !thr->xsp[-1]->red)
	fno1 = 0;
      else if (!evalr_with_frame(thr, &fno1,
				 fno, lvals, rp, rc, ip, xbp, modno, lineno,
				 info_addr, info_offs))
	break;
    }
    /* check whether the resulting term is irreducible; then we're done */
    if (fno1 == 0) {
      ip++;
      break;
    }
    /* check for errors in builtins */
    if (thr->qmstat == OK)
      ;
    else if (thr->qmstat == XCEPT_CATCH) {
      long xp;
      EXPR *h, *x;
      /* handle an exception */
    __handle_exception:
      x = NULL;
      get_mark(thr, &xp, &h);
      pop_mark(thr);
      /* save throw argument */
      if (thr->qmstat_save == XCEPT) x = *--thr->xsp;
      /* rewind the stack */
      while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
      /* evaluate the handler (special argument) */
      thr->qmstat = OK;
      /* we need an extra stack frame here */
      if (!eval_with_frame(thr, h, fno, lvals, rp, rc, ip, xbp, modno, lineno,
			   info_addr, info_offs)) {
	thr->sigpend = 0;
	qmfree(thr, x);
	if (thr->qmstat == XCEPT_FAIL)
	  /* fail in handler; let the current rule fail (see also case
	     qmstat == XCEPT_FAIL below) */
	  goto __failed_rule;
	else if (thr->qmstat == XCEPT_FAIL2)
	  goto __failed_reduction;
	else
	  /* exception in handler will be handled by caller */
	  goto __fail;
      }
      /* pop catch arguments, replace with handler and exception */
      retval(thr, thr->xsp-3);
      if (thr->qmstat_save != XCEPT) {
	/* exception comes from interpreter, push error code */
	if (!pushfun(thr, SYSERROP) || !pushint(thr, thr->qmstat_save) ||
	    !pushfun(thr, APPOP)) goto __fail;
      } else
	/* exception comes from throw, push argument saved above */
	*(thr->xsp++) = x;
      /* now handler and exception are on the stack and in the next loop
	 cycle we will reexecute the current (apply) instruction from the
	 original catch call */
      break;
    } else if (thr->qmstat == XCEPT_FAIL) {
      /* pretend that the current rule has failed */
    __failed_rule:
      {
	/* KLUDGE ALERT: Here we rely on proper jump address and stack offset
	   information being supplied through the addr and offs fields of the
	   previous INFOP instruction. */
	long bp = (fno == APPOP?xbp+2:xbp)+info_offs;
	thr->qmstat = OK;
	/* proceed as with failed qualifier */
	while (thr->xsp-thr->xst > bp)
	  qmfree(thr, *--thr->xsp);
	if (info_addr >= 0) {
	  ip = codesp + info_addr;
	  modno = lineno = NONE;
	  goto __debug;
	}
	goto __next_rule;
      }
    } else if (thr->qmstat == XCEPT_FAIL2) {
      /* pretend that *all* rules have failed */
    __failed_reduction:
      thr->qmstat = OK;
      /* make it look as if all rules have already been processed */
      if (rc > 1) {
	rp += rc-1;
	rc = 1;
      }
      /* proceed as with failed qualifier */
      while (thr->xsp-thr->xst > (fno == APPOP?xbp+2:xbp))
	qmfree(thr, *--thr->xsp);
      goto __next_rule;
    } else if ((thr->qmstat == EXT_ERR || thr->qmstat == XCEPT) &&
	       !have_mark(thr)) {
      int _debug;
    __exception:
      _debug = thr->debug;
      if (thr->brkdbg) {
	set_debug(thr, 1);
	dbg_rule;
	if (thr->qmstat == OK) thr->qmstat = HALT;
      }
      set_debug(thr, _debug);
      goto __fail;
    } else
      /* error condition in builtin */
      goto __fail;
    /* if we come here, we have a possibly reducible term to which all
       possible builtin rules have been applied; check for an applicable
       user-defined rule */
    {
      EXPR *lvals1[2];

      if (fno1 == APPOP) {
	lvals1[0] = thr->xsp[-2]; lvals1[1] = thr->xsp[-1];
      } else {
	lvals1[0] = lvals1[1] = NULL;
      }
      if (!match(thr, fno1, lvals1, &rp1, &rc1)) {
	/* no matching rule; push default value: */
	if (!pushfun(thr, fno1)) goto __fail;
	goto __check;
      } else {
	/* user-defined rule */
	if (abp+level+1 < 0) {
	  /* integer range overflow -- bail out */
	  thr->qmstat = AST_OVF;
	  qmfree(thr, lvals1[0]); qmfree(thr, lvals1[1]);
	  goto __fail;
	} else if (!pushact(thr, fno, lvals, rp, rc, ip, xbp, modno, lineno,
			    info_addr, info_offs)) {
	  qmfree(thr, lvals1[0]); qmfree(thr, lvals1[1]);
	  goto __fail;
	} else
	  level++;
	fno = fno1;
	lvals[0] = lvals1[0]; lvals[1] = lvals1[1];
	rp = rp1;
	rc = rc1;
	modno = lineno = NONE;
	goto __start;
      }
    }
    break;
  }		/* switch(ip->opcode) */
	  
  goto __loop;
}

/* evaldef() is a specialized version of evalu() used to execute the
   initialization code */

int evaldef1(int offs, int *modno, int *lineno)
{
  OPREC	       *ip;
  long	        start;
  long		xbp = thr0->xsp - thr0->xst;

  *modno = NONE; *lineno = NONE;
  thr0->mode = 0;
  for (ip = codesp + offs; ; ip++) {
    switch (ip->opcode) {
		
    case INFOP:
      /* process debugging information */
      *modno = ip->opargs.info.modno;
      *lineno = ip->opargs.info.lineno;
      break;
      
    case RETOP:
      /* pop value from stack and return: */
      if (thr0->xsp-thr0->xst > xbp) qmfree(thr0, *--thr0->xsp);
      return 1;

    case POPOP:
      /* pop value from stack */
      qmfree(thr0, *--thr0->xsp);
      break;

    case MATCHOP:
      /* match against lhs of definition */
      {
	EXPR *pat = thr0->xsp[-1];
	if (ip->opargs.qual.m < 0)
	  /* irrefutable match */;
	else if (!matchp(thr0, matchtb[ip->opargs.qual.m], &pat)) {
	  thr0->qmstat = MATCH_ERR;
	  return 0;
	} else {
	  qmfree(thr0, thr0->xsp[-1]); thr0->xsp[-1] = pat;
	}
      }
      break;

    case INTVALOP:
      /* push an int value: */
      {
	mpz_t z;
	if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	  thr0->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushmpz(thr0, z))
	  return 0;
      }
      break;

    case FLOATVALOP:
      /* push a float value: */
      if (!pushfloat(thr0, ip->opargs.fv))
	return 0;
      break;
      
    case STRVALOP:
      /* push a string value: */
      {
	char           *s;
	if ((s = strdup(strsp + ip->opargs.sv)) ==
	    NULL) {
	  thr0->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushstr(thr0, s))
	  return 0;
      }
      break;

    case APPOP:
      if (thr0->mode && thr0->xsp-thr0->xst-1 == start)
	thr0->mode = 0;
      /* falls through to default case */

    default:
      /* function call: */
      if (thr0->mode && !(ip->opcode == APPOP &&
			  (thr0->xsp[-2]->fno == UNQUOTEOP ||
			   thr0->xsp[-2]->fno == FORCEOP))) {
	/* processing special form */
	if (ip->opcode == APPOP && thr0->xsp[-2]->fno == MEMOP) {
	  EXPR *z = qmnew(memexpr(thr0, thr0->xsp[-1]));
	  if (!z) return 0;
	  qmfree(thr0, *--thr0->xsp);
	  qmfree(thr0, thr0->xsp[-1]);
	  thr0->xsp[-1] = z;
	} else if (!pushfun(thr0, ip->opcode))
	  return 0;
      } else if (evalf(thr0, ip->opcode)) {
	if (!thr0->mode && ip->mode) {
	  EXPR *x = thr0->xsp[-1];
	  unsigned long argv = (x->fno==APPOP)?x->data.args.argv:
	    symtb[x->fno].argv;
	  if (argv & 1) {
	    thr0->mode = 1;
	    start = thr0->xsp-thr0->xst;
	  }
	}
	break;
      } else
	return 0;
      
    }
  }
}

int evaldef(int offs, int *modno, int *lineno)
{
  char base, *baseptr = thr0->baseptr;
  bool mode = thr0->mode;
  int res;
  lock_debug(thr0);
  if (!baseptr)
    thr0->baseptr = &base;
  if (cstackmax > 0 && stack_dir*(&base - thr0->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr0->qmstat = AST_OVF;
    return 0;
  }
  res = evaldef1(offs, modno, lineno);
  if (!baseptr)
    thr0->baseptr = NULL;
  unlock_debug(thr0);
  thr0->mode = mode;
  if (thr0->sentinels) process_sentinels(thr0);
  return res;
}

/* evalf(): evaluate a function call */

static inline int eval_if_red(THREAD *thr)
{
  if (thr->xsp[-1]->red) {
    EXPR *x = *--thr->xsp;
    int res = eval(thr, x);
    qmfree(thr, x);
    if (!res)
      return 0;
  }
  return 1;
}

static int evalf(THREAD *thr, int fno)
{
  bool _mode = thr->mode;
  int res;
  long sig;
  if (evalb(thr, fno) && eval_if_red(thr)) {
    /* builtin: */
    if ((sig = get_sig(thr))) {
      if (pushfun(thr, SYSERROP) && pushint(thr, -sig) &&
	  pushfun(thr, APPOP))
	thr->qmstat = XCEPT;
      res = 0;
    } else
      res = 1;
  } else if (thr->qmstat == XCEPT_CATCH) {
    long xp;
    EXPR *h, *x = NULL;
    /* handle an exception */
    get_mark(thr, &xp, &h);
    pop_mark(thr);
    if (thr->qmstat_save == XCEPT) x = *--thr->xsp;
    while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
    thr->qmstat = OK;
    if (!eval(thr, h)) {
      thr->sigpend = 0;
      qmfree(thr, x);
      return 0;
    }
    retval(thr, thr->xsp-3);
    if (thr->qmstat_save != XCEPT) {
      if (!pushfun(thr, SYSERROP) || !pushint(thr, thr->qmstat_save) ||
	  !pushfun(thr, APPOP)) return 0;
    } else
      *(thr->xsp++) = x;
    res = evalu(thr, APPOP);
  } else if (thr->qmstat != OK)
    /* error condition in builtin: */
    res = 0;
  else
    /* check for user-defined function: */
    res = evalu(thr, fno);
  thr->mode = _mode;
  return res;
}

/* eval(): evaluate an expression and push the result onto the stack. */

static inline EXPR *copyexpr(THREAD *thr, EXPR *x)
{
  /* shallow copy of an expression */
  int fno = x->fno;
  switch (fno) {
  case INTVALOP: {
    mpz_t z;
    mpz_init(z);
    mpz_set(z, x->data.z);
    return mpzexpr(thr, z);
  }
  case FLOATVALOP:
    return floatexpr(thr, x->data.f);
  case STRVALOP:
    return strexpr(thr, strdup(x->data.s));
  case FILEVALOP: case BADFILEVALOP: case USRVALOP:
    /* no way to copy these */
    return NULL;
  case VECTOP: {
    int n = x->data.vect.n;
    EXPR **xv1 = malloc(n*sizeof(EXPR*)), **xv = x->data.vect.xv;
    if (xv1) {
      int i;
      for (i = 0; i < n; i++)
	xv1[i] = qmnew(xv[i]);
      return vectexpr(thr, n, xv1);
    } else {
      thr->qmstat = MEM_OVF;
      return NULL;
    }
  }
  case CONSOP: case PAIROP: case APPOP:
    return consexpr(thr, fno, x->data.args.x1, x->data.args.x2);
  default:
    return funexpr2(thr, fno);
  }
}

static inline void clearexpr(THREAD *thr, EXPR *x)
{
  /* get rid of the previous contents of x */
  if (x->fno == VECTOP) {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv;
    for (i = 0; i < n; i++) qmfree(thr, xv[i]);
    free(xv);
  } else if (x->fno < BINARY) {
    qmfree(thr, x->data.args.x1);
    qmfree(thr, x->data.args.x2);
  }
}

static int memoize(THREAD *thr, EXPR *x)
{
  EXPR *y = thr->xsp[-1];
  /* this shouldn't happen, but we play it safe */
  if (x->fno < RESERVED && x->fno != VECTOP)
    return 1;
  /* FIXME: Currently there is no safe way to copy these types of
     expressions, so we just refuse to memoize them if we can't do it
     in-place. */
  if (y->refc > 1 && (y->fno == FILEVALOP || y->fno == BADFILEVALOP ||
		      y->fno == USRVALOP))
    return 1;
#if 0
  printf("memoize(%p)\n", x);
#endif
  if (y->refc == 1) {
    /* in this case we can do the memoizing "in-place", so we just cannibalize
       y and install x in its place */
    int refc;
    clearexpr(thr, x);
    refc = x->refc; *x = *y; x->refc = refc;
    y->refc = 0; x_free(thr, y);
  } else {
    /* here we have to copy the dynamic contents of y */
    EXPR *z = copyexpr(thr, y);
    int refc;
    if (!z) return 0;
    qmfree(thr, y);
    clearexpr(thr, x);
    refc = x->refc; *x = *z; x->refc = refc;
    z->refc = 0; x_free(thr, z);
  }
  thr->xsp[-1] = qmnew(x);
  return 1;
}

#if 1
/* by default this is implemented as a macro to ensure inlining */
#define memoize_if_mem(thr,x) (((x)->mem)?memoize((thr),(x)):1)
#else
/* enable this instead in order to facilitate debugging */
static int memoize_if_mem(THREAD *thr, EXPR *x)
{
  if (x->mem)
    return memoize(thr, x);
  else
    return 1;
}
#endif

static int eval_internal(THREAD *thr, EXPR *x)
{
  char test;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if (!x->red || !x->raw && thr->mode)
    return push(thr, x);
#if 0
  printf("eval(%p;mem=%d;mode=%d): ", x, x->mem, thr->mode); printx(x); printf("\n");
#endif
  switch (x->fno) {
  case CONSOP:
  case PAIROP: {
    /* we do this non-recursively, to prevent stack overflows */
    EXPR *h = x->data.args.x1, *t = x->data.args.x2;
    int n = 0, res;
    while ((res = eval_internal(thr, h)) && ++n && t->fno == x->fno)
      h = t->data.args.x1, t = t->data.args.x2;
    if (!res || !eval_internal(thr, t)) return 0;
    while (n > 0 && (res = pushfun(thr, x->fno))) n--;
    return res && memoize_if_mem(thr, x);
  }
  case APPOP:
    /* function application: */
    if (!eval_internal(thr, x->data.args.x1))
      return 0;
    if (!thr->mode) {
      EXPR *x1 = thr->xsp[-1];
      unsigned long argv = (x1->fno==APPOP)?x1->data.args.argv:
	symtb[x1->fno].argv;
      if (argv & 1) {
	int res;
	thr->mode = 1;
	res = eval_internal(thr, x->data.args.x2);
	thr->mode = 0;
	if (!res) return 0;
      } else if (!eval_internal(thr, x->data.args.x2))
	return 0;
    } else if (!eval_internal(thr, x->data.args.x2))
      return 0;
    if (thr->mode && thr->xsp[-2]->fno != UNQUOTEOP &&
	thr->xsp[-2]->fno != FORCEOP) {
      if (thr->xsp[-2]->fno == MEMOP) {
	EXPR *z = qmnew(memexpr(thr, thr->xsp[-1]));
	if (!z) return 0;
	qmfree(thr, *--thr->xsp);
	qmfree(thr, thr->xsp[-1]);
	thr->xsp[-1] = z;
	return 1;
      } else
	return pushfun(thr, APPOP);
    } else
      return evalf(thr, APPOP) && memoize_if_mem(thr, x);
  default:
    if (x->fno == VECTOP) {
      int i, n = x->data.vect.n;
      EXPR **xv = n?malloc(n*sizeof(EXPR*)):NULL;
      if (n && !xv) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      for (i = 0; i < n; i++)
	if (!eval_internal(thr, x->data.vect.xv[i])) {
	  int j;
	  for (j = 0; j < i; j++)
	    qmfree(thr, xv[j]);
	  free(xv);
	  return 0;
	} else
	  xv[i] = *--thr->xsp;
      return pushvect(thr, n, xv) && memoize_if_mem(thr, x);
    } else if (thr->mode)
      return pushfun(thr, x->fno);
    else
      return evalf(thr, x->fno) && memoize_if_mem(thr, x);
  }
}

int eval(THREAD *thr, EXPR *x)
{
  char base, *baseptr = thr->baseptr;
  bool _mode = thr->mode;
  int res;
  thr->mode = 0;
  if (thr->level == 0) {
    lock_debug(thr);
    thr->stoplevel = 0;
  }
  if (!baseptr)
    thr->baseptr = &base;
  if (thr->level+1 <= 0 ||
      cstackmax > 0 && stack_dir*(&base - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->mode = _mode;
    thr->qmstat = AST_OVF;
    return 0;
  }
  thr->level++;
  res = eval_internal(thr, x);
  thr->level--;
  if (!baseptr)
    thr->baseptr = NULL;
  if (thr->level == 0) {
    unlock_debug(thr);
    thr->stoplevel = 0;
  }
  thr->mode = _mode;
  if (thr->sentinels) process_sentinels(thr);
  return res;
}

/* eval with extra stack frame */

static int eval_with_frame(THREAD *thr, 
			   EXPR *x, int fno, EXPR *lvals[2], int *rp, int rc,
			   OPREC *ip, long xbp, int modno, int lineno,
			   int info_addr, byte info_offs)
{
  int res;
  if (!pushact(thr, fno, lvals, rp, rc, ip, xbp, modno, lineno,
	       info_addr, info_offs))
    return 0;
  res = eval(thr, x);
  free(*--thr->asp);
  return res;
}

/* The following routines implement a simple debugging facility which
   allows to trace the reductions performed by the Q machine. */

static int printp(THREAD *thr, int fno, long xbp);
static int printc(THREAD *thr, int fno, EXPR *lvals[2],
		  long xbp, int addr, OPREC *ip0);
static int buildvartb(THREAD *thr, int addr, OPREC *ip0);
static int lastoffs(THREAD *thr, int addr, OPREC *ip0);
static int listvars(THREAD *thr, int offs);
static int savevars(THREAD *thr, int fno, EXPR *lvals[2], long xbp, int offs);
static void restorevars(THREAD *thr);
static int getvar(THREAD *thr, char *name, int offs);
static void set_print_params(void);
static void reset_print_params(void);

#define MAXITEMS 6	/* maximum number of stacked rules to print */
#define MAXLEVEL 2	/* maximum display depth of expressions */
#define MAXLIST 3	/* maximum number of list/tuple items to print */
#define MAXCHARS 33	/* maximum string length to print */

static int maxlevel0 = MAXLEVEL, maxlist0 = MAXLIST, maxchars0 = MAXCHARS;
static int maxitems0 = MAXITEMS;

static void check_thread(THREAD *thr)
{
  static int act_thrno = -1;
  int thrno = thr-thr0;
  if (thrno != act_thrno && (act_thrno >= 0 || thrno > 0)) {
    flush_shift();
    printf("[switching to thread #%d]\n", thrno);
  }
  act_thrno = thrno;
}

static void help(void)
{
	printf(
"Debugger Commands:\n\n\
? or help	print this list\n\
break [on|off]	set break flag, list breakpoints\n\
break F ...	set breakpoints on function symbols F\n\
tbreak F ...	set temporary breakpoint (cleared when hit)\n\
clear [F ...]	clear breakpoints\n\
.		print the current rule\n\
. ARG ...	change options (`help options' for details)\n\
l [+K|-K] [N]	list source lines (N lines, K = offset)\n\
p [N]		print rule stack (N rules)\n\
m		print memory usage\n\
v		list local variables\n\
u [N], d [N]	move up or down N stack levels\n\
t, b		move to the top or bottom of the rule stack\n\
<CR>		step into the current rule\n\
n		step over the current rule\n\
c		continue\n\
h		halt\n\
q or <EOF>	quit\n\
\n\
All other input is an expression to be evaluated. Use `? EXPR' to escape\n\
an expression which looks like a debugger command.\n");
}

static void help_options(void)
{
	printf(
"Option Commands:\n\n\
. options	print current settings\n\
\n\
. pathnames=y|n	print long script pathnames yes/no\n\
. detail=N	set detail to N expression levels\n\
. maxitems=N	set number of list or tuple items to print\n\
. maxchars=N	set number of string characters to print\n\
. maxstack=N	set number of stack levels to print\n\
\n\
The numeric argument N can also be zero or `all' to specify that all\n\
corresponding items are to be printed.\n\
\n\
Multiple options, separated with whitespace, can be specified in a single\n\
`.' command. Single options must not contain any whitespace.\n");
}

static void lock_debug(THREAD *thr)
{
#ifdef USE_THREADS
  if (thr->debug && !thr->debug_lock) {
    suspend_input();
    thr->debug_lock = 1;
    debug_lock++;
  }
#endif
}

static void unlock_debug(THREAD *thr)
{
#ifdef USE_THREADS
  if (thr->debug_lock) {
    thr->debug_lock = 0;
    if (!--debug_lock) resume_input();
  }
#endif
}

static void set_debug(THREAD *thr, bool debug)
{
  if (thr->debug != debug) {
    thr->debug = debug;
#ifdef USE_THREADS
    if (debug)
      lock_debug(thr);
    else
      unlock_debug(thr);
#endif
  }
}

static void debug_thread(THREAD *thr, char *msg)
{
  char s[MAXSTRLEN];
  push_sigint(SIG_IGN);
  flush_shift();
  sprintf(s, msg, thr-thr0);
  printf("[%s]\n", s);
  pop_sigint();
}

/* temporarily increase the maximum stack size for debugging */

static void dbg_stack(THREAD *thr, int n)
{
  if (thr->maxxstsz > 0) thr->maxxstsz += n;
}

static void end_stack(THREAD *thr, int n)
{
  if (thr->maxxstsz > 0) thr->maxxstsz -= n;
}

static void print_source(int modno, int lineno, int lines)
{
  FILE *fp;
  if (modno != NONE && lineno != NONE &&
      (fp = fopen(strsp+fnametb[modno], "r"))) {
    int c, actline = 1;
    while (actline < lineno && (c = getc(fp)) != EOF)
      if (c == '\n')
	actline++;
    while (actline < lineno+lines && (c = getc(fp)) != EOF) {
      putchar(c);
      if (c == '\n')
	actline++;
    }
    fclose(fp);
  } else
    error("Source file not found");
}

static void print_rule(THREAD *thr, int fno, EXPR *lvals[2],
		       long xbp, int addr, OPREC *ip, int modno, int lineno)
{
  char fname[MAXSTRLEN];
  if (modno != NONE && lineno != NONE) {
    if (debug_long)
      strcpy(fname, strsp+fnametb[modno]);
    else
      basename(fname, strsp+fnametb[modno], 0);
    printf("%s, line %d: ", fname, lineno);
  }
  dbg_stack(thr, MAXDEPTH+10);
  if (printp(thr, fno, xbp)) {
    printf("  ==>  ");
    printc(thr, fno, lvals, xbp, addr, ip);
  }
  end_stack(thr, MAXDEPTH+10);
  putchar('\n');
}

static void print_stacked_rule(THREAD *thr, int base, EXPR *lvals[2])
{
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, 
	     thr->asp[-base]->fno, thr->asp[-base]->lvals,
	     thr->asp[-base]->xbp, *thr->asp[-base]->rp,
	     thr->asp[-base]->ip,
	     thr->asp[-base]->modno, thr->asp[-base]->lineno);
}

static void print_stack(THREAD *thr, 
			int base, int fno, EXPR *lvals[2],
			long xbp, int addr, OPREC *ip,
			int modno, int lineno, int maxitems)
{
  AREC	      **abp, **asp1 = thr->asp - base;
  if (maxitems > 0 && asp1-thr->ast > maxitems-1)
    abp = asp1 - maxitems + 1;
  else
    abp = thr->ast;
  if (base) {
    fno = asp1[0]->fno; xbp = asp1[0]->xbp; addr = *asp1[0]->rp;
    modno = asp1[0]->modno; lineno = asp1[0]->lineno;
  }
  printf("stack size: %d\n", thr->asp-thr->ast+1);
  for (; abp < asp1; abp++) {
    printf("%3d>  ", abp-thr->ast);
    print_rule(thr, (*abp)->fno, (*abp)->lvals,
	       (*abp)->xbp, *(*abp)->rp, (*abp)->ip,
	       (*abp)->modno, (*abp)->lineno);
    if (thr->qmstat != OK) return;
  }
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, fno, lvals, xbp, addr, ip, modno, lineno);
}

static char *debug_parse_opt(char *s, char *t)
{
  int l = strlen(t);
  if (strncmp(s, t, l) == 0)
    return s+l;
  else
    return NULL;
}

char *debug_get_opts(char *s)
{
  char opt[MAXSTRLEN];
  sprintf(opt, "pathnames=%c", debug_long?'y':'n');
  strcpy(s, opt);
  if (maxlevel0)
    sprintf(opt, " detail=%d", maxlevel0);
  else
    sprintf(opt, " detail=all");
  strcat(s, opt);
  if (maxlist0)
    sprintf(opt, " maxitems=%d", maxlist0);
  else
    sprintf(opt, " maxitems=all");
  strcat(s, opt);
  if (maxchars0)
    sprintf(opt, " maxchars=%d", maxchars0);
  else
    sprintf(opt, " maxchars=all");
  strcat(s, opt);
  if (maxitems0)
    sprintf(opt, " maxstack=%d", maxitems0);
  else
    sprintf(opt, " maxstack=all");
  strcat(s, opt);
  return s;
}

int debug_parse_opts(char *s)
{
  int n;
  char *t, *arg;
  for (; *s && isspace(*s); s++) ;
  for (t = strtok(s, " \t\n\r\f"); t; t = strtok(NULL, " \t\n\r\f"))
    if (*t) {
      if (strcmp(t,"options") == 0) {
	char opts[MAXSTRLEN];
	printf("%s\n", debug_get_opts(opts));
      } else if ((arg = debug_parse_opt(t, "pathnames=")))
	if (strlen(arg)==1 && (*arg=='y'||*arg=='n'))
	  debug_long = *arg=='y';
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "detail=")))
	if (strcmp(arg,"all") == 0)
	  maxlevel0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxlevel0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxitems=")))
	if (strcmp(arg,"all") == 0)
	  maxlist0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxlist0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxchars=")))
	if (strcmp(arg,"all") == 0)
	  maxchars0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxchars0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxstack=")))
	if (strcmp(arg,"all") == 0)
	  maxitems0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxitems0 = n;
	else
	  return 0;
      else
	return 0;
    }
  return 1;
}

static void debug_eval(THREAD *thr, int base, int fno, EXPR *lvals[2],
		       long xbp, int addr, OPREC *ip, char *s)
{
  for (; *s && isspace(*s); s++) ;
  if (*s) {
    long xp = thr->xsp - thr->xst;
    long ap = thr->asp-thr->ast;
    bool _brkdbg = thr->brkdbg, _debug = thr->debug;
    pop_sigint();
#ifdef USE_THREADS
    release_tty();
#endif
    thr->brkdbg = thr->debug = 0;
    if (sparsex(s)) {
      EXPR *x = *--thr->xsp;
      int ret;
      long _fno = base?thr->asp[-base]->fno:fno;
      long _xbp = base?thr->asp[-base]->xbp:xbp;
      int _addr = base?(*thr->asp[-base]->rp):addr;
      OPREC *_ip = base?thr->asp[-base]->ip:ip;
      int offs = lastoffs(thr, _addr, _ip);

      buildvartb(thr, _addr, _ip);
      savevars(thr, _fno, base?thr->asp[-base]->lvals:lvals, _xbp, offs);
      ret = eval(thr, x);
      qmfree(thr, x);
      check_thread(thr);
      flush_shift();
      if (ret) {
	printx(thr->xsp[-1]);
	flush_shift();
	printf("\n");
	qmfree(thr, *--thr->xsp);
      }
      restorevars(thr);
    }
    if (thr->qmstat != OK) {
      error(qmmsg[thr->qmstat]);
      if (thr->qmstat == XCEPT && thr->xsp-thr->xst > xp) {
	printx(thr->xsp[-1]);
	flush_shift();
	printf("\n");
      }
    }
    thr->qmstat = OK;
    while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
    while (thr->asp-thr->ast > ap) free(*--thr->asp);
    if (thr->sentinels) process_sentinels(thr);
    thr->brkdbg = _brkdbg; thr->debug = _debug;
#ifdef USE_THREADS
    acquire_tty();
#endif
    push_sigint(SIG_IGN);
    fflush(stderr); fflush(stdout);
    clearerr(stdin);
  }
}

static char *skipws(char *s)
{
  while (*s && isspace(*s)) s++;
  return s;
}

extern print_breakpoints();

static int rule(THREAD *thr, 
		int fno, EXPR *lvals[2], long xbp, int addr, OPREC *ip,
		int modno, int lineno)
{
  char          *buf = NULL, *s;
  long		stoplevel = 0;
  long		base = 0;
  static int	debug_init = 1;
  static int	nlines = 1, noffs = 0;
  
  /* further breaks are masked while in the debugger: */
#ifdef USE_THREADS
  acquire_tty();
#endif
  push_sigint(SIG_IGN); if (thr == thr0) brkflag = 0;
  check_thread(thr);
  if (thr->qmstat != OK)
    error(qmmsg[thr->qmstat]);
  flush_shift();
  thr->qmstat = OK;
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, fno, lvals, xbp, addr, ip, modno, lineno);
  /* check whether running interactively: */
  if (!iflag && (!isatty(fileno(stdin)) || !isatty(fileno(stdout))))
    goto exit;
  /* clear EOF on stdin if set: */
  clearerr(stdin);

 command:			/* command loop: */

  if (thr->qmstat != OK)
    goto exit;
  if (debug_init) {
    printf("(type ? for help)\n");
    debug_init = 0;
  }
  switch_history();
  buf = mygetline(stdin, prompt3, 0);
  switch_history();
  if (buf == NULL) {
    thr->qmstat = QUIT;
    if (iflag) putchar('\n');
    goto exit;
  } else {
    int l = strlen((s = skipws(buf)));
    if (l > 0 && s[l-1] == '\n')
      s[l-1] = '\0', l--;
    while (l > 0 && isspace(s[l-1]))
      s[l-1] = '\0', l--;
  }
  
  if (strlen(s) == 0)
    goto exit;
  else if (strcmp(s, "?") == 0)
    help();
  else if (strncmp(s, "help", 4) == 0 && (!s[4] || isspace(s[4]))) {
    char *t = skipws(s+4);
    if (!*t)
      help();
    else if (strcmp(t, "options") == 0)
      help_options();
    else
      debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
  } else if (strncmp(s, "break", 5) == 0 && (!s[5] || isspace(s[5]))) {
    char *t = skipws(s+5);
    if (!*t)
      print_breakpoints();
    else if (strcmp(t, "on") == 0) {
      brkdbg = thr->brkdbg = 1;
      if (gflag) gcmd_b("q-break-cmd", 1);
    } else if (strcmp(t, "off") == 0) {
      brkdbg = thr->brkdbg = 0;
      if (gflag) gcmd_b("q-break-cmd", 0);
    } else {
      int modno = (mainno>=0)?mainno:0, fno;
      char *f;
      for (f = strtok(t, " \t\n\r\f"); f; f = strtok(NULL, " \t\n\r\f"))
	if ((fno = getsym(f, modno)) != NONE && !(symtb[fno].flags&VSYM)) {
	  if (symtb[fno].flags & TBRK) {
	    symtb[fno].flags &= ~TBRK;
	    nbreak--;
	  }
	  if (!(symtb[fno].flags & BRK)) {
	    symtb[fno].flags |= BRK;
	    nbreak++;
	  }
	  if (!brkdbg || !thr->brkdbg) {
	    brkdbg = thr->brkdbg = 1;
	    if (gflag) gcmd_b("q-break-cmd", 1);
	  }
	} else {
	  error(qmmsg[BAD_SYM]);
	  break;
	}
    }
  } else if (strncmp(s, "tbreak", 6) == 0 && (!s[6] || isspace(s[6]))) {
    char *t = skipws(s+6);
    int modno = (mainno>=0)?mainno:0, fno;
    char *f;
    for (f = strtok(t, " \t\n\r\f"); f; f = strtok(NULL, " \t\n\r\f"))
      if ((fno = getsym(f, modno)) != NONE && !(symtb[fno].flags&VSYM)) {
	if (symtb[fno].flags & BRK) {
	  symtb[fno].flags &= ~BRK;
	  nbreak--;
	}
	if (!(symtb[fno].flags & TBRK)) {
	  symtb[fno].flags |= TBRK;
	  nbreak++;
	}
	if (!brkdbg || !thr->brkdbg) {
	  brkdbg = thr->brkdbg = 1;
	  if (gflag) gcmd_b("q-break-cmd", 1);
	}
      } else {
	error(qmmsg[BAD_SYM]);
	break;
      }
  } else if (strncmp(s, "clear", 5) == 0 && (!s[5] || isspace(s[5]))) {
    char *t = skipws(s+5);
    if (!*t) {
      int i;
      for (i = RESERVED; i<symtbsz; i++) {
	if (symtb[i].flags & BRK) {
	  symtb[i].flags &= ~BRK;
	  nbreak--;
	}
	if (symtb[i].flags & TBRK) {
	  symtb[i].flags &= ~TBRK;
	  nbreak--;
	}
      }
    } else {
      int modno = (mainno>=0)?mainno:0, fno;
      char *f;
      for (f = strtok(t, " \t\n\r\f"); f; f = strtok(NULL, " \t\n\r\f"))
	if ((fno = getsym(f, modno)) != NONE && !(symtb[fno].flags&VSYM)) {
	  if (symtb[fno].flags & BRK) {
	    symtb[fno].flags &= ~BRK;
	    nbreak--;
	  }
	  if (symtb[fno].flags & TBRK) {
	    symtb[fno].flags &= ~TBRK;
	    nbreak--;
	  }
	} else {
	  error(qmmsg[BAD_SYM]);
	  break;
	}
    }
  } else if (strncmp(s, "completion_matches", 18) == 0 &&
	     (!s[18] || isspace(s[18]))) {
    char *t = skipws(s+18), *tmp;
    int l = strlen(t);
    /* parse possibly quoted string */
    if ((t[0] == '"' || t[0] == '\'') && l >= 2 && t[l-1] == t[0]) {
      t[l-1] = 0; t++;
      l -= 2;
    }
    tmp = malloc(l+1);
    if (tmp) {
      scanstr(tmp, t);
      list_completions(tmp);
      free(tmp);
    }
  } else if (strchr("mvtbnchq", s[0]) && strlen(s) > 1)
    debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
  else
    switch (s[0]) {
    case '?':
      debug_eval(thr, base, fno, lvals, xbp, addr, ip, s+1);
      break;
    case '.':
      if (s[1] != 0) {
	if (!debug_parse_opts(s+1))
	  debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
	else if (gflag) {
	  char opts[MAXSTRLEN];
	  /* prepare to fork */
	  pop_sigint();
#ifdef USE_THREADS
	  release_tty();
#endif
	  gcmd_s("q-debug-options-cmd", debug_get_opts(opts));
	  /* after fork */
#ifdef USE_THREADS
	  acquire_tty();
#endif
	  push_sigint(SIG_IGN);
	}
	break;
      }
    reprint:
      if (base)
	print_stacked_rule(thr, base, lvals);
      else {
	printf("%3d>  ", thr->asp-thr->ast-base);
	print_rule(thr, fno, lvals, xbp, addr, ip, modno, lineno);
      }
      break;
    case 'v': {
      int _addr = base?(*thr->asp[-base]->rp):addr;
      OPREC *_ip = base?thr->asp[-base]->ip:ip;
      int offs = lastoffs(thr, _addr, _ip);
      buildvartb(thr, _addr, _ip);
      listvars(thr, offs);
      break;
    }
    case 'm': {
      int nstack = thr->xsp - thr->xst;
      unsigned long nheap;
      unsigned long mexprs;
      
      nheap = ((unsigned long)(xnblks-1))*XBLKSZ+(xheap-xblk->x);
      mexprs = nheap-fexprs;
      if (stackmax > 0)
	printf("stack: %d cells (%d used) out of %d\n",
	       thr->xstsz, nstack, stackmax);
      else
	printf("stack: %d cells (%d used)\n",
	       thr->xstsz, nstack);
      if (memmax > 0)
	printf("heap: %d cells (%d used, %d free) out of %d\n",
	       xnblks*XBLKSZ, mexprs, fexprs, memmax);
      else
	printf("heap: %d cells (%d used, %d free)\n",
	       xnblks*XBLKSZ, mexprs, fexprs);
      break;
    }
    case 'p': {
      char *t = skipws(s+1);
      if (*t) {
	int n1, l = 0;
	if (strcmp(t, "all") == 0)
	  maxitems0 = 0;
	else if (sscanf(t, "%d%n", &n1, &l) > 0 && l > 0 && !t[l] && n1 >= 0)
	  maxitems0 = n1;
	else {
	  debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
	  break;
	}
      }
      print_stack(thr, base, fno, lvals, xbp, addr, ip, modno, lineno, maxitems0);
      break;
    }
    case 'l': {
      long _modno = base?thr->asp[-base]->modno:modno;
      long _lineno = base?thr->asp[-base]->lineno:lineno;
      char *t = skipws(s+1);
      if (*t) {
	char c;
	int n, k, l = 0;
	if (sscanf(t, "%c%d%n", &c, &k, &l) > 0 && l > 0 &&
	    (c=='+' || c=='-')) {
	  if (c=='-') k = -k;
	  t = skipws(t+l);
	} else
	  k = noffs;
	if (*t) {
	  l = 0;
	  if (sscanf(t, "%d%n", &n, &l) > 0 && l > 0 && !t[l] && n >= 0)
	    ;
	  else {
	    debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
	    break;
	  }
	} else
	  n = nlines;
	noffs = k;
	nlines = n;
      }
      if (_lineno+noffs < 1) noffs = -(_lineno-1);
      print_source(_modno, _lineno+noffs, nlines);
      break;
    }
    case 'n':
      thr->debug = 0;
      stoplevel = thr->asp-thr->ast-base+1;
      goto exit;
    case 'c':
      set_debug(thr, 0);
      goto exit;
    case 'u': {
      int n = 1;
      char *t = skipws(s+1);
      if (*t) {
	int n1, l = 0;
	if (sscanf(t, "%d%n", &n1, &l) > 0 && l > 0 && !t[l] && n1 >= 0)
	  n = n1;
	else {
	  debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
	  break;
	}
      }
      if (base+n > thr->asp-thr->ast) n = thr->asp-thr->ast-base;
      if (n > 0)
	base += n;
      else
	error("Already at top of stack");
      goto reprint;
    }
    case 'd': {
      int n = 1;
      char *t = skipws(s+1);
      if (*t) {
	int n1, l = 0;
	if (sscanf(t, "%d%n", &n1, &l) > 0 && l > 0 && !t[l] && n1 >= 0)
	  n = n1;
	else {
	  debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
	  break;
	}
      }
      if (base < n) n = base;
      if (n > 0)
	base -= n;
      else
	error("Already at bottom of stack");
      goto reprint;
    }
    case 't':
      if (base < thr->asp-thr->ast)
	base = thr->asp-thr->ast;
      goto reprint;
    case 'b':
      if (base > 0)
	base = 0;
      goto reprint;
    case 'h':
      thr->qmstat = HALT;
      goto exit;
    case 'q':
      thr->qmstat = QUIT;
      goto exit;
    default:
      debug_eval(thr, base, fno, lvals, xbp, addr, ip, s);
      break;
    }
  
  goto command;
  
 exit:
  if (buf) free(buf);
  /* reinstall break handler: */
  pop_sigint();
#ifdef USE_THREADS
  release_tty();
#endif
  return stoplevel;
}

static bool _brkflag, _unparseflag;
static int _maxlevel, _maxlist, _maxchars;

static void set_print_params(void)
{
  _brkflag = brkflag;
  _unparseflag = unparseflag;
  _maxlevel = maxlevel;
  _maxlist = maxlist;
  _maxchars = maxchars;
  brkflag = 0;
  unparseflag = 0;
  maxlevel = maxlevel0;
  maxlist = maxlist0;
  maxchars = maxchars0;
}

static void reset_print_params(void)
{
  brkflag = _brkflag;
  unparseflag = _unparseflag;
  maxlevel = _maxlevel;
  maxlist = _maxlist;
  maxchars = _maxchars;
}

/* rhs varsym table */

static void clearvartb(THREAD *thr)
{
  int i;
  for (i = 0; i < thr->nvarsyms; i++)
    if (thr->vartb[i].pname)
      free(thr->vartb[i].pname);
  if (thr->vartb) {
    free(thr->vartb);
    thr->vartb = NULL;
  }
  thr->nvarsyms = thr->avarsyms = 0;
}

static int vareq(VARREC *v1, VARREC *v2)
{
  return v1->offs == v2->offs && v1->plen == v2->plen &&
    v1->p == v2->p;
}

static int vartbadd(THREAD *thr, VARREC *v, int vsym)
{
  char s[MAXSTRLEN];

  if (thr->nvarsyms >= thr->avarsyms) {
    VARREC *vartb1 = (VARREC*) arealloc(thr->vartb, thr->avarsyms, 10,
					sizeof(VARREC));
    if (!vartb1) return NONE;
    thr->vartb = vartb1; thr->avarsyms += 10;
  }
  if (vsym >= 0)
    strcpy(s, strsp+vsym);
  else
    /*sprintf(s, "$%d", thr->nvarsyms+1);*/
    strcpy(s, "_");
  thr->vartb[thr->nvarsyms].pname = strdup(s);
  thr->vartb[thr->nvarsyms].offs = v->offs;
  thr->vartb[thr->nvarsyms].plen = v->plen;
  thr->vartb[thr->nvarsyms].p = v->p;
  return thr->nvarsyms++;
}

static VARREC *vtb;
static
varcmp(v1, v2)
     int	       *v1, *v2;
{
  return strcmp(vtb[*v1].pname, vtb[*v2].pname);
}

static int listvars(THREAD *thr, int offs)
{
  int             i, n, *v;

  vtb = thr->vartb;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) n++;
  if (n == 0)
    return 1;
  else if (!(v = (int *)calloc(n, sizeof(int))))
    return 0;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) v[n++] = i;
  /* sort variables */
  qsort(v, n, sizeof(int), (int(*)())varcmp);
  /* remove duplicates */
  for (i = 0; i < n-1; )
    if (strcmp(vtb[v[i]].pname, vtb[v[i+1]].pname) == 0) {
      int j;
      for (j = i+2; j < n; j++) v[j-1] = v[j];
      n--;
    } else
      i++;
  /* print variables */
  for (i = 0; i < n; i++) {
    if (i > 0)
      if (i % 4 == 0)
	printf("\n");
      else
	printf("\t");
    printf("%-15s", utf8_to_sys(vtb[v[i]].pname));
  }
  printf("\n");
  free(v);
  return 1;
}

typedef struct {
  int sym;
  EXPR *x;
} display;

static display *dpy = NULL;
static int ndpy = 0;

static int savevars(THREAD *thr, int fno, EXPR *lvals[2], long xbp, int offs)
{
  int             i, n, *v;

  vtb = thr->vartb;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) n++;
  if (n == 0)
    return 1;
  else if (!(v = (int *)calloc(n, sizeof(int))))
    return 0;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) v[n++] = i;
  /* sort variables */
  qsort(v, n, sizeof(int), (int(*)())varcmp);
  /* remove duplicates */
  for (i = 0; i < n-1; )
    if (strcmp(vtb[v[i]].pname, vtb[v[i+1]].pname) == 0) {
      int j;
      if (vtb[v[i]].offs > vtb[v[i+1]].offs)
	for (j = i+2; j < n; j++) v[j-1] = v[j];
      else
	for (j = i+1; j < n; j++) v[j-1] = v[j];
      n--;
    } else
      i++;
  /* construct the display */
  if (!(dpy = (display*)calloc(n, sizeof(display)))) {
    free(v);
    return 0;
  }
  for (i = 0; i < n; i++) {
    int sym = mksym(vtb[v[i]].pname);
    if (sym == NONE ||
	!pushlval(thr, fno, lvals, xbp, vtb[v[i]].offs, vtb[v[i]].plen,
		  vtb[v[i]].p)) {
      free(v); ndpy = i;
      restorevars(thr);
      return 0;
    }
    dpy[i].sym = sym;
    dpy[i].x = symtb[sym].x;
    symtb[sym].x = *--thr->xsp;
  }
  free(v);
  ndpy = n;
  return 1;
}

static void restorevars(THREAD *thr)
{
  if (dpy) {
    int i;
    for (i = 0; i < ndpy; i++) {
      qmfree(thr, symtb[dpy[i].sym].x);
      symtb[dpy[i].sym].x = dpy[i].x;
    }
    free(dpy);
    dpy = NULL; ndpy = 0;
  }
}

static int getvar(THREAD *thr, char *name, int offs)
{
  int             i, last = NONE;

  if (strcmp(name, "_") == 0) return NONE;
  for (i = 0; i < thr->nvarsyms; i++)
    if (strcmp(thr->vartb[i].pname, name) == 0 &&
	thr->vartb[i].offs <= offs &&
	(last == NONE || thr->vartb[i].offs > thr->vartb[last].offs))
	last = i;
  return last;
}

static int mkvar(THREAD *thr, byte offs, byte plen, PATH p, int vsym)
{
  VARREC          v;
  int             i;

  v.offs = offs; v.plen = plen; v.p = p;
  for (i = 0; i < thr->nvarsyms; i++)
    if (vareq(thr->vartb+i, &v))
      break;
  if (i >= thr->nvarsyms)
    i = vartbadd(thr, &v, vsym);
  if (i != NONE && thr->vartb[i].pname)
    return mksym(thr->vartb[i].pname);
  else
    return NONE;
}

static int pushtmp(THREAD *thr, byte offs, byte plen, PATH p)
{
  int sym = mkvar(thr, offs, plen, p, -1);
  if (sym != NONE)
    return pushfun(thr, sym);
  else
    return 0;
}

static int arity(int fno)
{
  switch (fno) {
  case CONSOP:
  case PAIROP:
  case APPOP:
    return (2);
  default:
    return (0);
  }
}

static int s;
static byte _offs, _plen;
static PATH _p;

static int pushm(THREAD *thr)
{
  int k = statetb[s].trans;

  s = transtb[k].next;
  if (transtb[k].fno) {
    if (arity(transtb[k].fno)) {
      _plen++;
      setpath(&_p, _plen-1, 0);
      if (!pushm(thr)) return 0;
      setpath(&_p, _plen-1, 1);
      if (!pushm(thr)) return 0;
      setpath(&_p, _plen-1, 0);
      _plen--;
    }
    return pushfun(thr, transtb[k].fno);
  } else
    return pushtmp(thr, _offs, _plen, _p);
}

static int printm(THREAD *thr, int m, byte offs)
{
  if (m >= 0) {
    s = matchtb[m]; _offs = offs; _plen = 0; _p = 0;
    if (!pushm(thr)) return 0;
  } else if (!pushtmp(thr, offs, 0, 0))
    return 0;
  if (thr->xsp[-1]->fno == APPOP && thr->xsp[-1]->data.args.x1->fno == APPOP &&
      thr->xsp[-1]->data.args.x1->data.args.x1->fno == EQOP) {
    printf("(");
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    flush_shift();
    printf(")");
  } else {
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    flush_shift();
  }
  qmfree(thr, *--thr->xsp);
  return 1;
}

static void binding(THREAD *thr, int failed, int m, byte offs)
{
  push_sigint(SIG_IGN);
  flush_shift();
  check_thread(thr);
  dbg_stack(thr, 10);
  printf("--  def ");
  if (printm(thr, m, offs)) {
    printf(" = ");
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    flush_shift();
    if (failed) printf("  :( FAILED )");
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void reduction(THREAD *thr, int fno, long xbp)
{
  push_sigint(SIG_IGN);
  flush_shift();
  check_thread(thr);
  dbg_stack(thr, 10);
  if (fno == APPOP && thr->xst[xbp]->fno == APPOP &&
      thr->xst[xbp]->data.args.x1->fno == DEFOP) {
    EXPR *p = thr->xst[xbp]->data.args.x2, *v = thr->xst[xbp+1];
    printf("--  def ");
    if (p->fno == APPOP && p->data.args.x1->fno == APPOP &&
	p->data.args.x1->data.args.x1->fno == EQOP) {
      printf("(");
      set_print_params(), printx(p), reset_print_params();
      flush_shift();
      printf(")");
    } else {
      set_print_params(), printx(p), reset_print_params();
      flush_shift();
    }
    printf(" = ");
    set_print_params(), printx(v), reset_print_params();
    flush_shift();
  } else if (fno == APPOP &&
	     thr->xst[xbp]->fno == UNDEFOP) {
    char s[MAXSTRLEN];
    printf("--  undef %s",
	   utf8_to_sys(pname(s, thr->xst[xbp+1]->fno)));
  } else {
    printf("**  ");
    if (printp(thr, fno, xbp)) {
      printf("  ==>  ");
      set_print_params(), printx(thr->xsp[-1]), reset_print_params();
      flush_shift();
    }
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void default_reduction(THREAD *thr, int fno, long xbp)
{
  push_sigint(SIG_IGN);
  flush_shift();
  check_thread(thr);
  dbg_stack(thr, 10);
  if (fno == APPOP && thr->xst[xbp]->fno == APPOP &&
      thr->xst[xbp]->data.args.x1->fno == DEFOP) {
    EXPR *p = thr->xst[xbp]->data.args.x2, *v = thr->xst[xbp+1];
    printf("--  def ");
    if (p->fno == APPOP && p->data.args.x1->fno == APPOP &&
	p->data.args.x1->data.args.x1->fno == EQOP) {
      printf("(");
      set_print_params(), printx(p), reset_print_params();
      flush_shift();
      printf(")");
    } else {
      set_print_params(), printx(p), reset_print_params();
      flush_shift();
    }
    printf(" = ");
    set_print_params(), printx(v), reset_print_params();
    flush_shift();
    printf("  :( FAILED )");
  } else if (fno == APPOP &&
	     thr->xst[xbp]->fno == UNDEFOP) {
    char s[MAXSTRLEN];
    printf("-- undef %s  :( FAILED )",
	   utf8_to_sys(pname(s, thr->xst[xbp+1]->fno)));
  } else {
#if 1
    return;
#else
    printf("**  ");
    printp(thr, fno, xbp);
#endif
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void tail_reduction(THREAD *thr, int fno, long xbp, int fno1)
{
  push_sigint(SIG_IGN);
  flush_shift();
  check_thread(thr);
  dbg_stack(thr, 10);
  printf("++  "); 
  if (printp(thr, fno, xbp)) {
    printf("  ==>  ");
    printp(thr, fno1, (fno1<BINARY)?thr->xsp-thr->xst-2:0);
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static int printp(THREAD *thr, int fno, long xbp)
{
  char		s[MAXSTRLEN];
  if (fno < BINARY) {
    if (push(thr, thr->xst[xbp]))
      if (push(thr, thr->xst[xbp+1]))
	if (pushfun(thr, fno)) {
	  set_print_params(), printx(thr->xsp[-1]), reset_print_params();
	  flush_shift();
	  qmfree(thr, *--thr->xsp);
	} else {
	  qmfree(thr, *--thr->xsp);
	  qmfree(thr, *--thr->xsp);
	  return 0;
	}
      else {
	qmfree(thr, *--thr->xsp);
	return 0;
      }
    else
      return 0;
  } else
    printf("%s", utf8_to_sys(pname(s, fno)));
  return 1;
}

static int xlat_pops(THREAD *thr, EXPR **mark)
{
  if (thr->xsp-mark > 1) {
    /* translate "dangling POPs" (toplevel ||) to ordinary applications */
    EXPR *x, *y, **act;
    for (x = *mark, act = mark+1; act < thr->xsp; act++) {
      if ((y = consexpr(thr, APPOP, funexpr(thr, SEQOP), x)) != NULL)
	x = y;
      else {
	if (x != *mark) qmfree(thr, x);
	return 0;
      }
      if ((y = consexpr(thr, APPOP, x, *act)) != NULL)
	x = y;
      else {
	qmfree(thr, x);
	return 0;
      }
    }
    while (thr->xsp > mark) qmfree(thr, *--thr->xsp);
    push(thr, x);
    return 1;
  } else
    return 1;
}

static inline int get_offs(int addr)
{
  OPREC	       *ip = codesp + addr;
  if (ip->opcode == INFOP)
    return ip->opargs.info.offs;
  else
    /* this shouldn't happen */
    return 0;
}

static inline int locate_addr(int addr, OPREC *ip0)
{
  int addr1 = ip0-codesp;
  if (addr > addr1) return addr;
  while (codesp[addr].opcode == INFOP &&
	 codesp[addr].opargs.info.addr >= 0 &&
	 codesp[addr].opargs.info.addr <= addr1)
    addr = codesp[addr].opargs.info.addr;
  return addr;
}

static int buildvartb(THREAD *thr, int addr, OPREC *ip0)
{
  OPREC	       *ip;

  addr = locate_addr(addr, ip0);
  if (thr->lastaddr == addr) return 1;

  thr->lastaddr = addr;
  clearvartb(thr);

  /* enter variables into the var table: */
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    if (ip->opcode == LVALOP && ip->opargs.lval.vsym >= 0)
	if (!mkvar(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		   ip->opargs.lval.p, ip->opargs.lval.vsym))
	  return 0;
  return 1;
}

static int lastoffs(THREAD *thr, int addr, OPREC *ip0)
{
  OPREC *ip;
  int p = 0, addr1 = locate_addr(addr, ip0);
  if (addr1 != addr) {
    p = get_offs(addr);
    addr = addr1;
  }
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    if (ip->opcode == MATCHOP)
      if (ip <= ip0)
	p++;
      else
	return p;
  return p;
}

static int printc(THREAD *thr, int fno, EXPR *lvals[2],
		  long xbp, int addr, OPREC *ip0)
{
  long          top = thr->xsp-thr->xst;
  int           maxoffs = (fno==APPOP)?(top-xbp-2):(top-xbp);
  OPREC	       *ip;
  int		p = 0, q = 0, m = -2, addr1 = codesp[addr].opargs.info.addr;

  if (!buildvartb(thr, addr, ip0)) return 0;

  /* count shared local variable defs */
  if (addr1 >= 0 && ip0-codesp >= addr1)
    p = codesp[addr].opargs.info.offs;

  addr = locate_addr(addr, ip0);

  /* build the rhs on the stack: */
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    switch(ip->opcode) {
    case INFOP:
    case RETOP:
      break;
    case POPOP:
      /* these are treated later */
      break;
    case LVALOP:
      if (ip->opargs.lval.offs > p) {
	if (!pushtmp(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		     ip->opargs.lval.p))
	  return 0;
      } else {
	if (ip->opargs.lval.offs)
	  mkvar(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		ip->opargs.lval.p, -1);
	if (!pushlval(thr, fno, lvals, xbp, ip->opargs.lval.offs,
		      ip->opargs.lval.plen,
		      ip->opargs.lval.p))
	  return 0;
      }
      break;
    case QUERYOP:
      if (!xlat_pops(thr, thr->xst+top+q)) return 0;
      if (ip < ip0 || q)
	qmfree(thr, *--thr->xsp);
      else
	q++;
      break;
    case MATCHOP:
      if (!xlat_pops(thr, thr->xst+top+q)) return 0;
      if (ip < ip0) p++;
      if (ip < ip0 || q)
	qmfree(thr, *--thr->xsp);
      else {
	m = ip->opargs.qual.m;
	q++;
      }
      break;
    case INTVALOP:
      { mpz_t z;
      if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	thr->qmstat = MEM_OVF;
	return 0;
      } else if (!pushmpz(thr, z))
	return (0);
      }
      break;
    case FLOATVALOP:
      if (!pushfloat(thr, ip->opargs.fv))
	return 0;
      break;
    case STRVALOP:
      {
	char           *s;
	if ((s = strdup(strsp + ip->opargs.sv)) ==
	    NULL) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushstr(thr, s))
	  return 0;
      }
      break;
    default:
      if (!pushfun(thr, ip->opcode))
	return 0;
      break;
    }
  if (!xlat_pops(thr, thr->xst+top+q)) return 0;
  thr->qmstat = OK;
  set_print_params(), printx(thr->xsp[-1]), reset_print_params();
  flush_shift();
  qmfree(thr, *--thr->xsp);
  if (q) {
    if (m >= -1) {
      printf(" where ");
      if (printm(thr, m, p+1)) {
	printf(" = ");
	set_print_params(), printx(thr->xsp[-1]), reset_print_params();
	flush_shift();
      }
    } else {
      /* suppress generated IDOP queries: */
      EXPR *x;
      for (x = thr->xsp[-1]; x->fno == APPOP; x = x->data.args.x1) ;
      if (x->fno != IDOP) {
	printf(" if ");
	set_print_params(), printx(thr->xsp[-1]), reset_print_params();
	flush_shift();
      }
    }
    qmfree(thr, *--thr->xsp);
  }
  return 1;
}

/* STRING SPACE: */

static int putstr(char *s)
{
  int             k = strspsz + tmpspsz, l = strlen(s);

  while (l >= strspsz + atmpspsz - k) {
    char *strsp1;
    if (!(strsp1 = (char*) arealloc(strsp, strspsz+atmpspsz,
				    TMPSPSZ, sizeof(char))))
      return(NONE);
    else {
      strsp = strsp1;
      atmpspsz += TMPSPSZ;
    }
  }
  strcpy(strsp+k, s);
  tmpspsz += l + 1;
  return (k);
}

/* SYMBOL TABLE: */

static int strhash(char *s, int sz)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h % sz;
}

#define streq(s1,s2) (!strcmp(s1,s2))

#define NIL (-2)
#define ANY (-3)

static int splitid(char *s, char *mnm)
{
  char *p;
  int mno = ANY;
  *mnm = 0;
  if ((p = strstr(s, "::"))) {
    char t[MAXSTRLEN];
    strcpy(t, p+2);
    *p = 0;
    strcpy(mnm, s);
    if (!*s)
      mno = NONE;
    else {
      mno = getmodno(s);
      if (mno == NONE) mno = NIL;
    }
    strcpy(s, t);
  }
  return mno;
}

static int symprio(int sym)
{
  if (symtb[sym].modno == NONE)
    return -1;
  else if (globs[symtb[sym].modno] & 2)
    return 0;
  else
    return 1;
}

static int isvsym(char *s)
{
  if (!*s)
    return 0;
  else {
#ifdef HAVE_UNICODE
    long c = u8decode(s);
    if (c < 0) c = (unsigned char)*s;
    return u_isupper(c);
#else
    return isupper(s[0]);
#endif
  }
}

/* mksym(): look up a (possibly qualified) function or variable symbol in the
   global scope, and create a new symbol if the symbol does not yet exist;
   sets qmstat to either BAD_REF, BAD_SYM or SYMTB_OVF if symbol is not
   defined or could not be created */

/* CAVEAT: Thrashes the given symbol name if it contains a qualifier, by
   splitting up the symbol in qualifier and id part. Make sure you back up the
   original string if necessary. */

#define matchsym(fno,s) (!(symtb[fno].flags & TSYM)&&\
			 streq((s), strsp+symtb[fno].pname))

int mksym(char *s)
{
  char mnm[MAXSTRLEN];
  int modno = (mainno>=0)?mainno:0, mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE, pname;

  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all global imports */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   globs[symtb[fno].modno]))
	if (symtb[fno].modno == modno) {
	  /* found symbol in main module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  int r1 = fno1, r = fno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (symprio(fno) == symprio(fno1)) {
	    /* multiple imports, error */
	    get_thr()->qmstat = BAD_REF;
	    return NONE;
	  } else
	    break;
	} else
	  fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) && symtb[fno].modno == mno) {
	fno1 = fno;
	break;
      }
  }
  fno = fno1;
  if (fno == NONE) {
    if (mno != ANY) {
      /* undefined symbol, not in global namespace */
      get_thr()->qmstat = BAD_SYM;
      return NONE;
    }
    if (symtbsz+tmptbsz > SHRT_MAX) {
      get_thr()->qmstat = SYMTB_OVF;
      return NONE;
    }
    if (tmptbsz >= atmptbsz) {
      SYMREC *symtb1;
      if (!(symtb1 =
	    (SYMREC*)arealloc(symtb,
			      symtbsz+atmptbsz, TMPTBSZ/10,
			      sizeof(SYMREC)))) {
	get_thr()->qmstat = SYMTB_OVF;
	return NONE;
      } else {
	symtb = symtb1;
	atmptbsz += TMPTBSZ/10;
      }
    }
    if ((pname = putstr(s)) == NONE) {
      get_thr()->qmstat = MEM_OVF;
      return NONE;
    }
    fno = symtbsz+tmptbsz++;
    symtb[fno].ref = 0;
    symtb[fno].prec = NONE;
    symtb[fno].flags = isvsym(s)?VSYM:0;
    symtb[fno].type = 0;
    symtb[fno].fno_min = symtb[fno].fno_max = 0;
    symtb[fno].argc = 0;
    symtb[fno].argv = 0;
    symtb[fno].modno = mainno>=0?mainno:modtbsz>0?0:NONE;
    symtb[fno].lineno = 0;
    symtb[fno].xfno = fno;
    symtb[fno].pname = pname;
    symtb[fno].x = symtb[fno].xp = NULL;
    symtb[fno].f = NULL;
    symtb[fno].next = hashtb[k];
    hashtb[k] = fno;
    symtb[fno].flags |= UNIQ|VIS;
  } else {
    while (symtb[fno].ref)
      fno = symtb[fno].ref;
  }
  return fno;
}

/* newsym(): create a new (unqualified) function or variable symbol in the
   global namespace, return the symbol or NONE if the symbol couldn't be
   created */

int newsym(char *s)
{
  int k = strhash(s, hashtbsz);
  int fno, pname;

  if (symtbsz+tmptbsz > SHRT_MAX) {
    get_thr()->qmstat = SYMTB_OVF;
    return NONE;
  }
  if (tmptbsz >= atmptbsz) {
    SYMREC *symtb1;
    if (!(symtb1 =
	  (SYMREC*)arealloc(symtb,
			    symtbsz+atmptbsz, TMPTBSZ/10,
			    sizeof(SYMREC)))) {
      get_thr()->qmstat = SYMTB_OVF;
      return NONE;
    } else {
      symtb = symtb1;
      atmptbsz += TMPTBSZ/10;
    }
  }
  if ((pname = putstr(s)) == NONE) {
    get_thr()->qmstat = MEM_OVF;
    return NONE;
  }
  fno = symtbsz+tmptbsz++;
  symtb[fno].ref = 0;
  symtb[fno].prec = NONE;
  symtb[fno].flags = isvsym(s)?VSYM:0;
  symtb[fno].type = 0;
  symtb[fno].fno_min = symtb[fno].fno_max = 0;
  symtb[fno].argc = 0;
  symtb[fno].argv = 0;
  symtb[fno].modno = mainno>=0?mainno:modtbsz>0?0:NONE;
  symtb[fno].lineno = 0;
  symtb[fno].xfno = fno;
  symtb[fno].pname = pname;
  symtb[fno].x = symtb[fno].xp = NULL;
  symtb[fno].f = NULL;
  symtb[fno].next = hashtb[k];
  hashtb[k] = fno;
  symtb[fno].flags |= UNIQ|VIS;
  return fno;
}

/* getsym(): look up a (possibly qualified) function or variable symbol in the
   context of a given module, return the symbol or NONE if it does not
   exist */

static int symprio2(int p, int sym)
{
  if (symtb[sym].modno == NONE)
    return -1;
  else if (impib[p] & 2)
    return 0;
  else
    return 1;
}

static int searchimp(int modno, int mno)
{
  int i, i1 = imports[modno],
    i2 = (modno+1<modtbsz)?imports[modno+1]:imptbsz;
  for (i = i1; i < i2; i++)
    if (imptb[i] == mno)
      return i;
  return NONE;
}

int getsym(char *s, int modno)
{
  char mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE, p = NONE, p1;

  if (mno >= 0) {
    /* check that module is in local scope */
    if (searchimp(modno, mno) == NONE)
      mno = NIL;
  }
  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all local imports */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   (p = searchimp(modno, symtb[fno].modno)) != NONE))
	if (symtb[fno].modno == modno) {
	  /* found symbol in this module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  if (symprio2(p,fno) == symprio2(p1,fno1)) {
	    /* multiple imports, error */
	    return NONE;
	  } else
	    break;
	} else
	  p1 = p, fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) && symtb[fno].modno == mno) {
	if (!(symtb[fno].flags & PRIV) || mno == modno)
	  fno1 = fno;
	break;
      }
  }
  if (fno1 != NONE) {
    while (symtb[fno1].ref)
      fno1 = symtb[fno1].ref;
  }
  return fno1;
}

/* gettype(): look up a (possibly qualified) type symbol in the context of a
   given module, return the symbol or NONE if it does not exist */

#define matchtyp(type,s) ((symtb[type].flags & TSYM)&&\
			   streq((s), strsp+symtb[type].pname))

int gettype(char *s, int modno)
{
  char mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int type, type1 = NONE, p = NONE, p1;

  if (mno >= 0) {
    /* check that module is in local scope */
    if (searchimp(modno, mno) == NONE)
      mno = NIL;
  }
  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all local imports */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) &&
	  (symtb[type].modno == NONE || symtb[type].modno == modno ||
	   (p = searchimp(modno, symtb[type].modno)) != NONE))
	if (symtb[type].modno == modno) {
	  /* found symbol in this module, done */
	  type1 = type;
	  break;
	} else if (symtb[type].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (type1 != NONE) {
	  if (symprio2(p,type) == symprio2(p1,type1)) {
	    /* multiple imports, error */
	    return NONE;
	  } else
	    break;
	} else
	  p1 = p, type1 = type;
  } else {
    /* look for qualified symbol in given module */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) && symtb[type].modno == mno) {
	if (!(symtb[type].flags & PRIV) || mno == modno)
	  type1 = type;
	break;
      }
  }
  return type1;
}

/* getmodno() returns the module index of a module name, NONE if
   nonexistent. */

int getmodno(char *name)
{
  int i;
  for (i = 0; i < modtbsz; i++)
    if (streq(name, strsp+modtb[i]))
      return i;
  return NONE;
}

