import { IRenderMime } from '@jupyterlab/rendermime-interfaces';
import { ITranslator } from '@jupyterlab/translation';
import { Message } from '@lumino/messaging';
import { Widget } from '@lumino/widgets';
/**
 * A common base class for mime renderers.
 */
export declare abstract class RenderedCommon extends Widget implements IRenderMime.IRenderer {
    /**
     * Construct a new rendered common widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * The mimetype being rendered.
     */
    readonly mimeType: string;
    /**
     * The sanitizer used to sanitize untrusted html inputs.
     */
    readonly sanitizer: IRenderMime.ISanitizer;
    /**
     * The resolver object.
     */
    readonly resolver: IRenderMime.IResolver | null;
    /**
     * The link handler.
     */
    readonly linkHandler: IRenderMime.ILinkHandler | null;
    /**
     * The latexTypesetter.
     */
    readonly latexTypesetter: IRenderMime.ILatexTypesetter | null;
    /**
     * The markdownParser.
     */
    readonly markdownParser: IRenderMime.IMarkdownParser | null;
    /**
     * The translator.
     */
    readonly translator: ITranslator;
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @param keepExisting - Whether to keep the existing rendering.
     *
     * @returns A promise which resolves when rendering is complete.
     *
     * #### Notes
     * By default, if the DOM node for this widget already has content, it
     * is emptied before rendering. Subclasses that do not want this behavior
     * (if, for instance, they are using DOM diffing), should override this
     * method or call `super.renderModel(model, true)`.
     */
    renderModel(model: IRenderMime.IMimeModel, keepExisting?: boolean): Promise<void>;
    /**
     * Render the mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    abstract render(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * Set the URI fragment identifier.
     *
     * @param fragment - The URI fragment identifier.
     */
    protected setFragment(fragment: string): void;
}
/**
 * A common base class for HTML mime renderers.
 */
export declare abstract class RenderedHTMLCommon extends RenderedCommon {
    /**
     * Construct a new rendered HTML common widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    setFragment(fragment: string): void;
}
/**
 * A mime renderer for displaying HTML and math.
 */
export declare class RenderedHTML extends RenderedHTMLCommon {
    /**
     * Construct a new rendered HTML widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * A message handler invoked on an `'after-attach'` message.
     */
    onAfterAttach(msg: Message): void;
}
/**
 * A mime renderer for displaying LaTeX output.
 */
export declare class RenderedLatex extends RenderedCommon {
    /**
     * Construct a new rendered LaTeX widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * A message handler invoked on an `'after-attach'` message.
     */
    onAfterAttach(msg: Message): void;
}
/**
 * A mime renderer for displaying images.
 */
export declare class RenderedImage extends RenderedCommon {
    /**
     * Construct a new rendered image widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
}
/**
 * A mime renderer for displaying Markdown with embedded latex.
 */
export declare class RenderedMarkdown extends RenderedHTMLCommon {
    /**
     * Construct a new rendered markdown widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    renderModel(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * A message handler invoked on an `'after-attach'` message.
     */
    onAfterAttach(msg: Message): void;
}
/**
 * A widget for displaying SVG content.
 */
export declare class RenderedSVG extends RenderedCommon {
    /**
     * Construct a new rendered SVG widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
    /**
     * A message handler invoked on an `'after-attach'` message.
     */
    onAfterAttach(msg: Message): void;
}
/**
 * A widget for displaying plain text and console text.
 */
export declare class RenderedText extends RenderedCommon {
    /**
     * Construct a new rendered text widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
}
/**
 * A widget for displaying JavaScript output.
 */
export declare class RenderedJavaScript extends RenderedCommon {
    /**
     * Construct a new rendered text widget.
     *
     * @param options - The options for initializing the widget.
     */
    constructor(options: IRenderMime.IRendererOptions);
    /**
     * Render a mime model.
     *
     * @param model - The mime model to render.
     *
     * @returns A promise which resolves when rendering is complete.
     */
    render(model: IRenderMime.IMimeModel): Promise<void>;
}
