#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (C) Duncan Macleod (2014-2020)
#
# This file is part of GWpy.
#
# GWpy is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# GWpy is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GWpy.  If not, see <http://www.gnu.org/licenses/>.

"""Plotting an `EventTable` as 2-d tiles

I would like to study the event triggers generated by the `ExcessPower <link>`_
gravitational-wave burst detection algorithm, over a small stretch of data.

The data from which these events were generated contain a simulated
gravitational-wave signal, or hardware injection, used to validate
the performance of the LIGO detectors and downstream data analysis procedures.
"""

__author__ = "Duncan Macleod <duncan.macleod@ligo.org>"
__currentmodule__ = 'gwpy.table'

# First, we import the `EventTable` object and read in a set of events from
# a LIGO_LW-format XML file containing a
# :class:`sngl_burst <ligo.lw.lsctables.SnglBurstTable>` table
from gwpy.table import EventTable
events = EventTable.read(
    'H1-LDAS_STRAIN-968654552-10.xml.gz', tablename='sngl_burst',
    columns=['peak', 'central_freq', 'bandwidth', 'duration', 'snr'])

# .. note::
#
#    Here we manually specify the `columns` to read in order to optimise
#    the `read()` operation to parse only the data we actually need.

# We can make a plot of these events as 2-dimensional tiles by specifying
# the x- and y-axis columns, and the widths in those directions:
plot = events.tile('peak', 'central_freq', 'duration', 'bandwidth',
                   color='snr')
ax = plot.gca()
ax.set_yscale('log')
ax.set_ylabel('Frequency [Hz]')
ax.set_epoch(968654552)
ax.set_xlim(968654552, 968654552+10)
ax.set_title('LIGO-Hanford event tiles for HW100916')
ax.colorbar(clim=[1, 8], cmap='YlGnBu', label='Signal-to-noise ratio (SNR)')
plot.show()
