// SPDX-License-Identifier: GPL-3.0-or-later

package functions

import (
	"context"
	"encoding/json"
	"fmt"
	"log/slog"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/netdata/netdata/go/plugins/logger"
	"github.com/netdata/netdata/go/plugins/pkg/netdataapi"
	"github.com/netdata/netdata/go/plugins/pkg/safewriter"
)

type functionSet struct {
	direct   func(Function)            // for globally-unique names
	prefixes map[string]func(Function) // for prefix-multiplexed names
}

func NewManager() *Manager {
	return &Manager{
		Logger: logger.New().With(
			slog.String("component", "functions manager"),
		),
		api:              netdataapi.New(safewriter.Stdout),
		input:            stdinInput,
		mux:              &sync.Mutex{},
		FunctionRegistry: make(map[string]*functionSet),
	}
}

type Manager struct {
	*logger.Logger

	api *netdataapi.API

	input input

	mux              *sync.Mutex
	FunctionRegistry map[string]*functionSet
}

func (m *Manager) Run(ctx context.Context, quitCh chan struct{}) {
	m.Info("instance is started")
	defer func() { m.Info("instance is stopped") }()

	var wg sync.WaitGroup

	wg.Add(1)
	go func() { defer wg.Done(); m.run(ctx, quitCh) }()

	wg.Wait()

	<-ctx.Done()
}

func (m *Manager) run(ctx context.Context, quitCh chan struct{}) {
	parser := newInputParser()

	for {
		select {
		case <-ctx.Done():
			return
		case line, ok := <-m.input.lines():
			if !ok {
				return
			}
			if line == "QUIT" {
				if quitCh != nil {
					quitCh <- struct{}{}
					return
				}
				continue
			}

			fn, err := parser.parse(line)
			if err != nil {
				m.Warningf("parse function: %v ('%s')", err, line)
				continue
			}
			if fn == nil {
				continue
			}

			handler, ok := m.lookupFunction(fn.Name)
			if !ok {
				m.Infof("skipping execution of '%s': unregistered function", fn.Name)
				m.respf(fn, 501, "unregistered function: %s", fn.Name)
				continue
			}
			if handler == nil {
				m.Warningf("skipping execution of '%s': nil function registered", fn.Name)
				m.respf(fn, 501, "nil function: %s", fn.Name)
				continue
			}

			handler(*fn)
		}
	}
}

func (m *Manager) lookupFunction(name string) (func(Function), bool) {
	m.mux.Lock()
	fs, ok := m.FunctionRegistry[name]
	m.mux.Unlock()

	if !ok || fs == nil {
		return nil, false
	}

	return func(f Function) {
		if len(fs.prefixes) > 0 {
			m.handlePrefixRouting(f, fs)
			return
		}

		if fs.direct != nil {
			fs.direct(f)
			return
		}

		m.respf(&f, 503, "unknown function '%s' (%v)", f.Name, f.Args)
	}, true
}

func (m *Manager) handlePrefixRouting(f Function, fs *functionSet) {
	if len(f.Args) == 0 {
		m.respf(&f, 503, "unknown function '%s' (%v)", f.Name, f.Args)
		return
	}

	id := f.Args[0]
	for prefix, handler := range fs.prefixes {
		if strings.HasPrefix(id, prefix) {
			handler(f)
			return
		}
	}

	m.respf(&f, 503, "unknown function '%s' (%v)", f.Name, f.Args)
}

func (m *Manager) respf(fn *Function, code int, msgf string, a ...any) {
	bs, _ := json.Marshal(struct {
		Status  int    `json:"status"`
		Message string `json:"message"`
	}{
		Status:  code,
		Message: fmt.Sprintf(msgf, a...),
	})

	m.api.FUNCRESULT(netdataapi.FunctionResult{
		UID:             fn.UID,
		ContentType:     "application/json",
		Payload:         string(bs),
		Code:            strconv.Itoa(code),
		ExpireTimestamp: strconv.FormatInt(time.Now().Unix(), 10),
	})
}
