//===-- Utilities for trigonometric functions -------------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_H
#define LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_H

#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/nearest_integer.h"

namespace __llvm_libc {

namespace generic {

static constexpr uint32_t FAST_PASS_BOUND = 0x4a80'0000U; // 2^22

static constexpr int N_ENTRIES = 8;

// We choose to split bits of 16/pi into 28-bit precision pieces, so that the
// product of x * SIXTEEN_OVER_PI_28[i] is exact.
// These are generated by Sollya with:
// > a1 = D(round(16/pi, 28, RN)); a1;
// > a2 = D(round(16/pi - a1, 28, RN)); a2;
// > a3 = D(round(16/pi - a1 - a2, 28, RN)); a3;
// > a4 = D(round(16/pi - a1 - a2 - a3, 28, RN)); a4;
// ...
static constexpr double SIXTEEN_OVER_PI_28[N_ENTRIES] = {
    0x1.45f306ep+2,   -0x1.b1bbeaep-29,  0x1.3f84ebp-58,    -0x1.7056592p-88,
    0x1.c0db62ap-117, -0x1.4cd8778p-146, -0x1.bef806cp-175, 0x1.63abdecp-205};

// Exponents of the least significant bits of the corresponding entries in
// SIXTEEN_OVER_PI_28.
static constexpr int SIXTEEN_OVER_PI_28_LSB_EXP[N_ENTRIES] = {
    -25, -56, -82, -115, -144, -171, -201, -231};

// Return k and y, where
//   k = round(x * 16 / pi) and y = (x * 16 / pi) - k.
static inline int64_t small_range_reduction(double x, double &y) {
  double prod = x * SIXTEEN_OVER_PI_28[0];
  double kd = fputil::nearest_integer(prod);
  y = prod - kd;
  y = fputil::multiply_add(x, SIXTEEN_OVER_PI_28[1], y);
  y = fputil::multiply_add(x, SIXTEEN_OVER_PI_28[2], y);
  return static_cast<int64_t>(kd);
}

// Return k and y, where
//   k = round(x * 16 / pi) and y = (x * 16 / pi) - k.
// For large range, there are at most 2 parts of SIXTEEN_OVER_PI_28 contributing
// to the lowest 5 binary digits (k & 31).  If the least significant bit of
// x * the least significant bit of SIXTEEN_OVER_PI_28[i] >= 32, we can
// completely ignore SIXTEEN_OVER_PI_28[i].
static inline int64_t large_range_reduction(double x, int x_exp, double &y) {
  int idx = 0;
  y = 0;
  int x_lsb_exp_m4 = x_exp - fputil::FloatProperties<float>::MANTISSA_WIDTH;

  // Skipping the first parts of 16/pi such that:
  //   LSB of x * LSB of SIXTEEN_OVER_PI_28[i] >= 32.
  while (x_lsb_exp_m4 + SIXTEEN_OVER_PI_28_LSB_EXP[idx] > 4)
    ++idx;

  double prod_hi = x * SIXTEEN_OVER_PI_28[idx];
  // Get the integral part of x * SIXTEEN_OVER_PI_28[idx]
  double k_hi = fputil::nearest_integer(prod_hi);
  // Get the fractional part of x * SIXTEEN_OVER_PI_28[idx]
  double frac = prod_hi - k_hi;
  double prod_lo = fputil::multiply_add(x, SIXTEEN_OVER_PI_28[idx + 1], frac);
  double k_lo = fputil::nearest_integer(prod_lo);

  // Now y is the fractional parts.
  y = prod_lo - k_lo;
  y = fputil::multiply_add(x, SIXTEEN_OVER_PI_28[idx + 2], y);
  y = fputil::multiply_add(x, SIXTEEN_OVER_PI_28[idx + 3], y);

  return static_cast<int64_t>(k_hi) + static_cast<int64_t>(k_lo);
}

// Exceptional cases.
static constexpr int N_EXCEPTS = 3;

static constexpr fputil::ExceptionalValues<float, N_EXCEPTS> SinfExcepts{
    /* inputs */ {
        0x3fa7832a, // x = 0x1.4f0654p0
        0x46199998, // x = 0x1.33333p13
        0x55cafb2a, // x = 0x1.95f654p44
    },
    /* outputs (RZ, RU offset, RD offset, RN offset) */
    {
        {0x3f7741b5, 1, 0, 1}, // x = 0x1.4f0654p0, sin(x) = 0x1.ee836ap-1 (RZ)
        {0xbeb1fa5d, 0, 1, 0}, // x = 0x1.33333p13, sin(x) = -0x1.63f4bap-2 (RZ)
        {0xbf7e7a16, 0, 1,
         1}, // x = 0x1.95f654p44, sin(x) = -0x1.fcf42cp-1 (RZ)
    }};

} // namespace generic

} // namespace __llvm_libc

#endif // LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_H
