//
//  Book.swift
//  SwiftyXL
//
//  Created by Dmitry Rodionov on 17/10/2016.
//  Copyright © 2016 Internals Exposed. All rights reserved.
//

import Cocoa
import Libxl

public enum BookError: Error {
    case InvalidName
    case InternalError(message: String)
}

public struct LicenseInformation {
    public let name: String
    public let key: String

    public init(name: String, key: String)
    {
        self.name = name
        self.key = key
    }
}

public class Book {

    let handle: BookHandle
    public let format: Book.FileFormat

    // MARK: - Initialization

    public enum FileFormat {
        case Binary
        case XML
    }

    /// Initialize a workbook with the given file format (XML by default)
    public init(format: Book.FileFormat = .XML)
    {
        switch format {
        case .Binary:
            handle = xlCreateBookCA()
        case .XML:
            handle = xlCreateXMLBookCA()
        }
        self.format = format
    }

    /// Initialize a workbook with the given file format (XML by default) and license information
    public init(format: Book.FileFormat = .XML, license: LicenseInformation)
    {
        switch format {
        case .Binary:
            handle = xlCreateBookCA()
        case .XML:
            handle = xlCreateXMLBookCA()
        }
        self.format = format
        use(license: license)
    }

    /// Initialize a workbook with the given file format (XML by default) and contents of the given file
    public init(format: Book.FileFormat = .XML, withContentsOfFile file: String) throws
    {
        switch format {
        case .Binary:
            handle = xlCreateBookCA()
        case .XML:
            handle = xlCreateXMLBookCA()
        }
        self.format = format
        try load(fromFile: file)
    }

    deinit {
        xlBookReleaseA(handle)
    }

    // MARK: - Properties

    public enum ColorMode: Int {
        case Index = 0
        case RGB = 1
    }
    /// A color mode used in this workbook
    public var rgbMode: Book.ColorMode {
        get {
            // We explicitly unwrap optional here because this is a programming error
            return Book.ColorMode(rawValue: Int(xlBookRgbModeA(handle)))!
        }
        set {
            xlBookSetRgbModeA(handle, Int32(newValue.rawValue))
        }
    }

    /// Whether the R1C1 reference mode is active for this workbook
    public var R1C1Mode: Bool {
        get {
            return Bool(xlBookRefR1C1A(handle) == 1)
        }
        set {
            xlBookSetRefR1C1A(handle, newValue ? 1 : 0)
        }
    }

    /// Returns BIFF version of binary file. Returns 0 for XML workbooks.
    public var biffVersion: Int {
        get {
            return Int(xlBookBiffVersionA(handle))
        }
    }

    public enum DateMode: Int {
        /// The lower date limit is January 1, 1900
        case Date1900 = 0
        /// The lower date limit is January 1, 1904
        case Date1904 = 1
    }

    /// A date system used in this workbook
    public var dateMode: Book.DateMode {
        get {
            // We explicitly unwrap optional here because this is a programming error
            return Book.DateMode(rawValue: Int(xlBookIsDate1904A(handle)))!
        }
        set {
            xlBookSetDate1904A(handle, Int32(newValue.rawValue))
        }
    }

    /// Is this workbook a template or not
    public var template: Bool {
        get {
            return Bool(xlBookIsTemplateA(handle) == 1)
        }
        set {
            xlBookSetTemplateA(handle, newValue ? 1 : 0)
        }
    }

    /// Returns the last error message generated by a previous failed operation
    public var lastErrorMessage: String {
        let raw: UnsafePointer<Int8>? = xlBookErrorMessageA(handle)
        return raw.map { String(cString: $0) } ?? ""
    }

    // MARK: - License

    /// Sets a customer's license information
    public func use(license: LicenseInformation)
    {
        license.name.utf8CString.withUnsafeBufferPointer { nameBuffer in
            license.key.utf8CString.withUnsafeBufferPointer { keyBuffer in
                xlBookSetKeyA(handle, nameBuffer.baseAddress, keyBuffer.baseAddress)
            }
        }
    }

    /// Sets a locale used for this workbook.
    /// The locale argument is the same as the locale argument in setlocale() function
    /// from C standart Library.
    /// For example, value "en_US.UTF-8" allows to use non-ascii characters in Linux or Mac.
    /// It also accepts the special value "UTF-8" for using UTF-8 character encoding.
    public func use(locale: String) throws
    {
        let result = locale.utf8CString.withUnsafeBufferPointer { xlBookSetLocaleA(handle, $0.baseAddress) }
        if result != 1 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
    }


    // MARK: - Loading

    /// Loads the content of an xls(x) file into current workbook
    public func load(fromFile file: String) throws
    {
        let result = file.data(using: .utf8)?.withUnsafeBytes { xlBookLoadA(handle, $0) }
        if result == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
    }

    /// Loads the content of an xls(x) file from data into current workbook
    public func load(fromData data: Data) throws
    {
        let status = data.withUnsafeBytes { xlBookLoadRawA(handle, $0, UInt32(data.count)) }
        if status == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
    }

    // MARK: - Saving

    /// Saves the current workbook into an xls(x) file
    public func save(toFile destination: String) throws
    {
        let result = destination.utf8CString.withUnsafeBufferPointer { xlBookSaveA(handle, $0.baseAddress) }
        if result == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
    }

    /// Saves the current workbook into a data buffer
    public func save() throws -> Data?
    {
        // First ask for a size of the storage needed
        var data = Data()
        var size: UInt32 = 0
        let sizeFetchStatus = data.withUnsafeMutableBytes {
            xlBookSaveRawA(handle, $0, &size)
        }
        if sizeFetchStatus == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        // Then actually fill the data
        data = Data(count: Int(size))
        size = 0
        let status = data.withUnsafeMutableBytes {
            xlBookSaveRawA(handle, $0, &size)
        }
        if status == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return data
    }

    // MARK: - Sheets

    /// Adds a new sheet to this book, returns the sheet handle.
    /// Set byCopying to nil if you wish to add a new empty sheet or provide an existing
    /// sheet for copying (this sheet must be from the same workbook).
    public func add(sheetWithName name: String, byCopying original: Sheet? = nil) throws -> Sheet
    {
        let result: SheetHandle? = name.utf8CString.withUnsafeBufferPointer {
            xlBookAddSheetA(handle, $0.baseAddress, original?.handle)
        }
        guard let newSheetHandle = result else {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return Sheet(withHandle: newSheetHandle, parentBook: self)
    }

    /// Inserts a new sheet to this workbook at the given position.
    /// Set byCopying to nil if you wish to add a new empty sheet or provide an existing
    /// sheet for copying (this sheet must be from the same workbook).
    public func insert(sheetWithName name: String, atIndex idx: Int, byCopying original: Sheet? = nil) throws -> Sheet
    {
        let result: SheetHandle? = name.utf8CString.withUnsafeBufferPointer {
            xlBookInsertSheetA(handle, Int32(idx), $0.baseAddress, original?.handle)
        }
        guard let newSheetHandle = result else {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return Sheet(withHandle: newSheetHandle, parentBook: self)
    }

    /// Returns a sheet with the specified index
    public func sheet(atIndex idx: Int) -> Sheet?
    {
        let handleMaybe: SheetHandle? =  xlBookGetSheetA(handle, Int32(idx))
        return handleMaybe.map { Sheet(withHandle: $0, parentBook: self) }
    }

    /// Returns a type of sheet with the specified index
    public func type(ofSheetAtIndex idx: Int) -> SheetType
    {
        return SheetType.from(rawValue: xlBookSheetTypeA(handle, Int32(idx)))
    }

    /// Deletes a sheet with the specified index
    public func delete(sheetAtIndex idx: Int) throws
    {
        if xlBookDelSheetA(handle, Int32(idx)) == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
    }

    /// Returns a total number of sheets in this workbook
    public var numberOfSheets: Int {
        return Int(xlBookSheetCountA(handle))
    }

    /// An index of the active sheet in this workbook
    public var activeSheet: Int {
        get {
            return Int(xlBookActiveSheetA(handle))
        }
        set {
            xlBookSetActiveSheetA(handle, Int32(newValue))
        }
    }

    // MARK: - Formats

    /// Adds a new format to the workbook. Can be a copy of another existing format
    public func add(formatByCopying original: Format?) throws -> Format
    {
        let maybeFormatHandle = xlBookAddFormatA(handle, original?.handle)
        guard let formatHandle = maybeFormatHandle else {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return Format(withHandle: formatHandle)
    }

    /// Returns a format by index
    public func format(atIndex idx: Int) -> Format?
    {
        let handleMaybe: FormatHandle? =  xlBookFormatA(handle, Int32(idx))
        return handleMaybe.map { Format(withHandle: $0) }
    }

    /// Returns a total number of formats in this workbook
    public var numberOfFormats: Int {
        return Int(xlBookFormatSizeA(handle))
    }

    // MARK: Custom Formats

    public typealias CustomNumberFormatIdentifier = Int32

    /// Adds a custom number format with the given specification
    public func add(customNumberFormat format: String) throws -> CustomNumberFormatIdentifier
    {
        let status = format.utf8CString.withUnsafeBufferPointer {
            xlBookAddCustomNumFormatA(handle, $0.baseAddress)
        }
        if status == 0 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return CustomNumberFormatIdentifier(status)
    }

    /// Returns a custom number with the given identifier
    public func customNumberFormat(withIdentifier id: CustomNumberFormatIdentifier) throws -> String
    {
        let raw: UnsafePointer<Int8>? = xlBookCustomNumFormatA(handle, id)
        guard let format = raw else {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return String(cString: format)
    }

    // MARK: - Fonts

    /// Returns a total number of fonts used in this workbook
    public var numberOfFonts: Int {
        return Int(xlBookFontSizeA(handle))
    }

    /// Adds a new font. May be a copy of another font from the same workbook
    public func add(fontByCopying original: Font?) throws -> Font
    {
        let maybeFontHandle: FontHandle? = xlBookAddFontA(handle, original?.handle)
        guard let fontHandle = maybeFontHandle else {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return Font(withHandle: fontHandle)
    }

    /// Returns a font by index
    public func font(atIndex idx: Int) -> Font?
    {
        let handleMaybe: FontHandle? =  xlBookFontA(handle, Int32(idx))
        return handleMaybe.map { Font(withHandle: $0) }
    }

    /// The default font used in this workbook
    public var defaultFont: DefaultBookFont? {
        get {
            var size: Int32 = 0
            let rawName = xlBookDefaultFontA(handle, &size)
            guard let name = rawName else {
                return nil
            }
            return DefaultBookFont(name: String(cString: name), size: Int(size))
        }
        set {
            guard let font = newValue else {
                return
            }
            font.name.utf8CString.withUnsafeBufferPointer {
                xlBookSetDefaultFontA(handle, $0.baseAddress, Int32(font.size))
            }
        }
    }

    // MARK: - Value Packing & Unpacking

    /// Packs date and time information into a single Double value
    public func pack(dateWithYear year: Int, month: Int, day: Int, hour: Int = 0, minute: Int = 0, sec: Int = 0, msec: Int = 0) -> Double
    {
        return xlBookDatePackA(handle, Int32(year), Int32(month), Int32(day), Int32(hour),
                               Int32(minute), Int32(sec),
                               Int32(msec))
    }

    /// Packs date and time information into a single Double value
    public func pack(date: Date) -> Double
    {
        let units: Set<Calendar.Component> = [.year, .day, .month, .hour, .minute, .second, .nanosecond]
        let components = Calendar.current.dateComponents(units, from: date)
        // TODO: deside on all these bangs
        return pack(dateWithYear: components.year!, month: components.month!, day: components.day!,
                    hour: components.hour!, minute: components.minute!, sec: components.second!,
                    msec: components.nanosecond! / 1_000_000)
    }

    /// Packs red, green and blue components into a single integer value
    public func pack(colorWithRed red: Int, green: Int, blue: Int) -> Int32
    {
        return xlBookColorPackA(handle, Int32(red), Int32(green), Int32(blue))
    }

    /// Packs the given color into a single integer value
    public func pack(color: NSColor) -> Int32
    {
        return pack(colorWithRed:Int(color.redComponent * 255), green: Int(color.greenComponent * 255),
                    blue: Int(color.blueComponent * 255))
    }

    /// Unpacks date and time information from a packed value
    public func unpack(date rawValue: Double, calendar: Calendar = Calendar.current) -> Date?
    {
        return calendar.date(from: unpack(date: rawValue))
    }

    /// Unpacks date and time information from a packed value
    public func unpack(date: Double) -> DateComponents
    {
        var year: Int32 = 0, month: Int32 = 0, day: Int32 = 0, hour: Int32 = 0, min: Int32 = 0,
             sec: Int32 = 0, msec: Int32 = 0
        xlBookDateUnpackA(handle, date, &year, &month, &day, &hour, &min, &sec, &msec)

        return DateComponents(calendar: nil, timeZone: nil, era: nil, year: Int(year), month: Int(month),
                              day: Int(day), hour: Int(hour), minute: Int(min), second: Int(sec),
                              nanosecond: Int(msec) * 1_000_000)
    }

    /// Unpacks a packed color value to red, green and blue components
    public func unpack(color: Int32) -> (Int, Int, Int)
    {
        var red: Int32 = 0, green: Int32 = 0, blue: Int32 = 0
        xlBookColorUnpackA(handle, color, &red, &green, &blue)
        return (Int(red), Int(green), Int(blue))
    }

    /// Unpacks a packed color value to a proper Color
    public func unpack(color: Int32) -> NSColor
    {
        let (r, g, b) =  unpack(color: color)
        return NSColor(red: CGFloat(r) / 255, green: CGFloat(g) / 255, blue: CGFloat(b) / 255, alpha: 1.0)
    }


    // MARK: - Pictures

    /// Returns a total number of pictures in this workbook
    public var numberOfPictures: Int {
        get {
            return Int(xlBookPictureSizeA(handle))
        }
    }

    /// Returns a picture at the given index
    public func picture(atIndex idx: Int) throws -> Picture
    {
        // First ask for a size of the storage needed
        var data = Data()
        var size: UInt32 = 0
        let status = data.withUnsafeMutableBytes {
            xlBookGetPictureA(handle, Int32(idx), $0, &size)
        }

        typealias SizeFetchStatus = PictureType
        if SizeFetchStatus.from(rawValue: status) == .Error {
            throw BookError.InternalError(message: lastErrorMessage)
        }

        // Then actually ask for data
        data = Data(count: Int(size))
        size = 0
        let rawType = data.withUnsafeMutableBytes {
            xlBookGetPictureA(handle, Int32(idx), $0, &size)
        }
        let type = PictureType.from(rawValue: rawType)
        if type == .Error {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return Picture(type: type, data: data)
    }


    /// Inserts a new picture from a file into this workbook
    public func add(pictureFromFile file: String) throws -> PictureIdentiifer
    {
        let id = file.utf8CString.withUnsafeBufferPointer { xlBookAddPictureA(handle, $0.baseAddress) }
        if id == -1 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return PictureIdentiifer(id)
    }

    /// Inserts a new picture from a data buffer into this workbook
    public func add(pictureFromData data: Data) throws -> PictureIdentiifer
    {
        let id = data.withUnsafeBytes { xlBookAddPicture2A(handle, $0, UInt32(data.count)) }
        if id == -1 {
            throw BookError.InternalError(message: lastErrorMessage)
        }
        return PictureIdentiifer(id)
    }
}
