/* -*- mode: c++; c-basic-offset: 4; -*- */

#include <ldastoolsal_config.h>

#include <pthread.h>

#include <sstream>

#include "ldastoolsal/DeadLockDetector.hh"
#include "ldastoolsal/mutexlock.hh"

#include "MutexLockBaton.cc"

#if DEAD_LOCK_DETECTOR_ENABLED
using LDASTools::AL::DeadLockDetector;
using LDASTools::AL::MutexLock;

namespace
{
    
    /// \brief  Log the thread state
    inline void log_lock( DeadLockDetector::state_type State,
			  MutexLock::baton_type Lock,
			  const char* File,
			  unsigned int Line )
    {
	if ( Lock.pimpl_->Logging( ) )
	{
	    DeadLockDetector::SetState( DeadLockDetector::MUTEX,
					Lock,
					State,
					File, Line );
	}
    }
}

#else /* DEAD_LOCK_DETECTOR_ENABLED */
#define log_lock( State, Lock, File, Line )
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

namespace LDASTools
{
    namespace AL
    {
	//===============================================================
	//===============================================================

	MutexLock::BusyError::
	BusyError( )
	    : std::runtime_error( msg( ) )
	{
	}

	std::string MutexLock::BusyError::
	msg( )
	{
	    static std::string	m( strerror( EBUSY ) );

	    return m;
	}

	//=======================================================================
	//=======================================================================

	MutexLock::baton_type::
	baton_type( const char* File,
		    unsigned int Line,
		    bool Logging )
	    : pimpl_( new impl( Logging ) )
	{
	    log_lock( DeadLockDetector::INITIALIZED, *this,
		      File, Line );
	}

	void* MutexLock::baton_type::
	Handle( )
	{
	    return pimpl_->ref( );
	}

	void MutexLock::baton_type::
	Lock( const char* File, const size_t Line )
	{
	    log_lock( DeadLockDetector::PENDING, *this,
		      File, Line );
	    try
	    {
		pimpl_->lock( );
		log_lock( DeadLockDetector::ACQUIRED, *this,
			  File, Line );
	    }
	    catch( const std::exception& Exception )
	    {
		log_lock( DeadLockDetector::ACQUISITION_ERROR, *this,
			  File, Line );
		throw;
	    }
	}

	void MutexLock::baton_type::
	TryLock( const char* File, const size_t Line )
	{
	    pimpl_->trylock( );
	}

	void MutexLock::baton_type::
	Unlock( const char* File, const size_t Line )
	{
	    try
	    {
		pimpl_->unlock( );
		log_lock( DeadLockDetector::RELEASED, *this,
			  File, Line );
	    }
	    catch( const std::exception& Exception )
	    {
		throw;
	    }
	}

	///----------------------------------------------------------------------
	///
	/// This contructor creates an object to ensure the releasing
	/// of the exclusive lock once the object goes out of scope.
	/// The lock held by the object can be released early by calling
	/// the Release method.
	///
	/// \see Release
	///----------------------------------------------------------------------
	MutexLock::
	MutexLock( baton_type Baton,
		   const char* File, const unsigned int Line )
	    : baton( Baton )
	{
	    baton.Lock( File, Line );
	}

	MutexLock::
	~MutexLock()
	{
	    Release( __FILE__, __LINE__ );
	}

	MutexLock::baton_type MutexLock::
	Baton( )
	{
	    return baton_type( );
	}

	void MutexLock::
	Release( const char* File, const unsigned int Line )
	{
	    if ( baton )
	    {
		baton.Unlock( File, Line);
	    }
	}

	//-----------------------------------------------------------------------
	/// This routine is used when a thread is canceled and the lock needs to
	/// be release.
	/// It is currently only written to support pthread_cancel_push().
	//-----------------------------------------------------------------------
	void MutexLock::
	ThreadCancel( void* VLock, const char* File, const unsigned int Line )
	{
	    if ( VLock )
	    {
		MutexLock*	l ( reinterpret_cast< MutexLock*>( VLock ) );

		l->Release( File, Line );
	    }
	}
    } // namespace - AL
} // namespace - LDASTools
