 /*
 *  Copyright (C) 2019 Edward Jacob Fauchon-Jones, Jonathan E. Thompson, Sebastian Khan
 *  Test code for LALSimIMRPhenomNSBH
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with with program; see the file COPYING. If not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *  MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>

#include <lal/Units.h>
#include <lal/LALAdaptiveRungeKuttaIntegrator.h>
#include <lal/LALConstants.h>
#include <lal/FindRoot.h>
#include <lal/SeqFactories.h>
#include <lal/LALSimInspiral.h>
#include <lal/LALSimIMR.h>

#include <lal/LALSimNoise.h>
#include <lal/ComplexFFT.h>

#include <lal/ComplexFFT.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_math.h>

#include "../lib/LALSimIMRPhenomNSBH.c"

#define MYUNUSED(expr) do { (void)(expr); } while (0)

bool approximatelyEqual(REAL8 a, REAL8 b, REAL8 epsilon);
bool approximatelyEqual(REAL8 a, REAL8 b, REAL8 epsilon) {
  if (a == 0)
    return fabs(b) < epsilon;
  else if (b == 0)
    return fabs(a) < epsilon;
  else
    return !gsl_fcmp(a, b, epsilon);
}

void print_difference(const char *name, REAL8 u, REAL8 u_expected);
void print_difference(const char *name, REAL8 u, REAL8 u_expected) {
  printf("%s: %-20.17g\t%-20.17g\t%-20.17g\n", name, u, u_expected, u - u_expected);
}

#define COUNT_X_D 36
#define TOLERANCE_X_D 1e-8
double input_x_D[COUNT_X_D][4] = {
  {0.0234970529550629, 0.14, 2.00, -0.90}, {0.1299454661328111, 0.14, 2.00, 0.00},
  {0.2882657031643916, 0.14, 2.00, 0.90}, {0.0000000000000000, 0.14, 3.00, -0.90},
  {0.0643792049443779, 0.14, 3.00, 0.00}, {0.2941998033123807, 0.14, 3.00, 0.90},
  {0.0000000000000000, 0.14, 4.00, -0.90}, {0.0000000000000000, 0.14, 4.00, 0.00},
  {0.2887085776862365, 0.14, 4.00, 0.90}, {0.0000000000000000, 0.14, 5.00, -0.90},
  {0.0000000000000000, 0.14, 5.00, 0.00}, {0.2763968308942356, 0.14, 5.00, 0.90},
  {0.0000000000000000, 0.17, 2.00, -0.90}, {0.0430073416291199, 0.17, 2.00, 0.00},
  {0.2332155204085186, 0.17, 2.00, 0.90}, {0.0000000000000000, 0.17, 3.00, -0.90},
  {0.0000000000000000, 0.17, 3.00, 0.00}, {0.2231866136920418, 0.17, 3.00, 0.90},
  {0.0000000000000000, 0.17, 4.00, -0.90}, {0.0000000000000000, 0.17, 4.00, 0.00},
  {0.2028602793140871, 0.17, 4.00, 0.90}, {0.0000000000000000, 0.17, 5.00, -0.90},
  {0.0000000000000000, 0.17, 5.00, 0.00}, {0.1765433406768961, 0.17, 5.00, 0.90},
  {0.0000000000000000, 0.20, 2.00, -0.90}, {0.0000000000000000, 0.20, 2.00, 0.00},
  {0.1778008350569429, 0.20, 2.00, 0.90}, {0.0000000000000000, 0.20, 3.00, -0.90},
  {0.0000000000000000, 0.20, 3.00, 0.00}, {0.1517306063389851, 0.20, 3.00, 0.90},
  {0.0000000000000000, 0.20, 4.00, -0.90}, {0.0000000000000000, 0.20, 4.00, 0.00},
  {0.1165992019782404, 0.20, 4.00, 0.90}, {0.0000000000000000, 0.20, 5.00, -0.90},
  {0.0000000000000000, 0.20, 5.00, 0.00}, {0.0764832851666699, 0.20, 5.00, 0.90}};
double expected_x_D[COUNT_X_D] = {
  0.3088924923327651, 0.3608377155105134, 0.4646547625420939,
  0.2714360204588185, 0.2813120354031964, 0.4566294437711992,
  0.2594324700000000, 0.2049292800000000, 0.4391346676862365,
  0.2494980801484723, 0.1949948901484723, 0.4168885310427079,
  0.2981427993777023, 0.2866469510068221, 0.4223519397862209,
  0.2841833804588185, 0.2296801904588185, 0.3983636141508602,
  0.2721798300000000, 0.2176766400000000, 0.3660337293140871,
  0.2622454401484722, 0.2077422501484723, 0.3297824008253684,
  0.3108901593777023, 0.2563869693777023, 0.3796846144346452,
  0.2969307404588185, 0.2424275504588185, 0.3396549667978037,
  0.2849271900000000, 0.2304240000000000, 0.2925200119782405,
  0.2749928001484723, 0.2204896101484723, 0.2424697053151421};
static void Test_x_D(void);
static void Test_x_D(void) {
  printf("\n## Test_x_D\n\n");
  for (int i=0; i<COUNT_X_D; i++) {
    double output = XLALSimIMRPhenomNSBH_x_D(
      input_x_D[i][0],
      input_x_D[i][1],
      input_x_D[i][2],
      input_x_D[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_x_D", output, expected_x_D[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_x_D[i], TOLERANCE_X_D));
  }
}

#define COUNT_EPSILON_INS_WITH_TORUS_MASS 36
#define TOLERANCE_EPSILON_INS_WITH_TORUS_MASS 1e-8
double input_epsilon_ins_with_torus_mass[COUNT_EPSILON_INS_WITH_TORUS_MASS][4] = {
  {0.0234970529550629, 0.14, 2.00, -0.90}, {0.1299454661328111, 0.14, 2.00, 0.00},
  {0.2882657031643916, 0.14, 2.00, 0.90}, {0.0000000000000000, 0.14, 3.00, -0.90},
  {0.0643792049443779, 0.14, 3.00, 0.00}, {0.2941998033123807, 0.14, 3.00, 0.90},
  {0.0000000000000000, 0.14, 4.00, -0.90}, {0.0000000000000000, 0.14, 4.00, 0.00},
  {0.2887085776862365, 0.14, 4.00, 0.90}, {0.0000000000000000, 0.14, 5.00, -0.90},
  {0.0000000000000000, 0.14, 5.00, 0.00}, {0.2763968308942356, 0.14, 5.00, 0.90},
  {0.0000000000000000, 0.17, 2.00, -0.90}, {0.0430073416291199, 0.17, 2.00, 0.00},
  {0.2332155204085186, 0.17, 2.00, 0.90}, {0.0000000000000000, 0.17, 3.00, -0.90},
  {0.0000000000000000, 0.17, 3.00, 0.00}, {0.2231866136920418, 0.17, 3.00, 0.90},
  {0.0000000000000000, 0.17, 4.00, -0.90}, {0.0000000000000000, 0.17, 4.00, 0.00},
  {0.2028602793140871, 0.17, 4.00, 0.90}, {0.0000000000000000, 0.17, 5.00, -0.90},
  {0.0000000000000000, 0.17, 5.00, 0.00}, {0.1765433406768961, 0.17, 5.00, 0.90},
  {0.0000000000000000, 0.20, 2.00, -0.90}, {0.0000000000000000, 0.20, 2.00, 0.00},
  {0.1778008350569429, 0.20, 2.00, 0.90}, {0.0000000000000000, 0.20, 3.00, -0.90},
  {0.0000000000000000, 0.20, 3.00, 0.00}, {0.1517306063389851, 0.20, 3.00, 0.90},
  {0.0000000000000000, 0.20, 4.00, -0.90}, {0.0000000000000000, 0.20, 4.00, 0.00},
  {0.1165992019782404, 0.20, 4.00, 0.90}, {0.0000000000000000, 0.20, 5.00, -0.90},
  {0.0000000000000000, 0.20, 5.00, 0.00}, {0.0764832851666699, 0.20, 5.00, 0.90}};
double expected_epsilon_ins_with_torus_mass[COUNT_EPSILON_INS_WITH_TORUS_MASS] = {
  0.8001567056997589, 0.7161488129677773, 0.5482517318264241,
  0.8607328102731804, 0.8447609238645346, 0.5612305983554656,
  0.8801454322171999, 0.9682901712127998, 0.5895238500311107,
  0.8962117248606847, 0.9843564638562846, 0.6255011920564910,
  0.8175415391344047, 0.8361330849537270, 0.6166655489001320,
  0.8401172697867804, 0.9282620087823803, 0.6554604286506627,
  0.8595298917307999, 0.9476746307263999, 0.7077456116040858,
  0.8755961843742848, 0.9637409233698846, 0.7663727100891812,
  0.7969259986480047, 0.8850707376436047, 0.6856688541517143,
  0.8195017293003803, 0.9076464682959803, 0.7504064014959200,
  0.8389143512443999, 0.9270590902399999, 0.8266349358283103,
  0.8549806438878846, 0.9431253828834847, 0.9075782937761394};
static void Test_epsilon_ins_with_torus_mass(void);
static void Test_epsilon_ins_with_torus_mass(void) {
  printf("\n## Test_epsilon_ins_with_torus_mass\n\n");
  for (int i=0; i<COUNT_EPSILON_INS_WITH_TORUS_MASS; i++) {
    double output = XLALSimIMRPhenomNSBH_epsilon_ins_with_torus_mass(
      input_epsilon_ins_with_torus_mass[i][0],
      input_epsilon_ins_with_torus_mass[i][1],
      input_epsilon_ins_with_torus_mass[i][2],
      input_epsilon_ins_with_torus_mass[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_epsilon_ins_with_torus_mass", output, expected_epsilon_ins_with_torus_mass[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_epsilon_ins_with_torus_mass[i], TOLERANCE_EPSILON_INS_WITH_TORUS_MASS));
  }
}

#define COUNT_X_D_PRIME 36
#define TOLERANCE_X_D_PRIME 1e-8
double input_x_D_prime[COUNT_X_D_PRIME][4] = {
  {0.0234970529550629, 0.14, 2.00, -0.90}, {0.1299454661328111, 0.14, 2.00, 0.00},
  {0.2882657031643916, 0.14, 2.00, 0.90}, {0.0000000000000000, 0.14, 3.00, -0.90},
  {0.0643792049443779, 0.14, 3.00, 0.00}, {0.2941998033123807, 0.14, 3.00, 0.90},
  {0.0000000000000000, 0.14, 4.00, -0.90}, {0.0000000000000000, 0.14, 4.00, 0.00},
  {0.2887085776862365, 0.14, 4.00, 0.90}, {0.0000000000000000, 0.14, 5.00, -0.90},
  {0.0000000000000000, 0.14, 5.00, 0.00}, {0.2763968308942356, 0.14, 5.00, 0.90},
  {0.0000000000000000, 0.17, 2.00, -0.90}, {0.0430073416291199, 0.17, 2.00, 0.00},
  {0.2332155204085186, 0.17, 2.00, 0.90}, {0.0000000000000000, 0.17, 3.00, -0.90},
  {0.0000000000000000, 0.17, 3.00, 0.00}, {0.2231866136920418, 0.17, 3.00, 0.90},
  {0.0000000000000000, 0.17, 4.00, -0.90}, {0.0000000000000000, 0.17, 4.00, 0.00},
  {0.2028602793140871, 0.17, 4.00, 0.90}, {0.0000000000000000, 0.17, 5.00, -0.90},
  {0.0000000000000000, 0.17, 5.00, 0.00}, {0.1765433406768961, 0.17, 5.00, 0.90},
  {0.0000000000000000, 0.20, 2.00, -0.90}, {0.0000000000000000, 0.20, 2.00, 0.00},
  {0.1778008350569429, 0.20, 2.00, 0.90}, {0.0000000000000000, 0.20, 3.00, -0.90},
  {0.0000000000000000, 0.20, 3.00, 0.00}, {0.1517306063389851, 0.20, 3.00, 0.90},
  {0.0000000000000000, 0.20, 4.00, -0.90}, {0.0000000000000000, 0.20, 4.00, 0.00},
  {0.1165992019782404, 0.20, 4.00, 0.90}, {0.0000000000000000, 0.20, 5.00, -0.90},
  {0.0000000000000000, 0.20, 5.00, 0.00}, {0.0764832851666699, 0.20, 5.00, 0.90}};
double expected_x_D_prime[COUNT_X_D_PRIME] = {
  0.3355511601551063, 0.3832042797328545, 0.3853372531644350,
  0.2899147353874842, 0.2954986467318621, 0.3691319814998649,
  0.2708773336000000, 0.2120820400000000, 0.3446033540862365,
  0.2551215810194710, 0.1963262874194710, 0.3165358547137066,
  0.3080714872000434, 0.2922835352291633, 0.3263044504085620,
  0.2859321153874842, 0.2271368217874842, 0.2941361718795260,
  0.2668947136000000, 0.2080994200000000, 0.2547724357140870,
  0.2511389610194710, 0.1923436674194710, 0.2126997444963671,
  0.3040888672000434, 0.2452935736000434, 0.2669071450569863,
  0.2819494953874842, 0.2231542017874842, 0.2186975445264693,
  0.2629120936000000, 0.2041168000000000, 0.1645287383782404,
  0.2471563410194710, 0.1883610474194710, 0.1086570689861409};
static void Test_x_D_prime(void);
static void Test_x_D_prime(void) {
  printf("\n## Test_x_D_prime\n\n");
  for (int i=0; i<COUNT_X_D_PRIME; i++) {
    double output = XLALSimIMRPhenomNSBH_x_D_prime(
      input_x_D_prime[i][0],
      input_x_D_prime[i][1],
      input_x_D_prime[i][2],
      input_x_D_prime[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_x_D_prime", output, expected_x_D_prime[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_x_D_prime[i], TOLERANCE_X_D_PRIME));
  }
}

#define COUNT_SIGMA_TIDE_WITH_TORUS_MASS 36
#define TOLERANCE_SIGMA_TIDE_WITH_TORUS_MASS 1e-8
double input_sigma_tide_with_torus_mass[COUNT_SIGMA_TIDE_WITH_TORUS_MASS][4] = {
  {0.0234970529550629, 0.14, 2.00, -0.90}, {0.1299454661328111, 0.14, 2.00, 0.00},
  {0.2882657031643916, 0.14, 2.00, 0.90}, {0.0000000000000000, 0.14, 3.00, -0.90},
  {0.0643792049443779, 0.14, 3.00, 0.00}, {0.2941998033123807, 0.14, 3.00, 0.90},
  {0.0000000000000000, 0.14, 4.00, -0.90}, {0.0000000000000000, 0.14, 4.00, 0.00},
  {0.2887085776862365, 0.14, 4.00, 0.90}, {0.0000000000000000, 0.14, 5.00, -0.90},
  {0.0000000000000000, 0.14, 5.00, 0.00}, {0.2763968308942356, 0.14, 5.00, 0.90},
  {0.0000000000000000, 0.17, 2.00, -0.90}, {0.0430073416291199, 0.17, 2.00, 0.00},
  {0.2332155204085186, 0.17, 2.00, 0.90}, {0.0000000000000000, 0.17, 3.00, -0.90},
  {0.0000000000000000, 0.17, 3.00, 0.00}, {0.2231866136920418, 0.17, 3.00, 0.90},
  {0.0000000000000000, 0.17, 4.00, -0.90}, {0.0000000000000000, 0.17, 4.00, 0.00},
  {0.2028602793140871, 0.17, 4.00, 0.90}, {0.0000000000000000, 0.17, 5.00, -0.90},
  {0.0000000000000000, 0.17, 5.00, 0.00}, {0.1765433406768961, 0.17, 5.00, 0.90},
  {0.0000000000000000, 0.20, 2.00, -0.90}, {0.0000000000000000, 0.20, 2.00, 0.00},
  {0.1778008350569429, 0.20, 2.00, 0.90}, {0.0000000000000000, 0.20, 3.00, -0.90},
  {0.0000000000000000, 0.20, 3.00, 0.00}, {0.1517306063389851, 0.20, 3.00, 0.90},
  {0.0000000000000000, 0.20, 4.00, -0.90}, {0.0000000000000000, 0.20, 4.00, 0.00},
  {0.1165992019782404, 0.20, 4.00, 0.90}, {0.0000000000000000, 0.20, 5.00, -0.90},
  {0.0000000000000000, 0.20, 5.00, 0.00}, {0.0764832851666699, 0.20, 5.00, 0.90}};
double expected_sigma_tide_with_torus_mass[COUNT_SIGMA_TIDE_WITH_TORUS_MASS] = {
  0.0393259844495971, 0.0253523266239769, 0.0247268598938206,
  0.0527082727391803, 0.0510708633282890, 0.0294788451409241,
  0.0582907433271368, 0.0755316988365200, 0.0366715462578143,
  0.0629109129465934, 0.0801518684559766, 0.0449019755713168,
  0.0473840413079209, 0.0520136529800058, 0.0420374618755445,
  0.0538761242801203, 0.0711170797895035, 0.0514703913665634,
  0.0594585948680768, 0.0766995503774600, 0.0630132952685083,
  0.0640787644875334, 0.0813197199969166, 0.0753505650231188,
  0.0485518928488609, 0.0657928483582441, 0.0594549495049245,
  0.0550439758210603, 0.0722849313304435, 0.0735917881356917,
  0.0606264464090168, 0.0778674019184000, 0.0894760863441799,
  0.0652466160284734, 0.0824875715378566, 0.1058597270617110};
static void Test_sigma_tide_with_torus_mass(void);
static void Test_sigma_tide_with_torus_mass(void) {
  printf("\n## Test_sigma_tide_with_torus_mass\n\n");
  for (int i=0; i<COUNT_SIGMA_TIDE_WITH_TORUS_MASS; i++) {
    double output = XLALSimIMRPhenomNSBH_sigma_tide_with_torus_mass(
      input_sigma_tide_with_torus_mass[i][0],
      input_sigma_tide_with_torus_mass[i][1],
      input_sigma_tide_with_torus_mass[i][2],
      input_sigma_tide_with_torus_mass[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_sigma_tide_with_torus_mass", output, expected_sigma_tide_with_torus_mass[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_sigma_tide_with_torus_mass[i], TOLERANCE_SIGMA_TIDE_WITH_TORUS_MASS));
  }
}

#define COUNT_EPSILON_TIDE_ND 108
#define TOLERANCE_EPSILON_TIDE_ND 1e-8
double input_epsilon_tide_ND[COUNT_EPSILON_TIDE_ND][1] = {
  {0.0750891905667151}, {0.0205860905667151}, {-0.0339170094332849}, {0.0811293896551724},
  {0.0266262896551724}, {-0.0278768103448276}, {0.0864809986779085}, {0.0319778986779085},
  {-0.0225252013220915}, {0.0912559101484723}, {0.0367528101484723}, {-0.0177502898515277},
  {0.0899611105667151}, {0.0354580105667151}, {-0.0190450894332849}, {0.0960013096551724},
  {0.0414982096551724}, {-0.0130048903448276}, {0.1013529186779085}, {0.0468498186779085},
  {-0.0076532813220915}, {0.1061278301484723}, {0.0516247301484723}, {-0.0028783698515277},
  {0.1048330305667151}, {0.0503299305667151}, {-0.0041731694332849}, {0.1108732296551724},
  {0.0563701296551724}, {0.0018670296551724}, {0.1162248386779085}, {0.0617217386779085},
  {0.0072186386779085}, {0.1209997501484723}, {0.0664966501484723}, {0.0119935501484722},
  {0.2050891905667151}, {0.1505860905667151}, {0.0960829905667151}, {0.2111293896551724},
  {0.1566262896551724}, {0.1021231896551724}, {0.2164809986779085}, {0.1619778986779085},
  {0.1074747986779085}, {0.2212559101484723}, {0.1667528101484723}, {0.1122497101484723},
  {0.2199611105667151}, {0.1654580105667151}, {0.1109549105667151}, {0.2260013096551724},
  {0.1714982096551724}, {0.1169951096551724}, {0.2313529186779085}, {0.1768498186779085},
  {0.1223467186779085}, {0.2361278301484722}, {0.1816247301484722}, {0.1271216301484722},
  {0.2348330305667151}, {0.1803299305667151}, {0.1258268305667151}, {0.2408732296551724},
  {0.1863701296551724}, {0.1318670296551724}, {0.2462248386779085}, {0.1917217386779085},
  {0.1372186386779085}, {0.2509997501484723}, {0.1964966501484723}, {0.1419935501484723},
  {0.3350891905667152}, {0.2805860905667151}, {0.2260829905667151}, {0.3411293896551725},
  {0.2866262896551725}, {0.2321231896551725}, {0.3464809986779085}, {0.2919778986779085},
  {0.2374747986779085}, {0.3512559101484723}, {0.2967528101484723}, {0.2422497101484723},
  {0.3499611105667151}, {0.2954580105667151}, {0.2409549105667151}, {0.3560013096551725},
  {0.3014982096551724}, {0.2469951096551724}, {0.3613529186779085}, {0.3068498186779085},
  {0.2523467186779085}, {0.3661278301484723}, {0.3116247301484723}, {0.2571216301484722},
  {0.3648330305667151}, {0.3103299305667151}, {0.2558268305667151}, {0.3708732296551724},
  {0.3163701296551724}, {0.2618670296551724}, {0.3762248386779086}, {0.3217217386779085},
  {0.2672186386779085}, {0.3809997501484723}, {0.3264966501484723}, {0.2719935501484723}};
double expected_epsilon_tide_ND[COUNT_EPSILON_TIDE_ND] = {
  0.9999998046379084, 0.9999548831569878, 0.9896872570770727,
  0.9999998931171316, 0.9999753160110837, 0.9943314088158876,
  0.9999999373622024, 0.9999855340106710, 0.9966701567238552,
  0.9999999611157933, 0.9999910197737862, 0.9979302909525667,
  0.9999999557491098, 0.9999897803621014, 0.9976453084404011,
  0.9999999757902789, 0.9999944087966979, 0.9987103698215647,
  0.9999999858120989, 0.9999967233153314, 0.9992438195259769,
  0.9999999911924543, 0.9999979659016727, 0.9995304446762743,
  0.9999999899768628, 0.9999976851620673, 0.9994656726625458,
  0.9999999945163283, 0.9999987335477748, 0.9997075980461685,
  0.9999999967863409, 0.9999992578060686, 0.9998286195805649,
  0.9999999980050291, 0.9999995392617429, 0.9998936037842391,
  0.9999999999995498, 0.9999999998960247, 0.9999999759869289,
  0.9999999999997538, 0.9999999999431151, 0.9999999868624169,
  0.9999999999998557, 0.9999999999666631, 0.9999999923008313,
  0.9999999999999104, 0.9999999999793051, 0.9999999952205205,
  0.9999999999998981, 0.9999999999764488, 0.9999999945608710,
  0.9999999999999443, 0.9999999999871152, 0.9999999970242452,
  0.9999999999999674, 0.9999999999924489, 0.9999999982560843,
  0.9999999999999797, 0.9999999999953124, 0.9999999989174144,
  0.9999999999999769, 0.9999999999946655, 0.9999999987679991,
  0.9999999999999873, 0.9999999999970814, 0.9999999993259707,
  0.9999999999999927, 0.9999999999982896, 0.9999999996049909,
  0.9999999999999953, 0.9999999999989382, 0.9999999997547868,
  1.0000000000000000, 0.9999999999999998, 0.9999999999999447,
  1.0000000000000000, 0.9999999999999999, 0.9999999999999698,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999822,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999890,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999875,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999931,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999960,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999976,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999971,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999984,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999991,
  1.0000000000000000, 1.0000000000000000, 0.9999999999999994};
static void Test_epsilon_tide_ND(void);
static void Test_epsilon_tide_ND(void) {
  printf("\n## Test_epsilon_tide_ND\n\n");
  for (int i=0; i<COUNT_EPSILON_TIDE_ND; i++) {
    double output = XLALSimIMRPhenomNSBH_epsilon_tide_ND(
      input_epsilon_tide_ND[i][0]);
    print_difference("XLALSimIMRPhenomNSBH_epsilon_tide_ND", output, expected_epsilon_tide_ND[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_epsilon_tide_ND[i], TOLERANCE_EPSILON_TIDE_ND));
  }
}

#define COUNT_SIGMA_TIDE_ND 108
#define TOLERANCE_SIGMA_TIDE_ND 1e-8
double input_sigma_tide_ND[COUNT_SIGMA_TIDE_ND][1] = {
  {11.4913585699999974}, {11.4679606399999976}, {11.4445627099999978}, {11.4782100899999975},
  {11.4548121599999977}, {11.4314142299999979}, {11.4650616099999976}, {11.4416636799999978},
  {11.4182657499999980}, {11.4519131299999959}, {11.4285151999999961}, {11.4051172699999963},
  {5.7479675665397911}, {5.7245696365397913}, {5.7011717065397916}, {5.7348190865397912},
  {5.7114211565397914}, {5.6880232265397916}, {5.7216706065397913}, {5.6982726765397915},
  {5.6748747465397917}, {5.7085221265397914}, {5.6851241965397916}, {5.6617262665397918},
  {3.9313585699999996}, {3.9079606399999998}, {3.8845627100000000}, {3.9182100899999996},
  {3.8948121599999999}, {3.8714142300000001}, {3.9050616099999997}, {3.8816636799999999},
  {3.8582657500000002}, {3.8919131299999998}, {3.8685152000000000}, {3.8451172700000003},
  {0.5713585699999997}, {0.5479606399999997}, {0.5245627099999997}, {0.5582100899999997},
  {0.5348121599999996}, {0.5114142299999996}, {0.5450616099999996}, {0.5216636799999996},
  {0.4982657499999996}, {0.5319131299999997}, {0.5085151999999997}, {0.4851172699999997},
  {0.7099052828027679}, {0.6865073528027679}, {0.6631094228027679}, {0.6967568028027679},
  {0.6733588728027678}, {0.6499609428027678}, {0.6836083228027678}, {0.6602103928027678},
  {0.6368124628027678}, {0.6704598428027679}, {0.6470619128027679}, {0.6236639828027679},
  {0.7716363477777777}, {0.7482384177777777}, {0.7248404877777777}, {0.7584878677777777},
  {0.7350899377777776}, {0.7116920077777776}, {0.7453393877777776}, {0.7219414577777776},
  {0.6985435277777776}, {0.7321909077777777}, {0.7087929777777777}, {0.6853950477777777},
  {3.1713585699999998}, {3.1479606400000000}, {3.1245627100000002}, {3.1582100899999999},
  {3.1348121600000001}, {3.1114142300000003}, {3.1450616100000000}, {3.1216636800000002},
  {3.0982657500000004}, {3.1319131300000000}, {3.1085152000000003}, {3.0851172700000005},
  {0.3500436911072667}, {0.3266457611072667}, {0.3032478311072667}, {0.3368952111072667},
  {0.3134972811072667}, {0.2900993511072667}, {0.3237467311072668}, {0.3003488011072667},
  {0.2769508711072667}, {0.3105982511072667}, {0.2872003211072667}, {0.2638023911072667},
  {-0.0408636522222222}, {-0.0642615822222222}, {-0.0876595122222222}, {-0.0540121322222222},
  {-0.0774100622222222}, {-0.1008079922222222}, {-0.0671606122222222}, {-0.0905585422222222},
  {-0.1139564722222222}, {-0.0803090922222222}, {-0.1037070222222222}, {-0.1271049522222222}};
double expected_sigma_tide_ND[COUNT_SIGMA_TIDE_ND] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.221245327087672e-12, 2.787214903321455e-12,
  6.361189353043528e-12, 1.941724558918168e-12, 4.431566225093775e-12,
  1.011390970973025e-11, 3.087252675726404e-12, 7.045919403481093e-12,
  1.608063682212446e-11, 4.908573547623973e-12, 1.120264991882891e-11,
  2.556738154524396e-11, 9.214851104388799e-15, 2.103872631664672e-14,
  4.801714581503802e-14, 1.465494392505207e-14, 3.347322419244847e-14,
  7.638334409421077e-14, 2.331468351712829e-14, 5.3179682879545e-14,
  1.214028877427609e-13, 3.708144902248023e-14, 8.459899447643693e-14,
  1.930677839823147e-13, 1.054711873393899e-15, 2.386979502944087e-15,
  5.440092820663267e-15, 1.665334536937735e-15, 3.774758283725532e-15,
  8.659739592076221e-15, 2.664535259100376e-15, 6.050715484207103e-15,
  1.376676550535194e-14, 4.218847493575595e-15, 9.603429163007604e-15,
  2.187139358511558e-14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  2.995577952358275e-09, 6.836670896070984e-09, 1.560302204106634e-08,
  4.762812699787844e-09, 1.086995016219561e-08, 2.48079912168464e-08,
  7.572623861307193e-09, 1.728265386358174e-08, 3.944341192552159e-08,
  1.20400769909601e-08, 2.747851823858838e-08, 6.271296681958916e-08,
  0.002899872666640546, 0.006593729188420216, 0.01492240094346903,
  0.004602772033608815, 0.01044306254129324, 0.02351879316771205,
  0.007298349629951184, 0.01650225678554373, 0.03688192323742007,
  0.01155424686739143, 0.0259847540081769, 0.05739158511822012};
static void Test_sigma_tide_ND(void);
static void Test_sigma_tide_ND(void) {
  printf("\n## Test_sigma_tide_ND\n\n");
  for (int i=0; i<COUNT_SIGMA_TIDE_ND; i++) {
    double output = XLALSimIMRPhenomNSBH_sigma_tide_ND(
      input_sigma_tide_ND[i][0]);
    print_difference("XLALSimIMRPhenomNSBH_sigma_tide_ND", output, expected_sigma_tide_ND[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_sigma_tide_ND[i], TOLERANCE_SIGMA_TIDE_ND));
  }
}

#define COUNT_X_ND 108
#define TOLERANCE_X_ND 1e-8
double input_x_ND[COUNT_X_ND][4] = {
  {-0.120, 0.050, 0.140, -0.900}, {-0.120, 0.050, 0.140, 0.000}, {-0.120, 0.050, 0.140, 0.900},
  {-0.120, 0.050, 0.160, -0.900}, {-0.120, 0.050, 0.160, 0.000}, {-0.120, 0.050, 0.160, 0.900},
  {-0.120, 0.050, 0.180, -0.900}, {-0.120, 0.050, 0.180, 0.000}, {-0.120, 0.050, 0.180, 0.900},
  {-0.120, 0.050, 0.200, -0.900}, {-0.120, 0.050, 0.200, 0.000}, {-0.120, 0.050, 0.200, 0.900},
  {-0.120, 0.085, 0.140, -0.900}, {-0.120, 0.085, 0.140, 0.000}, {-0.120, 0.085, 0.140, 0.900},
  {-0.120, 0.085, 0.160, -0.900}, {-0.120, 0.085, 0.160, 0.000}, {-0.120, 0.085, 0.160, 0.900},
  {-0.120, 0.085, 0.180, -0.900}, {-0.120, 0.085, 0.180, 0.000}, {-0.120, 0.085, 0.180, 0.900},
  {-0.120, 0.085, 0.200, -0.900}, {-0.120, 0.085, 0.200, 0.000}, {-0.120, 0.085, 0.200, 0.900},
  {-0.120, 0.120, 0.140, -0.900}, {-0.120, 0.120, 0.140, 0.000}, {-0.120, 0.120, 0.140, 0.900},
  {-0.120, 0.120, 0.160, -0.900}, {-0.120, 0.120, 0.160, 0.000}, {-0.120, 0.120, 0.160, 0.900},
  {-0.120, 0.120, 0.180, -0.900}, {-0.120, 0.120, 0.180, 0.000}, {-0.120, 0.120, 0.180, 0.900},
  {-0.120, 0.120, 0.200, -0.900}, {-0.120, 0.120, 0.200, 0.000}, {-0.120, 0.120, 0.200, 0.900},
  {0.010, 0.050, 0.140, -0.900}, {0.010, 0.050, 0.140, 0.000}, {0.010, 0.050, 0.140, 0.900},
  {0.010, 0.050, 0.160, -0.900}, {0.010, 0.050, 0.160, 0.000}, {0.010, 0.050, 0.160, 0.900},
  {0.010, 0.050, 0.180, -0.900}, {0.010, 0.050, 0.180, 0.000}, {0.010, 0.050, 0.180, 0.900},
  {0.010, 0.050, 0.200, -0.900}, {0.010, 0.050, 0.200, 0.000}, {0.010, 0.050, 0.200, 0.900},
  {0.010, 0.085, 0.140, -0.900}, {0.010, 0.085, 0.140, 0.000}, {0.010, 0.085, 0.140, 0.900},
  {0.010, 0.085, 0.160, -0.900}, {0.010, 0.085, 0.160, 0.000}, {0.010, 0.085, 0.160, 0.900},
  {0.010, 0.085, 0.180, -0.900}, {0.010, 0.085, 0.180, 0.000}, {0.010, 0.085, 0.180, 0.900},
  {0.010, 0.085, 0.200, -0.900}, {0.010, 0.085, 0.200, 0.000}, {0.010, 0.085, 0.200, 0.900},
  {0.010, 0.120, 0.140, -0.900}, {0.010, 0.120, 0.140, 0.000}, {0.010, 0.120, 0.140, 0.900},
  {0.010, 0.120, 0.160, -0.900}, {0.010, 0.120, 0.160, 0.000}, {0.010, 0.120, 0.160, 0.900},
  {0.010, 0.120, 0.180, -0.900}, {0.010, 0.120, 0.180, 0.000}, {0.010, 0.120, 0.180, 0.900},
  {0.010, 0.120, 0.200, -0.900}, {0.010, 0.120, 0.200, 0.000}, {0.010, 0.120, 0.200, 0.900},
  {0.140, 0.050, 0.140, -0.900}, {0.140, 0.050, 0.140, 0.000}, {0.140, 0.050, 0.140, 0.900},
  {0.140, 0.050, 0.160, -0.900}, {0.140, 0.050, 0.160, 0.000}, {0.140, 0.050, 0.160, 0.900},
  {0.140, 0.050, 0.180, -0.900}, {0.140, 0.050, 0.180, 0.000}, {0.140, 0.050, 0.180, 0.900},
  {0.140, 0.050, 0.200, -0.900}, {0.140, 0.050, 0.200, 0.000}, {0.140, 0.050, 0.200, 0.900},
  {0.140, 0.085, 0.140, -0.900}, {0.140, 0.085, 0.140, 0.000}, {0.140, 0.085, 0.140, 0.900},
  {0.140, 0.085, 0.160, -0.900}, {0.140, 0.085, 0.160, 0.000}, {0.140, 0.085, 0.160, 0.900},
  {0.140, 0.085, 0.180, -0.900}, {0.140, 0.085, 0.180, 0.000}, {0.140, 0.085, 0.180, 0.900},
  {0.140, 0.085, 0.200, -0.900}, {0.140, 0.085, 0.200, 0.000}, {0.140, 0.085, 0.200, 0.900},
  {0.140, 0.120, 0.140, -0.900}, {0.140, 0.120, 0.140, 0.000}, {0.140, 0.120, 0.140, 0.900},
  {0.140, 0.120, 0.160, -0.900}, {0.140, 0.120, 0.160, 0.000}, {0.140, 0.120, 0.160, 0.900},
  {0.140, 0.120, 0.180, -0.900}, {0.140, 0.120, 0.180, 0.000}, {0.140, 0.120, 0.180, 0.900},
  {0.140, 0.120, 0.200, -0.900}, {0.140, 0.120, 0.200, 0.000}, {0.140, 0.120, 0.200, 0.900}};
double expected_x_ND[COUNT_X_ND] = {
  11.4845653589999959, 11.4799892999999962, 11.4754132409999965,
  11.4731352589999958, 11.4685591999999961, 11.4639831409999964,
  11.4617051589999974, 11.4571290999999977, 11.4525530409999980,
  11.4502750589999973, 11.4456989999999976, 11.4411229409999979,
  5.7411743555397914, 5.7365982965397917, 5.7320222375397920,
  5.7297442555397913, 5.7251681965397916, 5.7205921375397919,
  5.7183141555397912, 5.7137380965397915, 5.7091620375397918,
  5.7068840555397911, 5.7023079965397914, 5.6977319375397917,
  3.9245653590000003, 3.9199893000000001, 3.9154132410000000,
  3.9131352590000001, 3.9085592000000000, 3.9039831409999999,
  3.9017051590000000, 3.8971290999999999, 3.8925530409999998,
  3.8902750589999999, 3.8856989999999998, 3.8811229409999997,
  0.5645653589999997, 0.5599892999999997, 0.5554132409999997,
  0.5531352589999997, 0.5485591999999997, 0.5439831409999997,
  0.5417051589999997, 0.5371290999999997, 0.5325530409999997,
  0.5302750589999997, 0.5256989999999997, 0.5211229409999997,
  0.7031120718027679, 0.6985360128027679, 0.6939599538027679,
  0.6916819718027679, 0.6871059128027679, 0.6825298538027679,
  0.6802518718027679, 0.6756758128027679, 0.6710997538027679,
  0.6688217718027679, 0.6642457128027679, 0.6596696538027679,
  0.7648431367777777, 0.7602670777777777, 0.7556910187777777,
  0.7534130367777777, 0.7488369777777777, 0.7442609187777777,
  0.7419829367777777, 0.7374068777777777, 0.7328308187777777,
  0.7305528367777777, 0.7259767777777777, 0.7214007187777777,
  3.1645653590000005, 3.1599893000000003, 3.1554132410000002,
  3.1531352590000004, 3.1485592000000002, 3.1439831410000001,
  3.1417051590000002, 3.1371291000000001, 3.1325530410000000,
  3.1302750590000001, 3.1256990000000000, 3.1211229409999999,
  0.3432504801072668, 0.3386744211072668, 0.3340983621072667,
  0.3318203801072668, 0.3272443211072668, 0.3226682621072667,
  0.3203902801072668, 0.3158142211072668, 0.3112381621072667,
  0.3089601801072668, 0.3043841211072668, 0.2998080621072667,
  -0.0476568632222222, -0.0522329222222222, -0.0568089812222222,
  -0.0590869632222222, -0.0636630222222222, -0.0682390812222222,
  -0.0705170632222222, -0.0750931222222222, -0.0796691812222222,
  -0.0819471632222222, -0.0865232222222222, -0.0910992812222222};
static void Test_x_ND(void);
static void Test_x_ND(void) {
  printf("\n## Test_x_ND\n\n");
  for (int i=0; i<COUNT_X_ND; i++) {
    double output = XLALSimIMRPhenomNSBH_x_ND(
      input_x_ND[i][0],
      input_x_ND[i][1],
      input_x_ND[i][2],
      input_x_ND[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_x_ND", output, expected_x_ND[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_x_ND[i], TOLERANCE_X_ND));
  }
}

#define COUNT_X_ND_PRIME 108
#define TOLERANCE_X_ND_PRIME 1e-8
double input_x_ND_prime[COUNT_X_ND_PRIME][4] = {
  {-0.120, 0.050, 0.140, -0.900}, {-0.120, 0.050, 0.140, 0.000}, {-0.120, 0.050, 0.140, 0.900},
  {-0.120, 0.050, 0.160, -0.900}, {-0.120, 0.050, 0.160, 0.000}, {-0.120, 0.050, 0.160, 0.900},
  {-0.120, 0.050, 0.180, -0.900}, {-0.120, 0.050, 0.180, 0.000}, {-0.120, 0.050, 0.180, 0.900},
  {-0.120, 0.050, 0.200, -0.900}, {-0.120, 0.050, 0.200, 0.000}, {-0.120, 0.050, 0.200, 0.900},
  {-0.120, 0.085, 0.140, -0.900}, {-0.120, 0.085, 0.140, 0.000}, {-0.120, 0.085, 0.140, 0.900},
  {-0.120, 0.085, 0.160, -0.900}, {-0.120, 0.085, 0.160, 0.000}, {-0.120, 0.085, 0.160, 0.900},
  {-0.120, 0.085, 0.180, -0.900}, {-0.120, 0.085, 0.180, 0.000}, {-0.120, 0.085, 0.180, 0.900},
  {-0.120, 0.085, 0.200, -0.900}, {-0.120, 0.085, 0.200, 0.000}, {-0.120, 0.085, 0.200, 0.900},
  {-0.120, 0.120, 0.140, -0.900}, {-0.120, 0.120, 0.140, 0.000}, {-0.120, 0.120, 0.140, 0.900},
  {-0.120, 0.120, 0.160, -0.900}, {-0.120, 0.120, 0.160, 0.000}, {-0.120, 0.120, 0.160, 0.900},
  {-0.120, 0.120, 0.180, -0.900}, {-0.120, 0.120, 0.180, 0.000}, {-0.120, 0.120, 0.180, 0.900},
  {-0.120, 0.120, 0.200, -0.900}, {-0.120, 0.120, 0.200, 0.000}, {-0.120, 0.120, 0.200, 0.900},
  {0.010, 0.050, 0.140, -0.900}, {0.010, 0.050, 0.140, 0.000}, {0.010, 0.050, 0.140, 0.900},
  {0.010, 0.050, 0.160, -0.900}, {0.010, 0.050, 0.160, 0.000}, {0.010, 0.050, 0.160, 0.900},
  {0.010, 0.050, 0.180, -0.900}, {0.010, 0.050, 0.180, 0.000}, {0.010, 0.050, 0.180, 0.900},
  {0.010, 0.050, 0.200, -0.900}, {0.010, 0.050, 0.200, 0.000}, {0.010, 0.050, 0.200, 0.900},
  {0.010, 0.085, 0.140, -0.900}, {0.010, 0.085, 0.140, 0.000}, {0.010, 0.085, 0.140, 0.900},
  {0.010, 0.085, 0.160, -0.900}, {0.010, 0.085, 0.160, 0.000}, {0.010, 0.085, 0.160, 0.900},
  {0.010, 0.085, 0.180, -0.900}, {0.010, 0.085, 0.180, 0.000}, {0.010, 0.085, 0.180, 0.900},
  {0.010, 0.085, 0.200, -0.900}, {0.010, 0.085, 0.200, 0.000}, {0.010, 0.085, 0.200, 0.900},
  {0.010, 0.120, 0.140, -0.900}, {0.010, 0.120, 0.140, 0.000}, {0.010, 0.120, 0.140, 0.900},
  {0.010, 0.120, 0.160, -0.900}, {0.010, 0.120, 0.160, 0.000}, {0.010, 0.120, 0.160, 0.900},
  {0.010, 0.120, 0.180, -0.900}, {0.010, 0.120, 0.180, 0.000}, {0.010, 0.120, 0.180, 0.900},
  {0.010, 0.120, 0.200, -0.900}, {0.010, 0.120, 0.200, 0.000}, {0.010, 0.120, 0.200, 0.900},
  {0.140, 0.050, 0.140, -0.900}, {0.140, 0.050, 0.140, 0.000}, {0.140, 0.050, 0.140, 0.900},
  {0.140, 0.050, 0.160, -0.900}, {0.140, 0.050, 0.160, 0.000}, {0.140, 0.050, 0.160, 0.900},
  {0.140, 0.050, 0.180, -0.900}, {0.140, 0.050, 0.180, 0.000}, {0.140, 0.050, 0.180, 0.900},
  {0.140, 0.050, 0.200, -0.900}, {0.140, 0.050, 0.200, 0.000}, {0.140, 0.050, 0.200, 0.900},
  {0.140, 0.085, 0.140, -0.900}, {0.140, 0.085, 0.140, 0.000}, {0.140, 0.085, 0.140, 0.900},
  {0.140, 0.085, 0.160, -0.900}, {0.140, 0.085, 0.160, 0.000}, {0.140, 0.085, 0.160, 0.900},
  {0.140, 0.085, 0.180, -0.900}, {0.140, 0.085, 0.180, 0.000}, {0.140, 0.085, 0.180, 0.900},
  {0.140, 0.085, 0.200, -0.900}, {0.140, 0.085, 0.200, 0.000}, {0.140, 0.085, 0.200, 0.900},
  {0.140, 0.120, 0.140, -0.900}, {0.140, 0.120, 0.140, 0.000}, {0.140, 0.120, 0.140, 0.900},
  {0.140, 0.120, 0.160, -0.900}, {0.140, 0.120, 0.160, 0.000}, {0.140, 0.120, 0.160, 0.900},
  {0.140, 0.120, 0.180, -0.900}, {0.140, 0.120, 0.180, 0.000}, {0.140, 0.120, 0.180, 0.900},
  {0.140, 0.120, 0.200, -0.900}, {0.140, 0.120, 0.200, 0.000}, {0.140, 0.120, 0.200, 0.900}};
double expected_x_ND_prime[COUNT_X_ND_PRIME] = {
  11.4913585699999974, 11.4679606399999976, 11.4445627099999978,
  11.4782100899999975, 11.4548121599999977, 11.4314142299999979,
  11.4650616099999976, 11.4416636799999978, 11.4182657499999980,
  11.4519131299999959, 11.4285151999999961, 11.4051172699999963,
  5.7479675665397911, 5.7245696365397913, 5.7011717065397916,
  5.7348190865397912, 5.7114211565397914, 5.6880232265397916,
  5.7216706065397913, 5.6982726765397915, 5.6748747465397917,
  5.7085221265397914, 5.6851241965397916, 5.6617262665397918,
  3.9313585699999996, 3.9079606399999998, 3.8845627100000000,
  3.9182100899999996, 3.8948121599999999, 3.8714142300000001,
  3.9050616099999997, 3.8816636799999999, 3.8582657500000002,
  3.8919131299999998, 3.8685152000000000, 3.8451172700000003,
  0.5713585699999997, 0.5479606399999997, 0.5245627099999997,
  0.5582100899999997, 0.5348121599999996, 0.5114142299999996,
  0.5450616099999996, 0.5216636799999996, 0.4982657499999996,
  0.5319131299999997, 0.5085151999999997, 0.4851172699999997,
  0.7099052828027679, 0.6865073528027679, 0.6631094228027679,
  0.6967568028027679, 0.6733588728027678, 0.6499609428027678,
  0.6836083228027678, 0.6602103928027678, 0.6368124628027678,
  0.6704598428027679, 0.6470619128027679, 0.6236639828027679,
  0.7716363477777777, 0.7482384177777777, 0.7248404877777777,
  0.7584878677777777, 0.7350899377777776, 0.7116920077777776,
  0.7453393877777776, 0.7219414577777776, 0.6985435277777776,
  0.7321909077777777, 0.7087929777777777, 0.6853950477777777,
  3.1713585699999998, 3.1479606400000000, 3.1245627100000002,
  3.1582100899999999, 3.1348121600000001, 3.1114142300000003,
  3.1450616100000000, 3.1216636800000002, 3.0982657500000004,
  3.1319131300000000, 3.1085152000000003, 3.0851172700000005,
  0.3500436911072667, 0.3266457611072667, 0.3032478311072667,
  0.3368952111072667, 0.3134972811072667, 0.2900993511072667,
  0.3237467311072668, 0.3003488011072667, 0.2769508711072667,
  0.3105982511072667, 0.2872003211072667, 0.2638023911072667,
  -0.0408636522222222, -0.0642615822222222, -0.0876595122222222,
  -0.0540121322222222, -0.0774100622222222, -0.1008079922222222,
  -0.0671606122222222, -0.0905585422222222, -0.1139564722222222,
  -0.0803090922222222, -0.1037070222222222, -0.1271049522222222};
static void Test_x_ND_prime(void);
static void Test_x_ND_prime(void) {
  printf("\n## Test_x_ND_prime\n\n");
  for (int i=0; i<COUNT_X_ND_PRIME; i++) {
    double output = XLALSimIMRPhenomNSBH_x_ND_prime(
      input_x_ND_prime[i][0],
      input_x_ND_prime[i][1],
      input_x_ND_prime[i][2],
      input_x_ND_prime[i][3]);
    print_difference("XLALSimIMRPhenomNSBH_x_ND_prime", output, expected_x_ND_prime[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_x_ND_prime[i], TOLERANCE_X_ND_PRIME));
  }
}

#define COUNT_DELTA2_PRIME 9
#define TOLERANCE_DELTA2_PRIME 1e-8
double input_delta2_prime[COUNT_DELTA2_PRIME][2] = {
  {-0.120, 0.050}, {-0.120, 0.085}, {-0.120, 0.120}, {0.010, 0.050}, {0.010, 0.085}, {0.010, 0.120},
  {0.140, 0.050}, {0.140, 0.085}, {0.140, 0.120}};
double expected_delta2_prime[COUNT_DELTA2_PRIME] = {
  0, 0, 0, 3.246794811282427e-09, 4.64277487690623e-10, 2.064565954817965e-10,
  0.81248, 0.8124797075947712, 0.7884780818898274};
static void Test_delta2_prime(void);
static void Test_delta2_prime(void) {
  printf("\n## Test_delta2_prime\n\n");
  for (int i=0; i<COUNT_DELTA2_PRIME; i++) {
    double output = XLALSimIMRPhenomNSBH_delta2_prime(
      input_delta2_prime[i][0],
      input_delta2_prime[i][1]);
    print_difference("XLALSimIMRPhenomNSBH_delta2_prime", output, expected_delta2_prime[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_delta2_prime[i], TOLERANCE_DELTA2_PRIME));
  }
}

#define COUNT_BARYONIC_MASS_FROM_C 9
#define TOLERANCE_BARYONIC_MASS_FROM_C 1e-8
double input_baryonic_mass_from_C[COUNT_BARYONIC_MASS_FROM_C][2] = {
  {0.1400, 1.3500}, {0.1400, 1.6750}, {0.1400, 2.0000}, {0.1700, 1.3500}, {0.1700, 1.6750},
  {0.1700, 2.0000}, {0.2000, 1.3500}, {0.2000, 1.6750}, {0.2000, 2.0000}};
double expected_baryonic_mass_from_C[COUNT_BARYONIC_MASS_FROM_C] = {
  1.4705869140000001, 1.8246170970000000, 2.1786472799999999,
  1.4973626385000001, 1.8578388292500001, 2.2183150199999999,
  1.5244686000000001, 1.8914703000000002, 2.2584720000000003};
static void Test_baryonic_mass_from_C(void);
static void Test_baryonic_mass_from_C(void) {
  printf("\n## Test_baryonic_mass_from_C\n\n");
  for (int i=0; i<COUNT_BARYONIC_MASS_FROM_C; i++) {
    double output = XLALSimIMRPhenomNSBH_baryonic_mass_from_C(
      input_baryonic_mass_from_C[i][0],
      input_baryonic_mass_from_C[i][1]);
    print_difference("XLALSimIMRPhenomNSBH_baryonic_mass_from_C", output, expected_baryonic_mass_from_C[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_baryonic_mass_from_C[i], TOLERANCE_BARYONIC_MASS_FROM_C));
  }
}

#define COUNT_OMEGA_TILDE 7
#define TOLERANCE_OMEGA_TILDE 1e-8
double input_omega_tilde[COUNT_OMEGA_TILDE][1] = {
  {-0.90}, {-0.60}, {-0.30}, {-0.00}, {0.30}, {0.60}, {0.90}};
COMPLEX16 expected_omega_tilde[COUNT_OMEGA_TILDE] = {
  0.2972594561409131 + I*0.08831455648737546,
  0.3171300978041187 + I*0.08877332447220707,
  0.34160908369694554 + I*0.08909934918507662,
  0.37365201926604874 + I*0.0890130676852164,
  0.4193077676007321 + I*0.0878758597418964,
  0.4940799241800059 + I*0.08379761075049783,
  0.6716700435394974 + I*0.06485409916444959};
static void Test_omega_tilde(void);
static void Test_omega_tilde(void) {
  printf("\n## Test_omega_tilde\n\n");
  for (int i=0; i<COUNT_OMEGA_TILDE; i++) {
    COMPLEX16 output = XLALSimIMRPhenomNSBH_omega_tilde(input_omega_tilde[i][0]);

    // Assert real part is approximately equal
    print_difference("XLALSimIMRPhenomNSBH_omega_tilde (real)", creal(output), creal(expected_omega_tilde[i]));
    XLAL_CHECK_EXIT(approximatelyEqual(creal(output), creal(expected_omega_tilde[i]), TOLERANCE_OMEGA_TILDE));

    // Assert imaginary part is approximately equal
    print_difference("XLALSimIMRPhenomNSBH_omega_tilde (imag)", cimag(output), cimag(expected_omega_tilde[i]));
    XLAL_CHECK_EXIT(approximatelyEqual(cimag(output), cimag(expected_omega_tilde[i]), TOLERANCE_OMEGA_TILDE));
  }
}

int main(int argc, char *argv[]) {
  MYUNUSED(argc);
  MYUNUSED(argv);

  Test_x_D();
  Test_epsilon_ins_with_torus_mass();
  Test_x_D_prime();
  Test_sigma_tide_with_torus_mass();
  Test_epsilon_tide_ND();
  Test_sigma_tide_ND();
  Test_x_ND();
  Test_x_ND_prime();
  Test_delta2_prime();

  Test_baryonic_mass_from_C();

  Test_omega_tilde();

  return 0;
}
