 /*
 *  Copyright (C) 2019 Edward Jacob Fauchon-Jones, Jonathan E. Thompson, Sebastian Khan
 *  Test code for LALSimIMRPhenomNSBH
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with with program; see the file COPYING. If not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *  MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include <complex.h>
#include <lal/LALStdlib.h>
#include <lal/LALConstants.h>
#include <lal/Date.h>
#include <lal/Units.h>
#include <lal/LALSimIMR.h>
#include <gsl/gsl_math.h>

#include "../lib/LALSimBHNSRemnantFits.c"

#define MYUNUSED(expr) do { (void)(expr); } while (0)

bool approximatelyEqual(REAL8 a, REAL8 b, REAL8 epsilon);
bool approximatelyEqual(REAL8 a, REAL8 b, REAL8 epsilon) {
  if (a == 0)
    return fabs(b) < epsilon;
  else if (b == 0)
    return fabs(a) < epsilon;
  else
    return !gsl_fcmp(a, b, epsilon);
}

void print_difference(const char *name, REAL8 u, REAL8 u_expected);
void print_difference(const char *name, REAL8 u, REAL8 u_expected) {
  printf("%s: %-20.17g\t%-20.17g\t%-20.17g\n", name, u, u_expected, u - u_expected);
}

#define COUNT_BHNS_SPIN_ALIGNED 48
#define TOLERANCE_BHNS_SPIN_ALIGNED 1e-7
double input_BHNS_spin_aligned[COUNT_BHNS_SPIN_ALIGNED][4] = {
  {0.66666667, 0.33333333, -0.90, 100.00}, {0.66666667, 0.33333333, -0.90, 800.00},
  {0.66666667, 0.33333333, -0.90, 1500.00}, {0.66666667, 0.33333333, -0.90, 2200.00},
  {0.66666667, 0.33333333, 0.00, 100.00}, {0.66666667, 0.33333333, 0.00, 800.00},
  {0.66666667, 0.33333333, 0.00, 1500.00}, {0.66666667, 0.33333333, 0.00, 2200.00},
  {0.66666667, 0.33333333, 0.90, 100.00}, {0.66666667, 0.33333333, 0.90, 800.00},
  {0.66666667, 0.33333333, 0.90, 1500.00}, {0.66666667, 0.33333333, 0.90, 2200.00},
  {0.75, 0.25, -0.90, 100.00}, {0.75, 0.25, -0.90, 800.00}, {0.75, 0.25, -0.90, 1500.00},
  {0.75, 0.25, -0.90, 2200.00}, {0.75, 0.25, 0.00, 100.00}, {0.75, 0.25, 0.00, 800.00},
  {0.75, 0.25, 0.00, 1500.00}, {0.75, 0.25, 0.00, 2200.00}, {0.75, 0.25, 0.90, 100.00},
  {0.75, 0.25, 0.90, 800.00}, {0.75, 0.25, 0.90, 1500.00}, {0.75, 0.25, 0.90, 2200.00},
  {0.8, 0.2, -0.90, 100.00}, {0.8, 0.2, -0.90, 800.00}, {0.8, 0.2, -0.90, 1500.00},
  {0.8, 0.2, -0.90, 2200.00}, {0.8, 0.2, 0.00, 100.00}, {0.8, 0.2, 0.00, 800.00},
  {0.8, 0.2, 0.00, 1500.00}, {0.8, 0.2, 0.00, 2200.00}, {0.8, 0.2, 0.90, 100.00},
  {0.8, 0.2, 0.90, 800.00}, {0.8, 0.2, 0.90, 1500.00}, {0.8, 0.2, 0.90, 2200.00},
  {0.83333333, 0.16666667, -0.90, 100.00}, {0.83333333, 0.16666667, -0.90, 800.00},
  {0.83333333, 0.16666667, -0.90, 1500.00}, {0.83333333, 0.16666667, -0.90, 2200.00},
  {0.83333333, 0.16666667, 0.00, 100.00}, {0.83333333, 0.16666667, 0.00, 800.00},
  {0.83333333, 0.16666667, 0.00, 1500.00}, {0.83333333, 0.16666667, 0.00, 2200.00},
  {0.83333333, 0.16666667, 0.90, 100.00}, {0.83333333, 0.16666667, 0.90, 800.00},
  {0.83333333, 0.16666667, 0.90, 1500.00}, {0.83333333, 0.16666667, 0.90, 2200.00}};
double expected_BHNS_spin_aligned[COUNT_BHNS_SPIN_ALIGNED] = {
  0.32531986806109, 0.32531986806109, 0.32531986806109, 0.32531986806109,
  0.6233940746778823, 0.6233940746778823, 0.6233940746778823,
  0.6233940746778823, 0.8933593070287664, 0.8933593070287664,
  0.8933593070287664, 0.8933593070287664, 0.1322915526236391,
  0.1322915526236391, 0.1322915526236391, 0.1322915526236391,
  0.5405636817226854, 0.5405636817226854, 0.5351118873433093,
  0.523184424888309, 0.9144046655280058, 0.9144046655280058,
  0.9144046655280058, 0.9140801081503465, -0.0152898311874099,
  -0.0152898311874099, -0.0152898311874099, -0.0152898311874099,
  0.4716225198815366, 0.4716225198815366, 0.4592291695111008,
  0.448324709134483, 0.9207205436502875, 0.9155233478254116,
  0.9122614149305155, 0.9101116289882841, -0.1282904163538133,
  -0.1282904163538133, -0.1282904163538133, -0.1282904163538133,
  0.416618630787216, 0.416618630787216, 0.40851912434322, 0.3989747180991385,
  0.9208154871629006, 0.9098290393520181, 0.9068297787096088, 0.90596110497524};
static void Test_BHNS_spin_aligned(void);
static void Test_BHNS_spin_aligned(void) {
  printf("\n## Test_BHNS_spin_aligned\n\n");
  for (int i=0; i<COUNT_BHNS_SPIN_ALIGNED; i++) {
    double output = XLALBHNS_spin_aligned(
      input_BHNS_spin_aligned[i][0],
      input_BHNS_spin_aligned[i][1],
      input_BHNS_spin_aligned[i][2],
      input_BHNS_spin_aligned[i][3]);
    print_difference("XLALBHNS_spin_aligned", output, expected_BHNS_spin_aligned[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_BHNS_spin_aligned[i], TOLERANCE_BHNS_SPIN_ALIGNED));
  }
}

#define COUNT_BHNS_MASS_ALIGNED 27
#define TOLERANCE_BHNS_MASS_ALIGNED 1e-7
double input_BHNS_mass_aligned[COUNT_BHNS_MASS_ALIGNED][5] = {
  {0.66666667, 0.33333333, -0.90, 100.00}, {0.66666667, 0.33333333, -0.90, 1150.00},
  {0.66666667, 0.33333333, -0.90, 2200.00}, {0.66666667, 0.33333333, 0.00, 100.00},
  {0.66666667, 0.33333333, 0.00, 1150.00}, {0.66666667, 0.33333333, 0.00, 2200.00},
  {0.66666667, 0.33333333, 0.90, 100.00}, {0.66666667, 0.33333333, 0.90, 1150.00},
  {0.66666667, 0.33333333, 0.90, 2200.00}, {0.77777778, 0.22222222, -0.90, 100.00},
  {0.77777778, 0.22222222, -0.90, 1150.00}, {0.77777778, 0.22222222, -0.90, 2200.00},
  {0.77777778, 0.22222222, 0.00, 100.00}, {0.77777778, 0.22222222, 0.00, 1150.00},
  {0.77777778, 0.22222222, 0.00, 2200.00}, {0.77777778, 0.22222222, 0.90, 100.00},
  {0.77777778, 0.22222222, 0.90, 1150.00}, {0.77777778, 0.22222222, 0.90, 2200.00},
  {0.83333333, 0.16666667, -0.90, 100.00}, {0.83333333, 0.16666667, -0.90, 1150.00},
  {0.83333333, 0.16666667, -0.90, 2200.00}, {0.83333333, 0.16666667, 0.00, 100.00},
  {0.83333333, 0.16666667, 0.00, 1150.00}, {0.83333333, 0.16666667, 0.00, 2200.00},
  {0.83333333, 0.16666667, 0.90, 100.00}, {0.83333333, 0.16666667, 0.90, 1150.00},
  {0.83333333, 0.16666667, 0.90, 2200.00}};
double expected_BHNS_mass_aligned[COUNT_BHNS_MASS_ALIGNED] = {
  0.9721892463752283, 0.9721892463752283, 0.9721892463752283,
  0.9612091161354085, 0.9552067472181308, 0.9388638553490046,
  0.9271901050008575, 0.8997642986449343, 0.8897816607983570,
  0.9829386159400940, 0.9829386159400940, 0.9829386159400940,
  0.9749212476580998, 0.9721096119194406, 0.9569331789707566,
  0.9402726792593378, 0.9149725534498141, 0.9099426996879995,
  0.9881860533109941, 0.9881860533109941, 0.9881860533109941,
  0.9823333002460519, 0.9823333002460519, 0.9790220473994081,
  0.9529747726329996, 0.9305878984670090, 0.9215091426291755};
static void Test_BHNS_mass_aligned(void);
static void Test_BHNS_mass_aligned(void) {
  printf("\n## Test_BHNS_mass_aligned\n\n");
  for (int i=0; i<COUNT_BHNS_MASS_ALIGNED; i++) {
    double output = XLALBHNS_mass_aligned(
      input_BHNS_mass_aligned[i][0],
      input_BHNS_mass_aligned[i][1],
      input_BHNS_mass_aligned[i][2],
      input_BHNS_mass_aligned[i][3]);
    print_difference("XLALBHNS_mass_aligned", output, expected_BHNS_mass_aligned[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_BHNS_mass_aligned[i], TOLERANCE_BHNS_MASS_ALIGNED));
  }
}

#define COUNT_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016 49
#define TOLERANCE_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016 1e-6
double input_bbh_final_spin_non_precessing_UIB2016[COUNT_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016][4] = {
  {0.66666667, 0.33333333, -0.90, 0.0}, {0.66666667, 0.33333333, -0.60, 0.0},
  {0.66666667, 0.33333333, -0.30, 0.0}, {0.66666667, 0.33333333, -0.00, 0.0},
  {0.66666667, 0.33333333, 0.30, 0.0}, {0.66666667, 0.33333333, 0.60, 0.0},
  {0.66666667, 0.33333333, 0.90, 0.0}, {0.71428571, 0.28571429, -0.90, 0.0},
  {0.71428571, 0.28571429, -0.60, 0.0}, {0.71428571, 0.28571429, -0.30, 0.0},
  {0.71428571, 0.28571429, -0.00, 0.0}, {0.71428571, 0.28571429, 0.30, 0.0},
  {0.71428571, 0.28571429, 0.60, 0.0}, {0.71428571, 0.28571429, 0.90, 0.0}, {0.75, 0.25, -0.90, 0.0},
  {0.75, 0.25, -0.60, 0.0}, {0.75, 0.25, -0.30, 0.0}, {0.75, 0.25, -0.00, 0.0},
  {0.75, 0.25, 0.30, 0.0}, {0.75, 0.25, 0.60, 0.0}, {0.75, 0.25, 0.90, 0.0},
  {0.77777778, 0.22222222, -0.90, 0.0}, {0.77777778, 0.22222222, -0.60, 0.0},
  {0.77777778, 0.22222222, -0.30, 0.0}, {0.77777778, 0.22222222, -0.00, 0.0},
  {0.77777778, 0.22222222, 0.30, 0.0}, {0.77777778, 0.22222222, 0.60, 0.0},
  {0.77777778, 0.22222222, 0.90, 0.0}, {0.8, 0.2, -0.90, 0.0}, {0.8, 0.2, -0.60, 0.0},
  {0.8, 0.2, -0.30, 0.0}, {0.8, 0.2, -0.00, 0.0}, {0.8, 0.2, 0.30, 0.0}, {0.8, 0.2, 0.60, 0.0},
  {0.8, 0.2, 0.90, 0.0}, {0.81818182, 0.18181818, -0.90, 0.0}, {0.81818182, 0.18181818, -0.60, 0.0},
  {0.81818182, 0.18181818, -0.30, 0.0}, {0.81818182, 0.18181818, -0.00, 0.0},
  {0.81818182, 0.18181818, 0.30, 0.0}, {0.81818182, 0.18181818, 0.60, 0.0},
  {0.81818182, 0.18181818, 0.90, 0.0}, {0.83333333, 0.16666667, -0.90, 0.0},
  {0.83333333, 0.16666667, -0.60, 0.0}, {0.83333333, 0.16666667, -0.30, 0.0},
  {0.83333333, 0.16666667, -0.00, 0.0}, {0.83333333, 0.16666667, 0.30, 0.0},
  {0.83333333, 0.16666667, 0.60, 0.0}, {0.83333333, 0.16666667, 0.90, 0.0}};
double expected_bbh_final_spin_non_precessing_UIB2016[COUNT_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016] = {
  0.32531986806109, 0.426811247693234, 0.5262816282723659, 0.6233940746778823,
  0.7176256005624781, 0.8081181824732344, 0.8933593070287664,
  0.2227051046418287, 0.3442933029242015, 0.4637591716373274,
  0.5806968749960987, 0.6944463986439515, 0.8038576147344049,
  0.9067316365893437, 0.1322915526236391, 0.2707009129437816,
  0.406938775219245, 0.5405636817226853, 0.6708305510903845,
  0.7963754449272744, 0.9144046655280058, 0.05344403493307415,
  0.206040738641565, 0.3564393270154197, 0.5041853113389944,
  0.6484874217929743, 0.7878347303236208, 0.918937716485219,
  -0.0152898311874099, 0.1493748390276979, 0.3118300811059859,
  0.4716225198815365, 0.6279414800852074, 0.7791806423169132,
  0.9216445513182814, -0.07541925103388192, 0.09959880790600101,
  0.2724090972820036, 0.4425677935576291, 0.6092626260836896,
  0.7708304036252198, 0.9232473877391176, -0.1282904163538133,
  0.05568406681932991, 0.237463578436505, 0.4166186307872159,
  0.5923469368641248, 0.7629568168378547, 0.9241622345560985};
static void Test_bbh_final_spin_non_precessing_UIB2016(void);
static void Test_bbh_final_spin_non_precessing_UIB2016(void) {
  printf("\n## Test_bbh_final_spin_non_precessing_UIB2016\n\n");
  for (int i=0; i<COUNT_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016; i++) {
    double output = XLALbbh_final_spin_non_precessing_UIB2016(
      input_bbh_final_spin_non_precessing_UIB2016[i][0],
      input_bbh_final_spin_non_precessing_UIB2016[i][1],
      input_bbh_final_spin_non_precessing_UIB2016[i][2],
      input_bbh_final_spin_non_precessing_UIB2016[i][3]);
    print_difference("XLALbbh_final_spin_non_precessing_UIB2016", output, expected_bbh_final_spin_non_precessing_UIB2016[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_bbh_final_spin_non_precessing_UIB2016[i], TOLERANCE_BBH_FINAL_SPIN_NON_PRECESSING_UIB2016));
  }
}

#define COUNT_BBH_FINAL_MASS_NON_PRECESSING_UIB2016 49
#define TOLERANCE_BBH_FINAL_MASS_NON_PRECESSING_UIB2016 1e-8
double input_bbh_final_mass_non_precessing_UIB2016[COUNT_BBH_FINAL_MASS_NON_PRECESSING_UIB2016][4] = {
  {0.66666667, 0.33333333, -0.90, 0.0}, {0.66666667, 0.33333333, -0.60, 0.0},
  {0.66666667, 0.33333333, -0.30, 0.0}, {0.66666667, 0.33333333, -0.00, 0.0},
  {0.66666667, 0.33333333, 0.30, 0.0}, {0.66666667, 0.33333333, 0.60, 0.0},
  {0.66666667, 0.33333333, 0.90, 0.0}, {0.71428571, 0.28571429, -0.90, 0.0},
  {0.71428571, 0.28571429, -0.60, 0.0}, {0.71428571, 0.28571429, -0.30, 0.0},
  {0.71428571, 0.28571429, -0.00, 0.0}, {0.71428571, 0.28571429, 0.30, 0.0},
  {0.71428571, 0.28571429, 0.60, 0.0}, {0.71428571, 0.28571429, 0.90, 0.0}, {0.75, 0.25, -0.90, 0.0},
  {0.75, 0.25, -0.60, 0.0}, {0.75, 0.25, -0.30, 0.0}, {0.75, 0.25, -0.00, 0.0},
  {0.75, 0.25, 0.30, 0.0}, {0.75, 0.25, 0.60, 0.0}, {0.75, 0.25, 0.90, 0.0},
  {0.77777778, 0.22222222, -0.90, 0.0}, {0.77777778, 0.22222222, -0.60, 0.0},
  {0.77777778, 0.22222222, -0.30, 0.0}, {0.77777778, 0.22222222, -0.00, 0.0},
  {0.77777778, 0.22222222, 0.30, 0.0}, {0.77777778, 0.22222222, 0.60, 0.0},
  {0.77777778, 0.22222222, 0.90, 0.0}, {0.8, 0.2, -0.90, 0.0}, {0.8, 0.2, -0.60, 0.0},
  {0.8, 0.2, -0.30, 0.0}, {0.8, 0.2, -0.00, 0.0}, {0.8, 0.2, 0.30, 0.0}, {0.8, 0.2, 0.60, 0.0},
  {0.8, 0.2, 0.90, 0.0}, {0.81818182, 0.18181818, -0.90, 0.0}, {0.81818182, 0.18181818, -0.60, 0.0},
  {0.81818182, 0.18181818, -0.30, 0.0}, {0.81818182, 0.18181818, -0.00, 0.0},
  {0.81818182, 0.18181818, 0.30, 0.0}, {0.81818182, 0.18181818, 0.60, 0.0},
  {0.81818182, 0.18181818, 0.90, 0.0}, {0.83333333, 0.16666667, -0.90, 0.0},
  {0.83333333, 0.16666667, -0.60, 0.0}, {0.83333333, 0.16666667, -0.30, 0.0},
  {0.83333333, 0.16666667, -0.00, 0.0}, {0.83333333, 0.16666667, 0.30, 0.0},
  {0.83333333, 0.16666667, 0.60, 0.0}, {0.83333333, 0.16666667, 0.90, 0.0}};
double expected_bbh_final_mass_non_precessing_UIB2016[COUNT_BBH_FINAL_MASS_NON_PRECESSING_UIB2016] = {
  0.9721892463752283, 0.9694119738670753, 0.9658485291704533,
  0.9612091161354085, 0.9549702082338648, 0.9460793967931898,
  0.9320618564969965, 0.9767340252207777, 0.9742714526109177,
  0.9710291544184678, 0.9667043606611544, 0.9607082019142325,
  0.9517232519172408, 0.9360722402912158, 0.9802316623246833,
  0.9780879021631663, 0.9751897931357891, 0.971248210181883,
  0.9656691198459941, 0.9570077253038091, 0.9405959334853373,
  0.982938615940094, 0.9810789379119643, 0.9784948592488221,
  0.9749212476580998, 0.969790204327316, 0.9616262072084677,
  0.9450263278811365, 0.985071427714766, 0.9834523313904443,
  0.9811397472589851, 0.9778948731043349, 0.9731894238502419,
  0.9655762410464163, 0.9491543700819247, 0.9867848155509094,
  0.9853653553170411, 0.983283197420709, 0.980324763824162, 0.9760053561768509,
  0.9689389496020029, 0.9529229375415428, 0.9881860533109941,
  0.9869312527994188, 0.9850439105054475, 0.9823333002460519,
  0.9783576056713429, 0.9718080109472624, 0.9563338892210131};
static void Test_bbh_final_mass_non_precessing_UIB2016(void);
static void Test_bbh_final_mass_non_precessing_UIB2016(void) {
  printf("\n## Test_bbh_final_mass_non_precessing_UIB2016\n\n");
  for (int i=0; i<COUNT_BBH_FINAL_MASS_NON_PRECESSING_UIB2016; i++) {
    double output = XLALbbh_final_mass_non_precessing_UIB2016(
      input_bbh_final_mass_non_precessing_UIB2016[i][0],
      input_bbh_final_mass_non_precessing_UIB2016[i][1],
      input_bbh_final_mass_non_precessing_UIB2016[i][2],
      input_bbh_final_mass_non_precessing_UIB2016[i][3]);
    print_difference("XLALbbh_final_mass_non_precessing_UIB2016", output, expected_bbh_final_mass_non_precessing_UIB2016[i]);
    XLAL_CHECK_EXIT(approximatelyEqual(output, expected_bbh_final_mass_non_precessing_UIB2016[i], TOLERANCE_BBH_FINAL_MASS_NON_PRECESSING_UIB2016));
  }
}

int main(int argc, char *argv[]) {
  MYUNUSED(argc);
  MYUNUSED(argv);

  Test_BHNS_spin_aligned();
  Test_BHNS_mass_aligned();
  Test_bbh_final_spin_non_precessing_UIB2016();
  Test_bbh_final_mass_non_precessing_UIB2016();

  return 0;
}
