/*
 * Copyright (C) 2022 Cardiff University
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with with program; see the file COPYING. If not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *  MA  02110-1301  USA
 */

#ifdef __cplusplus
extern "C"
{
#endif

  /**
   * \author Eleanor Hamilton, Sebastian Khan, Jonathan E. Thompson
   *
   */

#include "LALSimIMRPhenomX_PNR_coefficients.h"

#ifndef _OPENMP
#define omp ignore
#endif

#ifndef PHENOMXHMDEBUG
#define DEBUG 0
#else
#define DEBUG 1
#endif

  REAL8 IMRPhenomX_PNR_evaluate_coefficient_array(REAL8 coeff_array[4][4][5], REAL8 eta, REAL8 chi, REAL8 costheta)
  {
    REAL8 poly_sum = 0.0;

    for (int i = 0; i < 4; i++)
    {
      for (int j = 0; j < 4; j++)
      {
        for (int k = 0; k < 5; k++)
        {
          poly_sum += coeff_array[i][j][k] * pow(eta, i) * pow(chi, j) * pow(costheta, k);
        }
      }
    }

    return poly_sum;
  }

  /**
   * expressions for each of the coefficients that appear in the merger-ringdown expression for alpha
   */
  REAL8 IMRPhenomX_PNR_alpha_A1_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 A1 = -7.01729932E+00 * (x1) + -3.06075403E+00 * (x2) + -7.12384997E+00 * (x0) + 1.04459978E+00 + 3.04210473E+01 * (x0 * x2) + 2.54285862E+01 * (x0 * x0) + 6.38502724E+01 * (x0 * x1) + 2.95704570E+00 * (x2 * x2) + 1.62369754E+01 * (x1 * x1) + 2.36356947E+01 * (x1 * x2) + -1.60606542E+02 * (x0 * x1 * x1) + -2.19868327E+02 * (x0 * x1 * x2) + -1.11475857E+01 * (x1 * x1 * x1) + -5.16103690E+01 * (x1 * x1 * x2) + 7.48684859E+00 * (x2 * x2 * x2) + -1.73426975E+02 * (x0 * x0 * x1) + -6.89847827E+01 * (x0 * x0 * x2) + -8.83233850E+00 * (x1 * x2 * x2) + -4.56974441E+01 * (x0 * x2 * x2) + 4.77451870E+02 * (x0 * x1 * x1 * x2) + 1.15484744E+02 * (x0 * x1 * x1 * x1) + -5.53590259E+01 * (x1 * x2 * x2 * x2) + 4.15347868E+02 * (x0 * x0 * x1 * x1) + 7.51590132E+00 * (x1 * x1 * x2 * x2) + 4.86279790E+02 * (x0 * x0 * x1 * x2) + -4.31919865E+00 * (x2 * x2 * x2 * x2) + -8.29725789E+01 * (x0 * x2 * x2 * x2) + 1.40382881E+02 * (x0 * x0 * x2 * x2) + 1.84907979E+02 * (x0 * x1 * x2 * x2) + 3.35645858E+01 * (x1 * x1 * x1 * x2) + 1.18370165E+02 * (x1 * x1 * x2 * x2 * x2) + -1.07048007E+03 * (x0 * x0 * x1 * x1 * x2) + -2.95068093E+02 * (x0 * x0 * x1 * x1 * x1) + 6.00452453E+02 * (x0 * x1 * x2 * x2 * x2) + -6.61752262E+02 * (x0 * x0 * x1 * x2 * x2) + -2.63672148E+02 * (x0 * x1 * x1 * x2 * x2) + 6.53043804E+01 * (x0 * x2 * x2 * x2 * x2) + -3.09155358E+02 * (x0 * x1 * x1 * x1 * x2) + 1.49383286E+01 * (x1 * x2 * x2 * x2 * x2) + 2.09407834E+02 * (x0 * x0 * x2 * x2 * x2) + 1.21504291E+02 * (x0 * x1 * x1 * x1 * x2 * x2) + -1.27586539E+03 * (x0 * x1 * x1 * x2 * x2 * x2) + -1.50438712E+03 * (x0 * x0 * x1 * x2 * x2 * x2) + 1.12523251E+03 * (x0 * x0 * x1 * x1 * x2 * x2) + -1.87954067E+02 * (x0 * x0 * x2 * x2 * x2 * x2) + -2.80558563E+02 * (x0 * x1 * x2 * x2 * x2 * x2) + 7.03127824E+02 * (x0 * x0 * x1 * x1 * x1 * x2) + -7.70839731E+01 * (x1 * x1 * x1 * x2 * x2 * x2) + -1.37666457E+01 * (x1 * x1 * x2 * x2 * x2 * x2) + 9.25254367E+02 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + 3.98164795E+02 * (x0 * x1 * x1 * x2 * x2 * x2 * x2) + 3.22299321E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + 8.37135478E+02 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -6.36159668E+02 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + -1.62233651E+02 * (x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + -2.14803436E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + -1.52732767E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + 7.75401797E+02 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2);

    REAL8 coeff_array_A1[4][4][5] = {
        {{1.04459978e+00, -3.06075403e+00, 2.95704570e+00, 7.48684859e+00, -4.31919865e+00},
         {-7.01729932e+00, 2.36356947e+01, -8.83233850e+00, -5.53590259e+01, 1.49383286e+01},
         {1.62369754e+01, -5.16103690e+01, 7.51590132e+00, 1.18370165e+02, -1.37666457e+01},
         {-1.11475857e+01, 3.35645858e+01, 0.00000000e+00, -7.70839731e+01, 0.00000000e+00}},

        {{-7.12384997e+00, 3.04210473e+01, -4.56974441e+01, -8.29725789e+01, 6.53043804e+01},
         {6.38502724e+01, -2.19868327e+02, 1.84907979e+02, 6.00452453e+02, -2.80558563e+02},
         {-1.60606542e+02, 4.77451870e+02, -2.63672148e+02, -1.27586539e+03, 3.98164795e+02},
         {1.15484744e+02, -3.09155358e+02, 1.21504291e+02, 8.37135478e+02, -1.62233651e+02}},

        {{2.54285862e+01, -6.89847827e+01, 1.40382881e+02, 2.09407834e+02, -1.87954067e+02},
         {-1.73426975e+02, 4.86279790e+02, -6.61752262e+02, -1.50438712e+03, 9.25254367e+02},
         {4.15347868e+02, -1.07048007e+03, 1.12523251e+03, 3.22299321e+03, -1.52732767e+03},
         {-2.95068093e+02, 7.03127824e+02, -6.36159668e+02, -2.14803436e+03, 7.75401797e+02}},

        {{0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}}};

    REAL8 A1 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_A1, eta, chi, costheta);

    return chi * sqrt(1.0 - costheta * costheta) * A1 * A1;
  }

  REAL8 IMRPhenomX_PNR_alpha_A2_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 A2 = -4.78891038E+00 * (x1) + -5.51390083E+00 * (x0) + 7.08347660E-01 + -4.14264606E+00 * (x0 * x2) + 1.81136200E+01 * (x0 * x0) + 4.61263476E+01 * (x0 * x1) + 1.51680145E+00 * (x2 * x2) + 1.09348438E+01 * (x1 * x1) + -1.11931113E+02 * (x0 * x1 * x1) + 4.04546173E+01 * (x0 * x1 * x2) + -7.71924820E+00 * (x1 * x1 * x1) + 2.07574049E+01 * (x0 * x0 * x2) + 7.95447722E+00 * (x1 * x2 * x2) + -2.48290270E+01 * (x0 * x2 * x2) + -1.12976630E+00 * (x2 * x2 * x2) + -1.24497496E+02 * (x0 * x0 * x1) + -8.80207636E+01 * (x0 * x1 * x1 * x2) + 8.41956834E+01 * (x0 * x1 * x1 * x1) + 5.14517166E+00 * (x1 * x2 * x2 * x2) + 2.90326939E+02 * (x0 * x0 * x1 * x1) + -2.66171290E+01 * (x1 * x1 * x2 * x2) + -1.85643669E+02 * (x0 * x0 * x1 * x2) + -8.96225587E-01 * (x2 * x2 * x2 * x2) + 4.27226700E+01 * (x0 * x2 * x2 * x2) + 7.77696260E+01 * (x0 * x0 * x2 * x2) + -1.34011079E+02 * (x0 * x1 * x2 * x2) + -6.02467237E+00 * (x1 * x1 * x2 * x2 * x2) + 3.77825566E+02 * (x0 * x0 * x1 * x1 * x2) + 9.78335223E+00 * (x0 * x2 * x2 * x2 * x2) + -2.35039929E+02 * (x0 * x0 * x1 * x1 * x1) + -2.56156198E+02 * (x0 * x1 * x2 * x2 * x2) + -2.24073158E+01 * (x1 * x2 * x2 * x2 * x2) + 4.22254858E+02 * (x0 * x1 * x1 * x2 * x2) + 5.56575471E+01 * (x0 * x1 * x1 * x1 * x2) + 1.04563374E+03 * (x0 * x0 * x1 * x2 * x2) + 1.92361544E+01 * (x1 * x1 * x1 * x2 * x2) + -3.40634217E+02 * (x0 * x0 * x2 * x2 * x2) + 4.50060096E+02 * (x0 * x1 * x1 * x2 * x2 * x2) + 6.41488129E+01 * (x0 * x0 * x0 * x1 * x1 * x1) + -2.78531037E+03 * (x0 * x0 * x1 * x1 * x2 * x2) + -2.71250495E+03 * (x0 * x0 * x0 * x1 * x2 * x2) + -2.86460064E+02 * (x0 * x1 * x1 * x1 * x2 * x2) + 5.54407725E+01 * (x0 * x0 * x0 * x1 * x1 * x2) + 4.91231205E+01 * (x0 * x0 * x2 * x2 * x2 * x2) + 4.23612394E+02 * (x0 * x1 * x2 * x2 * x2 * x2) + 7.38662593E+02 * (x0 * x0 * x0 * x2 * x2 * x2) + 2.10619339E+03 * (x0 * x0 * x1 * x2 * x2 * x2) + -2.17428672E+02 * (x0 * x0 * x1 * x1 * x1 * x2) + 6.64680541E+01 * (x1 * x1 * x2 * x2 * x2 * x2) + -9.27544052E+01 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + -4.84350012E+01 * (x1 * x1 * x1 * x2 * x2 * x2 * x2) + -2.95316488E+03 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + -1.18188078E+03 * (x0 * x1 * x1 * x2 * x2 * x2 * x2) + -3.86114313E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -4.57681784E+03 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + 6.59102127E+03 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + -2.14207360E+02 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + 1.77987293E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.87089460E+02 * (x0 * x0 * x0 * x2 * x2 * x2 * x2) + 6.59296751E+03 * (x0 * x0 * x0 * x1 * x2 * x2 * x2 * x2) + 8.26652394E+02 * (x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + 8.55100850E+03 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + 2.04687886E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 7.56334211E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -4.16235846E+03 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2) + -5.06952341E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + -1.60298657E+04 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -4.76870795E+03 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 1.04968429E+04 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2);

    REAL8 coeff_array_A2[4][4][5] = {
        {{7.08347660e-01, 0.00000000e+00, 1.51680145e+00, -1.12976630e+00, -8.96225587e-01},
         {-4.78891038e+00, 0.00000000e+00, 7.95447722e+00, 5.14517166e+00, -2.24073158e+01},
         {1.09348438e+01, 0.00000000e+00, -2.66171290e+01, -6.02467237e+00, 6.64680541e+01},
         {-7.71924820e+00, 0.00000000e+00, 1.92361544e+01, 0.00000000e+00, -4.84350012e+01}},

        {{-5.51390083e+00, -4.14264606e+00, -2.48290270e+01, 4.27226700e+01, 9.78335223e+00},
         {4.61263476e+01, 4.04546173e+01, -1.34011079e+02, -2.56156198e+02, 4.23612394e+02},
         {-1.11931113e+02, -8.80207636e+01, 4.22254858e+02, 4.50060096e+02, -1.18188078e+03},
         {8.41956834e+01, 5.56575471e+01, -2.86460064e+02, -2.14207360e+02, 8.26652394e+02}},

        {{1.81136200e+01, 2.07574049e+01, 7.77696260e+01, -3.40634217e+02, 4.91231205e+01},
         {-1.24497496e+02, -1.85643669e+02, 1.04563374e+03, 2.10619339e+03, -2.95316488e+03},
         {2.90326939e+02, 3.77825566e+02, -2.78531037e+03, -3.86114313e+03, 7.56334211e+03},
         {-2.35039929e+02, -2.17428672e+02, 1.77987293e+03, 2.04687886e+03, -5.06952341e+03}},

        {{0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 7.38662593e+02, -2.87089460e+02},
         {0.00000000e+00, 0.00000000e+00, -2.71250495e+03, -4.57681784e+03, 6.59296751e+03},
         {0.00000000e+00, 5.54407725e+01, 6.59102127e+03, 8.55100850e+03, -1.60298657e+04},
         {6.41488129e+01, -9.27544052e+01, -4.16235846e+03, -4.76870795e+03, 1.04968429e+04}}};

    REAL8 A2 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_A2, eta, chi, costheta);

    return -chi * sqrt(1.0 - costheta * costheta) * A2 * A2;
  }

  REAL8 IMRPhenomX_PNR_alpha_A3_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 A3 = 2.29535857E-01 * (x1) + 8.53850677E-01 * (x2) + 2.37641854E-02 + -1.36518140E+01 * (x0 * x2) + -5.75632659E+00 * (x0 * x1) + 2.26634422E-01 * (x2 * x2) + -1.05830284E+00 * (x0 * x0) + -3.69774894E-01 * (x1 * x1) + -6.63002642E+00 * (x1 * x2) + 1.02674822E+01 * (x0 * x1 * x1) + 1.05782962E+02 * (x0 * x1 * x2) + 1.43739502E+01 * (x1 * x1 * x2) + 4.22174178E+01 * (x0 * x0 * x1) + 6.56968525E+01 * (x0 * x0 * x2) + 3.83732885E+00 * (x0 * x0 * x0) + -4.60399072E+00 * (x0 * x2 * x2) + -1.68372588E+00 * (x2 * x2 * x2) + -2.26132428E+00 * (x1 * x2 * x2) + -2.27904811E+02 * (x0 * x1 * x1 * x2) + -2.15345685E+00 * (x0 * x1 * x1 * x1) + 1.24225169E+01 * (x1 * x2 * x2 * x2) + -8.04678865E+01 * (x0 * x0 * x1 * x1) + 4.86077798E+00 * (x1 * x1 * x2 * x2) + -5.05493224E+02 * (x0 * x0 * x1 * x2) + 2.94176176E+01 * (x0 * x2 * x2 * x2) + 2.84160912E+01 * (x0 * x0 * x2 * x2) + 4.57920306E+01 * (x0 * x1 * x2 * x2) + -9.76185045E+01 * (x0 * x0 * x0 * x2) + -9.21827041E+01 * (x0 * x0 * x0 * x1) + -9.30697076E+00 * (x1 * x1 * x1 * x2) + -2.59086482E+01 * (x1 * x1 * x2 * x2 * x2) + 1.07807205E+03 * (x0 * x0 * x1 * x1 * x2) + 2.63933567E+01 * (x0 * x0 * x1 * x1 * x1) + -2.15952616E+02 * (x0 * x1 * x2 * x2 * x2) + 1.82667964E+02 * (x0 * x0 * x0 * x1 * x1) + 7.44071961E+02 * (x0 * x0 * x0 * x1 * x2) + -9.95024478E+01 * (x0 * x1 * x1 * x2 * x2) + -2.79807179E+00 * (x1 * x1 * x1 * x2 * x2) + -5.44153111E+01 * (x0 * x0 * x0 * x2 * x2) + 1.46405699E+02 * (x0 * x1 * x1 * x1 * x2) + -2.82597366E+02 * (x0 * x0 * x1 * x2 * x2) + -1.58222595E+02 * (x0 * x0 * x2 * x2 * x2) + 5.86426400E+01 * (x0 * x1 * x1 * x1 * x2 * x2) + 4.48148907E+02 * (x0 * x1 * x1 * x2 * x2 * x2) + -7.18225585E+01 * (x0 * x0 * x0 * x1 * x1 * x1) + -1.56625497E+03 * (x0 * x0 * x0 * x1 * x1 * x2) + 1.15097184E+03 * (x0 * x0 * x1 * x2 * x2 * x2) + 6.18415907E+02 * (x0 * x0 * x1 * x1 * x2 * x2) + -6.84842578E+02 * (x0 * x0 * x1 * x1 * x1 * x2) + 2.68829791E+02 * (x0 * x0 * x0 * x2 * x2 * x2) + 5.41525413E+02 * (x0 * x0 * x0 * x1 * x2 * x2) + 1.59684471E+01 * (x1 * x1 * x1 * x2 * x2 * x2) + 9.81227739E+02 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + -2.37206023E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -2.74864857E+02 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -1.93561667E+03 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + -1.18882155E+03 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + -3.68281254E+02 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + 3.96215587E+03 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + 1.44651605E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 7.10130827E+02 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.40403332E+03 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2);

    REAL8 coeff_array_A3[4][4][5] = {
        {{2.37641854e-02, 8.53850677e-01, 2.26634422e-01, -1.68372588e+00, 0.00000000e+00},
         {2.29535857e-01, -6.63002642e+00, -2.26132428e+00, 1.24225169e+01, 0.00000000e+00},
         {-3.69774894e-01, 1.43739502e+01, 4.86077798e+00, -2.59086482e+01, 0.00000000e+00},
         {0.00000000e+00, -9.30697076e+00, -2.79807179e+00, 1.59684471e+01, 0.00000000e+00}},

        {{0.00000000e+00, -1.36518140e+01, -4.60399072e+00, 2.94176176e+01, 0.00000000e+00},
         {-5.75632659e+00, 1.05782962e+02, 4.57920306e+01, -2.15952616e+02, 0.00000000e+00},
         {1.02674822e+01, -2.27904811e+02, -9.95024478e+01, 4.48148907e+02, 0.00000000e+00},
         {-2.15345685e+00, 1.46405699e+02, 5.86426400e+01, -2.74864857e+02, 0.00000000e+00}},

        {{-1.05830284e+00, 6.56968525e+01, 2.84160912e+01, -1.58222595e+02, 0.00000000e+00},
         {4.22174178e+01, -5.05493224e+02, -2.82597366e+02, 1.15097184e+03, 0.00000000e+00},
         {-8.04678865e+01, 1.07807205e+03, 6.18415907e+02, -2.37206023e+03, 0.00000000e+00},
         {2.63933567e+01, -6.84842578e+02, -3.68281254e+02, 1.44651605e+03, 0.00000000e+00}},

        {{3.83732885e+00, -9.76185045e+01, -5.44153111e+01, 2.68829791e+02, 0.00000000e+00},
         {-9.21827041e+01, 7.44071961e+02, 5.41525413e+02, -1.93561667e+03, 0.00000000e+00},
         {1.82667964e+02, -1.56625497e+03, -1.18882155e+03, 3.96215587e+03, 0.00000000e+00},
         {-7.18225585e+01, 9.81227739e+02, 7.10130827e+02, -2.40403332e+03, 0.00000000e+00}}};

    REAL8 A3 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_A3, eta, chi, costheta);

    return A3 * A3;
  }

  REAL8 IMRPhenomX_PNR_alpha_A4_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 A4 = 3.68521318E-01 * (x1) + 1.76157925E+00 * (x0) + -3.13330481E-01 * (x2) + 1.59708156E-01 + 4.37672477E+00 * (x0 * x2) + -1.01776826E+01 * (x0 * x0) + -8.10802054E+00 * (x0 * x1) + 3.82163433E-01 * (x2 * x2) + -8.32553332E-01 * (x1 * x1) + 2.65522323E+00 * (x1 * x2) + 1.87499491E+01 * (x0 * x1 * x1) + -3.83223991E+01 * (x0 * x1 * x2) + 8.05210158E-01 * (x1 * x1 * x1) + -6.02094408E+00 * (x1 * x1 * x2) + -2.06957701E+01 * (x0 * x0 * x2) + -2.23540184E+00 * (x1 * x2 * x2) + 2.05624058E+01 * (x0 * x0 * x0) + -3.65397391E+00 * (x0 * x2 * x2) + 6.14498512E-01 * (x2 * x2 * x2) + 5.43489123E+01 * (x0 * x0 * x1) + 9.08302756E+01 * (x0 * x1 * x1 * x2) + -1.63620472E+01 * (x0 * x1 * x1 * x1) + -4.37061988E+00 * (x1 * x2 * x2 * x2) + -1.27304367E+02 * (x0 * x0 * x1 * x1) + 4.86548073E+00 * (x1 * x1 * x2 * x2) + 1.91078478E+02 * (x0 * x0 * x1 * x2) + -3.91386611E-01 * (x2 * x2 * x2 * x2) + -9.43917739E+00 * (x0 * x2 * x2 * x2) + 8.82898695E+00 * (x0 * x0 * x2 * x2) + 2.30718712E+01 * (x0 * x1 * x2 * x2) + 3.29280099E+01 * (x0 * x0 * x0 * x2) + -1.10660415E+02 * (x0 * x0 * x0 * x1) + 4.06905683E+00 * (x1 * x1 * x1 * x2) + 8.36610850E+00 * (x1 * x1 * x2 * x2 * x2) + -4.65340116E+02 * (x0 * x0 * x1 * x1 * x2) + 2.79796061E+00 * (x0 * x2 * x2 * x2 * x2) + 1.07160127E+02 * (x0 * x0 * x1 * x1 * x1) + 6.83013158E+01 * (x0 * x1 * x2 * x2 * x2) + 2.60872767E+02 * (x0 * x0 * x0 * x1 * x1) + 2.01077124E+00 * (x1 * x2 * x2 * x2 * x2) + -5.62222338E+01 * (x0 * x1 * x1 * x2 * x2) + -6.32085162E+01 * (x0 * x1 * x1 * x1 * x2) + -3.19269499E+02 * (x0 * x0 * x0 * x1 * x2) + -8.10382470E+01 * (x0 * x0 * x1 * x2 * x2) + -3.59268661E+00 * (x1 * x1 * x1 * x2 * x2) + 4.68191983E+01 * (x0 * x0 * x2 * x2 * x2) + -1.34263772E+02 * (x0 * x1 * x1 * x2 * x2 * x2) + -2.17001178E+02 * (x0 * x0 * x0 * x1 * x1 * x1) + 2.43493364E+02 * (x0 * x0 * x1 * x1 * x2 * x2) + 8.40211777E+01 * (x0 * x0 * x0 * x1 * x2 * x2) + 4.63243515E+01 * (x0 * x1 * x1 * x1 * x2 * x2) + 7.88656773E+02 * (x0 * x0 * x0 * x1 * x1 * x2) + -1.66542213E+01 * (x0 * x1 * x2 * x2 * x2 * x2) + -7.52635673E+01 * (x0 * x0 * x0 * x2 * x2 * x2) + -3.45628809E+02 * (x0 * x0 * x1 * x2 * x2 * x2) + 3.33094494E+02 * (x0 * x0 * x1 * x1 * x1 * x2) + -4.39013590E+00 * (x1 * x1 * x1 * x2 * x2 * x2) + -2.56728295E+00 * (x1 * x1 * x2 * x2 * x2 * x2) + -5.76084011E+02 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + 5.98214021E-01 * (x1 * x1 * x1 * x2 * x2 * x2 * x2) + 6.96748172E+02 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + 3.77841307E+01 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + 1.75463323E+01 * (x0 * x1 * x1 * x2 * x2 * x2 * x2) + 5.64442664E+02 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + -3.46838253E+02 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + 7.33928698E+01 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -2.25400241E+02 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.10483740E+01 * (x0 * x0 * x0 * x2 * x2 * x2 * x2) + -1.15687118E+03 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + -3.93084818E+02 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + -4.14889950E+01 * (x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + 3.60191260E+02 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.62824389E+00 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + 1.30514173E+01 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + 6.62943701E+02 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2);

    REAL8 coeff_array_A4[4][4][5] = {
        {{1.59708156e-01, -3.13330481e-01, 3.82163433e-01, 6.14498512e-01, -3.91386611e-01},
         {3.68521318e-01, 2.65522323e+00, -2.23540184e+00, -4.37061988e+00, 2.01077124e+00},
         {-8.32553332e-01, -6.02094408e+00, 4.86548073e+00, 8.36610850e+00, -2.56728295e+00},
         {8.05210158e-01, 4.06905683e+00, -3.59268661e+00, -4.39013590e+00, 5.98214021e-01}},

        {{1.76157925e+00, 4.37672477e+00, -3.65397391e+00, -9.43917739e+00, 2.79796061e+00},
         {-8.10802054e+00, -3.83223991e+01, 2.30718712e+01, 6.83013158e+01, -1.66542213e+01},
         {1.87499491e+01, 9.08302756e+01, -5.62222338e+01, -1.34263772e+02, 1.75463323e+01},
         {-1.63620472e+01, -6.32085162e+01, 4.63243515e+01, 7.33928698e+01, 0.00000000e+00}},

        {{-1.01776826e+01, -2.06957701e+01, 8.82898695e+00, 4.68191983e+01, 0.00000000e+00},
         {5.43489123e+01, 1.91078478e+02, -8.10382470e+01, -3.45628809e+02, 3.77841307e+01},
         {-1.27304367e+02, -4.65340116e+02, 2.43493364e+02, 6.96748172e+02, -4.14889950e+01},
         {1.07160127e+02, 3.33094494e+02, -2.25400241e+02, -3.93084818e+02, -2.62824389e+00}},

        {{2.05624058e+01, 3.29280099e+01, 0.00000000e+00, -7.52635673e+01, -2.10483740e+01},
         {-1.10660415e+02, -3.19269499e+02, 8.40211777e+01, 5.64442664e+02, 0.00000000e+00},
         {2.60872767e+02, 7.88656773e+02, -3.46838253e+02, -1.15687118e+03, 1.30514173e+01},
         {-2.17001178e+02, -5.76084011e+02, 3.60191260e+02, 6.62943701e+02, 0.00000000e+00}}};

    REAL8 A4 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_A4, eta, chi, costheta);

    return A4 * A4;
  }

  /**
   * expressions for each of the co-efficients that appear in the merger-ringdown expression for beta
   */
  REAL8 IMRPhenomX_PNR_beta_B0_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    REAL8 theta = acos(costheta);

    // REAL8 B0 = 4.99601117E-02 * (x1) + -1.76204300E+01 * (x0) + 1.66666664E+00 + 9.42765416E+01 * (x0 * x0) + -1.56973167E+00 * (x2 * x2) + 1.63860030E+00 * (x1 * x2) + -3.07243357E+01 * (x0 * x1 * x2) + -7.31228312E-01 * (x2 * x2 * x2) + -1.90285349E+02 * (x0 * x0 * x0) + 1.25605160E+01 * (x0 * x2 * x2) + 1.68442510E+02 * (x0 * x0 * x1 * x2) + 3.01647084E+00 * (x2 * x2 * x2 * x2) + 5.10396051E+00 * (x0 * x2 * x2 * x2) + -2.87986635E+02 * (x0 * x0 * x0 * x1 * x2) + -4.98474167E+00 * (x0 * x0 * x1 * x2 * x2) + -2.33247834E+01 * (x0 * x2 * x2 * x2 * x2) + -8.95949243E+01 * (x0 * x0 * x0 * x2 * x2) + -4.93485987E+00 * (x1 * x2 * x2 * x2 * x2) + 8.12576632E+01 * (x0 * x1 * x2 * x2 * x2 * x2) + -3.54218713E+01 * (x0 * x0 * x0 * x2 * x2 * x2) + -4.19372351E+02 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + 1.69656947E+02 * (x0 * x0 * x0 * x2 * x2 * x2 * x2) + 7.11235264E+02 * (x0 * x0 * x0 * x1 * x2 * x2 * x2 * x2);

    REAL8 coeff_array_B0[4][4][5] = {
        {{1.66666664e+00, 0.00000000e+00, -1.56973167e+00, -7.31228312e-01, 3.01647084e+00},
         {4.99601117e-02, 1.63860030e+00, 0.00000000e+00, 0.00000000e+00, -4.93485987e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-1.76204300e+01, 0.00000000e+00, 1.25605160e+01, 5.10396051e+00, -2.33247834e+01},
         {0.00000000e+00, -3.07243357e+01, 0.00000000e+00, 0.00000000e+00, 8.12576632e+01},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{9.42765416e+01, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 1.68442510e+02, -4.98474167e+00, 0.00000000e+00, -4.19372351e+02},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-1.90285349e+02, 0.00000000e+00, -8.95949243e+01, -3.54218713e+01, 1.69656947e+02},
         {0.00000000e+00, -2.87986635e+02, 0.00000000e+00, 0.00000000e+00, 7.11235264e+02},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}}};

    REAL8 B0 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B0, eta, chi, costheta);

    return chi * sin(theta) * B0;
  }

  REAL8 IMRPhenomX_PNR_beta_B1_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 B1 = -1.91489844E+02 * (x0) + 8.90772876E+00 * (x1) + -3.32454515E+01 * (x2) + 9.90840259E+00 + 7.10589710E+02 * (x0 * x2) + -4.33793801E+01 * (x0 * x1) + -2.86081100E+01 * (x2 * x2) + 1.26143166E+03 * (x0 * x0) + -1.11362664E+01 * (x1 * x1) + 1.51402126E+02 * (x1 * x2) + 6.62671100E+01 * (x0 * x1 * x1) + -3.23610605E+03 * (x0 * x1 * x2) + -1.29998137E+02 * (x1 * x1 * x2) + -4.73928061E+03 * (x0 * x0 * x2) + 4.41611143E+01 * (x1 * x2 * x2) + -2.62496323E+03 * (x0 * x0 * x0) + 6.41633616E+02 * (x0 * x2 * x2) + 2.51567951E+01 * (x2 * x2 * x2) + 2.77517195E+03 * (x0 * x1 * x1 * x2) + -4.92342757E+01 * (x1 * x2 * x2 * x2) + -3.82595920E+01 * (x1 * x1 * x2 * x2) + 2.14432888E+04 * (x0 * x0 * x1 * x2) + -6.44370516E+02 * (x0 * x2 * x2 * x2) + -4.12120327E+03 * (x0 * x0 * x2 * x2) + -1.30528975E+03 * (x0 * x1 * x2 * x2) + 9.85262924E+03 * (x0 * x0 * x0 * x2) + -1.82864908E+04 * (x0 * x0 * x1 * x1 * x2) + 1.77860699E+03 * (x0 * x1 * x2 * x2 * x2) + -1.88093090E+02 * (x0 * x0 * x0 * x1 * x1) + -4.43458441E+04 * (x0 * x0 * x0 * x1 * x2) + 1.29179889E+03 * (x0 * x1 * x1 * x2 * x2) + 8.11282583E+03 * (x0 * x0 * x0 * x2 * x2) + 9.00514520E+03 * (x0 * x0 * x1 * x2 * x2) + 4.80089044E+03 * (x0 * x0 * x2 * x2 * x2) + -8.65826585E+02 * (x0 * x1 * x1 * x2 * x2 * x2) + 3.75890504E+04 * (x0 * x0 * x0 * x1 * x1 * x2) + -1.51266329E+04 * (x0 * x0 * x1 * x2 * x2 * x2) + -9.40456062E+03 * (x0 * x0 * x1 * x1 * x2 * x2) + -1.80428855E+04 * (x0 * x0 * x0 * x1 * x2 * x2) + -1.06934934E+04 * (x0 * x0 * x0 * x2 * x2 * x2) + 9.49293573E+03 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + 3.60864889E+04 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + 1.94233528E+04 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + -2.48379649E+04 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2);

    REAL8 coeff_array_B1[4][4][5] = {
        {{9.90840259e+00, -3.32454515e+01, -2.86081100e+01, 2.51567951e+01, 0.00000000e+00},
         {8.90772876e+00, 1.51402126e+02, 4.41611143e+01, -4.92342757e+01, 0.00000000e+00},
         {-1.11362664e+01, -1.29998137e+02, -3.82595920e+01, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-1.91489844e+02, 7.10589710e+02, 6.41633616e+02, -6.44370516e+02, 0.00000000e+00},
         {-4.33793801e+01, -3.23610605e+03, -1.30528975e+03, 1.77860699e+03, 0.00000000e+00},
         {6.62671100e+01, 2.77517195e+03, 1.29179889e+03, -8.65826585e+02, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{1.26143166e+03, -4.73928061e+03, -4.12120327e+03, 4.80089044e+03, 0.00000000e+00},
         {0.00000000e+00, 2.14432888e+04, 9.00514520e+03, -1.51266329e+04, 0.00000000e+00},
         {0.00000000e+00, -1.82864908e+04, -9.40456062e+03, 9.49293573e+03, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-2.62496323e+03, 9.85262924e+03, 8.11282583e+03, -1.06934934e+04, 0.00000000e+00},
         {0.00000000e+00, -4.43458441e+04, -1.80428855e+04, 3.60864889e+04, 0.00000000e+00},
         {-1.88093090e+02, 3.75890504e+04, 1.94233528e+04, -2.48379649e+04, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}}};

    REAL8 B1 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B1, eta, chi, costheta);

    return chi * sqrt(1.0 - costheta * costheta) * exp(B1);
  }

  REAL8 IMRPhenomX_PNR_beta_B2_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 B2 = 3.49258945E+00 * (x1) + -9.96099698E+00 * (x2) + -2.58211935E+02 * (x0) + 1.68444475E+01 + 3.33569187E+02 * (x0 * x2) + -7.68630145E+01 * (x2 * x2) + 1.63032394E+03 * (x0 * x0) + -7.44266403E+01 * (x1 * x2) + 6.37900482E+02 * (x0 * x1 * x2) + -8.31987801E+00 * (x1 * x1 * x1) + 4.89740754E+02 * (x1 * x1 * x2) + -4.21556808E+01 * (x2 * x2 * x2) + -2.65257994E+03 * (x0 * x0 * x2) + -3.32826175E+03 * (x0 * x0 * x0) + 1.38061489E+03 * (x0 * x2 * x2) + 3.39005879E+02 * (x1 * x2 * x2) + -8.37654765E+03 * (x0 * x1 * x1 * x2) + 3.98198351E+01 * (x0 * x1 * x1 * x1) + 5.29979449E+02 * (x1 * x2 * x2 * x2) + -5.52234827E+02 * (x1 * x1 * x2 * x2) + -1.24325251E+03 * (x0 * x0 * x1 * x2) + 7.33370180E+01 * (x2 * x2 * x2 * x2) + 4.58035577E+02 * (x0 * x2 * x2 * x2) + -7.65899068E+03 * (x0 * x0 * x2 * x2) + -5.60552556E+03 * (x0 * x1 * x2 * x2) + 5.92187248E+03 * (x0 * x0 * x0 * x2) + -2.53504341E+02 * (x0 * x0 * x0 * x1) + -4.70597369E+02 * (x1 * x1 * x1 * x2) + -1.32126338E+03 * (x1 * x1 * x2 * x2 * x2) + 4.92068135E+04 * (x0 * x0 * x1 * x1 * x2) + -8.01736623E+03 * (x0 * x1 * x2 * x2 * x2) + 2.83713129E+04 * (x0 * x0 * x1 * x2 * x2) + 8.18795918E+03 * (x0 * x1 * x1 * x2 * x2) + 2.62076484E+02 * (x1 * x1 * x1 * x2 * x2) + -1.07648973E+03 * (x0 * x2 * x2 * x2 * x2) + 1.35111259E+04 * (x0 * x0 * x0 * x2 * x2) + 8.66882648E+03 * (x0 * x1 * x1 * x1 * x2) + -6.04337081E+02 * (x1 * x2 * x2 * x2 * x2) + -1.15254729E+03 * (x0 * x0 * x2 * x2 * x2) + -3.03868645E+03 * (x0 * x1 * x1 * x1 * x2 * x2) + 2.20794318E+04 * (x0 * x1 * x1 * x2 * x2 * x2) + -9.66185024E+04 * (x0 * x0 * x0 * x1 * x1 * x2) + 3.98749465E+04 * (x0 * x0 * x1 * x2 * x2 * x2) + -3.67889222E+04 * (x0 * x0 * x1 * x1 * x2 * x2) + 4.93839236E+03 * (x0 * x0 * x2 * x2 * x2 * x2) + 9.31954814E+03 * (x0 * x1 * x2 * x2 * x2 * x2) + -4.43715296E+04 * (x0 * x0 * x0 * x1 * x2 * x2) + -5.31909605E+04 * (x0 * x0 * x1 * x1 * x1 * x2) + 8.86094957E+02 * (x1 * x1 * x1 * x2 * x2 * x2) + 1.20063914E+03 * (x1 * x1 * x2 * x2 * x2 * x2) + 1.06346795E+05 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + -7.25121024E+02 * (x1 * x1 * x1 * x2 * x2 * x2 * x2) + -4.54442750E+04 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + -1.82972414E+04 * (x0 * x1 * x1 * x2 * x2 * x2 * x2) + -1.23091363E+05 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -1.56932408E+04 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -6.61219185E+04 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + 4.72286307E+04 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + 8.86769471E+03 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + -7.07609947E+03 * (x0 * x0 * x0 * x2 * x2 * x2 * x2) + 6.92381252E+04 * (x0 * x0 * x0 * x1 * x2 * x2 * x2 * x2) + 1.09225254E+04 * (x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + 2.27703658E+05 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + 9.24026509E+04 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 8.73858633E+04 * (x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -5.10645576E+04 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + -1.27186811E+05 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -1.78622185E+05 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 7.07426805E+04 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2);

    REAL8 coeff_array_B2[4][4][5] = {
        {{1.68444475e+01, -9.96099698e+00, -7.68630145e+01, -4.21556808e+01, 7.33370180e+01},
         {3.49258945e+00, -7.44266403e+01, 3.39005879e+02, 5.29979449e+02, -6.04337081e+02},
         {0.00000000e+00, 4.89740754e+02, -5.52234827e+02, -1.32126338e+03, 1.20063914e+03},
         {-8.31987801e+00, -4.70597369e+02, 2.62076484e+02, 8.86094957e+02, -7.25121024e+02}},

        {{-2.58211935e+02, 3.33569187e+02, 1.38061489e+03, 4.58035577e+02, -1.07648973e+03},
         {0.00000000e+00, 6.37900482e+02, -5.60552556e+03, -8.01736623e+03, 9.31954814e+03},
         {0.00000000e+00, -8.37654765e+03, 8.18795918e+03, 2.20794318e+04, -1.82972414e+04},
         {3.98198351e+01, 8.66882648e+03, -3.03868645e+03, -1.56932408e+04, 1.09225254e+04}},

        {{1.63032394e+03, -2.65257994e+03, -7.65899068e+03, -1.15254729e+03, 4.93839236e+03},
         {0.00000000e+00, -1.24325251e+03, 2.83713129e+04, 3.98749465e+04, -4.54442750e+04},
         {0.00000000e+00, 4.92068135e+04, -3.67889222e+04, -1.23091363e+05, 8.73858633e+04},
         {0.00000000e+00, -5.31909605e+04, 8.86769471e+03, 9.24026509e+04, -5.10645576e+04}},

        {{-3.32826175e+03, 5.92187248e+03, 1.35111259e+04, 0.00000000e+00, -7.07609947e+03},
         {-2.53504341e+02, 0.00000000e+00, -4.43715296e+04, -6.61219185e+04, 6.92381252e+04},
         {0.00000000e+00, -9.66185024e+04, 4.72286307e+04, 2.27703658e+05, -1.27186811e+05},
         {0.00000000e+00, 1.06346795e+05, 0.00000000e+00, -1.78622185e+05, 7.07426805e+04}}};

    REAL8 B2 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B2, eta, chi, costheta);

    return chi * sqrt(1.0 - costheta * costheta) * (-exp(B2));
  }

  REAL8 IMRPhenomX_PNR_beta_B3_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 B3 = 1.66475047E+00 * (x1) + 4.10145391E+01 * (x2) + -9.21054892E+01 * (x0) + 1.10674893E+01 + -5.07904904E+02 * (x0 * x2) + -1.32076236E+02 * (x2 * x2) + 5.05319709E+02 * (x0 * x0) + -4.71864368E+00 * (x1 * x1) + -3.61980618E+02 * (x1 * x2) + 1.86230944E+02 * (x0 * x1 * x1) + 4.89896439E+03 * (x0 * x1 * x2) + 3.25316165E+00 * (x1 * x1 * x1) + 8.93190804E+02 * (x1 * x1 * x2) + -1.84329484E+02 * (x2 * x2 * x2) + 1.54084992E+02 * (x0 * x0 * x1) + 1.90048713E+03 * (x0 * x0 * x2) + -9.26104915E+02 * (x0 * x0 * x0) + 2.19722191E+03 * (x0 * x2 * x2) + 9.59447596E+02 * (x1 * x2 * x2) + -1.31096664E+04 * (x0 * x1 * x1 * x2) + -2.61918585E+02 * (x0 * x1 * x1 * x1) + 1.47930479E+03 * (x1 * x2 * x2 * x2) + -1.46915897E+03 * (x0 * x0 * x1 * x1) + -2.06302158E+03 * (x1 * x1 * x2 * x2) + -2.19094594E+04 * (x0 * x0 * x1 * x2) + 2.42572428E+02 * (x2 * x2 * x2 * x2) + 2.72460464E+03 * (x0 * x2 * x2 * x2) + -1.14552506E+04 * (x0 * x0 * x2 * x2) + -1.60398250E+04 * (x0 * x1 * x2 * x2) + -2.11224178E+03 * (x0 * x0 * x0 * x2) + -6.66132621E+02 * (x0 * x0 * x0 * x1) + -6.40128793E+02 * (x1 * x1 * x1 * x2) + -3.06024612E+03 * (x1 * x1 * x2 * x2 * x2) + 6.50229758E+04 * (x0 * x0 * x1 * x1 * x2) + 2.10490032E+03 * (x0 * x0 * x1 * x1 * x1) + -2.23550472E+04 * (x0 * x1 * x2 * x2 * x2) + 3.08266725E+03 * (x0 * x0 * x0 * x1 * x1) + 8.52189514E+04 * (x0 * x0 * x1 * x2 * x2) + 3.45203982E+04 * (x0 * x1 * x1 * x2 * x2) + 1.29904097E+03 * (x1 * x1 * x1 * x2 * x2) + -3.74384080E+03 * (x0 * x2 * x2 * x2 * x2) + 1.91858728E+04 * (x0 * x0 * x0 * x2 * x2) + 9.87646600E+03 * (x0 * x1 * x1 * x1 * x2) + -2.05602276E+03 * (x1 * x2 * x2 * x2 * x2) + 3.26263491E+04 * (x0 * x0 * x0 * x1 * x2) + -1.29334429E+04 * (x0 * x0 * x2 * x2 * x2) + -2.14822685E+04 * (x0 * x1 * x1 * x1 * x2 * x2) + 4.69061555E+04 * (x0 * x1 * x1 * x2 * x2 * x2) + -4.34258876E+03 * (x0 * x0 * x0 * x1 * x1 * x1) + -1.08213730E+05 * (x0 * x0 * x0 * x1 * x1 * x2) + 1.10216483E+05 * (x0 * x0 * x1 * x2 * x2 * x2) + -1.85763088E+05 * (x0 * x0 * x1 * x1 * x2 * x2) + 1.85473707E+04 * (x0 * x0 * x2 * x2 * x2 * x2) + 3.30354479E+04 * (x0 * x1 * x2 * x2 * x2 * x2) + 1.99782054E+04 * (x0 * x0 * x0 * x2 * x2 * x2) + -1.45621057E+05 * (x0 * x0 * x0 * x1 * x2 * x2) + -5.15537321E+04 * (x0 * x0 * x1 * x1 * x1 * x2) + 1.87648143E+03 * (x1 * x1 * x1 * x2 * x2 * x2) + 4.47587621E+03 * (x1 * x1 * x2 * x2 * x2 * x2) + 8.96784959E+04 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + -2.85924327E+03 * (x1 * x1 * x1 * x2 * x2 * x2 * x2) + -1.71528592E+05 * (x0 * x0 * x1 * x2 * x2 * x2 * x2) + -7.31278030E+04 * (x0 * x1 * x1 * x2 * x2 * x2 * x2) + -2.36448204E+05 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -2.92108696E+04 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -1.78381076E+05 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + 3.22014369E+05 * (x0 * x0 * x0 * x1 * x1 * x2 * x2) + 1.15343849E+05 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.97929760E+04 * (x0 * x0 * x0 * x2 * x2 * x2 * x2) + 2.88686822E+05 * (x0 * x0 * x0 * x1 * x2 * x2 * x2 * x2) + 4.72017088E+04 * (x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + 3.92667036E+05 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + 1.50171977E+05 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 3.86480237E+05 * (x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -1.99925823E+05 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2) + -2.52074452E+05 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2) + -6.60777075E+05 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2 * x2) + -2.54496967E+05 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + 4.34668358E+05 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2 * x2);

    REAL8 coeff_array_B3[4][4][5] = {
				     {{-3.84565012e+01, -3.72563299e+02,  4.15226702e+02, 1.00690394e+03, -8.39744987e+02},
				      { 0.00000000e+00,  3.46028868e+03, -2.13668892e+03, -8.66465191e+03,  6.59782412e+03},
				      { 0.00000000e+00, -8.98093407e+03,  2.77456494e+03, 1.96158493e+04, -1.21936587e+04},
				      { 2.71298988e+01,  6.57289583e+03, -9.16538843e+02, -1.30987997e+04,  7.03985911e+03}},
				     
				     {{ 6.41285217e+02,  6.43439871e+03, -6.12430709e+03, -1.69301376e+04,  1.21331375e+04},
				      { 3.33545476e+02, -6.18206463e+04,  2.59875058e+04, 1.49340484e+05, -9.65625710e+04},
				      {-9.31951414e+02,  1.65382217e+05, -1.85778489e+04, -3.45011414e+05,  1.68522923e+05},
				      { 2.48942253e+02, -1.23069045e+05, -6.07427694e+03, 2.34066117e+05, -9.29073962e+04}},
				     
				     {{-3.71186534e+03, -3.67048647e+04,  2.68404803e+04, 9.29790279e+04, -5.37333198e+04},
				      {-4.43880950e+03,  3.65054516e+05, -7.44198342e+04, -8.43552293e+05,  4.33017698e+05},
				      { 9.75983575e+03, -1.00029802e+06, -7.79273838e+04, 1.99051043e+06, -6.81841816e+05},
				      {-3.86313127e+03,  7.53226934e+05,  1.65966971e+05, -1.37070858e+06,  3.39981734e+05}},
				     
				     {{ 7.19881427e+03,  6.92187556e+04, -3.52397747e+04, -1.67861147e+05,  7.24438477e+04},
				      { 1.17058924e+04, -7.06752764e+05,  0.00000000e+00, 1.56142369e+06, -5.80532954e+05},
				      {-2.30996347e+04,  1.96859070e+06,  4.91882690e+05, -3.75576640e+06,  7.25646477e+05},
				      { 8.96350149e+03, -1.49311558e+06, -5.56575555e+05, 2.61876368e+06, -2.57044129e+05}}};
    
    REAL8 B3 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B3, eta, chi, costheta);

    return B3;
  }

  REAL8 IMRPhenomX_PNR_beta_B4_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 B4 = 6.08297089E+03 * (x2) + 6.17576977E+01 + -1.20759018E+05 * (x0 * x2) + -5.03606398E+02 * (x0 * x1) + -3.28698499E+01 * (x2 * x2) + -2.54963813E+02 * (x1 * x1) + -4.40828909E+04 * (x1 * x2) + 2.32342739E+03 * (x0 * x1 * x1) + 8.74143791E+05 * (x0 * x1 * x2) + 9.10415554E+04 * (x1 * x1 * x2) + 7.41502193E+05 * (x0 * x0 * x2) + 4.78799900E+02 * (x1 * x2 * x2) + -9.00899363E+03 * (x2 * x2 * x2) + -1.80424045E+06 * (x0 * x1 * x1 * x2) + 3.24392460E+03 * (x0 * x1 * x1 * x1) + 6.47461695E+04 * (x1 * x2 * x2 * x2) + -8.92432039E+01 * (x1 * x1 * x2 * x2) + -5.36676137E+06 * (x0 * x0 * x1 * x2) + 1.83101251E+05 * (x0 * x2 * x2 * x2) + -4.08363439E+02 * (x0 * x0 * x2 * x2) + -6.91844447E+03 * (x0 * x1 * x2 * x2) + -1.42623948E+06 * (x0 * x0 * x0 * x2) + -5.60479001E+04 * (x1 * x1 * x1 * x2) + -1.35277054E+05 * (x1 * x1 * x2 * x2 * x2) + 1.10742322E+07 * (x0 * x0 * x1 * x1 * x2) + -3.24694820E+04 * (x0 * x0 * x1 * x1 * x1) + -1.31847221E+06 * (x0 * x1 * x2 * x2 * x2) + 1.03241945E+07 * (x0 * x0 * x0 * x1 * x2) + 1.11193745E+06 * (x0 * x1 * x1 * x1 * x2) + 4.79979448E+04 * (x0 * x0 * x1 * x2 * x2) + -1.14406701E+06 * (x0 * x0 * x2 * x2 * x2) + 2.75883682E+06 * (x0 * x1 * x1 * x2 * x2 * x2) + 6.24935229E+04 * (x0 * x0 * x0 * x1 * x1 * x1) + -2.13005503E+07 * (x0 * x0 * x0 * x1 * x1 * x2) + 8.24912816E+06 * (x0 * x0 * x1 * x2 * x2 * x2) + -1.64024783E+04 * (x0 * x0 * x1 * x1 * x2 * x2) + -6.83199970E+06 * (x0 * x0 * x1 * x1 * x1 * x2) + 2.23076875E+06 * (x0 * x0 * x0 * x2 * x2 * x2) + -8.14473563E+04 * (x0 * x0 * x0 * x1 * x2 * x2) + 8.44304524E+04 * (x1 * x1 * x1 * x2 * x2 * x2) + 1.31513382E+07 * (x0 * x0 * x0 * x1 * x1 * x1 * x2) + -1.72640929E+07 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -1.72623318E+06 * (x0 * x1 * x1 * x1 * x2 * x2 * x2) + -1.60990267E+07 * (x0 * x0 * x0 * x1 * x2 * x2 * x2) + 1.23176652E+04 * (x0 * x0 * x1 * x1 * x1 * x2 * x2) + 3.36817613E+07 * (x0 * x0 * x0 * x1 * x1 * x2 * x2 * x2) + 1.08111926E+07 * (x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2) + -2.10935651E+07 * (x0 * x0 * x0 * x1 * x1 * x1 * x2 * x2 * x2);

    REAL8 coeff_array_B4[4][4][5] = {

       {{ 4.43516177e+00,  6.00923095e+03,  0.00000000e+00, -8.87403368e+03,  0.00000000e+00},
        { 0.00000000e+00, -4.37621347e+04,  4.62648713e+02, 6.41116102e+04,  0.00000000e+00},
        { 0.00000000e+00,  9.07775710e+04,  0.00000000e+00, -1.34626196e+05,  0.00000000e+00},
        {-2.91806956e+02, -5.60980213e+04,  0.00000000e+00, 8.43829555e+04,  0.00000000e+00}},

       {{ 0.00000000e+00, -1.19641275e+05,  0.00000000e+00, 1.80891623e+05,  0.00000000e+00},
        { 6.76653146e+03,  8.69092181e+05, -9.10600697e+03, -1.30756439e+06,  0.00000000e+00},
        {-1.71217123e+04, -1.79967454e+06,  5.23377635e+02, 2.74646543e+06,  0.00000000e+00},
        { 1.78197625e+04,  1.11232175e+06,  0.00000000e+00, -1.72414323e+06,  0.00000000e+00}},

       {{ 5.22673374e+03,  7.36732009e+05, -1.55862480e+03, -1.13336165e+06,  0.00000000e+00},
        {-8.09346012e+04, -5.34331917e+06,  6.60633209e+04, 8.19202160e+06,  0.00000000e+00},
        { 1.97663741e+05,  1.10486885e+07, -2.80356801e+04, -1.71897436e+07,  0.00000000e+00},
        {-1.73210655e+05, -6.82927341e+06,  1.70943961e+04, 1.07898804e+07,  0.00000000e+00}},

       {{-1.63063512e+04, -1.42058459e+06,  0.00000000e+00, 2.21498743e+06,  0.00000000e+00},   
        { 2.01426348e+05,  1.02916792e+07, -1.00924025e+05, -1.60054558e+07,  0.00000000e+00},
        {-4.84862499e+05, -2.12548744e+07,  0.00000000e+00, 3.35401658e+07,  0.00000000e+00},
        { 4.04834917e+05,  1.31368813e+07,  0.00000000e+00, -2.10371951e+07,  0.00000000e+00}}};

    REAL8 B4 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B4, eta, chi, costheta);

    return B4 * B4;
  }

  REAL8 IMRPhenomX_PNR_beta_B5_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {

    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    // REAL8 B5 = -7.47264130E-01 * (x0) + 2.68923234E-01 * (x1) + 2.59611799E-01 + -1.13259698E-01 * (x0 * x2) + -1.15034289E+00 * (x0 * x1) + -1.01759974E-01 * (x2 * x2) + 6.19349011E+00 * (x0 * x0) + -2.99635390E-01 * (x1 * x1) + 1.56410664E+00 * (x1 * x2) + 2.19968539E+00 * (x0 * x1 * x1) + -3.01781829E+01 * (x0 * x1 * x2) + -2.51949760E+00 * (x1 * x1 * x2) + -8.34460240E-01 * (x1 * x2 * x2) + -1.10778265E+01 * (x0 * x0 * x0) + 3.86831930E+00 * (x0 * x2 * x2) + -7.86816373E-01 * (x2 * x2 * x2) + 4.83446237E+01 * (x0 * x1 * x1 * x2) + 2.51157175E+00 * (x1 * x2 * x2 * x2) + -3.59555219E+00 * (x0 * x0 * x1 * x1) + 1.22381580E+00 * (x1 * x1 * x2 * x2) + 1.83372982E+02 * (x0 * x0 * x1 * x2) + 1.35408150E+01 * (x0 * x2 * x2 * x2) + -2.73053016E+01 * (x0 * x0 * x2 * x2) + 4.63583701E+00 * (x0 * x1 * x2 * x2) + -1.53171177E+00 * (x1 * x1 * x2 * x2 * x2) + -2.83554144E+02 * (x0 * x0 * x1 * x1 * x2) + -3.63125637E+01 * (x0 * x1 * x2 * x2 * x2) + -3.41305478E+02 * (x0 * x0 * x0 * x1 * x2) + -1.00638506E+01 * (x0 * x1 * x1 * x2 * x2) + 5.46594653E+01 * (x0 * x0 * x0 * x2 * x2) + -7.09847972E+01 * (x0 * x0 * x2 * x2 * x2) + 1.54589802E+01 * (x0 * x1 * x1 * x2 * x2 * x2) + 5.12910741E+02 * (x0 * x0 * x0 * x1 * x1 * x2) + 1.60130280E+02 * (x0 * x0 * x1 * x2 * x2 * x2) + 2.10964061E+01 * (x0 * x0 * x1 * x1 * x2 * x2) + -2.31451284E+01 * (x0 * x0 * x0 * x1 * x2 * x2) + 1.19828024E+02 * (x0 * x0 * x0 * x2 * x2 * x2) + -3.46919956E+01 * (x0 * x0 * x1 * x1 * x2 * x2 * x2) + -2.30544851E+02 * (x0 * x0 * x0 * x1 * x2 * x2 * x2);

    REAL8 coeff_array_B5[4][4][5] = {
       {{ 3.10386302e-01,  6.00813481e-02, -3.53776520e-01, -8.86045114e-01,  0.00000000e+00},
        { 0.00000000e+00,  1.25742267e+00,  3.84777116e-01, 3.01828258e+00,  0.00000000e+00},
        { 0.00000000e+00, -2.21073099e+00,  0.00000000e+00, -2.04284938e+00,  0.00000000e+00},
        { 0.00000000e+00,  0.00000000e+00,  0.00000000e+00, 0.00000000e+00,  0.00000000e+00}},

       {{-1.88945371e+00, -7.26051110e-01,  9.06175997e+00, 1.46766342e+01,  0.00000000e+00},
        { 4.86280161e+00, -2.66243788e+01, -2.05433977e+01, -4.27192221e+01,  0.00000000e+00},
        {-4.39245048e+00,  4.45640600e+01,  1.51720256e+01, 2.22127532e+01,  0.00000000e+00},
        { 0.00000000e+00,  0.00000000e+00,  0.00000000e+00, 0.00000000e+00,  0.00000000e+00}},

       {{ 1.40364230e+01,  0.00000000e+00, -6.04082590e+01, -7.25620325e+01,  0.00000000e+00},
        {-4.10909266e+01,  1.77965640e+02,  1.60792365e+02, 1.75894855e+02,  0.00000000e+00},
        { 4.08774229e+01, -2.75528934e+02, -1.39837130e+02, -5.43148597e+01,  0.00000000e+00},
        { 0.00000000e+00,  0.00000000e+00,  0.00000000e+00, 0.00000000e+00,  0.00000000e+00}},

       {{-2.75834452e+01,  6.53438283e+00,  1.20524698e+02, 1.13506432e+02,  0.00000000e+00},
        { 8.61828779e+01, -3.59365491e+02, -3.43549992e+02, -2.20083234e+02,  0.00000000e+00},
        {-9.24276057e+01,  5.23784793e+02,  3.20340101e+02, 0.00000000e+00,  0.00000000e+00},
        { 0.00000000e+00,  0.00000000e+00,  0.00000000e+00, 0.00000000e+00,  0.00000000e+00}}};

    REAL8 B5 = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_B5, eta, chi, costheta);

    return -B5 * B5;
  }

  REAL8 IMRPhenomX_PNR_beta_Bf_coefficient(REAL8 eta, REAL8 chi, REAL8 costheta)
  {
    // REAL8 x0 = eta;
    // REAL8 x1 = chi;
    // REAL8 x2 = costheta;

    REAL8 theta = acos(costheta);

    // REAL8 Bf = -3.63088945E+01 * (x0) + 1.34032610E+00 * (x2) + 3.09601897E+00 + -1.38476321E+01 * (x0 * x2) + 2.03218040E+02 * (x0 * x0) + -1.45826218E+00 * (x2 * x2) + -3.78818904E-01 * (x1 * x1) + -3.88910127E+00 * (x1 * x2) + 3.45434991E+00 * (x0 * x1 * x1) + 1.89698391E+01 * (x0 * x1 * x2) + 3.44727678E+00 * (x1 * x1 * x2) + 6.36061063E+01 * (x0 * x0 * x2) + 3.71319679E+00 * (x1 * x2 * x2) + -4.10212724E+02 * (x0 * x0 * x0) + 2.66368579E+01 * (x0 * x2 * x2) + -9.37928603E-01 * (x2 * x2 * x2) + -1.73678802E+01 * (x0 * x1 * x1 * x2) + -5.50316593E-01 * (x1 * x2 * x2 * x2) + -7.62767762E+00 * (x0 * x0 * x1 * x1) + -3.74449485E+00 * (x1 * x1 * x2 * x2) + 6.74798082E+00 * (x0 * x2 * x2 * x2) + -1.67985530E+02 * (x0 * x0 * x2 * x2) + -1.58854735E+01 * (x0 * x1 * x2 * x2) + -1.31527994E+02 * (x0 * x0 * x0 * x2) + 8.21928673E-01 * (x1 * x1 * x2 * x2 * x2) + -1.11744859E+00 * (x0 * x1 * x2 * x2 * x2) + 1.58282455E+01 * (x0 * x1 * x1 * x2 * x2) + 3.43405885E+02 * (x0 * x0 * x0 * x2 * x2) + -8.74477395E+00 * (x0 * x0 * x2 * x2 * x2);

    REAL8 coeff_array_Bf[4][4][5] = {
        {{3.09601897e+00, 1.34032610e+00, -1.45826218e+00, -9.37928603e-01, 0.00000000e+00},
         {0.00000000e+00, -3.88910127e+00, 3.71319679e+00, -5.50316593e-01, 0.00000000e+00},
         {-3.78818904e-01, 3.44727678e+00, -3.74449485e+00, 8.21928673e-01, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-3.63088945e+01, -1.38476321e+01, 2.66368579e+01, 6.74798082e+00, 0.00000000e+00},
         {0.00000000e+00, 1.89698391e+01, -1.58854735e+01, -1.11744859e+00, 0.00000000e+00},
         {3.45434991e+00, -1.73678802e+01, 1.58282455e+01, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{2.03218040e+02, 6.36061063e+01, -1.67985530e+02, -8.74477395e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {-7.62767762e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}},

        {{-4.10212724e+02, -1.31527994e+02, 3.43405885e+02, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00},
         {0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00}}};

    REAL8 Bf = IMRPhenomX_PNR_evaluate_coefficient_array(coeff_array_Bf, eta, chi, costheta);

    return chi * sin(theta) * Bf;
  }

#ifdef __cplusplus
}
#endif
