/* "SSH private key cracker" patch for JtR. Hacked together during
 * April of 2011 by Dhiru Kholia <dhiru.kholia at gmail.com> for GSoC.
 *
 * This software is Copyright (c) 2011, Dhiru Kholia <dhiru.kholia at gmail.com>,
 * and it is hereby released to the general public under the following terms:
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted.
 *
 * This patch is inspired by the ssh-privkey-crack program.
 * http://neophob.com/2007/10/ssh-private-key-cracker/
 *
 * PEM_read_bio_PrivateKey and related OpenSSL functions are too high
 * level for brute-forcing purposes. So we drill down and find suitable
 * low-level OpenSSL functions. */

#if FMT_EXTERNS_H
extern struct fmt_main fmt_ssh;
#elif FMT_REGISTERS_H
john_register_one(&fmt_ssh);
#else

#include <openssl/opensslv.h>
#include <openssl/crypto.h>
#include <openssl/ssl.h>
#include <openssl/bio.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#undef MEM_FREE
#include "options.h"
#ifdef _OPENMP
#include <omp.h>
#define OMP_SCALE               64
#endif
#include <string.h>
#include "arch.h"
#include "common.h"
#include "formats.h"
#include "params.h"
#include "misc.h"
#include "memdbg.h"

#define FORMAT_LABEL        "SSH"
#define FORMAT_NAME         ""
#define ALGORITHM_NAME      "RSA/DSA 32/" ARCH_BITS_STR
#define BENCHMARK_COMMENT   " (one 2048-bit RSA and one 1024-bit DSA key)"
#define BENCHMARK_LENGTH    -1001
#define PLAINTEXT_LENGTH    32
#define BINARY_SIZE         0
#define SALT_SIZE           sizeof(struct custom_salt)
#define BINARY_ALIGN		1
#define SALT_ALIGN			sizeof(long)
#define MIN_KEYS_PER_CRYPT  1
#define MAX_KEYS_PER_CRYPT  1

static char (*saved_key)[PLAINTEXT_LENGTH + 1];
static int any_cracked, *cracked;
static size_t cracked_size;

static struct custom_salt {
	long len;
	char data[4096];
	EVP_CIPHER_INFO cipher;
	EVP_PKEY pk;
} *restored_custom_salt;

static struct fmt_tests ssh_tests[] = {
	{"$ssh2$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*1743", "kingdom"},
	{"$ssh2$2d2d2d2d2d424547494e204453412050524956415445204b45592d2d2d2d2d0a50726f632d547970653a20342c454e435259505445440a44454b2d496e666f3a204145532d3132382d4342432c35413830363832373943304634364539383230373135304133433245433631340a0a2f756954696e4a3452556a6f5a76302b705931694d763163695661724369347a2f62365a694c4161565970794a31685854327463692b593266334c61614578630a6f357772316141464d3437786d526d476f3832492f76434847413952786735776147433970574f475a5675555172447355367463556b434d422b325a344753390a354f44474364444b32674e6574446e62324a764873714154736d3443633633476468695a30734346594c71796d2b576531774359616c78734f3231572b4f676f0a42336f6746464977327232462b714a7a714d37415543794c466869357a476d7536534e6558765534477a784750464a4e47306d414f55497761614e3161446a630a4e326b3462437266796271337a366e436533444273384b3232694e2b3875526e534162434f717a5a5845645971555959354b6b6a326e654354525458494e64670a512b61535359673379355937626f4b6b6a494f727650555748654f796475512b74657273414577376e43564a7a72394e387452673271563450557631434b66700a4f49467742372f39736f6d6a59496a71576f61537a6a784b30633852777a305331706d722b7571726277792b50656f75354d3373656d486c426b4769553237660a776f684b792b4d554e4862734e6a7973535a53456c4e4b734d4950715449567a5a45316d5646412f30754d477164705133627a424f6a58325a6f36656446434f0a6d4a34775961765735774d2b6a6d75564b5056564e7939395a78796570304645644c50354b623263345a6c3053396631342f62366836415069785665377a75760a5662536b4279664a6e797a68494f5942497954374d64773134723441584a56362b5a6f457730397769774d3d0a2d2d2d2d2d454e44204453412050524956415445204b45592d2d2d2d2d0a*771", "12345"},
	/* following test vectors are from CMIYC 2012 */
	{"$ssh2$2d2d2d2d2d424547494e205253412050524956415445204b45592d2d2d2d2d0a50726f632d547970653a20342c454e435259505445440a44454b2d496e666f3a204145532d3132382d4342432c41304238464341423242363535424133443034423230323042383941313432450a0a704e75307853616d76714f757969626d324a384d476576667552564d354d3230752f773049502f4f574e317134516439757064546f4147794b77666b4a4975790a314b50457679793635442b4d4b6656635044624756717079597630765170715066377845504264576837494d654f302b514a6f442b33734b2b694c764c36316a0a3159444f414f4d5a5342635868366947486a545579576251785530706c595866645870326f6e6a436a6f6d626578622b644e4f4b32665969436936373072516e0a6f2f4b554e7636793441436a38624a714f5162726844464b6a34334345613672446c316364327455705a77324d4b6c74355162392f50656d6d527575547044760a4c322b5a31616b73654f336348337658577053385279777937344b31596d47496d6d44377275344c4646784b7034572f2b4c6855754d5966336a38426a6844700a744e35767631716e2f7a445a6861693432683556525a68567a5a59486251336c2f7a476c6b38702f394c72624234686f434d596b7a712b5656576b546a4666390a6b41614866596f585338347a5237636b6b4e555948594f694251446353656757445164575762566f676772436549705171364249696c6d4d625a43434543624a0a7069452f4a357548632b73385732436e506b675a6e74664c706d5756352f495a784e4431346a45686c686e2f76543137306472557261692f6a5a3362315447660a39486b69356f574a526e654e727735725235633974333957334d4e5761527a4d5a537a4e55396e356956794a625a6e5044456d4f576f31784c79364b465a71410a364b506d6934456d55504464734f457742446935465054436a534d735244646f764d7272494549535355654341444e44706339745256755637386c4d6a5a56550a54624d7342546e5134667743694c58732f4c784c7437596e696c514a4f6c6273437457536a42453671576f5254582f547273467a645a3963434238644369384a0a496b4d2f38586b526b4277504435514c487a526446683252374e524662707545574b464f443875764b3355434163454d2f7a79504f484145755a766c466652630a4143414f2f71546a62594e53536e594d4946474e6b4334343168496237796d306565322f524c354d55584d4d4f37794733577138514c5a30634f4572676a584c0a4578653232754e4e6d5353504f6f2b597033324567324441476d526664734f72772f5a71384e4877393775336554437a2b46517742692b78714378612f784e510a76626f456d6f7648764d4d2b5966302b68495335357455656f53457a653356544b457a5249696f6b61654848465938542f324d776454416b4f764a62537a62520a6d36426751684c7279304b30554d5231346a6a43756641684349735771732b32355754752b4768672f51642f6b4e354848384a6d495459474365556d524535310a5672754e626563444a78704c714e3743564442633052592f6b4b4d74695762316d5a4139344f53324c6a714e735664542b776d64466b326231457746385750390a6c75397a4f4378336b685449374a4e4a6a7939666f7870307257724445326f337876624365783358755477654b533875384e56705742394679776448376c6f760a79365634484e795775624c7765397576662f54675179687a427945532f5264354c4a46556735504e574b667269784a76462b345166355a77566e774b747561350a6f706c77564e6c6f2f724b6652352b34743469424f4a636146434b4c7252685532594250472b70334155742f4452413352454634316d3074735a746d4450586f0a52796179397a426d4b6833354f727855726c4964505261522b302b6b6977682b725a33387a30356a5a396d6958306967476161762b2b79457767726e735177710a46707a364236545362464131774330376651466f472f5672395769724833514d6875357651376930514f366838524f6169667062786c4d5752436146626a70650a4976396637515736656756506255594a3678517538524f69547757354b366373514d326233676e59456c31474c394d4c7178584c52784d506f776377736d774e0a4f5a30557563744337466266416b7535753830597130306e6e4d2b437373482b36503236336f7656426e6b616239555453624a4d3344627354596a6d33554f6f0a57354b6e46466951676e48536d476766564b7057636d4c386343594d7756312f56566d72312b667554537838636e7631413278476d326b4f7a7061556743514d0a2d2d2d2d2d454e44205253412050524956415445204b45592d2d2d2d2d0a*1766", "Olympics"},
	{"$ssh2$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*1766", "C0Ld.FUS10N"},
	{"$ssh2$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*1766", "extuitive"},
	{NULL}
};

struct fmt_main fmt_ssh;

static void init(struct fmt_main *self)
{
	/* OpenSSL init, cleanup part is left to OS */
	SSL_load_error_strings();
	SSL_library_init();
	OpenSSL_add_all_algorithms();

#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
	if (SSLeay() < 0x10000000) {
		fprintf(stderr, "Warning: compiled against OpenSSL 1.0+, "
		    "but running with an older version -\n"
		    "disabling OpenMP for SSH because of thread-safety issues "
		    "of older OpenSSL\n");
		fmt_ssh.params.min_keys_per_crypt =
		    fmt_ssh.params.max_keys_per_crypt = 1;
		fmt_ssh.params.flags &= ~FMT_OMP;
	}
	else {
		int omp_t = 1;
		omp_t = omp_get_max_threads();
		self->params.min_keys_per_crypt *= omp_t;
		omp_t *= OMP_SCALE;
		self->params.max_keys_per_crypt *= omp_t;
	}
#endif
	saved_key = mem_calloc_tiny(sizeof(*saved_key) *
			self->params.max_keys_per_crypt, MEM_ALIGN_WORD);
	any_cracked = 0;
	cracked_size = sizeof(*cracked) * self->params.max_keys_per_crypt;
	cracked = mem_calloc_tiny(cracked_size, MEM_ALIGN_WORD);
}

static int ishex(char *q)
{
	while (atoi16[ARCH_INDEX(*q)] != 0x7F)
		q++;
	return !*q;
}

static int valid(char *ciphertext, struct fmt_main *self)
{
	char *ctcopy;
	char *keeptr;
	char *p;
	int res;
	if (strncmp(ciphertext, "$ssh2$", 6))
		return 0;
	ctcopy = strdup(ciphertext);
	keeptr = ctcopy;
	ctcopy += 6;
	/* find length field */
	p = strrchr(ctcopy, '*');
	if (!p)
		goto err;
	p += 1;
	res = atoi(p);
	if ((p = strtok(ctcopy, "*")) == NULL)	/* data */
		goto err;
	if (!ishex(p))
		goto err;
	if(strlen(p) != res * 2)
		goto err;
	MEM_FREE(keeptr);
	return 1;

err:
	MEM_FREE(keeptr);
	return 0;
}

#define M_do_cipher(ctx, out, in, inl) ctx->cipher->do_cipher(ctx, out, in, inl)
int EVP_DecryptFinal_ex_safe(EVP_CIPHER_CTX *ctx, unsigned char *out, int *outl)
{
	int i,n;
        unsigned int b;
        *outl=0;


#ifndef EVP_CIPH_FLAG_CUSTOM_CIPHER
#define EVP_CIPH_FLAG_CUSTOM_CIPHER 0x100000
#endif
	if (ctx->cipher->flags & EVP_CIPH_FLAG_CUSTOM_CIPHER) {
		i = M_do_cipher(ctx, out, NULL, 0);
                if (i < 0)
                        return 0;
                else
                        *outl = i;
                return 1;
	}

        b=ctx->cipher->block_size;
#ifndef EVP_CIPH_NO_PADDING
#define EVP_CIPH_NO_PADDING 0x100
#endif
        if (ctx->flags & EVP_CIPH_NO_PADDING) {
		if(ctx->buf_len) {
			return 0;
		}
		*outl = 0;
		return 1;
	}
        if (b > 1) {
		if (ctx->buf_len || !ctx->final_used) {
			return(0);
		}
		OPENSSL_assert(b <= sizeof ctx->final);
                n=ctx->final[b-1];
                if (n == 0 || n > (int)b) {
			return(0);
		}
                for (i=0; i<n; i++) {
			if (ctx->final[--b] != n) {
				return(0);
			}
		}
                n=ctx->cipher->block_size-n;
		for (i=0; i<n; i++)
			out[i]=ctx->final[i];
                *outl=n;
	}
	else
		*outl=0;
	return(1);
}

int PEM_do_header_safe(EVP_CIPHER_INFO *cipher, unsigned char *data, long *plen,
             pem_password_cb *callback,void *u)
{
	int i,j,o,klen;
	long len;
        EVP_CIPHER_CTX ctx;
        unsigned char key[EVP_MAX_KEY_LENGTH];
        char buf[PEM_BUFSIZE];

        len= *plen;

        if (cipher->cipher == NULL) return(1);
        if (callback == NULL)
                klen=PEM_def_callback(buf,PEM_BUFSIZE,0,u);
        else
                klen=callback(buf,PEM_BUFSIZE,0,u);
        if (klen <= 0) {
		return(0);
	}

        EVP_BytesToKey(cipher->cipher,EVP_md5(),&(cipher->iv[0]),
                (unsigned char *)buf,klen,1,key,NULL);

        j=(int)len;
        EVP_CIPHER_CTX_init(&ctx);
        EVP_DecryptInit_ex(&ctx,cipher->cipher,NULL, key,&(cipher->iv[0]));
        EVP_DecryptUpdate(&ctx,data,&i,data,j);
        o=EVP_DecryptFinal_ex_safe(&ctx,&(data[i]),&j);
        EVP_CIPHER_CTX_cleanup(&ctx);
        OPENSSL_cleanse((char *)buf,sizeof(buf));
        OPENSSL_cleanse((char *)key,sizeof(key));
        j+=i;
        if (!o) {
		return(0);
	}
	*plen=j;
	return(1);
}


static void *get_salt(char *ciphertext)
{
	int i, filelength;
	char *decoded_data;
	char *copy = strdup(ciphertext);
	char *encoded_data = strtok(copy, "*");
	BIO *bp;
	char *nm = NULL, *header = NULL;
	unsigned char *data = NULL;
	EVP_CIPHER_INFO cipher;
	EVP_PKEY pk;
	long len;
	static struct custom_salt cs;

	if (!copy || !encoded_data) {
		fprintf(stderr, "BUG in parsing ciphertext, aborting!\n");
		exit(-1);
	}
	filelength = atoi(strtok(NULL, "*"));
	encoded_data += 6;	/* skip over "$ssh2$ marker */
	decoded_data = (char *) mem_alloc(filelength + 1);
	for (i = 0; i < filelength; i++)
		decoded_data[i] =
		    atoi16[ARCH_INDEX(encoded_data[i * 2])] * 16 +
		    atoi16[ARCH_INDEX(encoded_data[i * 2 + 1])];
	decoded_data[filelength] = 0;

	/* load decoded data into OpenSSL structures */
	bp = BIO_new(BIO_s_mem());
	if (!bp) {
		fprintf(stderr, "OpenSSL BIO allocation failure\n");
		exit(-2);
	}
	BIO_write(bp, decoded_data, filelength);

	/* PEM_bytes_read_bio function in crypto/pem/pem_lib.c
	 * check_pem function in crypto/pem/pem_lib.c */
	for (;;) {
		if (!PEM_read_bio(bp, &nm, &header, &data, &len)) {
			if (ERR_GET_REASON(ERR_peek_error()) ==
			    PEM_R_NO_START_LINE) {
				ERR_print_errors_fp(stderr);
				exit(-3);
			}
		}
		/* only PEM encoded DSA and RSA private keys are supported. */
		if (!strcmp(nm, PEM_STRING_DSA)) {
			pk.save_type = EVP_PKEY_DSA;
			pk.type = EVP_PKEY_type(EVP_PKEY_DSA);
			break;
		}
		if (!strcmp(nm, PEM_STRING_RSA)) {
			pk.save_type = EVP_PKEY_RSA;
			pk.type = EVP_PKEY_type(EVP_PKEY_RSA);
			break;
		}
		OPENSSL_free(nm);
		OPENSSL_free(header);
		OPENSSL_free(data);
		OPENSSL_free(bp);
	}
	if (!PEM_get_EVP_CIPHER_INFO(header, &cipher)) {
		ERR_print_errors_fp(stderr);
		exit(-4);
	}
#ifdef SSH_FMT_DEBUG
	printf("Header Information:\n%s\n", header);
#endif

	/* save custom_salt information */
	memset(&cs, 0, sizeof(cs));
	memcpy(&cs.cipher, &cipher, sizeof(cipher));
	memcpy(&cs.pk, &pk, sizeof(pk));
	memcpy(cs.data, data, len);
	cs.len = len;

	OPENSSL_free(nm);
	OPENSSL_free(header);
	OPENSSL_free(data);
	BIO_free(bp);
	MEM_FREE(copy);
	MEM_FREE(decoded_data);
	return (void *) &cs;
}

static void set_salt(void *salt)
{
	/* restore custom_salt back */
	restored_custom_salt = (struct custom_salt *) salt;
}

static void ssh_set_key(char *key, int index)
{
	int len = strlen(key);
	if (len > PLAINTEXT_LENGTH)
		len = PLAINTEXT_LENGTH;
	memcpy(saved_key[index], key, len);
	saved_key[index][len] = 0;
}

static char *get_key(int index)
{
	return saved_key[index];
}

static int crypt_all(int *pcount, struct db_salt *salt)
{
	int count = *pcount;
	int index = 0;

	if (any_cracked) {
		memset(cracked, 0, cracked_size);
		any_cracked = 0;
	}

#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
#pragma omp parallel for default(none) private(index) shared(count, any_cracked, cracked, saved_key, restored_custom_salt)
	for (index = 0; index < count; index++)
#endif
	{
		/* copy restored items into working copy */
		unsigned char working_data[4096];
		long working_len = restored_custom_salt->len;
		EVP_CIPHER_INFO cipher = restored_custom_salt->cipher;
		EVP_PKEY pk = restored_custom_salt->pk;
		const char unsigned *dc = working_data;
		DSA *dsapkc = NULL;
		RSA *rsapkc = NULL;

		memcpy(working_data, restored_custom_salt->data, working_len);
		if (PEM_do_header_safe(&cipher, working_data, &working_len, NULL,
			(char *) saved_key[index])) {
			if (pk.save_type == EVP_PKEY_DSA) {
				if ((dsapkc =
					d2i_DSAPrivateKey(NULL, &dc,
					    working_len)) != NULL) {
					DSA_free(dsapkc);
					{
						cracked[index] = 1;
#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
#pragma omp atomic
#endif
						any_cracked |= 1;
					}
				}
			} else if (pk.save_type == EVP_PKEY_RSA) {
				if ((rsapkc =
					d2i_RSAPrivateKey(NULL, &dc,
					    working_len)) != NULL) {
					RSA_free(rsapkc);
					{
						cracked[index] = 1;
#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
#pragma omp atomic
#endif
						any_cracked |= 1;
					}
				}
			}
		}
	}
	return count;
}

static int cmp_all(void *binary, int count)
{
	return any_cracked;
}

static int cmp_one(void *binary, int index)
{
	return cracked[index];
}

static int cmp_exact(char *source, int index)
{
	return cracked[index];
}

struct fmt_main fmt_ssh = {
	{
		FORMAT_LABEL,
		FORMAT_NAME,
		ALGORITHM_NAME,
		BENCHMARK_COMMENT,
		BENCHMARK_LENGTH,
		PLAINTEXT_LENGTH,
		BINARY_SIZE,
		BINARY_ALIGN,
		SALT_SIZE,
		SALT_ALIGN,
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
		FMT_OMP |
#endif
		FMT_CASE | FMT_8_BIT,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		ssh_tests
	}, {
		init,
		fmt_default_done,
		fmt_default_reset,
		fmt_default_prepare,
		valid,
		fmt_default_split,
		fmt_default_binary,
		get_salt,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		fmt_default_source,
		{
			fmt_default_binary_hash
		},
		fmt_default_salt_hash,
		set_salt,
		ssh_set_key,
		get_key,
		fmt_default_clear_keys,
		crypt_all,
		{
			fmt_default_get_hash
		},
		cmp_all,
		cmp_one,
		cmp_exact
	}
};

#endif /* plugin stanza */
