/* pfx cracker patch for JtR. Hacked together during  June of 2012 by
 * Dhiru Kholia <dhiru.kholia at gmail.com>.
 *
 * This software is Copyright (c) 2021, Dhiru Kholia <dhiru.kholia at gmail.com>,
 * and it is hereby released to the general public under the following terms:
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted.
 *
 * Generating pfx files:
 *
 * keytool -genkeypair -alias my_certificate -keystore my_keystore.pfx -storepass
 * my_password -validity 365 -keyalg RSA -keysize 2048 -storetype pkcs12 */

#if FMT_EXTERNS_H
extern struct fmt_main fmt_pfx;
#elif FMT_REGISTERS_H
john_register_one(&fmt_pfx);
#else

#include <openssl/opensslv.h>
#include <openssl/crypto.h>
#include <openssl/ssl.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/crypto.h>
#include <openssl/pkcs12.h>
#include <openssl/ssl.h>

#undef MEM_FREE
#include "options.h"
#ifdef _OPENMP
#include <omp.h>
#define OMP_SCALE               2 // tuned on core i7
#endif
#include <string.h>
#include "arch.h"
#include "common.h"
#include "formats.h"
#include "params.h"
#include "misc.h"
#include "memory.h"
#include "memdbg.h"

#define FORMAT_LABEL        "PFX"
#define FORMAT_NAME         "PKCS12 (.pfx, .p12)"
#define ALGORITHM_NAME      "32/" ARCH_BITS_STR
#define BENCHMARK_COMMENT   ""
#define BENCHMARK_LENGTH    -1001
#define PLAINTEXT_LENGTH    32
#define BINARY_SIZE         0
#define SALT_SIZE           sizeof(struct custom_salt)
#define BINARY_ALIGN	1
#define SALT_ALIGN	sizeof(int)
#define MIN_KEYS_PER_CRYPT  1
#define MAX_KEYS_PER_CRYPT  1

static char (*saved_key)[PLAINTEXT_LENGTH + 1];
static int any_cracked, *cracked;
static size_t cracked_size;

static struct custom_salt {
	int len;
	PKCS12 pfx;
} *cur_salt;

static struct fmt_tests pfx_tests[] = {
	{"$pfx$*2136*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", "usr0052"},
	{"$pfx$*2604*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", "my_password"},
	{"$pfx$*1702*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", "openwall"},
	{NULL}
};

struct fmt_main fmt_pfx;

static void init(struct fmt_main *self)
{
	/* OpenSSL init, cleanup part is left to OS */
	SSL_load_error_strings();
	SSL_library_init();
	OpenSSL_add_all_algorithms();

#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
	if (SSLeay() < 0x10000000) {
		fprintf(stderr, "Warning: compiled against OpenSSL 1.0+, "
		    "but running with an older version -\n"
		    "disabling OpenMP for pfx because of thread-safety issues "
		    "of older OpenSSL\n");
		fmt_pfx.params.min_keys_per_crypt =
		    fmt_pfx.params.max_keys_per_crypt = 1;
		fmt_pfx.params.flags &= ~FMT_OMP;
	}
	else {
		int omp_t = 1;
		omp_t = omp_get_max_threads();
		self->params.min_keys_per_crypt *= omp_t;
		omp_t *= OMP_SCALE;
		self->params.max_keys_per_crypt *= omp_t;
	}
#endif
	saved_key = mem_calloc_tiny(sizeof(*saved_key) *
			self->params.max_keys_per_crypt, MEM_ALIGN_WORD);
	any_cracked = 0;
	cracked_size = sizeof(*cracked) * self->params.max_keys_per_crypt;
	cracked = mem_calloc_tiny(cracked_size, MEM_ALIGN_WORD);
}

static int ishex(char *q)
{
	while (atoi16[ARCH_INDEX(*q)] != 0x7F)
		q++;
	return !*q;
}

static int isdec(char *q)
{
	char buf[24];
	int x = atoi(q);
	sprintf(buf, "%d", x);
	return !strcmp(q,buf) && *q != '-';
}

static int valid(char *ciphertext, struct fmt_main *self)
{
	char *ctcopy, *p, *keeptr, *decoded = NULL;
	PKCS12 *p12 = NULL;
	BIO *bp = NULL;
	int len, i;
	if (strncmp(ciphertext, "$pfx$", 5))
		return 0;
	ctcopy = strdup(ciphertext);
	keeptr = ctcopy;
	ctcopy += 6;
	if ((p = strtok(ctcopy, "*")) == NULL)	/* length */
		goto err;
	len = atoi(p);
	if (!isdec(p))
		goto err;
	if ((p = strtok(NULL, "*")) == NULL)	/* data */
		goto err;
	if (!ishex(p))
		goto err;
	if(strlen(p) != len * 2)
		goto err;
	decoded = (char *) mem_alloc(len + 1);
	for (i = 0; i < len; i++)
		decoded[i] = atoi16[ARCH_INDEX(p[i * 2])] * 16 +
			atoi16[ARCH_INDEX(p[i * 2 + 1])];
	decoded[len] = 0;
	bp = BIO_new(BIO_s_mem());
	if (!bp)
		goto err;
	BIO_write(bp, decoded, len);
	if(!(p12 = d2i_PKCS12_bio(bp, NULL)))
		goto err;

	if (bp)	BIO_free(bp);
	MEM_FREE(decoded);
	MEM_FREE(keeptr);
	return 1;

err:
	if (bp)	BIO_free(bp);
	MEM_FREE(decoded);
	MEM_FREE(keeptr);
	return 0;
}

static void *get_salt(char *ciphertext)
{
	char *decoded_data;
	int i;
	char *ctcopy = strdup(ciphertext);
	char *keeptr = ctcopy;
	char *p;
	static struct custom_salt cs;
	PKCS12 *p12 = NULL;
	BIO *bp;
	ctcopy += 6;	/* skip over "$pfx$*" */
	p = strtok(ctcopy, "*");
	cs.len = atoi(p);
	decoded_data = (char *) mem_alloc(cs.len + 1);
	p = strtok(NULL, "*");
	for (i = 0; i < cs.len; i++)
		decoded_data[i] = atoi16[ARCH_INDEX(p[i * 2])] * 16 +
			atoi16[ARCH_INDEX(p[i * 2 + 1])];
	decoded_data[cs.len] = 0;
	/* load decoded data into OpenSSL structures */
	bp = BIO_new(BIO_s_mem());
	if (!bp) {
		fprintf(stderr, "OpenSSL BIO allocation failure\n");
		exit(-2);
	}
	BIO_write(bp, decoded_data, cs.len);
	if(!(p12 = d2i_PKCS12_bio(bp, NULL))) {
		perror("Unable to create PKCS12 object from bio\n");
		exit(-3);
	}
	/* save custom_salt information */
	memset(&cs, 0, sizeof(cs));
	memcpy(&cs.pfx, p12, sizeof(PKCS12));
	BIO_free(bp);
	MEM_FREE(decoded_data);
	MEM_FREE(keeptr);
	return (void *) &cs;
}

static void set_salt(void *salt)
{
	cur_salt = (struct custom_salt *) salt;
}

static void pfx_set_key(char *key, int index)
{
	int len = strlen(key);
	if (len > PLAINTEXT_LENGTH)
		len = PLAINTEXT_LENGTH;
	memcpy(saved_key[index], key, len);
	saved_key[index][len] = 0;
}

static char *get_key(int index)
{
	return saved_key[index];
}

static int crypt_all(int *pcount, struct db_salt *salt)
{
	int count = *pcount;
	int index = 0;

	if (any_cracked) {
		memset(cracked, 0, cracked_size);
		any_cracked = 0;
	}

#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
#pragma omp parallel for
	for (index = 0; index < count; index++)
#endif
	{
		if(PKCS12_verify_mac(&cur_salt->pfx, saved_key[index], -1)) {
			cracked[index] = 1;
#ifdef _OPENMP
#pragma omp atomic
#endif
			any_cracked |= 1;
		}
	}
	return count;
}

static int cmp_all(void *binary, int count)
{
	return any_cracked;
}

static int cmp_one(void *binary, int index)
{
	return cracked[index];
}

static int cmp_exact(char *source, int index)
{
	return cracked[index];
}

struct fmt_main fmt_pfx = {
	{
		FORMAT_LABEL,
		FORMAT_NAME,
		ALGORITHM_NAME,
		BENCHMARK_COMMENT,
		BENCHMARK_LENGTH,
		PLAINTEXT_LENGTH,
		BINARY_SIZE,
		BINARY_ALIGN,
		SALT_SIZE,
		SALT_ALIGN,
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
#if defined(_OPENMP) && OPENSSL_VERSION_NUMBER >= 0x10000000
		FMT_OMP |
#endif
		FMT_CASE | FMT_8_BIT,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		pfx_tests
	}, {
		init,
		fmt_default_done,
		fmt_default_reset,
		fmt_default_prepare,
		valid,
		fmt_default_split,
		fmt_default_binary,
		get_salt,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		fmt_default_source,
		{
			fmt_default_binary_hash
		},
		fmt_default_salt_hash,
		set_salt,
		pfx_set_key,
		get_key,
		fmt_default_clear_keys,
		crypt_all,
		{
			fmt_default_get_hash
		},
		cmp_all,
		cmp_one,
		cmp_exact
	}
};

#endif /* plugin stanza */
