
open Syntax
open Prettyutil

type 'a strhash = 'a Data.StringCols.Hash.t


(* ------------------------------------------------------------
 * The Typechecker environment
 * ------------------------------------------------------------ *)

type env

val new_env : unit -> env


(* ------------------------------------------------------------
 * Error handling
 * ------------------------------------------------------------ *)

val tyerr : env -> pretty -> 'a
val tywrong : env -> typ -> pretty -> 'a
val tywarn : env -> pretty -> unit
val error_mismatch : env -> pretty -> 'a


(* ------------------------------------------------------------
 * Pretty print errors
 * ------------------------------------------------------------ *)

val pprint_type_mismatch : pretty -> typ -> typ -> unit -> pretty
val pprint_funsig_mismatch : declaration -> declaration -> unit -> pretty

(* ------------------------------------------------------------
 * What is our current context
 * ------------------------------------------------------------ *)

val localenv : env -> env
val withlambdavars : env -> typ strhash -> env
val withmeta : env -> metadata -> env
val withstmt : env -> statement -> env
val withdecl : env -> declaration -> env
val with_tinit : env -> env
val withunsafe : env -> env
val with_expect_typ : env -> typ -> env
val with_unknown_typ : env -> env
val withtemps : env -> env
val withfwddecls : env -> env
val withlambdas : env -> id -> env
val with_repcheck : env -> env
val with_unify : env -> (unit -> pretty) -> env
val with_scrutinee : env -> (expression * typ) option -> env
val withdictenv : env -> dictenv -> env


(* ------------------------------------------------------------
 * Imperative modifications to the current environment
 * ------------------------------------------------------------ *)

val set_repchanged : env -> unit
val set_rettype : env -> typ option -> unit
val set_conttype : env -> typ option -> unit


(* ------------------------------------------------------------
 * Get the current state
 * ------------------------------------------------------------ *)

val get_repchanged : env -> bool
val get_meta : env -> metadata
val get_funlambdas : env -> lamdef list
val get_conttype : env -> typ option
val get_rettype : env -> typ option
val get_expectty : env -> typ option
val get_scrutinee : env -> expression * typ
val get_block_fwds : env -> (namevar * typ) list
val get_block_temps : env -> (namevar * typ) list
val get_block_lamenvs : env -> localfuninfo list
val get_block_dictenvs : env -> (string * id * id * dictenv) list
val get_stmt_temps : env -> tinitialiser list
val get_expect_typ : env -> typ
val get_unsafe_allowed : env -> bool


(* ------------------------------------------------------------
 * Add things
 * ------------------------------------------------------------ *)

val add_structdef : env -> structkind -> id -> structdetails -> unit
val add_enumdef : env -> id -> enumdetails -> unit
val add_constrdef : env -> id -> (id list * typ * id) -> unit
val add_localvar : env -> id -> typ -> unit
val add_typedef : env -> id -> typ -> unit
val add_localdecl : env -> id -> declaration -> unit
val add_localty : env -> id -> id -> unit 
val add_dictproto : env -> id -> dictdecl -> unit 
val add_methoddef : env -> id -> id -> unit
val add_dict_tyvar : env -> id -> unit
val add_ifacedef : env -> id -> ifacedef -> unit
val add_stdtype : env -> id -> typ -> unit


(* ------------------------------------------------------------
 * Find named things in scope
 * ------------------------------------------------------------ *)

val find_struct : env -> id -> structdetails
val find_iface : env -> id -> ifacedef
val find_typaram_ifaces : env -> id -> ifacespec list
val find_dict : env -> id -> id -> dictdecl
val find_method_iface : env -> id -> id
val find_id_decl : env -> id -> declaration 
val find_id_type : env -> id -> typ
val find_constr : env -> id -> (id list * typ * id)

val is_dict_tyvar : env -> id -> bool
val is_global : env -> id -> bool
val is_method : env -> id -> bool

(* ------------------------------------------------------------
 * Create temporaries
 * ------------------------------------------------------------ *)

val new_lambda : env -> typ -> declaration list -> block -> typ strhash -> unit
val new_dictenv : env -> id -> id -> dictenv -> dictkind
val new_temp : env -> typ -> tinitialiser -> unit
val new_fwddecl : env -> id -> typ -> unit 


(* ------------------------------------------------------------
 * Utils
 * ------------------------------------------------------------ *)

val resolve_typ : env -> typ -> typ
val tc_iter2 : env -> string -> ('a -> 'b -> unit) -> 'a list -> 'b list -> unit

