open Util
open Metadata
open Syntax
open Syntaxutil
open Prettyutil
open Prettyprint
open Names
open Typecheckenv
open Typecompare

module SH = Data.StringCols.Hash


(* ------------------------------------------------------------
 * Util functions
 * ------------------------------------------------------------ *)

let argdetails_typs env argdetails = match argdetails with
	| ArgsFull (m,tyifaces,decls) -> 
			tyifaces,List.map simpledecl_typ decls
	| _ -> tyerr env (str "Function is not completely specified") 

let check_single env kind id l = match l with
	| [x] -> x
	| [] -> tyerr env (str "No"<++> str kind <++> 
			str "are called"<++>pp_id id)
	| _ -> tyerr env (str "Multiple"<++>str kind<++>
			str "are called"<++>pp_id id)

let typ_name env typ = 
	try typ_name_nocheck typ 
	with _ -> tywrong env typ (str "cannot implement interfaces")


(* ------------------------------------------------------------
 * Get Info about a Type or Declaration
 * ------------------------------------------------------------ *)

let rec fields_for_init_decl (specs,coretyp) ((declmods,id_opt),init) = 
	match id_opt with
	| Some id -> Some ((specs,coretyp,declmods),id)
	| None -> None

and decl_fields (m,(basetyp,init_decls)) =
	option_map (fields_for_init_decl basetyp) init_decls 

and structdetails_fields env tyargs (m,typarams,decls) = 
	let subst = mksubst env typarams tyargs in
	let decls = List.map (subst_decl subst) decls in
	concat_map decl_fields decls

and struct_fields env typ = match resolve_typ env typ with
	| _,TyStruct (SKStruct,Some id,None),[DWithArgs tyargs] ->
			structdetails_fields env tyargs (find_struct env id)
	| _,TyStruct (SKStruct,Some id,None),[] ->
			structdetails_fields env [] (find_struct env id)
	| _,TyStruct (SKStruct,_,Some details),[] ->
			structdetails_fields env [] details
	| _ -> tywrong env typ (str "is not a struct")	


(* ------------------------------------------------------------
 * Type Check a Declaration
 * ------------------------------------------------------------ *)



(* TODO: check that the combination of specifiers/qualifiers is legal *)
and check_specs env specs = 
	if List.mem STaggedOnly specs then 
		tyerr env (str "_void cannot appear here")

and add_tag_decl env tyvars id decl = 
	let name = simpledecl_name decl in
	let typ = simpledecl_typ decl in
	add_constrdef env name (tyvars,typ,id)
	
and check_coretyp env coretyp = match coretyp with
	| TyStruct (SKTagged,Some id,Some (m,tyvars,decls as structdetails)) ->
			List.iter (add_tag_decl env tyvars id) decls;
			add_structdef env SKTagged id structdetails 
	| TyStruct (structkind,Some id,Some structdetails) ->
			add_structdef env structkind id structdetails 
	| TyEnum (Some id,Some enumdetails) ->
			let itemty = ([SCoreHere],TyEnum (Some id,None),[]) in
			List.iter (check_enumitem env itemty) enumdetails;
			add_enumdef env id enumdetails 
	| TyEnum (None,Some enumdetails) ->
			let itemty = ([SInt;SCoreHere],TyBasic,[]) in
			List.iter (check_enumitem env itemty) enumdetails
	| _ -> ()
	
	
(* TODO: make the tag be of enum type, and make casts from int to enum
 * unsafe *)	
and check_enumitem env itemty (tagid,exp_opt) =
	add_localvar env tagid itemty;
	match exp_opt with
	| Some e -> ignore (check_expr env e)
	| None -> () 

and check_vardecl env typ id_opt = match id_opt with
	| Some id -> add_localvar env id typ
	| None -> ()
	
and check_typedef env typ id_opt init = match id_opt, init with
	| Some id,None -> add_typedef env id typ
	| _ -> tyerr env (str "Malformed typedef")

and check_initialiser env typ init = match init with
	| IExp exp -> asserttyp_expr env typ exp
	| IFields (inits,_) ->
		let fields = struct_fields env typ in
		let fieldtyps,fieldids = List.split fields in
		tc_iter2 env "field initialisers" (check_initialiser env) fieldtyps inits
		
and check_init_declarator env (specs,coretyp) ((declmods,id_opt),init) =
	if List.mem STypeDef specs then
		let specs = List.filter (fun x -> x <> STypeDef) specs in
		let typ = specs,coretyp,declmods in
		check_typedef env typ id_opt init
	else begin
		let typ = specs,coretyp,declmods in
		option_iter (check_initialiser env typ) init;
		check_vardecl env typ id_opt		
	end

and check_decl env (m,((specs,coretyp as basetyp),init_declarators) as decl) =
	let env = withdecl env decl in
	check_specs env specs;
 	check_coretyp env coretyp;
	let names = decl_names decl in
	List.iter (fun id -> add_localdecl env id decl) names;
	List.iter (check_init_declarator env basetyp) init_declarators


(* ------------------------------------------------------------
 * Type Check a Function Definition
 * ------------------------------------------------------------ *)
	
and check_fundef_generic env m (tyifaces : tyiface list) argdecls block =
	let env = localenv env in
	List.iter (check_decl env) argdecls;
	List.iter (fun (id,iface) -> add_localty env id iface) tyifaces;	
	check_block env m block

and select_argdecls env argdecls krdecls = match argdecls, krdecls with
	| ArgsFull (_,tyifaces,argdecls), [] -> tyifaces,argdecls
	| ArgsNamed names, krdecls 
		when same_elements names (concat_map decl_names krdecls) ->
			[], krdecls
	| ArgsNamed names,_ ->
		tyerr env (str "K&R Decls do not match argument names")
	| ArgsNoinfo,_ ->
		[],[]
	| _, _ -> 
		tyerr env (str "Argument types must be provided in function decl")
	
and check_fundef env m (decl,krdecls,block : fundef) =
	let id,ret_ty,args,va = split_fundecl decl in
	let env = withlambdas env id in
	let tyifaces,argdecls = select_argdecls env args krdecls in
	set_rettype env (Some ret_ty);
	check_fundef_generic env m tyifaces argdecls block;
	set_lambdas (get_meta env) (get_funlambdas env) 
	
and check_lambda env m argdecls block = 
	let envvars = SH.create () in
	let env = withlambdavars env envvars in
	set_conttype env None;  
	check_fundef_generic env m [] argdecls block;
	let spec,core,mods = match get_conttype env with
		| None -> tyvoid
		| Some ty -> ty in
	new_lambda env (spec,core,mods) argdecls block envvars;
	spec,core,
		DFun(ArgsFull (cpm m,[],argdecls),Closure)
		::mods
		

(* ------------------------------------------------------------
 * Type Check a Dictionary Definition
 * ------------------------------------------------------------ *)
(* TODO: check prototype matches new declaration *)

and dictenv_from_dictparams dictparams = 
	List.map (fun (id,iface as typaram) -> typaram,(iface,id,DictEnv)) dictparams
	
and add_dictenv_prefix env args tyname ifaceid = match args with
	| ArgsFull (m,_,_) ->
		set_envarg m (EnvDict (ifaceid,tyname))
	| _ -> tyerr env (str "Method args must be fully specified")

and check_dictfun env typ ifaceid tyname tyiface (m,(decl,krdecls,block)) =
	let env = withmeta env m in
	let decl = as_dictmethod ifaceid tyname tyiface decl in
	let methid,_,args,_ = split_fundecl decl in
	check_fundef env m (decl,krdecls,block);
	add_dictenv_prefix env args tyname ifaceid;
	let right_funsig = find_specialised_method env methid typ in
	same_funsig env decl right_funsig

and check_dict_protos env ifaceid typ tyname tyiface =
	let thisvar,methods = find_iface env ifaceid in
	let subst = mksubst env [thisvar] [typ] in
	let protos = List.map (subst_funsig subst) methods in
	let protos = List.map (as_dictmethod ifaceid tyname tyiface) protos in
	set_dict_protos (get_meta env) protos

and check_dictdef env (typ,ifaceid,body) = 
	let env = localenv env in
	let tyname = typ_name env typ in
	match body with
		| DictProto tyiface ->
			add_dictproto env tyname (typ,ifaceid,tyiface);
			check_dict_protos env ifaceid typ tyname tyiface
		| DictImpl (tyiface,methods) ->
			let env = withdictenv env (dictenv_from_dictparams tyiface) in
			List.iter (fun (id,iface) -> add_localty env id iface) tyiface;
			List.iter (fun (id,iface) -> add_dict_tyvar env id) tyiface;
			List.iter (check_dictfun env typ ifaceid tyname tyiface) methods;
			set_iface_protos (get_meta env) (snd (find_iface env ifaceid))


(* ------------------------------------------------------------
 * Find a dictionary to implement in interface 
 * ------------------------------------------------------------ *)

and ptr_typ (specs,core,mods) = (specs,core,DPtr []::mods)

and deref_typ env (specs,core,mods as typ) = match mods with
	| DPtr _::xs -> (specs,core,xs)
	| DArray _::xs -> (specs,core,xs)
	| DFatPtr _::xs -> (specs,core,xs)
	| _ -> tywrong env typ (str "is not of pointer or array type")

and check_typaram_iface env id ifaceid =
	let ifacespecs = find_typaram_ifaces env id in
	match List.filter (fun id -> id_equal id ifaceid) ifacespecs with
	| [x] -> ()
	| [] -> tyerr env (str "Type param"<++>pp_id id<++>
				str "does not implement" <++>pp_id ifaceid)
	| _ ->  tyerr env (str "Type param"<++>pp_id id<++>
				str "multiply implements" <++>pp_id ifaceid)				

(* how does this type implement the given interface *)
and impl_for_ty_iface env typ ifaceid = match resolve_typ env typ with
	| _,TyWild id,[] when is_dict_tyvar env id -> 
		check_typaram_iface env id ifaceid;
		ifaceid,id,DictEnv
	| _,TyWild id,[] ->
		check_typaram_iface env id ifaceid;
		ifaceid,id,DictArg
	| typ ->
		let tyname = typ_name env typ in
		let dict = find_dict env tyname ifaceid in
		impl_for_dict env tyname typ dict

(* what ifacespec does this interface implement *)
and impl_spec (spec,ty,kind) = spec
		
(* check that a typaram can be met, and fill in any free vars in it *)
and process_typaram env subst (paramid,ifaceid) =
	if is_bound_var subst paramid then
		let ty = subst_find subst paramid in
		impl_for_ty_iface env ty ifaceid
	else
		tyerr env (str "Cannot resolve type param inferface:" <++>
			pprint_id ifaceid <++> pprint_id paramid)
			
(* how does this dict implement this ifacespec *)
and impl_for_dict env id ty (dict_ty,ifacespec,requirements) =
	let subst = new_subst (List.map tyiface_id requirements) in
	fill_in_generic_dict_type subst env ifacespec dict_ty ty;
	let impls = List.map (process_typaram env subst) requirements in
	check_subst_complete env subst;
	let dictenv = new_dictenv env id ifacespec 
			(List.combine requirements impls) in
	ifacespec, id, dictenv
 
and funsig_has_name id decl = 
	let funid,_,_,_ = split_fundecl decl in
	id_equal id funid
 	
and find_method env methodid = 
	let iface = find_method_iface env methodid in
	let thisvar,funsigs = find_iface env iface in
	let sigswithname = List.filter (funsig_has_name methodid) funsigs in
	let funsig = check_single env "methods" methodid sigswithname in
	funsig,iface,thisvar	
	
and find_specialised_method env methodid typ =
	let funsig,iface,thisvar = find_method env methodid in
	let subst = mksubst env [thisvar] [typ] in
	subst_funsig subst funsig	
	

(* ------------------------------------------------------------
 * Type Check a Function Call
 * ------------------------------------------------------------ *)

(* TODO: deal with basic ops more properly *)
and check_builtinfun_call env f argtys = 
	match List.map (resolve_typ env) argtys with
	| [(s,TyBasic,[]);(_,TyBasic,[])] 
				when member f builtin_twoint_ops -> Some (s,TyBasic,[])
	| [(s,c,(DFatPtr s2::ds));(_,TyBasic,[])] 
				when member f builtin_twoint_ops -> 
					Some (s,c,DFatPtr s2::ds)			
	| [(_,TyBasic,[]);(_,TyBasic,[])] 
				when member f builtin_intcmp_ops -> Some ([SInt],TyBasic,[])
	| [(_,_,(DPtr _|DFatPtr _)::_);(_,_,(DPtr _|DFatPtr _)::_)] 
				when member f builtin_intcmp_ops -> Some ([SInt],TyBasic,[])
	| [s,TyBasic,[]] 
				when member f builtin_oneint_ops -> Some (s,TyBasic,[])
	| [s,TyBasic,[]] 
				when member f builtin_postfix_ops -> Some (s,TyBasic,[])
	| [(s,TyBasic,[]);(_,TyBasic,[])] 
				when member f builtin_twobool_ops -> Some (s,TyBasic,[])
	| [s,TyBasic,[]] 
				when member f builtin_onebool_ops -> Some (s,TyBasic,[])
	| [_;_] when member f builtin_ptrcmp_ops -> Some ([SInt],TyBasic,[])
	| [s,core,DPtr _::mods] 
				when f = "*" -> Some (s,core,mods)
	| [s,core,mods] 
				when f = "&" -> Some (s,core,DPtr []::mods)

	| [ty1;ty2] when member f builtin_ops ->
				tyerr env (str "Operator"<++>str f<++>
				str "cannot be applied to arguments of types"<++>pp_ty ty1<++>
				str "and"<++>pp_ty ty2)
	| [ty] when member f builtin_ops ->
				tyerr env (str "Operator"<++>str f<++>
				str "cannot be applied to an argument of type"<++>pp_ty ty) 
	| _ -> None

and check_generic_call env argtys retty freevars typarams argenvs paramtys =
	let subst = new_subst freevars in
	if List.length paramtys <> List.length argtys then
		tyerr env (str "Wrong number of arguments passed");
	list_iter3 (fill_in_generic_arg_types subst) argenvs paramtys argtys;
	let impls = List.map (process_typaram env subst) typarams in
	set_dict_args (get_meta env) impls;
	subst_typ subst retty, subst

and check_funsig_call env argenvs argtys decl =
	let id,ret_ty,argdecls,funking = split_fundecl decl in
	let tyifaces,paramtys = argdetails_typs env argdecls in
	let freevars = decl_freevars decl in
	check_generic_call env argtys ret_ty freevars tyifaces argenvs paramtys
		
and check_topfun_call env f argenvs argtys = 
	match check_builtinfun_call env (id_str f) argtys with
	| None ->
		let retty,subst = check_funsig_call 
			env argenvs argtys (find_id_decl env f) in
		check_subst_complete env subst;
		retty
	| Some ty -> set_dict_args (get_meta env) []; ty
		
and check_expfun_call env exp argenvs argtys = 
	let funty = check_expr env exp in
	match funty with
	| (spec,base,DFun(args,kind)::mods) ->
		let tyifaces,paramtys = argdetails_typs env args in
		let retty,subst = (check_generic_call env argtys
			(spec,base,mods) [] tyifaces argenvs paramtys) in
		check_subst_complete env subst;
		retty
	| _ -> tywrong env funty (str "is not a function type")

and check_methodcall env methodid argenvs argtys = 
	let funsig,iface,thisvar = find_method env methodid in
	let ret_ty, subst = check_funsig_call env argenvs argtys funsig in
	infer_return_typ env subst ret_ty thisvar;
	check_subst_complete env subst;
	let implty = subst_find subst thisvar in
	let impl = impl_for_ty_iface env implty iface in
	set_calldict (get_meta env) impl (id_str methodid);
	ret_ty
	
and infer_return_typ env subst retty id = match get_expectty env with
	| _ when is_bound_var subst id -> ()
	| Some typ -> unify_typs env subst retty typ 
	| None -> tyerr env (str "Ambiguous return type")	
	

(* TODO: allow the user to explicitly specify the return type *)
and check_funcall env exp args =
	let argtys = List.map (check_expr env) args in
	set_argtys (get_meta env) argtys;
	let argenvs = List.map (fun a -> withmeta env (fst a)) args in
	match exp with
	| (m,Var f) when is_global env f -> 
			check_topfun_call env f argenvs argtys
	| (m,Var f) when is_method env f ->
			check_methodcall env f argenvs argtys
	| (m,Var f) when is_closure_typ (find_id_type env f) ->
			set_funenv (get_meta env) ("_fe_" ^ id_str f);
			check_expfun_call env exp argenvs argtys
	| _,_ -> check_expfun_call env exp argenvs argtys

	
(* ------------------------------------------------------------
 * Check Field Lookup Expressions
 * ------------------------------------------------------------ *)

and check_ty_field env typ fldid = 
	let fields = struct_fields env (resolve_typ env typ) in
	let matched = List.filter (fun (_,id) -> id_equal id fldid) fields in
	let typ,id = check_single env "fields" fldid matched in
	typ

and check_field env deref exp fieldid =	
	let ty = check_expr env exp in
	let ty = if deref then deref_typ env ty else ty in
	check_ty_field env ty fieldid


(* ------------------------------------------------------------
 * Simple Expressions
 * ------------------------------------------------------------ *)
(* TODO: check cast is safe *)

and check_const env const = match const with
	| ConstInt _ -> [SInt],TyBasic,[]
	| ConstString _ -> [SChar],TyBasic,[DPtr[]]
	| ConstFloat _ -> [SDouble],TyBasic,[]
	| ConstSizeTy _ | ConstSizeExp _ | 
		ConstAlignExp _ | ConstAlignTy _ -> [SInt],TyBasic,[]
	

and check_cast env cast_typ exp =
	let env = with_expect_typ env cast_typ in
	ignore (check_expr env exp);
	cast_typ

and check_expr_opt env e_opt = match e_opt with
	| None -> tyvoid
	| Some e -> check_expr env e

and check_return env e_opt tyo setter = match tyo with
	| None -> setter (Some (check_expr_opt (with_unknown_typ env) e_opt))
	| Some ty -> option_iter (asserttyp_expr env ty) e_opt

and check_cond env cond iftrue iffalse =
	asserttyp_expr env tybool cond;
	let t_typ = check_expr env iftrue in
	let f_typ = check_expr env iffalse in
	same_type env t_typ f_typ;
	t_typ
	
and check_index env arr idx = 
	asserttyp_expr env tyint idx;
	let typ = check_expr env arr in
	deref_typ env typ
	

(* ------------------------------------------------------------
 * Assignment
 * ------------------------------------------------------------ *)

and check_exp_is_lval env (m,expr) = match expr with
	| Field _ | Var _ | Index _ -> ()
	| Parens inner -> check_exp_is_lval env inner
	| Cast (_,inner) -> check_exp_is_lval env inner
	| Unsafe inner -> check_exp_is_lval env inner
	| Choice (cond,iftrue,iffalse) -> 
			check_exp_is_lval env iftrue;
			check_exp_is_lval env iffalse
	| FunCall ((m,Var i),exp) when id_str i = "*" -> ()
	| LocalFun _ | Const _ | FunCall _ | Assign _ | Init _ 
	| EBlock _ | JklNonDet _ -> 
			tyerr env (str "Invalid lvalue")

and check_assign_op_type env op ltyp = 
	match op,resolve_typ env ltyp with
	| "=",_ -> ()
	| ("*="|"+="|"-="|"/="|"<<="|">>="|"^="),(specs,TyBasic,[]) -> ()
	| _ -> tyerr env (str "assignment operator" <++> 
					str op <++> str "cannot be used with type" <++>
					pp_ty ltyp)

(* TODO: check that lhs isn't const *)
and check_assign env lhs op rhs =
	let ltyp = check_expr env lhs in
	asserttyp_expr env ltyp rhs;
	check_exp_is_lval env lhs;
	check_assign_op_type env op ltyp;
	ltyp
	

(* ------------------------------------------------------------
 * Type Check an extended Initialiser
 * ------------------------------------------------------------ *)

and check_tinit env (m,tinit) = try
	let typ = get_expect_typ env in
	new_temp env typ (m,tinit);
	let env = with_tinit env in
	(match tinit with
	| TConApp (tagid,body_opt) ->
		let bodyty = constr_typ env typ tagid in
		option_iter (check_tinit_expr env bodyty) body_opt
	| TStruct mems ->
		let fields = struct_fields env typ in
		let fieldtyps,fieldids = List.split fields in
		let memes = List.map snd mems in
		tc_iter2 env "field initialisers" 
			(check_tinit_expr env) fieldtyps memes;
		List.iter2 
			(fun id (m,_) -> set_fieldname m (id_str id)) fieldids memes
	| TAlloc (kind,body) ->
		let typ = deref_typ env typ in
		set_type m typ;
		check_tinit_expr env typ body);
	typ
	with
	  | _ -> tyerr env (str "Type error in initialiser")	


and check_tinit_expr env typ (m,expr) = 
	let env = withmeta env m in
	let env = with_expect_typ env typ in
	let env = with_tinit env in
	match expr with
	| Init tinit -> ignore (check_tinit env tinit)
	| _ -> asserttyp_expr env typ (m,expr)		
	
and asserttyp_expr env expect_typ (m,expr) = 
	let env = withmeta env m in
	let env = with_expect_typ env expect_typ in
	let typ = check_expr env (m,expr) in
	same_type env expect_typ typ		 

and assert_unsafe env = if not (get_unsafe_allowed env) then
	tywarn env (str "unsafe operation")
	

(* ------------------------------------------------------------
 * Type Check an Expression
 * ------------------------------------------------------------ *)

(* TODO: check that the expression really is const *)
and check_constexpr env exp = check_expr env exp

and check_expr env (m,expr) = try
	let env = withmeta env m in
	let ty = match expr with
	| LocalFun (params,body) -> check_lambda env m params body
	| FunCall (exp,args) -> check_funcall env exp args
	| Var id -> find_id_type env id
	| Field (deref,exp,id) -> check_field env deref exp id
	| Const c -> check_const env c
	| Cast (cast_ty,exp) -> check_cast env cast_ty exp
	| Parens exp -> check_expr env exp
	| Assign (lhs,op,rhs) -> check_assign env lhs op rhs
	| Choice (cond,iftrue,iffalse) -> check_cond env cond iftrue iffalse
	| Index (arr,idx) -> check_index env arr idx
	| Unsafe inner -> check_expr (withunsafe env) inner
	| Init tinit -> check_tinit env tinit 
	| EBlock block -> check_eblock env m block
	| JklNonDet _ -> tyerr env (str "Unexpected non-determinism")
	in
	set_type m ty;
	ty
	with
	  | _ -> tyerr env (str "Type error in expression")	


(* ------------------------------------------------------------
 * Type Check a Switch Block
 * ------------------------------------------------------------ *)

and check_switch_scrut env (m,exp) =
	let typ = check_expr env (m,exp) in 
	if (is_tagged_typ (resolve_typ env typ)) then 
		(check_exp_is_lval env (m,exp); typ)
	else if is_basic_typ typ then typ
	else pfatal m (str "Scrutinee cannot be of type"<++>pp_ty typ)
	
and check_switch env scrut s = 
	let typ = check_switch_scrut env scrut in
	let env = with_scrutinee env (Some (scrut,typ)) in
	check_stmt env s


(* ------------------------------------------------------------
 * Type Check a Case Statement
 * ------------------------------------------------------------ *)
	
and check_pattern env pat stmt = 
	let var,pattyp = get_scrutinee env in
	match pat with
	| PTag (tag,id_opt) ->
		let bodyty = constr_typ env pattyp tag in
		set_bodyty (get_meta env) bodyty;
		set_scrutvar (id_meta tag) var; 
		(match id_opt with
			| Some id -> 
				add_localvar env id bodyty;
				new_fwddecl env id bodyty;
			| None -> ()) 
	| PDefault -> ()
	| PConst c -> same_type env pattyp (check_constexpr env c)

and constr_typ env fulltyp tagid =
	let tyvars,bodyty,fullid = find_constr env tagid in
	let tyargs = List.map tywild tyvars in
	let declmods = if tyvars <> [] then [DWithArgs tyargs] else [] in
	let expect_ty = 
			[SCoreHere],TyStruct (SKTagged,Some fullid,None),
			declmods in
	let subst = new_subst tyvars in
	fill_in_pattern_typ subst env expect_ty fulltyp;
	subst_typ subst bodyty
		
		
(* ------------------------------------------------------------
 * Type Check a Statement
 * ------------------------------------------------------------ *)
(* TODO: check break, continue, goto are valid *)

and check_jump env jump = match jump with
	| JBreak | JContinue | JGoto _ -> ()
	| JReturn exp_opt ->
		check_return env exp_opt (get_rettype env) (set_rettype env)
	| JRet exp_opt ->
		check_return env exp_opt (get_conttype env) (set_conttype env)

and check_eblock env m (decls,stmts) = 
	let env = localenv env in
	let env = withfwddecls env in
	List.iter (check_decl env) decls;
	List.iter (check_stmt env) stmts;
	set_fwddecls m (get_block_fwds env);
	set_tempdecls m (get_block_temps env);
	set_lamenvs m (get_block_lamenvs env);
	set_dictenvs m (get_block_dictenvs env);
	match list_last stmts with
		| _,SExp e -> check_expr env e
		| _ -> tyvoid
	
and check_block env m block = ignore (check_eblock env m block)
	


and check_stmt env (m,stmt) = 
	let env = withstmt env (m,stmt) in
	let env = withtemps env in
	(match stmt with
	| Label (l,s) -> check_stmt env s
	| SExp e -> ignore (check_expr env e)
	| Block block -> check_block env m block
	| Switch (scrut,s) -> check_switch env scrut s
	| Case (pat,stmt) -> 
			check_pattern env pat stmt;  
			check_stmt env stmt;	
	| If (cond,iftrue,iffalse) -> 
			ignore (check_expr env cond);
			check_stmt env iftrue;
			option_iter (check_stmt env) iffalse
	| While (cond,stmt) | Do (stmt,cond) -> 
			ignore (check_expr env cond);
			check_stmt env stmt
	| For (init_opt,cond_opt,step_opt,body) ->
			ignore (check_expr_opt env init_opt);
			ignore (check_expr_opt env cond_opt);
			ignore (check_expr_opt env step_opt);
			check_stmt env body
	| Jump jump -> check_jump env jump
	| Semicolon -> ());
	set_temps m (get_stmt_temps env)


(* ------------------------------------------------------------
 * Check a top-level declaration
 * ------------------------------------------------------------ *)
	
let add_method_decl env ifaceid decl =
	let funid,ret_ty,args,kind = split_fundecl decl in
	add_methoddef env funid ifaceid

let check_ifacefun env ifaceid decl =
	let methid,_,args,_ = split_fundecl decl in
	match args with
	| ArgsFull (m,_,_) -> set_envarg m EnvOpaque
	| _ -> tyerr env (str "Method args must be fully specified")
	
let check_macrotype env kind = match kind with
	| MDecl decl -> check_decl env decl
	| MSilent _ | MSilentFun _ -> ()	
	
let check_extdecl env (m,decl) = 
	let env = withmeta env m in
	try (match decl with
	| Include _ -> ()
	| Interface (ifaceid,(thisid,methods as ifacedef)) -> 
			add_ifacedef env ifaceid ifacedef;
			List.iter (add_method_decl env ifaceid) methods;
			List.iter (check_ifacefun env ifaceid) methods
	| Dict dictdef -> check_dictdef env dictdef
	| Decl decl -> check_decl env decl
	| Func (decl,krdecls,block) when get_lang m = PureC -> check_decl env decl
	| Func (decl,krdecls,block as fundef) -> 
			check_decl env decl;
			check_fundef env m fundef
	| MacroType kind -> check_macrotype env kind
	| DSemicolon -> ()
	| StdType decl -> 
		let stds = decl_fields decl in
		List.iter (fun (typ,id) -> add_stdtype env id typ) stds
	| NonDet _ -> tyerr env (str "Unexpected non-determinism")
	)
	with
	  | _ -> tyerr env (str "Type error in declaration")	


(* ------------------------------------------------------------
 * Type Check a Program
 * ------------------------------------------------------------ *)

let check_program program includes = 
	let env = new_env () in
	List.iter (check_extdecl env) includes;
	List.iter (check_extdecl env) program
