open Syntax
open Location
open Metadata
open Syntaxutil
open Util
open Parsing
open Names
open Errors
open Scanf
open Filename

module SH = Data.StringCols.Hash
type 'a strhash = 'a SH.t

(* ------------------------------------------------------------
 * Find #includes in a file
 * ------------------------------------------------------------ *)

let scan_line_for_include line = 
	try	sscanf line "#include %[^'\n']"
		(fun fname -> Some fname)
	with _ -> None

let rec scan_includes input =
	try
		let line = input_line input in
		let result = scan_line_for_include line in
		result :: scan_includes input
	with 
		| End_of_file -> []

let find_includes fname =
	let input = open_in fname in
	let includes = option_list_collapse 
		(scan_includes input) in
	close_in input;
	includes
	
let incname_to_localname incname = 
	try sscanf incname "\"%[^'\"']\""
		(fun fname -> Some fname)
	with _ -> None



(* ------------------------------------------------------------
 * Is a C file encoded Jekyll?
 * ------------------------------------------------------------ *)

let filetypes : bool strhash = SH.create ()	
		
let file_is_jekyll_c fname = match SH.maybe_find fname filetypes with
	| Some b -> b
	| None ->
		let includes = find_includes fname in
		let b = List.exists (fun n -> n = "<jekyll_1.h>") includes in
		SH.add fname b filetypes;
		b

let try_file_exists relativefile dir = 
	let name = Filename.concat dir relativefile in 
	if Sys.file_exists name then Some name else None	
	
let relative_file relativefile = 
	let possibledirs = !Cmdargs.includedirs in
	match list_first_option (try_file_exists relativefile) possibledirs with
		| Some filename -> Some filename
		| None -> None

let process_include fname = match incname_to_localname fname with
	| Some lname -> begin
		match relative_file lname with
		| Some x when file_is_jekyll_c x -> 
			"\"" ^ chop_extension lname ^ ".jkh\"" 
		| _ -> fname
		end
	| None -> fname
let process_includes includes = List.map process_include includes

let process_c_include incname =
	match incname_to_localname incname with
	| Some x when check_suffix x ".jkh" -> 
		"\"" ^ (chop_suffix x ".jkh") ^ ".h\""
	| _ -> incname
	

(* ------------------------------------------------------------
 * What language are we parsing?
 * ------------------------------------------------------------ *)

let current_lang = ref PureC
let get_current_lang () = !current_lang

let file_language fname = 
	if check_suffix fname "jkl" || check_suffix fname "jkh" then Jekyll
	else if file_is_jekyll_c fname then EncodedC 
	else PureC

let set_current_file fname = current_lang := file_language fname

let warn_bad_jekyll msg = 
	if get_current_lang () <> PureC then warning (current_span ()) msg

let lang_name lang = match lang with
	| Jekyll -> "jekyll"
	| PureC -> "raw c"
	| EncodedC -> "encoded C"

(* ------------------------------------------------------------
 * Parse Errors
 * ------------------------------------------------------------ *)

let parse_error msg =
	let lang = get_current_lang () in
	let msg = "syntax error while parsing " ^ lang_name lang in
	let span = current_span () in
	if !Cmdargs.short_errors then
		print_simple_error msg span
	else begin
		print_error_header msg span;
		print_span_source span; 
		if get_current_lang() = PureC then begin
			print_endline "macro-expands to:";
			print_current_line ();	
		end;
		print_span_guide span;
		print_error_trailer ()
	end;
	exit 2
		

(* ------------------------------------------------------------
 * Construct Metadata
 * ------------------------------------------------------------ *)

let mk a b : metadata = makemeta (mkspan a b)
let mkm () = mk (Parsing.symbol_start_pos ()) (Parsing.symbol_end_pos ())
let ident tknm = (makemeta (lt_span (fst tknm)), snd tknm)


(* ------------------------------------------------------------
 * Construct Expressions for Operator Application 
 * ------------------------------------------------------------ *)

let binop x op y = mkm (), FunCall ((mkm(),Var (strid op)),[x;y])
let unop op x = mkm (), FunCall ((mkm(), Var (strid op)),[x])
let punop x op = mkm (), FunCall ((mkm(), Var (strid op)),[x])


(* ------------------------------------------------------------
 * Which names represent types
 * ------------------------------------------------------------ *)

let type_names : unit SH.t = SH.of_list ["__builtin_va_list",()]
let add_typename str = SH.add str () type_names
let is_typename str = SH.mem str type_names

let struct_names : unit SH.t = SH.create ()
let add_struct str = SH.add str () struct_names
let is_struct str = SH.mem str struct_names
	
let method_names : unit SH.t = SH.create ()
let add_method str = SH.add str () method_names
let is_method s = SH.mem s method_names	
	
let tyvar_names : unit SH.t = SH.create ()
let add_tyvar str = SH.add str () tyvar_names
let is_tyvar str = SH.mem str tyvar_names

let define_names : defdetails SH.t = SH.create ()
let add_define name details = SH.add name details define_names
let is_define name = SH.mem name define_names

let constructor_names : unit SH.t = SH.create ()
let add_constructor str = SH.add str () constructor_names
let is_constructor str = SH.mem str constructor_names

let silentmacro_names : unit SH.t = SH.create ()
let add_silentmacro str = SH.add str () silentmacro_names
let is_silentmacro str = SH.mem str silentmacro_names

let silentmacro1_names : unit SH.t = SH.create ()
let add_silentmacro1 str = SH.add str () silentmacro1_names
let is_silentmacro1 str = SH.mem str silentmacro1_names


let reset_lexnames () = 
	SH.clear type_names;
	add_typename "__builtin_va_list";
	SH.clear tyvar_names;
	SH.clear define_names;
	SH.clear constructor_names;
	SH.clear silentmacro_names;
	SH.clear silentmacro1_names 


let add_macrotype kindstr ids =
	let names = List.map id_str ids in
    match kindstr with
	| "silent" -> 
		List.iter add_silentmacro names; MacroType (MSilent ids)
	| "silent-fun" -> 
		List.iter add_silentmacro1 names; MacroType (MSilentFun ids)
	| _ -> parse_error ()
	

(* ------------------------------------------------------------
 * Register constructor names 
 * ------------------------------------------------------------ *)

let add_decl_conname (m,_ as decl) = 
	if is_simpledecl decl then
		add_constructor (id_str (simpledecl_name decl))	
	else error m.span "Tagged members must be simple declarations"

let register_connames corety = match corety with
	| TyStruct (SKTagged,_,Some (_,_,decls)) ->
			List.iter add_decl_conname decls
	| _ -> ()


(* ------------------------------------------------------------
 * Set the correct function kind 
 * ------------------------------------------------------------ *)

let set_fundecl_kind kind (m,decl) = match decl with
	| base,[(DFun(args,_)::mods,Some id),None] ->
		m, (base,[(DFun(args,kind)::mods,Some id),None])
	| _ -> fatal m.span "Interface members must be functions"

let set_fundecl_kinds kind decls = List.map (set_fundecl_kind kind) decls


(* ------------------------------------------------------------
 * No args <-> void arg
 * ------------------------------------------------------------ *)
	
let process_funargs args = match args with
	| [_,((_,TyVoid),[(_,None),_])] -> []
	| _ -> args

	

(* ------------------------------------------------------------
 * Separate the coretype from the type qualifiers
 * ------------------------------------------------------------ *)

type spec_or_ty = 
	| OSpec of specqual
 	| OType of coretyp

let rec split_declspecs declspecs = match declspecs with
	| (OSpec specifier)::others ->
		let xs,ys = split_declspecs others in
		specifier::xs,ys
	| (OType coretype)::others ->
		let xs,ys = split_declspecs others in
		(SCoreHere::xs),(coretype::ys)
	| [] -> [],[]	

let specs_to_basetyp specs =
	let specifiers,ys = split_declspecs specs in
	let coretyp = match ys with
	| [] -> TyBasic
	| [y] -> y
	| x::xs -> parse_error "Two or more types specified" 
	in
	specifiers,coretyp

let make_typ specs declty = 
	let specifiers,coretyp = specs_to_basetyp specs in
	specifiers,coretyp,declty

let specs_to_typ specs = make_typ specs []

let mkdecl specs declarator = 
	mkm (),(specs_to_basetyp specs,[declarator,None])

		
(* ------------------------------------------------------------
 * Check line endings
 * ------------------------------------------------------------ *)

let rec has_cr_char input = 
	try let line = input_line input in
		if String.contains line '\r' then true
		else has_cr_char input
	with _ -> false

let file_is_dos fname = 
	let input = open_in_bin fname in
	let is_dos = has_cr_char input in
	close_in input;
	is_dos
