
%{
	open Syntax
	open Util
	open Syntaxutil
	open Metadata
	open Parseutils
%}

/* Derived from the Bison grammar given at:
http://www.lysator.liu.se/c/ANSI-C-grammar-y.html
*/

/* Pure C */
%token<Location.lextoken * string> IDENTIFIER TYPE_NAME CONSTANT
%token<string> STRING_LITERAL INT_LITERAL FLOAT_LITERAL
%token<unit> SIZEOF
%token<unit> PTR_OP INC_OP DEC_OP LEFT_OP RIGHT_OP 
%token<unit> LE_OP GE_OP EQ_OP NE_OP
%token<unit> ANDAND_OP OROR_OP MUL_ASSIGN 
%token<unit> DIV_ASSIGN MOD_ASSIGN ADD_ASSIGN
%token<unit> SUB_ASSIGN LEFT_ASSIGN RIGHT_ASSIGN AND_ASSIGN
%token<unit> XOR_ASSIGN OR_ASSIGN 
%token<unit> AND_OP BANG_OP TILDE_OP MINUS_OP PLUS_OP DIV_OP 
%token<unit> MOD_OP LT_OP GT_OP OR_OP XOR_OP

%token<unit> ASSIGN STAR QUESTION
%token<unit> TYPEDEF EXTERN STATIC AUTO REGISTER INLINE
%token<unit> CHAR SHORT INT LONG SIGNED UNSIGNED FLOAT 
%token<unit> DOUBLE CONST VOLATILE VOID
%token<unit> STRUCT UNION ENUM ELLIPSIS

%token<unit> CASE DEFAULT IF ELSE SWITCH WHILE DO 
%token<unit> FOR GOTO CONTINUE BREAK RETURN

%token<unit> LPAREN RPAREN LSQUARE RSQUARE LBRACE RBRACE
%token<unit> SEMICOLON COLON PERIOD COMMA


/* GCC Extensions */
%token<unit> TYPEOF U_TYPEOF ALIGNOF


/* Parsed proprocessor directives */

%token<string> INCLUDE
%token<unit> EOF 


/* Encoded C */

%token<unit> C_UNSAFE C_MACROTYPE
%token<unit> C_IMPLEMENT C_IMPLEMENT_PROTO C_ENDIMPLEMENT
%token<unit> RET
%token<unit> C_STDTYPE
%token<unit> C_P C_VOID C_NEEDS
%token<unit> C_ENVARG C_DICTENV  


/* Jekyll */
%token<Location.lextoken * string> TYVAR J_CONSTRUCTOR
%token<Location.lextoken> J_NEW C_NEW J_MALLOC C_MALLOC
%token<unit> J_UNSAFE J_MACROTYPE J_INTERFACE J_TAGGED J_IMPLEMENT
%token<unit> J_STDTYPE J_IMPORT
%token<Location.lextoken * string> J_OP_NAME 

/* parse errors */
%token<Location.lextoken> GARBAGE


%type <Syntax.expression> expression constant_expression
%type <Syntax.expression> primary_expression postfix_expression 
%type <Syntax.expression> unary_expression cast_expression
%type <string> unary_operator 
%type <Syntax.declaration> declaration
%type <Syntax.declaration list> declaration_list parameter_list
%type <spec_or_ty list> declaration_specifiers 
%type <Syntax.specqual> storage_class_specifier type_qualifier
%type <Syntax.coretyp> struct_or_union_specifier enum_specifier
%type <Syntax.init_declarator> init_declarator
%type <Syntax.init_declarator list> init_declarator_list
%type <Syntax.declaration> parameter_declaration
%type <Syntax.declarator> declarator direct_declarator struct_declarator
%type <Syntax.declmod list> abstract_declarator direct_abstract_declarator
%type <Syntax.declmod list> pointer
%type <Syntax.declaration> parameter_declaration
%type <Syntax.typ> type_name
%type <Syntax.declaration> struct_declaration
%type <Syntax.statement> statement 
%type <Syntax.block> compound_statement
%type <Syntax.extdeclaration> external_declaration
%type <Syntax.metadata * string> j_constructor

%type <Syntax.metadata * Syntax.typ list> withargs

%type <Syntax.program> translation_unit
%type <string> assignment_operator

%start translation_unit
%%

name
	: IDENTIFIER
		{$1}
	| TYPE_NAME
		{$1}
	| J_CONSTRUCTOR
		{$1}
	| TYVAR
		{$1}

primary_expression
	: IDENTIFIER
		{mkm (), Var (ident $1)}
	| INT_LITERAL
		{mkm (), Const (ConstInt $1)}
	| FLOAT_LITERAL
		{mkm (), Const (ConstFloat $1)}
	| STRING_LITERAL
		{mkm (), Const (ConstString $1)}
	| LPAREN expression RPAREN
		{mkm (), Parens $2}
	| LPAREN compound_statement RPAREN
		{mkm (), EBlock $2}
	;
		
lambda_expression
	: LBRACE parameter_list COLON compound_statement_body RBRACE
		{mkm (), LocalFun ($2,$4)}	 
		
lambda_expression_list
	: /* empty */
		{[]}
	| lambda_expression lambda_expression_list
		{$1 :: $2}

complex_function_args
	: LPAREN expression_list RPAREN lambda_expression_list
		{$2 @ $4}
	
postfix_expression
	: primary_expression
		{$1}
	| postfix_expression LSQUARE expression RSQUARE
		{mkm (), Index ($1,$3)}
	| postfix_expression LPAREN RPAREN
		{mkm (), FunCall ($1,[])}
	| postfix_expression complex_function_args
		{mkm (), FunCall ($1,$2)}
	| postfix_expression PERIOD name
		{mkm (), Field (false,$1,ident $3)}
	| postfix_expression PTR_OP name
		{mkm (), Field (true,$1,ident $3)}
	| postfix_expression INC_OP
		{punop $1 "++P"}
	| postfix_expression DEC_OP
		{punop $1 "--P"}
	;

unary_expression
	: postfix_expression
		{$1}
	| INC_OP unary_expression
		{unop "++" $2}
	| DEC_OP unary_expression
		{unop "--" $2}
	| unary_operator cast_expression
		{unop $1 $2}
	| SIZEOF unary_expression
		{mkm (), Const (ConstSizeExp $2)}
	| SIZEOF LPAREN type_name RPAREN
		{mkm (), Const (ConstSizeTy $3)}
	| ALIGNOF unary_expression
		{mkm (), Const (ConstAlignExp $2)}
	| ALIGNOF LPAREN type_name RPAREN
		{mkm (), Const (ConstAlignTy $3)}
	;

expression_list
	: init_expression 
		{[$1]}
	| expression_list COMMA init_expression
		{$1 @ [$3]} 

unary_operator
	: AND_OP
		{"&"}
	| STAR
		{"*"}
	| PLUS_OP
		{"+"}
	| MINUS_OP
		{"-"}
	| TILDE_OP
		{"~"}
	| BANG_OP
		{"!"}
	;

cast_expression
	: unary_expression
		{$1}
	| LPAREN type_name RPAREN cast_expression
		{mkm (), Cast ($2,$4)}
	| LPAREN type_name RPAREN structinit_expression
		{mkm (), Cast ($2,$4)} 
	;

multiplicative_expression
	: cast_expression
		{$1}
	| multiplicative_expression STAR cast_expression
		{binop $1 "*" $3}
	| multiplicative_expression DIV_OP cast_expression
		{binop $1 "/" $3}
	| multiplicative_expression MOD_OP cast_expression
		{binop $1 "%" $3}
	;

additive_expression
	: multiplicative_expression
		{$1}
	| additive_expression PLUS_OP multiplicative_expression
		{binop $1 "+" $3}
	| additive_expression MINUS_OP multiplicative_expression
		{binop $1 "-" $3}
	;

shift_expression
	: additive_expression
		{$1}
	| shift_expression LEFT_OP additive_expression
		{binop $1 "<<" $3}
	| shift_expression RIGHT_OP additive_expression
		{binop $1 ">>" $3}
	;

relational_expression
	: shift_expression
		{$1}
	| relational_expression LT_OP shift_expression
		{binop $1 "<" $3}
	| relational_expression GT_OP shift_expression
		{binop $1 ">" $3}
	| relational_expression LE_OP shift_expression
		{binop $1 "<=" $3}
	| relational_expression GE_OP shift_expression
		{binop $1 ">=" $3}
	;

equality_expression
	: relational_expression
		{$1}
	| equality_expression EQ_OP relational_expression
		{binop $1 "==" $3}
	| equality_expression NE_OP relational_expression
		{binop $1 "!=" $3}
	;

and_expression
	: equality_expression
		{$1}
	| and_expression AND_OP equality_expression
		{binop $1 "&" $3}
	;

exclusive_or_expression
	: and_expression
		{$1}
	| exclusive_or_expression XOR_OP and_expression
		{binop $1 "^" $3}
	;

inclusive_or_expression
	: exclusive_or_expression
		{$1}
	| inclusive_or_expression OR_OP exclusive_or_expression
		{binop $1 "|" $3}
	;

logical_and_expression
	: inclusive_or_expression
		{$1}
	| logical_and_expression ANDAND_OP inclusive_or_expression
		{binop $1 "&&" $3}
	;

logical_or_expression
	: logical_and_expression
		{$1}
	| logical_or_expression OROR_OP logical_and_expression
		{binop $1 "||" $3}
	;

conditional_expression
	: logical_or_expression
		{$1}
	| logical_or_expression QUESTION expression COLON conditional_expression
		{mkm (), Choice ($1,$3,$5)}
	;

storage_kind_j
	: J_NEW {New}
	| J_MALLOC {Malloc}
	
init_expression	
	: J_CONSTRUCTOR 
		{mkm (), Init (mkm (), TConApp (ident $1, None))}
	| J_CONSTRUCTOR nonstmt_expression
		{mkm (), Init (mkm (), TConApp (ident $1, Some $2))}
	| storage_kind_j nonstmt_expression
		{mkm (), Init (mkm (),TAlloc ($1, $2))}
	| unsafe_expression {$1}

nonstmt_expression
	: init_expression {$1}
	| structinit_expression {$1}

assignment_expression
	: conditional_expression
		{$1}
	| unary_expression assignment_operator nonstmt_expression
		{mkm (), Assign ($1,$2,$3)}
	;

unsafe_expression	
	: unsafe assignment_expression
		{mkm (), Unsafe $2}
	| assignment_expression 
		{$1}
	
	
unsafe 
	: J_UNSAFE {()}
	| C_UNSAFE {()}

structinit_expression
	: LBRACE expression_list RBRACE
		{mkm (), 
			Init (mkm (),TStruct 
			(List.map (fun f -> "",f) $2)
			)}
	;

assignment_operator
	: ASSIGN
		{"="}
	| MUL_ASSIGN
		{"*="}
	| DIV_ASSIGN
		{"/="}
	| MOD_ASSIGN
		{"%="}
	| ADD_ASSIGN
		{"&="}
	| SUB_ASSIGN
		{"-="}
	| LEFT_ASSIGN
		{"<<="}
	| RIGHT_ASSIGN
		{">>="}
	| AND_ASSIGN
		{"&="}
	| XOR_ASSIGN
		{"^="}
	| OR_ASSIGN
		{"|="}
	;

expression
	: init_expression
		{$1}
	| expression COMMA init_expression
		{binop $1 "," $3}
	;
	
noninit_expression
	: unsafe_expression
		{$1}
	| noninit_expression COMMA unsafe_expression
		{binop $1 "," $3}
	;


constant_expression
	: conditional_expression
		{$1}
	;

declaration
	: declaration_specifiers 
 		{mkm (),(specs_to_basetyp $1,[])}
	| declaration_specifiers init_declarator_list 
		{
		let (specs,coretyp as typ) = specs_to_basetyp $1 in
		let names = concat_map Syntaxutil.init_names $2 in
		if List.mem STypeDef specs then
			List.iter (fun id -> add_typename (id_str id)) names;
		mkm (),(typ,$2)}
	;

declaration_specifiers
	: storage_class_specifier
		{[OSpec $1]}
	| storage_class_specifier declaration_specifiers
		{OSpec $1::$2}
	| type_specifier
		{[$1]}
	| type_specifier declaration_specifiers
		{$1::$2}
	| type_qualifier
		{[OSpec $1]}
	| type_qualifier declaration_specifiers
		{OSpec $1::$2}
	;

init_declarator_list
	: init_declarator
		{[$1]}
	| init_declarator COMMA init_declarator_list
		{$1 :: $3}
	;

init_declarator
	: declarator
		{$1,None}
	| declarator ASSIGN initialiser
		{$1,(Some $3)}
	;

storage_class_specifier
	: TYPEDEF
		{STypeDef}
	| EXTERN
		{SExtern}
	| STATIC
		{SStatic}
	| AUTO
		{SAuto}
	| REGISTER
		{SRegister}
	| INLINE
		{SInline}
	;

type_specifier
	: VOID
		{OType TyVoid}
	| CHAR
		{OSpec SChar}
	| SHORT
		{OSpec SShort}
	| INT
		{OSpec SInt}
	| LONG
		{OSpec SLong}
	| FLOAT
		{OSpec SFloat}
	| DOUBLE
		{OSpec SDouble}
	| SIGNED
		{OSpec SSigned}
	| UNSIGNED
		{OSpec SUnsigned}
	| struct_or_union_specifier
		{register_connames $1; OType $1}
	| enum_specifier
		{OType $1}
	| typevar
		{$1}
	| TYPE_NAME
		{OType (TyName (ident $1))}
	| typeof 
		{OType $1}
	;

typeof
	: TYPEOF LPAREN type_name RPAREN
		{TyTypeofTyp (false,$3)}
	| U_TYPEOF LPAREN type_name RPAREN
		{TyTypeofTyp (true,$3)}
	| TYPEOF LPAREN expression RPAREN
		{TyTypeofExp (false,$3)}
	| U_TYPEOF LPAREN expression RPAREN
		{TyTypeofExp (true,$3)}

typevar
	: TYVAR
		{OType (TyWild (ident $1))}

struct_or_union_specifier
	: struct_or_union name LBRACE struct_declaration_list RBRACE
		{add_struct (snd $2);
		 TyStruct ($1,Some (ident $2),Some (mkm (),[],$4))}
	| struct_or_union withtyparams name LBRACE struct_declaration_list RBRACE
		{add_struct (snd $3);
		 TyStruct ($1,Some (ident $3),Some (mkm (),$2,$5))}
	| struct_or_union LBRACE struct_declaration_list RBRACE
		{TyStruct ($1,None,Some (mkm (),[],$3))}
	| struct_or_union name
		{add_struct (snd $2);
		 TyStruct ($1,Some (ident $2),None)}
	;

struct_or_union
	: STRUCT
		{SKStruct}
	| UNION
		{SKUnion}
	| J_TAGGED
		{SKTagged}
	;

struct_declaration_list
	: /* empty */
		{[]}
	| struct_declaration struct_declaration_list
		{$1 :: $2}
	;

struct_declaration
	: specifier_qualifier_list struct_declarator_list SEMICOLON
		{mkm (), (specs_to_basetyp $1, $2)}
	| C_VOID LPAREN IDENTIFIER RPAREN 
		{mkm (), (([STaggedOnly;SCoreHere],TyVoid),[([],Some (ident $3)),None])}
	;

specifier_qualifier_list
	: type_specifier specifier_qualifier_list
		{$1::$2}
	| type_specifier
		{[$1]}
	| type_qualifier specifier_qualifier_list
		{(OSpec $1)::$2}
	| type_qualifier
		{[OSpec $1]}
	;

struct_declarator_list
	: struct_declarator
		{[$1,None]}
	| struct_declarator COMMA struct_declarator_list
		{($1,None) :: $3}
	;

struct_declarator
	: declarator
		{$1}
	| COLON constant_expression
		{[DBitField $2],None}
	| declarator COLON constant_expression
		{DBitField $3::fst $1,snd $1}
	;

enum_specifier
	: ENUM LBRACE enumerator_list RBRACE
		{TyEnum (None,Some $3)}
	| ENUM IDENTIFIER LBRACE enumerator_list RBRACE
		{TyEnum (Some (ident $2),Some $4)}
	| ENUM IDENTIFIER
		{TyEnum (Some (ident $2),None)}
	;

enumerator_list
	: {[]}
	| enumerator
		{[$1]}
	| enumerator COMMA enumerator_list
		{$1 :: $3}
	;

enumerator
	: IDENTIFIER
		{ident $1,None}
	| IDENTIFIER ASSIGN constant_expression
		{ident $1,Some $3}
	;

type_qualifier
	: CONST
		{SConst}
	| VOLATILE
		{SVolatile}
	;

declarator
	: withargs pointer_declarator
		{fst $2 @ [DWithArgs (snd $1)],snd $2}
	| pointer_declarator
		{$1}
		
withargs
	: LT_OP type_name_list GT_OP
		{mkm (), $2}
	| C_P LPAREN type_name_semi_list RPAREN		
		{mkm (), $3}



tyvar_list
	: TYVAR
		{[ident $1]}
	| TYVAR COMMA tyvar_list
		{ident $1 :: $3}
		
tyvar_semi_list
	: TYVAR
		{[ident $1]}
	| TYVAR SEMICOLON tyvar_semi_list
		{ident $1 :: $3}

withtyparams
	: LT_OP tyvar_list GT_OP
		{$2}
	| C_P LPAREN tyvar_semi_list RPAREN
		{$3}

pointer_declarator
	: pointer direct_declarator
		{fst $2 @ $1,snd $2}
	| pointer TYPE_NAME
		{$1,Some (ident $2)}
	| direct_declarator
		{$1}
	;
	
typaram_j
	: IDENTIFIER TYVAR
		{ident $2,ident $1}	
	
typaram_j_list
	: typaram_j COMMA typaram_j_list
		{$1::$3}
	| typaram_j
		{[$1]}	

	
function_parameters
	: LPAREN parameter_list RPAREN
		{DFun(ArgsFull (mkm (),[],process_funargs $2),SimpleFun)}
	| LT_OP typaram_j_list GT_OP LPAREN parameter_list RPAREN
		{DFun(ArgsFull (mkm (),$2,process_funargs $5),SimpleFun)}
	| LPAREN parameter_list COMMA ELLIPSIS RPAREN
		{DFun(ArgsFull (mkm (),[],process_funargs $2),VarArgs)}
	| LPAREN RPAREN
		{DFun(ArgsNoinfo,SimpleFun)}
	| BANG_OP LPAREN parameter_list RPAREN
		{DFun(ArgsFull (mkm (),[],process_funargs $3),Closure)}
;

direct_declarator
	: IDENTIFIER
		{[], Some (ident $1)}
	| LPAREN declarator RPAREN
		{fst $2 @ [DParens],snd $2}
	| direct_declarator LSQUARE constant_expression RSQUARE
		{fst $1 @ [DArray (Some $3)],snd $1}
	| direct_declarator LSQUARE RSQUARE
		{fst $1 @ [DArray None],snd $1}
/* change from original - transformed parameter_type_list */
	| direct_declarator function_parameters
		{fst $1 @ [$2],snd $1}
	| direct_declarator LPAREN identifier_list RPAREN
		{fst $1 @ [DFun (ArgsNamed $3,SimpleFun)],snd $1}
	;

pointer
	: STAR
		{[DPtr []]}
	| STAR type_qualifier_list
		{[DPtr $2]}
	| STAR pointer
		{$2 @ [DPtr []]}
	| STAR type_qualifier_list pointer
		{$3 @ [DPtr $2]}
	| QUESTION
		{[DFatPtr []]}
	| QUESTION type_qualifier_list
		{[DFatPtr $2]}
	| QUESTION pointer
		{$2 @ [DFatPtr []]}
	| QUESTION type_qualifier_list pointer
		{$3 @ [DFatPtr $2]}

	;

type_qualifier_list
	: type_qualifier
		{[$1]}
	| type_qualifier type_qualifier_list
		{$1 :: $2}
	;

parameter_list
	: parameter_declaration
		{[$1]}
	| parameter_list COMMA parameter_declaration
		{$1 @ [$3]}
	;

parameter_declaration
	: declaration_specifiers declarator
		{mkm (), (specs_to_basetyp $1,[$2,None])}
	| declaration_specifiers abstract_declarator
		{mkm (), (specs_to_basetyp $1,[($2,None),None])}
	| declaration_specifiers
		{mkm (), (specs_to_basetyp $1,[([],None),None])}
	;

identifier_list
	: IDENTIFIER
		{[ident $1]}
	| IDENTIFIER COMMA name_list
		{ident $1 :: $3}
	;


name_list
	: name
		{[ident $1]}
	| name COMMA name_list
		{ident $1 :: $3}
	;

type_name
	: specifier_qualifier_list
		{specs_to_typ $1}
	| specifier_qualifier_list abstract_declarator
		{make_typ $1 $2}
	;
	
type_name_list
	: type_name
		{[$1]}
	| type_name COMMA type_name_list
		{$1 :: $3}
		
type_name_semi_list
	: type_name
		{[$1]}
	| type_name SEMICOLON type_name_semi_list
		{$1 :: $3}

abstract_declarator
	: abstract_pointer_declarator
		{$1}
	| withargs abstract_pointer_declarator
		{$2 @ [DWithArgs (snd $1)]}
	| withargs
		{[DWithArgs (snd $1)]}

abstract_pointer_declarator
	: pointer
		{$1}
	| direct_abstract_declarator
		{$1}
	| pointer direct_abstract_declarator
		{$2 @ $1}
	;

direct_abstract_declarator
	: LPAREN abstract_declarator RPAREN
		{$2 @ [DParens]}
	| LSQUARE RSQUARE
		{[DArray None]}
	| LSQUARE constant_expression RSQUARE
		{[DArray (Some $2)]}
	| direct_abstract_declarator LSQUARE RSQUARE
		{$1 @ [DArray None]}
	| direct_abstract_declarator LSQUARE constant_expression RSQUARE
		{$1 @ [DArray (Some $3)]}
	| function_parameters
		{[$1]}
/* change from original - inlined parameter_type_list */
	| direct_abstract_declarator function_parameters
		{$1 @ [$2]}
	;

initialiser
	: assignment_expression
		{IExp $1}
	| LBRACE initialiser_list RBRACE
		{IFields ($2,false)}
	| LBRACE initialiser_list COMMA RBRACE
		{IFields ($2,true)}
	;

initialiser_list
	: initialiser
		{[$1]}
	| initialiser_list COMMA initialiser
		{$1 @ [$3]}
	;

statement
	: labeled_statement
		{$1}
	| compound_statement
		{mkm (), Block $1}
	| expression_statement
		{fst $1,match snd $1 with None -> Semicolon | Some x -> SExp x}
	| selection_statement
		{$1}
	| iteration_statement
		{$1}
	| jump_statement
		{$1}
	;

labeled_statement
	: IDENTIFIER COLON statement
		{mkm (), Label (ident $1,$3)}
	| CASE constant_expression COLON statement
		{mkm (), Case (PConst $2,$4)}
	| tagcase_pattern 
		{mkm (), $1}
	| DEFAULT COLON statement
		{mkm (), Case (PDefault,$3)}
	;

j_constructor
	: J_CONSTRUCTOR
		{ident $1}
		
tagcase_pattern
	: CASE j_constructor COLON statement
		{Case (PTag ($2,None),$4)}
	| CASE j_constructor IDENTIFIER COLON statement
		{Case (PTag ($2,Some (ident $3)),$5)}

compound_statement
	: LBRACE compound_statement_body RBRACE 
		{$2}
	;

compound_statement_body
	: /* empty */
		{[],[]}
	| statement_list 
		{[],$1}
	| declaration_list
		{$1,[]}
	| declaration_list statement_list
		{$1,$2}
	;
		
declaration_list
	: declaration SEMICOLON
		{[$1]}
	| declaration SEMICOLON declaration_list
		{$1 :: $3}
	;

statement_list
	: statement
		{[$1]}
	| statement statement_list
		{$1 :: $2}
	;

expression_statement
	: SEMICOLON
		{mkm (), None}
	| noninit_expression SEMICOLON
		{mkm (), Some $1}
	;

selection_statement
	: IF LPAREN expression RPAREN statement
		{mkm (), If ($3,$5,None)}
	| IF LPAREN expression RPAREN statement ELSE statement
		{mkm (), If ($3,$5,Some $7)}
	| SWITCH LPAREN expression RPAREN statement
		{mkm (), Switch ($3,$5)} 
	;

iteration_statement
	: WHILE LPAREN expression RPAREN statement
		{mkm (), While ($3,$5)}
	| DO statement WHILE LPAREN expression RPAREN SEMICOLON
		{mkm (), Do ($2,$5)}
	| FOR LPAREN expression_statement expression_statement RPAREN statement
		{mkm (), For (snd $3,snd $4,None,$6)}
	| FOR LPAREN expression_statement expression_statement expression RPAREN statement
		{mkm (), For (snd $3,snd $4,Some $5,$7)}
	;

jump_statement
	: GOTO IDENTIFIER SEMICOLON
		{mkm (), Jump (JGoto (ident $2))}
	| CONTINUE SEMICOLON
		{mkm (), Jump JContinue}
	| BREAK SEMICOLON
		{mkm (), Jump JBreak}
	| RETURN SEMICOLON
		{mkm (), Jump (JReturn None)}
	| RETURN expression SEMICOLON
		{mkm (), Jump (JReturn (Some $2))}
	| RET SEMICOLON
		{mkm (), Jump (JRet None)}
	| RET expression SEMICOLON
		{mkm (), Jump (JRet (Some $2))}	
	;
	
translation_unit
	: external_declaration translation_unit
		{let m,extd = $1 in	
		 let m = with_tags [Lang (get_current_lang ())] m in
		 (m,extd) :: $2}
	| EOF
		{[]}
	;

macrotype
	: C_UNSAFE C_MACROTYPE LPAREN declaration RPAREN SEMICOLON
		{MacroType (MDecl $4)}
	| J_UNSAFE J_MACROTYPE declaration SEMICOLON
		{MacroType (MDecl $3)}
	| C_UNSAFE C_MACROTYPE LPAREN IDENTIFIER identifier_list RPAREN SEMICOLON
		{add_macrotype (snd $4) $5}
	| J_UNSAFE J_MACROTYPE IDENTIFIER identifier_list SEMICOLON
		{add_macrotype (snd $3) $4}

interface_declaration
	: J_INTERFACE IDENTIFIER TYVAR 
		LBRACE struct_declaration_list RBRACE SEMICOLON
		{List.iter 
			(fun d -> add_method (id_str (simpledecl_name d)))
			$5;
		 Interface (ident $2,(ident $3,set_fundecl_kinds IfaceMethod $5))}
		
tyiface_c
	: C_NEEDS LPAREN IDENTIFIER COMMA TYVAR RPAREN
		{ident $5,ident $3}

tyiface_j
	: IDENTIFIER TYVAR
		{ident $2, ident $1}
		
tyiface_j_list
	: tyiface_j
		{[$1]}
	| tyiface_j COMMA tyiface_j_list
		{$1 :: $3}		
		
simple_coretyp
	: struct_or_union name
		{TyStruct ($1,Some (ident $2),None)}
	| TYPE_NAME	
		{TyName (ident $1)}	
		
basictype
	: INT {[SInt]}
	| LONG {[SLong]}
	| LONG LONG {[SLong;SLong]}
	| CHAR {[SChar]}
	| FLOAT {[SFloat]}
	| DOUBLE {[SDouble]}		
		
simple_typ
	: simple_coretyp withargs
		{[SCoreHere],$1,[DWithArgs (snd $2)]}
	| simple_coretyp 
		{[SCoreHere],$1,[]}
	| basictype
		{$1,TyBasic,[]}
		
maybe_withargs
	: withargs
		{[DWithArgs (snd $1)]}
	| /* empty */
		{[]}
				
maybe_tyiface_j_list
	: COLON tyiface_j_list
		{$2}
	| /* empty */
		{[]}
		
c_impl_header
	: LPAREN IDENTIFIER struct_or_union name 
		maybe_withargs RPAREN SEMICOLON
		{([SCoreHere],TyStruct ($3,Some (ident $4),None),$5),ident $2
		}
	| LPAREN IDENTIFIER basictype RPAREN SEMICOLON
		{
		let typ = ($3,TyBasic,[]) in
		typ,ident $2
		}	

c_requirement
	: STRUCT IDENTIFIER STAR IDENTIFIER SEMICOLON VOID STAR IDENTIFIER SEMICOLON
	{
		let ifaceid = (ident $2) in
		let tyid = id_strip_prefix ("_dm" ^ id_str ifaceid ^ "_") (ident $4) in
		tyid,ifaceid}

c_requirement_list
	: c_requirement {[$1]}
	| c_requirement c_requirement_list {$1::$2}

c_dict_env
	: C_DICTENV STRUCT IDENTIFIER LBRACE c_requirement_list RBRACE SEMICOLON
	{$5}
	
maybe_c_dict_env
	: /* nothing */ {[]}
	| c_dict_env {$1}
	
maybe_c_tyiface_list
	: /* nothing */ {[]}
	| tyiface_c SEMICOLON maybe_c_tyiface_list {$1::$3}	
		
implementation_declaration
	: C_IMPLEMENT c_impl_header maybe_c_tyiface_list function_definition_list C_ENDIMPLEMENT declaration
	SEMICOLON
		{
			let typ,iface = $2 in
			Dict (typ,iface,DictImpl ($3,$4))
		}	
		
	| J_IMPLEMENT IDENTIFIER simple_typ maybe_tyiface_j_list LBRACE function_definition_list RBRACE
		{Dict ($3,ident $2,DictImpl ($4, $6))}
		
	| J_IMPLEMENT IDENTIFIER simple_typ maybe_tyiface_j_list SEMICOLON
		{Dict ($3,ident $2,DictProto $4)}	
		
	| C_IMPLEMENT_PROTO c_impl_header maybe_c_dict_env struct_declaration_list EXTERN declaration SEMICOLON
		{
			let typ,iface = $2 in
			Dict (typ,iface,DictProto $3)
		}	

external_declaration
	: function_definition
		{mkm (), Func (snd $1)}
	| interface_declaration
		{mkm (),$1}
	| implementation_declaration
		{mkm (),$1}
	| declaration SEMICOLON
		{mkm (),Decl $1}
	| macrotype
		{mkm (),$1}
	| SEMICOLON
		{mkm (),DSemicolon}
	| jekdecl
		{mkm (), $1}
	| INCLUDE 
		{mkm (), Include (process_include $1)}
	;

jekdecl
	: J_STDTYPE struct_declaration SEMICOLON
		{StdType $2}
	| C_STDTYPE LPAREN struct_declaration RPAREN SEMICOLON
		{StdType $3}


function_declarator
	: direct_declarator function_parameters
		{fst $1 @ [$2], snd $1}
	| withargs direct_declarator function_parameters
		{fst $2 @ [$3;DWithArgs(snd $1)], snd $2}
	| pointer direct_declarator function_parameters
		{fst $2 @ [$3] @ $1, snd $2}
	| withargs pointer direct_declarator function_parameters
		{fst $3 @ [$4] @ $2 @ [DWithArgs(snd $1)], snd $3}


function_definition
	: declaration_specifiers function_declarator declaration_list compound_statement
		{mkm (),(mkdecl $1 $2,$3,$4)}
	| declaration_specifiers function_declarator compound_statement
		{mkm (),(mkdecl $1 $2,[],$3)}
	| function_declarator declaration_list compound_statement
		{mkm (),(mkdecl [] $1,$2,$3)}
	| function_declarator compound_statement
		{mkm (),(mkdecl [] $1,[],$2)}
	;

function_definition_list
	: function_definition
		{[$1]}
	| function_definition function_definition_list
		{$1 :: $2}
		
