
open Util
open Syntax
open Location
open Prettyutil
open Errors


(* ------------------------------------------------------------
 * Create Metadata
 * ------------------------------------------------------------ *)

let makemeta span = {span = span; tags = []}
			
let nometa () = makemeta (mkspan nowhere nowhere)

let cpm m = makemeta m.span


(* ------------------------------------------------------------
 * Utility functions
 * ------------------------------------------------------------ *)

let add_tag meta tag = meta.tags <- tag :: meta.tags

let remove_tag meta tagfind = 
		meta.tags <- List.filter (fun x -> tagfind x = None) meta.tags 

let has_tag meta tag = member tag meta.tags
let find_tag meta propname dflt f = match list_first_option f meta.tags with
	| Some x -> x
	| None when !Cmdargs.j2c -> 
		error meta.span ("Property missing - "^propname); dflt
	| None -> dflt

let find_maybe_tag meta propname f = list_first_option f meta.tags
	
let with_tags tags meta = 
	{meta with tags = tags @ meta.tags}
	
let merge_tags m1 m2 = 
	{m1 with tags = m1.tags @ m2.tags}
	
	
let add_listtag meta extra tagcons tagfind = 
	match list_first_option tagfind meta.tags with
	| Some xs -> remove_tag meta tagfind;
				add_tag meta (tagcons (extra :: xs))
	| None -> add_tag meta (tagcons [extra])


(* ------------------------------------------------------------
 * Fake Default values
 * ------------------------------------------------------------ *)

let fakestr = "???"
let fakename () = new_namevar fakestr "?" 
let fake_funinfo () = {funname = fakename (); envname = fakename (); 
					strname = fakename (); envvars = []}
let fakeid = nometa (),fakestr
let fake_impl () = (fakeid,fakeid, DictEnv)
let fake_expr () = nometa (),Var fakeid
let fakety = [SCoreHere],TyBasic,[]


(* ------------------------------------------------------------
 * Read a tag
 * ------------------------------------------------------------ *)

let get_type meta = find_tag meta "type" fakety
		(function HasType t -> Some t | _ -> None)
let get_localfun_info meta = find_tag meta "localfun information"
		(fake_funinfo ())
		(function LocalFunInfo n -> Some n | _ -> None)
let get_dict_args meta = find_tag meta "dictargs" []
		(function DictArgs a -> Some a | _ -> None) 
let get_call_dict meta = find_maybe_tag meta "calldict"
		(function CallDict (d,m) -> Some (d,m) | _ -> None)
let get_scrutinee meta = find_tag meta "scrutinee" (fake_expr ())
		(function Scrutinee x -> Some x | _ -> None)
let get_opname meta = find_tag meta "operator name" fakestr
		(function OpName x -> Some x | _ -> None) 
let get_return_type meta = find_tag meta "return type" fakety
		(function ReturnType x -> Some x | _ -> None)
let get_envarg meta = find_maybe_tag meta "envarg tyname" 
		(function Envarg x -> Some x | _ -> None)
let get_tempname meta = find_tag meta "temporary var name" (fakename ())
		(function TempName x -> Some x | _ -> None)
let get_fieldname meta = find_tag meta "field name" fakestr
		(function FieldName x -> Some x | _ -> None)
let get_temps meta = find_tag meta "temporary variables" []
		(function Temps x -> Some x | _ -> None)
let get_fwddecls meta = find_tag meta "foward declarations" []
		(function FwdDecls x -> Some x | _ -> None)
let get_tempdecls meta = find_tag meta "temporary variable declarations" []
		(function TempDecls x -> Some x | _ -> None)
let get_lambdas meta = find_tag meta "lambda expressions" []
		(function Lambdas x -> Some x | _ -> None)
let get_funenv meta = find_maybe_tag meta "function environment"
		(function FunEnv x -> Some x | _ -> None)
let get_argtys meta = find_tag meta "function argument types" []
		(function ArgTys x -> Some x | _ -> None)
let get_needscast meta = find_maybe_tag meta "cast needed"
		(function NeedsCast x -> Some x | _ -> None)
let get_envname meta = find_tag meta "lambda environment" fakestr
		(function EnvName x -> Some x | _ -> None)
let get_is_envvar meta = has_tag meta EnvVar
let get_lamenvs meta = find_tag meta "lambda environments" []
		(function LamEnvs x -> Some x | _ -> None)
let get_sharedpretty meta = find_tag meta "shared prettyprint data" 
		(ref None) 
		(function SharedPretty x -> Some x | _ -> None)
let get_dict_protos meta = find_tag meta "implementation prototypes" []
		(function DictProtos x -> Some x | _ -> None)
let get_iface_protos meta = find_tag meta "interface prototypes" []
		(function IfaceProtos x -> Some x | _ -> None)
let get_dictenvs meta = find_tag meta "dictionary environments" []
		(function DictEnvs x -> Some x | _ -> None)
let get_lang meta = find_tag meta "source language" EncodedC
		(function Lang x -> Some x | _ -> None)
let get_maybehidden meta = find_maybe_tag meta "maybe temp" 
		(function MaybeHidden x -> Some x | _ -> None)


(* ------------------------------------------------------------
 * Add a tag
 * ------------------------------------------------------------ *)

let set_dictenvs meta d = add_tag meta (DictEnvs d)
let set_iface_protos meta p = add_tag meta (IfaceProtos p)
let set_dict_protos meta p = add_tag meta (DictProtos p)
let set_sharedpretty meta p = add_tag meta (SharedPretty p)
let set_lamenvs meta e = add_tag meta (LamEnvs e)
let set_is_envvar meta = add_tag meta EnvVar
let set_envname meta s = add_tag meta (EnvName s)
let set_needscast meta ty = add_tag meta (NeedsCast ty)		
let set_argtys meta argtys = add_tag meta (ArgTys argtys)				
let set_funenv meta funenv = add_tag meta (FunEnv funenv)		
let set_lambdas meta lambdas = add_tag meta (Lambdas lambdas)
let set_return_type meta typ = add_tag meta (ReturnType typ)
let set_localfun_info meta info = add_tag meta (LocalFunInfo info)
let set_tempdecls meta temps = add_tag meta (TempDecls temps)
let set_fwddecls meta fwds = add_tag meta (FwdDecls fwds)		
let set_temps meta temps = add_tag meta (Temps temps)
let set_fieldname meta fieldname = add_tag meta (FieldName fieldname)
let set_tempname meta tempname = add_tag meta (TempName tempname)
let set_dict_args meta dicts = add_tag meta (DictArgs dicts)
let set_calldict meta dictarg meth = add_tag meta (CallDict (dictarg,meth))

let set_type meta typ = add_tag meta (HasType typ)
let set_bodyty meta typ = add_tag meta (BodyTy typ)
let set_scrutvar meta s = add_tag meta (Scrutinee s)
let set_envarg meta envarg = add_tag meta (Envarg envarg)

