 {
	open Location
	open Metadata
	open Parser	
	open Util
	open Parseutils
	open Syntax
	open Scanf
	open Lexing
	open Cmdargs
	open Errors
	exception Eof

let decode_pure_c_name lexbuf =
	let name = Lexing.lexeme lexbuf in
	let a = mktoken lexbuf in
	if is_typename name then TYPE_NAME (a,name)
	else IDENTIFIER (a,name)

let decode_encoded_c_name lexbuf =
	let name = Lexing.lexeme lexbuf in
	let a = mktoken lexbuf in
	match name with
	| "unsafe" -> C_UNSAFE ()
	| "macrotype" -> C_MACROTYPE ()
	| "_implement" -> C_IMPLEMENT ()
	| "_implement_proto" -> C_IMPLEMENT_PROTO ()
	| "_endimplement" -> C_ENDIMPLEMENT ()
	| "ret" -> RET ()
	| "_needs" -> C_NEEDS ()
	| "_void" -> C_VOID ()
	| "_dictenv" -> C_DICTENV ()
	| "stdtype" -> C_STDTYPE ()
	| "_p"	-> C_P ()
	| _ when is_typename name -> TYPE_NAME (a,name)
	| _ when is_tyvar name -> TYVAR (a,name)
	| _ -> IDENTIFIER (a,name) 

let decode_jkl_name lexbuf =
	let name = Lexing.lexeme lexbuf in
	let a = mktoken lexbuf in
	match name with
	| "unsafe" -> J_UNSAFE ()
	| "macrotype" -> J_MACROTYPE ()
	| "interface" -> J_INTERFACE ()
	| "tagged" -> J_TAGGED ()
	| "implement" -> J_IMPLEMENT ()
	| "import" -> J_IMPORT ()
	| "new" -> J_NEW a
	| "alloc" -> J_MALLOC a
	| "stdtype" -> J_STDTYPE ()
	| "ret" -> RET ()
	| _ when is_typename name -> TYPE_NAME (a,name)
	| _ when is_constructor name -> J_CONSTRUCTOR (a,name)
	| _ when is_tyvar name -> TYVAR (a,name)
	| _ -> IDENTIFIER (a,name) 
	
let decode_name lexbuf = match get_current_lang () with
	| Jekyll -> decode_jkl_name lexbuf
	| EncodedC -> decode_encoded_c_name lexbuf
	| PureC -> decode_pure_c_name lexbuf
		
let handle_string_newlines s lexbuf =
	String.iter (fun c -> if c = '\n' then newline lexbuf) s
		
let decode_const s rhs =
	try
		sscanf s "%d" (fun _ -> DKConst (ConstInt rhs,s))
	with _ ->
		DKOpaque s 
		
	
let decode_define_directive lexbuf =
	let s = Lexing.lexeme lexbuf in
	handle_string_newlines s lexbuf;
	add_whitespace lexbuf;
	try
		sscanf s "#define %[^'('' '](" 
		(fun name -> add_define name (DKOpaque s))
	with _ ->
	try
		sscanf s "#define %s %s" 
		(fun name rhs -> add_define name (decode_const s rhs))
	with _ ->
	try
		sscanf s "#define %s (%s)" 
		(fun name rhs -> add_define name (decode_const s rhs))
	with _ ->
	try
		sscanf s "#define %s" 
		(fun name -> add_define name (DKOpaque s))
	with 
		_ -> ()

let decode_undef_directive lexbuf =
	dotoken lexbuf;
	let s = Lexing.lexeme lexbuf in
	handle_string_newlines s lexbuf;
	add_whitespace lexbuf;
	try
		sscanf s "#undef %s" 
		(fun name -> add_define name (DKUndef s))
	with 
		_ -> ()
		
let decode_line_directive lexbuf =
	let s = Lexing.lexeme lexbuf in
	handle_string_newlines s lexbuf;
	try
		sscanf s "# %d \"%[^'\"']\" " 
		(fun i s -> 
			set_current_file s;
			lexstate_set_file_line lexbuf s (i-1) lexbuf.lex_curr_pos;
			add_whitespace lexbuf;
			()
		);			
	with 
		_ -> ()

let warn_asm () = warn_bad_jekyll "inline assembler"
let lfatal msg = fatal (current_span ()) msg

let decode_include_directive lexbuf =
	dotoken lexbuf;
	let incline = Lexing.lexeme lexbuf in
	match scan_line_for_include incline with
	| Some fname -> INCLUDE fname
	| None -> lfatal ("malformed include : "^incline)
	

}

let number = ("0b")?['0'-'9']+("L"|'L'|'S'|'B'|'U'|'l'|'s'|'b'|'u')*
let hexnumber = ("0x")['a'-'f''A'-'F''0'-'9']+('L'|'S'|'B'|'U'|'l'|'s'|'b'|'u')*
let float = ['0'-'9']+('.')['0'-'9']+('e'|'E')?('+'|'-')?['0'-'9']*('F'|'L')?

let name = ['_''a'-'z''A'-'Z']['_''a'-'z''A'-'Z''0'-'9']*
let anything = ['!'-'~']*
let whitespace = [' ''\t''\r']*
let stringlit = "\""([^'\"']|"\\\"")*"\""
let line = ([^'\n''\r']|"\\\n"|"\\\r\n")*
let charlit = "\'" ([^'\'']|("\\" _)) "\'" 
let jkl_tyvar = "%" ['a'-'z']
let c_tyvar = "_v" ['a'-'z']
let operator = ['/''-''+''>''<''=''&''|''$''%''?'':''.''~']*

rule token = parse
	whitespace 	{add_whitespace lexbuf; token lexbuf}
	| "/*"		{add_whitespace lexbuf;
				 comment lexbuf; token lexbuf}
	| "//"		{add_whitespace lexbuf;
				 sline_comment lexbuf; token lexbuf}
	| "\n"		{add_whitespace lexbuf; newline lexbuf; token lexbuf}
	| stringlit	{STRING_LITERAL (tokenlexeme lexbuf)}
	| number		{INT_LITERAL(tokenlexeme lexbuf)}
	| hexnumber 	{INT_LITERAL(tokenlexeme lexbuf)}
	| float		{FLOAT_LITERAL(tokenlexeme lexbuf)}
	| charlit		{INT_LITERAL(tokenlexeme lexbuf)}
	| jkl_tyvar	{TYVAR(mktoken lexbuf, String.sub (Lexing.lexeme lexbuf) 1 1)}
	| c_tyvar	{TYVAR(mktoken lexbuf, String.sub (Lexing.lexeme lexbuf) 2 1)}
	| ("__attribute__" | "__attribute") whitespace "(" 
			{add_whitespace lexbuf; inparen lexbuf; token lexbuf}
	| "asm" | "__asm__" | "__asm" 
			{warn_asm (); 
			 ignore(mktoken lexbuf); infakefun lexbuf; token lexbuf}
	| "__extension__" | "__extension" {add_whitespace lexbuf; token lexbuf}
	| "__inline__" | "__inline" | "inline" {add_whitespace lexbuf; token lexbuf}
	| "__signed__" | "__signed" | "__volatile__" {add_whitespace lexbuf; token lexbuf}
	| "__restrict__" | "__restrict" {add_whitespace lexbuf; token lexbuf}

	| "__cacheline_aligned" | "__cacheline_aligned__" 
				{add_whitespace lexbuf; token lexbuf}

	| "__typeof__" {U_TYPEOF (dotoken lexbuf)}
	| "typeof" {TYPEOF (dotoken lexbuf)} 
	| "sizeof" { SIZEOF (dotoken lexbuf) }
	| "alignof" | "__alignof" | "__alignof__" { ALIGNOF (dotoken lexbuf) }

	| "typedef" { TYPEDEF (dotoken lexbuf) }
	| "extern" { EXTERN (dotoken lexbuf) }
	| "static" { STATIC (dotoken lexbuf) }
	| "inline" { INLINE (dotoken lexbuf) }
	| "auto" { AUTO (dotoken lexbuf) }
	| "register" { REGISTER (dotoken lexbuf) }

	| "char" { CHAR (dotoken lexbuf) }
	| "short" { SHORT (dotoken lexbuf) }
	| "int" { INT (dotoken lexbuf) }
	| "long" { LONG (dotoken lexbuf) }
	| "signed" { SIGNED (dotoken lexbuf) }
	| "unsigned" { UNSIGNED (dotoken lexbuf) }
	| "float" { FLOAT (dotoken lexbuf) }
	| "double" { DOUBLE (dotoken lexbuf) }
	| "void" { VOID (dotoken lexbuf) }
	
	| "const" | "__const" | "__const__" { CONST (dotoken lexbuf) }
	| "volatile" { VOLATILE (dotoken lexbuf) }
	
	| "struct" { STRUCT (dotoken lexbuf) }
	| "union" { UNION (dotoken lexbuf) }
	| "enum" { ENUM (dotoken lexbuf) }
	
	| "case" { CASE (dotoken lexbuf) }
	| "default" { DEFAULT (dotoken lexbuf) }
	| "if" { IF (dotoken lexbuf) }
	| "else" { ELSE (dotoken lexbuf) }
	| "switch" { SWITCH (dotoken lexbuf) }
	| "while" { WHILE (dotoken lexbuf) }
	| "do" { DO (dotoken lexbuf) }
	| "for" { FOR (dotoken lexbuf) }
	| "goto" { GOTO (dotoken lexbuf) }
	| "continue" { CONTINUE (dotoken lexbuf) }
	| "break" { BREAK (dotoken lexbuf) }
	| "return" { RETURN (dotoken lexbuf) }

	| "->" { PTR_OP (dotoken lexbuf) }
	| "++" { INC_OP (dotoken lexbuf) }
	| "--" { DEC_OP (dotoken lexbuf) }

	| "<<" { LEFT_OP (dotoken lexbuf) }
	| ">>" { RIGHT_OP (dotoken lexbuf) }

	| "<=" { LE_OP (dotoken lexbuf) }
	| ">=" { GE_OP (dotoken lexbuf) }
	| "==" { EQ_OP (dotoken lexbuf) }
	| "!=" { NE_OP (dotoken lexbuf) }

	| "&&" { ANDAND_OP (dotoken lexbuf) }
	| "||" { OROR_OP (dotoken lexbuf) }

	| "*=" { MUL_ASSIGN (dotoken lexbuf) }
	| "/=" { DIV_ASSIGN (dotoken lexbuf) }
	| "%=" { MOD_ASSIGN (dotoken lexbuf) }
	| "+=" { ADD_ASSIGN (dotoken lexbuf) }
	| "-=" { SUB_ASSIGN (dotoken lexbuf) }
	| "<<=" { LEFT_ASSIGN (dotoken lexbuf) }
	| ">>=" { RIGHT_ASSIGN (dotoken lexbuf) }
	| "&=" { AND_ASSIGN (dotoken lexbuf) }
	| "^=" { XOR_ASSIGN (dotoken lexbuf) }
	| "|=" { OR_ASSIGN (dotoken lexbuf) }
	
	| "&" { AND_OP (dotoken lexbuf) }
	| "!" { BANG_OP (dotoken lexbuf) }
	| "~" { TILDE_OP (dotoken lexbuf) }
	| "-" { MINUS_OP (dotoken lexbuf) }
	| "+" { PLUS_OP (dotoken lexbuf) }
	| "/" { DIV_OP (dotoken lexbuf) }
	| "%" { MOD_OP (dotoken lexbuf) }
	| ":" { COLON (dotoken lexbuf) }
	| "." { PERIOD (dotoken lexbuf) }
	| "..." { ELLIPSIS (dotoken lexbuf) }

	| "<" { LT_OP (dotoken lexbuf) }
	| ">" { GT_OP (dotoken lexbuf) }
	| "|" { OR_OP (dotoken lexbuf) }
	| "^" { XOR_OP (dotoken lexbuf) }

	| "=" { ASSIGN (dotoken lexbuf) }
	| "?" { QUESTION (dotoken lexbuf) }


	| name		{if is_silentmacro(Lexing.lexeme lexbuf) then
					(add_whitespace lexbuf; token lexbuf)
				else decode_name lexbuf}

	| "*"			{STAR(dotoken lexbuf)}
	| ";"			{SEMICOLON(dotoken lexbuf)}
	| ","			{COMMA(dotoken lexbuf)}
	| "("			{LPAREN(dotoken lexbuf)}
	| ")"			{RPAREN(dotoken lexbuf)}
	| "["			{LSQUARE(dotoken lexbuf)}
	| "]"			{RSQUARE(dotoken lexbuf)}
	| "{"			{LBRACE(dotoken lexbuf)}
	| "}"			{RBRACE(dotoken lexbuf)}

	| "#" whitespace "include" whitespace "<jekyll_1.h>" line "\n" 
				{newline lexbuf; token lexbuf} 

	| "#" whitespace "include" line {decode_include_directive lexbuf}

	| "#" whitespace "define" line {decode_define_directive lexbuf;token lexbuf}

	| "#" whitespace "undef" line {decode_undef_directive lexbuf;token lexbuf}

	| "#" whitespace "if" {parse_error "Jekyll does not currently support #if"}
			
	| "#" line	{decode_line_directive lexbuf;token lexbuf}			

	| eof			{EOF()} 
				 
	| _			{GARBAGE(mktoken lexbuf)}


(* ------------------------------------------------------------
 * Auxilary lexing modes used by the main lexer
 * ------------------------------------------------------------ *)

and infakefun = parse
| "__volatile__" {add_whitespace lexbuf; infakefun lexbuf}
| '('			{add_whitespace lexbuf; inparen lexbuf}
| _			{add_whitespace lexbuf; infakefun lexbuf}

and inparen = parse
| ')'			{add_whitespace lexbuf}
| '('			{add_whitespace lexbuf; inparen lexbuf; inparen lexbuf}
| eof			{lfatal "unclosed parenthesis at end of file"}
| _			{add_whitespace lexbuf; inparen lexbuf}

and comment = parse
| "*/"			{add_whitespace lexbuf}
| "\n"			{newline lexbuf; add_whitespace lexbuf; comment lexbuf}
| eof			{lfatal "unclosed comment at end of file"}
| _				{add_whitespace lexbuf; comment lexbuf}

and sline_comment = parse
| "\n"			{newline lexbuf; add_whitespace lexbuf}
| _				{add_whitespace lexbuf; sline_comment lexbuf}

and consume_defline = parse
| "\n"			{newline lexbuf; []}
| "\\\n"		{newline lexbuf; consume_defline lexbuf}
| line			{dotoken lexbuf; 
				 let s = Lexing.lexeme lexbuf in
				 s :: consume_defline lexbuf}
| _				{parse_error (mktoken lexbuf)}

