
(* ------------------------------------------------------------------------------------------------------------------------
 *
 * Basic data structures and operations on them.
 *
 * This code builds on the standard O'Caml data structures, adding new operations
 * and exposing a more consistent interface.
 *
 * ------------------------------------------------------------------------------------------------------------------------ *)


open Util


module type PRIMSET = sig
	type t 
	type elt 
	val empty : t
	val is_empty : t -> bool
	val choose : t -> elt 
	val mem : elt -> t -> bool
	val add : elt -> t -> t
	val remove : elt -> t -> t
	val subtract : t -> t -> t 
	val union : t -> t -> t
	val inter : t -> t -> t
	val equal : t -> t -> bool
	val subset : t -> t -> bool
	val iter : (elt -> unit) -> t -> unit
	val fold : (elt -> 'a -> 'a) -> t -> 'a -> 'a
	val filter : (elt -> bool) -> t -> t
end

module type SET = sig
	include PRIMSET
	val singleton : elt -> t
	val of_list : elt list -> t
	val of_option : elt option -> t
	val to_list : t -> elt list
	val add_list : elt list -> t -> t
	val union_of_list : t list -> t
	val inter_of_list : t list -> t
	val for_all : (elt -> bool) -> t -> bool
	val count : (elt -> bool) -> t -> int
	val map : (elt -> elt) -> t -> t
end

(* imperative set with destructive update *)
module type PRIMIMPSET = sig
	type t
	type elt
	val mem : elt -> t -> bool
	val add : elt -> t -> unit
	val remove : elt -> t -> unit
	val union : t -> t -> t
	val copy : t -> t
	val of_list : elt list -> t
	val create : unit -> t
	val equal : t -> t -> bool
	val union_of_list : t list -> t
	val subtract_as_list : t -> t -> elt list
	val subtract_from_list : elt list -> t -> elt list
	val empty : t -> unit
	val singleton : elt -> t -> unit
	val to_list : t -> elt list
	
	module EltSet : SET with type elt = elt
	val to_set : t -> EltSet.t
end

module type IMPSET = sig
	include PRIMIMPSET
	val maybe_union : t option -> t option -> t
	val add_list : elt list -> t -> unit
end

(* basic readable map operations - others can be derived from these *)
module type PRIMCOLLECTION = sig
	type 'a t
	type key
	val mem : key -> 'a t -> bool
	val find : key -> 'a t -> 'a
	val iter : (key -> 'a -> unit) -> 'a t -> unit
	val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b 
end

(* primitive pure map *)
module type PRIMMAP = sig
	include PRIMCOLLECTION
	val empty : 'a t
	val add : key -> 'a -> 'a t -> 'a t
end

(* primitive imperative hash collection *)
module type PRIMHASH = sig
	include PRIMCOLLECTION
	val create : unit -> 'a t
	val add : key -> 'a -> 'a t -> unit
	val copy : 'a t -> 'a t
	val clear : 'a t -> unit
	val remove : key -> 'a t -> unit
end

(* extra, derived collection operations *)
module type COLLECTION = sig
	include PRIMCOLLECTION
	val to_list : 'a t -> (key * 'a) list
	val find_dflt : key -> 'a -> 'a t -> 'a
	val maybe_find : key -> 'a t -> 'a option
	val equal_with : ('a -> 'a -> bool) -> 'a t -> 'a t -> bool

	module KeySet : SET with type elt = key
	val keys : 'a t -> KeySet.t
end

module type MAP = sig
	include PRIMMAP

	val of_list : (key * 'a) list -> 'a t
	val map : (key -> 'a -> 'b) -> 'a t -> 'b t
	val map2_dflt : ('a -> 'b -> 'c) -> 'a -> 'b -> 'a t -> 'b t -> 'c t
	val option_map : (key -> 'a -> 'b option) -> 'a t -> 'b t
	val combine : 'a t -> 'a t -> 'a t
	
	(* copied from collection*)	
	val to_list : 'a t -> (key * 'a) list
	val find_dflt : key -> 'a -> 'a t -> 'a
	val maybe_find : key -> 'a t -> 'a option
	val equal_with : ('a -> 'a -> bool) -> 'a t -> 'a t -> bool
	module KeySet : SET with type elt = key
	val keys : 'a t -> KeySet.t
end

module type HASH = sig
	include PRIMHASH

	val of_list : (key * 'a) list -> 'a t
	val find_or_create : key -> (unit -> 'a) -> 'a t -> 'a
	val add_if_missing : key -> (unit -> 'a) -> 'a t -> unit
	val map : (key -> 'a -> 'b) -> 'a t -> 'b t
	val map2_dflt : ('a -> 'b -> 'c) -> 'a -> 'b -> 'a t -> 'b t -> 'c t
	val map_many_dflt : ('a list -> 'b) -> 'a -> 'a t list -> 'b t
	val option_map : (key -> 'a -> 'b option) -> 'a t -> 'b t
	
	(* copied from collection*)	
	val to_list : 'a t -> (key * 'a) list
	val find_dflt : key -> 'a -> 'a t -> 'a
	val maybe_find : key -> 'a t -> 'a option
	val equal_with : ('a -> 'a -> bool) -> 'a t -> 'a t -> bool
	module KeySet : SET with type elt = key
	val keys : 'a t -> KeySet.t

end

module type OrderedType = Set.OrderedType
module type HashType = sig
	type t
	val hash : t -> int
	val compare : t -> t -> int
end

(* a set, hash and map together, with the types working correctly together *)
module type COLLECTIONGROUP = sig
	type t
	module Set : SET with type elt = t	
	module Map : MAP with type key = t and module KeySet = Set
	module Hash : HASH with type key = t and module KeySet = Set 
	module ImpSet : IMPSET with type elt = t
end


module MakeHashedType (H : HashType) = struct
	type t = H.t
	let hash = H.hash
	let equal x y = (H.compare x y = 0)
end


module MakePrimSet (O : OrderedType) = struct
	module Set = Set.Make (O)
	include Set
	let subtract = diff
end

module MakeSet (P : PRIMSET) = struct 
	include P
	
	let singleton elt = add elt empty
	let union_of_list = List.fold_left union empty
	let inter_of_list list = List.fold_left inter (List.hd list) (List.tl list)
	let for_all f set = fold (fun elt b -> f elt && b) set true 
	let of_list = List.fold_left (fun set elt -> add elt set) empty
	let to_list set = fold (fun elt l -> elt::l) set []
	let add_list list set = union (of_list list) set 
	let of_option opt = match opt with 
		| Some x -> singleton x
		| None -> empty
	let count f set = fold (fun elt c -> if f elt then c + 1 else c) set 0
	let map f set = fold (fun elt set -> add (f elt) set) set empty
end

	
module MakePrimImpSet (S : SET) : PRIMIMPSET with type elt = S.elt = struct
	type t = S.t ref
	type elt = S.elt
	module EltSet = S
	let mem elt s = S.mem elt !s
	let add elt s = s := S.add elt !s
	let remove elt s = s := S.remove elt !s
	let of_list list = ref (S.of_list list) 
	let create () = ref S.empty
	let equal x y = S.equal !x !y
	let union x y = ref (S.union !x !y)
	let copy x = ref (!x)
	let union_of_list l = ref (S.union_of_list (List.map (fun s -> !s) l))
	let subtract_as_list x y = S.to_list (S.subtract !x !y)
	let subtract_from_list x y = S.to_list (S.subtract (S.of_list x) !y)
	let to_list x = S.to_list !x
	let empty s = s := S.empty
	let singleton x s = s := S.singleton x
	let to_set x = !x
end

module MakeImpSet (P : PRIMIMPSET) : IMPSET with type elt = P.elt = struct
	include P
	let maybe_union x y = match x, y with
		| Some a, Some b -> union a b
		| Some a, _ -> copy a
		| _, Some b -> copy b
		| _ -> raise InternalError
	let add_list l s = List.iter (fun e -> add e s) l	
end


module MakeCollection (P: PRIMCOLLECTION) (S : SET with type elt = P.key) : COLLECTION 
	with type 'a t = 'a P.t and type key = P.key and module KeySet = S =  struct
	include P
		
	let to_list m = fold (fun key data l -> (key,data)::l) m []
	let find_dflt key dflt m = if mem key m then find key m else dflt
	let maybe_find key m = if mem key m then Some (find key m) else None	

	module KeySet = S
	let keys m = fold (fun key data s -> KeySet.add key s) m KeySet.empty
	
	let equal_with cmp a b =
		let a_keys = keys a in
		let b_keys = keys b in
		(KeySet.equal a_keys b_keys) &&
		(KeySet.for_all (fun k -> cmp (find k a) (find k b)) a_keys)
	
end

module MakePrimMap (O : OrderedType) = struct
	module Map = Map.Make (O)
	include Map
end


module MakeMap (P : PRIMMAP) (S : SET with type elt = P.key) : MAP
	with type 'a t = 'a P.t and type key = P.key and module KeySet = S = struct
	include P

	module C = MakeCollection (P) (S)
	let to_list = C.to_list
	let find_dflt = C.find_dflt
	let maybe_find = C.maybe_find
	let equal_with = C.equal_with
	module KeySet = C.KeySet
	let keys = C.keys
	
	let of_list list = List.fold_left (fun env (key,data) ->	add key data env) empty list

	let map f m = fold (fun k d m -> add k (f k d) m) m empty	
	let map2_dflt f d1 d2 m1 m2 = 
		let keyset = KeySet.union (keys m1) (keys m2) in
		KeySet.fold (fun k m -> add k (f (find_dflt k d1 m1) (find_dflt k d2 m2)) m) keyset empty
	let map_many_dflt f d ms =
		let keyset = KeySet.union_of_list (List.map keys ms) in
		KeySet.fold (fun k m -> add k (f (List.map (find_dflt k d) ms)) m) keyset empty
	let option_map f m = 
		fold (fun k d m -> match f k d with Some x -> add k x m | None -> m) m empty

	let first_binding x y k = if mem k x then find k x else find k y
	let combine x y = 
		let keyset = KeySet.union (keys x) (keys y) in
		KeySet.fold (fun k m -> add k (first_binding x y k) m) keyset empty
end


module MakePrimHash (H : HashType) = struct
	module Hash = Hashtbl.Make (MakeHashedType(H))
	type key = Hash.key
	type 'a t = 'a Hash.t
	
	(* changed order of args so consistent with Map *)
	let mem key hash = Hash.mem hash key
	let find key hash = Hash.find hash key
	let iter = Hash.iter 
	let fold = Hash.fold
	let copy = Hash.copy
	let create () = Hash.create 10
	let add key data hash = Hash.replace hash key data
	let clear hash = Hash.clear hash
	let remove key hash = Hash.remove hash key
end

module MakeHash (P : PRIMHASH) (S : SET with type elt = P.key) : HASH
	with type 'a t = 'a P.t and type key = P.key and module KeySet = S = struct
	include P
	
	module C = MakeCollection (P) (S)
	let to_list = C.to_list
	let find_dflt = C.find_dflt
	let maybe_find = C.maybe_find
	let equal_with = C.equal_with
	module KeySet = C.KeySet
	let keys = C.keys

	let of_list list = 
		let hash = create () in
		List.iter (fun (key,data) -> add key data hash) list;
		hash

	let find_or_create key creator hash = 
		if mem key hash then
			find key hash 
		else begin
			let initval = creator () in
			add key initval hash;
			initval
		end
		
	let add_if_missing key creator hash = 
		if not (mem key hash) then add key (creator ()) hash
	
	let map f hash = 
		let newhash = create () in
		iter (fun key data -> add key (f key data) newhash) hash;
		newhash	
	let map2_dflt f d1 d2 m1 m2 = 
		let newhash = create () in
		let keyset = KeySet.union (keys m1) (keys m2) in
		KeySet.iter (fun k -> add k (f (find_dflt k d1 m1) (find_dflt k d2 m2)) newhash) keyset;
		newhash
	let map_many_dflt f d ms =
		let newhash = create () in
		let keyset = KeySet.union_of_list (List.map keys ms) in
		KeySet.iter (fun k -> add k (f (List.map (find_dflt k d) ms)) newhash) keyset;
		newhash

	let option_map f m = 
		let newhash = create () in
		iter (fun key data -> match f key data with Some x -> add key x newhash | None -> ()) m;
		newhash

end	

module MakeCollections (H : HashType) : COLLECTIONGROUP with type t = H.t = struct
	type t = H.t
	module Set = MakeSet (MakePrimSet (H))
	module Map = MakeMap (MakePrimMap (H)) (Set)
	module Hash = MakeHash (MakePrimHash (H)) (Set)
	module ImpSet = MakeImpSet (MakePrimImpSet (Set))
end

type 'a topbot = Bot | Val of 'a | Top

	
let topbot_lub f x y = match x,y with
	| Val a, Val b -> f a b 
	| Top, _ -> Top
	| _, Top -> Top
	| Bot, _ -> y
	| _, Bot -> x

let pair_equal eq_x eq_y (x1,y1) (x2,y2) = eq_x x1 x2 && eq_y y1 y2	
	

(* sets of ints *)

module IntOrd = struct
	type t = int
	let compare = compare
end

module StringOrd = struct
	type t = string
	let compare = compare
	let hash = Hashtbl.hash
end


module IntCols = MakeCollections (struct type t = int let hash = Hashtbl.hash let compare = compare end)
module IPairCols = MakeCollections (struct type t = int*int let hash = Hashtbl.hash let compare = compare end)
module StringCols = MakeCollections (StringOrd)
module StringPairCols = MakeCollections (struct type t = string*string let hash = Hashtbl.hash let compare = compare end)

module IntSet = IntCols.Set
module IntMap = IntCols.Map
module IntHash = IntCols.Hash
module IntImpSet = IntCols.ImpSet

module IPairSet = IPairCols.Set
module IPairMap = IPairCols.Map
module IPairHash = IPairCols.Hash
module IPairImpSet = IPairCols.ImpSet

module StringSet = StringCols.Set
module StringMap = StringCols.Map
module StringHash = StringCols.Hash

type intset = IntSet.t
type 'a inthash = 'a IntHash.t 
type 'a intmap = 'a IntMap.t
type intimpset = IntImpSet.t
type ipairimpset = IPairImpSet.t

type stringset = StringSet.t
type 'a stringhash = 'a StringHash.t
type 'a stringmap = 'a StringMap.t
