subroutine mosaic_set_header(line,error)
  use gkernel_interfaces
  use image_def
  use clean_arrays
  use clean_default
  use imager_interfaces, except_this => mosaic_set_header
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER    
  !     Support for MOSAIC [Ra Dec Unit [ANGLE Angle]] command
  !
  ! Define the associated MOSAIC Table to enable use of the Sault et al
  ! method for Mosaic imaging, using a smaller image size for each field.
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line    ! Command line
  logical, intent(inout) :: error         ! Error flag
  !
  ! Constants
  character(len=*), parameter :: rname='MOSAIC'
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  !
  ! Local
  logical :: lshift, print
  real(8) :: newabs(3), freq
  real(4) :: uvmax, uvmin
  !
  ! Code
  error = .false.
  if (huv%loca%size.eq.0) return
  !
  ! Code valid only if there is some UV data
  print = sic_present(0,1)
  !
  ! 1) Define MAP_CENTER if any specified
  newabs = [huv%gil%a0,huv%gil%d0,huv%gil%pang]
  if (print) call map_center(line,rname,huv,lshift,newabs,error)
  if (error) return
  ! 2) Set the TABLE
  if (.not.allocated(hmosaic)) then
    call init_fields(print,error)
  else
    huv%r2d => duv
    call load_fields(rname,huv,abs(themap%nfields),hmosaic,error)
  endif
  if (error) return
  !
  ! 3) Use MAP_CENTER
  call change_fields(rname,abs(themap%nfields),hmosaic,newabs,error)    
  if (error) return
  !
  ! 4) Define MAP_PARAMETERS
  call uvgmax(huv,huv%r2d,uvmax,uvmin)
  freq = gdf_uv_frequency(huv)
  uvmin = uvmin*freq*f_to_k
  uvmax = uvmax*freq*f_to_k
  call map_copy_par(default_map,themap)
  call map_parameters(rname,themap,huv,freq,uvmax,uvmin,error,print=print) 
  !
  if (.not.user_method%mosaic) then
    user_method%mosaic = .true.
    call gprompt_set('MOSAIC')
  endif
end subroutine mosaic_set_header
!
subroutine init_fields(print,error)
  use gkernel_interfaces
  use image_def
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, only : load_fields, map_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER    
  !     Support routine for the Sault et al Mosaicking method
  !
  ! Define the associated MOSAIC Table and load the Field coordinates
  ! (Pointing and Phases)
  !---------------------------------------------------------------------
  logical, intent(in) :: print
  logical, intent(inout) :: error
  !
  ! Constants
  character(len=*), parameter :: rname='POINTING'
  !
  ! Local
  integer :: ier
  !
  ! Code
  call imager_tree('INIT_FIELDS')
  error = .false.
  if (.not.allocated(duvi)) then
    call map_message(seve%e,rname,'No UV data loaded')
    error = .true.
    return
  endif
  !
  ! 1) Set the TABLE
  if (allocated(hmosaic)) then
    deallocate(hmosaic,stat=ier)
    if (ier.ne.0) then 
      call map_message(seve%e,rname,'Mosaic Header Deallocation error') 
      error = .true.
      return
    endif
  endif
  !
  huv%r2d => duv ! Make sure it works
  call load_fields(rname,huv,abs(themap%nfields),hmosaic,error)
  if (error) return
  !
  mosaic_mode = 'SAULT'
  !
end subroutine init_fields
! Sault et al Mosaicking method 
!   See Sault, Staveley-Smith and Brouw
!   A&A Supp Ser 120, 375-384 (1995)
!
! The principle of the Sault & al Mosaicking technique is
! to minimize spatial distortion and intermediate image
! sizes by
!   - limiting each pointing field to about 2-3 primary beams
!   - and accordingly, limiting each individual dirty beam
!     in the same way
!   - properly reprojecting each pointing to the common tangent
!     plane
!   - phasing each pointing to the nearest pixel as a Phase center
!
! For proper use, it must carry along the individual Pixel shifts
! for each field, and the fractional offset centers.
!
! The gain in size can be substantial for Large mosaics.
! E.g. for a NxN square mosaic, about (N/4)^2
! A similar gain occurs in computation, since FFTs scale 
!   as M log(M)
! This also shows that there is no need for this method for
! "small" mosaics, as the gain only start for 16 pointings
! at least.
!
! Unfortunately, the Gildas Data Format does not easily accomodate
! a "pixel offset" compared to some reference.  This information can 
! readily be retrieved from an absolute coordinate difference,
! but there is no simple way to have it in e.g. a 3-D data cube
! of different pointings.
! 
! This handling is done in the code, through the HMOSAIC allocatable
! structure, but not stored externally so far, unless we can associate 
! some additional "pseudo-table" to data sets. 
!   This could be done
!     - either by a modification of the Gildas Data Format, similar
!       to the "Telescope" section or "Frequencies" section, by
!       adding a "Mosaic" section with the relevant information
!     - or by convention, writing the (internal) HMOSAIC structure
!       to a Gildas pseudo-table, e.g. 'name'.mosaic, to handle
!       the appropriate information. Command WRITE PRIMARY 
!       and READ PRIMARY could do automatically do this.
!
! The HMOSAIC structure contains for each field
!       Real(8) The absolute Pointing Centers (Ad,Dd) 
!       Real(8) The Phase tracking centers (A0,D0) 
!       integer The number of visibilities
!       Real(4) The offset Pointings relative to Mosaic center
!       Real(4) The Phase offsets relative to Mosaic center
!       Integer The Pixel offsets
!       Real(4) The fractional pixel Phase offsets (in ")
! The last 3 items (2 values each) are derived from the 
! the absolute values when imaging, with a given
! Pixel size and Mosaic center. This is done by the 
! MOSAIC_SET_HEADER routine. 
!
! The output pseudo-table should be Real(8) to hold the absolute 
! coordinates and be able to represent the HMOSAIC structure.
! 
!   The Mosaicked Image would then have a different size than
! the Primary & Dirty beams. 
!   The SHOW FIELDS tool could use directly this Table, 
! if it contains some idea of the Primary beam size, which 
! means the pseudo-table should have a complete Gimdas header,
! which could also serve as checking facts (e.g. pixel sizes,
! Mosaic center)
!
!   When using this imaging mode, the UV table is modified to
! have a field ID column, rather than either the Phase or 
! Pointing offsets columns. Note that this mode can even
! handle simultaneously Pointing and Phase offsets columns
! in case they would differ.
!
!   This structure is closer to the UVFITS handling convention
! (AIPS or CASA "Pointing" table)
!
!-----------------------------------------------------------------
subroutine load_fields(rname,huv,mfi,hmos,error)
  use gkernel_interfaces
  use gkernel_types
  use clean_def
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER    
  !     Scan a Mosaic UV data to find out the number of fields
  !     and their positions, and replace the corresponding columns
  !     by Fields ID numbers. 
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname     ! Calling command
  type(gildas), intent(inout) :: huv        ! UV header & data 
  integer, intent(in) :: mfi                ! Number of fields
  type(mosaic_par), intent(inout), allocatable :: hmos(:) ! Mosaic header
  logical, intent(inout) :: error
  ! 
  ! Local 
  integer :: np                   ! Size of a visibility
  integer :: nv                   ! Number of visibilities
  integer :: ixoff,iyoff          ! X,Y Pointing pointer
  integer :: iloff,imoff          ! L,M Phase pointer
  integer :: ioff,joff            ! Actual offset pointer
  integer :: iv, ier, ip, id
  integer :: ifi, kfi, nfi
  real(4), allocatable :: doff(:,:)
  real(8), allocatable :: poff(:,:), pabs(:,:)
  integer, allocatable :: dvis(:)
  type(projection_t) :: proj
  !
  ! Code
  call imager_tree('LOAD_FIELDS')
  !
  ixoff = huv%gil%column_pointer(code_uvt_xoff)
  iyoff = huv%gil%column_pointer(code_uvt_yoff)
  iloff = huv%gil%column_pointer(code_uvt_loff)
  imoff = huv%gil%column_pointer(code_uvt_moff)
  !
  if ((ixoff*iyoff).ne.0) then
    ! Usual Pointing Offsets Mosaic. A Common Phase center
    ioff = ixoff
    joff = iyoff
  else if ((iloff*imoff).ne.0) then
    ! Phase Offsets Mosaic. Pointing & Phase centers match.
    ioff = iloff
    joff = imoff
  else   
    if (huv%gil%column_pointer(code_uvt_id).eq.0) then
      call map_message(seve%e,rname,'Data is not a Mosaic')
      error = .true.
    else if (allocated(hmos)) then
      ! More checks needed (Number of fields at least ?) 
      id = huv%gil%column_pointer(code_uvt_id)
      ip = maxval(huv%r2d(id,:))
      if (ip.ne.mfi) then
        call map_message(seve%e,rname,'Mosaic and Mosaic Header mismatch')
        error = .true.
      else
        call map_message(seve%w,rname,'Mosaic already has an associated Mosaic Header')
      endif
    else
      call map_message(seve%e,rname,'Mosaic with no associated Mosaic Header')
      error = .true.
    endif
    return
  endif
  !
  if (allocated(hmos)) deallocate(hmos)
  !
  np = huv%gil%dim(1)   ! Size of a visibility
  nv = huv%gil%nvisi    ! Number of visibilities
  allocate(doff(nv,2),dvis(nv),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  ! Scan how many fields
  nfi = 1  
  doff(1,1) = huv%r2d(ioff,1)
  doff(1,2) = huv%r2d(joff,1)
  huv%r2d(ioff,1) = nfi
  dvis(1) = 1
  !
  do iv=2,nv
    kfi = 0
    do ifi=1,nfi
      if (huv%r2d(ioff,iv).eq.doff(ifi,1) .and. &
      & huv%r2d(joff,iv).eq.doff(ifi,2) ) then
        kfi = ifi
        huv%r2d(ioff,iv) = ifi
        dvis(ifi) = dvis(ifi)+1
        exit
      endif
    enddo
    !
    ! New field
    if (kfi.eq.0) then
      nfi = nfi+1
      doff(nfi,1) = huv%r2d(ioff,iv)
      doff(nfi,2) = huv%r2d(joff,iv)
      huv%r2d(ioff,iv) = nfi
      dvis(nfi) = 1
      !! Print *,'New field ',nfi,' at ',iv
    endif
  enddo
  if (nfi.eq.1) then
    call map_message(seve%w,rname,'Degenerate mosaic with 1 field only')
    return
  endif
  !
  if (nfi.ne.mfi) Print *,'I am in trouble ...', nfi, mfi
  allocate(hmos(nfi),pabs(nfi,2),poff(nfi,2),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Mosaic Header Memory allocation error')
    error = .true.
    return
  endif
  !
  poff(:,:) = doff(1:nfi,:)
  !
  ! Define the projection about the Tangent point
  call gwcs_projec(huv%gil%a0,huv%gil%d0,huv%gil%pang,huv%gil%ptyp,proj,error)
  call rel_to_abs (proj,poff(:,1),poff(:,2),pabs(:,1),pabs(:,2),nfi)
  !
  ! Store
  if ((iloff*imoff).ne.0) then
    ! A Pointing & Phase Centers match
    do ip=1,nfi
      hmos(ip)%apoint(1:2) = pabs(ip,1:2)
      hmos(ip)%aphase(1:2) = pabs(ip,1:2)
      hmos(ip)%opoint(1:2) = poff(ip,1:2)
      hmos(ip)%ophase(1:2) = poff(ip,1:2)
      hmos(ip)%nvisi = dvis(ip)
    enddo
    huv%gil%column_pointer(code_uvt_loff) = 0
    huv%gil%column_pointer(code_uvt_moff) = 0
    huv%gil%column_size(code_uvt_loff) = 0
    huv%gil%column_size(code_uvt_moff) = 0
  else
    ! Phase Center was common
    ! A Pointing & Phase Centers match
    do ip=1,nfi
      hmos(ip)%apoint(1:2) = pabs(ip,1:2)
      hmos(ip)%aphase(1:2) = [huv%gil%a0,huv%gil%d0]
      hmos(ip)%opoint(1:2) = poff(ip,1:2)
      hmos(ip)%ophase(1:2) = 0.
      hmos(ip)%nvisi = dvis(ip)
    enddo
    huv%gil%column_pointer(code_uvt_xoff) = 0
    huv%gil%column_pointer(code_uvt_yoff) = 0
    huv%gil%column_size(code_uvt_xoff) = 0
    huv%gil%column_size(code_uvt_yoff) = 0
  endif
  do ip=1,nfi
    hmos(ip)%jxy = 0
    hmos(ip)%oxy = 0
  enddo
  deallocate(poff,pabs,dvis,doff)
  !
  ! This leaves an Empty column at end
  huv%gil%column_pointer(code_uvt_id) = ioff
  huv%gil%column_size(code_uvt_id) = 1
end subroutine load_fields
!
subroutine mosaic_show
  use clean_arrays
  use gbl_message
  use imager_interfaces, only : map_message
  use gkernel_interfaces, only : rad2sexa
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER    
  !     Support for SHOW MOSAIC or MOSAIC ?
  !
  ! Show the MOSAIC Table values 
  !---------------------------------------------------------------------
  !
  ! Constants
  character(len=*), parameter :: rname='SHOW'
  real, parameter :: sec=180.*3600./acos(-1.0)
  !
  ! Local
  character(len=14) :: chra,chde,cha0,chd0
  character(len=80) :: mess
  integer :: nf, if
  !
  ! Code
  nf = abs(themap%nfields)
  if (nf.eq.0) then
    call map_message(seve%i,rname,'Only a single field')
    return
  endif
  !
  if (.not.allocated(hmosaic)) then
    write(mess,'(A,I0,A)') 'Mosaic of ',nf,' fields with no Mosaic Header'
    call map_message(seve%i,rname,mess)
    return
  endif
  write(mess,'(A,I0,A)') 'Mosaic of ',nf,' fields'
  !
  write(6,'(A)') "Field    Ra              Dec           A0              D0           Nvisi"
  do if=1,nf
    call rad2sexa (hmosaic(if)%apoint(1),24,chra)
    call rad2sexa (hmosaic(if)%apoint(2),360,chde)
    call rad2sexa (hmosaic(if)%aphase(1),24,cha0)
    call rad2sexa (hmosaic(if)%aphase(2),360,chd0)
    write(6,'(I4,4(1X,A),I9)') if,chra,chde,cha0,chd0,hmosaic(if)%nvisi
  enddo
  write(6,'(A)') "Field     dRa     dDec     dA0      dD0    DeltaX   DeltaY    Jx    Jy  Noise"
  do if=1,nf
    write(6,'(I4,4(1X,F8.2),2(1X,F8.3),2(1X,I5),1X,1PG10.3)') if, & 
      & hmosaic(if)%opoint(1)*sec, hmosaic(if)%opoint(2)*sec, & 
      & hmosaic(if)%ophase(1)*sec, hmosaic(if)%ophase(2)*sec, & 
      & hmosaic(if)%oxy(1)*sec, hmosaic(if)%oxy(2)*sec, & 
      & hmosaic(if)%jxy(1), hmosaic(if)%jxy(2), hmosaic(if)%sigma
  enddo
end subroutine mosaic_show
!
subroutine change_fields(rname,nfi,hmos,newabs,error)
  use gkernel_interfaces
  use gkernel_types
  use clean_def
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER    
  !     Change the associated Mosaic Table to a new Reference center
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname       ! Calling command
  integer, intent(in) :: nfi                  ! Number of fields
  type(mosaic_par), intent(inout) :: hmos(:)  ! Mosaic Header
  real(8), intent(in) :: newabs(3)            ! Coordinates of center
  logical, intent(inout) :: error             ! Error flag
  !
  ! Local
  type(projection_t) :: proj
  real(8) :: ra(nfi), dec(nfi), offx(nfi), offy(nfi) ! Automatic arrays
  !
  ! Code
  !  
  ! Define the projection about the Projection center
  call gwcs_projec(newabs(1),newabs(2),newabs(3),p_azimuthal,proj,error)
  if (error) return
  !
  ra = hmos(:)%apoint(1)
  dec = hmos(:)%apoint(2)
  call abs_to_rel (proj,ra,dec,offx,offy,nfi)
  hmos(:)%opoint(1) = offx
  hmos(:)%opoint(2) = offy
  !
  ra = hmos(:)%aphase(1)
  dec = hmos(:)%aphase(2)
  call abs_to_rel (proj,ra,dec,offx,offy,nfi)
  hmos(:)%ophase(1) = offx
  hmos(:)%ophase(2) = offy
  !
end subroutine change_fields
!
subroutine fraction_fields(rname,hmos,map,mx,my,nx,ny,error)
  use gkernel_interfaces
  use gkernel_types
  use clean_def
  use gbl_message
  use imager_interfaces, only : map_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !
  ! IMAGER    
  !     Change the associated Mosaic Header to a new Reference
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname  ! Input task name (UV_MAP or MX)
  type (uvmap_par), intent(in) :: map    ! Map parameters
  type(mosaic_par), intent(inout) :: hmos(:)   ! Mosaic header
  integer, intent(in) :: mx,my,nx,ny     ! Image sizes
  logical, intent(inout) :: error
  !
  ! Constants
  real(8), parameter :: sec=180D0*3600d0/acos(-1D0)
  !
  ! Local
  integer :: if, nfield, idx, jdx, idy, jdy
  real(8) :: rdx, rdy, xinc, yinc
  !
  ! Code
  if (any(map%xycell.eq.0)) then
    call map_message(seve%e,rname,'Map cell not initialized')
    error = .true.
    return
  endif
  !
  ! This could be different between Phase & Pointing offsets tables ?
  nfield = abs(map%nfields)
  xinc =  -dble(map%xycell(1))              ! Sign is reversed along X
  yinc =  dble(map%xycell(2))
  do if=1,nfield
    idx = idnint(hmos(if)%opoint(1)/xinc)   ! X Pixel of the center
    rdx = hmos(if)%opoint(1)-idx*xinc       ! Offset of phase center (< 1 pixel)
    jdx = (mx-nx)/2+idx                     ! Pixel shift
    !
    idy = idnint(hmos(if)%opoint(2)/yinc)   ! Y Pixel of the center
    rdy = hmos(if)%opoint(2)-idy*yinc       ! Offset of phase center (< 1 pixel)
    jdy = (my-ny)/2+idy                     ! Pixel shift
    !
    hmos(if)%jxy(1)  = jdx
    hmos(if)%jxy(2)  = jdy
    hmos(if)%oxy(1)  = rdx
    hmos(if)%oxy(2)  = rdy
  enddo
end subroutine fraction_fields
!
subroutine sault_uv_change(nu,nv,visi,ap,dp,a0,d0)  
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !     Support for Mosaicking in the Sault et al approach
  !
  ! Figure out 
  !     (u)        (u')
  !     ( )  = R2T (  )
  !     (v)  =     (v') 
  ! the re-projection matrix that transforms the
  ! original (u',v') coordinates relative to the
  ! delay tracking center into the appropriate
  ! (u,v) coordinates relative to the final projection center
  !
  ! See Sault, Staveley-Smith and Brouw
  ! A&A Supp Ser 120, 375-384 (1995)
  !
  !---------------------------------------------------------------------
  integer :: nu   ! Visibility size
  integer :: nv   ! Number of visibilities
  real(4), intent(inout) :: visi(nu,nv)
  real(8), intent(in) :: ap, dp  ! The original phase/pointing center
  real(8), intent(in) :: a0, d0  ! The common projection center
  !
  ! Local
  real(8) :: a11, a21, a12, a22, detn
  integer :: iv
  real(8) :: up, vp
  !
  ! Code 
  !
  ! The R2T matrix elements are
  ! - Determinant
  detn = 1.d0/(sin(d0)*sin(dp) + cos(ap-a0)*cos(dp)*cos(d0))
  ! - Coefficients,
  ! Normalized to appropriate determinant
  a11 = (cos(ap-a0)*sin(dp)*sin(d0)+cos(dp)*cos(d0))*detn
  a21 = -sin(ap-a0)*sin(d0)*detn
  a12 = sin(ap-a0)*sin(dp)*detn
  a22 = cos(ap-a0)*detn
  !
  do iv=1,nv
    up = visi(1,iv)
    vp = visi(2,iv)
    visi(1,iv) = up*a11 + vp*a21
    visi(2,iv) = up*a12 + vp*a22
  enddo
end subroutine sault_uv_change
