/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of H4H5TOOLS. The full H4H5TOOLS copyright notice,      *
 * including terms governing use, modification, and redistribution, is       *
 * contained in the files COPYING and Copyright.html.  COPYING can be found  *
 * at the root of the source code distribution tree; Copyright.html can be   *
 * found at the root level of an installed copy of the electronic H4H5TOOLS  *
 * document set, is linked from the top-level documents page, and can be     *
 * found at http://www.hdfgroup.org/h4toh5/Copyright.html.  If you do not    *
 * have access to either file, you may request a copy from help@hdfgroup.org.*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#ifndef _HDFEOS2_H
#define _HDFEOS2_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include "mfhdf.h"
#include "hdf.h"
#include "HdfEosDef.h"

#include "cvct.h"

/** 
 * @brief vector-like structures
 * the following preprocessor statements create C++ vector like structures and functions.
 * e.g. DECLARE_VECTOR_ALL(challoc, char) will make bunch of functions whose names are
 * begin with challoc_, and a structure challoc which contains 's' field for actual data and
 * 'len' field for the length.
 */
DECLARE_VECTOR_ALL(challoc, char)
DECLARE_VECTOR_ALLOCATOR(float64alloc, float64)
DECLARE_VECTOR_ALLOCATOR(int32alloc, int32)
DECLARE_VECTOR_ALLOCATOR(intalloc, int)

typedef struct {
  challoc name;
} namelist_t;
void init_namelist_t(namelist_t *e);
void free_namelist_t(namelist_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(namelistalloc, namelist_t)

/** 
 * @brief a dimension in EOS2 is a pair of dimension name and size
 */
typedef struct {
  challoc name;
  int32 dimsize;
} dimension_t;
void init_dimension_t(dimension_t *e);
void free_dimension_t(dimension_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(dimensionalloc, dimension_t)

/** 
 * @brief EOS2 Swath Datafield and Geofield are mapped to this structure
 */
typedef struct {
  challoc name;
  int32 rank;
  int32 type;
  dimensionalloc dims;
  challoc data;
  challoc filler;
} fieldinfo_t;
void init_fieldinfo_t(fieldinfo_t *e);
void free_fieldinfo_t(fieldinfo_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(fieldinfoalloc, fieldinfo_t)

/** 
 * @brief EOS2 attributes
 */
typedef struct {
  challoc name;
  int32 type;
  challoc value;
} attribute_t;
void init_attribute_t(attribute_t *e);
void free_attribute_t(attribute_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(attributealloc, attribute_t)

/** 
 * @brief Grid information
 */
typedef struct {
  int32 xdim;
  int32 ydim;
  float64 upleft[2];
  float64 lowright[2];
} gridinfo_t;
void init_gridinfo_t(gridinfo_t *e);
void free_gridinfo_t(gridinfo_t *e);

/** 
 * @brief Grid projection information plus converted lons/lats
 */
typedef struct {
  int32 proj;
  int32 zone;
  int32 sphere;
  float64 param[16];
  int32 pix;
  int32 origin;
  float64alloc lons; /* this field is generated by convert_proj */
  float64alloc lats; /* this field is generated by convert_proj */
} projinfo_t;
void init_projinfo_t(projinfo_t *e);
void free_projinfo_t(projinfo_t *e);

/** 
 * @brief one big data structure that holds everything in one EOS2 Grid
 */
typedef struct {
  challoc name;
  gridinfo_t info;
  projinfo_t proj;
  dimensionalloc dim;
  fieldinfoalloc field;
  attributealloc attr;
} grid_t;
void init_grid_t(grid_t *e);
void free_grid_t(grid_t *e);

/** 
 * @brief EOS2 dimension mapping
 */
typedef struct {
  challoc data;
  challoc geo;
  int32 offset;
  int32 increment;
} dimmap_t;
void init_dimmap_t(dimmap_t *e);
void free_dimmap_t(dimmap_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(dimmapalloc, dimmap_t)

/** 
 * @brief EOS2 index mapping
 */
typedef struct {
  challoc geo;
  challoc data;
  int32alloc indices;
} indexmap_t;
void init_indexmap_t(indexmap_t *e);
void free_indexmap_t(indexmap_t *e);
DECLARE_VECTOR_ALLOCATOR_FREE(indexmapalloc, indexmap_t)

/** 
 * @brief one big datastructure that holds everything in one EOS2 swath
 */
typedef struct {
  challoc name;
  dimensionalloc dim;
  dimmapalloc dimmap;
  indexmapalloc index;
  fieldinfoalloc geofield;
  fieldinfoalloc datafield;
  attributealloc attr;
} swath_t;
void init_swath_t(swath_t *e);
void free_swath_t(swath_t *e);


int parse_gridnames(const char *filename, namelistalloc *names);
int32 open_grids(const char *filename);
int read_grid(int32 readfd, const char *gridname, grid_t *grid);
void close_grids(int32 readfd);

int parse_swathnames(const char *filename, namelistalloc *names);
int32 open_swaths(const char *filename);
int read_swath(int32 readfd, const char *swathname, swath_t *swath);
void close_swaths(int32 readfd);

#ifdef __cplusplus
}
#endif

#endif

// vim:ts=8:sw=2:sts=2:cindent
