package github

import (
	"encoding/json"
	"testing"
	"time"

	"github.com/trufflesecurity/trufflehog/v3/pkg/analyzer/analyzers"
	"github.com/trufflesecurity/trufflehog/v3/pkg/common"
	"github.com/trufflesecurity/trufflehog/v3/pkg/context"
)

func TestAnalyzer_Analyze(t *testing.T) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	testSecrets, err := common.GetSecret(ctx, "trufflehog-testing", "detectors4")
	if err != nil {
		t.Fatalf("could not get test secrets from GCP: %s", err)
	}

	tests := []struct {
		name    string
		key     string
		want    string // JSON string
		wantErr bool
	}{
		{
			name: "v2 ghp",
			key:  testSecrets.MustGetField("GITHUB_VERIFIED_GHP"),
			want: `{
              "AnalyzerType": 0,
              "Bindings": [
                {
                  "Resource": {
                    "Name": "truffle-sandbox",
                    "FullyQualifiedName": "github.com/truffle-sandbox",
                    "Type": "user",
                    "Metadata": null,
                    "Parent": null
                  },
                  "Permission": {
                    "Value": "notifications",
                    "AccessLevel": "",
                    "Parent": null
                  }
                },
                {
                  "Resource": {
                    "Name": "public gist",
                    "FullyQualifiedName": "gist.github.com/truffle-sandbox/fecf272c606ddbc5f8486f9c44821312",
                    "Type": "gist",
                    "Metadata": null,
                    "Parent": {
                      "Name": "truffle-sandbox",
                      "FullyQualifiedName": "github.com/truffle-sandbox",
                      "Type": "user",
                      "Metadata": null,
                      "Parent": null
                    }
                  },
                  "Permission": {
                    "Value": "notifications",
                    "AccessLevel": "",
                    "Parent": null
                  }
                }
              ],
              "UnboundedResources": null,
              "Metadata": {
                "expiration": "0001-01-01T00:00:00Z",
                "fine_grained": false,
                "type": "Classic GitHub Personal Access Token"
              }
            }`,
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			a := Analyzer{}
			got, err := a.Analyze(ctx, map[string]string{"key": tt.key})
			if (err != nil) != tt.wantErr {
				t.Errorf("Analyzer.Analyze() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			// Marshal the actual result to JSON
			gotJSON, err := json.Marshal(got)
			if err != nil {
				t.Fatalf("could not marshal got to JSON: %s", err)
			}

			// Parse the expected JSON string
			var wantObj analyzers.AnalyzerResult
			if err := json.Unmarshal([]byte(tt.want), &wantObj); err != nil {
				t.Fatalf("could not unmarshal want JSON string: %s", err)
			}

			// Marshal the expected result to JSON (to normalize)
			wantJSON, err := json.Marshal(wantObj)
			if err != nil {
				t.Fatalf("could not marshal want to JSON: %s", err)
			}

			// Compare the JSON strings
			if string(gotJSON) != string(wantJSON) {
				// Pretty-print both JSON strings for easier comparison
				var gotIndented, wantIndented []byte
				gotIndented, err = json.MarshalIndent(got, "", "  ")
				if err != nil {
					t.Fatalf("could not marshal got to indented JSON: %s", err)
				}
				wantIndented, err = json.MarshalIndent(wantObj, "", "  ")
				if err != nil {
					t.Fatalf("could not marshal want to indented JSON: %s", err)
				}
				t.Errorf("Analyzer.Analyze() = %s, want %s", gotIndented, wantIndented)
			}
		})
	}
}
