/*
 * Copyright (c) 2021-present Fabien Potencier <fabien@symfony.com>
 *
 * This file is part of Symfony CLI project
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package main

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"os"

	"github.com/hashicorp/go-version"
)

func main() {
	generateLatestPhpVersion()
}

func generateLatestPhpVersion() {
	resp, err := http.Get("https://www.php.net/releases/active.php")
	if err != nil {
		panic(err)
	}
	defer resp.Body.Close()

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		panic(err)
	}

	var result map[int]map[string]struct {
		Announcement bool
		LatestMinor  string `json:"version"`
	}

	if err := json.Unmarshal(body, &result); err != nil {
		panic(err)
	}

	var latestVersion *version.Version

	for _, versions := range result {
		for _, versionInfo := range versions {
			if !versionInfo.Announcement {
				continue
			}

			ver, err := version.NewVersion(versionInfo.LatestMinor)
			if err != nil {
				panic(err)
			}

			if latestVersion == nil || ver.GreaterThan(latestVersion) {
				latestVersion = ver
			}
		}
	}

	f, err := os.Create("php_version.go")
	if err != nil {
		panic(err)
	}
	f.WriteString(`// Code generated by commands/generator/main.go
// DO NOT EDIT

/*
 * Copyright (c) 2021-present Fabien Potencier <fabien@symfony.com>
 *
 * This file is part of Symfony CLI project
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package commands

const LatestPhpMajorVersion = "` + fmt.Sprintf("%d.%d", latestVersion.Segments()[0], latestVersion.Segments()[1]) + `"
const LatestPhpMinorVersion = "` + latestVersion.Original() + `"
`)
}
