/*
 * Copyright (c) 2021-present Fabien Potencier <fabien@symfony.com>
 *
 * This file is part of Symfony CLI project
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package envs

import (
	"encoding/base64"
	"os"

	. "gopkg.in/check.v1"
)

type RemoteSuite struct{}

var _ = Suite(&RemoteSuite{})

func (s *RemoteSuite) TestMailer(c *C) {
	r := &Remote{}

	os.Setenv("MAILER_URL", "smtp://foobar")
	c.Assert(r.Mailer(), DeepEquals, Envs{"MAILER_ENABLED": "1"})
	os.Unsetenv("MAILER_URL")

	os.Setenv("MAILER_DSN", "smtp://foobar")
	c.Assert(r.Mailer(), DeepEquals, Envs{"MAILER_ENABLED": "1"})
	os.Unsetenv("MAILER_DSN")

	os.Setenv("MAILER_HOST", "foobar")
	c.Assert(r.Mailer(), DeepEquals, Envs{"MAILER_ENABLED": "1"})
	os.Unsetenv("MAILER_HOST")

	os.Setenv("PLATFORM_SMTP_HOST", "")
	c.Assert(r.Mailer(), DeepEquals, Envs{
		"MAILER_ENABLED":   "0",
		"MAILER_USER":      "",
		"MAILER_PASSWORD":  "",
		"MAILER_URL":       "null://localhost",
		"MAILER_DSN":       "null://localhost",
		"MAILER_HOST":      "localhost",
		"MAILER_PORT":      "25",
		"MAILER_TRANSPORT": "smtp",
		"MAILER_AUTH_MODE": "plain",
	})

	os.Setenv("PLATFORM_SMTP_HOST", "246.0.64.1")
	c.Assert(r.Mailer(), DeepEquals, Envs{
		"MAILER_ENABLED":   "1",
		"MAILER_USER":      "",
		"MAILER_PASSWORD":  "",
		"MAILER_URL":       "smtp://246.0.64.1:25?verify_peer=0",
		"MAILER_DSN":       "smtp://246.0.64.1:25?verify_peer=0",
		"MAILER_HOST":      "246.0.64.1",
		"MAILER_PORT":      "25",
		"MAILER_TRANSPORT": "smtp",
		"MAILER_AUTH_MODE": "plain",
	})

	os.Setenv("PLATFORM_SMTP_HOST", "246.0.64.1:25")
	c.Assert(r.Mailer(), DeepEquals, Envs{
		"MAILER_ENABLED":   "1",
		"MAILER_USER":      "",
		"MAILER_PASSWORD":  "",
		"MAILER_URL":       "smtp://246.0.64.1:25?verify_peer=0",
		"MAILER_DSN":       "smtp://246.0.64.1:25?verify_peer=0",
		"MAILER_HOST":      "246.0.64.1",
		"MAILER_PORT":      "25",
		"MAILER_TRANSPORT": "smtp",
		"MAILER_AUTH_MODE": "plain",
	})
}

func (s *RemoteSuite) TestSymfonyLegacyFallbacks(c *C) {
	c.Assert(os.Getenv("SYMFONY_ENV"), Equals, "")
	c.Assert(os.Getenv("APP_ENV"), Equals, "")

	r := &Remote{}
	c.Assert(r.Extra(), DeepEquals, Envs{"APP_ENV": "prod", "APP_DEBUG": "0", "SYMFONY_ENV": "prod", "SYMFONY_DEBUG": "0"})

	os.Setenv("SYMFONY_ENV", "dev")
	c.Check(r.Extra(), DeepEquals, Envs{"APP_ENV": "dev", "APP_DEBUG": "0", "SYMFONY_DEBUG": "0"})
	os.Unsetenv("SYMFONY_ENV")

	os.Setenv("APP_ENV", "dev")
	c.Check(r.Extra(), DeepEquals, Envs{"APP_DEBUG": "0", "SYMFONY_ENV": "dev", "SYMFONY_DEBUG": "0"})
	os.Unsetenv("APP_ENV")

	os.Setenv("SYMFONY_DEBUG", "1")
	c.Check(r.Extra(), DeepEquals, Envs{"APP_ENV": "prod", "SYMFONY_ENV": "prod", "APP_DEBUG": "1"})
	os.Unsetenv("SYMFONY_DEBUG")

	os.Setenv("APP_DEBUG", "1")
	c.Check(r.Extra(), DeepEquals, Envs{"APP_ENV": "prod", "SYMFONY_ENV": "prod", "SYMFONY_DEBUG": "1"})
	os.Unsetenv("APP_DEBUG")
}

func (s *RemoteSuite) TestIsWorker(c *C) {
	r := &Remote{}
	// By default: nothing
	c.Assert(r.Extra()["SYMFONY_IS_WORKER"], Equals, "")

	os.Setenv("PLATFORM_APPLICATION_NAME", "app")
	defer os.Unsetenv("PLATFORM_APPLICATION_NAME")
	c.Assert(r.Extra()["SYMFONY_IS_WORKER"], Equals, "0")

	os.Setenv("PLATFORM_APPLICATION_NAME", "app--mails")
	defer os.Unsetenv("PLATFORM_APPLICATION_NAME")
	c.Assert(r.Extra()["SYMFONY_IS_WORKER"], Equals, "1")
}

func (s *RemoteSuite) TestDefaultRoute(c *C) {
	r := &Remote{}
	_, hasUrl := r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"]
	_, hasProjectUrl := r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"]
	_, hasApplicationUrl := r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"]
	// By default: nothing
	c.Assert(hasUrl, Equals, false)
	c.Assert(hasProjectUrl, Equals, false)
	c.Assert(hasApplicationUrl, Equals, false)
	defer os.Unsetenv("PLATFORM_ROUTES")

	os.Setenv("PLATFORM_ROUTES", "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")
	// it should work the same from the application container or from the workers...
	for _, suffix := range []string{"", "--worker"} {
		os.Setenv("PLATFORM_APPLICATION_NAME", "app"+suffix)
		// {default} has priority
		for _, s := range []string{"APPLICATION_", "PROJECT_"} {
			c.Check(r.Extra()["SYMFONY_"+s+"DEFAULT_ROUTE_URL"], Equals, "https://symfony.com/")
			c.Check(r.Extra()["SYMFONY_"+s+"DEFAULT_ROUTE_SCHEME"], Equals, "https")
			c.Check(r.Extra()["SYMFONY_"+s+"DEFAULT_ROUTE_PORT"], Equals, "443")
			c.Check(r.Extra()["SYMFONY_"+s+"DEFAULT_ROUTE_HOST"], Equals, "symfony.com")
			c.Check(r.Extra()["SYMFONY_"+s+"DEFAULT_ROUTE_PATH"], Equals, "/")
		}

		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Unsetenv("PLATFORM_APPLICATION_NAME")
	}

	// it should work the same from the application container or from the workers...
	for _, suffix := range []string{"", "--worker"} {
		os.Setenv("PLATFORM_ROUTES", "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")
		// App does not match, https://{all}/ is the match for the project
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://git.sensiolabs.com/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		//nothing for the application as it does not match
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, "")

		os.Setenv("PLATFORM_APPLICATION_NAME", "test-app"+suffix)
		// then {all}
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://git.sensiolabs.com/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Setenv("PLATFORM_ROUTES", "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")
		// then {default}/path
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://tls-1-3.tucksaun.net/bar")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Setenv("PLATFORM_ROUTES", "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")
		// then {all}/path
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://test2.tucksaun.net/foo")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Setenv("PLATFORM_ROUTES", "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")
		// Fallback to first route available
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://test2.tucksaun.net/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Setenv("PLATFORM_ROUTES", "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")
		// https://{default}/ is the match
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://tls-1-3.tucksaun.net/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Setenv("PLATFORM_ROUTES", "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")
		// should work the same way in environments
		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://test-routes-ezema3q-c6yyhsnvmdojk.eu.s5y.io/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])

		os.Unsetenv("PLATFORM_APPLICATION_NAME")
	}

	// ... and with varnish?
	{
		os.Setenv("PLATFORM_ROUTES", "eyJodHRwOi8vcXNsemszaXJheS1qNGMyMno0dWJpNmZrLmV1LnM1eS5pby8iOiB7InByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbCwgImF0dHJpYnV0ZXMiOiB7fSwgInR5cGUiOiAicmVkaXJlY3QiLCAidG8iOiAiaHR0cHM6Ly9xc2x6azNpcmF5LWo0YzIyejR1Ymk2ZmsuZXUuczV5LmlvLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL3thbGx9LyJ9LCAiaHR0cHM6Ly9xc2x6azNpcmF5LWo0YzIyejR1Ymk2ZmsuZXUuczV5LmlvLyI6IHsicHJpbWFyeSI6IHRydWUsICJpZCI6IG51bGwsICJhdHRyaWJ1dGVzIjoge30sICJ0eXBlIjogInVwc3RyZWFtIiwgInVwc3RyZWFtIjogInZhcm5pc2giLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8ve2FsbH0vIn19")
		os.Setenv("PLATFORM_APPLICATION", "eyJzaXplIjogIkFVVE8iLCAiZGlzayI6IDUxMiwgImFjY2VzcyI6IHsic3NoIjogImNvbnRyaWJ1dG9yIn0sICJyZWxhdGlvbnNoaXBzIjogeyJyZWRpcyI6ICJyZWRpc2NhY2hlOnJlZGlzIiwgInJhYmJpdG1xIjogInF1ZXVlOnJhYmJpdG1xIiwgImRhdGFiYXNlIjogImRiOnBvc3RncmVzcWwifSwgIm1vdW50cyI6IHsiL3B1YmxpYy91cGxvYWRzIjogeyJzb3VyY2UiOiAic2VydmljZSIsICJzb3VyY2VfcGF0aCI6ICJ1cGxvYWRzIiwgInNlcnZpY2UiOiAiZmlsZXMifSwgIi92YXIiOiB7InNvdXJjZSI6ICJsb2NhbCIsICJzb3VyY2VfcGF0aCI6ICJ2YXIifX0sICJ0aW1lem9uZSI6IG51bGwsICJ2YXJpYWJsZXMiOiB7fSwgIm5hbWUiOiAiYXBwLS1tZXNzYWdlcyIsICJ0eXBlIjogInBocDo3LjMiLCAicnVudGltZSI6IHsiZXh0ZW5zaW9ucyI6IFsiYW1xcCIsICJhcGN1IiwgIm1ic3RyaW5nIiwgInNvZGl1bSIsICJjdHlwZSIsICJpY29udiIsICJwZG9fcGdzcWwiLCAicmVkaXMiLCAieHNsIl19LCAicHJlZmxpZ2h0IjogeyJlbmFibGVkIjogdHJ1ZSwgImlnbm9yZWRfcnVsZXMiOiBbXX0sICJ0cmVlX2lkIjogIjhiNGI2YmMyMTJkZTliMmM2MmQ3YjQ4ZTFkMjQ0OGY0ZGUwZjQyM2QiLCAic2x1Z19pZCI6ICJqNGMyMno0dWJpNmZrLWFwcC04YjRiNmJjMjEyZGU5YjJjNjJkN2I0OGUxZDI0NDhmNGRlMGY0MjNkIiwgImFwcF9kaXIiOiAiL2FwcCIsICJ3b3JrZXIiOiB7ImNvbW1hbmRzIjogeyJzdGFydCI6ICJzZXQgLXggLWUgXG5cbig+JjIgc3ltZm9ueS1kZXBsb3kpXG5waHAgYmluL2NvbnNvbGUgbWVzc2VuZ2VyOmNvbnN1bWUgYXN5bmMgLXZ2IC0tdGltZS1saW1pdCAzNjAwIC0tbWVtb3J5LWxpbWl0PTEyOE1cbiJ9fX0=")
		url, hasUrl := r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"]
		// App does not match
		c.Check(hasUrl, Equals, true)
		c.Check(url, Equals, "https://qslzk3iray-j4c22z4ubi6fk.eu.s5y.io/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, url)
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, "")
	}

	// when there's no clear winner, the one with `www` is the winner
	{
		// wwww prefix wins
		os.Setenv("PLATFORM_ROUTES", "eyJodHRwczovL21hcmtldC40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJ1cHN0cmVhbSIsICJ1cHN0cmVhbSI6ICJhcHAiLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8vbWFya2V0LnthbGx9LyJ9LCAiaHR0cHM6Ly93d3cuNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iOiB7InByaW1hcnkiOiB0cnVlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJ1cHN0cmVhbSIsICJ1cHN0cmVhbSI6ICJhcHAiLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8vd3d3LnthbGx9LyJ9LCAiaHR0cHM6Ly9hZG1pbi40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJ1cHN0cmVhbSIsICJ1cHN0cmVhbSI6ICJhcHAiLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8vYWRtaW4ue2FsbH0vIn0sICJodHRwOi8vbWFya2V0LjQzenFhc3RneDQtYzZ5eWhzbnZtZG9qay5ldS5zNXkuaW8vIjogeyJ0byI6ICJodHRwczovL21hcmtldC40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL21hcmtldC57YWxsfS8iLCAidHlwZSI6ICJyZWRpcmVjdCIsICJwcmltYXJ5IjogZmFsc2UsICJpZCI6IG51bGx9LCAiaHR0cDovL3d3dy40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsidG8iOiAiaHR0cHM6Ly93d3cuNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iLCAib3JpZ2luYWxfdXJsIjogImh0dHA6Ly93d3cue2FsbH0vIiwgInR5cGUiOiAicmVkaXJlY3QiLCAicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsfSwgImh0dHA6Ly9hZG1pbi40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsidG8iOiAiaHR0cHM6Ly9hZG1pbi40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL2FkbWluLnthbGx9LyIsICJ0eXBlIjogInJlZGlyZWN0IiwgInByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbH19")
		os.Setenv("PLATFORM_APPLICATION_NAME", "app")
		url, hasUrl := r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"]
		c.Check(hasUrl, Equals, true)
		c.Check(url, Equals, "https://www.43zqastgx4-c6yyhsnvmdojk.eu.s5y.io/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, url)
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, url)

		// the APEX should wine
		os.Setenv("PLATFORM_ROUTES", "eyJodHRwczovL21hcmtldC40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJ1cHN0cmVhbSIsICJ1cHN0cmVhbSI6ICJhcHAiLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8vbWFya2V0LnthbGx9LyJ9LCAiaHR0cHM6Ly80M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsicHJpbWFyeSI6IHRydWUsICJpZCI6IG51bGwsICJhdHRyaWJ1dGVzIjoge30sICJ0eXBlIjogInVwc3RyZWFtIiwgInVwc3RyZWFtIjogImFwcCIsICJvcmlnaW5hbF91cmwiOiAiaHR0cHM6Ly97YWxsfS8ifSwgImh0dHBzOi8vYWRtaW4uNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iOiB7InByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbCwgImF0dHJpYnV0ZXMiOiB7fSwgInR5cGUiOiAidXBzdHJlYW0iLCAidXBzdHJlYW0iOiAiYXBwIiwgIm9yaWdpbmFsX3VybCI6ICJodHRwczovL2FkbWluLnthbGx9LyJ9LCAiaHR0cDovL21hcmtldC40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsidG8iOiAiaHR0cHM6Ly9tYXJrZXQuNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iLCAib3JpZ2luYWxfdXJsIjogImh0dHA6Ly9tYXJrZXQue2FsbH0vIiwgInR5cGUiOiAicmVkaXJlY3QiLCAicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsfSwgImh0dHA6Ly9hZG1pbi40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyI6IHsidG8iOiAiaHR0cHM6Ly9hZG1pbi40M3pxYXN0Z3g0LWM2eXloc252bWRvamsuZXUuczV5LmlvLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL2FkbWluLnthbGx9LyIsICJ0eXBlIjogInJlZGlyZWN0IiwgInByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbH0sICJodHRwOi8vNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iOiB7InRvIjogImh0dHBzOi8vNDN6cWFzdGd4NC1jNnl5aHNudm1kb2prLmV1LnM1eS5pby8iLCAib3JpZ2luYWxfdXJsIjogImh0dHA6Ly97YWxsfS8iLCAidHlwZSI6ICJyZWRpcmVjdCIsICJwcmltYXJ5IjogZmFsc2UsICJpZCI6IG51bGx9fQ==")
		url, hasUrl = r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"]
		c.Check(hasUrl, Equals, true)
		c.Check(url, Equals, "https://43zqastgx4-c6yyhsnvmdojk.eu.s5y.io/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, url)
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, url)

		os.Unsetenv("PLATFORM_APPLICATION_NAME")
	}

	// When no matches, the {default} one wins
	{
		os.Setenv("PLATFORM_ROUTES", "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")
		url, hasUrl := r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"]
		c.Assert(url, Equals, "https://symfony.com/")
		c.Assert(hasUrl, Equals, true)
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, url)
	}

	// A more complicated use-case with multi app and several applications being public
	{
		os.Setenv("PLATFORM_ROUTES", "eyJodHRwOi8vd3d3LnN5bWZvbnljYXN0cy5jb20vIjogeyJwcmltYXJ5IjogZmFsc2UsICJpZCI6IG51bGwsICJhdHRyaWJ1dGVzIjoge30sICJ0eXBlIjogInJlZGlyZWN0IiwgInRvIjogImh0dHBzOi8vc3ltZm9ueWNhc3RzLmNvbS8iLCAib3JpZ2luYWxfdXJsIjogImh0dHA6Ly93d3cue2RlZmF1bHR9LyJ9LCAiaHR0cDovL2NoYWxsZW5nZXJ1bm5lci5zeW1mb255Y2FzdHMuY29tLyI6IHsicHJpbWFyeSI6IGZhbHNlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJyZWRpcmVjdCIsICJ0byI6ICJodHRwczovL2NoYWxsZW5nZXJ1bm5lci5zeW1mb255Y2FzdHMuY29tLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL2NoYWxsZW5nZXJ1bm5lci57ZGVmYXVsdH0vIn0sICJodHRwOi8vc3ltZm9ueWNhc3RzLmNvbS8iOiB7InByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbCwgImF0dHJpYnV0ZXMiOiB7fSwgInR5cGUiOiAicmVkaXJlY3QiLCAidG8iOiAiaHR0cHM6Ly9zeW1mb255Y2FzdHMuY29tLyIsICJvcmlnaW5hbF91cmwiOiAiaHR0cDovL3tkZWZhdWx0fS8ifSwgImh0dHBzOi8vd3d3LnN5bWZvbnljYXN0cy5jb20vIjogeyJwcmltYXJ5IjogZmFsc2UsICJpZCI6IG51bGwsICJhdHRyaWJ1dGVzIjoge30sICJ0eXBlIjogInJlZGlyZWN0IiwgInRvIjogImh0dHBzOi8vc3ltZm9ueWNhc3RzLmNvbS8iLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8vd3d3LntkZWZhdWx0fS8ifSwgImh0dHBzOi8vc3ltZm9ueWNhc3RzLmNvbS8iOiB7InByaW1hcnkiOiB0cnVlLCAiaWQiOiBudWxsLCAiYXR0cmlidXRlcyI6IHt9LCAidHlwZSI6ICJ1cHN0cmVhbSIsICJ1cHN0cmVhbSI6ICJzeW1mb255Y2FzdHMiLCAib3JpZ2luYWxfdXJsIjogImh0dHBzOi8ve2RlZmF1bHR9LyJ9LCAiaHR0cHM6Ly9jaGFsbGVuZ2VydW5uZXIuc3ltZm9ueWNhc3RzLmNvbS8iOiB7InByaW1hcnkiOiBmYWxzZSwgImlkIjogbnVsbCwgImF0dHJpYnV0ZXMiOiB7fSwgInR5cGUiOiAidXBzdHJlYW0iLCAidXBzdHJlYW0iOiAiY2hhbGxlbmdlcnVubmVyIiwgIm9yaWdpbmFsX3VybCI6ICJodHRwczovL2NoYWxsZW5nZXJ1bm5lci57ZGVmYXVsdH0vIn19")
		os.Setenv("PLATFORM_APPLICATION_NAME", "challengerunner")

		c.Check(r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"], Equals, "https://symfonycasts.com/")
		c.Check(r.Extra()["SYMFONY_APPLICATION_DEFAULT_ROUTE_URL"], Equals, "https://challengerunner.symfonycasts.com/")
		c.Check(r.Extra()["SYMFONY_DEFAULT_ROUTE_URL"], Equals, r.Extra()["SYMFONY_PROJECT_DEFAULT_ROUTE_URL"])
		os.Unsetenv("PLATFORM_APPLICATION_NAME")
	}
}

func (s *RemoteSuite) TestRelationships(c *C) {
	r := &Remote{}
	os.Setenv("PLATFORM_RELATIONSHIPS", "")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJzZWN1cml0eS1zZXJ2ZXIiOiBbeyJpcCI6ICIxNjkuMjU0LjI2LjIzMSIsICJob3N0IjogInNlY3VyaXR5LXNlcnZlci5pbnRlcm5hbCIsICJzY2hlbWUiOiAiaHR0cCIsICJwb3J0IjogODAsICJyZWwiOiAiaHR0cCJ9XSwgImRhdGFiYXNlIjogW3sidXNlcm5hbWUiOiAibWFpbiIsICJzY2hlbWUiOiAicGdzcWwiLCAiaXAiOiAiMTY5LjI1NC4xMjAuNDgiLCAiaG9zdCI6ICJkYXRhYmFzZS5pbnRlcm5hbCIsICJyZWwiOiAicG9zdGdyZXNxbCIsICJwYXRoIjogIm1haW4iLCAicXVlcnkiOiB7ImlzX21hc3RlciI6IHRydWV9LCAicGFzc3dvcmQiOiAibWFpbiIsICJwb3J0IjogNTQzMn1dfQ==")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"SECURITY_SERVER_IP":     "169.254.26.231",
		"SECURITY_SERVER_SCHEME": "http",
		"SECURITY_SERVER_HOST":   "security-server.internal",
		"SECURITY_SERVER_URL":    "http://security-server.internal:80",
		"SECURITY_SERVER_SERVER": "http://security-server.internal:80",
		"SECURITY_SERVER_PORT":   "80",
		"DATABASE_URL":           "postgres://main:main@database.internal:5432/main?sslmode=disable&charset=utf8",
		"DATABASE_DRIVER":        "postgres",
		"DATABASE_NAME":          "main",
		"DATABASE_DATABASE":      "main",
		"DATABASE_SERVER":        "postgres://database.internal:5432",
		"DATABASE_HOST":          "database.internal",
		"DATABASE_PORT":          "5432",
		"DATABASE_USER":          "main",
		"DATABASE_USERNAME":      "main",
		"DATABASE_PASSWORD":      "main",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJzZWN1cml0eS1zZXJ2ZXIiOiBbeyJpcCI6ICIxNjkuMjU0LjI2LjIzMSIsICJob3N0IjogInNlY3VyaXR5LXNlcnZlci5pbnRlcm5hbCIsICJzY2hlbWUiOiAiaHR0cCIsICJwb3J0IjogODAsICJyZWwiOiAiaHR0cCJ9XSwgImRhdGFiYXNlIjogW3sidXNlcm5hbWUiOiAibWFpbiIsICJzY2hlbWUiOiAicGdzcWwiLCAiaXAiOiAiMTY5LjI1NC4xMjAuNDgiLCAiaG9zdCI6ICJkYXRhYmFzZS5pbnRlcm5hbCIsICJyZWwiOiAicG9zdGdyZXNxbCIsICJwYXRoIjogIm1haW4iLCAicXVlcnkiOiB7ImlzX21hc3RlciI6IHRydWV9LCAicGFzc3dvcmQiOiAibWFpbiIsICJwb3J0IjogNTQzMn1dfQ==")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"SECURITY_SERVER_IP":     "169.254.26.231",
		"SECURITY_SERVER_SCHEME": "http",
		"SECURITY_SERVER_HOST":   "security-server.internal",
		"SECURITY_SERVER_URL":    "http://security-server.internal:80",
		"SECURITY_SERVER_SERVER": "http://security-server.internal:80",
		"SECURITY_SERVER_PORT":   "80",
		"DATABASE_URL":           "postgres://main:main@database.internal:5432/main?sslmode=disable&charset=utf8",
		"DATABASE_DRIVER":        "postgres",
		"DATABASE_NAME":          "main",
		"DATABASE_DATABASE":      "main",
		"DATABASE_SERVER":        "postgres://database.internal:5432",
		"DATABASE_HOST":          "database.internal",
		"DATABASE_PORT":          "5432",
		"DATABASE_USER":          "main",
		"DATABASE_USERNAME":      "main",
		"DATABASE_PASSWORD":      "main",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJkYXRhYmFzZSI6IFt7InVzZXJuYW1lIjogIm1haW4iLCAicGFzc3dvcmQiOiAibWFpbiIsICJzZXJ2aWNlIjogImRiIiwgImlwIjogIjE2OS4yNTQuMTI3LjExOSIsICJob3N0bmFtZSI6ICJ2dXJ6c3l5a2xzeGRiNnVhbDduaWVsYmtxdS5kYi5zZXJ2aWNlLl8uczV5LmlvIiwgImNsdXN0ZXIiOiAia2JqdW5hYjN6Z2V6Zy1kZW1vLWZrbHZjM2EiLCAiaG9zdCI6ICJkYXRhYmFzZS5pbnRlcm5hbCIsICJyZWwiOiAicG9zdGdyZXNxbCIsICJxdWVyeSI6IHsiaXNfbWFzdGVyIjogdHJ1ZX0sICJwYXRoIjogIm1haW4iLCAic2NoZW1lIjogInBnc3FsIiwgInR5cGUiOiAicG9zdGdyZXNxbDo5LjYiLCAicG9ydCI6IDU0MzJ9XX0=")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":      "postgres://main:main@database.internal:5432/main?sslmode=disable&charset=utf8&serverVersion=9.6",
		"DATABASE_DRIVER":   "postgres",
		"DATABASE_NAME":     "main",
		"DATABASE_DATABASE": "main",
		"DATABASE_SERVER":   "postgres://database.internal:5432",
		"DATABASE_HOST":     "database.internal",
		"DATABASE_PORT":     "5432",
		"DATABASE_USER":     "main",
		"DATABASE_USERNAME": "main",
		"DATABASE_PASSWORD": "main",
		"DATABASE_VERSION":  "9.6",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJyZWRpcyI6IFt7InVzZXJuYW1lIjogbnVsbCwgInBhc3N3b3JkIjogbnVsbCwgInNlcnZpY2UiOiAicmVkaXNfc2Vzc2lvbnMiLCAiZnJhZ21lbnQiOiBudWxsLCAiaXAiOiAiMTY5LjI1NC4yMy4yMDQiLCAiaG9zdG5hbWUiOiAidGVncTdqY3BqMjVuM3VnYjN0cm9rY2w1anEucmVkaXNfc2Vzc2lvbnMuc2VydmljZS5fLnM1eS5pbyIsICJwb3J0IjogNjM3OSwgImNsdXN0ZXIiOiAiN2NhbTRtbTUzN2ViZS1jbGVhbi11cC10bWVpd2hxIiwgImhvc3QiOiAicmVkaXMuaW50ZXJuYWwiLCAicmVsIjogInJlZGlzIiwgInBhdGgiOiBudWxsLCAicXVlcnkiOiB7fSwgInNjaGVtZSI6ICJyZWRpcyIsICJ0eXBlIjogInJlZGlzOjMuMiIsICJwdWJsaWMiOiBmYWxzZX1dLCAiZGF0YWJhc2UiOiBbeyJ1c2VybmFtZSI6ICJtYWluIiwgInBhc3N3b3JkIjogIjZlNjAyODg4NTc2NzAzMDMwZjUzYzE1NDA1MWJkNzc4IiwgInNlcnZpY2UiOiAibXlzcWwiLCAiaXAiOiAiMTY5LjI1NC4xMzQuMTEiLCAiaG9zdG5hbWUiOiAiaTNvNjJkbzV0eXh5MzV3NXdzdTY1YmdjcnUubXlzcWwuc2VydmljZS5fLnM1eS5pbyIsICJjbHVzdGVyIjogIjdjYW00bW01MzdlYmUtY2xlYW4tdXAtdG1laXdocSIsICJob3N0IjogImRhdGFiYXNlLmludGVybmFsIiwgInJlbCI6ICJtYWluIiwgInF1ZXJ5IjogeyJpc19tYXN0ZXIiOiB0cnVlfSwgInBhdGgiOiAibWFpbiIsICJzY2hlbWUiOiAibXlzcWwiLCAidHlwZSI6ICJteXNxbDoxMC4wIiwgInBvcnQiOiAzMzA2fV19")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":      "mysql://main:6e602888576703030f53c154051bd778@database.internal:3306/main?sslmode=disable&charset=utf8mb4&serverVersion=10.0.0-MariaDB",
		"DATABASE_DRIVER":   "mysql",
		"DATABASE_NAME":     "main",
		"DATABASE_DATABASE": "main",
		"DATABASE_SERVER":   "mysql://database.internal:3306",
		"DATABASE_HOST":     "database.internal",
		"DATABASE_PORT":     "3306",
		"DATABASE_USER":     "main",
		"DATABASE_USERNAME": "main",
		"DATABASE_PASSWORD": "6e602888576703030f53c154051bd778",
		"DATABASE_VERSION":  "10.0.0-MariaDB",
		"REDIS_URL":         "redis://redis.internal:6379",
		"REDIS_HOST":        "redis.internal",
		"REDIS_PORT":        "6379",
		"REDIS_SCHEME":      "redis",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJtcSI6IFt7InVzZXJuYW1lIjogImd1ZXN0IiwgInBhc3N3b3JkIjogImd1ZXN0IiwgInNlcnZpY2UiOiAibXlyYWJiaXRtcSIsICJpcCI6ICIxNjkuMjU0LjIxLjEzNCIsICJob3N0bmFtZSI6ICI3dWQydGRtcG5tYXoyZGQ3bHBmaW5henJkeS5teXJhYmJpdG1xLnNlcnZpY2UuXy5zNXkuaW8iLCAiY2x1c3RlciI6ICJyNWFpcW12bzViaHFlLW1hc3Rlci03cnF0d3RpIiwgImhvc3QiOiAibXEuaW50ZXJuYWwiLCAicmVsIjogInJhYmJpdG1xIiwgInNjaGVtZSI6ICJhbXFwIiwgInR5cGUiOiAicmFiYml0bXE6My43IiwgInBvcnQiOiA1NjcyfV0sICJkYXRhYmFzZSI6IFt7InVzZXJuYW1lIjogIm1haW4iLCAicGFzc3dvcmQiOiAibWFpbiIsICJzZXJ2aWNlIjogImRhdGFiYXNlIiwgImlwIjogIjE2OS4yNTQuMTUwLjE2IiwgImhvc3RuYW1lIjogInplNTM2bDZlZGtnbXRiZGVlcmttc3Zkenh1LmRhdGFiYXNlLnNlcnZpY2UuXy5zNXkuaW8iLCAiY2x1c3RlciI6ICJyNWFpcW12bzViaHFlLW1hc3Rlci03cnF0d3RpIiwgImhvc3QiOiAiZGF0YWJhc2UuaW50ZXJuYWwiLCAicmVsIjogInBvc3RncmVzcWwiLCAicXVlcnkiOiB7ImlzX21hc3RlciI6IHRydWV9LCAicGF0aCI6ICJtYWluIiwgInNjaGVtZSI6ICJwZ3NxbCIsICJ0eXBlIjogInBvc3RncmVzcWw6MTAiLCAicG9ydCI6IDU0MzJ9XX0=")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":           "postgres://main:main@database.internal:5432/main?sslmode=disable&charset=utf8&serverVersion=10",
		"DATABASE_SERVER":        "postgres://database.internal:5432",
		"DATABASE_DRIVER":        "postgres",
		"DATABASE_HOST":          "database.internal",
		"DATABASE_PORT":          "5432",
		"DATABASE_USER":          "main",
		"DATABASE_USERNAME":      "main",
		"DATABASE_PASSWORD":      "main",
		"DATABASE_VERSION":       "10",
		"DATABASE_DATABASE":      "main",
		"DATABASE_NAME":          "main",
		"MQ_SERVER":              "amqp://mq.internal:5672",
		"MQ_DSN":                 "amqp://guest:guest@mq.internal:5672",
		"MQ_URL":                 "amqp://guest:guest@mq.internal:5672",
		"MQ_SCHEME":              "amqp",
		"MQ_HOST":                "mq.internal",
		"MQ_PORT":                "5672",
		"MQ_USER":                "guest",
		"MQ_USERNAME":            "guest",
		"MQ_PASSWORD":            "guest",
		"MQ_MANAGEMENT_SERVER":   "http://mq.internal:15672",
		"MQ_MANAGEMENT_URL":      "http://guest:guest@mq.internal:15672",
		"MQ_MANAGEMENT_SCHEME":   "http",
		"MQ_MANAGEMENT_IP":       "169.254.21.134",
		"MQ_MANAGEMENT_HOST":     "mq.internal",
		"MQ_MANAGEMENT_PORT":     "15672",
		"MQ_MANAGEMENT_USER":     "guest",
		"MQ_MANAGEMENT_USERNAME": "guest",
		"MQ_MANAGEMENT_PASSWORD": "guest",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJkYXRhYmFzZSI6IFt7InNlcnZpY2UiOiAiZGF0YWJhc2VfbXlzcWxfbWFpbiIsICJpcCI6ICIxNjkuMjU0LjE4MC4yMjUiLCAiaG9zdG5hbWUiOiAibG03aDJld2Zvemc2cHVyamlhYmI1ZzRpYnUuZGF0YWJhc2VfbXlzcWxfbWFpbi5zZXJ2aWNlLl8uczV5LmlvIiwgImNsdXN0ZXIiOiAieHBzZm9vdGFwanc0YS1tYXN0ZXItN3JxdHd0aSIsICJob3N0IjogImRhdGFiYXNlLmludGVybmFsIiwgInJlbCI6ICJteXNxbCIsICJxdWVyeSI6IHt9LCAic2NoZW1lIjogIm15c3FsIiwgInR5cGUiOiAibXlzcWw6MTAuMSIsICJwb3J0IjogMzMwNn1dfQ==")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":      "mysql://database.internal:3306/main?sslmode=disable&charset=utf8mb4&serverVersion=10.1.0-MariaDB",
		"DATABASE_SERVER":   "mysql://database.internal:3306",
		"DATABASE_DRIVER":   "mysql",
		"DATABASE_HOST":     "database.internal",
		"DATABASE_PORT":     "3306",
		"DATABASE_VERSION":  "10.1.0-MariaDB",
		"DATABASE_DATABASE": "main",
		"DATABASE_NAME":     "main",
	})

	os.Setenv("PLATFORM_RELATIONSHIPS", "eyJkYXRhYmFzZSI6IFt7InNlcnZpY2UiOiAiZGF0YWJhc2VfbXlzcWxfbWFpbiIsICJpcCI6ICIxNjkuMjU0LjE4MC4yMjUiLCAiaG9zdG5hbWUiOiAibG03aDJld2Zvemc2cHVyamlhYmI1ZzRpYnUuZGF0YWJhc2VfbXlzcWxfbWFpbi5zZXJ2aWNlLl8uczV5LmlvIiwgImNsdXN0ZXIiOiAieHBzZm9vdGFwanc0YS1tYXN0ZXItN3JxdHd0aSIsICJob3N0IjogImRhdGFiYXNlLmludGVybmFsIiwgInJlbCI6ICJteXNxbCIsICJxdWVyeSI6IHt9LCAic2NoZW1lIjogIm15c3FsIiwgInR5cGUiOiAibXlzcWw6MTAuMiIsICJwb3J0IjogMzMwNn1dfQo=")
	// Maria DB 10.2 == 10.2.7
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":      "mysql://database.internal:3306/main?sslmode=disable&charset=utf8mb4&serverVersion=10.2.7-MariaDB",
		"DATABASE_SERVER":   "mysql://database.internal:3306",
		"DATABASE_DRIVER":   "mysql",
		"DATABASE_HOST":     "database.internal",
		"DATABASE_PORT":     "3306",
		"DATABASE_VERSION":  "10.2.7-MariaDB",
		"DATABASE_DATABASE": "main",
		"DATABASE_NAME":     "main",
	})

	os.Setenv("DATABASE_VERSION", "10.2.19-MariaDB")
	os.Setenv("DATABASE_CHARSET", "utf8")
	c.Assert(extractRelationshipsEnvs(r), DeepEquals, Envs{
		"DATABASE_URL":      "mysql://database.internal:3306/main?sslmode=disable&charset=utf8&serverVersion=10.2.19-MariaDB",
		"DATABASE_SERVER":   "mysql://database.internal:3306",
		"DATABASE_DRIVER":   "mysql",
		"DATABASE_HOST":     "database.internal",
		"DATABASE_PORT":     "3306",
		"DATABASE_VERSION":  "10.2.19-MariaDB",
		"DATABASE_DATABASE": "main",
		"DATABASE_NAME":     "main",
	})
	os.Unsetenv("DATABASE_VERSION")
	os.Unsetenv("DATABASE_CHARSET")
}

func (s *RemoteSuite) TestMySQLReadReplicaForDedicated(c *C) {
	r := &Remote{}
	value, err := os.ReadFile("testdata/dedicated/relationships_with_read_replica.json")
	if err != nil {
		panic(err)
	}
	if err := os.Setenv("PLATFORM_RELATIONSHIPS", base64.StdEncoding.EncodeToString(value)); err != nil {
		panic(err)
	}

	e := extractRelationshipsEnvs(r)

	c.Assert("mysql://mysql:xxx@dbread.internal:3306/main?sslmode=disable&charset=utf8mb4&serverVersion=10.6.0-MariaDB", DeepEquals, e["DBREAD_URL"])
	c.Assert("mysql://mysql:xxx@db.internal:3306/main?sslmode=disable&charset=utf8mb4&serverVersion=10.6.0-MariaDB", DeepEquals, e["DB_URL"])
}

func (s *RemoteSuite) TestNoIPsForDedicated(c *C) {
	r := &Remote{}
	value, err := os.ReadFile("testdata/dedicated/no_ips_for_dedicated.json")
	if err != nil {
		panic(err)
	}
	if err := os.Setenv("PLATFORM_RELATIONSHIPS", base64.StdEncoding.EncodeToString(value)); err != nil {
		panic(err)
	}

	rels := extractRelationshipsEnvs(r)
	c.Assert(rels, DeepEquals, Envs{
		"DATABASE_DATABASE":            "x_stg",
		"DATABASE_DRIVER":              "mysql",
		"DATABASE_HOST":                "127.0.0.1",
		"DATABASE_NAME":                "x_stg",
		"DATABASE_PASSWORD":            "x",
		"DATABASE_PORT":                "3306",
		"DATABASE_SERVER":              "mysql://127.0.0.1:3306",
		"DATABASE_URL":                 "mysql://xstg:x@127.0.0.1:3306/x_stg?sslmode=disable&charset=utf8mb4",
		"DATABASE_USER":                "xstg",
		"DATABASE_USERNAME":            "xstg",
		"RABBITMQ_VHOST":               "x_stg",
		"RABBITMQ_DSN":                 "amqp://x_stg:x@localhost:5672/x_stg",
		"RABBITMQ_HOST":                "localhost",
		"RABBITMQ_MANAGEMENT_HOST":     "localhost",
		"RABBITMQ_MANAGEMENT_PASSWORD": "x",
		"RABBITMQ_MANAGEMENT_PORT":     "15672",
		"RABBITMQ_MANAGEMENT_SCHEME":   "http",
		"RABBITMQ_MANAGEMENT_SERVER":   "http://localhost:15672",
		"RABBITMQ_MANAGEMENT_URL":      "http://x_stg:x@localhost:15672",
		"RABBITMQ_MANAGEMENT_USER":     "x_stg",
		"RABBITMQ_MANAGEMENT_USERNAME": "x_stg",
		"RABBITMQ_PASSWORD":            "x",
		"RABBITMQ_PORT":                "5672",
		"RABBITMQ_SCHEME":              "amqp",
		"RABBITMQ_SERVER":              "amqp://localhost:5672",
		"RABBITMQ_URL":                 "amqp://x_stg:x@localhost:5672/x_stg",
		"RABBITMQ_USER":                "x_stg",
		"RABBITMQ_USERNAME":            "x_stg",
		"REDISCACHE_HOST":              "localhost",
		"REDISCACHE_PORT":              "6379",
		"REDISCACHE_SCHEME":            "redis",
		"REDISCACHE_URL":               "redis://localhost:6379",
	})
}
