package core

import (
	"testing"

	"github.com/nalgeon/redka/internal/testx"
)

func TestValue(t *testing.T) {
	t.Run("bytes", func(t *testing.T) {
		v := Value([]byte("hello"))
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("hello"))
		testx.AssertEqual(t, v.String(), "hello")
		_, err := v.Bool()
		testx.AssertEqual(t, err.Error(), `strconv.ParseBool: parsing "hello": invalid syntax`)
		_, err = v.Int()
		testx.AssertEqual(t, err.Error(), `strconv.Atoi: parsing "hello": invalid syntax`)
		_, err = v.Float()
		testx.AssertEqual(t, err.Error(), `strconv.ParseFloat: parsing "hello": invalid syntax`)
	})
	t.Run("string", func(t *testing.T) {
		v := Value("hello")
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("hello"))
		testx.AssertEqual(t, v.String(), "hello")
		_, err := v.Bool()
		testx.AssertEqual(t, err.Error(), `strconv.ParseBool: parsing "hello": invalid syntax`)
		_, err = v.Int()
		testx.AssertEqual(t, err.Error(), `strconv.Atoi: parsing "hello": invalid syntax`)
		_, err = v.Float()
		testx.AssertEqual(t, err.Error(), `strconv.ParseFloat: parsing "hello": invalid syntax`)
	})
	t.Run("bool true", func(t *testing.T) {
		v := Value("1")
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("1"))
		testx.AssertEqual(t, v.String(), "1")
		vbool, err := v.Bool()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vbool, true)
		vint, err := v.Int()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vint, 1)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 1.0)
	})
	t.Run("bool false", func(t *testing.T) {
		v := Value("0")
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("0"))
		testx.AssertEqual(t, v.String(), "0")
		vbool, err := v.Bool()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vbool, false)
		vint, err := v.Int()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vint, 0)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 0.0)
	})
	t.Run("int", func(t *testing.T) {
		v := Value("42")
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("42"))
		testx.AssertEqual(t, v.String(), "42")
		_, err := v.Bool()
		testx.AssertEqual(t, err.Error(), `strconv.ParseBool: parsing "42": invalid syntax`)
		vint, err := v.Int()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vint, 42)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 42.0)
	})
	t.Run("float", func(t *testing.T) {
		v := Value("42.5")
		testx.AssertEqual(t, v.IsZero(), false)
		testx.AssertEqual(t, v.Bytes(), []byte("42.5"))
		testx.AssertEqual(t, v.String(), "42.5")
		_, err := v.Bool()
		testx.AssertEqual(t, err.Error(), `strconv.ParseBool: parsing "42.5": invalid syntax`)
		_, err = v.Int()
		testx.AssertEqual(t, err.Error(), `strconv.Atoi: parsing "42.5": invalid syntax`)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 42.5)
	})
	t.Run("empty string", func(t *testing.T) {
		v := Value("")
		testx.AssertEqual(t, v.IsZero(), true)
		testx.AssertEqual(t, v.Bytes(), []byte{})
		testx.AssertEqual(t, v.String(), "")
		vbool, err := v.Bool()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vbool, false)
		vint, err := v.Int()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vint, 0)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 0.0)
	})
	t.Run("nil", func(t *testing.T) {
		v := Value(nil)
		testx.AssertEqual(t, v.IsZero(), true)
		testx.AssertEqual(t, v.Bytes(), []byte(nil))
		testx.AssertEqual(t, v.String(), "")
		vbool, err := v.Bool()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vbool, false)
		vint, err := v.Int()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vint, 0)
		vfloat, err := v.Float()
		testx.AssertNoErr(t, err)
		testx.AssertEqual(t, vfloat, 0.0)
	})
}
