/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package awstasks

import (
	"context"
	"fmt"

	"github.com/aws/aws-sdk-go-v2/aws"
	"github.com/aws/aws-sdk-go-v2/service/ec2"
	ec2types "github.com/aws/aws-sdk-go-v2/service/ec2/types"
	"k8s.io/klog/v2"
	"k8s.io/kops/upup/pkg/fi"
	"k8s.io/kops/upup/pkg/fi/cloudup/awsup"
	"k8s.io/kops/upup/pkg/fi/cloudup/terraform"
	"k8s.io/kops/upup/pkg/fi/cloudup/terraformWriter"
)

// +kops:fitask
type RouteTable struct {
	Name      *string
	Lifecycle fi.Lifecycle

	ID  *string
	VPC *VPC

	// Shared is set if this is a shared RouteTable
	Shared *bool
	// Tags is a map of aws tags that are added to the RouteTable
	Tags map[string]string
}

var _ fi.CompareWithID = &RouteTable{}

func (e *RouteTable) CompareWithID() *string {
	return e.ID
}

func (e *RouteTable) Find(c *fi.CloudupContext) (*RouteTable, error) {
	ctx := c.Context()
	cloud := c.T.Cloud.(awsup.AWSCloud)

	var rt *ec2types.RouteTable
	var err error

	if e.ID != nil {
		rt, err = findRouteTableByID(ctx, cloud, *e.ID)
		if err != nil {
			return nil, err
		}
	}

	// Try finding by name
	if rt == nil && e.Tags["Name"] != "" {
		rt, err = findRouteTableByFilters(ctx, cloud, cloud.BuildFilters(e.Name))
		if err != nil {
			return nil, err
		}
	}

	// Try finding by shared cluster tag, along with role (so it isn't ambiguous)
	if rt == nil && e.Tags[awsup.TagNameKopsRole] != "" {
		var filters []ec2types.Filter
		filters = append(filters, ec2types.Filter{
			Name:   aws.String("tag-key"),
			Values: []string{"kubernetes.io/cluster/" + c.T.Cluster.Name},
		})
		filters = append(filters, ec2types.Filter{
			Name:   aws.String("tag:" + awsup.TagNameKopsRole),
			Values: []string{e.Tags[awsup.TagNameKopsRole]},
		})

		rt, err = findRouteTableByFilters(ctx, cloud, filters)
		if err != nil {
			return nil, err
		}
	}

	if rt == nil {
		return nil, nil
	}

	actual := &RouteTable{
		ID:   rt.RouteTableId,
		VPC:  &VPC{ID: rt.VpcId},
		Name: e.Name,
		Tags: intersectTags(rt.Tags, e.Tags),
	}
	klog.V(2).Infof("found matching RouteTable %q", *actual.ID)
	e.ID = actual.ID

	// Prevent spurious changes
	actual.Lifecycle = e.Lifecycle
	actual.Shared = e.Shared

	return actual, nil
}

func findRouteTableByID(ctx context.Context, cloud awsup.AWSCloud, id string) (*ec2types.RouteTable, error) {
	request := &ec2.DescribeRouteTablesInput{}
	request.RouteTableIds = []string{id}

	response, err := cloud.EC2().DescribeRouteTables(ctx, request)
	if err != nil {
		return nil, fmt.Errorf("error listing RouteTables: %v", err)
	}
	if response == nil || len(response.RouteTables) == 0 {
		return nil, nil
	}

	if len(response.RouteTables) != 1 {
		return nil, fmt.Errorf("found multiple RouteTables matching ID")
	}
	rt := response.RouteTables[0]

	return &rt, nil
}

func findRouteTableByFilters(ctx context.Context, cloud awsup.AWSCloud, filters []ec2types.Filter) (*ec2types.RouteTable, error) {
	request := &ec2.DescribeRouteTablesInput{}
	request.Filters = filters

	response, err := cloud.EC2().DescribeRouteTables(ctx, request)
	if err != nil {
		return nil, fmt.Errorf("error listing RouteTables: %v", err)
	}
	if response == nil || len(response.RouteTables) == 0 {
		return nil, nil
	}

	if len(response.RouteTables) != 1 {
		return nil, fmt.Errorf("found multiple RouteTables matching tags")
	}
	rt := response.RouteTables[0]
	return &rt, nil
}

func (e *RouteTable) Run(c *fi.CloudupContext) error {
	return fi.CloudupDefaultDeltaRunMethod(e, c)
}

func (s *RouteTable) CheckChanges(a, e, changes *RouteTable) error {
	if a == nil {
		if e.VPC == nil {
			return fi.RequiredField("VPC")
		}
	}
	if a != nil {
		if changes.VPC != nil && changes.VPC.ID != nil {
			return fi.CannotChangeField("VPC")
		}
	}
	return nil
}

func (_ *RouteTable) RenderAWS(t *awsup.AWSAPITarget, a, e, changes *RouteTable) error {
	ctx := context.TODO()
	if a == nil {
		vpcID := e.VPC.ID
		if vpcID == nil {
			return fi.RequiredField("VPC.ID")
		}

		klog.V(2).Infof("Creating RouteTable with VPC: %q", *vpcID)

		request := &ec2.CreateRouteTableInput{
			VpcId:             vpcID,
			TagSpecifications: awsup.EC2TagSpecification(ec2types.ResourceTypeRouteTable, e.Tags),
		}

		response, err := t.Cloud.EC2().CreateRouteTable(ctx, request)
		if err != nil {
			return fmt.Errorf("error creating RouteTable: %v", err)
		}

		rt := response.RouteTable
		e.ID = rt.RouteTableId
	}

	return t.AddAWSTags(*e.ID, e.Tags)
}

type terraformRouteTable struct {
	VPCID *terraformWriter.Literal `cty:"vpc_id"`
	Tags  map[string]string        `cty:"tags"`
}

func (_ *RouteTable) RenderTerraform(t *terraform.TerraformTarget, a, e, changes *RouteTable) error {
	// We use the role tag as a concise and stable identifier
	tag := e.Tags[awsup.TagNameKopsRole]
	if tag != "" {
		if err := t.AddOutputVariable("route_table_"+tag+"_id", e.TerraformLink()); err != nil {
			return err
		}
	}

	tf := &terraformRouteTable{
		VPCID: e.VPC.TerraformLink(),
		Tags:  e.Tags,
	}

	return t.RenderResource("aws_route_table", *e.Name, tf)
}

func (e *RouteTable) TerraformLink() *terraformWriter.Literal {
	return terraformWriter.LiteralProperty("aws_route_table", *e.Name, "id")
}
