// To change feature flags edit this file and run:
//     make gen-feature-toggles
//
// Alternatively run tests in:
//     pkg/services/featuremgmt/toggles_gen_test.go
// twice to generate and validate the feature flag files

package featuremgmt

import (
	"embed"
	"encoding/json"

	featuretoggle "github.com/grafana/grafana/pkg/apis/featuretoggle/v0alpha1"
)

var (
	// Register each toggle here
	standardFeatureFlags = []FeatureFlag{
		{
			Name:              "disableEnvelopeEncryption",
			Description:       "Disable envelope encryption (emergency only)",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaOperatorExperienceSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			Expression:        "false",
		},
		{
			Name:              "panelTitleSearch",
			Description:       "Search for dashboards using panel title",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
		},
		{
			Name:              "publicDashboardsEmailSharing",
			Description:       "Enables public dashboard sharing to be restricted to only allowed emails",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaOperatorExperienceSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "publicDashboardsScene",
			Description:  "Enables public dashboard rendering using scenes",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaOperatorExperienceSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:        "lokiExperimentalStreaming",
			Description: "Support new streaming approach for loki (prototype, needs special loki build)",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityLogsSquad,
		},
		{
			Name:           "featureHighlights",
			Description:    "Highlight Grafana Enterprise features",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaOperatorExperienceSquad,
			AllowSelfServe: true,
			Expression:     "false",
		},
		{
			Name:        "storage",
			Description: "Configurable storage for dashboards, datasources, and resources",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:           "correlations",
			Description:    "Correlations page",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaDatavizSquad,
			Expression:     "true", // enabled by default
			AllowSelfServe: true,
		},
		{
			Name:              "canvasPanelNesting",
			Description:       "Allow elements nesting",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaDatavizSquad,
			HideFromAdminPage: true,
		},
		{
			Name:        "logRequestsInstrumentedAsUnknown",
			Description: "Logs the path for requests that are instrumented as unknown",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaBackendGroup,
		},
		{
			Name:              "grpcServer",
			Description:       "Run the GRPC server",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
		},
		{
			Name:           "cloudWatchCrossAccountQuerying",
			Description:    "Enables cross-account querying in CloudWatch datasources",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          awsDatasourcesSquad,
			AllowSelfServe: true,
		},
		{
			Name:        "showDashboardValidationWarnings",
			Description: "Show warnings when dashboards do not validate against the schema",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaDashboardsSquad,
		},
		{
			Name:        "mysqlAnsiQuotes",
			Description: "Use double quotes to escape keyword in a MySQL query",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:        "alertingBacktesting",
			Description: "Rule backtesting API for alerting",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:         "editPanelCSVDragAndDrop",
			Description:  "Enables drag and drop for CSV and Excel files",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:           "logsContextDatasourceUi",
			Description:    "Allow datasource to provide custom UI for context view",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityLogsSquad,
			Expression:     "true", // turned on by default
			AllowSelfServe: true,
		},
		{
			Name:         "lokiShardSplitting",
			Description:  "Use stream shards to split queries into smaller subqueries",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:           "lokiQuerySplitting",
			Description:    "Split large interval queries into subqueries with smaller time intervals",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityLogsSquad,
			Expression:     "true", // turned on by default
			AllowSelfServe: true,
		},
		{
			Name:        "individualCookiePreferences",
			Description: "Support overriding cookie preferences per user",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaBackendGroup,
		},
		{
			Name:           "influxdbBackendMigration",
			Description:    "Query InfluxDB InfluxQL without the proxy",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaPartnerPluginsSquad,
			Expression:     "true", // enabled by default
			AllowSelfServe: false,
		},
		{
			Name:        "influxqlStreamingParser",
			Description: "Enable streaming JSON parser for InfluxDB datasource InfluxQL query language",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaPartnerPluginsSquad,
		},
		{
			Name:         "influxdbRunQueriesInParallel",
			Description:  "Enables running InfluxDB Influxql queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaPartnerPluginsSquad,
		},
		{
			Name:        "lokiLogsDataplane",
			Description: "Changes logs responses from Loki to be compliant with the dataplane specification.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityLogsSquad,
		},
		{
			Name:           "dataplaneFrontendFallback",
			Description:    "Support dataplane contract field name change for transformations and field name matchers where the name is different",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Expression:     "true",
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: true,
		},
		{
			Name:        "disableSSEDataplane",
			Description: "Disables dataplane specific processing in server side expressions.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityMetricsSquad,
		},
		{
			Name:              "unifiedRequestLog",
			Description:       "Writes error logs to the request logger",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaBackendGroup,
			Expression:        "true",
			HideFromAdminPage: true,
		},
		{
			Name:              "renderAuthJWT",
			Description:       "Uses JWT-based auth for rendering instead of relying on remote cache",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaOperatorExperienceSquad,
			HideFromAdminPage: true,
		},
		{
			Name:              "refactorVariablesTimeRange",
			Description:       "Refactor time range variables flow to reduce number of API calls made when query variables are chained",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaDashboardsSquad,
			HideFromAdminPage: true, // Non-feature, used to test out a bug fix that impacts the performance of template variables.
		},
		{
			Name:         "faroDatasourceSelector",
			Description:  "Enable the data source selector within the Frontend Apps section of the Frontend Observability",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        appO11ySquad,
		},
		{
			Name:         "enableDatagridEditing",
			Description:  "Enables the edit functionality in the datagrid panel",
			FrontendOnly: true,
			Stage:        FeatureStagePublicPreview,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "extraThemes",
			Description:  "Enables extra themes",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendPlatformSquad,
		},
		{
			Name:        "pluginsFrontendSandbox",
			Description: "Enables the plugins frontend sandbox",
			Stage:       FeatureStagePrivatePreview,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:           "recordedQueriesMulti",
			Description:    "Enables writing multiple items from a single query within Recorded Queries",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true",
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: false,
		},
		{
			Name:         "logsExploreTableVisualisation",
			Description:  "A table visualisation for logs in Explore",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true", // enabled by default,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:        "awsDatasourcesTempCredentials",
			Description: "Support temporary security credentials in AWS plugins for Grafana Cloud customers",
			Stage:       FeatureStageGeneralAvailability,
			Expression:  "true", // enabled by default
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:           "transformationsRedesign",
			Description:    "Enables the transformations redesign",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Expression:     "true", // enabled by default
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: true,
		},
		{
			Name:         "mlExpressions",
			Description:  "Enable support for Machine Learning in server-side expressions",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
		},
		{
			Name:            "datasourceAPIServers",
			Description:     "Expose some datasources as apiservers.",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "grafanaAPIServerWithExperimentalAPIs",
			Description:     "Register experimental APIs with the k8s API server, including all datasources",
			Stage:           FeatureStageExperimental,
			RequiresRestart: true,
			RequiresDevMode: true,
			Owner:           grafanaAppPlatformSquad,
		},
		{
			Name:            "provisioning",
			Description:     "Next generation provisioning... and git",
			Stage:           FeatureStageExperimental,
			RequiresRestart: true,
			Owner:           grafanaAppPlatformSquad,
		},
		{
			Name:            "grafanaAPIServerEnsureKubectlAccess",
			Description:     "Start an additional https handler and write kubectl options",
			Stage:           FeatureStageExperimental,
			RequiresDevMode: true,
			RequiresRestart: true,
			Owner:           grafanaAppPlatformSquad,
		},
		{
			Name:            "featureToggleAdminPage",
			Description:     "Enable admin page for managing feature toggles from the Grafana front-end. Grafana Cloud only.",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaOperatorExperienceSquad,
			RequiresRestart: true,
			HideFromDocs:    true,
		},
		{
			Name:        "awsAsyncQueryCaching",
			Description: "Enable caching for async queries for Redshift and Athena. Requires that the datasource has caching and async query support enabled",
			Stage:       FeatureStageGeneralAvailability,
			Expression:  "true", // enabled by default
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:        "permissionsFilterRemoveSubquery",
			Description: "Alternative permission filter implementation that does not use subqueries for fetching the dashboard folder",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:            "configurableSchedulerTick",
			Description:     "Enable changing the scheduler base interval via configuration option unified_alerting.scheduler_tick_interval",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaAlertingSquad,
			RequiresRestart: true,
			HideFromDocs:    true,
		},
		{
			Name:         "dashgpt",
			Description:  "Enable AI powered features in dashboards",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "aiGeneratedDashboardChanges",
			Description:  "Enable AI powered features for dashboards to auto-summary changes when saving",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:            "reportingRetries",
			Description:     "Enables rendering retries for the reporting feature",
			Stage:           FeatureStagePublicPreview,
			FrontendOnly:    false,
			Owner:           grafanaOperatorExperienceSquad,
			RequiresRestart: true,
		},
		{
			Name:        "sseGroupByDatasource",
			Description: "Send query to the same datasource in a single request when using server side expressions. The `cloudWatchBatchQueries` feature toggle should be enabled if this used with CloudWatch.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityMetricsSquad,
		},
		{
			Name:         "lokiRunQueriesInParallel",
			Description:  "Enables running Loki queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:              "externalServiceAccounts",
			Description:       "Automatic service account and token setup for plugins",
			HideFromAdminPage: true,
			Stage:             FeatureStagePublicPreview,
			Owner:             identityAccessTeam,
		},
		{
			Name:         "panelMonitoring",
			Description:  "Enables panel monitoring through logs and measurements",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true", // enabled by default
			Owner:        grafanaDatavizSquad,
			FrontendOnly: true,
		},
		{
			Name:              "enableNativeHTTPHistogram",
			Description:       "Enables native HTTP Histograms",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      false,
			Owner:             grafanaBackendServicesSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			RequiresRestart:   true,
		},
		{
			Name:              "disableClassicHTTPHistogram",
			Description:       "Disables classic HTTP Histogram (use with enableNativeHTTPHistogram)",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      false,
			Owner:             grafanaBackendServicesSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			RequiresRestart:   true,
		},
		{
			Name:         "formatString",
			Description:  "Enable format string transformer",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:            "kubernetesSnapshots",
			Description:     "Routes snapshot requests from /api to the /apis endpoint",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "kubernetesLibraryPanels",
			Description:     "Routes library panel requests from /api to the /apis endpoint",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:         "kubernetesDashboards",
			Description:  "Use the kubernetes API in the frontend for dashboards",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaDashboardsSquad,
			FrontendOnly: true,
			Expression:   "true", // enabled by default
		},
		{
			Name:            "kubernetesShortURLs",
			Description:     "Routes short url requests from /api to the /apis endpoint",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:        "dashboardDisableSchemaValidationV1",
			Description: "Disable schema validation for dashboards/v1",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAppPlatformSquad,
		},
		{
			Name:        "dashboardDisableSchemaValidationV2",
			Description: "Disable schema validation for dashboards/v2",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAppPlatformSquad,
		},
		{
			Name:        "dashboardSchemaValidationLogging",
			Description: "Log schema validation errors so they can be analyzed later",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAppPlatformSquad,
		},
		{
			Name:        "scanRowInvalidDashboardParseFallbackEnabled",
			Description: "Enable fallback parsing behavior when scan row encounters invalid dashboard JSON",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:            "datasourceQueryTypes",
			Description:     "Show query type endpoints in datasource API servers (currently hardcoded for testdata, expressions, and prometheus)",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "queryService",
			Description:     "Register /apis/query.grafana.app/ -- will eventually replace /api/ds/query",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaDatasourcesCoreServicesSquad,
			RequiresRestart: true, // Adds a route at startup
		},
		{
			Name:            "queryServiceRewrite",
			Description:     "Rewrite requests targeting /ds/query to the query service",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaDatasourcesCoreServicesSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:         "queryServiceFromUI",
			Description:  "Routes requests to the new query service",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatasourcesCoreServicesSquad,
			FrontendOnly: true, // and can change at startup
		},
		{
			Name:         "queryServiceFromExplore",
			Description:  "Routes explore requests to the new query service",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatasourcesCoreServicesSquad,
			FrontendOnly: true,
		},
		{
			Name:        "cloudWatchBatchQueries",
			Description: "Runs CloudWatch metrics queries as separate batches",
			Stage:       FeatureStagePublicPreview,
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:         "cachingOptimizeSerializationMemoryUsage",
			Description:  "If enabled, the caching backend gradually serializes query responses for the cache, comparing against the configured `[caching]max_value_mb` value as it goes. This can can help prevent Grafana from running out of memory while attempting to cache very large query responses.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaOperatorExperienceSquad,
			FrontendOnly: false,
		},
		{
			Name:         "addFieldFromCalculationStatFunctions",
			Description:  "Add cumulative and window functions to the add field from calculation transformation",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:        "alertmanagerRemoteSecondary",
			Description: "Enable Grafana to sync configuration and state with a remote Alertmanager.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:              "alertingProvenanceLockWrites",
			Description:       "Enables a feature to avoid issues with concurrent writes to the alerting provenance table in MySQL",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:        "alertmanagerRemotePrimary",
			Description: "Enable Grafana to have a remote Alertmanager instance as the primary Alertmanager.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:            "annotationPermissionUpdate",
			Description:     "Change the way annotation permissions work by scoping them to folders and dashboards.",
			Stage:           FeatureStageGeneralAvailability,
			RequiresDevMode: false,
			Expression:      "true", // enabled by default
			Owner:           identityAccessTeam,
		},
		{
			Name:         "extractFieldsNameDeduplication",
			Description:  "Make sure extracted field names are unique in the dataframe",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "dashboardSceneForViewers",
			Description:  "Enables dashboard rendering using Scenes for viewer roles",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "dashboardSceneSolo",
			Description:  "Enables rendering dashboards using scenes for solo panels",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "dashboardScene",
			Description:  "Enables dashboard rendering using scenes for all roles",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "dashboardNewLayouts",
			Description:  "Enables experimental new dashboard layouts",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:         "panelFilterVariable",
			Description:  "Enables use of the `systemPanelFilterVar` variable to filter panels in a dashboard",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			HideFromDocs: true,
		},
		{
			Name:         "pdfTables",
			Description:  "Enables generating table data as PDF in reporting",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: false,
			Owner:        grafanaOperatorExperienceSquad,
		},
		{
			Name:         "canvasPanelPanZoom",
			Description:  "Allow pan and zoom in canvas panel",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "timeComparison",
			Description:  "Enables time comparison option in supported panels",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "logsInfiniteScrolling",
			Description:  "Enables infinite scrolling for the Logs panel in Explore and Dashboards",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true",
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "logRowsPopoverMenu",
			Description:  "Enable filtering menu displayed when text of a log line is selected",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Expression:   "true",
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "pluginsSkipHostEnvVars",
			Description:  "Disables passing host environment variable to plugin processes",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:         "tableSharedCrosshair",
			Description:  "Enables shared crosshair in table panel",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "regressionTransformation",
			Description:  "Enables regression analysis transformation",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:              "kubernetesFeatureToggles",
			Description:       "Use the kubernetes API for feature toggle management in the frontend",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaOperatorExperienceSquad,
			AllowSelfServe:    false,
			HideFromAdminPage: true,
		},
		{
			Name:              "cloudRBACRoles",
			Description:       "Enabled grafana cloud specific RBAC roles",
			Stage:             FeatureStagePublicPreview,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			AllowSelfServe:    true,
			HideFromAdminPage: true,
			RequiresRestart:   true,
		},
		{
			Name:           "alertingQueryOptimization",
			Description:    "Optimizes eligible queries in order to reduce load on datasources",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaAlertingSquad,
			AllowSelfServe: false,
			Expression:     "false",
		},
		{
			Name:              "jitterAlertRulesWithinGroups",
			Description:       "Distributes alert rule evaluations more evenly over time, including spreading out rules within the same group. Disables sequential evaluation if enabled.",
			FrontendOnly:      false,
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: false,
			RequiresRestart:   true,
		},
		{
			Name:        "onPremToCloudMigrations",
			Description: "Enable the Grafana Migration Assistant, which helps you easily migrate various on-prem resources to your Grafana Cloud stack.",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaOperatorExperienceSquad,
			Expression:  "true",
		},
		{
			Name:        "secretsManagementAppPlatform",
			Description: "Enable the secrets management API and services under app platform",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaOperatorExperienceSquad,
		},
		{
			Name:        "secretsManagementAppPlatformUI",
			Description: "Enable the secrets management app platform UI",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaOperatorExperienceSquad,
		},
		{
			Name:         "alertingSaveStatePeriodic",
			Description:  "Writes the state periodically to the database, asynchronous to rule evaluation",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
		},
		{
			Name:         "alertingSaveStateCompressed",
			Description:  "Enables the compressed protobuf-based alert state storage",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
			Expression:   "true",
		},
		{
			Name:              "scopeApi",
			Description:       "In-development feature flag for the scope api using the app platform.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAppPlatformSquad,
			HideFromAdminPage: true,
			Expression:        "false",
		},
		{
			Name:              "useScopeSingleNodeEndpoint",
			Description:       "Use the single node endpoint for the scope api. This is used to fetch the scope parent node.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaOperatorExperienceSquad,
			Expression:        "false",
			FrontendOnly:      true,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "promQLScope",
			Description:       "In-development feature that will allow injection of labels into prometheus queries.",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaOSSBigTent,
			Expression:        "true",
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "logQLScope",
			Description:       "In-development feature that will allow injection of labels into loki queries.",
			Stage:             FeatureStagePrivatePreview,
			Owner:             grafanaObservabilityLogsSquad,
			Expression:        "false",
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "sqlExpressions",
			Description:  "Enables SQL Expressions, which can execute SQL queries against data source results.",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: false,
			Owner:        grafanaDatasourcesCoreServicesSquad,
		},
		{
			Name:         "sqlExpressionsColumnAutoComplete",
			Description:  "Enables column autocomplete for SQL Expressions",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDataProSquad,
		},
		{
			Name:         "groupToNestedTableTransformation",
			Description:  "Enables the group to nested table transformation",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default,
		},
		{
			Name:        "newPDFRendering",
			Description: "New implementation for the dashboard-to-PDF rendering",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaOperatorExperienceSquad,
			Expression:  "true", // enabled by default,
		},
		{
			Name:           "tlsMemcached",
			Description:    "Use TLS-enabled memcached in the enterprise caching feature",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaOperatorExperienceSquad,
			Expression:     "true",
			AllowSelfServe: false, // the non-tls implementation is slated for removal
		},
		{
			Name:            "kubernetesAggregator",
			Description:     "Enable grafana's embedded kube-aggregator",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:            "kubernetesAggregatorCapTokenAuth",
			Description:     "Enable CAP token based authentication in grafana's embedded kube-aggregator",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:              "groupByVariable",
			Description:       "Enable groupBy variable support in scenes dashboards",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "scopeFilters",
			Description:       "Enables the use of scope filters in Grafana",
			FrontendOnly:      false,
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			RequiresRestart:   false,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "oauthRequireSubClaim",
			Description:       "Require that sub claims is present in oauth tokens.",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "newDashboardWithFiltersAndGroupBy",
			Description:       "Enables filters and group by variables on all new dashboards. Variables are added only if default data source supports filtering.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:           "cloudWatchNewLabelParsing",
			Description:    "Updates CloudWatch label parsing to be more accurate",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          awsDatasourcesSquad,
			FrontendOnly:   false,
			AllowSelfServe: false,
		},
		{
			Name:            "disableNumericMetricsSortingInExpressions",
			Description:     "In server-side expressions, disable the sorting of numeric-kind metrics by their metric name or labels.",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaOSSBigTent,
			RequiresRestart: true,
		},
		{
			Name:              "grafanaManagedRecordingRules",
			Description:       "Enables Grafana-managed recording rules.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:           "queryLibrary",
			Description:    "Renamed feature toggle, enables Saved queries feature",
			Stage:          FeatureStagePrivatePreview,
			Owner:          grafanaSharingSquad,
			FrontendOnly:   false,
			AllowSelfServe: false,
		},
		{
			Name:           "savedQueries",
			Description:    "Enables Saved Queries feature",
			Stage:          FeatureStagePublicPreview,
			Owner:          grafanaSharingSquad,
			FrontendOnly:   false,
			AllowSelfServe: false,
		},
		{
			Name:         "logsExploreTableDefaultVisualization",
			Description:  "Sets the logs table as default visualisation in logs explore",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityLogsSquad,
			FrontendOnly: true,
		},
		{
			Name:         "newDashboardSharingComponent",
			Description:  "Enables the new sharing drawer design",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaSharingSquad,
			FrontendOnly: true,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "alertingListViewV2",
			Description:  "Enables the new alert list view design",
			Stage:        FeatureStagePrivatePreview,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:              "alertingDisableSendAlertsExternal",
			Description:       "Disables the ability to send alerts to an external Alertmanager datasource.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "preserveDashboardStateWhenNavigating",
			Description:       "Enables possibility to preserve dashboard variables and time range when navigating between dashboards",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "alertingCentralAlertHistory",
			Description:  "Enables the new central alert history.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:        "pluginProxyPreserveTrailingSlash",
			Description: "Preserve plugin proxy trailing slash.",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPluginsPlatformSquad,
			Expression:  "false", // disabled by default
		},
		{
			Name:        "azureMonitorPrometheusExemplars",
			Description: "Allows configuration of Azure Monitor as a data source that can provide Prometheus exemplars",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:        "pinNavItems",
			Description: "Enables pinning of nav items",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaFrontendPlatformSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:              "authZGRPCServer",
			Description:       "Enables the gRPC server for authorization",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:            "ssoSettingsLDAP",
			Description:     "Use the new SSO Settings API to configure LDAP",
			Stage:           FeatureStageGeneralAvailability,
			Owner:           identityAccessTeam,
			AllowSelfServe:  true,
			RequiresRestart: true,
			Expression:      "true", // enabled by default
		},
		{
			Name:              "zanzana",
			Description:       "Use openFGA as authorization engine.",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "reloadDashboardsOnParamsChange",
			Description:       "Enables reload of dashboards on scopes, time range and variables changes",
			FrontendOnly:      false,
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			RequiresRestart:   false,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "enableScopesInMetricsExplore",
			Description:       "Enables the scopes usage in Metrics Explore",
			FrontendOnly:      false,
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			RequiresRestart:   false,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "cloudWatchRoundUpEndTime",
			Description: "Round up end time for metric queries to the next minute to avoid missing data",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       awsDatasourcesSquad,
			Expression:  "true",
		},
		{
			Name:        "prometheusAzureOverrideAudience",
			Description: "Deprecated. Allow override default AAD audience for Azure Prometheus endpoint. Enabled by default. This feature should no longer be used and will be removed in the future.",
			Stage:       FeatureStageDeprecated,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "true", // Enabled by default for now
		}, {
			Name:         "alertingFilterV2",
			Description:  "Enable the new alerting search experience",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			HideFromDocs: true,
		},
		{
			Name:            "dataplaneAggregator",
			Description:     "Enable grafana dataplane aggregator",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:        "newFiltersUI",
			Description: "Enables new combobox style UI for the Ad hoc filters variable in scenes architecture",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaDashboardsSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:              "vizActionsAuth",
			Description:       "Allows authenticated API calls in actions",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaDatavizSquad,
			FrontendOnly:      true,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:         "alertingPrometheusRulesPrimary",
			Description:  "Uses Prometheus rules as the primary source of truth for ruler-enabled data sources",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "exploreLogsShardSplitting",
			Description:  "Used in Logs Drilldown to split queries into multiple queries based on the number of shards",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "exploreLogsAggregatedMetrics",
			Description:  "Used in Logs Drilldown to query by aggregated metrics",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "exploreLogsLimitedTimeRange",
			Description:  "Used in Logs Drilldown to limit the time range",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:              "appPlatformGrpcClientAuth",
			Description:       "Enables the gRPC client to authenticate with the App Platform by using ID & access tokens",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "groupAttributeSync",
			Description:  "Enable the groupsync extension for managing Group Attribute Sync feature",
			Stage:        FeatureStagePrivatePreview,
			Owner:        identityAccessTeam,
			HideFromDocs: true,
		},
		{
			Name:         "alertingQueryAndExpressionsStepMode",
			Description:  "Enables step mode for alerting queries and expressions",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
			Expression:   "true",
		},
		{
			Name:           "improvedExternalSessionHandling",
			Description:    "Enables improved support for OAuth external sessions. After enabling this feature, users might need to re-authenticate themselves.",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          identityAccessTeam,
			AllowSelfServe: true,
		},
		{
			Name:        "useSessionStorageForRedirection",
			Description: "Use session storage for handling the redirection after login",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       identityAccessTeam,
			Expression:  "true",
		},
		{
			Name:        "rolePickerDrawer",
			Description: "Enables the new role picker drawer design",
			Stage:       FeatureStageExperimental,
			Owner:       identityAccessTeam,
		},
		{
			Name:              "unifiedStorageSearch",
			Description:       "Enable unified storage search",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "unifiedStorageSearchSprinkles",
			Description:       "Enable sprinkles on unified storage search",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "managedDualWriter",
			Description:       "Pick the dual write mode from database configs",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "pluginsSriChecks",
			Description: "Enables SRI checks for plugin assets",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPluginsPlatformSquad,
			Expression:  "false", // disabled by default
		},
		{
			Name:        "unifiedStorageBigObjectsSupport",
			Description: "Enables to save big objects in blob storage",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:        "timeRangeProvider",
			Description: "Enables time pickers sync",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaFrontendPlatformSquad,
		},
		{
			Name:        "azureMonitorDisableLogLimit",
			Description: "Disables the log limit restriction for Azure Monitor when true. The limit is enabled by default.",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "false",
		},
		{
			Name:        "preinstallAutoUpdate",
			Description: "Enables automatic updates for pre-installed plugins",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPluginsPlatformSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:            "playlistsReconciler",
			Description:     "Enables experimental reconciler for playlists",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:              "passwordlessMagicLinkAuthentication",
			Description:       "Enable passwordless login via magic link authentication",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
		},
		{
			Name:         "exploreMetricsRelatedLogs",
			Description:  "Display Related Logs in Grafana Metrics Drilldown",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityMetricsSquad,
			FrontendOnly: true,
			HideFromDocs: false,
		},
		{
			Name:         "prometheusSpecialCharsInLabelValues",
			Description:  "Adds support for quotes and special characters in label values for Prometheus queries",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaOSSBigTent,
		},
		{
			Name:            "enableExtensionsAdminPage",
			Description:     "Enables the extension admin page regardless of development mode",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaPluginsPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:        "enableSCIM",
			Description: "Enables SCIM support for user and group management",
			Stage:       FeatureStagePublicPreview,
			Owner:       identityAccessTeam,
		},
		{
			Name:         "crashDetection",
			Description:  "Enables browser crash detection reporting to Faro.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "alertingUIOptimizeReducer",
			Description:  "Enables removing the reducer from the alerting UI when creating a new alert rule and using instant query",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaAlertingSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:        "azureMonitorEnableUserAuth",
			Description: "Enables user auth for Azure Monitor datasource only",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "true", // Enabled by default for now
		},
		{
			Name:              "alertingAIGenAlertRules",
			Description:       "Enable AI-generated alert rules.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingAIFeedback",
			Description:       "Enable AI-generated feedback from the Grafana UI.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingAIImproveAlertRules",
			Description:       "Enable AI-improve alert rules labels and annotations.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingAIGenTemplates",
			Description:       "Enable AI-generated alerting templates.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingEnrichmentPerRule",
			Description:       "Enable enrichment per rule in the alerting UI.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingAIAnalyzeCentralStateHistory",
			Description:       "Enable AI-analyze central state history.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:         "alertingNotificationsStepMode",
			Description:  "Enables simplified step mode in the notifications section",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
			Expression:   "true",
		},
		{
			Name:         "feedbackButton",
			Description:  "Enables a button to send feedback from the Grafana UI",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaOperatorExperienceSquad,
			HideFromDocs: true,
		},
		{
			Name:              "unifiedStorageSearchUI",
			Description:       "Enable unified storage search UI",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "elasticsearchCrossClusterSearch",
			Description: "Enables cross cluster search in the Elasticsearch datasource",
			Stage:       FeatureStagePublicPreview,
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:         "unifiedHistory",
			Description:  "Displays the navigation history so the user can navigate back to previous pages",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendPlatformSquad,
			FrontendOnly: true,
		},
		{
			// Remove this flag once Loki v4 is released and the min supported version is v3.0+,
			// since users on v2.9 need it to disable the feature, as it doesn't work for them.
			Name:        "lokiLabelNamesQueryApi",
			Description: "Defaults to using the Loki `/labels` API instead of `/series`",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaObservabilityLogsSquad,
			Expression:  "true",
		},
		{
			Name:        "investigationsBackend",
			Description: "Enable the investigations backend API",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAppPlatformSquad,
			Expression:  "false",
		},
		{
			Name:        "k8SFolderCounts",
			Description: "Enable folder's api server counts",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
			Expression:  "false",
		},
		{
			Name:        "k8SFolderMove",
			Description: "Enable folder's api server move",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
			Expression:  "false",
		},
		{
			Name:           "improvedExternalSessionHandlingSAML",
			Description:    "Enables improved support for SAML external sessions. Ensure the NameID format is correctly configured in Grafana for SAML Single Logout to function properly.",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          identityAccessTeam,
			AllowSelfServe: true,
		},
		{
			Name:         "teamHttpHeadersTempo",
			Description:  "Enables LBAC for datasources for Tempo to apply LBAC filtering of traces to the client requests for users in teams",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        identityAccessTeam,
		},
		{
			Name:         "templateVariablesUsesCombobox",
			Description:  "Use new **Combobox** component for template variables",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendPlatformSquad,
			FrontendOnly: true,
		},
		{
			Name:        "grafanaAdvisor",
			Description: "Enables Advisor app",
			Stage:       FeatureStagePrivatePreview,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:        "elasticsearchImprovedParsing",
			Description: "Enables less memory intensive Elasticsearch result parsing",
			Stage:       FeatureStageExperimental,
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:            "datasourceConnectionsTab",
			Description:     "Shows defined connections for a data source in the plugins detail page",
			Stage:           FeatureStagePrivatePreview,
			Owner:           grafanaPluginsPlatformSquad,
			RequiresDevMode: false,
			FrontendOnly:    true,
		},
		{
			Name:              "fetchRulesUsingPost",
			Description:       "Use a POST request to list rules by passing down the namespaces user has access to",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:         "newLogsPanel",
			Description:  "Enables the new logs panel in Explore",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:              "grafanaconThemes",
			Description:       "Enables the temporary themes for GrafanaCon",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaFrontendPlatformSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			RequiresRestart:   true,
			Expression:        "true",
		},
		{
			Name:         "alertingJiraIntegration",
			Description:  "Enables the new Jira integration for contact points in cloud alert managers.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
			HideFromDocs: true,
		},
		{
			Name:              "useScopesNavigationEndpoint",
			Description:       "Use the scopes navigation endpoint instead of the dashboardbindings endpoint",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaFrontendPlatformSquad,
			FrontendOnly:      true,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "scopeSearchAllLevels",
			Description:       "Enable scope search to include all levels of the scope node tree",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaFrontendPlatformSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "alertingRuleVersionHistoryRestore",
			Description:       "Enables the alert rule version history restore feature",
			FrontendOnly:      true,
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // enabled by default
		},
		{
			Name:              "newShareReportDrawer",
			Description:       "Enables the report creation drawer in a dashboard",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaOperatorExperienceSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "rendererDisableAppPluginsPreload",
			Description:       "Disable pre-loading app plugins when the request is coming from the renderer",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaOperatorExperienceSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
		},
		{
			Name:         "assetSriChecks",
			Description:  "Enables SRI checks for Grafana JavaScript assets",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendOpsWG,
			FrontendOnly: true,
		},
		{
			Name:        "alertRuleRestore",
			Description: "Enables the alert rule restore feature",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaAlertingSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:         "infinityRunQueriesInParallel",
			Description:  "Enables running Infinity queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaOSSBigTent,
		},
		{
			Name:              "inviteUserExperimental",
			Description:       "Renders invite user button along the app",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSharingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
		},
		{
			Name:         "alertingMigrationUI",
			Description:  "Enables the alerting migration UI, to migrate data source-managed rules to Grafana-managed rules",
			FrontendOnly: true,
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaAlertingSquad,
			Expression:   "true",
		},
		{
			Name:         "alertingImportYAMLUI",
			Description:  "Enables a UI feature for importing rules from a Prometheus file to Grafana-managed rules",
			FrontendOnly: true,
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaAlertingSquad,
			Expression:   "true",
		},
		{
			Name:              "unifiedStorageHistoryPruner",
			Description:       "Enables the unified storage history pruner",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // will be removed soon
		},
		{
			Name:        "azureMonitorLogsBuilderEditor",
			Description: "Enables the logs builder mode for the Azure Monitor data source",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "false",
		},
		{
			Name:        "localeFormatPreference",
			Description: "Specifies the locale so the correct format for numbers and dates can be shown",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaFrontendPlatformSquad,
		},
		{
			Name:              "unifiedStorageGrpcConnectionPool",
			Description:       "Enables the unified storage grpc connection pool",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "alertingRulePermanentlyDelete",
			Description:       "Enables UI functionality to permanently delete alert rules",
			FrontendOnly:      true,
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // enabled by default
		},
		{
			Name:              "alertingRuleRecoverDeleted",
			Description:       "Enables the UI functionality to recover and view deleted alert rules",
			FrontendOnly:      true,
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // enabled by default
		},
		{
			Name:         "multiTenantTempCredentials",
			Description:  "use multi-tenant path for awsTempCredentials",
			Stage:        FeatureStageExperimental,
			HideFromDocs: true,
			Owner:        awsDatasourcesSquad,
		},
		{
			Name:         "localizationForPlugins",
			Description:  "Enables localization for plugins",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaPluginsPlatformSquad,
			FrontendOnly: false,
		},
		{
			Name:         "unifiedNavbars",
			Description:  "Enables unified navbars",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaPluginsPlatformSquad,
			FrontendOnly: true,
			Expression:   "false", // enabled by default
		},
		{
			Name:         "logsPanelControls",
			Description:  "Enables a control component for the logs panel in Explore",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
			Expression:   "true",
		},
		{
			Name:         "metricsFromProfiles",
			Description:  "Enables creating metrics from profiles and storing them as recording rules",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "grafanaAssistantInProfilesDrilldown",
			Description:  "Enables integration with Grafana Assistant in Profiles Drilldown",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
			FrontendOnly: true,
			Expression:   "true",
		},
		{
			Name:        "postgresDSUsePGX",
			Description: "Enables using PGX instead of libpq for PostgreSQL datasource",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaOSSBigTent,
		},
		{
			Name:         "tempoAlerting",
			Description:  "Enables creating alerts from Tempo data source",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "pluginsAutoUpdate",
			Description:  "Enables auto-updating of users installed plugins",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:         "multiTenantFrontend",
			Description:  "Register MT frontend",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaFrontendPlatformSquad,
		},
		{
			Name:         "alertingListViewV2PreviewToggle",
			Description:  "Enables the alerting list view v2 preview toggle",
			FrontendOnly: true,
			Stage:        FeatureStagePrivatePreview,
			Owner:        grafanaAlertingSquad,
		},
		{
			Name:        "alertRuleUseFiredAtForStartsAt",
			Description: "Use FiredAt for StartsAt when sending alerts to Alertmaanger",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
			Expression:  "false",
		},
		{
			Name:              "alertingBulkActionsInUI",
			Description:       "Enables the alerting bulk actions in the UI",
			FrontendOnly:      true,
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // enabled by default
		},
		{
			Name:              "kubernetesAuthzApis",
			Description:       "Registers AuthZ /apis endpoint",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "kubernetesAuthzResourcePermissionApis",
			Description:       "Registers AuthZ resource permission /apis endpoints",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "kubernetesAuthnMutation",
			Description:       "Enables create, delete, and update mutations for resources owned by IAM identity",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "restoreDashboards",
			Description:       "Enables restore deleted dashboards feature",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaFrontendPlatformSquad,
			HideFromAdminPage: true,
			Expression:        "false",
		},
		{
			Name:              "skipTokenRotationIfRecent",
			Description:       "Skip token rotation if it was already rotated less than 5 seconds ago",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "true", // enabled by default
		},
		{
			Name:              "alertEnrichment",
			Description:       "Enable configuration of alert enrichments in Grafana Cloud.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingImportAlertmanagerAPI",
			Description:       "Enables the API to import Alertmanager configuration",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "alertingImportAlertmanagerUI",
			Description:       "Enables the UI to see imported Alertmanager configuration",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:         "sharingDashboardImage",
			Description:  "Enables image sharing functionality for dashboards",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaSharingSquad,
			HideFromDocs: true,
			FrontendOnly: true,
		},
		{
			Name:        "preferLibraryPanelTitle",
			Description: "Prefer library panel title over viz panel title.",
			Stage:       FeatureStagePrivatePreview,
			Owner:       grafanaDashboardsSquad,
			Expression:  "false",
		},
		{
			Name:        "tabularNumbers",
			Description: "Use fixed-width numbers globally in the UI",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaFrontendPlatformSquad,
			Expression:  "false",
		},
		{
			Name:         "newInfluxDSConfigPageDesign",
			Description:  "Enables new design for the InfluxDB data source configuration page",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaPartnerPluginsSquad,
			Expression:   "false",
		},
		{
			Name:              "enableAppChromeExtensions",
			Description:       "Set this to true to enable all app chrome extensions registered by plugins.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaPluginsPlatformSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
			Expression:        "false", // extensions will be disabled by default
		},
		{
			Name:              "foldersAppPlatformAPI",
			Description:       "Enables use of app platform API for folders",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaFrontendSearchNavOrganise,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
			Expression:        "false",
		},
		{
			Name:              "enablePluginImporter",
			Description:       "Set this to true to use the new PluginImporter functionality",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaPluginsPlatformSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
			Expression:        "false",
		},
		{
			Name:         "otelLogsFormatting",
			Description:  "Applies OTel formatting templates to displayed logs",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:              "alertingNotificationHistory",
			Description:       "Enables the notification history feature",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
		},
		{
			Name:              "pluginAssetProvider",
			Description:       "Allows decoupled core plugins to load from the Grafana CDN",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaPluginsPlatformSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			Expression:        "false",
			RequiresRestart:   true,
		},
		{
			Name:              "unifiedStorageSearchDualReaderEnabled",
			Description:       "Enable dual reader for unified storage search",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:         "dashboardDsAdHocFiltering",
			Description:  "Enables adhoc filtering support for the dashboard datasource",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaDataProSquad,
			FrontendOnly: true,
			Expression:   "true",
		},
		{
			Name:         "dashboardLevelTimeMacros",
			Description:  "Supports __from and __to macros that always use the dashboard level time range",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDashboardsSquad,
			FrontendOnly: true,
		},
		{
			Name:              "alertmanagerRemoteSecondaryWithRemoteState",
			Description:       "Starts Grafana in remote secondary mode pulling the latest state from the remote Alertmanager to avoid duplicate notifications.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:              "restrictedPluginApis",
			Description:       "Enables sharing a list of APIs with a list of plugins",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaPluginsPlatformSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			FrontendOnly:      true,
			Expression:        "false",
		},
		{
			Name:         "adhocFiltersInTooltips",
			Description:  "Enable adhoc filter buttons in visualization tooltips",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaDataProSquad,
			FrontendOnly: true,
			Expression:   "true",
		},
		{
			Name:         "favoriteDatasources",
			Description:  "Enable favorite datasources",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaPluginsPlatformSquad,
			FrontendOnly: true,
		},
		{
			Name:         "newLogContext",
			Description:  "New Log Context component",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityLogsSquad,
			FrontendOnly: true,
		},
		{
			Name:         "newClickhouseConfigPageDesign",
			Description:  "Enables new design for the Clickhouse data source configuration page",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaPartnerPluginsSquad,
			Expression:   "false",
		},
		{
			Name:              "unifiedStorageSearchAfterWriteExperimentalAPI",
			Description:       "Enable experimental search-after-write guarantees to unified-storage search endpoints",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
			HideFromDocs:      true,
			RequiresRestart:   true,
			Expression:        "false",
		},
		{
			Name:         "teamFolders",
			Description:  "Enables team folders functionality",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaFrontendSearchNavOrganise,
			Expression:   "false",
		},
		{
			Name:              "alertingTriage",
			Description:       "Enables the alerting triage feature",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaAlertingSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
			Expression:        "false",
		},
		{
			Name:         "graphiteBackendMode",
			Description:  "Enables the Graphite data source full backend mode",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaPartnerPluginsSquad,
			Expression:   "false",
		},
		{
			Name:         "azureResourcePickerUpdates",
			Description:  "Enables the updated Azure Monitor resource picker",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaPartnerPluginsSquad,
			Expression:   "false",
		},
		{
			Name:            "prometheusTypeMigration",
			Description:     "Checks for deprecated Prometheus authentication methods (SigV4 and Azure), installs the relevant data source, and migrates the Prometheus data sources",
			Stage:           FeatureStageExperimental,
			RequiresRestart: true,
			Owner:           grafanaPartnerPluginsSquad,
			Expression:      "false",
		},
	}
)

//go:embed toggles_gen.json
var f embed.FS

// Get the cached feature list (exposed as a k8s resource)
func GetEmbeddedFeatureList() (featuretoggle.FeatureList, error) {
	features := featuretoggle.FeatureList{}
	body, err := f.ReadFile("toggles_gen.json")
	if err == nil {
		err = json.Unmarshal(body, &features)
	}
	return features, err
}
