package proposallineage_test

import (
	"testing"

	"github.com/git-town/git-town/v22/internal/forge/forgedomain"
	"github.com/git-town/git-town/v22/internal/proposallineage"
	. "github.com/git-town/git-town/v22/pkg/prelude"
	"github.com/shoenig/test/must"
)

func TestRenderTree(t *testing.T) {
	t.Parallel()

	t.Run("all branches have proposals", func(t *testing.T) {
		t.Parallel()
		tree := proposallineage.TreeNodeWithProposal{
			Branch: "main",
			Children: []proposallineage.TreeNodeWithProposal{
				{
					Branch: "feature-1",
					Children: []proposallineage.TreeNodeWithProposal{
						{
							Branch:   "feature-1",
							Children: []proposallineage.TreeNodeWithProposal{},
							Proposal: Some(forgedomain.Proposal{
								Data: forgedomain.ProposalData{
									URL: "https://www.github.com/git-town/pull/2",
								},
							}),
						},
					},
					Proposal: Some(forgedomain.Proposal{
						Data: forgedomain.ProposalData{
							URL: "https://www.github.com/git-town/pull/1",
						},
					}),
				},
			},
			Proposal: None[forgedomain.Proposal](),
		}
		want := `

-------------------------
- main
  - https://www.github.com/git-town/pull/1 :point_left:
    - https://www.github.com/git-town/pull/2

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		have := proposallineage.RenderTree(tree, "feature-1")
		must.Eq(t, want, have)
	})

	t.Run("no proposals", func(t *testing.T) {
		t.Parallel()
		tree := proposallineage.TreeNodeWithProposal{
			Branch: "main",
			Children: []proposallineage.TreeNodeWithProposal{
				{
					Branch: "feature-1",
					Children: []proposallineage.TreeNodeWithProposal{
						{
							Branch: "feature-2",
							Children: []proposallineage.TreeNodeWithProposal{
								{
									Branch:   "feature-3",
									Children: []proposallineage.TreeNodeWithProposal{},
									Proposal: None[forgedomain.Proposal](),
								},
							},
							Proposal: None[forgedomain.Proposal](),
						},
					},
					Proposal: None[forgedomain.Proposal](),
				},
			},
			Proposal: None[forgedomain.Proposal](),
		}
		want := `

-------------------------
- main
  - feature-1
    - feature-2 :point_left:

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		have := proposallineage.RenderTree(tree, "feature-2")
		must.Eq(t, want, have)
	})

	t.Run("some branches have proposals", func(t *testing.T) {
		t.Parallel()
		tree := proposallineage.TreeNodeWithProposal{
			Branch: "main",
			Children: []proposallineage.TreeNodeWithProposal{
				{
					Branch: "feature-1",
					Children: []proposallineage.TreeNodeWithProposal{
						{
							Branch:   "feature-1A",
							Children: []proposallineage.TreeNodeWithProposal{},
							Proposal: None[forgedomain.Proposal](),
						},
						{
							Branch:   "feature-1B",
							Children: []proposallineage.TreeNodeWithProposal{},
							Proposal: Some(forgedomain.Proposal{
								Data: forgedomain.ProposalData{
									URL: "https://www.github.com/git-town/pull/1B",
								},
							}),
						},
					},
					Proposal: None[forgedomain.Proposal](),
				},
			},
			Proposal: None[forgedomain.Proposal](),
		}
		want := `

-------------------------
- main
  - feature-1 :point_left:
    - https://www.github.com/git-town/pull/1B

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		have := proposallineage.RenderTree(tree, "feature-1")
		must.Eq(t, want, have)
	})
}
